/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.remoting.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.UnknownHostException;
import java.rmi.AccessException;
import java.rmi.NotBoundException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.RMIClientSocketFactory;
import java.rmi.server.RMIServerSocketFactory;
import java.rmi.server.UnicastRemoteObject;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;
import javax.net.ServerSocketFactory;
import javax.net.SocketFactory;

import org.jboss.remoting.Remoting;
import org.jboss.remoting.loading.RemotingClassLoader;
import org.jboss.remoting.security.ServerSocketFactoryMBean;
import org.jboss.remoting.transport.rmi.RMIServerInvokerInf;
import org.jboss.serial.io.JBossObjectInputStream;
import org.jboss.serial.io.JBossObjectOutputStream;

/**
 * SecurityUtility provides a central point for making security sensitive calls.
 * 
 * It is divided into five sections:
 * 
 *   1. calls requiring FilePermissions
 *   2. calls requiring MBeanPermissions
 *   3. calls requiring PropertyPermissions
 *   4. calls requiring RuntimePermissions
 *   5. calls requiring SocketPermissions
 *   
 * When the SecurityUtility class is loaded, it checks for two conditions:
 * 
 *   1. there is no security manager
 *   2. the system property Remoting.SKIP_ACCESS_CONTROL ("skipAccessControl") is 
 *      set to true.
 *      
 * If either condition is true, then every method in SecurityUtility will
 * bypass its call to AccessController.doPrivileged().
 * 
 * @author <a href="ron.sigal@jboss.com">Ron Sigal</a>
 * @version $Revision: 1.1 $
 * <p>
 * Copyright Mar 31, 2008
 * </p>
 */
public class SecurityUtility
{
   static boolean skipAccessControl;
   
   static
   {
      try
      {
         skipAccessControl = ((Boolean)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               boolean b1 = System.getSecurityManager() == null;
               boolean b2 = Boolean.getBoolean(Remoting.SKIP_ACCESS_CONTROL);
//               System.out.println("security manager: " + System.getSecurityManager());
               return new Boolean(b1 || b2);
            }
         })).booleanValue();
      }
      catch (PrivilegedActionException e)
      {
         e.getCause().printStackTrace();
      }
   }
   
   
   static public boolean skipAccessControl()
   {
      return skipAccessControl;
   }
   
   
   ///////////////////////////////////////////////////////////////////////////////////////
   // FilePermission methods
   ///////////////////////////////////////////////////////////////////////////////////////
   
   static public File createTempFile(final String prefix, final String suffix, final boolean deleteOnExit) throws IOException
   {
      if (skipAccessControl)
      {
         File file =  File.createTempFile(prefix, suffix);
         if (deleteOnExit) file.deleteOnExit();
         return file;
      }
      
      try
      {
         return (File)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               File file =  File.createTempFile(prefix, suffix);
               if (deleteOnExit) file.deleteOnExit();
               return file;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   static public void deleteOnExit(final File file)
   {
      if (file == null)
         return;
      
      if (skipAccessControl)
      {
         file.deleteOnExit();
         return;
      }

      AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            file.deleteOnExit();
            return null;
         }
      });
   }
   
   static public boolean fileExists(final File file)
   {
      if (file == null)
         return false;
      
      if (skipAccessControl)
      {
         return file.exists();
      }

      return ((Boolean)AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return new Boolean(file.exists());
         }
      })).booleanValue();
   }

   
   static public boolean mkdirs(final File dir)
   {
      if (skipAccessControl)
      {
         return dir.mkdirs();
      }
      
      return ((Boolean) AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return new Boolean(dir.mkdirs());
         }
      })).booleanValue();
   }
   
   
   static public FileInputStream getFileInputStream(final File file) throws FileNotFoundException
   {
      if (skipAccessControl)
      {
         return new FileInputStream(file);
      }
      
      try
      {
         return (FileInputStream)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws FileNotFoundException
            {
               return new FileInputStream(file);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (FileNotFoundException) e.getCause();
      }
   }
   
   static public FileInputStream getFileInputStream(final String path) throws FileNotFoundException
   {
      if (skipAccessControl)
      {
         return new FileInputStream(path);
      }
      
      try
      {
         return (FileInputStream)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws FileNotFoundException
            {
               return new FileInputStream(path);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (FileNotFoundException) e.getCause();
      }
   }
   
   
   static public FileOutputStream getFileOutputStream(final File file)
   throws FileNotFoundException
   {
      if (skipAccessControl)
      {
         return new FileOutputStream(file);
      }
      
      try
      {
         return (FileOutputStream)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws FileNotFoundException
            {
               return new FileOutputStream(file);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (FileNotFoundException) e.getCause();
      }
   }
   
   
   static public FileOutputStream getFileOutputStream(final File file, final boolean append)
   throws FileNotFoundException
   {
      if (skipAccessControl)
      {
         return new FileOutputStream(file, append);
      }
      
      try
      {
         return (FileOutputStream)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws FileNotFoundException
            {
               return new FileOutputStream(file, append);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (FileNotFoundException) e.getCause();
      }
   }
   
   
   static public boolean canRead(final File file)
   {
      if (skipAccessControl)
      {
         return file.canRead();
      }
      
      return ((Boolean)AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return new Boolean(file.canRead());
         }
      })).booleanValue();
   }
   
   
   static public boolean createNewFile(final File file) throws IOException
   {
      if (skipAccessControl)
      {
         return file.createNewFile();
      }
      
      try
      {
         return ((Boolean)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return new Boolean(file.createNewFile());
            }
         })).booleanValue();
      }
      catch (Exception e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   ///////////////////////////////////////////////////////////////////////////////////////
   // MBeanPermission methods
   ///////////////////////////////////////////////////////////////////////////////////////
   
   static public MBeanServer createMBeanServer() throws Exception
   {
      if (skipAccessControl)
      {
         return MBeanServerFactory.createMBeanServer();
      }
      
      try
      {
         return (MBeanServer) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return MBeanServerFactory.createMBeanServer();
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (Exception) e.getCause();
      }   
   }
   
   
   static public ArrayList findMBeanServer(final String agentId)
   {
      if (skipAccessControl)
      {
         return MBeanServerFactory.findMBeanServer(agentId);
      }
      
      return (ArrayList)AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return MBeanServerFactory.findMBeanServer(agentId);
         }
      });
   }
   
   
   static public Object getMBeanAttribute(final MBeanServer server, final ObjectName objectName, final String attribute)
   throws Exception
   {
      if (skipAccessControl)
      {
         return server.getAttribute(objectName, attribute);
      }
      
      try
      {
         return AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return server.getAttribute(objectName, attribute);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (Exception) e.getCause();
      }  
   }
   
   
   static public MBeanServer getPlatformMBeanServer()
   throws NoSuchMethodException, IllegalAccessException, InvocationTargetException
   {
      if (skipAccessControl)
      {
         Class c = null;
         try
         {
            c = Class.forName("java.lang.management.ManagementFactory");
         }
         catch (Exception e)
         {
            System.out.println("Unable to access java.lang.management.ManagementFactory: must be using jdk 1.4");
            return null;
         }
         Method m = c.getMethod("getPlatformMBeanServer", new Class[] {});
         MBeanServer s = (MBeanServer) m.invoke(null, new Object[] {});
         return s;
      }
      
      try
      {
         return (MBeanServer) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run()
            throws NoSuchMethodException, IllegalAccessException, InvocationTargetException
            {
               Class c = null;
               try
               {
                  c = Class.forName("java.lang.management.ManagementFactory");
               }
               catch (Exception e)
               {
                  System.out.println("Unable to access java.lang.management.ManagementFactory: must be using jdk 1.4");
                  return null;
               }
               Method m = c.getMethod("getPlatformMBeanServer", new Class[] {});
               MBeanServer s = (MBeanServer) m.invoke(null, new Object[] {});
               return s;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
        Throwable cause = e.getCause();
        if (cause instanceof NoSuchMethodException)
           throw (NoSuchMethodException) cause;
        else if (cause instanceof IllegalAccessException)
           throw (IllegalAccessException) cause;
        else
           throw (InvocationTargetException) cause;
      }  
   }
   
   
   static public boolean isInstanceOf(final MBeanServer server, final ObjectName objectName, final String className)
   throws InstanceNotFoundException
   {
      if (skipAccessControl)
      {
         return server.isInstanceOf(objectName, className);
      }
      
      try
      {
         return ((Boolean)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return new Boolean(server.isInstanceOf(objectName, className));
            }
         })).booleanValue();
      }
      catch (PrivilegedActionException e)
      {
         throw (InstanceNotFoundException) e.getCause();
      }
   }
   
   
   static public void registerMBean(final MBeanServer server, final Object o, final ObjectName name)
   throws Exception
   {
      if (skipAccessControl)
      {
         server.registerMBean(o, name);
         return;
      }
      
      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               server.registerMBean(o, name);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (Exception) e.getCause();
      }
   }
     
   
   static public void unregisterMBean(final MBeanServer server, final ObjectName name)
   throws Exception
   {
      if (skipAccessControl)
      {
         server.unregisterMBean(name);
         return;
      }
      
      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               server.unregisterMBean(name);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (Exception) e.getCause();
      }
   }
   

   ///////////////////////////////////////////////////////////////////////////////////////
   // PropertyPermission methods
   ///////////////////////////////////////////////////////////////////////////////////////
   
   static public String getSystemProperty(final String name, final String defaultValue)
   {
      if (skipAccessControl)
         return System.getProperty(name, defaultValue);
         
      String value = null;
      try
      {
         value = (String)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return System.getProperty(name, defaultValue);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (RuntimeException) e.getCause();
      }
      
      return value;
   }
   
   
   static public String getSystemProperty(final String name)
   {
      if (skipAccessControl)
         return System.getProperty(name);
      
      String value = null;
      try
      {
         value = (String)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return System.getProperty(name);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (RuntimeException) e.getCause();
      }
      
      return value;
   }
   
   
   static public void setSystemProperty(final String name, final String value)
   {
      if (skipAccessControl)
      {
         System.setProperty(name, value);
         return;
      }
      
      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return System.setProperty(name, value);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (RuntimeException) e.getCause();
      }
   }
   
   
   ///////////////////////////////////////////////////////////////////////////////////////
   // RuntimePermission methods
   ///////////////////////////////////////////////////////////////////////////////////////
   
   static public RemotingClassLoader createRemotingClassLoader(final ClassLoader remotingClassLoader, final ClassLoader userClassLoader)
   {
      if (skipAccessControl)
      {
         return new RemotingClassLoader(remotingClassLoader, userClassLoader);
      }

      return (RemotingClassLoader)AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return new RemotingClassLoader(remotingClassLoader, userClassLoader);
         }
      });
   }
   
   
   static public Object forName(final String className) throws ClassNotFoundException
   {
      if (skipAccessControl)
      {
         return Class.forName(className);
      }
      
      try
      {
         return  AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return Class.forName(className);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (ClassNotFoundException) e.getCause();
      }
   }
   
   
   static public ClassLoader getClassLoader(final Class c)
   {
      if (skipAccessControl)
      {
         return c.getClassLoader();
      }

      return (ClassLoader)AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return c.getClassLoader();
         }
      });
   }
   
   
   static public ClassLoader getContextClassLoader(final Thread thread)
   {
      if (skipAccessControl)
      {
         return thread.getContextClassLoader();
      }

      return (ClassLoader) AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return thread.getContextClassLoader();
         }
      });
   }
   
   
   static public ClassLoader getSystemClassLoader()
   {
      if (skipAccessControl)
      {
         return ClassLoader.getSystemClassLoader();
      }

      return (ClassLoader)AccessController.doPrivileged( new PrivilegedAction()
      {
         public Object run()
         {
            return ClassLoader.getSystemClassLoader();
         }
      });
   }
   
   
   static public Method getMethod(final Class c, final String name, final Class[] parameterTypes)
   throws NoSuchMethodException
   {
      if (skipAccessControl)
      {
         return c.getMethod(name, parameterTypes);
      }

      try
      {
         return (Method) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws NoSuchMethodException
            {
               return c.getMethod(name, parameterTypes);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (NoSuchMethodException) e.getCause();
      }
   }
   
   
   static public Method getDeclaredMethod(final Class c, final String name, final Class[] parameterTypes)
   throws NoSuchMethodException
   {
      if (skipAccessControl)
      {
         Method m = c.getDeclaredMethod(name, parameterTypes);
         m.setAccessible(true);
         return m;
      }

      try
      {
         return (Method) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws NoSuchMethodException
            {
               Method m = c.getDeclaredMethod(name, parameterTypes);
               m.setAccessible(true);
               return m;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (NoSuchMethodException) e.getCause();
      }
   }
   
   static public Object readObject(final ObjectInputStream ois)
   throws IOException, ClassNotFoundException
   {
      if (skipAccessControl || !(ois instanceof JBossObjectInputStream))
      {
         return ois.readObject();
      }

      try
      {
         return AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException, ClassNotFoundException
            {
               return ois.readObject();
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         Throwable cause = e.getCause();
         if (cause instanceof IOException)
            throw (IOException) cause;
         else if (cause instanceof ClassNotFoundException)
            throw (ClassNotFoundException) cause;
         else
            throw (RuntimeException) cause;
      }
   }
   
   static public void writeObject(final ObjectOutputStream oos, final Object o)
   throws IOException
   {
      if (skipAccessControl || !(oos instanceof JBossObjectOutputStream))
      {
         oos.writeObject(o);
         return;
      }

      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               oos.writeObject(o);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         Throwable cause = e.getCause();
         if (cause instanceof IOException)
            throw (IOException) cause;
         else
            throw (RuntimeException) cause;
      }
   }
   
   
   ///////////////////////////////////////////////////////////////////////////////////////
   // SocketPermission methods
   ///////////////////////////////////////////////////////////////////////////////////////
   
   static public Socket accept(final ServerSocket ss) throws IOException
   {
      if (skipAccessControl)
      {
         return ss.accept();
      }
      
      try
      {
          return (Socket)AccessController.doPrivileged( new PrivilegedExceptionAction()
          {
             public Object run() throws Exception
             {
                 return ss.accept();
             }
          });
      }
      catch (PrivilegedActionException e)
      {
          throw (IOException) e.getCause();
      }
   }
   
   
   static public void bind(final ServerSocket ss, final SocketAddress address)
   throws IOException
   {
      if (skipAccessControl)
      {
         ss.bind(address);
         return;
      }

      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               ss.bind(address);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }


   static public void bind(final ServerSocket ss, final SocketAddress address,
                           final int backlog) throws IOException
   {
      if (skipAccessControl)
      {
         ss.bind(address, backlog);
         return;
      }
      
      try
      {
          AccessController.doPrivileged( new PrivilegedExceptionAction()
          {
             public Object run() throws Exception
             {
                ss.bind(address, backlog);
                return null;
             }
          });
      }
      catch (PrivilegedActionException e)
      {
          throw (IOException) e.getCause();
      }
   }
   
   
   static public void connect(final Socket socket, final InetSocketAddress address)
   throws IOException
   {
      if (skipAccessControl)
      {
         socket.connect(address);
         return;
      }
      
      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               socket.connect(address);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }   
   }
   
   
   static public void connect(final Socket socket, final InetSocketAddress address, final int timeout)
   throws IOException
   {
      if (skipAccessControl)
      {
         socket.connect(address, timeout);
         return;
      }
      
      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               socket.connect(address, timeout);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }   
   }
   
   
   static public void connect(final HttpURLConnection conn) throws IOException
   {
      if (skipAccessControl)
      {
         conn.connect();
         return;
      }

      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               conn.connect();
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public ServerSocket createServerSocket(final ServerSocketFactoryMBean ssf) throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket();
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return ssf.createServerSocket();
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public ServerSocket createServerSocket(final ServerSocketFactoryMBean ssf,
                                                 final int port) throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket(port);
      }
      
      try
      {
          return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
          {
             public Object run() throws Exception
             {
                 return ssf.createServerSocket(port);
             }
          });
      }
      catch (PrivilegedActionException e)
      {
          throw (IOException) e.getCause();
      }
   }

   
   static public ServerSocket createServerSocket(final ServerSocketFactoryMBean ssf,
                                                 final int port, final int backlog)
   throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket(port, backlog);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return ssf.createServerSocket(port, backlog);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }

   
   static public ServerSocket createServerSocket(final ServerSocketFactoryMBean ssf,
                                                 final int port, final int backlog,
                                                 final InetAddress inetAddress)
   throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket(port, backlog, inetAddress);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return ssf.createServerSocket(port, backlog, inetAddress);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }

   
   static public ServerSocket createServerSocket(final ServerSocketFactory ssf) throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket();
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return ssf.createServerSocket();
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public ServerSocket createServerSocket(final ServerSocketFactory ssf,
                                                 final int port) throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket(port);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return ssf.createServerSocket(port);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }


   static public ServerSocket createServerSocket(final ServerSocketFactory ssf,
                                                 final int port, final int backlog)
   throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket(port, backlog);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return ssf.createServerSocket(port, backlog);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }


   static public ServerSocket createServerSocket(final ServerSocketFactory ssf,
                                                 final int port, final int backlog,
                                                 final InetAddress inetAddress)
   throws IOException
   {
      if (skipAccessControl)
      {
         return ssf.createServerSocket(port, backlog, inetAddress);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return ssf.createServerSocket(port, backlog, inetAddress);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public ServerSocket createServerSocket(final int port) throws IOException
   {
      if (skipAccessControl)
      {
         return new ServerSocket(port);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return new ServerSocket(port);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }


   static public ServerSocket createServerSocket(final int port, final int backlog)
   throws IOException
   {
      if (skipAccessControl)
      {
         return new ServerSocket(port, backlog);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return new ServerSocket(port, backlog);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }


   static public ServerSocket createServerSocket(final int port, final int backlog,
                                                 final InetAddress inetAddress)
   throws IOException
   {
      if (skipAccessControl)
      {
         return new ServerSocket(port, backlog, inetAddress);
      }

      try
      {
         return (ServerSocket)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return new ServerSocket(port, backlog, inetAddress);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }


   static public Socket createSocket(final String host, final int port) throws IOException
   {
      if (skipAccessControl)
      {
         return new Socket(host, port);
      }
      
      try
      {
          return (Socket)AccessController.doPrivileged( new PrivilegedExceptionAction()
          {
             public Object run() throws IOException
             {
                return new Socket(host, port);
             }
          });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }


   static public Socket createSocket(final SocketFactory sf, final String host, final int port)
   throws IOException
   {
      if (skipAccessControl)
      {
         return sf.createSocket(host, port);
      }
      
      try
      {
          return (Socket)AccessController.doPrivileged( new PrivilegedExceptionAction()
          {
             public Object run() throws IOException
             {
                return sf.createSocket(host, port);
             }
          });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public InetAddress getLocalHost() throws UnknownHostException
   {
      if (skipAccessControl)
      {
         try
         {
            return InetAddress.getLocalHost();
         }
         catch (IOException e)
         {
            return InetAddress.getByName("127.0.0.1");
         }
      }

      try
      {
         return (InetAddress) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               try
               {
                  return InetAddress.getLocalHost();
               }
               catch (IOException e)
               {
                  return InetAddress.getByName("127.0.0.1");
               }
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (UnknownHostException) e.getCause();
      }
   }

   
   static public String getLocalHostName() throws UnknownHostException
   {
      if (skipAccessControl)
      {
         return getLocalHost().getHostName();
      }

      try
      {
         return (String) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               InetAddress address = null;
               try
               {
                  address = InetAddress.getLocalHost();
               }
               catch (IOException e)
               {
                  address = InetAddress.getByName("127.0.0.1");
               }
               
               return address.getHostName();
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (UnknownHostException) e.getCause();
      }
   }
   
   
   static public InetAddress getAddressByName(final String host) throws UnknownHostException
   {
      if (skipAccessControl)
      {
         return InetAddress.getByName(host);
      }
      
      try
      {
         return (InetAddress)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return InetAddress.getByName(host);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (UnknownHostException) e.getCause();
      }
   }
   
   
   static public OutputStream getOutputStream(final HttpURLConnection conn)
   throws IOException
   {
      if (skipAccessControl)
      {
         return conn.getOutputStream();
      }
      
      try
      {
         return (OutputStream)AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return conn.getOutputStream();
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public int getResponseCode(final HttpURLConnection conn)
   throws IOException
   {
      if (skipAccessControl)
      {
         return conn.getResponseCode();
      }
      
      try
      {
         return ((Integer) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return new Integer(conn.getResponseCode());
            }
         })).intValue();
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public Object callTransport(final RMIServerInvokerInf server, final Object payload)
   throws IOException
   {
      if (skipAccessControl)
      {
         return server.transport(payload);
      }

      try
      {
         return AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               return server.transport(payload);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      } 
   }


   static public Registry createRegistry(final int port) throws RemoteException
   {
      if (skipAccessControl)
      {
         return LocateRegistry.createRegistry(port);
      }
      
      try
      {
         return (Registry) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws RemoteException
            {
               return LocateRegistry.createRegistry(port);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (RemoteException) e.getCause();
      } 
   }
   
   
   static public Remote exportObject(final Remote object,
                                     final int port,
                                     final RMIClientSocketFactory csf,
                                     final RMIServerSocketFactory ssf)
   throws RemoteException
   {
      if (skipAccessControl)
      {
         return UnicastRemoteObject.exportObject(object, port, csf, ssf);
      }
      
      try
      {
         return (Remote) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws RemoteException
            {
               return UnicastRemoteObject.exportObject(object, port, csf, ssf);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (RemoteException) e.getCause();
      }
   }

   
   static public Registry getRegistry(final int port) throws RemoteException
   {
      if (skipAccessControl)
      {
         return LocateRegistry.getRegistry(port);
      }
      
      try
      {
         return (Registry) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws RemoteException
            {
               return LocateRegistry.getRegistry(port);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (RemoteException) e.getCause();
      } 
   }
   
   
   static public Remote lookup(final Registry registry, final String name)
   throws RemoteException, NotBoundException
   {
      if (skipAccessControl)
      {
         return registry.lookup(name);
      }
      
      try
      {
         return (Remote) AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return registry.lookup(name);
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         Throwable cause = e.getCause();
         if (cause instanceof RemoteException)
            throw (RemoteException) cause;
         else
            throw (NotBoundException) cause;
      }
   }
   
   
   static public void rebind(final Registry registry, final String name, final Remote object)
   throws IOException
   {
      if (skipAccessControl)
      {
         registry.rebind(name, object);
         return;
      }
      
      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws IOException
            {
               registry.rebind(name, object);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         throw (IOException) e.getCause();
      }
   }
   
   
   static public void unbind(final Registry registry, final String name)
   throws  AccessException, RemoteException, NotBoundException
   {
      if (skipAccessControl)
      {
         registry.unbind(name);
         return;
      }
      
      try
      {
         AccessController.doPrivileged( new PrivilegedExceptionAction()
         {
            public Object run() throws AccessException, RemoteException, NotBoundException
            {
               registry.unbind(name);
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         Throwable cause = e.getCause();
         if (cause instanceof AccessException)
            throw (AccessException) cause;
         else if (cause instanceof RemoteException)
            throw (RemoteException) cause;
         else
            throw (NotBoundException) cause;
      }
   }
}