/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.remoting.samples.callback;

import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.callback.Callback;
import org.jboss.remoting.callback.HandleCallbackException;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;

import java.util.Iterator;
import java.util.List;

/**
 * Sample client showing how to register for callbacks from remoting server.
 * The callback server we will make invocations to will generate random
 * callback messages that we can receive.
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class CallbackClient
{
   // Default locator values
   private static String transport = "socket";
   private static String host = "localhost";
   private static int port = 5400;

   private Client remotingClient;

   // callback server, which receives push callbacks.
   private Connector callbackServerConnector;

   /**
    * Create the remoting client to use to make calls on the remoting server.
    *
    * @param locatorURI
    * @throws Exception
    */
   public void createRemotingClient(String locatorURI) throws Exception
   {
      InvokerLocator locator = new InvokerLocator(locatorURI);
      System.out.println("Calling remoting server with locator uri of: " + locatorURI);

      remotingClient = new Client(locator);
      remotingClient.connect();
   }

   /**
    * Makes a call on the remoting server.
    *
    * @throws Throwable
    */
   public void makeInvocation() throws Throwable
   {
      Object response = remotingClient.invoke("Do something", null);
      System.out.println("Invocation response: " + response);
   }

   /**
    * Shows how to register for pull callbacks and then get any callbacks
    * that are waiting on the server.
    *
    * @throws Throwable
    */
   public void testPullCallback() throws Throwable
   {
      // our impplementation of the InvokerCallbackHandler interface, which
      // is defined as an inner class below.
      CallbackHandler callbackHandler = new CallbackHandler();
      // by passing only the callback handler, will indicate pull callbacks
      remotingClient.addListener(callbackHandler);

      // the callback server generates callback messages on its own and need
      // to give a few seconds to generate them.
      Thread.currentThread().sleep(2000);

      // call also make regular invocations on the server at any time.
      makeInvocation();

      // go get our callbacks residing on the server
      List callbacks = remotingClient.getCallbacks(callbackHandler);
      Iterator itr = callbacks.iterator();
      while(itr.hasNext())
      {
         Callback callbackObject = (Callback) itr.next();
         System.out.println("Pull Callback value = " + callbackObject.getCallbackObject());
      }

      // remove callback handler from server
      remotingClient.removeListener(callbackHandler);
   }

   /**
    * Shows how to register for push callbacks where the server will
    * callback on our callback handler when it generates a callback.
    *
    * @throws Throwable
    */
   public void testPushCallback() throws Throwable
   {
      // First, need to create remoting server to receive callbacks.
      // Using loctor with port value one higher than the target server
      String callbackLocatorURI = transport + "://" + host + ":" + (port + 1);
      InvokerLocator callbackLocator = new InvokerLocator(callbackLocatorURI);

      // call to create remoting server to the
      // callbacks generated by the target remoting server.
      setupServer(callbackLocator);

      CallbackHandler callbackHandler = new CallbackHandler();
      // Callback handle object will be passed back as part of the callback object
      // This can be used at the time the callback is received to identify context
      String callbackHandleObject = "myCallbackHandleObject";
      // by passing the callback handler and callback locator, will indicate push callbacks
      // could also have not included callbackHandleObject as last parameter if did not care
      // callback handle (i.e. addListener(callbackHandler, callbackLocator); )
      remotingClient.addListener(callbackHandler, callbackLocator, callbackHandleObject);

      // need to wait for brief moment so server can callback
      Thread.sleep(2000);

      // remove callback handler from server
      remotingClient.removeListener(callbackHandler);

      // shut down callback server
      callbackServerConnector.stop();
      callbackServerConnector.destroy();
   }

   /**
    * Sets up the callback server that the target remoting server can call on
    * with the callbacks that it generates.
    *
    * @param locator
    * @throws Exception
    */
   public void setupServer(InvokerLocator locator) throws Exception
   {
      System.out.println("Starting remoting server with locator uri of: " + locator);
      callbackServerConnector = new Connector();
      callbackServerConnector.setInvokerLocator(locator.getLocatorURI());
      callbackServerConnector.start();
   }


   /**
    * Can pass transport and port to be used as parameters.
    * Valid transports are 'rmi' and 'socket'.
    *
    * @param args
    */
   public static void main(String[] args)
   {
      if(args != null && args.length == 2)
      {
         transport = args[0];
         port = Integer.parseInt(args[1]);
      }
      String locatorURI = transport + "://" + host + ":" + port;
      CallbackClient client = new CallbackClient();
      try
      {
         client.createRemotingClient(locatorURI);

         // make call to test pull callbacks
         client.testPullCallback();

         // make call to test push callbacks
         client.testPushCallback();
      }
      catch(Throwable e)
      {
         e.printStackTrace();
      }
   }

   /**
    * Our implementation of the InvokerCallbackHandler.  Simply
    * prints out the callback object's message upon receiving it.
    */
   public class CallbackHandler implements InvokerCallbackHandler
   {
      /**
       * Will take the callback and print out its values.
       *
       * @param callback
       * @throws org.jboss.remoting.callback.HandleCallbackException
       *
       */
      public void handleCallback(Callback callback) throws HandleCallbackException
      {
         System.out.println("Received push callback.");
         System.out.println("Received callback value of: " + callback.getCallbackObject());
         System.out.println("Received callback handle object of: " + callback.getCallbackHandleObject());
         System.out.println("Received callback server invoker of: " + callback.getServerLocator());
      }
   }

}