/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.spi.helpers;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashSet;
import java.util.Set;

import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.logging.Logger;
import org.jboss.scanning.spi.Scanner;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.scanning.spi.ScanningPlugin;

/**
 * Url based scanner.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public abstract class UrlScanner extends AbstractScanner
{
   private Logger log = Logger.getLogger(getClass());
   private URL[] roots;
   private boolean ignoreIndexedHandles;
   private boolean ignoreRuntimeScanning;

   protected UrlScanner(URL... roots)
   {
      if (roots == null)
         throw new IllegalArgumentException("Null roots");
      this.roots = roots;
   }

   protected URL[] getRoots()
   {
      return roots;
   }

   /**
    * Should we actually perform the scan.
    *
    * @return true if we should scan, false otherwise
    */
   public abstract boolean doScan();

   @SuppressWarnings("unchecked")
   public void scan() throws Exception
   {
      if (doScan())
      {
         Set<ScanningPlugin> federatedPlugins = new HashSet<ScanningPlugin>();
         for (ScanningPlugin plugin : getPlugins())
         {
            ScanningHandle handle = createHandle(plugin);
            ScanningPluginWrapper wrapper = new ScanningPluginWrapper(plugin);
            federatedPlugins.add(wrapper);
            if (ignoreIndexedHandles == false)
            {
               String fileName = plugin.getFileName();
               for (URL root : roots)
               {
                  InputStream is = getInputStream(root, Scanner.META_INF + fileName);
                  if (is != null)
                  {
                     ScanningHandle pre = plugin.readHandle(is);
                     handle.merge(pre);
                     wrapper.addURL(root); // exclude this url -- we already have pre-existing handle
                  }
               }
            }
         }
         // do we scan, or should everything be in pre-indexed handles
         if (ignoreRuntimeScanning == false && federatedPlugins.isEmpty() == false)
            scan(federatedPlugins);
      }
   }

   protected ResourceFilter[] getRecurseFilters(ScanningPlugin[] plugins)
   {
      boolean explicitFilter = false;
      ResourceFilter[] recurseFilters = new ResourceFilter[plugins.length];
      for (int i = 0; i < plugins.length; i++)
      {
         ResourceFilter filter = plugins[i].getRecurseFilter();
         recurseFilters[i] = filter != null ? filter : AllResourceFilter.INSTANCE;
         explicitFilter |= filter != null;
      }
      return explicitFilter ? recurseFilters : null;
   }

   /**
    * Do scan using these plugins.
    *
    * These are already wrapper plugins,
    * which are aware of existing pre-indexed handles,
    * so you don't have to write any additional logic to
    * avoid or include pre-indexed handles.
    *
    * @param plugins the plugins used to scan
    * @throws Exception for any error
    */
   protected abstract void scan(Set<ScanningPlugin> plugins) throws Exception;

   protected InputStream getInputStream(URL url, String key)
   {
      try
      {
         String urlString = url.toExternalForm();
         if (urlString.endsWith("/") == false)
            urlString += "/";
         urlString += key;
         URL scanURL = new URL(urlString);
         return scanURL.openStream();
      }
      catch (IOException e)
      {
         if (log.isTraceEnabled())
            log.trace("Cannot access scanning .mcs file: " + e);
         return null;
      }
   }

   /**
    * Create a handle.
    *
    * This is a hook that allows you to store new handles.
    * e.g. into a deployment unit
    *
    * @param plugin the plugin
    * @return new handle
    */
   protected abstract ScanningHandle createHandle(ScanningPlugin plugin);

   /**
    * Don't use existing indexed handles.
    *
    * @param ignoreIndexedHandles the handles check flag
    */
   public void setIgnoreIndexedHandles(boolean ignoreIndexedHandles)
   {
      this.ignoreIndexedHandles = ignoreIndexedHandles;
   }

   /**
    * Don't scan at runtime.
    *
    * @param ignoreRuntimeScanning runtime scanning flag
    */
   public void setIgnoreRuntimeScanning(boolean ignoreRuntimeScanning)
   {
      this.ignoreRuntimeScanning = ignoreRuntimeScanning;
   }
}
