package org.jboss.set.mavendependencyupdater.core.processingstrategies;

import org.commonjava.maven.atlas.ident.ref.ArtifactRef;
import org.jboss.logging.Logger;
import org.jboss.set.mavendependencyupdater.DependencyEvaluator.ComponentUpgrade;
import org.jboss.set.mavendependencyupdater.configuration.Configuration;
import org.jboss.set.mavendependencyupdater.core.aggregation.ComponentUpgradeAggregator;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Prints upgradable dependencies report to stdout or to given file.
 * <p>
 * Non thread safe.
 */
public class TextReportProcessingStrategy implements UpgradeProcessingStrategy {

    protected static final Logger LOG = Logger.getLogger(TextReportProcessingStrategy.class);

    protected static final String PROJECT_URL = "https://github.com/jboss-set/maven-dependency-updater";
    protected static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("HH:mm:ss z yyyy-MM-dd");
    protected static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    final protected File pomFile;
    final protected Configuration configuration;
    protected PrintStream outputStream;
    protected File outputFile;

    public TextReportProcessingStrategy(Configuration configuration, File pomFile) {
        this.configuration = configuration;
        this.pomFile = pomFile;
    }

    public TextReportProcessingStrategy(Configuration configuration, File pomFile, String outputFileName) {
        this(configuration, pomFile);
        this.outputFile = new File(outputFileName);
    }

    public TextReportProcessingStrategy(Configuration configuration, File pomFile, PrintStream outputStream) {
        this(configuration, pomFile);
        this.outputStream = outputStream;
    }

    protected void initOutputStream() {
        try {
            if (this.outputStream == null) {
                if (this.outputFile == null) {
                    this.outputStream = System.out;
                } else {
                    this.outputStream = new PrintStream(this.outputFile);
                }
            }
        } catch (FileNotFoundException e) {
            throw new RuntimeException("Can't create output stream", e);
        }
    }

    @Override
    public boolean process(List<ComponentUpgrade> upgrades) throws Exception {
        try {
            if (upgrades.size() == 0) {
                LOG.info("No components to upgrade.");
                return true;
            }
            initOutputStream();

            List<ComponentUpgrade> sortedUpgrades =
                    upgrades.stream().sorted(new ComponentUpgradeComparator())
                            .collect(Collectors.toList());
            List<ComponentUpgrade> aggregatedUpgrades = ComponentUpgradeAggregator.aggregateComponentUpgrades(pomFile, sortedUpgrades);

            outputStream.println("Generated at " + DATE_TIME_FORMATTER.format(ZonedDateTime.now()));
            outputStream.println();
            outputStream.println("Searched in following repositories:\n");
            for (Map.Entry<String, String> entry : configuration.getRepositories().entrySet()) {
                outputStream.println("* " + entry.getKey() + ": " + entry.getValue());
            }
            outputStream.println();
            outputStream.println("Possible upgrades:\n");

            for (ComponentUpgrade upgrade : aggregatedUpgrades) {
                ArtifactRef artifact = upgrade.getArtifact();
                outputStream.println(
                        String.format("%s:%s:%s -> %s (%s) - %s",
                                artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersionString(),
                                upgrade.getNewVersion(), upgrade.getRepository(),
                                upgrade.getFirstSeen() == null ?
                                        "new" : "since " + upgrade.getFirstSeen().format(DATE_FORMATTER)
                        ));
            }

            outputStream.println("\n" + aggregatedUpgrades.size() + " items");
            outputStream.println("\nReport generated by Maven Dependency Updater");
            outputStream.println(PROJECT_URL);
            return true;
        } catch (Exception e) {
            throw new RuntimeException("Report generation failed", e);
        } finally {
            if (outputStream != null && outputStream != System.out) {
                outputStream.close();
            }
        }
    }

    /**
     * Comparator for sorting component upgrades. Sort primarily by first seen date, then alphabetically.
     */
    protected static class ComponentUpgradeComparator implements Comparator<ComponentUpgrade> {
        @Override
        public int compare(ComponentUpgrade o1, ComponentUpgrade o2) {
            // order null values first
            if (o1.getFirstSeen() == null && o2.getFirstSeen() == null) {
                return o1.getArtifact().compareTo(o2.getArtifact());
            } else if (o1.getFirstSeen() == null) {
                return -1;
            } else if (o2.getFirstSeen() == null) {
                return 1;
            } else {
                // order by age from new to old
                int res = -o1.getFirstSeen().toLocalDate().compareTo(o2.getFirstSeen().toLocalDate());
                if (res == 0) {
                    // when the date is the same, order alphabetically
                    res = o1.getArtifact().toString().compareTo(o2.getArtifact().toString());
                }
                return res;
            }
        }
    }
}
