/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.switchboard.impl;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.switchboard.spi.EnvironmentEntryType;
import org.jboss.switchboard.spi.ResourceProvider;

/**
 * ResourceProviderRegistry
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class ResourceProviderRegistry
{

   private Map<Class<?>, ResourceProvider<?, ? extends EnvironmentEntryType>> bindingProviders = new ConcurrentHashMap<Class<?>, ResourceProvider<?, ? extends EnvironmentEntryType>>();

   public void registerProviders(Collection<ResourceProvider<?, ? extends EnvironmentEntryType>> providers)
   {
      if (providers == null)
      {
         throw new IllegalArgumentException("ENCBindingProvider(s) cannot be null during registration");
      }
      for (ResourceProvider<?, ?> provider : providers)
      {
         this.registerProvider(provider);
      }
   }

   public void registerProvider(ResourceProvider<?, ? extends EnvironmentEntryType> provider)
   {
      if (provider == null)
      {
         throw new IllegalArgumentException("ENCBindingProvider cannot be null during registration");
      }
      Class<? extends EnvironmentEntryType> type = provider.getEnvironmentEntryType(); 
      this.bindingProviders.put(type, provider);
   }

   public ResourceProvider<?, ? extends EnvironmentEntryType> getResourceProvider(
         Class<? extends EnvironmentEntryType> type)
   {

      ResourceProvider<?, ? extends EnvironmentEntryType> provider = this.bindingProviders.get(type);
      if (provider != null)
      {
         return provider;
      }
      Class<?> entryType = type.getSuperclass();
      while (entryType != null && EnvironmentEntryType.class.isAssignableFrom(entryType))
      {
         provider = this.getResourceProvider((Class<? extends EnvironmentEntryType>) entryType);
         if (provider != null)
         {
            return provider;
         }
         entryType = entryType.getSuperclass();

      }

      Class<?>[] interfaces = type.getInterfaces();
      for (Class<?> intf : interfaces)
      {
         if (EnvironmentEntryType.class.isAssignableFrom(intf))
         {
            provider = this.getResourceProvider((Class<? extends EnvironmentEntryType>) intf);
            if (provider != null)
            {
               return provider;
            }
         }
      }

      return provider;

   }

   
//
//   /**
//    * Determine the Processor<T, ?> T generic processorType class.
//    * 
//    * @param processor
//    * @return The Class for the T parameter type.
//    */
//   private Class<?> getProviderEnvEntryType(ResourceProvider<C, ? extends EnvironmentEntryType> provider)
//   {
//      // Find the ResourceProvider<C, T extends EnvironmentEntryType> interface
////      Type[] interfaces = provider.getClass().getGenericInterfaces();
////      Type resourceProviderIntf = null;
////      for (Type t : interfaces)
////      {
////         ParameterizedType pt = (ParameterizedType) t;
////         Type rawType = pt.getRawType();
////         if ((rawType instanceof Class) && ResourceProvider.class.isAssignableFrom((Class<?>) rawType))
////         {
////            resourceProviderIntf = t;
////            break;
////         }
////      }
//      ParameterizedType resourceProviderIntf = this.getResourceProviderType(provider.getClass());
//      if (resourceProviderIntf == null)
//         throw new IllegalStateException("No generic Processor interface found on: " + provider);
//
//      // Get the type of the T parameter
//      ParameterizedType pt = (ParameterizedType) resourceProviderIntf;
//      Type envEntryTypeParameter = pt.getActualTypeArguments()[1];
//      Class<?> evnEntryType = null;
//      if (envEntryTypeParameter instanceof Class)
//         evnEntryType = (Class<?>) envEntryTypeParameter;
//      else if (envEntryTypeParameter instanceof TypeVariable)
//      {
//         TypeVariable tv = (TypeVariable) envEntryTypeParameter;
//         evnEntryType = (Class<?>) tv.getBounds()[0];
//      }
//
//      return evnEntryType;
//   }
//
//   private ParameterizedType getResourceProviderType(Class<?> klass)
//   {
//      // Find the ResourceProvider<C, T extends EnvironmentEntryType> interface
//      Type[] interfaces = klass.getGenericInterfaces();
//      for (Type t : interfaces)
//      {
//         ParameterizedType pt = (ParameterizedType) t;
//         Type rawType = pt.getRawType();
//         if ((rawType instanceof Class) && ResourceProvider.class.isAssignableFrom((Class<?>) rawType))
//         {
//            if (((Class<?>)rawType).getName().equals(ResourceProvider.class.getName()))
//            {
//               return pt;
//            }
//            return this.getResourceProviderType((Class<?>) rawType);
//         }
//      }
//      return null;
//   }
   
}
