/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.switchboard.mc;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.beans.metadata.spi.DependencyMetaData;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.logging.Logger;
import org.jboss.reloaded.naming.spi.JavaEEComponent;
import org.jboss.reloaded.naming.spi.JavaEEModule;
import org.jboss.switchboard.mc.dependency.SwitchBoardDependencyMetaData;
import org.jboss.switchboard.spi.Barrier;
import org.jboss.switchboard.spi.Resource;
import org.jboss.util.naming.Util;

/**
 * SwitchBoardImpl
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class SwitchBoardImpl implements Barrier
{

   private static Logger logger = Logger.getLogger(SwitchBoardImpl.class);
   
   private String id;
   
   private JavaEEComponent javaEEComponent;
   
   private DeploymentUnit unit;

   private Map<String, Resource> bindings = new HashMap<String, Resource>();
   
   private boolean contextPopulated;
   
   /**
    * java:global context
    */
   private Context javaGlobalContext;
   
   public SwitchBoardImpl(String barrierId, Map<String, Resource> bindings, DeploymentUnit unit)
   {
      this.id = barrierId;
      if (bindings != null)
      {
         this.bindings = bindings;
      }
      this.unit = unit;
   }
   
   public void start() throws NamingException
   {
      logger.debug("Binding entries in java:comp for switchboard: " + this.id);

      this.contextPopulated = true;
      for (Map.Entry<String, Resource> binding : this.bindings.entrySet())
      {
         String jndiName = binding.getKey();
         Object jndiObject = binding.getValue().getTarget();
         this.bindInRelevantContext(jndiName, jndiObject);
      }
   }
   
   public void stop() throws NamingException
   {
      logger.debug("Un-binding entries from java:comp for switchboard: " + this.id);

      this.contextPopulated = false;
      for (Map.Entry<String, Resource> binding : this.bindings.entrySet())
      {
         String jndiName = binding.getKey();
         this.unbindFromRelevantContext(jndiName);
      }
   }
   
   public JavaEEComponent getJavaEEComponent()
   {
      return this.javaEEComponent;
   }
   
   public void setJavaEEComponent(JavaEEComponent javaComp)
   {
      this.javaEEComponent = javaComp;
   }

   public void setJavaEEModule(final JavaEEModule javaeeModule)
   {
      JavaEEComponent javaComp = new JavaEEComponent()
      {
         
         @Override
         public String getName()
         {
            return javaeeModule.getName();
         }
         
         @Override
         public JavaEEModule getModule()
         {
            return javaeeModule;
         }
         
         @Override
         public Context getContext()
         {
            return javaeeModule.getContext();
         }
      };
      this.setJavaEEComponent(javaComp);
   }

   /**
    * Sets the java:global context which the {@link SwitchBoardImpl switchboard}
    * can use, if it has to bind environment entries to java:global context
    */
   public void setJavaGlobalContext(Context ctx)
   {
      if (ctx == null)
      {
         throw new IllegalArgumentException("java:global context cannot be set to null");
      }
      
      this.javaGlobalContext = ctx;
   }
   
   @Override
   public Context getContext()
   {
      return this.javaEEComponent == null ? null : this.javaEEComponent.getContext();
   }
   
   @Override
   public String getId()
   {
      return this.id;
   }
   
   public void addENCBinding(Map<String, Resource> resources)
   {
      if (resources == null)
      {
         throw new IllegalArgumentException("Resources cannot be null while adding ENC bindings to switchboard barrier: " + this.id);
      }
      
      if (this.contextPopulated)
      {
         throw new IllegalStateException("ENC context for switchboard barrier: " + this.id + " is already populated, cannot add more ENC bindings");
      }
      this.bindings.putAll(resources);
   }
   
   public Collection<DependencyMetaData> getDependencies()
   {
      Collection<DependencyMetaData> dependencies = new ArrayList<DependencyMetaData>();
      for (Resource encBinding : this.bindings.values())
      {
         Object dependency = encBinding.getDependency();
         if (dependency != null)
         {
            DependencyMetaData mcDependency = new SwitchBoardDependencyMetaData(this, dependency);
            dependencies.add(mcDependency);
         }
      }
      return dependencies;
   }
   
   public ClassLoader getClassLoader()
   {
      return this.unit.getClassLoader();
   }
   
   public Map<String, Resource> getBindings()
   {
      return Collections.unmodifiableMap(this.bindings);
   }
   

   private void bindInRelevantContext(String jndiName, Object jndiObject) throws NamingException
   {
      String relativeJndiName = jndiName;
      Context ctx = this.javaEEComponent.getContext();
      
      if (jndiName.trim().startsWith("java:comp/"))
      {
         relativeJndiName = jndiName.trim().substring("java:comp/".length());
         ctx = this.javaEEComponent.getContext();
      }
      else if (jndiName.trim().startsWith("java:module/"))
      {
         relativeJndiName = jndiName.trim().substring("java:module/".length());
         ctx = this.javaEEComponent.getModule().getContext();
      }
      else if (jndiName.trim().startsWith("java:app/"))
      {
         relativeJndiName = jndiName.trim().substring("java:app/".length());
         ctx = this.javaEEComponent.getModule().getApplication().getContext();
      }
      else if (jndiName.trim().startsWith("java:global/"))
      {
         relativeJndiName = jndiName.trim().substring("java:global/".length());
         
         if (this.javaGlobalContext == null)
         {
            logger.debug("java:global context not set on Switchboard: " + this.id + " will do a manual lookup of java:global context");
            ctx = (Context) new InitialContext().lookup("java:global/");
         }
         else
         {
            ctx = this.javaGlobalContext;
         }
      }
      logger.debug("Switchboard " + this.id + " binding ENC resource at jndi name: " + jndiName);
      Util.rebind(ctx, relativeJndiName, jndiObject);
   }
   
   private void unbindFromRelevantContext(String jndiName) throws NamingException
   {
      String relativeJndiName = jndiName;
      Context ctx = this.javaEEComponent.getContext();
      
      if (jndiName.trim().startsWith("java:comp/"))
      {
         relativeJndiName = jndiName.trim().substring("java:comp/".length());
         ctx = this.javaEEComponent.getContext();
      }
      else if (jndiName.trim().startsWith("java:module/"))
      {
         relativeJndiName = jndiName.trim().substring("java:module/".length());
         ctx = this.javaEEComponent.getModule().getContext();
      }
      else if (jndiName.trim().startsWith("java:app/"))
      {
         relativeJndiName = jndiName.trim().substring("java:app/".length());
         ctx = this.javaEEComponent.getModule().getApplication().getContext();
      }
      else if (jndiName.trim().startsWith("java:global/"))
      {
         relativeJndiName = jndiName.trim().substring("java:global/".length());
         
         if (this.javaGlobalContext == null)
         {
            logger.debug("java:global context not set on Switchboard: " + this.id + " will do a manual lookup of java:global context");
            ctx = (Context) new InitialContext().lookup("java:global/");
         }
         else
         {
            ctx = this.javaGlobalContext;
         }
      }
      logger.debug("Switchboard " + this.id + " unbinding ENC resource from jndi name: " + jndiName);
      Util.unbind(ctx, relativeJndiName);
   }
}
