/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v20.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.jboss.tools.rsp.eclipse.debug.core.model;


import org.jboss.tools.rsp.eclipse.debug.core.DebugPluginConstants;
import org.jboss.tools.rsp.eclipse.debug.core.DebugException;
import org.jboss.tools.rsp.eclipse.debug.core.ILaunch;

/**
 * A process represents a program running in normal (non-debug) mode.
 * Processes support setting and getting of client defined attributes.
 * This way, clients can annotate a process with any extra information
 * important to them. For example, classpath annotations, or command
 * line arguments used to launch the process may be important to a client.
 * <p>
 * Clients may implement this interface, however, the debug plug-in
 * provides an implementation of this interface for a
 * <code>java.lang.Process</code>.
 * </p>
 * @see org.jboss.tools.rsp.eclipse.debug.core.DebugPluginConstants#newProcess(ILaunch, Process, String)
 */
public interface IProcess extends ITerminate {

	/**
	 * Attribute key for a common, optional, process property. The value of this
	 * attribute is the command line a process was launched with.
	 *
	 * @since 2.1
	 */
	public final static String ATTR_CMDLINE= DebugPluginConstants.DEBUG_CORE_ID + ".ATTR_CMDLINE"; //$NON-NLS-1$

	/**
	 * Attribute key for a common, optional, process property. The value of this
	 * attribute is an identifier for the type of this process. Process types
	 * are client defined - whoever creates a process may define its type. For
	 * example, a process type could be "java", "javadoc", or "ant".
	 *
	 * @since 2.1
	 */
	public final static String ATTR_PROCESS_TYPE = DebugPluginConstants.DEBUG_CORE_ID + ".ATTR_PROCESS_TYPE"; //$NON-NLS-1$

	/**
	 * Attribute key for a common, optional, process property. The value of this
	 * attribute specifies an alternate dynamic label for a process, displayed by
	 * the console.
	 *
	 * @since 3.0
	 */
	public final static String ATTR_PROCESS_LABEL = DebugPluginConstants.DEBUG_CORE_ID + ".ATTR_PROCESS_LABEL"; //$NON-NLS-1$

	/**
	 * Returns a human-readable label for this process.
	 *
	 * @return a label for this process
	 */
	public String getLabel();
	
	/**
	 * Returns the launch this element originated from.
	 *
	 * @return the launch this process is contained in
	 */
	public ILaunch getLaunch();
	
	
	/**
	 * Returns a proxy to the standard input, output, and error streams
	 * for this process, or <code>null</code> if not supported.
	 *
	 * @return a streams proxy, or <code>null</code> if not supported
	 */
	public IStreamsProxy getStreamsProxy();

	/**
	 * Sets the value of a client defined attribute.
	 *
	 * @param key the attribute key
	 * @param value the attribute value
	 */
	public void setAttribute(String key, String value);

	/**
	 * Returns the value of a client defined attribute.
	 *
	 * @param key the attribute key
	 * @return value the String attribute value, or <code>null</code> if undefined
	 */
	public String getAttribute(String key);

	/**
	 * Returns the exit value of this process. Conventionally, 0 indicates
	 * normal termination.
	 *
	 * @return the exit value of this process
	 * @exception DebugException if this process has not yet terminated
	 */
	public int getExitValue() throws DebugException;
}
