/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.test.framework.impl.generic.config;

import org.w3c.dom.Element;
import org.w3c.dom.Document;
import org.jboss.portal.test.framework.common.io.IOTools;
import org.jboss.portal.test.framework.common.xml.XMLTools;

import java.net.URL;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.io.InputStream;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class ServerConfig
{

   /** . */
   private final String name;

   /** . */
   private final Map nodes;

   public ServerConfig(String serverName)
   {
      if (serverName == null)
      {
         throw new IllegalArgumentException();
      }
      this.name = serverName;
      this.nodes = new HashMap();
   }

   public String getName()
   {
      return name;
   }

   public void addNode(NodeConfig node)
   {
      nodes.put(node.getId(), node);
   }

   public Map getNodes()
   {
      return nodes;
   }

   public static Map createConfigs(URL url) throws Exception
   {
      Map configs = new LinkedHashMap();
      InputStream in = null;
      try
      {
         in = IOTools.safeBufferedWrapper(url.openStream());
         Document doc = XMLTools.getDocumentBuilderFactory().newDocumentBuilder().parse(in);
         for (Iterator i = XMLTools.getChildrenIterator(doc.getDocumentElement(), "server"); i.hasNext();)
         {
            Element serverElt = (Element)i.next();
            ServerConfig serverCfg = createConfig(serverElt);
            configs.put(serverCfg.getName(), serverCfg);
         }
         return configs;
      }
      finally
      {
         IOTools.safeClose(in);
      }
   }

   public static ServerConfig createConfig(Element serverElt)
   {
      Element serverNameElt = XMLTools.getUniqueChild(serverElt, "server-name", true);

      //
      ServerConfig serverCfg = new ServerConfig(XMLTools.asString(serverNameElt));

      //
      List nodeElts = XMLTools.getChildren(serverElt, "node");
      for (Iterator i = nodeElts.iterator(); i.hasNext();)
      {
         Element nodeElt = (Element)i.next();

         //
         Element nodeIdElt = XMLTools.getUniqueChild(nodeElt, "node-id", true);
         List serviceElts = XMLTools.getChildren(nodeElt, "service");
         Element deployerElt = XMLTools.getUniqueChild(nodeElt, "deployer", true);

         //
         NodeConfig node = new NodeConfig(XMLTools.asString(nodeIdElt));

         //
         for (Iterator j = serviceElts.iterator();j.hasNext();)
         {
            Element serviceElt = (Element)j.next();

            //
            Element serviceNameElt = XMLTools.getUniqueChild(serviceElt, "service-name", true);
            Element interfaceElt = XMLTools.getUniqueChild(serviceElt, "interface", true);
            Element uriElt = XMLTools.getUniqueChild(serviceElt, "uri", true);

            //
            ServiceConfig service = new ServiceConfig(XMLTools.asString(serviceNameElt));
            service.setInterfaceClass(XMLTools.asString(interfaceElt));
            service.setURI(XMLTools.asString(uriElt));

            //
            node.addService(service);
         }

         //
         Element remoteElt = XMLTools.getUniqueChild(deployerElt, "remote", true);
         Element authenticationElt = XMLTools.getUniqueChild(deployerElt, "authentication", false);

         //
         Element remoteNameElt = XMLTools.getUniqueChild(remoteElt, "name", true);
         Element remoteHostElt = XMLTools.getUniqueChild(remoteElt, "host", true);
         Element remotePortElt = XMLTools.getUniqueChild(remoteElt, "port", true);

         //
         RemoteDeployerConfig cargoCfg = new RemoteDeployerConfig();
         cargoCfg.setName(XMLTools.asString(remoteNameElt));
         cargoCfg.setHost(XMLTools.asString(remoteHostElt));
         cargoCfg.setPort(XMLTools.asString(remotePortElt));
         node.setDeployer(cargoCfg);

         //
         if (authenticationElt != null)
         {
            Element authenticationUsernameElt = XMLTools.getUniqueChild(authenticationElt, "username", true);
            Element authenticationPasswordElt = XMLTools.getUniqueChild(authenticationElt, "password", true);

            //
            AuthenticationConfig authenticationCfg = new AuthenticationConfig();
            authenticationCfg.setUsername(XMLTools.asString(authenticationUsernameElt));
            authenticationCfg.setPassword(XMLTools.asString(authenticationPasswordElt));
            node.setAuthentication(authenticationCfg);
         }

         //
         serverCfg.addNode(node);
      }

      //
      return serverCfg;
   }
}
