/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.apache.tomcat.websocket.pojo;

import org.apache.tomcat.websocket.InstanceHandle;
import org.apache.tomcat.websocket.WsSession;

import static org.jboss.web.WebsocketsMessages.MESSAGES;

import java.util.Map;

import javax.websocket.EndpointConfig;
import javax.websocket.Session;
import javax.websocket.server.ServerEndpointConfig;

/**
 * Wrapper class for instances of POJOs annotated with
 * {@link javax.websocket.server.ServerEndpoint} so they appear as standard
 * {@link javax.websocket.Endpoint} instances.
 */
public class PojoEndpointServer extends PojoEndpointBase {

    public static final String POJO_PATH_PARAM_KEY =
            "org.apache.tomcat.websocket.pojo.PojoEndpoint.pathParams";
    public static final String POJO_METHOD_MAPPING_KEY =
            "org.apache.tomcat.websocket.pojo.PojoEndpoint.methodMapping";


    @Override
    public void onOpen(Session session, EndpointConfig endpointConfig) {

        ServerEndpointConfig sec = (ServerEndpointConfig) endpointConfig;

        InstanceHandle instanceHandle = null;
        Object pojo;
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(((WsSession)session).getClassLoader());
            pojo = sec.getConfigurator().getEndpointInstance(
                    sec.getEndpointClass());
            if(pojo instanceof InstanceHandle) {
                instanceHandle = (InstanceHandle) pojo;
                pojo = instanceHandle.getInstance();
            }
        } catch (InstantiationException e) {
            throw MESSAGES.pojoInstanceFailed(sec.getEndpointClass().getName(), e);
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
        setPojo(pojo);
        setInstanceHandle(instanceHandle);

        @SuppressWarnings("unchecked")
        Map<String,String> pathParameters =
                (Map<String, String>) sec.getUserProperties().get(
                        POJO_PATH_PARAM_KEY);
        setPathParameters(pathParameters);

        PojoMethodMapping methodMapping =
                (PojoMethodMapping) sec.getUserProperties().get(
                        POJO_METHOD_MAPPING_KEY);
        setMethodMapping(methodMapping);

        doOnOpen(session, endpointConfig);
    }
}
