/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.framework.transport;

import org.jboss.logging.Logger;
import org.jboss.wsf.spi.deployment.Deployment;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.spi.invocation.EndpointAssociation;
import org.jboss.wsf.spi.invocation.RequestHandler;
import org.jboss.wsf.spi.transport.HttpSpec;
import org.jboss.wsf.spi.transport.ListenerRef;
import org.jboss.wsf.spi.transport.TransportManager;
import org.jboss.wsf.spi.transport.HttpListenerRef;

import javax.servlet.Servlet;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class HttpTransportManager implements TransportManager<HttpSpec>
{
   private HttpDeamon httpDeamon;

   public static final int DEFAULT_PORT = 20000;
   public static final String DEFAULT_HOST = "localhost";

   private int port;
   private String host;

   public HttpTransportManager(String host, int port)
   {
      this.host = host;
      this.port = port;

      this.httpDeamon = TJWSHttpDeamon.getInstance(DEFAULT_PORT);
      this.httpDeamon.start();
   }

   public HttpTransportManager()
   {
      this(DEFAULT_HOST, DEFAULT_PORT);
   }

   public ListenerRef createListener(Endpoint endpoint, HttpSpec transportProperties)
   {
      Servlet listener = new EndpointServlet(endpoint);
      this.httpDeamon.addServletContext(
        transportProperties.getWebContext(),
        transportProperties.getUrlPattern(),
        listener
      );

      return createListenerRef(transportProperties);
   }

   private ListenerRef createListenerRef(HttpSpec transportProperties)
   {

      assert transportProperties.getWebContext().startsWith("/");
      assert transportProperties.getUrlPattern()!=null;

      try
      {
         String ctx = transportProperties.getWebContext();
         String urlPattern = transportProperties.getUrlPattern();

         URI address = new URI("http://"+host+":"+port + ctx + urlPattern);
         return new HttpListenerRef(ctx, urlPattern, address);
      } catch (URISyntaxException e)
      {
         throw new RuntimeException("Failed to create ListenerRef");
      }
   }

   public void destroyListener(ListenerRef ref)
   {
      String context = ref.getUUID().substring(0, ref.getUUID().lastIndexOf("/"));
      String urlPattern = ref.getUUID().substring(ref.getUUID().lastIndexOf("/"));
      this.httpDeamon.removeServletContext(context, urlPattern);
   }

   /**
    * Core listener component
    */
   public class EndpointServlet extends HttpServlet
   {
      // provide logging
      private final Logger log = Logger.getLogger(EndpointServlet.class);

      protected Endpoint endpoint;
      //protected EndpointRegistry epRegistry;

      private boolean started;

      public void init(ServletConfig servletConfig) throws ServletException
      {
         super.init(servletConfig);
         /*SPIProvider spiProvider = SPIProviderResolver.getInstance().getProvider();
         epRegistry = spiProvider.getSPI(EndpointRegistryFactory.class).getEndpointRegistry();*/
      }

      public EndpointServlet(Endpoint endpoint)
      {
         if(null==endpoint.getName())
            throw new IllegalArgumentException("Endpoint objectName is null");

         this.endpoint = endpoint;
      }

      protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
      {
         response.setContentType("text/plain");
         response.setStatus(HttpServletResponse.SC_OK);
         response.getWriter().println(endpoint.getName());
      }

      public void service(HttpServletRequest req, HttpServletResponse res) throws ServletException, IOException
      {
         // arguable solution ...
         // the endpoint should be started when the transport listener is created
         if(!started) startEndpoint();

         try
         {
            EndpointAssociation.setEndpoint(endpoint);
            RequestHandler requestHandler = endpoint.getRequestHandler();
            requestHandler.handleHttpRequest(endpoint, req, res, getServletContext());
         }
         finally
         {
            EndpointAssociation.removeEndpoint();
         }
      }

      private void startEndpoint()
      {
         // Start the endpoint
         Deployment dep = endpoint.getService().getDeployment();
         if (dep.getType() == Deployment.DeploymentType.JAXRPC_JSE ||
           dep.getType() == Deployment.DeploymentType.JAXWS_JSE)
         {
            if (endpoint.getState() == Endpoint.EndpointState.CREATED)
               endpoint.getLifecycleHandler().start(endpoint);
         }
      }      
   }
}
