/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.framework.transport;

import Acme.Serve.Serve;
import org.jboss.logging.Logger;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Properties;
import java.util.Map;
import java.util.HashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class TJWSHttpDeamon implements HttpDeamon
{
   private int port;
   private Acme.Serve.Serve server;

   private static final Logger LOG = Logger.getLogger(TJWSHttpDeamon.class);
   private static final Lock CLASS_LOCK = new ReentrantLock();
   private static HttpDeamon INSTANCE = null;

   private boolean started = false;

   private Map<String, Servlet> servlets = new HashMap<String, Servlet>();

   private TJWSHttpDeamon(int port)
   {
      this.port = port;
      server = new Acme.Serve.Serve();

      Properties properties = new java.util.Properties();
      properties.put("port", port);
      properties.setProperty(Acme.Serve.Serve.ARG_NOHUP, "nohup");

      server.arguments = properties;
      server.addDefaultServlets(null);
      server.addServlet("/", new HelloServlet("JBossWS HttpDeamon"));
   }

   public int getPort()
   {
      return this.port;
   }

   public void start()
   {
      CLASS_LOCK.lock();
      try
      {
         if(!started)
         {
            Thread t = new Thread( new DoServe(server));
            t.start();
            started = true;
         }
         
      } finally
      {
         CLASS_LOCK.unlock();
      }
   }

   public void stop()
   {
      try
      {
         server.notifyStop();
      } catch (IOException e)
      {
         //
      }

      server.destroyAllServlets();
   }

   public void addServletContext(String contextPath, String urlPattern, Servlet servlet)
   {
      CLASS_LOCK.lock();

      try
      {
         if(!contextPath.startsWith("/"))
         contextPath = "/"+contextPath;

         // TODO: urlpattern not used?

         server.addServlet(contextPath, servlet);

         // remeber servlet, required for subsequent remove 
         servlets.put(contextPath, servlet);
      }
      finally
      {
         CLASS_LOCK.unlock();   
      }
   }

   public void removeServletContext(String contextPath, String urlPattern)
   {
      CLASS_LOCK.lock();

      try
      {
         Servlet servlet = servlets.remove(contextPath);
         if(servlet!=null)
         {
            // TJWS is missing a removeContext method
            server.destroyAllServlets();
            for(String ctx : servlets.keySet())
            {
               server.addServlet(ctx, servlets.get(ctx));
            }

            server.addServlet("/", new HelloServlet("JBossWS HttpDeamon"));
         }

      } finally
      {
         CLASS_LOCK.unlock();
      }

   }

   // TODO: Move to factory base class
   public static HttpDeamon getInstance(int port)
   {
      CLASS_LOCK.lock();
      try
      {
         if (INSTANCE == null)
         {
            INSTANCE = new TJWSHttpDeamon(port);

            try
            {
               INSTANCE.start();
            }
            catch (Exception e)
            {
               throw new RuntimeException("Failed to start HttpDeamon", e);
            }

            Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {
               public void run()
               {
                  try
                  {
                     INSTANCE.stop();
                  } catch (Exception e)
                  {
                     //
                  }
               }
            }, "HttpDeamonShutDownHook"));
         }
         else
         {
            LOG.warn("There is already a HttpDeamon running on port: " + INSTANCE.getPort());
         }

         return INSTANCE;
      }
      finally
      {
         CLASS_LOCK.unlock();
      }
   }

   public static class HelloServlet extends HttpServlet
   {
      String greeting="Hello";
      public HelloServlet()
      {}

      public HelloServlet(String hi)
      {greeting=hi;}

      protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
      {
         response.setContentType("text/plain");
         response.setStatus(HttpServletResponse.SC_OK);
         response.getWriter().println(greeting);
      }
   }

   private class DoServe implements Runnable
   {
      private Acme.Serve.Serve server;


      public void run()
      {
         this.server.serve();
      }

      public DoServe(Serve server)
      {
         this.server = server;
      }
   }
}
