/*
 * Copyright 2019 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jbpm.process.workitem.docker;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.command.ListImagesCmd;
import com.github.dockerjava.api.model.Image;
import org.jbpm.process.workitem.core.AbstractLogOrThrowWorkItemHandler;
import org.jbpm.process.workitem.core.util.RequiredParameterValidator;
import org.jbpm.process.workitem.core.util.Wid;
import org.jbpm.process.workitem.core.util.WidMavenDepends;
import org.jbpm.process.workitem.core.util.WidParameter;
import org.jbpm.process.workitem.core.util.WidResult;
import org.jbpm.process.workitem.core.util.service.WidAction;
import org.jbpm.process.workitem.core.util.service.WidService;
import org.kie.api.runtime.process.WorkItem;
import org.kie.api.runtime.process.WorkItemManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Wid(widfile = "DockerListImagesDefinitions.wid", name = "DockerListImages",
        displayName = "DockerListImages",
        defaultHandler = "mvel: new org.jbpm.process.workitem.docker.ListImagesWorkitemHandler()",
        documentation = "docker-workitem/index.html",
        category = "docker-workitem",
        icon = "DockerListImages.png",
        parameters = {
                @WidParameter(name = "ShowAll")
        },
        results = {
                @WidResult(name = "Images", runtimeType = "java.util.List")
        },
        mavenDepends = {
                @WidMavenDepends(group = "org.jbpm.contrib", artifact = "docker-workitem", version = "7.34.0.Final")
        },
        serviceInfo = @WidService(category = "Docker", description = "Connect to and interact with a running Docker deamon",
                keywords = "Docker,list,images",
                action = @WidAction(title = "List Docker images")
        ))
public class ListImagesWorkitemHandler extends AbstractLogOrThrowWorkItemHandler {

    private static final Logger logger = LoggerFactory.getLogger(ListContainersWorkitemHandler.class);
    private static final String RESULTS_DOCUMENT = "Images";

    private DockerClient dockerClient;

    public void executeWorkItem(WorkItem workItem,
                                WorkItemManager workItemManager) {

        Map<String, Object> results = new HashMap<>();

        try {

            RequiredParameterValidator.validate(this.getClass(),
                                                workItem);

            String showAll = (String) workItem.getParameter("ShowAll");

            if (dockerClient == null) {
                DockerClientConnector connector = new DockerClientConnector();
                dockerClient = connector.getDockerClient();
            }

            ListImagesCmd listImagesCmd = dockerClient.listImagesCmd();
            if (showAll != null && Boolean.parseBoolean(showAll)) {
                listImagesCmd = listImagesCmd.withShowAll(true);
            }

            List<Image> images = listImagesCmd.exec();

            results.put(RESULTS_DOCUMENT,
                        images);

            workItemManager.completeWorkItem(workItem.getId(),
                                             results);
        } catch (Exception e) {
            logger.error("Unable to get list of containers: " + e.getMessage());
            handleException(e);
        }
    }

    public void abortWorkItem(WorkItem wi,
                              WorkItemManager wim) {
    }

    // for testing
    public void setDockerClient(DockerClient dockerClient) {
        this.dockerClient = dockerClient;
    }
}

