/*
 * Decompiled with CFR 0.152.
 */
package org.jbpm.services.ejb.timer;

import java.io.Serializable;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.ejb.ConcurrencyManagement;
import javax.ejb.ConcurrencyManagementType;
import javax.ejb.Lock;
import javax.ejb.LockType;
import javax.ejb.NoSuchObjectLocalException;
import javax.ejb.Singleton;
import javax.ejb.Startup;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerConfig;
import javax.ejb.TimerService;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;
import javax.transaction.UserTransaction;
import org.drools.core.time.JobHandle;
import org.drools.core.time.impl.TimerJobInstance;
import org.jbpm.process.core.timer.TimerServiceRegistry;
import org.jbpm.services.ejb.timer.EjbGlobalJobHandle;
import org.jbpm.services.ejb.timer.EjbTimerJob;
import org.jbpm.services.ejb.timer.EjbTimerJobRetry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
@Startup
@ConcurrencyManagement(value=ConcurrencyManagementType.CONTAINER)
@TransactionManagement(value=TransactionManagementType.BEAN)
@Lock(value=LockType.READ)
public class EJBTimerScheduler {
    private static final Logger logger = LoggerFactory.getLogger(EJBTimerScheduler.class);
    private static final Long TIMER_RETRY_INTERVAL = Long.parseLong(System.getProperty("org.kie.jbpm.timer.retry.interval", "5000"));
    private static final Integer TIMER_RETRY_LIMIT = Integer.parseInt(System.getProperty("org.kie.jbpm.timer.retry.limit", "3"));
    private static final TimerExceptionPolicy TIMER_RETRY_POLICY = Enum.valueOf(TimerExceptionPolicy.class, System.getProperty("org.kie.jbpm.timer.retry.policy", "PLATFORM"));
    private static final Integer OVERDUE_WAIT_TIME = Integer.parseInt(System.getProperty("org.jbpm.overdue.timer.wait", "20000"));
    private static final boolean USE_LOCAL_CACHE = Boolean.parseBoolean(System.getProperty("org.jbpm.ejb.timer.local.cache", "true"));
    private ConcurrentMap<String, TimerJobInstance> localCache = new ConcurrentHashMap<String, TimerJobInstance>();
    @Resource
    protected TimerService timerService;
    @Resource
    protected UserTransaction utx;

    @PostConstruct
    public void setup() {
        System.setProperty("org.jbpm.rm.init.timer", "false");
        logger.info("Using local cache for EJB timers: {}", (Object)USE_LOCAL_CACHE);
    }

    @Timeout
    public void executeTimerJob(Timer timer) {
        EjbTimerJob timerJob = (EjbTimerJob)timer.getInfo();
        TimerJobInstance timerJobInstance = timerJob.getTimerJobInstance();
        logger.debug("About to execute timer for job {}", (Object)timerJob);
        String timerServiceId = ((EjbGlobalJobHandle)timerJobInstance.getJobHandle()).getDeploymentId();
        long time = 0L;
        while (TimerServiceRegistry.getInstance().get(timerServiceId) == null) {
            logger.debug("waiting for timer service to be available, elapsed time {} ms", (Object)time);
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
            if ((time += 500L) <= (long)OVERDUE_WAIT_TIME.intValue()) continue;
            logger.debug("No timer service found after waiting {} ms", (Object)time);
            break;
        }
        try {
            this.transaction(this::executeTimerJobInstance, timerJobInstance);
        }
        catch (Exception e) {
            this.recoverTimerJobInstance(timerJob, e);
        }
    }

    private void executeTimerJobInstance(TimerJobInstance timerJobInstance) throws Exception {
        try {
            ((Callable)timerJobInstance).call();
        }
        catch (Exception e) {
            logger.warn("Execution of time failed due to {}", (Object)e.getMessage(), (Object)e);
            throw e;
        }
    }

    private void recoverTimerJobInstance(EjbTimerJob ejbTimerJob, Exception e) {
        if (ejbTimerJob.getTimerJobInstance().getTrigger().hasNextFireTime() != null) {
            logger.warn("Execution of time failed Interval Trigger failed {}", (Object)ejbTimerJob.getTimerJobInstance());
            return;
        }
        switch (TIMER_RETRY_POLICY) {
            case RETRY: {
                logger.warn("Execution of time failed. The timer will be retried {}", (Object)ejbTimerJob.getTimerJobInstance());
                Transaction<TimerJobInstance> operation = instance -> {
                    ZonedDateTime nextRetry = ZonedDateTime.now().plus(TIMER_RETRY_INTERVAL, ChronoUnit.MILLIS);
                    EjbTimerJobRetry info = null;
                    info = ejbTimerJob instanceof EjbTimerJobRetry ? ((EjbTimerJobRetry)ejbTimerJob).next() : new EjbTimerJobRetry((TimerJobInstance)instance);
                    if (TIMER_RETRY_LIMIT > 0 && info.getRetry() > TIMER_RETRY_LIMIT) {
                        logger.warn("The timer {} reached retry limit {}. It won't be retried again", instance, (Object)TIMER_RETRY_LIMIT);
                        return;
                    }
                    TimerConfig config = new TimerConfig((Serializable)info, true);
                    this.timerService.createSingleActionTimer(Date.from(nextRetry.toInstant()), config);
                };
                try {
                    this.transaction(operation, ejbTimerJob.getTimerJobInstance());
                }
                catch (Exception e1) {
                    logger.error("Failed to executed timer recovery {}", (Object)e1.getMessage(), (Object)e1);
                }
                break;
            }
            case PLATFORM: {
                logger.warn("Execution of time failed. Application server policy applied {}", (Object)ejbTimerJob.getTimerJobInstance());
                throw new RuntimeException(e);
            }
        }
    }

    private <I> void transaction(Transaction<I> operation, I item) throws Exception {
        try {
            this.utx.begin();
            operation.doWork(item);
            this.utx.commit();
        }
        catch (Exception e) {
            try {
                if (this.utx.getStatus() != 6) {
                    this.utx.rollback();
                }
            }
            catch (Exception re) {
                logger.error("transaction could not be rolled back", (Throwable)re);
            }
            throw e;
        }
    }

    public void internalSchedule(TimerJobInstance timerJobInstance) {
        TimerConfig config = new TimerConfig((Serializable)new EjbTimerJob(timerJobInstance), true);
        Date expirationTime = timerJobInstance.getTrigger().hasNextFireTime();
        logger.debug("Timer expiration date is {}", (Object)expirationTime);
        if (expirationTime != null) {
            this.timerService.createSingleActionTimer(expirationTime, config);
            logger.debug("Timer scheduled {} on {} scheduler service", (Object)timerJobInstance);
            if (USE_LOCAL_CACHE) {
                this.localCache.putIfAbsent(((EjbGlobalJobHandle)timerJobInstance.getJobHandle()).getUuid(), timerJobInstance);
            }
        } else {
            logger.info("Timer that was to be scheduled has already expired");
        }
    }

    public boolean removeJob(JobHandle jobHandle) {
        EjbGlobalJobHandle ejbHandle = (EjbGlobalJobHandle)jobHandle;
        for (Timer timer : this.timerService.getTimers()) {
            try {
                EjbTimerJob job;
                EjbGlobalJobHandle handle;
                Serializable info = timer.getInfo();
                if (!(info instanceof EjbTimerJob) || !(handle = (EjbGlobalJobHandle)(job = (EjbTimerJob)info).getTimerJobInstance().getJobHandle()).getUuid().equals(ejbHandle.getUuid())) continue;
                logger.debug("Job handle {} does match timer and is going to be canceled", (Object)jobHandle);
                if (USE_LOCAL_CACHE) {
                    this.localCache.remove(handle.getUuid());
                }
                try {
                    timer.cancel();
                }
                catch (Throwable e) {
                    logger.debug("Timer cancel error due to {}", (Object)e.getMessage());
                    return false;
                }
                return true;
            }
            catch (NoSuchObjectLocalException e) {
                logger.debug("Timer {} has already expired or was canceled ", (Object)timer);
            }
        }
        logger.debug("Job handle {} does not match any timer on {} scheduler service", (Object)jobHandle, (Object)this);
        return false;
    }

    public TimerJobInstance getTimerByName(String jobName) {
        if (USE_LOCAL_CACHE && this.localCache.containsKey(jobName)) {
            logger.debug("Found job {} in cache returning", (Object)jobName);
            return (TimerJobInstance)this.localCache.get(jobName);
        }
        TimerJobInstance found = null;
        for (Timer timer : this.timerService.getTimers()) {
            try {
                EjbTimerJob job;
                EjbGlobalJobHandle handle;
                Serializable info = timer.getInfo();
                if (!(info instanceof EjbTimerJob) || !(handle = (EjbGlobalJobHandle)(job = (EjbTimerJob)info).getTimerJobInstance().getJobHandle()).getUuid().equals(jobName)) continue;
                found = handle.getTimerJobInstance();
                if (USE_LOCAL_CACHE) {
                    this.localCache.putIfAbsent(jobName, found);
                }
                logger.debug("Job {} does match timer and is going to be returned {}", (Object)jobName, (Object)found);
                break;
            }
            catch (NoSuchObjectLocalException e) {
                logger.debug("Timer info for {} was not found ", (Object)timer);
            }
        }
        return found;
    }

    @FunctionalInterface
    private static interface Transaction<I> {
        public void doWork(I var1) throws Exception;
    }

    private static enum TimerExceptionPolicy {
        RETRY,
        PLATFORM;

    }
}

