/*
 * Decompiled with CFR 0.152.
 */
package org.jbpm.services.ejb.timer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.ejb.ConcurrencyManagement;
import javax.ejb.ConcurrencyManagementType;
import javax.ejb.Lock;
import javax.ejb.LockType;
import javax.ejb.NoSuchObjectLocalException;
import javax.ejb.Singleton;
import javax.ejb.Startup;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerConfig;
import javax.ejb.TimerService;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;
import javax.transaction.RollbackException;
import javax.transaction.UserTransaction;
import org.drools.core.time.JobHandle;
import org.drools.core.time.impl.TimerJobInstance;
import org.jbpm.process.core.timer.TimerServiceRegistry;
import org.jbpm.services.ejb.timer.EjbGlobalJobHandle;
import org.jbpm.services.ejb.timer.EjbTimerJob;
import org.jbpm.services.ejb.timer.EjbTimerJobRetry;
import org.kie.internal.runtime.manager.SessionNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
@Startup
@ConcurrencyManagement(value=ConcurrencyManagementType.CONTAINER)
@TransactionManagement(value=TransactionManagementType.BEAN)
@Lock(value=LockType.READ)
public class EJBTimerScheduler {
    private static final Logger logger = LoggerFactory.getLogger(EJBTimerScheduler.class);
    private static final Long TIMER_RETRY_INTERVAL = Long.parseLong(System.getProperty("org.kie.jbpm.timer.retry.interval", "5000"));
    private static final Integer TIMER_RETRY_LIMIT = Integer.parseInt(System.getProperty("org.kie.jbpm.timer.retry.limit", "3"));
    private static final Integer OVERDUE_WAIT_TIME = Integer.parseInt(System.getProperty("org.jbpm.overdue.timer.wait", "20000"));
    private static final boolean USE_LOCAL_CACHE = Boolean.parseBoolean(System.getProperty("org.jbpm.ejb.timer.local.cache", "true"));
    private ConcurrentMap<String, TimerJobInstance> localCache = new ConcurrentHashMap<String, TimerJobInstance>();
    @Resource
    protected TimerService timerService;
    @Resource
    protected UserTransaction utx;

    @PostConstruct
    public void setup() {
        System.setProperty("org.jbpm.rm.init.timer", "false");
        logger.info("Using local cache for EJB timers: {}", (Object)USE_LOCAL_CACHE);
    }

    @Timeout
    public void executeTimerJob(Timer timer) {
        EjbTimerJob timerJob = (EjbTimerJob)timer.getInfo();
        TimerJobInstance timerJobInstance = timerJob.getTimerJobInstance();
        logger.debug("About to execute timer for job {}", (Object)timerJob);
        String timerServiceId = ((EjbGlobalJobHandle)timerJobInstance.getJobHandle()).getDeploymentId();
        long time = 0L;
        while (TimerServiceRegistry.getInstance().get(timerServiceId) == null) {
            logger.debug("waiting for timer service to be available, elapsed time {} ms", (Object)time);
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
            if ((time += 500L) <= (long)OVERDUE_WAIT_TIME.intValue()) continue;
            logger.debug("No timer service found after waiting {} ms", (Object)time);
            break;
        }
        try {
            this.transaction(this::executeTimerJobInstance, timerJobInstance);
        }
        catch (SessionNotFoundException e) {
            logger.warn("Process instance is not found. More likely already completed. Timer {} won't be recovered", (Object)timerJobInstance, (Object)e);
            this.removeUnrecoverableTimer(timerJob);
        }
        catch (Exception e) {
            this.recoverTimerJobInstance(timerJob, e);
        }
    }

    private void executeTimerJobInstance(TimerJobInstance timerJobInstance) throws Exception {
        ((Callable)timerJobInstance).call();
    }

    private void removeUnrecoverableTimer(EjbTimerJob ejbTimerJob) {
        try {
            Transaction<TimerJobInstance> tx = timerJobInstance -> {
                if (!this.removeJob(timerJobInstance.getJobHandle())) {
                    logger.warn("Session not found for timer {}. Timer could not removed.", (Object)ejbTimerJob.getTimerJobInstance());
                }
            };
            this.transaction(tx, ejbTimerJob.getTimerJobInstance());
        }
        catch (Exception e1) {
            logger.warn("There was a problem during timer removal {}", (Object)ejbTimerJob.getTimerJobInstance(), (Object)e1);
        }
    }

    private void recoverTimerJobInstance(EjbTimerJob ejbTimerJob, Exception e) {
        if (this.isSessionNotFound(e)) {
            logger.warn("Trying to recover timer. Not possible due to process instance is not found. More likely already completed. Timer {} won't be recovered", (Object)ejbTimerJob.getTimerJobInstance(), (Object)e);
            this.removeUnrecoverableTimer(ejbTimerJob);
            return;
        }
        if (ejbTimerJob.getTimerJobInstance().getTrigger().hasNextFireTime() != null) {
            try {
                logger.warn("Execution of time failed Interval Trigger failed. Skipping {}", (Object)ejbTimerJob.getTimerJobInstance());
                Transaction<TimerJobInstance> tx = timerJobInstance -> {
                    if (this.removeJob(timerJobInstance.getJobHandle())) {
                        this.internalSchedule((TimerJobInstance)timerJobInstance);
                    } else {
                        logger.debug("Interval trigger {} was removed before rescheduling", (Object)ejbTimerJob.getTimerJobInstance());
                    }
                };
                this.transaction(tx, ejbTimerJob.getTimerJobInstance());
            }
            catch (Exception e1) {
                logger.warn("Could not schedule the interval trigger {}", (Object)ejbTimerJob.getTimerJobInstance(), (Object)e1);
            }
            return;
        }
        logger.warn("Execution of time failed. The timer will be retried {}", (Object)ejbTimerJob.getTimerJobInstance());
        Transaction<TimerJobInstance> operation = instance -> {
            ZonedDateTime nextRetry = ZonedDateTime.now().plus(TIMER_RETRY_INTERVAL, ChronoUnit.MILLIS);
            EjbTimerJobRetry info = null;
            info = ejbTimerJob instanceof EjbTimerJobRetry ? ((EjbTimerJobRetry)ejbTimerJob).next() : new EjbTimerJobRetry((TimerJobInstance)instance);
            if (TIMER_RETRY_LIMIT > 0 && info.getRetry() > TIMER_RETRY_LIMIT) {
                logger.warn("The timer {} reached retry limit {}. It won't be retried again", instance, (Object)TIMER_RETRY_LIMIT);
                return;
            }
            TimerConfig config = new TimerConfig((Serializable)info, true);
            this.timerService.createSingleActionTimer(Date.from(nextRetry.toInstant()), config);
        };
        try {
            this.transaction(operation, ejbTimerJob.getTimerJobInstance());
        }
        catch (Exception e1) {
            logger.error("Failed to executed timer recovery {}", (Object)e1.getMessage(), (Object)e1);
        }
    }

    private boolean isSessionNotFound(Exception e) {
        Throwable current = e;
        do {
            if (!(current instanceof SessionNotFoundException)) continue;
            return true;
        } while ((current = current.getCause()) != null);
        return false;
    }

    private <I> void transaction(Transaction<I> operation, I item) throws Exception {
        try {
            this.utx.begin();
            operation.doWork(item);
            this.utx.commit();
        }
        catch (RollbackException e) {
            logger.warn("Transaction was rolled back for {} with status {}", item, (Object)this.utx.getStatus());
            if (this.utx.getStatus() == 0) {
                this.utx.rollback();
            }
            throw new RuntimeException("jbpm timer has been rolledback", e);
        }
        catch (Exception e) {
            try {
                this.utx.rollback();
            }
            catch (Exception re) {
                logger.error("transaction could not be rolled back", (Throwable)re);
            }
            throw e;
        }
    }

    public void internalSchedule(TimerJobInstance timerJobInstance) {
        Serializable info = this.removeTransientFields(new EjbTimerJob(timerJobInstance));
        TimerConfig config = new TimerConfig(info, true);
        Date expirationTime = timerJobInstance.getTrigger().hasNextFireTime();
        logger.debug("Timer expiration date is {}", (Object)expirationTime);
        if (expirationTime != null) {
            this.timerService.createSingleActionTimer(expirationTime, config);
            logger.debug("Timer scheduled {} on {} scheduler service", (Object)timerJobInstance);
            if (USE_LOCAL_CACHE) {
                this.localCache.putIfAbsent(((EjbGlobalJobHandle)timerJobInstance.getJobHandle()).getUuid(), timerJobInstance);
            }
        } else {
            logger.info("Timer that was to be scheduled has already expired");
        }
    }

    private Serializable removeTransientFields(Serializable info) {
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ObjectOutputStream out = new ObjectOutputStream(bos);
            out.writeObject(info);
            out.flush();
            ObjectInputStream stream = new ObjectInputStream(new ByteArrayInputStream(bos.toByteArray()));
            return (Serializable)stream.readObject();
        }
        catch (IOException io) {
            logger.warn("Not possible to serialize the timer info", (Throwable)io);
        }
        catch (ClassNotFoundException cnf) {
            logger.warn("Class not found in class loader", (Throwable)cnf);
        }
        return info;
    }

    public boolean removeJob(JobHandle jobHandle) {
        EjbGlobalJobHandle ejbHandle = (EjbGlobalJobHandle)jobHandle;
        if (USE_LOCAL_CACHE) {
            boolean removedFromCache = this.localCache.remove(ejbHandle.getUuid()) != null;
            logger.debug("Job handle {} is {} removed from cache ", (Object)jobHandle, (Object)(removedFromCache ? "" : "not"));
        }
        for (Timer timer : this.timerService.getTimers()) {
            try {
                EjbTimerJob job;
                EjbGlobalJobHandle handle;
                Serializable info = timer.getInfo();
                if (!(info instanceof EjbTimerJob) || !(handle = (EjbGlobalJobHandle)(job = (EjbTimerJob)info).getTimerJobInstance().getJobHandle()).getUuid().equals(ejbHandle.getUuid())) continue;
                logger.debug("Job handle {} does match timer and is going to be canceled", (Object)jobHandle);
                try {
                    timer.cancel();
                }
                catch (Throwable e) {
                    logger.debug("Timer cancel error due to {}", (Object)e.getMessage());
                    return false;
                }
                return true;
            }
            catch (NoSuchObjectLocalException e) {
                logger.debug("Timer {} has already expired or was canceled ", (Object)timer);
            }
        }
        logger.debug("Job handle {} does not match any timer on {} scheduler service", (Object)jobHandle, (Object)this);
        return false;
    }

    public TimerJobInstance getTimerByName(String jobName) {
        if (USE_LOCAL_CACHE && this.localCache.containsKey(jobName)) {
            logger.debug("Found job {} in cache returning", (Object)jobName);
            return (TimerJobInstance)this.localCache.get(jobName);
        }
        TimerJobInstance found = null;
        for (Timer timer : this.timerService.getTimers()) {
            try {
                EjbTimerJob job;
                EjbGlobalJobHandle handle;
                Serializable info = timer.getInfo();
                if (!(info instanceof EjbTimerJob) || !(handle = (EjbGlobalJobHandle)(job = (EjbTimerJob)info).getTimerJobInstance().getJobHandle()).getUuid().equals(jobName)) continue;
                found = handle.getTimerJobInstance();
                if (USE_LOCAL_CACHE) {
                    this.localCache.putIfAbsent(jobName, found);
                }
                logger.debug("Job {} does match timer and is going to be returned {}", (Object)jobName, (Object)found);
                break;
            }
            catch (NoSuchObjectLocalException e) {
                logger.debug("Timer info for {} was not found ", (Object)timer);
            }
        }
        return found;
    }

    public void evictCache(JobHandle jobHandle) {
        String jobName = ((EjbGlobalJobHandle)jobHandle).getUuid();
        logger.debug("Invalidate job {} with job name {} in cache", (Object)jobName, this.localCache.remove(jobName));
    }

    @FunctionalInterface
    private static interface Transaction<I> {
        public void doWork(I var1) throws Exception;
    }
}

