/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.model;


import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.util.Date;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@XmlRootElement(name="processInstance")
public class ProcessInstance
{
   private long instanceId;
   private long parentId;

   private String key;
   public enum STATE {RUNNING, SUSPENDED, ENDED};

   private Date startDate;
   private Date endDate;

   private boolean suspended;

   private STATE state;  // gson is field based, hence why...

   private transient Lifecycle lifecycle;

   public ProcessInstance()
   {
      initLifecycle();
   }

   public ProcessInstance(long id, long parentId, Date startDate, Date endDate, boolean suspended)
   {

      if(null==startDate)
         throw new IllegalArgumentException("An instance requires a start date");
      
      if(endDate!=null && suspended)
         throw new IllegalArgumentException("An instance cannot be ended and suspended at the same time");

      this.instanceId = id;
      this.parentId = parentId;
      this.startDate = startDate;
      this.endDate = endDate;
      this.suspended = suspended;
      initLifecycle();
   }

   /**
    * If not ENDED or SUSPENDED the instance is RUNNING
    */
   private void initLifecycle()
   {
      if(hasEnded())
         this.lifecycle = new Lifecycle(this, STATE.ENDED);
      else if(isSuspended())
         this.lifecycle = new Lifecycle(this, STATE.SUSPENDED);
      else
         this.lifecycle = new Lifecycle(this, STATE.RUNNING);

      this.state = lifecycle.getState();
   }

   @XmlElement(name = "instanceId")
   public long getInstanceId()
   {
      return instanceId;
   }

   public void setInstanceId(long instanceId)
   {
      this.instanceId = instanceId;
   }

   @XmlElement(name = "parentId")
   public long getParentId()
   {
      return parentId;
   }

   public void setParentId(long parentId)
   {
      this.parentId = parentId;
   }

   @XmlElement(name = "key")
   public String getKey()
   {
      return key !=null ? key : "";
   }

   public void setKey(String key)
   {
      this.key = key;
   }

   @XmlElement(name = "status")
   public STATE getState()
   {
      return this.state;
   }

   public void setState(String nextState)
   {
      setState(STATE.valueOf(nextState));
   }

   public void setState(STATE nextState)
   {
      this.lifecycle = this.lifecycle.transitionTo(nextState);
      this.state = this.lifecycle.getState();
   }

   @XmlElement(name = "start")
   public Date getStartDate()
   {
      return startDate;
   }

   public void setStartDate(Date startDate)
   {
      this.startDate = startDate;
   }

   @XmlElement(name = "end")
   public Date getEndDate()
   {
      return endDate;
   }

   public void setEndDate(Date endDate)
   {
      this.endDate = endDate;
   }

   public boolean isRunning()
   {
      return this.startDate!=null && !isSuspended();
   }

   public boolean hasEnded()
   {
      return this.startDate!=null
        && this.endDate!=null;
   }

   public boolean isSuspended()
   {
      return null==this.endDate && suspended;
   }
   
   private class Lifecycle
   {
      private STATE current;
      private ProcessInstance instance;

      public Lifecycle(ProcessInstance instance, STATE current)
      {
         this.instance = instance;
         this.current = current;
      }

      public Lifecycle transitionTo(STATE next)
      {
         Lifecycle nextLifecycle = null;

         switch(next)
         {            
            case SUSPENDED: // only RUNNING instances can be SUSPENDED
               if(STATE.RUNNING.equals(current))
               {
                  nextLifecycle = new Lifecycle(instance, next);
                  instance.suspended = true;
                  break;
               }
               else
               {
                  throw new IllegalTransitionException(current, next);
               }
            case ENDED: // both RUNNING and SUSPENDED instances can be ENDED
               if(STATE.RUNNING.equals(current) || STATE.SUSPENDED.equals(current))
               {
                  nextLifecycle =  new Lifecycle(instance, next);
                  instance.suspended = false;
                  instance.endDate = new Date();
                  break;
               }
               else
               {
                  throw new IllegalTransitionException(current, next);
               }
            case RUNNING: // only SUSPENDED instances can become RUNNING
               if(STATE.SUSPENDED.equals(current))
               {
                  nextLifecycle =  new Lifecycle(instance, next);
                  instance.suspended = false;                  
                  break;
               }
               else
               {
                  throw new IllegalTransitionException(current, next);
               }
            default:
               throw new IllegalTransitionException(current, next);
         }

         return nextLifecycle;
      }

      public STATE getState()
      {
         return current;
      }
   }

   private class IllegalTransitionException extends IllegalArgumentException
   {

      public IllegalTransitionException(STATE current, STATE next)
      {
         super("Illegal transition current " + current + " next " + next);
      }
   }
}
