package org.jbpm.job.executor;

import java.io.Serializable;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmConfiguration;

public class JobExecutor implements Serializable {

  private static final long serialVersionUID = 1L;

  protected JbpmConfiguration jbpmConfiguration;
  protected String name;
  protected int nbrOfThreads;
  protected int idleInterval;
  protected int maxIdleInterval;
  /** @deprecated this field was never used */
  protected int historyMaxSize;

  protected int maxLockTime;
  protected int lockMonitorInterval;
  protected int lockBufferTime;

  protected Map threads = new HashMap();
  protected LockMonitorThread lockMonitorThread;
  protected Map monitoredJobIds = new Hashtable();

  protected boolean isStarted = false;

  protected static String hostName;

  public synchronized void start() {
    if (!isStarted) {
      log.debug("starting thread group '" + name + "'...");
      for (int i = 0; i < nbrOfThreads; i++) {
        startThread();
      }
      lockMonitorThread = new LockMonitorThread(getLockMonitorThreadName(), this);
      lockMonitorThread.start();
      isStarted = true;
    }
    else {
      log.debug("ignoring start: thread group '" + name + "' is already started'");
    }
  }

  /**
   * signals to all threads in this job executor to stop. Threads may be in the middle of processing
   * a job and they will finish that first. Use {@link #stopAndJoin()} in case you want a method
   * that blocks until all the threads are actually finished.
   * 
   * @return a list of the stopped threads. In case no threads were stopped an empty list will be
   * returned.
   */
  public synchronized List stop() {
    List stoppedThreads;
    if (isStarted) {
      log.debug("stopping thread group '" + name + "'...");
      isStarted = false;

      stoppedThreads = new ArrayList(threads.size());
      for (int i = 0; i < nbrOfThreads; i++) {
        stoppedThreads.add(stopThread());
      }

      if (lockMonitorThread != null) lockMonitorThread.deactivate();
    }
    else {
      log.debug("ignoring stop: thread group '" + name + "' not started");
      stoppedThreads = Collections.EMPTY_LIST;
    }
    return stoppedThreads;
  }

  public void stopAndJoin() throws InterruptedException {
    for (Iterator i = stop().iterator(); i.hasNext();) {
      Thread thread = (Thread) i.next();
      thread.join();
    }

    if (lockMonitorThread != null) lockMonitorThread.join();
  }
  
  public void ensureThreadsAreActive() {
    List deadThreads = new ArrayList(); 
    for (Iterator i = threads.values().iterator(); i.hasNext();) {
      Thread thread = (Thread) i.next();
      if (!thread.isAlive()) {      
        if(log.isDebugEnabled())
        {
    	  log.debug("detected dead thread '" + thread.getName() + "'");
        }
        deadThreads.add(thread.getName());
        i.remove();
      }
    }
    for (int i = 0; i<deadThreads.size(); i++) {
        startThread((String)deadThreads.get(i));
    }
  }
  
  protected void startThread() {
    startThread(getNextThreadName());
  }

  protected synchronized void startThread(String threadName) {
    Thread thread = createThread(threadName);
    threads.put(threadName, thread);

    log.debug("starting new job executor thread '" + threadName + "'");
    thread.start();
  }

  protected Thread createThread(String threadName) {
    return new JobExecutorThread(threadName, this);
  }

  protected String getNextThreadName() {
    return getThreadName(threads.size() + 1);
  }

  protected String getLastThreadName() {
    return getThreadName(threads.size());
  }

  private String getThreadName(int index) {
    return name + ":" + getHostAddress() + ":" + index;
  }
  
  private String getLockMonitorThreadName() {
    return name + ':' + LockMonitorThread.DEFAULT_NAME + '@' + getHostAddress();     
  }

  private static String getHostAddress() {
    if (hostName == null) {
      try {
        hostName = InetAddress.getLocalHost().getHostAddress();
      }
      catch (UnknownHostException e) {
        hostName = "127.0.0.1";
      }
    }
    return hostName;
  }

  protected synchronized Thread stopThread() {
    String threadName = getLastThreadName();
    log.debug("removing job executor thread '" + threadName + "'");

    Thread thread = (Thread) threads.remove(threadName);
    if (thread instanceof JobExecutorThread) {
      JobExecutorThread jobThread = (JobExecutorThread) thread;
      jobThread.deactivate();
    }
    return thread;
  }

  public Set getMonitoredJobIds() {
    return new HashSet(monitoredJobIds.values());
  }

  public void addMonitoredJobId(String threadName, long jobId) {
    monitoredJobIds.put(threadName, new Long(jobId));
  }

  public void removeMonitoredJobId(String threadName) {
    monitoredJobIds.remove(threadName);
  }

  /**
   * @throws UnsupportedOperationException to prevent invocation
   * @deprecated <code>monitoredJobIds</code> is an internal control field
   */
  public void setMonitoredJobIds(Map monitoredJobIds) {
    throw new UnsupportedOperationException();
  }

  /** @deprecated this property was never used */
  public int getHistoryMaxSize() {
    return historyMaxSize;
  }

  /** @deprecated this property was never used */
  public void setHistoryMaxSize(int historyMaxSize) {
    this.historyMaxSize = historyMaxSize;
  }

  public int getIdleInterval() {
    return idleInterval;
  }

  public void setIdleInterval(int idleInterval) {
    this.idleInterval = idleInterval;
  }

  /**
   * Tells whether this job executor has been {@linkplain #start() started}.
   */
  public boolean isStarted() {
    return isStarted;
  }

  /**
   * @throws UnsupportedOperationException to prevent invocation
   * @deprecated <code>isStarted</code> is an internal control field
   */
  public void setStarted(boolean isStarted) {
    throw new UnsupportedOperationException();
  }

  public JbpmConfiguration getJbpmConfiguration() {
    return jbpmConfiguration;
  }

  public void setJbpmConfiguration(JbpmConfiguration jbpmConfiguration) {
    this.jbpmConfiguration = jbpmConfiguration;
  }

  public int getMaxIdleInterval() {
    return maxIdleInterval;
  }

  public void setMaxIdleInterval(int maxIdleInterval) {
    this.maxIdleInterval = maxIdleInterval;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  /**
   * @deprecated Replaced by {@link #getNbrOfThreads()}
   */
  public int getSize() {
    return nbrOfThreads;
  }

  /**
   * @deprecated Replaced by {@link #setNbrOfThreads(int)}
   */
  public void setSize(int nbrOfThreads) {
    this.nbrOfThreads = nbrOfThreads;
  }

  public Map getThreads() {
    return threads;
  }

  /**
   * @throws UnsupportedOperationException to prevent invocation
   * @deprecated <code>threads</code> is an internal control field
   */
  public void setThreads(Map threads) {
    throw new UnsupportedOperationException();
  }

  public int getMaxLockTime() {
    return maxLockTime;
  }

  public void setMaxLockTime(int maxLockTime) {
    this.maxLockTime = maxLockTime;
  }

  public int getLockBufferTime() {
    return lockBufferTime;
  }

  public void setLockBufferTime(int lockBufferTime) {
    this.lockBufferTime = lockBufferTime;
  }

  public int getLockMonitorInterval() {
    return lockMonitorInterval;
  }

  public void setLockMonitorInterval(int lockMonitorInterval) {
    this.lockMonitorInterval = lockMonitorInterval;
  }

  public int getNbrOfThreads() {
    return nbrOfThreads;
  }

  public void setNbrOfThreads(int nbrOfThreads) {
    this.nbrOfThreads = nbrOfThreads;
  }

  private static Log log = LogFactory.getLog(JobExecutor.class);
}
