package org.jbpm.job.executor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.db.JobSession;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.job.Job;
import org.jbpm.persistence.db.DbPersistenceService;
import org.jbpm.persistence.db.StaleObjectLogConfigurer;

public class JobExecutorThread extends Thread {

  final JobExecutor jobExecutor;
  final JbpmConfiguration jbpmConfiguration;
  final int idleInterval;
  final int maxIdleInterval;
  final long maxLockTime;

  volatile boolean isActive = true;

  public JobExecutorThread(String name, JobExecutor jobExecutor) {
    super(name);
    this.jobExecutor = jobExecutor;

    jbpmConfiguration = jobExecutor.getJbpmConfiguration();
    idleInterval = jobExecutor.getIdleInterval();
    maxIdleInterval = jobExecutor.getMaxIdleInterval();
    maxLockTime = jobExecutor.getMaxLockTime();
  }

  /**
   * @deprecated use {@link #JobExecutorThread(String, JobExecutor)} instead
   */
  public JobExecutorThread(String name, JobExecutor jobExecutor,
      JbpmConfiguration jbpmConfiguration, int idleInterval, int maxIdleInterval,
      long maxLockTime, int maxHistory) {
    super(name);
    this.jobExecutor = jobExecutor;
    this.jbpmConfiguration = jbpmConfiguration;
    this.idleInterval = idleInterval;
    this.maxIdleInterval = maxIdleInterval;
    this.maxLockTime = maxLockTime;
  }

  public void run() {
    int currentIdleInterval = idleInterval;
    while (isActive) {
      try {
        Collection acquiredJobs = acquireJobs();
        for (Iterator i = acquiredJobs.iterator(); i.hasNext() && isActive;) {
          Job job = (Job) i.next();
          executeJob(job);
        }
        if (isActive) {
          long waitPeriod = getWaitPeriod(currentIdleInterval);
          if (waitPeriod > 0) {
            synchronized (jobExecutor) {
              jobExecutor.wait(waitPeriod);
            }
          }
        }
        // no exception, reset current idle interval
        currentIdleInterval = idleInterval;
      }
      catch (RuntimeException e) {
        if (isActive) {
          log.error("exception in " + getName() + ", waiting " + currentIdleInterval + " ms", e);
          try {
            synchronized (jobExecutor) {
              jobExecutor.wait(currentIdleInterval);
            }
          }
          catch (InterruptedException ie) {
            log.debug(getName() + " got interrupted: " + e.getMessage());
          }
          // after an exception, double the current idle interval to prevent
          // continuous exception throwing while an anomalous condition prevails
          currentIdleInterval *= 2;
          if (currentIdleInterval > maxIdleInterval || currentIdleInterval < 0) {
            currentIdleInterval = maxIdleInterval;
          }
        }
      }
      catch (InterruptedException e) {
        log.debug(getName() + " got interrupted: " + e.getMessage());
      }
    }
    log.info(getName() + " leaves cyberspace");
  }

  protected Collection acquireJobs() {
    Collection acquiredJobs;

    synchronized (jobExecutor) {
      List jobsToLock = Collections.EMPTY_LIST;

      JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
      try {
        JobSession jobSession = jbpmContext.getJobSession();

        String lockOwner = getName();
        Job job = jobSession.getFirstAcquirableJob(lockOwner);
        if (job != null) {
          if (job.isExclusive()) {
            ProcessInstance processInstance = job.getProcessInstance();
            log.debug("finding exclusive jobs for " + processInstance);
            jobsToLock = jobSession.findExclusiveJobs(lockOwner, processInstance);
            log.debug("acquiring " + jobsToLock + " for " + processInstance);
          }
          else {
            log.debug("acquiring " + job);
            jobsToLock = Collections.singletonList(job);
          }

          Date lockTime = new Date();
          for (Iterator i = jobsToLock.iterator(); i.hasNext();) {
            job = (Job) i.next();
            job.setLockOwner(lockOwner);
            job.setLockTime(lockTime);
          }
        }
      }
      catch (RuntimeException e) {
        jbpmContext.setRollbackOnly();
        throw e;
      }
      catch (Error e) {
        jbpmContext.setRollbackOnly();
        throw e;
      }
      finally {
        try {
          jbpmContext.close();
          acquiredJobs = jobsToLock;
          log.debug("acquired lock on jobs: " + acquiredJobs);
        }
        catch (RuntimeException e) {
          if (!DbPersistenceService.isLockingException(e)) throw e;
          // if this is a locking exception, keep it quiet
          StaleObjectLogConfigurer.getStaleObjectExceptionsLog()
              .error("failed to acquire lock on jobs " + jobsToLock);
          acquiredJobs = Collections.EMPTY_LIST;
        }
      }
    }
    return acquiredJobs;
  }

  protected void executeJob(Job job) {
    JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
    try {
      JobSession jobSession = jbpmContext.getJobSession();
      jobSession.reattachJob(job);

      // register process instance for automatic save
      // see https://jira.jboss.org/jira/browse/JBPM-1015
      jbpmContext.addAutoSaveProcessInstance(job.getProcessInstance());

      log.debug("executing " + job);
      try {
        if (job.execute(jbpmContext)) {
          jobSession.deleteJob(job);
        }
      }
      catch (Exception e) {
        log.debug("exception while executing " + job, e);
        if (!DbPersistenceService.isPersistenceException(e)) {
          StringWriter memoryWriter = new StringWriter();
          e.printStackTrace(new PrintWriter(memoryWriter));
          job.setException(memoryWriter.toString());
          job.setRetries(job.getRetries() - 1);
        }
        else {
          // prevent unsafe use of the session after an exception occurs
          jbpmContext.setRollbackOnly();
        }
      }
      catch (Error e) {
        jbpmContext.setRollbackOnly();
        throw e;
      }

      // if this job is locked too long
      long totalLockTimeInMillis = System.currentTimeMillis() - job.getLockTime().getTime();
      if (totalLockTimeInMillis > maxLockTime) {
        jbpmContext.setRollbackOnly();
      }
    }
    finally {
      try {
        jbpmContext.close();
      }
      catch (RuntimeException e) {
        if (!DbPersistenceService.isLockingException(e)) throw e;
        // if this is a locking exception, keep it quiet
        StaleObjectLogConfigurer.getStaleObjectExceptionsLog().error("failed to complete job "
            + job);
      }
    }
  }

  protected long getWaitPeriod(int currentIdleInterval) {
    long waitPeriod = currentIdleInterval;

    Date nextDueDate = getNextDueDate();
    if (nextDueDate != null) {
      long nextDueTime = nextDueDate.getTime();
      long currentTime = System.currentTimeMillis();

      if (nextDueTime < currentTime + currentIdleInterval) {
        waitPeriod = nextDueTime - currentTime;
        if (waitPeriod < 0) waitPeriod = 0;
      }
    }
    return waitPeriod;
  }

  protected Date getNextDueDate() {
    Date nextDueDate = null;
    String lockOwner = getName();
    JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
    try {
      JobSession jobSession = jbpmContext.getJobSession();
      Collection jobIdsToIgnore = jobExecutor.getMonitoredJobIds();
      Job job = jobSession.getFirstDueJob(lockOwner, jobIdsToIgnore);
      if (job != null) {
        nextDueDate = job.getDueDate();
        jobExecutor.addMonitoredJobId(lockOwner, job.getId());
      }
    }
    catch (RuntimeException e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    catch (Error e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    finally {
      try {
        jbpmContext.close();
      }
      catch (RuntimeException e) {
        if (!DbPersistenceService.isLockingException(e)) throw e;
        // if this is a locking exception, keep it quiet
        StaleObjectLogConfigurer.getStaleObjectExceptionsLog()
            .error("failed to determine next due date for job executor thread " + lockOwner);
        nextDueDate = null;
      }
    }
    return nextDueDate;
  }

  /**
   * @deprecated As of jBPM 3.2.3, replaced by {@link #deactivate()}
   */
  public void setActive(boolean isActive) {
    if (isActive == false) deactivate();
  }

  /**
   * Indicates that this thread should stop running. Execution will cease shortly afterwards.
   */
  public void deactivate() {
    if (isActive) {
      isActive = false;
      interrupt();
    }
  }

  private static Log log = LogFactory.getLog(JobExecutorThread.class);
}
