/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.db;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.LockMode;
import org.hibernate.Session;
import org.hibernate.criterion.Restrictions;

import org.jbpm.JbpmException;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.exe.Token;
import org.jbpm.graph.node.ProcessState;
import org.jbpm.persistence.JbpmPersistenceException;

/**
 * are the graph related database operations.
 */
public class GraphSession {

  final Session session;
  /** @deprecated */
  final JbpmSession jbpmSession;

  /** @deprecated use {@link #GraphSession(Session)} instead */
  public GraphSession(JbpmSession jbpmSession) {
    this.session = jbpmSession.getSession();
    this.jbpmSession = jbpmSession;
  }

  public GraphSession(Session session) {
    this.session = session;
    this.jbpmSession = null;
  }

  // process definitions //////////////////////////////////////////////////////

  public void deployProcessDefinition(ProcessDefinition processDefinition) {
    String processDefinitionName = processDefinition.getName();
    // versioning applies to named process definitions only
    if (processDefinitionName != null) {
      // find the current latest process definition
      ProcessDefinition previousLatestVersion = findLatestProcessDefinition(processDefinitionName);
      // if there is a current latest process definition
      if (previousLatestVersion != null) {
        // take the next version number
        processDefinition.setVersion(previousLatestVersion.getVersion() + 1);
      }
      else {
        // start from 1
        processDefinition.setVersion(1);
      }
      session.save(processDefinition);
    }
    else {
      throw new JbpmException("process definition does not have a name");
    }
  }

  /**
   * saves the process definitions. this method does not assign a version number. that is the
   * responsibility of the {@link #deployProcessDefinition(ProcessDefinition)
   * deployProcessDefinition} method.
   */
  public void saveProcessDefinition(ProcessDefinition processDefinition) {
    try {
      session.save(processDefinition);
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not save " + processDefinition, e);
    }
  }

  /**
   * loads a process definition from the database by the identifier.
   * 
   * @throws JbpmPersistenceException in case the referenced process definition doesn't exist.
   */
  public ProcessDefinition loadProcessDefinition(long processDefinitionId) {
    try {
      return (ProcessDefinition) session.load(ProcessDefinition.class, new Long(
          processDefinitionId));
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not load process definition "
          + processDefinitionId, e);
    }
  }

  /**
   * gets a process definition from the database by the identifier.
   * 
   * @return the referenced process definition or null in case it doesn't exist.
   */
  public ProcessDefinition getProcessDefinition(long processDefinitionId) {
    try {
      return (ProcessDefinition) session.get(ProcessDefinition.class, new Long(
          processDefinitionId));
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not get process definition "
          + processDefinitionId, e);
    }
  }

  /**
   * queries the database for a process definition with the given name and version.
   */
  public ProcessDefinition findProcessDefinition(String name, int version) {
    try {
      return (ProcessDefinition) session.getNamedQuery("GraphSession.findProcessDefinitionByNameAndVersion")
          .setString("name", name)
          .setInteger("version", version)
          .uniqueResult();
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not find process definition '" + name
          + "' at version " + version, e);
    }
  }

  /**
   * queries the database for the latest version of a process definition with the given name.
   */
  public ProcessDefinition findLatestProcessDefinition(String name) {
    try {
      return (ProcessDefinition) session.getNamedQuery("GraphSession.findLatestProcessDefinitionQuery")
          .setString("name", name)
          .setMaxResults(1)
          .uniqueResult();
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not find process definition '" + name + "'", e);
    }
  }

  /**
   * queries the database for the latest version of each process definition. Process definitions
   * are distinct by name.
   */
  public List findLatestProcessDefinitions() {
    try {
      List tuples = session.getNamedQuery("GraphSession.findLatestProcessDefinitions").list();
      List result = new ArrayList();
      for (Iterator i = tuples.iterator(); i.hasNext();) {
        Object[] tuple = (Object[]) i.next();
        String name = (String) tuple[0];
        Integer version = (Integer) tuple[1];
        result.add(findProcessDefinition(name, version.intValue()));
      }
      return result;
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException(
          "could not find latest versions of process definitions", e);
    }
  }

  public List findProcessDefinitions(Collection processDefinitionIds) {
    return session.createCriteria(ProcessDefinition.class)
        .add(Restrictions.in("id", processDefinitionIds))
        .list();
  }

  /**
   * queries the database for all process definitions, ordered by name (ascending), then by
   * version (descending).
   */
  public List findAllProcessDefinitions() {
    try {
      return session.getNamedQuery("GraphSession.findAllProcessDefinitions").list();
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not find all process definitions", e);
    }
  }

  /**
   * queries the database for all versions of process definitions with the given name, ordered
   * by version (descending).
   */
  public List findAllProcessDefinitionVersions(String name) {
    try {
      return session.getNamedQuery("GraphSession.findAllProcessDefinitionVersions")
          .setString("name", name)
          .list();
    }
    catch (HibernateException e) {
      log.error(e);
      throw new JbpmPersistenceException("could not find all versions of process definition '"
          + name + "'", e);
    }
  }

  public void deleteProcessDefinition(long processDefinitionId) {
    deleteProcessDefinition(loadProcessDefinition(processDefinitionId));
  }

  public void deleteProcessDefinition(ProcessDefinition processDefinition) {
    try {
      // delete all the process instances of this definition
      List processInstanceIds = session.getNamedQuery("GraphSession.findAllProcessInstanceIdsForDefinition")
          .setLong("processDefinitionId", processDefinition.getId())
          .list();
      for (Iterator i = processInstanceIds.iterator(); i.hasNext();) {
        Long processInstanceId = (Long) i.next();
        ProcessInstance processInstance = getProcessInstance(processInstanceId.longValue());
        if (processInstance != null) {
          deleteProcessInstance(processInstance);
        }
        else {
          log.debug("process instance " + processInstanceId + " has been deleted already");
        }
      }

      List referencingProcessStates = findReferencingProcessStates(processDefinition);
      for (Iterator i = referencingProcessStates.iterator(); i.hasNext();) {
        ProcessState processState = (ProcessState) i.next();
        processState.setSubProcessDefinition(null);
      }

      // then delete the process definition
      session.delete(processDefinition);
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not delete " + processDefinition, e);
    }
  }

  List findReferencingProcessStates(ProcessDefinition subProcessDefinition) {
    return session.getNamedQuery("GraphSession.findReferencingProcessStates")
        .setEntity("subProcessDefinition", subProcessDefinition)
        .list();
  }

  // process instances ////////////////////////////////////////////////////////

  /**
   * @deprecated use {@link org.jbpm.JbpmContext#save(ProcessInstance)} instead.
   * @throws UnsupportedOperationException to prevent invocation
   */
  public void saveProcessInstance(ProcessInstance processInstance) {
    throw new UnsupportedOperationException("use JbpmContext.save(ProcessInstance) instead");
  }

  /**
   * loads a process instance from the database by the identifier. This throws an exception in
   * case the process instance does not exist.
   * 
   * @see #getProcessInstance(long)
   * @throws JbpmPersistenceException in case the process instance doesn't exist.
   */
  public ProcessInstance loadProcessInstance(long processInstanceId) {
    try {
      return (ProcessInstance) session.load(ProcessInstance.class, new Long(processInstanceId));
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException(
          "could not load process instance " + processInstanceId, e);
    }
  }

  /**
   * gets a process instance from the database by the identifier. This method returns null in
   * case the given process instance does not exist.
   */
  public ProcessInstance getProcessInstance(long processInstanceId) {
    try {
      return (ProcessInstance) session.get(ProcessInstance.class, new Long(processInstanceId));
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not get process instance " + processInstanceId,
          e);
    }
  }

  /**
   * loads a token from the database by the identifier.
   * 
   * @return the token.
   * @throws JbpmPersistenceException in case the referenced token doesn't exist.
   */
  public Token loadToken(long tokenId) {
    try {
      return (Token) session.load(Token.class, new Long(tokenId));
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not load token " + tokenId, e);
    }
  }

  /**
   * gets a token from the database by the identifier.
   * 
   * @return the token or null in case the token doesn't exist.
   */
  public Token getToken(long tokenId) {
    try {
      return (Token) session.get(Token.class, new Long(tokenId));
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not get token " + tokenId, e);
    }
  }

  /**
   * locks a process instance in the database.
   */
  public void lockProcessInstance(long processInstanceId) {
    try {
      session.load(ProcessInstance.class, new Long(processInstanceId), LockMode.UPGRADE);
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException(
          "could not lock process instance " + processInstanceId, e);
    }
  }

  /**
   * locks a process instance in the database.
   */
  public void lockProcessInstance(ProcessInstance processInstance) {
    try {
      session.lock(processInstance, LockMode.UPGRADE);
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not lock " + processInstance, e);
    }
  }

  /**
   * fetches all processInstances for the given process definition from the database. The
   * returned list of process instances is sorted start date, youngest first.
   */
  public List findProcessInstances(long processDefinitionId) {
    try {
      return session.getNamedQuery("GraphSession.findAllProcessInstancesForDefinition")
          .setLong("processDefinitionId", processDefinitionId)
          .list();
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException(
          "could not find process instances for process definition " + processDefinitionId, e);
    }
  }

  public void deleteProcessInstance(long processInstanceId) {
    deleteProcessInstance(loadProcessInstance(processInstanceId));
  }

  public void deleteProcessInstance(ProcessInstance processInstance) {
    deleteProcessInstance(processInstance, true, true);
  }

  public void deleteProcessInstance(ProcessInstance processInstance, boolean includeTasks,
      boolean includeJobs) {
    if (processInstance == null) {
      throw new IllegalArgumentException("processInstance cannot be null");
    }

    try {
      // delete outstanding jobs
      if (includeJobs) deleteJobs(processInstance);

      // delete logs
      deleteLogs(processInstance);

      // detach from superprocess token
      Token superProcessToken = processInstance.getSuperProcessToken();
      if (superProcessToken != null)
        detachFromSuperProcess(processInstance, superProcessToken);

      // delete subprocess instances
      deleteSubProcesses(processInstance);

      // delete tasks; since TaskLogs reference tasks, logs are deleted first
      if (includeTasks) deleteTasks(processInstance);

      // delete the process instance
      log.debug("deleting " + processInstance);
      session.delete(processInstance);
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not delete " + processInstance, e);
    }
  }

  void deleteJobs(ProcessInstance processInstance) {
    log.debug("deleting jobs for " + processInstance);
    int entityCount = session.getNamedQuery("GraphSession.deleteJobsForProcessInstance")
        .setEntity("processInstance", processInstance)
        .executeUpdate();
    log.debug("deleted " + entityCount + " jobs for " + processInstance);
  }

  void deleteLogs(ProcessInstance processInstance) {
    log.debug("deleting logs for " + processInstance);
    List logs = session.getNamedQuery("GraphSession.findLogsForProcessInstance")
        .setEntity("processInstance", processInstance)
        .list();
    for (Iterator i = logs.iterator(); i.hasNext();) {
      session.delete(i.next());
    }
  }

  void detachFromSuperProcess(ProcessInstance processInstance, Token superProcessToken) {
    log.debug("detaching " + processInstance + " from " + superProcessToken);
    processInstance.setSuperProcessToken(null);
    superProcessToken.setSubProcessInstance(null);
  }

  void deleteSubProcesses(ProcessInstance processInstance) {
    log.debug("deleting subprocesses for " + processInstance);
    List subProcessInstances = session.getNamedQuery("GraphSession.findSubProcessInstances")
        .setEntity("processInstance", processInstance)
        .list();

    if (subProcessInstances.isEmpty()) {
      log.debug("no subprocesses to delete for " + processInstance);
      return;
    }

    for (Iterator i = subProcessInstances.iterator(); i.hasNext();) {
      ProcessInstance subProcessInstance = (ProcessInstance) i.next();
      log.debug("preparing to delete Sub" + subProcessInstance);
      deleteProcessInstance(subProcessInstance);
    }
  }

  void deleteTasks(ProcessInstance processInstance) {
    log.debug("deleting tasks for " + processInstance);
    List tasks = session.getNamedQuery("GraphSession.findTaskInstancesForProcessInstance")
        .setEntity("processInstance", processInstance)
        .list();
    for (Iterator i = tasks.iterator(); i.hasNext();) {
      session.delete(i.next());
    }
  }

  public static class AverageNodeTimeEntry {

    private long nodeId;
    private String nodeName;
    private int count;
    private long averageDuration;
    private long minDuration;
    private long maxDuration;

    public long getNodeId() {
      return nodeId;
    }

    public void setNodeId(final long nodeId) {
      this.nodeId = nodeId;
    }

    public String getNodeName() {
      return nodeName;
    }

    public void setNodeName(final String nodeName) {
      this.nodeName = nodeName;
    }

    public int getCount() {
      return count;
    }

    public void setCount(final int count) {
      this.count = count;
    }

    public long getAverageDuration() {
      return averageDuration;
    }

    public void setAverageDuration(final long averageDuration) {
      this.averageDuration = averageDuration;
    }

    public long getMinDuration() {
      return minDuration;
    }

    public void setMinDuration(final long minDuration) {
      this.minDuration = minDuration;
    }

    public long getMaxDuration() {
      return maxDuration;
    }

    public void setMaxDuration(final long maxDuration) {
      this.maxDuration = maxDuration;
    }
  }

  public List calculateAverageTimeByNode(long processDefinitionId, long minumumDurationMillis) {
    try {
      List tuples = session.getNamedQuery("GraphSession.calculateAverageTimeByNode")
          .setLong("processDefinitionId", processDefinitionId)
          .setDouble("minimumDuration", minumumDurationMillis)
          .list();

      List results;
      if (!tuples.isEmpty()) {
        results = new ArrayList();

        for (Iterator i = tuples.iterator(); i.hasNext();) {
          Object[] values = (Object[]) i.next();
          AverageNodeTimeEntry averageNodeTimeEntry = new AverageNodeTimeEntry();
          averageNodeTimeEntry.setNodeId(((Number) values[0]).longValue());
          averageNodeTimeEntry.setNodeName((String) values[1]);
          averageNodeTimeEntry.setCount(((Number) values[2]).intValue());
          averageNodeTimeEntry.setAverageDuration(((Number) values[3]).longValue());
          averageNodeTimeEntry.setMinDuration(((Number) values[4]).longValue());
          averageNodeTimeEntry.setMaxDuration(((Number) values[5]).longValue());

          results.add(averageNodeTimeEntry);
        }
      }
      else {
        results = Collections.EMPTY_LIST;
      }
      return results;
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not calculate average time by node for "
          + processDefinitionId, e);
    }
  }

  public List findActiveNodesByProcessInstance(ProcessInstance processInstance) {
    try {
      return session.getNamedQuery("GraphSession.findActiveNodesByProcessInstance")
          .setEntity("processInstance", processInstance)
          .list();
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not find active nodes for " + processInstance,
          e);
    }
  }

  public ProcessInstance getProcessInstance(ProcessDefinition processDefinition, String key) {
    try {
      return (ProcessInstance) session.getNamedQuery("GraphSession.findProcessInstanceByKey")
          .setEntity("processDefinition", processDefinition)
          .setString("key", key)
          .uniqueResult();
    }
    catch (HibernateException e) {
      handle(e);
      throw new JbpmPersistenceException("could not get process instance with key '" + key
          + "'", e);
    }
  }

  public ProcessInstance loadProcessInstance(ProcessDefinition processDefinition, String key) {
    ProcessInstance processInstance = getProcessInstance(processDefinition, key);
    if (processInstance == null) {
      throw new JbpmException("no process instance was found with key '" + key + "'");
    }
    return processInstance;
  }

  private void handle(HibernateException exception) {
    // exception will be rethrown, no need to log here at a verbose level
    log.debug(exception);
    if (jbpmSession != null) jbpmSession.handleException();
  }

  private static final Log log = LogFactory.getLog(GraphSession.class);
}
