package org.jbpm.job.executor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.Random;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.db.JobSession;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.job.Job;
import org.jbpm.persistence.db.DbPersistenceService;
import org.jbpm.persistence.db.StaleObjectLogConfigurer;

public class JobExecutorThread extends Thread {

  private final JobExecutor jobExecutor;
  private volatile boolean isActive = true;
  private final Random random = new Random();

  public JobExecutorThread(String name, JobExecutor jobExecutor) {
    super(jobExecutor.getThreadGroup(), name);
    this.jobExecutor = jobExecutor;
  }

  /**
   * @deprecated use {@link #JobExecutorThread(String, JobExecutor)} instead
   */
  public JobExecutorThread(String name, JobExecutor jobExecutor,
    JbpmConfiguration jbpmConfiguration, int idleInterval, int maxIdleInterval,
    long maxLockTime, int maxHistory) {
    super(jobExecutor.getThreadGroup(), name);
    this.jobExecutor = jobExecutor;
  }

  public void run() {
    int currentIdleInterval = jobExecutor.getIdleInterval();
    while (isActive) {
      // acquire jobs; on exception, call returns empty collection
      Collection acquiredJobs = acquireJobs();
      // execute jobs
      boolean success = true;
      for (Iterator i = acquiredJobs.iterator(); i.hasNext() && isActive;) {
        Job job = (Job) i.next();
        try {
          executeJob(job);
        }
        catch (Exception e) {
          // on exception, call returns normally
          saveJobException(job, e);
          success = false;
          break;
        }
      }

      // if still active wait for next batch
      if (isActive) {
        try {
          if (success) {
            // reset current idle interval
            currentIdleInterval = jobExecutor.getIdleInterval();
            // wait for next due job
            long waitPeriod = getWaitPeriod(currentIdleInterval);
            if (waitPeriod > 0) {
              synchronized (jobExecutor) {
                jobExecutor.wait(waitPeriod);
              }
            }
          }
          else {
            // wait a random period, at least half the current idle interval
            int waitPeriod = currentIdleInterval / 2;
            // sleep instead of waiting on jobExecutor
            // to prevent DbMessageService from waking up this thread
            sleep(waitPeriod + random.nextInt(waitPeriod));
            // after an exception, double the current idle interval
            // to avoid continuous failures during anomalous conditions
            currentIdleInterval *= 2;
            // enforce maximum idle interval
            int maxIdleInterval = jobExecutor.getMaxIdleInterval();
            if (currentIdleInterval > maxIdleInterval || currentIdleInterval < 0) {
              currentIdleInterval = maxIdleInterval;
            }
          }
        }
        catch (InterruptedException e) {
          if (log.isDebugEnabled()) log.debug(getName() + " got interrupted");
        }
      }
    }
    log.info(getName() + " leaves cyberspace");
  }

  protected Collection acquireJobs() {
    boolean debug = log.isDebugEnabled();
    Collection jobs;
    // acquire monitor before creating context and allocating resources
    synchronized (jobExecutor) {
      JbpmContext jbpmContext = jobExecutor.getJbpmConfiguration().createJbpmContext();
      try {
        // search for acquirable job
        String lockOwner = getName();
        JobSession jobSession = jbpmContext.getJobSession();
        Job firstJob = jobSession.getFirstAcquirableJob(lockOwner);
        // is there a job?
        if (firstJob != null) {
          // is job exclusive?
          if (firstJob.isExclusive()) {
            // find other exclusive jobs
            ProcessInstance processInstance = firstJob.getProcessInstance();
            jobs = jobSession.findExclusiveJobs(lockOwner, processInstance);
            if (debug) log.debug("acquiring exclusive " + jobs + " for " + processInstance);
          }
          else {
            jobs = Collections.singletonList(firstJob);
            if (debug) log.debug("acquiring " + firstJob);
          }

          // acquire jobs
          Date lockTime = new Date();
          for (Iterator i = jobs.iterator(); i.hasNext();) {
            // lock job
            Job job = (Job) i.next();
            job.setLockOwner(lockOwner);
            job.setLockTime(lockTime);
            // has job failed previously?
            if (job.getException() != null) {
              // decrease retry count
              int retries = job.getRetries() - 1;
              job.setRetries(retries);
              if (debug) log.debug(job + " has " + retries + " retries remaining");
            }
          }
          if (debug) log.debug("acquired " + jobs);
        }
        else {
          jobs = Collections.EMPTY_LIST;
          if (debug) log.debug("no acquirable job found");
        }
      }
      catch (RuntimeException e) {
        jbpmContext.setRollbackOnly();
        jobs = Collections.EMPTY_LIST;
        if (debug) log.debug("failed to acquire jobs", e);
      }
      catch (Error e) {
        jbpmContext.setRollbackOnly();
        throw e;
      }
      finally {
        try {
          jbpmContext.close();
        }
        catch (RuntimeException e) {
          jobs = Collections.EMPTY_LIST;
          if (debug) log.debug("failed to acquire jobs", e);
        }
      }
    }
    return jobs;
  }

  protected void executeJob(Job job) throws Exception {
    JbpmContext jbpmContext = jobExecutor.getJbpmConfiguration().createJbpmContext();
    try {
      JobSession jobSession = jbpmContext.getJobSession();
      jobSession.reattachJob(job);

      // register process instance for automatic save
      // see https://jira.jboss.org/jira/browse/JBPM-1015
      jbpmContext.addAutoSaveProcessInstance(job.getProcessInstance());

      if (log.isDebugEnabled()) log.debug("executing " + job);
      if (job.execute(jbpmContext)) jobSession.deleteJob(job);
    }
    catch (Exception e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    catch (Error e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    finally {
      jbpmContext.close();
    }
  }

  private void saveJobException(Job job, Exception exception) {
    // if this is a locking exception, keep it quiet
    if (DbPersistenceService.isLockingException(exception)) {
      StaleObjectLogConfigurer.getStaleObjectExceptionsLog()
        .error("failed to execute " + job, exception);
    }
    else {
      log.error("failed to execute " + job, exception);
    }

    boolean debug = log.isDebugEnabled();
    JbpmContext jbpmContext = jobExecutor.getJbpmConfiguration().createJbpmContext();
    try {
      // do not reattach existing job as it contains undesired updates
      JobSession jobSession = jbpmContext.getJobSession();
      job = jobSession.loadJob(job.getId());

      // print and save exception
      StringWriter out = new StringWriter();
      exception.printStackTrace(new PrintWriter(out));
      job.setException(out.toString());
    }
    catch (RuntimeException e) {
      jbpmContext.setRollbackOnly();
      if (debug) log.debug("failed to save job exception", e);
    }
    catch (Error e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    finally {
      try {
        jbpmContext.close();
      }
      catch (RuntimeException e) {
        if (debug) log.debug("failed to save job exception", e);
      }
    }
  }

  protected long getWaitPeriod(int currentIdleInterval) {
    Date nextDueDate = getNextDueDate();
    if (nextDueDate != null) {
      long waitPeriod = nextDueDate.getTime() - System.currentTimeMillis();
      if (waitPeriod < currentIdleInterval) return waitPeriod;
    }
    return currentIdleInterval;
  }

  protected Date getNextDueDate() {
    Date nextDueDate;
    JbpmContext jbpmContext = jobExecutor.getJbpmConfiguration().createJbpmContext();
    try {
      String lockOwner = getName();
      Job job = jbpmContext.getJobSession()
        .getFirstDueJob(lockOwner, jobExecutor.getMonitoredJobIds());
      if (job != null) {
        jobExecutor.addMonitoredJobId(lockOwner, job.getId());
        nextDueDate = job.getDueDate();
      }
      else {
        nextDueDate = null;
        if (log.isDebugEnabled()) log.debug("no due job found");
      }
    }
    catch (RuntimeException e) {
      jbpmContext.setRollbackOnly();
      nextDueDate = null;
      if (log.isDebugEnabled()) log.debug("failed to determine next due date", e);
    }
    catch (Error e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    finally {
      try {
        jbpmContext.close();
      }
      catch (RuntimeException e) {
        nextDueDate = null;
        if (log.isDebugEnabled()) log.debug("failed to determine next due date", e);
      }
    }
    return nextDueDate;
  }

  /**
   * @deprecated As of jBPM 3.2.3, replaced by {@link #deactivate()}
   */
  public void setActive(boolean isActive) {
    if (isActive == false) deactivate();
  }

  /**
   * Indicates that this thread should stop running. Execution will cease
   * shortly afterwards.
   */
  public void deactivate() {
    if (isActive) {
      isActive = false;
      interrupt();
    }
  }

  private static final Log log = LogFactory.getLog(JobExecutorThread.class);
}
