/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.cts.gateway.exclusive;

// $Id: ExclusiveGatewayMergeTest.java 1987 2008-08-22 18:56:09Z thomas.diesler@jboss.com $

import java.io.IOException;

import org.jboss.bpm.client.SignalListener;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.EventBuilder;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.MessageBuilder;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.TaskBuilder;
import org.jboss.bpm.model.Assignment.AssignTime;
import org.jboss.bpm.model.Expression.ExpressionLanguage;
import org.jboss.bpm.model.Gateway.GatewayType;
import org.jboss.bpm.model.Signal.SignalType;
import org.jboss.bpm.test.DefaultEngineTestCase;

/**
 * Exclusive data-based gateway that has multiple incoming sequence flows. The first token arriving from one of the
 * incoming sequence flows proceeds along the outgoing sequence flow. All other token are ignored until a token has
 * arrived from each incoming sequence flow.
 * 
 * @author thomas.diesler@jboss.com
 * @since 06-Aug-2008
 */
public class ExclusiveGatewayMergeTest extends DefaultEngineTestCase
{
  public void testGateA() throws Exception
  {
    Process proc = getProcess();

    // Add a signal listener that sends the other start trigger signal
    SignalManager sigManager = SignalManager.locateSignalManager();
    MergeListener sigListener = new MergeListener(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "B"));
    sigManager.addSignalListener(sigListener);

    try
    {
      // Send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      // Wait for the process to end
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Verify the result
    Message endMessage = getMessages().get(0);
    assertNotNull("EndMessage expected", endMessage);
    assertEquals("TaskA", endMessage.getPropertyValue("taskValue"));
  }

  public void testGateB() throws Exception
  {
    Process proc = getProcess();

    // Add a signal listener that sends the other start trigger signal
    SignalManager sigManager = SignalManager.locateSignalManager();
    MergeListener sigListener = new MergeListener(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));
    sigManager.addSignalListener(sigListener);

    try
    {
      // Send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "B"));

      // Wait for the process to end
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Verify the result
    Message endMessage = getMessages().get(0);
    assertNotNull("EndMessage expected", endMessage);
    assertEquals("TaskB", endMessage.getPropertyValue("taskValue"));
  }

  public void testInvalidToken() throws Exception
  {
    Process proc = getProcess();

    // Add a signal listener that sends the other start trigger signal
    SignalManager sigManager = SignalManager.locateSignalManager();
    MergeListener sigListener = new MergeListener(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));
    sigManager.addSignalListener(sigListener);

    try
    {
      // Send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      try
      {
        proc.waitForEnd();
        fail("Expected: Unexpected token from: SequenceFlow[TaskA->Merge]");
      }
      catch (RuntimeException rte)
      {
        // expected
      }
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Add a signal listener that sends the other start trigger signal
    sigListener = new MergeListener(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "B"));
    sigManager.addSignalListener(sigListener);

    try
    {
      // Send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      // Wait for the process to end
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Verify the result
    Message endMessage = getMessages().get(0);
    assertNotNull("EndMessage expected", endMessage);
    assertEquals("TaskA", endMessage.getPropertyValue("taskValue"));
  }

  public Process getProcess() throws IOException
  {
    ProcessBuilder procBuilder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    EventBuilder eventBuilder = procBuilder.addProcess(getName()).addStartEvent("StartA");
    eventBuilder.addEventDetail(EventDetail.EventDetailType.Signal).addSignalRef(SignalType.SYSTEM_START_TRIGGER, "A");
    TaskBuilder taskBuilder = procBuilder.addSequenceFlow("TaskA").addTask("TaskA");
    taskBuilder.addAssignment(AssignTime.Start, ExpressionLanguage.MVEL, "'TaskA'", "taskValue");
    taskBuilder.addSequenceFlow("Merge");
    eventBuilder = procBuilder.addStartEvent("StartB");
    eventBuilder.addEventDetail(EventDetail.EventDetailType.Signal).addSignalRef(SignalType.SYSTEM_START_TRIGGER, "B");
    taskBuilder = procBuilder.addSequenceFlow("TaskB").addTask("TaskB");
    taskBuilder.addAssignment(AssignTime.Start, ExpressionLanguage.MVEL, "'TaskB'", "taskValue");
    taskBuilder.addSequenceFlow("Merge");
    procBuilder.addGateway("Merge", GatewayType.Exclusive).addSequenceFlow("End");
    procBuilder.addEndEvent("End").addEventDetail(EventDetail.EventDetailType.Message).addMessageRef("EndMessage");
    MessageBuilder msgBuilder = procBuilder.addMessage("EndMessage");
    msgBuilder.addToRef(getTestID()).addProperty("taskValue", null, true);
    Process proc = procBuilder.getProcess();
    return proc;
  }

  class MergeListener implements SignalListener
  {
    private Signal nextSignal;

    public MergeListener(Signal nextSignal)
    {
      this.nextSignal = nextSignal;
    }

    public boolean acceptSignal(Signal signal)
    {
      return signal.getSignalType() == SignalType.SYSTEM_GATEWAY_ENTER;
    }

    public void catchSignal(Signal signal)
    {
      if (nextSignal != null)
      {
        SignalManager signalManager = SignalManager.locateSignalManager();
        signalManager.throwSignal(nextSignal);
        nextSignal = null;
      }
    }
  }
}
