/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.cts.startevent;

// $Id: StartEventSignalTest.java 1987 2008-08-22 18:56:09Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.util.List;

import org.jboss.bpm.client.SignalListener;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.EventBuilder;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.MessageBuilder;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.TaskBuilder;
import org.jboss.bpm.model.Assignment.AssignTime;
import org.jboss.bpm.model.EventDetail.EventDetailType;
import org.jboss.bpm.model.Expression.ExpressionLanguage;
import org.jboss.bpm.model.Signal.SignalType;
import org.jboss.bpm.test.DefaultEngineTestCase;

/**
 * Test start event with signal trigger
 * 
 * @author thomas.diesler@jboss.com
 * @since 06-Aug-2008
 */
public class StartEventSignalTest extends DefaultEngineTestCase
{
  public void testStart() throws Exception
  {
    Process proc = getProcess();

    // Start the process and send start trigger signal
    proc.startProcess();
    SignalManager signalManager = SignalManager.locateSignalManager();
    signalManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

    // Wait for the process to end
    proc.waitForEnd();

    // Verify the result
    Message endMessage = getMessages().get(0);
    assertEquals("TaskA", endMessage.getPropertyValue("taskValue"));
  }

  public void testRestart() throws Exception
  {
    Process proc = getProcess();

    // Start the process and send start trigger signal
    proc.startProcess();
    SignalManager signalManager = SignalManager.locateSignalManager();
    signalManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

    // Wait for the process to end
    proc.waitForEnd();

    // Start the process and send start trigger signal
    proc.startProcess();
    signalManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

    // Wait for the process to end
    proc.waitForEnd();

    // Verify the result
    List<Message> messages = getMessages();
    assertEquals(2, messages.size());
    assertEquals("TaskA", messages.get(0).getPropertyValue("taskValue"));
    assertEquals("TaskA", messages.get(1).getPropertyValue("taskValue"));
  }

  public void testRestartBeforeEnd() throws Exception
  {
    Process proc = getProcess();

    // Add a signal listener that sends the other start trigger signal
    TaskListener sigListener = new TaskListener(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));
    SignalManager sigManager = SignalManager.locateSignalManager();
    sigManager.addSignalListener(sigListener);

    try
    {
      // Start the process and send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      // Wait for the process to end
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Verify the result
    List<Message> messages = getMessages();
    assertEquals(2, messages.size());
    assertEquals("TaskA", messages.get(0).getPropertyValue("taskValue"));
    assertEquals("TaskA", messages.get(1).getPropertyValue("taskValue"));
  }

  public Process getProcess() throws IOException
  {
    ProcessBuilder procBuilder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    EventBuilder eventBuilder = procBuilder.addProcess("StartEventSignal").addStartEvent("StartA");
    eventBuilder.addEventDetail(EventDetailType.Signal).addSignalRef(SignalType.SYSTEM_START_TRIGGER, "A");
    TaskBuilder taskBuilder = procBuilder.addSequenceFlow("TaskA").addTask("TaskA");
    taskBuilder.addAssignment(AssignTime.Start, ExpressionLanguage.MVEL, "'TaskA'", "taskValue");
    taskBuilder.addSequenceFlow("End");
    procBuilder.addEndEvent("End").addEventDetail(EventDetailType.Message).addMessageRef("EndMessage");
    MessageBuilder msgBuilder = procBuilder.addMessage("EndMessage");
    msgBuilder.addToRef(getTestID()).addProperty("taskValue", null, true);
    Process proc = procBuilder.getProcess();
    return proc;
  }

  public static class TaskListener implements SignalListener
  {
    private Signal nextSignal;

    public TaskListener(Signal nextSignal)
    {
      this.nextSignal = nextSignal;
    }

    public boolean acceptSignal(Signal signal)
    {
      return signal.getSignalType() == SignalType.SYSTEM_TASK_ENTER;
    }

    public void catchSignal(Signal signal)
    {
      if (nextSignal != null)
      {
        SignalManager signalManager = SignalManager.locateSignalManager();
        signalManager.throwSignal(nextSignal);
        nextSignal = null;
      }
    }
  }
}
