/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.pattern.control.synchronization;

// $Id: SynchronizationTest.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.io.IOException;

import org.jboss.bpm.client.SignalListener;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.EventBuilder;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.MessageBuilder;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.TaskBuilder;
import org.jboss.bpm.model.Assignment.AssignTime;
import org.jboss.bpm.model.Expression.ExpressionLanguage;
import org.jboss.bpm.model.Signal.SignalType;
import org.jboss.bpm.test.DefaultEngineTestCase;

/**
 * Parallel gateway that that has multiple incoming sequence flows. Each token arriving from an incoming sequence flow
 * is stored in the gateway until a token has arrived from each incoming sequence flow. The tokens are merged together
 * and leave the gateway as a one.
 * 
 * @author thomas.diesler@jboss.com
 * @since 06-Aug-2008
 */
public class SynchronizationTest extends DefaultEngineTestCase
{
  public void testParallelMerge() throws Exception
  {
    Process proc = getProcess();

    // Add a signal listener that sends the other start trigger signal
    MergeListener sigListener = new MergeListener(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "B"));
    SignalManager sigManager = SignalManager.locateSignalManager();
    sigManager.addSignalListener(sigListener);

    try
    {
      // Start the process and send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      // Wait for the process to end
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Verify the result
    Message endMessage = getMessages().get(0);
    assertEquals("TaskA", endMessage.getPropertyValue("taskValueA"));
    assertEquals("TaskB", endMessage.getPropertyValue("taskValueB"));
  }

  public void testMergeTimeout() throws Exception
  {
    Process proc = getProcess();

    // Start the process and send start trigger signal
    proc.startProcess();
    SignalManager signalManager = SignalManager.locateSignalManager();
    signalManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

    // Wait for the process to end
    try
    {
      proc.waitForEnd(1000);
      fail("timeout expected");
    }
    catch (RuntimeException rte)
    {
      // expected
    }
  }

  public void testInvalidToken() throws Exception
  {
    Process proc = getProcess();

    // Add a signal listener that sends the other start trigger signal
    Signal startTrigger = new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A");
    MergeListener sigListener = new MergeListener(startTrigger);
    SignalManager sigManager = SignalManager.locateSignalManager();
    sigManager.addSignalListener(sigListener);

    try
    {
      // Start the process and send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      try
      {
        proc.waitForEnd();
        fail("Expected: Unexpected token from: SequenceFlow[TaskA->Merge]");
      }
      catch (RuntimeException rte)
      {
        // expected
      }
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Add a signal listener that sends the other start trigger signal
    startTrigger = new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "B");
    sigListener = new MergeListener(startTrigger);
    sigManager.addSignalListener(sigListener);

    try
    {
      // Start the process and send start trigger signal
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      // Wait for the process to end
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    // Verify the result
    Message endMessage = getMessages().get(0);
    assertEquals("TaskA", endMessage.getPropertyValue("taskValueA"));
    assertEquals("TaskB", endMessage.getPropertyValue("taskValueB"));
  }

  public Process getProcess() throws IOException
  {
    ProcessBuilder procBuilder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    EventBuilder eventBuilder = procBuilder.addProcess("Synchronization").addStartEvent("StartA");
    eventBuilder.addEventDetail(EventDetail.EventDetailType.Signal).addSignalRef(SignalType.SYSTEM_START_TRIGGER, "A");
    TaskBuilder taskBuilder = procBuilder.addSequenceFlow("TaskA").addTask("TaskA");
    taskBuilder.addAssignment(AssignTime.Start, ExpressionLanguage.MVEL, "'TaskA'", "taskValueA");
    taskBuilder.addSequenceFlow("Merge");
    eventBuilder = procBuilder.addStartEvent("StartB");
    eventBuilder.addEventDetail(EventDetail.EventDetailType.Signal).addSignalRef(SignalType.SYSTEM_START_TRIGGER, "B");
    taskBuilder = procBuilder.addSequenceFlow("TaskB").addTask("TaskB");
    taskBuilder.addAssignment(AssignTime.Start, ExpressionLanguage.MVEL, "'TaskB'", "taskValueB");
    taskBuilder.addSequenceFlow("Merge");
    procBuilder.addGateway("Merge", Gateway.GatewayType.Parallel).addSequenceFlow("End");
    procBuilder.addEndEvent("End").addEventDetail(EventDetail.EventDetailType.Message).addMessageRef("EndMessage");
    MessageBuilder msgBuilder = procBuilder.addMessage("EndMessage");
    msgBuilder.addToRef(getTestID()).addProperty("taskValueA", null, true).addProperty("taskValueB", null, true);
    Process proc = procBuilder.getProcess();
    return proc;
  }

  public static class MergeListener implements SignalListener
  {
    private Signal nextSignal;

    public MergeListener(Signal nextSignal)
    {
      this.nextSignal = nextSignal;
    }

    public boolean acceptSignal(Signal signal)
    {
      return signal.getSignalType() == SignalType.SYSTEM_GATEWAY_ENTER;
    }

    public void catchSignal(Signal signal)
    {
      if (nextSignal != null)
      {
        SignalManager signalManager = SignalManager.locateSignalManager();
        signalManager.throwSignal(nextSignal);
        nextSignal = null;
      }
    }
  }
}
