/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.dialect.api10;

// $Id: ProcessUnmarshaller.java 1993 2008-08-25 10:45:32Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.io.Reader;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;

import javax.management.ObjectName;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import org.jboss.bpm.InvalidProcessException;
import org.jboss.bpm.NotImplementedException;
import org.jboss.bpm.client.DialectHandler;
import org.jboss.bpm.client.ProcessManager;
import org.jboss.bpm.dialect.api10.model.JAXBAssignment;
import org.jboss.bpm.dialect.api10.model.JAXBCancelEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBCompensationEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBComplexGateway;
import org.jboss.bpm.dialect.api10.model.JAXBConditionalEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBEndEvent;
import org.jboss.bpm.dialect.api10.model.JAXBErrorEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBEvent;
import org.jboss.bpm.dialect.api10.model.JAXBEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBExclusiveGateway;
import org.jboss.bpm.dialect.api10.model.JAXBExpression;
import org.jboss.bpm.dialect.api10.model.JAXBFlow;
import org.jboss.bpm.dialect.api10.model.JAXBFlowObject;
import org.jboss.bpm.dialect.api10.model.JAXBGateway;
import org.jboss.bpm.dialect.api10.model.JAXBHandler;
import org.jboss.bpm.dialect.api10.model.JAXBInclude;
import org.jboss.bpm.dialect.api10.model.JAXBInclusiveGateway;
import org.jboss.bpm.dialect.api10.model.JAXBInputSet;
import org.jboss.bpm.dialect.api10.model.JAXBIntermediateEvent;
import org.jboss.bpm.dialect.api10.model.JAXBMessage;
import org.jboss.bpm.dialect.api10.model.JAXBMessageEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBMessageFlow;
import org.jboss.bpm.dialect.api10.model.JAXBMessageRef;
import org.jboss.bpm.dialect.api10.model.JAXBOutputSet;
import org.jboss.bpm.dialect.api10.model.JAXBParallelGateway;
import org.jboss.bpm.dialect.api10.model.JAXBProcess;
import org.jboss.bpm.dialect.api10.model.JAXBProperty;
import org.jboss.bpm.dialect.api10.model.JAXBSequenceFlow;
import org.jboss.bpm.dialect.api10.model.JAXBSignal;
import org.jboss.bpm.dialect.api10.model.JAXBSignalEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBStartEvent;
import org.jboss.bpm.dialect.api10.model.JAXBTask;
import org.jboss.bpm.dialect.api10.model.JAXBTerminateEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBTimerEventDetail;
import org.jboss.bpm.dialect.api10.model.ObjectFactory;
import org.jboss.bpm.model.EventBuilder;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.GatewayBuilder;
import org.jboss.bpm.model.MessageBuilder;
import org.jboss.bpm.model.ObjectNameFactory;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.TaskBuilder;
import org.jboss.bpm.model.SequenceFlow.ConditionType;
import org.jboss.bpm.runtime.Handler;

/**
 * A JAXB unmarshaller for a Process
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public class ProcessUnmarshaller
{
  public Process unmarshallProcess(Reader xml, boolean isInclude) throws JAXBException, IOException
  {
    JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
    Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
    unmarshaller.setProperty("com.sun.xml.bind.ObjectFactory", new ObjectFactory());
    JAXBProcess jaxbProc = (JAXBProcess)unmarshaller.unmarshal(xml);
    Process proc = adaptProcess(jaxbProc, isInclude);
    return proc;
  }

  private Process adaptProcess(JAXBProcess jaxbProc, boolean isInclude) throws IOException
  {
    ProcessBuilder procBuilder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    procBuilder.addProcess(jaxbProc.getName());

    // Process Includes
    processIncludes(procBuilder, jaxbProc);

    // Process Messages
    for (JAXBMessage jaxb : jaxbProc.getMessages())
    {
      MessageBuilder msgBuilder = procBuilder.addMessage(jaxb.getName());
      adaptJAXBMessage(msgBuilder, jaxb);
    }
    
    // Process Properties
    for (JAXBProperty jaxbProp : jaxbProc.getProperties())
    {
      procBuilder.addProperty(jaxbProp.getName(), jaxbProp.getValue());
    }
    
    // Process Assignments
    for (JAXBAssignment jaxbAss : jaxbProc.getAssignments())
    {
      JAXBExpression jaxbFrom = jaxbAss.getFrom();
      String jaxbTo = jaxbAss.getTo();
      procBuilder.addAssignment(jaxbAss.getAssignTime(), jaxbFrom.getLang(), jaxbFrom.getBody(), jaxbTo);
    }

    // Process FlowObjects
    for (JAXBFlowObject jaxbFlowObject : jaxbProc.getFlowObjects())
    {
      if (jaxbFlowObject instanceof JAXBEvent)
      {
        adaptJAXBEvent(procBuilder, jaxbProc, (JAXBEvent)jaxbFlowObject);
      }
      else if (jaxbFlowObject instanceof JAXBTask)
      {
        adaptTask(procBuilder, jaxbProc, (JAXBTask)jaxbFlowObject);
      }
      else if (jaxbFlowObject instanceof JAXBGateway)
      {
        adaptJAXBGateway(procBuilder, jaxbProc, (JAXBGateway)jaxbFlowObject);
      }
      else
      {
        throw new IllegalStateException("Unsupported flow object: " + jaxbFlowObject);
      }
      
      // FlowObject Assignments
      for (JAXBAssignment jaxbAss : jaxbFlowObject.getAssignments())
      {
        JAXBExpression jaxbFrom = jaxbAss.getFrom();
        String jaxbTo = jaxbAss.getTo();
        procBuilder.addAssignment(jaxbAss.getAssignTime(), jaxbFrom.getLang(), jaxbFrom.getBody(), jaxbTo);
      }
    }

    Process proc = (isInclude ? procBuilder.getProcessForInclude() : procBuilder.getProcess());
    return proc;
  }

  private void processIncludes(ProcessBuilder procBuilder, JAXBProcess jaxbProc) throws IOException
  {
    for (JAXBInclude incl : jaxbProc.getIncludes())
    {
      String nsURI = incl.getNamespace();
      String location = incl.getLocation();

      URL procURL;
      try
      {
        procURL = new URL(location);
      }
      catch (MalformedURLException ex)
      {
        ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
        procURL = ctxLoader.getResource(location);
      }
      if (procURL == null)
        throw new InvalidProcessException("Cannot find process include: " + location);

      ProcessManager pm = ProcessManager.locateProcessManager();
      DialectHandler dh = pm.getDialectHandler(nsURI);
      Process procIncl = dh.createProcess(procURL, true);

      // Debug the included process
      StringWriter strwr = new StringWriter();
      DialectHandler dhapi = pm.getDialectHandler(DialectHandler.DEFAULT_NAMESPACE_URI);
      dhapi.marshallProcess(procIncl, strwr);
      // System.out.println(strwr);

      procBuilder.addProcess(procIncl);
    }
  }

  private void adaptJAXBMessage(MessageBuilder msgBuilder, JAXBMessage jaxbMsg)
  {
    String fromRef = jaxbMsg.getFromRef();
    if (fromRef != null)
      msgBuilder.addFromRef(getObjectName(fromRef));
    
    String toRef = jaxbMsg.getToRef();
    if (toRef != null)
      msgBuilder.addToRef(getObjectName(toRef));
    
    for (JAXBProperty jaxbProp : jaxbMsg.getProperties())
    {
      boolean isCorrelation = jaxbProp.isCorrelation() != null ? jaxbProp.isCorrelation() : false;
      msgBuilder.addProperty(jaxbProp.getName(), jaxbProp.getValue(), isCorrelation);
    }
  }

  private ObjectName getObjectName(String objName)
  {
    return ObjectNameFactory.create(objName);
  }

  private void adaptJAXBEvent(ProcessBuilder procBuilder, JAXBProcess jaxbProc, JAXBEvent jaxb)
  {
    if (jaxb instanceof JAXBStartEvent)
    {
      JAXBStartEvent jaxbStart = (JAXBStartEvent)jaxb;
      EventBuilder eventBuilder = procBuilder.addStartEvent(jaxbStart.getName());
      addOutFlow(procBuilder, jaxbStart.getOutFlow());

      if (jaxbStart.getTrigger().size() > 1)
        throw new NotImplementedException("JBPM-1660", "StartTrigger Multiple");

      for (JAXBEventDetail jaxbTrigger : jaxbStart.getTrigger())
      {
        if (jaxbTrigger instanceof JAXBMessageEventDetail)
        {
          throw new NotImplementedException("JBPM-1657", "StartTrigger Message");
        }
        if (jaxbTrigger instanceof JAXBTimerEventDetail)
        {
          throw new NotImplementedException("JBPM-1658", "StartTrigger Timer");
        }
        if (jaxbTrigger instanceof JAXBConditionalEventDetail)
        {
          throw new NotImplementedException("JBPM-1659", "StartTrigger Conditional");
        }
        if (jaxbTrigger instanceof JAXBSignalEventDetail)
        {
          JAXBSignalEventDetail jaxbSignalTrigger = (JAXBSignalEventDetail)jaxbTrigger;
          JAXBSignal jaxbSignal = jaxbSignalTrigger.getSignal();
          eventBuilder.addEventDetail(EventDetail.EventDetailType.Signal).addSignalRef(jaxbSignal.getType(),
              jaxbSignal.getMessage());
        }
        else
        {
          throw new IllegalStateException("Unsupported start trigger: " + jaxbTrigger);
        }
      }
    }
    else if (jaxb instanceof JAXBIntermediateEvent)
    {
      throw new NotImplementedException("JBPM-1661", "IntermediateEvent");
    }
    else if (jaxb instanceof JAXBEndEvent)
    {
      JAXBEndEvent jaxbEnd = (JAXBEndEvent)jaxb;
      EventBuilder eventBuilder = procBuilder.addEndEvent(jaxbEnd.getName());

      if (jaxbEnd.getResult().size() > 1)
        throw new NotImplementedException("JBPM-1683", "EndEvent Multiple Result");

      for (JAXBEventDetail jaxbResult : jaxbEnd.getResult())
      {
        if (jaxbResult instanceof JAXBMessageEventDetail)
        {
          JAXBMessageEventDetail jaxbMessageResult = (JAXBMessageEventDetail)jaxbResult;
          eventBuilder.addEventDetail(EventDetail.EventDetailType.Message);
          JAXBMessageRef jaxbMsg = jaxbMessageResult.getMessageRef();
          eventBuilder.addMessageRef(jaxbMsg.getNameRef());
        }
        else if (jaxbResult instanceof JAXBErrorEventDetail)
        {
          throw new NotImplementedException("JBPM-1677", "EndEvent Error Result");
        }
        else if (jaxbResult instanceof JAXBCancelEventDetail)
        {
          throw new NotImplementedException("JBPM-1678", "EndEvent Cancel Result");
        }
        else if (jaxbResult instanceof JAXBCompensationEventDetail)
        {
          throw new NotImplementedException("JBPM-1679", "EndEvent Compensation Result");
        }
        else if (jaxbResult instanceof JAXBSignalEventDetail)
        {
          throw new NotImplementedException("JBPM-1651", "EndEvent Signal Result");
        }
        else if (jaxbResult instanceof JAXBTerminateEventDetail)
        {
          throw new NotImplementedException("JBPM-1680", "EndEvent Terminate Result");
        }
        else
        {
          throw new IllegalStateException("Unsupported end event result type: " + jaxbResult);
        }
      }
    }
    else
    {
      throw new IllegalStateException("Unsupported Event: " + jaxb);
    }
    procBuilder.addExecutionHandler(loadHandler(jaxb.getExecutionHandler()));
    procBuilder.addFlowHandler(loadHandler(jaxb.getFlowHandler()));
    procBuilder.addSignalHandler(loadHandler(jaxb.getSignalHandler()));
  }

  private void adaptTask(ProcessBuilder procBuilder, JAXBProcess jaxbProc, JAXBTask jaxbTask)
  {
    TaskBuilder taskBuilder = procBuilder.addTask(jaxbTask.getName(), jaxbTask.getTaskType());
    addOutFlow(taskBuilder, jaxbTask.getOutFlow());
    
    for (JAXBInputSet jaxbSet : jaxbTask.getInputSets())
    {
      taskBuilder.addInputSet();
      for (JAXBProperty jaxbProp : jaxbSet.getProperties())
        taskBuilder.addPropertyInput(jaxbProp.getName());
    }
    
    for (JAXBOutputSet jaxbSet : jaxbTask.getOutputSets())
    {
      taskBuilder.addOutputSet();
      for (JAXBProperty jaxbProp : jaxbSet.getProperties())
        taskBuilder.addPropertyOutput(jaxbProp.getName(), jaxbProp.getValue());
    }
    
    for (JAXBProperty jaxbProp : jaxbTask.getProperties())
    {
      taskBuilder.addProperty(jaxbProp.getName(), jaxbProp.getValue());
    }
    
    JAXBMessageRef jaxbMsgRef = jaxbTask.getMessageRef();
    if (jaxbMsgRef != null)
    {
      String msgName = jaxbMsgRef.getNameRef();
      taskBuilder.addMessageRef(msgName);
    }

    procBuilder.addExecutionHandler(loadHandler(jaxbTask.getExecutionHandler()));
    procBuilder.addFlowHandler(loadHandler(jaxbTask.getFlowHandler()));
    procBuilder.addSignalHandler(loadHandler(jaxbTask.getSignalHandler()));
  }

  private void adaptJAXBGateway(ProcessBuilder procBuilder, JAXBProcess jaxbProc, JAXBGateway jaxb)
  {
    GatewayBuilder gwBuilder;
    if (jaxb instanceof JAXBExclusiveGateway)
    {
      gwBuilder = procBuilder.addGateway(jaxb.getName(), Gateway.GatewayType.Exclusive);
    }
    else if (jaxb instanceof JAXBInclusiveGateway)
    {
      gwBuilder = procBuilder.addGateway(jaxb.getName(), Gateway.GatewayType.Inclusive);
    }
    else if (jaxb instanceof JAXBComplexGateway)
    {
      gwBuilder = procBuilder.addGateway(jaxb.getName(), Gateway.GatewayType.Complex);
    }
    else if (jaxb instanceof JAXBParallelGateway)
    {
      gwBuilder = procBuilder.addGateway(jaxb.getName(), Gateway.GatewayType.Parallel);
    }
    else
    {
      throw new IllegalStateException("Unsupported gateway: " + jaxb);
    }
    for (JAXBFlow jaxbFlow : jaxb.getOutFlows())
    {
      if (jaxbFlow instanceof JAXBSequenceFlow)
      {
        JAXBSequenceFlow jaxbSeq = (JAXBSequenceFlow)jaxbFlow;
        if (jaxbSeq.getConditionType() == ConditionType.Expression)
        {
          JAXBExpression jaxbExpr = jaxbSeq.getCondition();
          gwBuilder.addConditionalGate(jaxbFlow.getTargetName(), jaxbExpr.getLang(), jaxbExpr.getBody());
        }
        else if (jaxbSeq.getConditionType() == ConditionType.Default)
        {
          gwBuilder.addDefaultGate(jaxbFlow.getTargetName());
        }
        else
        {
          gwBuilder.addGate(jaxbFlow.getTargetName());
        }
      }
      else if (jaxbFlow instanceof JAXBMessageFlow)
      {
        throw new NotImplementedException("JBPM-1382", "Message Flow");
      }
      else
      {
        throw new IllegalStateException("Unsupported connectiong object: " + jaxbFlow);
      }
    }
    procBuilder.addExecutionHandler(loadHandler(jaxb.getExecutionHandler()));
    procBuilder.addFlowHandler(loadHandler(jaxb.getFlowHandler()));
    procBuilder.addSignalHandler(loadHandler(jaxb.getSignalHandler()));
  }

  private void addOutFlow(ProcessBuilder procBuilder, JAXBFlow jaxbFlow)
  {
    if (jaxbFlow != null)
    {
      if (jaxbFlow instanceof JAXBSequenceFlow)
      {
        procBuilder.addSequenceFlow(jaxbFlow.getTargetName());
      }
      else if (jaxbFlow instanceof JAXBMessageFlow)
      {
        throw new NotImplementedException("JBPM-1382", "Message Flow");
      }
      else
      {
        throw new IllegalStateException("Unsupported connectiong object: " + jaxbFlow);
      }
    }
  }

  @SuppressWarnings("unchecked")
  private Class<Handler> loadHandler(JAXBHandler jaxbHandler)
  {
    Class<Handler> handlerClass = null;
    if (jaxbHandler != null)
    {
      String className = jaxbHandler.getClassName();
      try
      {
        ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
        handlerClass = (Class<Handler>)ctxLoader.loadClass(className);
      }
      catch (ClassNotFoundException e)
      {
        throw new IllegalStateException("Cannot load handler class: " + className);
      }
    }
    return handlerClass;
  }
}
