/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.dialect.jpdl32;

// $Id: ProcessDefinitionAdapter.java 1930 2008-08-19 11:04:43Z thomas.diesler@jboss.com $

import java.util.List;

import org.jboss.bpm.InvalidProcessException;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32Action;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32EndState;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32Event;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32Fork;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32ProcessDefinition;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32StartState;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32State;
import org.jboss.bpm.dialect.jpdl32.model.JPDL32Transition;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Task;
import org.jboss.bpm.runtime.ExecutionHandler;

/**
 * Adapt the jpdl-3.2 ProcessDefinition to the API Process
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public class ProcessDefinitionAdapter
{
  public Process adaptProcessDefinition(JPDL32ProcessDefinition jpdlProcDef)
  {
    ProcessBuilder builder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    builder.addProcess(jpdlProcDef.getName());
    
    for (Object jpdlObj : jpdlProcDef.getDescriptionOrSwimlaneOrStartState())
    {
      if (jpdlObj instanceof JPDL32StartState)
      {
        adaptStartState(builder, (JPDL32StartState)jpdlObj);
      }
      else if (jpdlObj instanceof JPDL32State)
      {
        adaptState(builder, (JPDL32State)jpdlObj);
      }
      else if (jpdlObj instanceof JPDL32Fork)
      {
        adaptFork(builder, (JPDL32Fork)jpdlObj);
      }
      else if (jpdlObj instanceof JPDL32EndState)
      {
        adaptEndState(builder, (JPDL32EndState)jpdlObj);
      }
      else
      {
        throw new InvalidProcessException("Unsupported type: " + jpdlObj);
      }
    }

    Process proc = builder.getProcess();
    return proc;
  }

  private void adaptStartState(ProcessBuilder builder, JPDL32StartState jpdlStart)
  {
    builder.addStartEvent("Start");
    adaptTransitions(builder, jpdlStart.getDescriptionOrTaskOrTransition());
  }

  private void adaptEndState(ProcessBuilder builder, JPDL32EndState jpdlEnd)
  {
    builder.addEndEvent(jpdlEnd.getName());
  }

  private void adaptState(ProcessBuilder builder, JPDL32State jpdlState)
  {
    builder.addTask(jpdlState.getName(), Task.TaskType.None);
    adaptTransitions(builder, jpdlState.getDescriptionOrEventOrExceptionHandler());
    
    for (Object jpdlObj : jpdlState.getDescriptionOrEventOrExceptionHandler())
    {
      if (jpdlObj instanceof JPDL32Event)
      {
        JPDL32Event jpdlEvent = (JPDL32Event)jpdlObj;
        if (jpdlEvent.getType().equals("node-enter"))
        {
          for (Object jpdlAux : jpdlEvent.getActionOrScriptOrCreateTimer())
          {
            if (jpdlAux instanceof JPDL32Action)
            {
              JPDL32Action jpdlAction = (JPDL32Action)jpdlAux;
              Class<?> handler = loadHandler(jpdlAction.getClazz());
              if (ExecutionHandler.class.isAssignableFrom(handler) == false)
                throw new InvalidProcessException("Node action is not of type ExecutionHandler");

              builder.addExecutionHandler(handler);
            }
          }
        }
      }
    }
  }

  private void adaptFork(ProcessBuilder builder, JPDL32Fork jpdlObj)
  {
    builder.addGateway(jpdlObj.getName(), Gateway.GatewayType.Parallel);
    adaptTransitions(builder, jpdlObj.getScriptOrDescriptionOrEvent());
  }

  private void adaptTransitions(ProcessBuilder builder, List<Object> list)
  {
    for (Object subEl : list)
    {
      if (subEl instanceof JPDL32Transition)
      {
        JPDL32Transition trans = (JPDL32Transition)subEl;
        String targetName = trans.getTo();
        builder.addSequenceFlow(targetName);
      }
    }
  }

  private Class<?> loadHandler(String className)
  {
    try
    {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      return ctxLoader.loadClass(className);
    }
    catch (ClassNotFoundException e)
    {
      throw new IllegalStateException("Cannot load handler class: " + className);
    }
  }
}
