package org.mapstruct.ap.internal.gem;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.AnnotationValue;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.AbstractAnnotationValueVisitor8;
import javax.lang.model.util.ElementFilter;
import org.mapstruct.tools.gem.Gem;
import org.mapstruct.tools.gem.GemValue;


public class DeprecatedGem implements Gem {

    private final GemValue<String> since;
    private final GemValue<Boolean> forRemoval;
    private final boolean isValid;
    private final AnnotationMirror mirror;

    private DeprecatedGem( BuilderImpl builder ) {
        this.since = builder.since;
        this.forRemoval = builder.forRemoval;
        isValid = ( this.since != null ? this.since.isValid() : false )
               && ( this.forRemoval != null ? this.forRemoval.isValid() : false );
        mirror = builder.mirror;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link DeprecatedGem#since}
    */
    public GemValue<String> since( ) {
        return since;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link DeprecatedGem#forRemoval}
    */
    public GemValue<Boolean> forRemoval( ) {
        return forRemoval;
    }

    @Override
    public AnnotationMirror mirror( ) {
        return mirror;
    }

    @Override
    public boolean isValid( ) {
        return isValid;
    }

    public static DeprecatedGem  instanceOn(Element element) {
        return build( element, new BuilderImpl() );
    }

    public static DeprecatedGem instanceOn(AnnotationMirror mirror ) {
        return build( mirror, new BuilderImpl() );
    }

    public static  <T> T  build(Element element, Builder<T> builder) {
        AnnotationMirror mirror = element.getAnnotationMirrors().stream()
            .filter( a ->  "java.lang.Deprecated".contentEquals( ( ( TypeElement )a.getAnnotationType().asElement() ).getQualifiedName() ) )
            .findAny()
            .orElse( null );
        return build( mirror, builder );
    }

    public static <T> T build(AnnotationMirror mirror, Builder<T> builder ) {

        // return fast
        if ( mirror == null || builder == null ) {
            return null;
        }

        // fetch defaults from all defined values in the annotation type
        List<ExecutableElement> enclosed = ElementFilter.methodsIn( mirror.getAnnotationType().asElement().getEnclosedElements() );
        Map<String, AnnotationValue> defaultValues = new HashMap<>( enclosed.size() );
        enclosed.forEach( e -> defaultValues.put( e.getSimpleName().toString(), e.getDefaultValue() ) );

        // fetch all explicitely set annotation values in the annotation instance
        Map<String, AnnotationValue> values = new HashMap<>( enclosed.size() );
        mirror.getElementValues().entrySet().forEach( e -> values.put( e.getKey().getSimpleName().toString(), e.getValue() ) );

        // iterate and populate builder
        for ( String methodName : defaultValues.keySet() ) {

            if ( "since".equals( methodName ) ) {
                builder.setSince( GemValue.create( values.get( methodName ), defaultValues.get( methodName ), String.class ) );
            }
            else if ( "forRemoval".equals( methodName ) ) {
                builder.setForremoval( GemValue.create( values.get( methodName ), defaultValues.get( methodName ), Boolean.class ) );
            }
        }
        builder.setMirror( mirror );
        return builder.build();
    }

    /**
     * A builder that can be implemented by the user to define custom logic e.g. in the
     * build method, prior to creating the annotation gem.
     */
    public interface Builder<T> {

       /**
        * Sets the {@link GemValue} for {@link DeprecatedGem#since}
        *
        * @return the {@link Builder} for this gem, representing {@link DeprecatedGem}
        */
        Builder setSince(GemValue<String> methodName );

       /**
        * Sets the {@link GemValue} for {@link DeprecatedGem#forRemoval}
        *
        * @return the {@link Builder} for this gem, representing {@link DeprecatedGem}
        */
        Builder setForremoval(GemValue<Boolean> methodName );

        /**
         * Sets the annotation mirror
         *
         * @param mirror the mirror which this gem represents
         *
         * @return the {@link Builder} for this gem, representing {@link DeprecatedGem}
         */
          Builder setMirror( AnnotationMirror mirror );

        /**
         * The build method can be overriden in a custom custom implementation, which allows
         * the user to define his own custom validation on the annotation.
         *
         * @return the representation of the annotation
         */
        T build();
    }

    private static class BuilderImpl implements Builder<DeprecatedGem> {

        private GemValue<String> since;
        private GemValue<Boolean> forRemoval;
        private AnnotationMirror mirror;

        public Builder setSince(GemValue<String> since ) {
            this.since = since;
            return this;
        }

        public Builder setForremoval(GemValue<Boolean> forRemoval ) {
            this.forRemoval = forRemoval;
            return this;
        }

        public Builder  setMirror( AnnotationMirror mirror ) {
            this.mirror = mirror;
            return this;
        }

        public DeprecatedGem build() {
            return new DeprecatedGem( this );
        }
    }

}
