/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.bkd;

import java.io.IOException;
import java.util.Arrays;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.StringHelper;

public class BKDReader
implements Accountable {
    private final byte[] splitPackedValues;
    final long[] leafBlockFPs;
    private final int leafNodeOffset;
    final int numDims;
    final int bytesPerDim;
    final IndexInput in;
    final int maxPointsInLeafNode;
    final byte[] minPackedValue;
    final byte[] maxPackedValue;
    final long pointCount;
    final int docCount;
    protected final int packedBytesLength;

    public BKDReader(IndexInput in) throws IOException {
        CodecUtil.checkHeader(in, "BKD", 0, 0);
        this.numDims = in.readVInt();
        this.maxPointsInLeafNode = in.readVInt();
        this.bytesPerDim = in.readVInt();
        this.packedBytesLength = this.numDims * this.bytesPerDim;
        int numLeaves = in.readVInt();
        assert (numLeaves > 0);
        this.leafNodeOffset = numLeaves;
        this.minPackedValue = new byte[this.packedBytesLength];
        this.maxPackedValue = new byte[this.packedBytesLength];
        in.readBytes(this.minPackedValue, 0, this.packedBytesLength);
        in.readBytes(this.maxPackedValue, 0, this.packedBytesLength);
        this.pointCount = in.readVLong();
        this.docCount = in.readVInt();
        this.splitPackedValues = new byte[(1 + this.bytesPerDim) * numLeaves];
        in.readBytes(this.splitPackedValues, 0, this.splitPackedValues.length);
        long[] leafBlockFPs = new long[numLeaves];
        long lastFP = 0L;
        for (int i = 0; i < numLeaves; ++i) {
            long delta = in.readVLong();
            leafBlockFPs[i] = lastFP + delta;
            lastFP += delta;
        }
        if (this.numDims == 1 && numLeaves > 1) {
            int levelCount = 2;
            while (true) {
                if (numLeaves >= levelCount && numLeaves <= 2 * levelCount) {
                    int lastLevel = 2 * (numLeaves - levelCount);
                    assert (lastLevel >= 0);
                    if (lastLevel == 0) break;
                    long[] newLeafBlockFPs = new long[numLeaves];
                    System.arraycopy(leafBlockFPs, lastLevel, newLeafBlockFPs, 0, leafBlockFPs.length - lastLevel);
                    System.arraycopy(leafBlockFPs, 0, newLeafBlockFPs, leafBlockFPs.length - lastLevel, lastLevel);
                    leafBlockFPs = newLeafBlockFPs;
                    break;
                }
                levelCount *= 2;
            }
        }
        this.leafBlockFPs = leafBlockFPs;
        this.in = in;
    }

    protected BKDReader(IndexInput in, int numDims, int maxPointsInLeafNode, int bytesPerDim, long[] leafBlockFPs, byte[] splitPackedValues, byte[] minPackedValue, byte[] maxPackedValue, long pointCount, int docCount) throws IOException {
        this.in = in;
        this.numDims = numDims;
        this.maxPointsInLeafNode = maxPointsInLeafNode;
        this.bytesPerDim = bytesPerDim;
        this.packedBytesLength = numDims * bytesPerDim;
        this.leafNodeOffset = leafBlockFPs.length;
        this.leafBlockFPs = leafBlockFPs;
        this.splitPackedValues = splitPackedValues;
        this.minPackedValue = minPackedValue;
        this.maxPackedValue = maxPackedValue;
        this.pointCount = pointCount;
        this.docCount = docCount;
        assert (minPackedValue.length == this.packedBytesLength);
        assert (maxPackedValue.length == this.packedBytesLength);
    }

    public void verify(int maxDoc) throws IOException {
        byte[] rootMinPacked = new byte[this.packedBytesLength];
        byte[] rootMaxPacked = new byte[this.packedBytesLength];
        Arrays.fill(rootMaxPacked, (byte)-1);
        IntersectState state = new IntersectState(this.in.clone(), this.numDims, this.packedBytesLength, this.maxPointsInLeafNode, new VerifyVisitor(this.numDims, this.bytesPerDim, maxDoc));
        this.verify(state, 1, rootMinPacked, rootMaxPacked);
    }

    private void verify(IntersectState state, int nodeID, byte[] cellMinPacked, byte[] cellMaxPacked) throws IOException {
        if (nodeID >= this.leafNodeOffset) {
            int leafID = nodeID - this.leafNodeOffset;
            if (leafID < this.leafBlockFPs.length) {
                VerifyVisitor visitor = (VerifyVisitor)state.visitor;
                visitor.cellMinPacked = cellMinPacked;
                visitor.cellMaxPacked = cellMaxPacked;
                int count = this.readDocIDs(state.in, this.leafBlockFPs[leafID], state.scratchDocIDs);
                this.visitDocValues(state.commonPrefixLengths, state.scratchPackedValue, state.in, state.scratchDocIDs, count, state.visitor);
            }
        } else {
            int address = nodeID * (this.bytesPerDim + 1);
            int splitDim = this.splitPackedValues[address] & 0xFF;
            assert (splitDim < this.numDims);
            byte[] splitPackedValue = new byte[this.packedBytesLength];
            System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address + 1, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.verify(state, 2 * nodeID, cellMinPacked, splitPackedValue);
            System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address + 1, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.verify(state, 2 * nodeID + 1, splitPackedValue, cellMaxPacked);
        }
    }

    public void intersect(PointValues.IntersectVisitor visitor) throws IOException {
        IntersectState state = new IntersectState(this.in.clone(), this.numDims, this.packedBytesLength, this.maxPointsInLeafNode, visitor);
        this.intersect(state, 1, this.minPackedValue, this.maxPackedValue);
    }

    private void addAll(IntersectState state, int nodeID) throws IOException {
        if (nodeID >= this.leafNodeOffset) {
            this.visitDocIDs(state.in, this.leafBlockFPs[nodeID - this.leafNodeOffset], state.visitor);
        } else {
            this.addAll(state, 2 * nodeID);
            this.addAll(state, 2 * nodeID + 1);
        }
    }

    protected void visitDocIDs(IndexInput in, long blockFP, PointValues.IntersectVisitor visitor) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        visitor.grow(count);
        for (int i = 0; i < count; ++i) {
            visitor.visit(in.readInt());
        }
    }

    protected int readDocIDs(IndexInput in, long blockFP, int[] docIDs) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        for (int i = 0; i < count; ++i) {
            docIDs[i] = in.readInt();
        }
        return count;
    }

    protected void visitDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, int[] docIDs, int count, PointValues.IntersectVisitor visitor) throws IOException {
        visitor.grow(count);
        for (int dim = 0; dim < this.numDims; ++dim) {
            int prefix;
            commonPrefixLengths[dim] = prefix = in.readVInt();
            if (prefix <= 0) continue;
            in.readBytes(scratchPackedValue, dim * this.bytesPerDim, prefix);
        }
        for (int i = 0; i < count; ++i) {
            for (int dim = 0; dim < this.numDims; ++dim) {
                int prefix = commonPrefixLengths[dim];
                in.readBytes(scratchPackedValue, dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
            }
            visitor.visit(docIDs[i], scratchPackedValue);
        }
    }

    private void intersect(IntersectState state, int nodeID, byte[] cellMinPacked, byte[] cellMaxPacked) throws IOException {
        PointValues.Relation r = state.visitor.compare(cellMinPacked, cellMaxPacked);
        if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
            return;
        }
        if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
            this.addAll(state, nodeID);
            return;
        }
        if (nodeID >= this.leafNodeOffset) {
            int leafID = nodeID - this.leafNodeOffset;
            if (leafID < this.leafBlockFPs.length) {
                int count = this.readDocIDs(state.in, this.leafBlockFPs[leafID], state.scratchDocIDs);
                this.visitDocValues(state.commonPrefixLengths, state.scratchPackedValue, state.in, state.scratchDocIDs, count, state.visitor);
            }
        } else {
            int address = nodeID * (this.bytesPerDim + 1);
            int splitDim = this.splitPackedValues[address] & 0xFF;
            assert (splitDim < this.numDims);
            byte[] splitPackedValue = new byte[this.packedBytesLength];
            System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address + 1, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.intersect(state, 2 * nodeID, cellMinPacked, splitPackedValue);
            System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.splitPackedValues, address + 1, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            this.intersect(state, 2 * nodeID + 1, splitPackedValue, cellMaxPacked);
        }
    }

    @Override
    public long ramBytesUsed() {
        return this.splitPackedValues.length + this.leafBlockFPs.length * 8;
    }

    public byte[] getMinPackedValue() {
        return (byte[])this.minPackedValue.clone();
    }

    public byte[] getMaxPackedValue() {
        return (byte[])this.maxPackedValue.clone();
    }

    public int getNumDimensions() {
        return this.numDims;
    }

    public int getBytesPerDimension() {
        return this.bytesPerDim;
    }

    public long getPointCount() {
        return this.pointCount;
    }

    public int getDocCount() {
        return this.docCount;
    }

    static final class IntersectState {
        final IndexInput in;
        final int[] scratchDocIDs;
        final byte[] scratchPackedValue;
        final int[] commonPrefixLengths;
        final PointValues.IntersectVisitor visitor;

        public IntersectState(IndexInput in, int numDims, int packedBytesLength, int maxPointsInLeafNode, PointValues.IntersectVisitor visitor) {
            this.in = in;
            this.visitor = visitor;
            this.commonPrefixLengths = new int[numDims];
            this.scratchDocIDs = new int[maxPointsInLeafNode];
            this.scratchPackedValue = new byte[packedBytesLength];
        }
    }

    private static class VerifyVisitor
    implements PointValues.IntersectVisitor {
        byte[] cellMinPacked;
        byte[] cellMaxPacked;
        byte[] lastPackedValue;
        final int numDims;
        final int bytesPerDim;
        final int maxDoc;

        public VerifyVisitor(int numDims, int bytesPerDim, int maxDoc) {
            this.numDims = numDims;
            this.bytesPerDim = bytesPerDim;
            this.maxDoc = maxDoc;
        }

        @Override
        public void visit(int docID) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void visit(int docID, byte[] packedValue) {
            if (docID < 0 || docID >= this.maxDoc) {
                throw new RuntimeException("docID=" + docID + " is out of bounds of 0.." + this.maxDoc);
            }
            for (int dim = 0; dim < this.numDims; ++dim) {
                if (StringHelper.compare(this.bytesPerDim, this.cellMinPacked, dim * this.bytesPerDim, packedValue, dim * this.bytesPerDim) > 0) {
                    throw new RuntimeException("value=" + new BytesRef(packedValue, dim * this.bytesPerDim, this.bytesPerDim) + " for docID=" + docID + " dim=" + dim + " is less than this leaf block's minimum=" + new BytesRef(this.cellMinPacked, dim * this.bytesPerDim, this.bytesPerDim));
                }
                if (StringHelper.compare(this.bytesPerDim, this.cellMaxPacked, dim * this.bytesPerDim, packedValue, dim * this.bytesPerDim) >= 0) continue;
                throw new RuntimeException("value=" + new BytesRef(packedValue, dim * this.bytesPerDim, this.bytesPerDim) + " for docID=" + docID + " dim=" + dim + " is greater than this leaf block's maximum=" + new BytesRef(this.cellMaxPacked, dim * this.bytesPerDim, this.bytesPerDim));
            }
            if (this.numDims == 1) {
                if (this.lastPackedValue == null) {
                    this.lastPackedValue = Arrays.copyOf(packedValue, packedValue.length);
                } else {
                    if (StringHelper.compare(this.bytesPerDim, this.lastPackedValue, 0, packedValue, 0) > 0) {
                        throw new RuntimeException("value=" + new BytesRef(packedValue) + " for docID=" + docID + " dim=0" + " sorts before last value=" + new BytesRef(this.lastPackedValue));
                    }
                    System.arraycopy(packedValue, 0, this.lastPackedValue, 0, this.bytesPerDim);
                }
            }
        }

        @Override
        public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
            throw new UnsupportedOperationException();
        }
    }
}

