/*
 * Decompiled with CFR 0.152.
 */
package org.modeshape.jcr.locking;

import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import org.modeshape.common.annotation.ThreadSafe;
import org.modeshape.common.logging.Logger;
import org.modeshape.common.util.CheckArg;
import org.modeshape.jcr.locking.LockingService;

@ThreadSafe
public abstract class AbstractLockingService<T extends Lock>
implements LockingService {
    protected final Logger logger = Logger.getLogger(this.getClass());
    private final ConcurrentHashMap<String, T> locksByName = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, AtomicInteger> locksByWaiters = new ConcurrentHashMap();
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final long lockTimeoutMillis;

    protected AbstractLockingService() {
        this(0L);
    }

    protected AbstractLockingService(long lockTimeoutMillis) {
        CheckArg.isNonNegative((long)lockTimeoutMillis, (String)"lockTimeoutMillis");
        this.lockTimeoutMillis = lockTimeoutMillis;
        this.running.compareAndSet(false, true);
    }

    @Override
    public boolean tryLock(String ... names) throws InterruptedException {
        return this.tryLock(this.lockTimeoutMillis, TimeUnit.MILLISECONDS, names);
    }

    @Override
    public boolean tryLock(long time, TimeUnit unit, String ... names) throws InterruptedException {
        if (!this.running.get()) {
            throw new IllegalStateException("Service has been shut down");
        }
        LinkedHashSet<String> successfullyLocked = new LinkedHashSet<String>();
        for (String name : names) {
            this.logger.debug("attempting to lock {0}", new Object[]{name});
            this.locksByWaiters.computeIfAbsent(name, lockName -> new AtomicInteger(0)).incrementAndGet();
            boolean success = false;
            Lock lock = null;
            try {
                lock = this.locksByName.computeIfAbsent(name, this::createLock);
                success = this.doLock(lock, name, time, unit);
            }
            catch (Exception e) {
                this.logger.debug((Throwable)e, "unexpected exception while attempting to lock '{0}'", new Object[]{name});
            }
            if (!success) {
                this.locksByWaiters.computeIfPresent(name, (lockName, atomicInteger) -> {
                    if (atomicInteger.decrementAndGet() == 0) {
                        this.locksByName.computeIfPresent(name, (internalLockName, internalLock) -> null);
                        return null;
                    }
                    return atomicInteger;
                });
                if (!successfullyLocked.isEmpty()) {
                    this.logger.debug("Unable to acquire lock on {0}. Reverting back the already obtained locks: {1}", new Object[]{name, successfullyLocked});
                    this.unlock(successfullyLocked.toArray(new String[successfullyLocked.size()]));
                }
                return false;
            }
            this.logger.debug("{0} locked successfully (ref {1})", new Object[]{name, lock});
            successfullyLocked.add(name);
        }
        return true;
    }

    @Override
    public boolean unlock(String ... names) {
        if (!this.running.get()) {
            throw new IllegalStateException("Service has been shut down");
        }
        return Arrays.stream(names).map(this::unlock).allMatch(Boolean::booleanValue);
    }

    protected boolean unlock(String name) {
        this.logger.debug("attempting to unlock {0}", new Object[]{name});
        AtomicBoolean unlocked = new AtomicBoolean(false);
        this.locksByName.computeIfPresent(name, (key, lock) -> {
            this.validateLock(lock);
            AtomicInteger waiters = this.locksByWaiters.computeIfPresent(name, (lockName, atomicInteger) -> {
                if (this.releaseLock(lock)) {
                    this.logger.debug("{0} unlocked (ref {1})...", new Object[]{name, lock});
                    unlocked.compareAndSet(false, true);
                    if (atomicInteger.decrementAndGet() > 0) {
                        return atomicInteger;
                    }
                    return null;
                }
                this.logger.debug("{0} failed to unlock (ref {1})...", new Object[]{name, lock});
                return atomicInteger;
            });
            if (waiters != null) {
                this.logger.debug("lock '{0}' is not currently locked but will be", new Object[]{name});
                return lock;
            }
            this.logger.debug("lock '{0}' not used anymore; removing it from map", new Object[]{name});
            return null;
        });
        return unlocked.get();
    }

    @Override
    public synchronized boolean shutdown() {
        if (!this.running.get()) {
            return false;
        }
        this.logger.debug("Shutting down locking service...", new Object[0]);
        this.doShutdown();
        this.locksByName.clear();
        this.running.compareAndSet(true, false);
        return true;
    }

    protected void doShutdown() {
        this.locksByName.forEach((key, lock) -> {
            if (this.releaseLock(lock)) {
                this.logger.debug("{0} unlocked successfully ", new Object[]{key});
            } else {
                this.logger.debug("{0} cannot be released...", new Object[]{key});
            }
        });
    }

    protected boolean doLock(T lock, String name, long time, TimeUnit timeUnit) throws InterruptedException {
        return time > 0L ? lock.tryLock(time, timeUnit) : lock.tryLock();
    }

    protected long getLockTimeoutMillis() {
        return this.lockTimeoutMillis;
    }

    protected abstract T createLock(String var1);

    protected abstract void validateLock(T var1);

    protected abstract boolean releaseLock(T var1);
}

