/*
 * Decompiled with CFR 0.152.
 */
package org.modeshape.persistence.relational;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.modeshape.common.database.DatabaseType;
import org.modeshape.common.i18n.I18nResource;
import org.modeshape.common.logging.Logger;
import org.modeshape.common.util.StringUtil;
import org.modeshape.persistence.relational.DataSourceManager;
import org.modeshape.persistence.relational.DefaultStatements;
import org.modeshape.persistence.relational.OracleStatements;
import org.modeshape.persistence.relational.RelationalDbConfig;
import org.modeshape.persistence.relational.RelationalProviderException;
import org.modeshape.persistence.relational.RelationalProviderI18n;
import org.modeshape.persistence.relational.SQLServerStatements;
import org.modeshape.persistence.relational.Statements;
import org.modeshape.persistence.relational.TransactionalCaches;
import org.modeshape.persistence.relational.TransactionsHolder;
import org.modeshape.schematic.SchematicDb;
import org.modeshape.schematic.SchematicEntry;
import org.modeshape.schematic.document.Document;
import org.modeshape.schematic.document.EditableDocument;

public class RelationalDb
implements SchematicDb {
    private static final Logger LOGGER = Logger.getLogger(RelationalDb.class);
    private final ConcurrentMap<String, Connection> connectionsByTxId = new ConcurrentHashMap<String, Connection>();
    private final DataSourceManager dsManager;
    private final RelationalDbConfig config;
    private final Statements statements;
    private final TransactionalCaches transactionalCaches;

    protected RelationalDb(Document configDoc) {
        configDoc = Objects.requireNonNull(configDoc, "Configuration document cannot be null");
        this.config = new RelationalDbConfig(configDoc);
        this.dsManager = new DataSourceManager(this.config);
        DatabaseType dbType = this.dsManager.dbType();
        this.statements = this.createStatements(dbType);
        this.transactionalCaches = new TransactionalCaches();
    }

    private Statements createStatements(DatabaseType dbType) {
        Map<String, String> statementsFile = this.loadStatementsResource();
        switch (dbType.name()) {
            case ORACLE: {
                return new OracleStatements(this.config, statementsFile);
            }
            case SQLSERVER: {
                return new SQLServerStatements(this.config, statementsFile);
            }
        }
        return new DefaultStatements(this.config, statementsFile);
    }

    public String id() {
        return this.config.name();
    }

    public void start() {
        if (this.config.createOnStart()) {
            this.runWithConnection(this.statements::createTable, false);
        }
    }

    public void stop() {
        TransactionsHolder.clearActiveTransaction();
        this.cleanupConnections();
        if (this.config.dropOnExit()) {
            this.runWithConnection(this.statements::dropTable, false);
        }
        this.dsManager.close();
        this.transactionalCaches.stop();
    }

    private void cleanupConnections() {
        if (this.connectionsByTxId.isEmpty()) {
            return;
        }
        LOGGER.warn((I18nResource)RelationalProviderI18n.warnConnectionsNeedCleanup, new Object[]{this.connectionsByTxId.size()});
        Iterator iterator = this.connectionsByTxId.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = iterator.next();
            this.closeConnection((String)entry.getKey(), (Connection)entry.getValue());
            iterator.remove();
        }
    }

    private void closeConnection(String txId, Connection connection) {
        try {
            if (connection == null || connection.isClosed()) {
                return;
            }
        }
        catch (Throwable t) {
            LOGGER.debug(t, "Cannot determine DB connection status for transaction {0}", new Object[]{txId});
            return;
        }
        try {
            connection.close();
        }
        catch (Throwable t) {
            LOGGER.warn((I18nResource)RelationalProviderI18n.warnCannotCloseConnection, new Object[]{this.config.tableName(), txId, t.getMessage()});
            LOGGER.debug(t, "Cannot close connection", new Object[0]);
        }
    }

    public List<String> keys() {
        List persistedKeys = this.runWithConnection(this.statements::getAllIds, true);
        if (!TransactionsHolder.hasActiveTransaction()) {
            return persistedKeys;
        }
        persistedKeys.addAll(this.transactionalCaches.documentKeys());
        return persistedKeys.stream().filter(id -> !this.transactionalCaches.isRemoved((String)id)).collect(Collectors.toList());
    }

    public Document get(String key) {
        if (!TransactionsHolder.hasActiveTransaction()) {
            return this.runWithConnection(connection -> this.statements.getById(connection, key), true);
        }
        Document cachedDocument = this.transactionalCaches.search(key);
        if (cachedDocument != null) {
            this.logDebug("Getting {0} from cache; value {1}", key, cachedDocument);
            return cachedDocument != TransactionalCaches.REMOVED ? cachedDocument : null;
        }
        if (this.transactionalCaches.isNew(key)) {
            return null;
        }
        Document doc = this.runWithConnection(connection -> this.statements.getById(connection, key), false);
        if (doc != null) {
            this.transactionalCaches.putForReading(key, doc);
        } else {
            this.transactionalCaches.putNew(key);
        }
        return doc;
    }

    public List<SchematicEntry> load(Collection<String> keys) {
        List alreadyChangedInTransaction = Collections.emptyList();
        ArrayList alreadyChangedKeys = new ArrayList();
        if (TransactionsHolder.hasActiveTransaction()) {
            alreadyChangedInTransaction = keys.stream().map(this.transactionalCaches::getForWriting).filter(Objects::nonNull).map(SchematicEntry::fromDocument).collect(ArrayList::new, (list, schematicEntry) -> {
                alreadyChangedKeys.add(schematicEntry.id());
                if (TransactionalCaches.REMOVED != schematicEntry.source()) {
                    list.add(schematicEntry);
                }
            }, ArrayList::addAll);
        }
        keys.removeAll(alreadyChangedKeys);
        Function<Document, SchematicEntry> documentParser = document -> {
            SchematicEntry entry = SchematicEntry.fromDocument((Document)document);
            String id = entry.id();
            this.transactionalCaches.putForReading(id, (Document)document);
            return entry;
        };
        List results = this.runWithConnection(connection -> this.statements.load(connection, keys, documentParser), true);
        results.addAll(alreadyChangedInTransaction);
        this.transactionalCaches.putNew(keys);
        return results;
    }

    public boolean lockForWriting(List<String> locks) {
        if (locks.isEmpty()) {
            return false;
        }
        TransactionsHolder.requireActiveTransaction();
        return this.runWithConnection(connection -> this.statements.lockForWriting(connection, locks), true);
    }

    public void put(String key, SchematicEntry entry) {
        this.transactionalCaches.putForWriting(key, entry.source());
    }

    public EditableDocument editContent(String key, boolean createIfMissing) {
        Document entryDocument;
        SchematicEntry entry = this.getEntry(key);
        if (entry == null) {
            if (createIfMissing) {
                this.put(key, SchematicEntry.create((String)key));
            } else {
                return null;
            }
        }
        if ((entryDocument = this.transactionalCaches.getForWriting(key)) == null) {
            entryDocument = this.transactionalCaches.putForWriting(key, entry.source());
        }
        return SchematicEntry.content((Document)entryDocument).editable();
    }

    public SchematicEntry putIfAbsent(String key, Document content) {
        SchematicEntry existingEntry = this.getEntry(key);
        if (existingEntry != null) {
            return existingEntry;
        }
        this.put(key, SchematicEntry.create((String)key, (Document)content));
        return null;
    }

    public boolean remove(String key) {
        this.transactionalCaches.remove(key);
        return true;
    }

    public void removeAll() {
        this.runWithConnection(this.statements::removeAll, false);
    }

    public boolean containsKey(String key) {
        if (!TransactionsHolder.hasActiveTransaction()) {
            return this.runWithConnection(connection -> this.statements.exists(connection, key), true);
        }
        Document cachedDocument = this.transactionalCaches.search(key);
        if (cachedDocument != null) {
            return cachedDocument != TransactionalCaches.REMOVED;
        }
        if (this.transactionalCaches.isNew(key)) {
            return false;
        }
        boolean existsInDB = this.runWithConnection(connection -> this.statements.exists(connection, key), true);
        if (!existsInDB) {
            this.transactionalCaches.putNew(key);
        }
        return existsInDB;
    }

    public void txStarted(String id) {
        this.logDebug("New transaction '{0}' started by ModeShape...", id);
        String activeTx = TransactionsHolder.activeTransaction();
        if (activeTx != null && !activeTx.equals(id)) {
            LOGGER.warn((I18nResource)RelationalProviderI18n.threadAssociatedWithAnotherTransaction, new Object[]{Thread.currentThread().getName(), activeTx, id});
        }
        TransactionsHolder.setActiveTxId(id);
        this.connectionForActiveTx();
    }

    public void txCommitted(String id) {
        this.logDebug("Received committed notification for transaction '{0}'", id);
        try {
            Connection connection = (Connection)this.connectionsByTxId.get(id);
            this.persistContent(connection, id);
        }
        catch (SQLException e) {
            throw new RelationalProviderException(e);
        }
        finally {
            this.cleanupTransaction(id);
        }
    }

    private void cleanupTransaction(String id) {
        try {
            this.connectionsByTxId.computeIfPresent(id, (txId, connection) -> {
                this.closeConnection((String)txId, (Connection)connection);
                this.logDebug("Released DB connection for transaction '{0}'", id);
                return null;
            });
        }
        finally {
            this.transactionalCaches.clearCache(id);
            TransactionsHolder.clearActiveTransaction();
        }
    }

    private void persistContent(Connection tlConnection, String txId) throws SQLException {
        TransactionalCaches.TransactionalCache cache = this.transactionalCaches.cacheForTransaction(txId);
        if (cache == null) {
            tlConnection.commit();
            return;
        }
        ConcurrentMap<String, Document> writeCache = cache.writeCache();
        ConcurrentMap<String, Document> readCache = cache.readCache();
        this.logDebug("Committing the active connection for transaction {0} with the changes: {1}", txId, writeCache);
        Statements.BatchUpdate batchUpdate = this.statements.batchUpdate(tlConnection);
        HashMap<String, Document> toInsert = new HashMap<String, Document>();
        HashMap<String, Document> toUpdate = new HashMap<String, Document>();
        ArrayList<String> toRemove = new ArrayList<String>();
        writeCache.forEach((key, document) -> {
            if (TransactionalCaches.REMOVED == document) {
                toRemove.add((String)key);
            } else if (readCache.containsKey(key)) {
                toUpdate.put((String)key, (Document)document);
            } else {
                toInsert.put((String)key, (Document)document);
            }
        });
        try {
            batchUpdate.insert(toInsert);
            batchUpdate.update(toUpdate);
            batchUpdate.remove(toRemove);
        }
        catch (SQLException e) {
            throw new RelationalProviderException(e);
        }
        tlConnection.commit();
    }

    public void txRolledback(String id) {
        this.logDebug("Received rollback notification for transaction '{0}'", id);
        try {
            this.runWithConnection(this::rollback, false);
        }
        finally {
            this.cleanupTransaction(id);
        }
    }

    private Void rollback(Connection connection) throws SQLException {
        connection.rollback();
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected <R> R runWithConnection(SQLFunction<R> function, boolean readonly) {
        try {
            if (TransactionsHolder.hasActiveTransaction()) {
                Connection connection = this.connectionForActiveTx();
                return function.execute(connection);
            }
            try (Connection connection = this.newConnection(true, readonly);){
                R r = function.execute(connection);
                return r;
            }
        }
        catch (SQLException e) {
            throw new RelationalProviderException(e);
        }
    }

    protected Connection connectionForActiveTx() {
        String activeTxId = TransactionsHolder.requireActiveTransaction();
        Connection connection = (Connection)this.connectionsByTxId.get(activeTxId);
        if (connection != null) {
            return connection;
        }
        connection = this.dsManager.newConnection(false, false);
        this.connectionsByTxId.put(activeTxId, connection);
        this.logDebug("New DB connection allocated for tx '{0}'", activeTxId);
        return connection;
    }

    protected RelationalDbConfig config() {
        return this.config;
    }

    protected DataSourceManager dsManager() {
        return this.dsManager;
    }

    protected Connection newConnection(boolean autoCommit, boolean readonly) {
        return this.dsManager.newConnection(autoCommit, readonly);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Map<String, String> loadStatementsResource() {
        try (InputStream fileStream = this.loadStatementsFile(this.dsManager.dbType());){
            Properties statements = new Properties();
            statements.load(fileStream);
            Map<String, String> map = statements.entrySet().stream().collect(Collectors.toMap(entry -> entry.getKey().toString(), entry -> {
                String value = entry.getValue().toString();
                return !value.contains("{0}") ? value : StringUtil.createString((String)value, (Object[])new Object[]{this.config.tableName()});
            }));
            return map;
        }
        catch (IOException e) {
            throw new RelationalProviderException(e);
        }
    }

    private InputStream loadStatementsFile(DatabaseType dbType) {
        String filePrefix = RelationalDb.class.getPackage().getName().replaceAll("\\.", "/") + "/" + dbType.nameString().toLowerCase();
        String majorMinorFile = filePrefix + String.format("_%s.%s_database.properties", dbType.majorVersion(), dbType.minorVersion());
        String majorFile = filePrefix + String.format("_%s_database.properties", dbType.majorVersion());
        String defaultFile = filePrefix + "_database.properties";
        return Stream.of(majorMinorFile, majorFile, defaultFile).map(fileName -> {
            InputStream is = RelationalDb.class.getClassLoader().getResourceAsStream((String)fileName);
            if (LOGGER.isDebugEnabled()) {
                if (is != null) {
                    LOGGER.debug("located DB statements file '{0}'", new Object[]{fileName});
                } else {
                    LOGGER.debug("'{0}' statements file not found", new Object[]{fileName});
                }
            }
            return is;
        }).filter(Objects::nonNull).findFirst().orElseThrow(() -> new RelationalProviderException(RelationalProviderI18n.unsupportedDBError, dbType));
    }

    public String toString() {
        return "RelationalDB[" + this.config.toString() + "]";
    }

    private void logDebug(String message, Object ... args) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug(message, args);
        }
    }

    @FunctionalInterface
    private static interface SQLFunction<R> {
        public R execute(Connection var1) throws SQLException;
    }
}

