/*
 * Decompiled with CFR 0.152.
 */
package org.modeshape.schematic.internal.document;

import java.io.Reader;
import java.io.StringReader;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;
import java.util.UUID;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.modeshape.schematic.document.Binary;
import org.modeshape.schematic.document.Bson;
import org.modeshape.schematic.document.Code;
import org.modeshape.schematic.document.CodeWithScope;
import org.modeshape.schematic.document.Document;
import org.modeshape.schematic.document.Null;
import org.modeshape.schematic.document.ObjectId;
import org.modeshape.schematic.internal.document.CompactJsonWriter;
import org.modeshape.schematic.internal.document.JsonReader;
import org.modeshape.schematic.internal.document.JsonWriter;

public class JsonReaderParserTest {
    protected boolean print;
    protected Object value;
    protected Document bson;
    protected JsonWriter writer;

    @Before
    public void beforeTest() {
        this.print = false;
        this.writer = new CompactJsonWriter();
    }

    @After
    public void afterTest() {
        this.value = null;
        this.bson = null;
    }

    protected JsonReader.Parser parser(String string) {
        return new JsonReader.Parser(new JsonReader.Tokenizer((Reader)new StringReader(string)), JsonReader.VALUE_FACTORY, JsonReader.DATE_VALUE_MATCHER);
    }

    @Test
    public void shouldParseValueAsNullGivenNoContent() throws Exception {
        this.value = this.parser("").parseValue();
        Assert.assertTrue((boolean)Null.matches((Object)this.value));
    }

    @Test
    public void shouldParseValueAsNullGivenNullLiteral() throws Exception {
        this.value = this.parser("null").parseValue();
        Assert.assertTrue((boolean)Null.matches((Object)this.value));
    }

    @Test
    public void shouldParseValueAsNullGivenNullLiteralWithIncorrectCase() throws Exception {
        this.value = this.parser("nULl").parseValue();
        Assert.assertTrue((boolean)Null.matches((Object)this.value));
    }

    @Test
    public void shouldParseValueGivenTrueBooleanLiteral() throws Exception {
        this.value = this.parser("true").parseValue();
        Assert.assertTrue((boolean)Boolean.TRUE.equals(this.value));
    }

    @Test
    public void shouldParseValueGivenFalseBooleanLiteral() throws Exception {
        this.value = this.parser("false").parseValue();
        Assert.assertTrue((boolean)Boolean.FALSE.equals(this.value));
    }

    @Test
    public void shouldParseValueGivenTrueBooleanLiteralWithIncorrectCase() throws Exception {
        this.value = this.parser("TrUe").parseValue();
        Assert.assertTrue((boolean)Boolean.TRUE.equals(this.value));
    }

    @Test
    public void shouldParseValueGivenFalseBooleanLiteralWithIncorrectCase() throws Exception {
        this.value = this.parser("fAlSE").parseValue();
        Assert.assertTrue((boolean)Boolean.FALSE.equals(this.value));
    }

    @Test
    public void shouldParseValueGivenIntegerValues() throws Exception {
        Assert.assertEquals((Object)0, (Object)this.parser("0").parseValue());
        Assert.assertEquals((Object)-1, (Object)this.parser("-1").parseValue());
        Assert.assertEquals((Object)1, (Object)this.parser("1").parseValue());
        Assert.assertEquals((Object)123456, (Object)this.parser("123456").parseValue());
        Assert.assertEquals((Object)Integer.MAX_VALUE, (Object)this.parser("2147483647").parseValue());
        Assert.assertEquals((Object)Integer.MIN_VALUE, (Object)this.parser("-2147483648").parseValue());
    }

    @Test
    public void shouldParseValueGivenLongValues() throws Exception {
        Assert.assertEquals((Object)0x80000000L, (Object)this.parser("2147483648").parseValue());
        Assert.assertEquals((Object)-2147483649L, (Object)this.parser("-2147483649").parseValue());
        Assert.assertEquals((Object)Long.MAX_VALUE, (Object)this.parser("9223372036854775807").parseValue());
        Assert.assertEquals((Object)Long.MIN_VALUE, (Object)this.parser("-9223372036854775808").parseValue());
    }

    @Test
    public void shouldParseValueGivenDoubleValues() throws Exception {
        Assert.assertEquals((Object)0.1, (Object)this.parser("0.1").parseValue());
        Assert.assertEquals((Object)1.0, (Object)this.parser("1.0").parseValue());
        Assert.assertEquals((Object)-1.0, (Object)this.parser("-1.0").parseValue());
        Assert.assertEquals((Object)-1000.0, (Object)this.parser("-1.0e3").parseValue());
        Assert.assertEquals((Object)3.4028234663852886E38, (Object)this.parser("3.4028234663852886E38").parseValue());
        Assert.assertEquals((Object)-1.0, (Object)this.parser("-1.0").parseValue());
        Assert.assertEquals((Object)Double.MAX_VALUE, (Object)this.parser("1.7976931348623157E308").parseValue());
        Assert.assertEquals((Object)Double.MIN_VALUE, (Object)this.parser("4.9E-324").parseValue());
    }

    @Test
    public void shouldParseDocumentWithOneField() throws Exception {
        this.bson = (Document)this.parser("{ \"foo\" : 32 }").parseValue();
        this.assertField("foo", 32);
    }

    @Test
    public void shouldParseDocumentWithTwoFields() throws Exception {
        this.bson = (Document)this.parser("{ \"foo\" : 32 , \"bar\" : \"baz\" }").parseValue();
        this.assertField("foo", 32);
        this.assertField("bar", "baz");
    }

    @Test
    public void shouldParseDocumentWithThreeFields() throws Exception {
        this.bson = (Document)this.parser("{ \"foo\" : 32 , \"bar\" : \"baz\", \"bom\" : true }").parseValue();
        this.assertField("foo", 32);
        this.assertField("bar", "baz");
        this.assertField("bom", true);
    }

    @Test
    public void shouldParseDocumentWithNestedDocument() throws Exception {
        this.bson = (Document)this.parser("{ \"foo\" : 32 , \"nested\" : { \"bar\" : \"baz\", \"bom\" : true }}").parseValue();
        this.assertField("foo", 32);
        this.bson = this.bson.getDocument("nested");
        this.assertField("bar", "baz");
        this.assertField("bom", true);
    }

    @Test
    public void shouldParseDocumentWithExtraTrailingFieldDelimiter() throws Exception {
        this.bson = (Document)this.parser("{ \"foo\" : 32 , \"bar\" : \"baz\",  }").parseValue();
        this.assertField("foo", 32);
        this.assertField("bar", "baz");
    }

    @Test
    public void shouldParseDocumentWithExtraFieldDelimiters() throws Exception {
        this.bson = (Document)this.parser("{ \"foo\" : 32 , , \"bar\" : \"baz\",,  }").parseValue();
        this.assertField("foo", 32);
        this.assertField("bar", "baz");
    }

    @Test
    public void shouldParseDocumentWithUuid() throws Exception {
        UUID obj = UUID.randomUUID();
        this.value = this.parser(this.writer.write((Object)obj)).parseValue();
        Assert.assertEquals((Object)obj, (Object)this.value);
    }

    @Test
    public void shouldParseDocumentWithObjectId() throws Exception {
        ObjectId obj = new ObjectId(300, 200, 9, 15);
        this.value = this.parser(this.writer.write((Object)obj)).parseValue();
        Assert.assertEquals((Object)obj, (Object)this.value);
    }

    @Test
    public void shouldParseDocumentWithDate() throws Exception {
        Date obj = this.now();
        String dateDoc = this.writer.write((Object)obj);
        this.value = this.parser(dateDoc).parseValue();
        Assert.assertTrue((boolean)(this.value instanceof Date));
    }

    @Test
    public void shouldParseDocumentWithCode() throws Exception {
        Code obj = new Code("foo");
        this.value = this.parser(this.writer.write((Object)obj)).parseValue();
        Assert.assertEquals((Object)obj, (Object)this.value);
    }

    @Test
    public void shouldParseDocumentWithCodeAndSscope() throws Exception {
        Document scope = (Document)this.parser("{ \"foo\" : 32 }").parseValue();
        CodeWithScope obj = new CodeWithScope("foo", scope);
        this.value = this.parser(this.writer.write((Object)obj)).parseValue();
        Assert.assertEquals((Object)obj, (Object)this.value);
    }

    @Test
    public void shouldParseDocumentWithBinary() throws Exception {
        byte[] bytes = new byte[]{19, 34, 83, 0};
        Binary obj = new Binary(5, bytes);
        this.value = this.parser(this.writer.write((Object)obj)).parseValue();
        Assert.assertEquals((Object)obj, (Object)this.value);
    }

    @Test
    public void shouldParseIsoFormat() throws Exception {
        this.value = Bson.getDateParsingFormatter().parse("2011-06-14T16:05:11GMT+00:00");
        Date date = (Date)this.value;
        Assert.assertTrue((boolean)date.after(this.date(2011, 6, 13)));
        Assert.assertTrue((boolean)date.before(this.date(2011, 6, 16)));
    }

    @Test
    public void shouldParseMsDateFormat() throws Exception {
        this.value = JsonReader.DATE_VALUE_MATCHER.parseValue("\\/Date(2011-06-14T16:05:11GMT+00:00)\\/");
        Assert.assertTrue((boolean)(this.value instanceof Date));
        Date date = (Date)this.value;
        Assert.assertTrue((boolean)date.after(this.date(2011, 6, 13)));
        Assert.assertTrue((boolean)date.before(this.date(2011, 6, 16)));
    }

    @Test
    public void shouldParseMsDateFormatWithSpaces() throws Exception {
        this.value = JsonReader.DATE_VALUE_MATCHER.parseValue("\\/Date( 2011-06-14T16:05:11GMT+00:00 )\\/");
        Assert.assertTrue((boolean)(this.value instanceof Date));
        Date date = (Date)this.value;
        Assert.assertTrue((boolean)date.after(this.date(2011, 6, 13)));
        Assert.assertTrue((boolean)date.before(this.date(2011, 6, 16)));
    }

    @Test
    public void shouldParseEscapedDateFormat() throws Exception {
        this.value = JsonReader.DATE_VALUE_MATCHER.parseValue("/Date(2011-06-14T16:05:11GMT+00:00)/");
        Assert.assertTrue((boolean)(this.value instanceof Date));
        Date date = (Date)this.value;
        Assert.assertTrue((boolean)date.after(this.date(2011, 6, 13)));
        Assert.assertTrue((boolean)date.before(this.date(2011, 6, 16)));
    }

    protected Date date(int year, int month, int day) {
        Calendar cal = Calendar.getInstance();
        cal.clear();
        cal.setTimeZone(TimeZone.getTimeZone("GMT"));
        cal.set(year, month - 1, day);
        return cal.getTime();
    }

    @Test
    public void shouldParseEscapedDateFormatWithSpaces() throws Exception {
        this.value = JsonReader.DATE_VALUE_MATCHER.parseValue("/Date( 2011-06-14T16:05:11GMT+00:00 )/");
        Assert.assertTrue((boolean)(this.value instanceof Date));
        Date date = (Date)this.value;
        Assert.assertTrue((boolean)date.after(this.date(2011, 6, 13)));
        Assert.assertTrue((boolean)date.before(this.date(2011, 6, 16)));
    }

    @Test
    public void shouldEvaluateIsoDateWithGmtTimeZone() throws Exception {
        this.value = JsonReader.DATE_VALUE_MATCHER.parseValue("2011-06-14T16:05:11GMT+00:00");
        Assert.assertTrue((boolean)(this.value instanceof Date));
        Date date = (Date)this.value;
        Assert.assertTrue((boolean)date.after(this.date(2011, 6, 13)));
        Assert.assertTrue((boolean)date.before(this.date(2011, 6, 16)));
    }

    @Test
    public void shouldEvaluateIsoDateWithZuluTimeZone() throws Exception {
        this.value = JsonReader.DATE_VALUE_MATCHER.parseValue("2011-06-14T16:05:11Z");
        Assert.assertTrue((boolean)(this.value instanceof Date));
        Date date = (Date)this.value;
        Assert.assertTrue((boolean)date.after(this.date(2011, 6, 13)));
        Assert.assertTrue((boolean)date.before(this.date(2011, 6, 16)));
    }

    @Test
    public void shouldParseIsoDateWithZuluTimeZone() throws Exception {
        Date date = this.now();
        this.value = JsonReader.DATE_VALUE_MATCHER.parseValue(Bson.getDateFormatter().format(date));
        Assert.assertTrue((boolean)(this.value instanceof Date));
    }

    @Test
    public void dateMatcherShouldPreserveBackslashEscapeChars() throws Exception {
        String value1 = "one\\backslash";
        Assert.assertEquals((Object)value1, (Object)JsonReader.DATE_VALUE_MATCHER.parseValue(value1));
        String value2 = "two\\\\backslashes";
        Assert.assertEquals((Object)value2, (Object)JsonReader.DATE_VALUE_MATCHER.parseValue(value2));
        String value3 = "three\\\\\\backslashes";
        Assert.assertEquals((Object)value3, (Object)JsonReader.DATE_VALUE_MATCHER.parseValue(value3));
    }

    protected void print(Object object) {
        if (this.print) {
            System.out.println(object);
            System.out.flush();
        }
    }

    protected Date now() {
        return new Date();
    }

    protected void assertField(String name, Object value) {
        Object actual = this.bson.get(name);
        if (value == null) {
            Assert.assertTrue((boolean)Null.matches((Object)actual));
        } else {
            Assert.assertEquals((Object)value, (Object)actual);
        }
    }
}

