/*
 * Decompiled with CFR 0.152.
 */
package org.modeshape.common.statistic;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import net.jcip.annotations.NotThreadSafe;
import org.modeshape.common.math.MathOperations;
import org.modeshape.common.util.StringUtil;

@NotThreadSafe
public class Histogram<T extends Number> {
    public static final int DEFAULT_BUCKET_COUNT = 10;
    public static final int DEFAULT_SIGNIFICANT_FIGURES = 4;
    protected final MathOperations<T> math;
    protected final List<T> values;
    private int bucketCount = 10;
    private int significantFigures = 4;
    private BigDecimal bucketWidth;
    private LinkedList<Bucket> buckets;
    private BucketingStrategy actualValueStrategy;
    private BucketingStrategy bucketingStrategy = this.actualValueStrategy = new DefaultBucketingStrategy();

    public Histogram(MathOperations<T> operations, List<T> values) {
        this.math = operations;
        this.values = new LinkedList<T>(values);
        this.buckets = new LinkedList();
        this.bucketWidth = null;
        Collections.sort(this.values, this.math.getComparator());
    }

    public Histogram(MathOperations<T> operations, T ... values) {
        this(operations, Arrays.asList(values));
    }

    public BucketingStrategy getStrategy() {
        return this.bucketingStrategy;
    }

    public MathOperations<T> getMathOperations() {
        return this.math;
    }

    public void setStrategy(double median, double standardDeviation, int sigma) {
        this.bucketingStrategy = new StandardDeviationBucketingStrategy(median, standardDeviation, sigma);
        this.bucketWidth = null;
    }

    public void setStrategy(T minimum, T maximum) {
        this.bucketingStrategy = new ExplicitBucketingStrategy(this, minimum, maximum);
        this.bucketWidth = null;
    }

    public void setStrategyToDefault() {
        this.bucketingStrategy = this.actualValueStrategy;
        this.bucketWidth = null;
    }

    public int getSignificantFigures() {
        return this.significantFigures;
    }

    public Histogram<T> setSignificantFigures(int significantFigures) {
        if (significantFigures != this.significantFigures) {
            this.significantFigures = significantFigures;
            this.bucketWidth = null;
            this.buckets.clear();
        }
        return this;
    }

    public int getBucketCount() {
        return this.bucketCount;
    }

    public Histogram<T> setBucketCount(int count) {
        if (count != this.bucketCount) {
            this.bucketCount = count;
            this.bucketWidth = null;
            this.buckets.clear();
        }
        return this;
    }

    public List<Bucket> getBuckets() {
        this.compute();
        return this.buckets;
    }

    protected void compute() {
        if (this.bucketWidth != null) {
            return;
        }
        Object lowerBound = this.bucketingStrategy.getLowerBound();
        Object upperBound = this.bucketingStrategy.getUpperBound();
        Object actualMinimum = this.actualValueStrategy.getLowerBound();
        Object actualMaximum = this.actualValueStrategy.getUpperBound();
        List<T> boundaries = Histogram.getBucketBoundaries(this.math, lowerBound, upperBound, actualMinimum, actualMaximum, this.bucketCount, this.significantFigures);
        this.buckets.clear();
        int numBuckets = boundaries.isEmpty() ? 0 : boundaries.size() - 1;
        for (int i = 0; i != numBuckets; ++i) {
            this.buckets.add(new Bucket(this, (Number)boundaries.get(i), (Number)boundaries.get(i + 1)));
        }
        Iterator intervalIterator = this.buckets.iterator();
        Bucket currentInterval = null;
        for (Number value : this.values) {
            while ((currentInterval == null || currentInterval.checkValue(value, !intervalIterator.hasNext()) > 0) && intervalIterator.hasNext()) {
                currentInterval = (Bucket)intervalIterator.next();
            }
            if (currentInterval == null) continue;
            currentInterval.addValue(value);
        }
    }

    public long getTotalNumberOfValues() {
        return this.values.size();
    }

    protected float getMaximumPercentage() {
        float maxPercentage = 0.0f;
        for (Bucket bucket : this.buckets) {
            maxPercentage = Math.max(maxPercentage, bucket.getPercentageOfValues());
        }
        return maxPercentage;
    }

    protected long getMaximumCount() {
        long maxCount = 0L;
        for (Bucket bucket : this.buckets) {
            maxCount = Math.max(maxCount, bucket.getNumberOfValues());
        }
        return maxCount;
    }

    public List<String> getTextGraph(int maxBarLength) {
        this.compute();
        if (maxBarLength < 1) {
            maxBarLength = (int)this.getMaximumCount();
        }
        float barLengthForHundredPercent = this.buckets.isEmpty() ? (float)maxBarLength : 100.0f * (float)maxBarLength / this.getMaximumPercentage();
        String fullLengthBar = StringUtil.createString('*', (int)barLengthForHundredPercent);
        LinkedList<String> result = new LinkedList<String>();
        int maxLowerBoundLength = 0;
        int maxUpperBoundLength = 0;
        for (Bucket bucket : this.buckets) {
            maxLowerBoundLength = Math.max(bucket.getLowerBound().toString().length(), maxLowerBoundLength);
            maxUpperBoundLength = Math.max(bucket.getUpperBound().toString().length(), maxUpperBoundLength);
        }
        int rangeWidth = 1 + maxLowerBoundLength + 3 + maxUpperBoundLength + 1;
        int barWidth = maxBarLength + 20;
        result.add(StringUtil.justifyLeft("Ranges", rangeWidth, ' ') + " Distribution");
        result.add(StringUtil.createString('-', rangeWidth) + ' ' + StringUtil.createString('-', barWidth));
        for (Bucket bucket : this.buckets) {
            float percent = bucket.getPercentageOfValues();
            long number = bucket.getNumberOfValues();
            StringBuilder sb = new StringBuilder();
            sb.append("[");
            sb.append(StringUtil.justifyLeft(bucket.getLowerBound().toString(), maxLowerBoundLength, ' '));
            sb.append(" - ");
            sb.append(StringUtil.justifyLeft(bucket.getUpperBound().toString(), maxUpperBoundLength, ' '));
            sb.append("] ");
            int barLength = Math.max((int)(barLengthForHundredPercent * percent / 100.0f), 0);
            if (barLength == 0 && number != 0L) {
                barLength = 1;
            }
            sb.append(fullLengthBar.substring(0, barLength));
            if (number != 0L) {
                sb.append(" ");
                sb.append(number);
                sb.append(" (");
                sb.append(new DecimalFormat("###.#").format(percent));
                sb.append("%)");
            }
            result.add(sb.toString());
        }
        return result;
    }

    protected static <T> List<T> getBucketBoundaries(MathOperations<T> math, T lowerBound, T upperBound, T actualMinimum, T actualMaximum, int bucketCount, int bucketWidthSigFigs) {
        boolean extraUpperBucketNeeded;
        lowerBound = math.compare(lowerBound, actualMinimum) < 0 ? actualMinimum : lowerBound;
        T t = upperBound = math.compare(actualMaximum, upperBound) < 0 ? actualMaximum : upperBound;
        if (math.compare(lowerBound, upperBound) == 0) {
            ArrayList<T> boundaries = new ArrayList<T>();
            boundaries.add(lowerBound);
            boundaries.add(upperBound);
            return boundaries;
        }
        boolean extraLowerBucketNeeded = math.compare(lowerBound, actualMinimum) > 0;
        boolean bl = extraUpperBucketNeeded = math.compare(actualMaximum, upperBound) > 0;
        if (extraLowerBucketNeeded) {
            --bucketCount;
        }
        if (extraUpperBucketNeeded) {
            --bucketCount;
        }
        T totalWidth = math.subtract(upperBound, lowerBound);
        int totalWidthScale = math.getExponentInScientificNotation(totalWidth);
        T roundedLowerBound = math.roundDown(lowerBound, -totalWidthScale);
        T roundedUpperBound = math.roundUp(upperBound, -totalWidthScale);
        double finalLowerBound = math.doubleValue(roundedLowerBound);
        double finalUpperBound = math.doubleValue(roundedUpperBound);
        double finalBucketCount = bucketCount;
        double bucketWidth = (finalUpperBound - finalLowerBound) / finalBucketCount;
        ArrayList<T> boundaries = new ArrayList<T>();
        if (bucketWidth > 0.0) {
            if (extraLowerBucketNeeded) {
                boundaries.add(actualMinimum);
            }
            double nextBoundary = finalLowerBound;
            for (int i = 0; i != bucketCount; ++i) {
                boundaries.add(math.create(nextBoundary));
                nextBoundary += bucketWidth;
            }
            boundaries.add(roundedUpperBound);
            if (extraUpperBucketNeeded) {
                boundaries.add(actualMaximum);
            }
        }
        return boundaries;
    }

    public class StandardDeviationBucketingStrategy
    extends BucketingStrategy {
        private final double median;
        private final double standardDeviation;
        private final int numberOfDeviationsAboveAndBelow;

        protected StandardDeviationBucketingStrategy(double median, double standardDeviation, int numDeviationsAboveAndBelow) {
            this.median = median;
            this.standardDeviation = Math.abs(standardDeviation);
            this.numberOfDeviationsAboveAndBelow = Math.abs(numDeviationsAboveAndBelow);
        }

        @Override
        public T getLowerBound() {
            double lower = this.median - this.standardDeviation * (double)this.numberOfDeviationsAboveAndBelow;
            return (Number)Histogram.this.math.create(lower);
        }

        @Override
        public T getUpperBound() {
            double upper = this.median + this.standardDeviation * (double)this.numberOfDeviationsAboveAndBelow;
            return (Number)Histogram.this.math.create(upper);
        }
    }

    public static class ExplicitBucketingStrategy
    extends BucketingStrategy {
        private final T lowerBound;
        private final T upperBound;
        final /* synthetic */ Histogram this$0;

        protected ExplicitBucketingStrategy(T lowerBound, T upperBound) {
            this.this$0 = var1_1;
            this.lowerBound = lowerBound;
            this.upperBound = upperBound;
        }

        @Override
        public T getLowerBound() {
            return this.lowerBound;
        }

        @Override
        public T getUpperBound() {
            return this.upperBound;
        }
    }

    public class DefaultBucketingStrategy
    extends BucketingStrategy {
        @Override
        public T getLowerBound() {
            if (this.getValues().isEmpty()) {
                return (Number)Histogram.this.math.createZeroValue();
            }
            return (Number)this.getValues().get(0);
        }

        @Override
        public T getUpperBound() {
            if (this.getValues().isEmpty()) {
                return (Number)Histogram.this.math.createZeroValue();
            }
            return (Number)this.getValues().get(this.getValues().size() - 1);
        }
    }

    public abstract class BucketingStrategy {
        public List<T> getValues() {
            return Histogram.this.values;
        }

        public abstract T getLowerBound();

        public abstract T getUpperBound();
    }

    public static class Bucket
    implements Comparable<Bucket> {
        private final T lowerBound;
        private final T upperBound;
        private final T width;
        private long numValues;
        final /* synthetic */ Histogram this$0;

        protected Bucket(T lowerBound, T upperBound) {
            this.this$0 = var1_1;
            this.lowerBound = lowerBound;
            this.upperBound = upperBound;
            this.width = (Number)var1_1.math.subtract(upperBound, lowerBound);
        }

        public T getLowerBound() {
            return this.lowerBound;
        }

        public T getUpperBound() {
            return this.upperBound;
        }

        public T getWidth() {
            return this.width;
        }

        public float getPercentageOfValues() {
            float total = this.this$0.getTotalNumberOfValues();
            if (total == 0.0f) {
                return 0.0f;
            }
            float numValuesFloat = this.numValues;
            return 100.0f * numValuesFloat / total;
        }

        protected void addValue(T value) {
            ++this.numValues;
        }

        public long getNumberOfValues() {
            return this.numValues;
        }

        public int checkValue(T value, boolean isLast) {
            if (this.this$0.math.compare(this.lowerBound, value) > 0) {
                return -1;
            }
            if (isLast ? this.this$0.math.compare(value, this.upperBound) > 0 : this.this$0.math.compare(value, this.upperBound) >= 0) {
                return 1;
            }
            return 0;
        }

        @Override
        public int compareTo(Bucket that) {
            if (this.this$0.math.compare(this.lowerBound, that.lowerBound) < 0) {
                return -1;
            }
            if (this.this$0.math.compare(this.lowerBound, that.lowerBound) > 0) {
                return 1;
            }
            if (this.this$0.math.compare(this.upperBound, that.upperBound) < 0) {
                return -1;
            }
            if (this.this$0.math.compare(this.upperBound, that.upperBound) > 0) {
                return 1;
            }
            return 0;
        }

        protected Class<T> getNumberClass() {
            return this.this$0.math.getOperandClass();
        }

        public boolean equals(Object obj) {
            if (obj instanceof Bucket) {
                Bucket that = (Bucket)obj;
                if (this.getNumberClass().isAssignableFrom(that.getNumberClass())) {
                    if (this.this$0.math.compare(this.lowerBound, that.lowerBound) != 0) {
                        return false;
                    }
                    if (this.this$0.math.compare(this.upperBound, that.upperBound) != 0) {
                        return false;
                    }
                    return this.this$0.math.compare(this.width, that.width) == 0;
                }
            }
            return false;
        }

        public String toString() {
            return "[" + this.lowerBound + "," + this.upperBound + ")";
        }
    }
}

