/*
 * JBoss, Home of Professional Open Source
 *
 * Copyright 2013 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.picketlink.idm.internal.util;

import static org.picketlink.idm.IDMMessages.MESSAGES;

import org.picketlink.idm.config.FeatureSet.FeatureGroup;
import org.picketlink.idm.model.Agent;
import org.picketlink.idm.model.Group;
import org.picketlink.idm.model.IdentityType;
import org.picketlink.idm.model.Role;
import org.picketlink.idm.model.User;
import org.picketlink.idm.query.QueryParameter;

/**
 * General purpose Util
 *
 * @author anil saldhana
 * @since Sep 13, 2012
 *
 */
public class IDMUtil {

    public static boolean isGroupType(Class<? extends IdentityType> identityType) {
        return Group.class.isAssignableFrom(identityType);
    }

    public static boolean isRoleType(Class<? extends IdentityType> identityType) {
        return Role.class.isAssignableFrom(identityType);
    }

    public static boolean isUserType(Class<? extends IdentityType> identityType) {
        return User.class.isAssignableFrom(identityType);
    }

    public static boolean isAgentType(Class<? extends IdentityType> identityType) {
        return Agent.class.isAssignableFrom(identityType);
    }

    public static FeatureGroup getFeatureGroup(Class<? extends IdentityType> identityType) {
        if (User.class.isAssignableFrom(identityType)) {
            return FeatureGroup.user;
        } else if (Agent.class.isAssignableFrom(identityType)) {
            return FeatureGroup.agent;
        } else if (Group.class.isAssignableFrom(identityType)) {
            return FeatureGroup.group;
        } else if (Role.class.isAssignableFrom(identityType)) {
            return FeatureGroup.role;
        } else if (IdentityType.class.equals(identityType)) {
            // TODO: we need this in order to allow queries using the IdentityType. Maybe we should have a more specific feature
            // group for this cases.
            return FeatureGroup.user;
        } else {
            throw MESSAGES.identityTypeUnsupportedType(identityType);
        }
    }

    /**
     * Return default criterias for sorting query results. Those are used by default if there are not sorting criterias
     * specified by user
     *
     * @param clazz identity type
     * @return default sorting criteria for particular identity type
     */
    public static QueryParameter[] getDefaultParamsForSorting(Class<? extends IdentityType> clazz) {
        if (isAgentType(clazz)) {
            return new QueryParameter[] { Agent.LOGIN_NAME };
        } else if (isGroupType(clazz)) {
            return new QueryParameter[] { Group.NAME };
        } else if (isRoleType(clazz)) {
            return new QueryParameter[] { Role.NAME };
        } else {
            return new QueryParameter[] { IdentityType.ID };
        }
    }

    /**
     * Match two arrays for equality
     *
     * @param arr1
     * @param arr2
     * @return
     */
    public static boolean arraysEqual(String[] arr1, String[] arr2) {
        if (arr1 != null && arr2 == null) {
            return false;
        }
        if (arr1 == null && arr2 == null) {
            return true;
        }
        if (arr1 == null && arr2 != null) {
            return false;
        }

        int length1 = arr1.length;
        int length2 = arr2.length;
        if (length1 != length2) {
            return false;
        }
        boolean foundMatch = false;
        for (int i = 0; i < length1; i++) {
            for (int j = 0; j < length2; j++) {
                if (arr1[i].equals(arr2[j])) {
                    foundMatch = true;
                    break;
                }
            }
            if (foundMatch == false) {
                return false;
            }
            // reset
            foundMatch = false;
        }
        return true;
    }

    /**
     * Sleep given number of milliseconds
     *
     * @param ms
     */
    public static void sleep(int ms) {
        try {
            Thread.sleep(ms);
        } catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
        }
    }

}