// resource: org.richfaces:richfaces.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

window.RichFaces = window.RichFaces || {};
RichFaces.jQuery = RichFaces.jQuery || window.jQuery;

(function($, rf) {

    rf.RICH_CONTAINER = "rf";

    /**
     * All input elements which can hold value, which are enabled and visible.
     */
    rf.EDITABLE_INPUT_SELECTOR = ":not(:submit):not(:button):not(:image):input:visible:enabled";

    //keys codes
    rf.KEYS = {
        BACKSPACE: 8,
        TAB: 9,
        RETURN: 13,
        ESC: 27,
        PAGEUP: 33,
        PAGEDOWN: 34,
        END: 35,
        HOME: 36,
        LEFT: 37,
        UP: 38,
        RIGHT: 39,
        DOWN: 40,
        DEL: 46
    };

    if (window.jsf) {
        var jsfAjaxRequest = jsf.ajax.request;
        var jsfAjaxResponse = jsf.ajax.response;
    }

    // get DOM element by id or DOM element or jQuery object
    rf.getDomElement = function (source) {
        var type = typeof source;
        var element;
        if (source == null) {
            element = null;
        } else if (type == "string") {
            // id
            element = document.getElementById(source);
        } else if (type == "object") {
            if (source.nodeType) {
                // DOM element
                element = source;
            } else
            if (source instanceof $) {
                // jQuery object
                element = source.get(0);
            }
        }
        return element;
    };

    // get RichFaces component object by component id or DOM element or jQuery object
    rf.component = function (source) {
        var element = rf.getDomElement(source);

        if (element) {
            return $(element).data("rf.widget") || (element[rf.RICH_CONTAINER] || {})["component"];
        }
    };

    rf.$ = function(source) {
        rf.log.warn("The function `RichFaces.$` has been deprecated and renamed to `RichFaces.component`.  Please adjust your usage accordingly.");
        return rf.component.call(this, source);
    }

    /**
     * jQuery selector ":editable" which selects only input elements which can be edited, are visible and enabled
     */
    $.extend($.expr[':'], {
        editable : function(element) {
            return $(element).is(rf.EDITABLE_INPUT_SELECTOR);
        }
    });

    rf.$$ = function(componentName, element) {
        while (element.parentNode) {
            var e = element[rf.RICH_CONTAINER];
            if (e && e.component && e.component.name == componentName)
                return e.component;
            else
                element = element.parentNode;
        }
    };
    rf.findNonVisualComponents = function (source) {
        var element = rf.getDomElement(source);

        if (element) {
            return (element[rf.RICH_CONTAINER] || {})["attachedComponents"];
        }
    };

    // find component and call his method
    rf.invokeMethod = function(source, method) {
        var c = rf.component(source);
        var f;
        if (c && typeof (f = c[method]) == "function") {
            return f.apply(c, Array.prototype.slice.call(arguments, 2));
        }
    };

    //dom cleaner
    rf.cleanComponent = function (source) {
        var component = rf.component(source);
        if (component && !$(source).data('rf.bridge')) {
            //TODO fire destroy event
            component.destroy();
            component.detach(source);
        }
        var attachedComponents = rf.findNonVisualComponents(source);
        if (attachedComponents) {
            for (var i in attachedComponents) {
                if (attachedComponents.hasOwnProperty(i) && attachedComponents[i]) {
                    attachedComponents[i].destroy();
                }
            }
        }
    };

    rf.cleanDom = function(source) {
        var e = (typeof source == "string") ? document.getElementById(source) : $('body').get(0);
        if (source == "javax.faces.ViewRoot") {
            e = $('body').get(0);
        }
        if (e) {
            // Fire a DOM cleanup event
            $(e).trigger("beforeDomClean.RICH");
            var elements = e.getElementsByTagName("*");
            if (elements.length) {
                $.each(elements, function(index) {
                    rf.cleanComponent(this);
                });
                $.cleanData(elements);
            }
            // clean RF 4 BaseComponent components
            rf.cleanComponent(e);
            // cleans data and jQuery UI Widget Factory components
            $.cleanData([e]);
            $(e).trigger("afterDomClean.RICH");
        }
    };

    //form.js
    rf.submitForm = function(form, parameters, target) {
        if (typeof form === "string") {
            form = $(form)
        }
        var initialTarget = form.attr("target");
        var parameterInputs = new Array();
        try {
            form.attr("target", target);

            if (parameters) {
                for (var parameterName in parameters) {
                    if (parameters.hasOwnProperty(parameterName)) {
                        var parameterValue = parameters[parameterName];
    
                        var input = $("input[name='" + parameterName + "']", form);
                        if (input.length == 0) {
                            var newInput = $("<input />").attr({type: 'hidden', name: parameterName, value: parameterValue});
                            if (parameterName === 'javax.faces.portletbridge.STATE_ID' /* fix for fileUpload in portlets */) {
                                input = newInput.prependTo(form);
                            } else {
                                input = newInput.appendTo(form);
                            }
                        } else {
                            input.val(parameterValue);
                        }
    
                        input.each(function() {
                            parameterInputs.push(this)
                        });
                    }
                }
            }

            //TODO: inline onsubmit handler is not triggered - http://dev.jquery.com/ticket/4930
            form.trigger("submit");
        } finally {
            if (initialTarget === undefined) {
                form.removeAttr("target");
            } else {
                form.attr("target", initialTarget);
            }
            $(parameterInputs).remove();
        }
    };



    //utils.js
    $.fn.toXML = function () {
        var out = '';

        if (this.length > 0) {
            if (typeof XMLSerializer == 'function' ||
                typeof XMLSerializer == 'object') {

                var xs = new XMLSerializer();
                this.each(function() {
                    out += xs.serializeToString(this);
                });
            } else if (this[0].xml !== undefined) {
                this.each(function() {
                    out += this.xml;
                });
            } else {
                this.each(function() {
                    out += this;
                });
            }
        }

        return out;
    };

    //there is the same pattern in server-side code:
    //org.richfaces.javascript.ScriptUtils.escapeCSSMetachars(String)
    var CSS_METACHARS_PATTERN = /([#;&,.+*~':"!^$\[\]()=>|\/])/g;

    /**
     * Escapes CSS meta-characters in string according to
     *  <a href="http://api.jquery.com/category/selectors/">jQuery selectors</a> document.
     *
     * @param s - string to escape meta-characters in
     * @return string with meta-characters escaped
     */
    rf.escapeCSSMetachars = function(s) {
        //TODO nick - cache results

        return s.replace(CSS_METACHARS_PATTERN, "\\$1");
    };

    var logImpl;

    rf.setLog = function(newLogImpl) {
        logImpl = newLogImpl;
    };

    rf.log = {
        debug: function(text) {
            if (logImpl) {
                logImpl.debug(text);
            }
        },

        info: function(text) {
            if (logImpl) {
                logImpl.info(text);
            }
        },

        warn: function(text) {
            if (logImpl) {
                logImpl.warn(text);
            }
        },

        error: function(text) {
            if (logImpl) {
                logImpl.error(text);
            }
        },

        setLevel: function(level) {
            if (logImpl) {
                logImpl.setLevel(level);
            }
        },

        getLevel: function() {
            if (logImpl) {
                return logImpl.getLevel();
            }
            return 'info';
        },

        clear: function() {
            if (logImpl) {
                logImpl.clear();
            }
        }
    };

    /**
     * Evaluates chained properties for the "base" object.
     * For example, window.document.location is equivalent to
     * "propertyNamesString" = "document.location" and "base" = window
     * Evaluation is safe, so it stops on the first null or undefined object
     *
     * @param propertyNamesArray - array of strings that contains names of the properties to evaluate
     * @param base - base object to evaluate properties on
     * @return returns result of evaluation or empty string
     */
    rf.getValue = function(propertyNamesArray, base) {
        var result = base;
        var c = 0;
        do {
            result = result[propertyNamesArray[c++]];
        } while (result && c != propertyNamesArray.length);

        return result;
    };

    var VARIABLE_NAME_PATTERN_STRING = "[_A-Z,a-z]\\w*";
    var VARIABLES_CHAIN = new RegExp("^\\s*" + VARIABLE_NAME_PATTERN_STRING + "(?:\\s*\\.\\s*" + VARIABLE_NAME_PATTERN_STRING + ")*\\s*$");
    var DOT_SEPARATOR = /\s*\.\s*/;

    rf.evalMacro = function(macro, base) {
        var value = "";
        // variable evaluation
        if (VARIABLES_CHAIN.test(macro)) {
            // object's variable evaluation
            var propertyNamesArray = $.trim(macro).split(DOT_SEPARATOR);
            value = rf.getValue(propertyNamesArray, base);
            if (!value) {
                value = rf.getValue(propertyNamesArray, window);
            }
        } else {
            //js string evaluation
            try {
                if (base.eval) {
                    value = base.eval(macro);
                } else with (base) {
                    value = eval(macro);
                }
            } catch (e) {
                rf.log.warn("Exception: " + e.message + "\n[" + macro + "]");
            }
        }

        if (typeof value == 'function') {
            value = value(base);
        }
        //TODO 0 and false are also treated as null values
        return value || "";
    };

    var ALPHA_NUMERIC_MULTI_CHAR_REGEXP = /^\w+$/;

    rf.interpolate = function (placeholders, context) {
        var contextVarsArray = new Array();
        for (var contextVar in context) {
            if (ALPHA_NUMERIC_MULTI_CHAR_REGEXP.test(contextVar)) {
                //guarantees that no escaping for the below RegExp is necessary
                contextVarsArray.push(contextVar);
            }
        }

        var regexp = new RegExp("\\{(" + contextVarsArray.join("|") + ")\\}", "g");
        return placeholders.replace(regexp, function(str, contextVar) {
            return context[contextVar];
        });
    };

    rf.clonePosition = function(element, baseElement, positioning, offset) {

    };
    //

    var jsfEventsAdapterEventNames = {
        event: {
            'begin': ['begin'],
            'complete': ['beforedomupdate'],
            'success': ['success', 'complete']
        },
        error: ['error', 'complete']
    };

    var getExtensionResponseElement = function(responseXML) {
        return $("partial-response extension#org\\.richfaces\\.extension", responseXML);
    };

    var JSON_STRING_START = /^\s*(\[|\{)/;

    rf.parseJSON = function(dataString) {
        try {
            if (dataString) {
                if (JSON_STRING_START.test(dataString)) {
                    return $.parseJSON(dataString);
                } else {
                    var parsedData = $.parseJSON("{\"root\": " + dataString + "}");
                    return parsedData.root;
                }
            }
        } catch (e) {
            rf.log.warn("Error evaluating JSON data from element <" + elementName + ">: " + e.message);
        }

        return null;
    }

    var getJSONData = function(extensionElement, elementName) {
        var dataString = $.trim(extensionElement.children(elementName).text());
        return rf.parseJSON(dataString);
    };

    rf.createJSFEventsAdapter = function(handlers) {
        //hash of handlers
        //supported are:
        // - begin
        // - beforedomupdate
        // - success
        // - error
        // - complete
        var handlers = handlers || {};
        var ignoreSuccess;

        return function(eventData) {
            var source = eventData.source;
            //that's request status, not status control data
            var status = eventData.status;
            var type = eventData.type;

            if (type == 'event' && status == 'begin') {
                ignoreSuccess = false;
            } else if (type == 'error') {
                ignoreSuccess = true;
            } else if (ignoreSuccess) {
                return;
            } else if (status == 'complete' && rf.ajaxContainer && rf.ajaxContainer.isIgnoreResponse && rf.ajaxContainer.isIgnoreResponse()) {
                return;
            }

            var typeHandlers = jsfEventsAdapterEventNames[type];
            var handlerNames = (typeHandlers || {})[status] || typeHandlers;

            if (handlerNames) {
                for (var i = 0; i < handlerNames.length; i++) {
                    var eventType = handlerNames[i];
                    var handler = handlers[eventType];
                    if (handler) {
                        var event = {};
                        $.extend(event, eventData);
                        event.type = eventType;
                        if (type != 'error') {
                            delete event.status;

                            if (event.responseXML) {
                                var xml = getExtensionResponseElement(event.responseXML);
                                var data = getJSONData(xml, "data");
                                var componentData = getJSONData(xml, "componentData");

                                event.data = data;
                                event.componentData = componentData || {};
                            }
                        }
                        handler.call(source, event);
                    }
                }
            }
        };
    };

    rf.setGlobalStatusNameVariable = function(statusName) {
        //TODO: parallel requests
        if (statusName) {
            rf['statusName'] = statusName;
        } else {
            delete rf['statusName'];
        }
    }

    rf.setZeroRequestDelay = function(options) {
        if (typeof options.requestDelay == "undefined") {
            options.requestDelay = 0;
        }
    };

    var chain = function() {
        var functions = arguments;
        if (functions.length == 1) {
            return functions[0];
        } else {
            return function() {
                var callResult;
                for (var i = 0; i < functions.length; i++) {
                    var f = functions[i];
                    if (f) {
                        callResult = f.apply(this, arguments);
                    }
                }

                return callResult;
            };
        }
    };

    var createEventHandler = function(handlerCode) {
        if (handlerCode) {
            // ensure safe execution, errors would cause rf.queue to hang up (RF-12132)
            var safeHandlerCode = "try {" +
                    handlerCode + 
                "} catch (e) {" +
                    "window.RichFaces.log.error('Error in method execution: ' + e.message)" +
                "}";
            return new Function("event", safeHandlerCode);
        }

        return null;
    };

    //TODO take events just from .java code using EL-expression
    var AJAX_EVENTS = (function() {
        var serverEventHandler = function(clientHandler, event) {
            var xml = getExtensionResponseElement(event.responseXML);

            var serverHandler = createEventHandler(xml.children(event.type).text());

            if (clientHandler) {
                clientHandler.call(this, event);
            }

            if (serverHandler) {
                serverHandler.call(this, event);
            }
        };

        return {
            'error': null,
            'begin': null,
            'complete': serverEventHandler,
            'beforedomupdate': serverEventHandler
        }
    }());
    
    rf.requestParams = null;

    rf.ajax = function(source, event, options) {
        var options = options || {};

        var sourceId = getSourceId(source, options);
        var sourceElement = getSourceElement(source);

        // event source re-targeting finds a RichFaces component root
        // to setup javax.faces.source correctly - RF-12616)
        if (sourceElement) {
            source = searchForComponentRootOrReturn(sourceElement);
        }

        parameters = options.parameters || {}; // TODO: change "parameters" to "richfaces.ajax.params"
        parameters.execute = "@component";
        parameters.render = "@component";

        if (options.clientParameters) {
            $.extend(parameters, options.clientParameters);
        }

        if (!parameters["org.richfaces.ajax.component"]) {
            parameters["org.richfaces.ajax.component"] = sourceId;
        }

        if (options.incId) {
            parameters[sourceId] = sourceId;
        }

        if (rf.queue) {
            parameters.queueId = options.queueId;
        }

        var form = getFormElement(sourceElement);
        if (window.mojarra && form && form.enctype == "multipart/form-data"
            && jsf.specversion > 20000) {
            var input, name, value;
                rf.requestParams = [];
            // RF-13828: when inside multipart/form-data create hidden inputs for request parameters
            //   Mojarra is going to submit the form instead of sending a XmlHttpRequest
            for (var i in parameters) {
                if (parameters.hasOwnProperty(i)) {
                    value = parameters[i];
                    
                    if (i !== "javax.faces.source" &&
                        i !== "javax.faces.partial.event" &&
                        i !== "javax.faces.partial.execute" &&
                        i !== "javax.faces.partial.render" &&
                        i !== "javax.faces.partial.ajax" &&
                        i !== "javax.faces.behavior.event" &&
                        i !== "queueId") {                             
                            input = document.createElement("input");
                            input.setAttribute("type", "hidden");
                            input.setAttribute("id", i);
                            input.setAttribute("name", i);
                            input.setAttribute("value", value);
                            form.appendChild(input);
                            rf.requestParams.push(i);
                        }
                }
            }
        }

        // propagates some options to process it in jsf.ajax.request
        parameters.rfExt = {};
        parameters.rfExt.status = options.status;
        for (var eventName in AJAX_EVENTS) {
            if (AJAX_EVENTS.hasOwnProperty(eventName)) {
                parameters.rfExt[eventName] = options[eventName];
            }
        }

        jsf.ajax.request(source, event, parameters);
    };

    if (window.jsf) {
        jsf.ajax.request = function request(source, event, options) {

            // build parameters, taking options.rfExt into consideration
            var parameters = $.extend({}, options);
            parameters.rfExt = null;

            var eventHandlers;

            var sourceElement = getSourceElement(source);
            var form = getFormElement(sourceElement);

            for (var eventName in AJAX_EVENTS) {
                if (AJAX_EVENTS.hasOwnProperty(eventName)) {
                    var handlerCode, handler;
    
                    if (options.rfExt) {
                        handlerCode = options.rfExt[eventName];
                        handler = typeof handlerCode == "function" ? handlerCode : createEventHandler(handlerCode);
                    }
    
                    var serverHandler = AJAX_EVENTS[eventName];
                    if (serverHandler) {
                        handler = $.proxy(function(clientHandler, event) {
                          return serverHandler.call(this, clientHandler, event);
                        }, sourceElement, handler);
                    }
    
                    if (handler) {
                        eventHandlers = eventHandlers || {};
                        eventHandlers[eventName] = handler;
                    }
                }
            }

            if (options.rfExt && options.rfExt.status) {
                var namedStatusEventHandler = function() {
                    rf.setGlobalStatusNameVariable(options.rfExt.status);
                };

                //TODO add support for options.submit
                eventHandlers = eventHandlers || {};
                if (eventHandlers.begin) {
                    eventHandlers.begin = chain(namedStatusEventHandler, eventHandlers.begin);
                } else {
                    eventHandlers.begin = namedStatusEventHandler;
                }
            }

            // register handlers for form events: ajaxbegin and ajaxbeforedomupdate
            if (form) {
                eventHandlers.begin = chain(eventHandlers.begin, function() { $(form).trigger('ajaxbegin'); });
                eventHandlers.beforedomupdate = chain(eventHandlers.beforedomupdate, function() { $(form).trigger('ajaxbeforedomupdate'); });
                eventHandlers.complete = chain(eventHandlers.complete, function() { $(form).trigger('ajaxcomplete'); });
            }

            if (eventHandlers) {
                var eventsAdapter = rf.createJSFEventsAdapter(eventHandlers);
                parameters['onevent'] = chain(options.onevent, eventsAdapter);
                parameters['onerror'] = chain(options.onerror, eventsAdapter);
            }

            // trigger handler for form event: ajaxsubmit
            if (form) {
                $(form).trigger('ajaxsubmit');
            }

            return jsfAjaxRequest(source, event, parameters);
        };

        jsf.ajax.response = function(request, context) {
            // for all RichFaces.ajax requests
            if (context.render == '@component') {
                // get list of IDs updated on the server - replaces @render option which is normally available on client
                context.render = $("extension[id='org.richfaces.extension'] render", request.responseXML).text();
            }

            // remove hidden inputs if they were created before submission
            if (window.mojarra && rf.requestParams && rf.requestParams.length) {
                for (var i=0; i<rf.requestParams.length; i++) {
                    var elements = context.form.childNodes;
                    for (var j=0; j<elements.length; j++) {
                        if (!elements[j].type === "hidden") {
                            continue;
                        }
                        if (elements[j].name === rf.requestParams[i]) {
                            var node = context.form.removeChild(elements[j]);
                            node = null;                           
                            break;
                        }
                    }   
                }
            }
            return jsfAjaxResponse(request, context);
        };
    }

    /**
     * Returns RichFaces component root for given element in the list of ancestors of sourceElement.
     * Otherwise returns sourceElement if RichFaces component root can't be located.
     */
    var searchForComponentRootOrReturn = function(sourceElement) {
        if (sourceElement.id && !isRichFacesComponent(sourceElement)) {
            var parentElement = false;
            $(sourceElement).parents().each(function() {
                if (this.id && sourceElement.id.indexOf(this.id) == 0) { // otherwise parent element is definitely not JSF component
                    var suffix = sourceElement.id.substring(this.id.length); // extract suffix
                    if (suffix.match(/^[a-zA-Z]*$/) && isRichFacesComponent(this)) {
                        parentElement = this;
                        return false;
                    }
                }
            });
            if (parentElement !== false) {
                return parentElement;
            }
        }
        return sourceElement;
    };


    /**
     * Detects whether the element has bound RichFaces component.
     *
     * Supports detection of RichFaces 5 (bridge-base.js) and RichFaces 4 (richfaces-base-component.js) components.
     */
    var isRichFacesComponent = function(element) {
      return $(element).data('rf.bridge') || rf.component(element);
    };

    var getSourceElement = function(source) {
        if (typeof source === 'string') {
            return document.getElementById(source);
        } else if (typeof source === 'object') {
            return source;
        } else {
            throw new Error("jsf.request: source must be object or string");
        }
    };

    var getFormElement = function(sourceElement) {
        if ($(sourceElement).is('form')) {
            return sourceElement;
        } else {
            return $('form').has(sourceElement).get(0);
        }
    };

    var getSourceId = function(source, options) {
        if (options.sourceId) {
            return options.sourceId;
        } else {
            return (typeof source == 'object' && (source.id || source.name)) ? (source.id ? source.id : source.name) : source;
        }
    };

    var ajaxOnComplete = function (data) {
        var type = data.type;
        var responseXML = data.responseXML;

        if (data.type == 'event' && data.status == 'complete' && responseXML) {
            var partialResponse = $(responseXML).children("partial-response");
            if (partialResponse && partialResponse.length) {
                var elements = partialResponse.children('changes').children('update, delete');
                $.each(elements, function () {
                    rf.cleanDom($(this).attr('id'));
                });
            }
        }
    };

    rf.javascriptServiceComplete = function(event) {
        $(function() {
            $(document).trigger("javascriptServiceComplete");
        });
    };

    var attachAjaxDOMCleaner = function() {
        // move this code to somewhere
        if (typeof jsf != 'undefined' && jsf.ajax) {
            jsf.ajax.addOnEvent(ajaxOnComplete);

            return true;
        }

        return false;
    };

    if (!attachAjaxDOMCleaner()) {
        $(document).ready(attachAjaxDOMCleaner);
    }

    if (window.addEventListener) {
        window.addEventListener("unload", rf.cleanDom, false);
    } else {
        window.attachEvent("onunload", rf.cleanDom);
    }
    
    // browser detection, taken jQuery Migrate plugin (https://github.com/jquery/jquery-migrate)
    //     and atmosphere.js
    rf.browser = {};
    var ua = navigator.userAgent.toLowerCase(),
        match = /(chrome)[ \/]([\w.]+)/.exec(ua) ||
            /(webkit)[ \/]([\w.]+)/.exec(ua) ||
            /(opera)(?:.*version|)[ \/]([\w.]+)/.exec(ua) ||
            /(msie) ([\w.]+)/.exec(ua) ||
            /(trident)(?:.*? rv:([\w.]+)|)/.exec(ua) ||
            ua.indexOf("compatible") < 0 && /(mozilla)(?:.*? rv:([\w.]+)|)/.exec(ua) ||
            [];

    rf.browser[match[1] || ""] = true;
    rf.browser.version = match[2] || "0";
    
    // Chrome is Webkit, but Webkit is also Safari.
	if ( rf.browser.chrome ) {
		rf.browser.webkit = true;
	} else if ( rf.browser.webkit ) {
		rf.browser.safari = true;
	}

    // Trident is the layout engine of the Internet Explorer
    // IE 11 has no "MSIE: 11.0" token
    if (rf.browser.trident) {
        rf.browser.msie = true;
    }
    
    
    // MyFaces 2.2 workaround, skip input.rf-fu-inp (fileupload) when looking for a multipart candidate
    if (window.myfaces && myfaces._impl && myfaces._impl._util && myfaces._impl._util._Dom.isMultipartCandidate) {
        var oldIsMultipartCandidate = myfaces._impl._util._Dom.isMultipartCandidate,
            that = myfaces._impl._util._Dom;
        
        
        myfaces._impl._util._Dom.isMultipartCandidate = function (executes) {
            if (that._Lang.isString(executes)) {
                executes = that._Lang.strToArray(executes, /\s+/);
            }
    
            for (var cnt = 0, len = executes.length; cnt < len ; cnt ++) {
                var element = that.byId(executes[cnt]);
                var inputs = that.findByTagName(element, "input", true);
                for (var cnt2 = 0, len2 = inputs.length; cnt2 < len2 ; cnt2++) {
                    if (that.getAttribute(inputs[cnt2], "type") == "file" &&
                        (!that.getAttribute(inputs[cnt2], "class") ||
                        that.getAttribute(inputs[cnt2], "class").search("rf-fu-inp") == -1)) 
                    return true;
                }
            }
            return false;
        };
    }
}(RichFaces.jQuery, RichFaces));
;

// resource: org.richfaces:richfaces-queue.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

/**
 * @author Pavel Yaschenko
 */

(function($, rf, jsf) {

    /**
     * RichFaces Ajax container
     * @class
     * @memberOf RichFaces
     * @static
     * @name ajaxContainer
     * */
    rf.ajaxContainer = rf.ajaxContainer || {};

    if (rf.ajaxContainer.jsfRequest) {
        return;
    }

    /**
     * JSF 2.0 original method that sends an asynchronous ajax request to the server
     * see jsf.ajax.request method for parameter's description
     * @function
     * @name RichFaces.ajaxContainer.jsfRequest
     *
     * */
    rf.ajaxContainer.jsfRequest = jsf.ajax.request;

    /**
     * RichFaces wrapper function of JSF 2.0 original method jsf.ajax.request
     * @function
     * @name jsf.ajax.request
     *
     * @param {string|DOMElement} source - The DOM element or an id that triggered this ajax request
     * @param {object} [event] - The DOM event that triggered this ajax request
     * @param {object} [options] - The set name/value pairs that can be sent as request parameters to control client and/or server side request processing
     * */
    jsf.ajax.request = function(source, event, options) {
        rf.queue.push(source, event, options);
    };

    rf.ajaxContainer.jsfResponse = jsf.ajax.response;

    rf.ajaxContainer.isIgnoreResponse = function() {
        return rf.queue.isIgnoreResponse();
    };


    jsf.ajax.response = function(request, context) {
        rf.queue.response(request, context);
    };

    var QUEUE_MODE_PULL = 'pull';
    var QUEUE_MODE_PUSH = 'push';
    var QUEUE_MODE = QUEUE_MODE_PULL;
    var DEFAULT_QUEUE_ID = "org.richfaces.queue.global";

    /**
     * RichFaces Queue API container
     * @class
     * @memberOf RichFaces
     * @static
     * @name queue
     * */
    rf.queue = (function() {

        var defaultQueueOptions = {};
        //defaultQueueOptions[DEFAULT_QUEUE_ID] = {requestDelay:0, ignoreDupResponse:false, timeout:0};
        var eventHandlers = ["status", "onbeforedomupdate", "oncomplete", "onerror"];
        
        var addEventHandlers = function(queueOptions, options) {
            for (var i = 0; i < eventHandlers.length; i++) {
                var event = eventHandlers[i],
                    handler = event.match(/^on/) ? event.substr(2) : event;
                if (queueOptions[event] && options.rfExt && !options.rfExt[handler]) {
                    // add global event handler unless the item provides its own
                    options.rfExt[handler] = queueOptions[event];
                }
            }
        }

        var QueueEntry = function(queue, source, event, options) {
            this.queue = queue;
            this.source = source;
            this.options = $.extend({}, options || {});
            this.queueOptions = {}
            var id;

            // find default options for QueueEntry
            if (this.options.queueId) {
                if (defaultQueueOptions[this.options.queueId]) {
                    id = this.options.queueId;
                }
                delete this.options.queueId;
            } else {
                var element = rf.getDomElement(source);
                var form;
                if (element) {
                    element = $(element).closest("form");
                    if (element.length > 0) {
                        form = element.get(0);
                    }
                }
                if (form && form.id && defaultQueueOptions[form.id]) {
                    id = form.id;
                } else {
                    id = DEFAULT_QUEUE_ID;
                }
            }
            if (id) {
                this.queueOptions = defaultQueueOptions[id] || {};
                if (this.queueOptions.queueId) {
                    this.queueOptions = $.extend({}, (defaultQueueOptions[this.queueOptions.queueId] || {}), this.queueOptions);
                } else {
                    // TODO: clean duplicated code
                    var element = rf.getDomElement(source);
                    var form;
                    if (element) {
                        element = $(element).closest("form");
                        if (element.length > 0) {
                            form = element.get(0);
                        }
                    }
                    if (form && form.id && defaultQueueOptions[form.id]) {
                        id = form.id;
                    } else {
                        id = DEFAULT_QUEUE_ID;
                    }
                    if (id) {
                        this.queueOptions = $.extend({}, (defaultQueueOptions[id] || {}), this.queueOptions);
                    }
                }
            }
            
            addEventHandlers(this.queueOptions, this.options);
            
            if (this.queueOptions.onsubmit) {
                this.options.queueonsubmit = new Function("entry", this.queueOptions.onsubmit);
            }
            
            if (typeof this.queueOptions.requestGroupingId == "undefined") {
                this.queueOptions.requestGroupingId = typeof this.source == "string" ? this.source : this.source.id;
            }

            // Remove the layerX and layerY events (generated in WebKit browsers)
            if (event && event instanceof Object) {
                if ('layerX' in event) delete event.layerX;
                if ('layerY' in event) delete event.layerY;
            }

            // copy of event should be created otherwise IE will fail
            this.event = $.extend({}, event);

            //requestGroupingId is mutable, thus we need special field for it
            this.requestGroupingId = this.queueOptions.requestGroupingId;
            this.eventsCount = 1;
        };

        $.extend(QueueEntry.prototype, {
                // now unused functions: ondrop, clearEntry
                isIgnoreDupResponses: function() {
                    return this.queueOptions.ignoreDupResponses;
                },

                getRequestGroupId: function() {
                    return this.requestGroupingId;
                },

                setRequestGroupId: function(id) {
                    this.requestGroupingId = id;
                },

                resetRequestGroupId: function() {
                    this.requestGroupingId = undefined;
                },

                setReadyToSubmit: function(isReady) {
                    this.readyToSubmit = isReady;
                },

                getReadyToSubmit: function() {
                    return this.readyToSubmit;
                },

                ondrop: function() {
                    var callback = this.queueOptions.onqueuerequestdrop;
                    if (callback) {
                        callback.call(this.queue, this.source, this.options, this.event);
                    }
                },

                onRequestDelayPassed: function() {
                    this.readyToSubmit = true;
                    submitFirstEntry.call(this.queue);
                },

                startTimer: function() {
                    var delay = this.queueOptions.requestDelay;
                    if (typeof delay != "number") {
                        delay = this.queueOptions.requestDelay || 0;
                    }

                    rf.log.debug("Queue will wait " + (delay || 0) + "ms before submit");

                    if (delay) {
                        var _this = this;
                        this.timer = window.setTimeout(function() {
                            try {
                                _this.onRequestDelayPassed();
                            } finally {
                                _this.timer = undefined;
                                _this = undefined;
                            }
                        }, delay);
                    } else {
                        this.onRequestDelayPassed();
                    }
                },

                stopTimer: function() {
                    if (this.timer) {
                        window.clearTimeout(this.timer);
                        this.timer = undefined;
                    }
                },

                clearEntry: function() { //???
                    this.stopTimer();
                    if (this.request) {
                        this.request.shouldNotifyQueue = false;
                        this.request = undefined;
                    }
                },

                getEventsCount: function() {
                    return this.eventsCount;
                },

                setEventsCount: function(newCount) {
                    this.eventsCount = newCount;
                }
            });

        // TODO: add this two variables to richfaces and report bug to jsf about constants
        var JSF_EVENT_TYPE = 'event';
        var JSF_EVENT_SUCCESS = 'success';
        var JSF_EVENT_COMPLETE = 'complete';

        var items = [];
        var lastRequestedEntry;

        //TODO: instance of this function will be created for each queue
        var onError = function (data) {
            var message = "richfaces.queue: ajax submit error";
            if (data) {
                var description = data.message || data.description;
                if (description) {
                    message += ": " + description;
                }
            }
            rf.log.warn(message);

            lastRequestedEntry = null;
            //TODO: what if somebody is going to clear queue on error?
            submitFirstEntry();
        };

        var removeStaleEntriesFromQueue = function () {
            var entry;
            var foundValidEntry = false;
            while (items.length > 0 && !foundValidEntry) {
                entry = items[0];
                var element = rf.getDomElement(entry.source);
                if (element == null || $(element).closest("form").length == 0) {
                    var removedEntry = items.shift();
                    removedEntry.stopTimer();
                    rf.log.debug("richfaces.queue: removing stale entry from the queue (source element: " + element + ")");
                } else {
                    foundValidEntry = true;
                }
            }
        }

        var onComplete = function (data) {
            if (data.type == JSF_EVENT_TYPE && data.status == JSF_EVENT_SUCCESS) { // or JSF_EVENT_COMPLETE will be rather
                rf.log.debug("richfaces.queue: ajax submit successfull");
                lastRequestedEntry = null;
                removeStaleEntriesFromQueue();
                submitFirstEntry();
            }
        };

        jsf.ajax.addOnEvent(onComplete);
        jsf.ajax.addOnError(onError);

        var submitFirstEntry = function() {
            if (QUEUE_MODE == QUEUE_MODE_PULL && lastRequestedEntry) {
                rf.log.debug("richfaces.queue: Waiting for previous submit results");
                return;
            }
            if (isEmpty()) {
                rf.log.debug("richfaces.queue: Nothing to submit");
                return;
            }
            var entry;
            if (items[0].getReadyToSubmit()) {
            	try {
                    entry = lastRequestedEntry = items.shift();
                    rf.log.debug("richfaces.queue: will submit request NOW");
                    var o = lastRequestedEntry.options;
                    o["AJAX:EVENTS_COUNT"] = lastRequestedEntry.eventsCount;
                    rf.ajaxContainer.jsfRequest(lastRequestedEntry.source, lastRequestedEntry.event, o);

                    // call event handlers
                    if (o.queueonsubmit) {
                        o.queueonsubmit.call(entry);
                    }
                    callEventHandler("onrequestdequeue", entry);
            	} catch (error) {
                    onError(error);
            	}
            }
        };

        var isEmpty = function() {
            return (getSize() == 0)
        };
        var getSize = function() {
            return items.length;
        };

        var getLastEntry = function () {
            var lastIdx = items.length - 1;
            return items[lastIdx];
        };

        var updateLastEntry = function (entry) {
            var lastIdx = items.length - 1;
            items[lastIdx] = entry;
        };

        var callEventHandler = function (handlerName, entry) {
            var handler = entry.queueOptions[handlerName];
            if (handler) {
                if (typeof(handler) == "string") {
                    new Function(handler).call(null, entry);
                } else {
                    handler.call(null, entry);
                }
            }
            var opts, handler2;
            if (entry.queueOptions.queueId &&
                (opts = defaultQueueOptions[entry.queueOptions.queueId]) &&
                (handler2 = opts[handlerName])
                && handler2 != handler) {
                // the same about context
                handler2.call(null, entry);
            }
        }

        var pushEntry = function (entry) {
            items.push(entry);
            rf.log.debug("New request added to queue. Queue requestGroupingId changed to " + entry.getRequestGroupId());
            // call event handlers
            callEventHandler("onrequestqueue", entry);
        }

        return {
            /**
             * @constant
             * @name RichFaces.queue.DEFAULT_QUEUE_ID
             * @type string
             * */
            DEFAULT_QUEUE_ID: DEFAULT_QUEUE_ID,

            /**
             * Get current queue size
             * @function
             * @name RichFaces.queue.getSize
             *
             * @return {number} size of items in the queue
             * */
            getSize: getSize,

            /**
             * Check if queue is empty
             * @function
             * @name RichFaces.queue.isEmpty
             *
             * @return {boolean} returns true if queue is empty
             * */
            isEmpty: isEmpty,

            /**
             * Extract and submit first QueueEntry in the queue if QueueEntry is ready to submit
             * @function
             * @name RichFaces.queue.submitFirst
             * */
            submitFirst: function () {
                if (!isEmpty()) {
                    var entry = items[0];
                    entry.stopTimer();
                    entry.setReadyToSubmit(true);
                    submitFirstEntry();
                }
            },

            /**
             * Create and push QueueEntry to the queue for ajax requests
             * @function
             * @name RichFaces.queue.push
             *
             * @param {string|DOMElement} source - The DOM element or an id that triggered this ajax request
             * @param {object} [event] - The DOM event that triggered this ajax request
             * @param {object} [options] - The set name/value pairs that can be sent as request parameters to control client and/or server side request processing
             * */
            push: function (source, event, options) {
                var entry = new QueueEntry(this, source, event, options);
                var requestGroupingId = entry.getRequestGroupId();

                var lastEntry = getLastEntry();

                if (lastEntry) {
                    if (lastEntry.getRequestGroupId() == requestGroupingId) {
                        rf.log.debug("Similar request currently in queue");

                        rf.log.debug("Combine similar requests and reset timer");

                        lastEntry.stopTimer();
                        entry.setEventsCount(lastEntry.getEventsCount() + 1);

                        updateLastEntry(entry);
                        callEventHandler("onrequestqueue", entry);
                    } else {
                        rf.log.debug("Last queue entry is not the last anymore. Stopping requestDelay timer and marking entry as ready for submission")

                        lastEntry.stopTimer();
                        lastEntry.resetRequestGroupId();
                        lastEntry.setReadyToSubmit(true);

                        pushEntry(entry);
                        submitFirstEntry();
                    }
                } else {
                    pushEntry(entry);
                }

                // start timer
                entry.startTimer();

            },

            response: function (request, context) {
                if (this.isIgnoreResponse()) {
                    lastRequestedEntry = null;
                    submitFirstEntry();
                } else {
                    rf.ajaxContainer.jsfResponse(request, context);
                }
            },

            isIgnoreResponse: function () {
                var entry = items[0];
                return entry && lastRequestedEntry.isIgnoreDupResponses()
                    && lastRequestedEntry.queueOptions.requestGroupingId == entry.queueOptions.requestGroupingId;
            },

            /**
             * Remove all QueueEntry from the queue
             * @function
             * @name RichFaces.queue.clear
             * */
            clear: function () {
                var lastEntry = getLastEntry();
                if (lastEntry) {
                    lastEntry.stopTimer();
                }
                items = [];
            },

            /**
             * Set queue default options
             * @function
             * @name RichFaces.queue.setQueueOptions
             *
             * @param {string||object} [id] - Queue id for storing options or hash with options for multiple options set
             * @param {object} options - Queue options object
             * */
            setQueueOptions: function (id, options) {
                var tid = typeof id;
                if (tid == "string") {
                    // add named queue options
                    if (defaultQueueOptions[id]) {
                        throw "Queue already registered";
                    } else {
                        defaultQueueOptions[id] = options;
                    }
                } else if (tid == "object") {
                    // first parameter is hash with queue names and options
                    $.extend(defaultQueueOptions, id);
                }
                return rf.queue;
            },

            getQueueOptions: function (id) {
                return defaultQueueOptions[id] || {};
            }
        }
    }());
}(RichFaces.jQuery, RichFaces, jsf));
;

// resource: org.richfaces:richfaces-csv.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

window.RichFaces = window.RichFaces || {};
RichFaces.jQuery = RichFaces.jQuery || window.jQuery;

(function($, rf) {

    rf.csv = rf.csv || {};

    var _messages = {};

    var RE_MESSAGE_PATTERN = /\'?\{(\d+)\}\'?/g;

    var interpolateMessage = function (message, originalValue, values) {
        if (message) {
            var msgObject = message.replace(RE_MESSAGE_PATTERN, "\n$1\n").split("\n");
            var value;
            values[9] = originalValue;
            for (var i = 1; i < msgObject.length; i += 2) {
                value = values[msgObject[i]];
                msgObject[i] = typeof value == "undefined" ? "" : value;
            }
            return msgObject.join('');
        } else {
            return "";
        }
    }

    var _value_query = function(control) {
        if (null !== control.value && undefined != control.value) {
            return control.value;
        } else {
            return "";
        }
    };

    var _check_query = function(control) {
        if (control.checked) {
            return true;
        } else {
            return false;
        }
    };

    var _addOption = function(value, option) {
        if (option.selected) {
            return value[value.length] = option.value;
        }

    };

    var valueExtractors = {
        hidden : function(control) {
            return _value_query(control);
        },

        text : function(control) {
            return _value_query(control);
        },

        textarea : function(control) {
            return _value_query(control);
        },

        'select-one' : function(control) {
            if (control.selectedIndex != -1) {
                return _value_query(control);
            }
        },

        password : function(control) {
            return _value_query(control);
        },

        file : function(control) {
            return _value_query(control);
        },

        radio : function(control) {
            return _check_query(control);
        },

        checkbox : function(control) {
            return _check_query(control);
        },


        'select-multiple' : function(control) {
            var cname = control.name;
            var childs = control.childNodes;
            var value = [];
            for (var i = 0; i < childs.length; i++) {
                var child = childs[i];
                if (child.tagName === 'OPTGROUP') {
                    var options = child.childNodes;
                    for (var j = 0; j < options.length; j++) {
                        value = _addOption(value, options[j]);
                    }
                } else {
                    value = _addOption(value, child);
                }
            }
            return value;
        },

        // command inputs


        // same as link, but have additional field - control, for input
        // submit.
        input : function(control) {
            return _value_query(control);
        }
    };

    var getValue = function(element) {
        var value = "";
        if (valueExtractors[element.type]) {
            value = valueExtractors[element.type](element);
        } else if (undefined !== element.value) {
            value = element.value;
        } else {
            var component = $(element);
            // TODO: add getValue to baseComponent and change jsdocs
            if (component) {
                if (typeof rf.component(component)["getValue"] === "function") {
                    value = rf.component(component).getValue();
                } else {
                    var nestedComponents = $("*", component).filter(":editable");
                    if (nestedComponents) {
                        var nestedComponent = nestedComponents[0];
                        value = valueExtractors[nestedComponent.type](nestedComponent);
                    }
                }
            }
        }
        return value;
    }

    var getLabel = function(component, id) {
        if (component.p) {
            return component.p.label || id;
        }
        return id;
    }

    $.extend(rf.csv, {
            RE_DIGITS: /^-?\d+$/,
            RE_FLOAT: /^(-?\d+)?(\.(\d+)?(e[+-]?\d+)?)?$/,
            // Messages API
            addMessage: function (messagesObject) {
                $.extend(_messages, messagesObject);
            },
            getMessage: function(customMessage, originalValue, values, messageId) {
                var message = customMessage ? customMessage : _messages[messageId] || {detail:"",summary:"",severity:0};
                return {detail:interpolateMessage(message.detail, originalValue, values),summary:interpolateMessage(message.summary, originalValue, values),severity:message.severity};
            },
            sendMessage: function (componentId, message) {
                rf.Event.fire(window.document, rf.Event.MESSAGE_EVENT_TYPE, {'sourceId':componentId, 'message':message});
            },
            clearMessage: function(componentId) {
                rf.Event.fire(window.document, rf.Event.MESSAGE_EVENT_TYPE, {'sourceId':componentId });
            },
            validate: function (event, id, element, params) {
                var element = rf.getDomElement(element || id);
                var value = getValue(element);
                var convertedValue;
                var converter = params.c;
                rf.csv.clearMessage(id);
                if (converter) {
                    var label = getLabel(converter, id);
                    try {
                        if (converter.f)
                            convertedValue = converter.f(value, id, getLabel(converter, id), converter.m);
                    } catch (e) {
                        e.severity = 2;
                        rf.csv.sendMessage(id, e);
                        return false;
                    }
                } else {
                    convertedValue = value;
                }
                var result = true
                var validators = params.v;
                var validationErrorMessage;
                if (validators) {
                    var validatorFunction,validator;
                    for (var i = 0; i < validators.length; i++) {
                        try {
                            validator = validators[i];
                            validatorFunction = validator.f;
                            if (validatorFunction) {
                                validatorFunction(convertedValue, getLabel(validator, id), validator.p, validator.m);
                            }
                        } catch (e) {
                            validationErrorMessage = e;
                            e.severity = 2;
                            rf.csv.sendMessage(id, e);
                            result = false;
                        }
                    }
                }
                if (!result && params.oninvalid instanceof Function) {
                    params.oninvalid([validationErrorMessage]);
                }
                if (result) {
                    if (!params.da && params.a) {
                        params.a.call(element, event, id);
                    } else if (params.onvalid instanceof Function) {
                        params.onvalid();
                    }
                }
                return result;
            }
        });

    /*
     * convert all natural number formats
     *
     */
    var _convertNatural = function(value, label, msg, min, max, sample) {
        var result = null,
            originalValue = value;
        if (value) {
            value = $.trim(value);
            if (!rf.csv.RE_DIGITS.test(value) || (result = parseInt(value, 10)) < min || result > max) {
                throw rf.csv.getMessage(msg, originalValue, sample ? [value, sample, label] : [value,label]);
            }
        }
        return result;
    }

    var _convertReal = function(value, label, msg, sample) {
        var result = null,
            originalValue = value;
        if (value) {
            value = $.trim(value);
            if (!rf.csv.RE_FLOAT.test(value) || isNaN(result = parseFloat(value))) {
                // TODO - check Float limits.
                throw rf.csv.getMessage(msg, originalValue, sample ? [value, sample, label] : [value,label]);
            }
        }
        return result;
    }
    /*
     * Converters implementation
     */
    $.extend(rf.csv, {
            "convertBoolean": function (value, label, params, msg) {
                if (typeof value === "string") {
                    var lcvalue = $.trim(value).toLowerCase();
                    if (lcvalue === 'on' || lcvalue === 'true' || lcvalue === 'yes') {
                        return true;
                    }
                } else if (true === value) {
                    return true;
                }
                return false;
            },
            "convertDate": function (value, label, params, msg) {
                var result;
                value = $.trim(value);
                // TODO - JSF date converter options.
                result = Date.parse(value);
                return result;
            },
            "convertByte": function (value, label, params, msg) {
                return _convertNatural(value, label, msg, -128, 127, 254);
            },
            "convertNumber": function (value, label, params, msg) {
                var result,
                    originalValue = value;
                value = $.trim(value);
                result = parseFloat(value);
                if (isNaN(result)) {
                    throw rf.csv.getMessage(msg, originalValue, [value, 99, label]);
                }
                return result;
            },
            "convertFloat": function (value, label, params, msg) {
                return _convertReal(value, label, msg, 2000000000);
            },
            "convertDouble": function (value, label, params, msg) {
                return _convertReal(value, label, msg, 1999999);
            },
            "convertShort": function (value, label, params, msg) {
                return _convertNatural(value, label, msg, -32768, 32767, 32456);
            },
            "convertInteger": function (value, label, params, msg) {
                return _convertNatural(value, label, msg, -2147483648, 2147483648, 9346);
            },
            "convertCharacter": function (value, label, params, msg) {
                return _convertNatural(value, label, msg, 0, 65535);
            },
            "convertLong": function (value, label, params, msg) {
                return _convertNatural(value, label, msg, -9223372036854775808, 9223372036854775807, 98765432);
            }
        });

    var validateRange = function(originalValue, value, label, params, msg) {
        var isMinSet = typeof params.min === "number";// && params.min >0;
        var isMaxSet = typeof params.max === "number";// && params.max >0;

        if (isMaxSet && value > params.max) {
            throw rf.csv.getMessage(msg, originalValue, isMinSet ? [params.min,params.max,label] : [params.max,label]);
        }
        if (isMinSet && value < params.min) {
            throw rf.csv.getMessage(msg, originalValue, isMaxSet ? [params.min,params.max,label] : [params.min,label]);
        }
    };

    var validateRegex = function(value, label, pattern, msg) {
        if (typeof pattern != "string" || pattern.length == 0) {
            throw rf.csv.getMessage(msg, value, [], 'REGEX_VALIDATOR_PATTERN_NOT_SET');
        }

        var matchPattern = makePatternAMatch(pattern);
        var re;
        try {
            re = new RegExp(matchPattern);
        } catch (e) {
            throw rf.csv.getMessage(msg, value, [], 'REGEX_VALIDATOR_MATCH_EXCEPTION');
        }
        if (!re.test(value)) {
            throw rf.csv.getMessage(msg, value, [pattern,label]);
        }

    };

    var makePatternAMatch = function(pattern) {
        if (! (pattern.slice(0, 1) === '^') ) {
            pattern = '^' + pattern;
        }
        if (! (pattern.slice(-1) === '$') ) {
            pattern = pattern + '$';
        }
        return pattern;
    }
    /*
     * Validators implementation
     */
    $.extend(rf.csv, {
            "validateLongRange": function (value, label, params, msg) {
                var type = typeof value,
                    originalValue = value;
                if (type !== "number") {
                    if (type != "string") {
                        throw rf.csv.getMessage(msg, value, [componentId, ""], 'LONG_RANGE_VALIDATOR_TYPE');
                    } else {
                        value = $.trim(value);
                        if (!rf.csv.RE_DIGITS.test(value) || (value = parseInt(value, 10)) == NaN) {
                            throw rf.csv.getMessage(msg, value, [componentId, ""], 'LONG_RANGE_VALIDATOR_TYPE');
                        }
                    }
                }

                validateRange(originalValue, value, label, params, msg);
            },
            "validateDoubleRange": function (value, label, params, msg) {
                var type = typeof value,
                    originalValue = value;
                if (type !== "number") {
                    if (type !== "string") {
                        throw rf.csv.getMessage(msg, value, [componentId, ""], 'DOUBLE_RANGE_VALIDATOR_TYPE');
                    } else {
                        value = $.trim(value);
                        if (!rf.csv.RE_FLOAT.test(value) || (value = parseFloat(value)) == NaN) {
                            throw rf.csv.getMessage(msg, value, [componentId, ""], 'DOUBLE_RANGE_VALIDATOR_TYPE');
                        }
                    }
                }

                validateRange(originalValue, value, label, params, msg);
            },
            "validateLength": function (value, label, params, msg) {
                var length = value ? value.length : 0;
                validateRange(value, length, label, params, msg);
            },
            "validateSize": function (value, label, params, msg) {
                var length = value ? value.length : 0;
                validateRange(value, length, label, params, msg);
            },
            "validateRegex": function (value, label, params, msg) {
                validateRegex(value, label, params.pattern, msg);
            },
            "validatePattern": function (value, label, params, msg) {
                validateRegex(value, label, params.regexp, msg);
            },
            "validateRequired": function (value, label, params, msg) {
                if (undefined === value || null === value || "" === value) {
                    throw rf.csv.getMessage(msg, value, [label]);
                }
            },
            "validateTrue": function (value, label, params, msg) {
                if (value !== true) {
                    throw msg;
                }
            },
            "validateFalse": function (value, label, params, msg) {
                if (value !== false) {
                    throw msg;
                }
            },
            "validateMax": function (value, label, params, msg) {
                if (value > params.value) {
                    throw msg;
                }
            },
            "validateMin": function (value, label, params, msg) {
                if (value < params.value) {
                    throw msg;
                }
            }
        });

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:richfaces-base-component.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

/**
 * @author Pavel Yaschenko
 */

window.RichFaces = window.RichFaces || {};
RichFaces.jQuery = RichFaces.jQuery || window.jQuery;

(function ($, rf, params) {

    rf.blankFunction = function () {
    }; //TODO: add it to global library

    /**
     * @class Base class for all components.
     * All RichFaces components should use this class as base or another RichFaces class which based on it.
     *
     <pre><code>
     //Inheritance example:
     (function ($, richfaces, params) {

     // Constructor definition
     richfaces.MyComponent = function(componentId, [options]) {
     // call constructor of parent class
     $super.constructor.call(this, componentId, [options]);

     <span style="color:red">
     // call this.attachToDom method to attach component to dom element
     // its required for the client side API calls and to clean up after ajax request or page unload:
     // destroy method will be called if component attached to dom
     this.attachToDom(componentId);
     </span>
     };

     // define private method
     var myPrivateMethod = function () {
     }

     // Extend component class and add protected methods from parent class to our container
     richfaces.BaseComponent.extend(richfaces.BaseComponent, richfaces.MyComponent);

     // define super class link
     var $super = richfaces.MyComponent.$super;

     // Add new properties and methods
     $.extend(richfaces.MyComponent.prototype, (function (params) {
     return {
     name:"MyComponent",
     f:function (){alert("hello"),
     // destroy method definition for clean up
     destroy: function () {
     // clean up code here

     // call parent's destroy method
     $super.destroy.call(this);
     }
     }
     };
     })(params));
     })(jQuery, RichFaces);
     </code></pre>
     *
     * @memberOf RichFaces
     * @name BaseComponent
     *
     * @constructor
     * @param {String} componentId - component id
     * */
    rf.BaseComponent = function(componentId) {
        this.id = componentId;
        this.options = this.options || {};
    };

    var $p = {};

    var extend = function (parent, child, h) {
        h = h || {};
        var F = rf.blankFunction;
        F.prototype = parent.prototype;
        child.prototype = new F();
        child.prototype.constructor = child;
        child.$super = parent.prototype;
        if (child.$super == rf.BaseComponent.prototype) {
            var r = jQuery.extend({}, $p, h || {});
        }

        var _parent = child;

        // create wrapper with protected methods and variables
        child.extend = function (_child, _h) {
            _h = _h || {};
            var _r = jQuery.extend({}, r || h || {}, _h || {});
            return extend(_parent, _child, _r);
        }
        return r || h;
    };

    /**
     * Method extends child class prototype with parent prototype
     * and return the object with parent's protected methods
     *
     * @function
     * @name RichFaces.BaseComponent.extend
     *
     * @return {object}
     * */
    rf.BaseComponent.extend = function(child, h) {
        return extend(rf.BaseComponent, child, h);
    };


    /**
     * Easy way to create a subclass.
     *
     * Example:
     *
     * RichFaces.ui.MyClass = RichFaces.BaseComponent.extendClass({
     *     // Class name
     *     name: "MyClass",
     *
     *     // Constructor
     *     init : function (...) {
     *         // ...
     *     },
     *
     *     // public api
     *     publicFunction : function () {
     *         // ...
     *     },
     *
     *     // private api
     *     // names of private methods should start with '__' (2 underscore symbols)
     *     __privateFunction : function () {
     *         // ...
     *     },
     *
     *     __overrideMethod : function () {
     *         // if you need to use method from parent class use link to parent prototype
     *         // like in previous solution with extend method
     *         $super.__overrideMethod.call(this, ...params...);
     *
     *         //...
     *     }
     *
     * });
     *
     * RichFaces.ui.MySecondClass = RichFaces.ui.MyClass({
     *     //
     *     name : "MySecondClass",
     *
     *     // Constructor
     *     init : function (...) {
     *         // ...
     *     }
     *
     * })
     *
     * */
    rf.BaseComponent.extendClass = function (methods) {
        var DerivedClass = methods.init || rf.blankFunction;
        var SupperClass = this;

        SupperClass.extend(DerivedClass);

        DerivedClass.extendClass = SupperClass.extendClass;

        $.extend(DerivedClass.prototype, methods);

        return DerivedClass;
    };

    $.extend(rf.BaseComponent.prototype, (function (params) {
        return {
            /**
             * Component name.
             *
             * @name RichFaces.BaseComponent#name
             * @type String
             * */
            name: "BaseComponent",

            /**
             * Method for converting object to string
             *
             * @function
             * @name RichFaces.BaseComponent#toString
             *
             * @return {String}
             * */
            toString: function() {
                var result = [];
                if (this.constructor.$super) {
                    result[result.length] = this.constructor.$super.toString();
                }
                result[result.length] = this.name;
                return result.join(', ');
            },

            /** TODO: add jsdocs and qunit tests
             *
             */
            getValue: function() {
                return;
            },

            /**
             * Method returns element's id for event handlers binding.
             * Event API calls this method when binding by component object as selector was used.
             *
             * @function
             * @name RichFaces.BaseComponent#getEventElement
             *
             * @return {String}
             * */
            getEventElement: function() {
                return this.id;
            },

            /**
             * Attach component object to DOM element by component id, DOM element or jQuery object and returns the element
             * Its required for the client side API calls and to clean up after ajax request or document unload by
             * calling destroy method
             *
             * @function
             * @name RichFaces.BaseComponent#attachToDom
             * @param {string|DOMElement|jQuery} source - component id, DOM element or DOM elements wrapped by jQuery
             *
             * @return {DOMElement}
             * */
            attachToDom: function(source) {
                source = source || this.id;
                var element = rf.getDomElement(source);
                if (element) {
                    var container = element[rf.RICH_CONTAINER] = element[rf.RICH_CONTAINER] || {};
                    container.component = this;
                }
                return element;
            },

            /**
             * Detach component object from DOM element by component id, DOM element or jQuery object
             *
             * @function
             * @name RichFaces.BaseComponent#detach
             * @param {string|DOMElement|jQuery} source - component id, DOM element or DOM elements wrapped by jQuery
             *
             * */
            detach: function(source) {
                source = source || this.id;
                var element = rf.getDomElement(source);
                element && element[rf.RICH_CONTAINER] && (element[rf.RICH_CONTAINER].component = null);
            },

            /**
             * Invokes event on on the DOM element
             * @param eventType event type, e.g. "click"
             * @param element DOM element object
             * @param event jQuery Event
             * @param data additional data used for event handler
             * @return true if an event is successfully invoked
             */
            invokeEvent: function(eventType, element, event, data) {
                var handlerResult, result;
                var eventObj = $.extend({}, event, {type: eventType});

                if (!eventObj) {
                    if (document.createEventObject) {
                        eventObj = document.createEventObject();
                        eventObj.type = eventType;
                    }
                    else if (document.createEvent) {
                        eventObj = document.createEvent('Events');
                        eventObj.initEvent(eventType, true, false);
                    }
                }
                eventObj[rf.RICH_CONTAINER] = {component:this, data: data};

                var eventHandler = this.options['on' + eventType];

                if (typeof eventHandler == "function") {
                    handlerResult = eventHandler.call(element, eventObj);
                }

                if (rf.Event) {
                    result = rf.Event.callHandler(this, eventType, data);
                }

                if (result != false && handlerResult != false) result = true;

                return result;
            },

            /**
             * Destroy method. Will be called before remove component from the page
             *
             * @function
             * @name RichFaces.BaseComponent#destroy
             *
             * */
            destroy: function() {
            }
        };
    })(params));

    rf.BaseNonVisualComponent = function(componentId) {
        this.id = componentId;
        this.options = this.options || {};
    };

    rf.BaseNonVisualComponent.extend = function(child, h) {
        return extend(rf.BaseNonVisualComponent, child, h);
    };

    rf.BaseNonVisualComponent.extendClass = function (methods) {
        var DerivedClass = methods.init || rf.blankFunction;
        var SupperClass = this;

        SupperClass.extend(DerivedClass);

        DerivedClass.extendClass = SupperClass.extendClass;

        $.extend(DerivedClass.prototype, methods);

        return DerivedClass;
    };

    $.extend(rf.BaseNonVisualComponent.prototype, (function (params) {
        return {
            name: "BaseNonVisualComponent",

            toString: function() {
                var result = [];
                if (this.constructor.$super) {
                    result[result.length] = this.constructor.$super.toString();
                }
                result[result.length] = this.name;
                return result.join(', ');
            },

            getValue: function() {
                return;
            },
            /**
             * Attach component object to DOM element by component id, DOM element or jQuery object and returns the element
             * Its required for the client side API calls and to clean up after ajax request or document unload by
             * calling destroy method
             *
             * @function
             * @name RichFaces.BaseNonVisualComponent#attachToDom
             * @param {string|DOMElement|jQuery} source - component id, DOM element or DOM elements wrapped by jQuery
             *
             * @return {DOMElement}
             * */
            attachToDom: function(source) {
                source = source || this.id;
                var element = rf.getDomElement(source);
                if (element) {
                    var container = element[rf.RICH_CONTAINER] = element[rf.RICH_CONTAINER] || {};
                    if (container.attachedComponents) {
                        container.attachedComponents[this.name] = this;
                    } else {
                        container.attachedComponents = {};
                        container.attachedComponents[this.name] = this;
                    }
                }
                return element;
            },

            /**
             * Detach component object from DOM element by component id, DOM element or jQuery object
             *
             * @function
             * @name RichFaces.BaseNonVisualComponent#detach
             * @param {string|DOMElement|jQuery} source - component id, DOM element or DOM elements wrapped by jQuery
             *
             * */
            detach: function(source) {
                source = source || this.id;
                var element = rf.getDomElement(source);
                element && element[rf.RICH_CONTAINER] && (element[rf.RICH_CONTAINER].attachedComponents[this.name] = null);
            },

            /**
             * Destroy method. Will be called before remove component from the page
             *
             * @function
             * @name RichFaces.BaseNonVisualComponent#destroy
             *
             * */
            destroy: function() {
            }
        };
    })(params));


})(jQuery, window.RichFaces || (window.RichFaces = {}));

// RichFaces Base class for ui components
(function($, rf) {

    rf.ui = rf.ui || {};

    // Constructor definition
    rf.ui.Base = function(componentId, options, defaultOptions) {
        this.namespace = "." + rf.Event.createNamespace(this.name, componentId);
        // call constructor of parent class
        $super.constructor.call(this, componentId);
        this.options = $.extend(this.options, defaultOptions, options);
        this.attachToDom();
        this.__bindEventHandlers();
    };

    // Extend component class and add protected methods from parent class to our container
    rf.BaseComponent.extend(rf.ui.Base);

    // define super class link
    var $super = rf.ui.Base.$super;

    $.extend(rf.ui.Base.prototype, {
            __bindEventHandlers: function () {
            },
            destroy: function () {
                rf.Event.unbindById(this.id, this.namespace);
                $super.destroy.call(this);
            }
        });

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:status.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
(function($, rf) {

    var getGlobalStatusNameVariable = function() {
        return rf.statusName;
    }

    var RICHFACES_AJAX_STATUS = "richfaces:ajaxStatus";

    var getStatusDataAttributeName = function(statusName) {
        return statusName ? (RICHFACES_AJAX_STATUS + "@" + statusName) : RICHFACES_AJAX_STATUS;
    };

    var statusAjaxEventHandler = function(data, methodName) {
        if (methodName) {
            //global status name
            var statusName = getGlobalStatusNameVariable();
            var source = data.source;

            var statusApplied = false;
            var statusDataAttribute = getStatusDataAttributeName(statusName);

            var statusContainers;
            if (statusName) {
                statusContainers = [$(document)];
            } else {
                statusContainers = [$(source).parents('form'), $(document)];
            }

            for (var containerIdx = 0; containerIdx < statusContainers.length && !statusApplied;
                 containerIdx++) {

                var statusContainer = statusContainers[containerIdx];
                var statuses = statusContainer.data(statusDataAttribute);
                if (statuses) {
                    for (var statusId in statuses) {
                        if (statuses.hasOwnProperty(statusId)) {
                            var status = statuses[statusId];
                            var result = status[methodName].apply(status, arguments);
                            if (result) {
                                statusApplied = true;
                            } else {
                                delete statuses[statusId];
                            }
                        }
                    }

                    if (!statusApplied) {
                        statusContainer.removeData(statusDataAttribute);
                    }
                }
            }
        }
    };

    var initializeStatuses = function() {
        var thisFunction = arguments.callee;
        if (!thisFunction.initialized) {
            thisFunction.initialized = true;

            var jsfEventsListener = rf.createJSFEventsAdapter({
                    begin: function(event) {
                        statusAjaxEventHandler(event, 'start');
                    },
                    error: function(event) {
                        statusAjaxEventHandler(event, 'error');
                    },
                    success: function(event) {
                        statusAjaxEventHandler(event, 'success');
                    },
                    complete: function() {
                        rf.setGlobalStatusNameVariable(null);
                    }
                });

            jsf.ajax.addOnEvent(jsfEventsListener);
            //TODO blocks default alert error handler
            jsf.ajax.addOnError(jsfEventsListener);
        }
    };

    rf.ui = rf.ui || {};

    rf.ui.Status = rf.BaseComponent.extendClass({

            name: "Status",

            //TODO - support for parallel requests
            init: function(id, options) {
                this.id = id;
                this.attachToDom();
                this.options = options || {};
                this.register();
            },

            register: function() {
                initializeStatuses();

                var statusName = this.options.statusName;
                var dataStatusAttribute = getStatusDataAttributeName(statusName);

                var container;
                if (statusName) {
                    container = $(document);
                } else {
                    container = $(rf.getDomElement(this.id)).parents('form');
                    if (container.length == 0) {
                        container = $(document);
                    }
                    ;
                }

                var statuses = container.data(dataStatusAttribute);
                if (!statuses) {
                    statuses = {};
                    container.data(dataStatusAttribute, statuses);
                }

                statuses[this.id] = this;
            },

            start: function() {
                if (this.options.onstart) {
                    this.options.onstart.apply(this, arguments);
                }

                return this.__showHide('.rf-st-start');
            },

            stop: function() {
                this.__stop();
                return this.__showHide('.rf-st-stop');
            },

            success: function() {
                if (this.options.onsuccess) {
                    this.options.onsuccess.apply(this, arguments);
                }
                return this.stop();
            },

            error: function() {
                if (this.options.onerror) {
                    this.options.onerror.apply(this, arguments);
                }
                this.__stop();

                return this.__showHide(':not(.rf-st-error) + .rf-st-stop, .rf-st-error');
            },

            __showHide: function(selector) {
                var element = $(rf.getDomElement(this.id));
                if (element) {
                    var statusElts = element.children();
                    statusElts.each(function() {
                        var t = $(this);
                        t.css('display', t.is(selector) ? '' : 'none');
                    });

                    return true;
                }
                return false;
            },

            __stop: function () {
                if (this.options.onstop) {
                    this.options.onstop.apply(this, arguments);
                }
            }
        });
}(RichFaces.jQuery, window.RichFaces));
;

// resource: org.richfaces:jquery.position.js
/**
 * @fileOverview jQuery setPosition Plugin to place elements on the page
 * @author Pavel Yaschenko, 05.2010
 * @version 0.5
 */

// draft examples of usage
// jQuery('#tooltip').setPosition('#aaa',{from:'bottom-left', to:'auto-auto'});
// jQuery('#bbb').bind("click",function(e){jQuery('#tooltip').setPosition(e);});
// TODO: clear code
// TODO: optimization

// jQuery(target).setPosition(source,[params])
// source:
//			jQuery selector
//			object {id:}
//			object {left:,top:,width:,height} // all properties are optimal
//			jQuery object
//			dom element
//			event
//
//	params:
//			type: string // position type
//			collision: string // not implemented
//			offset: array [x,y] // implemented only for noPositionType
//			from: string // place target relative of source
//			to: string // direction for target

	/**
	  * @name jQuery
	  * @namespace jQuery 
	  * */

(function($) {
	/**
     * Place DOM element relative to another element or using position parameters. Elements with style.display='none' also supported.
     * 
     * @example jQuery('#tooltip').setPosition('#myDiv',{from:'LB', to:'AA'});
     * @example jQuery('#myClickDiv').bind("click",function(e){jQuery('#tooltip').setPosition(e);});
     *
     * @function
     * @name jQuery#setPosition
     * 
     * @param {object} source - object that provides information about new position. <p>
     * accepts:
     * <ul>
     * 		<li>jQuery selector or object</li>
     * 		<li>object with id: <code>{id:'myDiv'}</code></li>
     * 		<li>object with region settings: <code>{left:0, top:0, width:100, height:100}</code></li>
     * 		<li>DOM Element</li>
     * 		<li>Event object</li>
     * </ul>
     * </p>
     * @param {object} params - position parameters:
     * <dl><dt>
     * @param {string} [params.type] - position type that defines positioning and auto positioning rules ["TOOLTIP","DROPDOWN"]</dt><dt>
     * @param {string} [params.collision] - not implemented yet</dt><dt>
     * @param {array} [params.offset] - provides array(2) with x and y for manually position definition<br/>
     * affects only if "type", "from" and "to" not defined</dt><dt>
     * @param {string} [params.from] - place target relative of source // draft definition</dt><dt>
     * @param {string} [params.to] - direction for target // draft definition</dt>
     * </blockquote>
     *
     * @return {jQuery} jQuery wrapped DOM elements
     * */
	$.fn.setPosition = function(source, params) {
		var stype = typeof source;
		if (stype == "object" || stype == "string") {
			var rect = {};
			if (stype == "string" || source.nodeType || source instanceof jQuery || typeof source.length!="undefined") {
					rect = getElementRect(source);
			} else if (source.type) {
				rect = getPointerRect(source);
			} else if (source.id) {
				rect = getElementRect(document.getElementById(source.id));
			} else {
				rect = source;
			}
			
			var params = params || {};
			var def = params.type || params.from || params.to ? $.PositionTypes[params.type || defaultType] : {noPositionType:true};
			
			var options =  $.extend({}, defaults, def, params);
			if (!options.noPositionType) {
				if (options.from.length>2) {
					options.from = positionDefinition[options.from.toLowerCase()];
				}
				if (options.to.length>2) {
					options.to = positionDefinition[options.to.toLowerCase()];
				}
			}
			return this.each(function() {
					element = $(this);
					//alert(rect.left+" "+rect.top+" "+rect.width+" "+rect.height);
					position(rect, element, options);
				});
		}
		return this;
	};
	
	var defaultType = "TOOLTIP";
	var defaults = {
		collision: "",
		offset: [0,0]
	};
	var re = /^(left|right)-(top|buttom|auto)$/i;
	
	// TODO: make it private
	var positionDefinition = {
		'top-left':'LT',
		'top-right':'RT',
		'bottom-left':'LB',
		'bottom-right':'RB',
		'top-auto':'AT',
		'bottom-auto':'AB',
		'auto-left':'LA',
		'auto-right':'RA',
		'auto-auto':'AA'
	};
	$.PositionTypes = {
		// horisontal constants: L-left, R-right, C-center, A-auto
		// vertical constants:   T-top, B-bottom, M-middle, A-auto
		// for auto: list of joinPoint-Direction pairs
		TOOLTIP: {from:"AA", to:"AA", auto:["RTRT", "RBRT", "LTRT", "RTLT", "LTLT", "LBLT", "RTRB", "RBRB", "LBRB", "RBLB"]},
		DROPDOWN:{from:"AA", to:"AA", auto:["LBRB", "LTRT", "RBLB", "RTLT"]},
        DDMENUGROUP:{from:"AA", to:"AA", auto:["RTRB", "RBRT", "LTLB", "LBLT"]}
	};
	
	/** 
	  * Add or replace position type rules for auto positioning.
	  * Does not fully determinated with parameters yet, only draft version.
	  * 
	  * @function
	  * @name jQuery.addPositionType
	  * @param {string} type - name of position rules
	  * @param {object} option - options of position rules
	  * */
	$.addPositionType = function (type, options) {
		// TODO: change [options] to [from, to, auto]
		/*var obj = {};
		if (match=from.match(re))!=null ) {
			obj.from = [ match[1]=='right' ? 'R' : 'L', match[2]=='bottom' ? 'B' : 'T'];
		}
		if (match=to.match(re))!=null ) {
			obj.to = [ match[1]=='right' ? 'R' : match[1]=='left' ? 'L' : 'A', match[2]=='bottom' ? 'B' : match[2]=='top' ? 'T' : 'A'];
		}*/
		$.PositionTypes[type] = options;
	}
    
    function getPointerRect (event) {
		var e = $.event.fix(event);
		return {width: 0, height: 0, left: e.pageX, top: e.pageY};
	};

	function getElementRect (element) {
		var jqe = $(element);
		var offset = jqe.offset();
		var rect = {width: jqe.outerWidth(), height: jqe.outerHeight(), left: Math.floor(offset.left), top: Math.floor(offset.top)};
		if (jqe.length>1) {
			var width, height, offset;
			var e;
			for (var i=1;i<jqe.length;i++) {
				e = jqe.eq(i);
				if (e.css('display')=="none") continue;
				width = e.outerWidth();
				height = e.outerHeight();
				offset = e.offset();
				var d = rect.left - offset.left;
				if (d<0) {
					if (width-d > rect.width) rect.width = width - d;
				} else {
					rect.width += d;
				}
				var d = rect.top - offset.top;
				if (d<0) {
					if (height-d > rect.height) rect.height = height -d;
				} else {
					rect.height += d;
				}
				if (offset.left < rect.left) rect.left = offset.left;
				if (offset.top < rect.top) rect.top = offset.top;
			}
		}
		
		return rect;
	};
	
	function checkCollision (elementRect, windowRect) {
		// return 0 if elementRect in windowRect without collision
		if (elementRect.left >= windowRect.left &&
			elementRect.top >= windowRect.top &&
			elementRect.right <= windowRect.right &&  
			elementRect.bottom <= windowRect.bottom)
			return 0;
		// return collision squire
		var rect = {left:   (elementRect.left>windowRect.left ? elementRect.left : windowRect.left),
					top:    (elementRect.top>windowRect.top ? elementRect.top : windowRect.top)};
		rect.right = elementRect.right<windowRect.right ? (elementRect.right==elementRect.left ? rect.left : elementRect.right) : windowRect.right;
		rect.bottom = elementRect.bottom<windowRect.bottom ? (elementRect.bottom==elementRect.top ? rect.top : elementRect.bottom) : windowRect.bottom;

		return (rect.right-rect.left) * (rect.bottom-rect.top);
	};
	
	//function fromLeft() {
	/*
	 * params: {
	 * 	left,top,width,height, //baseRect
	 * 	ox,oy, //rectoffset
	 * 	w,h // elementDim
	 * }
	 */
	/*	return this.left;
	}
	
	function fromRight(params) {
			return this.left + this.width - this.w;
	}
	
	function (params) {
		var rect = {left:fromLeft.call(params), right:fromRight.call(params), top:}
	}*/
	
	function getPositionRect(baseRect, rectOffset, elementDim, pos) {
		var rect = {};
		// TODO: add support for center and middle // may be middle rename to center too
		
		var v = pos.charAt(0);
		if (v=='L') {
			rect.left = baseRect.left;
		} else if (v=='R') {
			rect.left = baseRect.left + baseRect.width;
		}
		
		v = pos.charAt(1);
		if (v=='T') {
			rect.top = baseRect.top;
		} else if (v=='B') {
			rect.top = baseRect.top + baseRect.height;
		}
		
		v = pos.charAt(2);
		if (v=='L') {
			rect.left -= rectOffset[0];
			rect.right = rect.left;
			rect.left -= elementDim.width;
		} else if (v=='R') {
			rect.left += rectOffset[0];
			rect.right = rect.left + elementDim.width;
		}		
		
		v = pos.charAt(3);
		if (v=='T') {
			rect.top -= rectOffset[1];
			rect.bottom = rect.top;
			rect.top -= elementDim.height;
		} else if (v=='B') {
			rect.top += rectOffset[1];
			rect.bottom = rect.top + elementDim.height;
		}
		
		return rect;
	}
	
	function __mergePos(s1,s2) {
		var result = "";
		var ch;
		while (result.length < s1.length) {
			ch = s1.charAt(result.length);
			result += ch == 'A' ? s2.charAt(result.length) : ch; 
		}
		return result;
	}
	
	function calculatePosition (baseRect, rectOffset, windowRect, elementDim, options) {

		var theBest = {square:0};
		var rect;
		var s;
		var ox, oy;
		var p = options.from+options.to;
		
		if (p.indexOf('A')<0) {
			return getPositionRect(baseRect, rectOffset, elementDim, p);
		} else {
			var flag = p=="AAAA";
			var pos;
			for (var i = 0; i<options.auto.length; i++) {
				
				// TODO: draft functional
				pos = flag ? options.auto[i] : __mergePos(p, options.auto[i]);
				rect = getPositionRect(baseRect, rectOffset, elementDim, pos);
				ox = rect.left; oy = rect.top;
				s = checkCollision(rect, windowRect);
				if (s!=0) {
					if (ox>=0 && oy>=0 && theBest.square<s) theBest = {x:ox, y:oy, square:s};
				} else break;
			}
			if (s!=0 && (ox<0 || oy<0 || theBest.square>s)) {
				ox=theBest.x; oy=theBest.y
			}
		}
		
		return {left:ox, top:oy};
	}
	
	function position (rect, element, options) {
		var width = element.width();
		var height = element.height();
		
		rect.width = rect.width || 0;
		rect.height = rect.height || 0;
		
		var left = parseInt(element.css('left'),10);
		if (isNaN(left) || left==0) {
			left = 0;
			element.css('left', '0px');
		}
		if (isNaN(rect.left)) rect.left = left;
		
		var top = parseInt(element.css('top'),10);
		if (isNaN(top) || top==0) {
			top = 0;
			element.css('top', '0px');
		}
		if (isNaN(rect.top)) rect.top = top;
		
		var pos = {};
		if (options.noPositionType) {
			pos.left = rect.left + rect.width + options.offset[0];
			pos.top = rect.top + options.offset[1];
		} else {
			var jqw = $(window);
			var winRect = {left:jqw.scrollLeft(), top:jqw.scrollTop()};
			winRect.right = winRect.left + jqw.width();
			winRect.bottom = winRect.top + jqw.height();
		
			pos = calculatePosition(rect, options.offset, winRect, {width:width, height:height}, options);		
		}
		
		// jQuery does not support to get offset for hidden elements
		var hideElement=false;
		var eVisibility;
		var e;
		if (element.css("display")=="none") {
			hideElement=true;
			e = element.get(0);
			eVisibility = e.style.visibility;
			e.style.visibility = 'hidden';
			e.style.display = 'block';
		}
		
		var elementOffset = element.offset();
		
		if (hideElement) {
			e.style.visibility = eVisibility;
			e.style.display = 'none';
		}
		
		pos.left += left - Math.floor(elementOffset.left);
		pos.top += top - Math.floor(elementOffset.top);	

		if (left!=pos.left) {
			element.css('left', (pos.left + 'px'));
		}
		if (top!=pos.top) {
			element.css('top', (pos.top + 'px'));
		}
	};

})(jQuery);

;

// resource: org.richfaces:jquery.component.js
if (!window.RichFaces) {
    window.RichFaces = {};
}

(function($, rf) {

    rf.ui = rf.ui || {};

    var evaluate = function(selector) {
        var result = selector;
        try {
            result = eval(selector);
        } catch (e) {
            //do nothing
        }
        return result;
    };

    var evaluateJQuery = function(element, selector) {
        var result = element || evaluate(selector);
        if (!(result instanceof $)) {
            result = $(result || "");
        }

        return result;
    };

    var createEventHandlerFunction = function(opts) {
        var newFunction = new Function("event", opts.query);
        
        return function() {
            var selector = evaluateJQuery(null, opts.selector);
            if (opts.attachType != "live") {
                selector[opts.attachType || "bind"](opts.event, null, newFunction);
            } else { // $.live is deprecated in jQuery 1.9+
                $(document).on(opts.event, selector.selector, null, newFunction);
            }
        };
    };

    var createDirectQueryFunction = function(opts) {
        var queryFunction = new Function("options", "arguments[1]." + opts.query);

        return function() {
            var element;
            var options;

            if (arguments.length == 1) {             
    	        if (!opts.selector) {
		        	//if selector is not present take selector as default
		         	element = arguments[0];
	        	}else{
		        	//if selector is allready presen from rich jQuery component take options as default.
			        options = arguments[0];	
	        	}		     
		
            } else {
                //function(element, options) { ...query()... }
                element = arguments[0];
                options = arguments[1];
            }

            var selector = evaluateJQuery(element, opts.selector);
            queryFunction.call(this, options, selector);
        };
    };

    var createQueryFunction = function(options) {
        if (options.event) {
            return createEventHandlerFunction(options);
        } else {
            return createDirectQueryFunction(options);
        }
    };

    var query = function(options) {
        if (options.timing == 'immediate') {
            createQueryFunction(options).call(this);
        } else {
            $(document).ready(createQueryFunction(options));
        }
    };

    rf.ui.jQueryComponent = {

        createFunction: createQueryFunction,

        query: query

    };

}(RichFaces.jQuery, RichFaces));
;

// resource: org.richfaces:dnd-indicator.js
/*
 * code review by Pavel Yaschenko
 * 
 * 1. No need to save DOM element (this.indicator). We can use id to get dom element. It helps to avoid memory leaks :)
 * 
 * 2. Name refactoring: change names acceptClass, rejectClass, draggingClass 
 * 						to more readable names: getAcceptClass, getRejectClass, getDragClass
 * 
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.DragIndicator = function(id, options) {
        $super.constructor.call(this, id);
        this.attachToDom(id);

        this.indicator = $(document.getElementById(id));
        this.options = options;
    };

    var defaultOptions = {
    };

    rf.BaseComponent.extend(rf.ui.DragIndicator);
    var $super = rf.ui.DragIndicator.$super;

    $.extend(rf.ui.DragIndicator.prototype, ( function () {
        return {
            show : function() {
                this.indicator.show();
            },

            hide: function() {
                this.indicator.hide();
            },

            getAcceptClass: function() {
                return this.options.acceptClass;
            },

            getRejectClass: function() {
                return this.options.rejectClass;
            },

            getDraggingClass: function() {
                return this.options.draggingClass;
            },

            getElement: function() {
                return this.indicator;
            }
        }
    })());

})(RichFaces.jQuery, window.RichFaces);

;

// resource: org.richfaces:jquery.watermark.js
/*	
 Watermark plugin for jQuery
 Version: 3.0.6
 http://jquery-watermark.googlecode.com/

 Copyright (c) 2009-2010 Todd Northrop
 http://www.speednet.biz/

 June 21, 2010

 Requires:  jQuery 1.2.3+

 Dual licensed under the MIT or GPL Version 2 licenses.
 See mit-license.txt and gpl2-license.txt in the project root for details.
 ------------------------------------------------------*/

(function ($) {

    var
    // Will speed up references to undefined
            undefined,

    // String constants for data names
            dataFlag = "watermark",
            dataClass = "watermarkClass",
            dataFocus = "watermarkFocus",
            dataFormSubmit = "watermarkSubmit",
            dataMaxLen = "watermarkMaxLength",
            dataPassword = "watermarkPassword",
            dataText = "watermarkText",

    // Includes only elements with watermark defined
            selWatermarkDefined = ":data(" + dataFlag + ")",

    // Includes only elements capable of having watermark
            selWatermarkAble = ":text,:password,:search,textarea",

    // triggerFns:
    // Array of function names to look for in the global namespace.
    // Any such functions found will be hijacked to trigger a call to
    // hideAll() any time they are called.  The default value is the
    // ASP.NET function that validates the controls on the page
    // prior to a postback.
    //
    // Am I missing other important trigger function(s) to look for?
    // Please leave me feedback:
    // http://code.google.com/p/jquery-watermark/issues/list
            triggerFns = [
                "Page_ClientValidate"
            ],

    // Holds a value of true if a watermark was displayed since the last
    // hideAll() was executed. Avoids repeatedly calling hideAll().
            pageDirty = false;

// Extends jQuery with a custom selector - ":data(...)"
// :data(<name>)  Includes elements that have a specific name defined in the jQuery data collection. (Only the existence of the name is checked; the value is ignored.)
// :data(<name>=<value>)  Includes elements that have a specific jQuery data name defined, with a specific value associated with it.
// :data(<name>!=<value>)  Includes elements that have a specific jQuery data name defined, with a value that is not equal to the value specified.
// :data(<name>^=<value>)  Includes elements that have a specific jQuery data name defined, with a value that starts with the value specified.
// :data(<name>$=<value>)  Includes elements that have a specific jQuery data name defined, with a value that ends with the value specified.
// :data(<name>*=<value>)  Includes elements that have a specific jQuery data name defined, with a value that contains the value specified.
    $.extend($.expr[":"], {
        "search":function (elem) {
            return "search" === (elem.type || "");
        },

        "data":function (element, index, matches, set) {
            var data, parts = /^((?:[^=!^$*]|[!^$*](?!=))+)(?:([!^$*]?=)(.*))?$/.exec(matches[3]);

            if (parts) {
                data = $(element).data(parts[1]);

                if (data !== undefined) {

                    if (parts[2]) {
                        data = "" + data;

                        switch (parts[2]) {
                            case "=":
                                return (data == parts[3]);
                            case "!=":
                                return (data != parts[3]);
                            case "^=":
                                return (data.slice(0, parts[3].length) == parts[3]);
                            case "$=":
                                return (data.slice(-parts[3].length) == parts[3]);
                            case "*=":
                                return (data.indexOf(parts[3]) !== -1);
                        }
                    }

                    return true;
                }
            }

            return false;
        }
    });

    $.watermark = {

        // Current version number of the plugin
        version:"3.0.6",

        // Default options used when watermarks are instantiated.
        // Can be changed to affect the default behavior for all
        // new or updated watermarks.
        // BREAKING CHANGE:  The $.watermark.className
        // property that was present prior to version 3.0.2 must
        // be changed to $.watermark.options.className
        options:{

            // Default class name for all watermarks
            className:"watermark",

            // If true, plugin will detect and use native browser support for
            // watermarks, if available. (e.g., WebKit's placeholder attribute.)
            useNative:true
        },

        // Hide one or more watermarks by specifying any selector type
        // i.e., DOM element, string selector, jQuery matched set, etc.
        hide:function (selector) {
            $(selector).filter(selWatermarkDefined).each(
                    function () {
                        $.watermark._hide($(this));
                    }
            );
        },

        // Internal use only.
        _hide:function ($input, focus) {
            var inputVal = $input.val() || "",
                    inputWm = $input.data(dataText) || "",
                    maxLen = $input.data(dataMaxLen) || 0,
                    className = $input.data(dataClass);

            if ((inputWm.length) && (inputVal == inputWm)) {
                $input.val("");

                // Password type?
                if ($input.data(dataPassword)) {

                    if (($input.attr("type") || "") === "text") {
                        var $pwd = $input.data(dataPassword) || [],
                                $wrap = $input.parent() || [];

                        if (($pwd.length) && ($wrap.length)) {
                            $wrap[0].removeChild($input[0]); // Can't use jQuery methods, because they destroy data
                            $wrap[0].appendChild($pwd[0]);
                            $input = $pwd;
                        }
                    }
                }

                if (maxLen) {
                    $input.attr("maxLength", maxLen);
                    $input.removeData(dataMaxLen);
                }

                if (focus) {
                    $input.attr("autocomplete", "off");  // Avoid NS_ERROR_XPC_JS_THREW_STRING error in Firefox

                    window.setTimeout(
                            function () {
                                $input.select();  // Fix missing cursor in IE
                            }
                            , 1);
                }
            }

            className && $input.removeClass(className);
        },

        // Display one or more watermarks by specifying any selector type
        // i.e., DOM element, string selector, jQuery matched set, etc.
        // If conditions are not right for displaying a watermark, ensures that watermark is not shown.
        show:function (selector) {
            $(selector).filter(selWatermarkDefined).each(
                    function () {
                        $.watermark._show($(this));
                    }
            );
        },

        // Internal use only.
        _show:function ($input) {
            var val = $input.val() || "",
                    text = $input.data(dataText) || "",
                    type = $input.attr("type") || "",
                    className = $input.data(dataClass);

            if (((val.length == 0) || (val == text)) && (!$input.data(dataFocus))) {
                pageDirty = true;

                // Password type?
                if ($input.data(dataPassword)) {

                    if (type === "password") {
                        var $pwd = $input.data(dataPassword) || [],
                                $wrap = $input.parent() || [];

                        if (($pwd.length) && ($wrap.length)) {
                            $wrap[0].removeChild($input[0]); // Can't use jQuery methods, because they destroy data
                            $wrap[0].appendChild($pwd[0]);
                            $input = $pwd;
                            $input.attr("maxLength", text.length);
                        }
                    }
                }

                // Ensure maxLength big enough to hold watermark (input of type="text" or type="search" only)
                if ((type === "text") || (type === "search")) {
                    var maxLen = $input.attr("maxLength") || 0;

                    if ((maxLen > 0) && (text.length > maxLen)) {
                        $input.data(dataMaxLen, maxLen);
                        $input.attr("maxLength", text.length);
                    }
                }

                className && $input.addClass(className);
                $input.val(text);
            }
            else {
                $.watermark._hide($input);
            }
        },

        // Hides all watermarks on the current page.
        hideAll:function () {
            if (pageDirty) {
                $.watermark.hide(selWatermarkAble);
                pageDirty = false;
            }
        },

        // Displays all watermarks on the current page.
        showAll:function () {
            $.watermark.show(selWatermarkAble);
        }
    };

    $.fn.watermark = function (text, options) {
        ///	<summary>
        ///		Set watermark text and class name on all input elements of type="text/password/search" and
        /// 	textareas within the matched set. If className is not specified in options, the default is
        /// 	"watermark". Within the matched set, only input elements with type="text/password/search"
        /// 	and textareas are affected; all other elements are ignored.
        ///	</summary>
        ///	<returns type="jQuery">
        ///		Returns the original jQuery matched set (not just the input and texarea elements).
        /// </returns>
        ///	<param name="text" type="String">
        ///		Text to display as a watermark when the input or textarea element has an empty value and does not
        /// 	have focus. The first time watermark() is called on an element, if this argument is empty (or not
        /// 	a String type), then the watermark will have the net effect of only changing the class name when
        /// 	the input or textarea element's value is empty and it does not have focus.
        ///	</param>
        ///	<param name="options" type="Object" optional="true">
        ///		Provides the ability to override the default watermark options ($.watermark.options). For backward
        /// 	compatibility, if a string value is supplied, it is used as the class name that overrides the class
        /// 	name in $.watermark.options.className. Properties include:
        /// 		className: When the watermark is visible, the element will be styled using this class name.
        /// 		useNative (Boolean or Function): Specifies if native browser support for watermarks will supersede
        /// 			plugin functionality. If useNative is a function, the return value from the function will
        /// 			determine if native support is used. The function is passed one argument -- a jQuery object
        /// 			containing the element being tested as the only element in its matched set -- and the DOM
        /// 			element being tested is the object on which the function is invoked (the value of "this").
        ///	</param>
        /// <remarks>
        ///		The effect of changing the text and class name on an input element is called a watermark because
        ///		typically light gray text is used to provide a hint as to what type of input is required. However,
        ///		the appearance of the watermark can be something completely different: simply change the CSS style
        ///		pertaining to the supplied class name.
        ///
        ///		The first time watermark() is called on an element, the watermark text and class name are initialized,
        ///		and the focus and blur events are hooked in order to control the display of the watermark.  Also, as
        /// 	of version 3.0, drag and drop events are hooked to guard against dropped text being appended to the
        /// 	watermark.  If native watermark support is provided by the browser, it is detected and used, unless
        /// 	the useNative option is set to false.
        ///
        ///		Subsequently, watermark() can be called again on an element in order to change the watermark text
        ///		and/or class name, and it can also be called without any arguments in order to refresh the display.
        ///
        ///		For example, after changing the value of the input or textarea element programmatically, watermark()
        /// 	should be called without any arguments to refresh the display, because the change event is only
        /// 	triggered by user actions, not by programmatic changes to an input or textarea element's value.
        ///
        /// 	The one exception to programmatic updates is for password input elements:  you are strongly cautioned
        /// 	against changing the value of a password input element programmatically (after the page loads).
        /// 	The reason is that some fairly hairy code is required behind the scenes to make the watermarks bypass
        /// 	IE security and switch back and forth between clear text (for watermarks) and obscured text (for
        /// 	passwords).  It is *possible* to make programmatic changes, but it must be done in a certain way, and
        /// 	overall it is not recommended.
        /// </remarks>

        if (!this.length) {
            return this;
        }

        var hasClass = false,
                hasText = (typeof(text) === "string");

        if (typeof(options) === "object") {
            hasClass = (typeof(options.className) === "string");
            options = $.extend({}, $.watermark.options, options);
        }
        else if (typeof(options) === "string") {
            hasClass = true;
            options = $.extend({}, $.watermark.options, {className:options});
        }
        else {
            options = $.watermark.options;
        }

        if (typeof(options.useNative) !== "function") {
            options.useNative = options.useNative ? function () {
                return true;
            } : function () {
                return false;
            };
        }

        return this.each(
                function () {
                    var $input = $(this);

                    if (!$input.is(selWatermarkAble)) {
                        return;
                    }

                    // Watermark already initialized?
                    if ($input.data(dataFlag)) {

                        // If re-defining text or class, first remove existing watermark, then make changes
                        if (hasText || hasClass) {
                            $.watermark._hide($input);

                            if (hasText) {
                                $input.data(dataText, text);
                            }

                            if (hasClass) {
                                $input.data(dataClass, options.className);
                            }
                        }
                    }
                    else {

                        // Detect and use native browser support, if enabled in options
                        if (options.useNative.call(this, $input)) {

                            // Placeholder attribute (WebKit)
                            // Big thanks to Opera for the wacky test required

                            if ((("" + $input.css("-webkit-appearance")).replace("undefined", "") !== "") && ((($input.attr("tagName") || "") !== "TEXTAREA")) && $input.size() > 0 && $input[0].tagName !== "TEXTAREA") {

                                // className is not set because WebKit doesn't appear to have
                                // a separate class name property for placeholders (watermarks).
                                if (hasText) {
                                    $input.attr("placeholder", text);
                                }

                                // Only set data flag for non-native watermarks (purposely commented-out)
                                // $input.data(dataFlag, 1);
                                return;
                            }
                        }

                        $input.data(dataText, hasText ? text : "");
                        $input.data(dataClass, options.className);
                        $input.data(dataFlag, 1); // Flag indicates watermark was initialized

                        // Special processing for password type
                        if (($input.attr("type") || "") === "password") {
                            var $wrap = $input.wrap("<span>").parent(),
                                    $wm = $($wrap.html().replace(/type=["']?password["']?/i, 'type="text"'));

                            $wm.data(dataText, $input.data(dataText));
                            $wm.data(dataClass, $input.data(dataClass));
                            $wm.data(dataFlag, 1);
                            $wm.attr("maxLength", text.length);

                            $wm.focus(
                                    function () {
                                        $.watermark._hide($wm, true);
                                    }
                            ).bind("dragenter",
                                    function () {
                                        $.watermark._hide($wm);
                                    }
                            ).bind("dragend",
                                    function () {
                                        window.setTimeout(function () {
                                            $wm.blur();
                                        }, 1);
                                    }
                            );
                            $input.blur(
                                    function () {
                                        $.watermark._show($input);
                                    }
                            ).bind("dragleave",
                                    function () {
                                        $.watermark._show($input);
                                    }
                            );

                            $wm.data(dataPassword, $input);
                            $input.data(dataPassword, $wm);
                        }
                        else {

                            $input.focus(
                                    function () {
                                        $input.data(dataFocus, 1);
                                        $.watermark._hide($input, true);
                                    }
                            ).blur(
                                    function () {
                                        $input.data(dataFocus, 0);
                                        $.watermark._show($input);
                                    }
                            ).bind("dragenter",
                                    function () {
                                        $.watermark._hide($input);
                                    }
                            ).bind("dragleave",
                                    function () {
                                        $.watermark._show($input);
                                    }
                            ).bind("dragend",
                                    function () {
                                        window.setTimeout(function () {
                                            $.watermark._show($input);
                                        }, 1);
                                    }
                            ).bind("drop",
                                    // Firefox makes this lovely function necessary because the dropped text
                                    // is merged with the watermark before the drop event is called.
                                    function (evt) {
                                        var dropText = evt.originalEvent.dataTransfer.getData("Text");

                                        if ($input.val().replace(dropText, "") === $input.data(dataText)) {
                                            $input.val(dropText);
                                        }

                                        $input.focus();
                                    }
                            );
                        }

                        // In order to reliably clear all watermarks before form submission,
                        // we need to replace the form's submit function with our own
                        // function.  Otherwise watermarks won't be cleared when the form
                        // is submitted programmatically.
                        if (this.form) {
                            var form = this.form,
                                    $form = $(form);

                            if (!$form.data(dataFormSubmit)) {
                                $form.submit($.watermark.hideAll);

                                // form.submit exists for all browsers except Google Chrome
                                // (see "else" below for explanation)
                                if (form.submit) {
                                    $form.data(dataFormSubmit, form.onsubmit || 1);

                                    form.onsubmit = (function (f, $f) {
                                        return function () {
                                            var nativeSubmit = $f.data(dataFormSubmit);
                                            $.watermark.hideAll();
                                            if (nativeSubmit instanceof Function) {
                                                nativeSubmit();
                                            } else {
                                                eval(nativeSubmit);
                                            }
                                        };
                                    })(form, $form);
                                } else {
                                    $form.data(dataFormSubmit, 1);

                                    // This strangeness is due to the fact that Google Chrome's
                                    // form.submit function is not visible to JavaScript (identifies
                                    // as "undefined").  I had to invent a solution here because hours
                                    // of Googling (ironically) for an answer did not turn up anything
                                    // useful.  Within my own form.submit function I delete the form's
                                    // submit function, and then call the non-existent function --
                                    // which, in the world of Google Chrome, still exists.
                                    form.submit = (function (f) {
                                        return function () {
                                            $.watermark.hideAll();
                                            delete f.submit;
                                            f.submit();
                                        };
                                    })(form);
                                }
                            }
                        }
                    }

                    $.watermark._show($input);
                }
        );
    };

// Hijack any functions found in the triggerFns list
    if (triggerFns.length) {

        // Wait until DOM is ready before searching
        $(function () {
            var i, name, fn;

            for (i = triggerFns.length - 1; i >= 0; i--) {
                name = triggerFns[i];
                fn = window[name];

                if (typeof(fn) === "function") {
                    window[name] = (function (origFn) {
                        return function () {
                            $.watermark.hideAll();
                            return origFn.apply(null, Array.prototype.slice.call(arguments));
                        };
                    })(fn);
                }
            }
        });
    }

})(jQuery);
;

// resource: org.richfaces:datatable.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.DataTable = function(id, options) {
        $super.constructor.call(this, id);
        this.options = $.extend(this.options, options || {});
        this.element = this.attachToDom();
        var self = this;
        var header = $(this.element).find('.rf-dt-thd');
        header.find(".rf-dt-c-srt").each(function() {
            $(this).bind("click", {sortHandle: this}, $.proxy(self.sortHandler, self));
        });
        header.find(".rf-dt-flt-i").each(function() {
            $(this).bind("blur", {filterHandle: this}, $.proxy(self.filterHandler, self));
            $(this).bind("keyup", {filterHandle: this}, $.proxy(self.filterHandler, self));
        });
        $(this.element).trigger("rich:ready", this);
    };

    rf.BaseComponent.extend(rf.ui.DataTable);
    var $super = rf.ui.DataTable.$super;

    $.extend(rf.ui.DataTable, {
            SORTING: "rich:sorting",
            FILTERING: "rich:filtering",
            SUBTABLE_SELECTOR:".rf-cst"
        });

    $.extend(rf.ui.DataTable.prototype, ( function () {

        var invoke = function(event, attributes) {
            rf.ajax(this.id, event, {"parameters" : attributes});
        };

        var createParameters = function(type, id, arg1, arg2) {
            var parameters = {};
            var key = this.id + type;
            parameters[key] = (id + ":" + (arg1 || "") + ":" + arg2);

            var eventOptions = this.options.ajaxEventOption;
            for (key in eventOptions) {
                if (eventOptions.hasOwnProperty(key) && !parameters[key]) {
                    parameters[key] = eventOptions[key];
                }
            }
            return parameters;
        };


        return {

            name : "RichFaces.ui.DataTable",

            sort: function(columnId, direction, isClear) {
                invoke.call(this, null, createParameters.call(this, rf.ui.DataTable.SORTING, columnId, direction, isClear));
            },

            clearSorting: function() {
                this.sort("", "", true);
            },

            sortHandler: function(event) {
                var sortHandle = $(event.data.sortHandle);
                var button = sortHandle.find('.rf-dt-srt-btn');
                var columnId = button.data('columnid');
                var sortOrder = button.hasClass('rf-dt-srt-asc') ? 'descending' : 'ascending';
                this.sort(columnId, sortOrder, false);
            },

            filter: function(columnId, filterValue, isClear) {
                invoke.call(this, null, createParameters.call(this, rf.ui.DataTable.FILTERING, columnId, filterValue, isClear));
            },

            clearFiltering: function() {
                this.filter("", "", true);
            },

            filterHandler: function(event) {
                if (event.type == "keyup" && event.keyCode != 13) {
                    return;
                }
                var filterHandle = $(event.data.filterHandle);
                var columnId = filterHandle.data('columnid');
                var filterValue = filterHandle.val();
                this.filter(columnId, filterValue, false);
            },

            expandAllSubTables: function() {
                this.invokeOnSubTables('expand');
            },

            collapseAllSubTables: function() {
                this.invokeOnSubTables('collapse');
            },

            switchSubTable: function(id) {
                this.getSubTable(id).switchState();
            },

            getSubTable: function(id) {
                return rf.component(id);
            },

            invokeOnSubTables: function(funcName) {
                var elements = $(document.getElementById(this.id)).children(rf.ui.DataTable.SUBTABLE_SELECTOR);
                var invokeOnComponent = this.invokeOnComponent;
                elements.each(
                    function() {
                        if (this.firstChild && this.firstChild[rf.RICH_CONTAINER] && this.firstChild[rf.RICH_CONTAINER].component) {
                            var component = this.firstChild[rf.RICH_CONTAINER].component;
                            if (component instanceof RichFaces.ui.CollapsibleSubTable) {
                                invokeOnComponent(component, funcName);
                            }
                        }
                    }
                );
            },

            invokeOnSubTable: function(id, funcName) {
                var subtable = this.getSubTable(id);
                this.invokeOnComponent(subtable, funcName);
            },

            invokeOnComponent: function(component, funcName) {
                if (component) {
                    var func = component[funcName];
                    if (typeof func == 'function') {
                        func.call(component);
                    }
                }
            },

            __getMenuSelector: function (menu) {
                var selector = "[id='" + this.element.id + "'] ";
                selector += (typeof menu.options.targetSelector === 'undefined')
                    ?  ".rf-dt-b td" : menu.options.targetSelector;
                selector = $.trim(selector);
                return selector;
            },

            contextMenuAttach: function (menu) {
                var selector = this.__getMenuSelector(menu);
                rf.Event.bind(selector, menu.options.showEvent, $.proxy(menu.__showHandler, menu), menu);
            },

            contextMenuDetach: function (menu) {
                var selector = this.__getMenuSelector(menu);
                rf.Event.unbind(selector, menu.options.showEvent);
            },

            destroy: function() {
                $super.destroy.call(this);
            }
        }

    })());

})(RichFaces.jQuery, window.RichFaces);

;

// resource: org.richfaces:log.js
(function($, rf) {
    var logLevels = ['debug', 'info', 'warn', 'error'];
    var logLevelsPadded = {'debug': 'debug', 'info': 'info ', 'warn': 'warn ', 'error': 'error'};
    var logLevelValues = {'debug': 1, 'info': 2, 'warn': 3, 'error': 4};

    var logClassMethods = {

        __import: function(doc, node) {
            if (doc === document) {
                return node;
            }

            var result = $();
            for (var i = 0; i < node.length; i++) {
                if (doc.importNode) {
                    result = result.add(doc.importNode(node[i], true));
                } else {
                    var container = doc.createElement("div");
                    container.innerHTML = node[i].outerHTML;
                    for (var child = container.firstChild; child; child = child.nextSibling) {
                        result = result.add(child);
                    }
                }
            }

            return result;
        },

        __getStyles: function() {
            var head = $("head");

            if (head.length == 0) {
                return "";
            }

            try {
                //TODO - BASE element support?
                var clonedHead = head.clone();
                if (clonedHead.children().length == head.children().length) {
                    return clonedHead.children(":not(style):not(link[rel='stylesheet'])").remove().end().html();
                } else {
                    var result = new Array();
                    head.children("style, link[rel='stylesheet']").each(function() {
                        result.push(this.outerHTML);
                    });

                    return result.join('');
                }
            } catch (e) {
                return "";
            }
        },

        __openPopup: function() {
            if (!this.__popupWindow || this.__popupWindow.closed) {
                this.__popupWindow = open("", "_richfaces_logWindow", "height=400, width=600, resizable = yes, status=no, " +
                    "scrollbars = yes, statusbar=no, toolbar=no, menubar=no, location=no");

                var doc = this.__popupWindow.document;

                doc.write("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">" +
                    "<html xmlns=\"http://www.w3.org/1999/xhtml\"><head>" + this.__getStyles() + "</head>" +
                    "<body onunload='window.close()'><div id='richfaces.log' clas='rf-log rf-log-popup'></div></body></html>");
                doc.close();
                this.__initializeControls(doc);
            } else {
                this.__popupWindow.focus();
            }
        },

        __hotkeyHandler: function(event) {
            if (event.ctrlKey && event.shiftKey) {
                if ((this.hotkey || 'l').toLowerCase() == String.fromCharCode(event.keyCode).toLowerCase()) {
                    this.__openPopup();
                }
            }
        },

        __getTimeAsString: function() {
            var date = new Date();

            var timeString = this.__lzpad(date.getHours(), 2) + ':' + this.__lzpad(date.getMinutes(), 2) + ':' +
                this.__lzpad(date.getSeconds(), 2) + '.' + this.__lzpad(date.getMilliseconds(), 3);

            return timeString;
        },

        __lzpad: function(s, length) {
            s = s.toString();
            var a = new Array();
            for (var i = 0; i < length - s.length; i++) {
                a.push('0');
            }
            a.push(s);
            return a.join('');
        },

        __getMessagePrefix: function(level) {
            return logLevelsPadded[level] + '[' + this.__getTimeAsString() + ']: ';
        },

        __setLevelFromSelect: function(event) {
            this.setLevel(event.target.value);
        },

        __initializeControls : function(doc) {
            var console = $("#richfaces\\.log", doc);

            var clearBtn = console.children("button.rf-log-element");
            if (clearBtn.length == 0) {
                clearBtn = $("<button type='button' name='clear' class='rf-log-element'>Clear</button>", doc).appendTo(console);
            }

            clearBtn.click($.proxy(this.clear, this));

            var levelSelect = console.children("select.rf-log-element");
            if (levelSelect.length == 0) {
                levelSelect = $("<select class='rf-log-element' name='richfaces.log' />", doc).appendTo(console);
            }

            if (levelSelect.children().length == 0) {
                for (var l = 0; l < logLevels.length; l++) {
                    $("<option value='" + logLevels[l] + "'>" + logLevels[l] + "</option>", doc).appendTo(levelSelect);
                }
            }

            levelSelect.val(this.getLevel());
            levelSelect.change($.proxy(this.__setLevelFromSelect, this));

            var consoleEntries = console.children(".rf-log-contents");
            if (consoleEntries.length == 0) {
                consoleEntries = $("<div class='rf-log-contents'></div>", doc).appendTo(console);
            }
            this.__contentsElement = consoleEntries;
        },

        __append: function(element) {
            var target = this.__contentsElement;
            if (this.mode == "popup") {
                var doc = this.__popupWindow.document;
                $(doc.createElement("div")).appendTo(target).append(this.__import(doc, element));
            } else {
                $(document.createElement("div")).appendTo(target).append(element);
            }
        },

        __log: function(level, message) {
            var configuredLevel = this.getLevel();

            if (!logLevelValues[configuredLevel]) {
                // unknown log level
                if (console.log) {
                  console.log('Warning: unknown log level "' + this.getLevel() + '" - using log level "debug"');
                }
                configuredLevel = 'debug';
            }
            if (logLevelValues[level] < logLevelValues[configuredLevel]) {
                // message is not loggable due to its level
                return;
            }

            if (this.mode == 'console') {
                var logMsg = 'RichFaces: ' + message;
                if (console[level]) {
                    if (message instanceof $) {
                        // spans containing the XML response, print only the response
                        var msg = "RichFaces: ";
                        for (var i = 0; i < message.length; i++ ) {
                            msg += message[i].textContent;
                        }
                        console[level](msg)
                    } else if (typeof message == "object") {
                        // print object as object, not string
                        console[level](message);
                    } else {
                        console[level](logMsg);
                    }
                } else if (console.log) {
                    if (message instanceof $) {
                        var msg = "RichFaces: ";
                        for (var i = 0; i < message.length; i++ ) {
                            msg += message[i].textContent;
                        }
                        console.log(msg);
                    } else if (typeof message == "object") {
                        console.log(message);
                    } else {
                        console.log(logMsg);
                    }
                }
                return;
            }
            
            if (!this.__contentsElement) {
                return;
            }

            var newEntry = $();
            newEntry = newEntry.add($("<span class='rf-log-entry-lbl rf-log-entry-lbl-" + level + "'></span>").text(this.__getMessagePrefix(level)));

            var entrySpan = $("<span class='rf-log-entry-msg rf-log-entry-msg-" + level + "'></span>");
            if (typeof message != 'object' || !message.appendTo) {
                entrySpan.text(message);
            } else {
                message.appendTo(entrySpan);
            }

            newEntry = newEntry.add(entrySpan);

            this.__append(newEntry);
        },

        init: function(options) {
            $super.constructor.call(this, 'richfaces.log');
            this.attachToDom();
            rf.setLog(this);

            options = options || {};

            this.level = (options.level || 'info').toLowerCase();
            this.hotkey = options.hotkey;
            this.mode = (options.mode || 'inline');

            if (this.mode == 'console') {
                // do nothing
            } else if (this.mode == 'popup') {
                this.__boundHotkeyHandler = $.proxy(this.__hotkeyHandler, this);
                $(document).bind('keydown', this.__boundHotkeyHandler);
            } else {
                this.__initializeControls(document);
            }
        },

        destroy: function() {
            rf.setLog(null);

            //TODO test this method
            if (this.__popupWindow) {
                this.__popupWindow.close();
            }
            this.__popupWindow = null;

            if (this.__boundHotkeyHandler) {
                $(document).unbind('keydown', this.__boundHotkeyHandler);
                this.__boundHotkeyHandler = null;
            }

            this.__contentsElement = null;
            $super.destroy.call(this);
        },

        setLevel: function(level) {
            this.level = level;
            this.clear();
        },

        getLevel: function() {
            return this.level || 'info';
        },

        clear: function() {
            if (this.__contentsElement) {
                this.__contentsElement.children().remove();
            }
        }
    };

    for (var i = 0; i < logLevels.length; i++) {
        logClassMethods[logLevels[i]] = (function() {
            var level = logLevels[i];
            return function(message) {
                this.__log(level, message);
            }
        }());
    }

    rf.HtmlLog = rf.BaseComponent.extendClass(logClassMethods);
    // define super class link
    var $super = rf.HtmlLog.$super;

    $(document).ready(function() {
        if (typeof jsf != 'undefined') {
            (function($, rf, jsf) {

                //JSF log adapter
                var identifyElement = function(elt) {
                    var identifier = '<' + elt.tagName.toLowerCase();
                    var e = $(elt);
                    if (e.attr('id')) {
                        identifier += (' id=' + e.attr('id'));
                    }
                    if (e.attr('class')) {
                        identifier += (' class=' + e.attr('class'));
                    }

                    identifier += ' ...>';

                    return identifier;
                }

                var formatPartialResponseElement = function(logElement, responseElement) {
                    var change = $(responseElement);

                    logElement.append("Element <b>" + responseElement.nodeName + "</b>");
                    if (change.attr("id")) {
                        logElement.append(document.createTextNode(" for id=" + change.attr("id")));
                    }

                    $(document.createElement("br")).appendTo(logElement);
                    $("<span class='rf-log-entry-msg-xml'></span>").appendTo(logElement).text(change.toXML());
                    $(document.createElement("br")).appendTo(logElement);
                }

                var formatPartialResponse = function(partialResponse) {
                    var logElement = $(document.createElement("span"));

                    partialResponse.children().each(function() {
                        var responseElement = $(this);
                        if (responseElement.is('changes')) {
                            logElement.append("Listing content of response <b>changes</b> element:<br />");
                            responseElement.children().each(function() {
                                formatPartialResponseElement(logElement, this);
                            });
                        } else {
                            formatPartialResponseElement(logElement, this);
                        }
                    });

                    return logElement;
                }

                var jsfAjaxLogAdapter = function(data) {
                    try {
                        var log = rf.log;

                        var source = data.source;
                        var type = data.type;

                        var responseCode = data.responseCode;
                        var responseXML = data.responseXML;
                        var responseText = data.responseText;

                        if (type != 'error') {
                            log.info("Received '" + type + "' event from " + identifyElement(source));

                            if (type == 'beforedomupdate') {
                                var partialResponse;

                                if (responseXML) {
                                    partialResponse = $(responseXML).children("partial-response");
                                }

                                var responseTextEntry = $("<span>Server returned responseText: </span><span class='rf-log-entry-msg-xml'></span>").eq(1).text(responseText).end();

                                if (partialResponse && partialResponse.length) {
                                    log.debug(responseTextEntry);
                                    log.info(formatPartialResponse(partialResponse));
                                } else {
                                    log.info(responseTextEntry);
                                }
                            }
                        } else {
                            var status = data.status;
                            log.error("Received '" + type + '@' + status + "' event from " + identifyElement(source));

                            var message = "[status=" + data.responseCode + "] ";
                            if (data.errorName && data.errorMessage) {
                                message += " " + data.errorName + ": " + data.errorMessage;
                            } else if (data.description) {
                                message += " " + data.description;
                            } else {
                                message += " no error details";
                            }
                            log.error(message);
                        }
                    } catch (e) {
                        //ignore logging errors
                    }
                };

                var eventsListener = rf.createJSFEventsAdapter({
                        begin: jsfAjaxLogAdapter,
                        beforedomupdate: jsfAjaxLogAdapter,
                        success: jsfAjaxLogAdapter,
                        complete: jsfAjaxLogAdapter,
                        error: jsfAjaxLogAdapter
                    });

                jsf.ajax.addOnEvent(eventsListener);
                jsf.ajax.addOnError(eventsListener);
                //
            }($, rf, jsf));
        }
    });

}(RichFaces.jQuery, RichFaces));;

// resource: org.richfaces:collapsible-subtable.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.CollapsibleSubTable = function(id, f, options) {
        this.id = id;
        this.options = $.extend(this.options, options || {});
        this.stateInput = options.stateInput;
        this.optionsInput = options.optionsInput;
        this.expandMode = options.expandMode || rf.ui.CollapsibleSubTable.MODE_CLNT;
        this.eventOptions = options.eventOptions;
        this.formId = f;
        this.isNested = options.isNested;

        if (! this.isNested) {
            var self = this;
            var tbody = $(document.getElementById(this.id)).parent();
            tbody.find(".rf-dt-c-srt").each(function() {
                $(this).bind("click", {sortHandle: this}, $.proxy(self.sortHandler, self));
            });
            tbody.find(".rf-dt-flt-i").each(function() {
                $(this).bind("blur", {filterHandle: this}, $.proxy(self.filterHandler, self));
                $(this).bind("keyup", {filterHandle: this}, $.proxy(self.filterHandler, self));
            });

        }

        this.attachToDom();
    };

    $.extend(rf.ui.CollapsibleSubTable, {
            MODE_AJAX: "ajax",
            MODE_SRV: "server",
            MODE_CLNT: "client",
            collapse: 0,
            expand: 1,
            SORTING: "rich:sorting",
            FILTERING: "rich:filtering"
        });

    rf.BaseComponent.extend(rf.ui.CollapsibleSubTable);
    var $super = rf.ui.CollapsibleSubTable.$super;

    $.extend(rf.ui.CollapsibleSubTable.prototype, (function () {

        var invoke = function(event, attributes) {
            rf.ajax(this.id, event, {"parameters" : attributes});
        };

        var createParameters = function(type, id, arg1, arg2) {
            var parameters = {};
            var key = this.id + type;
            parameters[key] = (id + ":" + (arg1 || "") + ":" + arg2);

            var eventOptions = this.options.ajaxEventOption;
            for (key in eventOptions) {
                if (eventOptions.hasOwnProperty(key) && !parameters[key]) {
                    parameters[key] = eventOptions[key];
                }
            }
            return parameters;
        };

        var element = function() {
                // use parent tbody as parent dom elem
            return $(document.getElementById(this.id)).parent();
        };
        
        var nestedElements = function() {
                // return children rows - id contains the id of parent
            var childrenRegex = new RegExp("^" + this.id + "\\:\\d+\\:");
            return $(document.getElementById(this.id)).parent().find("tr").filter(function() {
                return this.id.match(childrenRegex);
            });
        };

        var stateInputElem = function() {
            return $(document.getElementById(this.stateInput));
        };

        var optionsInputElem = function() {
            return $(document.getElementById(this.optionsInput));
        };

        var ajax = function(e, options) {
            this.__switchState();
            rf.ajax(this.id, e, options);
        };

        var server = function(options) {
            this.__switchState();
            $(document.getElementById(this.formId)).submit();
        };

        var client = function(options) {
            if (this.isExpanded()) {
                this.collapse(options);
            } else {
                this.expand(options);
            }
        };


        return {

            name: "CollapsibleSubTable",

            sort: function(columnId, direction, isClear) {
                invoke.call(this, null, createParameters.call(this, rf.ui.CollapsibleSubTable.SORTING, columnId, direction, isClear));
            },

            clearSorting: function() {
                this.sort("", "", true);
            },

            sortHandler: function(event) {
                var sortHandle = $(event.data.sortHandle);
                var button = sortHandle.find('.rf-dt-srt-btn');
                var columnId = button.data('columnid');
                var sortOrder = button.hasClass('rf-dt-srt-asc') ? 'descending' : 'ascending';
                this.sort(columnId, sortOrder, false);
            },

            filter: function(columnId, filterValue, isClear) {
                invoke.call(this, null, createParameters.call(this, rf.ui.CollapsibleSubTable.FILTERING, columnId, filterValue, isClear));
            },

            clearFiltering: function() {
                this.filter("", "", true);
            },

            filterHandler: function(event) {
                if (event.type == "keyup" && event.keyCode != 13) {
                    return;
                }
                var filterHandle = $(event.data.filterHandle);
                var columnId = filterHandle.data('columnid');
                var filterValue = filterHandle.val();
                this.filter(columnId, filterValue, false);
            },

            switchState: function(e, options) {
                if (this.expandMode == rf.ui.CollapsibleSubTable.MODE_AJAX) {
                    ajax.call(this, e, this.eventOptions, options);
                } else if (this.expandMode == rf.ui.CollapsibleSubTable.MODE_SRV) {
                    server.call(this, options);
                } else if (this.expandMode == rf.ui.CollapsibleSubTable.MODE_CLNT) {
                    client.call(this, options);
                }
            },

            collapse: function(options) {
                this.setState(rf.ui.CollapsibleSubTable.collapse);
                if (this.isNested) {
                    nestedElements.call(this).hide();
                }
                else {
                    element.call(this).hide();
                }
            },

            expand: function(options) {
                this.setState(rf.ui.CollapsibleSubTable.expand);
                
                if (!this.isNested) {
                    element.call(this).show();
                }
                else {
                        // return first level children only
                    var subtableRegex = new RegExp("^" + this.id + "\\:\\d+\\:[^\\:]+$");
                    nestedElements.call(this).filter(function () {
                        return this.id.match(subtableRegex);
                    }).each(function() {
                        if (this.rf) { // is a nested subtable, show content if necessary
                            if (this.rf.component.isExpanded()) {
                                this.rf.component.expand();
                            }
                        }
                        else { // is a subtable toggler, always show
                            $(this).show();
                        }
                    });
                }
            },

            isExpanded: function() {
                return (parseInt(this.getState()) == rf.ui.CollapsibleSubTable.expand);
            },

            __switchState: function(options) {
                var state = this.isExpanded() ? rf.ui.CollapsibleSubTable.collapse : rf.ui.CollapsibleSubTable.expand;
                this.setState(state);
            },

            getState: function() {
                return stateInputElem.call(this).val();
            },

            setState: function(state) {
                stateInputElem.call(this).val(state)
            },

            setOption: function(option) {
                optionsInputElem.call(this).val(option);
            },

            getMode: function() {
                return this.expandMode;
            },
            destroy: function() {
                $super.destroy.call(this);
            }
        };

    })());

})(RichFaces.jQuery, window.RichFaces);
;

// resource: org.richfaces:poll.js
(function($, rf) {
    rf.ui = rf.ui || {};
    var defaultOptions = {
    };
    rf.ui.Poll = function(componentId, options) {
        $super.constructor.call(this, componentId, options);
        this.id = componentId;
        this.attachToDom();
        this.interval = options.interval || 1000;
        this.ontimer = options.ontimer;

        this.pollElement = rf.getDomElement(this.id);

        rf.ui.pollTracker = rf.ui.pollTracker || {};

        if (options.enabled) {
            this.startPoll();
        }
    }

    rf.BaseComponent.extend(rf.ui.Poll);
    var $super = rf.ui.Poll.$super;
    $.extend(rf.ui.Poll.prototype, (function() {
        return {
            name: "Poll",

            startPoll: function() {
                this.stopPoll();
                var poll = this;
                rf.ui.pollTracker[poll.id] = window.setTimeout(function() {
                    try {
                        poll.ontimer.call(poll.pollElement || window);
                        poll.startPoll();
                    } catch (e) {
                        // TODO: handle exception
                    }
                }, poll.interval);
            },

            stopPoll : function() {
                if (rf.ui.pollTracker && rf.ui.pollTracker[this.id]) {
                    window.clearTimeout(rf.ui.pollTracker[this.id]);
                    delete rf.ui.pollTracker[this.id];
                }
            },

            setZeroRequestDelay : function(options) {
                if (typeof options.requestDelay == "undefined") {
                    options.requestDelay = 0;
                }
            },

            destroy : function() {
                this.stopPoll();
                this.detach(this.id);
                // call parent's destroy method
                $super.destroy.call(this);
            }

        };
    })());

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:toolbar.js
(function ($, rf) {

    rf.ui = rf.ui || {};

  rf.ui.toolbarHandlers = function(options) {
      if (options.id && options.events) {
          $('.rf-tb-itm', document.getElementById(options.id)).bind(
              options.events);
      }
      var groups = options.groups;
      if (groups && groups.length > 0) {
          var group;
          for (var i = 0; i < groups.length; i++) {
              group = groups[i];
              if (group) {
                  var groupIds = group.ids;
                  var y;
                  var groupElements = [];
                  for (var y = 0; y < groupIds.length; y++) {
                      groupElements.push(document.getElementById(groupIds[y]));
                  }
                  $(groupElements).bind(group.events);
              }
          }
      }
  }

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:richfaces-event.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

/**
 * @author Pavel Yaschenko
 */

// TODO: add support to bind multiple events using type param as an object with eventType,function pairs // see bindById method
// TODO: update js docs 

window.RichFaces = window.RichFaces || {};
RichFaces.jQuery = RichFaces.jQuery || window.jQuery;

(function($, rf) {

    /**
     * RichFaces Event API container
     * @class
     * @memberOf RichFaces
     * @static
     * @name Event
     * */
    rf.Event = rf.Event || {};

    var getEventElement = function (selector) {
        if (!selector) {
            throw "RichFaces.Event: empty selector";
        }
        var element;
        if (rf.BaseComponent && selector instanceof rf.BaseComponent) {
            element = $(rf.getDomElement(selector.getEventElement()));
        } else {
            element = $(selector);
        }

        return element;
    }

    var getHandlerWrapper = function (component, fn) {
        return function (e, d) {
            if (!e[rf.RICH_CONTAINER]) {
                e[rf.RICH_CONTAINER] = {data: d};
            }
            return fn.call(component || this, e, this, d);
        };
    }

    var getMultipleHandlerWrapper = function (object, component) {
        var result = {};
        for (var type in object) {
            if (object.hasOwnProperty(type)) {
                result[type] = getHandlerWrapper(component, object[type]);
            }
        }
        return result;
    }

    $.extend(rf.Event, {
            /**
             * @constant
             * @name RichFaces.Event.RICH_NAMESPACE
             * @type string
             * */
            RICH_NAMESPACE : "RICH",

            /**
             * @constant
             * @name RichFaces.Event.EVENT_NAMESPACE_SEPARATOR
             * @type string
             * */
            EVENT_NAMESPACE_SEPARATOR : ".",

            MESSAGE_EVENT_TYPE: "onmessage",

            /**
             * Attach an event handler to execute when the DOM is fully loaded.
             *
             * @function
             * @name RichFaces.Event.ready
             * @param {function} fn - event handler
             * @return {jQuery} document element wrapped by jQuery
             * */
            ready : function(fn) {
                // TODO: not completed yet
                return $(document).ready(fn);
                /*
                 function callback(jQueryReference) {
                 this; // document
                 }
                 */
            },

            /**
             * Attach a handler to an event for the elements.
             * @function
             * @name RichFaces.Event.bind
             *
             * @param {string|DOMElement|jQuery} selector - jQuery elements selector
             * @param {string} eventType - one or more JavaScript event types, such as "click" or "submit," or custom event names
             * @param {function} fn - event handler
             * @param {Object} [data] - component or object with additional data
             * It is a context for an event handler
             * @return {function} function that binded to the element's event
             * */
            bind : function(selector, eventType, fn, component, data) {
                // eventType: namespace can be used, like onclick.rf.conponentName
                if (typeof eventType == "object") {
                    // in this case fn == component object
                    getEventElement(selector).bind(getMultipleHandlerWrapper(eventType, fn), data);
                } else {
                    var f = getHandlerWrapper(component, fn);
                    getEventElement(selector).bind(eventType, data, f);
                    return f;
                }
            },

            /**
             * Attach a handler to an event for the element by element id.
             * @function
             * @name RichFaces.Event.bindById
             *
             * @param {string} id - DOM element id
             * @param {string} eventType - one or more JavaScript event types, such as "click" or "submit," or custom event names
             * @param {function} fn - event handler
             * @param {Object} [data] - component or object with additional data
             * It is a context for an event handler
             * @return {function} function that binded to the element's event
             * */
            bindById : function(id, eventType, fn, component, data) {
                // eventType: namespace can be used, like onclick.rf.conponentName
                if (typeof eventType == "object") {
                    // in this case fn == component object
                    $(document.getElementById(id)).bind(getMultipleHandlerWrapper(eventType, fn), data);
                } else {
                    var f = getHandlerWrapper(component, fn);
                    $(document.getElementById(id)).bind(eventType, data, f);
                }
                return f;
            },

            /**
             * Attach a handler to an event for the elements.
             * The handler will be called only once when event happened.
             * @function
             * @name RichFaces.Event.bindOne
             *
             * @param {string|DOMElement|jQuery} selector - jQuery elements selector
             * @param {string} eventType - one or more JavaScript event types, such as "click" or "submit," or custom event names
             * @param {function} fn - event handler
             * @param {Object} [data] - component or object with additional data
             * It is a context for an event handler
             * @return {function} function that binded to the element's event
             * */
            bindOne: function(selector, eventType, fn, component, data) {
                // eventType: namespace can be used, like onclick.rf.conponentName
                var f = getHandlerWrapper(component, fn);
                getEventElement(selector).one(eventType, data, f);
                return f;
            },

            /**
             * Attach a handler to an event for the element by element id.
             * The handler will be called only once when event happened.
             * @function
             * @name RichFaces.Event.bindOneById
             *
             * @param {string} id - DOM element id
             * @param {string} eventType - one or more JavaScript event types, such as "click" or "submit," or custom event names
             * @param {function} fn - event handler
             * @param {Object} [data] - component or object with additional data
             * It is a context for an event handler
             * @return {function} function that binded to the element's event
             * */
            bindOneById: function(id, eventType, fn, component, data) {
                // eventType: namespace can be used, like onclick.rf.conponentName
                var f = getHandlerWrapper(component, fn);
                $(document.getElementById(id)).one(eventType, data, f);
                return f;
            },

            /**
             * Remove a previously-attached event handler from the elements.
             * @function
             * @name RichFaces.Event.unbind
             *
             * @param {string|DOMElement|jQuery} selector - jQuery elements selector
             * @param {string} [eventType] - one or more JavaScript event types, such as "click" or "submit," or custom event names
             * @param {function} [fn] - event handler
             * @return {jQuery} element wrapped by jQuery
             * */
            unbind : function(selector, eventType, fn) {
                // eventType: namespace can be used, like onclick.rf.conponentName
                return getEventElement(selector).unbind(eventType, fn);
            },

            /**
             * Remove a previously-attached event handler from the elements by element id.
             * The handler will be called only once when event happened.
             * @function
             * @name RichFaces.Event.unbindById
             *
             * @param {string} id - DOM element id
             * @param {string} [eventType] - one or more JavaScript event types, such as "click" or "submit," or custom event names
             * @param {function} [fn] - event handler
             * @return {jQuery} element wrapped by jQuery
             * */
            unbindById : function(id, eventType, fn) {
                // eventType: namespace can be used, like onclick.rf.conponentName
                return $(document.getElementById(id)).unbind(eventType, fn);
            },

            // TODO add jsdocs and qunits
            bindScrollEventHandlers: function(element, handler, component) {
                var elements = [];
                element = rf.getDomElement(element).parentNode;
                while (element && element != window.document.body) {
                    if (element.offsetWidth != element.scrollWidth || element.offsetHeight != element.scrollHeight) {
                        elements.push(element);
                        rf.Event.bind(element, "scroll" + component.getNamespace(), handler, component);
                    }
                    element = element.parentNode;
                }
                return elements;
            },
            unbindScrollEventHandlers: function(elements, component) {
                rf.Event.unbind(elements, "scroll" + component.getNamespace());
            },

            /**
             * Execute all handlers and behaviors attached to the matched elements for the given event type.
             * @function
             * @name RichFaces.Event.fire
             *
             * @param {string|DOMElement|jQuery} selector - jQuery elements selector
             * @param {string} eventType - event type
             * @param {Object} [data] - a object of additional parameters to pass to the event handler
             * @return {jQuery} element wrapped by jQuery
             * */
            fire : function(selector, eventType, data) {
                var event = $.Event(eventType);
                getEventElement(selector).trigger(event, [data]);
                return !event.isDefaultPrevented();
            },

            /**
             * The same as the fire method, but selects element by id.
             * @function
             * @name RichFaces.Event.fireById
             *
             * @param {string} id - DOM element id
             * @param {string} eventType - event type
             * @param {Object} [data] - a object of additional parameters to pass to the event handler
             * @return {jQuery} element wrapped by jQuery
             * */
            fireById : function(id, eventType, data) {
                var event = $.Event(eventType);
                $(document.getElementById(id)).trigger(event, [data]);
                return !event.isDefaultPrevented();
            },

            /**
             * The same as the fire method, but:
             *  - does not cause the default behavior of an event to occur
             *  - does not bubble up event
             *  - call handler only for the first founded element
             *  - returns whatever value that was returned by the handler
             * @function
             * @name RichFaces.Event.callHandler
             *
             * @param {string|DOMElement|jQuery} selector - jQuery elements selector
             * @param {string} eventType - event type
             * @param {Object} [data] - a object of additional parameters to pass to the event handler
             * @return value that was returned by the handler
             * */
            callHandler : function(selector, eventType, data) {
                return getEventElement(selector).triggerHandler(eventType, [data]);
            },

            /**
             * The same as the callHandler method, but selects element by id.
             * @function
             * @name RichFaces.Event.callHandlerById
             *
             * @param {string} id - DOM element id
             * @param {string} eventType - event type
             * @param {Object} [data] - a object of additional parameters to pass to the event handler
             * @return value that was returned by the handler
             *  */
            callHandlerById : function(id, eventType, data) {
                return $(document.getElementById(id)).triggerHandler(eventType, [data]);
            },

            /**
             * Create an event namespace for the components.
             * @function
             * @name RichFaces.Event.createNamespace
             *
             * @param {string} [componentName] - component name
             * @param {string} [id] - element id
             * @param {string} [prefix=RichFaces.Event.RICH_NAMESPACE] - namespace prefix
             * @return {string} namespace string
             *  */
                // TODO: rename argument names
            createNamespace : function(componentName, id, prefix) {
                var a = [];
                a.push(prefix || rf.Event.RICH_NAMESPACE);
                if (componentName) {
                    a.push(componentName);
                }
                if (id) {
                    a.push(id);
                }
                return a.join(rf.Event.EVENT_NAMESPACE_SEPARATOR);
            }
        });

})(RichFaces.jQuery, RichFaces);

/*
 fn : function (eventObject, element) {
 this; // object passed as data to bind function or dom element if no data
 element; // dom element

 }
 */

// 	API usage example:
// 		RichFaces.Event.bind(selector, type, fn, data);
;

// resource: org.richfaces:placeholder.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    var defaultOptions = {
        useNative: false
    };

    rf.ui.Placeholder = rf.BaseComponent.extendClass({

        name:"Placeholder",

        init: function (componentId, options) {
            $super.constructor.call(this, componentId);
            options = $.extend({}, defaultOptions, options);
            this.attachToDom(this.id);
            $(function() {
                options.className = 'rf-plhdr ' + ((options.styleClass) ? options.styleClass : '');
                var elements = (options.selector) ? $(options.selector) : $(document.getElementById(options.targetId));
                // finds all inputs within the subtree of target elements
                var inputs = elements.find('*').andSelf().filter(':editable');
                inputs.watermark(options.text, options);
            });
        },
        // destructor definition
        destroy: function () {
            // define destructor if additional cleaning is needed but
            // in most cases its not nessesary.
            // call parent’s destructor
            $super.destroy.call(this);
        }
    });
    
    // once per all placeholders on a page
    $(function() {
        $(document).on('ajaxsubmit', 'form', $.watermark.hideAll);
        $(document).on('ajaxbegin', 'form', $.watermark.showAll);
            // need to use setTimeout to allow client's native reset to happen
        $(document).on('reset', 'form', function() {setTimeout( $.watermark.showAll, 0); });
    });
    
    // define super class reference - reference to the parent prototype
    var $super = rf.ui.Placeholder.$super;
})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:notifyMessage.js
(function($, rf) {

    rf.ui = rf.ui || {};

    // Constructor definition
    rf.ui.NotifyMessage = function(componentId, options, notifyOptions) {
        // call constructor of parent class
        $super.constructor.call(this, componentId, options, defaultOptions);
        this.notifyOptions = notifyOptions;
    };

    // Extend component class and add protected methods from parent class to our container
    rf.ui.Base.extend(rf.ui.NotifyMessage);

    // define super class link
    var $super = rf.ui.NotifyMessage.$super;

    var defaultOptions = {
        showSummary:true,
        level:0,
        isMessages: false,
        globalOnly: false
    };


    var onMessage = function (event, element, data) {
        var sourceId = data.sourceId;
        var message = data.message;
        if (!this.options.forComponentId) {
            if (!this.options.globalOnly && message) {
                renderMessage.call(this, sourceId, message);
            }
        } else if (this.options.forComponentId === sourceId) {
            renderMessage.call(this, sourceId, message);
        }
    }

    var renderMessage = function(index, message) {
        if (message && message.severity >= this.options.level) {
            showNotification.call(this, message);
        }
    }
    
    var showNotification = function(message) {
        rf.ui.Notify($.extend({}, this.notifyOptions, {
            'summary': this.options.showSummary ? message.summary : undefined,
            'detail': this.options.showDetail ? message.detail : undefined,
            'severity': message.severity
        }));
    }

    var bindEventHandlers = function () {
        rf.Event.bind(window.document, rf.Event.MESSAGE_EVENT_TYPE + this.namespace, onMessage, this);
    };

    $.extend(rf.ui.NotifyMessage.prototype, {
            name: "NotifyMessage",
            __bindEventHandlers: bindEventHandlers,
            
            destroy : function() {
                rf.Event.unbind(window.document, rf.Event.MESSAGE_EVENT_TYPE + this.namespace);
                $super.destroy.call(this);
            }
        });

})(RichFaces.jQuery, window.RichFaces || (window.RichFaces = {}));;

// resource: com.jqueryui:core.js
/*!
 * jQuery UI Core 1.11.4
 * http://jqueryui.com
 *
 * Copyright jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * http://api.jqueryui.com/category/ui-core/
 */
(function( factory ) {
	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [ "jquery" ], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
}(function( $ ) {

// $.ui might exist from components with no dependencies, e.g., $.ui.position
$.ui = $.ui || {};

$.extend( $.ui, {
	version: "1.11.4",

	keyCode: {
		BACKSPACE: 8,
		COMMA: 188,
		DELETE: 46,
		DOWN: 40,
		END: 35,
		ENTER: 13,
		ESCAPE: 27,
		HOME: 36,
		LEFT: 37,
		PAGE_DOWN: 34,
		PAGE_UP: 33,
		PERIOD: 190,
		RIGHT: 39,
		SPACE: 32,
		TAB: 9,
		UP: 38
	}
});

// plugins
$.fn.extend({
	scrollParent: function( includeHidden ) {
		var position = this.css( "position" ),
			excludeStaticParent = position === "absolute",
			overflowRegex = includeHidden ? /(auto|scroll|hidden)/ : /(auto|scroll)/,
			scrollParent = this.parents().filter( function() {
				var parent = $( this );
				if ( excludeStaticParent && parent.css( "position" ) === "static" ) {
					return false;
				}
				return overflowRegex.test( parent.css( "overflow" ) + parent.css( "overflow-y" ) + parent.css( "overflow-x" ) );
			}).eq( 0 );

		return position === "fixed" || !scrollParent.length ? $( this[ 0 ].ownerDocument || document ) : scrollParent;
	},

	uniqueId: (function() {
		var uuid = 0;

		return function() {
			return this.each(function() {
				if ( !this.id ) {
					this.id = "ui-id-" + ( ++uuid );
				}
			});
		};
	})(),

	removeUniqueId: function() {
		return this.each(function() {
			if ( /^ui-id-\d+$/.test( this.id ) ) {
				$( this ).removeAttr( "id" );
			}
		});
	}
});

// selectors
function focusable( element, isTabIndexNotNaN ) {
	var map, mapName, img,
		nodeName = element.nodeName.toLowerCase();
	if ( "area" === nodeName ) {
		map = element.parentNode;
		mapName = map.name;
		if ( !element.href || !mapName || map.nodeName.toLowerCase() !== "map" ) {
			return false;
		}
		img = $( "img[usemap='#" + mapName + "']" )[ 0 ];
		return !!img && visible( img );
	}
	return ( /^(input|select|textarea|button|object)$/.test( nodeName ) ?
		!element.disabled :
		"a" === nodeName ?
			element.href || isTabIndexNotNaN :
			isTabIndexNotNaN) &&
		// the element and all of its ancestors must be visible
		visible( element );
}

function visible( element ) {
	return $.expr.filters.visible( element ) &&
		!$( element ).parents().addBack().filter(function() {
			return $.css( this, "visibility" ) === "hidden";
		}).length;
}

$.extend( $.expr[ ":" ], {
	data: $.expr.createPseudo ?
		$.expr.createPseudo(function( dataName ) {
			return function( elem ) {
				return !!$.data( elem, dataName );
			};
		}) :
		// support: jQuery <1.8
		function( elem, i, match ) {
			return !!$.data( elem, match[ 3 ] );
		},

	focusable: function( element ) {
		return focusable( element, !isNaN( $.attr( element, "tabindex" ) ) );
	},

	tabbable: function( element ) {
		var tabIndex = $.attr( element, "tabindex" ),
			isTabIndexNaN = isNaN( tabIndex );
		return ( isTabIndexNaN || tabIndex >= 0 ) && focusable( element, !isTabIndexNaN );
	}
});

// support: jQuery <1.8
if ( !$( "<a>" ).outerWidth( 1 ).jquery ) {
	$.each( [ "Width", "Height" ], function( i, name ) {
		var side = name === "Width" ? [ "Left", "Right" ] : [ "Top", "Bottom" ],
			type = name.toLowerCase(),
			orig = {
				innerWidth: $.fn.innerWidth,
				innerHeight: $.fn.innerHeight,
				outerWidth: $.fn.outerWidth,
				outerHeight: $.fn.outerHeight
			};

		function reduce( elem, size, border, margin ) {
			$.each( side, function() {
				size -= parseFloat( $.css( elem, "padding" + this ) ) || 0;
				if ( border ) {
					size -= parseFloat( $.css( elem, "border" + this + "Width" ) ) || 0;
				}
				if ( margin ) {
					size -= parseFloat( $.css( elem, "margin" + this ) ) || 0;
				}
			});
			return size;
		}

		$.fn[ "inner" + name ] = function( size ) {
			if ( size === undefined ) {
				return orig[ "inner" + name ].call( this );
			}

			return this.each(function() {
				$( this ).css( type, reduce( this, size ) + "px" );
			});
		};

		$.fn[ "outer" + name] = function( size, margin ) {
			if ( typeof size !== "number" ) {
				return orig[ "outer" + name ].call( this, size );
			}

			return this.each(function() {
				$( this).css( type, reduce( this, size, true, margin ) + "px" );
			});
		};
	});
}

// support: jQuery <1.8
if ( !$.fn.addBack ) {
	$.fn.addBack = function( selector ) {
		return this.add( selector == null ?
			this.prevObject : this.prevObject.filter( selector )
		);
	};
}

// support: jQuery 1.6.1, 1.6.2 (http://bugs.jquery.com/ticket/9413)
if ( $( "<a>" ).data( "a-b", "a" ).removeData( "a-b" ).data( "a-b" ) ) {
	$.fn.removeData = (function( removeData ) {
		return function( key ) {
			if ( arguments.length ) {
				return removeData.call( this, $.camelCase( key ) );
			} else {
				return removeData.call( this );
			}
		};
	})( $.fn.removeData );
}

// deprecated
$.ui.ie = !!/msie [\w.]+/.exec( navigator.userAgent.toLowerCase() );

$.fn.extend({
	focus: (function( orig ) {
		return function( delay, fn ) {
			return typeof delay === "number" ?
				this.each(function() {
					var elem = this;
					setTimeout(function() {
						$( elem ).focus();
						if ( fn ) {
							fn.call( elem );
						}
					}, delay );
				}) :
				orig.apply( this, arguments );
		};
	})( $.fn.focus ),

	disableSelection: (function() {
		var eventType = "onselectstart" in document.createElement( "div" ) ?
			"selectstart" :
			"mousedown";

		return function() {
			return this.bind( eventType + ".ui-disableSelection", function( event ) {
				event.preventDefault();
			});
		};
	})(),

	enableSelection: function() {
		return this.unbind( ".ui-disableSelection" );
	},

	zIndex: function( zIndex ) {
		if ( zIndex !== undefined ) {
			return this.css( "zIndex", zIndex );
		}

		if ( this.length ) {
			var elem = $( this[ 0 ] ), position, value;
			while ( elem.length && elem[ 0 ] !== document ) {
				// Ignore z-index if position is set to a value where z-index is ignored by the browser
				// This makes behavior of this function consistent across browsers
				// WebKit always returns auto if the element is positioned
				position = elem.css( "position" );
				if ( position === "absolute" || position === "relative" || position === "fixed" ) {
					// IE returns 0 when zIndex is not specified
					// other browsers return a string
					// we ignore the case of nested elements with an explicit value of 0
					// <div style="z-index: -10;"><div style="z-index: 0;"></div></div>
					value = parseInt( elem.css( "zIndex" ), 10 );
					if ( !isNaN( value ) && value !== 0 ) {
						return value;
					}
				}
				elem = elem.parent();
			}
		}

		return 0;
	}
});

// $.ui.plugin is deprecated. Use $.widget() extensions instead.
$.ui.plugin = {
	add: function( module, option, set ) {
		var i,
			proto = $.ui[ module ].prototype;
		for ( i in set ) {
			proto.plugins[ i ] = proto.plugins[ i ] || [];
			proto.plugins[ i ].push( [ option, set[ i ] ] );
		}
	},
	call: function( instance, name, args, allowDisconnected ) {
		var i,
			set = instance.plugins[ name ];

		if ( !set ) {
			return;
		}

		if ( !allowDisconnected && ( !instance.element[ 0 ].parentNode || instance.element[ 0 ].parentNode.nodeType === 11 ) ) {
			return;
		}

		for ( i = 0; i < set.length; i++ ) {
			if ( instance.options[ set[ i ][ 0 ] ] ) {
				set[ i ][ 1 ].apply( instance.element, args );
			}
		}
	}
};

}));
;

// resource: org.richfaces:richfaces-utils.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

// RichFaces.utils
// RichFaces.utils.cache
(function ($, rf) {
    rf.utils = rf.utils || {};

    rf.utils.Cache = function (key, items, values, useCache) {
        this.key = key.toLowerCase();
        this.cache = {};
        this.cache[this.key] = items || [];
        this.originalValues = typeof values == "function" ? values(items) : values || this.cache[this.key];
        this.values = processValues(this.originalValues);
        this.useCache = useCache || checkValuesPrefix.call(this);
    };

    var processValues = function (values) {
        var processedValues = [];
        for (var i = 0; i < values.length; i++) {
            processedValues.push(values[i].toLowerCase());
        }
        return processedValues;
    };

    var checkValuesPrefix = function () {
        var result = true;
        for (var i = 0; i < this.values.length; i++) {
            if (this.values[i].indexOf(this.key) != 0) {
                result = false;
                break;
            }
        }
        return result;
    };

    var getItems = function (key, filterFunction) {
        key = key.toLowerCase();
        var newCache = [];

        if (key.length < this.key.length) {
            return newCache;
        }

        if (this.cache[key]) {
            newCache = this.cache[key];
        } else {
            var useCustomFilterFunction = typeof filterFunction == "function";
            var itemsCache = this.cache[this.key];
            for (var i = 0; i < this.values.length; i++) {
                var value = this.values[i];
                if (useCustomFilterFunction && filterFunction(key, value)) {
                    newCache.push(itemsCache[i]);
                } else {
                    var p = value.indexOf(key);
                    if (p == 0) {
                        newCache.push(itemsCache[i]);
                    }
                }
            }

            if ((!this.lastKey || key.indexOf(this.lastKey) != 0) && newCache.length > 0) {
                this.cache[key] = newCache;
                if (newCache.length == 1) {
                    this.lastKey = key;
                }
            }
        }

        return newCache;
    };

    var getItemValue = function (item) {
        return this.originalValues[this.cache[this.key].index(item)];
    };

    var isCached = function (key) {
        key = key.toLowerCase();
        return this.cache[key] || this.useCache && key.indexOf(this.key) == 0;
    };

    $.extend(rf.utils.Cache.prototype, (function () {
        return  {
            getItems: getItems,
            getItemValue: getItemValue,
            isCached: isCached
        };
    })());

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:togglePanel.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    /* SIMPLE INNER CLASS for handle switch operation*/
    function SwitchItems(comp) {
        this.comp = comp;
    }

    SwitchItems.prototype = {

        /**
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {void}
         * */
        exec : function (oldPanel, newPanel) {
            if (newPanel.switchMode == "server") {
                return this.execServer(oldPanel, newPanel);
            } else if (newPanel.switchMode == "ajax") {
                return this.execAjax(oldPanel, newPanel);
            } else if (newPanel.switchMode == "client") {
                return this.execClient(oldPanel, newPanel);
            } else {
                rf.log.error("SwitchItems.exec : unknown switchMode (" + this.comp.switchMode + ")");
            }
        },

        /**
         * @protected
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {Boolean} false
         * */
        execServer : function (oldPanel, newPanel) {
            if (oldPanel) {
                var continueProcess = oldPanel.__leave();
                if (!continueProcess) {
                    return false;
                }
            }

            this.__setActiveItem(newPanel.getName());

            rf.submitForm(this.__getParentForm());

            return false;
        },

        /**
         * @protected
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {Boolean} false
         * */
        execAjax : function (oldPanel, newPanel) {
            var options = $.extend({}, this.comp.options["ajax"], {}/*this.getParameters(newPanel)*/);

            this.__setActiveItem(newPanel.getName());
            rf.ajax(this.comp.id, null, options);

            if (oldPanel) {
                this.__setActiveItem(oldPanel.getName());
            }

            return false;
        },

        /**
         * @protected
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {undefined}
         *             - false - if process has been terminated
         *             - true  - in other cases
         * */
        execClient : function (oldPanel, newPanel) {
            if (oldPanel) {
                var continueProcess = oldPanel.__leave();
                if (!continueProcess) {
                    return false;
                }
            }

            this.__setActiveItem(newPanel.getName());

            newPanel.__enter();
            this.comp.__fireItemChange(oldPanel, newPanel);

            return true;
        },

        /**
         * @private
         * */
        __getParentForm : function () {
            return $(rf.getDomElement(this.comp.id)).parents('form:first');
        },

        /**
         * @private
         * */
        __setActiveItem : function (name) {
            rf.getDomElement(this.__getValueInputId()).value = name;
            this.comp.activeItem = name;
        },

        /**
         * @private
         * */
        __getValueInputId: function () {
            return this.comp.id + "-value"
        }
    };

    /**
     * @class TogglePanel
     * @name TogglePanel
     *
     * @constructor
     * @param {String} componentId - component id
     * @param {Hash} options - params
     * */
    rf.ui.TogglePanel = rf.BaseComponent.extendClass({

            // class name
            name:"TogglePanel",

            init : function (componentId, options) {
                $super.constructor.call(this, componentId);
                this.attachToDom();

                this.items = [];

                this.options = $.extend(this.options, options || {});
                this.activeItem = this.options.activeItem;

                this.__addUserEventHandler("itemchange");
                this.__addUserEventHandler("beforeitemchange");
            },

            /***************************** Public Methods  ****************************************************************/

            /**
             * @methodOf
             *
             * @name TogglePanel#getSelectItem
             *
             * @return {String} name of current selected panel item
             */
            getSelectItem: function () {
                return this.activeItem;
            },

            /**
             * @methodOf
             * @name TogglePanel#switchToItem
             *
             * @param {String} name - panel item name to switch
             *           we can use meta names @first, @prev, @next and @last
             * @return {Boolean} - false if something wrong and true if all is ok
             */
            switchToItem: function (name) {
                var newPanel = this.getNextItem(name);
                if (newPanel == null) {
                    rf.log.warn("TogglePanel.switchToItems(" + name + "): item with name '" + name + "' not found");
                    return false;
                }

                var oldPanel = this.__getItemByName(this.getSelectItem());

                var continueProcess = this.__fireBeforeItemChange(oldPanel, newPanel);
                if (!continueProcess) {
                    rf.log.warn("TogglePanel.switchToItems(" + name + "): switch has been canceled by beforeItemChange event");
                    return false
                }

                return this.__itemsSwitcher().exec(oldPanel, newPanel);
            },

            /**
             * @methodOf
             * @name TogglePanel#getNextItem
             *
             * @param {String} name of TogglePanelItem or meta name (@first | @prev | @next | @last)
             * @return {TogglePanelItem} null if item not found
             */
            getNextItem : function (name) {
                if (name) {
                    var newItemIndex = this.__ITEMS_META_NAMES[name];
                    if (newItemIndex) {
                        return this.__getItem(newItemIndex(this));
                    } else {
                        return this.__getItemByName(name);
                    }
                } else {
                    return this.__getItemByName(this.nextItem());
                }
            },

            /**
             * please, remove this method when client side ajax events will be added
             *
             * */
            onCompleteHandler : function (newItemName) {
                var oldItem = this.__getItemByName(this.activeItem);
                var newItem = this.__getItemByName(newItemName);

                // Don't do like this and remove it ASAP
                this.__itemsSwitcher().execClient(oldItem, newItem);
                $(document.getElementById(newItem.getTogglePanel().id)).trigger("resize");
            },

            /**
             * @methodOf
             * @name TogglePanel#getItems
             *
             * @return {TogglePanelItem[]} all defined panel items
             */
            getItems : function () {
                return this.items;
            },

            /**
             * @methodOf
             * @name TogglePanel#getItemsNames
             *
             * @return {String[]} names of all defined items
             */
            getItemsNames: function () {
                var res = [];
                for (var i = 0; i < this.items.length; i++) {
                    res.push(this.items[i].getName());
                }

                return res;
            },

            /**
             * @methodOf
             * @name TogglePanel#nextItem
             *
             * @param {String} [itemName = activeItem]
             * @return {String} name of next panel item
             */
            nextItem: function (itemName) {
                var itemIndex = this.__getItemIndex(itemName || this.activeItem);
                if (itemIndex == -1) {
                    return null;
                }

                return this.__getItemName(itemIndex + 1);
            },

            /**
             * @methodOf
             * @name TogglePanel#firstItem
             *
             * @return {String} name of first panel item
             */
            firstItem: function () {
                return this.__getItemName(0);
            },

            /**
             * @methodOf
             * @name TogglePanel#lastItem
             *
             * @return {String} name of last panel item
             */
            lastItem: function () {
                return this.__getItemName(this.items.length - 1);
            },

            /**
             * @methodOf
             * @name TogglePanel#prevItem
             *
             * @param {String} itemName
             * @return {String} name of prev panel item
             *                  null if it is first item
             */
            prevItem: function (itemName) {
                var itemIndex = this.__getItemIndex(itemName || this.activeItem);
                if (!this.options.cycledSwitching && itemIndex < 1) {
                    return null;
                }

                return this.__getItemName(itemIndex - 1);
            },

            /////////////////////////////////////////////////////////////////////////////////
            //// Private
            /////////////////////////////////////////////////////////////////////////////////

            /********************* Methods *************************/

            __itemsSwitcher : function () {
                return new SwitchItems(this);
            },

            __ITEMS_META_NAMES : (function () {
                function goFrom(comp, ind, step) {
                    var res = ind;
                    while ((!comp.items[res] || comp.items[res].disabled) && res < comp.items.length && res > 0) {
                        res += step;
                    }
                    return res;
                }


                return {
                    "@first" : function (comp) {
                        return goFrom(comp, 0, 1);
                    },

                    "@prev"  : function (comp) {
                        return goFrom(comp, parseInt(comp.__getItemIndex(comp.activeItem)) - 1, -1);
                    },

                    "@next"  : function (comp) {
                        return goFrom(comp, parseInt(comp.__getItemIndex(comp.activeItem)) + 1, 1);
                    },

                    "@last"  : function (comp) {
                        return goFrom(comp, comp.items.length - 1, -1);
                    }
                }
            })(),

            /**
             * @private
             * */
            __getItemIndex : function (itemName) {
                var item;
                for (var i = 0; i < this.items.length; i++) {
                    item = this.items[i];
                    if (!item.disabled && item.getName() == itemName) {
                        return i;
                    }
                }

                rf.log.info("TogglePanel.getItemIndex: item with name '" + itemName + "' not found");
                return -1;
            },

            /**
             * @private
             * */
            __addUserEventHandler : function (name) {
                var handler = this.options["on" + name];
                if (handler) {
                    rf.Event.bindById(this.id, name, handler);
                }
            },

            /**
             * @private
             * @param {Number} index - array index
             *
             * @return {TogglePanelItem}
             *    null - if item not found
             * */
            __getItem : function (index) {
                if (this.options.cycledSwitching) {
                    var size = this.items.length;
                    return this.items[(size + index) % size]
                } else if (index >= 0 && index < this.items.length) {
                    return this.items[index]
                } else {
                    return null;
                }
            },

            __getItemByName : function (name) {
                return this.__getItem(this.__getItemIndex(name));
            },

            __getItemName : function (index) {
                var item = this.__getItem(index);
                if (item == null) {
                    return null;
                }

                return item.getName();
            },

            /**
             * Fire Concealable Event
             * */

            __fireItemChange : function (oldItem, newItem) {
                return new rf.Event.fireById(this.id, "itemchange", {
                        id: this.id,
                        oldItem : oldItem,
                        newItem : newItem
                    });
            },

            __fireBeforeItemChange : function (oldItem, newItem) {
                return rf.Event.fireById(this.id, "beforeitemchange", {
                        id: this.id,
                        oldItem : oldItem,
                        newItem : newItem
                    });
            }
        });

    // define super class link
    var $super = rf.ui.TogglePanel.$super;
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:tree.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
(function($, rf) {

    var NEW_NODE_TOGGLE_STATE = "__NEW_NODE_TOGGLE_STATE";

    var TRIGGER_NODE_AJAX_UPDATE = "__TRIGGER_NODE_AJAX_UPDATE";

    var SELECTION_STATE = "__SELECTION_STATE";

    var TREE_CLASSES = ["rf-tr-nd-colps", "rf-tr-nd-exp"];

    var TREE_HANDLE_CLASSES = ["rf-trn-hnd-colps", "rf-trn-hnd-exp"];

    var TREE_ICON_CLASSES = ["rf-trn-ico-colps", "rf-trn-ico-exp"];

    rf.ui = rf.ui || {};

    rf.ui.TreeNode = rf.BaseComponent.extendClass({

            name: "TreeNode",

            init: function (id, commonOptions) {
                $superTreeNode.constructor.call(this, id);
                this.__rootElt = $(this.attachToDom());

                this.__children = new Array();

                this.__initializeChildren(commonOptions);

                var handlers = (commonOptions.clientEventHandlers || {})[this.getId().substring(commonOptions.treeId.length)] || {};

                if (handlers.bth) {
                    rf.Event.bind(this.__rootElt, "beforetoggle", new Function("event", handlers.bth));
                }

                if (handlers.th) {
                    rf.Event.bind(this.__rootElt, "toggle", new Function("event", handlers.th));
                }

                this.__addLastNodeClass();
            },

            destroy: function() {

                if (this.parent) {
                    this.parent.removeChild(this);
                    this.parent = null;
                }

                this.__clientToggleStateInput = null;

                this.__clearChildren();

                this.__rootElt = null;

                $superTreeNode.destroy.call(this);
            },

            __initializeChildren: function(commonOptions) {
                var _this = this;
                this.__rootElt.children(".rf-tr-nd").each(function() {
                    _this.addChild(new rf.ui.TreeNode(this, commonOptions));
                });
            },

            __addLastNodeClass: function() {
                if (this.__rootElt.next("div").length == 0) {
                    this.__rootElt.addClass("rf-tr-nd-last");
                }
            },

            __getNodeContainer: function() {
                return this.__rootElt.find(" > .rf-trn:first");
            },

            __getHandle: function() {
                return this.__getNodeContainer().find(" > .rf-trn-hnd:first");
            },

            __getContent: function() {
                return this.__getNodeContainer().find(" > .rf-trn-cnt:first");
            },

            __getIcons: function() {
                return this.__getContent().find(" > .rf-trn-ico");
            },

            getParent: function() {
                return this.__parent;
            },

            setParent: function(newParent) {
                this.__parent = newParent;
            },

            addChild: function(child, idx) {
                var start;
                if (typeof idx != 'undefined') {
                    start = idx;
                } else {
                    start = this.__children.length;
                }

                this.__children.splice(start, 0, child);
                child.setParent(this);
            },

            removeChild: function(child) {
                if (this.__children.length) {
                    var idx = this.__children.indexOf(child);
                    if (idx != -1) {
                        var removedChildren = this.__children.splice(idx, 1);
                        if (removedChildren) {
                            for (var i = 0; i < removedChildren.length; i++) {
                                removedChildren[i].setParent(undefined);
                            }
                        }
                    }
                }
            },

            __clearChildren: function() {
                for (var i = 0; i < this.__children.length; i++) {
                    this.__children[i].setParent(undefined);
                }

                this.__children = new Array();
            },

            isExpanded: function() {
                return !this.isLeaf() && this.__rootElt.hasClass("rf-tr-nd-exp");
            },

            isCollapsed: function() {
                return !this.isLeaf() && this.__rootElt.hasClass("rf-tr-nd-colps");
            },

            isLeaf: function() {
                return this.__rootElt.hasClass("rf-tr-nd-lf");
            },

            __canBeToggled: function() {
                return !this.isLeaf() && !this.__rootElt.hasClass("rf-tr-nd-exp-nc") && !this.__loading;
            },

            toggle: function() {
                if (!this.__canBeToggled()) {
                    return;
                }

                if (this.isCollapsed()) {
                    this.expand();
                } else {
                    this.collapse();
                }
            },

            __updateClientToggleStateInput: function(newState) {
                if (!this.__clientToggleStateInput) {
                    this.__clientToggleStateInput = $("<input type='hidden' />").appendTo(this.__rootElt)
                        .attr({name: this.getId() + NEW_NODE_TOGGLE_STATE});
                }

                this.__clientToggleStateInput.val(newState.toString());

            },

            __fireBeforeToggleEvent: function() {
                return rf.Event.callHandler(this.__rootElt, "beforetoggle");
            },

            __fireToggleEvent: function() {
                rf.Event.callHandler(this.__rootElt, "toggle");
            },

            __makeLoading: function() {
                this.__loading = true;
                this.__getNodeContainer().addClass("rf-trn-ldn");
            },

            __resetLoading: function() {
                this.__loading = false;
                this.__getNodeContainer().removeClass("rf-trn-ldn");
            },

            __changeToggleState: function(newState) {
                if (!this.isLeaf()) {
                    if (newState ^ this.isExpanded()) {

                        if (this.__fireBeforeToggleEvent() === false) {
                            return;
                        }

                        var tree = this.getTree();

                        switch (tree.getToggleType()) {
                            case 'client':
                                this.__rootElt.addClass(TREE_CLASSES[newState ? 1 : 0]).removeClass(TREE_CLASSES[!newState ? 1 : 0]);
                                this.__getHandle().addClass(TREE_HANDLE_CLASSES[newState ? 1 : 0]).removeClass(TREE_HANDLE_CLASSES[!newState ? 1 : 0]);

                                var icons = this.__getIcons();
                                if (icons.length == 1) {
                                    icons.addClass(TREE_ICON_CLASSES[newState ? 1 : 0]).removeClass(TREE_ICON_CLASSES[!newState ? 1 : 0]);
                                }

                                this.__updateClientToggleStateInput(newState);
                                this.__fireToggleEvent();
                                break;

                            case 'ajax':
                            case 'server':
                                //TODO - event?
                                tree.__sendToggleRequest(null, this, newState);
                                break;
                        }
                    }
                }
            },

            collapse: function() {
                this.__changeToggleState(false);
            },

            expand: function() {
                this.__changeToggleState(true);
            },

            __setSelected: function(value) {
                var content = this.__getContent();
                if (value) {
                    content.addClass("rf-trn-sel");
                } else {
                    content.removeClass("rf-trn-sel");
                }

                this.__selected = value;
            },

            isSelected: function() {
                return this.__selected;
            },

            getTree: function() {
                return this.getParent().getTree();
            },

            getId: function() {
                return this.__rootElt.attr('id');
            }

        });

    // define super class link for TreeNode
    var $superTreeNode = rf.ui.TreeNode.$super;

    rf.ui.TreeNode.initNodeByAjax = function(nodeId, commonOptions) {
        var node = $(document.getElementById(nodeId));

        var opts = commonOptions || {};

        var parent = node.parent(".rf-tr-nd, .rf-tr");

        var idx = node.prevAll(".rf-tr-nd").length;

        var parentNode = rf.component(parent[0]);
        opts.treeId = parentNode.getTree().getId();

        var newChild = new rf.ui.TreeNode(node[0], opts);
        parentNode.addChild(newChild, idx);

        var tree = parentNode.getTree();

        if (tree.getSelection().contains(newChild.getId())) {
            newChild.__setSelected(true);
        }
    };

    rf.ui.TreeNode.emitToggleEvent = function(nodeId) {
        var node = document.getElementById(nodeId);
        if (!node) {
            return;
        }

        rf.component(node).__fireToggleEvent();
    };

    var findTree = function(elt) {
        return rf.component($(elt).closest(".rf-tr"));
    };

    var findTreeNode = function(elt) {
        return rf.component($(elt).closest(".rf-tr-nd"));
    };

    var isEventForAnotherTree = function(tree, elt) {
        return tree != findTree(elt);
    };

    rf.ui.Tree = rf.ui.TreeNode.extendClass({

            name: "Tree",

            init: function (id, options) {
                this.__treeRootElt = $(rf.getDomElement(id));

                var commonOptions = {};
                commonOptions.clientEventHandlers = options.clientEventHandlers || {};
                commonOptions.treeId = id;

                $superTree.constructor.call(this, this.__treeRootElt, commonOptions);

                this.__toggleType = options.toggleType || 'ajax';
                this.__selectionType = options.selectionType || 'client';

                if (options.ajaxSubmitFunction) {
                    this.__ajaxSubmitFunction = new Function("event", "source", "params", "complete", options.ajaxSubmitFunction);
                }

                if (options.onbeforeselectionchange) {
                    rf.Event.bind(this.__treeRootElt, "beforeselectionchange", new Function("event", options.onbeforeselectionchange));
                }

                if (options.onselectionchange) {
                    rf.Event.bind(this.__treeRootElt, "selectionchange", new Function("event", options.onselectionchange));
                }

                this.__toggleNodeEvent = options.toggleNodeEvent;
                if (this.__toggleNodeEvent) {
                    this.__treeRootElt.delegate(".rf-trn", this.__toggleNodeEvent, this, this.__nodeToggleActivated);
                }
                if (!this.__toggleNodeEvent || this.__toggleNodeEvent != 'click') {
                    this.__treeRootElt.delegate(".rf-trn-hnd", "click", this, this.__nodeToggleActivated);
                }

                this.__treeRootElt.delegate(".rf-trn-cnt", "mousedown", this, this.__nodeSelectionActivated);

                this.__findSelectionInput();
                this.__selection = new rf.ui.TreeNodeSet(this.__selectionInput.val());

                $(document).ready($.proxy(this.__updateSelectionFromInput, this));
            },

            __findSelectionInput: function () {
                this.__selectionInput = $(" > .rf-tr-sel-inp", this.__treeRootElt);
            },

            __addLastNodeClass: function() {
                //stub function overriding parent class method
            },

            destroy: function() {
                if (this.__toggleNodeEvent) {
                    this.__treeRootElt.undelegate(".rf-trn", this.__toggleNodeEvent, this, this.__nodeToggleActivated);
                }
                if (!this.__toggleNodeEvent || this.__toggleNodeEvent != 'click') {
                    this.__treeRootElt.undelegate(".rf-trn-hnd", "click", this, this.__nodeToggleActivated);
                }

                this.__treeRootElt.undelegate(".rf-trn-cnt", "mousedown", this.__nodeSelectionActivated);
                this.__treeRootElt = null;

                this.__selectionInput = null;
                this.__ajaxSubmitFunction = null;
                $superTree.destroy.call(this);
            },

            __nodeToggleActivated: function(event) {
                var theTree = event.data;
                if (isEventForAnotherTree(theTree, this)) {
                    return;
                }

                var treeNode = findTreeNode(this);
                treeNode.toggle();
            },

            __nodeSelectionActivated: function(event) {
                var theTree = event.data;
                if (isEventForAnotherTree(theTree, this)) {
                    return;
                }

                var treeNode = findTreeNode(this);

                if (event.ctrlKey) {
                    theTree.__toggleSelection(treeNode);
                } else {
                    theTree.__addToSelection(treeNode);
                }
            },

            __sendToggleRequest: function(event, toggleSource, newNodeState) {
                var toggleSourceId = toggleSource.getId();

                var clientParams = {};
                clientParams[toggleSourceId + NEW_NODE_TOGGLE_STATE] = newNodeState;

                if (this.getToggleType() == 'server') {
                    var form = this.__treeRootElt.closest('form');
                    rf.submitForm(form, clientParams);
                } else {
                    toggleSource.__makeLoading();
                    clientParams[toggleSourceId + TRIGGER_NODE_AJAX_UPDATE] = newNodeState;
                    this.__ajaxSubmitFunction(event, toggleSourceId, clientParams, function() {
                        var treeNode = rf.component(toggleSourceId);
                        if (treeNode) {
                            treeNode.__resetLoading();
                        }
                    });
                }
            },

            getToggleType: function() {
                return this.__toggleType;
            },

            getSelectionType: function() {
                return this.__selectionType;
            },

            getTree: function() {
                return this;
            },

            __handleSelectionChange: function(newSelection) {
                var eventData = {
                    oldSelection: this.getSelection().getNodes(),
                    newSelection: newSelection.getNodes()
                };

                if (rf.Event.callHandler(this.__treeRootElt, "beforeselectionchange", eventData) === false) {
                    return;
                }

                this.__selectionInput.val(newSelection.getNodeString());

                if (this.getSelectionType() == 'client') {
                    this.__updateSelection(newSelection);
                } else {
                    this.__ajaxSubmitFunction(null, this.getId());
                }
            },

            __toggleSelection: function(node) {
                var newSelection = this.getSelection().cloneAndToggle(node);
                this.__handleSelectionChange(newSelection);
            },

            __addToSelection: function(node) {
                var newSelection = this.getSelection().cloneAndAdd(node);
                this.__handleSelectionChange(newSelection);
            },

            __updateSelectionFromInput: function() {
                this.__findSelectionInput();
                this.__updateSelection(new rf.ui.TreeNodeSet(this.__selectionInput.val()));
            },

            __updateSelection: function(newSelection) {

                var oldSelection = this.getSelection();

                oldSelection.each(function() {
                    this.__setSelected(false)
                });
                newSelection.each(function() {
                    this.__setSelected(true)
                });

                if (oldSelection.getNodeString() != newSelection.getNodeString()) {
                    rf.Event.callHandler(this.__treeRootElt, "selectionchange", {
                            oldSelection: oldSelection.getNodes(),
                            newSelection: newSelection.getNodes()
                        });
                }

                this.__selection = newSelection;
            },

            getSelection: function() {
                return this.__selection;
            },

            __getMenuSelector: function (menu) {
                var selector = "[id='" + this.id[0].id + "'] ";
                selector += (typeof menu.options.targetSelector === 'undefined')
                    ?  ".rf-trn-cnt" : menu.options.targetSelector;
                selector = $.trim(selector);
                return selector;
            },

            contextMenuAttach: function (menu) {
                var selector = this.__getMenuSelector(menu);
                rf.Event.bind(selector, menu.options.showEvent, $.proxy(menu.__showHandler, menu), menu);
            },

            contextMenuDetach: function (menu) {
                var selector = this.__getMenuSelector(menu);
                rf.Event.unbind(selector, menu.options.showEvent);
            }
        });

    // define super class link for Tree
    var $superTree = rf.ui.Tree.$super;

    rf.ui.TreeNodeSet = function() {
        this.init.apply(this, arguments);
    };

    //TODO - that's a single-node set, implement multi-node support!
    $.extend(rf.ui.TreeNodeSet.prototype, {

            init: function(nodeId) {
                this.__nodeId = nodeId;
            },

            contains: function(node) {
                if (node.getId) {
                    return this.__nodeId == node.getId();
                } else {
                    return this.__nodeId == node;
                }
            },

            getNodeString: function() {
                return this.__nodeId;
            },

            toString: function() {
                return this.getNodeString();
            },

            getNodes: function() {
                if (this.__nodeId) {
                    var node = rf.component(this.__nodeId);
                    if (node) {
                        return [node];
                    } else {
                        return null;
                    }
                }

                return [];
            },

            cloneAndAdd: function(node) {
                return new rf.ui.TreeNodeSet(node.getId());
            },

            cloneAndToggle: function(node) {
                var nodeId;
                if (this.contains(node)) {
                    nodeId = "";
                } else {
                    nodeId = node.getId();
                }

                return new rf.ui.TreeNodeSet(nodeId);
            },

            each: function(callback) {
                $.each(this.getNodes() || [], callback);
            }
        });

}(RichFaces.jQuery, RichFaces));;

// resource: net.java.dev.atmosphere:atmosphere.js
/*
 * Copyright 2015 Async-IO.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Atmosphere.js
 * https://github.com/Atmosphere/atmosphere-javascript
 *
 * API reference
 * https://github.com/Atmosphere/atmosphere/wiki/jQuery.atmosphere.js-API
 *
 * Highly inspired by
 * - Portal by Donghwan Kim http://flowersinthesand.github.io/portal/
 */
(function (root, factory) {
    if (typeof define === "function" && define.amd) {
        // AMD
        define(factory);
    } else if(typeof exports !== 'undefined') {
        // CommonJS
        module.exports = factory();
    } else {
        // Browser globals, Window
        root.atmosphere = factory();
    }
}(this, function () {

    "use strict";

    var version = "2.2.12-javascript",
        atmosphere = {},
        guid,
        offline = false,
        requests = [],
        callbacks = [],
        uuid = 0,
        hasOwn = Object.prototype.hasOwnProperty;

    atmosphere = {

        onError: function (response) {
        },
        onClose: function (response) {
        },
        onOpen: function (response) {
        },
        onReopen: function (response) {
        },
        onMessage: function (response) {
        },
        onReconnect: function (request, response) {
        },
        onMessagePublished: function (response) {
        },
        onTransportFailure: function (errorMessage, _request) {
        },
        onLocalMessage: function (response) {
        },
        onFailureToReconnect: function (request, response) {
        },
        onClientTimeout: function (request) {
        },
        onOpenAfterResume: function (request) {
        },

        /**
         * Creates an object based on an atmosphere subscription that exposes functions defined by the Websocket interface.
         *
         * @class WebsocketApiAdapter
         * @param {Object} request the request object to build the underlying subscription
         * @constructor
         */
        WebsocketApiAdapter: function (request) {
            var _socket, _adapter;

            /**
             * Overrides the onMessage callback in given request.
             *
             * @method onMessage
             * @param {Object} e the event object
             */
            request.onMessage = function (e) {
                _adapter.onmessage({data: e.responseBody});
            };

            /**
             * Overrides the onMessagePublished callback in given request.
             *
             * @method onMessagePublished
             * @param {Object} e the event object
             */
            request.onMessagePublished = function (e) {
                _adapter.onmessage({data: e.responseBody});
            };

            /**
             * Overrides the onOpen callback in given request to proxy the event to the adapter.
             *
             * @method onOpen
             * @param {Object} e the event object
             */
            request.onOpen = function (e) {
                _adapter.onopen(e);
            };

            _adapter = {
                close: function () {
                    _socket.close();
                },

                send: function (data) {
                    _socket.push(data);
                },

                onmessage: function (e) {
                },

                onopen: function (e) {
                },

                onclose: function (e) {
                },

                onerror: function (e) {

                }
            };
            _socket = new atmosphere.subscribe(request);

            return _adapter;
        },

        AtmosphereRequest: function (options) {

            /**
             * {Object} Request parameters.
             *
             * @private
             */
            var _request = {
                timeout: 300000,
                method: 'GET',
                headers: {},
                contentType: '',
                callback: null,
                url: '',
                data: '',
                suspend: true,
                maxRequest: -1,
                reconnect: true,
                maxStreamingLength: 10000000,
                lastIndex: 0,
                logLevel: 'info',
                requestCount: 0,
                fallbackMethod: 'GET',
                fallbackTransport: 'streaming',
                transport: 'long-polling',
                webSocketImpl: null,
                webSocketBinaryType: null,
                dispatchUrl: null,
                webSocketPathDelimiter: "@@",
                enableXDR: false,
                rewriteURL: false,
                attachHeadersAsQueryString: true,
                executeCallbackBeforeReconnect: false,
                readyState: 0,
                withCredentials: false,
                trackMessageLength: false,
                messageDelimiter: '|',
                connectTimeout: -1,
                reconnectInterval: 0,
                dropHeaders: true,
                uuid: 0,
                async: true,
                shared: false,
                readResponsesHeaders: false,
                maxReconnectOnClose: 5,
                enableProtocol: true,
                pollingInterval: 0,
                heartbeat: {
                    client: null,
                    server: null
                },
                ackInterval: 0,
                closeAsync: false,
                reconnectOnServerError: true,
                onError: function (response) {
                },
                onClose: function (response) {
                },
                onOpen: function (response) {
                },
                onMessage: function (response) {
                },
                onReopen: function (request, response) {
                },
                onReconnect: function (request, response) {
                },
                onMessagePublished: function (response) {
                },
                onTransportFailure: function (reason, request) {
                },
                onLocalMessage: function (request) {
                },
                onFailureToReconnect: function (request, response) {
                },
                onClientTimeout: function (request) {
                },
                onOpenAfterResume: function (request) {
                }
            };

            /**
             * {Object} Request's last response.
             *
             * @private
             */
            var _response = {
                status: 200,
                reasonPhrase: "OK",
                responseBody: '',
                messages: [],
                headers: [],
                state: "messageReceived",
                transport: "polling",
                error: null,
                request: null,
                partialMessage: "",
                errorHandled: false,
                closedByClientTimeout: false,
                ffTryingReconnect: false
            };

            /**
             * {websocket} Opened web socket.
             *
             * @private
             */
            var _websocket = null;

            /**
             * {SSE} Opened SSE.
             *
             * @private
             */
            var _sse = null;

            /**
             * {XMLHttpRequest, ActiveXObject} Opened ajax request (in case of http-streaming or long-polling)
             *
             * @private
             */
            var _activeRequest = null;

            /**
             * {Object} Object use for streaming with IE.
             *
             * @private
             */
            var _ieStream = null;

            /**
             * {Object} Object use for jsonp transport.
             *
             * @private
             */
            var _jqxhr = null;

            /**
             * {boolean} If request has been subscribed or not.
             *
             * @private
             */
            var _subscribed = true;

            /**
             * {number} Number of test reconnection.
             *
             * @private
             */
            var _requestCount = 0;

            /**
             * The Heartbeat interval send by the server.
             * @type {int}
             * @private
             */
            var _heartbeatInterval = 0;

            /**
             * The Heartbeat bytes send by the server.
             * @type {string}
             * @private
             */
            var _heartbeatPadding = 'X';

            /**
             * {boolean} If request is currently aborted.
             *
             * @private
             */
            var _abortingConnection = false;

            /**
             * A local "channel' of communication.
             *
             * @private
             */
            var _localSocketF = null;

            /**
             * The storage used.
             *
             * @private
             */
            var _storageService;

            /**
             * Local communication
             *
             * @private
             */
            var _localStorageService = null;

            /**
             * A Unique ID
             *
             * @private
             */
            var guid = atmosphere.util.now();

            /** Trace time */
            var _traceTimer;

            /** Key for connection sharing */
            var _sharingKey;

            /**
             * {boolean} If window beforeUnload event has been called.
             * Flag will be reset after 5000 ms
             *
             * @private
             */
            var _beforeUnloadState = false;

            // Automatic call to subscribe
            _subscribe(options);

            /**
             * Initialize atmosphere request object.
             *
             * @private
             */
            function _init() {
                _subscribed = true;
                _abortingConnection = false;
                _requestCount = 0;

                _websocket = null;
                _sse = null;
                _activeRequest = null;
                _ieStream = null;
            }

            /**
             * Re-initialize atmosphere object.
             *
             * @private
             */
            function _reinit() {
                _clearState();
                _init();
            }

            /**
             * Returns true if the given level is equal or above the configured log level.
             *
             * @private
             */
            function _canLog(level) {
                if (level == 'debug') {
                    return _request.logLevel === 'debug';
                } else if (level == 'info') {
                    return _request.logLevel === 'info' || _request.logLevel === 'debug';
                } else if (level == 'warn') {
                    return _request.logLevel === 'warn' || _request.logLevel === 'info' || _request.logLevel === 'debug';
                } else if (level == 'error') {
                    return _request.logLevel === 'error' || _request.logLevel === 'warn' || _request.logLevel === 'info' || _request.logLevel === 'debug';
                } else {
                    return false;
                }
            }

            function _debug(msg) {
                if (_canLog('debug')) {
                    atmosphere.util.debug(new Date() + " Atmosphere: " + msg);
                }
            }

            /**
             *
             * @private
             */
            function _verifyStreamingLength(ajaxRequest, rq) {
                // Wait to be sure we have the full message before closing.
                if (_response.partialMessage === "" && (rq.transport === 'streaming') && (ajaxRequest.responseText.length > rq.maxStreamingLength)) {
                    return true;
                }
                return false;
            }

            /**
             * Disconnect
             *
             * @private
             */
            function _disconnect() {
                if (_request.enableProtocol && !_request.firstMessage) {
                    var query = "X-Atmosphere-Transport=close&X-Atmosphere-tracking-id=" + _request.uuid;

                    atmosphere.util.each(_request.headers, function (name, value) {
                        var h = atmosphere.util.isFunction(value) ? value.call(this, _request, _request, _response) : value;
                        if (h != null) {
                            query += "&" + encodeURIComponent(name) + "=" + encodeURIComponent(h);
                        }
                    });

                    var url = _request.url.replace(/([?&])_=[^&]*/, query);
                    url = url + (url === _request.url ? (/\?/.test(_request.url) ? "&" : "?") + query : "");

                    var rq = {
                        connected: false
                    };
                    var closeR = new atmosphere.AtmosphereRequest(rq);
                    closeR.connectTimeout = _request.connectTimeout;
                    closeR.attachHeadersAsQueryString = false;
                    closeR.dropHeaders = true;
                    closeR.url = url;
                    closeR.contentType = "text/plain";
                    closeR.transport = 'polling';
                    closeR.method = 'GET';
                    closeR.data = '';
                    closeR.heartbeat = null;
                    if (_request.enableXDR) {
                        closeR.enableXDR = _request.enableXDR
                    }
                    closeR.async = _request.closeAsync;
                    _pushOnClose("", closeR);
                }
            }

            /**
             * Close request.
             *
             * @private
             */
            function _close() {
                _debug("Closing (AtmosphereRequest._close() called)");

                _abortingConnection = true;
                if (_request.reconnectId) {
                    clearTimeout(_request.reconnectId);
                    delete _request.reconnectId;
                }

                if (_request.heartbeatTimer) {
                    clearTimeout(_request.heartbeatTimer);
                }

                _request.reconnect = false;
                _response.request = _request;
                _response.state = 'unsubscribe';
                _response.responseBody = "";
                _response.status = 408;
                _response.partialMessage = "";
                _invokeCallback();
                _disconnect();
                _clearState();
            }

            function _clearState() {
                _response.partialMessage = "";
                if (_request.id) {
                    clearTimeout(_request.id);
                }

                if (_request.heartbeatTimer) {
                    clearTimeout(_request.heartbeatTimer);
                }

                // https://github.com/Atmosphere/atmosphere/issues/1860#issuecomment-74707226
                if(_request.reconnectId) {
                    clearTimeout(_request.reconnectId);
                    delete _request.reconnectId;
                }

                if (_ieStream != null) {
                    _ieStream.close();
                    _ieStream = null;
                }
                if (_jqxhr != null) {
                    _jqxhr.abort();
                    _jqxhr = null;
                }
                if (_activeRequest != null) {
                    _activeRequest.abort();
                    _activeRequest = null;
                }
                if (_websocket != null) {
                    if (_websocket.canSendMessage) {
                        _debug("invoking .close() on WebSocket object");
                        _websocket.close();
                    }
                    _websocket = null;
                }
                if (_sse != null) {
                    _sse.close();
                    _sse = null;
                }
                _clearStorage();
            }

            function _clearStorage() {
                // Stop sharing a connection
                if (_storageService != null) {
                    // Clears trace timer
                    clearInterval(_traceTimer);
                    // Removes the trace
                    document.cookie = _sharingKey + "=; expires=Thu, 01 Jan 1970 00:00:00 GMT; path=/";
                    // The heir is the parent unless unloading
                    _storageService.signal("close", {
                        reason: "",
                        heir: !_abortingConnection ? guid : (_storageService.get("children") || [])[0]
                    });
                    _storageService.close();
                }
                if (_localStorageService != null) {
                    _localStorageService.close();
                }
            }

            /**
             * Subscribe request using request transport. <br>
             * If request is currently opened, this one will be closed.
             *
             * @param {Object} Request parameters.
             * @private
             */
            function _subscribe(options) {
                _reinit();

                _request = atmosphere.util.extend(_request, options);
                // Allow at least 1 request
                _request.mrequest = _request.reconnect;
                if (!_request.reconnect) {
                    _request.reconnect = true;
                }
            }

            /**
             * Check if web socket is supported (check for custom implementation provided by request object or browser implementation).
             *
             * @returns {boolean} True if web socket is supported, false otherwise.
             * @private
             */
            function _supportWebsocket() {
                return _request.webSocketImpl != null || window.WebSocket || window.MozWebSocket;
            }

            /**
             * Check if server side events (SSE) is supported (check for custom implementation provided by request object or browser implementation).
             *
             * @returns {boolean} True if web socket is supported, false otherwise.
             * @private
             */
            function _supportSSE() {
                // Origin parts
                var url = atmosphere.util.getAbsoluteURL(_request.url.toLowerCase());
                var parts = /^([\w\+\.\-]+:)(?:\/\/([^\/?#:]*)(?::(\d+))?)?/.exec(url);
                var crossOrigin = !!(parts && (
                    // protocol
                parts[1] != window.location.protocol ||
                    // hostname
                parts[2] != window.location.hostname ||
                    // port
                (parts[3] || (parts[1] === "http:" ? 80 : 443)) != (window.location.port || (window.location.protocol === "http:" ? 80 : 443))
                ));
                return window.EventSource && (!crossOrigin || !atmosphere.util.browser.safari || atmosphere.util.browser.vmajor >= 7);
            }

            /**
             * Open request using request transport. <br>
             * If request transport is 'websocket' but websocket can't be opened, request will automatically reconnect using fallback transport.
             *
             * @private
             */
            function _execute() {
                // Shared across multiple tabs/windows.
                if (_request.shared) {
                    _localStorageService = _local(_request);
                    if (_localStorageService != null) {
                        if (_canLog('debug')) {
                            atmosphere.util.debug("Storage service available. All communication will be local");
                        }

                        if (_localStorageService.open(_request)) {
                            // Local connection.
                            return;
                        }
                    }

                    if (_canLog('debug')) {
                        atmosphere.util.debug("No Storage service available.");
                    }
                    // Wasn't local or an error occurred
                    _localStorageService = null;
                }

                // Protocol
                _request.firstMessage = uuid == 0 ? true : false;
                _request.isOpen = false;
                _request.ctime = atmosphere.util.now();

                // We carry any UUID set by the user or from a previous connection.
                if (_request.uuid === 0) {
                    _request.uuid = uuid;
                }
                _response.closedByClientTimeout = false;

                if (_request.transport !== 'websocket' && _request.transport !== 'sse') {
                    _executeRequest(_request);

                } else if (_request.transport === 'websocket') {
                    if (!_supportWebsocket()) {
                        _reconnectWithFallbackTransport("Websocket is not supported, using request.fallbackTransport (" + _request.fallbackTransport
                        + ")");
                    } else {
                        _executeWebSocket(false);
                    }
                } else if (_request.transport === 'sse') {
                    if (!_supportSSE()) {
                        _reconnectWithFallbackTransport("Server Side Events(SSE) is not supported, using request.fallbackTransport ("
                        + _request.fallbackTransport + ")");
                    } else {
                        _executeSSE(false);
                    }
                }
            }

            function _local(request) {
                var trace, connector, orphan, name = "atmosphere-" + request.url, connectors = {
                    storage: function () {
                        function onstorage(event) {
                            if (event.key === name && event.newValue) {
                                listener(event.newValue);
                            }
                        }

                        if (!atmosphere.util.storage) {
                            return;
                        }

                        var storage = window.localStorage,
                            get = function (key) {
                                return atmosphere.util.parseJSON(storage.getItem(name + "-" + key));
                            },
                            set = function (key, value) {
                                storage.setItem(name + "-" + key, atmosphere.util.stringifyJSON(value));
                            };

                        return {
                            init: function () {
                                set("children", get("children").concat([guid]));
                                atmosphere.util.on(window, "storage", onstorage);
                                return get("opened");
                            },
                            signal: function (type, data) {
                                storage.setItem(name, atmosphere.util.stringifyJSON({
                                    target: "p",
                                    type: type,
                                    data: data
                                }));
                            },
                            close: function () {
                                var children = get("children");

                                atmosphere.util.off(window, "storage", onstorage);
                                if (children) {
                                    if (removeFromArray(children, request.id)) {
                                        set("children", children);
                                    }
                                }
                            }
                        };
                    },
                    windowref: function () {
                        var win = window.open("", name.replace(/\W/g, ""));

                        if (!win || win.closed || !win.callbacks) {
                            return;
                        }

                        return {
                            init: function () {
                                win.callbacks.push(listener);
                                win.children.push(guid);
                                return win.opened;
                            },
                            signal: function (type, data) {
                                if (!win.closed && win.fire) {
                                    win.fire(atmosphere.util.stringifyJSON({
                                        target: "p",
                                        type: type,
                                        data: data
                                    }));
                                }
                            },
                            close: function () {
                                // Removes traces only if the parent is alive
                                if (!orphan) {
                                    removeFromArray(win.callbacks, listener);
                                    removeFromArray(win.children, guid);
                                }
                            }

                        };
                    }
                };

                function removeFromArray(array, val) {
                    var i, length = array.length;

                    for (i = 0; i < length; i++) {
                        if (array[i] === val) {
                            array.splice(i, 1);
                        }
                    }

                    return length !== array.length;
                }

                // Receives open, close and message command from the parent
                function listener(string) {
                    var command = atmosphere.util.parseJSON(string), data = command.data;

                    if (command.target === "c") {
                        switch (command.type) {
                            case "open":
                                _open("opening", 'local', _request);
                                break;
                            case "close":
                                if (!orphan) {
                                    orphan = true;
                                    if (data.reason === "aborted") {
                                        _close();
                                    } else {
                                        // Gives the heir some time to reconnect
                                        if (data.heir === guid) {
                                            _execute();
                                        } else {
                                            setTimeout(function () {
                                                _execute();
                                            }, 100);
                                        }
                                    }
                                }
                                break;
                            case "message":
                                _prepareCallback(data, "messageReceived", 200, request.transport);
                                break;
                            case "localMessage":
                                _localMessage(data);
                                break;
                        }
                    }
                }

                function findTrace() {
                    var matcher = new RegExp("(?:^|; )(" + encodeURIComponent(name) + ")=([^;]*)").exec(document.cookie);
                    if (matcher) {
                        return atmosphere.util.parseJSON(decodeURIComponent(matcher[2]));
                    }
                }

                // Finds and validates the parent socket's trace from the cookie
                trace = findTrace();
                if (!trace || atmosphere.util.now() - trace.ts > 1000) {
                    return;
                }

                // Chooses a connector
                connector = connectors.storage() || connectors.windowref();
                if (!connector) {
                    return;
                }

                return {
                    open: function () {
                        var parentOpened;

                        // Checks the shared one is alive
                        _traceTimer = setInterval(function () {
                            var oldTrace = trace;
                            trace = findTrace();
                            if (!trace || oldTrace.ts === trace.ts) {
                                // Simulates a close signal
                                listener(atmosphere.util.stringifyJSON({
                                    target: "c",
                                    type: "close",
                                    data: {
                                        reason: "error",
                                        heir: oldTrace.heir
                                    }
                                }));
                            }
                        }, 1000);

                        parentOpened = connector.init();
                        if (parentOpened) {
                            // Firing the open event without delay robs the user of the opportunity to bind connecting event handlers
                            setTimeout(function () {
                                _open("opening", 'local', request);
                            }, 50);
                        }
                        return parentOpened;
                    },
                    send: function (event) {
                        connector.signal("send", event);
                    },
                    localSend: function (event) {
                        connector.signal("localSend", atmosphere.util.stringifyJSON({
                            id: guid,
                            event: event
                        }));
                    },
                    close: function () {
                        // Do not signal the parent if this method is executed by the unload event handler
                        if (!_abortingConnection) {
                            clearInterval(_traceTimer);
                            connector.signal("close");
                            connector.close();
                        }
                    }
                };
            }

            function share() {
                var storageService, name = "atmosphere-" + _request.url, servers = {
                    // Powered by the storage event and the localStorage
                    // http://www.w3.org/TR/webstorage/#event-storage
                    storage: function () {
                        function onstorage(event) {
                            // When a deletion, newValue initialized to null
                            if (event.key === name && event.newValue) {
                                listener(event.newValue);
                            }
                        }

                        if (!atmosphere.util.storage) {
                            return;
                        }

                        var storage = window.localStorage;

                        return {
                            init: function () {
                                // Handles the storage event
                                atmosphere.util.on(window, "storage", onstorage);
                            },
                            signal: function (type, data) {
                                storage.setItem(name, atmosphere.util.stringifyJSON({
                                    target: "c",
                                    type: type,
                                    data: data
                                }));
                            },
                            get: function (key) {
                                return atmosphere.util.parseJSON(storage.getItem(name + "-" + key));
                            },
                            set: function (key, value) {
                                storage.setItem(name + "-" + key, atmosphere.util.stringifyJSON(value));
                            },
                            close: function () {
                                atmosphere.util.off(window, "storage", onstorage);
                                storage.removeItem(name);
                                storage.removeItem(name + "-opened");
                                storage.removeItem(name + "-children");
                            }

                        };
                    },
                    // Powered by the window.open method
                    // https://developer.mozilla.org/en/DOM/window.open
                    windowref: function () {
                        // Internet Explorer raises an invalid argument error
                        // when calling the window.open method with the name containing non-word characters
                        var neim = name.replace(/\W/g, ""), container = document.getElementById(neim), win;

                        if (!container) {
                            container = document.createElement("div");
                            container.id = neim;
                            container.style.display = "none";
                            container.innerHTML = '<iframe name="' + neim + '" />';
                            document.body.appendChild(container);
                        }

                        win = container.firstChild.contentWindow;

                        return {
                            init: function () {
                                // Callbacks from different windows
                                win.callbacks = [listener];
                                // In IE 8 and less, only string argument can be safely passed to the function in other window
                                win.fire = function (string) {
                                    var i;

                                    for (i = 0; i < win.callbacks.length; i++) {
                                        win.callbacks[i](string);
                                    }
                                };
                            },
                            signal: function (type, data) {
                                if (!win.closed && win.fire) {
                                    win.fire(atmosphere.util.stringifyJSON({
                                        target: "c",
                                        type: type,
                                        data: data
                                    }));
                                }
                            },
                            get: function (key) {
                                return !win.closed ? win[key] : null;
                            },
                            set: function (key, value) {
                                if (!win.closed) {
                                    win[key] = value;
                                }
                            },
                            close: function () {
                            }
                        };
                    }
                };

                // Receives send and close command from the children
                function listener(string) {
                    var command = atmosphere.util.parseJSON(string), data = command.data;

                    if (command.target === "p") {
                        switch (command.type) {
                            case "send":
                                _push(data);
                                break;
                            case "localSend":
                                _localMessage(data);
                                break;
                            case "close":
                                _close();
                                break;
                        }
                    }
                }

                _localSocketF = function propagateMessageEvent(context) {
                    storageService.signal("message", context);
                };

                function leaveTrace() {
                    document.cookie = _sharingKey + "=" +
                        // Opera's JSON implementation ignores a number whose a last digit of 0 strangely
                        // but has no problem with a number whose a last digit of 9 + 1
                    encodeURIComponent(atmosphere.util.stringifyJSON({
                        ts: atmosphere.util.now() + 1,
                        heir: (storageService.get("children") || [])[0]
                    })) + "; path=/";
                }

                // Chooses a storageService
                storageService = servers.storage() || servers.windowref();
                storageService.init();

                if (_canLog('debug')) {
                    atmosphere.util.debug("Installed StorageService " + storageService);
                }

                // List of children sockets
                storageService.set("children", []);

                if (storageService.get("opened") != null && !storageService.get("opened")) {
                    // Flag indicating the parent socket is opened
                    storageService.set("opened", false);
                }
                // Leaves traces
                _sharingKey = encodeURIComponent(name);
                leaveTrace();
                _traceTimer = setInterval(leaveTrace, 1000);

                _storageService = storageService;
            }

            /**
             * @private
             */
            function _open(state, transport, request) {
                if (_request.shared && transport !== 'local') {
                    share();
                }

                if (_storageService != null) {
                    _storageService.set("opened", true);
                }

                request.close = function () {
                    _close();
                };

                if (_requestCount > 0 && state === 're-connecting') {
                    request.isReopen = true;
                    _tryingToReconnect(_response);
                } else if (_response.error == null) {
                    _response.request = request;
                    var prevState = _response.state;
                    _response.state = state;
                    var prevTransport = _response.transport;
                    _response.transport = transport;

                    var _body = _response.responseBody;
                    _invokeCallback();
                    _response.responseBody = _body;

                    _response.state = prevState;
                    _response.transport = prevTransport;
                }
            }

            /**
             * Execute request using jsonp transport.
             *
             * @param request {Object} request Request parameters, if undefined _request object will be used.
             * @private
             */
            function _jsonp(request) {
                // When CORS is enabled, make sure we force the proper transport.
                request.transport = "jsonp";

                var rq = _request, script;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                _jqxhr = {
                    open: function () {
                        var callback = "atmosphere" + (++guid);

                        function _reconnectOnFailure() {
                            rq.lastIndex = 0;

                            if (rq.openId) {
                                clearTimeout(rq.openId);
                            }

                            if (rq.heartbeatTimer) {
                                clearTimeout(rq.heartbeatTimer);
                            }

                            if (rq.reconnect && _requestCount++ < rq.maxReconnectOnClose) {
                                _open('re-connecting', rq.transport, rq);
                                _reconnect(_jqxhr, rq, request.reconnectInterval);
                                rq.openId = setTimeout(function () {
                                    _triggerOpen(rq);
                                }, rq.reconnectInterval + 1000);
                            } else {
                                _onError(0, "maxReconnectOnClose reached");
                            }
                        }

                        function poll() {
                            var url = rq.url;
                            if (rq.dispatchUrl != null) {
                                url += rq.dispatchUrl;
                            }

                            var data = rq.data;
                            if (rq.attachHeadersAsQueryString) {
                                url = _attachHeaders(rq);
                                if (data !== '') {
                                    url += "&X-Atmosphere-Post-Body=" + encodeURIComponent(data);
                                }
                                data = '';
                            }

                            var head = document.head || document.getElementsByTagName("head")[0] || document.documentElement;

                            script = document.createElement("script");
                            script.src = url + "&jsonpTransport=" + callback;
                            //script.async = rq.async;
                            script.clean = function () {
                                script.clean = script.onerror = script.onload = script.onreadystatechange = null;
                                if (script.parentNode) {
                                    script.parentNode.removeChild(script);
                                }

                                if (++request.scriptCount === 2) {
                                    request.scriptCount = 1;
                                    _reconnectOnFailure();
                                }

                            };
                            script.onload = script.onreadystatechange = function () {
                                _debug("jsonp.onload");
                                if (!script.readyState || /loaded|complete/.test(script.readyState)) {
                                    script.clean();
                                }
                            };

                            script.onerror = function () {
                                _debug("jsonp.onerror");
                                request.scriptCount = 1;
                                script.clean();
                            };

                            head.insertBefore(script, head.firstChild);
                        }

                        // Attaches callback
                        window[callback] = function (msg) {
                            _debug("jsonp.window");
                            request.scriptCount = 0;
                            if (rq.reconnect && rq.maxRequest === -1 || rq.requestCount++ < rq.maxRequest) {

                                // _readHeaders(_jqxhr, rq);
                                if (!rq.executeCallbackBeforeReconnect) {
                                    _reconnect(_jqxhr, rq, rq.pollingInterval);
                                }

                                if (msg != null && typeof msg !== 'string') {
                                    try {
                                        msg = msg.message;
                                    } catch (err) {
                                        // The message was partial
                                    }
                                }
                                var skipCallbackInvocation = _trackMessageSize(msg, rq, _response);
                                if (!skipCallbackInvocation) {
                                    _prepareCallback(_response.responseBody, "messageReceived", 200, rq.transport);
                                }

                                if (rq.executeCallbackBeforeReconnect) {
                                    _reconnect(_jqxhr, rq, rq.pollingInterval);
                                }
                                _timeout(rq);
                            } else {
                                atmosphere.util.log(_request.logLevel, ["JSONP reconnect maximum try reached " + _request.requestCount]);
                                _onError(0, "maxRequest reached");
                            }
                        };
                        setTimeout(function () {
                            poll();
                        }, 50);
                    },
                    abort: function () {
                        if (script && script.clean) {
                            script.clean();
                        }
                    }
                };
                _jqxhr.open();
            }

            /**
             * Build websocket object.
             *
             * @param location {string} Web socket url.
             * @returns {websocket} Web socket object.
             * @private
             */
            function _getWebSocket(location) {
                if (_request.webSocketImpl != null) {
                    return _request.webSocketImpl;
                } else {
                    if (window.WebSocket) {
                        return new WebSocket(location);
                    } else {
                        return new MozWebSocket(location);
                    }
                }
            }

            /**
             * Build web socket url from request url.
             *
             * @return {string} Web socket url (start with "ws" or "wss" for secure web socket).
             * @private
             */
            function _buildWebSocketUrl() {
                return _attachHeaders(_request, atmosphere.util.getAbsoluteURL(_request.webSocketUrl || _request.url)).replace(/^http/, "ws");
            }

            /**
             * Build SSE url from request url.
             *
             * @return a url with Atmosphere's headers
             * @private
             */
            function _buildSSEUrl() {
                var url = _attachHeaders(_request);
                return url;
            }

            /**
             * Open SSE. <br>
             * Automatically use fallback transport if SSE can't be opened.
             *
             * @private
             */
            function _executeSSE(sseOpened) {

                _response.transport = "sse";

                var location = _buildSSEUrl();

                if (_canLog('debug')) {
                    atmosphere.util.debug("Invoking executeSSE");
                    atmosphere.util.debug("Using URL: " + location);
                }

                if (sseOpened && !_request.reconnect) {
                    if (_sse != null) {
                        _clearState();
                    }
                    return;
                }

                try {
                    _sse = new EventSource(location, {
                        withCredentials: _request.withCredentials
                    });
                } catch (e) {
                    _onError(0, e);
                    _reconnectWithFallbackTransport("SSE failed. Downgrading to fallback transport and resending");
                    return;
                }

                if (_request.connectTimeout > 0) {
                    _request.id = setTimeout(function () {
                        if (!sseOpened) {
                            _clearState();
                        }
                    }, _request.connectTimeout);
                }

                _sse.onopen = function (event) {
                    _debug("sse.onopen");
                    _timeout(_request);
                    if (_canLog('debug')) {
                        atmosphere.util.debug("SSE successfully opened");
                    }

                    if (!_request.enableProtocol) {
                        if (!sseOpened) {
                            _open('opening', "sse", _request);
                        } else {
                            _open('re-opening', "sse", _request);
                        }
                    } else if (_request.isReopen) {
                        _request.isReopen = false;
                        _open('re-opening', _request.transport, _request);
                    }

                    sseOpened = true;

                    if (_request.method === 'POST') {
                        _response.state = "messageReceived";
                        _sse.send(_request.data);
                    }
                };

                _sse.onmessage = function (message) {
                    _debug("sse.onmessage");
                    _timeout(_request);

                    if (!_request.enableXDR && message.origin && message.origin !== window.location.protocol + "//" + window.location.host) {
                        atmosphere.util.log(_request.logLevel, ["Origin was not " + window.location.protocol + "//" + window.location.host]);
                        return;
                    }

                    _response.state = 'messageReceived';
                    _response.status = 200;

                    message = message.data;
                    var skipCallbackInvocation = _trackMessageSize(message, _request, _response);

                    // https://github.com/remy/polyfills/blob/master/EventSource.js
                    // Since we polling.
                    /* if (_sse.URL) {
                     _sse.interval = 100;
                     _sse.URL = _buildSSEUrl();
                     } */

                    if (!skipCallbackInvocation) {
                        _invokeCallback();
                        _response.responseBody = '';
                        _response.messages = [];
                    }
                };

                _sse.onerror = function (message) {
                    _debug("sse.onerror");
                    clearTimeout(_request.id);

                    if (_request.heartbeatTimer) {
                        clearTimeout(_request.heartbeatTimer);
                    }

                    if (_response.closedByClientTimeout) {
                        return;
                    }

                    _invokeClose(sseOpened);
                    _clearState();

                    if (_abortingConnection) {
                        atmosphere.util.log(_request.logLevel, ["SSE closed normally"]);
                    } else if (!sseOpened) {
                        _reconnectWithFallbackTransport("SSE failed. Downgrading to fallback transport and resending");
                    } else if (_request.reconnect && (_response.transport === 'sse')) {
                        if (_requestCount++ < _request.maxReconnectOnClose) {
                            _open('re-connecting', _request.transport, _request);
                            if (_request.reconnectInterval > 0) {
                                _request.reconnectId = setTimeout(function () {
                                    _executeSSE(true);
                                }, _request.reconnectInterval);
                            } else {
                                _executeSSE(true);
                            }
                            _response.responseBody = "";
                            _response.messages = [];
                        } else {
                            atmosphere.util.log(_request.logLevel, ["SSE reconnect maximum try reached " + _requestCount]);
                            _onError(0, "maxReconnectOnClose reached");
                        }
                    }
                };
            }

            /**
             * Open web socket. <br>
             * Automatically use fallback transport if web socket can't be opened.
             *
             * @private
             */
            function _executeWebSocket(webSocketOpened) {

                _response.transport = "websocket";

                var location = _buildWebSocketUrl(_request.url);
                if (_canLog('debug')) {
                    atmosphere.util.debug("Invoking executeWebSocket, using URL: " + location);
                }

                if (webSocketOpened && !_request.reconnect) {
                    if (_websocket != null) {
                        _clearState();
                    }
                    return;
                }

                _websocket = _getWebSocket(location);
                if (_request.webSocketBinaryType != null) {
                    _websocket.binaryType = _request.webSocketBinaryType;
                }

                if (_request.connectTimeout > 0) {
                    _request.id = setTimeout(function () {
                        if (!webSocketOpened) {
                            var _message = {
                                code: 1002,
                                reason: "",
                                wasClean: false
                            };
                            _websocket.onclose(_message);
                            // Close it anyway
                            try {
                                _clearState();
                            } catch (e) {
                            }
                            return;
                        }

                    }, _request.connectTimeout);
                }

                _websocket.onopen = function (message) {
                    _debug("websocket.onopen");
                    _timeout(_request);
                    offline = false;

                    if (_canLog('debug')) {
                        atmosphere.util.debug("Websocket successfully opened");
                    }

                    var reopening = webSocketOpened;

                    if (_websocket != null) {
                        _websocket.canSendMessage = true;
                    }

                    if (!_request.enableProtocol) {
                        webSocketOpened = true;
                        if (reopening) {
                            _open('re-opening', "websocket", _request);
                        } else {
                            _open('opening', "websocket", _request);
                        }
                    }

                    if (_websocket != null) {
                        if (_request.method === 'POST') {
                            _response.state = "messageReceived";
                            _websocket.send(_request.data);
                        }
                    }
                };

                _websocket.onmessage = function (message) {
                    _debug("websocket.onmessage");
                    _timeout(_request);

                    // We only consider it opened if we get the handshake data
                    // https://github.com/Atmosphere/atmosphere-javascript/issues/74
                    if (_request.enableProtocol) {
                        webSocketOpened = true;
                    }

                    _response.state = 'messageReceived';
                    _response.status = 200;

                    message = message.data;
                    var isString = typeof (message) === 'string';
                    if (isString) {
                        var skipCallbackInvocation = _trackMessageSize(message, _request, _response);
                        if (!skipCallbackInvocation) {
                            _invokeCallback();
                            _response.responseBody = '';
                            _response.messages = [];
                        }
                    } else {
                        message = _handleProtocol(_request, message);
                        if (message === "")
                            return;

                        _response.responseBody = message;
                        _invokeCallback();
                        _response.responseBody = null;
                    }
                };

                _websocket.onerror = function (message) {
                    _debug("websocket.onerror");
                    clearTimeout(_request.id);

                    if (_request.heartbeatTimer) {
                        clearTimeout(_request.heartbeatTimer);
                    }
                };

                _websocket.onclose = function (message) {
                    _debug("websocket.onclose");
                    clearTimeout(_request.id);
                    if (_response.state === 'closed')
                        return;

                    var reason = message.reason;
                    if (reason === "") {
                        switch (message.code) {
                            case 1000:
                                reason = "Normal closure; the connection successfully completed whatever purpose for which it was created.";
                                break;
                            case 1001:
                                reason = "The endpoint is going away, either because of a server failure or because the "
                                + "browser is navigating away from the page that opened the connection.";
                                break;
                            case 1002:
                                reason = "The endpoint is terminating the connection due to a protocol error.";
                                break;
                            case 1003:
                                reason = "The connection is being terminated because the endpoint received data of a type it "
                                + "cannot accept (for example, a text-only endpoint received binary data).";
                                break;
                            case 1004:
                                reason = "The endpoint is terminating the connection because a data frame was received that is too large.";
                                break;
                            case 1005:
                                reason = "Unknown: no status code was provided even though one was expected.";
                                break;
                            case 1006:
                                reason = "Connection was closed abnormally (that is, with no close frame being sent).";
                                break;
                        }
                    }

                    if (_canLog('warn')) {
                        atmosphere.util.warn("Websocket closed, reason: " + reason + ' - wasClean: ' + message.wasClean);
                    }

                    if (_response.closedByClientTimeout || offline) {
                        if (_request.reconnectId) {
                            clearTimeout(_request.reconnectId);
                            delete _request.reconnectId;
                        }
                        return;
                    }

                    _invokeClose(webSocketOpened);

                    _response.state = 'closed';

                    if (_abortingConnection) {
                        atmosphere.util.log(_request.logLevel, ["Websocket closed normally"]);
                    } else if (!webSocketOpened) {
                        _reconnectWithFallbackTransport("Websocket failed on first connection attempt. Downgrading to " + _request.fallbackTransport + " and resending");

                    } else if (_request.reconnect && _response.transport === 'websocket' ) {
                        _clearState();
                        if (_requestCount++ < _request.maxReconnectOnClose) {
                            _open('re-connecting', _request.transport, _request);
                            if (_request.reconnectInterval > 0) {
                                _request.reconnectId = setTimeout(function () {
                                    _response.responseBody = "";
                                    _response.messages = [];
                                    _executeWebSocket(true);
                                }, _request.reconnectInterval);
                            } else {
                                _response.responseBody = "";
                                _response.messages = [];
                                _executeWebSocket(true);
                            }
                        } else {
                            atmosphere.util.log(_request.logLevel, ["Websocket reconnect maximum try reached " + _request.requestCount]);
                            if (_canLog('warn')) {
                                atmosphere.util.warn("Websocket error, reason: " + message.reason);
                            }
                            _onError(0, "maxReconnectOnClose reached");
                        }
                    }
                };

                var ua = navigator.userAgent.toLowerCase();
                var isAndroid = ua.indexOf("android") > -1;
                if (isAndroid && _websocket.url === undefined) {
                    // Android 4.1 does not really support websockets and fails silently
                    _websocket.onclose({
                        reason: "Android 4.1 does not support websockets.",
                        wasClean: false
                    });
                }
            }

            function _handleProtocol(request, message) {

                var nMessage = message;
                if (request.transport === 'polling') return nMessage;

                if (request.enableProtocol && request.firstMessage && atmosphere.util.trim(message).length !== 0) {
                    var pos = request.trackMessageLength ? 1 : 0;
                    var messages = message.split(request.messageDelimiter);

                    if (messages.length <= pos + 1) {
                        // Something went wrong, normally with IE or when a message is written before the
                        // handshake has been received.
                        return nMessage;
                    }

                    request.firstMessage = false;
                    request.uuid = atmosphere.util.trim(messages[pos]);

                    if (messages.length <= pos + 2) {
                        atmosphere.util.log('error', ["Protocol data not sent by the server. " +
                        "If you enable protocol on client side, be sure to install JavascriptProtocol interceptor on server side." +
                        "Also note that atmosphere-runtime 2.2+ should be used."]);
                    }

                    _heartbeatInterval = parseInt(atmosphere.util.trim(messages[pos + 1]), 10);
                    _heartbeatPadding = messages[pos + 2];

                    if (request.transport !== 'long-polling') {
                        _triggerOpen(request);
                    }
                    uuid = request.uuid;
                    nMessage = "";

                    // We have trailing messages
                    pos = request.trackMessageLength ? 4 : 3;
                    if (messages.length > pos + 1) {
                        for (var i = pos; i < messages.length; i++) {
                            nMessage += messages[i];
                            if (i + 1 !== messages.length) {
                                nMessage += request.messageDelimiter;
                            }
                        }
                    }

                    if (request.ackInterval !== 0) {
                        setTimeout(function () {
                            _push("...ACK...");
                        }, request.ackInterval);
                    }
                } else if (request.enableProtocol && request.firstMessage && atmosphere.util.browser.msie && +atmosphere.util.browser.version.split(".")[0] < 10) {
                    // In case we are getting some junk from IE
                    atmosphere.util.log(_request.logLevel, ["Receiving unexpected data from IE"]);
                } else {
                    _triggerOpen(request);
                }
                return nMessage;
            }

            function _timeout(_request) {
                clearTimeout(_request.id);
                if (_request.timeout > 0 && _request.transport !== 'polling') {
                    _request.id = setTimeout(function () {
                        _onClientTimeout(_request);
                        _disconnect();
                        _clearState();
                    }, _request.timeout);
                }
            }

            function _onClientTimeout(_request) {
                _response.closedByClientTimeout = true;
                _response.state = 'closedByClient';
                _response.responseBody = "";
                _response.status = 408;
                _response.messages = [];
                _invokeCallback();
            }

            function _onError(code, reason) {
                _clearState();
                clearTimeout(_request.id);
                _response.state = 'error';
                _response.reasonPhrase = reason;
                _response.responseBody = "";
                _response.status = code;
                _response.messages = [];
                _invokeCallback();
            }

            /**
             * Track received message and make sure callbacks/functions are only invoked when the complete message has been received.
             *
             * @param message
             * @param request
             * @param response
             */
            function _trackMessageSize(message, request, response) {
                message = _handleProtocol(request, message);
                if (message.length === 0)
                    return true;

                response.responseBody = message;

                if (request.trackMessageLength) {
                    // prepend partialMessage if any
                    message = response.partialMessage + message;

                    var messages = [];
                    var messageStart = message.indexOf(request.messageDelimiter);
                    if (messageStart != -1) {
                        while (messageStart !== -1) {
                            var str = message.substring(0, messageStart);
                            var messageLength = +str;
                            if (isNaN(messageLength))
                                throw new Error('message length "' + str + '" is not a number');
                            messageStart += request.messageDelimiter.length;
                            if (messageStart + messageLength > message.length) {
                                // message not complete, so there is no trailing messageDelimiter
                                messageStart = -1;
                            } else {
                                // message complete, so add it
                                messages.push(message.substring(messageStart, messageStart + messageLength));
                                // remove consumed characters
                                message = message.substring(messageStart + messageLength, message.length);
                                messageStart = message.indexOf(request.messageDelimiter);
                            }
                        }

                        /* keep any remaining data */
                        response.partialMessage = message;

                        if (messages.length !== 0) {
                            response.responseBody = messages.join(request.messageDelimiter);
                            response.messages = messages;
                            return false;
                        } else {
                            response.responseBody = "";
                            response.messages = [];
                            return true;
                        }
                    }
                }
                response.responseBody = message;
                response.messages = [message];
                return false;
            }

            /**
             * Reconnect request with fallback transport. <br>
             * Used in case websocket can't be opened.
             *
             * @private
             */
            function _reconnectWithFallbackTransport(errorMessage) {
                atmosphere.util.log(_request.logLevel, [errorMessage]);

                if (typeof (_request.onTransportFailure) !== 'undefined') {
                    _request.onTransportFailure(errorMessage, _request);
                } else if (typeof (atmosphere.util.onTransportFailure) !== 'undefined') {
                    atmosphere.util.onTransportFailure(errorMessage, _request);
                }

                _request.transport = _request.fallbackTransport;
                var reconnectInterval = _request.connectTimeout === -1 ? 0 : _request.connectTimeout;
                if (_request.reconnect && _request.transport !== 'none' || _request.transport == null) {
                    _request.method = _request.fallbackMethod;
                    _response.transport = _request.fallbackTransport;
                    _request.fallbackTransport = 'none';
                    if (reconnectInterval > 0) {
                        _request.reconnectId = setTimeout(function () {
                            _execute();
                        }, reconnectInterval);
                    } else {
                        _execute();
                    }
                } else {
                    _onError(500, "Unable to reconnect with fallback transport");
                }
            }

            /**
             * Get url from request and attach headers to it.
             *
             * @param request {Object} request Request parameters, if undefined _request object will be used.
             *
             * @returns {Object} Request object, if undefined, _request object will be used.
             * @private
             */
            function _attachHeaders(request, url) {
                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                if (url == null) {
                    url = rq.url;
                }

                // If not enabled
                if (!rq.attachHeadersAsQueryString)
                    return url;

                // If already added
                if (url.indexOf("X-Atmosphere-Framework") !== -1) {
                    return url;
                }

                url += (url.indexOf('?') !== -1) ? '&' : '?';
                url += "X-Atmosphere-tracking-id=" + rq.uuid;
                url += "&X-Atmosphere-Framework=" + version;
                url += "&X-Atmosphere-Transport=" + rq.transport;

                if (rq.trackMessageLength) {
                    url += "&X-Atmosphere-TrackMessageSize=" + "true";
                }

                if (rq.heartbeat !== null && rq.heartbeat.server !== null) {
                    url += "&X-Heartbeat-Server=" + rq.heartbeat.server;
                }

                if (rq.contentType !== '') {
                    //Eurk!
                    url += "&Content-Type=" + (rq.transport === 'websocket' ? rq.contentType : encodeURIComponent(rq.contentType));
                }

                if (rq.enableProtocol) {
                    url += "&X-atmo-protocol=true";
                }

                atmosphere.util.each(rq.headers, function (name, value) {
                    var h = atmosphere.util.isFunction(value) ? value.call(this, rq, request, _response) : value;
                    if (h != null) {
                        url += "&" + encodeURIComponent(name) + "=" + encodeURIComponent(h);
                    }
                });

                return url;
            }

            function _triggerOpen(rq) {
                if (!rq.isOpen) {
                    rq.isOpen = true;
                    _open('opening', rq.transport, rq);
                } else if (rq.isReopen) {
                    rq.isReopen = false;
                    _open('re-opening', rq.transport, rq);
                } else if (_response.state === 'messageReceived' && (rq.transport === 'jsonp' || rq.transport === 'long-polling')) {
                    _openAfterResume(_response);
                } else {
                    return;
                }

                _startHeartbeat(rq);
            }

            function _startHeartbeat(rq) {
                if (rq.heartbeatTimer != null) {
                    clearTimeout(rq.heartbeatTimer);
                }

                if (!isNaN(_heartbeatInterval) && _heartbeatInterval > 0) {
                    var _pushHeartbeat = function () {
                        if (_canLog('debug')) {
                            atmosphere.util.debug("Sending heartbeat");
                        }
                        _push(_heartbeatPadding);
                        rq.heartbeatTimer = setTimeout(_pushHeartbeat, _heartbeatInterval);
                    };
                    rq.heartbeatTimer = setTimeout(_pushHeartbeat, _heartbeatInterval);
                }
            }

            /**
             * Execute ajax request. <br>
             *
             * @param request {Object} request Request parameters, if undefined _request object will be used.
             * @private
             */
            function _executeRequest(request) {
                var rq = _request;
                if ((request != null) || (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                rq.lastIndex = 0;
                rq.readyState = 0;

                // CORS fake using JSONP
                if ((rq.transport === 'jsonp') || ((rq.enableXDR) && (atmosphere.util.checkCORSSupport()))) {
                    _jsonp(rq);
                    return;
                }

                if (atmosphere.util.browser.msie && +atmosphere.util.browser.version.split(".")[0] < 10) {
                    if ((rq.transport === 'streaming')) {
                        if (rq.enableXDR && window.XDomainRequest) {
                            _ieXDR(rq);
                        } else {
                            _ieStreaming(rq);
                        }
                        return;
                    }

                    if ((rq.enableXDR) && (window.XDomainRequest)) {
                        _ieXDR(rq);
                        return;
                    }
                }

                var reconnectFExec = function (force) {
                    rq.lastIndex = 0;
                    if (force || (rq.reconnect && _requestCount++ < rq.maxReconnectOnClose)) {
                        _response.ffTryingReconnect = true;
                        _open('re-connecting', request.transport, request);
                        _reconnect(ajaxRequest, rq, request.reconnectInterval);
                    } else {
                        _onError(0, "maxReconnectOnClose reached");
                    }
                };

                var reconnectF = function (force){
                    if(atmosphere._beforeUnloadState){
                        // ATMOSPHERE-JAVASCRIPT-143: Delay reconnect to avoid reconnect attempts before an actual unload (we don't know if an unload will happen, yet)
                        atmosphere.util.debug(new Date() + " Atmosphere: reconnectF: execution delayed due to _beforeUnloadState flag");
                        setTimeout(function () {
                            reconnectFExec(force);
                        }, 5000);
                    }else {
                        reconnectFExec(force);
                    }
                };

                var disconnected = function () {
                    // Prevent onerror callback to be called
                    _response.errorHandled = true;
                    _clearState();
                    reconnectF(false);
                };

                if (rq.force || (rq.reconnect && (rq.maxRequest === -1 || rq.requestCount++ < rq.maxRequest))) {
                    rq.force = false;

                    var ajaxRequest = atmosphere.util.xhr();
                    ajaxRequest.hasData = false;

                    _doRequest(ajaxRequest, rq, true);

                    if (rq.suspend) {
                        _activeRequest = ajaxRequest;
                    }

                    if (rq.transport !== 'polling') {
                        _response.transport = rq.transport;

                        ajaxRequest.onabort = function () {
                            _debug("ajaxrequest.onabort")
                            _invokeClose(true);
                        };

                        ajaxRequest.onerror = function () {
                            _debug("ajaxrequest.onerror")
                            _response.error = true;
                            _response.ffTryingReconnect = true;
                            try {
                                _response.status = XMLHttpRequest.status;
                            } catch (e) {
                                _response.status = 500;
                            }

                            if (!_response.status) {
                                _response.status = 500;
                            }
                            if (!_response.errorHandled) {
                                _clearState();
                                reconnectF(false);
                            }
                        };
                    }

                    ajaxRequest.onreadystatechange = function () {
                        _debug("ajaxRequest.onreadystatechange, new state: " + ajaxRequest.readyState);
                        if (_abortingConnection) {
                            _debug("onreadystatechange has been ignored due to _abortingConnection flag");
                            return;
                        }

                        _response.error = null;
                        var skipCallbackInvocation = false;
                        var update = false;

                        if (rq.transport === 'streaming' && rq.readyState > 2 && ajaxRequest.readyState === 4) {
                            _clearState();
                            reconnectF(false);
                            return;
                        }

                        rq.readyState = ajaxRequest.readyState;

                        if (rq.transport === 'streaming' && ajaxRequest.readyState >= 3) {
                            update = true;
                        } else if (rq.transport === 'long-polling' && ajaxRequest.readyState === 4) {
                            update = true;
                        }
                        _timeout(_request);

                        if (rq.transport !== 'polling') {
                            // MSIE 9 and lower status can be higher than 1000, Chrome can be 0
                            var status = 200;
                            if (ajaxRequest.readyState === 4) {
                                status = ajaxRequest.status > 1000 ? 0 : ajaxRequest.status;
                            }

                            if (!rq.reconnectOnServerError && (status >= 300 && status < 600)) {
                                _onError(status, ajaxRequest.statusText);
                                return;
                            }

                            if (status >= 300 || status === 0) {
                                disconnected();
                                return;
                            }

                            // Firefox incorrectly send statechange 0->2 when a reconnect attempt fails. The above checks ensure that onopen is not called for these
                            if ((!rq.enableProtocol || !request.firstMessage) && ajaxRequest.readyState === 2) {
                                // Firefox incorrectly send statechange 0->2 when a reconnect attempt fails. The above checks ensure that onopen is not called for these
                                // In that case, ajaxRequest.onerror will be called just after onreadystatechange is called, so we delay the trigger until we are
                                // guarantee the connection is well established.
                                if (atmosphere.util.browser.mozilla && _response.ffTryingReconnect) {
                                    _response.ffTryingReconnect = false;
                                    setTimeout(function () {
                                        if (!_response.ffTryingReconnect) {
                                            _triggerOpen(rq);
                                        }
                                    }, 500);
                                } else {
                                    _triggerOpen(rq);
                                }
                            }

                        } else if (ajaxRequest.readyState === 4) {
                            update = true;
                        }

                        if (update) {
                            var responseText = ajaxRequest.responseText;
                            _response.errorHandled = false;

                            // IE behave the same way when resuming long-polling or when the server goes down.
                            if (rq.transport === 'long-polling' && atmosphere.util.trim(responseText).length === 0) {
                                // For browser that aren't support onabort
                                if (!ajaxRequest.hasData) {
                                    reconnectF(true);
                                } else {
                                    ajaxRequest.hasData = false;
                                }
                                return;
                            }
                            ajaxRequest.hasData = true;

                            _readHeaders(ajaxRequest, _request);

                            if (rq.transport === 'streaming') {
                                if (!atmosphere.util.browser.opera) {
                                    var message = responseText.substring(rq.lastIndex, responseText.length);
                                    skipCallbackInvocation = _trackMessageSize(message, rq, _response);

                                    rq.lastIndex = responseText.length;
                                    if (skipCallbackInvocation) {
                                        return;
                                    }
                                } else {
                                    atmosphere.util.iterate(function () {
                                        if (_response.status !== 500 && ajaxRequest.responseText.length > rq.lastIndex) {
                                            try {
                                                _response.status = ajaxRequest.status;
                                                _response.headers = atmosphere.util.parseHeaders(ajaxRequest.getAllResponseHeaders());

                                                _readHeaders(ajaxRequest, _request);

                                            } catch (e) {
                                                _response.status = 404;
                                            }
                                            _timeout(_request);

                                            _response.state = "messageReceived";
                                            var message = ajaxRequest.responseText.substring(rq.lastIndex);
                                            rq.lastIndex = ajaxRequest.responseText.length;

                                            skipCallbackInvocation = _trackMessageSize(message, rq, _response);
                                            if (!skipCallbackInvocation) {
                                                _invokeCallback();
                                            }

                                            if (_verifyStreamingLength(ajaxRequest, rq)) {
                                                _reconnectOnMaxStreamingLength(ajaxRequest, rq);
                                                return;
                                            }
                                        } else if (_response.status > 400) {
                                            // Prevent replaying the last message.
                                            rq.lastIndex = ajaxRequest.responseText.length;
                                            return false;
                                        }
                                    }, 0);
                                }
                            } else {
                                skipCallbackInvocation = _trackMessageSize(responseText, rq, _response);
                            }
                            var closeStream = _verifyStreamingLength(ajaxRequest, rq);

                            try {
                                _response.status = ajaxRequest.status;
                                _response.headers = atmosphere.util.parseHeaders(ajaxRequest.getAllResponseHeaders());

                                _readHeaders(ajaxRequest, rq);
                            } catch (e) {
                                _response.status = 404;
                            }

                            if (rq.suspend) {
                                _response.state = _response.status === 0 ? "closed" : "messageReceived";
                            } else {
                                _response.state = "messagePublished";
                            }

                            var isAllowedToReconnect = !closeStream && request.transport !== 'streaming' && request.transport !== 'polling';
                            if (isAllowedToReconnect && !rq.executeCallbackBeforeReconnect) {
                                _reconnect(ajaxRequest, rq, rq.pollingInterval);
                            }

                            if (_response.responseBody.length !== 0 && !skipCallbackInvocation)
                                _invokeCallback();

                            if (isAllowedToReconnect && rq.executeCallbackBeforeReconnect) {
                                _reconnect(ajaxRequest, rq, rq.pollingInterval);
                            }

                            if (closeStream) {
                                _reconnectOnMaxStreamingLength(ajaxRequest, rq);
                            }
                        }
                    };

                    try {
                        ajaxRequest.send(rq.data);
                        _subscribed = true;
                    } catch (e) {
                        atmosphere.util.log(rq.logLevel, ["Unable to connect to " + rq.url]);
                        _onError(0, e);
                    }

                } else {
                    if (rq.logLevel === 'debug') {
                        atmosphere.util.log(rq.logLevel, ["Max re-connection reached."]);
                    }
                    _onError(0, "maxRequest reached");
                }
            }

            function _reconnectOnMaxStreamingLength(ajaxRequest, rq) {
                _response.messages = [];
                rq.isReopen = true;
                _close();
                _abortingConnection = false;
                _reconnect(ajaxRequest, rq, 500);
            }

            /**
             * Do ajax request.
             *
             * @param ajaxRequest Ajax request.
             * @param request Request parameters.
             * @param create If ajax request has to be open.
             */
            function _doRequest(ajaxRequest, request, create) {
                // Prevent Android to cache request
                var url = request.url;
                if (request.dispatchUrl != null && request.method === 'POST') {
                    url += request.dispatchUrl;
                }
                url = _attachHeaders(request, url);
                url = atmosphere.util.prepareURL(url);

                if (create) {
                    ajaxRequest.open(request.method, url, request.async);
                    if (request.connectTimeout > 0) {
                        request.id = setTimeout(function () {
                            if (request.requestCount === 0) {
                                _clearState();
                                _prepareCallback("Connect timeout", "closed", 200, request.transport);
                            }
                        }, request.connectTimeout);
                    }
                }

                if (_request.withCredentials && _request.transport !== 'websocket') {
                    if ("withCredentials" in ajaxRequest) {
                        ajaxRequest.withCredentials = true;
                    }
                }

                if (!_request.dropHeaders) {
                    ajaxRequest.setRequestHeader("X-Atmosphere-Framework", version);
                    ajaxRequest.setRequestHeader("X-Atmosphere-Transport", request.transport);

                    if (request.heartbeat !== null && request.heartbeat.server !== null) {
                        ajaxRequest.setRequestHeader("X-Heartbeat-Server", ajaxRequest.heartbeat.server);
                    }

                    if (request.trackMessageLength) {
                        ajaxRequest.setRequestHeader("X-Atmosphere-TrackMessageSize", "true");
                    }
                    ajaxRequest.setRequestHeader("X-Atmosphere-tracking-id", request.uuid);

                    atmosphere.util.each(request.headers, function (name, value) {
                        var h = atmosphere.util.isFunction(value) ? value.call(this, ajaxRequest, request, create, _response) : value;
                        if (h != null) {
                            ajaxRequest.setRequestHeader(name, h);
                        }
                    });
                }

                if (request.contentType !== '') {
                    ajaxRequest.setRequestHeader("Content-Type", request.contentType);
                }
            }

            function _reconnect(ajaxRequest, request, reconnectInterval) {

                if (_response.closedByClientTimeout) {
                    return;
                }

                if (request.reconnect || (request.suspend && _subscribed)) {
                    var status = 0;
                    if (ajaxRequest && ajaxRequest.readyState > 1) {
                        status = ajaxRequest.status > 1000 ? 0 : ajaxRequest.status;
                    }
                    _response.status = status === 0 ? 204 : status;
                    _response.reason = status === 0 ? "Server resumed the connection or down." : "OK";

                    clearTimeout(request.id);
                    if (request.reconnectId) {
                        clearTimeout(request.reconnectId);
                        delete request.reconnectId;
                    }

                    if (reconnectInterval > 0) {
                        // For whatever reason, never cancel a reconnect timeout as it is mandatory to reconnect.
                        _request.reconnectId = setTimeout(function () {
                            _executeRequest(request);
                        }, reconnectInterval);
                    } else {
                        _executeRequest(request);
                    }
                }
            }

            function _tryingToReconnect(response) {
                response.state = 're-connecting';
                _invokeFunction(response);
            }

            function _openAfterResume(response) {
                response.state = 'openAfterResume';
                _invokeFunction(response);
                response.state = 'messageReceived';
            }

            function _ieXDR(request) {
                if (request.transport !== "polling") {
                    _ieStream = _configureXDR(request);
                    _ieStream.open();
                } else {
                    _configureXDR(request).open();
                }
            }

            function _configureXDR(request) {
                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                var transport = rq.transport;
                var lastIndex = 0;
                var xdr = new window.XDomainRequest();
                var reconnect = function () {
                    if (rq.transport === "long-polling" && (rq.reconnect && (rq.maxRequest === -1 || rq.requestCount++ < rq.maxRequest))) {
                        xdr.status = 200;
                        _ieXDR(rq);
                    }
                };

                var rewriteURL = rq.rewriteURL || function (url) {
                        // Maintaining session by rewriting URL
                        // http://stackoverflow.com/questions/6453779/maintaining-session-by-rewriting-url
                        var match = /(?:^|;\s*)(JSESSIONID|PHPSESSID)=([^;]*)/.exec(document.cookie);

                        switch (match && match[1]) {
                            case "JSESSIONID":
                                return url.replace(/;jsessionid=[^\?]*|(\?)|$/, ";jsessionid=" + match[2] + "$1");
                            case "PHPSESSID":
                                return url.replace(/\?PHPSESSID=[^&]*&?|\?|$/, "?PHPSESSID=" + match[2] + "&").replace(/&$/, "");
                        }
                        return url;
                    };

                // Handles open and message event
                xdr.onprogress = function () {
                    handle(xdr);
                };
                // Handles error event
                xdr.onerror = function () {
                    // If the server doesn't send anything back to XDR will fail with polling
                    if (rq.transport !== 'polling') {
                        _clearState();
                        if (_requestCount++ < rq.maxReconnectOnClose) {
                            if (rq.reconnectInterval > 0) {
                                rq.reconnectId = setTimeout(function () {
                                    _open('re-connecting', request.transport, request);
                                    _ieXDR(rq);
                                }, rq.reconnectInterval);
                            } else {
                                _open('re-connecting', request.transport, request);
                                _ieXDR(rq);
                            }
                        } else {
                            _onError(0, "maxReconnectOnClose reached");
                        }
                    }
                };

                // Handles close event
                xdr.onload = function () {
                };

                var handle = function (xdr) {
                    clearTimeout(rq.id);
                    var message = xdr.responseText;

                    message = message.substring(lastIndex);
                    lastIndex += message.length;

                    if (transport !== 'polling') {
                        _timeout(rq);

                        var skipCallbackInvocation = _trackMessageSize(message, rq, _response);

                        if (transport === 'long-polling' && atmosphere.util.trim(message).length === 0)
                            return;

                        if (rq.executeCallbackBeforeReconnect) {
                            reconnect();
                        }

                        if (!skipCallbackInvocation) {
                            _prepareCallback(_response.responseBody, "messageReceived", 200, transport);
                        }

                        if (!rq.executeCallbackBeforeReconnect) {
                            reconnect();
                        }
                    }
                };

                return {
                    open: function () {
                        var url = rq.url;
                        if (rq.dispatchUrl != null) {
                            url += rq.dispatchUrl;
                        }
                        url = _attachHeaders(rq, url);
                        xdr.open(rq.method, rewriteURL(url));
                        if (rq.method === 'GET') {
                            xdr.send();
                        } else {
                            xdr.send(rq.data);
                        }

                        if (rq.connectTimeout > 0) {
                            rq.id = setTimeout(function () {
                                if (rq.requestCount === 0) {
                                    _clearState();
                                    _prepareCallback("Connect timeout", "closed", 200, rq.transport);
                                }
                            }, rq.connectTimeout);
                        }
                    },
                    close: function () {
                        xdr.abort();
                    }
                };
            }

            function _ieStreaming(request) {
                _ieStream = _configureIE(request);
                _ieStream.open();
            }

            function _configureIE(request) {
                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                var stop;
                var doc = new window.ActiveXObject("htmlfile");

                doc.open();
                doc.close();

                var url = rq.url;
                if (rq.dispatchUrl != null) {
                    url += rq.dispatchUrl;
                }

                if (rq.transport !== 'polling') {
                    _response.transport = rq.transport;
                }

                return {
                    open: function () {
                        var iframe = doc.createElement("iframe");

                        url = _attachHeaders(rq);
                        if (rq.data !== '') {
                            url += "&X-Atmosphere-Post-Body=" + encodeURIComponent(rq.data);
                        }

                        // Finally attach a timestamp to prevent Android and IE caching.
                        url = atmosphere.util.prepareURL(url);

                        iframe.src = url;
                        doc.body.appendChild(iframe);

                        // For the server to respond in a consistent format regardless of user agent, we polls response text
                        var cdoc = iframe.contentDocument || iframe.contentWindow.document;

                        stop = atmosphere.util.iterate(function () {
                            try {
                                if (!cdoc.firstChild) {
                                    return;
                                }

                                var res = cdoc.body ? cdoc.body.lastChild : cdoc;
                                var readResponse = function () {
                                    // Clones the element not to disturb the original one
                                    var clone = res.cloneNode(true);

                                    // If the last character is a carriage return or a line feed, IE ignores it in the innerText property
                                    // therefore, we add another non-newline character to preserve it
                                    clone.appendChild(cdoc.createTextNode("."));

                                    var text = clone.innerText;

                                    text = text.substring(0, text.length - 1);
                                    return text;

                                };

                                // To support text/html content type
                                if (!cdoc.body || !cdoc.body.firstChild || cdoc.body.firstChild.nodeName.toLowerCase() !== "pre") {
                                    // Injects a plaintext element which renders text without interpreting the HTML and cannot be stopped
                                    // it is deprecated in HTML5, but still works
                                    var head = cdoc.head || cdoc.getElementsByTagName("head")[0] || cdoc.documentElement || cdoc;
                                    var script = cdoc.createElement("script");

                                    script.text = "document.write('<plaintext>')";

                                    head.insertBefore(script, head.firstChild);
                                    head.removeChild(script);

                                    // The plaintext element will be the response container
                                    res = cdoc.body.lastChild;
                                }

                                if (rq.closed) {
                                    rq.isReopen = true;
                                }

                                // Handles message and close event
                                stop = atmosphere.util.iterate(function () {
                                    var text = readResponse();
                                    if (text.length > rq.lastIndex) {
                                        _timeout(_request);

                                        _response.status = 200;
                                        _response.error = null;

                                        // Empties response every time that it is handled
                                        res.innerText = "";
                                        var skipCallbackInvocation = _trackMessageSize(text, rq, _response);
                                        if (skipCallbackInvocation) {
                                            return "";
                                        }

                                        _prepareCallback(_response.responseBody, "messageReceived", 200, rq.transport);
                                    }

                                    rq.lastIndex = 0;

                                    if (cdoc.readyState === "complete") {
                                        _invokeClose(true);
                                        _open('re-connecting', rq.transport, rq);
                                        if (rq.reconnectInterval > 0) {
                                            rq.reconnectId = setTimeout(function () {
                                                _ieStreaming(rq);
                                            }, rq.reconnectInterval);
                                        } else {
                                            _ieStreaming(rq);
                                        }
                                        return false;
                                    }
                                }, null);

                                return false;
                            } catch (err) {
                                _response.error = true;
                                _open('re-connecting', rq.transport, rq);
                                if (_requestCount++ < rq.maxReconnectOnClose) {
                                    if (rq.reconnectInterval > 0) {
                                        rq.reconnectId = setTimeout(function () {
                                            _ieStreaming(rq);
                                        }, rq.reconnectInterval);
                                    } else {
                                        _ieStreaming(rq);
                                    }
                                } else {
                                    _onError(0, "maxReconnectOnClose reached");
                                }
                                doc.execCommand("Stop");
                                doc.close();
                                return false;
                            }
                        });
                    },

                    close: function () {
                        if (stop) {
                            stop();
                        }

                        doc.execCommand("Stop");
                        _invokeClose(true);
                    }
                };
            }

            /**
             * Send message. <br>
             * Will be automatically dispatch to other connected.
             *
             * @param {Object, string} Message to send.
             * @private
             */
            function _push(message) {

                if (_localStorageService != null) {
                    _pushLocal(message);
                } else if (_activeRequest != null || _sse != null) {
                    _pushAjaxMessage(message);
                } else if (_ieStream != null) {
                    _pushIE(message);
                } else if (_jqxhr != null) {
                    _pushJsonp(message);
                } else if (_websocket != null) {
                    _pushWebSocket(message);
                } else {
                    _onError(0, "No suspended connection available");
                    atmosphere.util.error("No suspended connection available. Make sure atmosphere.subscribe has been called and request.onOpen invoked before trying to push data");
                }
            }

            function _pushOnClose(message, rq) {
                if (!rq) {
                    rq = _getPushRequest(message);
                }
                rq.transport = "polling";
                rq.method = "GET";
                rq.withCredentials = false;
                rq.reconnect = false;
                rq.force = true;
                rq.suspend = false;
                rq.timeout = 1000;
                _executeRequest(rq);
            }

            function _pushLocal(message) {
                _localStorageService.send(message);
            }

            function _intraPush(message) {
                // IE 9 will crash if not.
                if (message.length === 0)
                    return;

                try {
                    if (_localStorageService) {
                        _localStorageService.localSend(message);
                    } else if (_storageService) {
                        _storageService.signal("localMessage", atmosphere.util.stringifyJSON({
                            id: guid,
                            event: message
                        }));
                    }
                } catch (err) {
                    atmosphere.util.error(err);
                }
            }

            /**
             * Send a message using currently opened ajax request (using http-streaming or long-polling). <br>
             *
             * @param {string, Object} Message to send. This is an object, string message is saved in data member.
             * @private
             */
            function _pushAjaxMessage(message) {
                var rq = _getPushRequest(message);
                _executeRequest(rq);
            }

            /**
             * Send a message using currently opened ie streaming (using http-streaming or long-polling). <br>
             *
             * @param {string, Object} Message to send. This is an object, string message is saved in data member.
             * @private
             */
            function _pushIE(message) {
                if (_request.enableXDR && atmosphere.util.checkCORSSupport()) {
                    var rq = _getPushRequest(message);
                    // Do not reconnect since we are pushing.
                    rq.reconnect = false;
                    _jsonp(rq);
                } else {
                    _pushAjaxMessage(message);
                }
            }

            /**
             * Send a message using jsonp transport. <br>
             *
             * @param {string, Object} Message to send. This is an object, string message is saved in data member.
             * @private
             */
            function _pushJsonp(message) {
                _pushAjaxMessage(message);
            }

            function _getStringMessage(message) {
                var msg = message;
                if (typeof (msg) === 'object') {
                    msg = message.data;
                }
                return msg;
            }

            /**
             * Build request use to push message using method 'POST' <br>. Transport is defined as 'polling' and 'suspend' is set to false.
             *
             * @return {Object} Request object use to push message.
             * @private
             */
            function _getPushRequest(message) {
                var msg = _getStringMessage(message);

                var rq = {
                    connected: false,
                    timeout: 60000,
                    method: 'POST',
                    url: _request.url,
                    contentType: _request.contentType,
                    headers: _request.headers,
                    reconnect: true,
                    callback: null,
                    data: msg,
                    suspend: false,
                    maxRequest: -1,
                    logLevel: 'info',
                    requestCount: 0,
                    withCredentials: _request.withCredentials,
                    async: _request.async,
                    transport: 'polling',
                    isOpen: true,
                    attachHeadersAsQueryString: true,
                    enableXDR: _request.enableXDR,
                    uuid: _request.uuid,
                    dispatchUrl: _request.dispatchUrl,
                    enableProtocol: false,
                    messageDelimiter: '|',
                    trackMessageLength: _request.trackMessageLength,
                    maxReconnectOnClose: _request.maxReconnectOnClose,
                    heartbeatTimer: _request.heartbeatTimer,
                    heartbeat: _request.heartbeat
                };

                if (typeof (message) === 'object') {
                    rq = atmosphere.util.extend(rq, message);
                }

                return rq;
            }

            /**
             * Send a message using currently opened websocket. <br>
             *
             */
            function _pushWebSocket(message) {
                var msg = atmosphere.util.isBinary(message) ? message : _getStringMessage(message);
                var data;
                try {
                    if (_request.dispatchUrl != null) {
                        data = _request.webSocketPathDelimiter + _request.dispatchUrl + _request.webSocketPathDelimiter + msg;
                    } else {
                        data = msg;
                    }

                    if (!_websocket.canSendMessage) {
                        atmosphere.util.error("WebSocket not connected.");
                        return;
                    }

                    _websocket.send(data);

                } catch (e) {
                    _websocket.onclose = function (message) {
                    };
                    _clearState();

                    _reconnectWithFallbackTransport("Websocket failed. Downgrading to " + _request.fallbackTransport + " and resending " + message);
                    _pushAjaxMessage(message);
                }
            }

            function _localMessage(message) {
                var m = atmosphere.util.parseJSON(message);
                if (m.id !== guid) {
                    if (typeof (_request.onLocalMessage) !== 'undefined') {
                        _request.onLocalMessage(m.event);
                    } else if (typeof (atmosphere.util.onLocalMessage) !== 'undefined') {
                        atmosphere.util.onLocalMessage(m.event);
                    }
                }
            }

            function _prepareCallback(messageBody, state, errorCode, transport) {

                _response.responseBody = messageBody;
                _response.transport = transport;
                _response.status = errorCode;
                _response.state = state;

                _invokeCallback();
            }

            function _readHeaders(xdr, request) {
                if (!request.readResponsesHeaders) {
                    if (!request.enableProtocol) {
                        request.uuid = guid;
                    }
                }
                else {
                    try {

                        var tempUUID = xdr.getResponseHeader('X-Atmosphere-tracking-id');
                        if (tempUUID && tempUUID != null) {
                            request.uuid = tempUUID.split(" ").pop();
                        }
                    } catch (e) {
                    }
                }
            }

            function _invokeFunction(response) {
                _f(response, _request);
                // Global
                _f(response, atmosphere.util);
            }

            function _f(response, f) {
                switch (response.state) {
                    case "messageReceived":
                        _debug("Firing onMessage");
                        _requestCount = 0;
                        if (typeof (f.onMessage) !== 'undefined')
                            f.onMessage(response);

                        if (typeof (f.onmessage) !== 'undefined')
                            f.onmessage(response);
                        break;
                    case "error":
                        var dbgReasonPhrase = (typeof(response.reasonPhrase) != 'undefined') ? response.reasonPhrase : 'n/a';
                        _debug("Firing onError, reasonPhrase: " + dbgReasonPhrase);
                        if (typeof (f.onError) !== 'undefined')
                            f.onError(response);

                        if (typeof (f.onerror) !== 'undefined')
                            f.onerror(response);
                        break;
                    case "opening":
                        delete _request.closed;
                        _debug("Firing onOpen");
                        if (typeof (f.onOpen) !== 'undefined')
                            f.onOpen(response);

                        if (typeof (f.onopen) !== 'undefined')
                            f.onopen(response);
                        break;
                    case "messagePublished":
                        _debug("Firing messagePublished");
                        if (typeof (f.onMessagePublished) !== 'undefined')
                            f.onMessagePublished(response);
                        break;
                    case "re-connecting":
                        _debug("Firing onReconnect");
                        if (typeof (f.onReconnect) !== 'undefined')
                            f.onReconnect(_request, response);
                        break;
                    case "closedByClient":
                        _debug("Firing closedByClient");
                        if (typeof (f.onClientTimeout) !== 'undefined')
                            f.onClientTimeout(_request);
                        break;
                    case "re-opening":
                        delete _request.closed;
                        _debug("Firing onReopen");
                        if (typeof (f.onReopen) !== 'undefined')
                            f.onReopen(_request, response);
                        break;
                    case "fail-to-reconnect":
                        _debug("Firing onFailureToReconnect");
                        if (typeof (f.onFailureToReconnect) !== 'undefined')
                            f.onFailureToReconnect(_request, response);
                        break;
                    case "unsubscribe":
                    case "closed":
                        var closed = typeof (_request.closed) !== 'undefined' ? _request.closed : false;

                        if (!closed) {
                            _debug("Firing onClose (" + response.state + " case)");
                            if (typeof (f.onClose) !== 'undefined') {
                                f.onClose(response);
                            }

                            if (typeof (f.onclose) !== 'undefined') {
                                f.onclose(response);
                            }
                        } else {
                            _debug("Request already closed, not firing onClose (" + response.state + " case)");
                        }
                        _request.closed = true;
                        break;
                    case "openAfterResume":
                        if (typeof (f.onOpenAfterResume) !== 'undefined')
                            f.onOpenAfterResume(_request);
                        break;
                }
            }

            function _invokeClose(wasOpen) {
                if (_response.state !== 'closed') {
                    _response.state = 'closed';
                    _response.responseBody = "";
                    _response.messages = [];
                    _response.status = !wasOpen ? 501 : 200;
                    _invokeCallback();
                }
            }

            /**
             * Invoke request callbacks.
             *
             * @private
             */
            function _invokeCallback() {
                var call = function (index, func) {
                    func(_response);
                };

                if (_localStorageService == null && _localSocketF != null) {
                    _localSocketF(_response.responseBody);
                }

                _request.reconnect = _request.mrequest;

                var isString = typeof (_response.responseBody) === 'string';
                var messages = (isString && _request.trackMessageLength) ? (_response.messages.length > 0 ? _response.messages : ['']) : new Array(
                    _response.responseBody);
                for (var i = 0; i < messages.length; i++) {

                    if (messages.length > 1 && messages[i].length === 0) {
                        continue;
                    }
                    _response.responseBody = (isString) ? atmosphere.util.trim(messages[i]) : messages[i];

                    if (_localStorageService == null && _localSocketF != null) {
                        _localSocketF(_response.responseBody);
                    }

                    if ((_response.responseBody.length === 0 ||
                        (isString && _heartbeatPadding === _response.responseBody)) && _response.state === "messageReceived") {
                        continue;
                    }

                    _invokeFunction(_response);

                    // Invoke global callbacks
                    if (callbacks.length > 0) {
                        if (_canLog('debug')) {
                            atmosphere.util.debug("Invoking " + callbacks.length + " global callbacks: " + _response.state);
                        }
                        try {
                            atmosphere.util.each(callbacks, call);
                        } catch (e) {
                            atmosphere.util.log(_request.logLevel, ["Callback exception" + e]);
                        }
                    }

                    // Invoke request callback
                    if (typeof (_request.callback) === 'function') {
                        if (_canLog('debug')) {
                            atmosphere.util.debug("Invoking request callbacks");
                        }
                        try {
                            _request.callback(_response);
                        } catch (e) {
                            atmosphere.util.log(_request.logLevel, ["Callback exception" + e]);
                        }
                    }
                }
            }

            this.subscribe = function (options) {
                _subscribe(options);
                _execute();
            };

            this.execute = function () {
                _execute();
            };

            this.close = function () {
                _close();
            };

            this.disconnect = function () {
                _disconnect();
            };

            this.getUrl = function () {
                return _request.url;
            };

            this.push = function (message, dispatchUrl) {
                if (dispatchUrl != null) {
                    var originalDispatchUrl = _request.dispatchUrl;
                    _request.dispatchUrl = dispatchUrl;
                    _push(message);
                    _request.dispatchUrl = originalDispatchUrl;
                } else {
                    _push(message);
                }
            };

            this.getUUID = function () {
                return _request.uuid;
            };

            this.pushLocal = function (message) {
                _intraPush(message);
            };

            this.enableProtocol = function (message) {
                return _request.enableProtocol;
            };

            this.init = function () {
                _init();
            };

            this.request = _request;
            this.response = _response;
        }
    };

    atmosphere.subscribe = function (url, callback, request) {
        if (typeof (callback) === 'function') {
            atmosphere.addCallback(callback);
        }

        if (typeof (url) !== "string") {
            request = url;
        } else {
            request.url = url;
        }

        // https://github.com/Atmosphere/atmosphere-javascript/issues/58
        uuid = ((typeof (request) !== 'undefined') && typeof (request.uuid) !== 'undefined') ? request.uuid : 0;

        var rq = new atmosphere.AtmosphereRequest(request);
        rq.execute();

        requests[requests.length] = rq;
        return rq;
    };

    atmosphere.unsubscribe = function () {
        if (requests.length > 0) {
            var requestsClone = [].concat(requests);
            for (var i = 0; i < requestsClone.length; i++) {
                var rq = requestsClone[i];
                rq.close();
                clearTimeout(rq.response.request.id);

                if (rq.heartbeatTimer) {
                    clearTimeout(rq.heartbeatTimer);
                }
            }
        }
        requests = [];
        callbacks = [];
    };

    atmosphere.unsubscribeUrl = function (url) {
        var idx = -1;
        if (requests.length > 0) {
            for (var i = 0; i < requests.length; i++) {
                var rq = requests[i];

                // Suppose you can subscribe once to an url
                if (rq.getUrl() === url) {
                    rq.close();
                    clearTimeout(rq.response.request.id);

                    if (rq.heartbeatTimer) {
                        clearTimeout(rq.heartbeatTimer);
                    }

                    idx = i;
                    break;
                }
            }
        }
        if (idx >= 0) {
            requests.splice(idx, 1);
        }
    };

    atmosphere.addCallback = function (func) {
        if (atmosphere.util.inArray(func, callbacks) === -1) {
            callbacks.push(func);
        }
    };

    atmosphere.removeCallback = function (func) {
        var index = atmosphere.util.inArray(func, callbacks);
        if (index !== -1) {
            callbacks.splice(index, 1);
        }
    };

    atmosphere.util = {
        browser: {},

        parseHeaders: function (headerString) {
            var match, rheaders = /^(.*?):[ \t]*([^\r\n]*)\r?$/mg, headers = {};
            while (match = rheaders.exec(headerString)) {
                headers[match[1]] = match[2];
            }
            return headers;
        },

        now: function () {
            return new Date().getTime();
        },

        isArray: function (array) {
            return Object.prototype.toString.call(array) === "[object Array]";
        },

        inArray: function (elem, array) {
            if (!Array.prototype.indexOf) {
                var len = array.length;
                for (var i = 0; i < len; ++i) {
                    if (array[i] === elem) {
                        return i;
                    }
                }
                return -1;
            }
            return array.indexOf(elem);
        },

        isBinary: function (data) {
            // True if data is an instance of Blob, ArrayBuffer or ArrayBufferView
            return /^\[object\s(?:Blob|ArrayBuffer|.+Array)\]$/.test(Object.prototype.toString.call(data));
        },

        isFunction: function (fn) {
            return Object.prototype.toString.call(fn) === "[object Function]";
        },

        getAbsoluteURL: function (url) {
            var div = document.createElement("div");

            // Uses an innerHTML property to obtain an absolute URL
            div.innerHTML = '<a href="' + url + '"/>';

            // encodeURI and decodeURI are needed to normalize URL between IE and non-IE,
            // since IE doesn't encode the href property value and return it - http://jsfiddle.net/Yq9M8/1/
            return encodeURI(decodeURI(div.firstChild.href));
        },

        prepareURL: function (url) {
            // Attaches a time stamp to prevent caching
            var ts = atmosphere.util.now();
            var ret = url.replace(/([?&])_=[^&]*/, "$1_=" + ts);

            return ret + (ret === url ? (/\?/.test(url) ? "&" : "?") + "_=" + ts : "");
        },

        trim: function (str) {
            if (!String.prototype.trim) {
                return str.toString().replace(/(?:(?:^|\n)\s+|\s+(?:$|\n))/g, "").replace(/\s+/g, " ");
            } else {
                return str.toString().trim();
            }
        },

        param: function (params) {
            var prefix, s = [];

            function add(key, value) {
                value = atmosphere.util.isFunction(value) ? value() : (value == null ? "" : value);
                s.push(encodeURIComponent(key) + "=" + encodeURIComponent(value));
            }

            function buildParams(prefix, obj) {
                var name;

                if (atmosphere.util.isArray(obj)) {
                    atmosphere.util.each(obj, function (i, v) {
                        if (/\[\]$/.test(prefix)) {
                            add(prefix, v);
                        } else {
                            buildParams(prefix + "[" + (typeof v === "object" ? i : "") + "]", v);
                        }
                    });
                } else if (Object.prototype.toString.call(obj) === "[object Object]") {
                    for (name in obj) {
                        buildParams(prefix + "[" + name + "]", obj[name]);
                    }
                } else {
                    add(prefix, obj);
                }
            }

            for (prefix in params) {
                buildParams(prefix, params[prefix]);
            }

            return s.join("&").replace(/%20/g, "+");
        },

        storage: function () {
            try {
                return !!(window.localStorage && window.StorageEvent);
            } catch (e) {
                //Firefox throws an exception here, see
                //https://bugzilla.mozilla.org/show_bug.cgi?id=748620
                return false;
            }
        },

        iterate: function (fn, interval) {
            var timeoutId;

            // Though the interval is 0 for real-time application, there is a delay between setTimeout calls
            // For detail, see https://developer.mozilla.org/en/window.setTimeout#Minimum_delay_and_timeout_nesting
            interval = interval || 0;

            (function loop() {
                timeoutId = setTimeout(function () {
                    if (fn() === false) {
                        return;
                    }

                    loop();
                }, interval);
            })();

            return function () {
                clearTimeout(timeoutId);
            };
        },

        each: function (obj, callback, args) {
            if (!obj) return;
            var value, i = 0, length = obj.length, isArray = atmosphere.util.isArray(obj);

            if (args) {
                if (isArray) {
                    for (; i < length; i++) {
                        value = callback.apply(obj[i], args);

                        if (value === false) {
                            break;
                        }
                    }
                } else {
                    for (i in obj) {
                        value = callback.apply(obj[i], args);

                        if (value === false) {
                            break;
                        }
                    }
                }

                // A special, fast, case for the most common use of each
            } else {
                if (isArray) {
                    for (; i < length; i++) {
                        value = callback.call(obj[i], i, obj[i]);

                        if (value === false) {
                            break;
                        }
                    }
                } else {
                    for (i in obj) {
                        value = callback.call(obj[i], i, obj[i]);

                        if (value === false) {
                            break;
                        }
                    }
                }
            }

            return obj;
        },

        extend: function (target) {
            var i, options, name;

            for (i = 1; i < arguments.length; i++) {
                if ((options = arguments[i]) != null) {
                    for (name in options) {
                        target[name] = options[name];
                    }
                }
            }

            return target;
        },
        on: function (elem, type, fn) {
            if (elem.addEventListener) {
                elem.addEventListener(type, fn, false);
            } else if (elem.attachEvent) {
                elem.attachEvent("on" + type, fn);
            }
        },
        off: function (elem, type, fn) {
            if (elem.removeEventListener) {
                elem.removeEventListener(type, fn, false);
            } else if (elem.detachEvent) {
                elem.detachEvent("on" + type, fn);
            }
        },

        log: function (level, args) {
            if (window.console) {
                var logger = window.console[level];
                if (typeof logger === 'function') {
                    logger.apply(window.console, args);
                }
            }
        },

        warn: function () {
            atmosphere.util.log('warn', arguments);
        },

        info: function () {
            atmosphere.util.log('info', arguments);
        },

        debug: function () {
            atmosphere.util.log('debug', arguments);
        },

        error: function () {
            atmosphere.util.log('error', arguments);
        },
        xhr: function () {
            try {
                return new window.XMLHttpRequest();
            } catch (e1) {
                try {
                    return new window.ActiveXObject("Microsoft.XMLHTTP");
                } catch (e2) {
                }
            }
        },
        parseJSON: function (data) {
            return !data ? null : window.JSON && window.JSON.parse ? window.JSON.parse(data) : new Function("return " + data)();
        },
        // http://github.com/flowersinthesand/stringifyJSON
        stringifyJSON: function (value) {
            var escapable = /[\\\"\x00-\x1f\x7f-\x9f\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g, meta = {
                '\b': '\\b',
                '\t': '\\t',
                '\n': '\\n',
                '\f': '\\f',
                '\r': '\\r',
                '"': '\\"',
                '\\': '\\\\'
            };

            function quote(string) {
                return '"' + string.replace(escapable, function (a) {
                        var c = meta[a];
                        return typeof c === "string" ? c : "\\u" + ("0000" + a.charCodeAt(0).toString(16)).slice(-4);
                    }) + '"';
            }

            function f(n) {
                return n < 10 ? "0" + n : n;
            }

            return window.JSON && window.JSON.stringify ? window.JSON.stringify(value) : (function str(key, holder) {
                var i, v, len, partial, value = holder[key], type = typeof value;

                if (value && typeof value === "object" && typeof value.toJSON === "function") {
                    value = value.toJSON(key);
                    type = typeof value;
                }

                switch (type) {
                    case "string":
                        return quote(value);
                    case "number":
                        return isFinite(value) ? String(value) : "null";
                    case "boolean":
                        return String(value);
                    case "object":
                        if (!value) {
                            return "null";
                        }

                        switch (Object.prototype.toString.call(value)) {
                            case "[object Date]":
                                return isFinite(value.valueOf()) ? '"' + value.getUTCFullYear() + "-" + f(value.getUTCMonth() + 1) + "-"
                                + f(value.getUTCDate()) + "T" + f(value.getUTCHours()) + ":" + f(value.getUTCMinutes()) + ":" + f(value.getUTCSeconds())
                                + "Z" + '"' : "null";
                            case "[object Array]":
                                len = value.length;
                                partial = [];
                                for (i = 0; i < len; i++) {
                                    partial.push(str(i, value) || "null");
                                }

                                return "[" + partial.join(",") + "]";
                            default:
                                partial = [];
                                for (i in value) {
                                    if (hasOwn.call(value, i)) {
                                        v = str(i, value);
                                        if (v) {
                                            partial.push(quote(i) + ":" + v);
                                        }
                                    }
                                }

                                return "{" + partial.join(",") + "}";
                        }
                }
            })("", {
                "": value
            });
        },

        checkCORSSupport: function () {
            if (atmosphere.util.browser.msie && !window.XDomainRequest && +atmosphere.util.browser.version.split(".")[0] < 11) {
                return true;
            } else if (atmosphere.util.browser.opera && +atmosphere.util.browser.version.split(".") < 12.0) {
                return true;
            }

            // KreaTV 4.1 -> 4.4
            else if (atmosphere.util.trim(navigator.userAgent).slice(0, 16) === "KreaTVWebKit/531") {
                return true;
            }
            // KreaTV 3.8
            else if (atmosphere.util.trim(navigator.userAgent).slice(-7).toLowerCase() === "kreatel") {
                return true;
            }

            // Force Android to use CORS as some version like 2.2.3 fail otherwise
            var ua = navigator.userAgent.toLowerCase();
            var isAndroid = ua.indexOf("android") > -1;
            if (isAndroid) {
                return true;
            }
            return false;
        }
    };

    guid = atmosphere.util.now();

    // Browser sniffing
    (function () {
        var ua = navigator.userAgent.toLowerCase(),
            match = /(chrome)[ \/]([\w.]+)/.exec(ua) ||
                /(opera)(?:.*version|)[ \/]([\w.]+)/.exec(ua) ||
                /(msie) ([\w.]+)/.exec(ua) ||
                /(trident)(?:.*? rv:([\w.]+)|)/.exec(ua) ||
                ua.indexOf("android") < 0 && /version\/(.+) (safari)/.exec(ua) ||
                ua.indexOf("compatible") < 0 && /(mozilla)(?:.*? rv:([\w.]+)|)/.exec(ua) ||
                [];

        // Swaps variables
        if (match[2] === "safari") {
            match[2] = match[1];
            match[1] = "safari";
        }
        atmosphere.util.browser[match[1] || ""] = true;
        atmosphere.util.browser.version = match[2] || "0";
        atmosphere.util.browser.vmajor = atmosphere.util.browser.version.split(".")[0];

        // Trident is the layout engine of the Internet Explorer
        // IE 11 has no "MSIE: 11.0" token
        if (atmosphere.util.browser.trident) {
            atmosphere.util.browser.msie = true;
        }

        // The storage event of Internet Explorer and Firefox 3 works strangely
        if (atmosphere.util.browser.msie || (atmosphere.util.browser.mozilla && +atmosphere.util.browser.version.split(".")[0] === 1)) {
            atmosphere.util.storage = false;
        }
    })();

    atmosphere.util.on(window, "unload", function (event) {
        atmosphere.util.debug(new Date() + " Atmosphere: " + "unload event");
        atmosphere.unsubscribe();
    });

    atmosphere.util.on(window, "beforeunload", function (event) {
        atmosphere.util.debug(new Date() + " Atmosphere: " + "beforeunload event");

        // ATMOSPHERE-JAVASCRIPT-143: Delay reconnect to avoid reconnect attempts before an actual unload (we don't know if an unload will happen, yet)
        atmosphere._beforeUnloadState = true;
        setTimeout(function () {
            atmosphere.util.debug(new Date() + " Atmosphere: " + "beforeunload event timeout reached. Reset _beforeUnloadState flag");
            atmosphere._beforeUnloadState = false;
        }, 5000);
    });

    // Pressing ESC key in Firefox kills the connection
    // for your information, this is fixed in Firefox 20
    // https://bugzilla.mozilla.org/show_bug.cgi?id=614304
    atmosphere.util.on(window, "keypress", function (event) {
        if (event.charCode === 27 || event.keyCode === 27) {
            if (event.preventDefault) {
                event.preventDefault();
            }
        }
    });

    atmosphere.util.on(window, "offline", function () {
        atmosphere.util.debug(new Date() + " Atmosphere: offline event");
        offline = true;
        if (requests.length > 0) {
            var requestsClone = [].concat(requests);
            for (var i = 0; i < requestsClone.length; i++) {
                var rq = requestsClone[i];
                rq.close();
                clearTimeout(rq.response.request.id);

                if (rq.heartbeatTimer) {
                    clearTimeout(rq.heartbeatTimer);
                }
            }
        }
    });

    atmosphere.util.on(window, "online", function () {
        atmosphere.util.debug(new Date() + " Atmosphere: online event");
        if (requests.length > 0) {
            for (var i = 0; i < requests.length; i++) {
                requests[i].init();
                requests[i].execute();
            }
        }
        offline = false;
    });

    return atmosphere;
}));
/* jshint eqnull:true, noarg:true, noempty:true, eqeqeq:true, evil:true, laxbreak:true, undef:true, browser:true, indent:false, maxerr:50 */
;

// resource: org.richfaces:json-dom.js
// AJAX-JSF AJAX-like library, for communicate with view Tree on server side.
// In case of XMLHttpRequest don't worked, use :
// JSHttpRequest v1.12. (C) Dmitry Koterov, 2005-01-27. 
// http://forum.dklab.ru/users/DmitryKoterov/
//
// Do not remove this comment if you want to use script!
// ?? ???????? ?????? ???????????, ???? ?? ?????? ???????????? ??????!
//
// Modified by Alexander J. Smirnov to use as JSF AJAX-like components. 

// DOM - like elements for JSRequest. JS serialiser encode
// XML sax events to creation of corresponding objects.  
JSNode =  function() {
};

// Base node  
JSNode.prototype = {
	tag : null,
	attrs : {},
	childs : [],
	value : "",
	_symbols : {
			'&':"&amp;",
			'<':"&lt;",
			'>':"&gt;",
			'"':"&quot;",
			'\'':"&apos;",
			'\u00A0':"&nbsp;"
	},

	// Public functions
	getInnerHTML : function(context) {
		var children = [];
		for (var i = 0; i < this.childs.length; i++)
		{
			children.push(this.childs[i].getContent(context)); 
		}
		return children.join('');
	},
	// Escape XML symbols - < > & ' ...
	xmlEscape : function(value) {
		return RichFaces.jQuery("<div></div>").text(value).html();
	}
};

// Element node
E = function(tagname,attributes,childnodes) {
	this.tag = tagname;
	if (attributes) this.attrs = attributes;
	if(childnodes) this.childs = childnodes;
};

E.prototype = new JSNode();
E.prototype.getContent = function(context) {
	var html = "<"+this.tag;
	var inner = this.getInnerHTML(context);
	if (inner=='') this.isEmpty = true; else this.isEmpty=false;
	for(var i in this.attrs) {
		if (!this.attrs.hasOwnProperty(i)) {
			continue ;
		}

		var attrValue = this.attrs[i];
		
		if (typeof attrValue == "function")
			attrValue = attrValue.call(this, context);
		
		if (attrValue) 
			html += " "+(i=='className'?'class':i)+'="'+this.xmlEscape(attrValue)+'"';
	}
	html+= ">"+inner+"</"+this.tag+">";
	return html;
};

// Escaped Text node
ET = function(text) {
	this.value = text;
};


//ET.prototype = new JSNode();
ET.prototype.getContent = function(context) {
  	var value = this.value;
  	if (typeof value=="function") value=value(context);
  	if (value && value.getContent) {
		value = value.getContent(context);
	}
	
	if (value) return value;
  
	return "";
};

// Text node
T = function(text) {
	this.value = text;
};

T.prototype = new JSNode();
T.prototype.getContent = function(context) {
  	var value = this.value;
  	if (typeof value=="function") value=value(context);
  		
	if (value) return this.xmlEscape(value);

	return "";
};

// Comment node
C = function(text) {
	this.value = text;
};

//C.prototype = new JSNode();
C.prototype.getContent = function(context) {
	return "<!--"+this.value+"-->";
};
  
// CDATA Section node.
D = function(text) {
	this.value = text;
};
  
//D.prototype = new JSNode();
D.prototype.getContent = function(context) {
	return "<![CDATA["+this.value+"]]>";
};


;

// resource: org.richfaces:message.js
(function($, rf) {

    rf.ui = rf.ui || {};
    
    // Constructor definition
    rf.ui.Message = function(componentId, options) {
        // call constructor of parent class
        $super.constructor.call(this, componentId, options, defaultOptions);
        if (this.options.isMessages) {
            this.severityClasses = ["rf-msgs-inf", "rf-msgs-wrn", "rf-msgs-err", "rf-msgs-ftl"];
            this.summaryClass = "rf-msgs-sum";
            this.detailClass = "rf-msgs-det";
        } else {
            this.severityClasses = ["rf-msg-inf", "rf-msg-wrn", "rf-msg-err", "rf-msg-ftl"];
            this.summaryClass = "rf-msg-sum";
            this.detailClass = "rf-msg-det";
        }
    };

    // Extend component class and add protected methods from parent class to our container
    rf.ui.Base.extend(rf.ui.Message);

    // define super class link
    var $super = rf.ui.Message.$super;

    var defaultOptions = {
        showSummary:true,
        level:0,
        isMessages: false,
        globalOnly: false
    };


    var onMessage = function (event, element, data) {
        var content = $(rf.getDomElement(this.id));
        var sourceId = data.sourceId;
        var message = data.message;
        if (!this.options.forComponentId) {
            if (!message || this.options.globalOnly) {
                // rf.csv.clearMessage
                var element;
                while (element = rf.getDomElement(this.id + ':' + sourceId)) {
                    $(element).remove();
                }
            } else {
                renderMessage.call(this, sourceId, message);
            }
        } else if (this.options.forComponentId === sourceId) {
            content.empty();
            renderMessage.call(this, sourceId, message);
        }
    }

    var renderMessage = function(index, message) {
        if (message && message.severity >= this.options.level) {

            var content = $(rf.getDomElement(this.id));
            var msgContent = $("<span/>", {'class':(this.severityClasses)[message.severity],"id":this.id + ':' + index});
            if (message.summary) {
                if (this.options.tooltip) {
                    msgContent.attr("title", message.summary);
                } else if (this.options.showSummary) {
                    msgContent.append($("<span/>", {"class":(this.summaryClass)}).text(message.summary));
                }
            }
            if (this.options.showDetail && message.detail) {
                msgContent.append($("<span/>", {"class":(this.detailClass)}).text(message.detail));
            }
            content.append(msgContent);
        }
    }

    var bindEventHandlers = function () {
        rf.Event.bind(window.document, rf.Event.MESSAGE_EVENT_TYPE + this.namespace, onMessage, this);
    };

    $.extend(rf.ui.Message.prototype, {
            name: "Message",
            __bindEventHandlers: bindEventHandlers,
            
            destroy : function() {
                rf.Event.unbind(window.document, rf.Event.MESSAGE_EVENT_TYPE + this.namespace);
                $super.destroy.call(this);
            }
        });

})(RichFaces.jQuery, window.RichFaces || (window.RichFaces = {}));
;

// resource: org.richfaces:jquery.mousewheel.js
/*! Copyright (c) 2013 Brandon Aaron (http://brandon.aaron.sh)
 * Licensed under the MIT License (LICENSE.txt).
 *
 * Version: 3.1.12
 *
 * Requires: jQuery 1.2.2+
 */

(function (factory) {
    if ( typeof define === 'function' && define.amd ) {
        // AMD. Register as an anonymous module.
        define(['jquery'], factory);
    } else if (typeof exports === 'object') {
        // Node/CommonJS style for Browserify
        module.exports = factory;
    } else {
        // Browser globals
        factory(jQuery);
    }
}(function ($) {

    var toFix  = ['wheel', 'mousewheel', 'DOMMouseScroll', 'MozMousePixelScroll'],
        toBind = ( 'onwheel' in document || document.documentMode >= 9 ) ?
                    ['wheel'] : ['mousewheel', 'DomMouseScroll', 'MozMousePixelScroll'],
        slice  = Array.prototype.slice,
        nullLowestDeltaTimeout, lowestDelta;

    if ( $.event.fixHooks ) {
        for ( var i = toFix.length; i; ) {
            $.event.fixHooks[ toFix[--i] ] = $.event.mouseHooks;
        }
    }

    var special = $.event.special.mousewheel = {
        version: '3.1.12',

        setup: function() {
            if ( this.addEventListener ) {
                for ( var i = toBind.length; i; ) {
                    this.addEventListener( toBind[--i], handler, false );
                }
            } else {
                this.onmousewheel = handler;
            }
            // Store the line height and page height for this particular element
            $.data(this, 'mousewheel-line-height', special.getLineHeight(this));
            $.data(this, 'mousewheel-page-height', special.getPageHeight(this));
        },

        teardown: function() {
            if ( this.removeEventListener ) {
                for ( var i = toBind.length; i; ) {
                    this.removeEventListener( toBind[--i], handler, false );
                }
            } else {
                this.onmousewheel = null;
            }
            // Clean up the data we added to the element
            $.removeData(this, 'mousewheel-line-height');
            $.removeData(this, 'mousewheel-page-height');
        },

        getLineHeight: function(elem) {
            var $elem = $(elem),
                $parent = $elem['offsetParent' in $.fn ? 'offsetParent' : 'parent']();
            if (!$parent.length) {
                $parent = $('body');
            }
            return parseInt($parent.css('fontSize'), 10) || parseInt($elem.css('fontSize'), 10) || 16;
        },

        getPageHeight: function(elem) {
            return $(elem).height();
        },

        settings: {
            adjustOldDeltas: true, // see shouldAdjustOldDeltas() below
            normalizeOffset: true  // calls getBoundingClientRect for each event
        }
    };

    $.fn.extend({
        mousewheel: function(fn) {
            return fn ? this.bind('mousewheel', fn) : this.trigger('mousewheel');
        },

        unmousewheel: function(fn) {
            return this.unbind('mousewheel', fn);
        }
    });


    function handler(event) {
        var orgEvent   = event || window.event,
            args       = slice.call(arguments, 1),
            delta      = 0,
            deltaX     = 0,
            deltaY     = 0,
            absDelta   = 0,
            offsetX    = 0,
            offsetY    = 0;
        event = $.event.fix(orgEvent);
        event.type = 'mousewheel';

        // Old school scrollwheel delta
        if ( 'detail'      in orgEvent ) { deltaY = orgEvent.detail * -1;      }
        if ( 'wheelDelta'  in orgEvent ) { deltaY = orgEvent.wheelDelta;       }
        if ( 'wheelDeltaY' in orgEvent ) { deltaY = orgEvent.wheelDeltaY;      }
        if ( 'wheelDeltaX' in orgEvent ) { deltaX = orgEvent.wheelDeltaX * -1; }

        // Firefox < 17 horizontal scrolling related to DOMMouseScroll event
        if ( 'axis' in orgEvent && orgEvent.axis === orgEvent.HORIZONTAL_AXIS ) {
            deltaX = deltaY * -1;
            deltaY = 0;
        }

        // Set delta to be deltaY or deltaX if deltaY is 0 for backwards compatabilitiy
        delta = deltaY === 0 ? deltaX : deltaY;

        // New school wheel delta (wheel event)
        if ( 'deltaY' in orgEvent ) {
            deltaY = orgEvent.deltaY * -1;
            delta  = deltaY;
        }
        if ( 'deltaX' in orgEvent ) {
            deltaX = orgEvent.deltaX;
            if ( deltaY === 0 ) { delta  = deltaX * -1; }
        }

        // No change actually happened, no reason to go any further
        if ( deltaY === 0 && deltaX === 0 ) { return; }

        // Need to convert lines and pages to pixels if we aren't already in pixels
        // There are three delta modes:
        //   * deltaMode 0 is by pixels, nothing to do
        //   * deltaMode 1 is by lines
        //   * deltaMode 2 is by pages
        if ( orgEvent.deltaMode === 1 ) {
            var lineHeight = $.data(this, 'mousewheel-line-height');
            delta  *= lineHeight;
            deltaY *= lineHeight;
            deltaX *= lineHeight;
        } else if ( orgEvent.deltaMode === 2 ) {
            var pageHeight = $.data(this, 'mousewheel-page-height');
            delta  *= pageHeight;
            deltaY *= pageHeight;
            deltaX *= pageHeight;
        }

        // Store lowest absolute delta to normalize the delta values
        absDelta = Math.max( Math.abs(deltaY), Math.abs(deltaX) );

        if ( !lowestDelta || absDelta < lowestDelta ) {
            lowestDelta = absDelta;

            // Adjust older deltas if necessary
            if ( shouldAdjustOldDeltas(orgEvent, absDelta) ) {
                lowestDelta /= 40;
            }
        }

        // Adjust older deltas if necessary
        if ( shouldAdjustOldDeltas(orgEvent, absDelta) ) {
            // Divide all the things by 40!
            delta  /= 40;
            deltaX /= 40;
            deltaY /= 40;
        }

        // Get a whole, normalized value for the deltas
        delta  = Math[ delta  >= 1 ? 'floor' : 'ceil' ](delta  / lowestDelta);
        deltaX = Math[ deltaX >= 1 ? 'floor' : 'ceil' ](deltaX / lowestDelta);
        deltaY = Math[ deltaY >= 1 ? 'floor' : 'ceil' ](deltaY / lowestDelta);

        // Normalise offsetX and offsetY properties
        if ( special.settings.normalizeOffset && this.getBoundingClientRect ) {
            var boundingRect = this.getBoundingClientRect();
            offsetX = event.clientX - boundingRect.left;
            offsetY = event.clientY - boundingRect.top;
        }

        // Add information to the event object
        event.deltaX = deltaX;
        event.deltaY = deltaY;
        event.deltaFactor = lowestDelta;
        event.offsetX = offsetX;
        event.offsetY = offsetY;
        // Go ahead and set deltaMode to 0 since we converted to pixels
        // Although this is a little odd since we overwrite the deltaX/Y
        // properties with normalized deltas.
        event.deltaMode = 0;

        // Add event and delta to the front of the arguments
        args.unshift(event, delta, deltaX, deltaY);

        // Clearout lowestDelta after sometime to better
        // handle multiple device types that give different
        // a different lowestDelta
        // Ex: trackpad = 3 and mouse wheel = 120
        if (nullLowestDeltaTimeout) { clearTimeout(nullLowestDeltaTimeout); }
        nullLowestDeltaTimeout = setTimeout(nullLowestDelta, 200);

        return ($.event.dispatch || $.event.handle).apply(this, args);
    }

    function nullLowestDelta() {
        lowestDelta = null;
    }

    function shouldAdjustOldDeltas(orgEvent, absDelta) {
        // If this is an older event and the delta is divisable by 120,
        // then we are assuming that the browser is treating this as an
        // older mouse wheel event and that we should divide the deltas
        // by 40 to try and get a more usable deltaFactor.
        // Side note, this actually impacts the reported scroll distance
        // in older browsers and can cause scrolling to be slower than native.
        // Turn this off by setting $.event.special.mousewheel.settings.adjustOldDeltas to false.
        return special.settings.adjustOldDeltas && orgEvent.type === 'mousewheel' && absDelta % 120 === 0;
    }

}));;

// resource: org.richfaces:popupPanel.js
(function ($, rf) {

    rf.ui = rf.ui || {};
    var selectionEventHandler = function(event) {
        event.stopPropagation();
        event.preventDefault();
    };

    var disableSelection = function (element) {
        if (typeof element.onselectstart != "undefined") //IE
        {
            $(rf.getDomElement(element)).bind('selectstart', selectionEventHandler);
        }
        else //All other (ie: Opera)
        {
            $(rf.getDomElement(element)).bind('mousedown', selectionEventHandler);
        }
    };

    var enableSelection = function (element) {
        if (typeof element.onselectstart != "undefined") //IE
        {
            $(rf.getDomElement(element)).unbind('selectstart', selectionEventHandler);
        }
        else //All other (ie: Opera)
        {
            $(rf.getDomElement(element)).unbind('mousedown', selectionEventHandler);
        }
    };

    var defaultOptions = {
        width:-1,
        height:-1,
        minWidth:-1,
        minHeight:-1,
        modal:true,
        moveable:true,
        resizeable: false,
        autosized: false,
        left: "auto",
        top : "auto",
        zindex:100,
        shadowDepth : 5,
        shadowOpacity: 0.1,
        attachToBody:true
    };


    rf.ui.PopupPanel = function(id, options) {

        $super.constructor.call(this, id);
        this.markerId = id;
        this.attachToDom(this.markerId);
        this.options = $.extend(this.options, defaultOptions, options || {});

        this.minWidth = this.getMinimumSize(this.options.minWidth);
        this.minHeight = this.getMinimumSize(this.options.minHeight);
        this.maxWidth = this.options.maxWidth;
        this.maxHeight = this.options.maxHeight;

        this.baseZIndex = this.options.zindex;

        this.div = $(rf.getDomElement(id));
        this.cdiv = $(rf.getDomElement(id + "_container"));
        this.contentDiv = $(rf.getDomElement(id + "_content"));
        this.shadowDiv = $(rf.getDomElement(id + "_shadow"));
        this.shadeDiv = $(rf.getDomElement(id + "_shade"));
        this.scrollerDiv = $(rf.getDomElement(id + "_content_scroller"));

        $(this.shadowDiv).css("opacity", this.options.shadowOpacity);
        this.shadowDepth = parseInt(this.options.shadowDepth);

        this.borders = new Array();
        this.firstHref = $(rf.getDomElement(id + "FirstHref"));
        if (this.options.resizeable) {
            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerN", this, "N-resize", rf.ui.PopupPanel.Sizer.N));
            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerE", this, "E-resize", rf.ui.PopupPanel.Sizer.E));
            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerS", this, "S-resize", rf.ui.PopupPanel.Sizer.S));
            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerW", this, "W-resize", rf.ui.PopupPanel.Sizer.W));

            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerNW", this, "NW-resize", rf.ui.PopupPanel.Sizer.NW));
            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerNE", this, "NE-resize", rf.ui.PopupPanel.Sizer.NE));
            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerSE", this, "SE-resize", rf.ui.PopupPanel.Sizer.SE));
            this.borders.push(new rf.ui.PopupPanel.Border(id + "ResizerSW", this, "SW-resize", rf.ui.PopupPanel.Sizer.SW));
        }

        if (this.options.moveable && rf.getDomElement(id + "_header")) {
            this.header = new rf.ui.PopupPanel.Border(id + "_header", this, "move", rf.ui.PopupPanel.Sizer.Header);
        } else {
            $(rf.getDomElement(id + "_header")).css('cursor', 'default');
        }

        this.resizeProxy = $.proxy(this.resizeListener, this);

        this.cdiv.resize(this.resizeProxy);
        this.findForm(this.cdiv).on("ajaxcomplete", this.resizeProxy);
    };

    rf.BaseComponent.extend(rf.ui.PopupPanel);
    var $super = rf.ui.PopupPanel.$super;
    $.extend(rf.ui.PopupPanel.prototype, (function (options) {

        return {

            name: "PopupPanel",
            saveInputValues: function(element) {
                /* Fix for RF-3856 - Checkboxes in modal panel does not hold their states after modal was closed and opened again */
                if (rf.browser.msie /* reproducible for checkbox/radio in IE6, radio in IE 7/8 beta 2 */) {
                    $('input[type=checkbox], input[type=radio]', element).each(function(index) {
                        $(this).defaultChecked = $(this).checked;
                    });
                }
            },

            width: function() {
                return this.getContentElement()[0].clientWidth;//TODO
            },

            height: function() {
                return this.getContentElement()[0].clientHeight;//TODO
            },

            getLeft : function () {
                return this.cdiv.css('left');
            },

            getTop : function () {
                return this.cdiv.css('top');
            },

            getInitialSize : function() {
                if (this.options.autosized) {
                    return 15;
                } else {
                    return $(rf.getDomElement(this.markerId + "_header_content")).height();
                }
            },

            getContentElement: function() {
                if (!this._contentElement) {
                    this._contentElement = this.cdiv;
                }

                return this._contentElement;
            },
            getSizeElement : function() {
                return document.body;
            },

            getMinimumSize : function(size) {
                return Math.max(size, 2 * this.getInitialSize() + 2);
            },

            __getParsedOption: function(options, name) {
                var value = parseInt(options[name], 10);

                if (value < 0 || isNaN(value)) {
                    value = this[name];
                }

                return value;
            },

            destroy: function() {
                this.findForm(this.cdiv).off("ajaxcomplete", this.resizeProxy);

                this._contentElement = null;
                this.firstOutside = null;
                this.lastOutside = null;
                this.firstHref = null;
                this.parent = null;
                if (this.header) {
                    this.header.destroy();
                    this.header = null;
                }

                for (var k = 0; k < this.borders.length; k++) {
                    this.borders[k].destroy();
                }
                this.borders = null;

                if (this.domReattached) {
                    this.div.remove();
                }

                this.markerId = null;
                this.options = null;

                this.div = null;
                this.cdiv = null;
                this.contentDiv = null;
                this.shadowDiv = null;
                this.scrollerDiv = null;
                this.userOptions = null;
                this.eIframe = null;

                $super.destroy.call(this);

            },

            initIframe : function() {
                if (this.contentWindow) {
                    $(this.contentWindow.document.body).css("margin", "0px 0px 0px 0px");
                } else {
                    //TODO opera etc.

                }

                if ("transparent" == $(document.body).css("background-color")) {
                    $(this).css('filter', "alpha(opacity=0)");
                    $(this).css('opacity', "0");
                }
            },

            setLeft: function(pos) {
                if (!isNaN(pos)) {
                    this.cdiv.css('left', pos + "px");
                }
            },

            setTop: function(pos) {
                if (!isNaN(pos)) {
                    this.cdiv.css('top', pos + "px");
                }
            },

            show: function(event, opts) {
                var element = this.cdiv;
                if (!this.shown && this.invokeEvent("beforeshow", event, null, element)) {
                    this.preventFocus();


                    if (!this.domReattached) {
                        this.parent = this.div.parent();

                        var domElementAttachment;
                        if (opts) {
                            domElementAttachment = opts.domElementAttachment;
                        }

                        if (!domElementAttachment) {
                            domElementAttachment = this.options.domElementAttachment;
                        }

                        var newParent;
                        if ('parent' == domElementAttachment) {
                            newParent = this.parent;
                        } else if ('form' == domElementAttachment) {
                            newParent = this.findForm(element)[0] || document.body;
                        } else {
                            //default - body
                            newParent = document.body;
                        }

                        if (newParent != this.parent) {
                            this.saveInputValues(element);
                            this.shadeDiv.length && newParent.appendChild(this.shadeDiv.get(0));
                            newParent.appendChild(this.cdiv.get(0));
                            this.domReattached = true;
                        } else {
                            this.parent.show();
                        }
                    }

                    var forms = $("form", element);

                    if (this.options.keepVisualState && forms) {
                        for (var i = 0; i < forms.length; i++) {
                            var popup = this;
                            $(forms[i]).bind("submit", {popup:popup}, this.setStateInput);
                        }
                    }

                    var options = {};
                    this.userOptions = {};
                    $.extend(options, this.options);

                    if (opts) {
                        $.extend(options, opts);
                        $.extend(this.userOptions, opts);
                    }

                    // reset dimensions
                    if (this.options.autosized) {
                        if (options.left) {
                            var _left;
                            if (options.left != "auto") {
                                _left = parseInt(options.left, 10);
                            } else {
                                var cw = this.__calculateWindowWidth();
                                var _width = this.width();
                                if (cw >= _width) {
                                    _left = (cw - _width) / 2;
                                } else {
                                    _left = 0;
                                }
                            }

                            this.setLeft(Math.round(_left));
                            $(this.shadowDiv).css("left", this.shadowDepth);
                        }

                        if (options.top) {
                            var _top;
                            if (options.top != "auto") {
                                _top = parseInt(options.top, 10);
                            } else {
                                var ch = this.__calculateWindowHeight();
                                var _height = this.height();
                                if (ch >= _height) {
                                    _top = (ch - _height) / 2;
                                } else {
                                    _top = 0;
                                }
                            }

                            this.setTop(Math.round(_top));
                            $(this.shadowDiv).css("top", this.shadowDepth);
                            $(this.shadowDiv).css("bottom", -this.shadowDepth);
                        }

                        this.doResizeOrMove(rf.ui.PopupPanel.Sizer.Diff.EMPTY);
                    }

                    this.currentMinHeight = this.getMinimumSize(this.__getParsedOption(options, 'minHeight'));
                    this.currentMinWidth = this.getMinimumSize(this.__getParsedOption(options, 'minWidth'));

                    var eContentElt = this.getContentElement();

                    if (!this.options.autosized) {
                        if (options.width && options.width == -1)
                            options.width = 300;
                        if (options.height && options.height == -1)
                            options.height = 200;
                    }

                    this.div.css('visibility', '');
                    if (rf.browser.msie) {
                        $(this.cdiv).find('input').each(function() {
                            // Force a CSS "touch" of all popupPanel children to ensure visibility in IE for RF-12850
                            var $this = $(this);
                            if ($this.parents(".rf-pp-cntr").first().attr('id') === element.attr('id')) {
                                $this.css('visibility', $this.css('visibility'));
                            }
                        })
                    }
                    this.div.css('display', 'block');
                    if (this.options.autosized) {
                        this.shadowDiv.css('width', this.cdiv[0].clientWidth);

                    }

                    if (options.width && options.width != -1 || options.autosized) {
                        var width;
                        if (options.autosized) {
                            width = this.getStyle(this.getContentElement(), "width");
                            if (this.currentMinWidth > width) {
                                width = this.currentMinWidth;
                            }
                            if (width > this.maxWidth) {
                                width = this.maxWidth;
                            }
                        } else {
                            if (this.currentMinWidth > options.width) {
                                options.width = this.currentMinWidth;
                            }
                            if (options.width > this.maxWidth) {
                                options.width = this.maxWidth;
                            }
                            width = options.width;
                        }
                        $(rf.getDomElement(eContentElt)).css('width', width + (/px/.test(width) ? '' : 'px'));
                        this.shadowDiv.css('width', width + (/px/.test(width) ? '' : 'px'));
                        this.scrollerDiv.css('width', width + (/px/.test(width) ? '' : 'px'));
                    }

                    if (options.height && options.height != -1 || options.autosized) {
                        var height;
                        if (options.autosized) {
                            height = this.getStyle(this.getContentElement(), "height");
                            if (this.currentMinHeight > height) {
                                height = this.currentMinHeight;
                            }
                            if (height > this.maxHeight) {
                                height = this.maxHeight;
                            }
                        } else {
                            if (this.currentMinHeight > options.height) {
                                options.height = this.currentMinHeight;
                            }
                            if (options.height > this.maxHeight) {
                                options.height = this.maxHeight;
                            }
                            height = options.height;
                        }
                        $(rf.getDomElement(eContentElt)).css('height', height + (/px/.test(height) ? '' : 'px'));
                        var headerHeight = $(rf.getDomElement(this.markerId + "_header")) ? $(rf.getDomElement(this.markerId + "_header")).innerHeight() : 0;
                        this.shadowDiv.css('height', height + (/px/.test(height) ? '' : 'px'));
                        this.scrollerDiv.css('height', height - headerHeight + (/px/.test(height) ? '' : 'px'));
                    }

                    var eIframe;
                    if (this.options.overlapEmbedObjects && !this.iframe) {
                        this.iframe = this.markerId + "IFrame";
                        $("<iframe src=\"javascript:''\" frameborder=\"0\" scrolling=\"no\" id=\"" + this.iframe + "\" " +
                            "class=\"rf-pp-ifr\" style=\"width:" + this.options.width + "px; height:" + this.options.height + "px;\">" +
                            "</iframe>").insertBefore($(':first-child', this.cdiv)[0]);

                        eIframe = $(rf.getDomElement(this.iframe));

                        eIframe.bind('load', this.initIframe);
                        this.eIframe = eIframe;
                    }

                    if (options.left) {
                        var _left;
                        if (options.left != "auto") {
                            _left = parseInt(options.left, 10);
                        } else {
                            var cw = this.__calculateWindowWidth();
                            var _width = this.width();
                            if (cw >= _width) {
                                _left = (cw - _width) / 2;
                            } else {
                                _left = 0;
                            }
                        }

                        this.setLeft(Math.round(_left));
                        $(this.shadowDiv).css("left", this.shadowDepth);
                    }

                    if (options.top) {
                        var _top;
                        if (options.top != "auto") {
                            _top = parseInt(options.top, 10);
                        } else {
                            var ch = this.__calculateWindowHeight();
                            var _height = this.height();
                            if (ch >= _height) {
                                _top = (ch - _height) / 2;
                            } else {
                                _top = 0;
                            }
                        }

                        this.setTop(Math.round(_top));
                        $(this.shadowDiv).css("top", this.shadowDepth);
                        $(this.shadowDiv).css("bottom", -this.shadowDepth);
                    }

                    var showEvent = {};
                    showEvent.parameters = opts || {};
                    this.shown = true;
                    // Cache the height difference between the shadoww div and the scroller div for later height calculations
                    this.scrollerSizeDelta = parseInt(this.shadowDiv.css('height')) - parseInt(this.scrollerDiv.css('height'));
                    this.invokeEvent("show", showEvent, null, element);
                }
            },

            __calculateWindowHeight: function() {
                var documentElement = document.documentElement;
                return self.innerHeight || (documentElement && documentElement.clientHeight) || document.body.clientHeight;
            },

            __calculateWindowWidth: function() {
                var documentElement = document.documentElement;
                return self.innerWidth || (documentElement && documentElement.clientWidth) || document.body.clientWidth;
            },

            startDrag: function(border) {
                disableSelection(document.body);
            },
            firstOnfocus: function(event) {
                var e = $(event.data.popup.firstHref);
                if (e) {
                    e.focus();
                }
            },

            processAllFocusElements: function(root, callback) {
                var idx = -1;
                var tagName;
                var formElements = "|a|input|select|button|textarea|";

                if (root.focus && root.nodeType == 1 && (tagName = root.tagName) &&
                    // Many not visible elements have focus method, we is had to avoid processing them.
                    (idx = formElements.indexOf(tagName.toLowerCase())) != -1 &&
                    formElements.charAt(idx - 1) === '|' &&
                    formElements.charAt(idx + tagName.length) === '|' &&
                    !root.disabled && root.type != "hidden") {
                    callback.call(this, root);
                } else {
                    if (root != this.cdiv.get(0)) {
                        var child = root.firstChild;
                        while (child) {
                            if (!child.style || child.style.display != 'none') {
                                this.processAllFocusElements(child, callback);
                            }
                            child = child.nextSibling;
                        }
                    }
                }
            },

            processTabindexes:    function(input) {
                if (!this.firstOutside) {
                    this.firstOutside = input;
                }
                if (!input.prevTabIndex) {
                    input.prevTabIndex = input.tabIndex;
                    input.tabIndex = -1;
                }

                if (!input.prevAccessKey) {
                    input.prevAccessKey = input.accessKey;
                    input.accessKey = "";
                }
            },

            restoreTabindexes:    function(input) {
                if (input.prevTabIndex != undefined) {
                    if (input.prevTabIndex == 0) {
                        $(input).removeAttr('tabindex');
                    } else {
                        input.tabIndex = input.prevTabIndex;
                    }
                    input.prevTabIndex = undefined;
                }
                if (input.prevAccessKey != undefined) {
                    if (input.prevAccessKey == "") {
                        $(input).removeAttr('accesskey');
                    } else {
                        input.accessKey = input.prevAccessKey;
                    }
                    input.prevAccessKey = undefined;
                }
            },

            preventFocus:    function() {
                if (this.options.modal) {
                    this.processAllFocusElements(document, this.processTabindexes);
                    var popup = this;
                    if (this.firstOutside) {

                        $(rf.getDomElement(this.firstOutside)).bind("focus", {popup: popup}, this.firstOnfocus);
                    }
                }
            },

            restoreFocus: function() {
                if (this.options.modal) {
                    this.processAllFocusElements(document, this.restoreTabindexes);

                    if (this.firstOutside) {
                        $(rf.getDomElement(this.firstOutside)).unbind("focus", this.firstOnfocus);
                        this.firstOutside = null;
                    }
                }
            },

            endDrag: function(border) {
                for (var k = 0; k < this.borders.length; k++) {
                    this.borders[k].show();
                    this.borders[k].doPosition();
                }
                enableSelection(document.body);
            },

            hide: function(event, opts) {
                var element = this.cdiv;
                this.restoreFocus();
                if (this.shown && this.invokeEvent("beforehide", event, null, element)) {

                    this.currentMinHeight = undefined;
                    this.currentMinWidth = undefined;

                    this.div.hide();

                    if (this.parent) {
                        if (this.domReattached) {
                            this.saveInputValues(element);
                            var div = this.div.get(0);
                            this.shadeDiv.length && div.appendChild(this.shadeDiv.get(0));
                            div.appendChild(element.get(0));

                            this.domReattached = false;
                        }
                    }

                    var hideEvent = {};
                    hideEvent.parameters = opts || {};

                    var forms = $("form", element);
                    if (this.options.keepVisualState && forms) {
                        for (var i = 0; i < forms.length; i++) {
                            $(forms[i]).unbind("submit", this.setStateInput);
                        }
                    }

                    this.shown = false;
                    this.invokeEvent("hide", hideEvent, null, element);

                    // reset position for proper resizing
                    this.setLeft(10);
                    this.setTop(10);
                }
            },

            getStyle: function(elt, name) {
                return parseInt($(rf.getDomElement(elt)).css(name).replace("px", ""), 10);
            },

            resizeListener: function(event, diff) {
                this.doResizeOrMove(rf.ui.PopupPanel.Sizer.Diff.EMPTY);
            },

            doResizeOrMove: function(diff) {
                var vetoes = {};
                var shadowHash = {};
                var cssHash = {};
                var cssHashWH = {};
                var shadowHashWH = {};
                var contentHashWH = {};
                var scrollerHashWH = {};
                var newSize;
                var scrollerHeight = this.scrollerSizeDelta;
                var scrollerWidth = 0;
                var eContentElt = this.getContentElement();

                var doResize = diff === rf.ui.PopupPanel.Sizer.Diff.EMPTY || diff.deltaWidth || diff.deltaHeight;

                if (doResize) {
                    if (this.options.autosized) {
                        this.resetHeight();
                        this.resetWidth();
                    }

                    newSize = this.getStyle(eContentElt, "width");

                    var oldWidthSize = newSize;
                    newSize += diff.deltaWidth || 0;

                    if (newSize >= this.currentMinWidth) {
                        cssHashWH.width = newSize + 'px';
                        shadowHashWH.width = newSize + 'px';
                        contentHashWH.width = newSize - scrollerWidth + 'px';
                        scrollerHashWH.width = newSize - scrollerWidth + 'px';
                    } else {
                        cssHashWH.width = this.currentMinWidth + 'px';
                        shadowHashWH.width = this.currentMinWidth + 'px';
                        contentHashWH.width = this.currentMinWidth - scrollerWidth + 'px';
                        scrollerHashWH.width = this.currentMinWidth - scrollerWidth + 'px';

                        if (diff.deltaWidth) {
                            vetoes.vx = oldWidthSize - this.currentMinWidth;
                            vetoes.x = true;
                        }
                    }

                    if (newSize > this.options.maxWidth) {
                        cssHashWH.width = this.options.maxWidth + 'px';
                        shadowHashWH.width = this.options.maxWidth + 'px';
                        contentHashWH.width = this.options.maxWidth - scrollerWidth + 'px';
                        scrollerHashWH.width = this.options.maxWidth - scrollerWidth + 'px';

                        if (diff.deltaWidth) {
                            vetoes.vx = oldWidthSize - this.options.maxWidth;
                            vetoes.x = true;
                        }
                    }
                }

                if (vetoes.vx && diff.deltaX) {
                    diff.deltaX = -vetoes.vx;
                }

                var eCdiv = $(this.cdiv);

                if (diff.deltaX && (vetoes.vx || !vetoes.x)) {
                    if (vetoes.vx) {
                        diff.deltaX = vetoes.vx;
                    }

                    var newLeftPos = this.getStyle(eCdiv, "left");
                    newLeftPos += diff.deltaX;
                    cssHash.left = newLeftPos + 'px';

                }

                if (doResize) {
                    newSize = this.getStyle(eContentElt, "height");

                    var oldHeightSize = newSize;
                    newSize += diff.deltaHeight || 0;

                    if (newSize >= this.currentMinHeight) {
                        cssHashWH.height = newSize + 'px';
                        shadowHashWH.height = newSize + 'px';
                        scrollerHashWH.height = newSize - scrollerHeight + 'px';
                    } else {
                        cssHashWH.height = this.currentMinHeight + 'px';
                        shadowHashWH.height = this.currentMinHeight + 'px';
                        scrollerHashWH.height = this.currentMinHeight - scrollerHeight + 'px';

                        if (diff.deltaHeight) {
                            vetoes.vy = oldHeightSize - this.currentMinHeight;
                            vetoes.y = true;
                        }
                    }

                    if (newSize > this.options.maxHeight) {
                        cssHashWH.height = this.options.maxHeight + 'px';
                        shadowHashWH.height = this.options.maxHeight + 'px';
                        scrollerHashWH.height = this.options.maxHeight - scrollerHeight + 'px';

                        if (diff.deltaHeight) {
                            vetoes.vy = oldHeightSize - this.options.maxHeight;
                            vetoes.y = true;
                        }
                    }
                }

                if (vetoes.vy && diff.deltaY) {
                    diff.deltaY = -vetoes.vy;
                }

                if (diff.deltaY && (vetoes.vy || !vetoes.y)) {
                    if (vetoes.vy) {
                        diff.deltaY = vetoes.vy;
                    }

                    var newTopPos = this.getStyle(eCdiv, "top");
                    newTopPos += diff.deltaY;
                    cssHash.top = newTopPos + 'px';
                }

                eContentElt.css(cssHashWH);
                this.scrollerDiv.css(scrollerHashWH);
                if (this.eIframe) {
                    this.eIframe.css(scrollerHashWH);
                }
                this.shadowDiv.css(shadowHashWH);

                eCdiv.css(cssHash);
                this.shadowDiv.css(shadowHash);

                $.extend(this.userOptions, cssHash);
                $.extend(this.userOptions, cssHashWH);
                var w = this.width();
                var h = this.height();

                this.reductionData = null;

                if (w <= 2 * this.getInitialSize()) {
                    this.reductionData = {};
                    this.reductionData.w = w;
                }

                if (h <= 2 * this.getInitialSize()) {
                    if (!this.reductionData) {
                        this.reductionData = {};
                    }

                    this.reductionData.h = h;
                }

                if (this.header) {
                    this.header.doPosition();
                }

                return vetoes;
            },

            resetWidth: function() {
                this.getContentElement().css('width', '');
                this.scrollerDiv.css('width', '');
                if (this.eIframe) {
                    this.eIframe.css('width', '');
                }
                this.shadowDiv.css('width', '');
                $(this.cdiv).css('width', '');
            },

            resetHeight: function() {
                this.getContentElement().css('height', '');
                this.scrollerDiv.css('height', '');
                if (this.eIframe) {
                    this.eIframe.css('height', '');
                }
                this.shadowDiv.css('height', '');
                $(this.cdiv).css('height', '');
            },

            setSize : function (width, height) {
                var w = width - this.width();
                var h = height - this.height();
                var diff = new rf.ui.PopupPanel.Sizer.Diff(0, 0, w, h);
                this.doResizeOrMove(diff);
            },

            moveTo : function (top, left) {
                this.cdiv.css('top', top);
                this.cdiv.css('left', left);
            },

            move : function (dx, dy) {
                var diff = new rf.ui.PopupPanel.Sizer.Diff(dx, dy, 0, 0);
                this.doResizeOrMove(diff);
            },

            resize : function (dx, dy) {
                var diff = new rf.ui.PopupPanel.Sizer.Diff(0, 0, dx, dy);
                this.doResizeOrMove(diff);
            },

            findForm: function(elt) {
                var target = elt;
                while (target) {
                    if (target[0] && (!target[0].tagName /* document node doesn't have tagName */
                        || target[0].tagName.toLowerCase() != "form")) {

                        target = $(target).parent();
                    } else {
                        break;
                    }
                }

                return target;
            },

            setStateInput: function(event) {
                // Concret input but not entire form is a target element for onsubmit in FF
                var popup = event.data.popup;
                target = $(popup.findForm(event.currentTarget));

                var input = document.createElement("input");
                input.type = "hidden";
                input.id = popup.markerId + "OpenedState";
                input.name = popup.markerId + "OpenedState";
                input.value = popup.shown ? "true" : "false";
                target.append(input);

                $.each(popup.userOptions, function(key, value) {
                    input = document.createElement("input");
                    input.type = "hidden";
                    input.id = popup.markerId + "StateOption_" + key;
                    input.name = popup.markerId + "StateOption_" + key;
                    input.value = value;
                    target.append(input);
                });

                return true;
            }


        }

    })());
    $.extend(rf.ui.PopupPanel, {

            showPopupPanel : function (id, opts, event) {
                rf.Event.ready(function() {
                    rf.component(id).show()
                });
            },

            hidePopupPanel : function (id, opts, event) {
                rf.Event.ready(function() {
                    rf.component(id).hide()
                });
            }
        });

})(RichFaces.jQuery, window.RichFaces);
;

// resource: org.richfaces:component-control.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.ComponentControl = rf.ui.ComponentControl || {};

    $.extend(rf.ui.ComponentControl, {

            execute: function(event, parameters) {
                var targetList = parameters.target;
                var selector = parameters.selector;
                var callback = parameters.callback;

                if (parameters.onbeforeoperation && typeof parameters.onbeforeoperation == "function") {
                    var result = parameters.onbeforeoperation(event);
                    if (result == "false" || result == 0) return;
                }

                if (targetList) {
                    for (var i = 0; i < targetList.length; i++) {
                        var component = document.getElementById(targetList[i]);
                        if (component) {
                            rf.ui.ComponentControl.invokeOnComponent(event, component, callback);
                        }
                    }
                }

                if (selector) {
                    rf.ui.ComponentControl.invokeOnComponent(event, selector, callback);
                }
            },

            invokeOnComponent : function(event, target, callback) {
                if (callback && typeof callback == 'function') {
                    $(target).each(function() {
                        var component = rf.component(this) || this;
                        callback(event, component);
                    });
                }
            }
        });

})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:fileupload.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
(function($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.FileUpload = function(id, options) {
        this.id = id;
        this.items = [];
        this.submitedItems = [];

        $.extend(this, options);
        if (this.acceptedTypes) {
            this.acceptedTypes = $.trim(this.acceptedTypes).toUpperCase().split(/\s*,\s*\.?/);
        }
        if (this.maxFilesQuantity) {
            this.maxFilesQuantity = parseInt($.trim(this.maxFilesQuantity));
        }
        this.element = $(this.attachToDom());
        this.form = this.element.parents("form:first");
        var header = this.element.children(".rf-fu-hdr:first");
        var leftButtons = header.children(".rf-fu-btns-lft:first");
        this.addButton = leftButtons.children(".rf-fu-btn-add:first");
        this.uploadButton = this.addButton.next();
        this.clearButton = leftButtons.next().children(".rf-fu-btn-clr:first");
        this.inputContainer = this.addButton.find(".rf-fu-inp-cntr:first");
        this.input = this.inputContainer.children("input");
        this.list = header.next();
        this.element.bind('dragenter', function(e) {e.stopPropagation(); e.preventDefault();});
        this.element.bind('dragover', function(e) {e.stopPropagation(); e.preventDefault();});
        this.element.bind('drop', $.proxy(this.__addItemsFromDrop, this));

        this.hiddenContainer = this.list.next();
        this.cleanInput = this.input.clone();
        this.addProxy = $.proxy(this.__addItems, this);
        this.input.change(this.addProxy);
        this.addButton.mousedown(pressButton).mouseup(unpressButton).mouseout(unpressButton);
        this.uploadButton.click($.proxy(this.__startUpload, this)).mousedown(pressButton)
            .mouseup(unpressButton).mouseout(unpressButton);
        this.clearButton.click($.proxy(this.__removeAllItems, this)).mousedown(pressButton)
            .mouseup(unpressButton).mouseout(unpressButton);
        if (this.onfilesubmit) {
            rf.Event.bind(this.element, "onfilesubmit", new Function("event", this.onfilesubmit));
        }
        if (this.ontyperejected) {
            rf.Event.bind(this.element, "ontyperejected", new Function("event", this.ontyperejected));
        }
        if (this.onsizerejected) {
            rf.Event.bind(this.element, "onsizerejected", new Function("event", this.onsizerejected));
        }
        if (this.onuploadcomplete) {
            rf.Event.bind(this.element, "onuploadcomplete", new Function("event", this.onuploadcomplete));
        }
        if (this.onclear) {
            rf.Event.bind(this.element, "onclear", new Function("event", this.onclear));
        }
        if (this.onfileselect) {
            rf.Event.bind(this.element, "onfileselect", new Function("event", this.onfileselect));
        }
    };

    var UID = "rf_fu_uid";
    var UID_ALT = "rf_fu_uid_alt";
    var FAKE_PATH = "C:\\fakepath\\";
    var ITEM_HTML = '<div class="rf-fu-itm">'
        + '<span class="rf-fu-itm-lft"><span class="rf-fu-itm-lbl"/><span class="rf-fu-itm-st" />'
        + '<div class="progress progress-striped active">'
        + '<div class="progress-bar" role="progressbar" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100" style="width: 0%">'
        + '<span></span></div></div></span>'
        + '<span class="rf-fu-itm-rgh"><a href="javascript:void(0)" class="rf-fu-itm-lnk"/></span></div>';

    var ITEM_STATE = {
        NEW: "new",
        UPLOADING: "uploading",
        DONE: "done",
        SIZE_EXCEEDED: "sizeExceeded",
        STOPPED: "stopped",
        SERVER_ERROR_PROCESS: "serverErrorProc",
        SERVER_ERROR_UPLOAD: "serverErrorUp"
    };

    var pressButton = function(event) {
        $(this).children(":first").css("background-position", "3px 3px").css("padding", "4px 4px 2px 22px");
    };

    var unpressButton = function(event) {
        $(this).children(":first").css("background-position", "2px 2px").css("padding", "3px 5px 3px 21px");
    };

    rf.BaseComponent.extend(rf.ui.FileUpload);

    function TypeRejectedException(fileName) {
        this.name = "TypeRejectedException";
        this.message = "The type of file " + fileName + " is not accepted";
        this.fileName = fileName;
    }

    $.extend(rf.ui.FileUpload.prototype, (function () {

        return {
            name: "FileUpload",

            doneLabel: "Done",
            sizeExceededLabel: "File size is exceeded",
            stoppedLabel: "",
            serverErrorProcLabel: "Server error: error in processing",
            serverErrorUpLabel: "Server error: upload failed",
            clearLabel: "Clear",
            deleteLabel: "Delete",

            __addFiles : function(files) {
                var context = {
                    acceptedFileNames: [],
                    rejectedFileNames: []
                };

                if (files) {
                    for (var i = 0 ; i < files.length; i++) {
                        this.__tryAddItem(context, files[i]);

                        if (this.maxFilesQuantity && this.__getTotalItemCount() >= this.maxFilesQuantity) {
                            this.addButton.hide();
                            break;
                        }
                    }
                } else {
                    var fileName = this.input.val();
                    this.__tryAddItem(context, fileName);
                }

                if (context.rejectedFileNames.length > 0) {
                    rf.Event.fire(this.element, "ontyperejected", context.rejectedFileNames.join(','));
                }

                if (this.immediateUpload) {
                    this.__startUpload();
                }
            },

            __addItems : function() {
                this.__addFiles(this.input.prop("files"));
            },

            __addItemsFromDrop: function(dropEvent) {
                dropEvent.stopPropagation();
                dropEvent.preventDefault();
                
                if (this.maxFilesQuantity && this.__getTotalItemCount() >= this.maxFilesQuantity) {
                    return;
                }

                this.__addFiles(dropEvent.originalEvent.dataTransfer.files);
            },

            __tryAddItem: function(context, file) {
                if (this.maxFileSize && file.size > this.maxFileSize) {
                    rf.Event.fire(this.element, "onsizerejected", file);
                    return;
                }
                try {
                    if (this.__addItem(file)) {
                        context.acceptedFileNames.push(file.name);
                    }
                } catch (e) {
                    if (e instanceof TypeRejectedException) {
                        context.rejectedFileNames.push(file.name);
                    } else {
                        throw e;
                    }
                }
            },

            __addItem: function(file) {
                var fileName = file.name;
                if (!navigator.platform.indexOf("Win")) {
                    fileName = fileName.match(/[^\\]*$/)[0];
                } else {
                    if (!fileName.indexOf(FAKE_PATH)) {
                        fileName = fileName.substr(FAKE_PATH.length);
                    } else {
                        fileName = fileName.match(/[^\/]*$/)[0];
                    }
                }
                if (this.__accept(fileName) && (!this.noDuplicate || !this.__isFileAlreadyAdded(fileName))) {
                    this.input.remove();
                    this.input.unbind("change", this.addProxy);
                    var item = new Item(this, file);
                    this.list.append(item.getJQuery());
                    this.items.push(item);
                    this.input = this.cleanInput.clone();
                    this.inputContainer.append(this.input);
                    this.input.change(this.addProxy);
                    this.__updateButtons();
                    rf.Event.fire(this.element, "onfileselect", fileName);
                    return true;
                }

                return false;
            },

            __removeItem: function(item) {
                var inItems = $.inArray(item, this.items),
                    inSItems = $.inArray(item, this.submitedItems);
                if (inItems != -1) {    
                    this.items.splice(inItems, 1);
                }
                if (inSItems != -1) {
                    this.submitedItems.splice(inSItems, 1);
                }
                this.__updateButtons();
                rf.Event.fire(this.element, "onclear", [item.model]);
            },

            __removeAllItems: function(item) {
                var itemsRemoved = [];
                for (var i = 0; i < this.submitedItems.length; i++) {
                    itemsRemoved.push(this.submitedItems[i].model);
                }
                for (var i = 0; i < this.items.length; i++) {
                    itemsRemoved.push(this.items[i].model);
                }
                this.list.empty();
                this.items.splice(0, this.items.length);
                this.submitedItems.splice(0, this.submitedItems.length);
                this.__updateButtons();
                rf.Event.fire(this.element, "onclear", itemsRemoved);
            },

            __updateButtons: function() {
                if (!this.loadableItem && this.list.children(".rf-fu-itm").size()) {
                    if (this.items.length) {
                        this.uploadButton.css("display", "inline-block");
                    } else {
                        this.uploadButton.hide();
                    }
                    this.clearButton.css("display", "inline-block");
                } else {
                    this.uploadButton.hide();
                    this.clearButton.hide();
                }
                if (this.maxFilesQuantity && this.__getTotalItemCount() >= this.maxFilesQuantity) {
                    this.addButton.hide();
                } else {
                    this.addButton.css("display", "inline-block");
                }
            },

            __startUpload: function() {
                if (!this.items.length) {
                    this.__finishUpload();
                    return;
                }
                rf.setGlobalStatusNameVariable(this.status);
                this.loadableItem = this.items.shift();
                this.__updateButtons();
                this.loadableItem.startUploading();
            },

            __accept: function(fileName) {
                fileName = fileName.toUpperCase();
                var result = !this.acceptedTypes;
                for (var i = 0; !result && i < this.acceptedTypes.length; i++) {
                    var extension = this.acceptedTypes[i];

                    if (extension === "" && fileName.indexOf(".") < 0) {
                        // no extension
                        result = true;
                    } else {
                        result = fileName.indexOf(extension, fileName.length - extension.length) !== -1;
                    }
                }
                if (!result) {
                    throw new TypeRejectedException(fileName);
                }
                return result;
            },

            __isFileAlreadyAdded: function(fileName) {
                var result = false;
                for (var i = 0; !result && i < this.items.length; i++) {
                    result = this.items[i].model.name == fileName;
                }
                result = result || (this.loadableItem && this.loadableItem.model.name == fileName);
                for (var i = 0; !result && i < this.submitedItems.length; i++) {
                    result = this.submitedItems[i].model.name == fileName;
                }
                return result;
            },


            __getTotalItemCount : function() {
                return this.__getItemCountByState(this.items, ITEM_STATE.NEW)
                    + this.__getItemCountByState(this.submitedItems, ITEM_STATE.DONE);
            },

            __getItemCountByState : function(items) {
                var statuses = {};
                var s = 0;
                for ( var i = 1; i < arguments.length; i++) {
                    statuses[arguments[i]] = true;
                }
                for ( var i = 0; i < items.length; i++) {
                    if (statuses[items[i].model.state]) {
                        s++;
                    }
                }
                return s;
            },

            __finishUpload : function() {
                this.loadableItem = null;
                this.__updateButtons();
                var items = [];
                for (var i = 0; i < this.submitedItems.length; i++) {
                    items.push(this.submitedItems[i].model);
                }
                for (var i = 0; i < this.items.length; i++) {
                    items.push(this.items[i].model);
                }
                rf.Event.fire(this.element, "onuploadcomplete", items);
            }
        };
    })());


    var Item = function(fileUpload, file) {
        this.fileUpload = fileUpload;
        this.model = {name: file.name, state: ITEM_STATE.NEW, file: file};
    };

    $.extend(Item.prototype, {
            getJQuery: function() {
                this.element = $(ITEM_HTML);
                var leftArea = this.element.children(".rf-fu-itm-lft:first");
                this.label = leftArea.children(".rf-fu-itm-lbl:first");
                this.state = this.label.nextAll(".rf-fu-itm-st:first");
                this.progressBar = leftArea.find(".progress-bar");
                this.progressBar.parent().hide();
                this.progressLabel = this.progressBar.find('span');
                this.link = leftArea.next().children("a");
                this.label.html(this.model.name);
                this.link.html(this.fileUpload["deleteLabel"]);
                this.link.click($.proxy(this.removeOrStop, this));
                return this.element;
            },

            removeOrStop: function() {
                this.element.remove();
                this.fileUpload.__removeItem(this);
            },

            startUploading: function() {
                this.state.css("display", "block");
                this.progressBar.parent().show();
                this.progressLabel.html("0 %");
                this.link.html("");
                this.model.state = ITEM_STATE.UPLOADING;
                this.uid = Math.random();

                var formData = new FormData(),
                    fileName = this.model.file.name,
                    viewState = this.fileUpload.form.find("input[name='javax.faces.ViewState']").val();

                formData.append(this.fileUpload.form[0].id, this.fileUpload.form[0].id);
                formData.append("javax.faces.ViewState", viewState);
                formData.append(this.fileUpload.id, this.model.file);

                var originalAction = this.fileUpload.form.attr("action"),
                    delimiter = originalAction.indexOf("?") == -1 ? "?" : "&",
                    encodedId = encodeURIComponent(this.fileUpload.id),
                    newAction =  originalAction + delimiter + UID + "=" + this.uid + 
                        "&javax.faces.partial.ajax=true" + 
                        "&javax.faces.source="           + encodedId +
                        "&javax.faces.partial.execute="  + encodedId +
                        "&org.richfaces.ajax.component=" + encodedId + 
                        "&javax.faces.ViewState=" + encodeURIComponent(viewState);

                if (jsf.getClientWindow && jsf.getClientWindow()) {
                    newAction += "&javax.faces.ClientWindow=" + encodeURIComponent(jsf.getClientWindow());
                };

                var eventHandler = function(handlerCode) {
                    if (handlerCode) {
                        var safeHandlerCode = "try {" +
                                handlerCode + 
                            "} catch (e) {" +
                                "window.RichFaces.log.error('Error in method execution: ' + e.message)" +
                            "}";

                        return new Function("event", safeHandlerCode);
                    }
                }
                
                this.onerror = eventHandler(this.fileUpload.onerror);
                
                this.xhr = new XMLHttpRequest();

                this.xhr.open('POST', newAction, true);
                this.xhr.setRequestHeader('Faces-Request', 'partial/ajax');
                
                this.xhr.upload.onprogress = $.proxy(function(e) {
                        if (e.lengthComputable) {
                            var progress = Math.floor((e.loaded / e.total) * 100);
                            this.progressLabel.html( progress + " %" );
                            this.progressBar.attr("aria-valuenow", progress);
                            this.progressBar.css("width", progress + "%");
                        }
                    }, this);

                this.xhr.upload.onerror = $.proxy(function (e) {
                        this.fileUpload.loadableItem = null;
                        this.finishUploading(ITEM_STATE.SERVER_ERROR_UPLOAD);
                    }, this);
                
                this.xhr.onload = $.proxy(function (e) {
                    switch (e.target.status) {
                        case 413:
                            responseStatus = ITEM_STATE.SIZE_EXCEEDED;
                            break;
                        case 200:
                            responseStatus = ITEM_STATE.DONE;
                            break;
                        default: // 500 - error in processing parts
                            responseStatus = ITEM_STATE.SERVER_ERROR_PROCESS;
                    }
                    
                    var handlerFunction = function(handlerName) {
                        return function (event) {
                            var xml = $("partial-response extension#org\\.richfaces\\.extension", event.responseXML)
                            
                            var handlerCode = xml.children(handlerName).text();
                            
                            var handler = eventHandler(handlerCode);
                            
                            if (handler) {
                                handler.call(this,event);
                            }
                            $("form").trigger('ajaxcomplete');
                        }
                    }

                    var eventsAdapter = rf.createJSFEventsAdapter({
                        complete: handlerFunction('complete'),
                        error: handlerFunction('error')
                    });

                    var responseContext = {
                            source: this.fileUpload.element[0],
                            element: this.fileUpload.element[0],
                            /* hack for MyFaces */
                            _mfInternal: {
                                _mfSourceControlId: this.fileUpload.element.attr('id')
                            },
                            onevent: eventsAdapter,
                            onerror: eventsAdapter
                        };
                        
                    var onbeforedomupdate = eventHandler(this.fileUpload.onbeforedomupdate);
                    
                    if (onbeforedomupdate) {
                        var data = {};
                        data.type = "event";
                        data.status = 'complete';
                        data.source = this.fileUpload.element[0];
                        data.responseCode = this.xhr.status;
                        data.responseXML = this.xhr.responseXML;
                        data.responseText = this.xhr.responseText
    
                        onbeforedomupdate.call(this.fileUpload, data);
                    }
                    this.fileUpload.form.trigger('ajaxbeforedomupdate');
                    jsf.ajax.response(this.xhr, responseContext);
                    this.finishUploading(responseStatus);
                    this.fileUpload.__startUpload();
                    }, this);
                
                var onbegin = eventHandler(this.fileUpload.onbegin);
                
                if (onbegin) {
                    onbegin.call(this.fileUpload, {
                        source: this.fileUpload.element[0],
                        type: 'event',
                        status: 'begin'
                    })
                }
                this.fileUpload.form.trigger('ajaxbegin');
                this.xhr.send(formData);

                rf.Event.fire(this.fileUpload.element, "onfilesubmit", this.model);
            },

            finishUploading: function(state) {
                if (state != ITEM_STATE.DONE && this.onerror) {
                    this.onerror.call(this.fileUpload, {state: state, error: this.fileUpload[state + "Label"]});
                }
                this.state.html(this.fileUpload[state + "Label"]);
                this.progressBar.parent().hide();
                this.link.html(this.fileUpload["clearLabel"]);
                this.model.state = state;
                this.fileUpload.submitedItems.push(this);
            }
        });
}(RichFaces.jQuery, window.RichFaces));;

// resource: org.richfaces:collapsible-subtable-toggler.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.CollapsibleSubTableToggler = function(id, options) {
        this.id = id;
        this.eventName = options.eventName;
        this.expandedControl = options.expandedControl;
        this.collapsedControl = options.collapsedControl;
        this.forId = options.forId;
        this.element = $(document.getElementById(this.id));

        if (this.element && this.eventName) {
            this.element.bind(this.eventName, $.proxy(this.switchState, this));
        }
    };

    $.extend(rf.ui.CollapsibleSubTableToggler.prototype, (function () {

        var getElementById = function(id) {
            return $(document.getElementById(id))
        }

        return {

            switchState: function(e) {
                var subtable = rf.component(this.forId);
                if (subtable) {
                    var mode = subtable.getMode();

                    if (rf.ui.CollapsibleSubTable.MODE_CLNT == mode) {
                        this.toggleControl(subtable.isExpanded());
                    }

                    subtable.setOption(this.id);
                    subtable.switchState(e);
                }
            },

            toggleControl: function(collapse) {
                var expandedControl = getElementById(this.expandedControl);
                var collapsedControl = getElementById(this.collapsedControl);

                if (collapse) {
                    expandedControl.hide();
                    collapsedControl.show();
                } else {
                    collapsedControl.hide();
                    expandedControl.show();
                }
            }
        };
    })());

})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:chart.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {
	var defaultOptions = {

	      /**
	       * Specifies the type of chart.
	       * According to this property and xtypex ytype, proper default options corresponding to chart type
	       * are applied and data are transformed to the format expected by flot.
	       * Following values are supported: `line`,`bar`,`pie`
	       * Options is not required. If it is not used the options and data format remains the same as flot uses by default.
	       * @property charttype
	       * @default null
	       */
	      charttype: '',

	      /**
	       * Specify the data type of values plotted on x-axis
	       * Following options are supported: number,string,date
	       * @property xtype
	       * @default null
	       */
	      xtype: '',

	      /**
	       * Specify the data type of values plotted on x-axis
	       * Following options are supported: number,string,date
	       * @property xtype
	       * @default null
	       */
	      ytype: '',

	      /**
	       * Allows to zoom the chart. Supported only when line chart is used. Requires charttype to be set.
	       * @property zoom
	       * @default false
	       */
	      zoom: false,



	     /**
	      * Options customizing the chart grid.
	      * @type Object
	      * @default {clickable:true, hoverable:true}
	      * @property grid
	      *   @property clickable {boolean}
	      *   Grid accepts click events.
	      *   @property hoverable {boolean}
	      *   Grid fires mouseover event. Necessary for tooltip to work.
	      *
	      */
	     grid: {
	        clickable: true,
	        hoverable: true
	      },

	      /**
	       * Turns on tooltip (text shown when mouse points to a part of a chart)
	       * @property tooltip
	       * @type boolean
	       * @default true
	       */
	      tooltip: true,

	      /**
	       * Customizes the tooltip.
	       * @type Object
	       * @default { content: '%s [%x,%y]'}
	       * @property tooltipOpts
	       *    @property content {String}
	       *    Specify the tooltip format. Use %s for series label, %x for X values, %y for Y value
	       *    @property defaultTheme
	       */
	      tooltipOpts: {
	        content: '%s  [%x,%y]',
	        shifts: {
	          x: 20,
	          y: 0
	        },
	        defaultTheme: false
	      },

	      /**
	       * Legend properties
	       * @type Object
	       * @default {postion:'ne', sorted: 'ascending'}
	       * @property legend
	       *    @property position {String}
	       *    Defines the placement of the legend in the grid. One of ne,nw,se,sw
	       *    @property sorted {String}
	       *    Defines the order of labels in the legend. One of ascending,descending,false.
	       */
	      legend: {
	        postion:'ne',
	        sorted: 'ascending'
	      },

	      /**
	       * Customizes the horizontal axis
	       * @type Object
	       * @default {min: null, max: null,autoscaleMargin: null, axisLabel: ''}
	       * @property xaxis
	       *   @property min {Number}
	       *   Minimal value shown on axis
	       *   @property max {Number}
	       *   Maximal values show on axis
	       *   @property autoscaleMargin {Number}
	       *   It's the fraction of margin that the scaling algorithm will add
	       *   to avoid that the outermost points ends up on the grid border
	       *   @property axisLabel {String}
	       *   Axis description
	       */
	      xaxis:{
	        min: null,
	        max: null,
	        autoscaleMargin: null,
	        axisLabel: ''
	      },

	      /**
	       * Customizes the vertical axis
	       * @type Object
	       * @default {min: null, max: null,autoscaleMargin: 0.2,axisLabel: ''}
	       * @property yaxis
	       *   @property min {Number}
	       *   Minimal value shown on axis
	       *   @property max {Number}
	       *   Maximal values show on axis
	       *   @property autoscaleMargin {Number}
	       *   It's the fraction of margin that the scaling algorithm will add to
	       *   avoid that the outermost points ends up on the grid border.
	       *   @property axisLabel {String}
	       *   Axis description
	       */
	      yaxis:{
	        min: null,
	        max: null,
	        autoscaleMargin: 0.2,
	        axisLabel: ''
	      },




	      /**
	       * Data to be plotted. The format is the same used by flot. The format may differ if the charttype
	       * option is set to pie or bar.
	       * @property data
	       * @default []
	       */
	      data:[]

	    };
	    
	var pieDefaults = {
	      series: {
	          pie: {
	            show: true
	          }
	        },
	        tooltipOpts: {
	          content: ' %p.0%, %s'
	        }
	      };
	 var _plotClickServerSide = function (event, clientId) {
	      var params = {};
	      params[clientId + 'name'] = "plotclick";
	      params[clientId + 'seriesIndex'] = event.data.seriesIndex;
	      params[clientId + 'dataIndex'] = event.data.dataIndex;
	      params[clientId + 'x'] = event.data.x;
	      params[clientId + 'y'] = event.data.y;

	      rf.ajax(clientId, event, {
	        parameters: params,
	        incId: 1
	      });

	    };

    
    rf.ui = rf.ui || {};

    rf.ui.Chart = rf.BaseComponent.extendClass({
            // class name
            name:"Chart",

            init : function (componentId, options) {
            	$super.constructor.call(this, componentId, options);

                this.namespace = this.namespace || "." + RichFaces.Event.createNamespace(this.name, this.id);
                this.attachToDom();
            	this.options = $.extend(true,{},defaultOptions,options);
            	
            	this.element = $(document.getElementById(componentId));
                this.chartElement = this.element.find(".chart");
            	
            	if (this.options.charttype === 'pie') {
                    this.options = $.extend(true,{},this.options,pieDefaults);
                    //transform data from
                    // [[{data:1, label:"label1"},{data:2,label:"label2"},...]]
                    //to
                    // [{data:1, label:"label1"},{data:2,label:"label2"},...]
                    this.options.data = this.options.data[0]; //pie chart data should not be in a collection
                }
                else if (this.options.charttype === 'bar') {                    
                    if (this.options.xtype === 'string') {
                        /*
                         * data transformation:
                         * data: [["label1", value1], ["label2", value2"], …]
                         * 
                         * =>
                         * 
                         * data: [[1, value1], [2, value2], …] 
                         * ticks: [[1, "label1"], [2, "label2"], …]
                         *
                         */
                        this.options.xaxis.tickLength = 0;
                        
                        var seriesLength = this.options.data[0].data.length,
                            seriesTotal = this.options.data.length,
                            ticks = [],
                            ordered = false;
                            
                        this.options.bars = this.options.bars || {};
                        this.options.bars.barWidth = 1 / (seriesTotal + 1);
                        for (var i = 0; i < seriesLength; i++) {
                            ticks.push( [i, this.options.data[0].data[i][0]] );
                            for (var j = 0; j < seriesTotal; j++) { 
                                this.options.data[j].data[i][0] = i;
                                if (!ordered) {
                                    this.options.data[j].bars.order = j;
                                }
                            }
                            ordered = true;
                        }
                        
                        this.options.xaxis.ticks = ticks;
                    }
                    
                    
                  }
                  else if (options.charttype === 'line') {
                    if (this.options.xtype === 'string') {
                        /*
                         * data transformation:
                         * data: [["label1", value1], ["label2", value2"], …]
                         * 
                         * =>
                         * 
                         * data: [[1, value1], [2, value2], …] 
                         * ticks: [[1, "label1"], [2, "label2"], …]
                         *
                         */
                        this.options.xaxis.tickLength = 0;
                        
                        var seriesLength = this.options.data[0].data.length,
                            seriesTotal = this.options.data.length,
                            ticks = [];
                        for (var i = 0; i < seriesLength; i++) {
                            ticks.push( [i, this.options.data[0].data[i][0]] );
                            for (var j = 0; j < seriesTotal; j++) {
                                this.options.data[j].data[i][0] = i;
                            }
                        }
                        
                        this.options.xaxis.ticks = ticks;
                    }
                    if (options.zoom) {
                      this.options.selection = {mode: 'xy'};
                    }
                    if (this.options.xtype === 'date') {
                      this.options = $.extend({},this.options, dateDefaults);
                      if (this.options.xaxis.format) {
                        this.options.xaxis.timeformat = this.options.xaxis.format;
                      }
                    }
                  }
            	
                this.plot = $.plot(this.chartElement,this.options.data,this.options);
                this.__bindEventHandlers(this.chartElement,this.options);
            },

            /***************************** Public Methods  ****************************************************************/

            /**
             * Returns chart object
             *
             * @method getPlotObject
             */
            getPlotObject: function () {
              return this.plot;
            },

            /**
             * Highlights the point in chart selected by seriesIndex or point index. Does not work for pie charts.
             * @param seriesIndex {int}
             * @param pointIndex {int}
             * @method highlight
             */
            highlight: function(seriesIndex,pointIndex){
               this.plot.highlight(seriesIndex,pointIndex);
            },
            /**
             * Removes highlighting of point. If method is called without parameters it unhighlights all points.
             * @param seriesIndex {int}
             * @param pointIndex {int}
             * @method unghighlight
             */
            unhighlight: function(seriesIndex,pointIndex){
               this.plot.unhighlight(seriesIndex,pointIndex);
            },

            /***************************** Private Methods ********************************************************/
            __bindEventHandlers:function(element,options){
            	
                element.on('plotclick', this._getPlotClickHandler(options, element.get(0), _plotClickServerSide));
                element.on('plothover', this._getPlotHoverHandler(options, element.get(0)));
            	if (options.handlers && options.handlers.onmouseout) {
                    element.on('mouseout', options.handlers.onmouseout);
                }

                if (options.zoom) {
                    element.on('plotselected', $.proxy(this._zoomFunction, this));
                }
                
            },
          //function handles plotclick event. it calls server-side, client-side and particular series handlers if set.
            _getPlotClickHandler: function (options, element, serverSideHandler) {
            	
              var clickHandler = options.handlers['onplotclick'];	
              var particularClickHandlers= options.particularSeriesHandlers['onplotclick'];
              var clientId = this.element.attr('id');
              return function (event, pos, item) {
                if (item !== null) {
                  //point in a chart clicked
                  event.data = {
                    seriesIndex: item.seriesIndex,
                    dataIndex: item.dataIndex,
                    x: item.datapoint[0],
                    y: item.datapoint[1],
                    item: item
                  };
                  if(options.charttype=="pie"){
                	  event.data.x = options.data[item.seriesIndex].label;
                	  event.data.y = item.datapoint[1][0][1];
                  }
                  else if(options.charttype=="bar" && options.xtype=="string"){
                    event.data.x = options.xaxis.ticks[item.dataIndex][1];

                  }
                } else {
                    event.data = {
                        x: pos.x,
                        y: pos.y
                    };
                }
                  //sent request only if a server-side listener attached
                  if (options.serverSideListener) {
                    //server-side
                    if (serverSideHandler) {
                      serverSideHandler(event, clientId);
                    }
                  }

                  //client-side
                  if (clickHandler) {
                	 clickHandler.call(element, event);
                  }
                  //client-side particular series handler
                  if (particularClickHandlers[event.data.seriesIndex]) {
                	  particularClickHandlers[event.data.seriesIndex].call(element, event);
                  }
              };
            },
            
            //function handles plothover event. it calls client-side and particular series handlers if set.
            _getPlotHoverHandler: function (options, element) {
              var hoverHandler = options.handlers['onplothover'];	
              var particularHoverHandlers =	options.particularSeriesHandlers['onplothover']; 
              
              return function (event, pos, item) {
                if (item !== null) {
                  //point in a chart clicked
                  event.data = {
                    seriesIndex: item.seriesIndex,
                    dataIndex: item.dataIndex,
                    x: item.datapoint[0],
                    y: item.datapoint[1],
                    item: item
                  };
                } else {
                    event.data = {
                        x: pos.x,
                        y: pos.y
                    };
                }
                  //client-side
                  if (hoverHandler) {
                    hoverHandler.call(element, event);
                  }

                  //client-side particular series handler
                  if (particularHoverHandlers[event.data.seriesIndex]) {
                	  particularHoverHandlers[event.data.seriesIndex].call(element, event);
                  }
                
              };
            },

            _zoomFunction: function (event, ranges) {
                var plot = this.getPlotObject();
                $.each(plot.getXAxes(), function(_, axis) {
                    var opts = axis.options;
                    opts.min = ranges.xaxis.from;
                    opts.max = ranges.xaxis.to;
                });
                plot.setupGrid();
                plot.draw();
                plot.clearSelection();
            },

            resetZoom: function () {
                this.plot = $.plot(this.chartElement,this.options.data,this.options);
            },

            destroy: function () {
                rf.Event.unbindById(this.id, "." + this.namespace);
                $super.destroy.call(this);
            }
        });

    // define super class link
    var $super = rf.ui.Chart.$super;
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:datascroller.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    var initButtons = function(buttons, css, component) {
        var id;

        var fn = function(e) {
            e.data.fn.call(e.data.component, e);
        };

        var data = {};
        data.component = component;

        for (id in buttons) {
            if (buttons.hasOwnProperty(id)) {
                var element = $(document.getElementById(id));
    
                data.id = id;
                data.page = buttons[id];
                data.element = element;
                data.fn = component.processClick;
    
                element.bind('click', copy(data), fn);
            }
        }
    };

    var copy = function(data) {
        var key;
        var eventData = {};

        for (key in data) {
            if (data.hasOwnProperty(key)) {
                eventData[key] = data[key];
            }
        }

        return eventData;
    };

    var togglePressClass = function(el, event) {
        if (event.type == 'mousedown') {
            el.addClass('rf-ds-press');
        } else if (event.type == 'mouseup' || event.type == 'mouseout') {
            el.removeClass('rf-ds-press');
        }
    };

    rf.ui.DataScroller = function(id, submit, options) {

        $super.constructor.call(this, id);

        var dataScrollerElement = this.attachToDom();

        this.options = options;
        this.currentPage = options.currentPage;

        if (submit && typeof submit == 'function') {
            RichFaces.Event.bindById(id, this.getScrollEventName(), submit);
        }

        var css = {};

        if (options.buttons) {

            $(dataScrollerElement).delegate('.rf-ds-btn', 'mouseup mousedown mouseout', function(event) {
                if ($(this).hasClass('rf-ds-dis')) {
                    $(this).removeClass('rf-ds-press');
                } else {
                    togglePressClass($(this), event);
                }
            });

            initButtons(options.buttons.left, css, this);
            initButtons(options.buttons.right, css, this);
        }

        if (options.digitals) {

            $(dataScrollerElement).delegate('.rf-ds-nmb-btn', 'mouseup mousedown mouseout', function(event) {
                togglePressClass($(this), event);
            });

            initButtons(options.digitals, css, this);
        }
    };

    rf.BaseComponent.extend(rf.ui.DataScroller);
    var $super = rf.ui.DataScroller.$super;

    $.extend(rf.ui.DataScroller.prototype, (function () {

        var scrollEventName = "rich:datascroller:onscroll";

        return {

            name: "RichFaces.ui.DataScroller",

            processClick: function(event) {
                var data = event.data;
                if (data) {
                    var page = data.page;
                    if (page) {
                        this.switchToPage(page);
                    }
                }
            },

            switchToPage: function(page) {
                if (typeof page != 'undefined' && page != null) {
                    RichFaces.Event.fireById(this.id, this.getScrollEventName(), {'page' : page});
                }
            },

            fastForward: function() {
                this.switchToPage("fastforward");
            },

            fastRewind: function() {
                this.switchToPage("fastrewind");
            },

            next: function() {
                this.switchToPage("next");
            },

            previous: function() {
                this.switchToPage("previous");
            },

            first: function() {
                this.switchToPage("first");
            },

            last: function() {
                this.switchToPage("last");
            },

            getScrollEventName: function() {
                return scrollEventName;
            },
            destroy: function() {
                $super.destroy.call(this);
            }
        }

    })());

})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:progressBar.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    var defaultOptions = {
        interval: 1000,
        minValue: 0,
        maxValue: 100
    };

    var stateSelectors = {
        initial: '> .rf-pb-init',
        progress: '> .rf-pb-rmng',
        finish: '> .rf-pb-fin'
    };

    // Constructor definition
    rf.ui.ProgressBar = function(componentId, options) {
        // call constructor of parent class
        $super.constructor.call(this, componentId);
        this.__elt = this.attachToDom();
        this.options = $.extend(this.options, defaultOptions, options || {});
        this.enabled = this.options.enabled;
        this.minValue = this.options.minValue;
        this.maxValue = this.options.maxValue;

        this.__setValue(this.options.value || this.options.minValue /* TODO - check with Ilya */);

        if (this.options.resource) {
            this.__poll();
        } else if (this.options.submitFunction) {
            this.submitFunction = new Function("beforeUpdateHandler", "afterUpdateHandler", "params", "event", this.options.submitFunction);
            this.__poll();
        }

        if (this.options.onfinish) {
            rf.Event.bind(this.__elt, "finish", new Function("event", this.options.onfinish));
        }
    };

    // Extend component class and add protected methods from parent class to our container
    rf.BaseComponent.extend(rf.ui.ProgressBar);

    // define super class link
    var $super = rf.ui.ProgressBar.$super;

    $.extend(rf.ui.ProgressBar.prototype, (function() {
        return {
            name: "ProgressBar",

            __isInitialState: function() {
                return parseFloat(this.value) < parseFloat(this.getMinValue());
            },

            __isProgressState: function() {
                return !this.__isInitialState() && !this.__isFinishState();
            },

            __isFinishState: function() {
                return parseFloat(this.value) >= parseFloat(this.getMaxValue());
            },

            __beforeUpdate: function(event) {
                if (event.componentData && typeof event.componentData[this.id] != 'undefined') {
                    this.setValue(event.componentData[this.id]);
                }
            },

            __afterUpdate: function(event) {
                this.__poll();
            },

            __onResourceDataAvailable: function(data) {
                var parsedData = rf.parseJSON(data);
                if (parsedData instanceof Number || typeof parsedData == 'number') {
                    this.setValue(parsedData);
                }

                this.__poll();
            },

            __submit: function() {
                if (this.submitFunction) {
                    this.submitFunction.call(this, $.proxy(this.__beforeUpdate, this), $.proxy(this.__afterUpdate, this),
                        this.__params || {});
                } else {
                    $.get(this.options.resource, this.__params || {}, $.proxy(this.__onResourceDataAvailable, this), 'text');
                }
            },

            __poll: function(immediate) {
                if (this.enabled) {
                    if (immediate) {
                        this.__submit();
                    } else {
                        this.__pollTimer = setTimeout($.proxy(this.__submit, this), this.options.interval);
                    }
                }
            },

            __calculatePercent: function(v) {
                var min = parseFloat(this.getMinValue());
                var max = parseFloat(this.getMaxValue());
                var value = parseFloat(v);
                if (min < value && value < max) {
                    return (100 * (value - min)) / (max - min);
                } else if (value <= min) {
                    return 0;
                } else if (value >= max) {
                    return 100;
                }
            },

            __getPropertyOrObject: function(obj, propName) {
                if ($.isPlainObject(obj) && obj.propName) {
                    return obj.propName;
                }

                return obj;
            },

            getValue: function() {
                return this.value;
            },

            __showState: function (state) {
                var stateElt = $(stateSelectors[state], this.__elt);

                if (stateElt.length == 0 && (state == 'initial' || state == 'finish')) {
                    stateElt = $(stateSelectors['progress'], this.__elt)
                }

                stateElt.show().siblings().hide();
            },

            __setValue: function(val, initialStateSetup) {
                this.value = parseFloat(this.__getPropertyOrObject(val, "value"));

                if (this.__isFinishState() || this.__isInitialState()) {
                    this.disable();
                }
            },

            __updateVisualState: function() {
                if (this.__isInitialState()) {
                    this.__showState("initial");
                } else if (this.__isFinishState()) {
                    this.__showState("finish");
                } else {
                    this.__showState("progress");
                }

                var p = this.__calculatePercent(this.value);
                $(".rf-pb-prgs", this.__elt).css('width', p + "%");
            },

            setValue: function(val) {
                var wasInFinishState = this.__isFinishState();

                this.__setValue(val);
                this.__updateVisualState();

                if (!wasInFinishState && this.__isFinishState()) {
                    rf.Event.callHandler(this.__elt, "finish");
                }
            },

            getMaxValue: function() {
                return this.maxValue;
            },

            getMinValue: function() {
                return this.minValue;
            },

            isAjaxMode: function () {
                return !!this.submitFunction || !!this.options.resource;
            },

            disable: function () {
                this.__params = null;
                if (this.__pollTimer) {
                    clearTimeout(this.__pollTimer);
                    this.__pollTimer = null;
                }

                this.enabled = false;
            },

            enable: function (params) {
                if (this.isEnabled()) {
                    return;
                }

                this.__params = params;
                this.enabled = true;

                if (this.isAjaxMode()) {
                    this.__poll(true);
                }

            },

            isEnabled: function() {
                return this.enabled;
            },

            destroy: function() {
                this.disable();
                this.__elt = null;
                $super.destroy.call(this);
            }
        }
    }()));

})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:jquery.hotkeys.js
/*
 * jQuery Hotkeys Plugin
 * Copyright 2010, John Resig
 * Dual licensed under the MIT or GPL Version 2 licenses.
 *
 * Based upon the plugin by Tzury Bar Yochay:
 * http://github.com/tzuryby/hotkeys
 *
 * Original idea by:
 * Binny V A, http://www.openjs.com/scripts/events/keyboard_shortcuts/
*/

(function(jQuery){

	jQuery.hotkeys = {
		version: "0.8",

		specialKeys: {
			8: "backspace", 9: "tab", 13: "return", 16: "shift", 17: "ctrl", 18: "alt", 19: "pause",
			20: "capslock", 27: "esc", 32: "space", 33: "pageup", 34: "pagedown", 35: "end", 36: "home",
			37: "left", 38: "up", 39: "right", 40: "down", 45: "insert", 46: "del", 
			96: "0", 97: "1", 98: "2", 99: "3", 100: "4", 101: "5", 102: "6", 103: "7",
			104: "8", 105: "9", 106: "*", 107: "+", 109: "-", 110: ".", 111 : "/", 
			112: "f1", 113: "f2", 114: "f3", 115: "f4", 116: "f5", 117: "f6", 118: "f7", 119: "f8", 
			120: "f9", 121: "f10", 122: "f11", 123: "f12", 144: "numlock", 145: "scroll", 191: "/", 224: "meta"
		},

		shiftNums: {
			"`": "~", "1": "!", "2": "@", "3": "#", "4": "$", "5": "%", "6": "^", "7": "&", 
			"8": "*", "9": "(", "0": ")", "-": "_", "=": "+", ";": ": ", "'": "\"", ",": "<", 
			".": ">",  "/": "?",  "\\": "|"
		}
	};
	
	var defaultOptions = {
        key : '',
        enabledInInput : false
    };

	function keyHandler( handleObj ) {
		var options = ( typeof handleObj.data == "string" ) ? {key: handleObj.data} : handleObj.data;
		options = jQuery.extend({}, defaultOptions, options);

		var origHandler = handleObj.handler,
			keys = options.key.toLowerCase().split(" ");
		
		// when no key binding is defined
		if (keys.length === 1 && keys[0] === '') {
		    return;
		}
		
		handleObj.handler = function( event ) {
			var character = String.fromCharCode( event.which ).toLowerCase(),
					isTextInput = (/textarea|select/i.test( event.target.nodeName ) || event.target.type === "text");
			// Don't fire in text-accepting inputs that we didn't directly bind to
			if ( this !== event.target && isTextInput && !options.enabledInInput ) {
				return;
			}

			// Keypress represents characters, not special keys
			var special = event.type !== "keypress" && jQuery.hotkeys.specialKeys[ event.which ],
				key, modif = "", possible = {};

			// check combinations (alt|ctrl|shift+anything)
			if ( event.altKey && special !== "alt" ) {
				modif += "alt+";
			}

			if ( event.ctrlKey && special !== "ctrl" ) {
				modif += "ctrl+";
			}

			// TODO: Need to make sure this works consistently across platforms
			if ( event.metaKey && !event.ctrlKey && special !== "meta" ) {
				modif += "meta+";
			}

			if ( event.shiftKey && special !== "shift" ) {
				modif += "shift+";
			}

			if ( special ) {
				possible[ modif + special ] = true;

			} else {
				possible[ modif + character ] = true;
				possible[ modif + jQuery.hotkeys.shiftNums[ character ] ] = true;

				// "$" can be triggered as "Shift+4" or "Shift+$" or just "$"
				if ( modif === "shift+" ) {
					possible[ jQuery.hotkeys.shiftNums[ character ] ] = true;
				}
			}
			for ( var i = 0, l = keys.length; i < l; i++ ) {
				if ( possible[ keys[i] ] ) {
					return origHandler.apply( this, arguments );
				}
			}
		};
	}

	jQuery.each([ "keydown", "keyup", "keypress" ], function() {
		jQuery.event.special[ this ] = { add: keyHandler };
	});

})( jQuery );
;

// resource: org.richfaces:panelMenu.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */


(function ($, rf) {

    rf.ui = rf.ui || {};

    var __DEFAULT_OPTIONS = {
        expandSingle : true,
        bubbleSelection : true
    };

    rf.ui.PanelMenu = rf.BaseComponent.extendClass({
            // class name
            name:"PanelMenu",

            /**
             * @class PanelMenu
             * @name PanelMenu
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                $super.constructor.call(this, componentId);
                this.items = {};
                this.attachToDom();

                this.options = $.extend(this.options, __DEFAULT_OPTIONS, options || {});
                this.activeItem = this.__getValueInput().value;
                this.nestingLevel = 0;

                this.__addUserEventHandler("collapse");
                this.__addUserEventHandler("expand");
            },

            addItem: function(item) {
                this.items[item.itemName] = item;
            },

            deleteItem: function(item) {
                delete this.items[item.itemName];
            },

            getSelectedItem: function() {
                return this.getItem(this.selectedItem());
            },

            getItem: function (name) {
                return this.items[name];
            },

            /***************************** Public Methods  ****************************************************************/
            /**
             * @methodOf
             * @name PanelMenu#selectItem
             *
             * TODO ...
             *
             * @param {String} name
             * @return {void} TODO ...
             */
            selectItem: function (name) {
                // TODO implement
            },

            /**
             * @methodOf
             * @name PanelMenu#selectedItem
             *
             * TODO ...
             *
             * @return {String} TODO ...
             */
            selectedItem: function (id) {
                if (typeof id != "undefined") {
                    var valueInput = this.__getValueInput();
                    var prevActiveItem = valueInput.value;

                    this.activeItem = id;
                    valueInput.value = id;

                    for (var itemName in this.items) {
                        if (this.items.hasOwnProperty(itemName)) {
                            var item = this.items[itemName];
                            if (item.__isSelected()) {
                                item.__unselect();
                            }
                        }
                    }

                    return prevActiveItem;
                } else {
                    return this.activeItem;
                }
            },

            __getValueInput : function() {
                return document.getElementById(this.id + "-value");
            },

            /**
             * @methodOf
             * @name PanelMenu#expandAll
             *
             * TODO ...
             *
             * @return {void} TODO ...
             */
            expandAll: function () {
                // TODO implement
            },

            /**
             * @methodOf
             * @name PanelMenu#collapseAll
             *
             * TODO ...
             *
             * @return {void} TODO ...
             */
            collapseAll: function () {
                // TODO implement
            },

            /**
             * @methodOf
             * @name PanelMenu#expandGroup
             *
             * TODO ...
             *
             * @param {String} groupName
             * @return {void} TODO ...
             */
            expandGroup: function (groupName) {
                // TODO implement
            },

            /**
             * @methodOf
             * @name PanelMenu#collapseGroup
             *
             * TODO ...
             *
             * @param {String} groupName
             * @return {void} TODO ...
             */
            collapseGroup: function (groupName) {
                // TODO implement
            },


            /***************************** Private Methods ****************************************************************/


            __panelMenu  : function () {
                return $(rf.getDomElement(this.id));
            },

            __childGroups : function () {
                return this.__panelMenu().children(".rf-pm-top-gr")
            },

            /**
             * @private
             * */
            __addUserEventHandler : function (name) {
                var handler = this.options["on" + name];
                if (handler) {
                    //TODO nick - this will cause slowdowns in IE
                    rf.Event.bindById(this.id, name, handler);
                }
            },

            __isActiveItem: function(item) {
                return item.itemName == this.activeItem;
            },

            __collapseGroups : function (source) {
                var topGroup = source.__rfTopGroup();
                this.__childGroups().each(function (index, group) {
                    if (group.id != source.getEventElement() && (!topGroup || group.id != topGroup.id)) {
                        rf.component(group).__collapse();
                    }
                });

            },

            destroy: function () {
                rf.Event.unbindById(this.id, "." + this.namespace);
                $super.destroy.call(this);
            }
        });

    // define super class link
    var $super = rf.ui.PanelMenu.$super;

})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:jquery.pnotify.js
/*
 * jQuery Pines Notify (pnotify) Plugin 1.0.2-richfaces
 *
 * Copyright (c) 2009-2011 Hunter Perrin
 *
 * Triple license under the GPL, LGPL, and MPL:
 *    http://www.gnu.org/licenses/gpl.html
 *    http://www.gnu.org/licenses/lgpl.html
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *    
 * Modified by: RichFaces team
 */

(function($, rf) {
    var history_handle_top, timer;
    var body;
    var jwindow;
    $.extend({
        pnotify_remove_all: function () {
            var body_data = body.data("pnotify");
            /* POA: Added null-check */
            if (body_data && body_data.length) {
                $.each(body_data, function() {
                    if (this.pnotify_remove)
                        this.pnotify_remove();
                });
            }
        },
        pnotify_position_all: function () {
            if (timer)
                clearTimeout(timer);
            timer = null;
            var body_data = body.data("pnotify");
            if (!body_data || !body_data.length)
                return;
            $.each(body_data, function() {
                var s = this.opts.pnotify_stack;
                if (!s) return;
                if (!s.nextpos1)
                    s.nextpos1 = s.firstpos1;
                if (!s.nextpos2)
                    s.nextpos2 = s.firstpos2;
                if (!s.addpos2)
                    s.addpos2 = 0;
                if (this.css("display") != "none") {
                    var curpos1, curpos2;
                    var animate = {};
                    // Calculate the current pos1 value.
                    var csspos1;
                    switch (s.dir1) {
                        case "down":
                            csspos1 = "top";
                            break;
                        case "up":
                            csspos1 = "bottom";
                            break;
                        case "left":
                            csspos1 = "right";
                            break;
                        case "right":
                            csspos1 = "left";
                            break;
                    }
                    curpos1 = parseInt(this.css(csspos1));
                    if (isNaN(curpos1))
                        curpos1 = 0;
                    // Remember the first pos1, so the first visible notice goes there.
                    if (typeof s.firstpos1 == "undefined") {
                        s.firstpos1 = curpos1;
                        s.nextpos1 = s.firstpos1;
                    }
                    // Calculate the current pos2 value.
                    var csspos2;
                    switch (s.dir2) {
                        case "down":
                            csspos2 = "top";
                            break;
                        case "up":
                            csspos2 = "bottom";
                            break;
                        case "left":
                            csspos2 = "right";
                            break;
                        case "right":
                            csspos2 = "left";
                            break;
                    }
                    curpos2 = parseInt(this.css(csspos2));
                    if (isNaN(curpos2))
                        curpos2 = 0;
                    // Remember the first pos2, so the first visible notice goes there.
                    if (typeof s.firstpos2 == "undefined") {
                        s.firstpos2 = curpos2;
                        s.nextpos2 = s.firstpos2;
                    }
                    // Check that it's not beyond the viewport edge.
                    if ((s.dir1 == "down" && s.nextpos1 + this.height() > jwindow.height()) ||
                            (s.dir1 == "up" && s.nextpos1 + this.height() > jwindow.height()) ||
                            (s.dir1 == "left" && s.nextpos1 + this.width() > jwindow.width()) ||
                            (s.dir1 == "right" && s.nextpos1 + this.width() > jwindow.width())) {
                        // If it is, it needs to go back to the first pos1, and over on pos2.
                        s.nextpos1 = s.firstpos1;
                        s.nextpos2 += s.addpos2 + 10;
                        s.addpos2 = 0;
                    }
                    // Animate if we're moving on dir2.
                    if (s.animation && s.nextpos2 < curpos2) {
                        switch (s.dir2) {
                            case "down":
                                animate.top = s.nextpos2 + "px";
                                break;
                            case "up":
                                animate.bottom = s.nextpos2 + "px";
                                break;
                            case "left":
                                animate.right = s.nextpos2 + "px";
                                break;
                            case "right":
                                animate.left = s.nextpos2 + "px";
                                break;
                        }
                    } else
                        this.css(csspos2, s.nextpos2 + "px");
                    // Keep track of the widest/tallest notice in the column/row, so we can push the next column/row.
                    switch (s.dir2) {
                        case "down":
                        case "up":
                            if (this.outerHeight(true) > s.addpos2)
                                s.addpos2 = this.height();
                            break;
                        case "left":
                        case "right":
                            if (this.outerWidth(true) > s.addpos2)
                                s.addpos2 = this.width();
                            break;
                    }
                    // Move the notice on dir1.
                    if (s.nextpos1) {
                        // Animate if we're moving toward the first pos.
                        if (s.animation && (curpos1 > s.nextpos1 || animate.top || animate.bottom || animate.right || animate.left)) {
                            switch (s.dir1) {
                                case "down":
                                    animate.top = s.nextpos1 + "px";
                                    break;
                                case "up":
                                    animate.bottom = s.nextpos1 + "px";
                                    break;
                                case "left":
                                    animate.right = s.nextpos1 + "px";
                                    break;
                                case "right":
                                    animate.left = s.nextpos1 + "px";
                                    break;
                            }
                        } else
                            this.css(csspos1, s.nextpos1 + "px");
                    }
                    if (animate.top || animate.bottom || animate.right || animate.left)
                        this.animate(animate, {duration: 500, queue: false});
                    // Calculate the next dir1 position.
                    switch (s.dir1) {
                        case "down":
                        case "up":
                            s.nextpos1 += this.height() + 10;
                            break;
                        case "left":
                        case "right":
                            s.nextpos1 += this.width() + 10;
                            break;
                    }
                }
            });
            // Reset the next position data.
            $.each(body_data, function() {
                var s = this.opts.pnotify_stack;
                if (!s) return;
                s.nextpos1 = s.firstpos1;
                s.nextpos2 = s.firstpos2;
                s.addpos2 = 0;
                s.animation = true;
            });
        },
        pnotify: function(options) {
            if (!body)
                body = $("body");
            if (!jwindow)
                jwindow = $(window);

            var animating;

            // Build main options.
            var opts;
            if (typeof options != "object") {
                opts = $.extend({}, $.pnotify.defaults);
                opts.pnotify_text = options;
            } else {
                opts = $.extend({}, $.pnotify.defaults, options);
                if (opts['pnotify_animation'] instanceof Object) {
                    opts['pnotify_animation'] = $.extend({
                        effect_in:$.pnotify.defaults.pnotify_animation,
                        effect_out:$.pnotify.defaults.pnotify_animation
                    }, opts['pnotify_animation']);
                }
            }

            if (opts.pnotify_before_init) {
                if (opts.pnotify_before_init(opts) === false)
                    return null;
            }

            // This keeps track of the last element the mouse was over, so
            // mouseleave, mouseenter, etc can be called.
            var nonblock_last_elem;
            // This is used to pass events through the notice if it is non-blocking.
            var nonblock_pass = function(e, e_name) {
                pnotify.css("display", "none");
                var element_below = document.elementFromPoint(e.clientX, e.clientY);
                pnotify.css("display", "block");
                var jelement_below = $(element_below);
                var cursor_style = jelement_below.css("cursor");
                pnotify.css("cursor", cursor_style != "auto" ? cursor_style : "default");
                // If the element changed, call mouseenter, mouseleave, etc.
                if (!nonblock_last_elem || nonblock_last_elem.get(0) != element_below) {
                    if (nonblock_last_elem) {
                        dom_event.call(nonblock_last_elem.get(0), "mouseleave", e.originalEvent);
                        dom_event.call(nonblock_last_elem.get(0), "mouseout", e.originalEvent);
                    }
                    dom_event.call(element_below, "mouseenter", e.originalEvent);
                    dom_event.call(element_below, "mouseover", e.originalEvent);
                }
                dom_event.call(element_below, e_name, e.originalEvent);
                // Remember the latest element the mouse was over.
                nonblock_last_elem = jelement_below;
            };

            // Create our widget.
            // Stop animation, reset the removal timer, and show the close
            // button when the user mouses over.
            var pnotify = $("<div />", {
                "class": "rf-ntf " + opts.pnotify_addclass,
                "css": {"display": "none"},
                "mouseenter": function(e) {
                    if (opts.pnotify_nonblock) e.stopPropagation();
                    if (opts.pnotify_mouse_reset && animating == "out") {
                        // If it's animating out, animate back in really quick.
                        pnotify.stop(true);
                        animating = "in";
                        pnotify.css("height", "auto").animate({"width": opts.pnotify_width, "opacity": opts.pnotify_nonblock ? opts.pnotify_nonblock_opacity : opts.pnotify_opacity}, "fast");
                    }
                    if (opts.pnotify_nonblock) {
                        // If it's non-blocking, animate to the other opacity.
                        pnotify.animate({"opacity": opts.pnotify_nonblock_opacity}, "fast");
                    }
                    if (opts.pnotify_hide && opts.pnotify_mouse_reset) pnotify.pnotify_cancel_remove();
                    //                    Do not update
                    if (opts.pnotify_closer && !opts.pnotify_nonblock) pnotify.closer.css("visibility", "visible");
                },
                "mouseleave": function(e) {
                    if (opts.pnotify_nonblock) e.stopPropagation();
                    nonblock_last_elem = null;
                    pnotify.css("cursor", "auto");
                    if (opts.pnotify_nonblock && animating != "out")
                        pnotify.animate({"opacity": opts.pnotify_opacity}, "fast");
                    if (opts.pnotify_hide && opts.pnotify_mouse_reset) pnotify.pnotify_queue_remove();
                    //                    Do not update
                    pnotify.closer.css("visibility", "hidden");
                    $.pnotify_position_all();
                },
                "mouseover": function(e) {
                    if (opts.pnotify_nonblock) e.stopPropagation();
                },
                "mouseout": function(e) {
                    if (opts.pnotify_nonblock) e.stopPropagation();
                },
                "mousemove": function(e) {
                    if (opts.pnotify_nonblock) {
                        e.stopPropagation();
                        nonblock_pass(e, "onmousemove");
                    }
                },
                "mousedown": function(e) {
                    if (opts.pnotify_nonblock) {
                        e.stopPropagation();
                        e.preventDefault();
                        nonblock_pass(e, "onmousedown");
                    }
                },
                "mouseup": function(e) {
                    if (opts.pnotify_nonblock) {
                        e.stopPropagation();
                        e.preventDefault();
                        nonblock_pass(e, "onmouseup");
                    }
                },
                "click": function(e) {
                    if (opts.pnotify_nonblock) {
                        e.stopPropagation();
                        nonblock_pass(e, "onclick");
                    }
                },
                "dblclick": function(e) {
                    if (opts.pnotify_nonblock) {
                        e.stopPropagation();
                        nonblock_pass(e, "ondblclick");
                    }
                },
                "dir": opts.dir,
                "lang": opts.lang
            });
            pnotify.opts = opts;
            // Create a drop shadow.
            if (opts.pnotify_shadow && !rf.browser.msie)
                pnotify.shadow_container = $("<div />", {"class": "rf-ntf-shdw"}).prependTo(pnotify);
            // Create a container for the notice contents.
            pnotify.container = $("<div />", {"class": "rf-ntf-cnt"})
                    .appendTo(pnotify);

            pnotify.pnotify_version = "1.0.2";

            // This function is for updating the notice.
            pnotify.pnotify = function(options) {
                // Update the notice.
                var old_opts = opts;
                if (typeof options == "string")
                    opts.pnotify_text = options;
                else
                    opts = $.extend({}, opts, options);
                pnotify.opts = opts;
                // Update the shadow.
                if (opts.pnotify_shadow != old_opts.pnotify_shadow) {
                    if (opts.pnotify_shadow && !rf.browser.msie)
                        pnotify.shadow_container = $("<div />", {"class": "rf-ntf-shdw"}).prependTo(pnotify);
                    else
                        pnotify.children(".rf-ntf-shdw").remove();
                }
                // Update the additional classes.
                if (opts.pnotify_addclass === false)
                    pnotify.removeClass(old_opts.pnotify_addclass);
                else if (opts.pnotify_addclass !== old_opts.pnotify_addclass)
                    pnotify.removeClass(old_opts.pnotify_addclass).addClass(opts.pnotify_addclass);
                // Update the title.
                if (opts.pnotify_title === false)
                    pnotify.title_container.hide("fast");
                else if (opts.pnotify_title !== old_opts.pnotify_title)
                    pnotify.title_container.html(opts.pnotify_title).show(200);
                // Update the text.
                if (opts.pnotify_text === false) {
                    pnotify.text_container.hide("fast");
                } else if (opts.pnotify_text !== old_opts.pnotify_text) {
                    if (opts.pnotify_insert_brs)
                        opts.pnotify_text = opts.pnotify_text.replace(/\n/g, "<br />");
                    pnotify.text_container.html(opts.pnotify_text).show(200);
                }
                pnotify.pnotify_history = opts.pnotify_history;
                // Change the notice type.
                if (opts.pnotify_type != old_opts.pnotify_type)
                    pnotify.container.toggleClass("rf-ntf-cnt rf-ntf-cnt-hov");
                if ((opts.pnotify_notice_icon != old_opts.pnotify_notice_icon && opts.pnotify_type == "notice") ||
                        (opts.pnotify_error_icon != old_opts.pnotify_error_icon && opts.pnotify_type == "error") ||
                        (opts.pnotify_type != old_opts.pnotify_type)) {
                    // Remove any old icon.
                    pnotify.container.find("div.rf-ntf-ico").remove();
                    //					if ((opts.pnotify_error_icon && opts.pnotify_type == "error") || (opts.pnotify_notice_icon)) {
                    // Build the new icon.
                    $("<div />", {"class": "rf-ntf-ico"})
                            .append($("<span />", {"class": opts.pnotify_type == "error" ? opts.pnotify_error_icon : opts.pnotify_notice_icon}))
                            .prependTo(pnotify.container);
                    //					}
                }
                // Update the width.
                if (opts.pnotify_width !== old_opts.pnotify_width)
                    pnotify.animate({width: opts.pnotify_width});
                // Update the minimum height.
                if (opts.pnotify_min_height !== old_opts.pnotify_min_height)
                    pnotify.container.animate({minHeight: opts.pnotify_min_height});
                // Update the opacity.
                if (opts.pnotify_opacity !== old_opts.pnotify_opacity)
                    pnotify.fadeTo(opts.pnotify_animate_speed, opts.pnotify_opacity);
                if (!opts.pnotify_hide)
                    pnotify.pnotify_cancel_remove();
                else if (!old_opts.pnotify_hide)
                    pnotify.pnotify_queue_remove();
                pnotify.pnotify_queue_position();
                return pnotify;
            };

            // Queue the position function so it doesn't run repeatedly and use
            // up resources.
            pnotify.pnotify_queue_position = function() {
                if (timer)
                    clearTimeout(timer);
                timer = setTimeout($.pnotify_position_all, 10);
            };

            // Display the notice.
            pnotify.pnotify_display = function() {
                // If the notice is not in the DOM, append it.
                if (!pnotify.parent().length)
                    pnotify.appendTo(body);
                // Run callback.
                if (opts.pnotify_before_open) {
                    if (opts.pnotify_before_open(pnotify) === false)
                        return;
                }
                pnotify.pnotify_queue_position();
                // First show it, then set its opacity, then hide it.
                if (opts.pnotify_animation == "fade" || opts.pnotify_animation.effect_in == "fade") {
                    // If it's fading in, it should start at 0.
                    pnotify.show().fadeTo(0, 0).hide();
                } else {
                    // Or else it should be set to the opacity.
                    if (opts.pnotify_opacity != 1)
                        pnotify.show().fadeTo(0, opts.pnotify_opacity).hide();
                }
                pnotify.animate_in(function() {
                    if (opts.pnotify_after_open)
                        opts.pnotify_after_open(pnotify);

                    pnotify.pnotify_queue_position();

                    // Now set it to hide.
                    if (opts.pnotify_hide)
                        pnotify.pnotify_queue_remove();
                });
            };

            // Remove the notice.
            pnotify.pnotify_remove = function() {
                if (pnotify.timer) {
                    window.clearTimeout(pnotify.timer);
                    pnotify.timer = null;
                }
                // Run callback.
                if (opts.pnotify_before_close) {
                    if (opts.pnotify_before_close(pnotify) === false)
                        return;
                }
                pnotify.animate_out(function() {
                    if (opts.pnotify_after_close) {
                        if (opts.pnotify_after_close(pnotify) === false)
                            return;
                    }
                    pnotify.pnotify_queue_position();
                    // If we're supposed to remove the notice from the DOM, do it.
                    if (opts.pnotify_remove)
                        pnotify.detach();
                });
            };

            // Animate the notice in.
            pnotify.animate_in = function(callback) {
                // Declare that the notice is animating in. (Or has completed animating in.)
                animating = "in";
                var animation;
                if (typeof opts.pnotify_animation.effect_in != "undefined")
                    animation = opts.pnotify_animation.effect_in;
                else
                    animation = opts.pnotify_animation;
                if (animation == "none") {
                    pnotify.show();
                    callback();
                } else if (animation == "show")
                    pnotify.show(opts.pnotify_animate_speed, callback);
                else if (animation == "fade")
                    pnotify.show().fadeTo(opts.pnotify_animate_speed, opts.pnotify_opacity, callback);
                else if (animation == "slide")
                    pnotify.slideDown(opts.pnotify_animate_speed, callback);
                else if (typeof animation == "function")
                    animation("in", callback, pnotify);
                else if (pnotify.effect)
                    pnotify.effect(animation, {}, opts.pnotify_animate_speed, callback);
            };

            // Animate the notice out.
            pnotify.animate_out = function(callback) {
                // Declare that the notice is animating out. (Or has completed animating out.)
                animating = "out";
                var animation;
                if (typeof opts.pnotify_animation.effect_out != "undefined")
                    animation = opts.pnotify_animation.effect_out;
                else
                    animation = opts.pnotify_animation;
                if (animation == "none") {
                    pnotify.hide();
                    callback();
                } else if (animation == "show")
                    pnotify.hide(opts.pnotify_animate_speed, callback);
                else if (animation == "fade")
                    pnotify.fadeOut(opts.pnotify_animate_speed, callback);
                else if (animation == "slide")
                    pnotify.slideUp(opts.pnotify_animate_speed, callback);
                else if (typeof animation == "function")
                    animation("out", callback, pnotify);
                else if (pnotify.effect)
                    pnotify.effect(animation, {}, opts.pnotify_animate_speed, callback);
            };

            // Cancel any pending removal timer.
            pnotify.pnotify_cancel_remove = function() {
                if (pnotify.timer)
                    window.clearTimeout(pnotify.timer);
            };

            // Queue a removal timer.
            pnotify.pnotify_queue_remove = function() {
                // Cancel any current removal timer.
                pnotify.pnotify_cancel_remove();
                pnotify.timer = window.setTimeout(function() {
                    pnotify.pnotify_remove();
                }, (isNaN(opts.pnotify_delay) ? 0 : opts.pnotify_delay));
            };

            // Provide a button to close the notice.
            pnotify.closer = $("<div />", {
                "class": "rf-ntf-cls",
                "css": {"cursor": "pointer", "visibility": "hidden"},
                "click": function() {
                    pnotify.pnotify_remove();
                    //                    Do not update
                    pnotify.closer.css("visibility", "hidden");
                }
            })
                    .append($("<span />", {"class": "rf-ntf-cls-ico"}))
                    .appendTo(pnotify.container);

            // Add the appropriate icon.
            //			if ((opts.pnotify_error_icon && opts.pnotify_type == "error") || (opts.pnotify_notice_icon)) {
            $("<div />", {"class": "rf-ntf-ico"})
                    .append($("<span />", {"class": opts.pnotify_type == "error" ? opts.pnotify_error_icon : opts.pnotify_notice_icon}))
                    .appendTo(pnotify.container);
            //			}

            // Add a title.
            pnotify.title_container = $("<div />", {
                "class": "rf-ntf-sum",
                "html": opts.pnotify_title
            })
                    .appendTo(pnotify.container);
            if (opts.pnotify_title === false)
                pnotify.title_container.hide();

            // Replace new lines with HTML line breaks.
            if (opts.pnotify_insert_brs && typeof opts.pnotify_text == "string")
                opts.pnotify_text = opts.pnotify_text.replace(/\n/g, "<br />");
            // Add text.
            pnotify.text_container = $("<div />", {
                "class": "rf-ntf-det",
                "html": opts.pnotify_text
            })
                    .appendTo(pnotify.container);
            if (opts.pnotify_text === false)
                pnotify.text_container.hide();

            //Append div with clear:both class
            $("<div />", {"class":"rf-ntf-clr"}).appendTo(pnotify.container);

            // Set width and min height.
            if (typeof opts.pnotify_width == "string")
                pnotify.css("width", opts.pnotify_width);
            if (typeof opts.pnotify_min_height == "string")
                pnotify.container.css("min-height", opts.pnotify_min_height);

            // The history variable controls whether the notice gets redisplayed
            // by the history pull down.
            pnotify.pnotify_history = opts.pnotify_history;

            // Add the notice to the notice array.
            var body_data = body.data("pnotify");
            if (body_data == null || typeof body_data != "object")
                body_data = [];
            if (opts.pnotify_stack.push == "top")
                body_data = $.merge([pnotify], body_data);
            else
                body_data = $.merge(body_data, [pnotify]);
            body.data("pnotify", body_data);

            // Run callback.
            if (opts.pnotify_after_init)
                opts.pnotify_after_init(pnotify);

            if (opts.pnotify_history) {
                // If there isn't a history pull down, create one.
                var body_history = body.data("pnotify_history");
                if (typeof body_history == "undefined") {
                    body_history = $("<div />", {
                        "class": "rf-ntf-hstr",
                        "mouseleave": function() {
                            body_history.animate({top: "-" + history_handle_top + "px"}, {duration: 100, queue: false});
                        }
                    })
                            .append($("<div />", {"class": "rf-ntf-hstr-hdr", "text": "Redisplay"}))
                            .append($("<button />", {
                        "class": "rf-ntf-hstr-all",
                        "text": "All",
                        //							"mouseenter": function(){
                        //								$(this).addClass("ui-state-hover");
                        //							},
                        //							"mouseleave": function(){
                        //								$(this).removeClass("ui-state-hover");
                        //							},
                        "click": function() {
                            // Display all notices. (Disregarding non-history notices.)
                            //                            Don't change it to pnotify's new version, cause using body_data here is a bug
                            $.each(body.data("pnotify"), function() {
                                if (this.pnotify_history && this.pnotify_display)
                                    this.pnotify_display();
                            });
                            return false;
                        }
                    }))
                            .append($("<button />", {
                        "class": "rf-ntf-hstr-last",
                        "text": "Last",
                        //							"mouseenter": function(){
                        //								$(this).addClass("ui-state-hover");
                        //							},
                        //							"mouseleave": function(){
                        //								$(this).removeClass("ui-state-hover");
                        //							},
                        "click": function() {
                            // Look up the last history notice, and display it.
                            var i = 1;
                            var body_data = body.data("pnotify");
                            while (!body_data[body_data.length - i] || !body_data[body_data.length - i].pnotify_history || body_data[body_data.length - i].is(":visible")) {
                                if (body_data.length - i === 0)
                                    return false;
                                i++;
                            }
                            var n = body_data[body_data.length - i];
                            if (n.pnotify_display)
                                n.pnotify_display();
                            return false;
                        }
                    }))
                            .appendTo(body);

                    // Make a handle so the user can pull down the history pull down.
                    var handle = $("<span />", {
                        "class": "rf-ntf-hstr-hndl",
                        "mouseenter": function() {
                            body_history.animate({top: "0"}, {duration: 100, queue: false});
                        }
                    })
                            .appendTo(body_history);

                    // Get the top of the handle.
                    history_handle_top = handle.offset().top + 2;
                    // Hide the history pull down up to the top of the handle.
                    body_history.css({top: "-" + history_handle_top + "px"});
                    // Save the history pull down.
                    body.data("pnotify_history", body_history);
                }
            }

            // Mark the stack so it won't animate the new notice.
            opts.pnotify_stack.animation = false;

            // Display the notice.
            pnotify.pnotify_display();

            return pnotify;
        }
    });

    // Some useful regexes.
    var re_on = /^on/;
    var re_mouse_events = /^(dbl)?click$|^mouse(move|down|up|over|out|enter|leave)$|^contextmenu$/;
    var re_ui_events = /^(focus|blur|select|change|reset)$|^key(press|down|up)$/;
    var re_html_events = /^(scroll|resize|(un)?load|abort|error)$/;
    // Fire a DOM event.
    var dom_event = function(e, orig_e) {
        var event_object;
        e = e.toLowerCase();
        if (document.createEvent && this.dispatchEvent) {
            // FireFox, Opera, Safari, Chrome
            e = e.replace(re_on, '');
            if (e.match(re_mouse_events)) {
                // This allows the click event to fire on the notice. There is
                // probably a much better way to do it.
                $(this).offset();
                event_object = document.createEvent("MouseEvents");
                event_object.initMouseEvent(
                        e, orig_e.bubbles, orig_e.cancelable, orig_e.view, orig_e.detail,
                        orig_e.screenX, orig_e.screenY, orig_e.clientX, orig_e.clientY,
                        orig_e.ctrlKey, orig_e.altKey, orig_e.shiftKey, orig_e.metaKey, orig_e.button, orig_e.relatedTarget
                        );
            } else if (e.match(re_ui_events)) {
                event_object = document.createEvent("UIEvents");
                event_object.initUIEvent(e, orig_e.bubbles, orig_e.cancelable, orig_e.view, orig_e.detail);
            } else if (e.match(re_html_events)) {
                event_object = document.createEvent("HTMLEvents");
                event_object.initEvent(e, orig_e.bubbles, orig_e.cancelable);
            }
            if (!event_object) return;
            this.dispatchEvent(event_object);
        } else {
            // Internet Explorer
            if (!e.match(re_on)) e = "on" + e;
            event_object = document.createEventObject(orig_e);
            this.fireEvent(e, event_object);
        }
    };

    $.pnotify.defaults = {
        // The notice's title.
        pnotify_title: false,
        // The notice's text.
        pnotify_text: false,
        // Additional classes to be added to the notice. (For custom styling.)
        pnotify_addclass: "",
        // Create a non-blocking notice. It lets the user click elements underneath it.
        pnotify_nonblock: false,
        // The opacity of the notice (if it's non-blocking) when the mouse is over it.
        pnotify_nonblock_opacity: .2,
        // Display a pull down menu to redisplay previous notices, and place the notice in the history.
        pnotify_history: true,
        // Width of the notice.
        pnotify_width: "300px",
        // Minimum height of the notice. It will expand to fit content.
        pnotify_min_height: "16px",
        // Type of the notice. "notice" or "error".
        pnotify_type: "notice",
        // The icon class to use if type is notice.
        pnotify_notice_icon: "",
        // The icon class to use if type is error.
        pnotify_error_icon: "",
        // The animation to use when displaying and hiding the notice. "none", "show", "fade", and "slide" are built in to jQuery. Others require jQuery UI. Use an object with effect_in and effect_out to use different effects.
        pnotify_animation: "fade",
        // Speed at which the notice animates in and out. "slow", "def" or "normal", "fast" or number of milliseconds.
        pnotify_animate_speed: "slow",
        // Opacity of the notice.
        pnotify_opacity: 1,
        // Display a drop shadow.
        pnotify_shadow: false,
        // Provide a button for the user to manually close the notice.
        pnotify_closer: true,
        // After a delay, remove the notice.
        pnotify_hide: true,
        // Delay in milliseconds before the notice is removed.
        pnotify_delay: 8000,
        // Reset the hide timer if the mouse moves over the notice.
        pnotify_mouse_reset: true,
        // Remove the notice's elements from the DOM after it is removed.
        pnotify_remove: true,
        // Change new lines to br tags.
        pnotify_insert_brs: true,
        // The stack on which the notices will be placed. Also controls the direction the notices stack.
        pnotify_stack: {"dir1": "down", "dir2": "left", "push": "bottom"}
    };
})(jQuery, RichFaces);;

// resource: com.jqueryui:widget.js
/*!
 * jQuery UI Widget 1.11.4
 * http://jqueryui.com
 *
 * Copyright jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * http://api.jqueryui.com/jQuery.widget/
 */
(function( factory ) {
	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [ "jquery" ], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
}(function( $ ) {

var widget_uuid = 0,
	widget_slice = Array.prototype.slice;

$.cleanData = (function( orig ) {
	return function( elems ) {
		var events, elem, i;
		for ( i = 0; (elem = elems[i]) != null; i++ ) {
			try {

				// Only trigger remove when necessary to save time
				events = $._data( elem, "events" );
				if ( events && events.remove ) {
					$( elem ).triggerHandler( "remove" );
				}

			// http://bugs.jquery.com/ticket/8235
			} catch ( e ) {}
		}
		orig( elems );
	};
})( $.cleanData );

$.widget = function( name, base, prototype ) {
	var fullName, existingConstructor, constructor, basePrototype,
		// proxiedPrototype allows the provided prototype to remain unmodified
		// so that it can be used as a mixin for multiple widgets (#8876)
		proxiedPrototype = {},
		namespace = name.split( "." )[ 0 ];

	name = name.split( "." )[ 1 ];
	fullName = namespace + "-" + name;

	if ( !prototype ) {
		prototype = base;
		base = $.Widget;
	}

	// create selector for plugin
	$.expr[ ":" ][ fullName.toLowerCase() ] = function( elem ) {
		return !!$.data( elem, fullName );
	};

	$[ namespace ] = $[ namespace ] || {};
	existingConstructor = $[ namespace ][ name ];
	constructor = $[ namespace ][ name ] = function( options, element ) {
		// allow instantiation without "new" keyword
		if ( !this._createWidget ) {
			return new constructor( options, element );
		}

		// allow instantiation without initializing for simple inheritance
		// must use "new" keyword (the code above always passes args)
		if ( arguments.length ) {
			this._createWidget( options, element );
		}
	};
	// extend with the existing constructor to carry over any static properties
	$.extend( constructor, existingConstructor, {
		version: prototype.version,
		// copy the object used to create the prototype in case we need to
		// redefine the widget later
		_proto: $.extend( {}, prototype ),
		// track widgets that inherit from this widget in case this widget is
		// redefined after a widget inherits from it
		_childConstructors: []
	});

	basePrototype = new base();
	// we need to make the options hash a property directly on the new instance
	// otherwise we'll modify the options hash on the prototype that we're
	// inheriting from
	basePrototype.options = $.widget.extend( {}, basePrototype.options );
	$.each( prototype, function( prop, value ) {
		if ( !$.isFunction( value ) ) {
			proxiedPrototype[ prop ] = value;
			return;
		}
		proxiedPrototype[ prop ] = (function() {
			var _super = function() {
					return base.prototype[ prop ].apply( this, arguments );
				},
				_superApply = function( args ) {
					return base.prototype[ prop ].apply( this, args );
				};
			return function() {
				var __super = this._super,
					__superApply = this._superApply,
					returnValue;

				this._super = _super;
				this._superApply = _superApply;

				returnValue = value.apply( this, arguments );

				this._super = __super;
				this._superApply = __superApply;

				return returnValue;
			};
		})();
	});
	constructor.prototype = $.widget.extend( basePrototype, {
		// TODO: remove support for widgetEventPrefix
		// always use the name + a colon as the prefix, e.g., draggable:start
		// don't prefix for widgets that aren't DOM-based
		widgetEventPrefix: existingConstructor ? (basePrototype.widgetEventPrefix || name) : name
	}, proxiedPrototype, {
		constructor: constructor,
		namespace: namespace,
		widgetName: name,
		widgetFullName: fullName
	});

	// If this widget is being redefined then we need to find all widgets that
	// are inheriting from it and redefine all of them so that they inherit from
	// the new version of this widget. We're essentially trying to replace one
	// level in the prototype chain.
	if ( existingConstructor ) {
		$.each( existingConstructor._childConstructors, function( i, child ) {
			var childPrototype = child.prototype;

			// redefine the child widget using the same prototype that was
			// originally used, but inherit from the new version of the base
			$.widget( childPrototype.namespace + "." + childPrototype.widgetName, constructor, child._proto );
		});
		// remove the list of existing child constructors from the old constructor
		// so the old child constructors can be garbage collected
		delete existingConstructor._childConstructors;
	} else {
		base._childConstructors.push( constructor );
	}

	$.widget.bridge( name, constructor );

	return constructor;
};

$.widget.extend = function( target ) {
	var input = widget_slice.call( arguments, 1 ),
		inputIndex = 0,
		inputLength = input.length,
		key,
		value;
	for ( ; inputIndex < inputLength; inputIndex++ ) {
		for ( key in input[ inputIndex ] ) {
			value = input[ inputIndex ][ key ];
			if ( input[ inputIndex ].hasOwnProperty( key ) && value !== undefined ) {
				// Clone objects
				if ( $.isPlainObject( value ) ) {
					target[ key ] = $.isPlainObject( target[ key ] ) ?
						$.widget.extend( {}, target[ key ], value ) :
						// Don't extend strings, arrays, etc. with objects
						$.widget.extend( {}, value );
				// Copy everything else by reference
				} else {
					target[ key ] = value;
				}
			}
		}
	}
	return target;
};

$.widget.bridge = function( name, object ) {
	var fullName = object.prototype.widgetFullName || name;
	$.fn[ name ] = function( options ) {
		var isMethodCall = typeof options === "string",
			args = widget_slice.call( arguments, 1 ),
			returnValue = this;

		if ( isMethodCall ) {
			this.each(function() {
				var methodValue,
					instance = $.data( this, fullName );
				if ( options === "instance" ) {
					returnValue = instance;
					return false;
				}
				if ( !instance ) {
					return $.error( "cannot call methods on " + name + " prior to initialization; " +
						"attempted to call method '" + options + "'" );
				}
				if ( !$.isFunction( instance[options] ) || options.charAt( 0 ) === "_" ) {
					return $.error( "no such method '" + options + "' for " + name + " widget instance" );
				}
				methodValue = instance[ options ].apply( instance, args );
				if ( methodValue !== instance && methodValue !== undefined ) {
					returnValue = methodValue && methodValue.jquery ?
						returnValue.pushStack( methodValue.get() ) :
						methodValue;
					return false;
				}
			});
		} else {

			// Allow multiple hashes to be passed on init
			if ( args.length ) {
				options = $.widget.extend.apply( null, [ options ].concat(args) );
			}

			this.each(function() {
				var instance = $.data( this, fullName );
				if ( instance ) {
					instance.option( options || {} );
					if ( instance._init ) {
						instance._init();
					}
				} else {
					$.data( this, fullName, new object( options, this ) );
				}
			});
		}

		return returnValue;
	};
};

$.Widget = function( /* options, element */ ) {};
$.Widget._childConstructors = [];

$.Widget.prototype = {
	widgetName: "widget",
	widgetEventPrefix: "",
	defaultElement: "<div>",
	options: {
		disabled: false,

		// callbacks
		create: null
	},
	_createWidget: function( options, element ) {
		element = $( element || this.defaultElement || this )[ 0 ];
		this.element = $( element );
		this.uuid = widget_uuid++;
		this.eventNamespace = "." + this.widgetName + this.uuid;

		this.bindings = $();
		this.hoverable = $();
		this.focusable = $();

		if ( element !== this ) {
			$.data( element, this.widgetFullName, this );
			this._on( true, this.element, {
				remove: function( event ) {
					if ( event.target === element ) {
						this.destroy();
					}
				}
			});
			this.document = $( element.style ?
				// element within the document
				element.ownerDocument :
				// element is window or document
				element.document || element );
			this.window = $( this.document[0].defaultView || this.document[0].parentWindow );
		}

		this.options = $.widget.extend( {},
			this.options,
			this._getCreateOptions(),
			options );

		this._create();
		this._trigger( "create", null, this._getCreateEventData() );
		this._init();
	},
	_getCreateOptions: $.noop,
	_getCreateEventData: $.noop,
	_create: $.noop,
	_init: $.noop,

	destroy: function() {
		this._destroy();
		// we can probably remove the unbind calls in 2.0
		// all event bindings should go through this._on()
		this.element
			.unbind( this.eventNamespace )
			.removeData( this.widgetFullName )
			// support: jquery <1.6.3
			// http://bugs.jquery.com/ticket/9413
			.removeData( $.camelCase( this.widgetFullName ) );
		this.widget()
			.unbind( this.eventNamespace )
			.removeAttr( "aria-disabled" )
			.removeClass(
				this.widgetFullName + "-disabled " +
				"ui-state-disabled" );

		// clean up events and states
		this.bindings.unbind( this.eventNamespace );
		this.hoverable.removeClass( "ui-state-hover" );
		this.focusable.removeClass( "ui-state-focus" );
	},
	_destroy: $.noop,

	widget: function() {
		return this.element;
	},

	option: function( key, value ) {
		var options = key,
			parts,
			curOption,
			i;

		if ( arguments.length === 0 ) {
			// don't return a reference to the internal hash
			return $.widget.extend( {}, this.options );
		}

		if ( typeof key === "string" ) {
			// handle nested keys, e.g., "foo.bar" => { foo: { bar: ___ } }
			options = {};
			parts = key.split( "." );
			key = parts.shift();
			if ( parts.length ) {
				curOption = options[ key ] = $.widget.extend( {}, this.options[ key ] );
				for ( i = 0; i < parts.length - 1; i++ ) {
					curOption[ parts[ i ] ] = curOption[ parts[ i ] ] || {};
					curOption = curOption[ parts[ i ] ];
				}
				key = parts.pop();
				if ( arguments.length === 1 ) {
					return curOption[ key ] === undefined ? null : curOption[ key ];
				}
				curOption[ key ] = value;
			} else {
				if ( arguments.length === 1 ) {
					return this.options[ key ] === undefined ? null : this.options[ key ];
				}
				options[ key ] = value;
			}
		}

		this._setOptions( options );

		return this;
	},
	_setOptions: function( options ) {
		var key;

		for ( key in options ) {
			this._setOption( key, options[ key ] );
		}

		return this;
	},
	_setOption: function( key, value ) {
		this.options[ key ] = value;

		if ( key === "disabled" ) {
			this.widget()
				.toggleClass( this.widgetFullName + "-disabled", !!value );

			// If the widget is becoming disabled, then nothing is interactive
			if ( value ) {
				this.hoverable.removeClass( "ui-state-hover" );
				this.focusable.removeClass( "ui-state-focus" );
			}
		}

		return this;
	},

	enable: function() {
		return this._setOptions({ disabled: false });
	},
	disable: function() {
		return this._setOptions({ disabled: true });
	},

	_on: function( suppressDisabledCheck, element, handlers ) {
		var delegateElement,
			instance = this;

		// no suppressDisabledCheck flag, shuffle arguments
		if ( typeof suppressDisabledCheck !== "boolean" ) {
			handlers = element;
			element = suppressDisabledCheck;
			suppressDisabledCheck = false;
		}

		// no element argument, shuffle and use this.element
		if ( !handlers ) {
			handlers = element;
			element = this.element;
			delegateElement = this.widget();
		} else {
			element = delegateElement = $( element );
			this.bindings = this.bindings.add( element );
		}

		$.each( handlers, function( event, handler ) {
			function handlerProxy() {
				// allow widgets to customize the disabled handling
				// - disabled as an array instead of boolean
				// - disabled class as method for disabling individual parts
				if ( !suppressDisabledCheck &&
						( instance.options.disabled === true ||
							$( this ).hasClass( "ui-state-disabled" ) ) ) {
					return;
				}
				return ( typeof handler === "string" ? instance[ handler ] : handler )
					.apply( instance, arguments );
			}

			// copy the guid so direct unbinding works
			if ( typeof handler !== "string" ) {
				handlerProxy.guid = handler.guid =
					handler.guid || handlerProxy.guid || $.guid++;
			}

			var match = event.match( /^([\w:-]*)\s*(.*)$/ ),
				eventName = match[1] + instance.eventNamespace,
				selector = match[2];
			if ( selector ) {
				delegateElement.delegate( selector, eventName, handlerProxy );
			} else {
				element.bind( eventName, handlerProxy );
			}
		});
	},

	_off: function( element, eventName ) {
		eventName = (eventName || "").split( " " ).join( this.eventNamespace + " " ) +
			this.eventNamespace;
		element.unbind( eventName ).undelegate( eventName );

		// Clear the stack to avoid memory leaks (#10056)
		this.bindings = $( this.bindings.not( element ).get() );
		this.focusable = $( this.focusable.not( element ).get() );
		this.hoverable = $( this.hoverable.not( element ).get() );
	},

	_delay: function( handler, delay ) {
		function handlerProxy() {
			return ( typeof handler === "string" ? instance[ handler ] : handler )
				.apply( instance, arguments );
		}
		var instance = this;
		return setTimeout( handlerProxy, delay || 0 );
	},

	_hoverable: function( element ) {
		this.hoverable = this.hoverable.add( element );
		this._on( element, {
			mouseenter: function( event ) {
				$( event.currentTarget ).addClass( "ui-state-hover" );
			},
			mouseleave: function( event ) {
				$( event.currentTarget ).removeClass( "ui-state-hover" );
			}
		});
	},

	_focusable: function( element ) {
		this.focusable = this.focusable.add( element );
		this._on( element, {
			focusin: function( event ) {
				$( event.currentTarget ).addClass( "ui-state-focus" );
			},
			focusout: function( event ) {
				$( event.currentTarget ).removeClass( "ui-state-focus" );
			}
		});
	},

	_trigger: function( type, event, data ) {
		var prop, orig,
			callback = this.options[ type ];

		data = data || {};
		event = $.Event( event );
		event.type = ( type === this.widgetEventPrefix ?
			type :
			this.widgetEventPrefix + type ).toLowerCase();
		// the original event may come from any element
		// so we need to reset the target on the new event
		event.target = this.element[ 0 ];

		// copy original event properties over to the new event
		orig = event.originalEvent;
		if ( orig ) {
			for ( prop in orig ) {
				if ( !( prop in event ) ) {
					event[ prop ] = orig[ prop ];
				}
			}
		}

		this.element.trigger( event, data );
		return !( $.isFunction( callback ) &&
			callback.apply( this.element[0], [ event ].concat( data ) ) === false ||
			event.isDefaultPrevented() );
	}
};

$.each( { show: "fadeIn", hide: "fadeOut" }, function( method, defaultEffect ) {
	$.Widget.prototype[ "_" + method ] = function( element, options, callback ) {
		if ( typeof options === "string" ) {
			options = { effect: options };
		}
		var hasOptions,
			effectName = !options ?
				method :
				options === true || typeof options === "number" ?
					defaultEffect :
					options.effect || defaultEffect;
		options = options || {};
		if ( typeof options === "number" ) {
			options = { duration: options };
		}
		hasOptions = !$.isEmptyObject( options );
		options.complete = callback;
		if ( options.delay ) {
			element.delay( options.delay );
		}
		if ( hasOptions && $.effects && $.effects.effect[ effectName ] ) {
			element[ method ]( options );
		} else if ( effectName !== method && element[ effectName ] ) {
			element[ effectName ]( options.duration, options.easing, callback );
		} else {
			element.queue(function( next ) {
				$( this )[ method ]();
				if ( callback ) {
					callback.call( element[ 0 ] );
				}
				next();
			});
		}
	};
});

return $.widget;

}));
;

// resource: org.richfaces:richfaces-selection.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

window.RichFaces = window.RichFaces || {};
RichFaces.jQuery = RichFaces.jQuery || window.jQuery;

(function (rf) {

    rf.Selection = rf.Selection || {};

    rf.Selection.set = function (field, start, end) {
        if (field.setSelectionRange) {
            field.focus();
            field.setSelectionRange(start, end);
        } else if (field.createTextRange) {
            var range = field.createTextRange();
            range.collapse(true);
            range.moveEnd('character', end);
            range.moveStart('character', start);
            range.select();
        }
    }

    rf.Selection.getStart = function(field) {
        if (field.setSelectionRange) {
            return field.selectionStart;
        } else if (document.selection && document.selection.createRange) {
            var r = document.selection.createRange().duplicate();
            r.moveEnd('character', field.value.length);
            if (r.text == '') return field.value.length;
            return field.value.lastIndexOf(r.text);
        }
    }

    rf.Selection.getEnd = function(field) {
        if (field.setSelectionRange) {
            return field.selectionEnd;
        } else if (document.selection && document.selection.createRange) {
            var r = document.selection.createRange().duplicate();
            r.moveStart('character', -field.value.length);
            return r.text.length;
        }
    }

    rf.Selection.setCaretTo = function (field, pos) {
        if (!pos) pos = field.value.length;
        rf.Selection.set(field, pos, pos);
    }
})(RichFaces);;

// resource: org.richfaces:tabPanel.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    var ITEMS_SWITCHER = {

        /**
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {void}
         * */
        exec : function (oldPanel, newPanel) {
            if (newPanel.switchMode == "server") {
                return this.execServer(oldPanel, newPanel);
            } else if (newPanel.switchMode == "ajax") {
                return this.execAjax(oldPanel, newPanel);
            } else if (newPanel.switchMode == "client") {
                return this.execClient(oldPanel, newPanel);
            } else {
                rf.log.error("SwitchItems.exec : unknown switchMode (" + newPanel.switchMode + ")");
            }
        },

        /**
         * @protected
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {Boolean} false
         * */
        execServer : function (oldPanel, newPanel) {
            if (oldPanel) {
                var continueProcess = oldPanel.__leave();
                if (!continueProcess) {
                    return false;
                }
            }

            this.__setActiveItem(newPanel);

            var params = {};

            params[newPanel.getTogglePanel().id] = newPanel.name;
            params[newPanel.id] = newPanel.id;

            $.extend(params, newPanel.getTogglePanel().options["ajax"] || {});

            rf.submitForm(this.__getParentForm(newPanel), params);

            return false;
        },

        /**
         * @protected
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {Boolean} false
         * */
        execAjax : function (oldPanel, newPanel) {
            var options = $.extend({}, newPanel.getTogglePanel().options["ajax"], oldPanel.options.ajax);

            this.__setActiveItem(newPanel);
            rf.ajax(newPanel.id, null, options);

            if (oldPanel) {
                this.__setActiveItem(oldPanel);
            }

            return false;
        },

        /**
         * @protected
         * @param {TogglePanelItem} oldPanel
         * @param {TogglePanelItem} newPanel
         *
         * @return {undefined}
         *             - false - if process has been terminated
         *             - true  - in other cases
         * */
        execClient : function (oldPanel, newPanel) {
            if (oldPanel) {
                var continueProcess = oldPanel.__leave();
                if (!continueProcess) {
                    return false;
                }
            }

            this.__setActiveItem(newPanel);

            newPanel.__enter();
            newPanel.getTogglePanel().__fireItemChange(oldPanel, newPanel);

            return true;
        },

        /**
         * @private
         * */
        __getParentForm : function (comp) {
            return $(rf.getDomElement(comp.id)).parents('form:first');
        },

        /**
         * @private
         * */
        __setActiveItem : function (item) {
            rf.getDomElement(item.togglePanelId + "-value").value = item.getName(); // todo it is should be toogle panel method
            item.getTogglePanel().activeItem = item.getName();
        }
    };


    rf.ui.TabPanel = rf.ui.TogglePanel.extendClass({
            // class name
            name:"TabPanel",

            /**
             * @class TabPanel
             * @name TabPanel
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                rf.ui.TogglePanel.call(this, componentId, options);
                this.items = [];

                this.isKeepHeight = options["isKeepHeight"] || false;

                this.element = document.getElementById(componentId);
                var $element = $(this.element);

                $element.on("click", ".rf-tab-hdr-act", $.proxy(this.__clickListener, this))
                $element.on("click", ".rf-tab-hdr-inact", $.proxy(this.__clickListener, this))
            },

            __clickListener: function(event) {
                var header = $(event.target);
                if (! header.hasClass("rf-tab-hdr")) {
                    header = header.parents(".rf-tab-hdr").first();
                }
                var tabname = header.data('tabname');
                this.switchToItem(tabname);
            },

            __itemsSwitcher : function () {
                return ITEMS_SWITCHER;
            }

        });
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:focus.js
(function($, rf) {

    rf.ui = rf.ui || {};

    var defaultOptions = {
        useNative : false
    };

    rf.ui.Focus = rf.BaseComponent.extendClass({

        name : "Focus",

        init : function(componentId, options) {
            $super.constructor.call(this, componentId);
            options = this.options = $.extend({}, defaultOptions, options);
            this.attachToDom(this.id);

            var focusInput = $(document.getElementById(componentId + 'InputFocus'));
            var focusCandidates = this.options.focusCandidates;

            $(document).on('focus', ':tabbable', function(e) {
                var target = $(e.target);
                if (!target.is(':editable')) {
                    return;
                }
                var ids = e.target.id || '';
                target.parents().each(function() {
                    var id = $(this).attr('id');
                    if (id) {
                        ids += ' ' + id;
                    }
                });
                focusInput.val(ids);
                rf.log.debug('Focus - clientId candidates for components: ' + ids);
            });

            if (this.options.mode === 'VIEW') {
                $(document).on('ajaxsubmit submit', 'form', function(e) {
                    var form = $(e.target);
                    var input = $("input[name='org.richfaces.focus']", form);
                    if (!input.length) {
                        input = $('<input name="org.richfaces.focus" type="hidden" />').appendTo(form);
                    }
                    input.val(focusInput.val());
                });
            }

            this.options.applyFocus = $.proxy(function() {
                var tabbables = $();

                if (focusCandidates) {
                    var candidates = focusCandidates;
                    rf.log.debug('Focus - focus candidates: ' + candidates);
                    candidates = candidates.split(' ');
                    $.each(candidates, function(i, v) {
                        var candidate = $(document.getElementById(v));
                        tabbables = tabbables.add($(":tabbable", candidate));

                        if (candidate.is(":tabbable")) {
                            tabbables = tabbables.add(candidate);
                        }
                    });

                    if (tabbables.length == 0) {
                        tabbables = $('form').has(focusInput).find(':tabbable')
                    }
                } else if (this.options.mode == 'VIEW') {
                    tabbables = $("body form:first :tabbable");
                }

                if (tabbables.length > 0) {
                    tabbables = tabbables.sort(sortTabindex);
                    tabbables.get(0).focus();
                }
            }, this);
        },

        applyFocus : function() {
            $(this.options.applyFocus);
        },

        // destructor definition
        destroy : function() {
            // define destructor if additional cleaning is needed but
            // in most cases its not nessesary.
            // call parent’s destructor
            $super.destroy.call(this);
        }
    });

    /**
     * Returns the tabindex sort order of two elements based on their tabindex and position in the DOM, following real tabbing
     * order implemented by browsers.
     * 
     * Returns negative number when element A has lesser tabindex than B or it is closer the start of the DOM; returns negative
     * number when element B has lesser tabindex than A or it is closer the start of the DOM; returns 0 if both A and B points
     * to same element.
     */
    var sortTabindex = function(a, b) {
        var result = sortTabindexNums($(a).attr('tabindex'), $(b).attr('tabindex'));

        return (result != 0) ? result : sortByDOMOrder(a, b);
    };

    /**
     * Sorts two tabindex values (positive number or undefined).
     * 
     * Returns negative number when tabindex A is lesser than B; returns positive number when tabindex B is lesser than A;
     * returns 0 if both A and B has same values.
     */
    var sortTabindexNums = function(a, b) {
        if (a) {
            if (b) {
                return a - b;
            } else {
                return -1;
            }
        } else {
            if (b) {
                return +1;
            } else {
                return 0;
            }
        }
    };

    /**
     * Detects absolute order of two elements in the DOM tree.
     * 
     * Returns negative number when element A is closer the start of the DOM; returns positive number when element B is closer
     * the start of the DOM; returns 0 if both A and B points to same element
     */
    var sortByDOMOrder = function(a, b) {
        var r = searchCommonParent(a, b);
        if (a == b) {
            return 0;
        } else if (r.parent == a) {
            return -1;
        } else if (r.parent == b) {
            return +1;
        } else {
            return $(r.aPrevious).index() - $(r.bPrevious).index();
        }
    };

    /**
     * Search for common parent for two given elements.
     * 
     * returns object containing following parameters:
     * 
     * result.parent - the commnon parent for A and B result.aPrevious - the parent's direct child which is on the branch
     * leading to A in DOM tree result.bPrevious - the parent's direct child which is on the branch leading to B in DOM tree
     */
    var searchCommonParent = function(a, b) {
        var aParents = $(a).add($(a).parents()).get().reverse();
        var bParents = $(b).add($(b).parents()).get().reverse();
        var r = {
            aPrevious : a,
            bPrevious : b
        };
        $.each(aParents, function(i, ap) {
            $.each(bParents, function(j, bp) {
                if (ap == bp) {
                    r.parent = ap;
                    return false;
                }
                r.bPrevious = bp;
            });
            if (r.parent) {
                return false;
            }
            r.aPrevious = ap;
        });
        if (!r.parent) {
            return null;
        }
        return r;
    };

    /**
     * Exposes sortTabindex family of functions for testing
     */
    rf.ui.Focus.__fn = {
        'sortTabindex' : sortTabindex,
        'sortTabindexNums' : sortTabindexNums,
        'searchCommonParent' : searchCommonParent,
        'sortByDOMOrder' : sortByDOMOrder
    }

    // define super class reference - reference to the parent prototype
    var $super = rf.ui.Focus.$super;
})(RichFaces.jQuery, RichFaces);;

// resource: net.java.dev.atmosphere:jquery-atmosphere.js
/*
 * Copyright 2015 Async-IO.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Atmosphere.js
 * https://github.com/Atmosphere/atmosphere-javascript
 *
 * Requires
 * - jQuery 2.0.3 http://jquery.com/
 *
 * API reference
 * https://github.com/Atmosphere/atmosphere/wiki/jQuery.atmosphere.js-API
 *
 * Highly inspired by
 * - Portal by Donghwan Kim http://flowersinthesand.github.io/portal/
 */
(function (factory) {
    if (typeof define === 'function' && define.amd) {
        // AMD
        define(['jquery'], factory);
    } else {
        // Browser globals, Window
        factory(jQuery);
    }
}(function (jQuery) {

    jQuery(window).bind("unload.atmosphere", function () {
        jQuery.atmosphere.debug(new Date() + " Atmosphere: " + "unload event");
        jQuery.atmosphere.unsubscribe();
    });

    jQuery(window).bind("beforeunload.atmosphere", function () {
        jQuery.atmosphere.debug(new Date() + " Atmosphere: " + "beforeunload event");

        // ATMOSPHERE-JAVASCRIPT-143: Delay reconnect to avoid reconnect attempts before an actual unload (we don't know if an unload will happen, yet)
        jQuery.atmosphere._beforeUnloadState = true;
        setTimeout(function () {
            jQuery.atmosphere.debug(new Date() + " Atmosphere: " + "beforeunload event timeout reached. Reset _beforeUnloadState flag");
            jQuery.atmosphere._beforeUnloadState = false;
        }, 5000);
    });

    jQuery(window).bind("offline", function () {
        jQuery.atmosphere.offline = true;
        var requestsClone = [].concat(jQuery.atmosphere.requests);
        for (var i = 0; i < requestsClone.length; i++) {
            var rq = requestsClone[i];
            rq.close();
            clearTimeout(rq.response.request.id);

            if (rq.heartbeatTimer) {
                clearTimeout(rq.heartbeatTimer);
            }
        }
    });

    jQuery(window).bind("online", function () {
        jQuery.atmosphere.offline = false;
        if (jQuery.atmosphere.requests.length > 0) {
            for (var i = 0; i < jQuery.atmosphere.requests.length; i++) {
                jQuery.atmosphere.requests[i].init();
                jQuery.atmosphere.requests[i].execute();
            }
        }
    });

    // Prevent ESC to kill the connection from Firefox.
    jQuery(window).keypress(function (e) {
        if (e.keyCode === 27) {
            e.preventDefault();
        }
    });

    var parseHeaders = function (headerString) {
        var match, rheaders = /^(.*?):[ \t]*([^\r\n]*)\r?$/mg, headers = {};
        while (match = rheaders.exec(headerString)) {
            headers[match[1]] = match[2];
        }
        return headers;
    };

    jQuery.atmosphere = {
        version: "2.2.12-jquery",
        uuid: 0,
        offline: false,
        requests: [],
        callbacks: [],

        onError: function (response) {
        },
        onClose: function (response) {
        },
        onOpen: function (response) {
        },
        onMessage: function (response) {
        },
        onReconnect: function (request, response) {
        },
        onMessagePublished: function (response) {
        },
        onTransportFailure: function (errorMessage, _request) {
        },
        onLocalMessage: function (response) {
        },
        onClientTimeout: function (request) {
        },
        onFailureToReconnect: function (request, response) {
        },

        /**
         * Creates an object based on an atmosphere subscription that exposes functions defined by the Websocket interface.
         *
         * @class WebsocketApiAdapter
         * @param {Object} request the request object to build the underlying subscription
         * @constructor
         */
        WebsocketApiAdapter: function (request) {
            var _socket, _adapter;

            /**
             * Overrides the onMessage callback in given request.
             *
             * @method onMessage
             * @param {Object} e the event object
             */
            request.onMessage = function (e) {
                _adapter.onmessage({data: e.responseBody});
            };

            /**
             * Overrides the onMessagePublished callback in given request.
             *
             * @method onMessagePublished
             * @param {Object} e the event object
             */
            request.onMessagePublished = function (e) {
                _adapter.onmessage({data: e.responseBody});
            };

            /**
             * Overrides the onOpen callback in given request to proxy the event to the adapter.
             *
             * @method onOpen
             * @param {Object} e the event object
             */
            request.onOpen = function (e) {
                _adapter.onopen(e);
            };

            _adapter = {
                close: function () {
                    _socket.close();
                },

                send: function (data) {
                    _socket.push(data);
                },

                onmessage: function (e) {
                },

                onopen: function (e) {
                },

                onclose: function (e) {
                },

                onerror: function (e) {

                }
            };
            _socket = new $.atmosphere.subscribe(request);

            return _adapter;
        },

        AtmosphereRequest: function (options) {

            /**
             * {Object} Request parameters.
             *
             * @private
             */
            var _request = {
                timeout: 300000,
                method: 'GET',
                headers: {},
                contentType: '',
                callback: null,
                url: '',
                data: '',
                suspend: true,
                maxRequest: -1,
                reconnect: true,
                maxStreamingLength: 10000000,
                lastIndex: 0,
                logLevel: 'info',
                requestCount: 0,
                fallbackMethod: 'GET',
                fallbackTransport: 'streaming',
                transport: 'long-polling',
                webSocketImpl: null,
                webSocketBinaryType: null,
                dispatchUrl: null,
                webSocketPathDelimiter: "@@",
                enableXDR: false,
                rewriteURL: false,
                attachHeadersAsQueryString: true,
                executeCallbackBeforeReconnect: false,
                readyState: 0,
                withCredentials: false,
                trackMessageLength: false,
                messageDelimiter: '|',
                connectTimeout: -1,
                reconnectInterval: 0,
                dropHeaders: true,
                uuid: 0,
                shared: false,
                readResponsesHeaders: false,
                maxReconnectOnClose: 5,
                enableProtocol: true,
                pollingInterval: 0,
                heartbeat: {
                    client: null,
                    server: null
                },
                ackInterval: 0,
                closeAsync: false,
                reconnectOnServerError: true,
                onError: function (response) {
                },
                onClose: function (response) {
                },
                onOpen: function (response) {
                },
                onMessage: function (response) {
                },
                onReopen: function (request, response) {
                },
                onReconnect: function (request, response) {
                },
                onMessagePublished: function (response) {
                },
                onTransportFailure: function (reason, request) {
                },
                onLocalMessage: function (request) {
                },
                onFailureToReconnect: function (request, response) {
                },
                onClientTimeout: function (request) {
                }
            };

            /**
             * {Object} Request's last response.
             *
             * @private
             */
            var _response = {
                status: 200,
                reasonPhrase: "OK",
                responseBody: '',
                messages: [],
                headers: [],
                state: "messageReceived",
                transport: "polling",
                error: null,
                request: null,
                partialMessage: "",
                errorHandled: false,
                closedByClientTimeout: false,
                ffTryingReconnect: false
            };

            /**
             * {websocket} Opened web socket.
             *
             * @private
             */
            var _websocket = null;

            /**
             * {SSE} Opened SSE.
             *
             * @private
             */
            var _sse = null;

            /**
             * {XMLHttpRequest, ActiveXObject} Opened ajax request (in case of http-streaming or long-polling)
             *
             * @private
             */
            var _activeRequest = null;

            /**
             * {Object} Object use for streaming with IE.
             *
             * @private
             */
            var _ieStream = null;

            /**
             * {Object} Object use for jsonp transport.
             *
             * @private
             */
            var _jqxhr = null;

            /**
             * {boolean} If request has been subscribed or not.
             *
             * @private
             */
            var _subscribed = true;

            /**
             * {number} Number of test reconnection.
             *
             * @private
             */
            var _requestCount = 0;

            /**
             * The Heartbeat interval send by the server.
             * @type {int}
             * @private
             */
            var _heartbeatInterval = 0;

            /**
             * The Heartbeat bytes send by the server.
             * @type {string}
             * @private
             */
            var _heartbeatPadding = 'X';

            /**
             * {boolean} If request is currently aborted.
             *
             * @private
             */
            var _abortingConnection = false;

            /**
             * A local "channel' of communication.
             *
             * @private
             */
            var _localSocketF = null;

            /**
             * The storage used.
             *
             * @private
             */
            var _storageService;

            /**
             * Local communication
             *
             * @private
             */
            var _localStorageService = null;

            /**
             * A Unique ID
             *
             * @private
             */
            var guid = jQuery.now();

            /** Trace time */
            var _traceTimer;

            /** Key for connection sharing */
            var _sharingKey;

            /**
             * {boolean} If window beforeUnload event has been called.
             * Flag will be reset after 5000 ms
             *
             * @private
             */
            var _beforeUnloadState = false;

            // Automatic call to subscribe
            _subscribe(options);

            /**
             * Initialize atmosphere request object.
             *
             * @private
             */
            function _init() {
                _subscribed = true;
                _abortingConnection = false;
                _requestCount = 0;

                _websocket = null;
                _sse = null;
                _activeRequest = null;
                _ieStream = null;
            }

            /**
             * Re-initialize atmosphere object.
             *
             * @private
             */
            function _reinit() {
                _clearState();
                _init();
            }

            /**
             * Returns true if the given level is equal or above the configured log level.
             *
             * @private
             */
            function _canLog(level) {
                if (level == 'debug') {
                    return _request.logLevel === 'debug';
                } else if (level == 'info') {
                    return _request.logLevel === 'info' || _request.logLevel === 'debug';
                } else if (level == 'warn') {
                    return _request.logLevel === 'warn' || _request.logLevel === 'info' || _request.logLevel === 'debug';
                } else if (level == 'error') {
                    return _request.logLevel === 'error' || _request.logLevel === 'warn' || _request.logLevel === 'info' || _request.logLevel === 'debug';
                } else {
                    return false;
                }
            }

            function _debug(msg) {
                if (_canLog('debug')) {
                    jQuery.atmosphere.debug(new Date() + " Atmosphere: " + msg);
                }
            }

            /**
             * Subscribe request using request transport. <br>
             * If request is currently opened, this one will be closed.
             *
             * @param {Object} Request parameters.
             * @private
             */
            function _subscribe(options) {
                _reinit();

                _request = jQuery.extend(_request, options);
                // Allow at least 1 request
                _request.mrequest = _request.reconnect;
                if (!_request.reconnect) {
                    _request.reconnect = true;
                }
            }

            /**
             * Check if web socket is supported (check for custom implementation provided by request object or browser implementation).
             *
             * @returns {boolean} True if web socket is supported, false otherwise.
             * @private
             */
            function _supportWebsocket() {
                return _request.webSocketImpl != null || window.WebSocket || window.MozWebSocket;
            }

            /**
             * Check if server side events (SSE) is supported (check for custom implementation provided by request object or browser implementation).
             *
             * @returns {boolean} True if web socket is supported, false otherwise.
             * @private
             */
            function _supportSSE() {
                function makeAbsolute(url) {
                    var div = document.createElement("div");
                    // Uses an innerHTML property to obtain an absolute URL
                    div.innerHTML = '<a href="' + url + '"/>';
                    // encodeURI and decodeURI are needed to normalize URL between Internet Explorer and non-Internet Explorer,
                    // since Internet Explorer doesn't encode the href property value and return it - http://jsfiddle.net/Yq9M8/1/
                    return encodeURI(decodeURI(div.firstChild.href));
                }

                // Origin parts
                var url = makeAbsolute(_request.url.toLowerCase());
                var parts = /^([\w\+\.\-]+:)(?:\/\/([^\/?#:]*)(?::(\d+))?)?/.exec(url);
                var crossOrigin = !!(parts && (
                    // protocol
                parts[1] != window.location.protocol ||
                    // hostname
                parts[2] != window.location.hostname ||
                    // port
                (parts[3] || (parts[1] === "http:" ? 80 : 443)) != (window.location.port || (window.location.protocol === "http:" ? 80 : 443))
                ));
                return window.EventSource && (!crossOrigin || !jQuery.browser.safari || jQuery.browser.vmajor >= 7);
            }

            /**
             * Open request using request transport. <br>
             * If request transport is 'websocket' but websocket can't be opened, request will automatically reconnect using fallback transport.
             *
             * @private
             */
            function _execute() {
                // Shared across multiple tabs/windows.
                if (_request.shared) {
                    _localStorageService = _local(_request);
                    if (_localStorageService != null) {
                        if (_canLog('debug')) {
                            jQuery.atmosphere.debug("Storage service available. All communication will be local");
                        }

                        if (_localStorageService.open(_request)) {
                            // Local connection.
                            return;
                        }
                    }

                    if (_canLog('debug')) {
                        jQuery.atmosphere.debug("No Storage service available.");
                    }
                    // Wasn't local or an error occurred
                    _localStorageService = null;
                }

                // Protocol
                _request.firstMessage = jQuery.atmosphere.uuid == 0 ? true : false;
                _request.isOpen = false;
                _request.ctime = jQuery.now();

                // We carry any UUID set by the user or from a previous connection.
                if (_request.uuid === 0) {
                    _request.uuid = jQuery.atmosphere.uuid;
                }
                _request.closedByClientTimeout = false;

                if (_request.transport !== 'websocket' && _request.transport !== 'sse') {
                    _executeRequest(_request);

                } else if (_request.transport === 'websocket') {
                    if (!_supportWebsocket()) {
                        _reconnectWithFallbackTransport("Websocket is not supported, using request.fallbackTransport (" + _request.fallbackTransport
                        + ")");
                    } else {
                        _executeWebSocket(false);
                    }
                } else if (_request.transport === 'sse') {
                    if (!_supportSSE()) {
                        _reconnectWithFallbackTransport("Server Side Events(SSE) is not supported, using request.fallbackTransport ("
                        + _request.fallbackTransport + ")");
                    } else {
                        _executeSSE(false);
                    }
                }
            }

            function _local(request) {
                var trace, connector, orphan, name = "atmosphere-" + request.url, connectors = {
                    storage: function () {
                        if (!jQuery.atmosphere.supportStorage()) {
                            return;
                        }

                        var storage = window.localStorage, get = function (key) {
                            return jQuery.parseJSON(storage.getItem(name + "-" + key));
                        }, set = function (key, value) {
                            storage.setItem(name + "-" + key, jQuery.stringifyJSON(value));
                        };

                        return {
                            init: function () {
                                set("children", get("children").concat([guid]));
                                jQuery(window).on("storage.socket", function (event) {
                                    event = event.originalEvent;
                                    if (event.key === name && event.newValue) {
                                        listener(event.newValue);
                                    }
                                });
                                return get("opened");
                            },
                            signal: function (type, data) {
                                storage.setItem(name, jQuery.stringifyJSON({
                                    target: "p",
                                    type: type,
                                    data: data
                                }));
                            },
                            close: function () {
                                var index, children = get("children");

                                jQuery(window).off("storage.socket");
                                if (children) {
                                    index = jQuery.inArray(request.id, children);
                                    if (index > -1) {
                                        children.splice(index, 1);
                                        set("children", children);
                                    }
                                }
                            }
                        };
                    },
                    windowref: function () {
                        var win = window.open("", name.replace(/\W/g, ""));

                        if (!win || win.closed || !win.callbacks) {
                            return;
                        }

                        return {
                            init: function () {
                                win.callbacks.push(listener);
                                win.children.push(guid);
                                return win.opened;
                            },
                            signal: function (type, data) {
                                if (!win.closed && win.fire) {
                                    win.fire(jQuery.stringifyJSON({
                                        target: "p",
                                        type: type,
                                        data: data
                                    }));
                                }
                            },
                            close: function () {
                                function remove(array, e) {
                                    var index = jQuery.inArray(e, array);
                                    if (index > -1) {
                                        array.splice(index, 1);
                                    }
                                }

                                // Removes traces only if the parent is alive
                                if (!orphan) {
                                    remove(win.callbacks, listener);
                                    remove(win.children, guid);
                                }
                            }

                        };
                    }
                };

                // Receives open, close and message command from the parent
                function listener(string) {
                    var command = jQuery.parseJSON(string), data = command.data;

                    if (command.target === "c") {
                        switch (command.type) {
                            case "open":
                                _open("opening", 'local', _request);
                                break;
                            case "close":
                                if (!orphan) {
                                    orphan = true;
                                    if (data.reason === "aborted") {
                                        _close();
                                    } else {
                                        // Gives the heir some time to reconnect
                                        if (data.heir === guid) {
                                            _execute();
                                        } else {
                                            setTimeout(function () {
                                                _execute();
                                            }, 100);
                                        }
                                    }
                                }
                                break;
                            case "message":
                                _prepareCallback(data, "messageReceived", 200, request.transport);
                                break;
                            case "localMessage":
                                _localMessage(data);
                                break;
                        }
                    }
                }

                function findTrace() {
                    var matcher = new RegExp("(?:^|; )(" + encodeURIComponent(name) + ")=([^;]*)").exec(document.cookie);
                    if (matcher) {
                        return jQuery.parseJSON(decodeURIComponent(matcher[2]));
                    }
                }

                // Finds and validates the parent socket's trace from the cookie
                trace = findTrace();
                if (!trace || jQuery.now() - trace.ts > 1000) {
                    return;
                }

                // Chooses a connector
                connector = connectors.storage() || connectors.windowref();
                if (!connector) {
                    return;
                }

                return {
                    open: function () {
                        var parentOpened;

                        // Checks the shared one is alive
                        _traceTimer = setInterval(function () {
                            var oldTrace = trace;
                            trace = findTrace();
                            if (!trace || oldTrace.ts === trace.ts) {
                                // Simulates a close signal
                                listener(jQuery.stringifyJSON({
                                    target: "c",
                                    type: "close",
                                    data: {
                                        reason: "error",
                                        heir: oldTrace.heir
                                    }
                                }));
                            }
                        }, 1000);

                        parentOpened = connector.init();
                        if (parentOpened) {
                            // Firing the open event without delay robs the user of the opportunity to bind connecting event handlers
                            setTimeout(function () {
                                _open("opening", 'local', request);
                            }, 50);
                        }
                        return parentOpened;
                    },
                    send: function (event) {
                        connector.signal("send", event);
                    },
                    localSend: function (event) {
                        connector.signal("localSend", jQuery.stringifyJSON({
                            id: guid,
                            event: event
                        }));
                    },
                    close: function () {
                        // Do not signal the parent if this method is executed by the unload event handler
                        if (!_abortingConnection) {
                            clearInterval(_traceTimer);
                            connector.signal("close");
                            connector.close();
                        }
                    }
                };
            }

            function share() {
                var storageService, name = "atmosphere-" + _request.url, servers = {
                    // Powered by the storage event and the localStorage
                    // http://www.w3.org/TR/webstorage/#event-storage
                    storage: function () {
                        if (!jQuery.atmosphere.supportStorage()) {
                            return;
                        }

                        var storage = window.localStorage;

                        return {
                            init: function () {
                                // Handles the storage event
                                jQuery(window).on("storage.socket", function (event) {
                                    event = event.originalEvent;
                                    // When a deletion, newValue initialized to null
                                    if (event.key === name && event.newValue) {
                                        listener(event.newValue);
                                    }
                                });
                            },
                            signal: function (type, data) {
                                storage.setItem(name, jQuery.stringifyJSON({
                                    target: "c",
                                    type: type,
                                    data: data
                                }));
                            },
                            get: function (key) {
                                return jQuery.parseJSON(storage.getItem(name + "-" + key));
                            },
                            set: function (key, value) {
                                storage.setItem(name + "-" + key, jQuery.stringifyJSON(value));
                            },
                            close: function () {
                                jQuery(window).off("storage.socket");
                                storage.removeItem(name);
                                storage.removeItem(name + "-opened");
                                storage.removeItem(name + "-children");
                            }

                        };
                    },
                    // Powered by the window.open method
                    // https://developer.mozilla.org/en/DOM/window.open
                    windowref: function () {
                        // Internet Explorer raises an invalid argument error
                        // when calling the window.open method with the name containing non-word characters
                        var neim = name.replace(/\W/g, ""), win = (jQuery('iframe[name="' + neim + '"]')[0] || jQuery(
                            '<iframe name="' + neim + '" />').hide().appendTo("body")[0]).contentWindow;

                        return {
                            init: function () {
                                // Callbacks from different windows
                                win.callbacks = [listener];
                                // In IE 8 and less, only string argument can be safely passed to the function in other window
                                win.fire = function (string) {
                                    var i;

                                    for (i = 0; i < win.callbacks.length; i++) {
                                        win.callbacks[i](string);
                                    }
                                };
                            },
                            signal: function (type, data) {
                                if (!win.closed && win.fire) {
                                    win.fire(jQuery.stringifyJSON({
                                        target: "c",
                                        type: type,
                                        data: data
                                    }));
                                }
                            },
                            get: function (key) {
                                return !win.closed ? win[key] : null;
                            },
                            set: function (key, value) {
                                if (!win.closed) {
                                    win[key] = value;
                                }
                            },
                            close: function () {
                            }
                        };
                    }
                };

                // Receives send and close command from the children
                function listener(string) {
                    var command = jQuery.parseJSON(string), data = command.data;

                    if (command.target === "p") {
                        switch (command.type) {
                            case "send":
                                _push(data);
                                break;
                            case "localSend":
                                _localMessage(data);
                                break;
                            case "close":
                                _close();
                                break;
                        }
                    }
                }

                _localSocketF = function propagateMessageEvent(context) {
                    storageService.signal("message", context);
                };

                function leaveTrace() {
                    document.cookie = _sharingKey + "=" +
                        // Opera's JSON implementation ignores a number whose a last digit of 0 strangely
                        // but has no problem with a number whose a last digit of 9 + 1
                    encodeURIComponent(jQuery.stringifyJSON({
                        ts: jQuery.now() + 1,
                        heir: (storageService.get("children") || [])[0]
                    })) + "; path=/";
                }

                // Chooses a storageService
                storageService = servers.storage() || servers.windowref();
                storageService.init();

                if (_canLog('debug')) {
                    jQuery.atmosphere.debug("Installed StorageService " + storageService);
                }

                // List of children sockets
                storageService.set("children", []);

                if (storageService.get("opened") != null && !storageService.get("opened")) {
                    // Flag indicating the parent socket is opened
                    storageService.set("opened", false);
                }
                // Leaves traces
                _sharingKey = encodeURIComponent(name);
                leaveTrace();
                _traceTimer = setInterval(leaveTrace, 1000);

                _storageService = storageService;
            }

            /**
             * @private
             */
            function _open(state, transport, request) {
                if (_request.shared && transport !== 'local') {
                    share();
                }

                if (_storageService != null) {
                    _storageService.set("opened", true);
                }

                request.close = function () {
                    _close();
                };

                if (_requestCount > 0 && state === 're-connecting') {
                    request.isReopen = true;
                    _tryingToReconnect(_response);
                } else if (_response.error == null) {
                    _response.request = request;
                    var prevState = _response.state;
                    _response.state = state;
                    var prevTransport = _response.transport;
                    _response.transport = transport;

                    var _body = _response.responseBody;
                    _invokeCallback();
                    _response.responseBody = _body;

                    _response.state = prevState;
                    _response.transport = prevTransport;
                }
            }

            /**
             * Execute request using jsonp transport.
             *
             * @param request {Object} request Request parameters, if undefined _request object will be used.
             * @private
             */
            function _jsonp(request) {
                // When CORS is enabled, make sure we force the proper transport.
                request.transport = "jsonp";

                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                var url = rq.url;
                if (rq.dispatchUrl != null) {
                    url += rq.dispatchUrl;
                }

                var data = rq.data;
                if (rq.attachHeadersAsQueryString) {
                    url = _attachHeaders(rq);
                    if (data !== '') {
                        url += "&X-Atmosphere-Post-Body=" + encodeURIComponent(data);
                    }
                    data = '';
                }

                _jqxhr = jQuery.ajax({
                    url: url,
                    type: rq.method,
                    dataType: "jsonp",
                    error: function (jqXHR, textStatus, errorThrown) {
                        _response.error = true;

                        if (rq.openId) {
                            clearTimeout(rq.openId);
                        }

                        if (rq.heartbeatTimer) {
                            clearTimeout(rq.heartbeatTimer);
                        }

                        if (rq.reconnect && _requestCount++ < rq.maxReconnectOnClose) {
                            _open('re-connecting', rq.transport, rq);
                            _reconnect(_jqxhr, rq, rq.reconnectInterval);
                            rq.openId = setTimeout(function () {
                                _triggerOpen(rq);
                            }, rq.reconnectInterval + 1000);
                        } else {
                            _onError(jqXHR.status, errorThrown);
                        }
                    },
                    jsonp: "jsonpTransport",
                    success: function (json) {
                        if (rq.reconnect) {
                            if (rq.maxRequest === -1 || rq.requestCount++ < rq.maxRequest) {
                                _readHeaders(_jqxhr, rq);

                                var msg = json.message;
                                if (msg != null && typeof msg !== 'string') {
                                    try {
                                        msg = jQuery.stringifyJSON(msg);
                                    } catch (err) {
                                        // The message was partial
                                    }
                                }

                                var skipCallbackInvocation = _trackMessageSize(msg, rq, _response);

                                if (!rq.executeCallbackBeforeReconnect) {
                                    _reconnect(_jqxhr, rq, rq.pollingInterval);
                                }

                                if (!skipCallbackInvocation) {
                                    _prepareCallback(_response.responseBody, "messageReceived", 200, rq.transport);
                                }

                                if (rq.executeCallbackBeforeReconnect) {
                                    _reconnect(_jqxhr, rq, rq.pollingInterval);
                                }
                                _timeout(rq);
                            } else {
                                jQuery.atmosphere.log(_request.logLevel, ["JSONP reconnect maximum try reached " + _request.requestCount]);
                                _onError(0, "maxRequest reached");
                            }
                        }
                    },
                    data: rq.data,
                    beforeSend: function (jqXHR) {
                        _doRequest(jqXHR, rq, false);
                    }
                });
            }

            /**
             * Execute request using ajax transport.
             *
             * @param request {Object} request Request parameters, if undefined _request object will be used.
             * @private
             */
            function _ajax(request) {
                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                var url = rq.url;
                if (rq.dispatchUrl != null) {
                    url += rq.dispatchUrl;
                }

                var data = rq.data;
                if (rq.attachHeadersAsQueryString) {
                    url = _attachHeaders(rq);
                    if (data !== '') {
                        url += "&X-Atmosphere-Post-Body=" + encodeURIComponent(data);
                    }
                    data = '';
                }

                var async = typeof (rq.async) !== 'undefined' ? rq.async : true;
                _jqxhr = jQuery.ajax({
                    url: url,
                    type: rq.method,
                    error: function (jqXHR, textStatus, errorThrown) {
                        _response.error = true;
                        if (jqXHR.status < 300) {
                            _reconnect(_jqxhr, rq);
                        } else {
                            _onError(jqXHR.status, errorThrown);
                        }
                    },
                    success: function (data, textStatus, jqXHR) {

                        if (rq.reconnect) {
                            if (rq.maxRequest === -1 || rq.requestCount++ < rq.maxRequest) {
                                if (!rq.executeCallbackBeforeReconnect) {
                                    _reconnect(_jqxhr, rq, rq.pollingInterval);
                                }
                                var skipCallbackInvocation = _trackMessageSize(data, rq, _response);
                                if (!skipCallbackInvocation) {
                                    _prepareCallback(_response.responseBody, "messageReceived", 200, rq.transport);
                                }

                                if (rq.executeCallbackBeforeReconnect) {
                                    _reconnect(_jqxhr, rq, rq.pollingInterval);
                                }
                            } else {
                                jQuery.atmosphere.log(_request.logLevel, ["AJAX reconnect maximum try reached " + _request.requestCount]);
                                _onError(0, "maxRequest reached");
                            }
                        }
                    },
                    beforeSend: function (jqXHR) {
                        _doRequest(jqXHR, rq, false);
                    },
                    crossDomain: rq.enableXDR,
                    async: async
                });
            }

            /**
             * Build websocket object.
             *
             * @param location {string} Web socket url.
             * @returns {websocket} Web socket object.
             * @private
             */
            function _getWebSocket(location) {
                if (_request.webSocketImpl != null) {
                    return _request.webSocketImpl;
                } else {
                    if (window.WebSocket) {
                        return new WebSocket(location);
                    } else {
                        return new MozWebSocket(location);
                    }
                }
            }

            /**
             * Build web socket url from request url.
             *
             * @return {string} Web socket url (start with "ws" or "wss" for secure web socket).
             * @private
             */
            function _buildWebSocketUrl() {
                var url = _attachHeaders(_request);

                return decodeURI(jQuery('<a href="' + url + '"/>')[0].href.replace(/^http/, "ws"));
            }

            /**
             * Build SSE url from request url.
             *
             * @return a url with Atmosphere's headers
             * @private
             */
            function _buildSSEUrl() {
                var url = _attachHeaders(_request);
                return url;
            }

            /**
             * Open SSE. <br>
             * Automatically use fallback transport if SSE can't be opened.
             *
             * @private
             */
            function _executeSSE(sseOpened) {

                _response.transport = "sse";

                var location = _buildSSEUrl(_request.url);

                if (_canLog('debug')) {
                    jQuery.atmosphere.debug("Invoking executeSSE");
                    jQuery.atmosphere.debug("Using URL: " + location);
                }

                if (sseOpened && !_request.reconnect) {
                    if (_sse != null) {
                        _clearState();
                    }
                    return;
                }

                try {
                    _sse = new EventSource(location, {
                        withCredentials: _request.withCredentials
                    });
                } catch (e) {
                    _onError(0, e);
                    _reconnectWithFallbackTransport("SSE failed. Downgrading to fallback transport and resending");
                    return;
                }

                if (_request.connectTimeout > 0) {
                    _request.id = setTimeout(function () {
                        if (!sseOpened) {
                            _clearState();
                        }
                    }, _request.connectTimeout);
                }

                _sse.onopen = function (event) {
                    _debug("websocket.onopen");
                    _timeout(_request);
                    if (_canLog('debug')) {
                        jQuery.atmosphere.debug("SSE successfully opened");
                    }

                    if (!_request.enableProtocol) {
                        if (!sseOpened) {
                            _open('opening', "sse", _request);
                        } else {
                            _open('re-opening', "sse", _request);
                        }
                    } else if (_request.isReopen) {
                        _request.isReopen = false;
                        _open('re-opening', _request.transport, _request);
                    }
                    sseOpened = true;

                    if (_request.method === 'POST') {
                        _response.state = "messageReceived";
                        _sse.send(_request.data);
                    }
                };

                _sse.onmessage = function (message) {
                    _debug("websocket.onmessage");
                    _timeout(_request);
                    if (!_request.enableXDR && message.origin !== window.location.protocol + "//" + window.location.host) {
                        jQuery.atmosphere.log(_request.logLevel, ["Origin was not " + window.location.protocol + "//" + window.location.host]);
                        return;
                    }

                    _response.state = 'messageReceived';
                    _response.status = 200;

                    message = message.data;
                    var skipCallbackInvocation = _trackMessageSize(message, _request, _response);
                    if (!skipCallbackInvocation) {
                        _invokeCallback();
                        _response.responseBody = '';
                        _response.messages = [];
                    }
                };

                _sse.onerror = function (message) {
                    _debug("websocket.onerror");
                    clearTimeout(_request.id);

                    if (_request.heartbeatTimer) {
                        clearTimeout(_request.heartbeatTimer);
                    }

                    if (_response.closedByClientTimeout) {
                        return;
                    }

                    _invokeClose(sseOpened);
                    _clearState();

                    if (_abortingConnection) {
                        jQuery.atmosphere.log(_request.logLevel, ["SSE closed normally"]);
                    } else if (!sseOpened) {
                        _reconnectWithFallbackTransport("SSE failed. Downgrading to fallback transport and resending");
                    } else if (_request.reconnect && (_response.transport === 'sse')) {
                        if (_requestCount++ < _request.maxReconnectOnClose) {
                            _open('re-connecting', _request.transport, _request);
                            if (_request.reconnectInterval > 0) {
                                _request.reconnectId = setTimeout(function () {
                                    _executeSSE(true);
                                }, _request.reconnectInterval);
                            } else {
                                _executeSSE(true);
                            }
                            _response.responseBody = "";
                            _response.messages = [];
                        } else {
                            jQuery.atmosphere.log(_request.logLevel, ["SSE reconnect maximum try reached " + _requestCount]);
                            _onError(0, "maxReconnectOnClose reached");
                        }
                    }
                };
            }

            /**
             * Open web socket. <br>
             * Automatically use fallback transport if web socket can't be opened.
             *
             * @private
             */
            function _executeWebSocket(webSocketOpened) {
                _response.transport = "websocket";

                var location = _buildWebSocketUrl(_request.url);
                if (_canLog('debug')) {
                    jQuery.atmosphere.debug("Invoking executeWebSocket");
                    jQuery.atmosphere.debug("Using URL: " + location);
                }

                if (webSocketOpened && !_request.reconnect) {
                    if (_websocket != null) {
                        _clearState();
                    }
                    return;
                }

                _websocket = _getWebSocket(location);
                if (_request.webSocketBinaryType != null) {
                    _websocket.binaryType = _request.webSocketBinaryType;
                }

                if (_request.connectTimeout > 0) {
                    _request.id = setTimeout(function () {
                        if (!webSocketOpened) {
                            var _message = {
                                code: 1002,
                                reason: "",
                                wasClean: false
                            };
                            _websocket.onclose(_message);
                            // Close it anyway
                            try {
                                _clearState();
                            } catch (e) {
                            }
                            return;
                        }

                    }, _request.connectTimeout);
                }

                _websocket.onopen = function (message) {
                    _debug("websocket.onopen");
                    _timeout(_request);
                    if (_canLog('debug')) {
                        jQuery.atmosphere.debug("Websocket successfully opened");
                    }
                    jQuery.atmosphere.offline = false;

                    var reopening = webSocketOpened;

                    if (_websocket != null) {
                        _websocket.canSendMessage = true;
                    }

                    if (!_request.enableProtocol) {
                        webSocketOpened = true;
                        if (reopening) {
                            _open('re-opening', "websocket", _request);
                        } else {
                            _open('opening', "websocket", _request);
                        }
                    }

                    if (_websocket != null) {
                        if (_request.method === 'POST') {
                            _response.state = "messageReceived";
                            _websocket.send(_request.data);
                        }
                    }
                };

                _websocket.onmessage = function (message) {
                    _debug("websocket.onmessage: " + message);
                    _timeout(_request);

                    // We only consider it opened if we get the handshake data
                    // https://github.com/Atmosphere/atmosphere-javascript/issues/74
                    if (_request.enableProtocol) {
                        webSocketOpened = true;
                    }

                    _response.state = 'messageReceived';
                    _response.status = 200;

                    message = message.data;
                    var isString = typeof (message) === 'string';
                    if (isString) {
                        var skipCallbackInvocation = _trackMessageSize(message, _request, _response);
                        if (!skipCallbackInvocation) {
                            _invokeCallback();
                            _response.responseBody = '';
                            _response.messages = [];
                        }
                    } else {
                        message = _handleProtocol(_request, message);
                        if (message === "")
                            return;

                        _response.responseBody = message;
                        _invokeCallback();
                        _response.responseBody = null;
                    }
                };

                _websocket.onerror = function (message) {
                    _debug("websocket.onerror");
                    clearTimeout(_request.id);

                    if (_request.heartbeatTimer) {
                        clearTimeout(_request.heartbeatTimer);
                    }
                };

                _websocket.onclose = function (message) {
                    _debug("websocket.onclose");
                    if (_response.state === 'closed')
                        return;
                    clearTimeout(_request.id);

                    var reason = message.reason;
                    if (reason === "") {
                        switch (message.code) {
                            case 1000:
                                reason = "Normal closure; the connection successfully completed whatever purpose for which it was created.";
                                break;
                            case 1001:
                                reason = "The endpoint is going away, either because of a server failure or because the "
                                + "browser is navigating away from the page that opened the connection.";
                                break;
                            case 1002:
                                reason = "The endpoint is terminating the connection due to a protocol error.";
                                break;
                            case 1003:
                                reason = "The connection is being terminated because the endpoint received data of a type it "
                                + "cannot accept (for example, a text-only endpoint received binary data).";
                                break;
                            case 1004:
                                reason = "The endpoint is terminating the connection because a data frame was received that is too large.";
                                break;
                            case 1005:
                                reason = "Unknown: no status code was provided even though one was expected.";
                                break;
                            case 1006:
                                reason = "Connection was closed abnormally (that is, with no close frame being sent).";
                                break;
                        }
                    }

                    if (_canLog('warn')) {
                        jQuery.atmosphere.warn("Websocket closed, reason: " + reason);
                        jQuery.atmosphere.warn("Websocket closed, wasClean: " + message.wasClean);
                    }

                    if (_response.closedByClientTimeout || jQuery.atmosphere.offline) {
                        if (_request.reconnectId) {
                            clearTimeout(_request.reconnectId);
                            delete _request.reconnectId;
                        }
                        return;
                    }

                    _invokeClose(webSocketOpened);

                    _response.state = 'closed';

                    if (_abortingConnection) {
                        jQuery.atmosphere.log(_request.logLevel, ["Websocket closed normally"]);
                    } else if (!webSocketOpened) {
                        _reconnectWithFallbackTransport("Websocket failed. Downgrading to Comet and resending");

                    } else if (_request.reconnect && _response.transport === 'websocket') {
                        _clearState();
                        if (_requestCount++ < _request.maxReconnectOnClose) {
                            _open('re-connecting', _request.transport, _request);
                            if (_request.reconnectInterval > 0) {
                                _request.reconnectId = setTimeout(function () {
                                    _response.responseBody = "";
                                    _response.messages = [];
                                    _executeWebSocket(true);
                                }, _request.reconnectInterval);
                            } else {
                                _response.responseBody = "";
                                _response.messages = [];
                                _executeWebSocket(true);
                            }
                        } else {
                            jQuery.atmosphere.log(_request.logLevel, ["Websocket reconnect maximum try reached " + _request.requestCount]);
                            if (_canLog('warn')) {
                                jQuery.atmosphere.warn("Websocket error, reason: " + message.reason);
                            }
                            _onError(0, "maxReconnectOnClose reached");
                        }
                    }
                };

                var ua = navigator.userAgent.toLowerCase();
                var isAndroid = ua.indexOf("android") > -1;
                if (isAndroid && _websocket.url === undefined) {
                    // Android 4.1 does not really support websockets and fails silently
                    _websocket.onclose({
                        reason: "Android 4.1 does not support websockets.",
                        wasClean: false
                    });
                }
            }

            function _handleProtocol(request, message) {
                var nMessage = message;
                if (request.transport === 'polling') return nMessage;

                if (request.enableProtocol && request.firstMessage && jQuery.trim(message).length !== 0) {
                    var pos = request.trackMessageLength ? 1 : 0;
                    var messages = message.split(request.messageDelimiter);

                    if (messages.length <= pos + 1) {
                        // Something went wrong, normally with IE or when a message is written before the
                        // handshake has been received.
                        return nMessage;
                    }

                    request.firstMessage = false;
                    request.uuid = jQuery.trim(messages[pos]);

                    if (messages.length <= pos + 2) {
                        jQuery.atmosphere.log('error', ["Protocol data not sent by the server. " +
                        "If you enable protocol on client side, be sure to install JavascriptProtocol interceptor on server side." +
                        "Also note that atmosphere-runtime 2.2+ should be used."]);
                    }

                    _heartbeatInterval = parseInt(jQuery.trim(messages[pos + 1]), 10);
                    _heartbeatPadding = messages[pos + 2];

                    b = false;
                    if (request.transport !== 'long-polling') {
                        _triggerOpen(request);
                    }
                    jQuery.atmosphere.uuid = request.uuid;
                    nMessage = "";

                    // We have trailing messages
                    pos = request.trackMessageLength ? 4 : 3;
                    if (messages.length > pos + 1) {
                        for (var i = pos; i < messages.length; i++) {
                            nMessage += messages[i];
                            if (i + 1 !== messages.length) {
                                nMessage += request.messageDelimiter;
                            }
                        }
                    }

                    if (request.ackInterval !== 0) {
                        setTimeout(function () {
                            _push("...ACK...");
                        }, request.ackInterval);
                    }
                } else if (request.enableProtocol && request.firstMessage && jQuery.browser.msie && +jQuery.browser.version.split(".")[0] < 10) {
                    // In case we are getting some junk from IE
                    jQuery.atmosphere.log(_request.logLevel, ["Receiving unexpected data from IE"]);
                } else {
                    _triggerOpen(request);
                }
                return nMessage;
            }

            function _timeout(_request) {
                clearTimeout(_request.id);
                if (_request.timeout > 0 && _request.transport !== 'polling') {
                    _request.id = setTimeout(function () {
                        _onClientTimeout(_request);
                        _disconnect();
                        _clearState();
                    }, _request.timeout);
                }
            }

            function _onClientTimeout(_request) {
                _response.closedByClientTimeout = true;
                _response.state = 'closedByClient';
                _response.responseBody = "";
                _response.status = 408;
                _response.messages = [];
                _invokeCallback();
            }

            function _onError(code, reason) {
                _clearState();
                clearTimeout(_request.id);
                _response.state = 'error';
                _response.reasonPhrase = reason;
                _response.responseBody = "";
                _response.status = code;
                _response.messages = [];
                _invokeCallback();
            }

            /**
             * Track received message and make sure callbacks/functions are only invoked when the complete message has been received.
             *
             * @param message
             * @param request
             * @param response
             */
            function _trackMessageSize(message, request, response) {
                message = _handleProtocol(request, message);
                if (message.length === 0)
                    return true;

                response.responseBody = message;

                if (request.trackMessageLength) {
                    // prepend partialMessage if any
                    message = response.partialMessage + message;

                    var messages = [];
                    var messageStart = message.indexOf(request.messageDelimiter);
                    if (messageStart != -1) {
                        while (messageStart !== -1) {
                            var str = message.substring(0, messageStart);
                            var messageLength = parseInt(str, 10);
                            if (isNaN(messageLength))
                                throw 'message length "' + str + '" is not a number';
                            messageStart += request.messageDelimiter.length;
                            if (messageStart + messageLength > message.length) {
                                // message not complete, so there is no trailing messageDelimiter
                                messageStart = -1;
                            } else {
                                // message complete, so add it
                                messages.push(message.substring(messageStart, messageStart + messageLength));
                                // remove consumed characters
                                message = message.substring(messageStart + messageLength, message.length);
                                messageStart = message.indexOf(request.messageDelimiter);
                            }
                        }

                        /* keep any remaining data */
                        response.partialMessage = message;

                        if (messages.length !== 0) {
                            response.responseBody = messages.join(request.messageDelimiter);
                            response.messages = messages;
                            return false;
                        } else {
                            response.responseBody = "";
                            response.messages = [];
                            return true;
                        }
                    }
                }
                response.responseBody = message;
                response.messages = [message];
                return false;
            }

            /**
             * Reconnect request with fallback transport. <br>
             * Used in case websocket can't be opened.
             *
             * @private
             */
            function _reconnectWithFallbackTransport(errorMessage) {
                jQuery.atmosphere.log(_request.logLevel, [errorMessage]);

                if (typeof (_request.onTransportFailure) !== 'undefined') {
                    _request.onTransportFailure(errorMessage, _request);
                } else if (typeof (jQuery.atmosphere.onTransportFailure) !== 'undefined') {
                    jQuery.atmosphere.onTransportFailure(errorMessage, _request);
                }

                _request.transport = _request.fallbackTransport;
                var reconnectInterval = _request.connectTimeout === -1 ? 0 : _request.connectTimeout;
                if (_request.reconnect && _request.transport !== 'none' || _request.transport == null) {
                    _request.method = _request.fallbackMethod;
                    _response.transport = _request.fallbackTransport;
                    _request.fallbackTransport = 'none';
                    if (reconnectInterval > 0) {
                        _request.reconnectId = setTimeout(function () {
                            _execute();
                        }, reconnectInterval);
                    } else {
                        _execute();
                    }
                } else {
                    _onError(500, "Unable to reconnect with fallback transport");
                }
            }

            /**
             * Get url from request and attach headers to it.
             *
             * @param request {Object} request Request parameters, if undefined _request object will be used.
             *
             * @returns {Object} Request object, if undefined, _request object will be used.
             * @private
             */
            function _attachHeaders(request, url) {
                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                if (url == null) {
                    url = rq.url;
                }

                // If not enabled
                if (!rq.attachHeadersAsQueryString)
                    return url;

                // If already added
                if (url.indexOf("X-Atmosphere-Framework") !== -1) {
                    return url;
                }

                url += (url.indexOf('?') !== -1) ? '&' : '?';
                url += "X-Atmosphere-tracking-id=" + rq.uuid;
                url += "&X-Atmosphere-Framework=" + jQuery.atmosphere.version;
                url += "&X-Atmosphere-Transport=" + rq.transport;

                if (rq.trackMessageLength) {
                    url += "&X-Atmosphere-TrackMessageSize=" + "true";
                }

                if (rq.heartbeat !== null && rq.heartbeat.server !== null) {
                    url += "&X-Heartbeat-Server=" + rq.heartbeat.server;
                }

                if (rq.contentType !== '') {
                    url += "&Content-Type=" + (rq.transport === 'websocket' ? rq.contentType : encodeURIComponent(rq.contentType));
                }

                if (rq.enableProtocol) {
                    url += "&X-atmo-protocol=true";
                }

                jQuery.each(rq.headers, function (name, value) {
                    var h = jQuery.isFunction(value) ? value.call(this, rq, request, _response) : value;
                    if (h != null) {
                        url += "&" + encodeURIComponent(name) + "=" + encodeURIComponent(h);
                    }
                });

                return url;
            }

            function _triggerOpen(rq) {
                if (!rq.isOpen) {
                    rq.isOpen = true;
                    _open('opening', rq.transport, rq);
                } else if (rq.isReopen) {
                    rq.isReopen = false;
                    _open('re-opening', rq.transport, rq);
                } else {
                    return;
                }

                _startHeartbeat(rq);
            }

            function _startHeartbeat(rq) {
                if (rq.heartbeatTimer != null) {
                    clearTimeout(rq.heartbeatTimer);
                }

                if (!isNaN(_heartbeatInterval) && _heartbeatInterval > 0) {
                    var _pushHeartbeat = function () {
                        if (_canLog('debug')) {
                            jQuery.atmosphere.debug("Sending heartbeat");
                        }
                        _push(_heartbeatPadding);
                        rq.heartbeatTimer = setTimeout(_pushHeartbeat, _heartbeatInterval);
                    };
                    rq.heartbeatTimer = setTimeout(_pushHeartbeat, _heartbeatInterval);
                }
            }

            /**
             * Execute ajax request. <br>
             *
             * @param request {Object} request Request parameters, if undefined _request object will be used.
             * @private
             */
            function _executeRequest(request) {
                var rq = _request;
                if ((request != null) || (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                rq.lastIndex = 0;
                rq.readyState = 0;

                // CORS fake using JSONP
                if ((rq.transport === 'jsonp') || ((rq.enableXDR) && (jQuery.atmosphere.checkCORSSupport()))) {
                    _jsonp(rq);
                    return;
                }

                if (rq.transport === 'ajax') {
                    _ajax(request);
                    return;
                }

                if (jQuery.browser.msie && +jQuery.browser.version.split(".")[0] < 10) {
                    if ((rq.transport === 'streaming')) {
                        if (rq.enableXDR && window.XDomainRequest) {
                            _ieXDR(rq);
                        } else {
                            _ieStreaming(rq);
                        }
                        return;
                    }

                    if ((rq.enableXDR) && (window.XDomainRequest)) {
                        _ieXDR(rq);
                        return;
                    }
                }

                var reconnectFExec = function (force) {
                    rq.lastIndex = 0;
                    if (force || (rq.reconnect && _requestCount++ < rq.maxReconnectOnClose)) {
                        _response.ffTryingReconnect = true;
                        _open('re-connecting', request.transport, request);
                        _reconnect(ajaxRequest, rq, request.reconnectInterval);
                    } else {
                        _onError(0, "maxReconnectOnClose reached");
                    }
                };

                var reconnectF = function (force){
                    if(jQuery.atmosphere._beforeUnloadState){
                        // ATMOSPHERE-JAVASCRIPT-143: Delay reconnect to avoid reconnect attempts before an actual unload (we don't know if an unload will happen, yet)
                        jQuery.atmosphere.debug(new Date() + " Atmosphere: reconnectF: execution delayed due to _beforeUnloadState flag");
                        setTimeout(function () {
                            reconnectFExec(force);
                        }, 5000);
                    }else {
                        reconnectFExec(force);
                    }
                };

                var disconnected = function () {
                    // Prevent onerror callback to be called
                    _response.errorHandled = true;
                    _clearState();
                    reconnectF(false);
                };

                if (rq.reconnect && (rq.maxRequest === -1 || rq.requestCount++ < rq.maxRequest)) {
                    var ajaxRequest = jQuery.ajaxSettings.xhr();
                    ajaxRequest.hasData = false;

                    _doRequest(ajaxRequest, rq, true);

                    if (rq.suspend) {
                        _activeRequest = ajaxRequest;
                    }

                    if (rq.transport !== 'polling') {
                        _response.transport = rq.transport;

                        ajaxRequest.onabort = function () {
                            _debug("ajaxrequest.onabort");
                            _invokeClose(true);
                        };

                        ajaxRequest.onerror = function () {
                            _debug("ajaxrequest.onerror");
                            _response.error = true;
                            _response.ffTryingReconnect = true;
                            try {
                                _response.status = XMLHttpRequest.status;
                            } catch (e) {
                                _response.status = 500;
                            }

                            if (!_response.status) {
                                _response.status = 500;
                            }
                            if (!_response.errorHandled) {
                                _clearState();
                                reconnectF(false);
                            }
                        };
                    }

                    ajaxRequest.onreadystatechange = function () {
                        _debug("ajaxRequest.onreadystatechange, new state: " + ajaxRequest.readyState);
                        if (_abortingConnection) {
                            return;
                        }

                        _response.error = null;
                        var skipCallbackInvocation = false;
                        var update = false;

                        if (rq.transport === 'streaming' && rq.readyState > 2 && ajaxRequest.readyState === 4) {
                            _clearState();
                            reconnectF(false);
                            return;
                        }

                        rq.readyState = ajaxRequest.readyState;

                        if (rq.transport === 'streaming' && ajaxRequest.readyState >= 3) {
                            update = true;
                        } else if (rq.transport === 'long-polling' && ajaxRequest.readyState === 4) {
                            update = true;
                        }
                        _timeout(_request);

                        if (rq.transport !== 'polling') {
                            // MSIE 9 and lower status can be higher than 1000, Chrome can be 0
                            var status = 200;
                            if (ajaxRequest.readyState === 4) {
                                status = ajaxRequest.status > 1000 ? 0 : ajaxRequest.status;
                            }

                            if (!rq.reconnectOnServerError && (status >= 300 && status < 600)) {
                                _onError(status, ajaxRequest.statusText);
                                return;
                            }

                            if (status >= 300 || status === 0) {
                                disconnected();
                                return;
                            }

                            // Firefox incorrectly send statechange 0->2 when a reconnect attempt fails. The above checks ensure that onopen is not called for these
                            if ((!rq.enableProtocol || !request.firstMessage) && ajaxRequest.readyState === 2) {
                                // Firefox incorrectly send statechange 0->2 when a reconnect attempt fails. The above checks ensure that onopen is not called for these
                                // In that case, ajaxRequest.onerror will be called just after onreadystatechange is called, so we delay the trigger until we are
                                // guarantee the connection is well established.
                                if (jQuery.browser.mozilla && _response.ffTryingReconnect) {
                                    _response.ffTryingReconnect = false;
                                    setTimeout(function () {
                                        if (!_response.ffTryingReconnect) {
                                            _triggerOpen(rq);
                                        }
                                    }, 500);
                                } else {
                                    _triggerOpen(rq);
                                }
                            }
                        } else if (ajaxRequest.readyState === 4) {
                            update = true;
                        }

                        if (update) {
                            var responseText = ajaxRequest.responseText;

                            if (rq.transport === 'long-polling' && jQuery.trim(responseText).length === 0) {
                                // For browser that aren't support onabort
                                if (!ajaxRequest.hasData) {
                                    reconnectF(true);
                                } else {
                                    ajaxRequest.hasData = false;
                                }
                                return;
                            }
                            ajaxRequest.hasData = true;

                            _readHeaders(ajaxRequest, _request);

                            if (rq.transport === 'streaming') {
                                if (!jQuery.browser.opera) {
                                    var message = responseText.substring(rq.lastIndex, responseText.length);
                                    skipCallbackInvocation = _trackMessageSize(message, rq, _response);

                                    rq.lastIndex = responseText.length;
                                    if (skipCallbackInvocation) {
                                        return;
                                    }
                                } else {
                                    jQuery.atmosphere.iterate(function () {
                                        if (_response.status !== 500 && ajaxRequest.responseText.length > rq.lastIndex) {
                                            try {
                                                _response.status = ajaxRequest.status;
                                                _response.headers = parseHeaders(ajaxRequest.getAllResponseHeaders());

                                                _readHeaders(ajaxRequest, _request);

                                            } catch (e) {
                                                _response.status = 404;
                                            }
                                            _timeout(_request);

                                            _response.state = "messageReceived";
                                            var message = ajaxRequest.responseText.substring(rq.lastIndex);
                                            rq.lastIndex = ajaxRequest.responseText.length;

                                            skipCallbackInvocation = _trackMessageSize(message, rq, _response);
                                            if (!skipCallbackInvocation) {
                                                _invokeCallback();
                                            }

                                            if (_verifyStreamingLength(ajaxRequest, rq)) {
                                                _reconnectOnMaxStreamingLength(ajaxRequest, rq);
                                                return;
                                            }
                                        } else if (_response.status > 400) {
                                            // Prevent replaying the last message.
                                            rq.lastIndex = ajaxRequest.responseText.length;
                                            return false;
                                        }
                                    }, 0);
                                }
                            } else {
                                skipCallbackInvocation = _trackMessageSize(responseText, rq, _response);
                            }
                            var closeStream = _verifyStreamingLength(ajaxRequest, rq);

                            try {
                                _response.status = ajaxRequest.status;
                                _response.headers = parseHeaders(ajaxRequest.getAllResponseHeaders());

                                _readHeaders(ajaxRequest, rq);
                            } catch (e) {
                                _response.status = 404;
                            }

                            if (rq.suspend) {
                                _response.state = _response.status === 0 ? "closed" : "messageReceived";
                            } else {
                                _response.state = "messagePublished";
                            }

                            var isAllowedToReconnect = !closeStream && request.transport !== 'streaming' && request.transport !== 'polling';
                            ;
                            if (isAllowedToReconnect && !rq.executeCallbackBeforeReconnect) {
                                _reconnect(ajaxRequest, rq, rq.pollingInterval);
                            }

                            if (_response.responseBody.length !== 0 && !skipCallbackInvocation)
                                _invokeCallback();

                            if (isAllowedToReconnect && rq.executeCallbackBeforeReconnect) {
                                _reconnect(ajaxRequest, rq, rq.pollingInterval);
                            }

                            if (closeStream) {
                                _reconnectOnMaxStreamingLength(ajaxRequest, rq);
                            }
                        }
                    };

                    ajaxRequest.send(rq.data);
                    _subscribed = true;
                } else {
                    if (rq.logLevel === 'debug') {
                        jQuery.atmosphere.log(rq.logLevel, ["Max re-connection reached."]);
                    }
                    _onError(0, "maxRequest reached");
                }
            }

            function _reconnectOnMaxStreamingLength(ajaxRequest, rq) {
                _response.messages = [];
                rq.isReopen = true;
                _close();
                _abortingConnection = false;
                _reconnect(ajaxRequest, rq, 500);
            }

            /**
             * Do ajax request.
             *
             * @param ajaxRequest Ajax request.
             * @param request Request parameters.
             * @param create If ajax request has to be open.
             */
            function _doRequest(ajaxRequest, request, create) {
                // Prevent Android to cache request
                var url = request.url;
                if (request.dispatchUrl != null && request.method === 'POST') {
                    url += request.dispatchUrl;
                }
                url = _attachHeaders(request, url);
                url = jQuery.atmosphere.prepareURL(url);

                if (create) {
                    ajaxRequest.open(request.method, url, true);
                    if (request.connectTimeout > 0) {
                        request.id = setTimeout(function () {
                            if (request.requestCount === 0) {
                                _clearState();
                                _prepareCallback("Connect timeout", "closed", 200, request.transport);
                            }
                        }, request.connectTimeout);
                    }
                }

                if (_request.withCredentials && _request.transport !== 'websocket') {
                    if ("withCredentials" in ajaxRequest) {
                        ajaxRequest.withCredentials = true;
                    }
                }

                if (!_request.dropHeaders) {
                    ajaxRequest.setRequestHeader("X-Atmosphere-Framework", jQuery.atmosphere.version);
                    ajaxRequest.setRequestHeader("X-Atmosphere-Transport", request.transport);

                    if (request.heartbeat !== null && request.heartbeat.server !== null) {
                        ajaxRequest.setRequestHeader("X-Heartbeat-Server", ajaxRequest.heartbeat.server);
                    }

                    if (request.trackMessageLength) {
                        ajaxRequest.setRequestHeader("X-Atmosphere-TrackMessageSize", "true");
                    }
                    ajaxRequest.setRequestHeader("X-Atmosphere-tracking-id", request.uuid);

                    jQuery.each(request.headers, function (name, value) {
                        var h = jQuery.isFunction(value) ? value.call(this, ajaxRequest, request, create, _response) : value;
                        if (h != null) {
                            ajaxRequest.setRequestHeader(name, h);
                        }
                    });
                }

                if (request.contentType !== '') {
                    ajaxRequest.setRequestHeader("Content-Type", request.contentType);
                }
            }

            function _reconnect(ajaxRequest, request, reconnectInterval) {

                if (_response.closedByClientTimeout) {
                    return;
                }

                if (request.reconnect || (request.suspend && _subscribed)) {
                    var status = 0;
                    if (ajaxRequest.readyState > 1) {
                        status = ajaxRequest.status > 1000 ? 0 : ajaxRequest.status;
                    }
                    _response.status = status === 0 ? 204 : status;
                    _response.reason = status === 0 ? "Server resumed the connection or down." : "OK";

                    // Reconnect immediately
                    clearTimeout(request.id);
                    if (request.reconnectId) {
                        clearTimeout(request.reconnectId);
                        delete request.reconnectId;
                    }

                    if (reconnectInterval > 0) {
                        setTimeout(function () {
                            _request.reconnectId = _executeRequest(request);
                        }, reconnectInterval);
                    } else {
                        _executeRequest(request);
                    }
                }
            }

            function _tryingToReconnect(response) {
                response.state = 're-connecting';
                _invokeFunction(response);
            }

            // From jquery-stream, which is APL2 licensed as well.
            function _ieXDR(request) {
                if (request.transport !== "polling") {
                    _ieStream = _configureXDR(request);
                    _ieStream.open();
                } else {
                    _configureXDR(request).open();
                }
            }

            function _configureXDR(request) {
                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                var transport = rq.transport;
                var lastIndex = 0;
                var xdr = new window.XDomainRequest();

                var reconnect = function () {
                    if (rq.transport === "long-polling" && (rq.reconnect && (rq.maxRequest === -1 || rq.requestCount++ < rq.maxRequest))) {
                        xdr.status = 200;
                        _open('re-connecting', request.transport, request);
                        _ieXDR(rq);
                    }
                };

                var rewriteURL = rq.rewriteURL || function (url) {
                        // Maintaining session by rewriting URL
                        // http://stackoverflow.com/questions/6453779/maintaining-session-by-rewriting-url
                        var match = /(?:^|;\s*)(JSESSIONID|PHPSESSID)=([^;]*)/.exec(document.cookie);

                        switch (match && match[1]) {
                            case "JSESSIONID":
                                return url.replace(/;jsessionid=[^\?]*|(\?)|$/, ";jsessionid=" + match[2] + "$1");
                            case "PHPSESSID":
                                return url.replace(/\?PHPSESSID=[^&]*&?|\?|$/, "?PHPSESSID=" + match[2] + "&").replace(/&$/, "");
                        }
                        return url;
                    };

                // Handles open and message event
                xdr.onprogress = function () {
                    handle(xdr);
                };
                // Handles error event
                xdr.onerror = function () {
                    // If the server doesn't send anything back to XDR will fail with polling
                    if (rq.transport !== 'polling') {
                        _clearState();
                        if (_requestCount++ < rq.maxReconnectOnClose) {
                            if (rq.reconnectInterval > 0) {
                                rq.reconnectId = setTimeout(function () {
                                    _open('re-connecting', request.transport, request);
                                    _ieXDR(rq);
                                }, rq.reconnectInterval);
                            } else {
                                _open('re-connecting', request.transport, request);
                                _ieXDR(rq);
                            }
                        } else {
                            _onError(0, "maxReconnectOnClose reached");
                        }
                    }
                };

                // Handles close event
                xdr.onload = function () {
                };

                var handle = function (xdr) {
                    clearTimeout(rq.id);
                    var message = xdr.responseText;

                    message = message.substring(lastIndex);
                    lastIndex += message.length;

                    if (transport !== 'polling') {
                        _timeout(rq);

                        var skipCallbackInvocation = _trackMessageSize(message, rq, _response);

                        if (transport === 'long-polling' && jQuery.trim(message).length === 0)
                            return;

                        if (rq.executeCallbackBeforeReconnect) {
                            reconnect();
                        }

                        if (!skipCallbackInvocation) {
                            _prepareCallback(_response.responseBody, "messageReceived", 200, transport);
                        }

                        if (!rq.executeCallbackBeforeReconnect) {
                            reconnect();
                        }
                    }
                };

                return {
                    open: function () {
                        var url = rq.url;
                        if (rq.dispatchUrl != null) {
                            url += rq.dispatchUrl;
                        }
                        url = _attachHeaders(rq, url);
                        xdr.open(rq.method, rewriteURL(url));
                        if (rq.method === 'GET') {
                            xdr.send();
                        } else {
                            xdr.send(rq.data);
                        }

                        if (rq.connectTimeout > 0) {
                            rq.id = setTimeout(function () {
                                if (rq.requestCount === 0) {
                                    _clearState();
                                    _prepareCallback("Connect timeout", "closed", 200, rq.transport);
                                }
                            }, rq.connectTimeout);
                        }
                    },
                    close: function () {
                        xdr.abort();
                    }
                };
            }

            function _ieStreaming(request) {
                _ieStream = _configureIE(request);
                _ieStream.open();
            }

            function _configureIE(request) {
                var rq = _request;
                if ((request != null) && (typeof (request) !== 'undefined')) {
                    rq = request;
                }

                var stop;
                var doc = new window.ActiveXObject("htmlfile");

                doc.open();
                doc.close();

                var url = rq.url;
                if (rq.dispatchUrl != null) {
                    url += rq.dispatchUrl;
                }

                if (rq.transport !== 'polling') {
                    _response.transport = rq.transport;
                }

                return {
                    open: function () {
                        var iframe = doc.createElement("iframe");

                        url = _attachHeaders(rq);
                        if (rq.data !== '') {
                            url += "&X-Atmosphere-Post-Body=" + encodeURIComponent(rq.data);
                        }

                        // Finally attach a timestamp to prevent Android and IE caching.
                        url = jQuery.atmosphere.prepareURL(url);

                        iframe.src = url;
                        doc.body.appendChild(iframe);

                        // For the server to respond in a consistent format regardless of user agent, we polls response text
                        var cdoc = iframe.contentDocument || iframe.contentWindow.document;

                        stop = jQuery.atmosphere.iterate(function () {
                            try {
                                if (!cdoc.firstChild) {
                                    return;
                                }

                                // Detects connection failure
                                if (cdoc.readyState === "complete") {
                                    try {
                                        jQuery.noop(cdoc.fileSize);
                                    } catch (e) {
                                        _prepareCallback("Connection Failure", "error", 500, rq.transport);
                                        return false;
                                    }
                                }

                                var res = cdoc.body ? cdoc.body.lastChild : cdoc;
                                var readResponse = function () {
                                    // Clones the element not to disturb the original one
                                    var clone = res.cloneNode(true);

                                    // If the last character is a carriage return or a line feed, IE ignores it in the innerText property
                                    // therefore, we add another non-newline character to preserve it
                                    clone.appendChild(cdoc.createTextNode("."));

                                    var text = clone.innerText;

                                    text = text.substring(0, text.length - 1);
                                    return text;

                                };

                                // To support text/html content type
                                if (!jQuery.nodeName(res, "pre")) {
                                    // Injects a plaintext element which renders text without interpreting the HTML and cannot be stopped
                                    // it is deprecated in HTML5, but still works
                                    var head = cdoc.head || cdoc.getElementsByTagName("head")[0] || cdoc.documentElement || cdoc;
                                    var script = cdoc.createElement("script");

                                    script.text = "document.write('<plaintext>')";

                                    head.insertBefore(script, head.firstChild);
                                    head.removeChild(script);

                                    // The plaintext element will be the response container
                                    res = cdoc.body.lastChild;
                                }

                                if (rq.closed) {
                                    rq.isReopen = true;
                                }

                                // Handles message and close event
                                stop = jQuery.atmosphere.iterate(function () {
                                    var text = readResponse();
                                    if (text.length > rq.lastIndex) {
                                        _timeout(_request);

                                        _response.status = 200;
                                        _response.error = null;

                                        // Empties response every time that it is handled
                                        res.innerText = "";
                                        var skipCallbackInvocation = _trackMessageSize(text, rq, _response);
                                        if (skipCallbackInvocation) {
                                            return "";
                                        }

                                        _prepareCallback(_response.responseBody, "messageReceived", 200, rq.transport);
                                    }

                                    rq.lastIndex = 0;

                                    if (cdoc.readyState === "complete") {
                                        _invokeClose(true);
                                        _open('re-connecting', rq.transport, rq);
                                        if (rq.reconnectInterval > 0) {
                                            rq.reconnectId = setTimeout(function () {
                                                _ieStreaming(rq);
                                            }, rq.reconnectInterval);
                                        } else {
                                            _ieStreaming(rq);
                                        }
                                        return false;
                                    }
                                }, null);

                                return false;
                            } catch (err) {
                                _response.error = true;
                                _open('re-connecting', rq.transport, rq);
                                if (_requestCount++ < rq.maxReconnectOnClose) {
                                    if (rq.reconnectInterval > 0) {
                                        rq.reconnectId = setTimeout(function () {
                                            _ieStreaming(rq);
                                        }, rq.reconnectInterval);
                                    } else {
                                        _ieStreaming(rq);
                                    }
                                } else {
                                    _onError(0, "maxReconnectOnClose reached");
                                }
                                doc.execCommand("Stop");
                                doc.close();
                                return false;
                            }
                        });
                    },

                    close: function () {
                        if (stop) {
                            stop();
                        }

                        doc.execCommand("Stop");
                        _invokeClose(true);
                    }
                };
            }

            /**
             * Send message. <br>
             * Will be automatically dispatch to other connected.
             *
             * @param {Object, string} Message to send.
             * @private
             */
            function _push(message) {

                if (_localStorageService != null) {
                    _pushLocal(message);
                } else if (_activeRequest != null || _sse != null) {
                    _pushAjaxMessage(message);
                } else if (_ieStream != null) {
                    _pushIE(message);
                } else if (_jqxhr != null) {
                    _pushJsonp(message);
                } else if (_websocket != null) {
                    _pushWebSocket(message);
                } else {
                    _onError(0, "No suspended connection available");
                    jQuery.atmosphere.error("No suspended connection available. Make sure atmosphere.subscribe has been called and request.onOpen invoked before invoking trying to push data");
                }
            }

            function _pushLocal(message) {
                _localStorageService.send(message);
            }

            function _intraPush(message) {
                // IE 9 will crash if not.
                if (message.length === 0)
                    return;

                try {
                    if (_localStorageService) {
                        _localStorageService.localSend(message);
                    } else if (_storageService) {
                        _storageService.signal("localMessage", jQuery.stringifyJSON({
                            id: guid,
                            event: message
                        }));
                    }
                } catch (err) {
                    jQuery.atmosphere.error(err);
                }
            }

            /**
             * Send a message using currently opened ajax request (using http-streaming or long-polling). <br>
             *
             * @param {string, Object} Message to send. This is an object, string message is saved in data member.
             * @private
             */
            function _pushAjaxMessage(message) {
                var rq = _getPushRequest(message);
                _executeRequest(rq);
            }

            /**
             * Send a message using currently opened ie streaming (using http-streaming or long-polling). <br>
             *
             * @param {string, Object} Message to send. This is an object, string message is saved in data member.
             * @private
             */
            function _pushIE(message) {
                if (_request.enableXDR && jQuery.atmosphere.checkCORSSupport()) {
                    var rq = _getPushRequest(message);
                    // Do not reconnect since we are pushing.
                    rq.reconnect = false;
                    _jsonp(rq);
                } else {
                    _pushAjaxMessage(message);
                }
            }

            /**
             * Send a message using jsonp transport. <br>
             *
             * @param {string, Object} Message to send. This is an object, string message is saved in data member.
             * @private
             */
            function _pushJsonp(message) {
                _pushAjaxMessage(message);
            }

            function _getStringMessage(message) {
                var msg = message;
                if (typeof (msg) === 'object') {
                    msg = message.data;
                }
                return msg;
            }

            /**
             * Build request use to push message using method 'POST' <br>. Transport is defined as 'polling' and 'suspend' is set to false.
             *
             * @return {Object} Request object use to push message.
             * @private
             */
            function _getPushRequest(message) {
                var msg = _getStringMessage(message);

                var rq = {
                    connected: false,
                    timeout: 60000,
                    method: 'POST',
                    url: _request.url,
                    contentType: _request.contentType,
                    headers: _request.headers,
                    reconnect: true,
                    callback: null,
                    data: msg,
                    suspend: false,
                    maxRequest: -1,
                    logLevel: 'info',
                    requestCount: 0,
                    withCredentials: _request.withCredentials,
                    transport: 'polling',
                    isOpen: true,
                    attachHeadersAsQueryString: true,
                    enableXDR: _request.enableXDR,
                    uuid: _request.uuid,
                    dispatchUrl: _request.dispatchUrl,
                    enableProtocol: false,
                    messageDelimiter: '|',
                    trackMessageLength: _request.trackMessageLength,
                    maxReconnectOnClose: _request.maxReconnectOnClose,
                    heartbeatTimer: _request.heartbeatTimer,
                    heartbeat: _request.heartbeat
                };

                if (typeof (message) === 'object') {
                    rq = jQuery.extend(rq, message);
                }

                return rq;
            }

            /**
             * Send a message using currently opened websocket. <br>
             *
             */
            function _pushWebSocket(message) {
                var msg = jQuery.atmosphere.isBinary(message) ? message : _getStringMessage(message);
                var data;
                try {
                    if (_request.dispatchUrl != null) {
                        data = _request.webSocketPathDelimiter + _request.dispatchUrl + _request.webSocketPathDelimiter + msg;
                    } else {
                        data = msg;
                    }

                    if (!_websocket.canSendMessage) {
                        jQuery.atmosphere.error("WebSocket not connected.");
                        return;
                    }

                    _websocket.send(data);

                } catch (e) {
                    _websocket.onclose = function (message) {
                    };
                    _clearState();

                    _reconnectWithFallbackTransport("Websocket failed. Downgrading to Comet and resending " + message);
                    _pushAjaxMessage(message);
                }
            }

            function _localMessage(message) {
                var m = jQuery.parseJSON(message);
                if (m.id !== guid) {
                    if (typeof (_request.onLocalMessage) !== 'undefined') {
                        _request.onLocalMessage(m.event);
                    } else if (typeof (jQuery.atmosphere.onLocalMessage) !== 'undefined') {
                        jQuery.atmosphere.onLocalMessage(m.event);
                    }
                }
            }

            function _prepareCallback(messageBody, state, errorCode, transport) {

                _response.responseBody = messageBody;
                _response.transport = transport;
                _response.status = errorCode;
                _response.state = state;

                _invokeCallback();
            }

            function _readHeaders(xdr, request) {
                if (!request.readResponsesHeaders) {
                    if (!request.enableProtocol) {
                        request.uuid = jQuery.atmosphere.guid();
                    }
                }
                else {
                    try {
                        var tempUUID = xdr.getResponseHeader('X-Atmosphere-tracking-id');
                        if (tempUUID && tempUUID != null) {
                            request.uuid = tempUUID.split(" ").pop();
                        }
                    } catch (e) {
                    }
                }
            }

            function _invokeFunction(response) {
                _f(response, _request);
                // Global
                _f(response, jQuery.atmosphere);
            }

            function _f(response, f) {
                switch (response.state) {
                    case "messageReceived":
                        _debug("Firing onMessage");
                        _requestCount = 0;
                        if (typeof (f.onMessage) !== 'undefined')
                            f.onMessage(response);
                        break;
                    case "error":
                        _debug("Firing onError");
                        if (typeof (f.onError) !== 'undefined')
                            f.onError(response);
                        break;
                    case "opening":
                        _debug("Firing onOpen");
                        delete _request.closed;
                        if (typeof (f.onOpen) !== 'undefined')
                            f.onOpen(response);
                        break;
                    case "messagePublished":
                        _debug("Firing onMessagePublished");
                        if (typeof (f.onMessagePublished) !== 'undefined')
                            f.onMessagePublished(response);
                        break;
                    case "re-connecting":
                        _debug("Firing onReconnect");
                        if (typeof (f.onReconnect) !== 'undefined')
                            f.onReconnect(_request, response);
                        break;
                    case "closedByClient":
                        _debug("Firing onClientTimeout");
                        if (typeof (f.onClientTimeout) !== 'undefined')
                            f.onClientTimeout(_request);
                        break;
                    case "re-opening":
                        _debug("Firing onReopen");
                        delete _request.closed;
                        if (typeof (f.onReopen) !== 'undefined')
                            f.onReopen(_request, response);
                        break;
                    case "fail-to-reconnect":
                        _debug("Firing onFailureToReconnect");
                        if (typeof (f.onFailureToReconnect) !== 'undefined')
                            f.onFailureToReconnect(_request, response);
                        break;
                    case "unsubscribe":
                    case "closed":
                        var closed = typeof (_request.closed) !== 'undefined' ? _request.closed : false;
                        if (!closed) {
                            _debug("Firing onClose");
                            if (typeof (f.onClose) !== 'undefined') {
                                f.onClose(response);
                            }
                        } else {
                            _debug("Closed but not firing onClose");
                        }

                        _request.closed = true;
                        break;
                }
            }

            function _invokeClose(wasOpen) {
                if (_response.state !== 'closed') {
                    _response.state = 'closed';
                    _response.responseBody = "";
                    _response.messages = [];
                    _response.status = !wasOpen ? 501 : 200;
                    _invokeCallback();
                }
            }

            /**
             * Invoke request callbacks.
             *
             * @private
             */
            function _invokeCallback() {
                var call = function (index, func) {
                    func(_response);
                };

                if (_localStorageService == null && _localSocketF != null) {
                    _localSocketF(_response.responseBody);
                }

                _request.reconnect = _request.mrequest;

                var isString = typeof (_response.responseBody) === 'string';
                var messages = (isString && _request.trackMessageLength) ? (_response.messages.length > 0 ? _response.messages : ['']) : new Array(
                    _response.responseBody);
                for (var i = 0; i < messages.length; i++) {

                    if (messages.length > 1 && messages[i].length === 0) {
                        continue;
                    }
                    _response.responseBody = (isString) ? jQuery.trim(messages[i]) : messages[i];

                    if (_localStorageService == null && _localSocketF != null) {
                        _localSocketF(_response.responseBody);
                    }

                    if ((_response.responseBody.length === 0 ||
                        (isString && _heartbeatPadding === _response.responseBody)) && _response.state === "messageReceived") {
                        continue;
                    }

                    _invokeFunction(_response);

                    // Invoke global callbacks
                    if (jQuery.atmosphere.callbacks.length > 0) {
                        if (_canLog('debug')) {
                            jQuery.atmosphere.debug("Invoking " + jQuery.atmosphere.callbacks.length + " global callbacks: " + _response.state);
                        }
                        try {
                            jQuery.each(jQuery.atmosphere.callbacks, call);
                        } catch (e) {
                            jQuery.atmosphere.log(_request.logLevel, ["Callback exception" + e]);
                        }
                    }

                    // Invoke request callback
                    if (typeof (_request.callback) === 'function') {
                        if (_canLog('debug')) {
                            jQuery.atmosphere.debug("Invoking request callbacks");
                        }
                        try {
                            _request.callback(_response);
                        } catch (e) {
                            jQuery.atmosphere.log(_request.logLevel, ["Callback exception" + e]);
                        }
                    }
                }

            }

            /**
             *
             * @private
             */
            function _verifyStreamingLength(ajaxRequest, rq) {
                // Wait to be sure we have the full message before closing.
                if (_response.partialMessage === "" && (rq.transport === 'streaming') && (ajaxRequest.responseText.length > rq.maxStreamingLength)) {
                    return true;
                }
                return false;
            }

            /**
             * Disconnect
             *
             * @private
             */
            function _disconnect() {
                if (_request.enableProtocol && !_request.firstMessage) {
                    var query = "X-Atmosphere-Transport=close&X-Atmosphere-tracking-id=" + _request.uuid;

                    jQuery.each(_request.headers, function (name, value) {
                        var h = jQuery.isFunction(value) ? value.call(this, _request, _request, _response) : value;
                        if (h != null) {
                            query += "&" + encodeURIComponent(name) + "=" + encodeURIComponent(h);
                        }
                    });

                    var url = _request.url.replace(/([?&])_=[^&]*/, query);
                    url = url + (url === _request.url ? (/\?/.test(_request.url) ? "&" : "?") + query : "");

                    if (_request.connectTimeout > 0) {
                        jQuery.ajax({
                            url: url,
                            async: _request.closeAsync,
                            timeout: _request.connectTimeout,
                            cache: false,
                            crossDomain: _request.enableXDR,
                            mimeType: "text/plain"
                        });
                    } else {
                        jQuery.ajax({
                            url: url,
                            async: _request.closeAsync,
                            cache: false,
                            crossDomain: _request.enableXDR,
                            mimeType: "text/plain"
                        });
                    }
                }
            }

            /**
             * Close request.
             *
             * @private
             */
            function _close() {
                if (_request.reconnectId) {
                    clearTimeout(_request.reconnectId);
                    delete _request.reconnectId;
                }

                if (_request.heartbeatTimer) {
                    clearTimeout(_request.heartbeatTimer);
                }

                _request.reconnect = false;
                _abortingConnection = true;
                _response.request = _request;
                _response.state = 'unsubscribe';
                _response.responseBody = "";
                _response.status = 408;
                _invokeCallback();
                _disconnect();
                _clearState();
            }

            function _clearState() {
                _response.partialMessage = "";
                if (_request.id) {
                    clearTimeout(_request.id);
                }

                if (_request.heartbeatTimer) {
                    clearTimeout(_request.heartbeatTimer);
                }

                // https://github.com/Atmosphere/atmosphere/issues/1860#issuecomment-74707226
                if(_request.reconnectId) {
                    clearTimeout(_request.reconnectId);
                    delete _request.reconnectId;
                }

                if (_ieStream != null) {
                    _ieStream.close();
                    _ieStream = null;
                }
                if (_jqxhr != null) {
                    _jqxhr.abort();
                    _jqxhr = null;
                }
                if (_activeRequest != null) {
                    _activeRequest.abort();
                    _activeRequest = null;
                }
                if (_websocket != null) {
                    if (_websocket.canSendMessage) {
                        _websocket.close();
                    }
                    _websocket = null;
                }
                if (_sse != null) {
                    _sse.close();
                    _sse = null;
                }

                _clearStorage();
            }

            function _clearStorage() {
                // Stop sharing a connection
                if (_storageService != null) {
                    // Clears trace timer
                    clearInterval(_traceTimer);
                    // Removes the trace
                    document.cookie = _sharingKey + "=; expires=Thu, 01 Jan 1970 00:00:00 GMT; path=/";
                    // The heir is the parent unless unloading
                    _storageService.signal("close", {
                        reason: "",
                        heir: !_abortingConnection ? guid : (_storageService.get("children") || [])[0]
                    });
                    _storageService.close();
                }
                if (_localStorageService != null) {
                    _localStorageService.close();
                }
            }

            this.subscribe = function (options) {
                _subscribe(options);
                _execute();
            };

            this.execute = function () {
                _execute();
            };

            this.invokeCallback = function () {
                _invokeCallback();
            };

            this.close = function () {
                _close();
            };

            this.disconnect = function () {
                _disconnect();
            };

            this.getUrl = function () {
                return _request.url;
            };

            this.push = function (message, dispatchUrl) {
                if (dispatchUrl != null) {
                    var originalDispatchUrl = _request.dispatchUrl;
                    _request.dispatchUrl = dispatchUrl;
                    _push(message);
                    _request.dispatchUrl = originalDispatchUrl;
                } else {
                    _push(message);
                }
            };

            this.getUUID = function () {
                return _request.uuid;
            };

            this.pushLocal = function (message) {
                _intraPush(message);
            };

            this.enableProtocol = function (message) {
                return _request.enableProtocol;
            };

            this.init = function () {
                _init();
            };

            this.request = _request;
            this.response = _response;
        },

        subscribe: function (url, callback, request) {
            if (typeof (callback) === 'function') {
                jQuery.atmosphere.addCallback(callback);
            }

            if (typeof (url) !== "string") {
                request = url;
            } else {
                request.url = url;
            }

            // https://github.com/Atmosphere/atmosphere-javascript/issues/58
            jQuery.atmosphere.uuid = ((typeof (request) !== 'undefined') && typeof (request.uuid) !== 'undefined') ? request.uuid : 0;

            var rq = new jQuery.atmosphere.AtmosphereRequest(request);
            rq.execute();

            jQuery.atmosphere.requests[jQuery.atmosphere.requests.length] = rq;
            return rq;
        },

        addCallback: function (func) {
            if (jQuery.inArray(func, jQuery.atmosphere.callbacks) === -1) {
                jQuery.atmosphere.callbacks.push(func);
            }
        },

        removeCallback: function (func) {
            var index = jQuery.inArray(func, jQuery.atmosphere.callbacks);
            if (index !== -1) {
                jQuery.atmosphere.callbacks.splice(index, 1);
            }
        },

        unsubscribe: function () {
            if (jQuery.atmosphere.requests.length > 0) {
                var requestsClone = [].concat(jQuery.atmosphere.requests);
                for (var i = 0; i < requestsClone.length; i++) {
                    var rq = requestsClone[i];
                    rq.close();
                    clearTimeout(rq.response.request.id);

                    if (rq.heartbeatTimer) {
                        clearTimeout(rq.heartbeatTimer);
                    }
                }
            }
            jQuery.atmosphere.requests = [];
            jQuery.atmosphere.callbacks = [];
        },

        unsubscribeUrl: function (url) {
            var idx = -1;
            if (jQuery.atmosphere.requests.length > 0) {
                for (var i = 0; i < jQuery.atmosphere.requests.length; i++) {
                    var rq = jQuery.atmosphere.requests[i];

                    // Suppose you can subscribe once to an url
                    if (rq.getUrl() === url) {
                        rq.close();
                        clearTimeout(rq.response.request.id);

                        if (rq.heartbeatTimer) {
                            clearTimeout(rq.heartbeatTimer);
                        }

                        idx = i;
                        break;
                    }
                }
            }
            if (idx >= 0) {
                jQuery.atmosphere.requests.splice(idx, 1);
            }
        },

        publish: function (request) {
            if (typeof (request.callback) === 'function') {
                jQuery.atmosphere.addCallback(request.callback);
            }
            request.transport = "polling";

            var rq = new jQuery.atmosphere.AtmosphereRequest(request);
            jQuery.atmosphere.requests[jQuery.atmosphere.requests.length] = rq;
            return rq;
        },

        checkCORSSupport: function () {
            if (jQuery.browser.msie && !window.XDomainRequest && +jQuery.browser.version.split(".")[0] < 11) {
                return true;
            } else if (jQuery.browser.opera && +jQuery.browser.version.split(".")[0] < 12.0) {
                return true;
            }

            // KreaTV 4.1 -> 4.4
            else if (jQuery.trim(navigator.userAgent).slice(0, 16) === "KreaTVWebKit/531") {
                return true;
            }
            // KreaTV 3.8
            else if (jQuery.trim(navigator.userAgent).slice(-7).toLowerCase() === "kreatel") {
                return true;
            }

            // Force Android to use CORS as some version like 2.2.3 fail otherwise
            var ua = navigator.userAgent.toLowerCase();
            var isAndroid = ua.indexOf("android") > -1;
            if (isAndroid) {
                return true;
            }
            return false;
        },

        S4: function () {
            return (((1 + Math.random()) * 0x10000) | 0).toString(16).substring(1);
        },

        guid: function () {
            return (jQuery.atmosphere.S4() + jQuery.atmosphere.S4() + "-" + jQuery.atmosphere.S4() + "-" + jQuery.atmosphere.S4() + "-"
            + jQuery.atmosphere.S4() + "-" + jQuery.atmosphere.S4() + jQuery.atmosphere.S4() + jQuery.atmosphere.S4());
        },

        // From jQuery-Stream
        prepareURL: function (url) {
            // Attaches a time stamp to prevent caching
            var ts = jQuery.now();
            var ret = url.replace(/([?&])_=[^&]*/, "$1_=" + ts);

            return ret + (ret === url ? (/\?/.test(url) ? "&" : "?") + "_=" + ts : "");
        },

        // From jQuery-Stream
        param: function (data) {
            return jQuery.param(data, jQuery.ajaxSettings.traditional);
        },

        supportStorage: function () {
            var storage = window.localStorage;
            if (storage) {
                try {
                    storage.setItem("t", "t");
                    storage.removeItem("t");
                    // The storage event of Internet Explorer and Firefox 3 works strangely
                    return window.StorageEvent && !jQuery.browser.msie && !(jQuery.browser.mozilla && jQuery.browser.version.split(".")[0] === "1");
                } catch (e) {
                }
            }

            return false;
        },

        iterate: function (fn, interval) {
            var timeoutId;

            // Though the interval is 0 for real-time application, there is a delay between setTimeout calls
            // For detail, see https://developer.mozilla.org/en/window.setTimeout#Minimum_delay_and_timeout_nesting
            interval = interval || 0;

            (function loop() {
                timeoutId = setTimeout(function () {
                    if (fn() === false) {
                        return;
                    }

                    loop();
                }, interval);
            })();

            return function () {
                clearTimeout(timeoutId);
            };
        },

        log: function (level, args) {
            if (window.console) {
                var logger = window.console[level];
                if (typeof logger === 'function') {
                    logger.apply(window.console, args);
                }
            }
        },

        warn: function () {
            jQuery.atmosphere.log('warn', arguments);
        },

        info: function () {
            jQuery.atmosphere.log('info', arguments);
        },

        debug: function () {
            jQuery.atmosphere.log('debug', arguments);
        },

        error: function () {
            jQuery.atmosphere.log('error', arguments);
        },

        // TODO extract to utils or something
        isBinary: function (data) {
            // True if data is an instance of Blob, ArrayBuffer or ArrayBufferView 
            return /^\[object\s(?:Blob|ArrayBuffer|.+Array)\]$/.test(Object.prototype.toString.call(data));
        }
    };


    // http://stackoverflow.com/questions/9645803/whats-the-replacement-for-browser
    // Limit scope pollution from any deprecated API
    (function () {

        var matched, browser;

        // Use of jQuery.browser is frowned upon.
        // More details: http://api.jquery.com/jQuery.browser
        // jQuery.uaMatch maintained for back-compat
        jQuery.uaMatch = function (ua) {
            ua = ua.toLowerCase();

            var match = /(chrome)[ \/]([\w.]+)/.exec(ua) ||
                /(opera)(?:.*version|)[ \/]([\w.]+)/.exec(ua) ||
                /(msie) ([\w.]+)/.exec(ua) ||
                /(trident)(?:.*? rv:([\w.]+)|)/.exec(ua) ||
                ua.indexOf("android") < 0 && /version\/(.+) (safari)/.exec(ua) ||
                ua.indexOf("compatible") < 0 && /(mozilla)(?:.*? rv:([\w.]+)|)/.exec(ua) ||
                [];

            // Swaps variables
            if (match[2] === "safari") {
                match[2] = match[1];
                match[1] = "safari";
            }
            return {
                browser: match[1] || "",
                version: match[2] || "0"
            };
        };

        matched = jQuery.uaMatch(navigator.userAgent);
        browser = {};

        if (matched.browser) {
            browser[matched.browser] = true;
            browser.version = matched.version;
            browser.vmajor = browser.version.split(".")[0];
        }

        // Trident is the layout engine of the Internet Explorer
        // IE 11 has no "MSIE: 11.0" token
        if (browser.trident) {
            browser.msie = true;
        }

        jQuery.browser = browser;

        jQuery.sub = function () {
            function jQuerySub(selector, context) {
                return new jQuerySub.fn.init(selector, context);
            }

            jQuery.extend(true, jQuerySub, this);
            jQuerySub.superclass = this;
            jQuerySub.fn = jQuerySub.prototype = this();
            jQuerySub.fn.constructor = jQuerySub;
            jQuerySub.sub = this.sub;
            jQuerySub.fn.init = function init(selector, context) {
                if (context && context instanceof jQuery && !(context instanceof jQuerySub)) {
                    context = jQuerySub(context);
                }

                return jQuery.fn.init.call(this, selector, context, rootjQuerySub);
            };
            jQuerySub.fn.init.prototype = jQuerySub.fn;
            var rootjQuerySub = jQuerySub(document);
            return jQuerySub;
        };

    })();

    /*
     * jQuery stringifyJSON
     * http://github.com/flowersinthesand/jquery-stringifyJSON
     *
     * Copyright 2011, Donghwan Kim
     * Licensed under the Apache License, Version 2.0
     * http://www.apache.org/licenses/LICENSE-2.0
     */
    // This plugin is heavily based on Douglas Crockford's reference implementation
    (function (jQuery) {

        var escapable = /[\\\"\x00-\x1f\x7f-\x9f\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g, meta = {
            '\b': '\\b',
            '\t': '\\t',
            '\n': '\\n',
            '\f': '\\f',
            '\r': '\\r',
            '"': '\\"',
            '\\': '\\\\'
        };

        function quote(string) {
            return '"' + string.replace(escapable, function (a) {
                    var c = meta[a];
                    return typeof c === "string" ? c : "\\u" + ("0000" + a.charCodeAt(0).toString(16)).slice(-4);
                }) + '"';
        }

        function f(n) {
            return n < 10 ? "0" + n : n;
        }

        function str(key, holder) {
            var i, v, len, partial, value = holder[key], type = typeof value;

            if (value && typeof value === "object" && typeof value.toJSON === "function") {
                value = value.toJSON(key);
                type = typeof value;
            }

            switch (type) {
                case "string":
                    return quote(value);
                case "number":
                    return isFinite(value) ? String(value) : "null";
                case "boolean":
                    return String(value);
                case "object":
                    if (!value) {
                        return "null";
                    }

                    switch (Object.prototype.toString.call(value)) {
                        case "[object Date]":
                            return isFinite(value.valueOf()) ? '"' + value.getUTCFullYear() + "-" + f(value.getUTCMonth() + 1) + "-" + f(value.getUTCDate())
                            + "T" + f(value.getUTCHours()) + ":" + f(value.getUTCMinutes()) + ":" + f(value.getUTCSeconds()) + "Z" + '"' : "null";
                        case "[object Array]":
                            len = value.length;
                            partial = [];
                            for (i = 0; i < len; i++) {
                                partial.push(str(i, value) || "null");
                            }

                            return "[" + partial.join(",") + "]";
                        default:
                            partial = [];
                            for (i in value) {
                                if (Object.prototype.hasOwnProperty.call(value, i)) {
                                    v = str(i, value);
                                    if (v) {
                                        partial.push(quote(i) + ":" + v);
                                    }
                                }
                            }

                            return "{" + partial.join(",") + "}";
                    }
            }
        }

        jQuery.stringifyJSON = function (value) {
            if (window.JSON && window.JSON.stringify) {
                return window.JSON.stringify(value);
            }

            return str("", {
                "": value
            });
        };

    }(jQuery));
}));
/* jshint noarg:true, noempty:true, eqeqeq:true, evil:true, laxbreak:true, undef:true, browser:true, jquery:true, indent:false, maxerr:50, eqnull:true */
;

// resource: com.jqueryui:effect.js
/*!
 * jQuery UI Effects 1.11.4
 * http://jqueryui.com
 *
 * Copyright jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * http://api.jqueryui.com/category/effects-core/
 */
(function( factory ) {
	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [ "jquery" ], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
}(function( $ ) {

var dataSpace = "ui-effects-",

	// Create a local jQuery because jQuery Color relies on it and the
	// global may not exist with AMD and a custom build (#10199)
	jQuery = $;

$.effects = {
	effect: {}
};

/*!
 * jQuery Color Animations v2.1.2
 * https://github.com/jquery/jquery-color
 *
 * Copyright 2014 jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * Date: Wed Jan 16 08:47:09 2013 -0600
 */
(function( jQuery, undefined ) {

	var stepHooks = "backgroundColor borderBottomColor borderLeftColor borderRightColor borderTopColor color columnRuleColor outlineColor textDecorationColor textEmphasisColor",

	// plusequals test for += 100 -= 100
	rplusequals = /^([\-+])=\s*(\d+\.?\d*)/,
	// a set of RE's that can match strings and generate color tuples.
	stringParsers = [ {
			re: /rgba?\(\s*(\d{1,3})\s*,\s*(\d{1,3})\s*,\s*(\d{1,3})\s*(?:,\s*(\d?(?:\.\d+)?)\s*)?\)/,
			parse: function( execResult ) {
				return [
					execResult[ 1 ],
					execResult[ 2 ],
					execResult[ 3 ],
					execResult[ 4 ]
				];
			}
		}, {
			re: /rgba?\(\s*(\d+(?:\.\d+)?)\%\s*,\s*(\d+(?:\.\d+)?)\%\s*,\s*(\d+(?:\.\d+)?)\%\s*(?:,\s*(\d?(?:\.\d+)?)\s*)?\)/,
			parse: function( execResult ) {
				return [
					execResult[ 1 ] * 2.55,
					execResult[ 2 ] * 2.55,
					execResult[ 3 ] * 2.55,
					execResult[ 4 ]
				];
			}
		}, {
			// this regex ignores A-F because it's compared against an already lowercased string
			re: /#([a-f0-9]{2})([a-f0-9]{2})([a-f0-9]{2})/,
			parse: function( execResult ) {
				return [
					parseInt( execResult[ 1 ], 16 ),
					parseInt( execResult[ 2 ], 16 ),
					parseInt( execResult[ 3 ], 16 )
				];
			}
		}, {
			// this regex ignores A-F because it's compared against an already lowercased string
			re: /#([a-f0-9])([a-f0-9])([a-f0-9])/,
			parse: function( execResult ) {
				return [
					parseInt( execResult[ 1 ] + execResult[ 1 ], 16 ),
					parseInt( execResult[ 2 ] + execResult[ 2 ], 16 ),
					parseInt( execResult[ 3 ] + execResult[ 3 ], 16 )
				];
			}
		}, {
			re: /hsla?\(\s*(\d+(?:\.\d+)?)\s*,\s*(\d+(?:\.\d+)?)\%\s*,\s*(\d+(?:\.\d+)?)\%\s*(?:,\s*(\d?(?:\.\d+)?)\s*)?\)/,
			space: "hsla",
			parse: function( execResult ) {
				return [
					execResult[ 1 ],
					execResult[ 2 ] / 100,
					execResult[ 3 ] / 100,
					execResult[ 4 ]
				];
			}
		} ],

	// jQuery.Color( )
	color = jQuery.Color = function( color, green, blue, alpha ) {
		return new jQuery.Color.fn.parse( color, green, blue, alpha );
	},
	spaces = {
		rgba: {
			props: {
				red: {
					idx: 0,
					type: "byte"
				},
				green: {
					idx: 1,
					type: "byte"
				},
				blue: {
					idx: 2,
					type: "byte"
				}
			}
		},

		hsla: {
			props: {
				hue: {
					idx: 0,
					type: "degrees"
				},
				saturation: {
					idx: 1,
					type: "percent"
				},
				lightness: {
					idx: 2,
					type: "percent"
				}
			}
		}
	},
	propTypes = {
		"byte": {
			floor: true,
			max: 255
		},
		"percent": {
			max: 1
		},
		"degrees": {
			mod: 360,
			floor: true
		}
	},
	support = color.support = {},

	// element for support tests
	supportElem = jQuery( "<p>" )[ 0 ],

	// colors = jQuery.Color.names
	colors,

	// local aliases of functions called often
	each = jQuery.each;

// determine rgba support immediately
supportElem.style.cssText = "background-color:rgba(1,1,1,.5)";
support.rgba = supportElem.style.backgroundColor.indexOf( "rgba" ) > -1;

// define cache name and alpha properties
// for rgba and hsla spaces
each( spaces, function( spaceName, space ) {
	space.cache = "_" + spaceName;
	space.props.alpha = {
		idx: 3,
		type: "percent",
		def: 1
	};
});

function clamp( value, prop, allowEmpty ) {
	var type = propTypes[ prop.type ] || {};

	if ( value == null ) {
		return (allowEmpty || !prop.def) ? null : prop.def;
	}

	// ~~ is an short way of doing floor for positive numbers
	value = type.floor ? ~~value : parseFloat( value );

	// IE will pass in empty strings as value for alpha,
	// which will hit this case
	if ( isNaN( value ) ) {
		return prop.def;
	}

	if ( type.mod ) {
		// we add mod before modding to make sure that negatives values
		// get converted properly: -10 -> 350
		return (value + type.mod) % type.mod;
	}

	// for now all property types without mod have min and max
	return 0 > value ? 0 : type.max < value ? type.max : value;
}

function stringParse( string ) {
	var inst = color(),
		rgba = inst._rgba = [];

	string = string.toLowerCase();

	each( stringParsers, function( i, parser ) {
		var parsed,
			match = parser.re.exec( string ),
			values = match && parser.parse( match ),
			spaceName = parser.space || "rgba";

		if ( values ) {
			parsed = inst[ spaceName ]( values );

			// if this was an rgba parse the assignment might happen twice
			// oh well....
			inst[ spaces[ spaceName ].cache ] = parsed[ spaces[ spaceName ].cache ];
			rgba = inst._rgba = parsed._rgba;

			// exit each( stringParsers ) here because we matched
			return false;
		}
	});

	// Found a stringParser that handled it
	if ( rgba.length ) {

		// if this came from a parsed string, force "transparent" when alpha is 0
		// chrome, (and maybe others) return "transparent" as rgba(0,0,0,0)
		if ( rgba.join() === "0,0,0,0" ) {
			jQuery.extend( rgba, colors.transparent );
		}
		return inst;
	}

	// named colors
	return colors[ string ];
}

color.fn = jQuery.extend( color.prototype, {
	parse: function( red, green, blue, alpha ) {
		if ( red === undefined ) {
			this._rgba = [ null, null, null, null ];
			return this;
		}
		if ( red.jquery || red.nodeType ) {
			red = jQuery( red ).css( green );
			green = undefined;
		}

		var inst = this,
			type = jQuery.type( red ),
			rgba = this._rgba = [];

		// more than 1 argument specified - assume ( red, green, blue, alpha )
		if ( green !== undefined ) {
			red = [ red, green, blue, alpha ];
			type = "array";
		}

		if ( type === "string" ) {
			return this.parse( stringParse( red ) || colors._default );
		}

		if ( type === "array" ) {
			each( spaces.rgba.props, function( key, prop ) {
				rgba[ prop.idx ] = clamp( red[ prop.idx ], prop );
			});
			return this;
		}

		if ( type === "object" ) {
			if ( red instanceof color ) {
				each( spaces, function( spaceName, space ) {
					if ( red[ space.cache ] ) {
						inst[ space.cache ] = red[ space.cache ].slice();
					}
				});
			} else {
				each( spaces, function( spaceName, space ) {
					var cache = space.cache;
					each( space.props, function( key, prop ) {

						// if the cache doesn't exist, and we know how to convert
						if ( !inst[ cache ] && space.to ) {

							// if the value was null, we don't need to copy it
							// if the key was alpha, we don't need to copy it either
							if ( key === "alpha" || red[ key ] == null ) {
								return;
							}
							inst[ cache ] = space.to( inst._rgba );
						}

						// this is the only case where we allow nulls for ALL properties.
						// call clamp with alwaysAllowEmpty
						inst[ cache ][ prop.idx ] = clamp( red[ key ], prop, true );
					});

					// everything defined but alpha?
					if ( inst[ cache ] && jQuery.inArray( null, inst[ cache ].slice( 0, 3 ) ) < 0 ) {
						// use the default of 1
						inst[ cache ][ 3 ] = 1;
						if ( space.from ) {
							inst._rgba = space.from( inst[ cache ] );
						}
					}
				});
			}
			return this;
		}
	},
	is: function( compare ) {
		var is = color( compare ),
			same = true,
			inst = this;

		each( spaces, function( _, space ) {
			var localCache,
				isCache = is[ space.cache ];
			if (isCache) {
				localCache = inst[ space.cache ] || space.to && space.to( inst._rgba ) || [];
				each( space.props, function( _, prop ) {
					if ( isCache[ prop.idx ] != null ) {
						same = ( isCache[ prop.idx ] === localCache[ prop.idx ] );
						return same;
					}
				});
			}
			return same;
		});
		return same;
	},
	_space: function() {
		var used = [],
			inst = this;
		each( spaces, function( spaceName, space ) {
			if ( inst[ space.cache ] ) {
				used.push( spaceName );
			}
		});
		return used.pop();
	},
	transition: function( other, distance ) {
		var end = color( other ),
			spaceName = end._space(),
			space = spaces[ spaceName ],
			startColor = this.alpha() === 0 ? color( "transparent" ) : this,
			start = startColor[ space.cache ] || space.to( startColor._rgba ),
			result = start.slice();

		end = end[ space.cache ];
		each( space.props, function( key, prop ) {
			var index = prop.idx,
				startValue = start[ index ],
				endValue = end[ index ],
				type = propTypes[ prop.type ] || {};

			// if null, don't override start value
			if ( endValue === null ) {
				return;
			}
			// if null - use end
			if ( startValue === null ) {
				result[ index ] = endValue;
			} else {
				if ( type.mod ) {
					if ( endValue - startValue > type.mod / 2 ) {
						startValue += type.mod;
					} else if ( startValue - endValue > type.mod / 2 ) {
						startValue -= type.mod;
					}
				}
				result[ index ] = clamp( ( endValue - startValue ) * distance + startValue, prop );
			}
		});
		return this[ spaceName ]( result );
	},
	blend: function( opaque ) {
		// if we are already opaque - return ourself
		if ( this._rgba[ 3 ] === 1 ) {
			return this;
		}

		var rgb = this._rgba.slice(),
			a = rgb.pop(),
			blend = color( opaque )._rgba;

		return color( jQuery.map( rgb, function( v, i ) {
			return ( 1 - a ) * blend[ i ] + a * v;
		}));
	},
	toRgbaString: function() {
		var prefix = "rgba(",
			rgba = jQuery.map( this._rgba, function( v, i ) {
				return v == null ? ( i > 2 ? 1 : 0 ) : v;
			});

		if ( rgba[ 3 ] === 1 ) {
			rgba.pop();
			prefix = "rgb(";
		}

		return prefix + rgba.join() + ")";
	},
	toHslaString: function() {
		var prefix = "hsla(",
			hsla = jQuery.map( this.hsla(), function( v, i ) {
				if ( v == null ) {
					v = i > 2 ? 1 : 0;
				}

				// catch 1 and 2
				if ( i && i < 3 ) {
					v = Math.round( v * 100 ) + "%";
				}
				return v;
			});

		if ( hsla[ 3 ] === 1 ) {
			hsla.pop();
			prefix = "hsl(";
		}
		return prefix + hsla.join() + ")";
	},
	toHexString: function( includeAlpha ) {
		var rgba = this._rgba.slice(),
			alpha = rgba.pop();

		if ( includeAlpha ) {
			rgba.push( ~~( alpha * 255 ) );
		}

		return "#" + jQuery.map( rgba, function( v ) {

			// default to 0 when nulls exist
			v = ( v || 0 ).toString( 16 );
			return v.length === 1 ? "0" + v : v;
		}).join("");
	},
	toString: function() {
		return this._rgba[ 3 ] === 0 ? "transparent" : this.toRgbaString();
	}
});
color.fn.parse.prototype = color.fn;

// hsla conversions adapted from:
// https://code.google.com/p/maashaack/source/browse/packages/graphics/trunk/src/graphics/colors/HUE2RGB.as?r=5021

function hue2rgb( p, q, h ) {
	h = ( h + 1 ) % 1;
	if ( h * 6 < 1 ) {
		return p + ( q - p ) * h * 6;
	}
	if ( h * 2 < 1) {
		return q;
	}
	if ( h * 3 < 2 ) {
		return p + ( q - p ) * ( ( 2 / 3 ) - h ) * 6;
	}
	return p;
}

spaces.hsla.to = function( rgba ) {
	if ( rgba[ 0 ] == null || rgba[ 1 ] == null || rgba[ 2 ] == null ) {
		return [ null, null, null, rgba[ 3 ] ];
	}
	var r = rgba[ 0 ] / 255,
		g = rgba[ 1 ] / 255,
		b = rgba[ 2 ] / 255,
		a = rgba[ 3 ],
		max = Math.max( r, g, b ),
		min = Math.min( r, g, b ),
		diff = max - min,
		add = max + min,
		l = add * 0.5,
		h, s;

	if ( min === max ) {
		h = 0;
	} else if ( r === max ) {
		h = ( 60 * ( g - b ) / diff ) + 360;
	} else if ( g === max ) {
		h = ( 60 * ( b - r ) / diff ) + 120;
	} else {
		h = ( 60 * ( r - g ) / diff ) + 240;
	}

	// chroma (diff) == 0 means greyscale which, by definition, saturation = 0%
	// otherwise, saturation is based on the ratio of chroma (diff) to lightness (add)
	if ( diff === 0 ) {
		s = 0;
	} else if ( l <= 0.5 ) {
		s = diff / add;
	} else {
		s = diff / ( 2 - add );
	}
	return [ Math.round(h) % 360, s, l, a == null ? 1 : a ];
};

spaces.hsla.from = function( hsla ) {
	if ( hsla[ 0 ] == null || hsla[ 1 ] == null || hsla[ 2 ] == null ) {
		return [ null, null, null, hsla[ 3 ] ];
	}
	var h = hsla[ 0 ] / 360,
		s = hsla[ 1 ],
		l = hsla[ 2 ],
		a = hsla[ 3 ],
		q = l <= 0.5 ? l * ( 1 + s ) : l + s - l * s,
		p = 2 * l - q;

	return [
		Math.round( hue2rgb( p, q, h + ( 1 / 3 ) ) * 255 ),
		Math.round( hue2rgb( p, q, h ) * 255 ),
		Math.round( hue2rgb( p, q, h - ( 1 / 3 ) ) * 255 ),
		a
	];
};

each( spaces, function( spaceName, space ) {
	var props = space.props,
		cache = space.cache,
		to = space.to,
		from = space.from;

	// makes rgba() and hsla()
	color.fn[ spaceName ] = function( value ) {

		// generate a cache for this space if it doesn't exist
		if ( to && !this[ cache ] ) {
			this[ cache ] = to( this._rgba );
		}
		if ( value === undefined ) {
			return this[ cache ].slice();
		}

		var ret,
			type = jQuery.type( value ),
			arr = ( type === "array" || type === "object" ) ? value : arguments,
			local = this[ cache ].slice();

		each( props, function( key, prop ) {
			var val = arr[ type === "object" ? key : prop.idx ];
			if ( val == null ) {
				val = local[ prop.idx ];
			}
			local[ prop.idx ] = clamp( val, prop );
		});

		if ( from ) {
			ret = color( from( local ) );
			ret[ cache ] = local;
			return ret;
		} else {
			return color( local );
		}
	};

	// makes red() green() blue() alpha() hue() saturation() lightness()
	each( props, function( key, prop ) {
		// alpha is included in more than one space
		if ( color.fn[ key ] ) {
			return;
		}
		color.fn[ key ] = function( value ) {
			var vtype = jQuery.type( value ),
				fn = ( key === "alpha" ? ( this._hsla ? "hsla" : "rgba" ) : spaceName ),
				local = this[ fn ](),
				cur = local[ prop.idx ],
				match;

			if ( vtype === "undefined" ) {
				return cur;
			}

			if ( vtype === "function" ) {
				value = value.call( this, cur );
				vtype = jQuery.type( value );
			}
			if ( value == null && prop.empty ) {
				return this;
			}
			if ( vtype === "string" ) {
				match = rplusequals.exec( value );
				if ( match ) {
					value = cur + parseFloat( match[ 2 ] ) * ( match[ 1 ] === "+" ? 1 : -1 );
				}
			}
			local[ prop.idx ] = value;
			return this[ fn ]( local );
		};
	});
});

// add cssHook and .fx.step function for each named hook.
// accept a space separated string of properties
color.hook = function( hook ) {
	var hooks = hook.split( " " );
	each( hooks, function( i, hook ) {
		jQuery.cssHooks[ hook ] = {
			set: function( elem, value ) {
				var parsed, curElem,
					backgroundColor = "";

				if ( value !== "transparent" && ( jQuery.type( value ) !== "string" || ( parsed = stringParse( value ) ) ) ) {
					value = color( parsed || value );
					if ( !support.rgba && value._rgba[ 3 ] !== 1 ) {
						curElem = hook === "backgroundColor" ? elem.parentNode : elem;
						while (
							(backgroundColor === "" || backgroundColor === "transparent") &&
							curElem && curElem.style
						) {
							try {
								backgroundColor = jQuery.css( curElem, "backgroundColor" );
								curElem = curElem.parentNode;
							} catch ( e ) {
							}
						}

						value = value.blend( backgroundColor && backgroundColor !== "transparent" ?
							backgroundColor :
							"_default" );
					}

					value = value.toRgbaString();
				}
				try {
					elem.style[ hook ] = value;
				} catch ( e ) {
					// wrapped to prevent IE from throwing errors on "invalid" values like 'auto' or 'inherit'
				}
			}
		};
		jQuery.fx.step[ hook ] = function( fx ) {
			if ( !fx.colorInit ) {
				fx.start = color( fx.elem, hook );
				fx.end = color( fx.end );
				fx.colorInit = true;
			}
			jQuery.cssHooks[ hook ].set( fx.elem, fx.start.transition( fx.end, fx.pos ) );
		};
	});

};

color.hook( stepHooks );

jQuery.cssHooks.borderColor = {
	expand: function( value ) {
		var expanded = {};

		each( [ "Top", "Right", "Bottom", "Left" ], function( i, part ) {
			expanded[ "border" + part + "Color" ] = value;
		});
		return expanded;
	}
};

// Basic color names only.
// Usage of any of the other color names requires adding yourself or including
// jquery.color.svg-names.js.
colors = jQuery.Color.names = {
	// 4.1. Basic color keywords
	aqua: "#00ffff",
	black: "#000000",
	blue: "#0000ff",
	fuchsia: "#ff00ff",
	gray: "#808080",
	green: "#008000",
	lime: "#00ff00",
	maroon: "#800000",
	navy: "#000080",
	olive: "#808000",
	purple: "#800080",
	red: "#ff0000",
	silver: "#c0c0c0",
	teal: "#008080",
	white: "#ffffff",
	yellow: "#ffff00",

	// 4.2.3. "transparent" color keyword
	transparent: [ null, null, null, 0 ],

	_default: "#ffffff"
};

})( jQuery );

/******************************************************************************/
/****************************** CLASS ANIMATIONS ******************************/
/******************************************************************************/
(function() {

var classAnimationActions = [ "add", "remove", "toggle" ],
	shorthandStyles = {
		border: 1,
		borderBottom: 1,
		borderColor: 1,
		borderLeft: 1,
		borderRight: 1,
		borderTop: 1,
		borderWidth: 1,
		margin: 1,
		padding: 1
	};

$.each([ "borderLeftStyle", "borderRightStyle", "borderBottomStyle", "borderTopStyle" ], function( _, prop ) {
	$.fx.step[ prop ] = function( fx ) {
		if ( fx.end !== "none" && !fx.setAttr || fx.pos === 1 && !fx.setAttr ) {
			jQuery.style( fx.elem, prop, fx.end );
			fx.setAttr = true;
		}
	};
});

function getElementStyles( elem ) {
	var key, len,
		style = elem.ownerDocument.defaultView ?
			elem.ownerDocument.defaultView.getComputedStyle( elem, null ) :
			elem.currentStyle,
		styles = {};

	if ( style && style.length && style[ 0 ] && style[ style[ 0 ] ] ) {
		len = style.length;
		while ( len-- ) {
			key = style[ len ];
			if ( typeof style[ key ] === "string" ) {
				styles[ $.camelCase( key ) ] = style[ key ];
			}
		}
	// support: Opera, IE <9
	} else {
		for ( key in style ) {
			if ( typeof style[ key ] === "string" ) {
				styles[ key ] = style[ key ];
			}
		}
	}

	return styles;
}

function styleDifference( oldStyle, newStyle ) {
	var diff = {},
		name, value;

	for ( name in newStyle ) {
		value = newStyle[ name ];
		if ( oldStyle[ name ] !== value ) {
			if ( !shorthandStyles[ name ] ) {
				if ( $.fx.step[ name ] || !isNaN( parseFloat( value ) ) ) {
					diff[ name ] = value;
				}
			}
		}
	}

	return diff;
}

// support: jQuery <1.8
if ( !$.fn.addBack ) {
	$.fn.addBack = function( selector ) {
		return this.add( selector == null ?
			this.prevObject : this.prevObject.filter( selector )
		);
	};
}

$.effects.animateClass = function( value, duration, easing, callback ) {
	var o = $.speed( duration, easing, callback );

	return this.queue( function() {
		var animated = $( this ),
			baseClass = animated.attr( "class" ) || "",
			applyClassChange,
			allAnimations = o.children ? animated.find( "*" ).addBack() : animated;

		// map the animated objects to store the original styles.
		allAnimations = allAnimations.map(function() {
			var el = $( this );
			return {
				el: el,
				start: getElementStyles( this )
			};
		});

		// apply class change
		applyClassChange = function() {
			$.each( classAnimationActions, function(i, action) {
				if ( value[ action ] ) {
					animated[ action + "Class" ]( value[ action ] );
				}
			});
		};
		applyClassChange();

		// map all animated objects again - calculate new styles and diff
		allAnimations = allAnimations.map(function() {
			this.end = getElementStyles( this.el[ 0 ] );
			this.diff = styleDifference( this.start, this.end );
			return this;
		});

		// apply original class
		animated.attr( "class", baseClass );

		// map all animated objects again - this time collecting a promise
		allAnimations = allAnimations.map(function() {
			var styleInfo = this,
				dfd = $.Deferred(),
				opts = $.extend({}, o, {
					queue: false,
					complete: function() {
						dfd.resolve( styleInfo );
					}
				});

			this.el.animate( this.diff, opts );
			return dfd.promise();
		});

		// once all animations have completed:
		$.when.apply( $, allAnimations.get() ).done(function() {

			// set the final class
			applyClassChange();

			// for each animated element,
			// clear all css properties that were animated
			$.each( arguments, function() {
				var el = this.el;
				$.each( this.diff, function(key) {
					el.css( key, "" );
				});
			});

			// this is guarnteed to be there if you use jQuery.speed()
			// it also handles dequeuing the next anim...
			o.complete.call( animated[ 0 ] );
		});
	});
};

$.fn.extend({
	addClass: (function( orig ) {
		return function( classNames, speed, easing, callback ) {
			return speed ?
				$.effects.animateClass.call( this,
					{ add: classNames }, speed, easing, callback ) :
				orig.apply( this, arguments );
		};
	})( $.fn.addClass ),

	removeClass: (function( orig ) {
		return function( classNames, speed, easing, callback ) {
			return arguments.length > 1 ?
				$.effects.animateClass.call( this,
					{ remove: classNames }, speed, easing, callback ) :
				orig.apply( this, arguments );
		};
	})( $.fn.removeClass ),

	toggleClass: (function( orig ) {
		return function( classNames, force, speed, easing, callback ) {
			if ( typeof force === "boolean" || force === undefined ) {
				if ( !speed ) {
					// without speed parameter
					return orig.apply( this, arguments );
				} else {
					return $.effects.animateClass.call( this,
						(force ? { add: classNames } : { remove: classNames }),
						speed, easing, callback );
				}
			} else {
				// without force parameter
				return $.effects.animateClass.call( this,
					{ toggle: classNames }, force, speed, easing );
			}
		};
	})( $.fn.toggleClass ),

	switchClass: function( remove, add, speed, easing, callback) {
		return $.effects.animateClass.call( this, {
			add: add,
			remove: remove
		}, speed, easing, callback );
	}
});

})();

/******************************************************************************/
/*********************************** EFFECTS **********************************/
/******************************************************************************/

(function() {

$.extend( $.effects, {
	version: "1.11.4",

	// Saves a set of properties in a data storage
	save: function( element, set ) {
		for ( var i = 0; i < set.length; i++ ) {
			if ( set[ i ] !== null ) {
				element.data( dataSpace + set[ i ], element[ 0 ].style[ set[ i ] ] );
			}
		}
	},

	// Restores a set of previously saved properties from a data storage
	restore: function( element, set ) {
		var val, i;
		for ( i = 0; i < set.length; i++ ) {
			if ( set[ i ] !== null ) {
				val = element.data( dataSpace + set[ i ] );
				// support: jQuery 1.6.2
				// http://bugs.jquery.com/ticket/9917
				// jQuery 1.6.2 incorrectly returns undefined for any falsy value.
				// We can't differentiate between "" and 0 here, so we just assume
				// empty string since it's likely to be a more common value...
				if ( val === undefined ) {
					val = "";
				}
				element.css( set[ i ], val );
			}
		}
	},

	setMode: function( el, mode ) {
		if (mode === "toggle") {
			mode = el.is( ":hidden" ) ? "show" : "hide";
		}
		return mode;
	},

	// Translates a [top,left] array into a baseline value
	// this should be a little more flexible in the future to handle a string & hash
	getBaseline: function( origin, original ) {
		var y, x;
		switch ( origin[ 0 ] ) {
			case "top": y = 0; break;
			case "middle": y = 0.5; break;
			case "bottom": y = 1; break;
			default: y = origin[ 0 ] / original.height;
		}
		switch ( origin[ 1 ] ) {
			case "left": x = 0; break;
			case "center": x = 0.5; break;
			case "right": x = 1; break;
			default: x = origin[ 1 ] / original.width;
		}
		return {
			x: x,
			y: y
		};
	},

	// Wraps the element around a wrapper that copies position properties
	createWrapper: function( element ) {

		// if the element is already wrapped, return it
		if ( element.parent().is( ".ui-effects-wrapper" )) {
			return element.parent();
		}

		// wrap the element
		var props = {
				width: element.outerWidth(true),
				height: element.outerHeight(true),
				"float": element.css( "float" )
			},
			wrapper = $( "<div></div>" )
				.addClass( "ui-effects-wrapper" )
				.css({
					fontSize: "100%",
					background: "transparent",
					border: "none",
					margin: 0,
					padding: 0
				}),
			// Store the size in case width/height are defined in % - Fixes #5245
			size = {
				width: element.width(),
				height: element.height()
			},
			active = document.activeElement;

		// support: Firefox
		// Firefox incorrectly exposes anonymous content
		// https://bugzilla.mozilla.org/show_bug.cgi?id=561664
		try {
			active.id;
		} catch ( e ) {
			active = document.body;
		}

		element.wrap( wrapper );

		// Fixes #7595 - Elements lose focus when wrapped.
		if ( element[ 0 ] === active || $.contains( element[ 0 ], active ) ) {
			$( active ).focus();
		}

		wrapper = element.parent(); //Hotfix for jQuery 1.4 since some change in wrap() seems to actually lose the reference to the wrapped element

		// transfer positioning properties to the wrapper
		if ( element.css( "position" ) === "static" ) {
			wrapper.css({ position: "relative" });
			element.css({ position: "relative" });
		} else {
			$.extend( props, {
				position: element.css( "position" ),
				zIndex: element.css( "z-index" )
			});
			$.each([ "top", "left", "bottom", "right" ], function(i, pos) {
				props[ pos ] = element.css( pos );
				if ( isNaN( parseInt( props[ pos ], 10 ) ) ) {
					props[ pos ] = "auto";
				}
			});
			element.css({
				position: "relative",
				top: 0,
				left: 0,
				right: "auto",
				bottom: "auto"
			});
		}
		element.css(size);

		return wrapper.css( props ).show();
	},

	removeWrapper: function( element ) {
		var active = document.activeElement;

		if ( element.parent().is( ".ui-effects-wrapper" ) ) {
			element.parent().replaceWith( element );

			// Fixes #7595 - Elements lose focus when wrapped.
			if ( element[ 0 ] === active || $.contains( element[ 0 ], active ) ) {
				$( active ).focus();
			}
		}

		return element;
	},

	setTransition: function( element, list, factor, value ) {
		value = value || {};
		$.each( list, function( i, x ) {
			var unit = element.cssUnit( x );
			if ( unit[ 0 ] > 0 ) {
				value[ x ] = unit[ 0 ] * factor + unit[ 1 ];
			}
		});
		return value;
	}
});

// return an effect options object for the given parameters:
function _normalizeArguments( effect, options, speed, callback ) {

	// allow passing all options as the first parameter
	if ( $.isPlainObject( effect ) ) {
		options = effect;
		effect = effect.effect;
	}

	// convert to an object
	effect = { effect: effect };

	// catch (effect, null, ...)
	if ( options == null ) {
		options = {};
	}

	// catch (effect, callback)
	if ( $.isFunction( options ) ) {
		callback = options;
		speed = null;
		options = {};
	}

	// catch (effect, speed, ?)
	if ( typeof options === "number" || $.fx.speeds[ options ] ) {
		callback = speed;
		speed = options;
		options = {};
	}

	// catch (effect, options, callback)
	if ( $.isFunction( speed ) ) {
		callback = speed;
		speed = null;
	}

	// add options to effect
	if ( options ) {
		$.extend( effect, options );
	}

	speed = speed || options.duration;
	effect.duration = $.fx.off ? 0 :
		typeof speed === "number" ? speed :
		speed in $.fx.speeds ? $.fx.speeds[ speed ] :
		$.fx.speeds._default;

	effect.complete = callback || options.complete;

	return effect;
}

function standardAnimationOption( option ) {
	// Valid standard speeds (nothing, number, named speed)
	if ( !option || typeof option === "number" || $.fx.speeds[ option ] ) {
		return true;
	}

	// Invalid strings - treat as "normal" speed
	if ( typeof option === "string" && !$.effects.effect[ option ] ) {
		return true;
	}

	// Complete callback
	if ( $.isFunction( option ) ) {
		return true;
	}

	// Options hash (but not naming an effect)
	if ( typeof option === "object" && !option.effect ) {
		return true;
	}

	// Didn't match any standard API
	return false;
}

$.fn.extend({
	effect: function( /* effect, options, speed, callback */ ) {
		var args = _normalizeArguments.apply( this, arguments ),
			mode = args.mode,
			queue = args.queue,
			effectMethod = $.effects.effect[ args.effect ];

		if ( $.fx.off || !effectMethod ) {
			// delegate to the original method (e.g., .show()) if possible
			if ( mode ) {
				return this[ mode ]( args.duration, args.complete );
			} else {
				return this.each( function() {
					if ( args.complete ) {
						args.complete.call( this );
					}
				});
			}
		}

		function run( next ) {
			var elem = $( this ),
				complete = args.complete,
				mode = args.mode;

			function done() {
				if ( $.isFunction( complete ) ) {
					complete.call( elem[0] );
				}
				if ( $.isFunction( next ) ) {
					next();
				}
			}

			// If the element already has the correct final state, delegate to
			// the core methods so the internal tracking of "olddisplay" works.
			if ( elem.is( ":hidden" ) ? mode === "hide" : mode === "show" ) {
				elem[ mode ]();
				done();
			} else {
				effectMethod.call( elem[0], args, done );
			}
		}

		return queue === false ? this.each( run ) : this.queue( queue || "fx", run );
	},

	show: (function( orig ) {
		return function( option ) {
			if ( standardAnimationOption( option ) ) {
				return orig.apply( this, arguments );
			} else {
				var args = _normalizeArguments.apply( this, arguments );
				args.mode = "show";
				return this.effect.call( this, args );
			}
		};
	})( $.fn.show ),

	hide: (function( orig ) {
		return function( option ) {
			if ( standardAnimationOption( option ) ) {
				return orig.apply( this, arguments );
			} else {
				var args = _normalizeArguments.apply( this, arguments );
				args.mode = "hide";
				return this.effect.call( this, args );
			}
		};
	})( $.fn.hide ),

	toggle: (function( orig ) {
		return function( option ) {
			if ( standardAnimationOption( option ) || typeof option === "boolean" ) {
				return orig.apply( this, arguments );
			} else {
				var args = _normalizeArguments.apply( this, arguments );
				args.mode = "toggle";
				return this.effect.call( this, args );
			}
		};
	})( $.fn.toggle ),

	// helper functions
	cssUnit: function(key) {
		var style = this.css( key ),
			val = [];

		$.each( [ "em", "px", "%", "pt" ], function( i, unit ) {
			if ( style.indexOf( unit ) > 0 ) {
				val = [ parseFloat( style ), unit ];
			}
		});
		return val;
	}
});

})();

/******************************************************************************/
/*********************************** EASING ***********************************/
/******************************************************************************/

(function() {

// based on easing equations from Robert Penner (http://www.robertpenner.com/easing)

var baseEasings = {};

$.each( [ "Quad", "Cubic", "Quart", "Quint", "Expo" ], function( i, name ) {
	baseEasings[ name ] = function( p ) {
		return Math.pow( p, i + 2 );
	};
});

$.extend( baseEasings, {
	Sine: function( p ) {
		return 1 - Math.cos( p * Math.PI / 2 );
	},
	Circ: function( p ) {
		return 1 - Math.sqrt( 1 - p * p );
	},
	Elastic: function( p ) {
		return p === 0 || p === 1 ? p :
			-Math.pow( 2, 8 * (p - 1) ) * Math.sin( ( (p - 1) * 80 - 7.5 ) * Math.PI / 15 );
	},
	Back: function( p ) {
		return p * p * ( 3 * p - 2 );
	},
	Bounce: function( p ) {
		var pow2,
			bounce = 4;

		while ( p < ( ( pow2 = Math.pow( 2, --bounce ) ) - 1 ) / 11 ) {}
		return 1 / Math.pow( 4, 3 - bounce ) - 7.5625 * Math.pow( ( pow2 * 3 - 2 ) / 22 - p, 2 );
	}
});

$.each( baseEasings, function( name, easeIn ) {
	$.easing[ "easeIn" + name ] = easeIn;
	$.easing[ "easeOut" + name ] = function( p ) {
		return 1 - easeIn( 1 - p );
	};
	$.easing[ "easeInOut" + name ] = function( p ) {
		return p < 0.5 ?
			easeIn( p * 2 ) / 2 :
			1 - easeIn( p * -2 + 2 ) / 2;
	};
});

})();

return $.effects;

}));
;

// resource: org.richfaces:togglePanelItem.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.TogglePanelItem = rf.BaseComponent.extendClass({

            // class name
            name:"TogglePanelItem",

            init : function (componentId, options) {
                $super.constructor.call(this, componentId);
                this.attachToDom(this.id);

                this.options = $.extend(this.options, options || {});
                this.name = this.options.name;
                this.togglePanelId = this.options.togglePanelId;
                this.switchMode = this.options.switchMode;
                this.disabled = this.options.disabled || false;

                this.index = options["index"];
                this.getTogglePanel().getItems()[this.index] = this;

                this.__addUserEventHandler("enter");
                this.__addUserEventHandler("leave");
            },

            /***************************** Public Methods *****************************************************************/
            /**
             * @methodOf TogglePanelItem
             * @name TogglePanelItem#getName
             *
             * @return {String} panel item name
             */
            getName: function () {
                return this.options.name;
            },

            /**
             * @methodOf
             * @name TogglePanelItem#getTogglePanel
             *
             * @return {TogglePanel} parent TogglePanel
             * */
            getTogglePanel : function () {
                return rf.component(this.togglePanelId);
            },

            /**
             * @methodOf
             * @name TogglePanelItem#isSelected
             *
             * @return {Boolean} true if this panel item is selected in the parent toggle panel
             * */
            isSelected : function () {
                return this.getName() == this.getTogglePanel().getSelectItem();
            },


            /***************************** Private Methods ****************************************************************/

            /**
             * @private
             * */
            __addUserEventHandler : function (name) {
                var handler = this.options["on" + name];
                if (handler) {
                    rf.Event.bindById(this.id, name, handler);
                }
            },

            /**
             * @private
             *
             * used in TogglePanel
             * */
            __enter : function () {
                rf.getDomElement(this.id).style.display = "block";

                return this.__fireEnter();
            },

            /**
             * @private
             *
             * used in TogglePanel
             * */
            __leave : function () {
                var continueProcess = this.__fireLeave();
                if (!continueProcess) {
                    return false;
                }

                rf.getDomElement(this.id).style.display = "none";
                return true;
            },

            __fireLeave : function () {
                return rf.Event.fireById(this.id, "leave");
            },

            __fireEnter : function () {
                return rf.Event.fireById(this.id, "enter");
            },

            // class stuff
            destroy: function () {
                var parent = this.getTogglePanel();
                if (parent) {
                    delete parent.getItems()[this.index];
                }

                $super.destroy.call(this);
            }
        });

    // define super class link
    var $super = rf.ui.TogglePanelItem.$super;
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:inputNumberSlider.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
(function($, rf) {
    rf.ui = rf.ui || {};

    rf.ui.InputNumberSlider = rf.BaseComponent.extendClass({

            name: "InputNumberSlider",

            delay: 200,
            maxValue: 100,
            minValue: 0,
            step: 1,
            tabIndex: 0,

            decreaseSelectedClass: "rf-insl-dec-sel",
            handleSelectedClass: "rf-insl-hnd-sel",
            increaseSelectedClass: "rf-insl-inc-sel",

            init: function (id, options, selectedClasses) {
                $superInputNumberSlider.constructor.call(this, id);
                $.extend(this, options);
                this.range = this.maxValue - this.minValue;
                this.id = id;
                this.element = $(this.attachToDom());
                this.input = this.element.children(".rf-insl-inp-cntr").children(".rf-insl-inp");
                this.track = this.element.children(".rf-insl-trc-cntr").children(".rf-insl-trc");
                this.handleContainer = this.track.children("span");
                this.handle = this.handleContainer.children(".rf-insl-hnd, .rf-insl-hnd-dis");
                this.tooltip = this.element.children(".rf-insl-tt");

                var value = Number(this.input.val());
                if (isNaN(value)) {
                    value = this.minValue;
                }
                this.handleContainer.css("display", "block");
                this.track.css("padding-right", this.handle.width() + "px");
                this.__setValue(value, null, true);

                if (!this.disabled) {
                    this.decreaseButton = this.element.children(".rf-insl-dec");
                    this.increaseButton = this.element.children(".rf-insl-inc");

                    this.track[0].tabIndex = this.tabIndex;

                    for (var i in selectedClasses) {
                        if (selectedClasses.hasOwnProperty(i)) {
                            this[i] += " " + selectedClasses[i];
                        }
                    }
                    var proxy = $.proxy(this.__inputHandler, this);
                    this.input.change(proxy);
                    this.input.submit(proxy);
                    this.element.mousewheel($.proxy(this.__mousewheelHandler, this));
                    this.track.keydown($.proxy(this.__keydownHandler, this));
                    this.decreaseButton.mousedown($.proxy(this.__decreaseHandler, this));
                    this.increaseButton.mousedown($.proxy(this.__increaseHandler, this));
                    this.track.mousedown($.proxy(this.__mousedownHandler, this));
                }
            },

            decrease: function (event) {
                var value = this.value - this.step;
                value = this.roundFloat(value);
                this.setValue(value, event);
            },

            increase: function (event) {
                var value = this.value + this.step;
                value = this.roundFloat(value);
                this.setValue(value, event);
            },

            getValue: function () {
                return this.value;
            },

            setValue: function (value, event) {
                if (!this.disabled) {
                    this.__setValue(value, event);
                }
            },

            roundFloat: function(x){
                var str = this.step.toString();
                var power = 0;
                if (!/\./.test(str)) {
                    if (this.step >= 1) {
                        return x;
                    }
                    if (/e/.test(str)) {
                        power = str.split("-")[1];
                    }
                } else {
                    power = str.length - str.indexOf(".") - 1;
                }
                var ret = x.toFixed(power);
                return parseFloat(ret);
            },

            focus: function() {
                this.input.focus();
            },

            __setValue: function (value, event, skipOnchange) {
                if (!isNaN(value)) {
                    var changed = false;
                    if (this.input.val() == "") {
                        // value already changed from "" to 0, compare to real value to track changes
                        changed = true;
                    }

                    if (value > this.maxValue) {
                        value = this.maxValue;
                        this.input.val(value);
                        changed = true;
                    } else if (value < this.minValue) {
                        value = this.minValue;
                        this.input.val(value);
                        changed = true;
                    }
                    if (value != this.value || changed) {
                        this.input.val(value);
                        var left = 100 * (value - this.minValue) / this.range;
                        if(this.handleType=='bar') {
                            this.handleContainer.css("width", left + "%");
                        } else {
                            this.handleContainer.css("padding-left", left + "%");
                        }
                        this.tooltip.text(value);
                        this.tooltip.setPosition(this.handle, {from: 'LT', offset: [0, 5]}); //TODO Seems offset doesn't work now.
                        this.value = value;
                        if (this.onchange && !skipOnchange) {
                            this.onchange.call(this.element[0], event);
                        }
                    }
                }
            },

            __inputHandler: function (event) {
                var value = Number(this.input.val());
                if (isNaN(value)) {
                    this.input.val(this.value);
                } else {
                    this.__setValue(value, event);
                }
            },

            __mousewheelHandler: function (event, delta, deltaX, deltaY) {
                delta = deltaX || deltaY;
                if (delta > 0) {
                    this.increase(event);
                } else if (delta < 0) {
                    this.decrease(event);
                }
                return false;
            },

            __keydownHandler: function (event) {
                if (event.keyCode == 37) { //LEFT
                    var value = Number(this.input.val()) - this.step;
                    value = this.roundFloat(value);
                    this.__setValue(value, event);
                    event.preventDefault();
                } else if (event.keyCode == 39) { //RIGHT
                    var value = Number(this.input.val()) + this.step;
                    value = this.roundFloat(value);
                    this.__setValue(value, event);
                    event.preventDefault();
                }
            },

            __decreaseHandler: function (event) {
                var component = this;
                component.decrease(event);
                this.intervalId = window.setInterval(function() {
                    component.decrease(event);
                }, this.delay);
                $(document).one("mouseup", true, $.proxy(this.__clearInterval, this));
                this.decreaseButton.addClass(this.decreaseSelectedClass);
                event.preventDefault();
            },

            __increaseHandler: function (event) {
                var component = this;
                component.increase(event);
                this.intervalId = window.setInterval(function() {
                    component.increase(event);
                }, this.delay);
                $(document).one("mouseup", $.proxy(this.__clearInterval, this));
                this.increaseButton.addClass(this.increaseSelectedClass);
                event.preventDefault();
            },

            __clearInterval: function (event) {
                window.clearInterval(this.intervalId);
                if (event.data) { // decreaseButton
                    this.decreaseButton.removeClass(this.decreaseSelectedClass);
                } else {
                    this.increaseButton.removeClass(this.increaseSelectedClass);
                }
            },

            __mousedownHandler: function (event) {
                this.__mousemoveHandler(event);
                this.track.focus();
                var jQueryDocument = $(document);
                jQueryDocument.mousemove($.proxy(this.__mousemoveHandler, this));
                jQueryDocument.one("mouseup", $.proxy(this.__mouseupHandler, this));
                this.handle.addClass(this.handleSelectedClass);
                this.tooltip.show();
            },

            __mousemoveHandler: function (event) {
                var value = this.range * (event.pageX - this.track.offset().left - this.handle.width() / 2) / (this.track.width()
                    - this.handle.width()) + this.minValue;
                value = Math.round(value / this.step) * this.step;
                value = this.roundFloat(value);
                this.__setValue(value, event);
                event.preventDefault();
            },

            __mouseupHandler: function () {
                this.handle.removeClass(this.handleSelectedClass);
                this.tooltip.hide();
                $(document).unbind("mousemove", this.__mousemoveHandler);
            },

            destroy: function (event) {
                $(document).unbind("mousemove", this.__mousemoveHandler);
                $superInputNumberSlider.destroy.call(this);
            }
        });
    $superInputNumberSlider = rf.ui.InputNumberSlider.$super;
}(RichFaces.jQuery, window.RichFaces));
;

// resource: org.richfaces:inputNumberSpinner.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
(function($, rf) {
    rf.ui = rf.ui || {};

    rf.ui.InputNumberSpinner = rf.BaseComponent.extendClass({

            name: "InputNumberSpinner",

            cycled: true,
            delay: 200,
            maxValue: 100,
            minValue: 0,
            step: 1,

            init: function (id, options) {
                $super.constructor.call(this, id);
                $.extend(this, options);
                this.element = $(this.attachToDom());
                this.input = this.element.children(".rf-insp-inp");

                var value = Number(this.input.val());
                if (isNaN(value)) {
                    value = this.minValue;
                }
                this.__setValue(value, null, true);

                if (!this.input.attr("disabled")) {
                    var buttonsArea = this.element.children(".rf-insp-btns");
                    this.decreaseButton = buttonsArea.children(".rf-insp-dec");
                    this.increaseButton = buttonsArea.children(".rf-insp-inc");

                    var proxy = $.proxy(this.__inputHandler, this)
                    this.input.change(proxy);
                    this.input.submit(proxy);
                    this.input.submit(proxy);
                    this.input.mousewheel($.proxy(this.__mousewheelHandler, this));
                    this.input.keydown($.proxy(this.__keydownHandler, this));
                    this.decreaseButton.mousedown($.proxy(this.__decreaseHandler, this));
                    this.increaseButton.mousedown($.proxy(this.__increaseHandler, this));
                }
            },

            decrease: function (event) {
                var value = this.value - this.step;
                value = this.roundFloat(value);
                if (value < this.minValue && this.cycled) {
                    value = this.maxValue;
                }
                this.__setValue(value, event);
            },

            increase: function (event) {
                var value = this.value + this.step;
                value = this.roundFloat(value);

                if (value > this.maxValue && this.cycled) {
                    value = this.minValue;
                }
                this.__setValue(value, event);
            },

            getValue: function () {
                return this.value;
            },

            setValue: function (value, event) {
                if (!this.input.attr("disabled")) {
                    this.__setValue(value);
                }
            },

            roundFloat: function(x){
                var str = this.step.toString();
                var power = 0;
                if (!/\./.test(str)) {
                    if (this.step >= 1) {
                        return x;
                    }
                    if (/e/.test(str)) {
                        power = str.split("-")[1];
                    }
                } else {
                    power = str.length - str.indexOf(".") - 1;
                }
                var ret = x.toFixed(power);
                return parseFloat(ret);
            },

            focus: function() {
                this.input.focus();
            },

            destroy: function (event) {
                if (this.intervalId) {
                    window.clearInterval(this.intervalId);
                    this.decreaseButton.css("backgroundPosition", " 50% 40%").unbind("mouseout", this.destroy)
                        .unbind("mouseup", this.destroy);
                    this.increaseButton.css("backgroundPosition", " 50% 40%").unbind("mouseout", this.destroy)
                        .unbind("mouseup", this.destroy);
                    this.intervalId = null;
                }
                $super.destroy.call(this);
            },

            __setValue: function (value, event, skipOnchange) {
                if (!isNaN(value)) {
                    if (value > this.maxValue) {
                        value = this.maxValue;
                        this.input.val(value);
                    } else if (value < this.minValue) {
                        value = this.minValue;
                        this.input.val(value);
                    }
                    if (value != this.value) {
                        this.input.val(value);
                        this.value = value;
                        if (this.onchange && !skipOnchange) {
                            this.onchange.call(this.element[0], event);
                        }
                    }
                }
            },

            __inputHandler: function (event) {
                var value = Number(this.input.val());
                if (isNaN(value)) {
                    this.input.val(this.value);
                } else {
                    this.__setValue(value, event);
                }
            },

            __mousewheelHandler: function (event, delta, deltaX, deltaY) {
                delta = deltaX || deltaY;
                if (delta > 0) {
                    this.increase(event);
                } else if (delta < 0) {
                    this.decrease(event);
                }
                return false;
            },

            __keydownHandler: function (event) {
                if (event.keyCode == 40) { //DOWN
                    this.decrease(event);
                    event.preventDefault();
                } else if (event.keyCode == 38) { //UP
                    this.increase(event);
                    event.preventDefault();
                }
            },

            __decreaseHandler: function (event) {
                var component = this;
                component.decrease(event);
                this.intervalId = window.setInterval(function() {
                    component.decrease(event);
                }, this.delay);
                var proxy = $.proxy(this.destroy, this);
                this.decreaseButton.bind("mouseup", proxy).bind("mouseout", proxy)
                    .css("backgroundPosition", "60% 60%");
                event.preventDefault();
            },

            __increaseHandler: function (event) {
                var component = this;
                component.increase(event);
                this.intervalId = window.setInterval(function() {
                    component.increase(event);
                }, this.delay);
                var proxy = $.proxy(this.destroy, this);
                this.increaseButton.bind("mouseup", proxy).bind("mouseout", proxy)
                    .css("backgroundPosition", "60% 60%");
                event.preventDefault();
            }
        });

    // define super class link
    var $super = rf.ui.InputNumberSpinner.$super;
}(RichFaces.jQuery, window.RichFaces));;

// resource: org.richfaces:accordion.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.Accordion = rf.ui.TogglePanel.extendClass({
            // class name
            name:"Accordion",

            /**
             * @class Accordion
             * @name Accordion
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                $super.constructor.call(this, componentId, options);
                this.items = [];

                this.isKeepHeight = options["isKeepHeight"] || false
            },

            /***************************** Public Methods  ****************************************************************/

            getHeight : function (recalculate) {
                if (recalculate || !this.__height) {
                    this.__height = $(rf.getDomElement(this.id)).outerHeight(true)
                }

                return this.__height;
            },

            getInnerHeight : function (recalculate) {
                if (recalculate || !this.__innerHeight) {
                    this.__innerHeight = $(rf.getDomElement(this.id)).innerHeight()
                }

                return this.__innerHeight;
            },

            /***************************** Private Methods ********************************************************/


            destroy: function () {
                rf.Event.unbindById(this.id, "." + this.namespace);
                $super.destroy.call(this);
            }
        });

    // define super class link
    var $super = rf.ui.Accordion.$super;
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:popupPanelBorders.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.PopupPanel.Border = function(id, modalPanel, cursor, sizer) {

        $super.constructor.call(this, id);

        this.element = $(rf.getDomElement(id));
        this.element.css('cursor', cursor);
        var border = this;
        this.element.bind('mousedown', {border:border}, this.startDrag);

        this.modalPanel = modalPanel;
        this.sizer = sizer;
    };

    var $super = rf.BaseComponent.extend(rf.ui.PopupPanel.Border);
    var $super = rf.ui.PopupPanel.Border.$super;

    $.extend(rf.ui.PopupPanel.Border.prototype, (function (options) {

        return {

            name: "RichFaces.ui.PopupPanel.Border",

            destroy: function() {
                if (this.doingDrag) {
                    $(document).unbind('mousemove', this.doDrag);
                    $(document).unbind('mouseup', this.endDrag);
                }

                this.element.unbind('mousedown', this.startDrag);
                this.element = null;
                this.modalPanel = null;
            },

            show: function() {
                this.element.show();
            },

            hide: function() {
                this.element.hide();
            },

            startDrag: function(event) {
                var border = event.data.border;
                border.doingDrag = true;

                border.dragX = event.clientX;
                border.dragY = event.clientY;
                $(document).bind('mousemove', {border:border}, border.doDrag);
                $(document).bind('mouseup', {border:border}, border.endDrag);

                border.modalPanel.startDrag(border);

                border.onselectStartHandler = document.onselectstart;
                document.onselectstart = function() {
                    return false;
                }
            },

            getWindowSize : function() {
                var myWidth = 0, myHeight = 0;
                if (typeof( window.innerWidth ) == 'number') {
                    myWidth = window.innerWidth;
                    myHeight = window.innerHeight;
                } else if (document.documentElement && ( document.documentElement.clientWidth || document.documentElement.clientHeight )) {
                    myWidth = document.documentElement.clientWidth;
                    myHeight = document.documentElement.clientHeight;
                } else if (document.body && ( document.body.clientWidth || document.body.clientHeight )) {
                    myWidth = document.body.clientWidth;
                    myHeight = document.body.clientHeight;
                }
                return {"width":myWidth,"height":myHeight};
            },

            doDrag: function(event) {
                var border = event.data.border;
                if (!border.doingDrag) {
                    return;
                }

                var evtX = event.clientX;
                var evtY = event.clientY;

                var winSize = border.getWindowSize();

                if (evtX < 0) {
                    evtX = 0;
                } else if (evtX >= winSize.width) {
                    evtX = winSize.width - 1;
                }

                if (evtY < 0) {
                    evtY = 0;
                } else if (evtY >= winSize.height) {
                    evtY = winSize.height - 1;
                }

                var dx = evtX - border.dragX;
                var dy = evtY - border.dragY;

                if (dx != 0 || dy != 0) {

                    var id = border.id;

                    var diff = border.sizer.prototype.doDiff(dx, dy);//TODO
                    var doResize;

                    var element = border.modalPanel.cdiv;

                    if (diff.deltaWidth || diff.deltaHeight) {
                        doResize = border.modalPanel.invokeEvent("resize", event, null, element);
                    } else if (diff.deltaX || diff.deltaY) {
                        doResize = border.modalPanel.invokeEvent("move", event, null, element);
                    }

                    var vetoes;

                    if (doResize) {
                        vetoes = border.modalPanel.doResizeOrMove(diff);
                    }

                    if (vetoes) {
                        if (!vetoes.x) {
                            border.dragX = evtX;
                        } else {
                            if (!diff.deltaX) {
                                border.dragX -= vetoes.vx || 0;
                            } else {
                                border.dragX += vetoes.vx || 0;
                            }
                        }

                        if (!vetoes.y) {
                            border.dragY = evtY;
                        } else {
                            if (!diff.deltaY) {
                                border.dragY -= vetoes.vy || 0;
                            } else {
                                border.dragY += vetoes.vy || 0;
                            }
                        }
                    }
                }
            },

            endDrag: function(event) {
                var border = event.data.border;
                border.doingDrag = undefined;

                $(document).unbind('mousemove', border.doDrag);
                $(document).unbind('mouseup', border.endDrag);

                border.modalPanel.endDrag(border);

                document.onselectstart = border.onselectStartHandler;
                border.onselectStartHandler = null;
            },

            doPosition: function() {
                this.sizer.prototype.doPosition(this.modalPanel, this.element);	//TODO remove prototype
            }
        }

    })());

})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:extendedDataTable.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
(function($, rf) {
    rf.utils = rf.utils || {};

    rf.utils.addCSSText = function(cssText, elementId) {
        var style = $("<style></style>").attr({type: 'text/css', id: elementId}).appendTo("head");
        try {
            style.html(cssText);
        } catch (e) {
            //IE
            style[0].styleSheet.cssText = cssText;
        }
    };
    
    rf.utils.Ranges = function() {
        this.ranges = [];
    };

    rf.utils.Ranges.prototype = {

        add: function(index) {
            var i = 0;
            while (i < this.ranges.length && index >= this.ranges[i++][1]);
            i--;
            if (this.ranges[i - 1] && index == (this.ranges[i - 1][1] + 1)) {
                if (index == (this.ranges[i][0] - 1)) {
                    this.ranges[i - 1][1] = this.ranges[i][1];
                    this.ranges.splice(i, 1);
                } else {
                    this.ranges[i - 1][1]++;
                }
            } else {
                if (this.ranges[i]) {
                    if (this.ranges[i] && index == (this.ranges[i][0] - 1)) {
                        this.ranges[i][0]--;
                    } else {
                        if (index == (this.ranges[i][1] + 1)) {
                            this.ranges[i][1]++;
                        } else {
                            if (index < this.ranges[i][1]) {
                                this.ranges.splice(i, 0, [index, index]);
                            } else {
                                this.ranges.splice(i + 1, 0, [index, index]);
                            }
                        }
                    }
                } else {
                    this.ranges.splice(i, 0, [index, index]);
                }
            }
        },

        remove: function(index) {
            var i = 0;
            while (i < this.ranges.length && index > this.ranges[i++][1]);
            i--;
            if (this.ranges[i]) {
                if (index == (this.ranges[i][1])) {
                    if (index == (this.ranges[i][0])) {
                        this.ranges.splice(i, 1);
                    } else {
                        this.ranges[i][1]--;
                    }
                } else {
                    if (index == (this.ranges[i][0])) {
                        this.ranges[i][0]++;
                    } else {
                        this.ranges.splice(i + 1, 0, [index + 1, this.ranges[i][1]]);
                        this.ranges[i][1] = index - 1;
                    }
                }
            }
        },

        clear: function() {
            this.ranges = [];
        },

        contains: function(index) {
            var i = 0;
            while (i < this.ranges.length && index >= this.ranges[i][0]) {
                if (index >= this.ranges[i][0] && index <= this.ranges[i][1]) {
                    return true;
                } else {
                    i++;
                }
            }
            return false;
        },

        toString: function() {
            var ret = new Array(this.ranges.length);
            for (var i = 0; i < this.ranges.length; i++) {
                ret[i] = this.ranges[i].join();
            }
            return ret.join(";");
        }
    };

    var WIDTH_CLASS_NAME_BASE = "rf-edt-c-";
    var MIN_WIDTH = 20;

    rf.ui = rf.ui || {};

    rf.ui.ExtendedDataTable = rf.BaseComponent.extendClass({

            name: "ExtendedDataTable",

            init: function (id, rowCount, ajaxFunction, options) {
                $super.constructor.call(this, id);
                this.ranges = new rf.utils.Ranges();
                this.rowCount = rowCount;
                this.ajaxFunction = ajaxFunction;
                this.options = options || {};
                this.element = this.attachToDom();
                this.newWidths = {};
                this.storeDomReferences();
                if (this.options['onready'] && typeof this.options['onready'] == 'function') {
                    rf.Event.bind(this.element, "rich:ready", this.options['onready']);
                }
                this.resizeEventName = "resize.rf.edt." + this.id;
                $(document).ready($.proxy(this.initialize, this));
                this.activateResizeListener();
                var body = $(this.element).find(".rf-edt-b .rf-edt-cnt");
                var delayedScroll = function(table, body) {
                    return function () {
                        // let browser scroll the element
                        setTimeout(function() {
                            table.scrollElement.scrollLeft = body.scrollLeft();
                            table.updateScrollPosition();
                        }, 0);
                    };
                };
                
                body.bind("scroll", delayedScroll(this, body));
                body.bind("mousewheel", $.proxy(this.horizontalScrollHandler, this));
                $(this.scrollElement).bind("scroll", $.proxy(this.updateScrollPosition, this));
                this.bindHeaderHandlers();
                $(this.element).bind("rich:onajaxcomplete", $.proxy(this.ajaxComplete, this));
                
                this.resizeData = {};
                this.idOfReorderingColumn = "";
                this.timeoutId = null;
            },

            storeDomReferences: function() {
                this.dragElement = document.getElementById(this.id + ":d");
                this.reorderElement = document.getElementById(this.id + ":r");
                this.reorderMarkerElement = document.getElementById(this.id + ":rm");
                this.widthInput = document.getElementById(this.id + ":wi");
                this.selectionInput = document.getElementById(this.id + ":si");


                this.header = $(this.element).children(".rf-edt-hdr");
                this.headerCells = this.header.find(".rf-edt-hdr-c");
                this.footerCells = $(this.element).children(".rf-edt-ftr").find(".rf-edt-ftr-c");
                this.resizerHolders = this.header.find(".rf-edt-rsz-cntr");

                this.frozenHeaderPartElement = document.getElementById(this.id + ":frozenHeader");
                this.frozenColumnCount = this.frozenHeaderPartElement ? this.frozenHeaderPartElement.children[0].rows[0].cells.length : 0;//TODO Richfaces.firstDescendant;

                this.headerElement = document.getElementById(this.id + ":header");
                this.footerElement = document.getElementById(this.id + ":footer");
                this.scrollElement = document.getElementById(this.id + ":scrl");
                this.scrollContentElement = document.getElementById(this.id + ":scrl-cnt");

            },

            getColumnPosition: function(id) {
                var position;
                for (var i = 0; i < this.headerCells.length; i++) {
                    if (id == this.headerCells[i].className.match(new RegExp(WIDTH_CLASS_NAME_BASE + "([^\\W]*)"))[1]) {
                        position = i;
                    }
                }
                return position;
            },

            setColumnPosition: function(id, position) {
                var colunmsOrder = "";
                var before;
                for (var i = 0; i < this.headerCells.length; i++) {
                    var current = this.headerCells[i].className.match(new RegExp(WIDTH_CLASS_NAME_BASE + "([^\\W]*)"))[1];
                    if (i == position) {
                        if (before) {
                            colunmsOrder += current + "," + id + ",";
                        } else {
                            colunmsOrder += id + "," + current + ",";
                        }
                    } else {
                        if (id != current) {
                            colunmsOrder += current + ",";
                        } else {
                            before = true;
                        }
                    }
                }
                this.ajaxFunction(null, {"rich:columnsOrder" : colunmsOrder}); // TODO Maybe, event model should be used here.
            },

            setColumnWidth: function(columnId, width) {
                width = width + "px";
                var $table = $(document.getElementById(this.element.id));
                $table.find("." + WIDTH_CLASS_NAME_BASE + columnId).parent().css('width', width);
                $table.find("." + WIDTH_CLASS_NAME_BASE + columnId).css('width', width);
                this.newWidths[columnId] = width;
                var widthsArray = new Array();
                for (var id in this.newWidths) {
                    if (this.newWidths.hasOwnProperty(id)) {
                        widthsArray.push(id + ":" + this.newWidths[id]);
                    }
                }
                this.widthInput.value = widthsArray.toString();
                this.updateLayout();
                this.adjustResizers();
                this.ajaxFunction(); // TODO Maybe, event model should be used here.
            },

            filter: function(colunmId, filterValue, isClear) {
                if (typeof(filterValue) == "undefined" || filterValue == null) {
                    filterValue = "";
                }
                var map = {};
                map[this.id + "rich:filtering"] = colunmId + ":" + filterValue + ":" + isClear;
                this.ajaxFunction(null, map); // TODO Maybe, event model should be used here.
            },

            clearFiltering: function() {
                this.filter("", "", true);
            },

            sortHandler: function(event) {
                var sortHandle = $(event.data.sortHandle);
                var button = sortHandle.find('.rf-edt-srt-btn');
                var columnId = button.data('columnid');
                var sortOrder = button.hasClass('rf-edt-srt-asc') ? 'descending' : 'ascending';
                this.sort(columnId, sortOrder, false);
            },

            filterHandler: function(event) {
                if (event.type == "keyup" && event.keyCode != 13) {
                    return;
                }
                var filterHandle = $(event.data.filterHandle);
                var columnId = filterHandle.data('columnid');
                var filterValue = filterHandle.val();
                this.filter(columnId, filterValue, false);
            },


            sort: function(colunmId, sortOrder, isClear) {
                if (typeof(sortOrder) == "string") {
                    sortOrder = sortOrder.toLowerCase();
                }
                var map = {}
                map[this.id + "rich:sorting"] = colunmId + ":" + sortOrder + ":" + isClear;
                this.ajaxFunction(null, map); // TODO Maybe, event model should be used here.
            },

            clearSorting: function() {
                this.sort("", "", true);
            },

            destroy: function() {
                $(window).unbind("resize", this.updateLayout);
                $(rf.getDomElement(this.id + ':st')).remove();
                $super.destroy.call(this);
            },

            bindHeaderHandlers: function() {
                this.header.find(".rf-edt-rsz").bind("mousedown", $.proxy(this.beginResize, this));
                this.headerCells.bind("mousedown", $.proxy(this.beginReorder, this));
                var self = this;
                this.header.find(".rf-edt-c-srt").each(function() {
                    $(this).bind("click", {sortHandle: this}, $.proxy(self.sortHandler, self));
                });
                this.header.find(".rf-edt-flt-i").each(function() {
                    $(this).bind("blur", {filterHandle: this}, $.proxy(self.filterHandler, self));
                    $(this).bind("keyup", {filterHandle: this}, $.proxy(self.filterHandler, self));
                });
            },

            updateLayout: function() {
                this.deActivateResizeListener();
                this.headerCells.height("auto");
                var headerCellHeight = 0;
                this.headerCells.each(function() {
                    if (this.clientHeight > headerCellHeight) {
                        headerCellHeight = this.clientHeight;
                    }
                });
                this.headerCells.height(headerCellHeight + "px");
                this.footerCells.height("auto");
                var footerCellHeight = 0;
                this.footerCells.each(function() {
                    if (this.clientHeight > footerCellHeight) {
                        footerCellHeight = this.clientHeight;
                    }
                });
                this.footerCells.height(footerCellHeight + "px");
                this.contentDivElement.css('width', 'auto');
                var offsetWidth = this.frozenHeaderPartElement ? this.frozenHeaderPartElement.offsetWidth : 0;
                var width = Math.max(0, this.element.clientWidth - offsetWidth);
                if (width) {
                    this.parts.each(function() {
                        this.style.width = "auto";
                    });
                    var contentWidth = this.parts.width();
                    if (contentWidth > width) {
                        this.contentDivElement.css('width', width + 'px');
                    }
                    this.contentDivElement.css('display', 'block');
                    // update scroller and scroll-content
                    if (contentWidth > width) {
                        this.parts.each(function() {
                            this.style.width = width + "px";
                        });
                        this.scrollElement.style.display = "block";
                        this.scrollElement.style.overflowX = "scroll";
                        this.scrollElement.style.width = width + "px";
                        this.scrollContentElement.style.width = contentWidth + "px";
                        this.updateScrollPosition();
                    } else {
                        this.parts.each(function() {
                            this.style.width = "";
                        });
                        this.scrollElement.style.display = "none";
                    }
                } else if (this.element.clientWidth != 0) {
                    this.contentDivElement.css('display', 'none');
                }
                var height = this.element.clientHeight;
                var el = this.element.firstChild;
                while (el && (!el.nodeName || el.nodeName.toUpperCase() != "TABLE")) {
                    if (el.nodeName && el.nodeName.toUpperCase() == "DIV" && el != this.bodyElement) {
                        height -= el.offsetHeight;
                    }
                    el = el.nextSibling;
                }
                if (this.bodyElement.offsetHeight != height || !this.contentElement) {
                    this.bodyElement.style.height = height > this.contentDivElement.height() ? "" : height + "px";
                }
                this.activateResizeListener();
            },

            adjustResizers: function() { //For IE7 only.
                var scrollLeft = this.scrollElement ? this.scrollElement.scrollLeft : 0;
                var clientWidth = this.element.clientWidth - 3;
                var i = 0;
                for (; i < this.frozenColumnCount; i++) {
                    if (clientWidth > 0) {
                        this.resizerHolders[i].style.display = "none";
                        this.resizerHolders[i].style.display = "";
                        clientWidth -= this.resizerHolders[i].offsetWidth;
                    }
                    if (clientWidth <= 0) {
                        this.resizerHolders[i].style.display = "none";
                    }
                }
                scrollLeft -= 3;
                for (; i < this.resizerHolders.length; i++) {
                    if (clientWidth > 0) {
                        this.resizerHolders[i].style.display = "none";
                        if (scrollLeft > 0) {
                            this.resizerHolders[i].style.display = "";
                            scrollLeft -= this.resizerHolders[i].offsetWidth;
                            if (scrollLeft > 0) {
                                this.resizerHolders[i].style.display = "none";
                            } else {
                                clientWidth += scrollLeft;
                            }
                        } else {
                            this.resizerHolders[i].style.display = "";
                            clientWidth -= this.resizerHolders[i].offsetWidth;
                        }
                    }
                    if (clientWidth <= 0) {
                        this.resizerHolders[i].style.display = "none";
                    }
                }
            },

            updateScrollPosition: function() {
                if (this.scrollElement) {
                    var scrollLeft = this.scrollElement.scrollLeft;
                    this.parts.each(function() {
                        this.scrollLeft = scrollLeft;
                    });
                }
                this.adjustResizers();
            },
            
            horizontalScrollHandler: function(event) {
                var scrollDelta = event.deltaFactor * event.deltaX;
                
                // ignore vertical scroll (deltaX = 0)
                if (scrollDelta && this.scrollElement) {
                    this.scrollElement.scrollLeft += scrollDelta;
                    this.updateScrollPosition();
                }  
            },

            initialize: function() {
                this.deActivateResizeListener();
                if (! $(this.element).is(":visible")) {
                    this.showOffscreen(this.element);
                }
                this.bodyElement = document.getElementById(this.id + ":b");
                this.bodyElement.tabIndex = -1; //TODO don't use tabIndex.
                this.contentDivElement = $(this.bodyElement).find('.rf-edt-cnt');
                var bodyJQuery = $(this.bodyElement);
                this.contentElement = bodyJQuery.children("div:not(.rf-edt-ndt):first")[0];
                if (this.contentElement) {
                    this.spacerElement = this.contentElement.children[0];//TODO this.marginElement = Richfaces.firstDescendant(this.contentElement);
                    this.dataTableElement = this.contentElement.lastChild;//TODO this.dataTableElement = Richfaces.lastDescendant(this.contentElement);
                    this.tbodies = $(document.getElementById(this.id + ":tbf")).add(document.getElementById(this.id + ":tbn"));
                    this.rows = this.tbodies[0].rows.length;
                    this.rowHeight = this.dataTableElement.offsetHeight / this.rows;
                    if (this.rowCount != this.rows) {
                        this.contentElement.style.height = (this.rowCount * this.rowHeight) + "px";
                    }
                    bodyJQuery.bind("scroll", $.proxy(this.bodyScrollListener, this));
                    if (this.options.selectionMode != "none") {
                        this.tbodies.bind("click", $.proxy(this.selectionClickListener, this));
                        bodyJQuery.bind(window.opera ? "keypress" : "keydown", $.proxy(this.selectionKeyDownListener, this));
                        this.initializeSelection();
                    }
                } else {
                    this.spacerElement = null;
                    this.dataTableElement = null;
                }
                var edt = this.element;
                this.parts = $(this.element).find(".rf-edt-cnt, .rf-edt-ftr-cnt").filter(function() {
                    return $(this).parents('.rf-edt').get(0) === edt;
                });
                this.updateLayout();
                this.updateScrollPosition(); //TODO Restore horizontal scroll position
                if ($(this.element).data('offscreenElements')) {
                    this.hideOffscreen(this.element);
                }
                this.activateResizeListener();
                $(this.element).trigger("rich:ready", this);
            },

            showOffscreen: function(element) {
                var $element = $(element);
                var offscreenElements = $element.parents(":not(:visible)").addBack().toArray().reverse();
                var that = this;
                $.each(offscreenElements, function() {
                    $this = $(this);
                    if ($this.css('display') === 'none') {
                        that.showOffscreenElement($(this));
                    }
                })
                $element.data('offscreenElements', offscreenElements);
            },

            hideOffscreen: function(element) {
                var $element = $(element);
                var offscreenElements = $element.data('offscreenElements');
                var that = this;
                $.each(offscreenElements, function() {
                    $this = $(this);
                    if ($this.data('offscreenOldValues')) {
                        that.hideOffscreenElement($(this));
                    }
                })
                $element.removeData('offscreenElements');
            },

            showOffscreenElement: function($element) {
                var offscreenOldValues = {};
                offscreenOldValues.oldPosition = $element.css('position');
                offscreenOldValues.oldLeft = $element.css('left');
                offscreenOldValues.oldDisplay = $element.css('display');
                $element.css('position', 'absolute');
                $element.css('left', '-10000');
                $element.css('display', 'block');
                $element.data('offscreenOldValues', offscreenOldValues);
            },

            hideOffscreenElement: function($element) {
                var offscreenOldValues = $element.data('offscreenOldValues');
                $element.css('display', offscreenOldValues.oldDisplay);
                $element.css('left', offscreenOldValues.oldLeft);
                $element.css('position', offscreenOldValues.oldPosition);
                $element.removeData('offscreenOldValues');
            },

            drag: function(event) {
                $(this.dragElement).setPosition({left:Math.max(this.resizeData.left + MIN_WIDTH, event.pageX), top: $(this.dragElement).offset().top});
                return false;
            },

            beginResize: function(event) {
                var id = event.currentTarget.parentNode.className.match(new RegExp(WIDTH_CLASS_NAME_BASE + "([^\\W]*)"))[1];
                this.resizeData = {
                    id : id,
                    left : $(event.currentTarget).parent().offset().left
                };
                this.dragElement.style.height = this.element.offsetHeight + "px";
                $(this.dragElement).setPosition({top:$(this.element).offset().top, left:event.pageX});
                this.dragElement.style.display = "block";
                $(document).bind("mousemove", $.proxy(this.drag, this));
                $(document).one("mouseup", $.proxy(this.endResize, this));
                return false;
            },

            endResize: function(event) {
                $(document).unbind("mousemove", this.drag);
                this.dragElement.style.display = "none";
                var width = Math.max(MIN_WIDTH, event.pageX - this.resizeData.left);
                this.setColumnWidth(this.resizeData.id, width);
            },

            reorder: function(event) {
                $(this.reorderElement).setPosition(event, {offset:[5,5]});
                this.reorderElement.style.display = "block";
                return false;
            },

            beginReorder: function(event) {
                if (!$(event.target).is("a, img, :input")) {
                    this.idOfReorderingColumn = event.currentTarget.className.match(new RegExp(WIDTH_CLASS_NAME_BASE + "([^\\W]*)"))[1];
                    $(document).bind("mousemove", $.proxy(this.reorder, this));
                    this.headerCells.bind("mouseover", $.proxy(this.overReorder, this));
                    $(document).one("mouseup", $.proxy(this.cancelReorder, this));
                    return false;
                }
            },

            overReorder: function(event) {
                if (this.idOfReorderingColumn != event.currentTarget.className.match(new RegExp(WIDTH_CLASS_NAME_BASE + "([^\\W]*)"))[1]) {
                    var eventElement = $(event.currentTarget);
                    var offset = eventElement.offset();
                    $(this.reorderMarkerElement).setPosition({top:offset.top + eventElement.height(), left:offset.left - 5});
                    this.reorderMarkerElement.style.display = "block";
                    eventElement.one("mouseout", $.proxy(this.outReorder, this));
                    eventElement.one("mouseup", $.proxy(this.endReorder, this));
                }
            },

            outReorder: function(event) {
                this.reorderMarkerElement.style.display = "";
                $(event.currentTarget).unbind("mouseup", this.endReorder);
            },

            endReorder: function(event) {
                this.reorderMarkerElement.style.display = "";
                $(event.currentTarget).unbind("mouseout", this.outReorder);
                var id = event.currentTarget.className.match(new RegExp(WIDTH_CLASS_NAME_BASE + "([^\\W]*)"))[1];
                var colunmsOrder = "";
                var _this = this;
                this.headerCells.each(function() {
                    var i = this.className.match(new RegExp(WIDTH_CLASS_NAME_BASE + "([^\\W]*)"))[1];
                    if (i == id) {
                        colunmsOrder += _this.idOfReorderingColumn + "," + id + ",";
                    } else if (i != _this.idOfReorderingColumn) {
                        colunmsOrder += i + ",";
                    }
                });
                this.ajaxFunction(event, {"rich:columnsOrder" : colunmsOrder}); // TODO Maybe, event model should be used here.
            },

            cancelReorder: function(event) {
                $(document).unbind("mousemove", this.reorder);
                this.headerCells.unbind("mouseover", this.overReorder);
                this.reorderElement.style.display = "none";
            },

            loadData: function(event) {
                var clientFirst = Math.round((this.bodyElement.scrollTop + this.bodyElement.clientHeight / 2) / this.rowHeight - this.rows / 2);
                if (clientFirst <= 0) {
                    clientFirst = 0;
                } else {
                    clientFirst = Math.min(this.rowCount - this.rows, clientFirst);
                }
                this.ajaxFunction(event, {"rich:clientFirst" : clientFirst});// TODO Maybe, event model should be used here.
            },

            bodyScrollListener: function(event) {
                if (this.timeoutId) {
                    window.clearTimeout(this.timeoutId);
                    this.timeoutId = null;
                }
                if (Math.max(event.currentTarget.scrollTop - this.rowHeight, 0) < this.spacerElement.offsetHeight
                    || Math.min(event.currentTarget.scrollTop + this.rowHeight + event.currentTarget.clientHeight, event.currentTarget.scrollHeight) > this.spacerElement.offsetHeight + this.dataTableElement.offsetHeight) {
                    var _this = this;
                    this.timeoutId = window.setTimeout(function (event) {
                        _this.loadData(event)
                    }, 1000);
                }
            },

            showActiveRow: function() {
                if (this.bodyElement.scrollTop > this.activeIndex * this.rowHeight + this.spacerElement.offsetHeight) { //UP
                    this.bodyElement.scrollTop = Math.max(this.bodyElement.scrollTop - this.rowHeight, 0);
                } else if (this.bodyElement.scrollTop + this.bodyElement.clientHeight
                    < (this.activeIndex + 1) * this.rowHeight + this.spacerElement.offsetHeight) { //DOWN
                    this.bodyElement.scrollTop = Math.min(this.bodyElement.scrollTop + this.rowHeight, this.bodyElement.scrollHeight - this.bodyElement.clientHeight);
                }
            },

            selectRow: function(index) {
                this.ranges.add(index);
                for (var i = 0; i < this.tbodies.length; i++) {
                    $(this.tbodies[i].rows[index]).addClass("rf-edt-r-sel");
                }
            },

            deselectRow: function (index) {
                this.ranges.remove(index);
                for (var i = 0; i < this.tbodies.length; i++) {
                    $(this.tbodies[i].rows[index]).removeClass("rf-edt-r-sel");
                }
            },

            setActiveRow: function (index) {
                if (typeof this.activeIndex == "number") {
                    for (var i = 0; i < this.tbodies.length; i++) {
                        $(this.tbodies[i].rows[this.activeIndex]).removeClass("rf-edt-r-act");
                    }

                }
                this.activeIndex = index;
                for (var i = 0; i < this.tbodies.length; i++) {
                    $(this.tbodies[i].rows[this.activeIndex]).addClass("rf-edt-r-act");
                }
            },

            resetShiftRow: function () {
                if (typeof this.shiftIndex == "number") {
                    for (var i = 0; i < this.tbodies.length; i++) {
                        $(this.tbodies[i].rows[this.shiftIndex]).removeClass("rf-edt-r-sht");
                    }

                }
                this.shiftIndex = null;
            },

            setShiftRow: function (index) {
                this.resetShiftRow();
                this.shiftIndex = index;
                if (typeof index == "number") {
                    for (var i = 0; i < this.tbodies.length; i++) {
                        $(this.tbodies[i].rows[this.shiftIndex]).addClass("rf-edt-r-sht");
                    }
                }
            },

            initializeSelection: function() {
                this.ranges.clear();
                var strings = this.selectionInput.value.split("|");
                this.activeIndex = strings[1] || null;
                this.shiftIndex = strings[2] || null;
                this.selectionFlag = null;
                var rows = this.tbodies[0].rows;
                for (var i = 0; i < rows.length; i++) {
                    var row = $(rows[i]);
                    if (row.hasClass("rf-edt-r-sel")) {
                        this.ranges.add(row[0].rowIndex)
                    }
                    if (row.hasClass("rf-edt-r-act")) {
                        this.activeIndex = row[0].rowIndex;
                    }
                    if (row.hasClass("rf-edt-r-sht")) {
                        this.shiftIndex = row[0].rowIndex;
                    }
                }
                this.writeSelection();
            },

            writeSelection: function() {
                this.selectionInput.value = [this.ranges, this.activeIndex, this.shiftIndex, this.selectionFlag].join("|");
            },

            selectRows: function(range) {
                if (typeof range == "number") {
                    range = [range, range];
                }
                var changed;
                var i = 0;
                for (; i < range[0]; i++) {
                    if (this.ranges.contains(i)) {
                        this.deselectRow(i);
                        changed = true;
                    }
                }
                for (; i <= range[1]; i++) {
                    if (!this.ranges.contains(i)) {
                        this.selectRow(i);
                        changed = true;
                    }
                }
                for (; i < this.rows; i++) {
                    if (this.ranges.contains(i)) {
                        this.deselectRow(i);
                        changed = true;
                    }
                }
                this.selectionFlag = typeof this.shiftIndex == "string" ? this.shiftIndex : "x";
                return changed;
            },

            processSlectionWithShiftKey: function(index) {
                if (this.shiftIndex == null) {
                    this.setShiftRow(this.activeIndex != null ? this.activeIndex : index);
                }
                var range;
                if ("u" == this.shiftIndex) {
                    range = [0, index];
                } else if ("d" == this.shiftIndex) {
                    range = [index, this.rows - 1];
                } else if (index >= this.shiftIndex) {
                    range = [this.shiftIndex, index];
                } else {
                    range = [index, this.shiftIndex];
                }
                return this.selectRows(range);
            },

            onbeforeselectionchange: function (event) {
                return !this.options.onbeforeselectionchange || this.options.onbeforeselectionchange.call(this.element, event) != false;
            },

            onselectionchange: function (event, index, changed) {
                if (!event.shiftKey) {
                    this.resetShiftRow();
                }
                if (this.activeIndex != index) {
                    this.setActiveRow(index);
                    this.showActiveRow();
                }
                if (changed) {
                    this.writeSelection();
                    if (this.options.onselectionchange) {
                        this.options.onselectionchange.call(this.element, event);
                    }
                }
            },

            selectionClickListener: function (event) {
                if (!this.onbeforeselectionchange(event)) {
                    return;
                }
                var changed;
                if (event.shiftKey || event.ctrlKey) {
                    if (window.getSelection) { //TODO Try to find other way.
                        window.getSelection().removeAllRanges();
                    } else if (document.selection) {
                        document.selection.empty();
                    }
                }
                var tr = event.target;
                while (this.tbodies.index(tr.parentNode) == -1) {
                    tr = tr.parentNode;
                }
                var index = tr.rowIndex;
                if (typeof(index) === 'undefined') {
                    return;
                }
                if (this.options.selectionMode == "single" || (this.options.selectionMode != "multipleKeyboardFree"
                    && !event.shiftKey && !event.ctrlKey)) {
                    changed = this.selectRows(index);
                } else if (this.options.selectionMode == "multipleKeyboardFree" || (!event.shiftKey && event.ctrlKey)) {
                    if (this.ranges.contains(index)) {
                        this.deselectRow(index);
                    } else {
                        this.selectRow(index);
                    }
                    changed = true;
                } else {
                    changed = this.processSlectionWithShiftKey(index);
                }
                this.onselectionchange(event, index, changed);
            },

            selectionKeyDownListener: function(event) {
                if (event.ctrlKey && this.options.selectionMode != "single" && (event.keyCode == 65 || event.keyCode == 97) //Ctrl-A
                    && this.onbeforeselectionchange(event)) {
                    this.selectRows([0, this.rows]);
                    this.selectionFlag = "a";
                    this.onselectionchange(event, this.activeIndex, true); //TODO Is there a way to know that selection haven't changed?
                    event.preventDefault();
                } else {
                    var index;
                    if (event.keyCode == 38) { //UP
                        index = -1;
                    } else if (event.keyCode == 40) { //DOWN
                        index = 1;
                    }
                    if (index != null && this.onbeforeselectionchange(event)) {
                        if (typeof this.activeIndex == "number") {
                            index += this.activeIndex;
                            if (index >= 0 && index < this.rows) {
                                var changed;
                                if (this.options.selectionMode == "single" || (!event.shiftKey && !event.ctrlKey)) {
                                    changed = this.selectRows(index);
                                } else if (event.shiftKey) {
                                    changed = this.processSlectionWithShiftKey(index);
                                }
                                this.onselectionchange(event, index, changed);
                            }
                        }
                        event.preventDefault();
                    }
                }
            },

            ajaxComplete: function (event, data) {
                this.storeDomReferences();
                if (data.reinitializeHeader) {
                    this.bindHeaderHandlers();
                    this.updateLayout();
                } else {
                    this.selectionInput = document.getElementById(this.id + ":si");
                    if (data.reinitializeBody) {
                        this.rowCount = data.rowCount;
                        this.initialize();
                    } else if (this.options.selectionMode != "none") {
                        this.initializeSelection();
                    }
                    if (this.spacerElement) {
                        this.spacerElement.style.height = (data.first * this.rowHeight) + "px";
                    }
                }
                
                // resize columns if necessary
                var $table = $(document.getElementById(this.element.id)),
                    widthsArray = new Array();
                for (var id in this.newWidths) {
                    if (this.newWidths.hasOwnProperty(id)) {
                        $table.find("." + WIDTH_CLASS_NAME_BASE + id).css('width', this.newWidths[id])
                            .parent().css('width', this.newWidths[id]);
                        widthsArray.push(id + ":" + this.newWidths[id]);
                    }
                }
                this.widthInput.value = widthsArray.toString();
                this.updateLayout();
                this.adjustResizers();
            },

            activateResizeListener: function() {
                if (typeof this.resizeEventName !== "undefined") {
                    $(window).on(this.resizeEventName, $.proxy(this.updateLayout, this));
                }
            },

            deActivateResizeListener: function() {
                if (typeof this.resizeEventName !== "undefined") {
                    $(window).off(this.resizeEventName);
                }
            },

            __getMenuSelector: function (menu) {
                var selector = "[id='" + this.element.id + "'] ";
                selector += (typeof menu.options.targetSelector === 'undefined')
                    ?  ".rf-edt-b td" : menu.options.targetSelector;
                selector = $.trim(selector);
                return selector;
            },

            contextMenuAttach: function (menu) {
                var selector = this.__getMenuSelector(menu);
                rf.Event.bind(selector, menu.options.showEvent, $.proxy(menu.__showHandler, menu), menu);
            },

            contextMenuDetach: function (menu) {
                var selector = this.__getMenuSelector(menu);
                rf.Event.unbind(selector, menu.options.showEvent);
            },

            contextMenuShow: function (menu, event) {
                var tr = event.target;
                while (this.tbodies.index(tr.parentNode) == -1) {
                    tr = tr.parentNode;
                }
                var index = tr.rowIndex;
                if (! this.ranges.contains(index) ) {
                    this.selectionClickListener(event);
                }
            }
        });

    var $super = rf.ui.ExtendedDataTable.$super;
}(RichFaces.jQuery, window.RichFaces));
;

// resource: org.richfaces:hotkey.js
/**
 * @author Ilya Shaikovsky
 * @author Lukas Fryc
 */

(function($, rf) {

    rf.ui = rf.ui || {};

    var defaultOptions = {
        enabledInInput : false,
        preventDefault : true
    };
    
    var types = [ 'keydown', 'keyup' ];

    rf.ui.HotKey = function(componentId, options) {
        $super.constructor.call(this, componentId);
        this.namespace = this.namespace || "." + rf.Event.createNamespace(this.name, this.id);
        this.attachToDom(this.componentId);
        this.options = $.extend({}, defaultOptions, options);
        this.__handlers = {};
        
        this.options.selector = (this.options.selector) ? this.options.selector : document;

        $(document).ready($.proxy(function() {
            this.__bindDefinedHandlers();
        }, this));
    };

    rf.BaseComponent.extend(rf.ui.HotKey);

    var $super = rf.ui.HotKey.$super;

    $.extend(rf.ui.HotKey.prototype, {

        name : "HotKey",
        
        __bindDefinedHandlers : function() {
            for (var i = 0; i < types.length; i++) {
                if (this.options['on' + types[i]]) {
                    this.__bindHandler(types[i]);
                }
            }
        },
        
        __bindHandler : function(type) {
            this.__handlers[type] = $.proxy(function(event) {
                var result = this.invokeEvent.call(this, type, document.getElementById(this.id), event);
                if (this.options.preventDefault) {
                    event.stopPropagation();
                    event.preventDefault();
                    return false;
                }
                return result;
            }, this);
            $(this.options.selector).bind(type + this.namespace, this.options, this.__handlers[type]);
        },

        destroy : function() {
            rf.Event.unbindById(this.id, this.namespace);

            for (var type in this.__handlers) {
                if (this.__handlers.hasOwnProperty(type)) {
                    $(this.options.selector).unbind(type + this.namespace, this.__handlers[type]);
                }
            }

            $super.destroy.call(this);
        }
    });

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:panelMenuItem.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */


(function ($, rf) {

    rf.ui = rf.ui || {};

    var __DEFAULT_OPTIONS = {
        disabled : false,
        selectable: true,
        unselectable: false,
        mode: "client",
        stylePrefix: "rf-pm-itm",
        itemStep: 20
    };

    var SELECT_ITEM = {

        /**
         *
         * @return {void}
         * */
        exec : function (item) {

            if (item.expanded) {
                var flag = item.options.expandEvent == item.options.collapseEvent && item.options.collapseEvent == "click";
                if (flag && item.__fireEvent("beforeswitch") == false) return false;
                if (!item.expanded()) {
                    if (item.options.expandEvent == "click" && item.__fireEvent("beforeexpand") == false) return false;
                } else {
                    if (item.options.collapseEvent == "click" && item.__fireEvent("beforecollapse") == false) return false;
                }
            }

            var mode = item.mode;
            if (mode == "server") {
                return this.execServer(item);
            } else if (mode == "ajax") {
                return this.execAjax(item);
            } else if (mode == "client" || mode == "none") {
                return this.execClient(item);
            } else {
                rf.log.error("SELECT_ITEM.exec : unknown mode (" + mode + ")");
            }
        },

        /**
         * @protected
         *
         * @return {Boolean} false
         * */
        execServer : function (item) {
            item.__changeState();
            //TODO nick - 'target' attribute?

            var params = {};
            params[item.__panelMenu().id] = item.itemName; // TODO
            params[item.id] = item.id;

            $.extend(params, item.options["ajax"]["parameters"] || {});

            rf.submitForm(this.__getParentForm(item), params);

            return false;
        },

        /**
         * @protected
         *
         * @return {Boolean} false
         * */
        execAjax : function (item) {
            var oldItem = item.__changeState();
            //TODO nick - check for interaction with queue
            rf.ajax(item.id, null, $.extend({}, item.options["ajax"], {}));
            item.__restoreState(oldItem);

            return true;
        },

        /**
         * @protected
         *
         * @return {undefined}
         *             - false - if process has been terminated
         *             - true  - in other cases
         * */
        execClient : function (item) {
            var panelMenu = item.__rfPanelMenu();
            var prevItem = panelMenu.getSelectedItem();
            if (prevItem) {
                prevItem.unselect();
            }

            panelMenu.selectedItem(item.itemName);

            item.__select();
            var result = item.__fireSelect();

            if (item.__switch) {
                var mode = item.mode;
                if (mode == "client" || mode == "none") {
                    item.__switch(!item.expanded());
                }
            }

            return result;
        },

        /**
         * @private
         * */
        __getParentForm : function (item) {
            return $($(rf.getDomElement(item.id)).parents("form")[0]);
        }
    };

    rf.ui.PanelMenuItem = rf.BaseComponent.extendClass({
            // class name
            name:"PanelMenuItem",

            /**
             * @class PanelMenuItem
             * @name PanelMenuItem
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                $super.constructor.call(this, componentId);
                var rootElt = $(this.attachToDom());

                this.options = $.extend(this.options, __DEFAULT_OPTIONS, options || {});

                this.mode = this.options.mode;
                this.itemName = this.options.name;
                var panelMenu = this.__rfPanelMenu();
                panelMenu.addItem(this);

                // todo move it
                this.selectionClass = this.options.stylePrefix + "-sel";

                if (!this.options.disabled) {
                    var item = this;

                    if (this.options.selectable) {
                        this.__header().bind("click", function() {
                            if (item.__rfPanelMenu().selectedItem() == item.id) {
                                if (item.options.unselectable) {
                                    return item.unselect();
                                }

                                // we shouldn't select one item several times
                            } else {
                                return item.select();
                            }
                        });
                    }
                }

                item = this;
                $(this.__panelMenu()).ready(function () {
                    item.__renderNestingLevel();
                });

                this.__addUserEventHandler("select");
                this.__addUserEventHandler("beforeselect");
            },

            /***************************** Public Methods  ****************************************************************/

            selected : function () {
                return this.__header().hasClass(this.selectionClass);
            },

            /**
             * @methodOf
             * @name PanelMenuItem#select
             *
             * TODO ...
             *
             * @return {void} TODO ...
             */
            select: function () {
                var continueProcess = this.__fireBeforeSelect();
                if (!continueProcess) {
                    return false;
                }

                return SELECT_ITEM.exec(this)
            },

            /**
             * please, remove this method when client side ajax events will be added
             *
             * */
            onCompleteHandler : function () {
                SELECT_ITEM.execClient(this);
            },

            unselect: function () {
                var panelMenu = this.__rfPanelMenu();
                if (panelMenu.selectedItem() == this.itemName) {
                    panelMenu.selectedItem(null);
                } else {
                    rf.log.warn("You tried to unselect item (name=" + this.itemName + ") that isn't seleted")
                }

                this.__unselect();

                return this.__fireUnselect();
            },

            /***************************** Private Methods ****************************************************************/
            __rfParentItem : function () {
                var res = this.__item().parents(".rf-pm-gr")[0];
                if (!res) {
                    res = this.__item().parents(".rf-pm-top-gr")[0];
                }

                if (!res) {
                    res = this.__panelMenu();
                }

                return res ? rf.component(res) : null;
            },

            __getNestingLevel : function () {
                if (!this.nestingLevel) {
                    var parentItem = this.__rfParentItem();
                    if (parentItem && parentItem.__getNestingLevel) {
                        this.nestingLevel = parentItem.__getNestingLevel() + 1;
                    } else {
                        this.nestingLevel = 0;
                    }
                }

                return this.nestingLevel;
            },

            __renderNestingLevel : function () {
                this.__item().find("td").first().css("padding-left", this.options.itemStep * this.__getNestingLevel());
            },

            __panelMenu : function () {
                return this.__item().parents(".rf-pm")[0];
            },

            __rfPanelMenu : function () {
                return rf.component(this.__panelMenu());
            },

            __changeState : function () {
                return this.__rfPanelMenu().selectedItem(this.itemName);
            },

            __restoreState : function (state) {
                if (state) {
                    this.__rfPanelMenu().selectedItem(state);
                }
            },

            __item : function () {
                return $(rf.getDomElement(this.id));
            },

            __header : function () {
                return this.__item();
            },

            __isSelected: function() {
                return this.__header().hasClass(this.selectionClass);
            },

            __select: function () {
                this.__header().addClass(this.selectionClass);
            },

            __unselect: function () {
                this.__header().removeClass(this.selectionClass);
            },

            __fireBeforeSelect : function () {
                return rf.Event.fireById(this.id, "beforeselect", {
                        item: this
                    });
            },

            __fireSelect : function () {
                return rf.Event.fireById(this.id, "select", {
                        item: this
                    });
            },

            __fireUnselect : function () {
                return rf.Event.fireById(this.id, "unselect", {
                        item: this
                    });
            },

            __fireEvent : function (eventType, event) {
                return this.invokeEvent(eventType, rf.getDomElement(this.id), event, {id: this.id, item: this});
            },

            /**
             * @private
             * */
            __addUserEventHandler : function (name) {
                var handler = this.options["on" + name];
                if (handler) {
                    rf.Event.bindById(this.id, name, handler);
                }
            },

            __rfTopGroup : function () {
                var res = this.__item().parents(".rf-pm-top-gr")[0];
                return res ? res : null;
            },

            destroy: function () {
                var panelMenu = this.__rfPanelMenu();
                if (panelMenu) {
                    panelMenu.deleteItem(this);
                }

                $super.destroy.call(this);
            }
        });

    // define super class link
    var $super = rf.ui.PanelMenuItem.$super;
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:notify.js
(function($, rf) {

    rf.ui = rf.ui || {};
    
    var defaultOptions = {
        styleClass: '',
        nonblocking: false,
        nonblockingOpacity: 0.2,
        showHistory: false,
        animationSpeed: 'slow',
        opacity: '1',
        showShadow: false,
        showCloseButton: true,
        appearAnimation: 'fade',
        hideAnimation: 'fade',
        sticky: false,
        stayTime: 8000,
        delay: 0
    };
    
    var defaultStackId = "org.richfaces.notifyStack.default";
    
    var events = "click dblclick  keydown keypress keyup mousedown mousemove mouseout mouseover mouseup";
    
    var propertyTranslation = {
        'summary':'pnotify_title',
        'detail': 'pnotify_text',
        'styleClass': 'pnotify_addclass',
        'nonblocking': 'pnotify_nonblock',
        'nonblockingOpacity': 'pnotify_nonblock_opacity',
        'showHistory': 'pnotify_history',
        'animation': 'pnotify_animation',
        'appearAnimation': 'effect_in',
        'hideAnimation': 'effect_out',
        'animationSpeed': 'pnotify_animate_speed',
        'opacity': 'pnotify_opacity',
        'showShadow': 'pnotify_shadow',
        'showCloseButton': 'pnotify_closer',
        'sticky': 'pnotify_hide',
        'stayTime': 'pnotify_delay'
    };
    
    var severityClasses = ["rf-ntf-inf", "rf-ntf-wrn", "rf-ntf-err", "rf-ntf-ftl"];
    
    var translateProperties = function(target, source, translation) {
        for (var attr in source) {
            if (source.hasOwnProperty(attr)) {
                var targetAttr = translation[attr] != null ? translation[attr] : attr;
                target[targetAttr] = source[attr];
                if (target[targetAttr] instanceof Object) {
                    target[targetAttr] = $.extend({}, target[targetAttr], translation);
                }
            }
        }
        return target;
    };
    
    var getDefaultStack = function() {
        if (!document.getElementById(defaultStackId)) {
            var stackElement = $('<span id="' + defaultStackId + '" class="rf-ntf-stck" />');
            $('body').append(stackElement);
            new rf.ui.NotifyStack(defaultStackId);
        }
        return getStack(defaultStackId);
    };
    
    var getStack = function(stackId) {
        if (!stackId) {
            return getDefaultStack();
        }
        return rf.component(stackId).getStack();
    };
    

    var array_remove = function(array, from, to) {
        var rest = array.slice((to || from) + 1 || array.length);
        array.length = from < 0 ? array.length + from : from;
        return array.push.apply(array, rest);
    };
    
    rf.ui.Notify = function(options) {
        var options = $.extend({}, defaultOptions, options);
        
        if (typeof options.severity == "number") {
            var severity = severityClasses[options.severity];
            options.styleClass = options.styleClass ? severity + " " + options.styleClass : severity;
        }
        
        options.showCloseButton = options.sticky || options.showCloseButton;
        
        var pnotifyOptions = translateProperties({}, options, propertyTranslation);

        var display = function() {
            var stack = getStack(options.stackId);
            pnotifyOptions.pnotify_stack = stack;
            pnotifyOptions.pnotify_addclass += ' rf-ntf-pos-' + stack.position;
            pnotifyOptions.pnotify_after_close = function(pnotify) {
                var index = $.inArray(pnotify, stack.notifications);
                if (index >= 0) {
                    array_remove(stack.notifications, index);
                }
            }
            var pnotify = $.pnotify(pnotifyOptions);
            pnotify.on(events, function(e) {
                if (options['on' + e.type]) {
                    options['on' + e.type].call(this, e);
                }
            });
            stack.addNotification(pnotify);
        }
        
        if (options.sticky !== null) {
            pnotifyOptions.pnotify_hide = !options.sticky;
        }
        
        $(document).ready(function() {
            if (options.delay) {
                setTimeout(function() {
                    display();
                }, options.delay);
            } else {
                display();
            }
        });
    };

})(RichFaces.jQuery, RichFaces);;

// resource: com.jqueryui:mouse.js
/*!
 * jQuery UI Mouse 1.11.4
 * http://jqueryui.com
 *
 * Copyright jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * http://api.jqueryui.com/mouse/
 */
(function( factory ) {
	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define([
			"jquery",
			"./widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
}(function( $ ) {

var mouseHandled = false;
$( document ).mouseup( function() {
	mouseHandled = false;
});

return $.widget("ui.mouse", {
	version: "1.11.4",
	options: {
		cancel: "input,textarea,button,select,option",
		distance: 1,
		delay: 0
	},
	_mouseInit: function() {
		var that = this;

		this.element
			.bind("mousedown." + this.widgetName, function(event) {
				return that._mouseDown(event);
			})
			.bind("click." + this.widgetName, function(event) {
				if (true === $.data(event.target, that.widgetName + ".preventClickEvent")) {
					$.removeData(event.target, that.widgetName + ".preventClickEvent");
					event.stopImmediatePropagation();
					return false;
				}
			});

		this.started = false;
	},

	// TODO: make sure destroying one instance of mouse doesn't mess with
	// other instances of mouse
	_mouseDestroy: function() {
		this.element.unbind("." + this.widgetName);
		if ( this._mouseMoveDelegate ) {
			this.document
				.unbind("mousemove." + this.widgetName, this._mouseMoveDelegate)
				.unbind("mouseup." + this.widgetName, this._mouseUpDelegate);
		}
	},

	_mouseDown: function(event) {
		// don't let more than one widget handle mouseStart
		if ( mouseHandled ) {
			return;
		}

		this._mouseMoved = false;

		// we may have missed mouseup (out of window)
		(this._mouseStarted && this._mouseUp(event));

		this._mouseDownEvent = event;

		var that = this,
			btnIsLeft = (event.which === 1),
			// event.target.nodeName works around a bug in IE 8 with
			// disabled inputs (#7620)
			elIsCancel = (typeof this.options.cancel === "string" && event.target.nodeName ? $(event.target).closest(this.options.cancel).length : false);
		if (!btnIsLeft || elIsCancel || !this._mouseCapture(event)) {
			return true;
		}

		this.mouseDelayMet = !this.options.delay;
		if (!this.mouseDelayMet) {
			this._mouseDelayTimer = setTimeout(function() {
				that.mouseDelayMet = true;
			}, this.options.delay);
		}

		if (this._mouseDistanceMet(event) && this._mouseDelayMet(event)) {
			this._mouseStarted = (this._mouseStart(event) !== false);
			if (!this._mouseStarted) {
				event.preventDefault();
				return true;
			}
		}

		// Click event may never have fired (Gecko & Opera)
		if (true === $.data(event.target, this.widgetName + ".preventClickEvent")) {
			$.removeData(event.target, this.widgetName + ".preventClickEvent");
		}

		// these delegates are required to keep context
		this._mouseMoveDelegate = function(event) {
			return that._mouseMove(event);
		};
		this._mouseUpDelegate = function(event) {
			return that._mouseUp(event);
		};

		this.document
			.bind( "mousemove." + this.widgetName, this._mouseMoveDelegate )
			.bind( "mouseup." + this.widgetName, this._mouseUpDelegate );

		event.preventDefault();

		mouseHandled = true;
		return true;
	},

	_mouseMove: function(event) {
		// Only check for mouseups outside the document if you've moved inside the document
		// at least once. This prevents the firing of mouseup in the case of IE<9, which will
		// fire a mousemove event if content is placed under the cursor. See #7778
		// Support: IE <9
		if ( this._mouseMoved ) {
			// IE mouseup check - mouseup happened when mouse was out of window
			if ($.ui.ie && ( !document.documentMode || document.documentMode < 9 ) && !event.button) {
				return this._mouseUp(event);

			// Iframe mouseup check - mouseup occurred in another document
			} else if ( !event.which ) {
				return this._mouseUp( event );
			}
		}

		if ( event.which || event.button ) {
			this._mouseMoved = true;
		}

		if (this._mouseStarted) {
			this._mouseDrag(event);
			return event.preventDefault();
		}

		if (this._mouseDistanceMet(event) && this._mouseDelayMet(event)) {
			this._mouseStarted =
				(this._mouseStart(this._mouseDownEvent, event) !== false);
			(this._mouseStarted ? this._mouseDrag(event) : this._mouseUp(event));
		}

		return !this._mouseStarted;
	},

	_mouseUp: function(event) {
		this.document
			.unbind( "mousemove." + this.widgetName, this._mouseMoveDelegate )
			.unbind( "mouseup." + this.widgetName, this._mouseUpDelegate );

		if (this._mouseStarted) {
			this._mouseStarted = false;

			if (event.target === this._mouseDownEvent.target) {
				$.data(event.target, this.widgetName + ".preventClickEvent", true);
			}

			this._mouseStop(event);
		}

		mouseHandled = false;
		return false;
	},

	_mouseDistanceMet: function(event) {
		return (Math.max(
				Math.abs(this._mouseDownEvent.pageX - event.pageX),
				Math.abs(this._mouseDownEvent.pageY - event.pageY)
			) >= this.options.distance
		);
	},

	_mouseDelayMet: function(/* event */) {
		return this.mouseDelayMet;
	},

	// These are placeholder methods, to be overriden by extending plugin
	_mouseStart: function(/* event */) {},
	_mouseDrag: function(/* event */) {},
	_mouseStop: function(/* event */) {},
	_mouseCapture: function(/* event */) { return true; }
});

}));
;

// resource: org.richfaces:inputBase.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.InputBase = function(id, options) {
        $super.constructor.call(this, id);
        this.namespace = this.getNamespace() || "." + rf.Event.createNamespace(this.getName(), this.getId());

        this.namespace = this.namespace || "." + rf.Event.createNamespace(this.name, this.id);

        this.input = $(document.getElementById(id + "Input"));
        this.attachToDom();

        var inputEventHandlers = {};
        inputEventHandlers["keydown" + this.namespace] = $.proxy(this.__keydownHandler, this);
        inputEventHandlers["blur" + this.namespace] = $.proxy(this.__blurHandler, this);
        inputEventHandlers["change" + this.namespace] = $.proxy(this.__changeHandler, this);
        inputEventHandlers["focus" + this.namespace] = $.proxy(this.__focusHandler, this);
        rf.Event.bind(this.input, inputEventHandlers, this);
    };

    rf.BaseComponent.extend(rf.ui.InputBase);

    // define super class link
    var $super = rf.ui.InputBase.$super;

    $.extend(rf.ui.InputBase.prototype, ( function () {

        return {

            name : "inputBase",


            getName: function() {
                return this.name;
            },

            getNamespace: function() {
                return this.namespace;
            },

            __focusHandler: function(e) {
            },

            __keydownHandler: function(e) {
            },

            __blurHandler: function(e) {
            },

            __changeHandler: function(e) {
            },

            __setInputFocus: function() {
                this.input.focus();
            },

            __getValue: function() {
                return this.input.val();
            },

            __setValue: function(value) {
                this.input.val(value);
                if (this.defaultLabelClass) {
                    if (this.defaultLabel && value == this.defaultLabel) {
                        this.input.addClass(this.defaultLabelClass);
                    } else {
                        this.input.removeClass(this.defaultLabelClass);
                    }
                }
            },

            getValue: function() {
                return this.__getValue();
            },

            setValue: function(value) {
                this.__setValue(value);
            },

            getInput: function() {
                return this.input;
            },

            getId: function() {
                return this.id;
            },

            focus: function() {
                this.__setInputFocus();
            },

            destroy: function() {
                rf.Event.unbindById(this.input, this.namespace);
                this.input = null;
                $super.destroy.call(this);
            }
        }
    })());

})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:push.js
(function($, rf, jsf) {


  /**
   * Global object accessing general state of Push functionality.
   *
   * Push internally uses Atmosphere to handle connection.
   *
   * @module RichFaces.push
   */
  rf.push = {

    options: {
      transport: "websocket",
      fallbackTransport: "long-polling",
      logLevel: "info"
    },

    /**
     * topics that we subscribed to in current atmosphere connection
     */
    _subscribedTopics: {},

    /**
     * topics to be added to subscription during next invocation updateConnection
     */
    _addedTopics: {},

    /**
     * topics to be removed from subscription during next invocation updateConnection
     */
    _removedTopics: {},

    /**
     * Object that holds mapping of addresses to number of subscribed widgets
     *
     * { address: counter }
     *
     * If counter reaches 0, it is safe to unsubscribe given address.
     */
    _handlersCounter: {},

    /**
     * current Push session identifier
     */
    _pushSessionId: null,
    /**
     * sequence number which identifies which messages were already processed
     */
    _lastMessageNumber: -1,

    /**
     * the URL that handles Push subscriptions
     */
    _pushResourceUrl: null,
    /**
     * the URL that handles Atmosphere requests
     */
    _pushHandlerUrl: null,

    /**
     * Checks whether there are requests for adding or removing topics to the list of subscribed topics.
     *
     * If yes, it reconnects Atmosphere connection.
     *
     * @method updateConnection
     */
    updateConnection: function() {
      if ($.isEmptyObject(this._handlersCounter)) {
        this._disconnect();
      } else if (!$.isEmptyObject(this._addedTopics) || !$.isEmptyObject(this._removedTopics)) {
        this._disconnect();
        this._connect();
      }

      this._addedTopics = {};
      this._removedTopics = {};
    },

    /**
     * Increases number of subscriptions to given topic
     *
     * @method increaseSubscriptionCounters
     * @param topic
     */
    increaseSubscriptionCounters: function(topic) {
      if (isNaN(this._handlersCounter[topic]++)) {
        this._handlersCounter[topic] = 1;
        this._addedTopics[topic] = true;
      }
    },

    /**
     * Decreases number of subscriptions to given topic
     *
     * @method decreaseSubscriptionCounters
     * @param topic
     */
    decreaseSubscriptionCounters: function(topic) {
      if (--this._handlersCounter[topic] == 0) {
        delete this._handlersCounter[topic];
        this._removedTopics[topic] = true;
        this._subscribedTopics[topic] = false;
      }
    },

    /**
     * Setups the URL that handles Push subscriptions
     *
     * @method setPushResourceUrl
     * @param url
     */
    setPushResourceUrl: function(url) {
      this._pushResourceUrl = qualifyUrl(url);
    },

    /**
     * Setups the URL that handles Atmosphere requests
     *
     * @method setPushHandlerUrl
     * @param url
     */
    setPushHandlerUrl: function(url) {
      this._pushHandlerUrl = qualifyUrl(url);
    },

    /**
     * Handles messages transported using Atmosphere
     */
    _messageCallback: function(response) {
      if (response.state && response.state === "opening") {
          this._lastMessageNumber = -1;
          return;
      }
      var suspendMessageEndMarker = /^(<!--[^>]+-->\s*)+/;
      var messageTokenExpr = /<msg topic="([^"]+)" number="([^"]+)">([^<]*)<\/msg>/g;

      var dataString = response.responseBody.replace(suspendMessageEndMarker, "");
      if (dataString) {
        var messageToken;
        while (messageToken = messageTokenExpr.exec(dataString)) {
          if (!messageToken[1]) {
            continue;
          }

          var message = {
              topic: messageToken[1],
              number: parseInt(messageToken[2]),
              data: $.parseJSON(messageToken[3])
          };

          if (message.number <= this._lastMessageNumber) {
            continue;
          }

          var event = new jQuery.Event('push.push.RICH.' + message.topic, { rf: { data: message.data } });

          (function(event) {
            $(function() {
              $(document).trigger(event);
            });
          })(event);

          this._lastMessageNumber = message.number;
        }
      }
    },

    /**
     * Handles errors during Atmosphere initialization and transport
     */
    _errorCallback: function(response) {
      for (var address in this.newlySubcribed) {
        if (this.newlySubcribed.hasOwnProperty(address)) {
          this._subscribedTopics[address] = true;
          $(document).trigger('error.push.RICH.' + address, response);
        }
      }
    },

    /**
     * Initializes Atmosphere connection
     */
    _connect: function() {
      this.newlySubcribed = {};

      var topics = [];
      for (var address in this._handlersCounter) {
        if (this._handlersCounter.hasOwnProperty(address)) {
          topics.push(address);
          if (!this._subscribedTopics[address]) {
            this.newlySubcribed[address] = true;
          }
        }
      }

      var data = {
        'pushTopic': topics
      };

      if (this._pushSessionId) {
        data['forgetPushSessionId'] = this._pushSessionId;
      }

      //TODO handle request errors
      $.ajax({
        data: data,
        dataType: 'text',
        traditional: true,
        type: 'POST',
        url: this._pushResourceUrl,
        success: $.proxy(function(response) {
          var data = $.parseJSON(response);

          for (var address in data.failures) {
            $(document).trigger('error.push.RICH.' + address);
          }

          if (data.sessionId) {
            this._pushSessionId = data.sessionId;

            var url = this._pushHandlerUrl || this._pushResourceUrl;
            url += "?__richfacesPushAsync=1&pushSessionId="
            url += this._pushSessionId;

            var messageCallback = $.proxy(this._messageCallback, this);
            var errorCallback = $.proxy(this._errorCallback, this);

            $.atmosphere.subscribe(url, messageCallback, {
              transport: this.options.transport,
              fallbackTransport: this.options.fallbackTransport,
              logLevel: this.options.logLevel,
              onError: errorCallback
            });

            // fire subscribed events
            for (var address in this.newlySubcribed) {
              this._subscribedTopics[address] = true;
              $(document).trigger('subscribed.push.RICH.' + address);
            }
          }
        }, this)
      });
    },

    /**
     * Ends Atmosphere connection
     */
    _disconnect: function() {
      $.atmosphere.unsubscribe();
    }
  };

  /**
   * jQuery plugin which mades easy to bind the Push to the DOM element
   * and manage plugins lifecycle and event handling
   *
   * @function $.fn.richpush
   */

  $.fn.richpush = function( options ) {
    var widget = $.extend({}, $.fn.richpush);

    // for all selected elements
    return this.each(function() {
      widget.element = this;
      widget.options = $.extend({}, widget.options, options);
      widget.eventNamespace = '.push.RICH.' + widget.element.id;

      // call constructor
      widget._create();

      // listen for global DOM destruction event
      $(document).on('beforeDomClean' + widget.eventNamespace, function(event) {
        // is the push component under destructed DOM element (event target)?
        if (event.target && (event.target === widget.element || $.contains(event.target, widget.element))) {
          widget._destroy();
        }
      });
    });
  };

  $.extend($.fn.richpush, {

    options: {

      /**
       * Specifies which address (topic) will Push listen on
       *
       * @property address
       * @required
       */
      address: null,

      /**
       * Fired when Push is subscribed to the specified address
       *
       * @event subscribed
       */
      subscribed: null,

      /**
       * Triggered when Push receives data
       *
       * @event push
       */
      push: null,

      /**
       * Triggered when error is observed during Push initialization or communication
       *
       * @event error
       */
      error: null
    },

    _create: function() {
      var widget = this;

      this.address = this.options.address;

      this.handlers = {
        subscribed: null,
        push: null,
        error: null
      };

      $.each(this.handlers, function(eventName) {
        if (widget.options[eventName]) {

          var handler = function(event, data) {
            if (data) {
              $.extend(event, {
                rf: {
                  data: data
                }
              });
            }

            widget.options[eventName].call(widget.element, event);
          };

          widget.handlers[eventName] = handler;
          $(document).on(eventName + widget.eventNamespace + '.' + widget.address, handler);
        }
      });

      rf.push.increaseSubscriptionCounters(this.address);
    },

    _destroy: function() {
      rf.push.decreaseSubscriptionCounters(this.address);
      $(document).off(this.eventNamespace);
    }

  });

  /*
   * INITIALIZE PUSH
   *
   * when document is ready and when JSF errors happens
   */

  $(document).ready(function() {
    rf.push.updateConnection();
  });

  jsf.ajax.addOnEvent(jsfErrorHandler);
  jsf.ajax.addOnError(jsfErrorHandler);


  /* PRIVATE FUNCTIONS */

  function jsfErrorHandler(event) {
    if (event.type == 'event') {
      if (event.status != 'success') {
        return;
      }
    } else if (event.type != 'error') {
      return;
    }

    rf.push.updateConnection();
  }

  function qualifyUrl(url) {
    var result = url;
    if (url.charAt(0) == '/') {
      result = location.protocol + '//' + location.host + url;
    }
    return result;
  }

}(RichFaces.jQuery, RichFaces, jsf));;

// resource: com.jqueryui:effect-highlight.js
/*!
 * jQuery UI Effects Highlight 1.11.4
 * http://jqueryui.com
 *
 * Copyright jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * http://api.jqueryui.com/highlight-effect/
 */
(function( factory ) {
	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define([
			"jquery",
			"./effect"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
}(function( $ ) {

return $.effects.effect.highlight = function( o, done ) {
	var elem = $( this ),
		props = [ "backgroundImage", "backgroundColor", "opacity" ],
		mode = $.effects.setMode( elem, o.mode || "show" ),
		animation = {
			backgroundColor: elem.css( "backgroundColor" )
		};

	if (mode === "hide") {
		animation.opacity = 0;
	}

	$.effects.save( elem, props );

	elem
		.show()
		.css({
			backgroundImage: "none",
			backgroundColor: o.color || "#ffff99"
		})
		.animate( animation, {
			queue: false,
			duration: o.duration,
			easing: o.easing,
			complete: function() {
				if ( mode === "hide" ) {
					elem.hide();
				}
				$.effects.restore( elem, props );
				done();
			}
		});
};

}));
;

// resource: org.richfaces:AutocompleteBase.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    // Constructor definition
    rf.ui.AutocompleteBase = function(componentId, selectId, fieldId, options) {
        // call constructor of parent class
        $super.constructor.call(this, componentId);
        this.selectId = selectId;
        this.fieldId = fieldId;
        this.options = $.extend({}, defaultOptions, options);
        this.namespace = this.namespace || "." + rf.Event.createNamespace(this.name, this.selectId);
        this.currentValue = $(rf.getDomElement(fieldId)).val();
        this.tempValue = this.getValue();
        this.isChanged = false;
        bindEventHandlers.call(this);
    };

    // Extend component class and add protected methods from parent class to our container
    rf.BaseComponent.extend(rf.ui.AutocompleteBase);

    // define super class link
    var $super = rf.ui.AutocompleteBase.$super;

    var defaultOptions = {
        changeDelay:8
    };

    var bindEventHandlers = function() {

        var inputEventHandlers = {};

        if (this.options.buttonId) {
            inputEventHandlers["mousedown" + this.namespace] = onButtonShow;
            inputEventHandlers["mouseup" + this.namespace] = onSelectMouseUp;
            rf.Event.bindById(this.options.buttonId, inputEventHandlers, this);
        }

        inputEventHandlers = {};
        inputEventHandlers["focus" + this.namespace] = onFocus;
        inputEventHandlers["blur" + this.namespace] = onBlur;
        inputEventHandlers["click" + this.namespace] = onClick;
        inputEventHandlers["keydown" + this.namespace] = onKeyDown;
        inputEventHandlers["change" + this.namespace] = function (event) {
            if (this.focused) {
                event.stopPropagation()
            }
        };
        rf.Event.bindById(this.fieldId, inputEventHandlers, this);

        inputEventHandlers = {};
        inputEventHandlers["mousedown" + this.namespace] = onSelectMouseDown;
        inputEventHandlers["mouseup" + this.namespace] = onSelectMouseUp;
        rf.Event.bindById(this.selectId, inputEventHandlers, this);
    };

    var onSelectMouseDown = function () {
        this.isMouseDown = true;
    };
    var onSelectMouseUp = function () {
        rf.getDomElement(this.fieldId).focus();
    };

    var onButtonShow = function (event) {
        this.isMouseDown = true;
        if (this.timeoutId) {
            window.clearTimeout(this.timeoutId);
            this.timeoutId = null;
        }

        rf.getDomElement(this.fieldId).focus();
        if (this.isVisible) {
            this.__hide(event);
        } else {
            onShow.call(this, event);
        }
    };

    var onFocus = function (event) {
        if (!this.focused) {
            this.__focusValue = this.getValue();
            this.focused = true;
            this.invokeEvent("focus", rf.getDomElement(this.fieldId), event);
        }
    };

    var onBlur = function (event) {
        if (this.isMouseDown) {
            rf.getDomElement(this.fieldId).focus();
            this.isMouseDown = false;
        } else if (!this.isMouseDown) {
            if (this.isVisible) {
                var _this = this;
                this.timeoutId = window.setTimeout(function() {
                    if (_this.isVisible) {
                        _this.__hide(event);
                    }
                }, 200);
            }
            if (this.focused) {
                this.focused = false;
                this.invokeEvent("blur", rf.getDomElement(this.fieldId), event);
                if (this.__focusValue != this.getValue()) {
                    this.invokeEvent("change", rf.getDomElement(this.fieldId), event);
                }
            }
        }
    };

    var onClick = function (event) {
    };

    var onChange = function (event) {
        if (this.isChanged) {
            if (this.getValue() == this.tempValue) return;
        }
        this.isChanged = false;
        var value = this.getValue();
        var flag = value != this.currentValue;
        //TODO: is it needed to chesk keys?
        //TODO: we need to set value when autoFill used when LEFT or RIGHT was pressed
        if (event.keyCode == rf.KEYS.LEFT || event.keyCode == rf.KEYS.RIGHT || flag) {
            if (flag) {
                this.currentValue = this.getValue();
                this.__onChangeValue(event, undefined, (!this.isVisible ? this.__show : undefined));
            } else if (this.isVisible) {
                this.__onChangeValue(event);
            }
        }
    };

    var onShow = function (event) {
        if (this.isChanged) {
            this.isChanged = false;
            onChange.call(this, {});
        } else {
            !this.__updateState(event) && this.__show(event);
        }
    };

    var onKeyDown = function (event) {
        switch (event.keyCode) {
            case rf.KEYS.UP:
                event.preventDefault();
                if (this.isVisible) {
                    this.__onKeyUp(event);
                }
                break;
            case rf.KEYS.DOWN:
                event.preventDefault();
                if (this.isVisible) {
                    this.__onKeyDown(event);
                } else {
                    onShow.call(this, event);
                }
                break;
            case rf.KEYS.PAGEUP:
                if (this.isVisible) {
                    event.preventDefault();
                    this.__onPageUp(event);
                }
                break;
            case rf.KEYS.PAGEDOWN:
                if (this.isVisible) {
                    event.preventDefault();
                    this.__onPageDown(event);
                }
                break;
            case rf.KEYS.HOME:
                if (this.isVisible) {
                    event.preventDefault();
                    this.__onKeyHome(event);
                }
                break;
            case rf.KEYS.END:
                if (this.isVisible) {
                    event.preventDefault();
                    this.__onKeyEnd(event);
                }
                break;
            case rf.KEYS.RETURN:
                if (this.isVisible) {
                    event.preventDefault();
                    this.__onEnter(event);
                    //TODO: bind form submit event handler to cancel form submit under the opera
                    //cancelSubmit = true;
                    this.__hide(event);
                    return false;
                }
                break;
            case rf.KEYS.ESC:
                this.__hide(event);
                break;
            default:
                if (!this.options.selectOnly) {
                    var _this = this;
                    window.clearTimeout(this.changeTimerId);
                    this.changeTimerId = window.setTimeout(function() {
                        onChange.call(_this, event);
                    }, this.options.changeDelay)
                }
                break;
        }
    };

    /*
     * public API functions definition
     */
    var show = function (event) {
        if (!this.isVisible) {
            if (this.__onBeforeShow(event) != false) {
                this.scrollElements = rf.Event.bindScrollEventHandlers(this.selectId, this.__hide, this, this.namespace);
                var element = rf.getDomElement(this.selectId);
                if (this.options.attachToBody) {
                    this.parentElement = element.parentNode;
                    document.body.appendChild(element);
                }
                $(element).setPosition({id: this.fieldId}, {type:"DROPDOWN"}).show();
                this.isVisible = true;
                this.__onShow(event);
            }
        }
    };
    var hide = function (event) {
        if (this.isVisible) {
            this.__conceal();
            this.isVisible = false;
            this.__onHide(event);
        }
    };
    
    var conceal = function () {
        if (this.isVisible) {
            if (this.scrollElements) {
                rf.Event.unbindScrollEventHandlers(this.scrollElements, this);
                this.scrollElements = null;
            }
            $(rf.getDomElement(this.selectId)).hide();
            if (this.options.attachToBody && this.parentElement) {
                this.parentElement.appendChild(rf.getDomElement(this.selectId));
                this.parentElement = null;
            }
        }
    };

    var updateInputValue = function (value) {
        if (this.fieldId) {
            rf.getDomElement(this.fieldId).value = value;
            return value;
        } else {
            return "";
        }
    };

    /*
     * Prototype definition
     */
    $.extend(rf.ui.AutocompleteBase.prototype, (function () {
        return {
            /*
             * public API functions
             */
            name:"AutocompleteBase",
            showPopup: function (event) {
                if (!this.focused) {
                    rf.getDomElement(this.fieldId).focus();
                }
                onShow.call(this, event);
            },
            hidePopup: function (event) {
                this.__hide(event)
            },
            getNamespace: function () {
                return this.namespace;
            },
            getValue: function () {
                return this.fieldId ? rf.getDomElement(this.fieldId).value : "";
            },
            setValue: function (value) {
                if (value == this.currentValue) return;
                updateInputValue.call(this, value);
                this.isChanged = true;
            },
            focus: function () {
                rf.getDomElement(this.fieldId).focus();
            },
            /*
             * Protected methods
             */
            __updateInputValue: updateInputValue,
            __show: show,
            __hide: hide,
            __conceal: conceal,
            /*
             * abstract protected methods
             */
            __onChangeValue: function (event) {
            },
            __onKeyUp: function (event) {
            },
            __onKeyDown: function (event) {
            },
            __onPageUp: function (event) {
            },
            __onPageDown: function (event) {
            },
            __onKeyHome: function (event) {
            },
            __onKeyEnd: function (event) {
            },
            __onBeforeShow: function (event) {
            },
            __onShow: function (event) {
            },
            __onHide: function (event) {
            },
            /*
             * Destructor
             */
            destroy: function () {
                this.parentNode = null;
                if (this.scrollElements) {
                    rf.Event.unbindScrollEventHandlers(this.scrollElements, this);
                    this.scrollElements = null;
                }
                this.options.buttonId && rf.Event.unbindById(this.options.buttonId, this.namespace);
                rf.Event.unbindById(this.fieldId, this.namespace);
                rf.Event.unbindById(this.selectId, this.namespace);
                $super.destroy.call(this);
            }
        };
    })());
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:collapsiblePanel.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.CollapsiblePanel = rf.ui.TogglePanel.extendClass({

            name:"CollapsiblePanel",

            /**
             * @class CollapsiblePanel
             * @name CollapsiblePanel
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                rf.ui.TogglePanel.call(this, componentId, options);
                this.switchMode = options.switchMode;

                this.__addUserEventHandler("beforeswitch");
                this.__addUserEventHandler("switch");

                this.options.cycledSwitching = true;

                var panel = this;
                $(document.getElementById(this.id)).ready(function () { // TODO
                    rf.Event.bindById(panel.id + ":header", "click", panel.__onHeaderClick, panel);

                    new RichFaces.ui.CollapsiblePanelItem(
                        panel.id + ":content", {"index":0, "togglePanelId":panel.id, "switchMode":panel.switchMode, "name":"true"}),

                        new RichFaces.ui.CollapsiblePanelItem(
                            panel.id + ":empty", {"index":1, "togglePanelId":panel.id, "switchMode":panel.switchMode, "name":"false"})
                })
            },

            switchPanel : function (to) {
                this.switchToItem(to || "@next");
            },

            expand: function() {
                this.switchToItem("true");
            },
            
            collapse: function() {
                this.switchToItem("false");
            },
            
            isExpanded: function() {
                return this.activeItem == "true";
            },
            
            /***************************** Private Methods ********************************************************/

            __onHeaderClick : function () {
                this.switchToItem("@next");
            },

            __fireItemChange : function (oldItem, newItem) {
                return new rf.Event.fireById(this.id, "switch", {
                        id: this.id,
                        isExpanded : newItem.getName()
                    });
            },

            __fireBeforeItemChange : function (oldItem, newItem) {
                return rf.Event.fireById(this.id, "beforeswitch", {
                        id: this.id,
                        isExpanded : newItem.getName()
                    });
            }
        });
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:popupPanelSizer.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.PopupPanel.Sizer = function(id, modalPanel, cursor, sizer) {

        $super.constructor.call(this, id);

    };

    var $super = rf.BaseComponent.extend(rf.ui.PopupPanel.Sizer);
    var $super = rf.ui.PopupPanel.Sizer.$super;
    $.extend(rf.ui.PopupPanel.Sizer.prototype, (function (options) {
        return {

            name: "richfaces.ui.PopupPanel.Sizer",

            doSetupSize: function (modalPanel, elt) {
                var width = 0;
                var height = 0;
                var element = $(rf.getDomElement(elt));
                var reductionData = modalPanel.reductionData;

                if (reductionData) {
                    if (reductionData.w) {
                        width = reductionData.w / 2;
                    }

                    if (reductionData.h) {
                        height = reductionData.h / 2;
                    }
                }

                if (width > 0) {
                    if (elt.clientWidth > width) {
                        if (!elt.reducedWidth) {
                            elt.reducedWidth = element.css('width');
                        }
                        element.css('width', width + 'px');
                    } else if (width < 4 && elt.reducedWidth == 4 + 'px') {
                        element.css('width', width + 'px');
                    }
                } else {
                    if (elt.reducedWidth) {
                        element.css('width', elt.reducedWidth);
                        elt.reducedWidth = undefined;
                    }
                }

                if (height > 0) {
                    if (elt.clientHeight > height) {
                        if (!elt.reducedHeight) {
                            elt.reducedHeight = element.css('height');
                        }
                        elt.style.height = height + 'px';
                    } else if (height < 4 && elt.reducedHeight == 4 + 'px') {
                        element.css('height', height + 'px');
                    }
                } else {
                    if (elt.reducedHeight) {
                        element.css('height', elt.reducedHeight);
                        elt.reducedHeight = undefined;
                    }
                }
            },

            doSetupPosition: function (modalPanel, elt, left, top) {
                var element = $(rf.getDomElement(elt));
                if (!isNaN(left) && !isNaN(top)) {
                    element.css('left', left + 'px');
                    element.css('top', top + 'px');
                }
            },

            doPosition: function (modalPanel, elt) {

            },

            doDiff: function (dx, dy) {

            }
        }

    })());
    rf.ui.PopupPanel.Sizer.Diff = function(dX, dY, dWidth, dHeight) {

        this.deltaX = dX;
        this.deltaY = dY;

        this.deltaWidth = dWidth;
        this.deltaHeight = dHeight;

    };

    rf.ui.PopupPanel.Sizer.Diff.EMPTY = new rf.ui.PopupPanel.Sizer.Diff(0, 0, 0, 0),

        rf.ui.PopupPanel.Sizer.N = function() {

        }

    $.extend(rf.ui.PopupPanel.Sizer.N.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.N.prototype, {


            name: "richfaces.ui.PopupPanel.Sizer.N",

            doPosition : function (popupPanel, elt) {
                var element = $(rf.getDomElement(elt));
                element.css('width', popupPanel.width() + 'px');
                this.doSetupPosition(popupPanel, elt, 0, 0);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(0, dy, 0, -dy);
            }

        });

    rf.ui.PopupPanel.Sizer.NW = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.NW.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.NW.prototype, {

            name: "richfaces.ui.PopupPanel.Sizer.NW",

            doPosition : function (popupPanel, elt) {
                this.doSetupSize(popupPanel, elt);
                this.doSetupPosition(popupPanel, elt, 0, 0);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(dx, dy, -dx, -dy);
            }

        });

    rf.ui.PopupPanel.Sizer.NE = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.NE.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.NE.prototype, {

            name: "richfaces.ui.PopupPanel.Sizer.NE",

            doPosition : function (popupPanel, elt) {
                this.doSetupSize(popupPanel, elt);
                this.doSetupPosition(popupPanel, elt, popupPanel.width() - elt.clientWidth, 0);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(0, dy, dx, -dy);
            }

        });

    rf.ui.PopupPanel.Sizer.E = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.E.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.E.prototype, {

            name: "richfaces.ui.PopupPanel.Sizer.E",

            doPosition : function (popupPanel, elt) {
                var element = $(rf.getDomElement(elt));
                element.css('height', popupPanel.height() + 'px');
                this.doSetupPosition(popupPanel, elt, popupPanel.width() - elt.clientWidth, 0);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(0, 0, dx, 0);
            }

        });

    rf.ui.PopupPanel.Sizer.SE = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.SE.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.SE.prototype, {

            name: "richfaces.ui.PopupPanel.Sizer.SE",

            doPosition : function (popupPanel, elt) {
                this.doSetupSize(popupPanel, elt);
                this.doSetupPosition(popupPanel, elt, popupPanel.width() - elt.clientWidth,
                    popupPanel.height() - elt.clientHeight);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(0, 0, dx, dy);
            }

        });

    rf.ui.PopupPanel.Sizer.S = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.S.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.S.prototype, {

            name: "richfaces.ui.PopupPanel.Sizer.S",

            doPosition : function (popupPanel, elt) {
                var element = $(rf.getDomElement(elt));
                element.css('width', popupPanel.width() + 'px');
                this.doSetupPosition(popupPanel, elt, 0, popupPanel.height() - elt.clientHeight);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(0, 0, 0, dy);
            }

        });


    rf.ui.PopupPanel.Sizer.SW = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.SW.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.SW.prototype, {


            name: "richfaces.ui.PopupPanel.Sizer.SW",

            doPosition : function (popupPanel, elt) {
                this.doSetupSize(popupPanel, elt);
                this.doSetupPosition(popupPanel, elt, 0, popupPanel.height() - elt.clientHeight);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(dx, 0, -dx, dy);
            }

        });

    rf.ui.PopupPanel.Sizer.W = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.W.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.W.prototype, {


            name: "richfaces.ui.PopupPanel.Sizer.W",

            doPosition : function (popupPanel, elt) {
                var element = $(rf.getDomElement(elt));
                element.css('height', popupPanel.height() + 'px');
                this.doSetupPosition(popupPanel, elt, 0, 0);
            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(dx, 0, -dx, 0);
            }


        });


    rf.ui.PopupPanel.Sizer.Header = function() {

    }
    $.extend(rf.ui.PopupPanel.Sizer.Header.prototype, rf.ui.PopupPanel.Sizer.prototype);
    $.extend(rf.ui.PopupPanel.Sizer.Header.prototype, {


            name: "richfaces.ui.PopupPanel.Sizer.Header",

            doPosition : function (popupPanel, elt) {

            },

            doDiff : function(dx, dy) {
                return new rf.ui.PopupPanel.Sizer.Diff(dx, dy, 0, 0);
            }


        });


})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:tab.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.Tab = rf.ui.TogglePanelItem.extendClass({
            // class name
            name:"Tab",

            /**
             * @class AccordionItem
             * @name AccordionItem
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                $super.constructor.call(this, componentId, options);
                this.attachToDom();
                this.index = options["index"];
                this.getTogglePanel().getItems()[this.index] = this;
            },

            /**
             * @param newState {string} = inactive | active | disabled
             *     in that case looking header by css class appropriate to this state
             *
             * @return {jQuery Object}
             * */
            __header : function (newState) {
                var header = $(rf.getDomElement(this.id + ":header"));

                for (var state in stateMap) {
                    if (stateMap.hasOwnProperty(state)) {
                        if (state !== newState) {
                            header.removeClass(stateMap[state]);
                        }
                        if (!header.hasClass(stateMap[newState])) {
                            header.addClass(stateMap[newState]);
                        }
                    }
                }
                return header;
            },

            /**
             * @return {jQuery Object}
             * */
            __content : function () {
                if (!this.__content_) {
                    this.__content_ = $(rf.getDomElement(this.id));
                }
                return this.__content_;
            },

            /**
             * @private
             *
             * used in TogglePanel
             * */
            __enter : function () {

                this.__content().show();
                this.__header("active");

                return this.__fireEnter();
            },

            __fireLeave : function () {
                return rf.Event.fireById(this.id + ":content", "leave");
            },

            __fireEnter : function () {
                return rf.Event.fireById(this.id + ":content", "enter");
            },

            __addUserEventHandler : function (name) {
                var handler = this.options["on" + name];
                if (handler) {
                    var userHandler = rf.Event.bindById(this.id + ":content", name, handler);
                }
            },

            getHeight : function (recalculate) {
                if (recalculate || !this.__height) {
                    this.__height = $(rf.getDomElement(this.id)).outerHeight(true)
                }

                return this.__height;
            },

            /**
             * @private
             *
             * used in TogglePanel
             * */
            __leave : function () {
                var continueProcess = this.__fireLeave();
                if (!continueProcess) {
                    return false;
                }

                this.__content().hide();
                this.__header("inactive");

                return true;
            },

            /***************************** Private Methods ********************************************************/


            destroy: function () {
                var parent = this.getTogglePanel();
                if (parent && parent.getItems && parent.getItems()[this.index]) {
                    delete parent.getItems()[this.index];
                }

                rf.Event.unbindById(this.id);

                $super.destroy.call(this);
            }
        });

    // define super class link
    var $super = rf.ui.Tab.$super;

    var stateMap = {
        active: 'rf-tab-hdr-act',
        inactive: 'rf-tab-hdr-inact',
        disabled: 'rf-tab-hdr-dis'
    };
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:panelMenuGroup.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */


(function ($, rf) {

    rf.ui = rf.ui || {};

    var __DEFAULT_OPTIONS = {
        expanded : false,
        stylePrefix : "rf-pm-gr",
        expandEvent: "click",
        collapseEvent: "click",

        // TODO we should use selectionType = {none, selectable, unselectable}
        selectable : false,
        unselectable : false // unselectable can be only selectable item => if selectable == false than unselectable = false
    };

    var EXPAND_ITEM = {

        /**
         *
         * @return {void}
         * */
        exec : function (group, expand) {
            var mode = group.mode;
            if (mode == "server") {
                return this.execServer(group);
            } else if (mode == "ajax") {
                return this.execAjax(group);
            } else if (mode == "client" || mode == "none") {
                return this.execClient(group, expand);
            } else {
                rf.log.error("EXPAND_ITEM.exec : unknown mode (" + mode + ")");
            }
        },

        /**
         * @protected
         *
         * @return {Boolean} false
         * */
        execServer : function (group) {
            group.__changeState();
            rf.submitForm(this.__getParentForm(group), group.options["ajax"]["parameters"] || {});

            return false;
        },

        /**
         * @protected
         *
         * @return {Boolean} false
         * */
        execAjax : function (group) {
            var oldState = group.__changeState();
            rf.ajax(group.id, null, $.extend({}, group.options["ajax"], {}));
            group.__restoreState(oldState);

            return true;
        },

        /**
         * @protected
         *
         * @param {PanelMenuGroup} group
         * @param {Boolean} expand
         * @return {undefined}
         *             - false - if process has been terminated
         *             - true  - in other cases
         * */
        execClient : function (group, expand) {
            if (expand) {
                group.__expand();
            } else {
                group.__collapse();
            }

            return group.__fireEvent("switch");
        },

        /**
         * @private
         * */
        __getParentForm : function (item) {
            return $($(rf.getDomElement(item.id)).parents("form")[0]);
        }
    };

    rf.ui.PanelMenuGroup = rf.ui.PanelMenuItem.extendClass({
            // class name
            name:"PanelMenuGroup",

            /**
             * @class PanelMenuGroup
             * @name PanelMenuGroup
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                $super.constructor.call(this, componentId, $.extend({}, __DEFAULT_OPTIONS, options || {}));

                this.options.bubbleSelection = this.__rfPanelMenu().options.bubbleSelection;
                this.options.expandSingle = this.__rfPanelMenu().options.expandSingle;

                if (!this.options.disabled) {
                    var menuGroup = this;

                    if (!this.options.selectable) {

                        //TODO nick - this can be replaced by jQuery.delegate on menu itself
                        if (this.options.expandEvent == this.options.collapseEvent) {
                            this.__header().bind(this.options.expandEvent, function () {
                                menuGroup.switchExpantion();
                            });

                        } else {
                            this.__header().bind(this.options.expandEvent, function () {
                                if (menuGroup.collapsed()) {
                                    return menuGroup.expand();
                                }
                            });

                            this.__header().bind(this.options.collapseEvent, function () {
                                if (menuGroup.expanded()) {
                                    return menuGroup.collapse();
                                }
                            });
                        }
                    } else {

                        if (this.options.expandEvent == this.options.collapseEvent) {
                            if (this.options.expandEvent != 'click') {
                                this.__header().bind(this.options.expandEvent, function () {
                                    menuGroup.switchExpantion();
                                });
                            }

                        } else {
                            if (this.options.expandEvent != 'click') {
                                this.__header().bind(this.options.expandEvent, function () {
                                    if (menuGroup.collapsed()) {
                                        return menuGroup.expand();
                                    }
                                });
                            }

                            if (this.options.collapseEvent != 'click') {
                                this.__header().bind(this.options.collapseEvent, function () {
                                    if (menuGroup.expanded()) {
                                        return menuGroup.collapse();
                                    }
                                });
                            }
                        }

                    }

                    if (this.options.selectable || this.options.bubbleSelection) {
                        this.__content().bind("select", function (event) {
                            if (menuGroup.options.selectable && menuGroup.__isMyEvent(event)) {
                                menuGroup.expand();
                            }

                            if (menuGroup.options.bubbleSelection && !menuGroup.__isMyEvent(event)) {
                                menuGroup.__select();
                                if (!menuGroup.expanded()) {
                                    menuGroup.expand();
                                }
                            }
                        });

                        this.__content().bind("unselect", function (event) {
                            if (menuGroup.options.selectable && menuGroup.__isMyEvent(event)) {
                                menuGroup.collapse();
                            }

                            if (menuGroup.options.bubbleSelection && !menuGroup.__isMyEvent(event)) {
                                menuGroup.__unselect();
                            }
                        });
                    }

                    /*this.__addUserEventHandler("beforecollapse");
                     this.__addUserEventHandler("collapse");
                     this.__addUserEventHandler("beforeexpand");
                     this.__addUserEventHandler("expand");
                     this.__addUserEventHandler("beforeswitch");
                     this.__addUserEventHandler("switch");*/
                }
            },

            /***************************** Public Methods  ****************************************************************/
            expanded : function () {
                // TODO check invariant in dev mode
                // return this.__content().hasClass("rf-pm-exp")
                return this.__getExpandValue();
            },

            expand : function () {
                if (this.expanded()) return;
                if (!this.__fireEvent("beforeexpand")) {
                    return false;
                }

                EXPAND_ITEM.exec(this, true);
            },

            __expand : function () {
                this.__updateStyles(true);
                this.__collapseForExpandSingle();

                return this.__fireEvent("expand");
            },

            collapsed : function () {
                // TODO check invariant in dev mode
                // return this.__content().hasClass("rf-pm-colps")
                return !this.__getExpandValue();
            },

            collapse : function () {
                if (!this.expanded()) return;
                if (!this.__fireEvent("beforecollapse")) {
                    return false;
                }

                EXPAND_ITEM.exec(this, false);
            },

            __collapse : function () {
                this.__updateStyles(false);

                this.__childGroups().each(function(index, group) {
                    //TODO nick - why not group.collapse()?
                    rf.component(group.id).__collapse();
                });

                return this.__fireEvent("collapse");
            },

            __updateStyles : function (expand) {
                if (expand) {
                    //expand
                    this.__content().removeClass("rf-pm-colps").addClass("rf-pm-exp");
                    this.__header().removeClass("rf-pm-hdr-colps").addClass("rf-pm-hdr-exp");

                    this.__setExpandValue(true);
                } else {
                    this.__content().addClass("rf-pm-colps").removeClass("rf-pm-exp");
                    this.__header().addClass("rf-pm-hdr-colps").removeClass("rf-pm-hdr-exp");

                    this.__setExpandValue(false);
                }
            },

            /**
             * @methodOf
             * @name PanelMenuGroup#switch
             *
             * TODO ...
             *
             * @param {boolean} expand
             * @return {void} TODO ...
             */
            switchExpantion : function () { // TODO rename
                var continueProcess = this.__fireEvent("beforeswitch");
                if (!continueProcess) {
                    return false;
                }

                if (this.expanded()) {
                    this.collapse();
                } else {
                    this.expand();
                }
            },

            /**
             * please, remove this method when client side ajax events will be added
             *
             * */
            onCompleteHandler : function () {
                if (this.options.selectable) {
                    $super.onCompleteHandler.call(this);
                }

                EXPAND_ITEM.execClient(this, this.expanded());
            },

            __switch : function (expand) {
                if (expand) {
                    this.__expand();
                } else {
                    this.__collapse();
                }
                return this.__fireEvent("switch");
            },

            /***************************** Private Methods ****************************************************************/
            __childGroups : function () {
                return this.__content().children(".rf-pm-gr")
            },

            __group : function () {
                return $(rf.getDomElement(this.id))
            },

            __header : function () {
                return $(rf.getDomElement(this.id + ":hdr"))
            },

            __content : function () {
                return $(rf.getDomElement(this.id + ":cnt"))
            },

            __expandValueInput : function () {
                return document.getElementById(this.id + ":expanded");
            },

            __getExpandValue : function () {
                return this.__expandValueInput().value == "true";
            },

            __collapseForExpandSingle: function() {
                if (this.options.expandSingle) {
                    this.__rfPanelMenu().__collapseGroups(this);
                }
            },

            /**
             * @methodOf
             * @name PanelMenuGroup#__setExpandValue
             *
             * @param {boolean} value - is group expanded?
             * @return {boolean} preview value
             */
            __setExpandValue : function (value) {
                var input = this.__expandValueInput();
                var oldValue = input.value;

                input.value = value;

                return oldValue;
            },

            __changeState : function () {
                if (!this.__getExpandValue()) {
                    this.__collapseForExpandSingle();
                }

                var state = {};
                state["expanded"] = this.__setExpandValue(!this.__getExpandValue());
                if (this.options.selectable) {
                    state["itemName"] = this.__rfPanelMenu().selectedItem(this.itemName); // TODO bad function name for function which change component state
                }

                return state;
            },

            __restoreState : function (state) {
                if (!state) {
                    return;
                }

                if (state["expanded"]) {
                    this.__setExpandValue(state["expanded"]);
                }

                if (state["itemName"]) {
                    this.__rfPanelMenu().selectedItem(state["itemName"]);
                }
            },

            __isMyEvent: function (event) {
                return this.id == event.target.id;
            },

            destroy: function () {
                rf.Event.unbindById(this.id, "." + this.namespace);

                $super.destroy.call(this);
            }
        });

    // define super class link
    var $super = rf.ui.PanelMenuGroup.$super;
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:accordionItem.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.AccordionItem = rf.ui.TogglePanelItem.extendClass({
            // class name
            name:"AccordionItem",

            /**
             * @class AccordionItem
             * @name AccordionItem
             *
             * @constructor
             * @param {String} componentId - component id
             * @param {Hash} options - params
             * */
            init : function (componentId, options) {
                $super.constructor.call(this, componentId, options);

                if (!this.disabled) {
                    rf.Event.bindById(this.id + ":header", "click", this.__onHeaderClick, this);
                }

                if (this.isSelected()) {
                    var item = this;
                    $(document).ready(function () {
                        item.__fitToHeight(item.getTogglePanel());
                    }).one("javascriptServiceComplete", function () {
                        item.__fitToHeight(item.getTogglePanel());
                    });
                }
            },

            /***************************** Public Methods  ****************************************************************/

            __onHeaderClick : function (comp) {
                this.getTogglePanel().switchToItem(this.getName());
            },

            /**
             * @return {jQuery Object}
             * */
            __header : function () {
                return $(rf.getDomElement(this.id + ":header"));
            },

            /**
             * @return {jQuery Object}
             * */
            __content : function () {
                if (!this.__content_) {
                    this.__content_ = $(rf.getDomElement(this.id + ":content"));
                }
                return this.__content_;
            },

            /**
             * @private
             *
             * used in TogglePanel
             * */
            __enter : function () {
                var parentPanel = this.getTogglePanel();
                if (parentPanel.isKeepHeight) {
                    this.__content().hide(); // TODO ?
                    this.__fitToHeight(parentPanel);
                }

                this.__content().show();
                this.__header().addClass("rf-ac-itm-hdr-act").removeClass("rf-ac-itm-hdr-inact");

                return this.__fireEnter();
            },

            __fitToHeight : function (parentPanel) {
                var h = parentPanel.getInnerHeight();

                var items = parentPanel.getItems();
                for (var i = 0; i < items.length; i++) {
                    h -= items[i].__header().outerHeight();
                }

                this.__content().height(h - 20); // 20 it is padding top and bottom
            },

            getHeight : function (recalculate) {
                if (recalculate || !this.__height) {
                    this.__height = $(rf.getDomElement(this.id)).outerHeight(true)
                }

                return this.__height;
            },

            /**
             * @private
             *
             * used in TogglePanel
             * */
            __leave : function () {
                var continueProcess = this.__fireLeave();
                if (!continueProcess) {
                    return false;
                }

                this.__content().hide();
                this.__header().removeClass("rf-ac-itm-hdr-act").addClass("rf-ac-itm-hdr-inact");

                return true;
            }
        });

    // define super class link
    var $super = rf.ui.AccordionItem.$super;
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:notifyStack.js
(function($, rf) {

    rf.ui = rf.ui || {};
    
    var defaultOptions = {
        position: "tr",
        direction: "vertical",
        method: "last",
        notifications: [],
        addNotification: function(pnotify) {
            this.notifications.push(pnotify);
        }
    };

    rf.ui.NotifyStack = rf.BaseComponent.extendClass({
        
        name : "NotifyStack",

        init : function(componentId, options) {
            $super.constructor.call(this, componentId);
            this.attachToDom(this.id);
            this.__initializeStack(options);
        },
        
        __initializeStack : function(options) {
            var stack = $.extend({}, $.pnotify.defaults.pnotify_stack, defaultOptions, options);
            
            var isVertical = (stack.direction == 'vertical');
            var isFirst = (stack.method == 'first');
            
            stack.push = isFirst ? 'top' : 'bottom';
            
            switch (stack.position) {
                case "tl": // topLeft
                    stack.dir1 = isVertical ? 'down' : 'right';
                    stack.dir2 = isVertical ? 'right' : 'down';
                    break;
                case "tr": // topRight
                    stack.dir1 = isVertical ? 'down' : 'left';
                    stack.dir2 = isVertical ? 'left' : 'down';
                    break;
                case "bl": // bottomLeft
                    stack.dir1 = isVertical ? 'up' : 'right';
                    stack.dir2 = isVertical ? 'right' : 'up';
                    break;
                case "br": // bottomRight
                    stack.dir1 = isVertical ? 'up' : 'left';
                    stack.dir2 = isVertical ? 'left' : 'up';
                    break;
                default:
                    throw "wrong stack position: " + stack.position;
            }
            
            this.stack = stack;
        },
    
        getStack : function() {
            return this.stack;
        },
        
        removeNotifications: function() {
            var pnotify;
            while (pnotify = this.stack.notifications.pop()) {
                pnotify.pnotify_remove();
            }
        },
        
        destroy : function() {
            this.removeNotifications();
            this.stack = null;
            $super.destroy.call(this);
        }
    });
    
    var $super = rf.ui.NotifyStack.$super;

})(RichFaces.jQuery, RichFaces);;

// resource: com.jqueryui:draggable.js
/*!
 * jQuery UI Draggable 1.11.4
 * http://jqueryui.com
 *
 * Copyright jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * http://api.jqueryui.com/draggable/
 */
(function( factory ) {
	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define([
			"jquery",
			"./core",
			"./mouse",
			"./widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
}(function( $ ) {

$.widget("ui.draggable", $.ui.mouse, {
	version: "1.11.4",
	widgetEventPrefix: "drag",
	options: {
		addClasses: true,
		appendTo: "parent",
		axis: false,
		connectToSortable: false,
		containment: false,
		cursor: "auto",
		cursorAt: false,
		grid: false,
		handle: false,
		helper: "original",
		iframeFix: false,
		opacity: false,
		refreshPositions: false,
		revert: false,
		revertDuration: 500,
		scope: "default",
		scroll: true,
		scrollSensitivity: 20,
		scrollSpeed: 20,
		snap: false,
		snapMode: "both",
		snapTolerance: 20,
		stack: false,
		zIndex: false,

		// callbacks
		drag: null,
		start: null,
		stop: null
	},
	_create: function() {

		if ( this.options.helper === "original" ) {
			this._setPositionRelative();
		}
		if (this.options.addClasses){
			this.element.addClass("ui-draggable");
		}
		if (this.options.disabled){
			this.element.addClass("ui-draggable-disabled");
		}
		this._setHandleClassName();

		this._mouseInit();
	},

	_setOption: function( key, value ) {
		this._super( key, value );
		if ( key === "handle" ) {
			this._removeHandleClassName();
			this._setHandleClassName();
		}
	},

	_destroy: function() {
		if ( ( this.helper || this.element ).is( ".ui-draggable-dragging" ) ) {
			this.destroyOnClear = true;
			return;
		}
		this.element.removeClass( "ui-draggable ui-draggable-dragging ui-draggable-disabled" );
		this._removeHandleClassName();
		this._mouseDestroy();
	},

	_mouseCapture: function(event) {
		var o = this.options;

		this._blurActiveElement( event );

		// among others, prevent a drag on a resizable-handle
		if (this.helper || o.disabled || $(event.target).closest(".ui-resizable-handle").length > 0) {
			return false;
		}

		//Quit if we're not on a valid handle
		this.handle = this._getHandle(event);
		if (!this.handle) {
			return false;
		}

		this._blockFrames( o.iframeFix === true ? "iframe" : o.iframeFix );

		return true;

	},

	_blockFrames: function( selector ) {
		this.iframeBlocks = this.document.find( selector ).map(function() {
			var iframe = $( this );

			return $( "<div>" )
				.css( "position", "absolute" )
				.appendTo( iframe.parent() )
				.outerWidth( iframe.outerWidth() )
				.outerHeight( iframe.outerHeight() )
				.offset( iframe.offset() )[ 0 ];
		});
	},

	_unblockFrames: function() {
		if ( this.iframeBlocks ) {
			this.iframeBlocks.remove();
			delete this.iframeBlocks;
		}
	},

	_blurActiveElement: function( event ) {
		var document = this.document[ 0 ];

		// Only need to blur if the event occurred on the draggable itself, see #10527
		if ( !this.handleElement.is( event.target ) ) {
			return;
		}

		// support: IE9
		// IE9 throws an "Unspecified error" accessing document.activeElement from an <iframe>
		try {

			// Support: IE9, IE10
			// If the <body> is blurred, IE will switch windows, see #9520
			if ( document.activeElement && document.activeElement.nodeName.toLowerCase() !== "body" ) {

				// Blur any element that currently has focus, see #4261
				$( document.activeElement ).blur();
			}
		} catch ( error ) {}
	},

	_mouseStart: function(event) {

		var o = this.options;

		//Create and append the visible helper
		this.helper = this._createHelper(event);

		this.helper.addClass("ui-draggable-dragging");

		//Cache the helper size
		this._cacheHelperProportions();

		//If ddmanager is used for droppables, set the global draggable
		if ($.ui.ddmanager) {
			$.ui.ddmanager.current = this;
		}

		/*
		 * - Position generation -
		 * This block generates everything position related - it's the core of draggables.
		 */

		//Cache the margins of the original element
		this._cacheMargins();

		//Store the helper's css position
		this.cssPosition = this.helper.css( "position" );
		this.scrollParent = this.helper.scrollParent( true );
		this.offsetParent = this.helper.offsetParent();
		this.hasFixedAncestor = this.helper.parents().filter(function() {
				return $( this ).css( "position" ) === "fixed";
			}).length > 0;

		//The element's absolute position on the page minus margins
		this.positionAbs = this.element.offset();
		this._refreshOffsets( event );

		//Generate the original position
		this.originalPosition = this.position = this._generatePosition( event, false );
		this.originalPageX = event.pageX;
		this.originalPageY = event.pageY;

		//Adjust the mouse offset relative to the helper if "cursorAt" is supplied
		(o.cursorAt && this._adjustOffsetFromHelper(o.cursorAt));

		//Set a containment if given in the options
		this._setContainment();

		//Trigger event + callbacks
		if (this._trigger("start", event) === false) {
			this._clear();
			return false;
		}

		//Recache the helper size
		this._cacheHelperProportions();

		//Prepare the droppable offsets
		if ($.ui.ddmanager && !o.dropBehaviour) {
			$.ui.ddmanager.prepareOffsets(this, event);
		}

		// Reset helper's right/bottom css if they're set and set explicit width/height instead
		// as this prevents resizing of elements with right/bottom set (see #7772)
		this._normalizeRightBottom();

		this._mouseDrag(event, true); //Execute the drag once - this causes the helper not to be visible before getting its correct position

		//If the ddmanager is used for droppables, inform the manager that dragging has started (see #5003)
		if ( $.ui.ddmanager ) {
			$.ui.ddmanager.dragStart(this, event);
		}

		return true;
	},

	_refreshOffsets: function( event ) {
		this.offset = {
			top: this.positionAbs.top - this.margins.top,
			left: this.positionAbs.left - this.margins.left,
			scroll: false,
			parent: this._getParentOffset(),
			relative: this._getRelativeOffset()
		};

		this.offset.click = {
			left: event.pageX - this.offset.left,
			top: event.pageY - this.offset.top
		};
	},

	_mouseDrag: function(event, noPropagation) {
		// reset any necessary cached properties (see #5009)
		if ( this.hasFixedAncestor ) {
			this.offset.parent = this._getParentOffset();
		}

		//Compute the helpers position
		this.position = this._generatePosition( event, true );
		this.positionAbs = this._convertPositionTo("absolute");

		//Call plugins and callbacks and use the resulting position if something is returned
		if (!noPropagation) {
			var ui = this._uiHash();
			if (this._trigger("drag", event, ui) === false) {
				this._mouseUp({});
				return false;
			}
			this.position = ui.position;
		}

		this.helper[ 0 ].style.left = this.position.left + "px";
		this.helper[ 0 ].style.top = this.position.top + "px";

		if ($.ui.ddmanager) {
			$.ui.ddmanager.drag(this, event);
		}

		return false;
	},

	_mouseStop: function(event) {

		//If we are using droppables, inform the manager about the drop
		var that = this,
			dropped = false;
		if ($.ui.ddmanager && !this.options.dropBehaviour) {
			dropped = $.ui.ddmanager.drop(this, event);
		}

		//if a drop comes from outside (a sortable)
		if (this.dropped) {
			dropped = this.dropped;
			this.dropped = false;
		}

		if ((this.options.revert === "invalid" && !dropped) || (this.options.revert === "valid" && dropped) || this.options.revert === true || ($.isFunction(this.options.revert) && this.options.revert.call(this.element, dropped))) {
			$(this.helper).animate(this.originalPosition, parseInt(this.options.revertDuration, 10), function() {
				if (that._trigger("stop", event) !== false) {
					that._clear();
				}
			});
		} else {
			if (this._trigger("stop", event) !== false) {
				this._clear();
			}
		}

		return false;
	},

	_mouseUp: function( event ) {
		this._unblockFrames();

		//If the ddmanager is used for droppables, inform the manager that dragging has stopped (see #5003)
		if ( $.ui.ddmanager ) {
			$.ui.ddmanager.dragStop(this, event);
		}

		// Only need to focus if the event occurred on the draggable itself, see #10527
		if ( this.handleElement.is( event.target ) ) {
			// The interaction is over; whether or not the click resulted in a drag, focus the element
			this.element.focus();
		}

		return $.ui.mouse.prototype._mouseUp.call(this, event);
	},

	cancel: function() {

		if (this.helper.is(".ui-draggable-dragging")) {
			this._mouseUp({});
		} else {
			this._clear();
		}

		return this;

	},

	_getHandle: function(event) {
		return this.options.handle ?
			!!$( event.target ).closest( this.element.find( this.options.handle ) ).length :
			true;
	},

	_setHandleClassName: function() {
		this.handleElement = this.options.handle ?
			this.element.find( this.options.handle ) : this.element;
		this.handleElement.addClass( "ui-draggable-handle" );
	},

	_removeHandleClassName: function() {
		this.handleElement.removeClass( "ui-draggable-handle" );
	},

	_createHelper: function(event) {

		var o = this.options,
			helperIsFunction = $.isFunction( o.helper ),
			helper = helperIsFunction ?
				$( o.helper.apply( this.element[ 0 ], [ event ] ) ) :
				( o.helper === "clone" ?
					this.element.clone().removeAttr( "id" ) :
					this.element );

		if (!helper.parents("body").length) {
			helper.appendTo((o.appendTo === "parent" ? this.element[0].parentNode : o.appendTo));
		}

		// http://bugs.jqueryui.com/ticket/9446
		// a helper function can return the original element
		// which wouldn't have been set to relative in _create
		if ( helperIsFunction && helper[ 0 ] === this.element[ 0 ] ) {
			this._setPositionRelative();
		}

		if (helper[0] !== this.element[0] && !(/(fixed|absolute)/).test(helper.css("position"))) {
			helper.css("position", "absolute");
		}

		return helper;

	},

	_setPositionRelative: function() {
		if ( !( /^(?:r|a|f)/ ).test( this.element.css( "position" ) ) ) {
			this.element[ 0 ].style.position = "relative";
		}
	},

	_adjustOffsetFromHelper: function(obj) {
		if (typeof obj === "string") {
			obj = obj.split(" ");
		}
		if ($.isArray(obj)) {
			obj = { left: +obj[0], top: +obj[1] || 0 };
		}
		if ("left" in obj) {
			this.offset.click.left = obj.left + this.margins.left;
		}
		if ("right" in obj) {
			this.offset.click.left = this.helperProportions.width - obj.right + this.margins.left;
		}
		if ("top" in obj) {
			this.offset.click.top = obj.top + this.margins.top;
		}
		if ("bottom" in obj) {
			this.offset.click.top = this.helperProportions.height - obj.bottom + this.margins.top;
		}
	},

	_isRootNode: function( element ) {
		return ( /(html|body)/i ).test( element.tagName ) || element === this.document[ 0 ];
	},

	_getParentOffset: function() {

		//Get the offsetParent and cache its position
		var po = this.offsetParent.offset(),
			document = this.document[ 0 ];

		// This is a special case where we need to modify a offset calculated on start, since the following happened:
		// 1. The position of the helper is absolute, so it's position is calculated based on the next positioned parent
		// 2. The actual offset parent is a child of the scroll parent, and the scroll parent isn't the document, which means that
		//    the scroll is included in the initial calculation of the offset of the parent, and never recalculated upon drag
		if (this.cssPosition === "absolute" && this.scrollParent[0] !== document && $.contains(this.scrollParent[0], this.offsetParent[0])) {
			po.left += this.scrollParent.scrollLeft();
			po.top += this.scrollParent.scrollTop();
		}

		if ( this._isRootNode( this.offsetParent[ 0 ] ) ) {
			po = { top: 0, left: 0 };
		}

		return {
			top: po.top + (parseInt(this.offsetParent.css("borderTopWidth"), 10) || 0),
			left: po.left + (parseInt(this.offsetParent.css("borderLeftWidth"), 10) || 0)
		};

	},

	_getRelativeOffset: function() {
		if ( this.cssPosition !== "relative" ) {
			return { top: 0, left: 0 };
		}

		var p = this.element.position(),
			scrollIsRootNode = this._isRootNode( this.scrollParent[ 0 ] );

		return {
			top: p.top - ( parseInt(this.helper.css( "top" ), 10) || 0 ) + ( !scrollIsRootNode ? this.scrollParent.scrollTop() : 0 ),
			left: p.left - ( parseInt(this.helper.css( "left" ), 10) || 0 ) + ( !scrollIsRootNode ? this.scrollParent.scrollLeft() : 0 )
		};

	},

	_cacheMargins: function() {
		this.margins = {
			left: (parseInt(this.element.css("marginLeft"), 10) || 0),
			top: (parseInt(this.element.css("marginTop"), 10) || 0),
			right: (parseInt(this.element.css("marginRight"), 10) || 0),
			bottom: (parseInt(this.element.css("marginBottom"), 10) || 0)
		};
	},

	_cacheHelperProportions: function() {
		this.helperProportions = {
			width: this.helper.outerWidth(),
			height: this.helper.outerHeight()
		};
	},

	_setContainment: function() {

		var isUserScrollable, c, ce,
			o = this.options,
			document = this.document[ 0 ];

		this.relativeContainer = null;

		if ( !o.containment ) {
			this.containment = null;
			return;
		}

		if ( o.containment === "window" ) {
			this.containment = [
				$( window ).scrollLeft() - this.offset.relative.left - this.offset.parent.left,
				$( window ).scrollTop() - this.offset.relative.top - this.offset.parent.top,
				$( window ).scrollLeft() + $( window ).width() - this.helperProportions.width - this.margins.left,
				$( window ).scrollTop() + ( $( window ).height() || document.body.parentNode.scrollHeight ) - this.helperProportions.height - this.margins.top
			];
			return;
		}

		if ( o.containment === "document") {
			this.containment = [
				0,
				0,
				$( document ).width() - this.helperProportions.width - this.margins.left,
				( $( document ).height() || document.body.parentNode.scrollHeight ) - this.helperProportions.height - this.margins.top
			];
			return;
		}

		if ( o.containment.constructor === Array ) {
			this.containment = o.containment;
			return;
		}

		if ( o.containment === "parent" ) {
			o.containment = this.helper[ 0 ].parentNode;
		}

		c = $( o.containment );
		ce = c[ 0 ];

		if ( !ce ) {
			return;
		}

		isUserScrollable = /(scroll|auto)/.test( c.css( "overflow" ) );

		this.containment = [
			( parseInt( c.css( "borderLeftWidth" ), 10 ) || 0 ) + ( parseInt( c.css( "paddingLeft" ), 10 ) || 0 ),
			( parseInt( c.css( "borderTopWidth" ), 10 ) || 0 ) + ( parseInt( c.css( "paddingTop" ), 10 ) || 0 ),
			( isUserScrollable ? Math.max( ce.scrollWidth, ce.offsetWidth ) : ce.offsetWidth ) -
				( parseInt( c.css( "borderRightWidth" ), 10 ) || 0 ) -
				( parseInt( c.css( "paddingRight" ), 10 ) || 0 ) -
				this.helperProportions.width -
				this.margins.left -
				this.margins.right,
			( isUserScrollable ? Math.max( ce.scrollHeight, ce.offsetHeight ) : ce.offsetHeight ) -
				( parseInt( c.css( "borderBottomWidth" ), 10 ) || 0 ) -
				( parseInt( c.css( "paddingBottom" ), 10 ) || 0 ) -
				this.helperProportions.height -
				this.margins.top -
				this.margins.bottom
		];
		this.relativeContainer = c;
	},

	_convertPositionTo: function(d, pos) {

		if (!pos) {
			pos = this.position;
		}

		var mod = d === "absolute" ? 1 : -1,
			scrollIsRootNode = this._isRootNode( this.scrollParent[ 0 ] );

		return {
			top: (
				pos.top	+																// The absolute mouse position
				this.offset.relative.top * mod +										// Only for relative positioned nodes: Relative offset from element to offset parent
				this.offset.parent.top * mod -										// The offsetParent's offset without borders (offset + border)
				( ( this.cssPosition === "fixed" ? -this.offset.scroll.top : ( scrollIsRootNode ? 0 : this.offset.scroll.top ) ) * mod)
			),
			left: (
				pos.left +																// The absolute mouse position
				this.offset.relative.left * mod +										// Only for relative positioned nodes: Relative offset from element to offset parent
				this.offset.parent.left * mod	-										// The offsetParent's offset without borders (offset + border)
				( ( this.cssPosition === "fixed" ? -this.offset.scroll.left : ( scrollIsRootNode ? 0 : this.offset.scroll.left ) ) * mod)
			)
		};

	},

	_generatePosition: function( event, constrainPosition ) {

		var containment, co, top, left,
			o = this.options,
			scrollIsRootNode = this._isRootNode( this.scrollParent[ 0 ] ),
			pageX = event.pageX,
			pageY = event.pageY;

		// Cache the scroll
		if ( !scrollIsRootNode || !this.offset.scroll ) {
			this.offset.scroll = {
				top: this.scrollParent.scrollTop(),
				left: this.scrollParent.scrollLeft()
			};
		}

		/*
		 * - Position constraining -
		 * Constrain the position to a mix of grid, containment.
		 */

		// If we are not dragging yet, we won't check for options
		if ( constrainPosition ) {
			if ( this.containment ) {
				if ( this.relativeContainer ){
					co = this.relativeContainer.offset();
					containment = [
						this.containment[ 0 ] + co.left,
						this.containment[ 1 ] + co.top,
						this.containment[ 2 ] + co.left,
						this.containment[ 3 ] + co.top
					];
				} else {
					containment = this.containment;
				}

				if (event.pageX - this.offset.click.left < containment[0]) {
					pageX = containment[0] + this.offset.click.left;
				}
				if (event.pageY - this.offset.click.top < containment[1]) {
					pageY = containment[1] + this.offset.click.top;
				}
				if (event.pageX - this.offset.click.left > containment[2]) {
					pageX = containment[2] + this.offset.click.left;
				}
				if (event.pageY - this.offset.click.top > containment[3]) {
					pageY = containment[3] + this.offset.click.top;
				}
			}

			if (o.grid) {
				//Check for grid elements set to 0 to prevent divide by 0 error causing invalid argument errors in IE (see ticket #6950)
				top = o.grid[1] ? this.originalPageY + Math.round((pageY - this.originalPageY) / o.grid[1]) * o.grid[1] : this.originalPageY;
				pageY = containment ? ((top - this.offset.click.top >= containment[1] || top - this.offset.click.top > containment[3]) ? top : ((top - this.offset.click.top >= containment[1]) ? top - o.grid[1] : top + o.grid[1])) : top;

				left = o.grid[0] ? this.originalPageX + Math.round((pageX - this.originalPageX) / o.grid[0]) * o.grid[0] : this.originalPageX;
				pageX = containment ? ((left - this.offset.click.left >= containment[0] || left - this.offset.click.left > containment[2]) ? left : ((left - this.offset.click.left >= containment[0]) ? left - o.grid[0] : left + o.grid[0])) : left;
			}

			if ( o.axis === "y" ) {
				pageX = this.originalPageX;
			}

			if ( o.axis === "x" ) {
				pageY = this.originalPageY;
			}
		}

		return {
			top: (
				pageY -																	// The absolute mouse position
				this.offset.click.top	-												// Click offset (relative to the element)
				this.offset.relative.top -												// Only for relative positioned nodes: Relative offset from element to offset parent
				this.offset.parent.top +												// The offsetParent's offset without borders (offset + border)
				( this.cssPosition === "fixed" ? -this.offset.scroll.top : ( scrollIsRootNode ? 0 : this.offset.scroll.top ) )
			),
			left: (
				pageX -																	// The absolute mouse position
				this.offset.click.left -												// Click offset (relative to the element)
				this.offset.relative.left -												// Only for relative positioned nodes: Relative offset from element to offset parent
				this.offset.parent.left +												// The offsetParent's offset without borders (offset + border)
				( this.cssPosition === "fixed" ? -this.offset.scroll.left : ( scrollIsRootNode ? 0 : this.offset.scroll.left ) )
			)
		};

	},

	_clear: function() {
		this.helper.removeClass("ui-draggable-dragging");
		if (this.helper[0] !== this.element[0] && !this.cancelHelperRemoval) {
			this.helper.remove();
		}
		this.helper = null;
		this.cancelHelperRemoval = false;
		if ( this.destroyOnClear ) {
			this.destroy();
		}
	},

	_normalizeRightBottom: function() {
		if ( this.options.axis !== "y" && this.helper.css( "right" ) !== "auto" ) {
			this.helper.width( this.helper.width() );
			this.helper.css( "right", "auto" );
		}
		if ( this.options.axis !== "x" && this.helper.css( "bottom" ) !== "auto" ) {
			this.helper.height( this.helper.height() );
			this.helper.css( "bottom", "auto" );
		}
	},

	// From now on bulk stuff - mainly helpers

	_trigger: function( type, event, ui ) {
		ui = ui || this._uiHash();
		$.ui.plugin.call( this, type, [ event, ui, this ], true );

		// Absolute position and offset (see #6884 ) have to be recalculated after plugins
		if ( /^(drag|start|stop)/.test( type ) ) {
			this.positionAbs = this._convertPositionTo( "absolute" );
			ui.offset = this.positionAbs;
		}
		return $.Widget.prototype._trigger.call( this, type, event, ui );
	},

	plugins: {},

	_uiHash: function() {
		return {
			helper: this.helper,
			position: this.position,
			originalPosition: this.originalPosition,
			offset: this.positionAbs
		};
	}

});

$.ui.plugin.add( "draggable", "connectToSortable", {
	start: function( event, ui, draggable ) {
		var uiSortable = $.extend( {}, ui, {
			item: draggable.element
		});

		draggable.sortables = [];
		$( draggable.options.connectToSortable ).each(function() {
			var sortable = $( this ).sortable( "instance" );

			if ( sortable && !sortable.options.disabled ) {
				draggable.sortables.push( sortable );

				// refreshPositions is called at drag start to refresh the containerCache
				// which is used in drag. This ensures it's initialized and synchronized
				// with any changes that might have happened on the page since initialization.
				sortable.refreshPositions();
				sortable._trigger("activate", event, uiSortable);
			}
		});
	},
	stop: function( event, ui, draggable ) {
		var uiSortable = $.extend( {}, ui, {
			item: draggable.element
		});

		draggable.cancelHelperRemoval = false;

		$.each( draggable.sortables, function() {
			var sortable = this;

			if ( sortable.isOver ) {
				sortable.isOver = 0;

				// Allow this sortable to handle removing the helper
				draggable.cancelHelperRemoval = true;
				sortable.cancelHelperRemoval = false;

				// Use _storedCSS To restore properties in the sortable,
				// as this also handles revert (#9675) since the draggable
				// may have modified them in unexpected ways (#8809)
				sortable._storedCSS = {
					position: sortable.placeholder.css( "position" ),
					top: sortable.placeholder.css( "top" ),
					left: sortable.placeholder.css( "left" )
				};

				sortable._mouseStop(event);

				// Once drag has ended, the sortable should return to using
				// its original helper, not the shared helper from draggable
				sortable.options.helper = sortable.options._helper;
			} else {
				// Prevent this Sortable from removing the helper.
				// However, don't set the draggable to remove the helper
				// either as another connected Sortable may yet handle the removal.
				sortable.cancelHelperRemoval = true;

				sortable._trigger( "deactivate", event, uiSortable );
			}
		});
	},
	drag: function( event, ui, draggable ) {
		$.each( draggable.sortables, function() {
			var innermostIntersecting = false,
				sortable = this;

			// Copy over variables that sortable's _intersectsWith uses
			sortable.positionAbs = draggable.positionAbs;
			sortable.helperProportions = draggable.helperProportions;
			sortable.offset.click = draggable.offset.click;

			if ( sortable._intersectsWith( sortable.containerCache ) ) {
				innermostIntersecting = true;

				$.each( draggable.sortables, function() {
					// Copy over variables that sortable's _intersectsWith uses
					this.positionAbs = draggable.positionAbs;
					this.helperProportions = draggable.helperProportions;
					this.offset.click = draggable.offset.click;

					if ( this !== sortable &&
							this._intersectsWith( this.containerCache ) &&
							$.contains( sortable.element[ 0 ], this.element[ 0 ] ) ) {
						innermostIntersecting = false;
					}

					return innermostIntersecting;
				});
			}

			if ( innermostIntersecting ) {
				// If it intersects, we use a little isOver variable and set it once,
				// so that the move-in stuff gets fired only once.
				if ( !sortable.isOver ) {
					sortable.isOver = 1;

					// Store draggable's parent in case we need to reappend to it later.
					draggable._parent = ui.helper.parent();

					sortable.currentItem = ui.helper
						.appendTo( sortable.element )
						.data( "ui-sortable-item", true );

					// Store helper option to later restore it
					sortable.options._helper = sortable.options.helper;

					sortable.options.helper = function() {
						return ui.helper[ 0 ];
					};

					// Fire the start events of the sortable with our passed browser event,
					// and our own helper (so it doesn't create a new one)
					event.target = sortable.currentItem[ 0 ];
					sortable._mouseCapture( event, true );
					sortable._mouseStart( event, true, true );

					// Because the browser event is way off the new appended portlet,
					// modify necessary variables to reflect the changes
					sortable.offset.click.top = draggable.offset.click.top;
					sortable.offset.click.left = draggable.offset.click.left;
					sortable.offset.parent.left -= draggable.offset.parent.left -
						sortable.offset.parent.left;
					sortable.offset.parent.top -= draggable.offset.parent.top -
						sortable.offset.parent.top;

					draggable._trigger( "toSortable", event );

					// Inform draggable that the helper is in a valid drop zone,
					// used solely in the revert option to handle "valid/invalid".
					draggable.dropped = sortable.element;

					// Need to refreshPositions of all sortables in the case that
					// adding to one sortable changes the location of the other sortables (#9675)
					$.each( draggable.sortables, function() {
						this.refreshPositions();
					});

					// hack so receive/update callbacks work (mostly)
					draggable.currentItem = draggable.element;
					sortable.fromOutside = draggable;
				}

				if ( sortable.currentItem ) {
					sortable._mouseDrag( event );
					// Copy the sortable's position because the draggable's can potentially reflect
					// a relative position, while sortable is always absolute, which the dragged
					// element has now become. (#8809)
					ui.position = sortable.position;
				}
			} else {
				// If it doesn't intersect with the sortable, and it intersected before,
				// we fake the drag stop of the sortable, but make sure it doesn't remove
				// the helper by using cancelHelperRemoval.
				if ( sortable.isOver ) {

					sortable.isOver = 0;
					sortable.cancelHelperRemoval = true;

					// Calling sortable's mouseStop would trigger a revert,
					// so revert must be temporarily false until after mouseStop is called.
					sortable.options._revert = sortable.options.revert;
					sortable.options.revert = false;

					sortable._trigger( "out", event, sortable._uiHash( sortable ) );
					sortable._mouseStop( event, true );

					// restore sortable behaviors that were modfied
					// when the draggable entered the sortable area (#9481)
					sortable.options.revert = sortable.options._revert;
					sortable.options.helper = sortable.options._helper;

					if ( sortable.placeholder ) {
						sortable.placeholder.remove();
					}

					// Restore and recalculate the draggable's offset considering the sortable
					// may have modified them in unexpected ways. (#8809, #10669)
					ui.helper.appendTo( draggable._parent );
					draggable._refreshOffsets( event );
					ui.position = draggable._generatePosition( event, true );

					draggable._trigger( "fromSortable", event );

					// Inform draggable that the helper is no longer in a valid drop zone
					draggable.dropped = false;

					// Need to refreshPositions of all sortables just in case removing
					// from one sortable changes the location of other sortables (#9675)
					$.each( draggable.sortables, function() {
						this.refreshPositions();
					});
				}
			}
		});
	}
});

$.ui.plugin.add("draggable", "cursor", {
	start: function( event, ui, instance ) {
		var t = $( "body" ),
			o = instance.options;

		if (t.css("cursor")) {
			o._cursor = t.css("cursor");
		}
		t.css("cursor", o.cursor);
	},
	stop: function( event, ui, instance ) {
		var o = instance.options;
		if (o._cursor) {
			$("body").css("cursor", o._cursor);
		}
	}
});

$.ui.plugin.add("draggable", "opacity", {
	start: function( event, ui, instance ) {
		var t = $( ui.helper ),
			o = instance.options;
		if (t.css("opacity")) {
			o._opacity = t.css("opacity");
		}
		t.css("opacity", o.opacity);
	},
	stop: function( event, ui, instance ) {
		var o = instance.options;
		if (o._opacity) {
			$(ui.helper).css("opacity", o._opacity);
		}
	}
});

$.ui.plugin.add("draggable", "scroll", {
	start: function( event, ui, i ) {
		if ( !i.scrollParentNotHidden ) {
			i.scrollParentNotHidden = i.helper.scrollParent( false );
		}

		if ( i.scrollParentNotHidden[ 0 ] !== i.document[ 0 ] && i.scrollParentNotHidden[ 0 ].tagName !== "HTML" ) {
			i.overflowOffset = i.scrollParentNotHidden.offset();
		}
	},
	drag: function( event, ui, i  ) {

		var o = i.options,
			scrolled = false,
			scrollParent = i.scrollParentNotHidden[ 0 ],
			document = i.document[ 0 ];

		if ( scrollParent !== document && scrollParent.tagName !== "HTML" ) {
			if ( !o.axis || o.axis !== "x" ) {
				if ( ( i.overflowOffset.top + scrollParent.offsetHeight ) - event.pageY < o.scrollSensitivity ) {
					scrollParent.scrollTop = scrolled = scrollParent.scrollTop + o.scrollSpeed;
				} else if ( event.pageY - i.overflowOffset.top < o.scrollSensitivity ) {
					scrollParent.scrollTop = scrolled = scrollParent.scrollTop - o.scrollSpeed;
				}
			}

			if ( !o.axis || o.axis !== "y" ) {
				if ( ( i.overflowOffset.left + scrollParent.offsetWidth ) - event.pageX < o.scrollSensitivity ) {
					scrollParent.scrollLeft = scrolled = scrollParent.scrollLeft + o.scrollSpeed;
				} else if ( event.pageX - i.overflowOffset.left < o.scrollSensitivity ) {
					scrollParent.scrollLeft = scrolled = scrollParent.scrollLeft - o.scrollSpeed;
				}
			}

		} else {

			if (!o.axis || o.axis !== "x") {
				if (event.pageY - $(document).scrollTop() < o.scrollSensitivity) {
					scrolled = $(document).scrollTop($(document).scrollTop() - o.scrollSpeed);
				} else if ($(window).height() - (event.pageY - $(document).scrollTop()) < o.scrollSensitivity) {
					scrolled = $(document).scrollTop($(document).scrollTop() + o.scrollSpeed);
				}
			}

			if (!o.axis || o.axis !== "y") {
				if (event.pageX - $(document).scrollLeft() < o.scrollSensitivity) {
					scrolled = $(document).scrollLeft($(document).scrollLeft() - o.scrollSpeed);
				} else if ($(window).width() - (event.pageX - $(document).scrollLeft()) < o.scrollSensitivity) {
					scrolled = $(document).scrollLeft($(document).scrollLeft() + o.scrollSpeed);
				}
			}

		}

		if (scrolled !== false && $.ui.ddmanager && !o.dropBehaviour) {
			$.ui.ddmanager.prepareOffsets(i, event);
		}

	}
});

$.ui.plugin.add("draggable", "snap", {
	start: function( event, ui, i ) {

		var o = i.options;

		i.snapElements = [];

		$(o.snap.constructor !== String ? ( o.snap.items || ":data(ui-draggable)" ) : o.snap).each(function() {
			var $t = $(this),
				$o = $t.offset();
			if (this !== i.element[0]) {
				i.snapElements.push({
					item: this,
					width: $t.outerWidth(), height: $t.outerHeight(),
					top: $o.top, left: $o.left
				});
			}
		});

	},
	drag: function( event, ui, inst ) {

		var ts, bs, ls, rs, l, r, t, b, i, first,
			o = inst.options,
			d = o.snapTolerance,
			x1 = ui.offset.left, x2 = x1 + inst.helperProportions.width,
			y1 = ui.offset.top, y2 = y1 + inst.helperProportions.height;

		for (i = inst.snapElements.length - 1; i >= 0; i--){

			l = inst.snapElements[i].left - inst.margins.left;
			r = l + inst.snapElements[i].width;
			t = inst.snapElements[i].top - inst.margins.top;
			b = t + inst.snapElements[i].height;

			if ( x2 < l - d || x1 > r + d || y2 < t - d || y1 > b + d || !$.contains( inst.snapElements[ i ].item.ownerDocument, inst.snapElements[ i ].item ) ) {
				if (inst.snapElements[i].snapping) {
					(inst.options.snap.release && inst.options.snap.release.call(inst.element, event, $.extend(inst._uiHash(), { snapItem: inst.snapElements[i].item })));
				}
				inst.snapElements[i].snapping = false;
				continue;
			}

			if (o.snapMode !== "inner") {
				ts = Math.abs(t - y2) <= d;
				bs = Math.abs(b - y1) <= d;
				ls = Math.abs(l - x2) <= d;
				rs = Math.abs(r - x1) <= d;
				if (ts) {
					ui.position.top = inst._convertPositionTo("relative", { top: t - inst.helperProportions.height, left: 0 }).top;
				}
				if (bs) {
					ui.position.top = inst._convertPositionTo("relative", { top: b, left: 0 }).top;
				}
				if (ls) {
					ui.position.left = inst._convertPositionTo("relative", { top: 0, left: l - inst.helperProportions.width }).left;
				}
				if (rs) {
					ui.position.left = inst._convertPositionTo("relative", { top: 0, left: r }).left;
				}
			}

			first = (ts || bs || ls || rs);

			if (o.snapMode !== "outer") {
				ts = Math.abs(t - y1) <= d;
				bs = Math.abs(b - y2) <= d;
				ls = Math.abs(l - x1) <= d;
				rs = Math.abs(r - x2) <= d;
				if (ts) {
					ui.position.top = inst._convertPositionTo("relative", { top: t, left: 0 }).top;
				}
				if (bs) {
					ui.position.top = inst._convertPositionTo("relative", { top: b - inst.helperProportions.height, left: 0 }).top;
				}
				if (ls) {
					ui.position.left = inst._convertPositionTo("relative", { top: 0, left: l }).left;
				}
				if (rs) {
					ui.position.left = inst._convertPositionTo("relative", { top: 0, left: r - inst.helperProportions.width }).left;
				}
			}

			if (!inst.snapElements[i].snapping && (ts || bs || ls || rs || first)) {
				(inst.options.snap.snap && inst.options.snap.snap.call(inst.element, event, $.extend(inst._uiHash(), { snapItem: inst.snapElements[i].item })));
			}
			inst.snapElements[i].snapping = (ts || bs || ls || rs || first);

		}

	}
});

$.ui.plugin.add("draggable", "stack", {
	start: function( event, ui, instance ) {
		var min,
			o = instance.options,
			group = $.makeArray($(o.stack)).sort(function(a, b) {
				return (parseInt($(a).css("zIndex"), 10) || 0) - (parseInt($(b).css("zIndex"), 10) || 0);
			});

		if (!group.length) { return; }

		min = parseInt($(group[0]).css("zIndex"), 10) || 0;
		$(group).each(function(i) {
			$(this).css("zIndex", min + i);
		});
		this.css("zIndex", (min + group.length));
	}
});

$.ui.plugin.add("draggable", "zIndex", {
	start: function( event, ui, instance ) {
		var t = $( ui.helper ),
			o = instance.options;

		if (t.css("zIndex")) {
			o._zIndex = t.css("zIndex");
		}
		t.css("zIndex", o.zIndex);
	},
	stop: function( event, ui, instance ) {
		var o = instance.options;

		if (o._zIndex) {
			$(ui.helper).css("zIndex", o._zIndex);
		}
	}
});

return $.ui.draggable;

}));
;

// resource: org.richfaces:inplaceBase.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.InplaceBase = function(id, options) {
        $super.constructor.call(this, id);
        var mergedOptions = $.extend({}, defaultOptions, options);
        this.editEvent = mergedOptions.editEvent;
        this.noneCss = mergedOptions.noneCss;
        this.changedCss = mergedOptions.changedCss;
        this.editCss = mergedOptions.editCss;
        this.defaultLabel = mergedOptions.defaultLabel;
        this.state = mergedOptions.state;

        this.options = mergedOptions;

        this.element = $(document.getElementById(id));
        this.editContainer = $(document.getElementById(id + "Edit"));
        this.element.bind(this.editEvent, $.proxy(this.__editHandler, this));
        this.isSaved = false;
        this.useDefaultLabel = false;
        this.editState = false;
    };

    rf.ui.InputBase.extend(rf.ui.InplaceBase);
    var $super = rf.ui.InplaceBase.$super;

    var defaultOptions = {
        editEvent: "click",
        state: "ready"
    };

    $.extend(rf.ui.InplaceBase.prototype, ( function () {

        var STATE = {
            READY : 'ready',
            CHANGED: 'changed',
            DISABLE: 'disable',
            EDIT: 'edit'
        };

        return {

            getLabel: function() {
            },

            setLabel: function(value) {
            },

            onshow: function() {
            },

            onhide: function() {
            },

            onsave: function() {
            },

            oncancel: function() {
            },

            save: function() {
                var value = this.__getValue()
                if (value.length > 0) {
                    this.setLabel(value);
                    this.useDefaultLabel = false;
                } else {
                    this.setLabel(this.defaultLabel);
                    this.useDefaultLabel = true;
                }

                this.isSaved = true;

                this.__applyChangedStyles();
                this.onsave();
            },

            cancel: function() {
                var text = "";
                if (!this.useDefaultLabel) {
                    text = this.getLabel();
                }
                this.__setValue(text);
                this.isSaved = true;
                this.oncancel();
            },

            open: function() {
                this.__editHandler();
            },

            isValueSaved: function() {
                return this.isSaved;
            },

            isEditState: function() {
                return this.editState;
            },

            __applyChangedStyles: function() {
                if (this.isValueChanged()) {
                    this.element.addClass(this.changedCss);
                } else {
                    this.element.removeClass(this.changedCss);
                }
            },

            __show: function() {
                this.scrollElements = rf.Event.bindScrollEventHandlers(this.id, this.__scrollHandler, this);
                this.editState = true;
                this.onshow();
            },

            __hide: function() {
                if (this.scrollElements) {
                    rf.Event.unbindScrollEventHandlers(this.scrollElements, this);
                    this.scrollElements = null;
                }
                this.editState = false;
                this.editContainer.addClass(this.noneCss);
                this.element.removeClass(this.editCss);
                this.onhide();
            },

            __editHandler: function(e) {
                this.isSaved = false;
                this.element.addClass(this.editCss);
                this.editContainer.removeClass(this.noneCss);
                this.__show();
            },
            __scrollHandler: function(e) {
                this.cancel();
            },

            destroy: function () {
                $super.destroy.call(this);
            }
        }

    })());

})(RichFaces.jQuery, window.RichFaces);
;

// resource: org.richfaces:JQuerySpinBtn.js
/* SpinButton control
 *
 * Adds bells and whistles to any ordinary textbox to
 * make it look and feel like a SpinButton Control.
 *
 * Originally written by George Adamson, Software Unity (george.jquery@softwareunity.com) August 2006.
 * - Added min/max options
 * - Added step size option
 * - Added bigStep (page up/down) option
 *
 * Modifications made by Mark Gibson, (mgibson@designlinks.net) September 2006:
 * - Converted to jQuery plugin
 * - Allow limited or unlimited min/max values
 * - Allow custom class names, and add class to input element
 * - Removed global vars
 * - Reset (to original or through config) when invalid value entered
 * - Repeat whilst holding mouse button down (with initial pause, like keyboard repeat)
 * - Support mouse wheel in Firefox
 * - Fix double click in IE
 * - Refactored some code and renamed some vars
 *
 * Tested in IE6, Opera9, Firefox 1.5
 * v1.0  11 Aug 2006 - George Adamson	- First release
 * v1.1     Aug 2006 - George Adamson	- Minor enhancements
 * v1.2  27 Sep 2006 - Mark Gibson		- Major enhancements
 * v1.3a 28 Sep 2006 - George Adamson	- Minor enhancements
 * rf1.3a 15 Nov 2007 - Pavel Yaschenko - some changes
 
 Sample usage:
 
	// Create group of settings to initialise spinbutton(s). (Optional)
	var myOptions = {
					min: 0,						// Set lower limit.
					max: 100,					// Set upper limit.
					step: 1,					// Set increment size.
					spinClass: mySpinBtnClass,	// CSS class to style the spinbutton. (Class also specifies url of the up/down button image.)
					upClass: mySpinUpClass,		// CSS class for style when mouse over up button.
					downClass: mySpinDnClass	// CSS class for style when mouse over down button.
					}
 
	$(document).ready(function(){

		// Initialise INPUT element(s) as SpinButtons: (passing options if desired)
		$("#myInputElement").SpinButton(myOptions);

	});
 
 */
var sbjQuery = jQuery;
sbjQuery.fn.SpinButton = function(cfg){
	return this.each(function(){

		// Apply specified options or defaults:
		// (Ought to refactor this some day to use $.extend() instead)
		this.spinCfg = {
			//min: cfg && cfg.min ? Number(cfg.min) : null,
			//max: cfg && cfg.max ? Number(cfg.max) : null,
			min: cfg && !isNaN(parseFloat(cfg.min)) ? Number(cfg.min) : null,	// Fixes bug with min:0
			max: cfg && !isNaN(parseFloat(cfg.max)) ? Number(cfg.max) : null,
			step: cfg && cfg.step ? Number(cfg.step) : 1,
			page: cfg && cfg.page ? Number(cfg.page) : 10,
			upClass: cfg && cfg.upClass ? cfg.upClass : 'up',
			downClass: cfg && cfg.downClass ? cfg.downClass : 'down',
			reset: cfg && cfg.reset ? cfg.reset : this.value,
			delay: cfg && cfg.delay ? Number(cfg.delay) : 500,
			interval: cfg && cfg.interval ? Number(cfg.interval) : 100,
			_btn_width: 20,
			_btn_height: 12,
			_direction: null,
			_delay: null,
			_repeat: null,
			
			digits: cfg && cfg.digits ? Number(cfg.digits) : 1			
		};
		
		this.adjustValue = function(i){
			var v = this.value.toLowerCase();
			if (v=="am") 
			{
				this.value="PM";
				return; 
			}
			else if (v=="pm") {
				this.value="AM";
				return;
			} 
			v = (isNaN(this.value) ? this.spinCfg.reset : Number(this.value)) + Number(i);
			if (this.spinCfg.min !== null) v = (v<this.spinCfg.min ? (this.spinCfg.max != null ? this.spinCfg.max : this.spinCfg.min) : v);
			if (this.spinCfg.max !== null) v = (v>this.spinCfg.max ? (this.spinCfg.min != null ? this.spinCfg.min : this.spinCfg.max) : v);

			var value = String(v);
			while (value.length<this.spinCfg.digits) value="0"+value;
			
			this.value = value;
		};
		
		sbjQuery(this)
//		.addClass(cfg && cfg.spinClass ? cfg.spinClass : 'spin-button')
//		
//		.mousemove(function(e){
//			// Determine which button mouse is over, or not (spin direction):
//			var x = e.pageX || e.x;
//			var y = e.pageY || e.y;
//			var el = e.target || e.srcElement;
//			var direction = 
//				(x > coord(el,'offsetLeft') + el.offsetWidth - this.spinCfg._btn_width)
//				? ((y < coord(el,'offsetTop') + this.spinCfg._btn_height) ? 1 : -1) : 0;
//			
//			if (direction !== this.spinCfg._direction) {
//				// Style up/down buttons:
//				switch(direction){
//					case 1: // Up arrow:
//						sbjQuery(this).removeClass(this.spinCfg.downClass).addClass(this.spinCfg.upClass);
//						break;
//					case -1: // Down arrow:
//						sbjQuery(this).removeClass(this.spinCfg.upClass).addClass(this.spinCfg.downClass);
//						break;
//					default: // Mouse is elsewhere in the textbox
//						sbjQuery(this).removeClass(this.spinCfg.upClass).removeClass(this.spinCfg.downClass);
//				}
//				
//				// Set spin direction:
//				this.spinCfg._direction = direction;
//			}
//		})
//		
//		.mouseout(function(){
//			// Reset up/down buttons to their normal appearance when mouse moves away:
//			sbjQuery(this).removeClass(this.spinCfg.upClass).removeClass(this.spinCfg.downClass);
//			this.spinCfg._direction = null;
//		})
		
//		.mousedown(function(e){
//			if (this.spinCfg._direction != 0) {
//				// Respond to click on one of the buttons:
//				var self = this;
//				var adjust = function() {
//					self.adjustValue(self.spinCfg._direction * self.spinCfg.step);
//				};
//			
//				adjust();
//				
//				// Initial delay before repeating adjustment
//				self.spinCfg._delay = window.setTimeout(function() {
//					adjust();
//					// Repeat adjust at regular intervals
//					self.spinCfg._repeat = window.setInterval(adjust, self.spinCfg.interval);
//				}, self.spinCfg.delay);
//			}
//		})
//		
//		.mouseup(function(e){
//			// Cancel repeating adjustment
//			window.clearInterval(this.spinCfg._repeat);
//			window.clearTimeout(this.spinCfg._delay);
//		})
//		
//		.dblclick(function(e) {
//			if (sbjQuery.browser.msie)
//				this.adjustValue(this.spinCfg._direction * this.spinCfg.step);
//		})
		
		.keydown(function(e){
			// Respond to up/down arrow keys.
			switch(e.keyCode){
				case 38: this.adjustValue(this.spinCfg.step);  break; // Up
				case 40: this.adjustValue(-this.spinCfg.step); break; // Down
				case 33: this.adjustValue(this.spinCfg.page);  break; // PageUp
				case 34: this.adjustValue(-this.spinCfg.page); break; // PageDown
			}
		})

		.bind("mousewheel", function(e){
			// Respond to mouse wheel in IE. (It returns up/dn motion in multiples of 120)
			if (e.wheelDelta >= 120)
				this.adjustValue(this.spinCfg.step);
			else if (e.wheelDelta <= -120)
				this.adjustValue(-this.spinCfg.step);
			
			e.preventDefault();
		})
		
		.change(function(e){
			this.adjustValue(0);
		});
		
		var self = this;
		
		var btnUp = document.getElementById(this.id + 'BtnUp');
		sbjQuery(btnUp)
			.mousedown(function(e){
				// Respond to click on one of the buttons:
				var adjust = function() {
					self.adjustValue(self.spinCfg.step);
				};
			
				adjust();
				
				// Initial delay before repeating adjustment
				self.spinCfg._delay = window.setTimeout(function() {
					adjust();
					// Repeat adjust at regular intervals
					self.spinCfg._repeat = window.setInterval(adjust, self.spinCfg.interval);
				}, self.spinCfg.delay);
				self.spinCfg._repeater = true;
				return false;
			})
			
			.mouseup(function(e){
				// Cancel repeating adjustment
				self.spinCfg._repeater = false;
				window.clearInterval(self.spinCfg._repeat);
				window.clearTimeout(self.spinCfg._delay);
			})
			
			.dblclick(function(e) {
				if (RichFaces.browser.msie)
					self.adjustValue(self.spinCfg.step);
			})
			.mouseout(function(e){
				// Cancel repeating adjustment
				if (self.spinCfg._repeater)
				{
					self.spinCfg._repeater = false
					window.clearInterval(self.spinCfg._repeat);
					window.clearTimeout(self.spinCfg._delay);
				}
			});
		
		var btnDown = document.getElementById(this.id + 'BtnDown');
		sbjQuery(btnDown)
			.mousedown(function(e){
				// Respond to click on one of the buttons:
				var adjust = function() {
					self.adjustValue(-self.spinCfg.step);
				};
			
				adjust();
				
				// Initial delay before repeating adjustment
				self.spinCfg._delay = window.setTimeout(function() {
					adjust();
					// Repeat adjust at regular intervals
					self.spinCfg._repeat = window.setInterval(adjust, self.spinCfg.interval);
				}, self.spinCfg.delay);
				self.spinCfg._repeater = true;
				return false;
			})
			
			.mouseup(function(e){
				// Cancel repeating adjustment
				self.spinCfg._repeater = false;
				window.clearInterval(self.spinCfg._repeat);
				window.clearTimeout(self.spinCfg._delay);
			})
			
			.dblclick(function(e) {
				if (RichFaces.browser.msie)
					self.adjustValue(-self.spinCfg.step);
			})
			.mouseout(function(e){
				// Cancel repeating adjustment
				if (self.spinCfg._repeater)
				{
					self.spinCfg._repeater = false
					window.clearInterval(self.spinCfg._repeat);
					window.clearTimeout(self.spinCfg._delay);
				}
			});
			
		
		if (this.addEventListener) {
			// Respond to mouse wheel in Firefox
			this.addEventListener('DOMMouseScroll', function(e) {
				if (e.detail > 0)
					this.adjustValue(-this.spinCfg.step);
				else if (e.detail < 0)
					this.adjustValue(this.spinCfg.step);
				
				e.preventDefault();
			}, false);
		}
	});
	
	function coord(el,prop) {
		var c = el[prop], b = document.body;
		
		while ((el = el.offsetParent) && (el != b)) {
			if (!RichFaces.browser.msie || (el.currentStyle.position != 'relative'))
				c += el[prop];
		}
		
		return c;
	}
};
;

// resource: org.richfaces:editor.js
/**
 * @author Lukas Fryc
 */

(function($, rf) {
    rf.ui = rf.ui || {};

    /**
     * Default component configuration
     */
    var defaultOptions = {
        toolbar : 'Basic',
        skin: 'moono',
        readonly : false,
        style : '',
        styleClass : '',
        editorStyle : '',
        editorClass : '',
        width : '100%',
        height : '200px'
    };
    
    var eventsForDirty = [ "key", "paste", "undo", "redo" ];

    rf.ui.Editor = function(componentId, options, config) {
        $super.constructor.call(this, componentId);
        this.options = $.extend({}, defaultOptions, options);

        this.componentId = componentId;
        this.textareaId = componentId + ':inp';
        this.editorElementId = 'cke_' + this.textareaId;
        this.valueChanged = false;
        this.dirtyState = false;
        this.config = $.extend({}, config);

        this.attachToDom(this.componentId);

        $(document).ready($.proxy(this.__initializationHandler, this));
        rf.Event.bindById(this.__getTextarea(), 'init', this.options.oninit, this);
        rf.Event.bindById(this.__getTextarea(), 'dirty', this.options.ondirty, this);
    };

    rf.BaseComponent.extend(rf.ui.Editor);

    var $super = rf.ui.Editor.$super;

    $.extend(rf.ui.Editor.prototype, {

        name : "Editor",

        __initializationHandler : function() {
            this.ckeditor = CKEDITOR.replace(this.textareaId, this.__getConfiguration());

            // register event handlers
            if (this.__getForm()) {
                this.__updateTextareaHandlerWrapper = rf.Event.bind(this.__getForm(), 'ajaxsubmit', $.proxy(this.__updateTextareaHandler, this));
            }
            this.ckeditor.on('instanceReady', $.proxy(this.__instanceReadyHandler, this));
            this.ckeditor.on('blur', $.proxy(this.__blurHandler, this));
            this.ckeditor.on('focus', $.proxy(this.__focusHandler, this));
            // register handlers for 'dirty' event
            for (var i = 0; i < eventsForDirty.length; i++) {
                this.ckeditor.on(eventsForDirty[i], $.proxy(this.__checkDirtyHandlerWithDelay, this));
            }
            // interval for dirty checking
            this.dirtyCheckingInterval = window.setInterval($.proxy(this.__checkDirtyHandler, this), 100);
        },
        
        __checkDirtyHandlerWithDelay : function() {
            window.setTimeout($.proxy(this.__checkDirtyHandler, this), 0);
        },
        
        __checkDirtyHandler : function() {
            if (this.ckeditor.checkDirty()) {
                this.dirtyState = true;
                this.valueChanged = true;
                this.ckeditor.resetDirty();
                this.__dirtyHandler();
            }
        },
        
        __dirtyHandler : function() {
            this.invokeEvent.call(this, "dirty", document.getElementById(this.textareaId));
        },
        
        __updateTextareaHandler : function() {
            this.ckeditor.updateElement();
        },

        __instanceReadyHandler : function(e) {
            this.__setupStyling();
            this.__setupPassThroughAttributes();

            this.invokeEvent.call(this, "init", document.getElementById(this.textareaId), e);
        },

        __blurHandler : function(e) {
            this.invokeEvent.call(this, "blur", document.getElementById(this.textareaId), e);
            if (this.isDirty()) {
                this.valueChanged = true;
                this.__changeHandler();
            }
            this.dirtyState = false;
        },

        __focusHandler : function(e) {
            this.invokeEvent.call(this, "focus", document.getElementById(this.textareaId), e);
        },

        __changeHandler : function(e) {
            this.invokeEvent.call(this, "change", document.getElementById(this.textareaId), e);
        },

        __getTextarea : function() {
            return $(document.getElementById(this.textareaId));
        },

        /**
         * Returns the form where this editor component is placed
         */
        __getForm : function() {
            return $('form').has(this.__getTextarea()).get(0);
        },

        __getConfiguration : function() {
            var textarea = this.__getTextarea();
            return $.extend({
                skin : this.options.skin,
                toolbar : this.__getToolbar(),
                readOnly : textarea.attr('readonly') || this.options.readonly,
                width : this.__resolveUnits(this.options.width),
                height : this.__resolveUnits(this.options.height),
                bodyClass : 'rf-ed-b',
                defaultLanguage : this.options.lang,
                contentsLanguage : this.options.lang
            }, this.config);
        },

        __setupStyling : function() {
            var span = $(document.getElementById(this.editorElementId));
            if (!span.hasClass('rf-ed')) {
                span.addClass('rf-ed');
            }
            var styleClass = $.trim(this.options.styleClass + ' ' + this.options.editorClass);
            if (this.initialStyle == undefined) {
                this.initialStyle = span.attr('style');
            }
            var style = this.__concatStyles(this.initialStyle, this.options.style, this.options.editorStyle);
            if (this.oldStyleClass !== styleClass) {
                if (this.oldStyleClass) {
                    span.removeClass(this.oldStyleClass);
                }
                span.addClass(styleClass);
                this.oldStyleClass = styleClass;
            }
            if (this.oldStyle !== style) {
                span.attr('style', style);
                this.oldStyle = style;
            }
        },

        __setupPassThroughAttributes : function() {
            var textarea = this.__getTextarea();
            var span = $(document.getElementById(this.editorElementId));

            // title
            span.attr('title', textarea.attr('title'));
        },

        __concatStyles : function() {
            var result = "";
            for ( var i = 0; i < arguments.length; i++) {
                var style = $.trim(arguments[i]);
                if (style) {
                    result = result + style + "; ";
                }
            }
            return result;
        },

        __getToolbar : function() {
            var toolbar = this.options.toolbar;

            var lowercase = toolbar.toLowerCase();
            if (lowercase === 'basic') {
                return 'Basic';
            }
            if (lowercase === 'full') {
                return 'Full';
            }

            return toolbar;
        },

        __setOptions : function(options) {
            this.options = $.extend({}, defaultOptions, options);
        },

        __resolveUnits : function(dimension) {
            var dimension = $.trim(dimension);
            if (dimension.match(/^[0-9]+$/)) {
                return dimension + 'px';
            } else {
                return dimension;
            }
        },

        getEditor : function() {
            return this.ckeditor;
        },

        setValue : function(newValue) {
            this.ckeditor.setData(newValue, $.proxy(function() {
                this.valueChanged = false;
                this.dirtyState = false;
                this.ckeditor.resetDirty();
            }, this));
        },

        getValue : function() {
            return this.ckeditor.getData();
        },

        getInput : function() {
            return document.getElementById(this.textareaId);
        },

        focus : function() {
            this.ckeditor.focus();
        },

        blur : function() {
            this.ckeditor.focusManager.blur(true);
        },

        isFocused : function() {
            return this.ckeditor.focusManager.hasFocus;
        },

        isDirty : function() {
            return this.dirtyState || this.ckeditor.checkDirty();
        },

        isValueChanged : function() {
            return this.valueChanged || this.isDirty();
        },

        setReadOnly : function(readOnly) {
            this.ckeditor.setReadOnly(readOnly !== false);
        },

        isReadOnly : function() {
            return this.ckeditor.readOnly;
        },

        destroy : function() {
            window.clearInterval(this.dirtyCheckingInterval);
            
            if (this.__getForm()) {
                rf.Event.unbind(this.__getForm(), 'ajaxsubmit', this.__updateTextareaHandlerWrapper);
            }

            if (this.ckeditor) {
                this.ckeditor.destroy();
                this.ckeditor = null;
            }

            this.__getTextarea().show();

            $super.destroy.call(this);
        }
    });
})(RichFaces.jQuery, RichFaces);
;

// resource: org.richfaces:Autocomplete.js
(function ($, rf) {

    /*
     * TODO: add user's event handlers call from options
     * TODO: add fire events
     */

    rf.ui = rf.ui || {};
    // Constructor definition
    rf.ui.Autocomplete = function(componentId, fieldId, options) {
        this.namespace = "." + rf.Event.createNamespace(this.name, componentId);
        this.options = {};
        // call constructor of parent class
        $super.constructor.call(this, componentId, componentId + ID.SELECT, fieldId, options);
        this.attachToDom();
        this.options = $.extend(this.options, defaultOptions, options);
        this.value = "";
        this.index = null;
        this.isFirstAjax = true;
        updateTokenOptions.call(this);
        bindEventHandlers.call(this);
        updateItemsList.call(this, "");
    };

    // Extend component class and add protected methods from parent class to our container
    rf.ui.AutocompleteBase.extend(rf.ui.Autocomplete);

    // define super class link
    var $super = rf.ui.Autocomplete.$super;

    var defaultOptions = {
        itemClass:'rf-au-itm',
        selectedItemClass:'rf-au-itm-sel',
        subItemClass:'rf-au-opt',
        selectedSubItemClass:'rf-au-opt-sel',
        autofill:true,
        minChars:1,
        selectFirst:true,
        ajaxMode:true,
        lazyClientMode:false,
        isCachedAjax:true,
        tokens: "",
        attachToBody:true,
        filterFunction: undefined
        //nothingLabel = "Nothing";
    };

    var ID = {
        SELECT:'List',
        ITEMS:'Items',
        VALUE:'Value'
    };

    var REGEXP_TRIM = /^[\n\s]*(.*)[\n\s]*$/;

    var getData = function (nodeList) {
        var data = [];
        nodeList.each(function () {
            data.push($(this).text().replace(REGEXP_TRIM, "$1"));
        });
        return data;
    }

    var updateTokenOptions = function () {
        this.useTokens = (typeof this.options.tokens == "string" && this.options.tokens.length > 0);
        if (this.useTokens) {
            var escapedTokens = this.options.tokens.split('').join("\\");
            this.REGEXP_TOKEN_RIGHT = new RegExp('[' + escapedTokens + ']', 'i');
            this.getLastTokenIndex = function(value) {
                return RichFaces.ui.Autocomplete.__getLastTokenIndex(escapedTokens, value);
            }
        }
    };

    var bindEventHandlers = function () {
        var list = $(rf.getDomElement(this.id + ID.ITEMS).parentNode);
        list.on("click" + this.namespace, "."+this.options.itemClass, $.proxy(onMouseClick, this));
        // The mouseenter event is available only natively in Internet Explorer, however jQuery emulates it in other browsers
        list.on("mouseenter" + this.namespace, "."+this.options.itemClass, $.proxy(onMouseEnter, this));
    };

    var onMouseEnter = function(event) {
        var element = $(event.target);

        if (element && !element.hasClass(this.options.itemClass)) {
            element = element.parents("." + this.options.itemClass).get(0);
        }

        if (element) {
            var index = this.items.index(element);
            selectItem.call(this, event, index);
        }
    };

    var onMouseClick = function(event) {
        var element = $(event.target);

        if (element && !element.hasClass(this.options.itemClass)) {
            element = element.parents("." + this.options.itemClass).get(0);
        }

        if (element) {
            this.__onEnter(event);
            rf.Selection.setCaretTo(rf.getDomElement(this.fieldId));
            this.__hide(event);
        }
    };

    var updateItemsList = function (value, fetchValues) {
        var itemsContainer = $(rf.getDomElement(this.id + ID.ITEMS));
        this.items = itemsContainer.find("." + this.options.itemClass);
        var componentData = itemsContainer.data("componentData");
        itemsContainer.removeData("componentData");
        if (this.items.length > 0) {
            this.cache = new rf.utils.Cache((this.options.ajaxMode ? value : ""), this.items, fetchValues || componentData || getData, !this.options.ajaxMode);
        }
    };

    var scrollToSelectedItem = function() {
        var offset = 0;
        this.items.slice(0, this.index).each(function() {
            offset += this.offsetHeight;
        });
        var parentContainer = $(rf.getDomElement(this.id + ID.ITEMS)).parent();
        if (offset < parentContainer.scrollTop()) {
            parentContainer.scrollTop(offset);
        } else {
            offset += this.items.eq(this.index).outerHeight();
            if (offset - parentContainer.scrollTop() > parentContainer.innerHeight()) {
                parentContainer.scrollTop(offset - parentContainer.innerHeight());
            }
        }
    };

    var autoFill = function (inputValue, value) {
        if (this.options.autofill && value.toLowerCase().indexOf(inputValue.toLowerCase()) == 0) {
            var field = rf.getDomElement(this.fieldId);
            var start = rf.Selection.getStart(field);
            this.__setInputValue(inputValue + value.substring(inputValue.length));
            var end = start + value.length - inputValue.length;
            rf.Selection.set(field, start, end);
        }
    };

    var callAjax = function(event, callback) {

        rf.getDomElement(this.id + ID.VALUE).value = this.value;

        var _this = this;
        var _event = event;
        var ajaxSuccess = function (event) {
            if (_this.options.minChars <= _this.value.length) {
                updateItemsList.call(_this, _this.value, event.componentData && event.componentData[_this.id]);
            }
            if (_this.options.lazyClientMode && _this.value.length != 0) {
                updateItemsFromCache.call(_this, _this.value);
            }
            if (_this.items.length != 0) {
                if (callback) {
                    (_this.focused || _this.isMouseDown) && callback.call(_this, _event);
                } else {
                    _this.isVisible && _this.options.selectFirst && selectItem.call(_this, _event, 0);
                }
            } else {
                _this.__hide(_event);
            }
        };

        var ajaxError = function (event) {
            _this.__hide(_event);
            clearItems.call(_this);
        };

        this.isFirstAjax = false;
        //caution: JSF submits inputs with empty names causing "WARNING: Parameters: Invalid chunk ignored." in Tomcat log
        var params = {};
        params[this.id + ".ajax"] = "1";
        rf.ajax(this.id, event, {parameters: params, error: ajaxError, complete:ajaxSuccess});
    };

    var clearSelection = function () {
        if (this.index != null) {
            var element = this.items.eq(this.index);
            if (element.removeClass(this.options.selectedItemClass).hasClass(this.options.subItemClass)) {
                element.removeClass(this.options.selectedSubItemClass);
            }
            this.index = null;
        }
    };

    var selectItem = function(event, index, isOffset) {
        if (this.items.length == 0 || (!isOffset && index == this.index)) return;

        if (index == null || index == undefined) {
            clearSelection.call(this);
            return;
        }

        if (isOffset) {
            if (this.index == null) {
                index = 0;
            } else {
                index = this.index + index;
            }
        }
        if (index < 0) {
            index = 0;
        } else if (index >= this.items.length) {
            index = this.items.length - 1;
        }
        if (index == this.index) return;

        clearSelection.call(this);
        this.index = index;

        var item = this.items.eq(this.index);
        if (item.addClass(this.options.selectedItemClass).hasClass(this.options.subItemClass)) {
            item.addClass(this.options.selectedSubItemClass);
        }
        scrollToSelectedItem.call(this);
        if (event &&
            event.keyCode != rf.KEYS.BACKSPACE &&
            event.keyCode != rf.KEYS.DEL &&
            event.keyCode != rf.KEYS.LEFT &&
            event.keyCode != rf.KEYS.RIGHT) {
            autoFill.call(this, this.value, getSelectedItemValue.call(this));
        }
    };

    var updateItemsFromCache = function (value) {
        var newItems = this.cache.getItems(value, this.options.filterFunction);
        this.items = $(newItems);
        //TODO: works only with simple markup, not with <tr>
        $(rf.getDomElement(this.id + ID.ITEMS)).empty().append(this.items);
    };

    var clearItems = function () {
        $(rf.getDomElement(this.id + ID.ITEMS)).removeData().empty();
        this.items = [];
    };

    var onChangeValue = function (event, value, callback) {
        selectItem.call(this, event);

        // value is undefined if called from AutocompleteBase onChange
        var subValue = (typeof value == "undefined") ? this.__getSubValue() : value;
        var oldValue = this.value;
        this.value = subValue;

        if ((this.options.isCachedAjax || !this.options.ajaxMode) &&
            this.cache && this.cache.isCached(subValue)) {
            if (oldValue != subValue) {
                updateItemsFromCache.call(this, subValue);
            }
            if (this.items.length != 0) {
                callback && callback.call(this, event);
            } else {
                this.__hide(event);
            }
            if (event.keyCode == rf.KEYS.RETURN || event.type == "click") {
                this.__setInputValue(subValue);
            } else if (this.options.selectFirst) {
                selectItem.call(this, event, 0);
            }
        } else {
            if (event.keyCode == rf.KEYS.RETURN || event.type == "click") {
                this.__setInputValue(subValue);
            }
            if (subValue.length >= this.options.minChars) {
                if ((this.options.ajaxMode || this.options.lazyClientMode) && (oldValue != subValue || (oldValue === '' && subValue === ''))) {
                    callAjax.call(this, event, callback);
                }
            } else {
                if (this.options.ajaxMode) {
                    clearItems.call(this);
                    this.__hide(event);
                }
            }
        }

    };

    var getSelectedItemValue = function () {
        if (this.index != null) {
            var element = this.items.eq(this.index);
            return this.cache.getItemValue(element);
        }
        return undefined;
    };

    var getSubValue = function () {
        //TODO: add posibility to use space chars before and after tokens if space not a token char
        if (this.useTokens) {
            var field = rf.getDomElement(this.fieldId);
            var value = field.value;
            
            var cursorPosition = rf.Selection.getStart(field);
            var beforeCursorStr = value.substring(0, cursorPosition);
            var afterCursorStr = value.substring(cursorPosition);
            var result = beforeCursorStr.substring(this.getLastTokenIndex(beforeCursorStr));
            r = afterCursorStr.search(this.REGEXP_TOKEN_RIGHT);
            if (r == -1) r = afterCursorStr.length;
            result += afterCursorStr.substring(0, r);

            return result;
        } else {
            return this.getValue();
        }
    };
    
    var getCursorPosition = function(field) {
        var pos = rf.Selection.getStart(field);
        if (pos <= 0) {
            // when cursorPosition is not determined (input is not focused),
            // use position of last token occurence) 
            pos = this.getLastTokenIndex(field.value);
        }
        return pos;
    }

    var updateInputValue = function (value) {
        var field = rf.getDomElement(this.fieldId);
        var inputValue = field.value;

        var cursorPosition = this.__getCursorPosition(field);
        var beforeCursorStr = inputValue.substring(0, cursorPosition);
        var afterCursorStr = inputValue.substring(cursorPosition);
        var pos = this.getLastTokenIndex(beforeCursorStr);
        var startPos = pos != -1 ? pos : beforeCursorStr.length;
        pos = afterCursorStr.search(this.REGEXP_TOKEN_RIGHT);
        var endPos = pos != -1 ? pos : afterCursorStr.length;

        var beginNewValue = inputValue.substring(0, startPos) + value;
        cursorPosition = beginNewValue.length;
        field.value = beginNewValue + afterCursorStr.substring(endPos);
        field.focus();
        rf.Selection.setCaretTo(field, cursorPosition);
        return field.value;
    };

    var getPageLastItem = function() {
        if (this.items.length == 0) return -1;
        var parentContainer = $(rf.getDomElement(this.id + ID.ITEMS)).parent();
        var h = parentContainer.scrollTop() + parentContainer.innerHeight() + this.items[0].offsetTop;
        var item;
        var i = (this.index != null && this.items[this.index].offsetTop <= h) ? this.index : 0;
        for (i; i < this.items.length; i++) {
            item = this.items[i];
            if (item.offsetTop + item.offsetHeight > h) {
                i--;
                break;
            }
        }
        if (i != this.items.length - 1 && i == this.index) {
            h += this.items[i].offsetTop - parentContainer.scrollTop();
            for (++i; i < this.items.length; i++) {
                item = this.items[i];
                if (item.offsetTop + item.offsetHeight > h) {
                    break;
                }
            }
        }
        return i;
    };

    var getPageFirstItem = function() {
        if (this.items.length == 0) return -1;
        var parentContainer = $(rf.getDomElement(this.id + ID.ITEMS)).parent();
        var h = parentContainer.scrollTop() + this.items[0].offsetTop;
        var item;
        var i = (this.index != null && this.items[this.index].offsetTop >= h) ? this.index - 1 : this.items.length - 1;
        for (i; i >= 0; i--) {
            item = this.items[i];
            if (item.offsetTop < h) {
                i++;
                break;
            }
        }
        if (i != 0 && i == this.index) {
            h = this.items[i].offsetTop - parentContainer.innerHeight();
            if (h < this.items[0].offsetTop) h = this.items[0].offsetTop;
            for (--i; i >= 0; i--) {
                item = this.items[i];
                if (item.offsetTop < h) {
                    i++;
                    break;
                }
            }
        }
        return i;
    };

    /*
     * Prototype definition
     */
    $.extend(rf.ui.Autocomplete.prototype, (function () {
        return {
            /*
             * public API functions
             */
            name:"Autocomplete",
            /*
             * Protected methods
             */
            __updateState: function (event) {
                var subValue = this.__getSubValue();
                // called from AutocompleteBase when not actually value changed
                if (this.items.length == 0 && this.isFirstAjax) {
                    if ((this.options.ajaxMode && subValue.length >= this.options.minChars) || this.options.lazyClientMode) {
                        this.value = subValue;
                        callAjax.call(this, event, this.__show);
                        return true;
                    }
                }
                return false;
            },
            __getSubValue: getSubValue,
            __getCursorPosition: getCursorPosition,
            __updateInputValue: function (value) {
                if (this.useTokens) {
                    return updateInputValue.call(this, value);
                } else {
                    return $super.__updateInputValue.call(this, value);
                }
            },
            __setInputValue: function (value) {
                this.currentValue = this.__updateInputValue(value);
            },
            __onChangeValue: onChangeValue,
            /*
             * Override abstract protected methods
             */
            __onKeyUp: function (event) {
                selectItem.call(this, event, -1, true);
            },
            __onKeyDown: function (event) {
                selectItem.call(this, event, 1, true);
            },
            __onPageUp: function (event) {
                selectItem.call(this, event, getPageFirstItem.call(this));
            },
            __onPageDown: function (event) {
                selectItem.call(this, event, getPageLastItem.call(this));
            },
            __onKeyHome: function (event) {
                selectItem.call(this, event, 0);
            },
            __onKeyEnd: function (event) {
                selectItem.call(this, event, this.items.length - 1);
            },
            __onBeforeShow: function (event) {
            },
            __onEnter: function (event) {
                var value = getSelectedItemValue.call(this);
                this.__onChangeValue(event, value);
                this.invokeEvent("selectitem", rf.getDomElement(this.fieldId), event, value);
            },
            __onShow: function (event) {
                if (this.options.selectFirst) {
                    selectItem.call(this, event, 0);
                }
            },
            __onHide: function (event) {
                selectItem.call(this, event);
            },
            /*
             * Destructor
             */
            destroy: function () {
                //TODO: add all unbind
                this.items = null;
                this.cache = null;
                var itemsContainer = rf.getDomElement(this.id + ID.ITEMS);
                $(itemsContainer).removeData();
                rf.Event.unbind(itemsContainer.parentNode, this.namespace);
                this.__conceal();
                $super.destroy.call(this);
            }
        };
    })());

    $.extend(rf.ui.Autocomplete, {
            setData: function (id, data) {
                $(rf.getDomElement(id)).data("componentData", data);
            },
            
            __getLastTokenIndex:  function (tokens, value) {
                var LAST_TOKEN_OCCURENCE = new RegExp("[" + tokens + "][^" + tokens + "]*$", "i");
                var AFTER_LAST_TOKEN_WITH_SPACES = new RegExp("[^" + tokens + " ]", "i");
                
                var value = value || "";

                var lastTokenIndex = value.search(LAST_TOKEN_OCCURENCE);
                if (lastTokenIndex < 0) {
                    return 0;
                }
                var beforeToken = value.substring(lastTokenIndex);
                var afterLastTokenIndex = beforeToken.search(AFTER_LAST_TOKEN_WITH_SPACES);
                if (afterLastTokenIndex <= 0) {
                    afterLastTokenIndex = beforeToken.length;
                }
            
                return lastTokenIndex + afterLastTokenIndex;
            }
        });

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:collapsiblePanelItem.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    var DEFAULT_OPTIONS = {switchMode: 'ajax'};

    rf.ui.CollapsiblePanelItem = rf.ui.TogglePanelItem.extendClass({

            init : function (componentId, options) {
                rf.ui.TogglePanelItem.call(this, componentId, $.extend({}, DEFAULT_OPTIONS, options));

                this.headerClass = "rf-cp-hdr-" + this.__state();
            },

            __enter : function () {
                this.__content().show();
                this.__header().addClass(this.headerClass);

                return true;
            },

            __leave : function () {
                this.__content().hide();

                if (this.options.switchMode == 'client') {
                    this.__header().removeClass(this.headerClass);
                }

                return true;
            },

            __state : function () {
                return this.getName() === "true" ? "exp" : "colps";
            },

            __content : function () {
                return $(rf.getDomElement(this.id));
            },

            __header : function () {
                return $(rf.getDomElement(this.togglePanelId + ":header"));
            }
        });
})(RichFaces.jQuery, RichFaces);
;

// resource: com.jqueryui:droppable.js
/*!
 * jQuery UI Droppable 1.11.4
 * http://jqueryui.com
 *
 * Copyright jQuery Foundation and other contributors
 * Released under the MIT license.
 * http://jquery.org/license
 *
 * http://api.jqueryui.com/droppable/
 */
(function( factory ) {
	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define([
			"jquery",
			"./core",
			"./widget",
			"./mouse",
			"./draggable"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
}(function( $ ) {

$.widget( "ui.droppable", {
	version: "1.11.4",
	widgetEventPrefix: "drop",
	options: {
		accept: "*",
		activeClass: false,
		addClasses: true,
		greedy: false,
		hoverClass: false,
		scope: "default",
		tolerance: "intersect",

		// callbacks
		activate: null,
		deactivate: null,
		drop: null,
		out: null,
		over: null
	},
	_create: function() {

		var proportions,
			o = this.options,
			accept = o.accept;

		this.isover = false;
		this.isout = true;

		this.accept = $.isFunction( accept ) ? accept : function( d ) {
			return d.is( accept );
		};

		this.proportions = function( /* valueToWrite */ ) {
			if ( arguments.length ) {
				// Store the droppable's proportions
				proportions = arguments[ 0 ];
			} else {
				// Retrieve or derive the droppable's proportions
				return proportions ?
					proportions :
					proportions = {
						width: this.element[ 0 ].offsetWidth,
						height: this.element[ 0 ].offsetHeight
					};
			}
		};

		this._addToManager( o.scope );

		o.addClasses && this.element.addClass( "ui-droppable" );

	},

	_addToManager: function( scope ) {
		// Add the reference and positions to the manager
		$.ui.ddmanager.droppables[ scope ] = $.ui.ddmanager.droppables[ scope ] || [];
		$.ui.ddmanager.droppables[ scope ].push( this );
	},

	_splice: function( drop ) {
		var i = 0;
		for ( ; i < drop.length; i++ ) {
			if ( drop[ i ] === this ) {
				drop.splice( i, 1 );
			}
		}
	},

	_destroy: function() {
		var drop = $.ui.ddmanager.droppables[ this.options.scope ];

		this._splice( drop );

		this.element.removeClass( "ui-droppable ui-droppable-disabled" );
	},

	_setOption: function( key, value ) {

		if ( key === "accept" ) {
			this.accept = $.isFunction( value ) ? value : function( d ) {
				return d.is( value );
			};
		} else if ( key === "scope" ) {
			var drop = $.ui.ddmanager.droppables[ this.options.scope ];

			this._splice( drop );
			this._addToManager( value );
		}

		this._super( key, value );
	},

	_activate: function( event ) {
		var draggable = $.ui.ddmanager.current;
		if ( this.options.activeClass ) {
			this.element.addClass( this.options.activeClass );
		}
		if ( draggable ){
			this._trigger( "activate", event, this.ui( draggable ) );
		}
	},

	_deactivate: function( event ) {
		var draggable = $.ui.ddmanager.current;
		if ( this.options.activeClass ) {
			this.element.removeClass( this.options.activeClass );
		}
		if ( draggable ){
			this._trigger( "deactivate", event, this.ui( draggable ) );
		}
	},

	_over: function( event ) {

		var draggable = $.ui.ddmanager.current;

		// Bail if draggable and droppable are same element
		if ( !draggable || ( draggable.currentItem || draggable.element )[ 0 ] === this.element[ 0 ] ) {
			return;
		}

		if ( this.accept.call( this.element[ 0 ], ( draggable.currentItem || draggable.element ) ) ) {
			if ( this.options.hoverClass ) {
				this.element.addClass( this.options.hoverClass );
			}
			this._trigger( "over", event, this.ui( draggable ) );
		}

	},

	_out: function( event ) {

		var draggable = $.ui.ddmanager.current;

		// Bail if draggable and droppable are same element
		if ( !draggable || ( draggable.currentItem || draggable.element )[ 0 ] === this.element[ 0 ] ) {
			return;
		}

		if ( this.accept.call( this.element[ 0 ], ( draggable.currentItem || draggable.element ) ) ) {
			if ( this.options.hoverClass ) {
				this.element.removeClass( this.options.hoverClass );
			}
			this._trigger( "out", event, this.ui( draggable ) );
		}

	},

	_drop: function( event, custom ) {

		var draggable = custom || $.ui.ddmanager.current,
			childrenIntersection = false;

		// Bail if draggable and droppable are same element
		if ( !draggable || ( draggable.currentItem || draggable.element )[ 0 ] === this.element[ 0 ] ) {
			return false;
		}

		this.element.find( ":data(ui-droppable)" ).not( ".ui-draggable-dragging" ).each(function() {
			var inst = $( this ).droppable( "instance" );
			if (
				inst.options.greedy &&
				!inst.options.disabled &&
				inst.options.scope === draggable.options.scope &&
				inst.accept.call( inst.element[ 0 ], ( draggable.currentItem || draggable.element ) ) &&
				$.ui.intersect( draggable, $.extend( inst, { offset: inst.element.offset() } ), inst.options.tolerance, event )
			) { childrenIntersection = true; return false; }
		});
		if ( childrenIntersection ) {
			return false;
		}

		if ( this.accept.call( this.element[ 0 ], ( draggable.currentItem || draggable.element ) ) ) {
			if ( this.options.activeClass ) {
				this.element.removeClass( this.options.activeClass );
			}
			if ( this.options.hoverClass ) {
				this.element.removeClass( this.options.hoverClass );
			}
			this._trigger( "drop", event, this.ui( draggable ) );
			return this.element;
		}

		return false;

	},

	ui: function( c ) {
		return {
			draggable: ( c.currentItem || c.element ),
			helper: c.helper,
			position: c.position,
			offset: c.positionAbs
		};
	}

});

$.ui.intersect = (function() {
	function isOverAxis( x, reference, size ) {
		return ( x >= reference ) && ( x < ( reference + size ) );
	}

	return function( draggable, droppable, toleranceMode, event ) {

		if ( !droppable.offset ) {
			return false;
		}

		var x1 = ( draggable.positionAbs || draggable.position.absolute ).left + draggable.margins.left,
			y1 = ( draggable.positionAbs || draggable.position.absolute ).top + draggable.margins.top,
			x2 = x1 + draggable.helperProportions.width,
			y2 = y1 + draggable.helperProportions.height,
			l = droppable.offset.left,
			t = droppable.offset.top,
			r = l + droppable.proportions().width,
			b = t + droppable.proportions().height;

		switch ( toleranceMode ) {
		case "fit":
			return ( l <= x1 && x2 <= r && t <= y1 && y2 <= b );
		case "intersect":
			return ( l < x1 + ( draggable.helperProportions.width / 2 ) && // Right Half
				x2 - ( draggable.helperProportions.width / 2 ) < r && // Left Half
				t < y1 + ( draggable.helperProportions.height / 2 ) && // Bottom Half
				y2 - ( draggable.helperProportions.height / 2 ) < b ); // Top Half
		case "pointer":
			return isOverAxis( event.pageY, t, droppable.proportions().height ) && isOverAxis( event.pageX, l, droppable.proportions().width );
		case "touch":
			return (
				( y1 >= t && y1 <= b ) || // Top edge touching
				( y2 >= t && y2 <= b ) || // Bottom edge touching
				( y1 < t && y2 > b ) // Surrounded vertically
			) && (
				( x1 >= l && x1 <= r ) || // Left edge touching
				( x2 >= l && x2 <= r ) || // Right edge touching
				( x1 < l && x2 > r ) // Surrounded horizontally
			);
		default:
			return false;
		}
	};
})();

/*
	This manager tracks offsets of draggables and droppables
*/
$.ui.ddmanager = {
	current: null,
	droppables: { "default": [] },
	prepareOffsets: function( t, event ) {

		var i, j,
			m = $.ui.ddmanager.droppables[ t.options.scope ] || [],
			type = event ? event.type : null, // workaround for #2317
			list = ( t.currentItem || t.element ).find( ":data(ui-droppable)" ).addBack();

		droppablesLoop: for ( i = 0; i < m.length; i++ ) {

			// No disabled and non-accepted
			if ( m[ i ].options.disabled || ( t && !m[ i ].accept.call( m[ i ].element[ 0 ], ( t.currentItem || t.element ) ) ) ) {
				continue;
			}

			// Filter out elements in the current dragged item
			for ( j = 0; j < list.length; j++ ) {
				if ( list[ j ] === m[ i ].element[ 0 ] ) {
					m[ i ].proportions().height = 0;
					continue droppablesLoop;
				}
			}

			m[ i ].visible = m[ i ].element.css( "display" ) !== "none";
			if ( !m[ i ].visible ) {
				continue;
			}

			// Activate the droppable if used directly from draggables
			if ( type === "mousedown" ) {
				m[ i ]._activate.call( m[ i ], event );
			}

			m[ i ].offset = m[ i ].element.offset();
			m[ i ].proportions({ width: m[ i ].element[ 0 ].offsetWidth, height: m[ i ].element[ 0 ].offsetHeight });

		}

	},
	drop: function( draggable, event ) {

		var dropped = false;
		// Create a copy of the droppables in case the list changes during the drop (#9116)
		$.each( ( $.ui.ddmanager.droppables[ draggable.options.scope ] || [] ).slice(), function() {

			if ( !this.options ) {
				return;
			}
			if ( !this.options.disabled && this.visible && $.ui.intersect( draggable, this, this.options.tolerance, event ) ) {
				dropped = this._drop.call( this, event ) || dropped;
			}

			if ( !this.options.disabled && this.visible && this.accept.call( this.element[ 0 ], ( draggable.currentItem || draggable.element ) ) ) {
				this.isout = true;
				this.isover = false;
				this._deactivate.call( this, event );
			}

		});
		return dropped;

	},
	dragStart: function( draggable, event ) {
		// Listen for scrolling so that if the dragging causes scrolling the position of the droppables can be recalculated (see #5003)
		draggable.element.parentsUntil( "body" ).bind( "scroll.droppable", function() {
			if ( !draggable.options.refreshPositions ) {
				$.ui.ddmanager.prepareOffsets( draggable, event );
			}
		});
	},
	drag: function( draggable, event ) {

		// If you have a highly dynamic page, you might try this option. It renders positions every time you move the mouse.
		if ( draggable.options.refreshPositions ) {
			$.ui.ddmanager.prepareOffsets( draggable, event );
		}

		// Run through all droppables and check their positions based on specific tolerance options
		$.each( $.ui.ddmanager.droppables[ draggable.options.scope ] || [], function() {

			if ( this.options.disabled || this.greedyChild || !this.visible ) {
				return;
			}

			var parentInstance, scope, parent,
				intersects = $.ui.intersect( draggable, this, this.options.tolerance, event ),
				c = !intersects && this.isover ? "isout" : ( intersects && !this.isover ? "isover" : null );
			if ( !c ) {
				return;
			}

			if ( this.options.greedy ) {
				// find droppable parents with same scope
				scope = this.options.scope;
				parent = this.element.parents( ":data(ui-droppable)" ).filter(function() {
					return $( this ).droppable( "instance" ).options.scope === scope;
				});

				if ( parent.length ) {
					parentInstance = $( parent[ 0 ] ).droppable( "instance" );
					parentInstance.greedyChild = ( c === "isover" );
				}
			}

			// we just moved into a greedy child
			if ( parentInstance && c === "isover" ) {
				parentInstance.isover = false;
				parentInstance.isout = true;
				parentInstance._out.call( parentInstance, event );
			}

			this[ c ] = true;
			this[c === "isout" ? "isover" : "isout"] = false;
			this[c === "isover" ? "_over" : "_out"].call( this, event );

			// we just moved out of a greedy child
			if ( parentInstance && c === "isout" ) {
				parentInstance.isout = false;
				parentInstance.isover = true;
				parentInstance._over.call( parentInstance, event );
			}
		});

	},
	dragStop: function( draggable, event ) {
		draggable.element.parentsUntil( "body" ).unbind( "scroll.droppable" );
		// Call prepareOffsets one final time since IE does not fire return scroll events when overflow was caused by drag (see #5003)
		if ( !draggable.options.refreshPositions ) {
			$.ui.ddmanager.prepareOffsets( draggable, event );
		}
	}
};

return $.ui.droppable;

}));
;

// resource: org.richfaces:popup.js
/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.Popup = function(id, options) {
        $super.constructor.call(this, id);
        this.options = $.extend({}, defaultOptions, options);
        this.positionOptions = {type: this.options.positionType, from:this.options.jointPoint, to:this.options.direction, offset: this.options.positionOffset};

        this.popup = $(document.getElementById(id));

        this.visible = this.options.visible;
        this.attachTo = this.options.attachTo;
        this.attachToBody = this.options.attachToBody;
        this.positionType = this.options.positionType;
        this.positionOffset = this.options.positionOffset;
    };

    rf.BaseComponent.extend(rf.ui.Popup);
    var $super = rf.ui.Popup.$super;

    var defaultOptions = {
        visible: false
    };

    $.extend(rf.ui.Popup.prototype, {

            name : "popup",

            show: function(event) {
                if (!this.visible) {
                    if (this.attachToBody) {
                        this.parentElement = this.popup.parent().get(0);
                        document.body.appendChild(this.popup.get(0));
                    }
                    this.visible = true;
                }

                this.popup.setPosition(event || {id: this.attachTo}, this.positionOptions).show();
            },

            hide: function() {
                if (this.visible) {
                    this.popup.hide();
                    this.visible = false;
                    if (this.attachToBody && this.parentElement) {
                        this.parentElement.appendChild(this.popup.get(0));
                        this.parentElement = null;
                    }
                }
            },

            isVisible: function() {
                return this.visible;
            },

            getId: function() {
                return this.id;
            },

            destroy: function() {
                if (this.attachToBody && this.parentElement) {
                    this.parentElement.appendChild(this.popup.get(0));
                    this.parentElement = null;
                }
            }
        });

})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:calendar-utils.js
(function ($, rf) {

    rf.calendarUtils = rf.calendarUtils || {};

    var getDefaultMonthNames = function(shortNames) {
        return (shortNames
            ? ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
            : ['January','February','March','April','May','June','July','August','September','October','November','December']);
    };

    $.extend(rf.calendarUtils, {
            // TODO: rewrite this function or use the same function if exists
            /*clonePosition: function (elements, source)
             {
             if (!elements.length) elements = [elements];
             var offset = Position.cumulativeOffset(source);
             offset = {left:offset[0], top:offset[1]};
             var offsetTemp;
             if (source.style.position!='absolute')
             {
             offsetTemp = Position.realOffset(source);
             offset.left -= offsetTemp.left;
             offset.top -= offsetTemp.top;
             offsetTemp = Richfaces.Calendar.getWindowScrollOffset();
             offset.left += offsetTemp.left;
             offset.top += offsetTemp.top;
             }

             for (var i=0;i<elements.length;i++)
             {
             offsetTemp = Richfaces.Calendar.getParentOffset(elements[i]);
             elements[i].style.left = (offset.left - offsetTemp.left) + 'px';
             elements[i].style.top = (offset.top - offsetTemp.top) + 'px';
             }
             return offset;
             }*/

            //TODO: not used
            /*Object.extend(Event, {
             findElementByAttr : function(event, tagName, attribute, value, flag) {
             var element = Event.findElement(event, tagName);
             while (!element[attribute] || (flag ? element[attribute].indexOf(value)!=0 : element[attribute]!=value) )
             {
             element = element.parentNode;
             }
             return element;
             }
             });

             Object.extend(Element, {
             replaceClassName : function (element, whichClassName, toClassName) {
             if (!(element = $(element))) return;
             var e = Element.classNames(element);
             e.remove(whichClassName);
             e.add(toClassName);
             return element;
             }
             });*/

            // TODO: move joinArray to richfaces utils
            joinArray: function(array, begin, end, separator) {
                var value = '';
                if (array.length != 0) value = begin + array.pop() + end;
                while (array.length)
                    value = begin + array.pop() + end + separator + value;
                return value;
            },

            getMonthByLabel: function (monthLabel, monthNames) {
                var toLowerMonthLabel = monthLabel.toLowerCase();
                var i = 0;
                while (i < monthNames.length) {
                    if (monthNames[i].toLowerCase() == toLowerMonthLabel) {
                        return i;
                    }

                    i++;
                }
            },

            createDate: function (yy, mm, dd, h, m, s) {
                h = h || 0;
                m = m || 0;
                s = s || 0;
                var date = new Date(yy, mm, dd, h, m, s);
                if (date.getDate() != dd) {
                    date = new Date(yy, mm);
                    date.setHours(h);
                    date.setMinutes(m);
                    date.setSeconds(s);
                    date.setUTCDate(dd);
                }
                return date;
            },

            /* Year:
             *	y,yy - 00-99
             *	yyy+ - 1999
             * Month:
             *	M - 1-12
             *	MM - 01-12
             *	MMM - short (Jul)
             *	MMMM+ - long (July)
             * Date:
             *	d - 1-31
             *	dd+ - 01-31 */
            parseDate: function(dateString, pattern, monthNames, monthNamesShort) {
                var re = /([.*+?^<>=!:${}()\[\]\/\\])/g;
                var monthNamesStr
                var monthNamesShortStr;
                if (!monthNames) {
                    monthNames = getDefaultMonthNames();
                    monthNamesStr = monthNames.join('|');
                } else {
                    monthNamesStr = monthNames.join('|').replace(re, '\\$1');
                }

                if (!monthNamesShort) {
                    monthNamesShort = getDefaultMonthNames(true);
                    monthNamesShortStr = monthNamesShort.join('|');
                } else {
                    monthNamesShortStr = monthNamesShort.join('|').replace(re, '\\$1');
                }

                var counter = 1;
                var y,m,d;
                var a,h,min,s;
                var shortLabel = false;

                pattern = pattern.replace(/([.*+?^<>=!:${}()|\[\]\/\\])/g, '\\$1');
                pattern = pattern.replace(/(y+|M+|d+|a|H{1,2}|h{1,2}|m{2}|s{2})/g,
                    function($1) {
                        switch ($1) {
                            case 'y'  :
                            case 'yy' :
                                y = counter;
                                counter++;
                                return '(\\d{2})';
                            case 'MM' :
                                m = counter;
                                counter++;
                                return '(\\d{2})';
                            case 'M'  :
                                m = counter;
                                counter++;
                                return '(\\d{1,2})';
                            case 'd'  :
                                d = counter;
                                counter++;
                                return '(\\d{1,2})';
                            case 'MMM':
                                m = counter;
                                counter++;
                                shortLabel = true;
                                return '(' + monthNamesShortStr + ')';
                            case 'a'  :
                                a = counter;
                                counter++;
                                return '(AM|am|PM|pm)?';
                            case 'HH' :
                            case 'hh' :
                                h = counter;
                                counter++;
                                return '(\\d{2})?';
                            case 'H'  :
                            case 'h'  :
                                h = counter;
                                counter++;
                                return '(\\d{1,2})?';
                            case 'mm' :
                                min = counter;
                                counter++;
                                return '(\\d{2})?';
                            case 'ss' :
                                s = counter;
                                counter++;
                                return '(\\d{2})?';

                        }
                        // y+,M+,d+
                        var ch = $1.charAt(0);
                        if (ch == 'y') {
                            y = counter;
                            counter++;
                            return '(\\d{3,4})'
                        }
                        ;
                        if (ch == 'M') {
                            m = counter;
                            counter++;
                            return '(' + monthNamesStr + ')'
                        }
                        ;
                        if (ch == 'd') {
                            d = counter;
                            counter++;
                            return '(\\d{2})'
                        }
                        ;
                    }
                );

                var re = new RegExp(pattern, 'i');
                var match = dateString.match(re);
                if (match != null && y != undefined && m != undefined) {
                    // set default century start
                    var correctYear = false;
                    var defaultCenturyStart = new Date();
                    defaultCenturyStart.setFullYear(defaultCenturyStart.getFullYear() - 80);

                    var yy = parseInt(match[y], 10);
                    if (isNaN(yy)) return null;
                    else if (yy < 100) {
                        // calculate full year if year has only two digits
                        var defaultCenturyStartYear = defaultCenturyStart.getFullYear();
                        var ambiguousTwoDigitYear = defaultCenturyStartYear % 100;
                        correctYear = yy == ambiguousTwoDigitYear;
                        yy += Math.floor(defaultCenturyStartYear / 100) * 100 + (yy < ambiguousTwoDigitYear ? 100 : 0);
                    }

                    var mm = parseInt(match[m], 10);
                    if (isNaN(mm)) mm = this.getMonthByLabel(match[m], shortLabel ? monthNamesShort : monthNames); else if (--mm < 0 || mm > 11) return null;
                    var addDay = correctYear ? 1 : 0;
                    var dd;
                    if (d != undefined) {
                        dd = parseInt(match[d], 10);
                    } else {
                        dd = 1; // default to the first of the month when the date is not selected
                    }
                    if (isNaN(dd) || dd < 1 || dd > this.daysInMonth(yy, mm) + addDay) return null;

                    var date;

                    // time parsing
                    if (min != undefined && h != undefined) {
                        var hh,mmin,aa;
                        mmin = parseInt(match[min], 10);
                        if (isNaN(mmin) || mmin < 0 || mmin > 59) return null;
                        hh = parseInt(match[h], 10);
                        if (isNaN(hh)) return null;
                        if (a != undefined) {
                            aa = match[a];
                            if (!aa) return null;
                            aa = aa.toLowerCase();
                            if ((aa != 'am' && aa != 'pm') || hh < 1 || hh > 12) return null;
                            if (aa == 'pm') {
                                if (hh != 12) hh += 12;
                            } else if (hh == 12) hh = 0;
                        }
                        else if (hh < 0 || hh > 23) return null;

                        date = this.createDate(yy, mm, dd, hh, mmin);
                        if (s != undefined) {
                            sec = parseInt(match[s], 10);
                            if (isNaN(sec) || sec < 0 || sec > 59) return null;
                            date.setSeconds(sec);
                        }
                    } else {
                        date = this.createDate(yy, mm, dd);
                    }

                    if (correctYear) {
                        if (date.getTime() < defaultCenturyStart.getTime()) {
                            date.setFullYear(yy + 100);
                        }
                        if (date.getMonth() != mm) return null;
                    }

                    return date;
                }
                return null;
            },

            formatDate: function(date, pattern, monthNames, monthNamesShort) {
                if (!monthNames) monthNames = getDefaultMonthNames();
                if (!monthNamesShort) monthNamesShort = getDefaultMonthNames(true);
                var mm,dd,hh,min,sec;
                var result = pattern.replace(/(\\\\|\\[yMdaHhms])|(y+|M+|d+|a|H{1,2}|h{1,2}|m{2}|s{2})/g,
                    function($1, $2, $3) {
                        if ($2) return $2.charAt(1);
                        switch ($3) {
                            case 'y':
                            case 'yy':
                                return date.getYear().toString().slice(-2);
                            case 'M':
                                return (date.getMonth() + 1);
                            case 'MM':
                                return ((mm = date.getMonth() + 1) < 10 ? '0' + mm : mm);
                            case 'MMM':
                                return monthNamesShort[date.getMonth()];
                            case 'd':
                                return date.getDate();
                            case 'a'  :
                                return (date.getHours() < 12 ? 'AM' : 'PM');
                            case 'HH' :
                                return ((hh = date.getHours()) < 10 ? '0' + hh : hh);
                            case 'H'  :
                                return date.getHours();
                            case 'hh' :
                                return ((hh = date.getHours()) == 0 ? '12' : (hh < 10 ? '0' + hh : (hh > 21 ? hh - 12 : (hh > 12) ? '0' + (hh - 12) : hh)));
                            case 'h'  :
                                return ((hh = date.getHours()) == 0 ? '12' : (hh > 12 ? hh - 12 : hh));
                            case 'mm' :
                                return ((min = date.getMinutes()) < 10 ? '0' + min : min);
                            case 'ss' :
                                return ((sec = date.getSeconds()) < 10 ? '0' + sec : sec);
                        }
                        // y+,M+,d+
                        var ch = $3.charAt(0);
                        if (ch == 'y') return date.getFullYear();
                        if (ch == 'M') return monthNames[date.getMonth()];
                        if (ch == 'd') return ((dd = date.getDate()) < 10 ? '0' + dd : dd);
                    }
                );
                return result;
            },

            isLeapYear: function(year) {
                return new Date(year, 1, 29).getDate() == 29;
            },

            daysInMonth: function(year, month) {
                return 32 - new Date(year, month, 32).getDate();
            },

            daysInMonthByDate: function(date) {
                return 32 - new Date(date.getFullYear(), date.getMonth(), 32).getDate();
            },

            getDay: function(date, firstWeekDay) {
                var value = date.getDay() - firstWeekDay;
                if (value < 0) value = 7 + value;
                return value;
            },

            getFirstWeek: function(year, mdifw, fdow) {
                var date = new Date(year, 0, 1);
                var firstday = this.getDay(date, fdow);

                var weeknumber = (7 - firstday < mdifw) ? 0 : 1;

                return {date:date, firstDay:firstday, weekNumber:weeknumber, mdifw:mdifw, fdow:fdow};
            },

            getLastWeekOfPrevYear: function(o) {
                var year = o.date.getFullYear() - 1;
                var days = (this.isLeapYear(year) ? 366 : 365);
                var obj = this.getFirstWeek(year, o.mdifw, o.fdow);
                days = (days - 7 + o.firstDay);
                var weeks = Math.ceil(days / 7);

                return  weeks + obj.weekNumber;
            },

            weekNumber: function(year, month, mdifw, fdow) {

                var o = this.getFirstWeek(year, mdifw, fdow);

                if (month == 0) {
                    if (o.weekNumber == 1) return 1;
                    return this.getLastWeekOfPrevYear(o);
                }
                var oneweek = 604800000;
                var d = new Date(year, month, 1);
                d.setDate(1 + o.firstDay + (this.getDay(d, fdow) == 0 ? 1 : 0));

                weeknumber = o.weekNumber + Math.floor((d.getTime() - o.date.getTime()) / oneweek);

                return weeknumber;
            }

        });

    rf.calendarTemplates = rf.calendarTemplates || {};

    $.extend(rf.calendarTemplates, (function () {

        var VARIABLE_NAME_PATTERN = /^\s*[_,A-Z,a-z][\w,_\.]*\s*$/;

        var getObjectValue = function (str, object) {
            var a = str.split(".");
            var value = object[a[0]];
            var c = 1;
            while (value && c < a.length) value = value[a[c++]];
            return (value ? value : "");
        };

        return  {
            evalMacro: function(template, object) {
                var _value_ = "";
                // variable evaluation
                if (VARIABLE_NAME_PATTERN.test(template)) {
                    if (template.indexOf('.') == -1) {
                        _value_ = object[template];
                        if (!_value_)    _value_ = window[template];
                    }
                    // object's variable evaluation
                    else {
                        _value_ = getObjectValue(template, object);
                        if (!_value_) _value_ = getObjectValue(template, window);
                    }
                    if (_value_ && typeof _value_ == 'function') _value_ = _value_(object);
                    if (!_value_) _value_ = "";
                }
                //js string evaluation
                else {
                    try {
                        if (object.eval) {
                            _value_ = object.eval(template);
                        }
                        else with (object) {
                            _value_ = eval(template);
                        }

                        if (typeof _value_ == 'function') {
                            _value_ = _value_(object);
                        }
                    } catch (e) {
                        LOG.warn("Exception: " + e.Message + "\n[" + template + "]");
                    }
                }
                return _value_;
            }
        };
    })());

})(RichFaces.jQuery, RichFaces);;

// resource: org.richfaces:dnd-droppable.js
/*
 * code review by Pavel Yaschenko
 * 
 * No event's unbindings when component would be destroyed Hint: easy way to
 * unbind - use namespaces when bind event handlers
 * 
 */

(function($, rf) {

    rf.ui = rf.ui || {};

    var defaultIndicatorClasses = {
        rejectClass : "rf-ind-rejt",
        acceptClass : "rf-ind-acpt",
        draggingClass : "rf-ind-drag"
    };

    var defaultOptions = {
    };

    rf.ui.Droppable = function(id, options) {
        this.options = {};
        $.extend(this.options, defaultOptions, options || {});
        $super.constructor.call(this, id);

        this.namespace = this.namespace || "."
            + rf.Event.createNamespace(this.name, this.id);

        this.id = id;

        this.parentId = this.options.parentId;

        this.attachToDom(this.parentId);

        this.dropElement = $(document.getElementById(this.parentId));
        this.dropElement.droppable({
                addClasses : false
            });
        this.dropElement.data("init", true);

        rf.Event.bind(this.dropElement, 'drop' + this.namespace, this.drop, this);
        rf.Event.bind(this.dropElement, 'dropover' + this.namespace, this.dropover, this);
        rf.Event.bind(this.dropElement, 'dropout' + this.namespace, this.dropout, this);

    };

    rf.BaseNonVisualComponent.extend(rf.ui.Droppable);

    var $super = rf.ui.Droppable.$super;

    $.extend(rf.ui.Droppable.prototype, (function() {
        return {
            drop : function(e) {
                var ui = e.rf.data;
                if (this.accept(ui.draggable)) {
                    this.__callAjax(e, ui);
                }
                var dragIndicatorObj = this.__getIndicatorObject(ui.helper);
                if (dragIndicatorObj) {
                    ui.helper.removeClass(dragIndicatorObj.getAcceptClass());
                    ui.helper.removeClass(dragIndicatorObj.getRejectClass());
                } else {
                    ui.helper.removeClass(defaultIndicatorClasses.acceptClass);
                    ui.helper.removeClass(defaultIndicatorClasses.rejectClass);
                }
            },

            dropover : function(e) {
                var ui = e.rf.data;
                var draggable = ui.draggable;
                var dragIndicatorObj = this.__getIndicatorObject(ui.helper);
                this.dropElement.addClass("rf-drp-hvr");
                if (dragIndicatorObj) {
                    if (this.accept(draggable)) {
                        ui.helper.removeClass(dragIndicatorObj.getRejectClass());
                        ui.helper.addClass(dragIndicatorObj.getAcceptClass());
                        this.dropElement.addClass("rf-drp-hlight");
                    } else {
                        ui.helper.removeClass(dragIndicatorObj.getAcceptClass());
                        ui.helper.addClass(dragIndicatorObj.getRejectClass());
                        this.dropElement.removeClass("rf-drp-hlight");
                    }
                } else {
                    if (this.accept(draggable)) {
                        ui.helper.removeClass(defaultIndicatorClasses.rejectClass);
                        ui.helper.addClass(defaultIndicatorClasses.acceptClass);
                        this.dropElement.addClass("rf-drp-hlight");
                    } else {
                        ui.helper.removeClass(defaultIndicatorClasses.acceptClass);
                        ui.helper.addClass(defaultIndicatorClasses.rejectClass);
                        this.dropElement.removeClass("rf-drp-hlight");
                    }
                }
            },

            dropout : function(e) {
                var ui = e.rf.data;
                var draggable = ui.draggable;
                var dragIndicatorObj = this.__getIndicatorObject(ui.helper);
                this.dropElement.removeClass("rf-drp-hvr rf-drp-hlight");
                if (dragIndicatorObj) {
                    ui.helper.removeClass(dragIndicatorObj.getAcceptClass());
                    ui.helper.removeClass(dragIndicatorObj.getRejectClass());
                } else {
                    ui.helper.removeClass(defaultIndicatorClasses.acceptClass);
                    ui.helper.removeClass(defaultIndicatorClasses.rejectClass);
                }
            },

            accept : function(draggable) {
                var accept = false;
                var acceptType = draggable.data("type");
                if (acceptType && this.options.acceptedTypes) {
                    $.each(this.options.acceptedTypes, function() {
                        if (this == "@none") {
                            return false;
                        }

                        if (this == acceptType || this == "@all") {
                            accept = true;
                            return false;
                        }
                    });
                }
                return accept;
            },
            
            __getIndicatorObject: function(helper) {
                var indicatorCloneId = helper.attr('id');
                if (indicatorCloneId) {
                    var indicatorId = indicatorCloneId.match(/(.*)Clone$/)[1];
                    return rf.component(indicatorId);
                }
            },

            __callAjax : function(e, ui) {
                if (ui.draggable) {
                    var dragSource = ui.draggable.data("id");
                    var ajaxFunc = this.options.ajaxFunction;
                    if (ajaxFunc && typeof ajaxFunc == 'function') {
                        ajaxFunc.call(this, e, dragSource);
                    }
                }
            },

            destroy : function() {
                // clean up code here
                this.detach(this.parentId);
                rf.Event.unbind(this.dropElement, this.namespace);

                // call parent's destroy method
                $super.destroy.call(this);

            }


        }
    })());

})(RichFaces.jQuery, window.RichFaces);;

// resource: org.richfaces:list.js
(function ($, rf) {

    rf.ui = rf.ui || {};

    rf.ui.List = function(id, options) {
        $super.constructor.call(this, id);
        this.namespace = this.namespace || "." + rf.Event.createNamespace(this.name, this.id);
        this.attachToDom();
        var mergedOptions = $.extend({}, defaultOptions, options);
        this.list = $(document.getElementById(id));
        this.selectListener = mergedOptions.selectListener;
        this.selectItemCss = mergedOptions.selectItemCss;
        this.selectItemCssMarker = mergedOptions.selectItemCss.split(" ", 1)[0];
        this.scrollContainer = $(mergedOptions.scrollContainer);
        this.parentContainer = mergedOptions.parentContainer ? $(mergedOptions.parentContainer) : this.scrollContainer;
        this.itemCss = mergedOptions.itemCss.split(" ", 1)[0]; // we only need one of the item css classes to identify the item
        this.listCss = mergedOptions.listCss;
        this.clickRequiredToSelect = mergedOptions.clickRequiredToSelect;
        this.index = -1;
        this.disabled = mergedOptions.disabled;

        this.focusKeeper = $(document.getElementById(id + "FocusKeeper"));
        this.focusKeeper.focused = false;

        this.isMouseDown = false;
        this.list
            .bind("mousedown", $.proxy(this.__onMouseDown, this))
            .bind("mouseup", $.proxy(this.__onMouseUp, this));


        bindEventHandlers.call(this);
        if (mergedOptions.focusKeeperEnabled) {
            bindFocusEventHandlers.call(this);
        }

        this.__updateItemsList(); // initialize this.items
        if (mergedOptions.clientSelectItems !== null) {
            this.__storeClientSelectItems(mergedOptions.clientSelectItems);
        }
    };

    rf.BaseComponent.extend(rf.ui.List);
    var $super = rf.ui.List.$super;

    var defaultOptions = {
        clickRequiredToSelect: false,
        disabled : false,
        selectListener : false,
        clientSelectItems : null,
        focusKeeperEnabled : true
    };

    var bindEventHandlers = function () {
        var handlers = {};
        handlers["click" + this.namespace] = $.proxy(this.onClick, this);
        handlers["dblclick" + this.namespace] = $.proxy(this.onDblclick, this);
        this.list.on("mouseover" + this.namespace, "."+this.itemCss, $.proxy(onMouseOver, this));
        rf.Event.bind(this.list, handlers, this);
    };

    var bindFocusEventHandlers = function () {
        var focusEventHandlers = {};
        focusEventHandlers["keydown" + this.namespace] = $.proxy(this.__keydownHandler, this);
        focusEventHandlers["blur" + this.namespace] = $.proxy(this.__blurHandler, this);
        focusEventHandlers["focus" + this.namespace] = $.proxy(this.__focusHandler, this);
        rf.Event.bind(this.focusKeeper, focusEventHandlers, this);
    };

    var onMouseOver = function(e) {
        var item = $(e.target);
        if (item && !this.clickRequiredToSelect && !this.disabled) {
            this.__select(item);
        }
    };

    $.extend(rf.ui.List.prototype, ( function () {

        return{

            name : "list",

            processItem: function(item) {
                if (this.selectListener.processItem && typeof this.selectListener.processItem == 'function') {
                    this.selectListener.processItem(item);
                }
            },

            isSelected: function(item) {
                return item.hasClass(this.selectItemCssMarker);
            },

            selectItem: function(item) {
                if (this.selectListener.selectItem && typeof this.selectListener.selectItem == 'function') {
                    this.selectListener.selectItem(item);
                } else {
                    item.addClass(this.selectItemCss);
                    rf.Event.fire(this, "selectItem", item);
                }
                this.__scrollToSelectedItem(this);
            },

            unselectItem: function(item) {
                if (this.selectListener.unselectItem && typeof this.selectListener.unselectItem == 'function') {
                    this.selectListener.unselectItem(item);
                } else {
                    item.removeClass(this.selectItemCss);
                    rf.Event.fire(this, "unselectItem", item);
                }
            },

            __focusHandler : function(e) {
                if (! this.focusKeeper.focused) {
                    this.focusKeeper.focused = true;
                    rf.Event.fire(this, "listfocus" + this.namespace, e);
                }
            },

            __blurHandler: function(e) {
                if (!this.isMouseDown) {
                    var that = this;
                    this.timeoutId = window.setTimeout(function() {
                        that.focusKeeper.focused = false;
                        that.invokeEvent.call(that, "blur", document.getElementById(that.id), e);
                        rf.Event.fire(that, "listblur" + that.namespace, e);
                    }, 200);
                } else {
                    this.isMouseDown = false;
                }
            },

            __onMouseDown: function(e) {
                this.isMouseDown = true;
            },

            __onMouseUp: function(e) {
                this.isMouseDown = false;
            },

            __keydownHandler: function(e) {
                if (e.isDefaultPrevented()) return;
                if (e.metaKey || e.ctrlKey) return;

                var code;
                if (e.keyCode) {
                    code = e.keyCode;
                } else if (e.which) {
                    code = e.which;
                }

                switch (code) {
                    case rf.KEYS.DOWN:
                        e.preventDefault();
                        this.__selectNext();
                        break;

                    case rf.KEYS.UP:
                        e.preventDefault();
                        this.__selectPrev();
                        break;

                    case rf.KEYS.HOME:
                        e.preventDefault();
                        this.__selectByIndex(0);
                        break;

                    case rf.KEYS.END:
                        e.preventDefault();
                        this.__selectByIndex(this.items.length - 1);
                        break;

                    default:
                        break;
                }
            },

            onClick: function(e) {
                this.setFocus();
                var item = this.__getItem(e);
                if (!item) return;
                this.processItem(item);
                var clickModified = e.metaKey || e.ctrlKey;
                if (!this.disabled) {
                    this.__select(item, clickModified && this.clickRequiredToSelect);
                }
            },

            onDblclick: function(e) {
                this.setFocus();
                var item = this.__getItem(e);
                if (!item) return;
                this.processItem(item);
                if (!this.disabled) {
                    this.__select(item, false);
                }
            },

            currentSelectItem: function() {
                if (this.items && this.index != -1) {
                    return $(t