/*
 * Copyright 2013 Red Hat Inc. and/or its affiliates and other contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.switchyard.component.common.knowledge.util;

import org.kie.api.KieServices;
import org.kie.api.io.KieResources;
import org.kie.internal.builder.KnowledgeBuilder;
import org.switchyard.common.io.resource.Resource;
import org.switchyard.common.io.resource.ResourceType;
import org.switchyard.component.common.knowledge.config.model.KnowledgeComponentImplementationModel;
import org.switchyard.component.common.knowledge.config.model.ManifestModel;
import org.switchyard.config.model.resource.ResourcesModel;

/**
 * Resource functions.
 *
 * @author David Ward &lt;<a href="mailto:dward@jboss.org">dward@jboss.org</a>&gt; &copy; 2012 Red Hat Inc.
 */
public final class Resources {

    /**
     * Installs resource types.
     * @param loader the class loader
     */
    public static void installTypes(ClassLoader loader) {
        ResourceType.install(loader);
    }

    /**
     * Adds resources.
     * @param model the model
     * @param loader the class loader
     * @param builder the knowledge builder
     */
    public static void addResources(KnowledgeComponentImplementationModel model, ClassLoader loader, KnowledgeBuilder builder) {
        ManifestModel manifestModel = model.getManifest();
        if (manifestModel != null) {
            ResourcesModel resourcesModel = manifestModel.getResources();
            if (resourcesModel != null) {
                KieResources kieResources = KieServices.Factory.get().getResources();
                for (Resource syResource : resourcesModel.getResources()) {
                    org.kie.api.io.Resource kieResource = kieResources.newUrlResource(syResource.getLocationURL(loader));
                    if (kieResource != null) {
                        org.kie.api.io.ResourceType kieResourceType = convertResourceType(syResource.getType());
                        if (kieResourceType != null) {
                            builder.add(kieResource, kieResourceType);
                        }
                    }
                }
            }
        }
    }

    /**
     * Adds a resource.
     * @param syResource the SwitchYard resource
     * @param loader the class loader
     * @param builder the knowledge builder
     */
    public static void addResource(Resource syResource, ClassLoader loader, KnowledgeBuilder builder) {
        KieResources kieResources = KieServices.Factory.get().getResources();
        org.kie.api.io.Resource kieResource = kieResources.newUrlResource(syResource.getLocationURL(loader));
        if (kieResource != null) {
            org.kie.api.io.ResourceType kieResourceType = convertResourceType(syResource.getType());
            if (kieResourceType != null) {
                builder.add(kieResource, kieResourceType);
            }
        }
    }

    /**
     * Converts a resource type.
     * @param syResourceType the SwitchYard resource type.
     * @return the KIE resource type
     */
    public static org.kie.api.io.ResourceType convertResourceType(ResourceType syResourceType) {
        if (syResourceType != null) {
            String resourceTypeName = syResourceType.getName();
            if ("BPMN".equals(resourceTypeName)) {
                // Drools ResourceType recognizes BPMN2, not BPMN
                resourceTypeName = "BPMN2";
            } else if ("XLS".equals(resourceTypeName)) {
                // Drools ResourceType recognizes DTABLE, not XLS
                resourceTypeName = "DTABLE";
            }
            return org.kie.api.io.ResourceType.getResourceType(resourceTypeName);
        }
        return null;
    }

    private Resources() {}

}
