/*
 * Decompiled with CFR 0.152.
 */
package org.milyn.archive;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.milyn.SmooksException;
import org.milyn.assertion.AssertArgument;
import org.milyn.io.FileUtils;
import org.milyn.io.StreamUtils;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class Archive {
    private static Log logger = LogFactory.getLog(Archive.class);
    private String archiveName;
    private File tmpDir;
    private LinkedHashMap<String, File> entries = new LinkedHashMap();

    public Archive() {
        this.archiveName = "Unknown";
        this.createTempDir();
    }

    public Archive(String archiveName) {
        AssertArgument.isNotNull(archiveName, "archiveName");
        this.archiveName = archiveName;
        this.createTempDir();
    }

    public Archive(ZipInputStream archiveStream) throws IOException {
        this.archiveName = "Unknown";
        this.createTempDir();
        this.addEntries(archiveStream);
    }

    public Archive(String archiveName, ZipInputStream archiveStream) throws IOException {
        AssertArgument.isNotNullAndNotEmpty(archiveName, "archiveName");
        this.archiveName = archiveName;
        this.addEntries(archiveStream);
        this.createTempDir();
    }

    public String getArchiveName() {
        return this.archiveName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Archive addEntry(String path, InputStream data) throws IOException {
        AssertArgument.isNotNullAndNotEmpty(path, "path");
        AssertArgument.isNotNull(data, "data");
        try {
            byte[] dataBytes = StreamUtils.readStream(data);
            this.addEntry(this.trimLeadingSlash(path.trim()), dataBytes);
        }
        finally {
            try {
                data.close();
            }
            catch (IOException e) {
                logger.warn((Object)("Unable to close input stream for archive entry '" + path + "'."));
            }
        }
        return this;
    }

    public Archive addEntry(Class<?> clazz) throws IOException {
        AssertArgument.isNotNull(clazz, "clazz");
        String className = clazz.getName();
        className = className.replace('.', '/') + ".class";
        this.addClasspathResourceEntry(className, "/" + className);
        return this;
    }

    public Archive addEntry(String path, byte[] data) {
        AssertArgument.isNotNullAndNotEmpty(path, "path");
        File entryFile = new File(this.tmpDir, path);
        if (entryFile.exists()) {
            entryFile.delete();
        }
        entryFile.getParentFile().mkdirs();
        if (data == null) {
            entryFile.mkdir();
        } else {
            try {
                FileUtils.writeFile(data, entryFile);
            }
            catch (IOException e) {
                throw new IllegalStateException("Unexpected error writing Archive file '" + entryFile.getAbsolutePath() + "'.", e);
            }
        }
        this.entries.put(this.trimLeadingSlash(path.trim()), entryFile);
        return this;
    }

    public Archive addEntry(String path) {
        AssertArgument.isNotNullAndNotEmpty(path, "path");
        path = path.trim();
        if (path.endsWith("/")) {
            this.entries.put(this.trimLeadingSlash(path), null);
        } else {
            this.entries.put(this.trimLeadingSlash(path) + "/", null);
        }
        return this;
    }

    public Archive addClasspathResourceEntry(String path, String resource) throws IOException {
        AssertArgument.isNotNull(path, "path");
        AssertArgument.isNotNull(resource, "resource");
        InputStream resourceStream = this.getClass().getResourceAsStream(resource);
        if (resourceStream == null) {
            throw new IOException("Classpath resource '" + resource + "' no found.");
        }
        this.addEntry(path, resourceStream);
        return this;
    }

    public Archive addEntry(String path, String data) {
        AssertArgument.isNotNullAndNotEmpty(path, "path");
        AssertArgument.isNotNull(data, "data");
        try {
            this.addEntry(this.trimLeadingSlash(path.trim()), data.getBytes("UTF-8"));
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unexpected UnsupportedEncodingException exception for encoding 'UTF-8' when writing Archive entry from a StringBuilder instance.", e);
        }
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Archive addEntries(ZipInputStream zipStream) throws IOException {
        AssertArgument.isNotNull(zipStream, "zipStream");
        try {
            ZipEntry zipEntry = zipStream.getNextEntry();
            ByteArrayOutputStream outByteStream = new ByteArrayOutputStream();
            byte[] byteReadBuffer = new byte[512];
            while (zipEntry != null) {
                if (zipEntry.isDirectory()) {
                    this.addEntry(zipEntry.getName(), (byte[])null);
                } else {
                    int byteReadCount;
                    while ((byteReadCount = zipStream.read(byteReadBuffer)) != -1) {
                        outByteStream.write(byteReadBuffer, 0, byteReadCount);
                    }
                    this.addEntry(zipEntry.getName(), outByteStream.toByteArray());
                    outByteStream.reset();
                }
                zipEntry = zipStream.getNextEntry();
            }
        }
        finally {
            try {
                zipStream.close();
            }
            catch (IOException e) {
                logger.debug((Object)"Unexpected error closing EDI Mapping Model Zip stream.", (Throwable)e);
            }
        }
        return this;
    }

    public Archive removeEntry(String path) {
        this.entries.remove(path);
        return this;
    }

    public Map<String, File> getEntries() {
        return Collections.unmodifiableMap(this.entries);
    }

    public String getEntryName(int index) {
        Set<Map.Entry<String, File>> entrySet = this.entries.entrySet();
        int i = 0;
        for (Map.Entry<String, File> entry : entrySet) {
            if (i == index) {
                return entry.getKey();
            }
            ++i;
        }
        throw new ArrayIndexOutOfBoundsException(index);
    }

    public byte[] getEntryValue(int index) {
        Set<Map.Entry<String, File>> entrySet = this.entries.entrySet();
        int i = 0;
        for (Map.Entry<String, File> entry : entrySet) {
            if (i == index) {
                File entryFile = entry.getValue();
                if (entryFile != null) {
                    try {
                        return FileUtils.readFile(entryFile);
                    }
                    catch (IOException e) {
                        throw new IllegalStateException("Unexpected error reading Archive file '" + entryFile.getAbsolutePath() + "'.", e);
                    }
                }
                return null;
            }
            ++i;
        }
        throw new ArrayIndexOutOfBoundsException(index);
    }

    public byte[] getEntryBytes(String resName) {
        File entryFile = this.getEntry(resName);
        if (entryFile != null) {
            try {
                return FileUtils.readFile(entryFile);
            }
            catch (IOException e) {
                throw new IllegalStateException("Unexpected error reading Archive file '" + entryFile.getAbsolutePath() + "'.", e);
            }
        }
        return null;
    }

    public File getEntry(String resName) {
        AssertArgument.isNotNullAndNotEmpty(resName, "resName");
        return this.entries.get(resName);
    }

    public URL getEntryURL(String resName) {
        File entry = this.getEntry(resName);
        if (entry != null) {
            try {
                return entry.toURI().toURL();
            }
            catch (MalformedURLException e) {
                throw new IllegalStateException("Unexpected error getting URL for Archive file '" + entry.getAbsolutePath() + "'.", e);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void toOutputStream(ZipOutputStream archiveStream) throws IOException {
        AssertArgument.isNotNull(archiveStream, "archiveStream");
        try {
            this.writeEntriesToArchive(archiveStream);
        }
        finally {
            try {
                archiveStream.flush();
            }
            finally {
                try {
                    archiveStream.close();
                }
                catch (IOException e) {
                    logger.info((Object)"Unable to close archive output stream.");
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void toFile(File file) throws IOException {
        FileOutputStream fileOutputStream = new FileOutputStream(file);
        try {
            this.toOutputStream(new ZipOutputStream(fileOutputStream));
        }
        finally {
            try {
                fileOutputStream.flush();
            }
            finally {
                fileOutputStream.close();
            }
        }
    }

    public void toFileSystem(File outputFolder) throws IOException {
        AssertArgument.isNotNull(outputFolder, "outputFolder");
        if (outputFolder.isFile()) {
            throw new IOException("Cannot write Archive entries to '" + outputFolder.getAbsolutePath() + "'.  This is a normal file i.e. not a directory.");
        }
        if (!outputFolder.exists()) {
            outputFolder.mkdirs();
        }
        Set<Map.Entry<String, File>> entrySet = this.entries.entrySet();
        for (Map.Entry<String, File> entry : entrySet) {
            File archEntryFile = entry.getValue();
            File entryFile = new File(outputFolder, entry.getKey());
            if (archEntryFile != null) {
                byte[] fileBytes = FileUtils.readFile(archEntryFile);
                entryFile.getParentFile().mkdirs();
                FileUtils.writeFile(fileBytes, entryFile);
                continue;
            }
            entryFile.mkdirs();
        }
    }

    public ZipInputStream toInputStream() throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        this.toOutputStream(new ZipOutputStream(outputStream));
        return new ZipInputStream(new ByteArrayInputStream(outputStream.toByteArray()));
    }

    private void writeEntriesToArchive(ZipOutputStream archiveStream) throws IOException {
        File manifestFile = this.entries.get("META-INF/MANIFEST.MF");
        if (manifestFile != null) {
            byte[] manifest = FileUtils.readFile(manifestFile);
            this.writeEntry("META-INF/MANIFEST.MF", manifest, archiveStream);
        }
        Set<Map.Entry<String, File>> entrySet = this.entries.entrySet();
        for (Map.Entry<String, File> entry : entrySet) {
            if (entry.getKey().equals("META-INF/MANIFEST.MF")) continue;
            File file = entry.getValue();
            if (file != null && !file.isDirectory()) {
                this.writeEntry(entry.getKey(), FileUtils.readFile(file), archiveStream);
                continue;
            }
            this.writeEntry(entry.getKey(), null, archiveStream);
        }
    }

    private void writeEntry(String entryName, byte[] entryValue, ZipOutputStream archiveStream) throws IOException {
        try {
            archiveStream.putNextEntry(new ZipEntry(entryName));
            if (entryValue != null) {
                archiveStream.write(entryValue);
            }
            archiveStream.closeEntry();
        }
        catch (Exception e) {
            throw (IOException)new IOException("Unable to create archive entry '" + entryName + "'.").initCause(e);
        }
    }

    private String trimLeadingSlash(String path) {
        StringBuilder builder = new StringBuilder(path);
        while (builder.length() > 0 && builder.charAt(0) == '/') {
            builder.deleteCharAt(0);
        }
        return builder.toString();
    }

    public Archive merge(Archive archive) {
        Map<String, File> entriesToMerge = archive.getEntries();
        for (Map.Entry<String, File> entry : entriesToMerge.entrySet()) {
            File file = entry.getValue();
            if (file != null && !file.isDirectory()) {
                try {
                    this.addEntry(entry.getKey(), FileUtils.readFile(file));
                    continue;
                }
                catch (IOException e) {
                    throw new IllegalStateException("Unexpected error reading Archive file '" + file.getAbsolutePath() + "'.", e);
                }
            }
            this.addEntry(entry.getKey(), (byte[])null);
        }
        return this;
    }

    public boolean contains(String path) {
        return this.entries.containsKey(path);
    }

    private void createTempDir() {
        if (this.tmpDir == null) {
            try {
                File tmpFile = File.createTempFile("tmp", "tmp");
                tmpFile.delete();
                this.tmpDir = new File(tmpFile.getParentFile(), UUID.randomUUID().toString());
                DeleteOnExitHook.add(this.tmpDir);
            }
            catch (IOException e) {
                throw new SmooksException("Unable to crete temp directory for archive.", e);
            }
        }
    }

    private static class DeleteOnExitHook {
        private static List<String> dirsToDelete;

        private DeleteOnExitHook() {
        }

        private static synchronized void add(File dir) {
            if (dirsToDelete == null) {
                throw new IllegalStateException("Shutdown in progress");
            }
            dirsToDelete.add(dir.getAbsolutePath());
        }

        private static synchronized void deleteDirs() {
            if (dirsToDelete == null) {
                throw new IllegalStateException("Shutdown in progress");
            }
            List<String> dirs = dirsToDelete;
            dirsToDelete = null;
            for (String dir : dirs) {
                try {
                    FileUtils.deleteDir(new File(dir));
                }
                catch (Exception exception) {}
            }
        }

        static {
            Runtime.getRuntime().addShutdownHook(new Thread(){

                public void run() {
                    DeleteOnExitHook.deleteDirs();
                }
            });
            dirsToDelete = new CopyOnWriteArrayList<String>();
        }
    }
}

