/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The frame name, metric type, and thread states. These are used to derive the value of the metric for the frame.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FrameMetric implements SdkPojo, Serializable, ToCopyableBuilder<FrameMetric.Builder, FrameMetric> {
    private static final SdkField<String> FRAME_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("frameName").getter(getter(FrameMetric::frameName)).setter(setter(Builder::frameName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("frameName").build()).build();

    private static final SdkField<List<String>> THREAD_STATES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("threadStates")
            .getter(getter(FrameMetric::threadStates))
            .setter(setter(Builder::threadStates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threadStates").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(FrameMetric::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FRAME_NAME_FIELD,
            THREAD_STATES_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String frameName;

    private final List<String> threadStates;

    private final String type;

    private FrameMetric(BuilderImpl builder) {
        this.frameName = builder.frameName;
        this.threadStates = builder.threadStates;
        this.type = builder.type;
    }

    /**
     * <p>
     * Name of the method common across the multiple occurrences of a frame in an application profile.
     * </p>
     * 
     * @return Name of the method common across the multiple occurrences of a frame in an application profile.
     */
    public final String frameName() {
        return frameName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ThreadStates property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasThreadStates() {
        return threadStates != null && !(threadStates instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of application runtime thread states used to get the counts for a frame a derive a metric value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasThreadStates} method.
     * </p>
     * 
     * @return List of application runtime thread states used to get the counts for a frame a derive a metric value.
     */
    public final List<String> threadStates() {
        return threadStates;
    }

    /**
     * <p>
     * A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
     * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is calculated
     * across the occurrences of all frames in a profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MetricType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
     *         <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
     *         calculated across the occurrences of all frames in a profile.
     * @see MetricType
     */
    public final MetricType type() {
        return MetricType.fromValue(type);
    }

    /**
     * <p>
     * A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
     * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is calculated
     * across the occurrences of all frames in a profile.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link MetricType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
     *         <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
     *         calculated across the occurrences of all frames in a profile.
     * @see MetricType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(frameName());
        hashCode = 31 * hashCode + Objects.hashCode(hasThreadStates() ? threadStates() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FrameMetric)) {
            return false;
        }
        FrameMetric other = (FrameMetric) obj;
        return Objects.equals(frameName(), other.frameName()) && hasThreadStates() == other.hasThreadStates()
                && Objects.equals(threadStates(), other.threadStates()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FrameMetric").add("FrameName", frameName())
                .add("ThreadStates", hasThreadStates() ? threadStates() : null).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "frameName":
            return Optional.ofNullable(clazz.cast(frameName()));
        case "threadStates":
            return Optional.ofNullable(clazz.cast(threadStates()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FrameMetric, T> g) {
        return obj -> g.apply((FrameMetric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FrameMetric> {
        /**
         * <p>
         * Name of the method common across the multiple occurrences of a frame in an application profile.
         * </p>
         * 
         * @param frameName
         *        Name of the method common across the multiple occurrences of a frame in an application profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder frameName(String frameName);

        /**
         * <p>
         * List of application runtime thread states used to get the counts for a frame a derive a metric value.
         * </p>
         * 
         * @param threadStates
         *        List of application runtime thread states used to get the counts for a frame a derive a metric value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadStates(Collection<String> threadStates);

        /**
         * <p>
         * List of application runtime thread states used to get the counts for a frame a derive a metric value.
         * </p>
         * 
         * @param threadStates
         *        List of application runtime thread states used to get the counts for a frame a derive a metric value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadStates(String... threadStates);

        /**
         * <p>
         * A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
         * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         * calculated across the occurrences of all frames in a profile.
         * </p>
         * 
         * @param type
         *        A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
         *        <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         *        calculated across the occurrences of all frames in a profile.
         * @see MetricType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricType
         */
        Builder type(String type);

        /**
         * <p>
         * A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
         * <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         * calculated across the occurrences of all frames in a profile.
         * </p>
         * 
         * @param type
         *        A type of aggregation that specifies how a metric for a frame is analyzed. The supported value
         *        <code>AggregatedRelativeTotalTime</code> is an aggregation of the metric value for one frame that is
         *        calculated across the occurrences of all frames in a profile.
         * @see MetricType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricType
         */
        Builder type(MetricType type);
    }

    static final class BuilderImpl implements Builder {
        private String frameName;

        private List<String> threadStates = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(FrameMetric model) {
            frameName(model.frameName);
            threadStates(model.threadStates);
            type(model.type);
        }

        public final String getFrameName() {
            return frameName;
        }

        public final void setFrameName(String frameName) {
            this.frameName = frameName;
        }

        @Override
        public final Builder frameName(String frameName) {
            this.frameName = frameName;
            return this;
        }

        public final Collection<String> getThreadStates() {
            if (threadStates instanceof SdkAutoConstructList) {
                return null;
            }
            return threadStates;
        }

        public final void setThreadStates(Collection<String> threadStates) {
            this.threadStates = ThreadStatesCopier.copy(threadStates);
        }

        @Override
        public final Builder threadStates(Collection<String> threadStates) {
            this.threadStates = ThreadStatesCopier.copy(threadStates);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder threadStates(String... threadStates) {
            threadStates(Arrays.asList(threadStates));
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(MetricType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public FrameMetric build() {
            return new FrameMetric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
