/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies which resource types Config records for configuration changes. By default, Config records configuration
 * changes for all current and future supported resource types in the Amazon Web Services Region where you have enabled
 * Config, excluding the global IAM resource types: IAM users, groups, roles, and customer managed policies.
 * </p>
 * <p>
 * In the recording group, you specify whether you want to record all supported current and future supported resource
 * types or to include or exclude specific resources types. For a list of supported resource types, see <a
 * href="https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
 * >Supported Resource Types</a> in the <i>Config developer guide</i>.
 * </p>
 * <p>
 * If you don't want Config to record all current and future supported resource types (excluding the global IAM resource
 * types), use one of the following recording strategies:
 * </p>
 * <ol>
 * <li>
 * <p>
 * <b>Record all current and future resource types with exclusions</b> (<code>EXCLUSION_BY_RESOURCE_TYPES</code>), or
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Record specific resource types</b> (<code>INCLUSION_BY_RESOURCE_TYPES</code>).
 * </p>
 * </li>
 * </ol>
 * <p>
 * If you use the recording strategy to <b>Record all current and future resource types</b> (
 * <code>ALL_SUPPORTED_RESOURCE_TYPES</code>), you can use the flag <code>includeGlobalResourceTypes</code> to include
 * the global IAM resource types in your recording.
 * </p>
 * <important>
 * <p>
 * <b>Aurora global clusters are recorded in all enabled Regions</b>
 * </p>
 * <p>
 * The <code>AWS::RDS::GlobalCluster</code> resource type will be recorded in all supported Config Regions where the
 * configuration recorder is enabled.
 * </p>
 * <p>
 * If you do not want to record <code>AWS::RDS::GlobalCluster</code> in all enabled Regions, use the
 * <code>EXCLUSION_BY_RESOURCE_TYPES</code> or <code>INCLUSION_BY_RESOURCE_TYPES</code> recording strategy.
 * </p>
 * </important>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecordingGroup implements SdkPojo, Serializable, ToCopyableBuilder<RecordingGroup.Builder, RecordingGroup> {
    private static final SdkField<Boolean> ALL_SUPPORTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("allSupported").getter(getter(RecordingGroup::allSupported)).setter(setter(Builder::allSupported))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allSupported").build()).build();

    private static final SdkField<Boolean> INCLUDE_GLOBAL_RESOURCE_TYPES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("includeGlobalResourceTypes")
            .getter(getter(RecordingGroup::includeGlobalResourceTypes))
            .setter(setter(Builder::includeGlobalResourceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("includeGlobalResourceTypes").build())
            .build();

    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceTypes")
            .getter(getter(RecordingGroup::resourceTypesAsStrings))
            .setter(setter(Builder::resourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ExclusionByResourceTypes> EXCLUSION_BY_RESOURCE_TYPES_FIELD = SdkField
            .<ExclusionByResourceTypes> builder(MarshallingType.SDK_POJO).memberName("exclusionByResourceTypes")
            .getter(getter(RecordingGroup::exclusionByResourceTypes)).setter(setter(Builder::exclusionByResourceTypes))
            .constructor(ExclusionByResourceTypes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exclusionByResourceTypes").build())
            .build();

    private static final SdkField<RecordingStrategy> RECORDING_STRATEGY_FIELD = SdkField
            .<RecordingStrategy> builder(MarshallingType.SDK_POJO).memberName("recordingStrategy")
            .getter(getter(RecordingGroup::recordingStrategy)).setter(setter(Builder::recordingStrategy))
            .constructor(RecordingStrategy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recordingStrategy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALL_SUPPORTED_FIELD,
            INCLUDE_GLOBAL_RESOURCE_TYPES_FIELD, RESOURCE_TYPES_FIELD, EXCLUSION_BY_RESOURCE_TYPES_FIELD,
            RECORDING_STRATEGY_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean allSupported;

    private final Boolean includeGlobalResourceTypes;

    private final List<String> resourceTypes;

    private final ExclusionByResourceTypes exclusionByResourceTypes;

    private final RecordingStrategy recordingStrategy;

    private RecordingGroup(BuilderImpl builder) {
        this.allSupported = builder.allSupported;
        this.includeGlobalResourceTypes = builder.includeGlobalResourceTypes;
        this.resourceTypes = builder.resourceTypes;
        this.exclusionByResourceTypes = builder.exclusionByResourceTypes;
        this.recordingStrategy = builder.recordingStrategy;
    }

    /**
     * <p>
     * Specifies whether Config records configuration changes for all supported resource types, excluding the global IAM
     * resource types.
     * </p>
     * <p>
     * If you set this field to <code>true</code>, when Config adds support for a new resource type, Config starts
     * recording resources of that type automatically.
     * </p>
     * <p>
     * If you set this field to <code>true</code>, you cannot enumerate specific resource types to record in the
     * <code>resourceTypes</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>, or to
     * exclude in the <code>resourceTypes</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
     * >ExclusionByResourceTypes</a>.
     * </p>
     * <note>
     * <p>
     * <b>Region availability</b>
     * </p>
     * <p>
     * Check <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html">Resource
     * Coverage by Region Availability</a> to see if a resource type is supported in the Amazon Web Services Region
     * where you set up Config.
     * </p>
     * </note>
     * 
     * @return Specifies whether Config records configuration changes for all supported resource types, excluding the
     *         global IAM resource types.</p>
     *         <p>
     *         If you set this field to <code>true</code>, when Config adds support for a new resource type, Config
     *         starts recording resources of that type automatically.
     *         </p>
     *         <p>
     *         If you set this field to <code>true</code>, you cannot enumerate specific resource types to record in the
     *         <code>resourceTypes</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>,
     *         or to exclude in the <code>resourceTypes</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
     *         >ExclusionByResourceTypes</a>.
     *         </p>
     *         <note>
     *         <p>
     *         <b>Region availability</b>
     *         </p>
     *         <p>
     *         Check <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
     *         >Resource Coverage by Region Availability</a> to see if a resource type is supported in the Amazon Web
     *         Services Region where you set up Config.
     *         </p>
     */
    public final Boolean allSupported() {
        return allSupported;
    }

    /**
     * <p>
     * This option is a bundle which only applies to the global IAM resource types: IAM users, groups, roles, and
     * customer managed policies. These global IAM resource types can only be recorded by Config in Regions where Config
     * was available before February 2022. You cannot be record the global IAM resouce types in Regions supported by
     * Config after February 2022. This list where you cannot record the global IAM resource types includes the
     * following Regions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Asia Pacific (Hyderabad)
     * </p>
     * </li>
     * <li>
     * <p>
     * Asia Pacific (Melbourne)
     * </p>
     * </li>
     * <li>
     * <p>
     * Canada West (Calgary)
     * </p>
     * </li>
     * <li>
     * <p>
     * Europe (Spain)
     * </p>
     * </li>
     * <li>
     * <p>
     * Europe (Zurich)
     * </p>
     * </li>
     * <li>
     * <p>
     * Israel (Tel Aviv)
     * </p>
     * </li>
     * <li>
     * <p>
     * Middle East (UAE)
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * <b>Aurora global clusters are recorded in all enabled Regions</b>
     * </p>
     * <p>
     * The <code>AWS::RDS::GlobalCluster</code> resource type will be recorded in all supported Config Regions where the
     * configuration recorder is enabled, even if <code>includeGlobalResourceTypes</code> is set<code>false</code>. The
     * <code>includeGlobalResourceTypes</code> option is a bundle which only applies to IAM users, groups, roles, and
     * customer managed policies.
     * </p>
     * <p>
     * If you do not want to record <code>AWS::RDS::GlobalCluster</code> in all enabled Regions, use one of the
     * following recording strategies:
     * </p>
     * <ol>
     * <li>
     * <p>
     * <b>Record all current and future resource types with exclusions</b> (<code>EXCLUSION_BY_RESOURCE_TYPES</code>),
     * or
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Record specific resource types</b> (<code>INCLUSION_BY_RESOURCE_TYPES</code>).
     * </p>
     * </li>
     * </ol>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/select-resources.html#select-resources-all"
     * >Selecting Which Resources are Recorded</a> in the <i>Config developer guide</i>.
     * </p>
     * </important> <important>
     * <p>
     * <b>includeGlobalResourceTypes and the exclusion recording strategy</b>
     * </p>
     * <p>
     * The <code>includeGlobalResourceTypes</code> field has no impact on the <code>EXCLUSION_BY_RESOURCE_TYPES</code>
     * recording strategy. This means that the global IAM resource types (IAM users, groups, roles, and customer managed
     * policies) will not be automatically added as exclusions for <code>exclusionByResourceTypes</code> when
     * <code>includeGlobalResourceTypes</code> is set to <code>false</code>.
     * </p>
     * <p>
     * The <code>includeGlobalResourceTypes</code> field should only be used to modify the <code>AllSupported</code>
     * field, as the default for the <code>AllSupported</code> field is to record configuration changes for all
     * supported resource types excluding the global IAM resource types. To include the global IAM resource types when
     * <code>AllSupported</code> is set to <code>true</code>, make sure to set <code>includeGlobalResourceTypes</code>
     * to <code>true</code>.
     * </p>
     * <p>
     * To exclude the global IAM resource types for the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy, you
     * need to manually add them to the <code>resourceTypes</code> field of <code>exclusionByResourceTypes</code>.
     * </p>
     * </important> <note>
     * <p>
     * <b>Required and optional fields</b>
     * </p>
     * <p>
     * Before you set this field to <code>true</code>, set the <code>allSupported</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
     * <code>true</code>. Optionally, you can set the <code>useOnly</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
     * <code>ALL_SUPPORTED_RESOURCE_TYPES</code>.
     * </p>
     * </note> <note>
     * <p>
     * <b>Overriding fields</b>
     * </p>
     * <p>
     * If you set this field to <code>false</code> but list global IAM resource types in the <code>resourceTypes</code>
     * field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>, Config
     * will still record configuration changes for those specified resource types <i>regardless</i> of if you set the
     * <code>includeGlobalResourceTypes</code> field to false.
     * </p>
     * <p>
     * If you do not want to record configuration changes to the global IAM resource types (IAM users, groups, roles,
     * and customer managed policies), make sure to not list them in the <code>resourceTypes</code> field in addition to
     * setting the <code>includeGlobalResourceTypes</code> field to false.
     * </p>
     * </note>
     * 
     * @return This option is a bundle which only applies to the global IAM resource types: IAM users, groups, roles,
     *         and customer managed policies. These global IAM resource types can only be recorded by Config in Regions
     *         where Config was available before February 2022. You cannot be record the global IAM resouce types in
     *         Regions supported by Config after February 2022. This list where you cannot record the global IAM
     *         resource types includes the following Regions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Asia Pacific (Hyderabad)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Asia Pacific (Melbourne)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Canada West (Calgary)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Europe (Spain)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Europe (Zurich)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Israel (Tel Aviv)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Middle East (UAE)
     *         </p>
     *         </li>
     *         </ul>
     *         <important>
     *         <p>
     *         <b>Aurora global clusters are recorded in all enabled Regions</b>
     *         </p>
     *         <p>
     *         The <code>AWS::RDS::GlobalCluster</code> resource type will be recorded in all supported Config Regions
     *         where the configuration recorder is enabled, even if <code>includeGlobalResourceTypes</code> is set
     *         <code>false</code>. The <code>includeGlobalResourceTypes</code> option is a bundle which only applies to
     *         IAM users, groups, roles, and customer managed policies.
     *         </p>
     *         <p>
     *         If you do not want to record <code>AWS::RDS::GlobalCluster</code> in all enabled Regions, use one of the
     *         following recording strategies:
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         <b>Record all current and future resource types with exclusions</b> (
     *         <code>EXCLUSION_BY_RESOURCE_TYPES</code>), or
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Record specific resource types</b> (<code>INCLUSION_BY_RESOURCE_TYPES</code>).
     *         </p>
     *         </li>
     *         </ol>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/config/latest/developerguide/select-resources.html#select-resources-all"
     *         >Selecting Which Resources are Recorded</a> in the <i>Config developer guide</i>.
     *         </p>
     *         </important> <important>
     *         <p>
     *         <b>includeGlobalResourceTypes and the exclusion recording strategy</b>
     *         </p>
     *         <p>
     *         The <code>includeGlobalResourceTypes</code> field has no impact on the
     *         <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy. This means that the global IAM resource
     *         types (IAM users, groups, roles, and customer managed policies) will not be automatically added as
     *         exclusions for <code>exclusionByResourceTypes</code> when <code>includeGlobalResourceTypes</code> is set
     *         to <code>false</code>.
     *         </p>
     *         <p>
     *         The <code>includeGlobalResourceTypes</code> field should only be used to modify the
     *         <code>AllSupported</code> field, as the default for the <code>AllSupported</code> field is to record
     *         configuration changes for all supported resource types excluding the global IAM resource types. To
     *         include the global IAM resource types when <code>AllSupported</code> is set to <code>true</code>, make
     *         sure to set <code>includeGlobalResourceTypes</code> to <code>true</code>.
     *         </p>
     *         <p>
     *         To exclude the global IAM resource types for the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording
     *         strategy, you need to manually add them to the <code>resourceTypes</code> field of
     *         <code>exclusionByResourceTypes</code>.
     *         </p>
     *         </important> <note>
     *         <p>
     *         <b>Required and optional fields</b>
     *         </p>
     *         <p>
     *         Before you set this field to <code>true</code>, set the <code>allSupported</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>
     *         to <code>true</code>. Optionally, you can set the <code>useOnly</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
     *         >RecordingStrategy</a> to <code>ALL_SUPPORTED_RESOURCE_TYPES</code>.
     *         </p>
     *         </note> <note>
     *         <p>
     *         <b>Overriding fields</b>
     *         </p>
     *         <p>
     *         If you set this field to <code>false</code> but list global IAM resource types in the
     *         <code>resourceTypes</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>,
     *         Config will still record configuration changes for those specified resource types <i>regardless</i> of if
     *         you set the <code>includeGlobalResourceTypes</code> field to false.
     *         </p>
     *         <p>
     *         If you do not want to record configuration changes to the global IAM resource types (IAM users, groups,
     *         roles, and customer managed policies), make sure to not list them in the <code>resourceTypes</code> field
     *         in addition to setting the <code>includeGlobalResourceTypes</code> field to false.
     *         </p>
     */
    public final Boolean includeGlobalResourceTypes() {
        return includeGlobalResourceTypes;
    }

    /**
     * <p>
     * A comma-separated list that specifies which resource types Config records.
     * </p>
     * <p>
     * For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a href=
     * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     * >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
     * </p>
     * <note>
     * <p>
     * <b>Required and optional fields</b>
     * </p>
     * <p>
     * Optionally, you can set the <code>useOnly</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
     * <code>INCLUSION_BY_RESOURCE_TYPES</code>.
     * </p>
     * <p>
     * To record all configuration changes, set the <code>allSupported</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
     * <code>true</code>, and either omit this field or don't specify any resource types in this field. If you set the
     * <code>allSupported</code> field to <code>false</code> and specify values for <code>resourceTypes</code>, when
     * Config adds support for a new type of resource, it will not record resources of that type unless you manually add
     * that type to your recording group.
     * </p>
     * </note> <note>
     * <p>
     * <b>Region availability</b>
     * </p>
     * <p>
     * Before specifying a resource type for Config to track, check <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html">Resource
     * Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web Services Region
     * where you set up Config. If a resource type is supported by Config in at least one Region, you can enable the
     * recording of that resource type in all Regions supported by Config, even if the specified resource type is not
     * supported in the Amazon Web Services Region where you set up Config.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return A comma-separated list that specifies which resource types Config records.</p>
     *         <p>
     *         For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a
     *         href=
     *         "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     *         >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         <b>Required and optional fields</b>
     *         </p>
     *         <p>
     *         Optionally, you can set the <code>useOnly</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
     *         >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
     *         </p>
     *         <p>
     *         To record all configuration changes, set the <code>allSupported</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>
     *         to <code>true</code>, and either omit this field or don't specify any resource types in this field. If
     *         you set the <code>allSupported</code> field to <code>false</code> and specify values for
     *         <code>resourceTypes</code>, when Config adds support for a new type of resource, it will not record
     *         resources of that type unless you manually add that type to your recording group.
     *         </p>
     *         </note> <note>
     *         <p>
     *         <b>Region availability</b>
     *         </p>
     *         <p>
     *         Before specifying a resource type for Config to track, check <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
     *         >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web
     *         Services Region where you set up Config. If a resource type is supported by Config in at least one
     *         Region, you can enable the recording of that resource type in all Regions supported by Config, even if
     *         the specified resource type is not supported in the Amazon Web Services Region where you set up Config.
     *         </p>
     */
    public final List<ResourceType> resourceTypes() {
        return ResourceTypeListCopier.copyStringToEnum(resourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypes() {
        return resourceTypes != null && !(resourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A comma-separated list that specifies which resource types Config records.
     * </p>
     * <p>
     * For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a href=
     * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     * >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
     * </p>
     * <note>
     * <p>
     * <b>Required and optional fields</b>
     * </p>
     * <p>
     * Optionally, you can set the <code>useOnly</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
     * <code>INCLUSION_BY_RESOURCE_TYPES</code>.
     * </p>
     * <p>
     * To record all configuration changes, set the <code>allSupported</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
     * <code>true</code>, and either omit this field or don't specify any resource types in this field. If you set the
     * <code>allSupported</code> field to <code>false</code> and specify values for <code>resourceTypes</code>, when
     * Config adds support for a new type of resource, it will not record resources of that type unless you manually add
     * that type to your recording group.
     * </p>
     * </note> <note>
     * <p>
     * <b>Region availability</b>
     * </p>
     * <p>
     * Before specifying a resource type for Config to track, check <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html">Resource
     * Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web Services Region
     * where you set up Config. If a resource type is supported by Config in at least one Region, you can enable the
     * recording of that resource type in all Regions supported by Config, even if the specified resource type is not
     * supported in the Amazon Web Services Region where you set up Config.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return A comma-separated list that specifies which resource types Config records.</p>
     *         <p>
     *         For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a
     *         href=
     *         "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     *         >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         <b>Required and optional fields</b>
     *         </p>
     *         <p>
     *         Optionally, you can set the <code>useOnly</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
     *         >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
     *         </p>
     *         <p>
     *         To record all configuration changes, set the <code>allSupported</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>
     *         to <code>true</code>, and either omit this field or don't specify any resource types in this field. If
     *         you set the <code>allSupported</code> field to <code>false</code> and specify values for
     *         <code>resourceTypes</code>, when Config adds support for a new type of resource, it will not record
     *         resources of that type unless you manually add that type to your recording group.
     *         </p>
     *         </note> <note>
     *         <p>
     *         <b>Region availability</b>
     *         </p>
     *         <p>
     *         Before specifying a resource type for Config to track, check <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
     *         >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web
     *         Services Region where you set up Config. If a resource type is supported by Config in at least one
     *         Region, you can enable the recording of that resource type in all Regions supported by Config, even if
     *         the specified resource type is not supported in the Amazon Web Services Region where you set up Config.
     *         </p>
     */
    public final List<String> resourceTypesAsStrings() {
        return resourceTypes;
    }

    /**
     * <p>
     * An object that specifies how Config excludes resource types from being recorded by the configuration recorder.
     * </p>
     * <note>
     * <p>
     * <b>Required fields</b>
     * </p>
     * <p>
     * To use this option, you must set the <code>useOnly</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
     * <code>EXCLUSION_BY_RESOURCE_TYPES</code>.
     * </p>
     * </note>
     * 
     * @return An object that specifies how Config excludes resource types from being recorded by the configuration
     *         recorder.</p> <note>
     *         <p>
     *         <b>Required fields</b>
     *         </p>
     *         <p>
     *         To use this option, you must set the <code>useOnly</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
     *         >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>.
     *         </p>
     */
    public final ExclusionByResourceTypes exclusionByResourceTypes() {
        return exclusionByResourceTypes;
    }

    /**
     * <p>
     * An object that specifies the recording strategy for the configuration recorder.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you set the <code>useOnly</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
     * <code>ALL_SUPPORTED_RESOURCE_TYPES</code>, Config records configuration changes for all supported resource types,
     * excluding the global IAM resource types. You also must set the <code>allSupported</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
     * <code>true</code>. When Config adds support for a new resource type, Config automatically starts recording
     * resources of that type.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you set the <code>useOnly</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
     * <code>INCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes for only the resource types you
     * specify in the <code>resourceTypes</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you set the <code>useOnly</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
     * <code>EXCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes for all supported resource types
     * except the resource types that you specify to exclude from being recorded in the <code>resourceTypes</code> field
     * of <a href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html">
     * ExclusionByResourceTypes</a>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * <b>Required and optional fields</b>
     * </p>
     * <p>
     * The <code>recordingStrategy</code> field is optional when you set the <code>allSupported</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
     * <code>true</code>.
     * </p>
     * <p>
     * The <code>recordingStrategy</code> field is optional when you list resource types in the
     * <code>resourceTypes</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
     * </p>
     * <p>
     * The <code>recordingStrategy</code> field is required if you list resource types to exclude from recording in the
     * <code>resourceTypes</code> field of <a
     * href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
     * >ExclusionByResourceTypes</a>.
     * </p>
     * </note> <note>
     * <p>
     * <b>Overriding fields</b>
     * </p>
     * <p>
     * If you choose <code>EXCLUSION_BY_RESOURCE_TYPES</code> for the recording strategy, the
     * <code>exclusionByResourceTypes</code> field will override other properties in the request.
     * </p>
     * <p>
     * For example, even if you set <code>includeGlobalResourceTypes</code> to false, global IAM resource types will
     * still be automatically recorded in this option unless those resource types are specifically listed as exclusions
     * in the <code>resourceTypes</code> field of <code>exclusionByResourceTypes</code>.
     * </p>
     * </note> <note>
     * <p>
     * <b>Global resources types and the resource exclusion recording strategy</b>
     * </p>
     * <p>
     * By default, if you choose the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy, when Config adds
     * support for a new resource type in the Region where you set up the configuration recorder, including global
     * resource types, Config starts recording resources of that type automatically.
     * </p>
     * <p>
     * Unless specifically listed as exclusions, <code>AWS::RDS::GlobalCluster</code> will be recorded automatically in
     * all supported Config Regions were the configuration recorder is enabled.
     * </p>
     * <p>
     * IAM users, groups, roles, and customer managed policies will be recorded in the Region where you set up the
     * configuration recorder if that is a Region where Config was available before February 2022. You cannot be record
     * the global IAM resouce types in Regions supported by Config after February 2022. This list where you cannot
     * record the global IAM resource types includes the following Regions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Asia Pacific (Hyderabad)
     * </p>
     * </li>
     * <li>
     * <p>
     * Asia Pacific (Melbourne)
     * </p>
     * </li>
     * <li>
     * <p>
     * Canada West (Calgary)
     * </p>
     * </li>
     * <li>
     * <p>
     * Europe (Spain)
     * </p>
     * </li>
     * <li>
     * <p>
     * Europe (Zurich)
     * </p>
     * </li>
     * <li>
     * <p>
     * Israel (Tel Aviv)
     * </p>
     * </li>
     * <li>
     * <p>
     * Middle East (UAE)
     * </p>
     * </li>
     * </ul>
     * </note>
     * 
     * @return An object that specifies the recording strategy for the configuration recorder.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you set the <code>useOnly</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
     *         >RecordingStrategy</a> to <code>ALL_SUPPORTED_RESOURCE_TYPES</code>, Config records configuration changes
     *         for all supported resource types, excluding the global IAM resource types. You also must set the
     *         <code>allSupported</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>
     *         to <code>true</code>. When Config adds support for a new resource type, Config automatically starts
     *         recording resources of that type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you set the <code>useOnly</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
     *         >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes
     *         for only the resource types you specify in the <code>resourceTypes</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you set the <code>useOnly</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
     *         >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes
     *         for all supported resource types except the resource types that you specify to exclude from being
     *         recorded in the <code>resourceTypes</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
     *         >ExclusionByResourceTypes</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         <b>Required and optional fields</b>
     *         </p>
     *         <p>
     *         The <code>recordingStrategy</code> field is optional when you set the <code>allSupported</code> field of
     *         <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>
     *         to <code>true</code>.
     *         </p>
     *         <p>
     *         The <code>recordingStrategy</code> field is optional when you list resource types in the
     *         <code>resourceTypes</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
     *         </p>
     *         <p>
     *         The <code>recordingStrategy</code> field is required if you list resource types to exclude from recording
     *         in the <code>resourceTypes</code> field of <a
     *         href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
     *         >ExclusionByResourceTypes</a>.
     *         </p>
     *         </note> <note>
     *         <p>
     *         <b>Overriding fields</b>
     *         </p>
     *         <p>
     *         If you choose <code>EXCLUSION_BY_RESOURCE_TYPES</code> for the recording strategy, the
     *         <code>exclusionByResourceTypes</code> field will override other properties in the request.
     *         </p>
     *         <p>
     *         For example, even if you set <code>includeGlobalResourceTypes</code> to false, global IAM resource types
     *         will still be automatically recorded in this option unless those resource types are specifically listed
     *         as exclusions in the <code>resourceTypes</code> field of <code>exclusionByResourceTypes</code>.
     *         </p>
     *         </note> <note>
     *         <p>
     *         <b>Global resources types and the resource exclusion recording strategy</b>
     *         </p>
     *         <p>
     *         By default, if you choose the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy, when Config
     *         adds support for a new resource type in the Region where you set up the configuration recorder, including
     *         global resource types, Config starts recording resources of that type automatically.
     *         </p>
     *         <p>
     *         Unless specifically listed as exclusions, <code>AWS::RDS::GlobalCluster</code> will be recorded
     *         automatically in all supported Config Regions were the configuration recorder is enabled.
     *         </p>
     *         <p>
     *         IAM users, groups, roles, and customer managed policies will be recorded in the Region where you set up
     *         the configuration recorder if that is a Region where Config was available before February 2022. You
     *         cannot be record the global IAM resouce types in Regions supported by Config after February 2022. This
     *         list where you cannot record the global IAM resource types includes the following Regions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Asia Pacific (Hyderabad)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Asia Pacific (Melbourne)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Canada West (Calgary)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Europe (Spain)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Europe (Zurich)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Israel (Tel Aviv)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Middle East (UAE)
     *         </p>
     *         </li>
     *         </ul>
     */
    public final RecordingStrategy recordingStrategy() {
        return recordingStrategy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allSupported());
        hashCode = 31 * hashCode + Objects.hashCode(includeGlobalResourceTypes());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypes() ? resourceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(exclusionByResourceTypes());
        hashCode = 31 * hashCode + Objects.hashCode(recordingStrategy());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecordingGroup)) {
            return false;
        }
        RecordingGroup other = (RecordingGroup) obj;
        return Objects.equals(allSupported(), other.allSupported())
                && Objects.equals(includeGlobalResourceTypes(), other.includeGlobalResourceTypes())
                && hasResourceTypes() == other.hasResourceTypes()
                && Objects.equals(resourceTypesAsStrings(), other.resourceTypesAsStrings())
                && Objects.equals(exclusionByResourceTypes(), other.exclusionByResourceTypes())
                && Objects.equals(recordingStrategy(), other.recordingStrategy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecordingGroup").add("AllSupported", allSupported())
                .add("IncludeGlobalResourceTypes", includeGlobalResourceTypes())
                .add("ResourceTypes", hasResourceTypes() ? resourceTypesAsStrings() : null)
                .add("ExclusionByResourceTypes", exclusionByResourceTypes()).add("RecordingStrategy", recordingStrategy())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "allSupported":
            return Optional.ofNullable(clazz.cast(allSupported()));
        case "includeGlobalResourceTypes":
            return Optional.ofNullable(clazz.cast(includeGlobalResourceTypes()));
        case "resourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypesAsStrings()));
        case "exclusionByResourceTypes":
            return Optional.ofNullable(clazz.cast(exclusionByResourceTypes()));
        case "recordingStrategy":
            return Optional.ofNullable(clazz.cast(recordingStrategy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecordingGroup, T> g) {
        return obj -> g.apply((RecordingGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecordingGroup> {
        /**
         * <p>
         * Specifies whether Config records configuration changes for all supported resource types, excluding the global
         * IAM resource types.
         * </p>
         * <p>
         * If you set this field to <code>true</code>, when Config adds support for a new resource type, Config starts
         * recording resources of that type automatically.
         * </p>
         * <p>
         * If you set this field to <code>true</code>, you cannot enumerate specific resource types to record in the
         * <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>, or
         * to exclude in the <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         * >ExclusionByResourceTypes</a>.
         * </p>
         * <note>
         * <p>
         * <b>Region availability</b>
         * </p>
         * <p>
         * Check <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         * >Resource Coverage by Region Availability</a> to see if a resource type is supported in the Amazon Web
         * Services Region where you set up Config.
         * </p>
         * </note>
         * 
         * @param allSupported
         *        Specifies whether Config records configuration changes for all supported resource types, excluding the
         *        global IAM resource types.</p>
         *        <p>
         *        If you set this field to <code>true</code>, when Config adds support for a new resource type, Config
         *        starts recording resources of that type automatically.
         *        </p>
         *        <p>
         *        If you set this field to <code>true</code>, you cannot enumerate specific resource types to record in
         *        the <code>resourceTypes</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a>, or to exclude in the <code>resourceTypes</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         *        >ExclusionByResourceTypes</a>.
         *        </p>
         *        <note>
         *        <p>
         *        <b>Region availability</b>
         *        </p>
         *        <p>
         *        Check <a href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         *        >Resource Coverage by Region Availability</a> to see if a resource type is supported in the Amazon Web
         *        Services Region where you set up Config.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allSupported(Boolean allSupported);

        /**
         * <p>
         * This option is a bundle which only applies to the global IAM resource types: IAM users, groups, roles, and
         * customer managed policies. These global IAM resource types can only be recorded by Config in Regions where
         * Config was available before February 2022. You cannot be record the global IAM resouce types in Regions
         * supported by Config after February 2022. This list where you cannot record the global IAM resource types
         * includes the following Regions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Asia Pacific (Hyderabad)
         * </p>
         * </li>
         * <li>
         * <p>
         * Asia Pacific (Melbourne)
         * </p>
         * </li>
         * <li>
         * <p>
         * Canada West (Calgary)
         * </p>
         * </li>
         * <li>
         * <p>
         * Europe (Spain)
         * </p>
         * </li>
         * <li>
         * <p>
         * Europe (Zurich)
         * </p>
         * </li>
         * <li>
         * <p>
         * Israel (Tel Aviv)
         * </p>
         * </li>
         * <li>
         * <p>
         * Middle East (UAE)
         * </p>
         * </li>
         * </ul>
         * <important>
         * <p>
         * <b>Aurora global clusters are recorded in all enabled Regions</b>
         * </p>
         * <p>
         * The <code>AWS::RDS::GlobalCluster</code> resource type will be recorded in all supported Config Regions where
         * the configuration recorder is enabled, even if <code>includeGlobalResourceTypes</code> is set
         * <code>false</code>. The <code>includeGlobalResourceTypes</code> option is a bundle which only applies to IAM
         * users, groups, roles, and customer managed policies.
         * </p>
         * <p>
         * If you do not want to record <code>AWS::RDS::GlobalCluster</code> in all enabled Regions, use one of the
         * following recording strategies:
         * </p>
         * <ol>
         * <li>
         * <p>
         * <b>Record all current and future resource types with exclusions</b> (<code>EXCLUSION_BY_RESOURCE_TYPES</code>
         * ), or
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Record specific resource types</b> (<code>INCLUSION_BY_RESOURCE_TYPES</code>).
         * </p>
         * </li>
         * </ol>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/select-resources.html#select-resources-all"
         * >Selecting Which Resources are Recorded</a> in the <i>Config developer guide</i>.
         * </p>
         * </important> <important>
         * <p>
         * <b>includeGlobalResourceTypes and the exclusion recording strategy</b>
         * </p>
         * <p>
         * The <code>includeGlobalResourceTypes</code> field has no impact on the
         * <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy. This means that the global IAM resource types
         * (IAM users, groups, roles, and customer managed policies) will not be automatically added as exclusions for
         * <code>exclusionByResourceTypes</code> when <code>includeGlobalResourceTypes</code> is set to
         * <code>false</code>.
         * </p>
         * <p>
         * The <code>includeGlobalResourceTypes</code> field should only be used to modify the <code>AllSupported</code>
         * field, as the default for the <code>AllSupported</code> field is to record configuration changes for all
         * supported resource types excluding the global IAM resource types. To include the global IAM resource types
         * when <code>AllSupported</code> is set to <code>true</code>, make sure to set
         * <code>includeGlobalResourceTypes</code> to <code>true</code>.
         * </p>
         * <p>
         * To exclude the global IAM resource types for the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy,
         * you need to manually add them to the <code>resourceTypes</code> field of
         * <code>exclusionByResourceTypes</code>.
         * </p>
         * </important> <note>
         * <p>
         * <b>Required and optional fields</b>
         * </p>
         * <p>
         * Before you set this field to <code>true</code>, set the <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>. Optionally, you can set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>ALL_SUPPORTED_RESOURCE_TYPES</code>.
         * </p>
         * </note> <note>
         * <p>
         * <b>Overriding fields</b>
         * </p>
         * <p>
         * If you set this field to <code>false</code> but list global IAM resource types in the
         * <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>,
         * Config will still record configuration changes for those specified resource types <i>regardless</i> of if you
         * set the <code>includeGlobalResourceTypes</code> field to false.
         * </p>
         * <p>
         * If you do not want to record configuration changes to the global IAM resource types (IAM users, groups,
         * roles, and customer managed policies), make sure to not list them in the <code>resourceTypes</code> field in
         * addition to setting the <code>includeGlobalResourceTypes</code> field to false.
         * </p>
         * </note>
         * 
         * @param includeGlobalResourceTypes
         *        This option is a bundle which only applies to the global IAM resource types: IAM users, groups, roles,
         *        and customer managed policies. These global IAM resource types can only be recorded by Config in
         *        Regions where Config was available before February 2022. You cannot be record the global IAM resouce
         *        types in Regions supported by Config after February 2022. This list where you cannot record the global
         *        IAM resource types includes the following Regions:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Asia Pacific (Hyderabad)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Asia Pacific (Melbourne)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Canada West (Calgary)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Europe (Spain)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Europe (Zurich)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Israel (Tel Aviv)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Middle East (UAE)
         *        </p>
         *        </li>
         *        </ul>
         *        <important>
         *        <p>
         *        <b>Aurora global clusters are recorded in all enabled Regions</b>
         *        </p>
         *        <p>
         *        The <code>AWS::RDS::GlobalCluster</code> resource type will be recorded in all supported Config
         *        Regions where the configuration recorder is enabled, even if <code>includeGlobalResourceTypes</code>
         *        is set<code>false</code>. The <code>includeGlobalResourceTypes</code> option is a bundle which only
         *        applies to IAM users, groups, roles, and customer managed policies.
         *        </p>
         *        <p>
         *        If you do not want to record <code>AWS::RDS::GlobalCluster</code> in all enabled Regions, use one of
         *        the following recording strategies:
         *        </p>
         *        <ol>
         *        <li>
         *        <p>
         *        <b>Record all current and future resource types with exclusions</b> (
         *        <code>EXCLUSION_BY_RESOURCE_TYPES</code>), or
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Record specific resource types</b> (<code>INCLUSION_BY_RESOURCE_TYPES</code>).
         *        </p>
         *        </li>
         *        </ol>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/select-resources.html#select-resources-all"
         *        >Selecting Which Resources are Recorded</a> in the <i>Config developer guide</i>.
         *        </p>
         *        </important> <important>
         *        <p>
         *        <b>includeGlobalResourceTypes and the exclusion recording strategy</b>
         *        </p>
         *        <p>
         *        The <code>includeGlobalResourceTypes</code> field has no impact on the
         *        <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy. This means that the global IAM resource
         *        types (IAM users, groups, roles, and customer managed policies) will not be automatically added as
         *        exclusions for <code>exclusionByResourceTypes</code> when <code>includeGlobalResourceTypes</code> is
         *        set to <code>false</code>.
         *        </p>
         *        <p>
         *        The <code>includeGlobalResourceTypes</code> field should only be used to modify the
         *        <code>AllSupported</code> field, as the default for the <code>AllSupported</code> field is to record
         *        configuration changes for all supported resource types excluding the global IAM resource types. To
         *        include the global IAM resource types when <code>AllSupported</code> is set to <code>true</code>, make
         *        sure to set <code>includeGlobalResourceTypes</code> to <code>true</code>.
         *        </p>
         *        <p>
         *        To exclude the global IAM resource types for the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording
         *        strategy, you need to manually add them to the <code>resourceTypes</code> field of
         *        <code>exclusionByResourceTypes</code>.
         *        </p>
         *        </important> <note>
         *        <p>
         *        <b>Required and optional fields</b>
         *        </p>
         *        <p>
         *        Before you set this field to <code>true</code>, set the <code>allSupported</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a> to <code>true</code>. Optionally, you can set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">
         *        RecordingStrategy</a> to <code>ALL_SUPPORTED_RESOURCE_TYPES</code>.
         *        </p>
         *        </note> <note>
         *        <p>
         *        <b>Overriding fields</b>
         *        </p>
         *        <p>
         *        If you set this field to <code>false</code> but list global IAM resource types in the
         *        <code>resourceTypes</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a>, Config will still record configuration changes for those specified resource types
         *        <i>regardless</i> of if you set the <code>includeGlobalResourceTypes</code> field to false.
         *        </p>
         *        <p>
         *        If you do not want to record configuration changes to the global IAM resource types (IAM users,
         *        groups, roles, and customer managed policies), make sure to not list them in the
         *        <code>resourceTypes</code> field in addition to setting the <code>includeGlobalResourceTypes</code>
         *        field to false.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeGlobalResourceTypes(Boolean includeGlobalResourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config records.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         * </p>
         * <note>
         * <p>
         * <b>Required and optional fields</b>
         * </p>
         * <p>
         * Optionally, you can set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         * </p>
         * <p>
         * To record all configuration changes, set the <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>, and either omit this field or don't specify any resource types in this field. If you set
         * the <code>allSupported</code> field to <code>false</code> and specify values for <code>resourceTypes</code>,
         * when Config adds support for a new type of resource, it will not record resources of that type unless you
         * manually add that type to your recording group.
         * </p>
         * </note> <note>
         * <p>
         * <b>Region availability</b>
         * </p>
         * <p>
         * Before specifying a resource type for Config to track, check <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         * >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web
         * Services Region where you set up Config. If a resource type is supported by Config in at least one Region,
         * you can enable the recording of that resource type in all Regions supported by Config, even if the specified
         * resource type is not supported in the Amazon Web Services Region where you set up Config.
         * </p>
         * </note>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config records.</p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        <b>Required and optional fields</b>
         *        </p>
         *        <p>
         *        Optionally, you can set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         *        </p>
         *        <p>
         *        To record all configuration changes, set the <code>allSupported</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a> to <code>true</code>, and either omit this field or don't specify any resource
         *        types in this field. If you set the <code>allSupported</code> field to <code>false</code> and specify
         *        values for <code>resourceTypes</code>, when Config adds support for a new type of resource, it will
         *        not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        </note> <note>
         *        <p>
         *        <b>Region availability</b>
         *        </p>
         *        <p>
         *        Before specifying a resource type for Config to track, check <a href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         *        >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon
         *        Web Services Region where you set up Config. If a resource type is supported by Config in at least one
         *        Region, you can enable the recording of that resource type in all Regions supported by Config, even if
         *        the specified resource type is not supported in the Amazon Web Services Region where you set up
         *        Config.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(Collection<String> resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config records.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         * </p>
         * <note>
         * <p>
         * <b>Required and optional fields</b>
         * </p>
         * <p>
         * Optionally, you can set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         * </p>
         * <p>
         * To record all configuration changes, set the <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>, and either omit this field or don't specify any resource types in this field. If you set
         * the <code>allSupported</code> field to <code>false</code> and specify values for <code>resourceTypes</code>,
         * when Config adds support for a new type of resource, it will not record resources of that type unless you
         * manually add that type to your recording group.
         * </p>
         * </note> <note>
         * <p>
         * <b>Region availability</b>
         * </p>
         * <p>
         * Before specifying a resource type for Config to track, check <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         * >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web
         * Services Region where you set up Config. If a resource type is supported by Config in at least one Region,
         * you can enable the recording of that resource type in all Regions supported by Config, even if the specified
         * resource type is not supported in the Amazon Web Services Region where you set up Config.
         * </p>
         * </note>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config records.</p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        <b>Required and optional fields</b>
         *        </p>
         *        <p>
         *        Optionally, you can set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         *        </p>
         *        <p>
         *        To record all configuration changes, set the <code>allSupported</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a> to <code>true</code>, and either omit this field or don't specify any resource
         *        types in this field. If you set the <code>allSupported</code> field to <code>false</code> and specify
         *        values for <code>resourceTypes</code>, when Config adds support for a new type of resource, it will
         *        not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        </note> <note>
         *        <p>
         *        <b>Region availability</b>
         *        </p>
         *        <p>
         *        Before specifying a resource type for Config to track, check <a href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         *        >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon
         *        Web Services Region where you set up Config. If a resource type is supported by Config in at least one
         *        Region, you can enable the recording of that resource type in all Regions supported by Config, even if
         *        the specified resource type is not supported in the Amazon Web Services Region where you set up
         *        Config.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(String... resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config records.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         * </p>
         * <note>
         * <p>
         * <b>Required and optional fields</b>
         * </p>
         * <p>
         * Optionally, you can set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         * </p>
         * <p>
         * To record all configuration changes, set the <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>, and either omit this field or don't specify any resource types in this field. If you set
         * the <code>allSupported</code> field to <code>false</code> and specify values for <code>resourceTypes</code>,
         * when Config adds support for a new type of resource, it will not record resources of that type unless you
         * manually add that type to your recording group.
         * </p>
         * </note> <note>
         * <p>
         * <b>Region availability</b>
         * </p>
         * <p>
         * Before specifying a resource type for Config to track, check <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         * >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web
         * Services Region where you set up Config. If a resource type is supported by Config in at least one Region,
         * you can enable the recording of that resource type in all Regions supported by Config, even if the specified
         * resource type is not supported in the Amazon Web Services Region where you set up Config.
         * </p>
         * </note>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config records.</p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        <b>Required and optional fields</b>
         *        </p>
         *        <p>
         *        Optionally, you can set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         *        </p>
         *        <p>
         *        To record all configuration changes, set the <code>allSupported</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a> to <code>true</code>, and either omit this field or don't specify any resource
         *        types in this field. If you set the <code>allSupported</code> field to <code>false</code> and specify
         *        values for <code>resourceTypes</code>, when Config adds support for a new type of resource, it will
         *        not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        </note> <note>
         *        <p>
         *        <b>Region availability</b>
         *        </p>
         *        <p>
         *        Before specifying a resource type for Config to track, check <a href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         *        >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon
         *        Web Services Region where you set up Config. If a resource type is supported by Config in at least one
         *        Region, you can enable the recording of that resource type in all Regions supported by Config, even if
         *        the specified resource type is not supported in the Amazon Web Services Region where you set up
         *        Config.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<ResourceType> resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config records.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         * </p>
         * <note>
         * <p>
         * <b>Required and optional fields</b>
         * </p>
         * <p>
         * Optionally, you can set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         * </p>
         * <p>
         * To record all configuration changes, set the <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>, and either omit this field or don't specify any resource types in this field. If you set
         * the <code>allSupported</code> field to <code>false</code> and specify values for <code>resourceTypes</code>,
         * when Config adds support for a new type of resource, it will not record resources of that type unless you
         * manually add that type to your recording group.
         * </p>
         * </note> <note>
         * <p>
         * <b>Region availability</b>
         * </p>
         * <p>
         * Before specifying a resource type for Config to track, check <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         * >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon Web
         * Services Region where you set up Config. If a resource type is supported by Config in at least one Region,
         * you can enable the recording of that resource type in all Regions supported by Config, even if the specified
         * resource type is not supported in the Amazon Web Services Region where you set up Config.
         * </p>
         * </note>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config records.</p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>Resource Type Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported Amazon Web Services resource Types</a> in the <i>Config developer guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        <b>Required and optional fields</b>
         *        </p>
         *        <p>
         *        Optionally, you can set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>.
         *        </p>
         *        <p>
         *        To record all configuration changes, set the <code>allSupported</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a> to <code>true</code>, and either omit this field or don't specify any resource
         *        types in this field. If you set the <code>allSupported</code> field to <code>false</code> and specify
         *        values for <code>resourceTypes</code>, when Config adds support for a new type of resource, it will
         *        not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        </note> <note>
         *        <p>
         *        <b>Region availability</b>
         *        </p>
         *        <p>
         *        Before specifying a resource type for Config to track, check <a href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/what-is-resource-config-coverage.html"
         *        >Resource Coverage by Region Availability</a> to see if the resource type is supported in the Amazon
         *        Web Services Region where you set up Config. If a resource type is supported by Config in at least one
         *        Region, you can enable the recording of that resource type in all Regions supported by Config, even if
         *        the specified resource type is not supported in the Amazon Web Services Region where you set up
         *        Config.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(ResourceType... resourceTypes);

        /**
         * <p>
         * An object that specifies how Config excludes resource types from being recorded by the configuration
         * recorder.
         * </p>
         * <note>
         * <p>
         * <b>Required fields</b>
         * </p>
         * <p>
         * To use this option, you must set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>.
         * </p>
         * </note>
         * 
         * @param exclusionByResourceTypes
         *        An object that specifies how Config excludes resource types from being recorded by the configuration
         *        recorder.</p> <note>
         *        <p>
         *        <b>Required fields</b>
         *        </p>
         *        <p>
         *        To use this option, you must set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionByResourceTypes(ExclusionByResourceTypes exclusionByResourceTypes);

        /**
         * <p>
         * An object that specifies how Config excludes resource types from being recorded by the configuration
         * recorder.
         * </p>
         * <note>
         * <p>
         * <b>Required fields</b>
         * </p>
         * <p>
         * To use this option, you must set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ExclusionByResourceTypes.Builder}
         * avoiding the need to create one manually via {@link ExclusionByResourceTypes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExclusionByResourceTypes.Builder#build()} is called immediately
         * and its result is passed to {@link #exclusionByResourceTypes(ExclusionByResourceTypes)}.
         * 
         * @param exclusionByResourceTypes
         *        a consumer that will call methods on {@link ExclusionByResourceTypes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #exclusionByResourceTypes(ExclusionByResourceTypes)
         */
        default Builder exclusionByResourceTypes(Consumer<ExclusionByResourceTypes.Builder> exclusionByResourceTypes) {
            return exclusionByResourceTypes(ExclusionByResourceTypes.builder().applyMutation(exclusionByResourceTypes).build());
        }

        /**
         * <p>
         * An object that specifies the recording strategy for the configuration recorder.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>ALL_SUPPORTED_RESOURCE_TYPES</code>, Config records configuration changes for
         * all supported resource types, excluding the global IAM resource types. You also must set the
         * <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>. When Config adds support for a new resource type, Config automatically starts recording
         * resources of that type.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes for
         * only the resource types you specify in the <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes for
         * all supported resource types except the resource types that you specify to exclude from being recorded in the
         * <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         * >ExclusionByResourceTypes</a>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * <b>Required and optional fields</b>
         * </p>
         * <p>
         * The <code>recordingStrategy</code> field is optional when you set the <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>.
         * </p>
         * <p>
         * The <code>recordingStrategy</code> field is optional when you list resource types in the
         * <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
         * </p>
         * <p>
         * The <code>recordingStrategy</code> field is required if you list resource types to exclude from recording in
         * the <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         * >ExclusionByResourceTypes</a>.
         * </p>
         * </note> <note>
         * <p>
         * <b>Overriding fields</b>
         * </p>
         * <p>
         * If you choose <code>EXCLUSION_BY_RESOURCE_TYPES</code> for the recording strategy, the
         * <code>exclusionByResourceTypes</code> field will override other properties in the request.
         * </p>
         * <p>
         * For example, even if you set <code>includeGlobalResourceTypes</code> to false, global IAM resource types will
         * still be automatically recorded in this option unless those resource types are specifically listed as
         * exclusions in the <code>resourceTypes</code> field of <code>exclusionByResourceTypes</code>.
         * </p>
         * </note> <note>
         * <p>
         * <b>Global resources types and the resource exclusion recording strategy</b>
         * </p>
         * <p>
         * By default, if you choose the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy, when Config adds
         * support for a new resource type in the Region where you set up the configuration recorder, including global
         * resource types, Config starts recording resources of that type automatically.
         * </p>
         * <p>
         * Unless specifically listed as exclusions, <code>AWS::RDS::GlobalCluster</code> will be recorded automatically
         * in all supported Config Regions were the configuration recorder is enabled.
         * </p>
         * <p>
         * IAM users, groups, roles, and customer managed policies will be recorded in the Region where you set up the
         * configuration recorder if that is a Region where Config was available before February 2022. You cannot be
         * record the global IAM resouce types in Regions supported by Config after February 2022. This list where you
         * cannot record the global IAM resource types includes the following Regions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Asia Pacific (Hyderabad)
         * </p>
         * </li>
         * <li>
         * <p>
         * Asia Pacific (Melbourne)
         * </p>
         * </li>
         * <li>
         * <p>
         * Canada West (Calgary)
         * </p>
         * </li>
         * <li>
         * <p>
         * Europe (Spain)
         * </p>
         * </li>
         * <li>
         * <p>
         * Europe (Zurich)
         * </p>
         * </li>
         * <li>
         * <p>
         * Israel (Tel Aviv)
         * </p>
         * </li>
         * <li>
         * <p>
         * Middle East (UAE)
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param recordingStrategy
         *        An object that specifies the recording strategy for the configuration recorder.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>ALL_SUPPORTED_RESOURCE_TYPES</code>, Config records configuration
         *        changes for all supported resource types, excluding the global IAM resource types. You also must set
         *        the <code>allSupported</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a> to <code>true</code>. When Config adds support for a new resource type, Config
         *        automatically starts recording resources of that type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration
         *        changes for only the resource types you specify in the <code>resourceTypes</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you set the <code>useOnly</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         *        >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration
         *        changes for all supported resource types except the resource types that you specify to exclude from
         *        being recorded in the <code>resourceTypes</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         *        >ExclusionByResourceTypes</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        <b>Required and optional fields</b>
         *        </p>
         *        <p>
         *        The <code>recordingStrategy</code> field is optional when you set the <code>allSupported</code> field
         *        of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup
         *        </a> to <code>true</code>.
         *        </p>
         *        <p>
         *        The <code>recordingStrategy</code> field is optional when you list resource types in the
         *        <code>resourceTypes</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html"
         *        >RecordingGroup</a>.
         *        </p>
         *        <p>
         *        The <code>recordingStrategy</code> field is required if you list resource types to exclude from
         *        recording in the <code>resourceTypes</code> field of <a
         *        href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         *        >ExclusionByResourceTypes</a>.
         *        </p>
         *        </note> <note>
         *        <p>
         *        <b>Overriding fields</b>
         *        </p>
         *        <p>
         *        If you choose <code>EXCLUSION_BY_RESOURCE_TYPES</code> for the recording strategy, the
         *        <code>exclusionByResourceTypes</code> field will override other properties in the request.
         *        </p>
         *        <p>
         *        For example, even if you set <code>includeGlobalResourceTypes</code> to false, global IAM resource
         *        types will still be automatically recorded in this option unless those resource types are specifically
         *        listed as exclusions in the <code>resourceTypes</code> field of <code>exclusionByResourceTypes</code>.
         *        </p>
         *        </note> <note>
         *        <p>
         *        <b>Global resources types and the resource exclusion recording strategy</b>
         *        </p>
         *        <p>
         *        By default, if you choose the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy, when Config
         *        adds support for a new resource type in the Region where you set up the configuration recorder,
         *        including global resource types, Config starts recording resources of that type automatically.
         *        </p>
         *        <p>
         *        Unless specifically listed as exclusions, <code>AWS::RDS::GlobalCluster</code> will be recorded
         *        automatically in all supported Config Regions were the configuration recorder is enabled.
         *        </p>
         *        <p>
         *        IAM users, groups, roles, and customer managed policies will be recorded in the Region where you set
         *        up the configuration recorder if that is a Region where Config was available before February 2022. You
         *        cannot be record the global IAM resouce types in Regions supported by Config after February 2022. This
         *        list where you cannot record the global IAM resource types includes the following Regions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Asia Pacific (Hyderabad)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Asia Pacific (Melbourne)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Canada West (Calgary)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Europe (Spain)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Europe (Zurich)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Israel (Tel Aviv)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Middle East (UAE)
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordingStrategy(RecordingStrategy recordingStrategy);

        /**
         * <p>
         * An object that specifies the recording strategy for the configuration recorder.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>ALL_SUPPORTED_RESOURCE_TYPES</code>, Config records configuration changes for
         * all supported resource types, excluding the global IAM resource types. You also must set the
         * <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>. When Config adds support for a new resource type, Config automatically starts recording
         * resources of that type.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>INCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes for
         * only the resource types you specify in the <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you set the <code>useOnly</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html"
         * >RecordingStrategy</a> to <code>EXCLUSION_BY_RESOURCE_TYPES</code>, Config records configuration changes for
         * all supported resource types except the resource types that you specify to exclude from being recorded in the
         * <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         * >ExclusionByResourceTypes</a>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * <b>Required and optional fields</b>
         * </p>
         * <p>
         * The <code>recordingStrategy</code> field is optional when you set the <code>allSupported</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a> to
         * <code>true</code>.
         * </p>
         * <p>
         * The <code>recordingStrategy</code> field is optional when you list resource types in the
         * <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingGroup.html">RecordingGroup</a>.
         * </p>
         * <p>
         * The <code>recordingStrategy</code> field is required if you list resource types to exclude from recording in
         * the <code>resourceTypes</code> field of <a
         * href="https://docs.aws.amazon.com/config/latest/APIReference/API_ExclusionByResourceTypes.html"
         * >ExclusionByResourceTypes</a>.
         * </p>
         * </note> <note>
         * <p>
         * <b>Overriding fields</b>
         * </p>
         * <p>
         * If you choose <code>EXCLUSION_BY_RESOURCE_TYPES</code> for the recording strategy, the
         * <code>exclusionByResourceTypes</code> field will override other properties in the request.
         * </p>
         * <p>
         * For example, even if you set <code>includeGlobalResourceTypes</code> to false, global IAM resource types will
         * still be automatically recorded in this option unless those resource types are specifically listed as
         * exclusions in the <code>resourceTypes</code> field of <code>exclusionByResourceTypes</code>.
         * </p>
         * </note> <note>
         * <p>
         * <b>Global resources types and the resource exclusion recording strategy</b>
         * </p>
         * <p>
         * By default, if you choose the <code>EXCLUSION_BY_RESOURCE_TYPES</code> recording strategy, when Config adds
         * support for a new resource type in the Region where you set up the configuration recorder, including global
         * resource types, Config starts recording resources of that type automatically.
         * </p>
         * <p>
         * Unless specifically listed as exclusions, <code>AWS::RDS::GlobalCluster</code> will be recorded automatically
         * in all supported Config Regions were the configuration recorder is enabled.
         * </p>
         * <p>
         * IAM users, groups, roles, and customer managed policies will be recorded in the Region where you set up the
         * configuration recorder if that is a Region where Config was available before February 2022. You cannot be
         * record the global IAM resouce types in Regions supported by Config after February 2022. This list where you
         * cannot record the global IAM resource types includes the following Regions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Asia Pacific (Hyderabad)
         * </p>
         * </li>
         * <li>
         * <p>
         * Asia Pacific (Melbourne)
         * </p>
         * </li>
         * <li>
         * <p>
         * Canada West (Calgary)
         * </p>
         * </li>
         * <li>
         * <p>
         * Europe (Spain)
         * </p>
         * </li>
         * <li>
         * <p>
         * Europe (Zurich)
         * </p>
         * </li>
         * <li>
         * <p>
         * Israel (Tel Aviv)
         * </p>
         * </li>
         * <li>
         * <p>
         * Middle East (UAE)
         * </p>
         * </li>
         * </ul>
         * </note> This is a convenience method that creates an instance of the {@link RecordingStrategy.Builder}
         * avoiding the need to create one manually via {@link RecordingStrategy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecordingStrategy.Builder#build()} is called immediately and its
         * result is passed to {@link #recordingStrategy(RecordingStrategy)}.
         * 
         * @param recordingStrategy
         *        a consumer that will call methods on {@link RecordingStrategy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordingStrategy(RecordingStrategy)
         */
        default Builder recordingStrategy(Consumer<RecordingStrategy.Builder> recordingStrategy) {
            return recordingStrategy(RecordingStrategy.builder().applyMutation(recordingStrategy).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Boolean allSupported;

        private Boolean includeGlobalResourceTypes;

        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private ExclusionByResourceTypes exclusionByResourceTypes;

        private RecordingStrategy recordingStrategy;

        private BuilderImpl() {
        }

        private BuilderImpl(RecordingGroup model) {
            allSupported(model.allSupported);
            includeGlobalResourceTypes(model.includeGlobalResourceTypes);
            resourceTypesWithStrings(model.resourceTypes);
            exclusionByResourceTypes(model.exclusionByResourceTypes);
            recordingStrategy(model.recordingStrategy);
        }

        public final Boolean getAllSupported() {
            return allSupported;
        }

        public final void setAllSupported(Boolean allSupported) {
            this.allSupported = allSupported;
        }

        @Override
        public final Builder allSupported(Boolean allSupported) {
            this.allSupported = allSupported;
            return this;
        }

        public final Boolean getIncludeGlobalResourceTypes() {
            return includeGlobalResourceTypes;
        }

        public final void setIncludeGlobalResourceTypes(Boolean includeGlobalResourceTypes) {
            this.includeGlobalResourceTypes = includeGlobalResourceTypes;
        }

        @Override
        public final Builder includeGlobalResourceTypes(Boolean includeGlobalResourceTypes) {
            this.includeGlobalResourceTypes = includeGlobalResourceTypes;
            return this;
        }

        public final Collection<String> getResourceTypes() {
            if (resourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypes;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
        }

        @Override
        public final Builder resourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesWithStrings(String... resourceTypes) {
            resourceTypesWithStrings(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public final Builder resourceTypes(Collection<ResourceType> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copyEnumToString(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(ResourceType... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final ExclusionByResourceTypes.Builder getExclusionByResourceTypes() {
            return exclusionByResourceTypes != null ? exclusionByResourceTypes.toBuilder() : null;
        }

        public final void setExclusionByResourceTypes(ExclusionByResourceTypes.BuilderImpl exclusionByResourceTypes) {
            this.exclusionByResourceTypes = exclusionByResourceTypes != null ? exclusionByResourceTypes.build() : null;
        }

        @Override
        public final Builder exclusionByResourceTypes(ExclusionByResourceTypes exclusionByResourceTypes) {
            this.exclusionByResourceTypes = exclusionByResourceTypes;
            return this;
        }

        public final RecordingStrategy.Builder getRecordingStrategy() {
            return recordingStrategy != null ? recordingStrategy.toBuilder() : null;
        }

        public final void setRecordingStrategy(RecordingStrategy.BuilderImpl recordingStrategy) {
            this.recordingStrategy = recordingStrategy != null ? recordingStrategy.build() : null;
        }

        @Override
        public final Builder recordingStrategy(RecordingStrategy recordingStrategy) {
            this.recordingStrategy = recordingStrategy;
            return this;
        }

        @Override
        public RecordingGroup build() {
            return new RecordingGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
