/*
 * Copyright 2002-2004 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 

package org.springframework.aop.framework.autoproxy.target;

import org.springframework.aop.target.AbstractPrototypeBasedTargetSource;
import org.springframework.aop.target.CommonsPoolTargetSource;
import org.springframework.aop.target.PrototypeTargetSource;
import org.springframework.aop.target.ThreadLocalTargetSource;
import org.springframework.beans.factory.BeanFactory;

/**
 * Convenient TargetSourceCreator using bean name prefixes to create one of three
 * well-known TargetSource types: 
 * <li>: CommonsPoolTargetSource
 * <li>% ThreadLocalTargetSource
 * <li>! PrototypeTargetSource
 * @author Rod Johnson
 * @see org.springframework.aop.target.CommonsPoolTargetSource
 * @see org.springframework.aop.target.ThreadLocalTargetSource
 * @see org.springframework.aop.target.PrototypeTargetSource
 */
public class QuickTargetSourceCreator extends AbstractPrototypeBasedTargetSourceCreator {

	public static final String PREFIX_COMMONS_POOL = ":";

	public static final String PREFIX_THREAD_LOCAL = "%";

	public static final String PREFIX_PROTOTYPE = "!";

	protected final AbstractPrototypeBasedTargetSource createPrototypeTargetSource(
			Object bean, String beanName, BeanFactory factory) {
		if (beanName.startsWith(PREFIX_COMMONS_POOL)) {
			CommonsPoolTargetSource cpts = new CommonsPoolTargetSource();
			cpts.setMaxSize(25);
			return cpts;
		}
		else if (beanName.startsWith(PREFIX_THREAD_LOCAL)) {
			return new ThreadLocalTargetSource();
		}
		else if (beanName.startsWith(PREFIX_PROTOTYPE)) {
			return new PrototypeTargetSource();
		}
		else {
			// No match. Don't create a custom target source.
			return null;
		}
	}
	
}
