/**
 *  Copyright 2005-2015 Red Hat, Inc.
 *
 *  Red Hat licenses this file to you under the Apache License, version
 *  2.0 (the "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *  implied.  See the License for the specific language governing
 *  permissions and limitations under the License.
 */
package io.fabric8.commands;

import io.fabric8.api.ZooKeeperClusterService;
import io.fabric8.api.scr.ValidatingReference;
import io.fabric8.boot.commands.support.AbstractCommandComponent;

import io.fabric8.commands.support.EnsembleAddCompleter;
import io.fabric8.zookeeper.bootstrap.BootstrapConfiguration;
import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.basic.AbstractCommand;
import org.apache.felix.scr.annotations.Activate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Deactivate;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.felix.service.command.Function;
import org.osgi.framework.BundleContext;

@Component(immediate = true)
@Service({ Function.class, AbstractCommand.class })
@org.apache.felix.scr.annotations.Properties({
        @Property(name = "osgi.command.scope", value = EnsembleAdd.SCOPE_VALUE),
        @Property(name = "osgi.command.function", value = EnsembleAdd.FUNCTION_VALUE)
})
public class EnsembleAdd extends AbstractCommandComponent {

    public static final String SCOPE_VALUE = "fabric";
    public static final String FUNCTION_VALUE = "ensemble-add";
    public static final String DESCRIPTION = "Extend the current fabric ensemble by converting the specified containers into ensemble servers";

    @Reference(referenceInterface = ZooKeeperClusterService.class)
    private final ValidatingReference<ZooKeeperClusterService> clusterService = new ValidatingReference<ZooKeeperClusterService>();

    @Reference(referenceInterface = BootstrapConfiguration.class)
    private final ValidatingReference<BootstrapConfiguration> bootstrapConfiguration = new ValidatingReference<BootstrapConfiguration>();


    // Completers
    @Reference(referenceInterface = EnsembleAddCompleter.class, bind = "bindContainerCompleter", unbind = "unbindContainerCompleter")
    private EnsembleAddCompleter containerCompleter; // dummy field

    private BundleContext bundleContext;

    @Activate
    void activate(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
        activateComponent();
    }

    @Deactivate
    void deactivate() {
        deactivateComponent();
    }

    @Override
    public Action createNewAction() {
        assertValid();
        return new EnsembleAddAction(bundleContext, clusterService.get(), bootstrapConfiguration.get());
    }

    void bindClusterService(ZooKeeperClusterService clusterService) {
        this.clusterService.bind(clusterService);
    }

    void unbindClusterService(ZooKeeperClusterService clusterService) {
        this.clusterService.unbind(clusterService);
    }

    void bindContainerCompleter(EnsembleAddCompleter completer) {
        bindCompleter(completer);
    }

    void unbindContainerCompleter(EnsembleAddCompleter completer) {
        unbindCompleter(completer);
    }

    void bindBootstrapConfiguration(BootstrapConfiguration configuration) {
        this.bootstrapConfiguration.bind(configuration);
    }

    void unbindBootstrapConfiguration(BootstrapConfiguration configuration) {
        this.bootstrapConfiguration.unbind(configuration);
    }
}
