/*
Copyright 2014 The Kubernetes Authors All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package host_path

import (
	"fmt"

	"k8s.io/kubernetes/pkg/api"
	"k8s.io/kubernetes/pkg/types"
	"k8s.io/kubernetes/pkg/util/mount"
	"k8s.io/kubernetes/pkg/volume"
)

// This is the primary entrypoint for volume plugins.
// The volumeConfig arg provides the ability to configure volume behavior.  It is implemented as a pointer to allow nils.
// The hostPathPlugin is used to store the volumeConfig and give it, when needed, to the func that creates HostPath Recyclers.
// Tests that exercise recycling should not use this func but instead use ProbeRecyclablePlugins() to override default behavior.
func ProbeVolumePlugins(volumeConfig *volume.VolumeConfig) []volume.VolumePlugin {
	return []volume.VolumePlugin{
		&hostPathPlugin{
			host:            nil,
			newRecyclerFunc: newRecycler,
			volumeConfig:    volumeConfig,
		},
	}
}

func ProbeRecyclableVolumePlugins(recyclerFunc func(spec *volume.Spec, host volume.VolumeHost, volumeConfig *volume.VolumeConfig) (volume.Recycler, error), volumeConfig *volume.VolumeConfig) []volume.VolumePlugin {
	return []volume.VolumePlugin{
		&hostPathPlugin{
			host:            nil,
			newRecyclerFunc: recyclerFunc,
			volumeConfig:    volumeConfig,
		},
	}
}

type hostPathPlugin struct {
	host volume.VolumeHost
	// decouple creating recyclers by deferring to a function.  Allows for easier testing.
	newRecyclerFunc func(spec *volume.Spec, host volume.VolumeHost, volumeConfig *volume.VolumeConfig) (volume.Recycler, error)
	volumeConfig    *volume.VolumeConfig
}

var _ volume.VolumePlugin = &hostPathPlugin{}
var _ volume.PersistentVolumePlugin = &hostPathPlugin{}
var _ volume.RecyclableVolumePlugin = &hostPathPlugin{}

const (
	hostPathPluginName = "kubernetes.io/host-path"
)

func (plugin *hostPathPlugin) Init(host volume.VolumeHost) {
	plugin.host = host
}

func (plugin *hostPathPlugin) Name() string {
	return hostPathPluginName
}

func (plugin *hostPathPlugin) CanSupport(spec *volume.Spec) bool {
	return (spec.PersistentVolume != nil && spec.PersistentVolume.Spec.HostPath != nil) ||
		(spec.Volume != nil && spec.Volume.HostPath != nil)
}

func (plugin *hostPathPlugin) GetAccessModes() []api.PersistentVolumeAccessMode {
	return []api.PersistentVolumeAccessMode{
		api.ReadWriteOnce,
	}
}

func (plugin *hostPathPlugin) NewBuilder(spec *volume.Spec, pod *api.Pod, _ volume.VolumeOptions, _ mount.Interface) (volume.Builder, error) {
	if spec.Volume != nil && spec.Volume.HostPath != nil {
		return &hostPathBuilder{
			hostPath: &hostPath{path: spec.Volume.HostPath.Path},
			readOnly: false,
		}, nil
	} else {
		return &hostPathBuilder{
			hostPath: &hostPath{path: spec.PersistentVolume.Spec.HostPath.Path},
			readOnly: spec.ReadOnly,
		}, nil
	}
}

func (plugin *hostPathPlugin) NewCleaner(volName string, podUID types.UID, _ mount.Interface) (volume.Cleaner, error) {
	return &hostPathCleaner{&hostPath{""}}, nil
}

func (plugin *hostPathPlugin) NewRecycler(spec *volume.Spec) (volume.Recycler, error) {
	if plugin.volumeConfig == nil {
		return nil, fmt.Errorf("VolumeConfig is nil for this plugin.  Recycler cannot be created.")
	}
	return plugin.newRecyclerFunc(spec, plugin.host, plugin.volumeConfig)
}

func newRecycler(spec *volume.Spec, host volume.VolumeHost, volumeConfig *volume.VolumeConfig) (volume.Recycler, error) {
	if spec.PersistentVolume == nil || spec.PersistentVolume.Spec.HostPath == nil {
		return nil, fmt.Errorf("spec.PersistentVolumeSource.HostPath is nil")
	}
	return &hostPathRecycler{
		name:         spec.Name,
		path:         spec.PersistentVolume.Spec.HostPath.Path,
		host:         host,
		volumeConfig: *volumeConfig,
		timeout:      volume.CalculateTimeoutForVolume(volumeConfig.PersistentVolumeRecyclerMinTimeoutHostPath, volumeConfig.PersistentVolumeRecyclerTimeoutIncrementHostPath, spec.PersistentVolume),
	}, nil
}

// HostPath volumes represent a bare host file or directory mount.
// The direct at the specified path will be directly exposed to the container.
type hostPath struct {
	path string
}

func (hp *hostPath) GetPath() string {
	return hp.path
}

type hostPathBuilder struct {
	*hostPath
	readOnly bool
}

var _ volume.Builder = &hostPathBuilder{}

// SetUp does nothing.
func (b *hostPathBuilder) SetUp() error {
	return nil
}

// SetUpAt does not make sense for host paths - probably programmer error.
func (b *hostPathBuilder) SetUpAt(dir string) error {
	return fmt.Errorf("SetUpAt() does not make sense for host paths")
}

func (b *hostPathBuilder) IsReadOnly() bool {
	return b.readOnly
}

func (b *hostPathBuilder) GetPath() string {
	return b.path
}

type hostPathCleaner struct {
	*hostPath
}

var _ volume.Cleaner = &hostPathCleaner{}

// TearDown does nothing.
func (c *hostPathCleaner) TearDown() error {
	return nil
}

// TearDownAt does not make sense for host paths - probably programmer error.
func (c *hostPathCleaner) TearDownAt(dir string) error {
	return fmt.Errorf("TearDownAt() does not make sense for host paths")
}

// hostPathRecycler scrubs a hostPath volume by running "rm -rf" on the volume in a pod
// This recycler only works on a single host cluster and is for testing purposes only.
type hostPathRecycler struct {
	name         string
	path         string
	host         volume.VolumeHost
	volumeConfig volume.VolumeConfig
	timeout      int64
}

func (r *hostPathRecycler) GetPath() string {
	return r.path
}

// Recycler provides methods to reclaim the volume resource.
// A HostPath is recycled by scheduling a pod to run "rm -rf" on the contents of the volume.  This is meant for
// development and testing in a single node cluster only.
// Recycle blocks until the pod has completed or any error occurs.
func (r *hostPathRecycler) Recycle() error {
	pod := r.volumeConfig.PersistentVolumeRecyclerDefaultScrubPod

	// overrides
	pod.Spec.ActiveDeadlineSeconds = &r.timeout
	pod.GenerateName = "pv-scrubber-hostpath-"
	pod.Spec.Volumes[0] = api.Volume{
		VolumeSource: api.VolumeSource{
			HostPath: &api.HostPathVolumeSource{
				Path: r.path,
			},
		},
	}
	return volume.ScrubPodVolumeAndWatchUntilCompletion(pod, r.host.GetKubeClient())
}
