/**
* @module Core
*/
var Core;
(function (Core) {
    Core.username = null;
    Core.password = null;

    /**
    * Controller that handles the login page and actually logging in
    *
    * @method LoginController
    * @for Core
    * @static
    * @param $scope
    * @param jolokia
    * @param userDetails
    * @param jolokiaUrl
    * @param workspace
    * @param localStorage
    * @param branding
    */
    function LoginController($scope, jolokia, userDetails, jolokiaUrl, workspace, localStorage, branding) {
        jolokia.stop();

        $scope.entity = {
            username: '',
            password: ''
        };
        $scope.backstretch = $.backstretch(branding.loginBg);

        $scope.rememberMe = false;
        $scope.branding = branding;

        var details = angular.fromJson(localStorage[jolokiaUrl]);
        if (details) {
            $scope.entity.username = details['username'];
            $scope.entity.password = details['password'];
            $scope.rememberMe = details['rememberMe'];
        }

        $scope.$on('$routeChangeStart', function () {
            if ($scope.backstretch) {
                $scope.backstretch.destroy();
            }
        });

        jQuery(window).bind("beforeunload", function () {
            // auto logout if we should not remember me
            if (!userDetails.rememberMe) {
                console.log("Auto logging out as remember me is off");
                Core.logout(jolokiaUrl, userDetails, localStorage, $scope);
            }
        });

        $scope.doLogin = function () {
            if (jolokiaUrl) {
                var url = jolokiaUrl.replace("jolokia", "auth/login/");

                $.ajax(url, {
                    type: "POST",
                    success: function (response) {
                        userDetails.username = $scope.entity.username;
                        userDetails.password = $scope.entity.password;
                        userDetails.rememberMe = $scope.rememberMe;
                        userDetails.loginDetails = response;

                        Core.username = $scope.entity.username;
                        Core.password = $scope.entity.password;
                        if ($scope.rememberMe) {
                            localStorage[jolokiaUrl] = angular.toJson(userDetails);
                        } else {
                            delete localStorage[jolokiaUrl];
                        }

                        jolokia.start();
                        workspace.loadTree();

                        Core.$apply($scope);
                    },
                    error: function (xhr, textStatus, error) {
                        switch (xhr.status) {
                            case 401:
                                notification('error', 'Failed to log in, ' + error);
                                break;
                            case 403:
                                notification('error', 'Failed to log in, ' + error);
                                break;
                            default:
                                notification('error', 'Failed to log in, ' + error);
                                break;
                        }
                        Core.$apply($scope);
                    },
                    beforeSend: function (xhr) {
                        xhr.setRequestHeader('Authorization', Core.getBasicAuthHeader($scope.entity.username, $scope.entity.password));
                    }
                });
            }
        };
    }
    Core.LoginController = LoginController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function WelcomeController($scope, $location, branding, localStorage) {
        var log = Logger.get("Welcome");

        $scope.stopShowingWelcomePage = function () {
            log.debug("Stop showing welcome page");
            localStorage['showWelcomePage'] = false;

            $location.path("/");
        };

        // load the welcome.md file
        $.ajax({
            url: "app/core/doc/welcome.md",
            dataType: 'html',
            cache: false,
            success: function (data, textStatus, jqXHR) {
                $scope.html = "Unable to download welcome.md";
                if (angular.isDefined(data)) {
                    $scope.html = marked(data);
                    $scope.branding = branding;
                }
                Core.$apply($scope);
            },
            error: function (jqXHR, textStatus, errorThrown) {
                $scope.html = "Unable to download welcome.md";
                Core.$apply($scope);
            }
        });
    }
    Core.WelcomeController = WelcomeController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    var HelpRegistry = (function () {
        function HelpRegistry($rootScope) {
            this.$rootScope = $rootScope;
            this.discoverableDocTypes = {
                user: 'help.md'
            };
            this.topicNameMappings = {
                activemq: 'ActiveMQ',
                camel: 'Camel',
                jboss: 'JBoss',
                jclouds: 'jclouds',
                jmx: 'JMX',
                jvm: 'JVM',
                log: 'Logs',
                openejb: 'OpenEJB'
            };
            this.subTopicNameMappings = {
                user: 'For Users',
                developer: 'For Developers',
                faq: 'FAQ'
            };
            // map plugin names to their path in the app
            this.pluginNameMappings = {
                hawtioCore: 'core',
                'hawtio-branding': 'branding',
                forceGraph: 'forcegraph',
                'hawtio-ui': 'ui',
                'hawtio-forms': 'forms',
                elasticjs: 'elasticsearch'
            };
            // let's not auto-discover help files in these plugins
            this.ignoredPlugins = [
                'core',
                'branding',
                'datatable',
                'forcegraph',
                'forms',
                'perspective',
                'tree',
                'ui'
            ];
            this.topics = {};
        }
        HelpRegistry.prototype.addUserDoc = function (topic, path, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            this.addSubTopic(topic, 'user', path, isValid);
        };

        HelpRegistry.prototype.addDevDoc = function (topic, path, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            this.addSubTopic(topic, 'developer', path, isValid);
        };

        HelpRegistry.prototype.addSubTopic = function (topic, subtopic, path, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            this.getOrCreateTopic(topic, isValid)[subtopic] = path;
        };

        HelpRegistry.prototype.getOrCreateTopic = function (topic, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            if (!angular.isDefined(this.topics[topic])) {
                if (isValid === null) {
                    isValid = function () {
                        return true;
                    };
                }

                this.topics[topic] = {
                    isValid: isValid
                };
                this.$rootScope.$broadcast('hawtioNewHelpTopic');
            }
            return this.topics[topic];
        };

        HelpRegistry.prototype.mapTopicName = function (name) {
            if (angular.isDefined(this.topicNameMappings[name])) {
                return this.topicNameMappings[name];
            }
            return name.capitalize();
        };

        HelpRegistry.prototype.mapSubTopicName = function (name) {
            if (angular.isDefined(this.subTopicNameMappings[name])) {
                return this.subTopicNameMappings[name];
            }
            return name.capitalize();
        };

        HelpRegistry.prototype.getTopics = function () {
            var answer = {};

            angular.forEach(this.topics, function (value, key) {
                if (value.isValid()) {
                    Core.log.debug(key, " is available");

                    // strip out any functions...
                    answer[key] = angular.fromJson(angular.toJson(value));
                } else {
                    Core.log.debug(key, " is not available");
                }
            });

            return answer;
        };

        HelpRegistry.prototype.disableAutodiscover = function (name) {
            this.ignoredPlugins.push(name);
        };

        HelpRegistry.prototype.discoverHelpFiles = function (plugins) {
            var self = this;

            console.log("Ignored plugins: ", self.ignoredPlugins);

            plugins.forEach(function (plugin) {
                var pluginName = self.pluginNameMappings[plugin];
                if (!angular.isDefined(pluginName)) {
                    pluginName = plugin;
                }

                if (!self.ignoredPlugins.any(function (p) {
                    return p === pluginName;
                })) {
                    angular.forEach(self.discoverableDocTypes, function (value, key) {
                        // avoid trying to discover these if plugins register them
                        if (!angular.isDefined(self[pluginName]) || !angular.isDefined(self[pluginName][key])) {
                            var target = 'app/' + pluginName + '/doc/' + value;
                            console.log("checking: ", target);

                            $.ajax(target, {
                                type: 'HEAD',
                                statusCode: {
                                    200: function () {
                                        self.getOrCreateTopic(plugin)[key] = target;
                                    }
                                }
                            });
                        }
                    });
                }
            });
        };
        return HelpRegistry;
    })();
    Core.HelpRegistry = HelpRegistry;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    // NOTE - $route is brought in here to ensure the factory for that service
    // has been called, otherwise the ng-include directive doesn't show the partial
    // after a refresh until you click a top-level link.
    function ViewController($scope, $route, $location, layoutTree, layoutFull, viewRegistry) {
        findViewPartial();

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            findViewPartial();
        });

        function searchRegistry(path) {
            var answer = undefined;
            Object.extended(viewRegistry).keys(function (key, value) {
                if (!answer) {
                    if (key.startsWith("/") && key.endsWith("/")) {
                        // assume its a regex
                        var text = key.substring(1, key.length - 1);
                        try  {
                            var reg = new RegExp(text, "");
                            if (reg.exec(path)) {
                                answer = value;
                            }
                        } catch (e) {
                            console.log("Invalid RegExp " + text + " for viewRegistry value: " + value);
                        }
                    } else {
                        if (path.startsWith(key)) {
                            answer = value;
                        }
                    }
                }
            });

            //console.log("Searching for: " + path + " returning: ", answer);
            return answer;
        }

        function findViewPartial() {
            var answer = null;
            var hash = $location.search();
            var tab = hash['tab'];
            if (angular.isString(tab)) {
                answer = searchRegistry(tab);
            }
            if (!answer) {
                var path = $location.path();
                if (path) {
                    if (path.startsWith("/")) {
                        path = path.substring(1);
                    }
                    answer = searchRegistry(path);
                }
            }
            if (!answer) {
                answer = layoutTree;
            }
            $scope.viewPartial = answer;

            console.log("Using view partial: " + answer);
            return answer;
        }
    }
    Core.ViewController = ViewController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * Parsers the given value as JSON if it is define
    */
    function parsePreferencesJson(value, key) {
        var answer = null;
        if (angular.isDefined(value)) {
            answer = Core.parseJsonText(value, "localStorage for " + key);
        }
        return answer;
    }
    Core.parsePreferencesJson = parsePreferencesJson;

    /**
    * Function to return the configured plugin for the given perspective. The returned
    * list is sorted in the configured order.
    * Notice the list contains plugins which may have been configured as disabled.
    */
    function configuredPluginsForPerspectiveId(perspectiveId, workspace, jolokia, localStorage) {
        // grab the top level tabs which is the plugins we can select as our default plugin
        var topLevelTabs = Perspective.topLevelTabsForPerspectiveId(workspace, perspectiveId);
        if (topLevelTabs && topLevelTabs.length > 0) {
            Core.log.debug("Found " + topLevelTabs.length + " plugins");

            // exclude invalid tabs at first
            topLevelTabs = topLevelTabs.filter(function (tab) {
                var href = tab.href();
                return href && isValidFunction(workspace, tab.isValid);
            });
            Core.log.debug("After filtering there are " + topLevelTabs.length + " plugins");

            var id = "plugins-" + perspectiveId;
            var initPlugins = parsePreferencesJson(localStorage[id], id);
            if (initPlugins) {
                // remove plugins which we cannot find active currently
                initPlugins = initPlugins.filter(function (p) {
                    return topLevelTabs.some(function (tab) {
                        return tab.id === p.id;
                    });
                });

                // add new active plugins which we didn't know about before
                topLevelTabs.forEach(function (tab) {
                    var knownPlugin = initPlugins.some(function (p) {
                        return p.id === tab.id;
                    });
                    if (!knownPlugin) {
                        Core.log.info("Discovered new plugin in JVM since loading configuration: " + tab.id);
                        initPlugins.push({ id: tab.id, index: -1, displayName: tab.content, enabled: true, isDefault: false });
                    }
                });
            } else {
                // okay no configured saved yet, so use what is active
                initPlugins = topLevelTabs;
            }
        }

        // okay push plugins to scope so we can see them in the UI
        var answer = safeTabsToPlugins(initPlugins);
        return answer;
    }
    Core.configuredPluginsForPerspectiveId = configuredPluginsForPerspectiveId;

    /**
    * Function which safely can turn tabs/plugins to plugins
    */
    function safeTabsToPlugins(tabs) {
        var answer = [];
        if (tabs) {
            tabs.forEach(function (tab, idx) {
                var name;
                if (angular.isUndefined(tab.displayName)) {
                    name = tab.content;
                } else {
                    name = tab.displayName;
                }
                var enabled;
                if (angular.isUndefined(tab.enabled)) {
                    enabled = true;
                } else {
                    enabled = tab.enabled;
                }
                var isDefault;
                if (angular.isUndefined(tab.isDefault)) {
                    isDefault = false;
                } else {
                    isDefault = tab.isDefault;
                }
                answer.push({ id: tab.id, index: idx, displayName: name, enabled: enabled, isDefault: isDefault });
            });
        }
        return answer;
    }
    Core.safeTabsToPlugins = safeTabsToPlugins;

    function filterTopLevelTabs(perspective, workspace, configuredPlugins) {
        var topLevelTabs = Perspective.topLevelTabsForPerspectiveId(workspace, perspective);
        if (perspective === "website")
            return topLevelTabs;

        // only include the tabs accordingly to configured
        var result = [];
        configuredPlugins.forEach(function (p) {
            if (p.enabled) {
                var pid = p.id;
                var tab = null;
                if (pid) {
                    tab = topLevelTabs.find(function (t) {
                        return t.id === pid;
                    });
                }
                if (tab) {
                    result.push(tab);
                }
            }
        });
        return result;
    }
    Core.filterTopLevelTabs = filterTopLevelTabs;

    /**
    * Returns true if there is no validFn defined or if its defined
    * then the function returns true.
    *
    * @method isValidFunction
    * @for Perspective
    * @param {Core.Workspace} workspace
    * @param {Function} validFn
    * @return {Boolean}
    */
    function isValidFunction(workspace, validFn) {
        return !validFn || validFn(workspace);
    }
    Core.isValidFunction = isValidFunction;

    /**
    * Gets the default configured plugin for the given perspective, or <tt>null</tt> if no default has been configured.
    */
    function getDefaultPlugin(perspectiveId, workspace, jolokia, localStorage) {
        var plugins = Core.configuredPluginsForPerspectiveId(perspectiveId, workspace, jolokia, localStorage);

        // find the default plugins
        var defaultPlugin = null;
        plugins.forEach(function (p) {
            if (p.isDefault) {
                defaultPlugin = p;
            }
        });
        return defaultPlugin;
    }
    Core.getDefaultPlugin = getDefaultPlugin;
})(Core || (Core = {}));
var __extends = this.__extends || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    __.prototype = b.prototype;
    d.prototype = new __();
};
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * @class TableWidget
    */
    // TODO would make sense to move this to UI
    var TableWidget = (function () {
        function TableWidget(scope, workspace, dataTableColumns, config) {
            if (typeof config === "undefined") { config = {}; }
            var _this = this;
            this.scope = scope;
            this.workspace = workspace;
            this.dataTableColumns = dataTableColumns;
            this.config = config;
            this.ignoreColumnHash = {};
            this.flattenColumnHash = {};
            this.detailTemplate = null;
            this.openMessages = [];
            this.addedExpandNodes = false;
            this.tableElement = null;
            this.sortColumns = null;
            this.dataTableConfig = {
                bPaginate: false,
                sDom: 'Rlfrtip',
                bDestroy: true,
                bAutoWidth: true
            };
            // the jQuery DataTable widget
            this.dataTable = null;
            // TODO is there an easier way of turning an array into a hash to true so it acts as a hash?
            angular.forEach(config.ignoreColumns, function (name) {
                _this.ignoreColumnHash[name] = true;
            });
            angular.forEach(config.flattenColumns, function (name) {
                _this.flattenColumnHash[name] = true;
            });

            var templateId = config.rowDetailTemplateId;
            if (templateId) {
                this.detailTemplate = workspace.$templateCache.get(templateId);
            }
        }
        /**
        * Adds new data to the table
        * @method addData
        * @for TableWidget
        * @param {Object} newData
        */
        TableWidget.prototype.addData = function (newData) {
            var dataTable = this.dataTable;
            dataTable.fnAddData(newData);
        };

        /**
        * Populates the table with the given data
        * @method populateTable
        * @for TableWidget
        * @param {Object} data
        */
        TableWidget.prototype.populateTable = function (data) {
            var _this = this;
            var $scope = this.scope;

            if (!data) {
                $scope.messages = [];
            } else {
                $scope.messages = data;

                var formatMessageDetails = function (dataTable, parentRow) {
                    var oData = dataTable.fnGetData(parentRow);
                    var div = $('<div>');
                    div.addClass('innerDetails');
                    _this.populateDetailDiv(oData, div);
                    return div;
                };

                var array = data;
                if (angular.isArray(data)) {
                } else if (angular.isObject(data)) {
                    array = [];
                    angular.forEach(data, function (object) {
                        return array.push(object);
                    });
                }

                var tableElement = this.tableElement;
                if (!tableElement) {
                    tableElement = $('#grid');
                }
                var tableTr = Core.getOrCreateElements(tableElement, ["thead", "tr"]);
                var tableBody = Core.getOrCreateElements(tableElement, ["tbody"]);
                var ths = $(tableTr).find("th");

                // lets add new columns based on the data...
                // TODO wont compile in TypeScript!
                //var columns = this.dataTableColumns.slice();
                var columns = [];
                angular.forEach(this.dataTableColumns, function (value) {
                    return columns.push(value);
                });

                //var columns = this.dataTableColumns.slice();
                var addColumn = function (key, title) {
                    columns.push({
                        "sDefaultContent": "",
                        "mData": null,
                        mDataProp: key
                    });

                    // lets see if we need to add another <th>
                    if (tableTr) {
                        $("<th>" + title + "</th>").appendTo(tableTr);
                    }
                };

                var checkForNewColumn = function (value, key, prefix) {
                    // lets check if we have a column data for it (if its not ignored)
                    //var keyName: string = key.toString();
                    //var config: Object = {mDataProp: key};
                    var found = _this.ignoreColumnHash[key] || columns.any(function (k, v) {
                        return "mDataProp" === k && v === key;
                    });

                    //var found = this.ignoreColumnHash[key] || columns.any(config);
                    if (!found) {
                        // lets check if its a flatten column
                        if (_this.flattenColumnHash[key]) {
                            // TODO so this only works on the first row - sucks! :)
                            if (angular.isObject(value)) {
                                var childPrefix = prefix + key + ".";
                                angular.forEach(value, function (value, key) {
                                    return checkForNewColumn(value, key, childPrefix);
                                });
                            }
                        } else {
                            addColumn(prefix + key, humanizeValue(key));
                        }
                    }
                };

                if (!this.config.disableAddColumns && angular.isArray(array) && array.length > 0) {
                    var first = array[0];
                    if (angular.isObject(first)) {
                        angular.forEach(first, function (value, key) {
                            return checkForNewColumn(value, key, "");
                        });
                    }
                }

                // lets default to column 1 sorting if there's no property on column 1 for expansion
                if (columns.length > 1) {
                    var col0 = columns[0];
                    if (!this.sortColumns && !col0["mDataProp"] && !col0["mData"]) {
                        var sortOrder = [[1, "asc"]];
                        this.sortColumns = sortOrder;
                    }
                }
                if (array.length && !angular.isArray(array[0])) {
                    //this.dataTableConfig["aoData"] = array;
                    this.dataTableConfig["aaData"] = array;
                } else {
                    this.dataTableConfig["aaData"] = array;
                }
                this.dataTableConfig["aoColumns"] = columns;
                if (this.sortColumns) {
                    this.dataTableConfig["aaSorting"] = this.sortColumns;
                }

                if (this.dataTable) {
                    this.dataTable.fnClearTable(false);
                    this.dataTable.fnAddData(array);
                    this.dataTable.fnDraw();
                    // lets try update it...
                } else {
                    this.dataTable = tableElement.dataTable(this.dataTableConfig);
                }

                var widget = this;

                if (this.dataTable) {
                    var keys = new KeyTable({
                        "table": tableElement[0],
                        "datatable": this.dataTable
                    });
                    keys.fnSetPosition(0, 0);

                    if (angular.isArray(data) && data.length) {
                        var selected = data[0];
                        var selectHandler = widget.config.selectHandler;
                        if (selected && selectHandler) {
                            selectHandler(selected);
                        }
                    }
                }

                // lets try focus on the table
                $(tableElement).focus();

                var widget = this;

                // add a handler for the expand/collapse column for all rows (and future rows)
                var expandCollapseNode = function () {
                    var dataTable = widget.dataTable;
                    var parentRow = this.parentNode;
                    var openMessages = widget.openMessages;
                    var i = $.inArray(parentRow, openMessages);

                    var element = $('i', this);
                    if (i === -1) {
                        element.removeClass('icon-plus');
                        element.addClass('icon-minus');
                        var dataDiv = formatMessageDetails(dataTable, parentRow);
                        var detailsRow = $(dataTable.fnOpen(parentRow, dataDiv, 'details'));
                        detailsRow.css("padding", "0");

                        setTimeout(function () {
                            detailsRow.find(".innerDetails").slideDown(400, function () {
                                $(parentRow).addClass('opened');
                                openMessages.push(parentRow);
                            });
                        }, 20);
                    } else {
                        $(parentRow.nextSibling).find(".innerDetails").slideUp(400, function () {
                            $(parentRow).removeClass('opened');
                            element.removeClass('icon-minus');
                            element.addClass('icon-plus');
                            dataTable.fnClose(parentRow);
                            openMessages.splice(i, 1);
                        });
                    }

                    // lets let angular render any new detail templates
                    Core.$apply($scope);
                };

                if (!this.addedExpandNodes) {
                    this.addedExpandNodes = true;

                    $(tableElement).on("click", "td.control", expandCollapseNode);

                    //$(document).on("click", "#grid td.control", expandCollapseNode);
                    keys.event.action(0, null, function (node) {
                        expandCollapseNode.call(node);
                    });
                }

                keys.event.focus(null, null, function (node) {
                    var dataTable = widget.dataTable;
                    var row = node;
                    if (node) {
                        var nodeName = node.nodeName;
                        if (nodeName) {
                            if (nodeName.toLowerCase() === "td") {
                                row = $(node).parents("tr")[0];
                            }
                            var selected = dataTable.fnGetData(row);
                            var selectHandler = widget.config.selectHandler;
                            if (selected && selectHandler) {
                                selectHandler(selected);
                            }
                        }
                    }
                });

                // $(document).on("click", "#grid td", function () {
                $(tableElement).find("td.control").on("click", function (event) {
                    var dataTable = widget.dataTable;
                    if ($(this).hasClass('selected')) {
                        $(this).removeClass('focus selected');
                    } else {
                        if (!widget.config.multiSelect) {
                            dataTable.$('td.selected').removeClass('focus selected');
                        }
                        $(this).addClass('focus selected');

                        var row = $(this).parents("tr")[0];
                        var selected = dataTable.fnGetData(row);
                        var selectHandler = widget.config.selectHandler;
                        if (selected && selectHandler) {
                            selectHandler(selected);
                        }
                    }
                });
            }
            Core.$apply($scope);
        };

        TableWidget.prototype.populateDetailDiv = function (row, div) {
            // lets remove the silly "0" property that gets shoved in there due to the expand/collapse row
            delete row["0"];
            var scope = this.scope.$new();
            scope.row = row;
            scope.templateDiv = div;
            var template = this.detailTemplate;
            if (!template) {
                var templateId = this.config.rowDetailTemplateId;
                if (templateId) {
                    this.detailTemplate = this.workspace.$templateCache.get(templateId);
                    template = this.detailTemplate;
                }
            }
            if (template) {
                div.html(template);
                this.workspace.$compile(div.contents())(scope);
            }
        };
        return TableWidget;
    })();
    Core.TableWidget = TableWidget;
})(Core || (Core = {}));

// TODO refactor other code to use Core.TableWidget
var TableWidget = (function (_super) {
    __extends(TableWidget, _super);
    function TableWidget() {
        _super.apply(this, arguments);
    }
    return TableWidget;
})(Core.TableWidget);
;
;
;
/**
* @module Core
*/
var Core;
(function (Core) {
    function NavBarController($scope, $location, workspace, $route, jolokia, localStorage) {
        $scope.hash = workspace.hash();
        $scope.topLevelTabs = [];
        $scope.subLevelTabs = workspace.subLevelTabs;
        $scope.currentPerspective = null;
        $scope.perspectiveDetails = {
            perspective: null
        };

        $scope.topLevelTabs = function () {
            reloadPerspective();

            // TODO transform the top level tabs based on the current perspective
            // TODO watch for changes to workspace.topLevelTabs and for the current perspective
            return workspace.topLevelTabs;
        };

        $scope.$on('jmxTreeUpdated', function () {
            reloadPerspective();
        });

        //$scope.subLevelTabs = () => workspace.subLevelTabs;
        $scope.validSelection = function (uri) {
            return workspace.validSelection(uri);
        };

        $scope.isValid = function (nav) {
            return nav && nav.isValid(workspace);
        };

        $scope.switchPerspective = function (perspective) {
            var searchPerspectiveId = $location.search()[Perspective.perspectiveSearchId];
            if (perspective && ($scope.currentPerspective !== perspective || perspective.id !== searchPerspectiveId)) {
                Logger.debug("Changed the perspective to " + JSON.stringify(perspective) + " from search id " + searchPerspectiveId);
                if ($scope.currentPerspective) {
                    $scope.currentPerspective.lastPage = $location.url();
                }
                var pid = perspective.id;
                $location.search(Perspective.perspectiveSearchId, pid);
                Logger.debug("Setting perspective to " + pid);
                $scope.currentPerspective = perspective;
                reloadPerspective();
                $scope.topLevelTabs = Perspective.getTopLevelTabsForPerspective($location, workspace, jolokia, localStorage);

                // is any of the top level tabs marked as default?
                var defaultPlugin = Core.getDefaultPlugin(pid, workspace, jolokia, localStorage);
                var defaultTab;
                var path;
                if (defaultPlugin) {
                    $scope.topLevelTabs.forEach(function (tab) {
                        if (tab.id === defaultPlugin.id) {
                            defaultTab = tab;
                        }
                    });
                    if (defaultTab) {
                        path = Core.trimLeading(defaultTab.href(), "#");
                    }
                } else {
                    // if no default plugin configured, then select the last page as the active location
                    if (perspective.lastPage) {
                        path = Core.trimLeading(perspective.lastPage, "#");
                    }
                }

                if (path) {
                    // lets avoid any old paths with ?p=" inside
                    var idx = path.indexOf("?p=") || path.indexOf("&p=");
                    if (idx > 0) {
                        path = path.substring(0, idx);
                    }
                    var sep = (path.indexOf("?") >= 0) ? "&" : "?";
                    path += sep + "p=" + pid;
                    $location.url(path);
                }
            }
        };

        $scope.$watch('hash', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                Core.log.debug("hash changed from ", oldValue, " to ", newValue);
            }
        });

        // when we change the view/selection lets update the hash so links have the latest stuff
        $scope.$on('$routeChangeSuccess', function () {
            $scope.hash = workspace.hash();
            reloadPerspective();
        });

        // use includePerspective = false as default as that was the previous behavior
        $scope.link = function (nav, includePerspective) {
            if (typeof includePerspective === "undefined") { includePerspective = false; }
            var href;
            if (angular.isString(nav)) {
                href = nav;
            } else {
                href = nav.href();
            }
            var removeParams = ['tab', 'nid', 'chapter', 'pref', 'q'];
            if (!includePerspective) {
                if (href.indexOf("?p=") >= 0 || href.indexOf("&p=") >= 0) {
                    removeParams.push("p");
                }
            }
            return Core.createHref($location, href, removeParams);
        };

        $scope.fullScreenLink = function () {
            var href = "#" + $location.path() + "?tab=notree";
            return Core.createHref($location, href, ['tab']);
        };

        $scope.addToDashboardLink = function () {
            var href = "#" + $location.path() + workspace.hash();

            var answer = "#/dashboard/add?tab=dashboard&href=" + encodeURIComponent(href);

            if ($location.url().has("/jmx/charts")) {
                var size = {
                    size_x: 4,
                    size_y: 3
                };

                answer += "&size=" + encodeURIComponent(angular.toJson(size));
            }

            return answer;
        };

        $scope.isActive = function (nav) {
            if (angular.isString(nav))
                return workspace.isLinkActive(nav);
            var fn = nav.isActive;
            if (fn) {
                return fn(workspace);
            }
            return workspace.isLinkActive(nav.href());
        };

        $scope.isTopTabActive = function (nav) {
            if (angular.isString(nav))
                return workspace.isTopTabActive(nav);
            var fn = nav.isActive;
            if (fn) {
                return fn(workspace);
            }
            return workspace.isTopTabActive(nav.href());
        };

        $scope.activeLink = function () {
            var tabs = $scope.topLevelTabs();
            if (!tabs) {
                return "Loading...";
            }
            var tab = tabs.find(function (nav) {
                return $scope.isActive(nav);
            });
            return tab ? tab['content'] : "";
        };

        function reloadPerspective() {
            var perspectives = Perspective.getPerspectives($location, workspace, jolokia, localStorage);
            var currentId = Perspective.currentPerspectiveId($location, workspace, jolokia, localStorage);

            console.log("reloading perspectives for " + currentId);

            if (currentId != $scope.perspectiveId || angular.toJson($scope.perspectives) !== angular.toJson(perspectives)) {
                $scope.perspectiveId = currentId;
                $scope.perspectives = perspectives;
                $scope.perspectiveDetails.perspective = $scope.perspectives.find(function (p) {
                    return p['id'] === currentId;
                });
                console.log("Current perspective ID: " + currentId);
                $scope.topLevelTabs = Perspective.getTopLevelTabsForPerspective($location, workspace, jolokia, localStorage);
            }
        }
        //reloadPerspective();
    }
    Core.NavBarController = NavBarController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * Simple helper class for creating <a href="http://angular-ui.github.io/bootstrap/#/modal">angular ui bootstrap modal dialogs</a>
    * @class Dialog
    */
    var Dialog = (function () {
        function Dialog() {
            this.show = false;
            this.options = {
                backdropFade: true,
                dialogFade: true
            };
        }
        /**
        * Opens the dialog
        * @method open
        */
        Dialog.prototype.open = function () {
            this.show = true;
        };

        /**
        * Closes the dialog
        * @method close
        */
        Dialog.prototype.close = function () {
            this.show = false;

            // lets make sure and remove any backgroup fades
            this.removeBackdropFadeDiv();
            setTimeout(this.removeBackdropFadeDiv, 100);
        };

        Dialog.prototype.removeBackdropFadeDiv = function () {
            $("div.modal-backdrop").remove();
        };
        return Dialog;
    })();
    Core.Dialog = Dialog;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function d3ForceGraph(scope, nodes, links, canvasElement) {
        // lets remove the old graph first
        if (scope.graphForce) {
            scope.graphForce.stop();
        }
        if (!canvasElement) {
            canvasElement = $("#canvas")[0];
        }
        var canvasDiv = $(canvasElement);
        canvasDiv.children("svg").remove();

        if (nodes.length) {
            var width = canvasDiv.parent().width();
            var height = canvasDiv.parent().height();

            if (height < 100) {
                //console.log("browse thinks the height is only " + height + " so calculating offset from doc height");
                var offset = canvasDiv.offset();
                height = $(document).height() - 5;
                if (offset) {
                    height -= offset['top'];
                }
            }

            //console.log("Using width " + width + " and height " + height);
            var svg = d3.select(canvasDiv[0]).append("svg").attr("width", width).attr("height", height);

            var force = d3.layout.force().distance(100).charge(-120 * 10).linkDistance(50).size([width, height]);

            scope.graphForce = force;

            /*
            var force = d3.layout.force()
            .gravity(.05)
            .distance(100)
            .charge(-100)
            .size([width, height]);
            */
            // prepare the arrows
            svg.append("svg:defs").selectAll("marker").data(["from"]).enter().append("svg:marker").attr("id", String).attr("viewBox", "0 -5 10 10").attr("refX", 25).attr("refY", -1.5).attr("markerWidth", 6).attr("markerHeight", 6).attr("orient", "auto").append("svg:path").attr("d", "M0,-5L10,0L0,5");

            force.nodes(nodes).links(links).start();

            var link = svg.selectAll(".link").data(links).enter().append("line").attr("class", "link");

            // draw the arrow
            link.attr("class", "link from");

            // end marker
            link.attr("marker-end", "url(#from)");

            var node = svg.selectAll(".node").data(nodes).enter().append("g").attr("class", "node").call(force.drag);

            node.append("image").attr("xlink:href", function (d) {
                return d.imageUrl;
            }).attr("x", -15).attr("y", -15).attr("width", 30).attr("height", 30);

            node.append("text").attr("dx", 20).attr("dy", ".35em").text(function (d) {
                return d.label;
            });

            force.on("tick", function () {
                link.attr("x1", function (d) {
                    return d.source.x;
                }).attr("y1", function (d) {
                    return d.source.y;
                }).attr("x2", function (d) {
                    return d.target.x;
                }).attr("y2", function (d) {
                    return d.target.y;
                });

                node.attr("transform", function (d) {
                    return "translate(" + d.x + "," + d.y + ")";
                });
            });
        }
    }
    Core.d3ForceGraph = d3ForceGraph;

    function createGraphStates(nodes, links, transitions) {
        var stateKeys = {};
        nodes.forEach(function (node) {
            var idx = node.id;
            if (idx === undefined) {
                console.log("No node found for node " + JSON.stringify(node));
            } else {
                if (node.edges === undefined)
                    node.edges = [];
                if (!node.label)
                    node.label = "node " + idx;
                stateKeys[idx] = node;
            }
        });
        var states = d3.values(stateKeys);
        links.forEach(function (d) {
            var source = stateKeys[d.source];
            var target = stateKeys[d.target];
            if (source === undefined || target === undefined) {
                console.log("Bad link!  " + source + " target " + target + " for " + d);
            } else {
                var edge = { source: source, target: target };
                transitions.push(edge);
                source.edges.push(edge);
                target.edges.push(edge);
                // TODO should we add the edge to the target?
            }
        });
        return states;
    }
    Core.createGraphStates = createGraphStates;

    // TODO Export as a service
    function dagreLayoutGraph(nodes, links, width, height, svgElement) {
        var nodePadding = 10;
        var transitions = [];
        var states = Core.createGraphStates(nodes, links, transitions);
        function spline(e) {
            var points = e.dagre.points.slice(0);
            var source = dagre.util.intersectRect(e.source.dagre, points.length > 0 ? points[0] : e.source.dagre);
            var target = dagre.util.intersectRect(e.target.dagre, points.length > 0 ? points[points.length - 1] : e.source.dagre);
            points.unshift(source);
            points.push(target);
            return d3.svg.line().x(function (d) {
                return d.x;
            }).y(function (d) {
                return d.y;
            }).interpolate("linear")(points);
        }

        // Translates all points in the edge using `dx` and `dy`.
        function translateEdge(e, dx, dy) {
            e.dagre.points.forEach(function (p) {
                p.x = Math.max(0, Math.min(svgBBox.width, p.x + dx));
                p.y = Math.max(0, Math.min(svgBBox.height, p.y + dy));
            });
        }

        // Now start laying things out
        var svg = svgElement ? d3.select(svgElement) : d3.select("svg");

        // lets remove all the old g elements
        if (svgElement) {
            $(svgElement).children("g").remove();
        }
        $(svg).children("g").remove();

        var svgGroup = svg.append("g").attr("transform", "translate(5, 5)");

        // `nodes` is center positioned for easy layout later
        var nodes = svgGroup.selectAll("g .node").data(states).enter().append("g").attr("class", "node").attr("data-cid", function (d) {
            return d.cid;
        }).attr("id", function (d) {
            return "node-" + d.label;
        });

        // lets add a tooltip
        nodes.append("title").text(function (d) {
            return d.tooltip || "";
        });

        var edges = svgGroup.selectAll("path .edge").data(transitions).enter().append("path").attr("class", "edge").attr("marker-end", "url(#arrowhead)");

        // Append rectangles to the nodes. We do this before laying out the text
        // because we want the text above the rectangle.
        var rects = nodes.append("rect").attr("rx", "5").attr("ry", "5").attr("filter", "url(#drop-shadow)");

        var images = nodes.append("image").attr("xlink:href", function (d) {
            return d.imageUrl;
        }).attr("x", -12).attr("y", -20).attr("height", 24).attr("width", 24);

        var counters = nodes.append("text").attr("text-anchor", "end").attr("class", "counter").attr("x", 0).attr("dy", 0).text(_counterFunction);

        // Append text
        var labels = nodes.append("text").attr("text-anchor", "middle").attr("x", 0);

        labels.append("tspan").attr("x", 0).attr("dy", 28).text(function (d) {
            return d.label;
        });

        var labelPadding = 12;
        var minLabelwidth = 80;

        labels.each(function (d) {
            var bbox = this.getBBox();
            d.bbox = bbox;
            if (bbox.width < minLabelwidth) {
                bbox.width = minLabelwidth;
            }
            d.width = bbox.width + 2 * nodePadding;
            d.height = bbox.height + 2 * nodePadding + labelPadding;
        });

        rects.attr("x", function (d) {
            return -(d.bbox.width / 2 + nodePadding);
        }).attr("y", function (d) {
            return -(d.bbox.height / 2 + nodePadding + (labelPadding / 2));
        }).attr("width", function (d) {
            return d.width;
        }).attr("height", function (d) {
            return d.height;
        });

        images.attr("x", function (d) {
            return -(d.bbox.width) / 2;
        });

        labels.attr("x", function (d) {
            return -d.bbox.width / 2;
        }).attr("y", function (d) {
            return -d.bbox.height / 2;
        });

        counters.attr("x", function (d) {
            var w = d.bbox.width;
            return w / 2;
        });

        // Create the layout and get the graph
        dagre.layout().nodeSep(50).edgeSep(10).rankSep(50).nodes(states).edges(transitions).debugLevel(1).run();

        nodes.attr("transform", function (d) {
            return 'translate(' + d.dagre.x + ',' + d.dagre.y + ')';
        });

        edges.attr('id', function (e) {
            return e.dagre.id;
        }).attr("d", function (e) {
            return spline(e);
        });

        // Resize the SVG element
        var svgNode = svg.node();
        if (svgNode) {
            var svgBBox = svgNode.getBBox();
            if (svgBBox) {
                svg.attr("width", svgBBox.width + 10);
                svg.attr("height", svgBBox.height + 10);
            }
        }

        // Drag handlers
        var nodeDrag = d3.behavior.drag().origin(function (d) {
            return d.pos ? { x: d.pos.x, y: d.pos.y } : { x: d.dagre.x, y: d.dagre.y };
        }).on('drag', function (d, i) {
            var prevX = d.dagre.x, prevY = d.dagre.y;

            // The node must be inside the SVG area
            d.dagre.x = Math.max(d.width / 2, Math.min(svgBBox.width - d.width / 2, d3.event.x));
            d.dagre.y = Math.max(d.height / 2, Math.min(svgBBox.height - d.height / 2, d3.event.y));
            d3.select(this).attr('transform', 'translate(' + d.dagre.x + ',' + d.dagre.y + ')');

            var dx = d.dagre.x - prevX, dy = d.dagre.y - prevY;

            // Edges position (inside SVG area)
            d.edges.forEach(function (e) {
                translateEdge(e, dx, dy);
                d3.select('#' + e.dagre.id).attr('d', spline(e));
            });
        });

        var edgeDrag = d3.behavior.drag().on('drag', function (d, i) {
            translateEdge(d, d3.event.dx, d3.event.dy);
            d3.select(this).attr('d', spline(d));
        });

        nodes.call(nodeDrag);
        edges.call(edgeDrag);

        return states;
    }
    Core.dagreLayoutGraph = dagreLayoutGraph;

    // TODO Export as a service
    function dagreUpdateGraphData(data) {
        var svg = d3.select("svg");
        svg.selectAll("text.counter").text(_counterFunction);

        // add tooltip
        svg.selectAll("g .node title").text(function (d) {
            return d.tooltip || "";
        });
        /*
        TODO can we reuse twitter bootstrap on an svg title?
        .each(function (d) {
        $(d).tooltip({
        'placement': "bottom"
        });
        });
        
        */
    }
    Core.dagreUpdateGraphData = dagreUpdateGraphData;

    function _counterFunction(d) {
        return d.counter || "";
    }
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    // TODO would be nice to use a directive instead; but couldn't get it working :(
    function EditorController($scope, workspace) {
        // TODO Do we have to deal with Async data loading?
        var options = {
            readOnly: true,
            mode: {
                name: CodeEditor.detectTextFormat($scope.row.Text)
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);
    }
    Core.EditorController = EditorController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function PreferencesController($scope, $location, jolokia, workspace, localStorage, userDetails, jolokiaUrl, branding) {
        var log = Logger.get("Preference");

        /**
        * Parsers the given value as JSON if it is define
        */
        function parsePreferencesJson(value, key) {
            var answer = null;
            if (angular.isDefined(value)) {
                answer = Core.parseJsonText(value, "localStorage for " + key);
            }
            return answer;
        }

        $scope.branding = branding;

        if (!angular.isDefined(localStorage['logLevel'])) {
            localStorage['logLevel'] = '{"value": 2, "name": "INFO"}';
        }

        $scope.localStorage = localStorage;

        Core.bindModelToSearchParam($scope, $location, "pref", "pref", "core-preference");

        $scope.logBuffer = 0;
        if ('logBuffer' in localStorage) {
            $scope.logBuffer = parseInt(localStorage['logBuffer']);
        }

        $scope.$watch('localStorage.logLevel', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                var level = JSON.parse(newValue);
                Logger.setLevel(level);
            }
        });

        $scope.$watch('logBuffer', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                localStorage['logBuffer'] = newValue;
                window['LogBuffer'] = newValue;
            }
        });

        $scope.updateRate = localStorage['updateRate'];
        $scope.url = localStorage['url'];
        $scope.autoRefresh = localStorage['autoRefresh'] === "true";
        $scope.showWelcomePage = localStorage['showWelcomePage'] === "true";

        $scope.hosts = [];
        $scope.newHost = {};

        $scope.addRegexDialog = new Core.Dialog();
        $scope.forms = {};

        $scope.perspectiveId;
        $scope.perspectives = [];

        // used by add dialog in preference.html
        $scope.hostSchema = {
            properties: {
                'name': {
                    description: 'Indicator name',
                    type: 'string',
                    required: true
                },
                'regex': {
                    description: 'Indicator regex',
                    type: 'string',
                    required: true
                }
            }
        };

        $scope.delete = function (index) {
            $scope.hosts.removeAt(index);
        };

        $scope.moveUp = function (index) {
            var tmp = $scope.hosts[index];
            $scope.hosts[index] = $scope.hosts[index - 1];
            $scope.hosts[index - 1] = tmp;
        };

        $scope.moveDown = function (index) {
            var tmp = $scope.hosts[index];
            $scope.hosts[index] = $scope.hosts[index + 1];
            $scope.hosts[index + 1] = tmp;
        };

        $scope.onOk = function (json, form) {
            $scope.addRegexDialog.close();
            $scope.newHost['color'] = UI.colors.sample();
            if (!angular.isArray($scope.hosts)) {
                $scope.hosts = [Object.clone($scope.newHost)];
            } else {
                $scope.hosts.push(Object.clone($scope.newHost));
            }

            $scope.newHost = {};
            Core.$apply($scope);
        };

        $scope.plugins = [];
        $scope.pluginDirty = false;

        $scope.pluginMoveUp = function (index) {
            $scope.pluginDirty = true;
            var tmp = $scope.plugins[index];
            $scope.plugins[index] = $scope.plugins[index - 1];
            $scope.plugins[index - 1] = tmp;
        };

        $scope.pluginMoveDown = function (index) {
            $scope.pluginDirty = true;
            var tmp = $scope.plugins[index];
            $scope.plugins[index] = $scope.plugins[index + 1];
            $scope.plugins[index + 1] = tmp;
        };

        $scope.pluginDisable = function (index) {
            $scope.pluginDirty = true;
            $scope.plugins[index].enabled = false;
            $scope.plugins[index].isDefault = false;
        };

        $scope.pluginEnable = function (index) {
            $scope.pluginDirty = true;
            $scope.plugins[index].enabled = true;
        };

        $scope.pluginDefault = function (index) {
            $scope.pluginDirty = true;
            $scope.plugins.forEach(function (p) {
                p.isDefault = false;
            });
            $scope.plugins[index].isDefault = true;
        };

        $scope.pluginApply = function () {
            $scope.pluginDirty = false;

            // set index before saving
            $scope.plugins.forEach(function (p, idx) {
                p.index = idx;
            });

            var json = angular.toJson($scope.plugins);
            if (json) {
                log.debug("Saving plugin settings for perspective " + $scope.perspectiveId + " -> " + json);
                var id = "plugins-" + $scope.perspectiveId;
                localStorage[id] = json;
            }

            // force UI to update by reloading the page which works
            setTimeout(function () {
                window.location.reload();
            }, 10);
        };

        $scope.$watch('hosts', function (oldValue, newValue) {
            if (!Object.equal(oldValue, newValue)) {
                if (angular.isDefined($scope.hosts)) {
                    localStorage['regexs'] = angular.toJson($scope.hosts);
                } else {
                    delete localStorage['regexs'];
                }
            } else {
                $scope.hosts = parsePreferencesJson(localStorage['regexs'], "hosts") || {};
            }
        }, true);

        var defaults = {
            showWelcomePage: true,
            logCacheSize: 1000,
            logSortAsc: true,
            logAutoScroll: true,
            fabricAlwaysPrompt: false,
            fabricEnableMaps: true,
            camelIgnoreIdForLabel: false,
            camelMaximumLabelWidth: Camel.defaultMaximumLabelWidth,
            camelMaximumTraceOrDebugBodyLength: Camel.defaultCamelMaximumTraceOrDebugBodyLength
        };

        var converters = {
            showWelcomePage: Core.parseBooleanValue,
            logCacheSize: parseInt,
            logSortAsc: Core.parseBooleanValue,
            logAutoScroll: Core.parseBooleanValue,
            fabricAlwaysPrompt: Core.parseBooleanValue,
            fabricEnableMaps: Core.parseBooleanValue,
            camelIgnoreIdForLabel: Core.parseBooleanValue,
            camelMaximumLabelWidth: parseInt,
            camelMaximumTraceOrDebugBodyLength: parseInt
        };

        $scope.$watch('updateRate', function () {
            localStorage['updateRate'] = $scope.updateRate;
            $scope.$emit('UpdateRate', $scope.updateRate);
        });

        $scope.$watch('autoRefresh', function (newValue, oldValue) {
            if (newValue === oldValue) {
                return;
            }
            localStorage['autoRefresh'] = $scope.autoRefresh;
        });

        $scope.$watch('showWelcomePage', function (newValue, oldValue) {
            if (newValue === oldValue) {
                return;
            }
            localStorage['showWelcomePage'] = $scope.showWelcomePage;
        });

        var names = [
            "showWelcomePage", "gitUserName", "gitUserEmail", "activemqUserName", "activemqPassword",
            "logCacheSize", "logSortAsc", "logAutoScroll", "fabricAlwaysPrompt", "fabricEnableMaps", "camelIgnoreIdForLabel", "camelMaximumLabelWidth",
            "camelMaximumTraceOrDebugBodyLength"];

        angular.forEach(names, function (name) {
            if (angular.isDefined(localStorage[name])) {
                $scope[name] = localStorage[name];
                var converter = converters[name];
                if (converter) {
                    $scope[name] = converter($scope[name]);
                }
            } else {
                $scope[name] = defaults[name] || "";
            }

            $scope.$watch(name, function () {
                var value = $scope[name];
                if (angular.isDefined(value)) {
                    var actualValue = value;
                    var converter = converters[name];
                    if (converter) {
                        actualValue = converter(value);
                    }
                    localStorage[name] = actualValue;
                }
            });
        });

        $scope.doReset = function () {
            log.info("Resetting");

            var doReset = function () {
                localStorage.clear();
                setTimeout(function () {
                    window.location.reload();
                }, 10);
            };
            if (Core.isBlank(userDetails.username) && Core.isBlank(userDetails.password)) {
                doReset();
            } else {
                Core.logout(jolokiaUrl, userDetails, localStorage, $scope, doReset);
            }
        };

        $scope.$watch('perspectiveId', function (newValue, oldValue) {
            if (newValue === oldValue) {
                return;
            }

            var perspective = Perspective.getPerspectiveById(newValue);
            if (perspective) {
                updateToPerspective(perspective);
                Core.$apply($scope);
            }
        });

        function updateToPerspective(perspective) {
            var plugins = Core.configuredPluginsForPerspectiveId(perspective.id, workspace, jolokia, localStorage);
            $scope.plugins = plugins;
            $scope.perspectiveId = perspective.id;
            log.debug("Updated to perspective " + $scope.perspectiveId + " with " + plugins.length + " plugins");
        }

        // initialize the controller, and pick the 1st perspective
        $scope.perspectives = Perspective.getPerspectives($location, workspace, jolokia, localStorage);
        log.debug("There are " + $scope.perspectives.length + " perspectives");

        // pick the current selected perspective
        var selectPerspective;
        var perspectiveId = Perspective.currentPerspectiveId($location, workspace, jolokia, localStorage);
        if (perspectiveId) {
            selectPerspective = $scope.perspectives.find(function (p) {
                return p.id === perspectiveId;
            });
        }
        if (!selectPerspective) {
            // just pick the 1st then
            selectPerspective = $scope.perspectives[0];
        }

        updateToPerspective(selectPerspective);

        // and force update the ui
        Core.$apply($scope);
    }
    Core.PreferencesController = PreferencesController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * Controller that's attached to hawtio's drop-down console, mainly handles the
    * clipboard icon at the bottom-right of the console.
    *
    * @method ConsoleController
    * @for Core
    * @static
    * @param {*} $scope
    * @param {*} $element
    * @param {*} $templateCache
    */
    function ConsoleController($scope, $element, $templateCache) {
        $scope.setHandler = function (clip) {
            clip.addEventListener('mouseDown', function (client, args) {
                // this is apparently a global event handler for zero clipboard
                // so you have to make sure you're handling the right click event
                var icon = $element.find('.icon-copy');
                var icon2 = $element.find('.icon-trash');
                if (this !== icon.get(0) && this !== icon2.get(0)) {
                    return;
                }

                if (this == icon.get(0)) {
                    copyToClipboard();
                } else {
                    clearLogs();
                    notification('info', "Cleared logging console");
                }
                Core.$apply($scope);
            });

            function copyToClipboard() {
                var text = $templateCache.get("logClipboardTemplate").lines();
                text.removeAt(0);
                text.removeAt(text.length - 1);
                $element.find('#log-panel-statements').children().each(function (index, child) {
                    text.push('  <li>' + child.innerHTML + '</li>');
                });
                text.push('</ul>');
                clip.setText(text.join('\n'));
            }

            function clearLogs() {
                $element.find('#log-panel-statements').children().remove();
            }
        };
    }
    Core.ConsoleController = ConsoleController;

    /**
    * Outermost controller attached to almost the root of the document, handles
    * logging in and logging out, the PID/container indicator at the bottom right
    * of the window and the document title
    *
    * @method AppController
    * @for Core
    * @static
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    * @param {*} jolokiaStatus
    * @param {*} $document
    * @param {Core.PageTitle} pageTitle
    * @param {*} localStorage
    * @param {*} userDetails
    * @param {*} lastLocation
    * @param {*} jolokiaUrl
    * @param {*} branding
    */
    function AppController($scope, $location, workspace, jolokia, jolokiaStatus, $document, pageTitle, localStorage, userDetails, lastLocation, jolokiaUrl, branding) {
        if (!userDetails) {
            userDetails = {};
        }
        if (userDetails.username === null) {
            $location.url(defaultPage());
        }

        $scope.collapse = '';
        $scope.match = null;
        $scope.pageTitle = [];
        $scope.userDetails = userDetails;

        $scope.confirmLogout = false;
        $scope.connectionFailed = false;
        $scope.connectFailure = {};

        $scope.branding = branding;

        $scope.hasMBeans = function () {
            return workspace.hasMBeans();
        };

        $scope.$watch('jolokiaStatus.xhr', function () {
            var failure = jolokiaStatus.xhr;
            $scope.connectionFailed = failure ? true : false;
            $scope.connectFailure.summaryMessage = null;
            if ($scope.connectionFailed) {
                $scope.connectFailure.status = failure.status;
                $scope.connectFailure.statusText = failure.statusText;
                var text = failure.responseText;
                if (text) {
                    try  {
                        var html = $(text);
                        var markup = html.find("body");
                        if (markup && markup.length) {
                            html = markup;
                        }

                        // lets tone down the size of the headers
                        html.each(function (idx, e) {
                            var name = e.localName;
                            if (name && name.startsWith("h")) {
                                $(e).addClass("ajaxError");
                            }
                        });
                        var container = $("<div></div>");
                        container.append(html);
                        $scope.connectFailure.summaryMessage = container.html();
                        console.log("Found HTML: " + $scope.connectFailure.summaryMessage);
                    } catch (e) {
                        if (text.indexOf('<') < 0) {
                            // lets assume its not html markup if it doesn't have a tag in it
                            $scope.connectFailure.summaryMessage = "<p>" + text + "</p>";
                        }
                    }
                }
            }
        });

        $scope.confirmConnectionFailed = function () {
            // I guess we should close the window now?
            window.close();
        };

        $scope.setPageTitle = function () {
            $scope.pageTitle = pageTitle.getTitleArrayExcluding([branding.appName]);
            var tab = workspace.getActiveTab();
            if (tab && tab.content) {
                Core.setPageTitleWithTab($document, pageTitle, tab.content);
            } else {
                Core.setPageTitle($document, pageTitle);
            }
        };

        $scope.setRegexIndicator = function () {
            try  {
                var regexs = angular.fromJson(localStorage['regexs']);
                if (regexs) {
                    regexs.reverse().each(function (regex) {
                        var r = new RegExp(regex.regex, 'g');
                        if (r.test($location.absUrl())) {
                            $scope.match = {
                                name: regex.name,
                                color: regex.color
                            };
                        }
                    });
                }
            } catch (e) {
                // ignore
            }
        };

        $scope.loggedIn = function () {
            return userDetails.username !== null && userDetails.username !== 'public';
        };

        $scope.showLogout = function () {
            return $scope.loggedIn() && angular.isDefined(userDetails.loginDetails);
        };

        $scope.logout = function () {
            $scope.confirmLogout = true;
        };

        $scope.getUsername = function () {
            if (userDetails.username && !userDetails.username.isBlank()) {
                return userDetails.username;
            } else {
                return 'user';
            }
        };

        $scope.doLogout = function () {
            $scope.confirmLogout = false;
            Core.logout(jolokiaUrl, userDetails, localStorage, $scope);
        };

        $scope.$watch(function () {
            return localStorage['regexs'];
        }, $scope.setRegexIndicator);

        $scope.maybeRedirect = function () {
            if (userDetails.username === null) {
                var currentUrl = $location.url();
                if (!currentUrl.startsWith('/login')) {
                    lastLocation.url = currentUrl;
                    $location.url('/login');
                }
            } else {
                if ($location.url().startsWith('/login')) {
                    var url = defaultPage();
                    if (angular.isDefined(lastLocation.url)) {
                        url = lastLocation.url;
                    }
                    $location.url(url);
                }
            }
        };

        $scope.$watch('userDetails', function (newValue, oldValue) {
            $scope.maybeRedirect();
        }, true);

        $scope.$on('$routeChangeStart', function () {
            $scope.maybeRedirect();
        });

        $scope.$on('$routeChangeSuccess', function () {
            $scope.setPageTitle();
            $scope.setRegexIndicator();
        });

        $scope.fullScreen = function () {
            if ($location.url().startsWith("/login")) {
                return branding.fullscreenLogin;
            }
            var tab = $location.search()['tab'];
            if (tab) {
                return tab === "fullscreen";
            }
            return false;
        };

        $scope.login = function () {
            return $location.url().startsWith("/login");
        };

        function defaultPage() {
            return Perspective.defaultPage($location, workspace, jolokia, localStorage);
        }
    }
    Core.AppController = AppController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    

    /**
    * @class Workspace
    */
    var Workspace = (function () {
        function Workspace(jolokia, jolokiaStatus, jmxTreeLazyLoadRegistry, $location, $compile, $templateCache, localStorage, $rootScope, userDetails) {
            this.jolokia = jolokia;
            this.jolokiaStatus = jolokiaStatus;
            this.jmxTreeLazyLoadRegistry = jmxTreeLazyLoadRegistry;
            this.$location = $location;
            this.$compile = $compile;
            this.$templateCache = $templateCache;
            this.localStorage = localStorage;
            this.$rootScope = $rootScope;
            this.userDetails = userDetails;
            this.operationCounter = 0;
            this.tree = new Core.Folder('MBeans');
            this.treeResponse = {};
            this.mbeanTypesToDomain = {};
            this.mbeanServicesToDomain = {};
            this.attributeColumnDefs = {};
            this.treePostProcessors = [];
            this.topLevelTabs = [];
            this.subLevelTabs = [];
            this.keyToNodeMap = {};
            this.pluginRegisterHandle = null;
            this.pluginUpdateCounter = null;
            this.treeWatchRegisterHandle = null;
            this.treeWatcherCounter = null;
            this.treeElement = null;
            // set defaults
            if (!('autoRefresh' in localStorage)) {
                localStorage['autoRefresh'] = true;
            }
            if (!('updateRate' in localStorage)) {
                localStorage['updateRate'] = 5000;
            }
        }
        /**
        * Creates a shallow copy child workspace with its own selection and location
        * @method createChildWorkspace
        * @param {ng.ILocationService} location
        * @return {Workspace}
        */
        Workspace.prototype.createChildWorkspace = function (location) {
            var child = new Workspace(this.jolokia, this.jolokiaStatus, this.jmxTreeLazyLoadRegistry, this.$location, this.$compile, this.$templateCache, this.localStorage, this.$rootScope, this.userDetails);

            // lets copy across all the properties just in case
            angular.forEach(this, function (value, key) {
                return child[key] = value;
            });
            child.$location = location;
            return child;
        };

        Workspace.prototype.getLocalStorage = function (key) {
            return this.localStorage[key];
        };

        Workspace.prototype.setLocalStorage = function (key, value) {
            this.localStorage[key] = value;
        };

        Workspace.prototype.loadTree = function () {
            // Make an initial blocking call to ensure the JMX tree is populated while the
            // app is initializing...
            //var flags = {error: initialLoadError, ajaxError: initialLoadError, maxDepth: 2};
            var flags = { ignoreErrors: true, maxDepth: 2 };
            var data = this.jolokia.list(null, onSuccess(null, flags));

            if (data) {
                this.jolokiaStatus.xhr = null;
            }
            this.populateTree({
                value: data
            });
            // we now only reload the tree if the TreeWatcher mbean is present...
            // Core.register(this.jolokia, this, {type: 'list', maxDepth: 2}, onSuccess(angular.bind(this, this.populateTree), {maxDepth: 2}));
        };

        /**
        * Adds a post processor of the tree to swizzle the tree metadata after loading
        * such as correcting any typeName values or CSS styles by hand
        * @method addTreePostProcessor
        * @param {Function} processor
        */
        Workspace.prototype.addTreePostProcessor = function (processor) {
            this.treePostProcessors.push(processor);

            var tree = this.tree;
            if (tree) {
                // the tree is loaded already so lets process it now :)
                processor(tree);
            }
        };

        Workspace.prototype.maybeMonitorPlugins = function () {
            if (this.treeContainsDomainAndProperties("hawtio", { type: "Registry" })) {
                if (this.pluginRegisterHandle === null) {
                    this.pluginRegisterHandle = this.jolokia.register(angular.bind(this, this.maybeUpdatePlugins), {
                        type: "read",
                        mbean: "hawtio:type=Registry",
                        attribute: "UpdateCounter"
                    });
                }
            } else {
                if (this.pluginRegisterHandle !== null) {
                    this.jolokia.unregister(this.pluginRegisterHandle);
                    this.pluginRegisterHandle = null;
                    this.pluginUpdateCounter = null;
                }
            }

            // lets also listen to see if we have a JMX tree watcher
            if (this.treeContainsDomainAndProperties("hawtio", { type: "TreeWatcher" })) {
                if (this.treeWatchRegisterHandle === null) {
                    this.treeWatchRegisterHandle = this.jolokia.register(angular.bind(this, this.maybeReloadTree), {
                        type: "read",
                        mbean: "hawtio:type=TreeWatcher",
                        attribute: "Counter"
                    });
                } else {
                    // lets keep using the existing register handle
                    /* we don't need to unregister lets keep using it
                    if (this.treeWatchRegisterHandle !== null) {
                    this.jolokia.unregister(this.treeWatchRegisterHandle);
                    this.treeWatchRegisterHandle = null;
                    this.treeWatcherCounter = null;
                    }
                    */
                }
            }
        };

        Workspace.prototype.maybeUpdatePlugins = function (response) {
            if (this.pluginUpdateCounter === null) {
                this.pluginUpdateCounter = response.value;
                return;
            }
            if (this.pluginUpdateCounter !== response.value) {
                if (Core.parseBooleanValue(localStorage['autoRefresh'])) {
                    window.location.reload();
                }
            }
        };

        Workspace.prototype.maybeReloadTree = function (response) {
            var counter = response.value;
            if (this.treeWatcherCounter === null) {
                this.treeWatcherCounter = counter;
                return;
            }
            if (this.treeWatcherCounter !== counter) {
                this.treeWatcherCounter = counter;
                var workspace = this;
                function wrapInValue(response) {
                    var wrapper = {
                        value: response
                    };
                    workspace.populateTree(wrapper);
                }
                this.jolokia.list(null, onSuccess(wrapInValue, { ignoreErrors: true, maxDepth: 2 }));
            }
        };

        Workspace.prototype.folderGetOrElse = function (folder, value) {
            if (folder) {
                try  {
                    return folder.getOrElse(value);
                } catch (e) {
                    Core.log.warn("Failed to find value " + value + " on folder " + folder);
                }
            }
            return null;
        };

        Workspace.prototype.populateTree = function (response) {
            if (!Object.equal(this.treeResponse, response.value)) {
                this.treeResponse = response.value;
                Core.log.debug("JMX tree has been loaded!");

                var rootId = 'root';
                var separator = '-';
                this.mbeanTypesToDomain = {};
                this.mbeanServicesToDomain = {};
                this.keyToNodeMap = {};
                var tree = new Core.Folder('MBeans');
                tree.key = rootId;
                var domains = response.value;
                for (var domain in domains) {
                    var domainClass = escapeDots(domain);
                    var mbeans = domains[domain];
                    for (var path in mbeans) {
                        var entries = {};
                        var folder = this.folderGetOrElse(tree, domain);

                        //if (!folder) continue;
                        folder.domain = domain;
                        if (!folder.key) {
                            folder.key = rootId + separator + domain;
                        }
                        var folderNames = [domain];
                        folder.folderNames = folderNames;
                        folderNames = folderNames.clone();
                        var items = path.split(',');
                        var paths = [];
                        var typeName = null;
                        var serviceName = null;
                        items.forEach(function (item) {
                            var kv = item.split('=');
                            var key = kv[0];
                            var value = kv[1] || key;
                            entries[key] = value;
                            var moveToFront = false;
                            var lowerKey = key.toLowerCase();
                            if (lowerKey === "type") {
                                typeName = value;

                                // if the type name value already exists in the root node
                                // of the domain then lets move this property around too
                                if (folder.map[value]) {
                                    moveToFront = true;
                                }
                            }
                            if (lowerKey === "service") {
                                serviceName = value;
                            }
                            if (moveToFront) {
                                paths.splice(0, 0, value);
                            } else {
                                paths.push(value);
                            }
                        });

                        var configureFolder = function (folder, name) {
                            folder.domain = domain;
                            if (!folder.key) {
                                folder.key = rootId + separator + folderNames.join(separator);
                            }
                            this.keyToNodeMap[folder.key] = folder;
                            folder.folderNames = folderNames.clone();

                            //var classes = escapeDots(folder.key);
                            var classes = "";
                            var entries = folder.entries;
                            var entryKeys = Object.keys(entries).filter(function (n) {
                                return n.toLowerCase().indexOf("type") >= 0;
                            });
                            if (entryKeys.length) {
                                angular.forEach(entryKeys, function (entryKey) {
                                    var entryValue = entries[entryKey];
                                    if (!folder.ancestorHasEntry(entryKey, entryValue)) {
                                        classes += " " + domainClass + separator + entryValue;
                                    }
                                });
                            } else {
                                var kindName = folderNames.last();

                                /*if (folder.parent && folder.parent.title === typeName) {
                                kindName = typeName;
                                } else */
                                if (kindName === name) {
                                    kindName += "-folder";
                                }
                                if (kindName) {
                                    classes += " " + domainClass + separator + kindName;
                                }
                            }
                            folder.addClass = escapeTreeCssStyles(classes);
                            return folder;
                        };

                        var lastPath = paths.pop();
                        var ws = this;
                        paths.forEach(function (value) {
                            folder = ws.folderGetOrElse(folder, value);
                            if (folder) {
                                folderNames.push(value);
                                angular.bind(ws, configureFolder, folder, value)();
                            }
                        });
                        var key = rootId + separator + folderNames.join(separator) + separator + lastPath;
                        var objectName = domain + ":" + path;

                        if (folder) {
                            folder = this.folderGetOrElse(folder, lastPath);
                            if (folder) {
                                // lets add the various data into the folder
                                folder.entries = entries;
                                folder.key = key;
                                angular.bind(this, configureFolder, folder, lastPath)();
                                folder.title = trimQuotes(lastPath);
                                folder.objectName = objectName;
                                folder.typeName = typeName;

                                var addFolderByDomain = function (owner, typeName) {
                                    var map = owner[typeName];
                                    if (!map) {
                                        map = {};
                                        owner[typeName] = map;
                                    }
                                    var value = map[domain];
                                    if (!value) {
                                        map[domain] = folder;
                                    } else {
                                        var array = null;
                                        if (angular.isArray(value)) {
                                            array = value;
                                        } else {
                                            array = [value];
                                            map[domain] = array;
                                        }
                                        array.push(folder);
                                    }
                                };

                                if (serviceName) {
                                    angular.bind(this, addFolderByDomain, this.mbeanServicesToDomain, serviceName)();
                                }
                                if (typeName) {
                                    angular.bind(this, addFolderByDomain, this.mbeanTypesToDomain, typeName)();
                                }
                            }
                        } else {
                            Core.log.info("No folder found for lastPath: " + lastPath);
                        }
                    }
                }

                tree.sortChildren(true);

                // now lets mark the nodes with no children as lazy loading...
                this.enableLazyLoading(tree);
                this.tree = tree;

                var processors = this.treePostProcessors;
                angular.forEach(processors, function (processor) {
                    return processor(tree);
                });

                this.maybeMonitorPlugins();

                var rootScope = this.$rootScope;
                if (rootScope) {
                    rootScope.$broadcast('jmxTreeUpdated');
                }
            }
        };

        Workspace.prototype.enableLazyLoading = function (folder) {
            var _this = this;
            var children = folder.children;
            if (children && children.length) {
                angular.forEach(children, function (child) {
                    _this.enableLazyLoading(child);
                });
            } else {
                // we have no children so enable lazy loading if we have a custom loader registered
                var lazyFunction = Jmx.findLazyLoadingFunction(this, folder);
                if (lazyFunction) {
                    folder.isLazy = true;
                }
            }
        };

        /**
        * Returns the hash query argument to append to URL links
        * @method hash
        * @return {String}
        */
        Workspace.prototype.hash = function () {
            var hash = this.$location.search();
            var params = Core.hashToString(hash);
            if (params) {
                return "?" + params;
            }
            return "";
        };

        /**
        * Returns the currently active tab
        * @method getActiveTab
        * @return {Boolean}
        */
        Workspace.prototype.getActiveTab = function () {
            var workspace = this;
            return this.topLevelTabs.find(function (tab) {
                if (!angular.isDefined(tab.isActive)) {
                    return workspace.isLinkActive(tab.href());
                } else {
                    return tab.isActive(workspace);
                }
            });
        };

        Workspace.prototype.getStrippedPathName = function () {
            var pathName = Core.trimLeading((this.$location.path() || '/'), "#");
            pathName = Core.trimLeading(pathName, "/");
            return pathName;
        };

        Workspace.prototype.linkContains = function () {
            var words = [];
            for (var _i = 0; _i < (arguments.length - 0); _i++) {
                words[_i] = arguments[_i + 0];
            }
            var pathName = this.getStrippedPathName();
            return words.all(function (word) {
                return pathName.has(word);
            });
        };

        /**
        * Returns true if the given link is active. The link can omit the leading # or / if necessary.
        * The query parameters of the URL are ignored in the comparison.
        * @method isLinkActive
        * @param {String} href
        * @return {Boolean} true if the given link is active
        */
        Workspace.prototype.isLinkActive = function (href) {
            // lets trim the leading slash
            var pathName = this.getStrippedPathName();

            var link = Core.trimLeading(href, "#");
            link = Core.trimLeading(link, "/");

            // strip any query arguments
            var idx = link.indexOf('?');
            if (idx >= 0) {
                link = link.substring(0, idx);
            }
            if (!pathName.length) {
                return link === pathName;
            } else {
                return pathName.startsWith(link);
            }
        };

        /**
        * Returns true if the given link is active. The link can omit the leading # or / if necessary.
        * The query parameters of the URL are ignored in the comparison.
        * @method isLinkActive
        * @param {String} href
        * @return {Boolean} true if the given link is active
        */
        Workspace.prototype.isLinkPrefixActive = function (href) {
            // lets trim the leading slash
            var pathName = this.getStrippedPathName();

            var link = Core.trimLeading(href, "#");
            link = Core.trimLeading(link, "/");

            // strip any query arguments
            var idx = link.indexOf('?');
            if (idx >= 0) {
                link = link.substring(0, idx);
            }
            return pathName.startsWith(link);
        };

        /**
        * Returns true if the tab query parameter is active or the URL starts with the given path
        * @method isTopTabActive
        * @param {String} path
        * @return {Boolean}
        */
        Workspace.prototype.isTopTabActive = function (path) {
            var tab = this.$location.search()['tab'];
            if (angular.isString(tab)) {
                return tab.startsWith(path);
            }
            return this.isLinkActive(path);
        };

        /**
        * Returns the selected mbean name if there is one
        * @method getSelectedMBeanName
        * @return {String}
        */
        Workspace.prototype.getSelectedMBeanName = function () {
            var selection = this.selection;
            if (selection) {
                return selection.objectName;
            }
            return null;
        };

        /**
        * Returns true if the path is valid for the current selection
        * @method validSelection
        * @param {String} uri
        * @return {Boolean}
        */
        Workspace.prototype.validSelection = function (uri) {
            var workspace = this;
            var filter = function (t) {
                var fn = t.href;
                if (fn) {
                    var href = fn();
                    if (href) {
                        if (href.startsWith("#/")) {
                            href = href.substring(2);
                        }
                        return href === uri;
                    }
                }
                return false;
            };
            var tab = this.subLevelTabs.find(filter);
            if (!tab) {
                tab = this.topLevelTabs.find(filter);
            }
            if (tab) {
                //console.log("Found tab " + JSON.stringify(tab));
                var validFn = tab['isValid'];
                return !angular.isDefined(validFn) || validFn(workspace);
            } else {
                Core.log.info("Could not find tab for " + uri);
                return false;
            }
            /*
            var value = this.uriValidations[uri];
            if (value) {
            if (angular.isFunction(value)) {
            return value();
            }
            }
            return true;
            */
        };

        /**
        * In cases where we have just deleted something we typically want to change
        * the selection to the parent node
        * @method removeAndSelectParentNode
        */
        Workspace.prototype.removeAndSelectParentNode = function () {
            var selection = this.selection;
            if (selection) {
                var parent = selection.parent;
                if (parent) {
                    // lets remove the selection from the parent so we don't do any more JMX attribute queries on the children
                    // or include it in table views etc
                    // would be nice to eagerly remove the tree node too?
                    var idx = parent.children.indexOf(selection);
                    if (idx < 0) {
                        idx = parent.children.findIndex({ key: selection.key });
                    }
                    if (idx >= 0) {
                        parent.children.splice(idx, 1);
                    }
                    this.updateSelectionNode(parent);
                }
            }
        };

        Workspace.prototype.selectParentNode = function () {
            var selection = this.selection;
            if (selection) {
                var parent = selection.parent;
                if (parent) {
                    this.updateSelectionNode(parent);
                }
            }
        };

        /**
        * Returns the view configuration key for the kind of selection
        * for example based on the domain and the node type
        * @method selectionViewConfigKey
        * @return {String}
        */
        Workspace.prototype.selectionViewConfigKey = function () {
            return this.selectionConfigKey("view/");
        };

        /**
        * Returns a configuration key for a node which is usually of the form
        * domain/typeName or for folders with no type, domain/name/folder
        * @method selectionConfigKey
        * @param {String} prefix
        * @return {String}
        */
        Workspace.prototype.selectionConfigKey = function (prefix) {
            if (typeof prefix === "undefined") { prefix = ""; }
            var key = null;
            var selection = this.selection;
            if (selection) {
                // lets make a unique string for the kind of select
                key = prefix + selection.domain;
                var typeName = selection.typeName;
                if (!typeName) {
                    typeName = selection.title;
                }
                key += "/" + typeName;
                if (selection.isFolder()) {
                    key += "/folder";
                }
            }
            return key;
        };

        Workspace.prototype.moveIfViewInvalid = function () {
            var workspace = this;
            var uri = Core.trimLeading(this.$location.path(), "/");
            if (this.selection) {
                var key = this.selectionViewConfigKey();
                if (this.validSelection(uri)) {
                    // lets remember the previous selection
                    this.setLocalStorage(key, uri);
                    return false;
                } else {
                    Core.log.info("the uri '" + uri + "' is not valid for this selection");

                    // lets look up the previous preferred value for this type
                    var defaultPath = this.getLocalStorage(key);
                    if (!defaultPath || !this.validSelection(defaultPath)) {
                        // lets find the first path we can find which is valid
                        defaultPath = null;
                        angular.forEach(this.subLevelTabs, function (tab) {
                            var fn = tab.isValid;
                            if (!defaultPath && tab.href && angular.isDefined(fn) && fn(workspace)) {
                                defaultPath = tab.href();
                            }
                        });
                    }
                    if (!defaultPath) {
                        defaultPath = "#/jmx/help";
                    }
                    Core.log.info("moving the URL to be " + defaultPath);
                    if (defaultPath.startsWith("#")) {
                        defaultPath = defaultPath.substring(1);
                    }
                    this.$location.path(defaultPath);
                    return true;
                }
            } else {
                return false;
            }
        };

        Workspace.prototype.updateSelectionNode = function (node) {
            var originalSelection = this.selection;
            this.selection = node;
            var key = null;
            if (node) {
                key = node['key'];
            }
            var $location = this.$location;
            var q = $location.search();
            if (key) {
                q['nid'] = key;
            }
            $location.search(q);

            // if we have updated the selection (rather than just loaded a page)
            // lets use the previous preferred view - otherwise we may be loading
            // a page from a bookmark so lets not change the view :)
            if (originalSelection) {
                key = this.selectionViewConfigKey();
                if (key) {
                    var defaultPath = this.getLocalStorage(key);
                    if (defaultPath) {
                        this.$location.path(defaultPath);
                    }
                }
            }
        };

        /**
        * Redraws the tree widget
        * @method redrawTree
        */
        Workspace.prototype.redrawTree = function () {
            var treeElement = this.treeElement;
            if (treeElement) {
                treeElement.dynatree("getTree").reload();
            }
        };

        /**
        * Expand / collapse the current active node
        * @method expandSelection
        * @param {Boolean} flag
        */
        Workspace.prototype.expandSelection = function (flag) {
            var treeElement = this.treeElement;
            if (treeElement) {
                var node = treeElement.dynatree("getActiveNode");
                if (node) {
                    node.expand(flag);
                }
            }
        };

        Workspace.prototype.matchesProperties = function (entries, properties) {
            if (!entries)
                return false;
            for (var key in properties) {
                var value = properties[key];
                if (!value || entries[key] !== value) {
                    return false;
                }
            }
            return true;
        };

        Workspace.prototype.treeContainsDomainAndProperties = function (domainName, properties) {
            if (typeof properties === "undefined") { properties = null; }
            var _this = this;
            var workspace = this;
            var tree = workspace.tree;
            if (tree) {
                var folder = tree.get(domainName);
                if (folder) {
                    if (properties) {
                        var children = folder.children || [];
                        var checkProperties = function (node) {
                            if (!_this.matchesProperties(node.entries, properties)) {
                                if (node.domain === domainName && node.children && node.children.length > 0) {
                                    return node.children.some(checkProperties);
                                } else {
                                    return false;
                                }
                            } else {
                                return true;
                            }
                        };
                        return children.some(checkProperties);
                    }
                    return true;
                } else {
                    // console.log("no hasMBean for " + objectName + " in tree " + tree);
                }
            } else {
                // console.log("workspace has no tree! returning false for hasMBean " + objectName);
            }
            return false;
        };

        Workspace.prototype.matches = function (folder, properties, propertiesCount) {
            if (folder) {
                var entries = folder.entries;
                if (properties) {
                    if (!entries)
                        return false;
                    for (var key in properties) {
                        var value = properties[key];
                        if (!value || entries[key] !== value) {
                            return false;
                        }
                    }
                }
                if (propertiesCount) {
                    return entries && Object.keys(entries).length === propertiesCount;
                }
                return true;
            }
            return false;
        };

        // only display stuff if we have an mbean with the given properties
        Workspace.prototype.hasDomainAndProperties = function (domainName, properties, propertiesCount) {
            if (typeof properties === "undefined") { properties = null; }
            if (typeof propertiesCount === "undefined") { propertiesCount = null; }
            var node = this.selection;
            if (node) {
                return this.matches(node, properties, propertiesCount) && node.domain === domainName;
            }
            return false;
        };

        // only display stuff if we have an mbean with the given properties
        Workspace.prototype.findMBeanWithProperties = function (domainName, properties, propertiesCount) {
            if (typeof properties === "undefined") { properties = null; }
            if (typeof propertiesCount === "undefined") { propertiesCount = null; }
            var tree = this.tree;
            if (tree) {
                return this.findChildMBeanWithProperties(tree.get(domainName), properties, propertiesCount);
            }
            return null;
        };

        Workspace.prototype.findChildMBeanWithProperties = function (folder, properties, propertiesCount) {
            if (typeof properties === "undefined") { properties = null; }
            if (typeof propertiesCount === "undefined") { propertiesCount = null; }
            var _this = this;
            var workspace = this;
            if (folder) {
                var children = folder.children;
                if (children) {
                    var answer = children.find(function (node) {
                        return _this.matches(node, properties, propertiesCount);
                    });
                    if (answer) {
                        return answer;
                    }
                    return children.map(function (node) {
                        return workspace.findChildMBeanWithProperties(node, properties, propertiesCount);
                    }).find(function (node) {
                        return node;
                    });
                }
            }
            return null;
        };

        Workspace.prototype.selectionHasDomainAndLastFolderName = function (objectName, lastName) {
            var lastNameLower = (lastName || "").toLowerCase();
            function isName(name) {
                return (name || "").toLowerCase() === lastNameLower;
            }
            var node = this.selection;
            if (node) {
                if (objectName === node.domain) {
                    var folders = node.folderNames;
                    if (folders) {
                        var last = folders.last();
                        return (isName(last) || isName(node.title)) && node.isFolder() && !node.objectName;
                    }
                }
            }
            return false;
        };

        Workspace.prototype.selectionHasDomain = function (domainName) {
            var node = this.selection;
            if (node) {
                return domainName === node.domain;
            }
            return false;
        };

        Workspace.prototype.selectionHasDomainAndType = function (objectName, typeName) {
            var node = this.selection;
            if (node) {
                return objectName === node.domain && typeName === node.typeName;
            }
            return false;
        };

        /**
        * Returns true if this workspace has any mbeans at all
        */
        Workspace.prototype.hasMBeans = function () {
            var answer = false;
            var tree = this.tree;
            if (tree) {
                var children = tree.children;
                if (angular.isArray(children) && children.length > 0) {
                    answer = true;
                }
            }
            return answer;
        };
        Workspace.prototype.hasFabricMBean = function () {
            return this.hasDomainAndProperties('io.fabric8', { type: 'Fabric' });
        };
        Workspace.prototype.isFabricFolder = function () {
            return this.hasDomainAndProperties('io.fabric8');
        };

        Workspace.prototype.isCamelContext = function () {
            return this.hasDomainAndProperties('org.apache.camel', { type: 'context' });
        };
        Workspace.prototype.isCamelFolder = function () {
            return this.hasDomainAndProperties('org.apache.camel');
        };
        Workspace.prototype.isEndpointsFolder = function () {
            return this.selectionHasDomainAndLastFolderName('org.apache.camel', 'endpoints');
        };
        Workspace.prototype.isEndpoint = function () {
            return this.hasDomainAndProperties('org.apache.camel', { type: 'endpoints' });
        };
        Workspace.prototype.isRoutesFolder = function () {
            return this.selectionHasDomainAndLastFolderName('org.apache.camel', 'routes');
        };
        Workspace.prototype.isRoute = function () {
            return this.hasDomainAndProperties('org.apache.camel', { type: 'routes' });
        };

        Workspace.prototype.isOsgiFolder = function () {
            return this.hasDomainAndProperties('osgi.core');
        };
        Workspace.prototype.isKarafFolder = function () {
            return this.hasDomainAndProperties('org.apache.karaf');
        };
        Workspace.prototype.isOsgiCompendiumFolder = function () {
            return this.hasDomainAndProperties('osgi.compendium');
        };
        return Workspace;
    })();
    Core.Workspace = Workspace;
})(Core || (Core = {}));

// TODO refactor other code to use Core.Workspace
var Workspace = (function (_super) {
    __extends(Workspace, _super);
    function Workspace() {
        _super.apply(this, arguments);
    }
    return Workspace;
})(Core.Workspace);
;
;
/**
* Module that contains several helper functions related to hawtio's code editor
*
* @module CodeEditor
* @main CodeEditor
*/
var CodeEditor;
(function (CodeEditor) {
    

    /**
    * @property GlobalCodeMirrorOptions
    * @for CodeEditor
    * @type CodeMirrorOptions
    */
    CodeEditor.GlobalCodeMirrorOptions = {
        theme: "default",
        tabSize: 4,
        lineNumbers: true,
        indentWithTabs: true,
        lineWrapping: true,
        autoCloseTags: true
    };

    /**
    * Controller used on the preferences page to configure the editor
    *
    * @method PreferencesController
    * @for CodeEditor
    * @static
    * @param $scope
    * @param workspace
    * @param localStorage
    * @param $templateCache
    */
    function PreferencesController($scope, workspace, localStorage, $templateCache) {
        $scope.exampleText = $templateCache.get("exampleText");
        $scope.codeMirrorEx = $templateCache.get("codeMirrorExTemplate");
        $scope.javascript = "javascript";

        $scope.preferences = CodeEditor.GlobalCodeMirrorOptions;

        // If any of the preferences change, make sure to save them automatically
        $scope.$watch("preferences", function (newValue, oldValue) {
            if (newValue !== oldValue) {
                // such a cheap and easy way to update the example view :-)
                $scope.codeMirrorEx += " ";
                localStorage['CodeMirrorOptions'] = angular.toJson(angular.extend(CodeEditor.GlobalCodeMirrorOptions, $scope.preferences));
            }
        }, true);
    }
    CodeEditor.PreferencesController = PreferencesController;

    /**
    * Tries to figure out what kind of text we're going to render in the editor, either
    * text, javascript or XML.
    *
    * @method detectTextFormat
    * @for CodeEditor
    * @static
    * @param value
    * @returns {string}
    */
    function detectTextFormat(value) {
        var answer = "text";
        if (value) {
            answer = "javascript";
            var trimmed = value.toString().trimLeft().trimRight();
            if (trimmed && trimmed.first() === '<' && trimmed.last() === '>') {
                answer = "xml";
            }
        }
        return answer;
    }
    CodeEditor.detectTextFormat = detectTextFormat;

    /**
    * Auto formats the CodeMirror editor content to pretty print
    *
    * @method autoFormatEditor
    * @for CodeEditor
    * @static
    * @param {CodeMirrorEditor} editor
    * @return {void}
    */
    function autoFormatEditor(editor) {
        if (editor) {
            var totalLines = editor.lineCount();

            //var totalChars = editor.getValue().length;
            var start = { line: 0, ch: 0 };
            var end = { line: totalLines - 1, ch: editor.getLine(totalLines - 1).length };
            editor.autoFormatRange(start, end);
            editor.setSelection(start, start);
        }
    }
    CodeEditor.autoFormatEditor = autoFormatEditor;

    /**
    * Used to configures the default editor settings (per Editor Instance)
    *
    * @method createEditorSettings
    * @for CodeEditor
    * @static
    * @param {Object} options
    * @return {Object}
    */
    function createEditorSettings(options) {
        if (typeof options === "undefined") { options = {}; }
        options.extraKeys = options.extraKeys || {};

        // Handle Mode
        (function (mode) {
            mode = mode || { name: "text" };

            if (typeof mode !== "object") {
                mode = { name: mode };
            }

            var modeName = mode.name;
            if (modeName === "javascript") {
                angular.extend(mode, {
                    "json": true
                });
            }
        })(options.mode);

        // Handle Code folding folding
        (function (options) {
            var javascriptFolding = CodeMirror.newFoldFunction(CodeMirror.braceRangeFinder);
            var xmlFolding = CodeMirror.newFoldFunction(CodeMirror.tagRangeFinder);

            // Mode logic inside foldFunction to allow for dynamic changing of the mode.
            // So don't have to listen to the options model and deal with re-attaching events etc...
            var foldFunction = function (codeMirror, line) {
                var mode = codeMirror.getOption("mode");
                var modeName = mode["name"];
                if (!mode || !modeName)
                    return;
                if (modeName === 'javascript') {
                    javascriptFolding(codeMirror, line);
                } else if (modeName === "xml" || modeName.startsWith("html")) {
                    xmlFolding(codeMirror, line);
                }
                ;
            };

            options.onGutterClick = foldFunction;
            options.extraKeys = angular.extend(options.extraKeys, {
                "Ctrl-Q": function (codeMirror) {
                    foldFunction(codeMirror, codeMirror.getCursor().line);
                }
            });
        })(options);

        var readOnly = options.readOnly;
        if (!readOnly) {
            /*
            options.extraKeys = angular.extend(options.extraKeys, {
            "'>'": function (codeMirror) {
            codeMirror.closeTag(codeMirror, '>');
            },
            "'/'": function (codeMirror) {
            codeMirror.closeTag(codeMirror, '/');
            }
            });
            */
            options.matchBrackets = true;
        }

        // Merge the global config in to this instance of CodeMirror
        angular.extend(options, CodeEditor.GlobalCodeMirrorOptions);

        return options;
    }
    CodeEditor.createEditorSettings = createEditorSettings;
})(CodeEditor || (CodeEditor = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * Directive that's used to ensure an ng-grid expands it's height to fit the viewport height
    * @class GridStyle
    */
    var GridStyle = (function () {
        function GridStyle($window) {
            var _this = this;
            this.$window = $window;
            this.restrict = 'C';
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        GridStyle.prototype.doLink = function (scope, element, attrs) {
            var lastHeight = 0;

            var resizeFunc = angular.bind(this, function (scope) {
                var top = element.position().top;
                var windowHeight = $(this.$window).height();
                var height = windowHeight - top - 15;
                var heightStr = height + 'px';

                element.css({
                    'min-height': heightStr,
                    'height': heightStr
                });

                if (lastHeight !== height) {
                    lastHeight = height;
                    element.trigger('resize');
                }
            });

            resizeFunc();

            scope.$watch(resizeFunc);

            $(this.$window).resize(function () {
                resizeFunc();
                Core.$apply(scope);
                return false;
            });
        };
        return GridStyle;
    })();
    Core.GridStyle = GridStyle;
})(Core || (Core = {}));
// TODO Get these functions and variables out of the global namespace
var logQueryMBean = 'org.fusesource.insight:type=LogQuery';

var _urlPrefix = null;

var numberTypeNames = {
    'byte': true,
    'short': true,
    'int': true,
    'long': true,
    'float': true,
    'double': true,
    'java.lang.Byte': true,
    'java.lang.Short': true,
    'java.lang.Integer': true,
    'java.lang.Long': true,
    'java.lang.Float': true,
    'java.lang.Double': true
};

/**
* Returns the number of lines in the given text
*
* @method lineCount
* @static
* @param {String} value
* @return {Number}
*
*/
function lineCount(value) {
    var rows = 0;
    if (value) {
        rows = 1;
        value.toString().each(/\n/, function () {
            return rows++;
        });
    }
    return rows;
}

function url(path) {
    if (path) {
        if (path.startsWith && path.startsWith("/")) {
            if (!_urlPrefix) {
                _urlPrefix = window.location.pathname || "";
                var idx = _urlPrefix.lastIndexOf("/");
                if (idx >= 0) {
                    _urlPrefix = _urlPrefix.substring(0, idx);
                }
            }
            if (_urlPrefix) {
                return _urlPrefix + path;
            }
        }
    }
    return path;
}

function humanizeValue(value) {
    if (value) {
        var text = value.toString();
        try  {
            text = text.underscore();
        } catch (e) {
            // ignore
        }
        try  {
            text = text.humanize();
        } catch (e) {
            // ignore
        }
        return trimQuotes(text);
    }
    return value;
}

function safeNull(value) {
    if (typeof value === 'boolean') {
        return value;
    } else if (typeof value === 'number') {
        // return numbers as-is
        return value;
    }
    if (value) {
        return value;
    } else {
        return "";
    }
}

function trimQuotes(text) {
    if (text) {
        while (text.endsWith('"') || text.endsWith("'")) {
            text = text.substring(0, text.length - 1);
        }
        while (text.startsWith('"') || text.startsWith("'")) {
            text = text.substring(1, text.length);
        }
    }
    return text;
}

/**
* Converts the given value to an array of query arguments.
*
* If the value is null an empty array is returned.
* If the value is a non empty string then the string is split by commas
*
* @method toSearchArgumentArray
* @static
* @param {*} value
* @return {String[]}
*
*/
function toSearchArgumentArray(value) {
    if (value) {
        if (angular.isArray(value))
            return value;
        if (angular.isString(value))
            return value.split(',');
    }
    return [];
}

function folderMatchesPatterns(node, patterns) {
    if (node) {
        var folderNames = node.folderNames;
        if (folderNames) {
            return patterns.any(function (ignorePaths) {
                for (var i = 0; i < ignorePaths.length; i++) {
                    var folderName = folderNames[i];
                    var ignorePath = ignorePaths[i];
                    if (!folderName)
                        return false;
                    var idx = ignorePath.indexOf(folderName);
                    if (idx < 0) {
                        return false;
                    }
                }
                return true;
            });
        }
    }
    return false;
}

function scopeStoreJolokiaHandle($scope, jolokia, jolokiaHandle) {
    // TODO do we even need to store the jolokiaHandle in the scope?
    if (jolokiaHandle) {
        $scope.$on('$destroy', function () {
            closeHandle($scope, jolokia);
        });
        $scope.jolokiaHandle = jolokiaHandle;
    }
}

function closeHandle($scope, jolokia) {
    var jolokiaHandle = $scope.jolokiaHandle;
    if (jolokiaHandle) {
        //console.log('Closing the handle ' + jolokiaHandle);
        jolokia.unregister(jolokiaHandle);
        $scope.jolokiaHandle = null;
    }
}

/**
* Pass in null for the success function to switch to sync mode
*
* @method onSuccess
* @static
* @param {Function} Success callback function
* @param {Object} Options object to pass on to Jolokia request
* @return {Object} initialized options object
*/
function onSuccess(fn, options) {
    if (typeof options === "undefined") { options = {}; }
    options['mimeType'] = 'application/json';
    if (angular.isDefined(fn)) {
        options['success'] = fn;
    }
    if (!options['method']) {
        options['method'] = "POST";
    }
    options['canonicalNaming'] = false;
    options['canonicalProperties'] = false;
    if (!options['error']) {
        options['error'] = function (response) {
            Core.defaultJolokiaErrorHandler(response, options);
        };
    }
    return options;
}

function supportsLocalStorage() {
    try  {
        return 'localStorage' in window && window['localStorage'] !== null;
    } catch (e) {
        return false;
    }
}

function isNumberTypeName(typeName) {
    if (typeName) {
        var text = typeName.toString().toLowerCase();
        var flag = numberTypeNames[text];
        return flag;
    }
    return false;
}

function encodeMBeanPath(mbean) {
    return mbean.replace(/\//g, '!/').replace(':', '/').escapeURL();
}

function escapeMBeanPath(mbean) {
    return mbean.replace(/\//g, '!/').replace(':', '/');
}

function encodeMBean(mbean) {
    return mbean.replace(/\//g, '!/').escapeURL();
}

function escapeDots(text) {
    return text.replace(/\./g, '-');
}

/**
* Escapes all dots and 'span' text in the css style names to avoid clashing with bootstrap stuff
*
* @method escapeTreeCssStyles
* @static
* @param {String} text
* @return {String}
*/
function escapeTreeCssStyles(text) {
    return escapeDots(text).replace(/span/g, 'sp-an');
}

function showLogPanel() {
    var log = $("#log-panel");
    var body = $('body');
    localStorage['showLog'] = 'true';
    log.css({ 'bottom': '50%' });
    body.css({
        'overflow-y': 'hidden'
    });
}

/**
* Displays an alert message which is typically the result of some asynchronous operation
*
* @method notification
* @static
* @param type which is usually "success" or "error" and matches css alert-* css styles
* @param message the text to display
*
*/
function notification(type, message, options) {
    if (typeof options === "undefined") { options = null; }
    var w = window;

    if (options === null) {
        options = {};
    }

    if (type === 'error' || type === 'warning') {
        if (!angular.isDefined(options.onclick)) {
            options.onclick = showLogPanel;
        }
    }

    w.toastr[type](message, '', options);
}

/**
* Clears all the pending notifications
* @method clearNotifications
* @static
*/
function clearNotifications() {
    var w = window;
    w.toastr.clear();
}

/**
* Returns the CSS class for a log level based on if its info, warn, error etc.
*
* @method logLevelClass
* @static
* @param {String} level
* @return {String}
*/
function logLevelClass(level) {
    if (level) {
        var first = level[0];
        if (first === 'w' || first === "W") {
            return "warning";
        } else if (first === 'e' || first === "E") {
            return "error";
        } else if (first === 'i' || first === "I") {
            return "info";
        } else if (first === 'd' || first === "D") {
            // we have no debug css style
            return "";
        }
    }
    return "";
}

if (!Object.keys) {
    Object.keys = function (obj) {
        var keys = [], k;
        for (k in obj) {
            if (Object.prototype.hasOwnProperty.call(obj, k)) {
                keys.push(k);
            }
        }
        return keys;
    };
}

/**
* @module Core
*/
var Core;
(function (Core) {
    function parseMBean(mbean) {
        var answer = {};
        var parts = mbean.split(":");
        if (parts.length > 1) {
            answer['domain'] = parts.first();
            parts = parts.exclude(parts.first());
            parts = parts.join(":");
            answer['attributes'] = {};
            var nameValues = parts.split(",");
            nameValues.forEach(function (str) {
                var nameValue = str.split('=');
                var name = nameValue.first().trim();
                nameValue = nameValue.exclude(nameValue.first());
                answer['attributes'][name] = nameValue.join('=').trim();
            });
        }
        return answer;
    }
    Core.parseMBean = parseMBean;

    /**
    * log out the current user
    * @for Core
    * @static
    * @method logout
    * @param {String} jolokiaUrl
    * @param {*} userDetails
    * @param {Object} localStorage
    * @param {Object} $scope
    * @param {Function} successCB
    * @param {Function} errorCB
    *
    */
    function logout(jolokiaUrl, userDetails, localStorage, $scope, successCB, errorCB) {
        if (typeof successCB === "undefined") { successCB = null; }
        if (typeof errorCB === "undefined") { errorCB = null; }
        if (jolokiaUrl) {
            var url = jolokiaUrl.replace("jolokia", "auth/logout/");

            $.ajax(url, {
                type: "POST",
                success: function () {
                    userDetails.username = null;
                    userDetails.password = null;
                    userDetails.loginDetails = null;
                    userDetails.rememberMe = false;
                    localStorage[jolokiaUrl] = angular.toJson(userDetails);
                    if (successCB && angular.isFunction(successCB)) {
                        successCB();
                    }
                    Core.$apply($scope);
                },
                error: function (xhr, textStatus, error) {
                    switch (xhr.status) {
                        case 401:
                            Core.log.error('Failed to log out, ', error);
                            break;
                        case 403:
                            Core.log.error('Failed to log out, ', error);
                            break;
                        default:
                            Core.log.error('Failed to log out, ', error);
                            break;
                    }
                    if (errorCB && angular.isFunction(errorCB)) {
                        errorCB();
                    }
                    Core.$apply($scope);
                }
            });
        }
    }
    Core.logout = logout;

    /**
    * Returns true if the string is either null or empty
    *
    * @method isBlank
    * @for Core
    * @static
    * @param {String} str
    * @return {Boolean}
    */
    function isBlank(str) {
        if (!str) {
            return true;
        }
        return str.isBlank();
    }
    Core.isBlank = isBlank;

    Core.log = Logger.get("Core");

    /**
    * Creates a link by appending the current $location.search() hash to the given href link,
    * removing any required parameters from the link
    * @method createHref
    * @for Core
    * @static
    * @param {ng.ILocationService} $location
    * @param {String} href the link to have any $location.search() hash parameters appended
    * @param {Array} removeParams any parameters to be removed from the $location.search()
    * @return {Object} the link with any $location.search() parameters added
    */
    function createHref($location, href, removeParams) {
        if (typeof removeParams === "undefined") { removeParams = null; }
        var hashMap = angular.copy($location.search());

        // lets remove any top level nav bar related hash searches
        if (removeParams) {
            angular.forEach(removeParams, function (param) {
                return delete hashMap[param];
            });
        }
        var hash = Core.hashToString(hashMap);
        if (hash) {
            var prefix = (href.indexOf("?") >= 0) ? "&" : "?";
            href += prefix + hash;
        }
        return href;
    }
    Core.createHref = createHref;

    /**
    * Trims the leading prefix from a string if its present
    * @method trimLeading
    * @for Core
    * @static
    * @param {String} text
    * @param {String} prefix
    * @return {String}
    */
    function trimLeading(text, prefix) {
        if (text && prefix) {
            if (text.startsWith(prefix)) {
                return text.substring(prefix.length);
            }
        }
        return text;
    }
    Core.trimLeading = trimLeading;

    /**
    * Trims the trailing postfix from a string if its present
    * @method trimTrailing
    * @for Core
    * @static
    * @param {String} trim
    * @param {String} postfix
    * @return {String}
    */
    function trimTrailing(text, postfix) {
        if (text && postfix) {
            if (text.endsWith(postfix)) {
                return text.substring(0, text.length - postfix.length);
            }
        }
        return text;
    }
    Core.trimTrailing = trimTrailing;

    /**
    * Turns the given search hash into a URI style query string
    * @method hashToString
    * @for Core
    * @static
    * @param {Object} hash
    * @return {String}
    */
    function hashToString(hash) {
        var keyValuePairs = [];
        angular.forEach(hash, function (value, key) {
            keyValuePairs.push(key + "=" + value);
        });
        var params = keyValuePairs.join("&");
        return encodeURI(params);
    }
    Core.hashToString = hashToString;

    /**
    * Parses the given string of x=y&bar=foo into a hash
    * @method stringToHash
    * @for Core
    * @static
    * @param {String} hashAsString
    * @return {Object}
    */
    function stringToHash(hashAsString) {
        var entries = {};
        if (hashAsString) {
            var text = decodeURI(hashAsString);
            var items = text.split('&');
            angular.forEach(items, function (item) {
                var kv = item.split('=');
                var key = kv[0];
                var value = kv[1] || key;
                entries[key] = value;
            });
        }
        return entries;
    }
    Core.stringToHash = stringToHash;

    /**
    * Register a JMX operation to poll for changes
    * @method register
    * @for Core
    * @static
    * @return a zero argument function for unregistering  this registration
    * @param {*} jolokia
    * @param {*} scope
    * @param {Object} arguments
    * @param {Function} callback
    */
    function register(jolokia, scope, arguments, callback) {
        if (!angular.isDefined(scope.$jhandle) || !angular.isArray(scope.$jhandle)) {
            scope.$jhandle = [];
        }
        if (angular.isDefined(scope.$on)) {
            scope.$on('$destroy', function (event) {
                unregister(jolokia, scope);
            });
        }
        if (angular.isArray(arguments)) {
            if (arguments.length >= 1) {
                // TODO can't get this to compile in typescript :)
                //var args = [callback].concat(arguments);
                var args = [callback];
                angular.forEach(arguments, function (value) {
                    return args.push(value);
                });

                //var args = [callback];
                //args.push(arguments);
                var registerFn = jolokia.register;
                var handle = registerFn.apply(jolokia, args);
                scope.$jhandle.push(handle);
                jolokia.request(arguments, callback);
            }
        } else {
            var handle = jolokia.register(callback, arguments);
            scope.$jhandle.push(handle);
            jolokia.request(arguments, callback);
        }
        return function () {
            Core.unregister(jolokia, scope);
        };
    }
    Core.register = register;

    /**
    * Register a JMX operation to poll for changes using a jolokia search using the given mbean pattern
    * @method registerSearch
    * @for Core
    * @static
    * @paran {*} jolokia
    * @param {*} scope
    * @param {String} mbeanPattern
    * @param {Function} callback
    */
    function registerSearch(jolokia, scope, mbeanPattern, callback) {
        if (!angular.isDefined(scope.$jhandle) || !angular.isArray(scope.$jhandle)) {
            scope.$jhandle = [];
        }
        if (angular.isDefined(scope.$on)) {
            scope.$on('$destroy', function (event) {
                unregister(jolokia, scope);
            });
        }
        if (angular.isArray(arguments)) {
            if (arguments.length >= 1) {
                // TODO can't get this to compile in typescript :)
                //var args = [callback].concat(arguments);
                var args = [callback];
                angular.forEach(arguments, function (value) {
                    return args.push(value);
                });

                //var args = [callback];
                //args.push(arguments);
                var registerFn = jolokia.register;
                var handle = registerFn.apply(jolokia, args);
                scope.$jhandle.push(handle);
                jolokia.search(mbeanPattern, callback);
            }
        } else {
            var handle = jolokia.register(callback, arguments);
            scope.$jhandle.push(handle);
            jolokia.search(mbeanPattern, callback);
        }
    }
    Core.registerSearch = registerSearch;

    function unregister(jolokia, scope) {
        if (angular.isDefined(scope.$jhandle)) {
            scope.$jhandle.forEach(function (handle) {
                jolokia.unregister(handle);
            });
            delete scope.$jhandle;
        }
    }
    Core.unregister = unregister;

    /**
    * The default error handler which logs errors either using debug or log level logging based on the silent setting
    * @param response the response from a jolokia request
    */
    function defaultJolokiaErrorHandler(response, options) {
        if (typeof options === "undefined") { options = {}; }
        //alert("Jolokia request failed: " + response.error);
        var stacktrace = response.stacktrace;
        if (stacktrace) {
            var silent = options['silent'];
            if (!silent) {
                var operation = Core.pathGet(response, ['request', 'operation']) || "unknown";
                if (stacktrace.indexOf("javax.management.InstanceNotFoundException") >= 0 || stacktrace.indexOf("javax.management.AttributeNotFoundException") >= 0 || stacktrace.indexOf("java.lang.IllegalArgumentException: No operation") >= 0) {
                    // ignore these errors as they can happen on timing issues
                    // such as its been removed
                    // or if we run against older containers
                    Core.log.debug("Operation ", operation, " failed due to: ", response['error']);
                    Core.log.debug("Stack trace: ", Logger.formatStackTraceString(response['stacktrace']));
                } else {
                    Core.log.warn("Operation ", operation, " failed due to: ", response['error']);
                    Core.log.info("Stack trace: ", Logger.formatStackTraceString(response['stacktrace']));
                }
            }
        }
    }
    Core.defaultJolokiaErrorHandler = defaultJolokiaErrorHandler;

    /**
    * Converts the given XML node to a string representation of the XML
    * @method xmlNodeToString
    * @for Core
    * @static
    * @param {Object} xmlNode
    * @return {Object}
    */
    function xmlNodeToString(xmlNode) {
        try  {
            // Gecko- and Webkit-based browsers (Firefox, Chrome), Opera.
            return (new XMLSerializer()).serializeToString(xmlNode);
        } catch (e) {
            try  {
                // Internet Explorer.
                return xmlNode.xml;
            } catch (e) {
                //Other browsers without XML Serializer
                console.log('WARNING: XMLSerializer not supported');
            }
        }
        return false;
    }
    Core.xmlNodeToString = xmlNodeToString;

    /**
    * Returns true if the given DOM node is a text node
    * @method isTextNode
    * @for Core
    * @static
    * @param {Object} node
    * @return {Boolean}
    */
    function isTextNode(node) {
        return node && node.nodeType === 3;
    }
    Core.isTextNode = isTextNode;

    /**
    * Performs a $scope.$apply() if not in a digest right now otherwise it will fire a digest later
    * @method $applyNowOrLater
    * @for Core
    * @static
    * @param {*} $scope
    */
    function $applyNowOrLater($scope) {
        if ($scope.$$phase || $scope.$root.$$phase) {
            setTimeout(function () {
                Core.$apply($scope);
            }, 50);
        } else {
            $scope.$apply();
        }
    }
    Core.$applyNowOrLater = $applyNowOrLater;

    /**
    * Performs a $scope.$apply() after the given timeout period
    * @method $applyLater
    * @for Core
    * @static
    * @param {*} $scope
    * @param {Integer} timeout
    */
    function $applyLater($scope, timeout) {
        if (typeof timeout === "undefined") { timeout = 50; }
        setTimeout(function () {
            Core.$apply($scope);
        }, timeout);
    }
    Core.$applyLater = $applyLater;

    /**
    * Performs a $scope.$apply() if not in a digest or apply phase on the given scope
    * @method $apply
    * @for Core
    * @static
    * @param {*} $scope
    */
    function $apply($scope) {
        var phase = $scope.$$phase || $scope.$root.$$phase;
        if (!phase) {
            $scope.$apply();
        }
    }
    Core.$apply = $apply;

    function $digest($scope) {
        var phase = $scope.$$phase || $scope.$root.$$phase;
        if (!phase) {
            $scope.$digest();
        }
    }
    Core.$digest = $digest;

    /**
    * Returns the lowercase file extension of the given file name or returns the empty
    * string if the file does not have an extension
    * @method fileExtension
    * @for Core
    * @static
    * @param {String} name
    * @param {String} defaultValue
    * @return {String}
    */
    function fileExtension(name, defaultValue) {
        if (typeof defaultValue === "undefined") { defaultValue = ""; }
        var extension = defaultValue;
        if (name) {
            var idx = name.lastIndexOf(".");
            if (idx > 0) {
                extension = name.substring(idx + 1, name.length).toLowerCase();
            }
        }
        return extension;
    }
    Core.fileExtension = fileExtension;

    function parseIntValue(value, description) {
        if (angular.isString(value)) {
            try  {
                return parseInt(value);
            } catch (e) {
                console.log("Failed to parse " + description + " with text '" + value + "'");
            }
        }
        return null;
    }
    Core.parseIntValue = parseIntValue;

    function parseFloatValue(value, description) {
        if (angular.isString(value)) {
            try  {
                return parseFloat(value);
            } catch (e) {
                console.log("Failed to parse " + description + " with text '" + value + "'");
            }
        }
        return null;
    }
    Core.parseFloatValue = parseFloatValue;

    /**
    * Look up a list of child element names or lazily create them.
    *
    * Useful for example to get the <tbody> <tr> element from a <table> lazily creating one
    * if not present.
    *
    * Usage: var trElement = getOrCreateElements(tableElement, ["tbody", "tr"])
    * @method getOrCreateElements
    * @for Core
    * @static
    * @param {Object} domElement
    * @param {Array} arrayOfElementNames
    * @return {Object}
    */
    function getOrCreateElements(domElement, arrayOfElementNames) {
        var element = domElement;
        angular.forEach(arrayOfElementNames, function (name) {
            if (element) {
                var children = $(element).children(name);
                if (!children || !children.length) {
                    $("<" + name + "></" + name + ">").appendTo(element);
                    children = $(element).children(name);
                }
                element = children;
            }
        });
        return element;
    }
    Core.getOrCreateElements = getOrCreateElements;

    function getUUID() {
        var d = new Date();
        var ms = (d.getTime() * 1000) + d.getUTCMilliseconds();
        var random = Math.floor((1 + Math.random()) * 0x10000);
        return ms.toString(16) + random.toString(16);
    }
    Core.getUUID = getUUID;

    /**
    * Navigates the given set of paths in turn on the source object
    * and returns the last most value of the path or null if it could not be found.
    * @method pathGet
    * @for Core
    * @static
    * @param {Object} object the start object to start navigating from
    * @param {Array} paths an array of path names to navigate or a string of dot separated paths to navigate
    * @return {*} the last step on the path which is updated
    */
    function pathGet(object, paths) {
        var pathArray = (angular.isArray(paths)) ? paths : (paths || "").split(".");
        var value = object;
        angular.forEach(pathArray, function (name) {
            if (value) {
                try  {
                    value = value[name];
                } catch (e) {
                    // ignore errors
                    return null;
                }
            } else {
                return null;
            }
        });
        return value;
    }
    Core.pathGet = pathGet;

    /**
    * Navigates the given set of paths in turn on the source object
    * and updates the last path value to the given newValue
    * @method pathSet
    * @for Core
    * @static
    * @param {Object} object the start object to start navigating from
    * @param {Array} paths an array of path names to navigate or a string of dot separated paths to navigate
    * @param {Object} newValue the value to update
    * @return {*} the last step on the path which is updated
    */
    function pathSet(object, paths, newValue) {
        var pathArray = (angular.isArray(paths)) ? paths : (paths || "").split(".");
        var value = object;
        var lastIndex = pathArray.length - 1;
        angular.forEach(pathArray, function (name, idx) {
            var next = value[name];
            if (idx >= lastIndex || !angular.isObject(next)) {
                next = (idx < lastIndex) ? {} : newValue;
                value[name] = next;
            }
            value = next;
        });
        return value;
    }
    Core.pathSet = pathSet;

    var _escapeHtmlChars = {
        "#": "&#35;",
        "'": "&#39;",
        "<": "&lt;",
        ">": "&gt;",
        "\"": "&quot;"
    };

    function unescapeHtml(str) {
        angular.forEach(_escapeHtmlChars, function (value, key) {
            var regex = new RegExp(value, "g");
            str = str.replace(regex, key);
        });
        str = str.replace(/&gt;/g, ">");
        return str;
    }
    Core.unescapeHtml = unescapeHtml;

    function escapeHtml(str) {
        if (angular.isString(str)) {
            var newStr = "";
            for (var i = 0; i < str.length; i++) {
                var ch = str.charAt(i);
                var ch = _escapeHtmlChars[ch] || ch;
                newStr += ch;
                /*
                var nextCode = str.charCodeAt(i);
                if (nextCode > 0 && nextCode < 48) {
                newStr += "&#" + nextCode + ";";
                }
                else {
                newStr += ch;
                }
                */
            }
            return newStr;
        } else {
            return str;
        }
    }
    Core.escapeHtml = escapeHtml;

    var _versionRegex = /[^\d]*(\d+)\.(\d+)(\.(\d+))?.*/;

    /**
    * Parses some text of the form "xxxx2.3.4xxxx"
    * to extract the version numbers as an array of numbers then returns an array of 2 or 3 numbers.
    *
    * Characters before the first digit are ignored as are characters after the last digit.
    * @method parseVersionNumbers
    * @for Core
    * @static
    * @param {String} text a maven like string containing a dash then numbers separated by dots
    * @return {Array}
    */
    function parseVersionNumbers(text) {
        if (text) {
            var m = text.match(_versionRegex);
            if (m && m.length > 4) {
                var m1 = m[1];
                var m2 = m[2];
                var m4 = m[4];
                if (angular.isDefined(m4)) {
                    return [parseInt(m1), parseInt(m2), parseInt(m4)];
                } else if (angular.isDefined(m2)) {
                    return [parseInt(m1), parseInt(m2)];
                } else if (angular.isDefined(m1)) {
                    return [parseInt(m1)];
                }
            }
        }
        return null;
    }
    Core.parseVersionNumbers = parseVersionNumbers;

    /**
    * Converts a version string with numbers and dots of the form "123.456.790" into a string
    * which is sortable as a string, by left padding each string between the dots to at least 4 characters
    * so things just sort as a string.
    *
    * @param text
    * @return {string} the sortable version string
    */
    function versionToSortableString(version, maxDigitsBetweenDots) {
        if (typeof maxDigitsBetweenDots === "undefined") { maxDigitsBetweenDots = 4; }
        return (version || "").split(".").map(function (x) {
            var length = x.length;
            return (length >= maxDigitsBetweenDots) ? x : x.padLeft(' ', maxDigitsBetweenDots - length);
        }).join(".");
    }
    Core.versionToSortableString = versionToSortableString;

    function time(message, fn) {
        var start = new Date().getTime();
        var answer = fn();
        var elapsed = new Date().getTime() - start;
        console.log(message + " " + elapsed);
        return answer;
    }
    Core.time = time;

    /**
    * Compares the 2 version arrays and returns -1 if v1 is less than v2 or 0 if they are equal or 1 if v1 is greater than v2
    * @method compareVersionNumberArrays
    * @for Core
    * @static
    * @param {Array} v1 an array of version numbers with the most significant version first (major, minor, patch).
    * @param {Array} v2
    * @return {Number}
    */
    function compareVersionNumberArrays(v1, v2) {
        if (v1 && !v2) {
            return 1;
        }
        if (!v1 && v2) {
            return -1;
        }
        if (v1 === v2) {
            return 0;
        }
        for (var i = 0; i < v1.length; i++) {
            var n1 = v1[i];
            if (i >= v2.length) {
                return 1;
            }
            var n2 = v2[i];
            if (!angular.isDefined(n1)) {
                return -1;
            }
            if (!angular.isDefined(n2)) {
                return 1;
            }
            if (n1 > n2) {
                return 1;
            } else if (n1 < n2) {
                return -1;
            }
        }
        return 0;
    }
    Core.compareVersionNumberArrays = compareVersionNumberArrays;

    /**
    * If the value is not an array then wrap it in one
    * @method asArray
    * @for Core
    * @static
    * @param {any} value
    * @return {Array}
    */
    function asArray(value) {
        return angular.isArray(value) ? value : [value];
    }
    Core.asArray = asArray;

    /**
    * Helper function which converts objects into tables of key/value properties and
    * lists into a <ul> for each value.
    * @method valueToHtml
    * @for Core
    * @static
    * @param {any} value
    * @return {String}
    */
    function valueToHtml(value) {
        if (angular.isArray(value)) {
            var size = value.length;
            if (!size) {
                return "";
            } else if (size === 1) {
                return valueToHtml(value[0]);
            } else {
                var buffer = "<ul>";
                angular.forEach(value, function (childValue) {
                    buffer += "<li>" + valueToHtml(childValue) + "</li>";
                });
                return buffer + "</ul>";
            }
        } else if (angular.isObject(value)) {
            var buffer = "<table>";
            angular.forEach(value, function (childValue, key) {
                buffer += "<tr><td>" + key + "</td><td>" + valueToHtml(childValue) + "</td></tr>";
            });
            return buffer + "</table>";
        } else if (angular.isString(value)) {
            var uriPrefixes = ["http://", "https://", "file://", "mailto:"];
            var answer = value;
            angular.forEach(uriPrefixes, function (prefix) {
                if (answer.startsWith(prefix)) {
                    answer = "<a href='" + value + "'>" + value + "</a>";
                }
            });
            return answer;
        }
        return value;
    }
    Core.valueToHtml = valueToHtml;

    /**
    * If the string starts and ends with [] {} then try parse as JSON and return the parsed content or return null
    * if it does not appear to be JSON
    * @method tryParseJson
    * @for Core
    * @static
    * @param {String} text
    * @return {Object}
    */
    function tryParseJson(text) {
        text = text.trim();
        if ((text.startsWith("[") && text.endsWith("]")) || (text.startsWith("{") && text.endsWith("}"))) {
            try  {
                return JSON.parse(text);
            } catch (e) {
                // ignore
            }
        }
        return null;
    }
    Core.tryParseJson = tryParseJson;

    /**
    * Given values (n, "person") will return either "1 person" or "2 people" depending on if a plural
    * is required using the String.pluralize() function from sugarjs
    * @method maybePlural
    * @for Core
    * @static
    * @param {Number} count
    * @param {String} word
    * @return {String}
    */
    function maybePlural(count, word) {
        var pluralWord = (count === 1) ? word : word.pluralize();
        return "" + count + " " + pluralWord;
    }
    Core.maybePlural = maybePlural;

    /**
    * given a JMX ObjectName of the form <code>domain:key=value,another=something</code> then return the object
    * <code>{key: "value", another: "something"}</code>
    * @method objectNameProperties
    * @for Core
    * @static
    * @param {String} name
    * @return {Object}
    */
    function objectNameProperties(objectName) {
        var entries = {};
        if (objectName) {
            var idx = objectName.indexOf(":");
            if (idx > 0) {
                var path = objectName.substring(idx + 1);
                var items = path.split(',');
                angular.forEach(items, function (item) {
                    var kv = item.split('=');
                    var key = kv[0];
                    var value = kv[1] || key;
                    entries[key] = value;
                });
            }
        }
        return entries;
    }
    Core.objectNameProperties = objectNameProperties;

    function setPageTitle($document, title) {
        $document.attr('title', title.getTitleWithSeparator(' '));
    }
    Core.setPageTitle = setPageTitle;

    function setPageTitleWithTab($document, title, tab) {
        $document.attr('title', title.getTitleWithSeparator(' ') + " " + tab);
    }
    Core.setPageTitleWithTab = setPageTitleWithTab;

    /**
    * Returns the Folder object for the given domain name and type name or null if it can not be found
    * @method getMBeanTypeFolder
    * @for Core
    * @static
    * @param {Workspace} workspace
    * @param {String} domain
    * @param {String} typeName}
    * @return {Folder}
    */
    function getMBeanTypeFolder(workspace, domain, typeName) {
        if (workspace) {
            var mbeanTypesToDomain = workspace.mbeanTypesToDomain || {};
            var types = mbeanTypesToDomain[typeName] || {};
            var answer = types[domain];
            if (angular.isArray(answer) && answer.length) {
                return answer[0];
            }
            return answer;
        }
        return null;
    }
    Core.getMBeanTypeFolder = getMBeanTypeFolder;

    /**
    * Returns the JMX objectName for the given jmx domain and type name
    * @method getMBeanTypeObjectName
    * @for Core
    * @static
    * @param {Workspace} workspace
    * @param {String} domain
    * @param {String} typeName
    * @return {String}
    */
    function getMBeanTypeObjectName(workspace, domain, typeName) {
        var folder = Core.getMBeanTypeFolder(workspace, domain, typeName);
        return Core.pathGet(folder, ["objectName"]);
    }
    Core.getMBeanTypeObjectName = getMBeanTypeObjectName;

    /**
    * Removes dodgy characters from a value such as '/' or '.' so that it can be used as a DOM ID value
    * and used in jQuery / CSS selectors
    * @method toSafeDomID
    * @for Core
    * @static
    * @param {String} text
    * @return {String}
    */
    function toSafeDomID(text) {
        return text ? text.replace(/(\/|\.)/g, "_") : text;
    }
    Core.toSafeDomID = toSafeDomID;

    /**
    * Invokes the given function on each leaf node in the array of folders
    * @method forEachLeafFolder
    * @for Core
    * @static
    * @param {Array[Folder]} folders
    * @param {Function} fn
    */
    function forEachLeafFolder(folders, fn) {
        angular.forEach(folders, function (folder) {
            var children = folder["children"];
            if (angular.isArray(children) && children.length > 0) {
                forEachLeafFolder(children, fn);
            } else {
                fn(folder);
            }
        });
    }
    Core.forEachLeafFolder = forEachLeafFolder;

    function extractHashURL(url) {
        var parts = url.split('#');
        if (parts.length === 0) {
            return url;
        }
        var answer = parts[1];
        if (parts.length > 1) {
            var remaining = parts.last(parts.length - 2);
            remaining.forEach(function (part) {
                answer = answer + "#" + part;
            });
        }
        return answer;
    }
    Core.extractHashURL = extractHashURL;

    function getBasicAuthHeader(username, password) {
        var authInfo = username + ":" + password;
        authInfo = btoa(authInfo);
        return "Basic " + authInfo;
    }
    Core.getBasicAuthHeader = getBasicAuthHeader;

    var ConnectToServerOptions = (function () {
        function ConnectToServerOptions() {
            this.scheme = "http";
            this.useProxy = true;
        }
        return ConnectToServerOptions;
    })();
    Core.ConnectToServerOptions = ConnectToServerOptions;

    function getDocHeight() {
        var D = document;
        return Math.max(Math.max(D.body.scrollHeight, D.documentElement.scrollHeight), Math.max(D.body.offsetHeight, D.documentElement.offsetHeight), Math.max(D.body.clientHeight, D.documentElement.clientHeight));
    }
    Core.getDocHeight = getDocHeight;

    /**
    * If a URL is external to the current web application, then
    * replace the URL with the proxy servlet URL
    * @method useProxyIfExternal
    * @for Core
    * @static
    * @param {String} connectUrl
    * @return {String}
    */
    function useProxyIfExternal(connectUrl) {
        if (Core.isChromeApp()) {
            return connectUrl;
        }
        var host = window.location.host;
        if (!connectUrl.startsWith("http://" + host + "/") && !connectUrl.startsWith("https://" + host + "/")) {
            // lets remove the http stuff
            var idx = connectUrl.indexOf("://");
            if (idx > 0) {
                connectUrl = connectUrl.substring(idx + 3);
            }

            // lets replace the : with a /
            connectUrl = connectUrl.replace(":", "/");
            connectUrl = Core.trimLeading(connectUrl, "/");
            connectUrl = Core.trimTrailing(connectUrl, "/");
            connectUrl = url("/proxy/" + connectUrl);
        }
        return connectUrl;
    }
    Core.useProxyIfExternal = useProxyIfExternal;

    function connectToServer(localStorage, options) {
        var connectUrl = options.jolokiaUrl;

        var userDetails = {
            username: options['userName'],
            password: options['password']
        };

        // TODO we should replace this and just store the real, final connectUrl!
        var json = angular.toJson(userDetails);
        if (connectUrl) {
            localStorage[connectUrl] = json;
        }
        var view = options.view;
        var full = "";
        var useProxy = options.useProxy && !Core.isChromeApp();
        if (connectUrl) {
            if (useProxy) {
                // lets remove the http stuff
                var idx = connectUrl.indexOf("://");
                if (idx > 0) {
                    connectUrl = connectUrl.substring(idx + 3);
                }

                // lets replace the : with a /
                connectUrl = connectUrl.replace(":", "/");
                connectUrl = Core.trimLeading(connectUrl, "/");
                connectUrl = Core.trimTrailing(connectUrl, "/");
                connectUrl = url("/proxy/" + connectUrl);
            } else {
                if (connectUrl.indexOf("://") < 0) {
                    connectUrl = options.scheme + "://" + connectUrl;
                }
            }
            console.log("going to server: " + connectUrl + " as user " + options.userName);
            localStorage[connectUrl] = json;

            full = "?url=" + encodeURIComponent(connectUrl);
            if (view) {
                full += "#" + view;
            }
        } else {
            var host = options.host || "localhost";
            var port = options.port;
            var path = Core.trimLeading(options.path || "jolokia", "/");
            path = Core.trimTrailing(path, "/");

            if (port > 0) {
                host += ":" + port;
            }
            var connectUrl = host + "/" + path;
            localStorage[connectUrl] = json;
            if (useProxy) {
                connectUrl = url("/proxy/" + connectUrl);
            } else {
                if (connectUrl.indexOf("://") < 0) {
                    connectUrl = options.scheme + "://" + connectUrl;
                }
            }
            console.log("going to server: " + connectUrl + " as user " + options.userName);
            localStorage[connectUrl] = json;

            full = "?url=" + encodeURIComponent(connectUrl);
            if (view) {
                full += "#" + view;
            }
        }
        if (full) {
            Core.log.info("Full URL is: " + full);
            window.open(full);
        }
    }
    Core.connectToServer = connectToServer;

    /**
    * Binds a $location.search() property to a model on a scope; so that its initialised correctly on startup
    * and its then watched so as the model changes, the $location.search() is updated to reflect its new value
    * @method bindModelToSearchParam
    * @for Core
    * @static
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {String} modelName
    * @param {String} paramName
    * @param {Object} initialValue
    */
    function bindModelToSearchParam($scope, $location, modelName, paramName, initialValue) {
        if (typeof initialValue === "undefined") { initialValue = null; }
        function currentValue() {
            return $location.search()[paramName] || initialValue;
        }

        var value = currentValue();
        Core.pathSet($scope, modelName, value);
        $scope.$watch(modelName, function () {
            var current = Core.pathGet($scope, modelName);
            if (current) {
                var params = $location.search();
                var old = currentValue();
                if (current !== old) {
                    $location.search(paramName, current);
                }
            } else {
                $location.search(paramName, null);
            }
        });
    }
    Core.bindModelToSearchParam = bindModelToSearchParam;

    /**
    * For controllers where reloading is disabled via "reloadOnSearch: false" on the registration; lets pick which
    * query parameters need to change to force the reload. We default to the JMX selection parameter 'nid'
    * @method reloadWhenParametersChange
    * @for Core
    * @static
    * @param {Object} $route
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {Array[String]} parameters
    */
    function reloadWhenParametersChange($route, $scope, $location, parameters) {
        if (typeof parameters === "undefined") { parameters = ["nid"]; }
        var initial = angular.copy($location.search());
        $scope.$on('$routeUpdate', function () {
            // lets check if any of the parameters changed
            var current = $location.search();
            var changed = [];
            angular.forEach(parameters, function (param) {
                if (current[param] !== initial[param]) {
                    changed.push(param);
                }
            });
            if (changed.length) {
                Core.log.info("Reloading page due to change to parameters: " + changed);
                $route.reload();
            }
        });
    }
    Core.reloadWhenParametersChange = reloadWhenParametersChange;

    /**
    * Creates a jolokia object for connecting to the container with the given remote jolokia URL,
    * username and password
    * @method createJolokia
    * @for Core
    * @static
    * @param {String} url
    * @param {String} username
    * @param {String} password
    * @return {Object}
    */
    function createJolokia(url, username, password) {
        var jolokiaParams = {
            url: url,
            username: username,
            password: password,
            canonicalNaming: false, ignoreErrors: true, mimeType: 'application/json'
        };
        return new Jolokia(jolokiaParams);
    }
    Core.createJolokia = createJolokia;

    /**
    * Returns a new function which ensures that the delegate function is only invoked at most once
    * within the given number of millseconds
    * @method throttled
    * @for Core
    * @static
    * @param {Function} fn the function to be invoked at most once within the given number of millis
    * @param {Number} millis the time window during which this function should only be called at most once
    * @return {Object}
    */
    function throttled(fn, millis) {
        var nextInvokeTime = 0;
        var lastAnswer = null;
        return function () {
            var now = Date.now();
            if (nextInvokeTime < now) {
                nextInvokeTime = now + millis;
                lastAnswer = fn();
            } else {
                Core.log.debug("Not invoking function as we did call " + (now - (nextInvokeTime - millis)) + " ms ago");
            }
            return lastAnswer;
        };
    }
    Core.throttled = throttled;

    /**
    * Attempts to parse the given JSON text and returns the JSON object structure or null.
    *Bad JSON is logged at info level.
    *
    * @param text a JSON formatted string
    * @param message description of the thing being parsed logged if its invalid
    */
    function parseJsonText(text, message) {
        if (typeof message === "undefined") { message = "JSON"; }
        var answer = null;
        try  {
            answer = angular.fromJson(text);
        } catch (e) {
            Core.log.info("Failed to parse " + message + " from: " + text + ". " + e);
        }
        return answer;
    }
    Core.parseJsonText = parseJsonText;

    /**
    * Returns the humanized markup of the given value
    */
    function humanizeValueHtml(value) {
        var formattedValue = "";
        if (value === true) {
            formattedValue = '<i class="icon-check"></i>';
        } else if (value === false) {
            formattedValue = '<i class="icon-check-empty"></i>';
        } else {
            formattedValue = humanizeValue(value);
        }
        return formattedValue;
    }
    Core.humanizeValueHtml = humanizeValueHtml;

    /**
    * Gets a query value from the given url
    *
    * @param url  url
    * @param parameterName the uri parameter value to get
    * @returns {*}
    */
    function getQueryParameterValue(url, parameterName) {
        var parts;

        var query = (url || '').split('?');
        if (query && query.length > 0) {
            parts = query[1];
        } else {
            parts = '';
        }

        var vars = parts.split('&');
        for (var i = 0; i < vars.length; i++) {
            var pair = vars[i].split('=');
            if (decodeURIComponent(pair[0]) == parameterName) {
                return decodeURIComponent(pair[1]);
            }
        }

        // not found
        return null;
    }
    Core.getQueryParameterValue = getQueryParameterValue;

    /**
    * Creates a remote workspace given a remote jolokia for querying the JMX MBeans inside the jolokia
    * @param remoteJolokia
    * @param $location
    * @param localStorage
    * @return {Core.Workspace|Workspace}
    */
    function createRemoteWorkspace(remoteJolokia, $location, localStorage, $rootScope, $compile, $templateCache, userDetails) {
        if (typeof $rootScope === "undefined") { $rootScope = null; }
        if (typeof $compile === "undefined") { $compile = null; }
        if (typeof $templateCache === "undefined") { $templateCache = null; }
        if (typeof userDetails === "undefined") { userDetails = null; }
        // lets create a child workspace object for the remote container
        var jolokiaStatus = {
            xhr: null
        };

        // disable reload notifications
        var jmxTreeLazyLoadRegistry = Jmx.lazyLoaders;
        var profileWorkspace = new Core.Workspace(remoteJolokia, jolokiaStatus, jmxTreeLazyLoadRegistry, $location, $compile, $templateCache, localStorage, $rootScope, userDetails);

        Core.log.info("Loading the profile using jolokia: " + remoteJolokia);
        profileWorkspace.loadTree();
        return profileWorkspace;
    }
    Core.createRemoteWorkspace = createRemoteWorkspace;

    /**
    * Takes a value in ms and returns a human readable
    * duration
    * @param value
    */
    function humanizeMilliseconds(value) {
        if (!angular.isNumber(value)) {
            return "XXX";
        }

        var seconds = value / 1000;
        var years = Math.floor(seconds / 31536000);
        if (years) {
            return maybePlural(years, "year");
        }
        var days = Math.floor((seconds %= 31536000) / 86400);
        if (days) {
            return maybePlural(days, "day");
        }
        var hours = Math.floor((seconds %= 86400) / 3600);
        if (hours) {
            return maybePlural(hours, 'hour');
        }
        var minutes = Math.floor((seconds %= 3600) / 60);
        if (minutes) {
            return maybePlural(minutes, 'minute');
        }
        seconds = Math.floor(seconds % 60);
        if (seconds) {
            return maybePlural(seconds, 'second');
        }
        return value + " ms";
    }
    Core.humanizeMilliseconds = humanizeMilliseconds;

    function storeConnectionRegex(regexs, name, json) {
        if (!regexs.any(function (r) {
            r['name'] === name;
        })) {
            var regex = '';

            if (json['useProxy']) {
                regex = '/hawtio/proxy/';
            } else {
                regex = '//';
            }
            regex += json['host'] + ':' + json['port'] + '/' + json['path'];
            regexs.push({
                name: name,
                regex: regex.escapeURL(true),
                color: UI.colors.sample()
            });
            writeRegexs(regexs);
        }
    }
    Core.storeConnectionRegex = storeConnectionRegex;

    function getRegexs() {
        var regexs = [];
        try  {
            regexs = angular.fromJson(localStorage['regexs']);
        } catch (e) {
            // corrupted config
            delete localStorage['regexs'];
        }
        return regexs;
    }
    Core.getRegexs = getRegexs;

    function removeRegex(name) {
        var regexs = Core.getRegexs();
        var hasFunc = function (r) {
            return r['name'] === name;
        };
        if (regexs.any(hasFunc)) {
            regexs = regexs.exclude(hasFunc);
            Core.writeRegexs(regexs);
        }
    }
    Core.removeRegex = removeRegex;

    function writeRegexs(regexs) {
        localStorage['regexs'] = angular.toJson(regexs);
    }
    Core.writeRegexs = writeRegexs;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function HelpController($scope, $routeParams, marked, helpRegistry, branding) {
        $scope.branding = branding;
        $scope.topics = helpRegistry.getTopics();

        if ('topic' in $routeParams) {
            $scope.topic = $routeParams['topic'];
        } else {
            $scope.topic = 'index';
        }

        if ('subtopic' in $routeParams) {
            $scope.subTopic = $routeParams['subtopic'];
        } else {
            $scope.subTopic = Object.extended($scope.topics[$scope.topic]).keys().first();
        }

        Core.log.debug("topic: ", $scope.topic, " subtopic: ", $scope.subTopic);

        // when on the index pages, filter the user subTopic unless on the dev page
        var isIndex = $scope.topic === "index";
        var filterSubTopic = $scope.subTopic;
        if (isIndex && filterSubTopic !== "developer") {
            filterSubTopic = "user";
        }

        $scope.breadcrumbs = [
            {
                topic: "index",
                subTopic: "user",
                label: "User Guide"
            },
            {
                topic: "index",
                subTopic: "faq",
                label: "FAQ"
            },
            {
                topic: "index",
                subTopic: "changes",
                label: "Changes"
            },
            {
                topic: "index",
                subTopic: "developer",
                label: "Developers"
            }
        ];

        // lets select the active tab
        var activeBreadcrumb = $scope.breadcrumbs.find(function (b) {
            return b.topic === $scope.topic && b.subTopic === $scope.subTopic;
        });
        if (activeBreadcrumb)
            activeBreadcrumb.active = true;

        $scope.sections = [];
        angular.forEach($scope.topics, function (details, topic) {
            // lets hide any index topics or any topics which don't have a filter sub topic
            if (topic !== "index" && details[filterSubTopic]) {
                $scope.sections.push({
                    topic: topic,
                    subTopic: filterSubTopic,
                    label: helpRegistry.mapTopicName(topic),
                    active: topic === $scope.topic
                });
            }
        });
        $scope.sections = $scope.sections.sortBy("label");

        $scope.$on('hawtioNewHelpTopic', function () {
            $scope.topics = helpRegistry.getTopics();
        });

        $scope.$watch('topics', function (newValue, oldValue) {
            Core.log.debug("Topics: ", $scope.topics);
        });

        if (!angular.isDefined($scope.topics[$scope.topic])) {
            $scope.html = "Unable to download help data for " + $scope.topic;
        } else {
            $.ajax({
                url: $scope.topics[$scope.topic][$scope.subTopic],
                dataType: 'html',
                cache: false,
                success: function (data, textStatus, jqXHR) {
                    $scope.html = "Unable to download help data for " + $scope.topic;
                    if (angular.isDefined(data)) {
                        $scope.html = marked(data);
                    }
                    Core.$apply($scope);
                },
                error: function (jqXHR, textStatus, errorThrown) {
                    $scope.html = "Unable to download help data for " + $scope.topic;
                    Core.$apply($scope);
                }
            });
        }
    }
    Core.HelpController = HelpController;
})(Core || (Core = {}));
var _this = this;
/**
* The main entry point for hawtio
*
* @module Core
* @main Core
*/
var Core;
(function (Core) {
    /**
    * Returns true if we are running inside a Chrome app or extension
    */
    function isChromeApp() {
        var answer = false;
        try  {
            answer = (chrome && chrome.app && chrome.extension) ? true : false;
        } catch (e) {
            answer = false;
        }

        //log.info("isChromeApp is: " + answer);
        return answer;
    }
    Core.isChromeApp = isChromeApp;

    /**
    * Name of plugin registered to hawtio's plugin loader and Angularjs module name
    *
    * @property pluginName
    * @for Core
    * @type String
    */
    Core.pluginName = 'hawtioCore';
})(Core || (Core = {}));

// Add any other known possible jolokia URLs here
var jolokiaUrls = [
    url("jolokia"),
    "/jolokia"
];

var jolokiaUrl = getJolokiaUrl();
console.log("jolokiaUrl " + jolokiaUrl);

function getJolokiaUrl() {
    var query = hawtioPluginLoader.parseQueryString();
    var localMode = query['localMode'];
    if (localMode) {
        console.log("local mode so not using jolokia URL");
        jolokiaUrls = [];
        return null;
    }
    var uri = query['url'];
    if (angular.isArray(uri)) {
        uri = uri[0];
    }
    return uri ? decodeURIComponent(uri) : null;
}

if (!jolokiaUrl) {
    jolokiaUrl = jolokiaUrls.find(function (url) {
        var jqxhr = $.ajax(url, {
            async: false,
            username: 'public',
            password: 'biscuit'
        });
        return jqxhr.status === 200 || jqxhr.status === 401 || jqxhr.status === 403;
    });
}

// bootstrap plugin loader
hawtioPluginLoader.addUrl(url("/plugin"));

if (jolokiaUrl) {
    // TODO replace with a jolokia call so we use authentication headers
    //hawtioPluginLoader.addUrl("jolokia:" + jolokiaUrl + ":hawtio:type=plugin,name=*");
}

/*
interface IMyAppScope extends ng.IRootScopeService, ng.IScope {
lineCount: (value:any) => number;
params: ng.IRouteParamsService;
is: (type:any, value:any) => boolean;
empty: (value:any) => boolean;
log: (variable:string) => void;
alert: (text:string) => void;
}
*/
hawtioPluginLoader.addModule(Core.pluginName);

var hawtioCoreModule = angular.module(Core.pluginName, ['bootstrap', 'ngResource', 'ui', 'ui.bootstrap.dialog', 'hawtio-ui']).config(function ($routeProvider, $dialogProvider) {
    $dialogProvider.options({
        backdropFade: true,
        dialogFade: true
    });

    $routeProvider.when('/login', { templateUrl: 'app/core/html/login.html' }).when('/welcome', { templateUrl: 'app/core/html/welcome.html' }).when('/preferences', { templateUrl: 'app/core/html/preferences.html' }).when('/about', { templateUrl: 'app/core/html/about.html' }).when('/help', {
        redirectTo: '/help/index'
    }).when('/help/:topic/', { templateUrl: 'app/core/html/help.html' }).when('/help/:topic/:subtopic', { templateUrl: 'app/core/html/help.html' }).otherwise({ redirectTo: '/perspective/defaultPage' });
}).constant('layoutTree', 'app/core/html/layoutTree.html').constant('layoutFull', 'app/core/html/layoutFull.html').service('localStorage', function () {
    // TODO Create correct implementation of windowLocalStorage
    var storage = window.localStorage || (function () {
        return {};
    })();
    return storage;
}).factory('marked', function () {
    marked.setOptions({
        gfm: true,
        tables: true,
        breaks: false,
        pedantic: true,
        sanitize: false,
        smartLists: true,
        langPrefix: 'language-'
    });
    return marked;
}).factory('pageTitle', function () {
    var answer = new Core.PageTitle();
    return answer;
}).factory('viewRegistry', function () {
    return {};
}).factory('lastLocation', function () {
    return {};
}).factory('helpRegistry', function ($rootScope) {
    return new Core.HelpRegistry($rootScope);
}).factory('jolokiaUrl', function () {
    return jolokiaUrl;
}).factory('jolokiaStatus', function () {
    return {
        xhr: null
    };
}).factory('jolokiaParams', function (jolokiaUrl) {
    return {
        url: jolokiaUrl,
        canonicalNaming: false,
        ignoreErrors: true,
        mimeType: 'application/json'
    };
}).factory('branding', function () {
    return {
        appName: 'hawtio',
        appLogo: 'img/logo-16px.png',
        loginBg: 'img/fire.jpg',
        fullscreenLogin: false
    };
}).factory('userDetails', function (jolokiaUrl, localStorage) {
    var answer = angular.fromJson(localStorage[jolokiaUrl]);
    if (!angular.isDefined(answer) && jolokiaUrl) {
        answer = {
            username: '',
            password: ''
        };

        Core.log.debug("No username set, checking if we have a session");

        // fetch the username if we've already got a session at the server
        var userUrl = jolokiaUrl.replace("jolokia", "user");
        $.ajax(userUrl, {
            type: "GET",
            success: function (response) {
                Core.log.debug("Got user response: ", response);

                // We'll only touch these if they're not set
                if (response !== '' && response !== null) {
                    answer.username = response;
                    if (!('loginDetails' in answer)) {
                        answer['loginDetails'] = {};
                    }
                }
            },
            error: function (xhr, textStatus, error) {
                Core.log.debug("Failed to get session username: ", error);
                // silently ignore, we could be using the proxy
            }
        });

        return answer;
    } else {
        return answer;
    }
}).factory('jolokia', function ($location, localStorage, jolokiaStatus, $rootScope, userDetails, jolokiaParams) {
    // TODO - Maybe have separate URLs or even jolokia instances for loading plugins vs. application stuff
    // var jolokiaUrl = $location.search()['url'] || url("/jolokia");
    console.log("Jolokia URL is " + jolokiaUrl);
    if (jolokiaUrl) {
        var credentials = hawtioPluginLoader.getCredentials(jolokiaUrl);

        // pass basic auth credentials down to jolokia if set
        var username = null;
        var password = null;

        //var userDetails = angular.fromJson(localStorage[jolokiaUrl]);
        if (credentials.length === 2) {
            username = credentials[0];
            password = credentials[1];
            // TODO we should try avoid both permutations of username / userName :)
        } else if (angular.isDefined(userDetails) && angular.isDefined(userDetails.username) && angular.isDefined(userDetails.password)) {
            username = userDetails.username;
            password = userDetails.password;
        } else if (angular.isDefined(userDetails) && angular.isDefined(userDetails.userName) && angular.isDefined(userDetails.password)) {
            username = userDetails.userName;
            password = userDetails.password;
        } else {
            // lets see if they are passed in via request parameter...
            var search = hawtioPluginLoader.parseQueryString();
            username = search["_user"];
            password = search["_pwd"];
            if (angular.isArray(username))
                username = username[0];
            if (angular.isArray(password))
                password = password[0];
        }

        if (username && password) {
            /*
            TODO can't use this, sets the username/password in the URL on every request, plus jolokia passes them on to $.ajax() which causes a fatal exception in firefox
            jolokiaParams['username'] = username;
            jolokiaParams['password'] = password;
            */
            //console.log("Using user / pwd " + username + " / " + password);
            userDetails.username = username;
            userDetails.password = password;

            $.ajaxSetup({
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('Authorization', Core.getBasicAuthHeader(userDetails.username, userDetails.password));
                }
            });

            var loginUrl = jolokiaUrl.replace("jolokia", "auth/login/");
            $.ajax(loginUrl, {
                type: "POST",
                success: function (response) {
                    Core.log.debug("Response from silent login: ", response);
                    if (angular.isDefined(response['credentials'] || angular.isDefined(response['principals']))) {
                        userDetails.loginDetails = response;
                    } else {
                        var doc = Core.pathGet(response, ['children', 0, 'innerHTML']);

                        // hmm, maybe we got an XML document, let's log it just in case...
                        if (doc) {
                            Core.log.debug("Response is a document (ignoring this): ", doc);
                        }
                    }
                },
                error: function (xhr, textStatus, error) {
                    // silently ignore, we could be using the proxy
                }
            });
        }

        jolokiaParams['ajaxError'] = function (xhr, textStatus, error) {
            if (xhr.status === 401 || xhr.status === 403) {
                userDetails.username = null;
                userDetails.password = null;
            } else {
                jolokiaStatus.xhr = xhr;
                if (!xhr.responseText && error) {
                    xhr.responseText = error.stack;
                }
            }
            Core.$apply($rootScope);
        };

        var jolokia = new Jolokia(jolokiaParams);
        localStorage['url'] = jolokiaUrl;
        jolokia.stop();
        return jolokia;
    } else {
        // empty jolokia that returns nothing
        return {
            request: function () {
                return null;
            },
            register: function () {
                return null;
            },
            list: function () {
                return null;
            },
            search: function () {
                return null;
            },
            read: function () {
                return null;
            },
            execute: function () {
                return null;
            },
            start: function () {
                _this.running = true;
                return null;
            },
            stop: function () {
                _this.running = false;
                return null;
            },
            isRunning: function () {
                return _this.running;
            },
            jobs: function () {
                return [];
            }
        };
    }
}).factory('toastr', function () {
    var win = window;
    var answer = win.toastr;
    if (!answer) {
        // lets avoid any NPEs
        answer = {};
        win.toaster = answer;
    }
    return answer;
}).factory('xml2json', function ($window) {
    var jquery = $;
    return jquery.xml2json;
}).factory('workspace', function ($location, jmxTreeLazyLoadRegistry, $compile, $templateCache, localStorage, jolokia, jolokiaStatus, $rootScope, userDetails) {
    var answer = new Workspace(jolokia, jolokiaStatus, jmxTreeLazyLoadRegistry, $location, $compile, $templateCache, localStorage, $rootScope, userDetails);
    answer.loadTree();
    return answer;
}).filter("valueToHtml", function () {
    return Core.valueToHtml;
}).filter('humanize', function () {
    return humanizeValue;
}).filter('humanizeMs', function () {
    return Core.humanizeMilliseconds;
}).directive('autofill', [
    '$timeout', function ($timeout) {
        return {
            restrict: "A",
            require: 'ngModel',
            link: function (scope, elem, attrs, ctrl) {
                var ngModel = attrs["ngModel"];
                if (ngModel) {
                    var log = Logger.get("Core");

                    function checkForDifference() {
                        // lets compare the current DOM node value with the model
                        // in case we can default it ourselves
                        var modelValue = scope.$eval(ngModel);
                        var value = elem.val();
                        if (value && !modelValue) {
                            Core.pathSet(scope, ngModel, value);
                            //log.info("autofill: Updated ngModel: " + ngModel + " original model value: " + modelValue + " UI value: " + value + " new value: " + scope.$eval(ngModel));
                        } else {
                            //log.info("Got invoked with ngModel: " + ngModel + " modelValue: " + modelValue + " value: " + value);
                            // lets try trigger input/change events just in case
                            // try both approaches just in case one doesn't work ;)
                            elem.trigger('input');
                            elem.trigger('change');
                            if (elem.length) {
                                var firstElem = $(elem[0]);
                                firstElem.trigger('input');
                                firstElem.trigger('change');
                            }
                        }
                    }

                    $timeout(checkForDifference, 200);
                    $timeout(checkForDifference, 800);
                    $timeout(checkForDifference, 1500);
                }
            }
        };
    }]).run(function ($rootScope, $routeParams, jolokia, workspace, localStorage, viewRegistry, layoutFull, helpRegistry, pageTitle, branding, toastr, userDetails) {
    $.support.cors = true;

    /*
    * Count the number of lines in the given text
    */
    $rootScope.lineCount = lineCount;

    /*
    * Easy access to route params
    */
    $rootScope.params = $routeParams;

    /*
    * Wrapper for angular.isArray, isObject, etc checks for use in the view
    *
    * @param type {string} the name of the check (casing sensitive)
    * @param value {string} value to check
    */
    $rootScope.is = function (type, value) {
        return angular['is' + type](value);
    };

    /*
    * Wrapper for $.isEmptyObject()
    *
    * @param value  {mixed} Value to be tested
    * @return booleanean
    */
    $rootScope.empty = function (value) {
        return $.isEmptyObject(value);
    };

    /*
    * Initialize jolokia polling and add handler to change poll
    * frequency
    */
    $rootScope.$on('UpdateRate', function (event, rate) {
        jolokia.stop();
        if (rate > 0) {
            jolokia.start(rate);
        }
        Core.log.debug("Set update rate to: ", rate);
    });

    $rootScope.$emit('UpdateRate', localStorage['updateRate']);

    /*
    * Debugging Tools
    *
    * Allows you to execute debug functions from the view
    */
    // TODO Doesn't support vargs like it should
    $rootScope.log = function (variable) {
        console.log(variable);
    };
    $rootScope.alert = function (text) {
        alert(text);
    };

    viewRegistry['fullscreen'] = layoutFull;
    viewRegistry['notree'] = layoutFull;
    viewRegistry['help'] = layoutFull;
    viewRegistry['welcome'] = layoutFull;
    viewRegistry['preferences'] = layoutFull;
    viewRegistry['about'] = layoutFull;
    viewRegistry['login'] = layoutFull;

    helpRegistry.addUserDoc('index', 'app/core/doc/overview.md');
    helpRegistry.addUserDoc('preference', 'app/core/doc/preference.md');
    helpRegistry.addSubTopic('index', 'faq', 'app/core/doc/FAQ.md');
    helpRegistry.addSubTopic('index', 'changes', 'app/core/doc/CHANGES.md');
    helpRegistry.addSubTopic('index', 'developer', 'app/core/doc/developer.md');
    helpRegistry.addDevDoc('Core', 'app/core/doc/coreDeveloper.md');

    //helpRegistry.discoverHelpFiles(hawtioPluginLoader.getModules());
    var opts = localStorage['CodeMirrorOptions'];
    if (opts) {
        opts = angular.fromJson(opts);
        CodeEditor.GlobalCodeMirrorOptions = angular.extend(CodeEditor.GlobalCodeMirrorOptions, opts);
    }

    toastr.options = {
        'closeButton': true,
        'showMethod': 'slideDown',
        'hideMethod': 'slideUp'
    };

    window['logInterceptors'].push(function (level, message) {
        if (level === "WARN") {
            notification('warning', message);
        }
        if (level === "ERROR") {
            notification('error', message);
        }
    });

    setTimeout(function () {
        $("#main-body").fadeIn(2000).after(function () {
            Core.log.info(branding.appName + " started");
            Core.$apply($rootScope);
            $(window).trigger('resize');
        });
    }, 500);
}).directive('compile', [
    '$compile', function ($compile) {
        return function (scope, element, attrs) {
            scope.$watch(function (scope) {
                // watch the 'compile' expression for changes
                return scope.$eval(attrs.compile);
            }, function (value) {
                // when the 'compile' expression changes
                // assign it into the current DOM
                element.html(value);

                // compile the new DOM and link it to the current
                // scope.
                // NOTE: we only compile .childNodes so that
                // we don't get into infinite loop compiling ourselves
                $compile(element.contents())(scope);
            });
        };
    }]).directive('noClick', function () {
    return function ($scope, $element, $attrs) {
        $element.click(function (event) {
            event.preventDefault();
        });
    };
}).directive('gridStyle', function ($window) {
    return new Core.GridStyle($window);
}).directive('logToggler', function (localStorage) {
    return {
        restrict: 'A',
        link: function ($scope, $element, $attr) {
            $element.click(function () {
                var log = $("#log-panel");
                var body = $('body');
                if (log.height() !== 0) {
                    localStorage['showLog'] = 'false';
                    log.css({ 'bottom': '110%' });
                    body.css({
                        'overflow-y': 'auto'
                    });
                } else {
                    localStorage['showLog'] = 'true';
                    log.css({ 'bottom': '50%' });
                    body.css({
                        'overflow-y': 'hidden'
                    });
                }
                return false;
            });
        }
    };
});

// for chrome packaged apps lets enable chrome-extension pages
if (hawtioCoreModule && Core.isChromeApp()) {
    hawtioCoreModule.config([
        '$compileProvider',
        function ($compileProvider) {
            //$compileProvider.aHrefSanitizationWhitelist(/^\s*(https?|ftp|mailto|chrome-extension):/);
            $compileProvider.urlSanitizationWhitelist(/^\s*(https?|ftp|mailto|chrome-extension):/);
            // Angular before v1.2 uses $compileProvider.urlSanitizationWhitelist(...)
        }
    ]);
}

// enable bootstrap tooltips
$(function () {
    $("a[title]").tooltip({
        selector: '',
        delay: { show: 1000, hide: 100 }
    });
});

var adjustHeight = function () {
    var windowHeight = $(window).height();
    var headerHeight = $("#main-nav").height();
    var containerHeight = windowHeight - headerHeight;
    $("#main").css("min-height", "" + containerHeight + "px");
};

$(function () {
    hawtioPluginLoader.loadPlugins(function () {
        var doc = $(document);
        angular.bootstrap(doc, hawtioPluginLoader.getModules());
        $(document.documentElement).attr('xmlns:ng', "http://angularjs.org");
        $(document.documentElement).attr('ng-app', 'hawtioCore');
        adjustHeight();
        $(window).resize(adjustHeight);
    });
});
/**
* @module Core
*/
var Core;
(function (Core) {
    function AboutController($scope, $location, jolokia, branding, localStorage) {
        var log = Logger.get("About");

        // load the about.md file
        $.ajax({
            url: "app/core/doc/about.md",
            dataType: 'html',
            cache: false,
            success: function (data, textStatus, jqXHR) {
                $scope.html = "Unable to download about.md";
                if (angular.isDefined(data)) {
                    $scope.html = marked(data);
                    $scope.branding = branding;
                    $scope.customBranding = Branding.enabled;
                    try  {
                        $scope.hawtioVersion = jolokia.request({
                            type: "read",
                            mbean: "hawtio:type=About",
                            attribute: "HawtioVersion"
                        }).value;
                    } catch (Error) {
                        // ignore
                        $scope.hawtioVersion = "N/A";
                    }
                    $scope.jolokiaVersion = jolokia.version().agent;
                    $scope.serverProduct = jolokia.version().info.product;
                    $scope.serverVendor = jolokia.version().info.vendor;
                    $scope.serverVersion = jolokia.version().info.version;
                }
                Core.$apply($scope);
            },
            error: function (jqXHR, textStatus, errorThrown) {
                $scope.html = "Unable to download about.md";
                Core.$apply($scope);
            }
        });
    }
    Core.AboutController = AboutController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    

    /**
    * @class Folder
    * @uses NodeSelection
    */
    var Folder = (function () {
        function Folder(title) {
            this.title = title;
            this.key = null;
            this.typeName = null;
            this.children = [];
            this.folderNames = [];
            this.domain = null;
            this.objectName = null;
            this.map = {};
            this.entries = {};
            this.addClass = null;
            this.parent = null;
            this.isLazy = false;
            this.icon = null;
            this.tooltip = null;
            this.entity = null;
            this.addClass = escapeTreeCssStyles(title);
        }
        Folder.prototype.get = function (key) {
            return this.map[key];
        };

        Folder.prototype.isFolder = function () {
            return this.children.length > 0;
        };

        /**
        * Navigates the given paths and returns the value there or null if no value could be found
        * @method navigate
        * @for Folder
        * @param {Array} paths
        * @return {NodeSelection}
        */
        Folder.prototype.navigate = function () {
            var paths = [];
            for (var _i = 0; _i < (arguments.length - 0); _i++) {
                paths[_i] = arguments[_i + 0];
            }
            var node = this;
            paths.forEach(function (path) {
                if (node) {
                    node = node.get(path);
                }
            });
            return node;
        };

        Folder.prototype.hasEntry = function (key, value) {
            var entries = this.entries;
            if (entries) {
                var actual = entries[key];
                return actual && value === actual;
            }
            return false;
        };

        Folder.prototype.parentHasEntry = function (key, value) {
            if (this.parent) {
                return this.parent.hasEntry(key, value);
            }
            return false;
        };

        Folder.prototype.ancestorHasEntry = function (key, value) {
            var parent = this.parent;
            while (parent) {
                if (parent.hasEntry(key, value))
                    return true;
                parent = parent.parent;
            }
            return false;
        };

        Folder.prototype.ancestorHasType = function (typeName) {
            var parent = this.parent;
            while (parent) {
                if (typeName === parent.typeName)
                    return true;
                parent = parent.parent;
            }
            return false;
        };

        Folder.prototype.getOrElse = function (key, defaultValue) {
            if (typeof defaultValue === "undefined") { defaultValue = new Folder(key); }
            var answer = this.map[key];
            if (!answer) {
                answer = defaultValue;
                this.map[key] = answer;
                this.children.push(answer);
                answer.parent = this;
                this.children = this.children.sortBy("title");
            }
            return answer;
        };

        Folder.prototype.sortChildren = function (recursive) {
            var children = this.children;
            if (children) {
                this.children = children.sortBy("title");
                if (recursive) {
                    angular.forEach(children, function (child) {
                        return child.sortChildren(recursive);
                    });
                }
            }
        };

        Folder.prototype.moveChild = function (child) {
            if (child && child.parent !== this) {
                child.detach();
                child.parent = this;
                this.children.push(child);
            }
        };

        /**
        * Removes this node from my parent if I have one
        * @method detach
        * @for Folder
        \   */
        Folder.prototype.detach = function () {
            var oldParent = this.parent;
            if (oldParent) {
                var oldParentChildren = oldParent.children;
                if (oldParentChildren) {
                    var idx = oldParentChildren.indexOf(this);
                    if (idx < 0) {
                        oldParent.children = oldParent.children.remove({ key: this.key });
                    } else {
                        oldParentChildren.splice(idx, 1);
                    }
                }
                this.parent = null;
            }
        };

        /**
        * Searches this folder and all its descendants for the first folder to match the filter
        * @method findDescendant
        * @for Folder
        * @param {Function} filter
        * @return {Folder}
        */
        Folder.prototype.findDescendant = function (filter) {
            if (filter(this)) {
                return this;
            }
            var answer = null;
            angular.forEach(this.children, function (child) {
                if (!answer) {
                    answer = child.findDescendant(filter);
                }
            });
            return answer;
        };
        return Folder;
    })();
    Core.Folder = Folder;
})(Core || (Core = {}));

;
var Folder = (function (_super) {
    __extends(Folder, _super);
    function Folder() {
        _super.apply(this, arguments);
    }
    return Folder;
})(Core.Folder);
;
/**
* @module Perspective
*/
var Perspective;
(function (Perspective) {
    Perspective.log = Logger.get("Perspective");

    /**
    * The location search parameter for specifying the perspective to view
    * @property perspectiveSearchId
    * @for Perspective
    * @type String
    */
    Perspective.perspectiveSearchId = "p";

    /**
    * Lets you specify which perspective to default to if there's not a single active one
    * @property defaultPerspective
    * @for Perspective
    * @type String
    */
    Perspective.defaultPerspective = null;

    /**
    * A hook so folks can specify the default start page explicitly if the first valid page in the
    * perspective is not the intended start page
    * @property defaultPageLocation
    * @for Perspective
    * @type String
    */
    Perspective.defaultPageLocation = null;

    /**
    * Returns the current perspective ID based on the query parameter or the current
    * discovered perspective
    * @method currentPerspectiveId
    * @for Perspective
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    * @paran {*} jolokia
    * @param {any} localStorage
    * @return {String}
    */
    function currentPerspectiveId($location, workspace, jolokia, localStorage) {
        var perspective = $location.search()[Perspective.perspectiveSearchId];
        if (!perspective) {
            perspective = Perspective.choosePerspective($location, workspace, jolokia, localStorage);
        }
        return perspective;
    }
    Perspective.currentPerspectiveId = currentPerspectiveId;

    /**
    * Returns an array of all the active perspectives
    * @method getPerspectives
    * @for Perspective
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    * @paran {*} jolokia
    * @param {any} localStorage
    */
    function getPerspectives($location, workspace, jolokia, localStorage) {
        var perspectives = [];
        angular.forEach(Perspective.metadata, function (perspective, key) {
            if (isValidFunction(workspace, perspective.isValid)) {
                if (!perspective.label) {
                    perspective.label = key;
                }
                if (!perspective.title) {
                    perspective.title = perspective.label;
                }
                perspective.id = key;
                perspectives.push(perspective);
            }
        });
        return perspectives;
    }
    Perspective.getPerspectives = getPerspectives;

    function getPerspectiveById(id) {
        var answer;
        angular.forEach(Perspective.metadata, function (perspective, key) {
            if (key === id) {
                answer = perspective;
            }
        });
        return answer;
    }
    Perspective.getPerspectiveById = getPerspectiveById;

    /**
    * Returns the top level tabs for the given perspectiveId
    * @method topLevelTabsForPerspectiveId
    * @for Perspective
    * @param {Core.Workspace} workspace
    * @param {String} perspective
    * @return {Array}
    */
    function topLevelTabsForPerspectiveId(workspace, perspective) {
        var data = perspective ? Perspective.metadata[perspective] : null;
        var metaData = data;
        var answer = [];
        if (!data) {
            answer = workspace.topLevelTabs;
        } else {
            // lets iterate through the available tabs in the perspective
            var topLevelTabs = data.topLevelTabs;

            var includes = filterTabs(topLevelTabs.includes, workspace);
            var excludes = filterTabs(topLevelTabs.excludes, workspace);

            // now do extra filtering of excludes, if they have some conditions in the meta data
            if (metaData) {
                excludes = excludes.filter(function (t) {
                    var metaTab = metaData.topLevelTabs.excludes.find(function (et) {
                        var etid = et.id;
                        return etid && etid === t.id;
                    });
                    if (metaTab != null && angular.isFunction(metaTab.onCondition)) {
                        // not all tabs has on condition function, so use try .. catch
                        var answer = metaTab.onCondition(workspace);
                        if (answer) {
                            Perspective.log.debug("Plugin " + t.id + " excluded in perspective " + perspective);
                            return true;
                        } else {
                            // the condition was false, so it does not apply
                            return false;
                        }
                    }
                    return true;
                });
            }

            // if the meta-data only had excludes, then it means all the top level tabs, excluding these
            if (!topLevelTabs.includes) {
                // lets exclude the excluded tabs
                answer = workspace.topLevelTabs;
            } else {
                // if the meta-data had includes, then its only these
                answer = includes;
            }

            // and remove any excludes
            answer = answer.subtract(excludes);
        }
        return answer;
    }
    Perspective.topLevelTabsForPerspectiveId = topLevelTabsForPerspectiveId;

    function filterTabs(tabs, workspace) {
        var matched = [];
        angular.forEach(tabs, function (tabSpec) {
            var href = tabSpec.href;
            var id = tabSpec.id;
            var rhref = tabSpec.rhref;
            if (href) {
                var hrefValue = href;
                if (angular.isFunction(href)) {
                    hrefValue = href();
                }
                var tab = workspace.topLevelTabs.find(function (t) {
                    var thref = t.href();
                    return thref && thref.startsWith(hrefValue);
                });
                if (!tab && !id && tabSpec.content) {
                    // lets assume the tab is the tabSpec
                    tab = tabSpec;
                }
                if (tab) {
                    matched.push(tab);
                }
            } else if (id) {
                var tab = workspace.topLevelTabs.find(function (t) {
                    var tid = t.id;
                    return tid && tid === id;
                });
                if (tab) {
                    matched.push(tab);
                }
            } else if (rhref) {
                var tab = workspace.topLevelTabs.find(function (t) {
                    var thref = t.href();
                    return thref && thref.match(rhref);
                });
                if (tab) {
                    matched.push(tab);
                }
            }
        });
        return matched;
    }

    /**
    * Filter the top level tabs to only include currently valid tabs.
    */
    function filterOnlyValidTopLevelTabs(workspace, topLevelTabs) {
        var answer = topLevelTabs.filter(function (tab) {
            var href = tab.href();
            return href && isValidFunction(workspace, tab.isValid);
        });
        return answer;
    }
    Perspective.filterOnlyValidTopLevelTabs = filterOnlyValidTopLevelTabs;

    /**
    * Filter the top level tabs to only include currently active tabs.
    */
    function filterOnlyActiveTopLevelTabs(workspace, topLevelTabs) {
        var answer = topLevelTabs.filter(function (tab) {
            var href = tab.href();
            return href && isValidFunction(workspace, tab.isActive);
        });
        return answer;
    }
    Perspective.filterOnlyActiveTopLevelTabs = filterOnlyActiveTopLevelTabs;

    /**
    * Returns the top level tabs for the given perspective (which are valid)
    * @method topLevelTabs
    * @for Perspective
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    * @paran {*} jolokia
    * @param {any} localStorage
    * @return {Array}
    */
    function getTopLevelTabsForPerspective($location, workspace, jolokia, localStorage) {
        var perspective = currentPerspectiveId($location, workspace, jolokia, localStorage);

        var plugins = Core.configuredPluginsForPerspectiveId(perspective, workspace, jolokia, localStorage);
        var tabs = Core.filterTopLevelTabs(perspective, workspace, plugins);
        tabs = Perspective.filterOnlyValidTopLevelTabs(workspace, tabs);

        return tabs;
    }
    Perspective.getTopLevelTabsForPerspective = getTopLevelTabsForPerspective;

    /**
    * Returns the perspective we should be using right now since none is specified
    * @method choosePerspective
    * @for Perspective
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    * @paran {*} jolokia
    * @param {any} localStorage
    * @return {String}
    */
    function choosePerspective($location, workspace, jolokia, localStorage) {
        var answer;

        var inFMC = Fabric.isFMCContainer(workspace);
        if (inFMC) {
            var url = $location.url();

            // noisy!
            //log.debug("Checking url: ", url);
            // we want first time users on welcome/index/default page to be in the fabric perspective
            if (url.startsWith("/perspective/defaultPage") || url.startsWith("/login") || url.startsWith("/welcome") || url.startsWith("/index") || url.startsWith("/fabric") || url.startsWith("/dashboard") || url.startsWith("/health") || (url.startsWith("/wiki") && url.has("/fabric/profiles")) || (url.startsWith("/wiki") && url.has("/editFeatures"))) {
                answer = "fabric";
            }
        }
        answer = answer || Perspective.defaultPerspective || "container";

        Perspective.log.debug("Choose perspective url: " + $location.url() + ", in fabric: " + inFMC + " -> " + answer);
        return answer;
    }
    Perspective.choosePerspective = choosePerspective;

    /**
    * Returns the default page after figuring out what the current perspective is
    * @method defaultPage
    * @for Perspective
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    * @paran {*} jolokia
    * @param {any} localStorage
    * @return {String}
    */
    function defaultPage($location, workspace, jolokia, localStorage) {
        if (shouldShowWelcomePage(localStorage) && !Core.isChromeApp()) {
            return "/welcome/";
        }

        // now find the configured default plugin, and then find the top level tab that matches the default plugin
        var answer = Perspective.defaultPageLocation;
        if (!answer && $location && workspace) {
            var perspectiveId = currentPerspectiveId($location, workspace, jolokia, localStorage);
            var defaultPlugin = Core.getDefaultPlugin(perspectiveId, workspace, jolokia, localStorage);
            var tabs = Perspective.topLevelTabsForPerspectiveId(workspace, perspectiveId);
            tabs = Perspective.filterOnlyValidTopLevelTabs(workspace, tabs);

            var defaultTab;
            if (defaultPlugin) {
                tabs.forEach(function (tab) {
                    if (tab.id === defaultPlugin.id) {
                        defaultTab = tab;
                    }
                });
            } else {
                // if no default plugin configured, then select the 1st tab as default
                defaultTab = tabs[0];
            }

            if (defaultTab) {
                // clip the href to get the path to the plugin
                answer = Core.trimLeading(defaultTab.href(), "#");
            }
        }

        return answer || '/help/index';
    }
    Perspective.defaultPage = defaultPage;

    /**
    * Whether to show the welcome page
    */
    function shouldShowWelcomePage(localStorage) {
        var value = localStorage["showWelcomePage"];
        if (angular.isString(value)) {
            return "true" === value;
        }
        return true;
    }
    Perspective.shouldShowWelcomePage = shouldShowWelcomePage;

    /**
    * Returns true if there is no validFn defined or if its defined
    * then the function returns true.
    *
    * @method isValidFunction
    * @for Perspective
    * @param {Core.Workspace} workspace
    * @param {Function} validFn
    * @return {Boolean}
    */
    function isValidFunction(workspace, validFn) {
        return !validFn || validFn(workspace);
    }
})(Perspective || (Perspective = {}));
/**
* @module Perspective
*/
var Perspective;
(function (Perspective) {
    /**
    * redirects the browser to the default page based on the detected profiles
    * @method DefaultPageController
    * @for Perspective
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {any} localStorage
    * @param {Core.Workspace} workspace
    */
    function DefaultPageController($scope, $location, localStorage, workspace, jolokia) {
        var url = Perspective.defaultPage($location, workspace, jolokia, localStorage);
        var path = Core.trimLeading(url, "#");
        if (path) {
            console.log("redirecting to default page: " + path);
            $location.url(path);
        } else {
            console.log("No default page could be chosen!");
        }
    }
    Perspective.DefaultPageController = DefaultPageController;
})(Perspective || (Perspective = {}));
/**
* @module Perspective
*/
var Perspective;
(function (Perspective) {
    Perspective.containerPerspectiveEnabled = true;

    /**
    * Configuration for the perspective plugin that defines what tabs are in which perspectives
    * @property metadata
    * @for Perspective
    * @type {any}
    */
    Perspective.metadata = {
        fabric: {
            label: "Fabric",
            isValid: function (workspace) {
                return Fabric.isFMCContainer(workspace);
            },
            lastPage: "#/fabric/containers",
            topLevelTabs: {
                includes: [
                    {
                        href: "#/fabric"
                    },
                    {
                        href: "#/wiki/branch/"
                    },
                    {
                        href: "#/wiki/profile"
                    },
                    {
                        href: "#/dashboard"
                    },
                    {
                        href: "#/health"
                    },
                    {
                        id: "fabric.insight"
                    }
                ]
            }
        },
        insight: {
            label: "Insight",
            isValid: function (workspace) {
                return Insight.hasInsight(workspace);
            },
            topLevelTabs: {
                includes: [
                    {
                        href: "#/kibanalogs"
                    },
                    {
                        href: "#/insight"
                    },
                    {
                        href: "#/kibanacamel"
                    },
                    {
                        href: "#/camin"
                    },
                    {
                        href: "#/eshead"
                    }
                ]
            }
        },
        container: {
            label: "Container",
            lastPage: "#/logs",
            isValid: function (workspace) {
                return workspace && workspace.tree && workspace.tree.children && workspace.tree.children.length;
            },
            topLevelTabs: {
                excludes: [
                    {
                        href: "#/fabric"
                    },
                    {
                        href: "#/insight"
                    },
                    {
                        href: "#/camin"
                    },
                    {
                        href: "#/kibanalogs"
                    },
                    {
                        href: "#/kibanacamel"
                    },
                    {
                        href: "#/eshead"
                    },
                    {
                        id: "dashboard",
                        // we only want to exclude dashboard if we are running in fabric (as they are in another perspective)
                        // (must use "id" attribute for the plugin, an not href, when using onCondition)
                        onCondition: function (workspace) {
                            return Fabric.isFMCContainer(workspace);
                        }
                    },
                    {
                        id: "health",
                        // we only want to exclude health if we are running in fabric (as they are in another perspective)
                        // (must use "id" attribute for the plugin, an not href, when using onCondition)
                        onCondition: function (workspace) {
                            return Fabric.isFMCContainer(workspace);
                        }
                    }
                ]
            }
        },
        limited: {
            label: "Limited",
            lastPage: "#/logs",
            isValid: function (workspace) {
                return false;
            },
            topLevelTabs: {
                includes: [
                    {
                        href: "#/jmx"
                    },
                    {
                        href: "#/camel"
                    },
                    {
                        href: "#/activemq"
                    },
                    {
                        href: "#/jetty"
                    },
                    {
                        href: "#/logs"
                    }
                ]
            }
        },
        website: {
            label: "WebSite",
            isValid: function (workspace) {
                return Site.sitePluginEnabled && Site.isSiteNavBarValid();
            },
            lastPage: "#/site/doc/index.md",
            topLevelTabs: {
                includes: [
                    {
                        content: "Get Started",
                        title: "How to get started using hawtio",
                        href: function () {
                            return "#/site/doc/GetStarted.md";
                        },
                        isValid: function () {
                            return Site.isSiteNavBarValid();
                        }
                    },
                    {
                        content: "FAQ",
                        title: "Frequently Asked Questions",
                        href: function () {
                            return "#/site/FAQ.md";
                        },
                        isValid: function () {
                            return Site.isSiteNavBarValid();
                        }
                    },
                    {
                        content: "User Guide",
                        title: "All the docs on using hawtio",
                        href: function () {
                            return "#/site/book/doc/index.md";
                        },
                        isValid: function () {
                            return Site.isSiteNavBarValid();
                        }
                    },
                    {
                        content: "Community",
                        title: "Come on in and join our community!",
                        href: function () {
                            return "#/site/doc/Community.html";
                        },
                        isValid: function () {
                            return Site.isSiteNavBarValid();
                        }
                    },
                    {
                        content: "Developers",
                        title: "Resources for developers if you want to hack on hawtio or provide your own plugins",
                        href: function () {
                            return "#/site/doc/developers/index.md";
                        },
                        isValid: function () {
                            return Site.isSiteNavBarValid();
                        }
                    },
                    {
                        content: "github",
                        title: "Hawtio's source code and issue tracker",
                        href: function () {
                            return "https://github.com/hawtio/hawtio";
                        },
                        isValid: function () {
                            return Site.isSiteNavBarValid();
                        }
                    }
                ]
            }
        }
    };
})(Perspective || (Perspective = {}));
/**
* @module Perspective
* @main Perspective
*/
var Perspective;
(function (Perspective) {
    var pluginName = 'perspective';
    angular.module(pluginName, ['hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/perspective/defaultPage', {
            templateUrl: 'app/perspective/html/defaultPage.html',
            controller: Perspective.DefaultPageController });
    }).run(function ($location, workspace, viewRegistry, layoutFull) {
        viewRegistry['perspective'] = layoutFull;
    });

    hawtioPluginLoader.addModule(pluginName);
})(Perspective || (Perspective = {}));
/**
* IDE integration
*
* @module IDE
*/
var IDE;
(function (IDE) {
    var pluginName = 'ide';

    angular.module(pluginName, ['bootstrap', 'hawtioCore']).directive('hawtioOpenIde', function (localStorage, workspace, jolokia) {
        return new IDE.OpenInIdeDirective(localStorage, workspace, jolokia);
    }).run(function (helpRegistry) {
        helpRegistry.addDevDoc('IDE', 'app/ide/doc/developer.md');
    });

    hawtioPluginLoader.addModule(pluginName);
})(IDE || (IDE = {}));
var IDE;
(function (IDE) {
    var log = Logger.get("IDE");

    var OpenInIdeDirective = (function () {
        function OpenInIdeDirective(localStorage, workspace, jolokia) {
            var _this = this;
            this.localStorage = localStorage;
            this.workspace = workspace;
            this.jolokia = jolokia;
            this.restrict = 'E';
            this.replace = true;
            this.transclude = false;
            this.scope = {
                fileName: '@',
                className: '@',
                line: '@',
                column: '@'
            };
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        OpenInIdeDirective.prototype.doLink = function ($scope, $element, $attrs) {
            var workspace = this.workspace;
            var jolokia = this.jolokia;

            var mbean = IDE.getIdeMBean(workspace);
            var fileName = $scope.fileName;
            if (mbean && fileName) {
                var className = $scope.className;
                var line = $scope.line;
                var col = $scope.col;
                if (!angular.isDefined(line) || line === null)
                    line = 0;
                if (!angular.isDefined(col) || col === null)
                    col = 0;

                if (IDE.isOpenInIdeaSupported(workspace, localStorage)) {
                    var ideaButton = $('<button class="btn btn-mini"><img src="app/ide/img/intellijidea.png" width="16" height="16"></button>');

                    function onResult(absoluteName) {
                        if (!absoluteName) {
                            log.info("Could not find file in source code: " + fileName + " class: " + className);
                            ideaButton.attr("title", "Could not find source file: " + fileName);
                        } else {
                            ideaButton.attr("title", "Opening in IDEA: " + absoluteName);
                            IDE.ideaOpenAndNavigate(mbean, jolokia, absoluteName, line, col);
                        }
                    }

                    ideaButton.on("click", function () {
                        log.info("Finding local file name: " + fileName + " className: " + className);
                        IDE.findClassAbsoluteFileName(mbean, jolokia, localStorage, fileName, className, onResult);
                    });
                    $element.append(ideaButton);
                }
            }
        };
        return OpenInIdeDirective;
    })();
    IDE.OpenInIdeDirective = OpenInIdeDirective;
})(IDE || (IDE = {}));
/**
* @module IDE
*/
var IDE;
(function (IDE) {
    /**
    * Returns the mbean name of the IDE Facade mbean if its available
    */
    function getIdeMBean(workspace) {
        return Core.getMBeanTypeObjectName(workspace, "hawtio", "IdeFacade");
    }
    IDE.getIdeMBean = getIdeMBean;

    /**
    * Returns true if open in IDEA is enabled
    */
    function isOpenInIdeaSupported(workspace, localStorage) {
        var value = localStorage["openInIDEA"];
        return value !== "false";
    }
    IDE.isOpenInIdeaSupported = isOpenInIdeaSupported;

    /**
    * Returns true if open in TextMate is enabled
    */
    function isOpenInTextMateSupported(workspace, localStorage) {
        var value = localStorage["openInTextMate"];
        return value !== "false";
    }
    IDE.isOpenInTextMateSupported = isOpenInTextMateSupported;

    /**
    * Attempts to find the absolute file name for the given file and class name
    */
    function findClassAbsoluteFileName(mbean, jolokia, localStorage, fileName, className, onResult) {
        var sourceRoots = [];

        // TODO load from localStorage
        var answer = null;
        if (mbean) {
            answer = jolokia.execute(mbean, "findClassAbsoluteFileName", fileName, className, sourceRoots, onSuccess(onResult));
        } else {
            onResult(answer);
        }
        return answer;
    }
    IDE.findClassAbsoluteFileName = findClassAbsoluteFileName;

    function asNumber(value, defaultValue) {
        if (typeof defaultValue === "undefined") { defaultValue = 0; }
        if (angular.isNumber(value)) {
            return value;
        } else if (angular.isString(value)) {
            return parseInt(value);
        } else {
            return defaultValue;
        }
    }

    function max(v1, v2) {
        return (v1 >= v2) ? v1 : v2;
    }

    /**
    * Opens the file in IDEA
    */
    function ideaOpenAndNavigate(mbean, jolokia, absoluteFileName, line, column, fn) {
        if (typeof fn === "undefined") { fn = null; }
        var answer = null;
        if (mbean) {
            line = max(asNumber(line) - 1, 0);
            column = max(asNumber(column) - 1, 0);
            answer = jolokia.execute(mbean, "ideaOpenAndNavigate", absoluteFileName, line, column, onSuccess(fn));
        }
        return answer;
    }
    IDE.ideaOpenAndNavigate = ideaOpenAndNavigate;
})(IDE || (IDE = {}));
/**
* @module Threads
* @main Threads
*/
var Threads;
(function (Threads) {
    Threads.pluginName = 'threads';
    Threads.templatePath = 'app/threads/html/';
    Threads.log = Logger.get("Threads");
    Threads.jmxDomain = 'java.lang';
    Threads.mbeanType = 'Threading';
    Threads.mbean = Threads.jmxDomain + ":type=" + Threads.mbeanType;

    Threads.angularModule = angular.module(Threads.pluginName, ['bootstrap', 'ngResource', 'hawtioCore', 'ui']);

    Threads.angularModule.config(function ($routeProvider) {
        $routeProvider.when('/threads', { templateUrl: Threads.templatePath + 'index.html' });
    });

    Threads.angularModule.run(function ($location, workspace, viewRegistry, layoutFull, helpRegistry) {
        viewRegistry['threads'] = layoutFull;
        helpRegistry.addUserDoc('threads', 'app/threads/doc/help.md');

        workspace.topLevelTabs.push({
            id: "threads",
            content: "Threads",
            title: "JVM Threads",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties(Threads.jmxDomain, { type: Threads.mbeanType });
            },
            href: function () {
                return "#/threads";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("threads");
            }
        });
    });

    hawtioPluginLoader.addModule(Threads.pluginName);
})(Threads || (Threads = {}));
/**
* @module Threads
*/
var Threads;
(function (Threads) {
    function ThreadsController($scope, $routeParams, $templateCache, jolokia) {
        $scope.selectedRowJson = '';

        $scope.lastThreadJson = '';
        $scope.getThreadInfoResponseJson = '';
        $scope.threads = [];
        $scope.totals = {};
        $scope.support = {};

        $scope.row = {};
        $scope.threadSelected = false;
        $scope.selectedRowIndex = -1;

        $scope.stateFilter = 'NONE';

        $scope.showRaw = {
            expanded: false
        };

        $scope.$watch('searchFilter', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.threadGridOptions.filterOptions.filterText = newValue;
            }
        });

        $scope.$watch('stateFilter', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if ($scope.stateFilter === 'NONE') {
                    $scope.threads = $scope.unfilteredThreads;
                } else {
                    $scope.threads = $scope.filterThreads($scope.stateFilter, $scope.unfilteredThreads);
                }
            }
        });

        $scope.threadGridOptions = {
            selectedItems: [],
            data: 'threads',
            showSelectionCheckbox: false,
            enableRowClickSelection: true,
            multiSelect: false,
            primaryKeyFn: function (entity, idx) {
                return entity.threadId;
            },
            filterOptions: {
                filterText: ''
            },
            sortInfo: {
                sortBy: 'threadId',
                ascending: false
            },
            columnDefs: [
                {
                    field: 'threadId',
                    displayName: 'ID'
                },
                {
                    field: 'threadState',
                    displayName: 'State',
                    cellTemplate: $templateCache.get("threadStateTemplate")
                },
                {
                    field: 'threadName',
                    displayName: 'Name'
                },
                {
                    field: 'waitedTime',
                    displayName: 'Waited Time',
                    cellTemplate: '<div class="ngCellText" ng-show="row.entity.waitedTime > 0">{{row.entity.waitedTime | humanizeMs}}</div>'
                },
                {
                    field: 'blockedTime',
                    displayName: 'Blocked Time',
                    cellTemplate: '<div class="ngCellText" ng-show="row.entity.blockedTime > 0">{{row.entity.blockedTime | humanizeMs}}</div>'
                },
                {
                    field: 'inNative',
                    displayName: 'Native',
                    cellTemplate: '<div class="ngCellText"><span ng-show="row.entity.inNative" class="orange">(in native)</span></div>'
                },
                {
                    field: 'suspended',
                    displayName: 'Suspended',
                    cellTemplate: '<div class="ngCellText"><span ng-show="row.entity.suspended" class="red">(suspended)</span></div>'
                }
            ]
        };

        $scope.$watch('threadGridOptions.selectedItems', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (newValue.length === 0) {
                    $scope.row = {};
                    $scope.threadSelected = false;
                    $scope.selectedRowIndex = -1;
                } else {
                    $scope.row = newValue.first();
                    $scope.threadSelected = true;
                    $scope.selectedRowIndex = Core.pathGet($scope, ['hawtioSimpleTable', 'threads', 'rows']).findIndex(function (t) {
                        return t.entity['threadId'] === $scope.row['threadId'];
                    });
                }
                $scope.selectedRowJson = angular.toJson($scope.row, true);
            }
        }, true);

        $scope.filterOn = function (state) {
            $scope.stateFilter = state;
        };

        $scope.filterThreads = function (state, threads) {
            Threads.log.debug("Filtering threads by: ", state);
            if (state === 'NONE') {
                return threads;
            }
            return threads.filter(function (t) {
                return t && t['threadState'] === state;
            });
        };

        $scope.selectedFilterClass = function (state) {
            if (state === $scope.stateFilter) {
                return "active";
            } else {
                return "";
            }
        };

        $scope.deselect = function () {
            $scope.threadGridOptions.selectedItems = [];
        };

        $scope.selectThreadById = function (id) {
            $scope.threadGridOptions.selectedItems = $scope.threads.filter(function (t) {
                return t.threadId === id;
            });
        };

        $scope.selectThreadByIndex = function (idx) {
            var selectedThread = Core.pathGet($scope, ['hawtioSimpleTable', 'threads', 'rows'])[idx];
            $scope.threadGridOptions.selectedItems = $scope.threads.filter(function (t) {
                return t && t['threadId'] == selectedThread.entity['threadId'];
            });
        };

        $scope.init = function () {
            jolokia.request([
                {
                    type: 'read',
                    mbean: Threads.mbean,
                    attribute: 'ThreadContentionMonitoringSupported'
                }, {
                    type: 'read',
                    mbean: Threads.mbean,
                    attribute: 'ObjectMonitorUsageSupported'
                }, {
                    type: 'read',
                    mbean: Threads.mbean,
                    attribute: 'SynchronizerUsageSupported'
                }], {
                method: 'post',
                success: [
                    function (response) {
                        $scope.support.threadContentionMonitoringSupported = response.value;
                        Threads.log.debug("ThreadContentionMonitoringSupported: ", $scope.support.threadContentionMonitoringSupported);
                        $scope.maybeRegister();
                    },
                    function (response) {
                        $scope.support.objectMonitorUsageSupported = response.value;
                        Threads.log.debug("ObjectMonitorUsageSupported: ", $scope.support.objectMonitorUsageSupported);
                        $scope.maybeRegister();
                    },
                    function (response) {
                        $scope.support.synchronizerUsageSupported = response.value;
                        Threads.log.debug("SynchronizerUsageSupported: ", $scope.support.synchronizerUsageSupported);
                        $scope.maybeRegister();
                    }],
                error: function (response) {
                    Threads.log.error('Failed to query for supported usages: ', response.error);
                }
            });
        };

        var initFunc = Core.throttled($scope.init, 500);

        $scope.maybeRegister = function () {
            if ('objectMonitorUsageSupported' in $scope.support && 'synchronizerUsageSupported' in $scope.support && 'threadContentionMonitoringSupported' in $scope.support) {
                Threads.log.debug("Registering dumpAllThreads polling");
                Core.register(jolokia, $scope, {
                    type: 'exec',
                    mbean: Threads.mbean,
                    operation: 'dumpAllThreads',
                    arguments: [$scope.support.objectMonitorUsageSupported, $scope.support.synchronizerUsageSupported]
                }, onSuccess(render));

                if ($scope.support.threadContentionMonitoringSupported) {
                    // check and see if it's actually turned on, if not
                    // enable it
                    jolokia.request({
                        type: 'read',
                        mbean: Threads.mbean,
                        attribute: 'ThreadContentionMonitoringEnabled'
                    }, onSuccess($scope.maybeEnableThreadContentionMonitoring));
                }
            }
        };

        function disabledContentionMonitoring(response) {
            Threads.log.info("Disabled contention monitoring: ", response);
            Core.$apply($scope);
        }

        function enabledContentionMonitoring(response) {
            $scope.$on('$routeChangeStart', function () {
                jolokia.setAttribute(Threads.mbean, 'ThreadContentionMonitoringEnabled', false, onSuccess(disabledContentionMonitoring));
            });
            Threads.log.info("Enabled contention monitoring");
            Core.$apply($scope);
        }

        $scope.maybeEnableThreadContentionMonitoring = function (response) {
            if (response.value === false) {
                Threads.log.info("Thread contention monitoring not enabled, enabling");
                jolokia.setAttribute(Threads.mbean, 'ThreadContentionMonitoringEnabled', true, onSuccess(enabledContentionMonitoring));
            } else {
                Threads.log.info("Thread contention monitoring already enabled");
            }
            Core.$apply($scope);
        };

        $scope.getMonitorClass = function (name, value) {
            return value.toString();
        };

        $scope.getMonitorName = function (name) {
            name = name.replace('Supported', '');
            return name.titleize();
        };

        function render(response) {
            var responseJson = angular.toJson(response.value, true);
            if ($scope.getThreadInfoResponseJson !== responseJson) {
                $scope.getThreadInfoResponseJson = responseJson;

                var threads = response.value.exclude(function (t) {
                    return t === null;
                });

                $scope.unfilteredThreads = threads;
                $scope.totals = {};
                threads.forEach(function (t) {
                    // calculate totals
                    var state = t.threadState;
                    if (!(state in $scope.totals)) {
                        $scope.totals[state] = 1;
                    } else {
                        $scope.totals[state]++;
                    }
                });

                threads = $scope.filterThreads($scope.stateFilter, threads);

                $scope.threads = threads;
                Core.$apply($scope);
            }
        }

        initFunc();
    }
    Threads.ThreadsController = ThreadsController;
})(Threads || (Threads = {}));
var Tomcat;
(function (Tomcat) {
    function filerTomcatOrCatalina(response) {
        if (response) {
            // Tomcat can have mbean server names with Catalina or Tomcat
            response = response.filter(function (name) {
                return name.startsWith("Catalina") || name.startsWith("Tomcat");
            });
        }
        return response;
    }
    Tomcat.filerTomcatOrCatalina = filerTomcatOrCatalina;
    ;

    function iconClass(state) {
        if (state) {
            switch (state.toString().toLowerCase()) {
                case '1':
                    return "green icon-play-circle";
                case 'started':
                    return "green icon-play-circle";
                case '0':
                    return "orange icon-off";
                case 'stopped':
                    return "orange icon-off";
            }
        }

        // Tomcat 5 uses 0 for stopped
        if (angular.isNumber(state)) {
            if (state.toString() === '0') {
                return "orange icon-off";
            }
        }

        return "icon-question-sign";
    }
    Tomcat.iconClass = iconClass;

    function millisToDateFormat(time) {
        if (time) {
            var date = new Date(time);
            return date.toLocaleDateString() + " " + date.toLocaleTimeString();
        } else {
            return "";
        }
    }
    Tomcat.millisToDateFormat = millisToDateFormat;

    function isTomcat5(name) {
        return name.toString().indexOf("Apache Tomcat/5") !== -1;
    }
    Tomcat.isTomcat5 = isTomcat5;

    function isTomcat6(name) {
        return name.toString().indexOf("Apache Tomcat/6") !== -1;
    }
    Tomcat.isTomcat6 = isTomcat6;
})(Tomcat || (Tomcat = {}));
var Tomcat;
(function (Tomcat) {
    function SessionsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | tomcatIconClass}}"></i></div>';

        $scope.sessions = [];
        $scope.search = "";

        var columnDefs = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'activeSessions',
                displayName: 'Active Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'expiredSessions',
                displayName: 'Expired Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'rejectedSessions',
                displayName: 'Rejected Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxActive',
                displayName: 'Max Active Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxActiveSessions',
                displayName: 'Max Active Sessions Allowed',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxInactiveInterval',
                displayName: 'Max Inactive Interval',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'sessionCounter',
                displayName: 'Session Counter',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'sessionCreateRate',
                displayName: 'Session Create Rate',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'sessionExpireRate',
                displayName: 'Session Expire Rate',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'sessions',
            displayFooter: false,
            displaySelectionCheckbox: false,
            canSelectRows: false,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            }
        };

        function render(response) {
            response = Tomcat.filerTomcatOrCatalina(response);

            $scope.sessions = [];

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    var mbean = obj.mbean;
                    if (mbean) {
                        // the context path is part of the mbean name
                        // grab the 2nd part of the mbean that has context=/name
                        var context = mbean.toString().split(",")[1];
                        if (context) {
                            if (context.toString().indexOf("path=") !== -1) {
                                // and remove the leading path=/ from the name (Tomcat 5 or 6)
                                obj.path = context.toString().substr(5);
                            } else {
                                // and remove the leading context=/ from the name (Tomcat 7)
                                obj.path = context.toString().substr(9);
                            }
                        } else {
                            obj.path = "";
                        }

                        $scope.sessions.push(obj);
                        Core.$apply($scope);
                    }
                }
            }

            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading tomcat session data...");
            jolokia.search("*:type=Manager,*", onSuccess(render));
        }
    }
    Tomcat.SessionsController = SessionsController;
})(Tomcat || (Tomcat = {}));
/**
* @module Tomcat
* @main Tomcat
*/
var Tomcat;
(function (Tomcat) {
    var pluginName = 'tomcat';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ui.bootstrap.dialog', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/tomcat/server', { templateUrl: 'app/tomcat/html/server.html' }).when('/tomcat/applications', { templateUrl: 'app/tomcat/html/applications.html' }).when('/tomcat/connectors', { templateUrl: 'app/tomcat/html/connectors.html' }).when('/tomcat/sessions', { templateUrl: 'app/tomcat/html/sessions.html' });
    }).filter('tomcatIconClass', function () {
        return Tomcat.iconClass;
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['tomcat'] = "app/tomcat/html/layoutTomcatTabs.html";
        helpRegistry.addUserDoc('tomcat', 'app/tomcat/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("Tomcat") || workspace.treeContainsDomainAndProperties("Catalina");
        });

        workspace.topLevelTabs.push({
            id: "tomcat",
            content: "Tomcat",
            title: "Manage your Tomcat container",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("Tomcat") || workspace.treeContainsDomainAndProperties("Catalina");
            },
            href: function () {
                return "#/tomcat/applications";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("tomcat");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Tomcat || (Tomcat = {}));
var Tomcat;
(function (Tomcat) {
    function ConnectorsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | tomcatIconClass}}"></i></div>';

        $scope.connectors = [];
        $scope.selected = [];

        var columnDefs = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'port',
                displayName: 'Port',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'scheme',
                displayName: 'Scheme',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'protocol',
                displayName: 'Protocol',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'secure',
                displayName: 'Secure',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'connectionLinger',
                displayName: 'Connection Linger',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'connectionTimeout',
                displayName: 'Connection Timeout',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'keepAliveTimeout',
                displayName: 'Keep Alive Timeout',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'minSpareThreads',
                displayName: 'Minimum Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxThreads',
                displayName: 'Maximum Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'connectors',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            }
        };

        function render(response) {
            response = Tomcat.filerTomcatOrCatalina(response);

            $scope.connectors = [];
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    $scope.connectors.push(obj);
                    Core.$apply($scope);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({
                    type: "read", mbean: mbean, attribute: [
                        "scheme", "port", "protocol", "secure",
                        "connectionLinger", "connectionTimeout", "keepAliveTimeout", "minSpareThreads", "maxThreads", "stateName"] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        // function to control the connectors
        $scope.controlConnector = function (op) {
            // grab id of mbean names to control
            var ids = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(ids)) {
                ids = [ids];
            }

            // execute operation on each mbean
            ids.forEach(function (id) {
                jolokia.request({
                    type: 'exec',
                    mbean: id,
                    operation: op,
                    arguments: null
                }, onSuccess($scope.onResponse, { error: $scope.onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlConnector('stop');
        };

        $scope.start = function () {
            $scope.controlConnector('start');
        };

        $scope.destroy = function () {
            $scope.controlConnector('destroy');
        };

        // function to trigger reloading page
        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
            loadData();
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading tomcat connector data...");
            jolokia.search("Catalina:type=Connector,*", onSuccess(render));
        }
    }
    Tomcat.ConnectorsController = ConnectorsController;
})(Tomcat || (Tomcat = {}));
var Tomcat;
(function (Tomcat) {
    function TomcatController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | tomcatIconClass}}"></i></div>';

        $scope.uninstallDialog = new Core.Dialog();

        $scope.webapps = [];
        $scope.selected = [];

        var columnDefsTomcat5 = [
            {
                field: 'state',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'startTime',
                displayName: 'Start Time',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        var columnDefsTomcat6 = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'startTime',
                displayName: 'Start Time',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        var columnDefsTomcat7 = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'displayName',
                displayName: 'Display Name',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'startTime',
                displayName: 'Start Time',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'webapps',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            filterOptions: {
                filterText: ''
            }
        };

        function render(response) {
            response = Tomcat.filerTomcatOrCatalina(response);

            $scope.webapps = [];
            $scope.mbeanIndex = {};
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    var mbean = obj.mbean;
                    if (mbean) {
                        // format the start time as readable date format
                        obj.startTime = Tomcat.millisToDateFormat(obj.startTime);

                        var idx = $scope.mbeanIndex[mbean];
                        if (angular.isDefined(idx)) {
                            $scope.webapps[mbean] = obj;
                        } else {
                            $scope.mbeanIndex[mbean] = $scope.webapps.length;
                            $scope.webapps.push(obj);
                        }

                        // ensure web page is updated
                        Core.$apply($scope);
                    }
                }
            }

            angular.forEach(response, function (value, key) {
                var mbean = value;
                if (Tomcat.isTomcat5($scope.tomcatServerVersion)) {
                    jolokia.request({
                        type: "read", mbean: mbean,
                        attribute: ["path", "state", "startTime"] }, onSuccess(onAttributes));
                } else if (Tomcat.isTomcat6($scope.tomcatServerVersion)) {
                    jolokia.request({
                        type: "read", mbean: mbean,
                        attribute: ["path", "stateName", "startTime"] }, onSuccess(onAttributes));
                } else {
                    jolokia.request({
                        type: "read", mbean: mbean,
                        attribute: ["displayName", "path", "stateName", "startTime"] }, onSuccess(onAttributes));
                }
            });
            Core.$apply($scope);
        }
        ;

        // function to control the web applications
        $scope.controlWebApps = function (op) {
            // grab id of mbean names to control
            var mbeanNames = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(mbeanNames)) {
                mbeanNames = [mbeanNames];
            }

            // execute operation on each mbean
            var lastIndex = (mbeanNames.length || 1) - 1;
            angular.forEach(mbeanNames, function (mbean, idx) {
                var onResponse = (idx >= lastIndex) ? $scope.onLastResponse : $scope.onResponse;
                jolokia.request({
                    type: 'exec',
                    mbean: mbean,
                    operation: op,
                    arguments: null
                }, onSuccess(onResponse, { error: onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlWebApps('stop');
        };

        $scope.start = function () {
            $scope.controlWebApps('start');
        };

        $scope.reload = function () {
            $scope.controlWebApps('reload');
        };

        $scope.uninstall = function () {
            $scope.controlWebApps('destroy');
            $scope.uninstallDialog.close();
        };

        // function to trigger reloading page
        $scope.onLastResponse = function (response) {
            $scope.onResponse(response);

            // we only want to force updating the data on the last response
            loadData();
        };

        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading tomcat webapp data...");
            jolokia.search("*:j2eeType=WebModule,*", onSuccess(render));
        }

        // grab server information once
        $scope.tomcatServerVersion = "";

        var servers = jolokia.search("*:type=Server");
        servers = Tomcat.filerTomcatOrCatalina(servers);
        if (servers && servers.length === 1) {
            $scope.tomcatServerVersion = jolokia.getAttribute(servers[0], "serverInfo");
        } else {
            console.log("Cannot find Tomcat server or there was more than one server. response is: " + servers);
        }

        // the columns shown in the applications view depends on the Tomcat version in use
        if (Tomcat.isTomcat5($scope.tomcatServerVersion)) {
            console.log("Using Tomcat 5");
            $scope.gridOptions.columnDefs = columnDefsTomcat5;
        } else if (Tomcat.isTomcat6($scope.tomcatServerVersion)) {
            console.log("Using Tomcat 6");
            $scope.gridOptions.columnDefs = columnDefsTomcat6;
        } else {
            console.log("Using Tomcat 7");
            $scope.gridOptions.columnDefs = columnDefsTomcat7;
        }
    }
    Tomcat.TomcatController = TomcatController;
})(Tomcat || (Tomcat = {}));
var Infinispan;
(function (Infinispan) {
    function TreeController($scope, $location, workspace) {
        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        $scope.$watch('workspace.tree', function () {
            if (workspace.moveIfViewInvalid())
                return;

            var children = [];

            // lets pull out each context
            var tree = workspace.tree;
            if (tree) {
                var domainName = Infinispan.jmxDomain;
                var folder = tree.get(domainName);
                if (folder) {
                    var cachesFolder = new Folder("Caches");
                    cachesFolder.domain = domainName;
                    cachesFolder.key = "root-Infinispan-Caches";
                    cachesFolder.typeName = "Caches";
                    children.push(cachesFolder);
                    addAllCacheStatistics(folder, cachesFolder);
                }

                var treeElement = $("#infinispantree");
                Jmx.enableTree($scope, $location, workspace, treeElement, children);

                // lets do this asynchronously to avoid Error: $digest already in progress
                setTimeout(updateSelectionFromURL, 50);
            }
        });

        function addAllCacheStatistics(folder, answer) {
            if (folder) {
                var children = folder.children;
                if (children) {
                    angular.forEach(folder.children, function (value, key) {
                        if (value.objectName && value.title === "Statistics") {
                            var cacheName = value.parent.parent.title || value.title;
                            var name = humanizeValue(cacheName);
                            var cacheFolder = new Folder(name);
                            cacheFolder.addClass = "org-infinispn-cache";
                            cacheFolder.typeName = "Cache";
                            cacheFolder.key = answer.key + "-" + cacheName;
                            cacheFolder.objectName = value.objectName;
                            cacheFolder.domain = value.domain;
                            cacheFolder.entries = value.entries;
                            answer.children.push(cacheFolder);
                        } else {
                            addAllCacheStatistics(value, answer);
                        }
                    });
                }
            }
        }

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURL($location, $("#infinispantree"), true);
        }
    }
    Infinispan.TreeController = TreeController;
})(Infinispan || (Infinispan = {}));
/**
* @module Infinispan
*/
var Infinispan;
(function (Infinispan) {
    /**
    * Returns the name of the cache from the mbean results
    * @method infinispanCacheName
    * @for Infinispan
    * @param {any} entity
    * @return {String}
    */
    function infinispanCacheName(entity) {
        // there's no name in the MBean so lets extract it from the JMX ObjectName
        if (entity) {
            var id = entity._id;
            if (id) {
                var prefix = 'name="';
                var idx = id.indexOf(prefix);
                if (idx > 0) {
                    idx += prefix.length;
                    var lastIdx = id.indexOf('"', idx + 1);
                    if (lastIdx > 0) {
                        return id.substring(idx, lastIdx);
                    } else {
                        return id.substring(idx);
                    }
                }
            }
            return id;
        }
        return null;
    }
    Infinispan.infinispanCacheName = infinispanCacheName;

    /**
    * Returns the MBean ObjectName for the interpreter
    * @method getInterpreterMBean
    * @for Infinispan
    * @param {Workspace} workspace
    * @return {String}
    */
    function getInterpreterMBean(workspace) {
        if (workspace) {
            var folder = workspace.findMBeanWithProperties(Infinispan.jmxDomain, { component: "Interpreter", type: "CacheManager" });
            if (folder) {
                return folder.objectName;
            }
        }
        return null;
    }
    Infinispan.getInterpreterMBean = getInterpreterMBean;

    function getSelectedCacheName(workspace) {
        var selection = workspace.selection;
        if (selection && selection.domain === Infinispan.jmxDomain) {
            // lets get the cache name
            return selection.entries["name"];
        }
        return null;
    }
    Infinispan.getSelectedCacheName = getSelectedCacheName;
})(Infinispan || (Infinispan = {}));
/**
* @module Infinispan
*/
var Infinispan;
(function (Infinispan) {
    /**
    * Returns the name of the cache from the mbean results
    * @class CLI
    */
    var CLI = (function () {
        function CLI(workspace, jolokia) {
            this.workspace = workspace;
            this.jolokia = jolokia;
            this.cacheName = null;
            this.sessionId = null;
            this.useSessionIds = true;
        }
        CLI.prototype.setCacheName = function (name) {
            if (name) {
                name = trimQuotes(name);
                var postfix = "(local)";
                if (name.endsWith(postfix)) {
                    name = name.substring(0, name.length - postfix.length);
                }
            }
            if (!this.cacheName || this.cacheName !== name) {
                if (this.sessionId) {
                    this.deleteSession(this.sessionId);
                }
                this.cacheName = name;
                this.createSession();
            }
        };

        CLI.prototype.createSession = function () {
            var _this = this;
            if (this.useSessionIds) {
                var mbean = Infinispan.getInterpreterMBean(this.workspace);
                if (mbean) {
                    var cli = this;
                    this.jolokia.execute(mbean, "createSessionId", this.cacheName, onSuccess(function (value) {
                        console.log("Has session ID: " + value);
                        _this.sessionId = value;
                    }));
                } else {
                    this.warnMissingMBean();
                }
            }
        };

        CLI.prototype.execute = function (sql, handler) {
            if (sql) {
                sql = sql.trim();
                if (!sql.endsWith(";")) {
                    sql += ";";
                }
                var sessionId = (this.useSessionIds) ? this.sessionId : null;
                if (!this.useSessionIds) {
                    sql = "cache " + this.cacheName + "; " + sql;
                }
                if (sessionId || !this.useSessionIds) {
                    var mbean = Infinispan.getInterpreterMBean(this.workspace);
                    if (mbean) {
                        this.jolokia.execute(mbean, "execute", sessionId, sql, onSuccess(handler));
                    } else {
                        this.warnMissingMBean();
                    }
                } else {
                    notification("warning", "Cannot evaluate SQL as we don't have a sessionId yet!");
                }
            }
        };

        CLI.prototype.deleteSession = function (sessionId) {
            // there is no delete API so far
        };

        CLI.prototype.warnMissingMBean = function () {
            notification("error", "No Interpreter MBean available");
        };
        return CLI;
    })();
    Infinispan.CLI = CLI;
})(Infinispan || (Infinispan = {}));
/**
* @module Infinispan
* @main Infinispan
*/
var Infinispan;
(function (Infinispan) {
    var pluginName = 'infinispan';
    Infinispan.jmxDomain = 'Infinispan';

    var toolBar = "app/infinispan/html/attributeToolBar.html";

    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/infinispan/query', { templateUrl: 'app/infinispan/html/query.html' });
    }).filter('infinispanCacheName', function () {
        return Infinispan.infinispanCacheName;
    }).run(function (workspace, viewRegistry, helpRegistry) {
        viewRegistry['infinispan'] = 'app/infinispan/html/layoutCacheTree.html';
        helpRegistry.addUserDoc('infinispan', 'app/infinispan/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties(Infinispan.jmxDomain);
        });

        /*
        Jmx.addAttributeToolBar(pluginName, jmxDomain, (selection: NodeSelection) => {
        // TODO there should be a nicer way to do this!
        var folderNames = selection.folderNames;
        if (folderNames && selection.domain === jmxDomain) {
        return toolBar;
        }
        return null;
        });
        */
        // register default attribute views
        var nameTemplate = '<div class="ngCellText" title="Infinispan Cache">{{row.entity | infinispanCacheName}}</div>';

        var attributes = workspace.attributeColumnDefs;
        attributes[Infinispan.jmxDomain + "/Caches/folder"] = [
            {
                field: '_id', displayName: 'Name',
                cellTemplate: nameTemplate, width: "**" },
            { field: 'numberOfEntries', displayName: 'Entries' },
            { field: 'hits', displayName: 'Hits' },
            { field: 'hitRatio', displayName: 'Hit Ratio' },
            { field: 'stores', displayName: 'Stores' },
            { field: 'averageReadTime', displayName: 'Avg Read Time' },
            { field: 'averageWriteTime', displayName: 'Avg Write Time' }
        ];

        workspace.topLevelTabs.push({
            id: "infinispan",
            content: "Infinispan",
            title: "View your distributed data",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties(Infinispan.jmxDomain);
            },
            href: function () {
                return "#/jmx/attributes?tab=infinispan";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("infinispan");
            }
        });

        workspace.subLevelTabs.push({
            content: '<i class="icon-pencil"></i> Query',
            title: "Perform InSQL commands on the cache",
            isValid: function (workspace) {
                return Infinispan.getSelectedCacheName(workspace) && Infinispan.getInterpreterMBean(workspace);
            },
            href: function () {
                return "#/infinispan/query";
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Infinispan || (Infinispan = {}));
var Infinispan;
(function (Infinispan) {
    function QueryController($scope, $location, workspace, jolokia) {
        var interpreter = new Infinispan.CLI(workspace, jolokia);
        $scope.logs = [];
        $scope.filteredLogs = [];
        $scope.selectedItems = [];
        $scope.searchText = "";
        $scope.filter = {
            // The default logging level to show, empty string => show all
            logLevelQuery: "",
            // The default value of the exact match logging filter
            logLevelExactMatch: false
        };

        var columnDefs = [
            {
                field: 'key',
                displayName: 'Key'
            },
            {
                field: 'value',
                displayName: 'Value'
            }
        ];

        $scope.gridOptions = {
            selectedItems: $scope.selectedItems,
            data: 'filteredLogs',
            displayFooter: false,
            showFilter: false,
            filterOptions: {
                filterText: "searchText"
            },
            columnDefs: columnDefs,
            rowDetailTemplateId: "logDetailTemplate"
        };

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(refreshCacheName, 50);
        });

        $scope.$watch('workspace.selection', function () {
            refreshCacheName();
        });

        $scope.doQuery = function () {
            if ($scope.sql) {
                interpreter.execute($scope.sql, handleResults);
            }
        };

        function handleResults(results) {
            $scope.output = null;
            if (!results) {
                console.log("no output...");
            } else {
                var error = results["ERROR"] || "";
                var stackTrace = results["STACKTRACE"] || "";
                if (error || stackTrace) {
                    if (stackTrace) {
                        error += "\n" + stackTrace;
                    }
                    notification("error", error);
                } else {
                    var output = results["OUTPUT"];
                    if (!output) {
                        notification("error", "No results!");
                    } else {
                        $scope.output = output;
                        console.log("==== output: " + output);
                        Core.$apply($scope);
                    }
                }
            }
        }

        function refreshCacheName() {
            var cacheName = Infinispan.getSelectedCacheName(workspace);
            console.log("selected cacheName is: " + cacheName);
            if (cacheName) {
                interpreter.setCacheName(cacheName);
            }
        }
    }
    Infinispan.QueryController = QueryController;
})(Infinispan || (Infinispan = {}));
/**
* @module Karaf
*/
var Karaf;
(function (Karaf) {
    function NavBarController($scope, workspace) {
        $scope.hash = workspace.hash();

        $scope.isKarafEnabled = workspace.treeContainsDomainAndProperties("org.apache.karaf");
        $scope.isFeaturesEnabled = Karaf.getSelectionFeaturesMBean(workspace);
        $scope.isScrEnabled = Karaf.getSelectionScrMBean(workspace);

        $scope.$on('$routeChangeSuccess', function () {
            $scope.hash = workspace.hash();
        });

        $scope.isActive = function (nav) {
            return workspace.isLinkActive(nav);
        };

        $scope.isPrefixActive = function (nav) {
            return workspace.isLinkPrefixActive(nav);
        };
    }
    Karaf.NavBarController = NavBarController;
})(Karaf || (Karaf = {}));
/**
* @module Karaf
* @main Karaf
*/
var Karaf;
(function (Karaf) {
    var pluginName = 'karaf';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/osgi/server', { templateUrl: 'app/karaf/html/server.html' }).when('/osgi/features', { templateUrl: 'app/karaf/html/features.html', reloadOnSearch: false }).when('/osgi/scr-components', { templateUrl: 'app/karaf/html/scr-components.html' }).when('/osgi/scr-component/:name', { templateUrl: 'app/karaf/html/scr-component.html' }).when('/osgi/feature/:name/:version', { templateUrl: 'app/karaf/html/feature.html' });
    }).run(function (workspace, viewRegistry, helpRegistry) {
        helpRegistry.addUserDoc('karaf', 'app/karaf/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties('org.apache.karaf');
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Karaf || (Karaf = {}));
/**
* @module Karaf
*/
var Karaf;
(function (Karaf) {
    function FeatureController($scope, jolokia, workspace, $routeParams) {
        $scope.hasFabric = Fabric.hasFabric(workspace);
        $scope.name = $routeParams.name;
        $scope.version = $routeParams.version;
        $scope.bundlesByLocation = {};
        $scope.props = "properties";

        updateTableContents();

        $scope.install = function () {
            Karaf.installFeature(workspace, jolokia, $scope.name, $scope.version, function () {
                notification('success', 'Installed feature ' + $scope.name);
            }, function (response) {
                notification('error', 'Failed to install feature ' + $scope.name + ' due to ' + response.error);
            });
        };

        $scope.uninstall = function () {
            Karaf.uninstallFeature(workspace, jolokia, $scope.name, $scope.version, function () {
                notification('success', 'Uninstalled feature ' + $scope.name);
            }, function (response) {
                notification('error', 'Failed to uninstall feature ' + $scope.name + ' due to ' + response.error);
            });
        };

        $scope.toProperties = function (elements) {
            var answer = '';
            angular.forEach(elements, function (value, name) {
                answer += value['Key'] + " = " + value['Value'] + "\n";
            });
            return answer.trim();
        };

        function populateTable(response) {
            $scope.row = Karaf.extractFeature(response.value, $scope.name, $scope.version);
            if ($scope.row) {
                addBundleDetails($scope.row);
                var dependencies = [];

                //TODO - if the version isn't set or is 0.0.0 then maybe we show the highest available?
                angular.forEach($scope.row.Dependencies, function (version, name) {
                    angular.forEach(version, function (data, version) {
                        dependencies.push({
                            Name: name,
                            Version: version
                        });
                    });
                });
                $scope.row.Dependencies = dependencies;
            }

            //console.log("row: ", $scope.row);
            Core.$apply($scope);
        }

        function setBundles(response) {
            var bundleMap = {};
            Osgi.defaultBundleValues(workspace, $scope, response.values);
            angular.forEach(response.value, function (bundle) {
                var location = bundle["Location"];
                $scope.bundlesByLocation[location] = bundle;
            });
        }
        ;

        function updateTableContents() {
            var featureMbean = Karaf.getSelectionFeaturesMBean(workspace);
            var bundleMbean = Osgi.getSelectionBundleMBean(workspace);
            var jolokia = workspace.jolokia;

            if (bundleMbean) {
                setBundles(jolokia.request({ type: 'exec', mbean: bundleMbean, operation: 'listBundles()' }));
            }

            if (featureMbean) {
                jolokia.request({ type: 'read', mbean: featureMbean }, onSuccess(populateTable));
            }
        }

        function addBundleDetails(feature) {
            var bundleDetails = [];
            angular.forEach(feature["Bundles"], function (bundleLocation) {
                var bundle = $scope.bundlesByLocation[bundleLocation];
                if (bundle) {
                    bundle["Installed"] = true;
                    bundleDetails.push(bundle);
                } else {
                    bundleDetails.push({
                        "Location": bundleLocation,
                        "Installed": false
                    });
                }
            });
            feature["BundleDetails"] = bundleDetails;
        }
    }
    Karaf.FeatureController = FeatureController;
})(Karaf || (Karaf = {}));
/**
* @module Karaf
*/
var Karaf;
(function (Karaf) {
    Karaf.log = Logger.get("Karaf");

    function setSelect(selection, group) {
        if (!angular.isDefined(selection)) {
            return group[0];
        }
        var answer = group.findIndex(function (item) {
            return item.id === selection.id;
        });
        if (answer !== -1) {
            return group[answer];
        } else {
            return group[0];
        }
    }
    Karaf.setSelect = setSelect;

    function installRepository(workspace, jolokia, uri, success, error) {
        Karaf.log.info("installing URI: ", uri);
        jolokia.request({
            type: 'exec', mbean: getSelectionFeaturesMBean(workspace),
            operation: 'addRepository(java.lang.String)',
            arguments: [uri]
        }, onSuccess(success, { error: error }));
    }
    Karaf.installRepository = installRepository;

    function uninstallRepository(workspace, jolokia, uri, success, error) {
        Karaf.log.info("uninstalling URI: ", uri);
        jolokia.request({
            type: 'exec', mbean: getSelectionFeaturesMBean(workspace),
            operation: 'removeRepository(java.lang.String)',
            arguments: [uri]
        }, onSuccess(success, { error: error }));
    }
    Karaf.uninstallRepository = uninstallRepository;

    function installFeature(workspace, jolokia, feature, version, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionFeaturesMBean(workspace),
            operation: 'installFeature(java.lang.String, java.lang.String)',
            arguments: [feature, version]
        }, onSuccess(success, { error: error }));
    }
    Karaf.installFeature = installFeature;

    function uninstallFeature(workspace, jolokia, feature, version, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionFeaturesMBean(workspace),
            operation: 'uninstallFeature(java.lang.String, java.lang.String)',
            arguments: [feature, version]
        }, onSuccess(success, { error: error }));
    }
    Karaf.uninstallFeature = uninstallFeature;

    // TODO move to core?
    function toCollection(values) {
        var collection = values;
        if (!angular.isArray(values)) {
            collection = [values];
        }
        return collection;
    }
    Karaf.toCollection = toCollection;

    function featureLinks(workspace, name, version) {
        return "<a href='" + url("#/karaf/feature/" + name + "/" + version + workspace.hash()) + "'>" + version + "</a>";
    }
    Karaf.featureLinks = featureLinks;

    function extractFeature(attributes, name, version) {
        var features = [];
        var repos = [];
        populateFeaturesAndRepos(attributes, features, repos);
        return features.find(function (feature) {
            return feature.Name == name && feature.Version == version;
        });
        /*
        var f = {};
        angular.forEach(attributes["Features"], (feature) => {
        angular.forEach(feature, (entry) => {
        if (entry["Name"] === name && entry["Version"] === version) {
        var deps = [];
        populateDependencies(attributes, entry["Dependencies"], deps);
        f["Name"] = entry["Name"];
        f["Version"] = entry["Version"];
        f["Bundles"] = entry["Bundles"];
        f["Dependencies"] = deps;
        f["Installed"] = entry["Installed"];
        f["Configurations"] = entry["Configurations"];
        f["Configuration Files"] = entry["Configuration Files"];
        f["Files"] = entry["Configuration Files"];
        }
        });
        });
        return f;
        */
    }
    Karaf.extractFeature = extractFeature;

    var platformBundlePatterns = [
        "^org.apache.aries",
        "^org.apache.karaf",
        "^activemq-karaf",
        "^org.apache.commons",
        "^org.apache.felix",
        "^io.fabric8",
        "^org.apache.geronimo.specs",
        "^org.apache.servicemix.bundles",
        "^org.objectweb.asm",
        "^io.hawt",
        "^javax.mail",
        "^org.jvnet",
        "^org.apache.mina.core",
        "^org.apache.sshd.core",
        "^org.apache.neethi",
        "^org.apache.servicemix.specs",
        "^org.apache.xbean",
        "^org.apache.santuario.xmlsec",
        "^biz.aQute.bndlib",
        "^groovy-all",
        "^com.google.guava",
        "jackson-\\w+-asl",
        "^org.ops4j",
        "^org.springframework",
        "^bcprov$",
        "^jline$",
        "^scala-library$",
        "^stax2-api$",
        "^woodstox-core-asl",
        "^org.jboss.amq.mq-fabric",
        "^joda-time$",
        "^org.apache.ws",
        "-commands$",
        "patch.patch",
        "org.fusesource.insight",
        "activeio-core",
        "activemq-osgi",
        "^org.eclipse.jetty",
        "org.codehaus.jettison.jettison"
    ];

    var platformBundleRegex = new RegExp(platformBundlePatterns.join('|'));

    var camelBundlePatterns = ["^org.apache.camel", "activemq-camel$"];
    var camelBundleRegex = new RegExp(camelBundlePatterns.join('|'));

    var cxfBundlePatterns = ["^org.apache.cxf"];
    var cxfBundleRegex = new RegExp(cxfBundlePatterns.join('|'));

    function isPlatformBundle(symbolicName) {
        return platformBundleRegex.test(symbolicName);
    }
    Karaf.isPlatformBundle = isPlatformBundle;

    function isCamelBundle(symbolicName) {
        return camelBundleRegex.test(symbolicName);
    }
    Karaf.isCamelBundle = isCamelBundle;

    function isCxfBundle(symbolicName) {
        return cxfBundleRegex.test(symbolicName);
    }
    Karaf.isCxfBundle = isCxfBundle;

    function populateFeaturesAndRepos(attributes, features, repositories) {
        var fullFeatures = attributes["Features"];
        angular.forEach(attributes["Repositories"], function (repo) {
            repositories.push({
                id: repo["Name"],
                uri: repo["Uri"]
            });

            if (!fullFeatures) {
                return;
            }

            angular.forEach(repo["Features"], function (feature) {
                angular.forEach(feature, function (entry) {
                    var f = Object.extended(fullFeatures[entry['Name']][entry['Version']]).clone();
                    f["Id"] = entry["Name"] + "/" + entry["Version"];
                    f["RepositoryName"] = repo["Name"];
                    f["RepositoryURI"] = repo["Uri"];
                    features.push(f);
                });
            });
        });
    }
    Karaf.populateFeaturesAndRepos = populateFeaturesAndRepos;

    function createScrComponentsView(workspace, jolokia, components) {
        var result = [];
        angular.forEach(components, function (component) {
            result.push({
                Name: component,
                State: getComponentStateDescription(getComponentState(workspace, jolokia, component))
            });
        });
        return result;
    }
    Karaf.createScrComponentsView = createScrComponentsView;

    function getComponentStateDescription(state) {
        switch (state) {
            case 2:
                return "Enabled";
            case 4:
                return "Unsatisfied";
            case 8:
                return "Activating";
            case 16:
                return "Active";
            case 32:
                return "Registered";
            case 64:
                return "Factory";
            case 128:
                return "Deactivating";
            case 256:
                return "Destroying";
            case 1024:
                return "Disabling";
            case 2048:
                return "Disposing";
        }
        return "Unknown";
    }
    Karaf.getComponentStateDescription = getComponentStateDescription;
    ;

    function getAllComponents(workspace, jolokia) {
        var scrMBean = getSelectionScrMBean(workspace);
        var response = jolokia.request({
            type: 'read', mbean: scrMBean,
            arguments: []
        });

        //Check if the MBean provides the Components attribute.
        if (!('Components' in response.value)) {
            response = jolokia.request({
                type: 'exec', mbean: scrMBean, operation: 'listComponents()'
            });
            return createScrComponentsView(workspace, jolokia, response.value);
        }
        return response.value['Components'].values;
    }
    Karaf.getAllComponents = getAllComponents;

    function getComponentByName(workspace, jolokia, componentName) {
        var components = getAllComponents(workspace, jolokia);
        return components.find(function (c) {
            return c.Name == componentName;
        });
    }
    Karaf.getComponentByName = getComponentByName;

    function isComponentActive(workspace, jolokia, component) {
        var response = jolokia.request({
            type: 'exec', mbean: getSelectionScrMBean(workspace),
            operation: 'isComponentActive(java.lang.String)',
            arguments: [component]
        });
        return response.value;
    }
    Karaf.isComponentActive = isComponentActive;

    function getComponentState(workspace, jolokia, component) {
        var response = jolokia.request({
            type: 'exec', mbean: getSelectionScrMBean(workspace),
            operation: 'componentState(java.lang.String)',
            arguments: [component]
        });
        return response.value;
    }
    Karaf.getComponentState = getComponentState;

    function activateComponent(workspace, jolokia, component, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionScrMBean(workspace),
            operation: 'activateComponent(java.lang.String)',
            arguments: [component]
        }, onSuccess(success, { error: error }));
    }
    Karaf.activateComponent = activateComponent;

    function deactivateComponent(workspace, jolokia, component, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionScrMBean(workspace),
            operation: 'deactiveateComponent(java.lang.String)',
            arguments: [component]
        }, onSuccess(success, { error: error }));
    }
    Karaf.deactivateComponent = deactivateComponent;

    function populateDependencies(attributes, dependencies, features) {
        angular.forEach(dependencies, function (feature) {
            angular.forEach(feature, function (entry) {
                var enhancedFeature = extractFeature(attributes, entry["Name"], entry["Version"]);
                enhancedFeature["id"] = entry["Name"] + "/" + entry["Version"];

                //enhancedFeature["repository"] = repo["Name"];
                features.push(enhancedFeature);
            });
        });
    }
    Karaf.populateDependencies = populateDependencies;

    function getSelectionFeaturesMBean(workspace) {
        if (workspace) {
            var featuresStuff = workspace.mbeanTypesToDomain["features"] || {};
            var karaf = featuresStuff["org.apache.karaf"] || {};
            var mbean = karaf.objectName;
            if (mbean) {
                return mbean;
            }

            // lets navigate to the tree item based on paths
            var folder = workspace.tree.navigate("org.apache.karaf", "features");
            if (!folder) {
                // sometimes the features mbean is inside the 'root' folder
                folder = workspace.tree.navigate("org.apache.karaf");
                if (folder) {
                    var children = folder.children;
                    folder = null;
                    angular.forEach(children, function (child) {
                        if (!folder) {
                            folder = child.navigate("features");
                        }
                    });
                }
            }
            if (folder) {
                var children = folder.children;
                if (children) {
                    var node = children[0];
                    if (node) {
                        return node.objectName;
                    }
                }
                return folder.objectName;
            }
        }
        return null;
    }
    Karaf.getSelectionFeaturesMBean = getSelectionFeaturesMBean;

    function getSelectionScrMBean(workspace) {
        if (workspace) {
            var scrStuff = workspace.mbeanTypesToDomain["scr"] || {};
            var karaf = scrStuff["org.apache.karaf"] || {};
            var mbean = karaf.objectName;
            if (mbean) {
                return mbean;
            }

            // lets navigate to the tree item based on paths
            var folder = workspace.tree.navigate("org.apache.karaf", "scr");
            if (!folder) {
                // sometimes the features mbean is inside the 'root' folder
                folder = workspace.tree.navigate("org.apache.karaf");
                if (folder) {
                    var children = folder.children;
                    folder = null;
                    angular.forEach(children, function (child) {
                        if (!folder) {
                            folder = child.navigate("scr");
                        }
                    });
                }
            }
            if (folder) {
                var children = folder.children;
                if (children) {
                    var node = children[0];
                    if (node) {
                        return node.objectName;
                    }
                }
                return folder.objectName;
            }
        }
        return null;
    }
    Karaf.getSelectionScrMBean = getSelectionScrMBean;
})(Karaf || (Karaf = {}));
/**
* @module Karaf
*/
var Karaf;
(function (Karaf) {
    function ServerController($scope, $location, workspace, jolokia) {
        $scope.data = {
            name: "",
            version: "",
            state: "",
            root: "",
            startLevel: "",
            framework: "",
            frameworkVersion: "",
            location: "",
            sshPort: "",
            rmiRegistryPort: "",
            rmiServerPort: "",
            pid: "" };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading Karaf data...");
            jolokia.search("org.apache.karaf:type=admin,*", onSuccess(render));
        }

        function render(response) {
            // grab the first mbean as there should ideally only be one karaf in the JVM
            if (angular.isArray(response)) {
                var mbean = response[0];
                if (mbean) {
                    jolokia.getAttribute(mbean, "Instances", onSuccess(function (response) {
                        onInstances(response, mbean);
                    }));
                }
            }
        }

        function onInstances(instances, mbean) {
            if (instances) {
                var parsedMBean = Core.parseMBean(mbean);
                var instanceName = 'root';
                if ('attributes' in parsedMBean) {
                    if ('name' in parsedMBean['attributes']) {
                        instanceName = parsedMBean['attributes']['name'];
                    }
                }

                //log.debug("mbean: ", Core.parseMBean(mbean));
                //log.debug("Instances: ", instances);
                // the name is the first child
                var rootInstance = instances[instanceName];
                $scope.data.name = rootInstance.Name;
                $scope.data.state = rootInstance.State;
                $scope.data.root = rootInstance["Is Root"];
                $scope.data.location = rootInstance.Location;
                $scope.data.sshPort = rootInstance["SSH Port"];
                $scope.data.rmiRegistryPort = rootInstance["RMI Registry Port"];
                $scope.data.rmiServerPort = rootInstance["RMI Server Port"];
                $scope.data.pid = rootInstance.Pid;

                // we need to get these data from the system mbean
                $scope.data.version = "?";
                $scope.data.startLevel = "?";
                $scope.data.framework = "?";
                $scope.data.frameworkVersion = "?";

                var systemMbean = "org.apache.karaf:type=system,name=" + rootInstance.Name;

                // get more data, and its okay to do this synchronously
                var response = jolokia.request({
                    type: "read", mbean: systemMbean,
                    attribute: ["StartLevel", "Framework", "Version"] }, onSuccess(null));

                var obj = response.value;
                if (obj) {
                    $scope.data.version = obj.Version;
                    $scope.data.startLevel = obj.StartLevel;
                    $scope.data.framework = obj.Framework;
                }

                // and the osgi framework version is the bundle version
                var response2 = jolokia.search("osgi.core:type=bundleState,*", onSuccess(null));
                if (angular.isArray(response2)) {
                    var mbean = response2[0];
                    if (mbean) {
                        // get more data, and its okay to do this synchronously
                        var response3 = jolokia.request({ type: 'exec', mbean: mbean, operation: 'getVersion(long)', arguments: [0] }, onSuccess(null));
                        var obj3 = response3.value;
                        if (obj3) {
                            $scope.data.frameworkVersion = obj3;
                        }
                    }
                }
            }

            // ensure web page is updated
            Core.$apply($scope);
        }
    }
    Karaf.ServerController = ServerController;
})(Karaf || (Karaf = {}));
/**
* @module Karaf
*/
var Karaf;
(function (Karaf) {
    function ScrComponentsController($scope, $location, workspace, jolokia) {
        $scope.component = empty();

        // caches last jolokia result
        $scope.result = [];

        // rows in components table
        $scope.components = [];

        // selected components
        $scope.selectedComponents = [];

        $scope.scrOptions = {
            //plugins: [searchProvider],
            data: 'components',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: false
            },
            sortInfo: { fields: ['Name'], directions: ['asc'] },
            selectedItems: $scope.selectedComponents,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'Name',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText"><a href="#/osgi/scr-component/{{row.entity.Name}}?p=container">{{row.getProperty(col.field)}}</a></div>',
                    width: 400
                },
                {
                    field: 'State',
                    displayName: 'State',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200
                }
            ]
        };

        var scrMBean = Karaf.getSelectionScrMBean(workspace);
        if (scrMBean) {
            render(Karaf.getAllComponents(workspace, jolokia));
        }

        $scope.activate = function () {
            $scope.selectedComponents.forEach(function (component) {
                Karaf.activateComponent(workspace, jolokia, component.Name, function () {
                    console.log("Activated!");
                }, function () {
                    console.log("Failed to activate!");
                });
            });
        };

        $scope.deactivate = function () {
            $scope.selectedComponents.forEach(function (component) {
                Karaf.deactivateComponent(workspace, jolokia, component.Name, function () {
                    console.log("Deactivated!");
                }, function () {
                    console.log("Failed to deactivate!");
                });
            });
        };

        function empty() {
            return [
                {
                    Name: "",
                    Status: false }
            ];
        }

        function render(components) {
            if (!Object.equal($scope.result, components)) {
                $scope.components = components;
                $scope.result = $scope.components;
                Core.$apply($scope);
            }
        }
    }
    Karaf.ScrComponentsController = ScrComponentsController;
})(Karaf || (Karaf = {}));
/**
* @module Karaf
*/
var Karaf;
(function (Karaf) {
    function ScrComponentController($scope, $location, workspace, jolokia, $routeParams) {
        $scope.name = $routeParams.name;
        populateTable();

        function populateTable() {
            $scope.row = Karaf.getComponentByName(workspace, jolokia, $scope.name);
            Core.$apply($scope);
        }

        $scope.activate = function () {
            Karaf.activateComponent(workspace, jolokia, $scope.row['Name'], function () {
                console.log("Activated!");
            }, function () {
                console.log("Failed to activate!");
            });
        };

        $scope.deactivate = function () {
            Karaf.deactivateComponent(workspace, jolokia, $scope.row['Name'], function () {
                console.log("Deactivated!");
            }, function () {
                console.log("Failed to deactivate!");
            });
        };
    }
    Karaf.ScrComponentController = ScrComponentController;
})(Karaf || (Karaf = {}));
/**
* @module Karaf
*/
var Karaf;
(function (Karaf) {
    function FeaturesController($scope, $location, workspace, jolokia) {
        $scope.hasFabric = Fabric.hasFabric(workspace);
        $scope.responseJson = '';
        $scope.filter = '';

        $scope.installedFeatures = [];

        $scope.features = [];
        $scope.repositories = [];
        $scope.selectedRepositoryId = '';
        $scope.selectedRepository = {};

        $scope.newRepositoryURI = '';

        $scope.init = function () {
            var selectedRepositoryId = $location.search()['repositoryId'];
            if (selectedRepositoryId) {
                $scope.selectedRepositoryId = selectedRepositoryId;
            }

            var filter = $location.search()['filter'];
            if (filter) {
                $scope.filter = filter;
            }
        };

        $scope.init();

        $scope.$watch('selectedRepository', function (newValue, oldValue) {
            //log.debug("selectedRepository: ", $scope.selectedRepository);
            if (newValue !== oldValue) {
                if (!newValue) {
                    $scope.selectedRepositoryId = '';
                } else {
                    $scope.selectedRepositoryId = newValue['repository'];
                }
                $location.search('repositoryId', $scope.selectedRepositoryId);
            }
        }, true);

        $scope.$watch('filter', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $location.search('filter', newValue);
            }
        });

        var featuresMBean = Karaf.getSelectionFeaturesMBean(workspace);

        Karaf.log.debug("Features mbean: ", featuresMBean);

        if (featuresMBean) {
            Core.register(jolokia, $scope, {
                type: 'read', mbean: featuresMBean
            }, onSuccess(render));
        }

        $scope.inSelectedRepository = function (feature) {
            if (!$scope.selectedRepository || !('repository' in $scope.selectedRepository)) {
                return "";
            }
            if (!feature || !('RepositoryName' in feature)) {
                return "";
            }
            if (feature['RepositoryName'] === $scope.selectedRepository['repository']) {
                return "in-selected-repository";
            }
            return "";
        };

        $scope.isValidRepository = function () {
            return Core.isBlank($scope.newRepositoryURI);
        };

        $scope.installRepository = function () {
            var repoURL = $scope.newRepositoryURI;
            notification('info', 'Adding feature repository URL');
            Karaf.installRepository(workspace, jolokia, repoURL, function () {
                notification('success', 'Added feature repository URL');
                $scope.selectedRepository = {};
                $scope.selectedRepositoryId = '';
                $scope.responseJson = null;
                $scope.triggerRefresh();
            }, function (response) {
                Karaf.log.error('Failed to add feature repository URL ', repoURL, ' due to ', response.error);
                Karaf.log.info('stack trace: ', response.stacktrace);
                Core.$apply($scope);
            });
        };

        $scope.uninstallRepository = function () {
            var repoURI = $scope.selectedRepository['uri'];
            notification('info', 'Removing feature repository ' + repoURI);
            Karaf.uninstallRepository(workspace, jolokia, repoURI, function () {
                notification('success', 'Removed feature repository ' + repoURI);
                $scope.responseJson = null;
                $scope.selectedRepositoryId = '';
                $scope.selectedRepository = {};
                $scope.triggerRefresh();
            }, function (response) {
                Karaf.log.error('Failed to remove feature repository ', repoURI, ' due to ', response.error);
                Karaf.log.info('stack trace: ', response.stacktrace);
                Core.$apply($scope);
            });
        };

        $scope.triggerRefresh = function () {
            jolokia.request({
                type: 'read',
                method: 'POST',
                mbean: featuresMBean
            }, onSuccess(render));
        };

        $scope.install = function (feature) {
            if ($scope.hasFabric) {
                return;
            }

            //$('.popover').remove();
            notification('info', 'Installing feature ' + feature.Name);
            Karaf.installFeature(workspace, jolokia, feature.Name, feature.Version, function () {
                notification('success', 'Installed feature ' + feature.Name);
                $scope.installedFeatures.add(feature);
                $scope.responseJson = null;
                $scope.triggerRefresh();
                //Core.$apply($scope);
            }, function (response) {
                Karaf.log.error('Failed to install feature ', feature.Name, ' due to ', response.error);
                Karaf.log.info('stack trace: ', response.stacktrace);
                Core.$apply($scope);
            });
        };

        $scope.uninstall = function (feature) {
            if ($scope.hasFabric) {
                return;
            }

            //$('.popover').remove();
            notification('info', 'Uninstalling feature ' + feature.Name);
            Karaf.uninstallFeature(workspace, jolokia, feature.Name, feature.Version, function () {
                notification('success', 'Uninstalled feature ' + feature.Name);
                $scope.installedFeatures.remove(feature);
                $scope.responseJson = null;
                $scope.triggerRefresh();
                //Core.$apply($scope);
            }, function (response) {
                Karaf.log.error('Failed to uninstall feature ', feature.Name, ' due to ', response.error);
                Karaf.log.info('stack trace: ', response.stacktrace);
                Core.$apply($scope);
            });
        };

        $scope.filteredRows = ['Bundles', 'Configurations', 'Configuration Files', 'Dependencies'];

        $scope.showRow = function (key, value) {
            if ($scope.filteredRows.any(key)) {
                return false;
            }

            if (angular.isArray(value)) {
                if (value.length === 0) {
                    return false;
                }
            }

            if (angular.isString(value)) {
                if (Core.isBlank(value)) {
                    return false;
                }
            }

            if (angular.isObject(value)) {
                if (!value || Object.equal(value, {})) {
                    return false;
                }
            }

            return true;
        };

        $scope.installed = function (installed) {
            var answer = Core.parseBooleanValue(installed);
            return answer;
        };

        $scope.showValue = function (value) {
            if (angular.isArray(value)) {
                var answer = ['<ul class="zebra-list">'];
                value.forEach(function (v) {
                    answer.push('<li>' + v + '</li>');
                });
                answer.push('</ul>');
                return answer.join('\n');
            }
            if (angular.isObject(value)) {
                var answer = ['<table class="table">', '<tbody>'];

                angular.forEach(value, function (value, key) {
                    answer.push('<tr>');
                    answer.push('<td>' + key + '</td>');
                    answer.push('<td>' + value + '</td>');
                    answer.push('</tr>');
                });

                answer.push('</tbody>');
                answer.push('</table>');

                return answer.join('\n');
            }
            return "" + value;
        };

        $scope.getStateStyle = function (feature) {
            if (Core.parseBooleanValue(feature.Installed)) {
                return "badge badge-success";
            }
            return "badge";
        };

        $scope.filterFeature = function (feature) {
            if (Core.isBlank($scope.filter)) {
                return true;
            }
            if (feature.Id.has($scope.filter)) {
                return true;
            }
            return false;
        };

        function render(response) {
            var responseJson = angular.toJson(response.value);
            if ($scope.responseJson !== responseJson) {
                $scope.responseJson = responseJson;

                //log.debug("Got response: ", response.value);
                if (response['value']['Features'] === null) {
                    $scope.featuresError = true;
                } else {
                    $scope.featuresError = false;
                }

                $scope.features = [];
                $scope.repositories = [];

                var features = [];
                var repositories = [];

                Karaf.populateFeaturesAndRepos(response.value, features, repositories);

                var installedFeatures = features.filter(function (f) {
                    return Core.parseBooleanValue(f.Installed);
                });
                var uninstalledFeatures = features.filter(function (f) {
                    return !Core.parseBooleanValue(f.Installed);
                });

                //log.debug("repositories: ", repositories);
                $scope.installedFeatures = installedFeatures.sortBy(function (f) {
                    return f['Name'];
                });
                uninstalledFeatures = uninstalledFeatures.sortBy(function (f) {
                    return f['Name'];
                });

                repositories.sortBy('id').forEach(function (repo) {
                    $scope.repositories.push({
                        repository: repo['id'],
                        uri: repo['uri'],
                        features: uninstalledFeatures.filter(function (f) {
                            return f['RepositoryName'] === repo['id'];
                        })
                    });
                });

                if (!Core.isBlank($scope.newRepositoryURI)) {
                    var selectedRepo = repositories.find(function (r) {
                        return r['uri'] === $scope.newRepositoryURI;
                    });
                    if (selectedRepo) {
                        $scope.selectedRepositoryId = selectedRepo['id'];
                    }
                    $scope.newRepositoryURI = '';
                }

                if (Core.isBlank($scope.selectedRepositoryId)) {
                    $scope.selectedRepository = $scope.repositories.first();
                } else {
                    $scope.selectedRepository = $scope.repositories.find(function (r) {
                        return r.repository === $scope.selectedRepositoryId;
                    });
                }

                Core.$apply($scope);
            }
        }
    }
    Karaf.FeaturesController = FeaturesController;
})(Karaf || (Karaf = {}));
/**
* @module Jclouds
* @main Jclouds
*/
var Jclouds;
(function (Jclouds) {
    var pluginName = 'jclouds';

    angular.module(pluginName, ['bootstrap', 'ngResource', 'ngGrid', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/jclouds/api', { templateUrl: 'app/jclouds/html/api-list.html' }).when('/jclouds/api/:apiId', { templateUrl: 'app/jclouds/html/api.html' }).when('/jclouds/provider', { templateUrl: 'app/jclouds/html/provider-list.html' }).when('/jclouds/provider/:providerId', { templateUrl: 'app/jclouds/html/provider.html' }).when('/jclouds/compute/service', { templateUrl: 'app/jclouds/html/compute/compute-service-list.html' }).when('/jclouds/compute/service/:computeId', { templateUrl: 'app/jclouds/html/compute/compute-service.html' }).when('/jclouds/compute/node/:computeId', { templateUrl: 'app/jclouds/html/compute/node-list.html' }).when('/jclouds/compute/node/:computeId/*nodeId', { templateUrl: 'app/jclouds/html/compute/node.html' }).when('/jclouds/compute/image/:computeId', { templateUrl: 'app/jclouds/html/compute/image-list.html' }).when('/jclouds/compute/image/:computeId/*imageId', { templateUrl: 'app/jclouds/html/compute/image.html' }).when('/jclouds/compute/hardware/:computeId', { templateUrl: 'app/jclouds/html/compute/hardware-list.html' }).when('/jclouds/compute/hardware/:computeId/*hardwareId', { templateUrl: 'app/jclouds/html/compute/hardware.html' }).when('/jclouds/compute/location/:computeId', { templateUrl: 'app/jclouds/html/compute/location-list.html' }).when('/jclouds/compute/location/:computeId/*locationId', { templateUrl: 'app/jclouds/html/compute/location.html' }).when('/jclouds/blobstore/service', { templateUrl: 'app/jclouds/html/blobstore/blobstore-service-list.html' }).when('/jclouds/blobstore/service/:blobstoreId', { templateUrl: 'app/jclouds/html/blobstore/blobstore-service.html' }).when('/jclouds/blobstore/location/:blobstoreId', { templateUrl: 'app/jclouds/html/blobstore/location-list.html' }).when('/jclouds/blobstore/location/:blobstoreId/*locationId', { templateUrl: 'app/jclouds/html/blobstore/location.html' }).when('/jclouds/blobstore/container/:blobstoreId', { templateUrl: 'app/jclouds/html/blobstore/container-list.html' }).when('/jclouds/blobstore/container/:blobstoreId/:containerId', { templateUrl: 'app/jclouds/html/blobstore/container.html' }).when('/jclouds/blobstore/container/:blobstoreId/:containerId/*directory', { templateUrl: 'app/jclouds/html/blobstore/container.html' });
    }).run(function (workspace, viewRegistry, helpRegistry) {
        viewRegistry['jclouds'] = "app/jclouds/html/layoutJclouds.html";
        helpRegistry.addUserDoc('jclouds', 'app/' + 'jclouds' + '/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("org.jclouds");
        });

        workspace.topLevelTabs.push({
            id: "jclouds",
            content: "jclouds",
            title: "Visualise and manage the Jclouds Compute/BlobStore providers and apis",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("org.jclouds");
            },
            href: function () {
                return "#/jclouds/api";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("jclouds");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ApiListController($scope, $location, workspace, jolokia) {
        $scope.result = {};
        $scope.apis = [];
        $scope.type = "";
        $scope.types = ["", "blobstore", "compute", "loadbalancer"];

        var key = $location.search()['type'];
        if (key) {
            $scope.type = key;
        }

        // selected apis
        $scope.selectedApis = [];

        /*
        var SearchProvider = function (scope, location) {
        var self = this;
        self.scope = scope;
        self.location = location;
        
        self.callback = function (newValue, oldValue) {
        if (newValue === oldValue) {
        return;
        }
        self.scope.apis = apisOfType(self.scope.apis, self.scope.type);
        self.scope.type = setSelect(self.scope.type, self.scope.types);
        
        var q = location.search();
        q['type'] = self.scope.type;
        location.search(q);
        self.evalFilter();
        };
        
        self.scope.$watch('type', self.callback);
        
        self.init = function (childScope, grid) {
        self.grid = grid;
        self.childScope = childScope;
        grid.searchProvider = self;
        };
        
        self.evalFilter = function () {
        var byType = self.grid.sortedData;
        if (self.scope.type !== "") {
        byType = self.grid.sortedData.findAll(function (item) {
        return item["type"] === self.scope.type
        });
        }
        self.grid.filteredData = byType;
        self.grid.rowFactory.filteredDataChanged();
        };
        }
        
        var searchProvider = new SearchProvider($scope, $location);
        */
        $scope.apiTable = {
            //plugins: [searchProvider],
            data: 'apis',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedApis,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/api/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'name',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 350
                },
                {
                    field: 'type',
                    displayName: 'Type',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'read', mbean: Jclouds.getSelectionJcloudsMBean(workspace)
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.apis = $scope.result["Apis"];
                Jclouds.populateTypeForApis($scope.apis);
                Core.$apply($scope);
            }
        }
    }
    Jclouds.ApiListController = ApiListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function HardwareController($scope, $filter, workspace, $routeParams) {
        $scope.computeId = $routeParams.computeId;
        $scope.hardwareId = $routeParams.hardwareId;

        updateTableContents();

        $scope.processorsTable = {
            plugins: [],
            data: 'processors',
            showFilter: false,
            displayFooter: false,
            displaySelectionCheckbox: false,
            showColumnMenu: false,
            rowHeight: 32,
            columnDefs: [
                {
                    field: 'cores',
                    displayName: 'Cores',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 50,
                    resizable: false
                },
                {
                    field: 'speed',
                    displayName: 'Speed',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100,
                    resizable: false
                }
            ]
        };

        $scope.volumesTable = {
            plugins: [],
            data: 'volumes',
            showFilter: false,
            displayFooter: false,
            displaySelectionCheckbox: false,
            showColumnMenu: false,
            rowHeight: 32,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100,
                    resizable: false
                },
                {
                    field: 'type',
                    displayName: 'Type',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100,
                    resizable: false
                },
                {
                    field: 'device',
                    displayName: 'Device',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100,
                    resizable: false
                },
                {
                    field: 'size',
                    displayName: 'Size',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100,
                    resizable: false
                },
                {
                    field: 'bootDevice',
                    displayName: 'Boot Device',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100,
                    resizable: false
                },
                {
                    field: 'durable',
                    displayName: 'Durable',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100,
                    resizable: false
                }
            ]
        };

        function setHardwareProfiles(hardwareProfiles) {
            $scope.row = findHardwareById(hardwareProfiles, $scope.hardwareId);
            $scope.processors = $scope.row["processors"];
            $scope.volumes = $scope.row["volumes"];
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var jolokia = workspace.jolokia;
            var computeMbean = Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId);

            if (computeMbean) {
                setHardwareProfiles(jolokia.request({ type: 'exec', mbean: computeMbean, operation: 'listHardwareProfiles()' }).value);
            }
        }

        function findHardwareById(hardwareProfiles, id) {
            return hardwareProfiles.find(function (hardware) {
                return hardware.id === id;
            });
        }
    }
    Jclouds.HardwareController = HardwareController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ImageListController($scope, $location, workspace, jolokia, $routeParams) {
        $scope.computeId = $routeParams.computeId;

        $scope.result = {};
        $scope.images = [];

        $scope.operatingSystemFamily = "";
        $scope.operatingSystemFamilies = [];
        var os = $location.search()['os'];
        if (os) {
            $scope.operatingSystemFamily = os;
        }

        // selected images
        $scope.selectedImages = [];

        /*
        var SearchProvider = function (scope, location) {
        var self = this;
        self.scope = scope;
        self.location = location;
        
        self.callback = function (newValue, oldValue) {
        if (newValue === oldValue) {
        return;
        }
        self.scope.images = filterImages(self.scope.images, self.scope.operatingSystemFamily);
        self.scope.operatingSystemFamily = setSelect(self.scope.operatingSystemFamily, self.scope.operatingSystemFamilies);
        
        var q = location.search();
        q['os'] = self.scope.operatingSystemFamily;
        location.search(q);
        self.evalFilter();
        };
        
        self.scope.$watch('operatingSystemFamily', self.callback);
        self.scope.$watch('location', self.callback);
        
        self.init = function (childScope, grid) {
        self.grid = grid;
        self.childScope = childScope;
        grid.searchProvider = self;
        };
        
        self.evalFilter = function () {
        var byOperatingSystemFamily = self.grid.sortedData;
        if (self.scope.operatingSystemFamily !== "") {
        byOperatingSystemFamily = self.grid.sortedData.findAll(function (item) {
        return item["operatingSystem"]["family"] === self.scope.operatingSystemFamily
        });
        }
        
        self.grid.filteredData = byOperatingSystemFamily;
        self.grid.rowFactory.filteredDataChanged();
        };
        }
        
        var searchProvider = new SearchProvider($scope, $location);
        */
        $scope.imageTable = {
            //plugins: [searchProvider],
            data: 'images',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedImages,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/compute/image/{{computeId}}/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'name',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'operatingSystem.family',
                    displayName: 'Operating System',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200
                },
                {
                    field: 'operatingSystem.is64Bit',
                    displayName: '64 bit',
                    cellTemplate: '<div class="ngCellText pagination-centered"><i class="icon1point5x {{is64BitIcon(row.getProperty(col.field))}}"></i></div>',
                    width: 200
                },
                {
                    field: 'status',
                    displayName: 'Status',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 300
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'exec', mbean: Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId), operation: 'listImages()'
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.images = $scope.result;
                $scope.operatingSystemFamilies = extractOperatingSystemFamilies($scope.images);
                Core.$apply($scope);
            }
        }

        function extractOperatingSystemFamilies(images) {
            var operatingSystemFamilies = [];
            operatingSystemFamilies.push("");
            angular.forEach(images, function (image) {
                var operatingSystemFamily = image["operatingSystem"]["family"];

                operatingSystemFamilies.push(operatingSystemFamily);
            });
            return operatingSystemFamilies.unique();
        }

        $scope.is64BitIcon = function (is64bit) {
            if (is64bit) {
                return 'icon-thumbs-up';
            } else {
                return 'icon-thumbs-down';
            }
        };
    }
    Jclouds.ImageListController = ImageListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function NodeController($scope, $filter, workspace, $routeParams) {
        $scope.computeId = $routeParams.computeId;
        $scope.nodeId = $routeParams.nodeId;

        updateTableContents();

        function setNode(api) {
            $scope.row = api;
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var computeMbean = Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId);
            var jolokia = workspace.jolokia;

            if (computeMbean) {
                setNode(jolokia.request({ type: 'exec', mbean: computeMbean, operation: 'getNode(java.lang.String)', arguments: [$scope.nodeId] }).value);
            }
        }

        $scope.resume = function () {
            Jclouds.resumeNode(workspace, workspace.jolokia, $scope.computeId, $scope.nodeId, function () {
                console.log("Resumed!");
            }, function () {
                console.log("Failed to resume!");
            });
        };

        $scope.suspend = function () {
            Jclouds.suspendNode(workspace, workspace.jolokia, $scope.computeId, $scope.nodeId, function () {
                console.log("Suspended!");
            }, function () {
                console.log("Failed to suspend!");
            });
        };

        $scope.reboot = function () {
            Jclouds.rebootNode(workspace, workspace.jolokia, $scope.computeId, $scope.nodeId, function () {
                console.log("Rebooted!");
            }, function () {
                console.log("Failed to reboot!");
            });
        };

        $scope.destroy = function () {
            Jclouds.destroyNode(workspace, workspace.jolokia, $scope.computeId, $scope.nodeId, function () {
                console.log("Destroyed!");
            }, function () {
                console.log("Failed to destroy!");
            });
        };
    }
    Jclouds.NodeController = NodeController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ImageController($scope, $filter, workspace, $routeParams) {
        $scope.computeId = $routeParams.computeId;
        $scope.imageId = $routeParams.imageId;

        updateTableContents();

        function setImage(api) {
            $scope.row = api;
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var jolokia = workspace.jolokia;
            var computeMbean = Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId);

            if (computeMbean) {
                setImage(jolokia.request({ type: 'exec', mbean: computeMbean, operation: 'getImage(java.lang.String)', arguments: [$scope.imageId] }).value);
            }
        }
    }
    Jclouds.ImageController = ImageController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ComputeLocationController($scope, $filter, workspace, $routeParams) {
        $scope.computeId = $routeParams.computeId;
        $scope.locationId = $routeParams.locationId;

        updateTableContents();

        function setLocationProfiles(locationProfiles) {
            $scope.row = findLocationById(locationProfiles, $scope.locationId);
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var jolokia = workspace.jolokia;
            var computeMbean = Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId);

            if (computeMbean) {
                setLocationProfiles(jolokia.request({ type: 'exec', mbean: computeMbean, operation: 'listAssignableLocations()' }).value);
            }
        }

        function findLocationById(locationProfiles, id) {
            return locationProfiles.find(function (location) {
                return location.id === id;
            });
        }
    }
    Jclouds.ComputeLocationController = ComputeLocationController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ComputeLocationListController($scope, $location, workspace, jolokia, $routeParams) {
        $scope.computeId = $routeParams.computeId;

        $scope.result = {};
        $scope.locations = [];

        // selected locations
        $scope.selectedLocations = [];

        $scope.locationTable = {
            plugins: [],
            data: 'locations',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedLocations,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/compute/location/{{computeId}}/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'description',
                    displayName: 'Description',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'exec', mbean: Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId), operation: 'listAssignableLocations()'
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.locations = $scope.result;
                Core.$apply($scope);
            }
        }
    }
    Jclouds.ComputeLocationListController = ComputeLocationListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function HardwareListController($scope, $location, workspace, jolokia, $routeParams) {
        $scope.computeId = $routeParams.computeId;

        $scope.result = {};
        $scope.hardwares = [];

        // selected hardwares
        $scope.selectedHardwares = [];

        $scope.hardwareTable = {
            plugins: [],
            data: 'hardwares',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedHardwares,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/compute/hardware/{{computeId}}/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'name',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'ram',
                    displayName: 'Ram',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'hypervisor',
                    displayName: 'Hypervisor',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'exec', mbean: Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId), operation: 'listHardwareProfiles()'
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.hardwares = $scope.result;
                Core.$apply($scope);
            }
        }

        $scope.is64BitIcon = function (is64bit) {
            if (is64bit) {
                return 'icon-thumbs-up';
            } else {
                return 'icon-thumbs-down';
            }
        };
    }
    Jclouds.HardwareListController = HardwareListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ComputeListController($scope, $location, workspace, jolokia) {
        $scope.result = {};
        $scope.computeServiceIds = [];
        $scope.computeServices = [];

        $scope.computeTable = {
            plugins: [],
            data: 'computeServices',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedComputeServices,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'name',
                    displayName: 'Service Name',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/compute/service/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'providerId',
                    displayName: 'Proivder',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'identity',
                    displayName: 'Identity',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                }
            ]
        };

        render(Jclouds.listJcloudsMBeanNameOfType(workspace, "compute"));

        function render(response) {
            if (!Object.equal($scope.result, response)) {
                $scope.result = response;
                $scope.computeServiceIds = $scope.result;
                var computeServices = [];
                angular.forEach($scope.computeServiceIds, function (id) {
                    computeServices.push(Jclouds.findContextByName(workspace, id));
                });
                $scope.computeServices = computeServices;
                Core.$apply($scope);
            }
        }
    }
    Jclouds.ComputeListController = ComputeListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function NodeListController($scope, $location, workspace, jolokia, $routeParams) {
        $scope.computeId = $routeParams.computeId;

        $scope.result = {};
        $scope.nodes = [];

        $scope.group = "";
        $scope.groups = [];

        $scope.location = "";
        $scope.locations = [];

        var grp = $location.search()['group'];
        if (grp) {
            $scope.group = grp;
        }

        var loc = $location.search()['location'];
        if (loc) {
            $scope.location = loc;
        }

        // selected nodes
        $scope.selectedNodes = [];

        /*
        var SearchProvider = function (scope, location) {
        var self = this;
        self.scope = scope;
        self.location = location;
        
        self.callback = function (newValue, oldValue) {
        if (newValue === oldValue) {
        return;
        }
        self.scope.nodes = filterNodes(self.scope.nodes, self.scope.group, self.scope.location);
        self.scope.group = setSelect(self.scope.group, self.scope.groups);
        self.scope.location = setSelect(self.scope.location, self.scope.locations);
        
        var q = location.search();
        q['group'] = self.scope.group;
        q['location'] = self.scope.location;
        location.search(q);
        self.evalFilter();
        };
        
        self.scope.$watch('group', self.callback);
        self.scope.$watch('location', self.callback);
        
        self.init = function (childScope, grid) {
        self.grid = grid;
        self.childScope = childScope;
        grid.searchProvider = self;
        };
        
        self.evalFilter = function () {
        var byGroup = self.grid.sortedData;
        if (self.scope.group !== "") {
        byGroup = self.grid.sortedData.findAll(function (item) {
        return item["group"] === self.scope.group
        });
        }
        
        var byLocation = byGroup;
        if (self.scope.location !== "") {
        byLocation = self.grid.sortedData.findAll(function (item) {
        return item["locationId"] === self.scope.location
        });
        }
        
        self.grid.filteredData = byLocation;
        self.grid.rowFactory.filteredDataChanged();
        };
        }
        
        var searchProvider = new SearchProvider($scope, $location);
        */
        $scope.nodeTable = {
            //plugins: [searchProvider],
            data: 'nodes',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedNodes,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/compute/node/{{computeId}}/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'group',
                    displayName: 'Group',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/compute/node/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'operatingSystem.family',
                    displayName: 'Operating System',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200
                },
                {
                    field: 'locationId',
                    displayName: 'Location',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200
                },
                {
                    field: 'hostname',
                    displayName: 'Host Name',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 300
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'exec', mbean: Jclouds.getSelectionJcloudsComputeMBean(workspace, $scope.computeId), operation: 'listNodes()'
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.nodes = $scope.result;
                $scope.locations = extractLocations($scope.nodes);
                $scope.groups = extractGroups($scope.nodes);
                Core.$apply($scope);
            }
        }

        function extractGroups(nodes) {
            var groups = [];
            groups.push("");
            angular.forEach(nodes, function (node) {
                var group = node["group"];
                groups.push(group);
            });
            return groups.unique();
        }

        function extractLocations(nodes) {
            var locations = [];
            locations.push("");
            angular.forEach(nodes, function (node) {
                var location = node["locationId"];
                locations.push(location);
            });
            return locations.unique();
        }

        $scope.resume = function () {
            $scope.selectedNodes.forEach(function (node) {
                Jclouds.resumeNode(workspace, jolokia, $scope.computeId, node.id, function () {
                    console.log("Resumed!");
                }, function () {
                    console.log("Failed to resume!");
                });
            });
        };

        $scope.suspend = function () {
            $scope.selectedNodes.forEach(function (node) {
                Jclouds.suspendNode(workspace, jolokia, $scope.computeId, node.id, function () {
                    console.log("Suspended!");
                }, function () {
                    console.log("Failed to suspend!");
                });
            });
        };

        $scope.reboot = function () {
            $scope.selectedNodes.forEach(function (node) {
                Jclouds.rebootNode(workspace, jolokia, $scope.computeId, node.id, function () {
                    console.log("Rebooted!");
                }, function () {
                    console.log("Failed to reboot!");
                });
            });
        };

        $scope.destroy = function () {
            $scope.selectedNodes.forEach(function (node) {
                Jclouds.destroyNode(workspace, jolokia, $scope.computeId, node.id, function () {
                    console.log("Destroyed!");
                }, function () {
                    console.log("Failed to destroy!");
                });
            });
        };
    }
    Jclouds.NodeListController = NodeListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ComputeNavigationController($scope, $routeParams, workspace) {
        $scope.computeId = $routeParams.computeId;

        $scope.isActive = function (nav) {
            if (angular.isString(nav))
                return workspace.isLinkActive(nav);
            var fn = nav.isActive;
            if (fn) {
                return fn(workspace);
            }
            return workspace.isLinkActive(nav.href());
        };
    }
    Jclouds.ComputeNavigationController = ComputeNavigationController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    Array.prototype.unique = function () {
        var a = [], l = this.length;
        for (var i = 0; i < l; i++) {
            for (var j = i + 1; j < l; j++)
                if (this[i] === this[j])
                    j = ++i;
            a.push(this[i]);
        }
        return a;
    };

    function setSelect(selection, group) {
        if (!angular.isDefined(selection)) {
            return group[0];
        }
        var answer = group.findIndex(function (item) {
            return item === selection;
        });
        if (answer !== -1) {
            return group[answer];
        } else {
            return group[0];
        }
    }
    Jclouds.setSelect = setSelect;

    /**
    * Returns context by name.
    * @method fundContextByName
    * @for Jclouds
    * @param {Workspace} workspace
    * @param {String} name
    * @return {any}
    */
    function findContextByName(workspace, name) {
        var jcloudsMBean = getSelectionJcloudsMBean(workspace);
        var response = workspace.jolokia.request({ type: 'read', mbean: jcloudsMBean });
        return response.value["Contexts"].find(function (context) {
            return context.name === name;
        });
    }
    Jclouds.findContextByName = findContextByName;

    /**
    * Add the type attribute to an array of apis
    * @method populateTypeForApis
    * @for Jclouds
    * @param {Array} apis
    */
    function populateTypeForApis(apis) {
        angular.forEach(apis, function (api) {
            populateTypeForApi(api);
        });
    }
    Jclouds.populateTypeForApis = populateTypeForApis;

    /**
    * Add the type attribute to a singe api.
    * @method populateTypeForApi
    * @for Jclouds
    * @param {any} api
    */
    function populateTypeForApi(api) {
        var views = api["views"];
        var found = false;
        angular.forEach(views, function (view) {
            if (!found) {
                if (view.has("blob")) {
                    api["type"] = "blobstore";
                    found = true;
                } else if (view.has("compute")) {
                    api["type"] = "compute";
                    found = true;
                }
            }
        });
    }
    Jclouds.populateTypeForApi = populateTypeForApi;

    /**
    * Filters Images based on Operating System.
    * @method filterImages
    * @for Jclouds
    * @param {Array} images
    * @param {any} operatingSystemFamily
    */
    function filterImages(images, operatingSystemFamily) {
        if (operatingSystemFamily === "") {
            return images;
        } else {
            return images.findAll(function (image) {
                return image["operatingSystem"]["family"] === operatingSystemFamily;
            });
        }
    }
    Jclouds.filterImages = filterImages;

    /**
    * Filters Nodes based on type.
    * @method filterNodes
    * @for Jclouds
    * @param {Array} nodes
    * @param {String} group
    * @param {String} location}
    * @return {any}
    */
    function filterNodes(nodes, group, location) {
        var filteredNodes = [];
        if (group === "") {
            filteredNodes = nodes;
        } else {
            filteredNodes = nodes.findAll(function (node) {
                return node.group === group;
            });
        }

        if (location === "") {
            return filteredNodes;
        } else {
            return filteredNodes.findAll(function (node) {
                return node.locationId === location;
            });
        }
    }
    Jclouds.filterNodes = filterNodes;

    function resumeNode(workspace, jolokia, compute, id, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionJcloudsComputeMBean(workspace, compute),
            operation: 'resumeNode(java.lang.String)',
            arguments: [id]
        }, onSuccess(success, { error: error }));
    }
    Jclouds.resumeNode = resumeNode;

    function suspendNode(workspace, jolokia, compute, id, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionJcloudsComputeMBean(workspace, compute),
            operation: 'suspendNode(java.lang.String)',
            arguments: [id]
        }, onSuccess(success, { error: error }));
    }
    Jclouds.suspendNode = suspendNode;

    function rebootNode(workspace, jolokia, compute, id, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionJcloudsComputeMBean(workspace, compute),
            operation: 'rebootNode(java.lang.String)',
            arguments: [id]
        }, onSuccess(success, { error: error }));
    }
    Jclouds.rebootNode = rebootNode;

    function destroyNode(workspace, jolokia, compute, id, success, error) {
        jolokia.request({
            type: 'exec', mbean: getSelectionJcloudsComputeMBean(workspace, compute),
            operation: 'destroyNode(java.lang.String)',
            arguments: [id]
        }, onSuccess(success, { error: error }));
    }
    Jclouds.destroyNode = destroyNode;

    /**
    * Filters Apis based on type.
    * @method apisOfType
    * @for Jclouds
    * @param {Array} apis
    * @param {String} type
    * @return {Array}
    */
    function apisOfType(apis, type) {
        if (type === "") {
            return apis;
        }

        return apis.findAll(function (api) {
            return api.type === type;
        });
    }
    Jclouds.apisOfType = apisOfType;

    /**
    * Add the type attribute to an array of providers.
    * @method populateTypeForProviders
    * @for Jclouds
    * @param {Array} providers
    */
    function populateTypeForProviders(providers) {
        angular.forEach(providers, function (provider) {
            populateTypeForProvider(provider);
        });
    }
    Jclouds.populateTypeForProviders = populateTypeForProviders;

    /**
    * Add the type attribute to a singe provider.
    * @method populateTypeForProvider
    * @for Jclouds
    * @param {any} provider
    */
    function populateTypeForProvider(provider) {
        var views = provider["api"]["views"];
        var found = false;
        angular.forEach(views, function (view) {
            if (!found) {
                if (view.has("blob")) {
                    provider["type"] = "blobstore";
                    found = true;
                } else if (view.has("compute")) {
                    provider["type"] = "compute";
                    found = true;
                }
            }
        });
    }
    Jclouds.populateTypeForProvider = populateTypeForProvider;

    /**
    * Filters Providers based on type.
    * @method providersOfType
    * @for Jclouds
    * @param {Array} providers
    * @param {String} type
    * @return {Array}
    */
    function providersOfType(providers, type) {
        if (type === "") {
            return providers;
        }

        return providers.findAll(function (provider) {
            return provider.type === type;
        });
    }
    Jclouds.providersOfType = providersOfType;

    /**
    * Walks the tree looking in the first child all the way down until we find an objectName
    * @method findFirstObjectName
    * @for Jclouds
    * @param {any} node
    * @return {any}
    */
    function findFirstObjectName(node) {
        if (node) {
            var answer = node.objectName;
            if (answer) {
                return answer;
            } else {
                var children = node.children;
                if (children && children.length) {
                    return findFirstObjectName(children[0]);
                }
            }
        }
        return null;
    }
    Jclouds.findFirstObjectName = findFirstObjectName;

    /**
    * Walks the tree looking for all available names.
    * @method childsOfType
    * @for Jclouds
    * @param {any} node
    * @return {Array}
    */
    function childsOfType(node) {
        var types = [];
        angular.forEach(node.children, function (child) {
            types.push(child.title);
        });

        return types;
    }
    Jclouds.childsOfType = childsOfType;

    /**
    * Jclouds MBeans are all listed under org.jclouds <type> <name>.
    * This method lists all <names> for the specified <type>.
    * @method listJcloudsMBeanNameOfType
    * @for Jclouds
    * @param {Workspace} workspace
    * @param {String} type
    * @return {any}
    */
    function listJcloudsMBeanNameOfType(workspace, type) {
        if (workspace) {
            // lets navigate to the tree item based on paths
            var folder = workspace.tree.navigate("org.jclouds", type);
            return childsOfType(folder);
        }
        return null;
    }
    Jclouds.listJcloudsMBeanNameOfType = listJcloudsMBeanNameOfType;

    /**
    * Returns the Jclouds Core Management MBean
    * @method getSelectionJcloudsMBean
    * @for Jclouds
    * @param {Workspace} workspace
    * @return {String}
    */
    function getSelectionJcloudsMBean(workspace) {
        if (workspace) {
            // lets navigate to the tree item based on paths
            var folder = workspace.tree.navigate("org.jclouds", "management", "core");
            return findFirstObjectName(folder);
        }
        return null;
    }
    Jclouds.getSelectionJcloudsMBean = getSelectionJcloudsMBean;

    /**
    * Returns the Jclouds Compute Management MBean for the specified name.
    * @method getSelectionJcloudsComputeMBean
    * @for Jclouds
    * @param {Workspace} workspace
    * @param {String} name
    * @return {String}
    */
    function getSelectionJcloudsComputeMBean(workspace, name) {
        if (workspace) {
            // lets navigate to the tree item based on paths
            var folder = workspace.tree.navigate("org.jclouds", "compute", name);
            return findFirstObjectName(folder);
        }
        return null;
    }
    Jclouds.getSelectionJcloudsComputeMBean = getSelectionJcloudsComputeMBean;

    /**
    * Returns the Jclouds Compute Management MBean for the specified name.
    * @method getSelectionJcloudsBlobstoreMBean
    * @for Jclouds
    * @param {Workspace} workspace
    * @param {String} name
    * @return {String}
    */
    function getSelectionJcloudsBlobstoreMBean(workspace, name) {
        if (workspace) {
            // lets navigate to the tree item based on paths
            var folder = workspace.tree.navigate("org.jclouds", "blobstore", name);
            return findFirstObjectName(folder);
        }
        return null;
    }
    Jclouds.getSelectionJcloudsBlobstoreMBean = getSelectionJcloudsBlobstoreMBean;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    /**
    * Controller to show the details of a Jclouds provider
    *
    * @method ProviderController
    * @for Jclouds
    * @param {*} $scope
    * @param {ng.IFilterService} $filter
    * @param {Workspace} workspace
    * @param {ng.IRouteParamsService} $routeParams
    */
    function ProviderController($scope, $filter, workspace, $routeParams) {
        $scope.providerId = $routeParams.providerId;

        updateTableContents();

        function setProvider(provider) {
            Jclouds.populateTypeForProvider(provider);
            $scope.row = provider;
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var jcloudsCoreMbean = Jclouds.getSelectionJcloudsMBean(workspace);
            var jolokia = workspace.jolokia;

            if (jcloudsCoreMbean) {
                setProvider(jolokia.request({ type: 'exec', mbean: Jclouds.getSelectionJcloudsMBean(workspace), operation: 'findProviderById(java.lang.String)', arguments: [$scope.providerId] }).value);
            }
        }
    }
    Jclouds.ProviderController = ProviderController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ApiController($scope, $filter, workspace, $routeParams) {
        $scope.apiId = $routeParams.apiId;

        updateTableContents();

        function setApi(api) {
            Jclouds.populateTypeForApi(api);
            $scope.row = api;
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var jcloudsCoreMbean = Jclouds.getSelectionJcloudsMBean(workspace);
            var jolokia = workspace.jolokia;

            if (jcloudsCoreMbean) {
                setApi(jolokia.request({ type: 'exec', mbean: Jclouds.getSelectionJcloudsMBean(workspace), operation: 'findApiById(java.lang.String)', arguments: [$scope.apiId] }).value);
            }
        }
    }
    Jclouds.ApiController = ApiController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function ProviderListController($scope, $location, workspace, jolokia) {
        $scope.result = {};
        $scope.providers = [];
        $scope.type = "";
        $scope.types = ["", "blobstore", "compute", "loadbalancer"];

        var key = $location.search()['type'];
        if (key) {
            $scope.type = key;
        }

        // selected providers
        $scope.selectedProviders = [];

        /*
        var SearchProvider = function (scope, location) {
        var self = this;
        self.scope = scope;
        self.location = location;
        
        self.callback = function (newValue, oldValue) {
        if (newValue === oldValue) {
        return;
        }
        self.scope.providers = providersOfType(self.scope.providers, self.scope.type);
        self.scope.type = setSelect(self.scope.type, self.scope.types);
        
        var q = location.search();
        q['type'] = self.scope.type;
        location.search(q);
        self.evalFilter();
        };
        
        self.scope.$watch('type', self.callback);
        
        self.init = function (childScope, grid) {
        self.grid = grid;
        self.childScope = childScope;
        grid.searchProvider = self;
        };
        
        self.evalFilter = function () {
        var byType = self.grid.sortedData;
        if (self.scope.type !== "") {
        byType = self.grid.sortedData.findAll(function (item) {
        return item["type"] === self.scope.type
        });
        }
        self.grid.filteredData = byType;
        self.grid.rowFactory.filteredDataChanged();
        };
        }
        
        var searchProvider = new SearchProvider($scope, $location);
        */
        $scope.providerTable = {
            //plugins: [searchProvider],
            data: 'providers',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedProviders,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/provider/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'name',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 350
                },
                {
                    field: 'type',
                    displayName: 'Type',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 100
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'read', mbean: Jclouds.getSelectionJcloudsMBean(workspace)
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.providers = $scope.result["Providers"];
                Jclouds.populateTypeForProviders($scope.providers);
                Core.$apply($scope);
            }
        }
    }
    Jclouds.ProviderListController = ProviderListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function BlobstoreContainerController($scope, $filter, workspace, $routeParams) {
        $scope.blobstoreId = $routeParams.blobstoreId;
        $scope.containerId = $routeParams.containerId;
        $scope.directory = $routeParams.directory;
        $scope.contents = [];
        $scope.breadcrumbs = loadBreadcrumbs($scope.blobstoreId, $scope.containerId, $scope.directory);

        $scope.contentTable = {
            data: 'contents',
            displayFooter: false,
            columnDefs: [
                {
                    field: 'name',
                    displayName: 'Content',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/blobstore/container/{{blobstoreId}}/{{containerId}}/{{row.entity.fullpath}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    cellFilter: ""
                },
                {
                    field: 'createdDate',
                    displayName: 'Created',
                    cellFilter: "date:'EEE, MMM d, y : hh:mm:ss a'"
                },
                {
                    field: 'lastModifiedDate',
                    displayName: 'Modified',
                    cellFilter: "date:'EEE, MMM d, y : hh:mm:ss a'"
                }
            ]
        };

        updateTableContents();

        function setContainers(containers) {
            $scope.contents = populatePathAndName(filterContainers(containers, $scope.directory), $scope.directory);
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var jolokia = workspace.jolokia;
            var blobstoreMbean = Jclouds.getSelectionJcloudsBlobstoreMBean(workspace, $scope.blobstoreId);

            if (blobstoreMbean) {
                if ($scope.directory) {
                    setContainers(jolokia.request({ type: 'exec', mbean: blobstoreMbean, operation: 'list(java.lang.String, java.lang.String)', arguments: [$scope.containerId, $scope.directory] }).value);
                } else {
                    setContainers(jolokia.request({ type: 'exec', mbean: blobstoreMbean, operation: 'list(java.lang.String)', arguments: [$scope.containerId] }).value);
                }
            }
        }

        function filterContainers(containers, directory) {
            return containers.filter(function (container) {
                return container.name !== directory;
            });
        }

        function populatePathAndName(containers, directory) {
            var updatedContainers = [];
            angular.forEach(containers, function (container) {
                var updateContainer = container;
                updateContainer.fullpath = container.name;
                if (updateContainer.name.startsWith(directory)) {
                    updateContainer.name = updateContainer.name.substring(directory.length + 1);
                }
                updatedContainers.push(updateContainer);
            });
            return updatedContainers;
        }

        $scope.isBlob = function (container) {
            return container.type === 'BLOB';
        };

        function loadBreadcrumbs(blobstore, container, directory) {
            var href = "#/jclouds/blobstore/container/" + blobstore + "/" + container;
            var breadcrumbs = [
                { href: href, name: "/" + container }
            ];

            var array = directory ? directory.split("/") : [];
            angular.forEach(array, function (name) {
                if (!name.startsWith("/") && !href.endsWith("/")) {
                    href += "/";
                }
                href += name;
                breadcrumbs.push({ href: href, name: name });
            });
            return breadcrumbs;
        }
    }
    Jclouds.BlobstoreContainerController = BlobstoreContainerController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function BlobstoreListController($scope, $location, workspace, jolokia) {
        $scope.result = {};
        $scope.blobstoreServiceIds = [];
        $scope.blobstoreServices = [];

        $scope.blobstoreTable = {
            plugins: [],
            data: 'blobstoreServices',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedBlobstoreServices,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'name',
                    displayName: 'Service Name',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/blobstore/service/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'providerId',
                    displayName: 'Proivder',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'identity',
                    displayName: 'Identity',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                }
            ]
        };

        render(Jclouds.listJcloudsMBeanNameOfType(workspace, "blobstore"));

        function render(response) {
            if (!Object.equal($scope.result, response)) {
                $scope.result = response;
                $scope.blobstoreServiceIds = $scope.result;
                var blobstoreServices = [];
                angular.forEach($scope.blobstoreServiceIds, function (id) {
                    blobstoreServices.push(Jclouds.findContextByName(workspace, id));
                });
                $scope.blobstoreServices = blobstoreServices;
                Core.$apply($scope);
            }
        }
    }
    Jclouds.BlobstoreListController = BlobstoreListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function BlobstoreLocationController($scope, $filter, workspace, $routeParams) {
        $scope.blobstoreId = $routeParams.blobstoreId;
        $scope.locationId = $routeParams.locationId;

        updateTableContents();

        function setLocationProfiles(locationProfiles) {
            $scope.row = findLocationById(locationProfiles, $scope.locationId);
            Core.$apply($scope);
        }
        ;

        function updateTableContents() {
            var jolokia = workspace.jolokia;
            var blobstoreMbean = Jclouds.getSelectionJcloudsBlobstoreMBean(workspace, $scope.blobstoreId);

            if (blobstoreMbean) {
                setLocationProfiles(jolokia.request({ type: 'exec', mbean: blobstoreMbean, operation: 'listAssignableLocations()' }).value);
            }
        }

        function findLocationById(locationProfiles, id) {
            return locationProfiles.find(function (location) {
                return location.id === id;
            });
        }
    }
    Jclouds.BlobstoreLocationController = BlobstoreLocationController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function BlobstoreLocationListController($scope, $location, workspace, jolokia, $routeParams) {
        $scope.blobstoreId = $routeParams.blobstoreId;

        $scope.result = {};
        $scope.locations = [];

        // selected locations
        $scope.selectedLocations = [];

        $scope.locationTable = {
            plugins: [],
            data: 'locations',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedLocations,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Id',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/blobstore/location/{{blobstoreId}}/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'description',
                    displayName: 'Description',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'exec', mbean: Jclouds.getSelectionJcloudsBlobstoreMBean(workspace, $scope.blobstoreId), operation: 'listAssignableLocations()'
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.locations = $scope.result;
                Core.$apply($scope);
            }
        }
    }
    Jclouds.BlobstoreLocationListController = BlobstoreLocationListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function BlobstoreNavigationController($scope, $routeParams, workspace) {
        $scope.blobstoreId = $routeParams.blobstoreId;

        $scope.isActive = function (nav) {
            if (angular.isString(nav))
                return workspace.isLinkActive(nav);
            var fn = nav.isActive;
            if (fn) {
                return fn(workspace);
            }
            return workspace.isLinkActive(nav.href());
        };
    }
    Jclouds.BlobstoreNavigationController = BlobstoreNavigationController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jclouds
*/
var Jclouds;
(function (Jclouds) {
    function BlobstoreContainerListController($scope, $location, workspace, jolokia, $routeParams) {
        $scope.blobstoreId = $routeParams.blobstoreId;

        $scope.result = {};
        $scope.containers = [];

        // selected containers
        $scope.selectedContainers = [];

        $scope.containerTable = {
            plugins: [],
            data: 'containers',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                useExternalFilter: true
            },
            selectedItems: $scope.selectedContainers,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'name',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText"><a href="#/jclouds/blobstore/container/{{blobstoreId}}/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 200,
                    resizable: false
                },
                {
                    field: 'creationDate',
                    displayName: 'Created',
                    cellTemplate: '<div class="ngCellText">{{row.getProperty(col.field)}}</div>',
                    width: 200,
                    resizable: false
                }
            ]
        };

        Core.register(jolokia, $scope, {
            type: 'exec', mbean: Jclouds.getSelectionJcloudsBlobstoreMBean(workspace, $scope.blobstoreId), operation: 'list()'
        }, onSuccess(render));

        function render(response) {
            if (!Object.equal($scope.result, response.value)) {
                $scope.result = response.value;
                $scope.containers = $scope.result;
                Core.$apply($scope);
            }
        }
    }
    Jclouds.BlobstoreContainerListController = BlobstoreContainerListController;
})(Jclouds || (Jclouds = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function ThreadPoolsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | jettyIconClass}}"></i></div>';

        $scope.threadpools = [];

        var columnDefs = [
            {
                field: 'running',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'threads',
                displayName: 'Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'minThreads',
                displayName: 'Min Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxThreads',
                displayName: 'Max Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'idleThreads',
                displayName: 'Idle Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'idleTimeout',
                displayName: 'Idle Timeout (ms)',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'name',
                displayName: 'Name',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'threadpools',
            displayFooter: true,
            canSelectRows: false,
            columnDefs: columnDefs,
            title: "Thread Pools"
        };

        function render78(response) {
            $scope.threadpools = [];

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    // jetty78 vs jetty9 is a bit different
                    obj.running = obj['running'] !== undefined ? obj['running'] : obj['state'] == "STARTED";
                    obj.idleTimeout = obj['idleTimeout'] !== undefined ? obj['idleTimeout'] : obj['maxIdleTimeMs'];
                    $scope.threadpools.push(obj);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading Jetty thread pool data...");
            var tree = workspace.tree;

            jolokia.search("org.eclipse.jetty.util.thread:type=queuedthreadpool,*", onSuccess(render78));
        }
    }
    Jetty.ThreadPoolsController = ThreadPoolsController;
})(Jetty || (Jetty = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function JettyController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | jettyIconClass}}"></i></div>';

        $scope.uninstallDialog = new Core.Dialog();

        $scope.webapps = [];
        $scope.selected = [];

        var columnDefs = [
            {
                field: 'state',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'displayName',
                displayName: 'Name',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'contextPath',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'webapps',
            displayFooter: true,
            selectedItems: $scope.selected,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            },
            title: "Web applications"
        };

        // function to control the web applications
        $scope.controlWebApps = function (op) {
            // grab id of mbean names to control
            var mbeanNames = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(mbeanNames)) {
                mbeanNames = [mbeanNames];
            }

            // execute operation on each mbean
            var lastIndex = (mbeanNames.length || 1) - 1;
            angular.forEach(mbeanNames, function (mbean, idx) {
                var onResponse = (idx >= lastIndex) ? $scope.onLastResponse : $scope.onResponse;
                jolokia.request({
                    type: 'exec',
                    mbean: mbean,
                    operation: op,
                    arguments: null
                }, onSuccess(onResponse, { error: onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlWebApps('stop');
        };

        $scope.start = function () {
            $scope.controlWebApps('start');
        };

        $scope.uninstall = function () {
            $scope.controlWebApps('destroy');
            $scope.uninstallDialog.close();
        };

        $scope.anySelectionHasState = function (state) {
            var selected = $scope.selected || [];
            return selected.length && selected.any(function (s) {
                return Jetty.isState(s, state);
            });
        };

        $scope.everySelectionHasState = function (state) {
            var selected = $scope.selected || [];
            return selected.length && selected.every(function (s) {
                return Jetty.isState(s, state);
            });
        };

        // function to trigger reloading page
        $scope.onLastResponse = function (response) {
            $scope.onResponse(response);

            // we only want to force updating the data on the last response
            loadData();
        };

        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        // grab server information once
        $scope.jettyServerVersion = "";
        $scope.jettyServerStartupTime = "";

        var servers = jolokia.search("org.eclipse.jetty.server:type=server,*");
        if (servers && servers.length === 1) {
            $scope.jettyServerVersion = jolokia.getAttribute(servers[0], "version");
            $scope.jettyServerStartupTime = jolokia.getAttribute(servers[0], "startupTime");
        } else {
            console.log("Cannot find jetty server or there was more than one server. response is: " + servers);
        }

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading Jetty webapp data...");

            // support embedded jetty which may use morbay mbean names
            jolokia.search("org.mortbay.jetty.plugin:type=jettywebappcontext,*", onSuccess(render));
            jolokia.search("org.eclipse.jetty.webapp:type=webappcontext,*", onSuccess(render));
            jolokia.search("org.eclipse.jetty.servlet:type=servletcontexthandler,*", onSuccess(render));
        }

        function render(response) {
            $scope.webapps = [];
            $scope.mbeanIndex = {};
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    if (!obj.state) {
                        // lets leave the state as it is if it is defined
                        obj.state = obj['running'] === undefined || obj['running'] ? "started" : "stopped";
                    }
                    var mbean = obj.mbean;
                    if (mbean) {
                        var idx = $scope.mbeanIndex[mbean];
                        if (angular.isDefined(idx)) {
                            $scope.webapps[mbean] = obj;
                        } else {
                            $scope.mbeanIndex[mbean] = $scope.webapps.length;
                            $scope.webapps.push(obj);
                        }
                        Core.$apply($scope);
                    }
                }
            }

            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
    }
    Jetty.JettyController = JettyController;
})(Jetty || (Jetty = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function iconClass(state) {
        if (state) {
            switch (state.toString().toLowerCase()) {
                case 'started':
                    return "green icon-play-circle";
                case 'true':
                    return "green icon-play-circle";
            }
        }
        return "orange icon-off";
    }
    Jetty.iconClass = iconClass;

    /**
    * Returns true if the state of the item begins with the given state - or one of the given states
    * @method isState
    * @for Jetty
    * @param {any} item the item which has a State
    * @param {any} state a value or an array of states
    * @return {Boolean}
    */
    function isState(item, state) {
        var value = (item.state || "").toLowerCase();
        if (angular.isArray(state)) {
            return state.any(function (stateText) {
                return value.startsWith(stateText);
            });
        } else {
            return value.startsWith(state);
        }
    }
    Jetty.isState = isState;
})(Jetty || (Jetty = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function ConnectorsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | jettyIconClass}}"></i></div>';

        $scope.connectors = [];
        $scope.selected = [];

        var columnDefs = [
            {
                field: 'running',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'port',
                displayName: 'Port',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'protocols',
                displayName: 'Protocols',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'default',
                displayName: 'Default',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'connectors',
            displayFooter: true,
            selectedItems: $scope.selected,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            },
            title: "Connectors"
        };

        // function to control the connectors
        $scope.controlConnectors = function (op) {
            // grab id of mbean names to control
            var mbeanNames = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(mbeanNames)) {
                mbeanNames = [mbeanNames];
            }

            // execute operation on each mbean
            var lastIndex = (mbeanNames.length || 1) - 1;
            angular.forEach(mbeanNames, function (mbean, idx) {
                var onResponse = (idx >= lastIndex) ? $scope.onLastResponse : $scope.onResponse;
                jolokia.request({
                    type: 'exec',
                    mbean: mbean,
                    operation: op,
                    arguments: null
                }, onSuccess(onResponse, { error: onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlConnectors('stop');
        };

        $scope.start = function () {
            $scope.controlConnectors('start');
        };

        $scope.anySelectionIsRunning = function () {
            var selected = $scope.selected || [];
            return selected.length && selected.any(function (s) {
                return s.running;
            });
        };

        $scope.everySelectionIsRunning = function (state) {
            var selected = $scope.selected || [];
            return selected.length && selected.every(function (s) {
                return s.running;
            });
        };

        function render78(response) {
            $scope.connectors = [];
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    // split each into 2 rows as we want http and https on each row
                    obj.mbean = response.request.mbean;
                    obj.protocols = "[http]";
                    obj.default = "http";
                    obj.port = obj.port;
                    obj.running = obj['running'] !== undefined ? obj['running'] : true;
                    $scope.connectors.push(obj);
                    if (obj.confidentialPort) {
                        // create a clone of obj for https
                        var copyObj = {
                            protocols: "[https]",
                            default: "https",
                            port: obj.confidentialPort,
                            running: obj.running,
                            mbean: obj.mbean
                        };
                        $scope.connectors.push(copyObj);
                    }
                    Core.$apply($scope);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        function render9(response) {
            $scope.connectors = [];
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    obj.protocols = obj['protocols'];
                    obj.default = obj['defaultProtocol'];
                    obj.port = obj.port;
                    obj.running = obj['state'] == "STARTED";
                    $scope.connectors.push(obj);
                    Core.$apply($scope);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        // function to trigger reloading page
        $scope.onLastResponse = function (response) {
            $scope.onResponse(response);

            // we only want to force updating the data on the last response
            loadData();
        };

        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading Jetty connector data...");
            var tree = workspace.tree;

            jolokia.search("org.eclipse.jetty.server.nio:type=selectchannelconnector,*", onSuccess(render78));
            jolokia.search("org.eclipse.jetty.server:type=serverconnector,*", onSuccess(render9));
        }
    }
    Jetty.ConnectorsController = ConnectorsController;
})(Jetty || (Jetty = {}));
/**
* @module Jetty
* @main Jetty
*/
var Jetty;
(function (Jetty) {
    var pluginName = 'jetty';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ui.bootstrap.dialog', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/jetty/server', { templateUrl: 'app/jetty/html/server.html' }).when('/jetty/applications', { templateUrl: 'app/jetty/html/applications.html' }).when('/jetty/connectors', { templateUrl: 'app/jetty/html/connectors.html' }).when('/jetty/threadpools', { templateUrl: 'app/jetty/html/threadpools.html' });
    }).filter('jettyIconClass', function () {
        return Jetty.iconClass;
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['jetty'] = "app/jetty/html/layoutJettyTabs.html";
        helpRegistry.addUserDoc('jetty', 'app/jetty/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("org.eclipse.jetty.server");
        });

        workspace.topLevelTabs.push({
            id: "jetty",
            content: "Jetty",
            title: "Manage your Jetty container",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("org.eclipse.jetty.server");
            },
            href: function () {
                return "#/jetty/applications";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("jetty");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Jetty || (Jetty = {}));
/**
* @module Site
* @main Site
*/
var Site;
(function (Site) {
    var pluginName = 'site';

    angular.module(pluginName, ['bootstrap', 'ngResource', 'ngGrid', 'datatable', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/site', { templateUrl: 'app/site/html/index.html' }).when('/site/', { templateUrl: 'app/site/html/index.html' }).when('/site/book/*page', { templateUrl: 'app/site/html/book.html', reloadOnSearch: false }).when('/site/*page', { templateUrl: 'app/site/html/page.html' });
    }).run(function ($location, workspace, viewRegistry, layoutFull, helpRegistry) {
        viewRegistry[pluginName] = layoutFull;

        workspace.topLevelTabs.push({
            id: "site",
            content: "Site",
            title: "View the documentation for Hawtio",
            isValid: function (workspace) {
                return false;
            },
            href: function () {
                return "#/site";
            }
        });
        /*
        helpRegistry.addUserDoc('log', 'app/log/doc/help.md', () => {
        return workspace.treeContainsDomainAndProperties('org.fusesource.insight', {type: 'LogQuery'});
        });
        
        */
    });

    hawtioPluginLoader.addModule(pluginName);
})(Site || (Site = {}));
/**
* @module Site
*/
var Site;
(function (Site) {
    Site.sitePluginEnabled = false;

    function isSiteNavBarValid() {
        return Site.sitePluginEnabled;
    }
    Site.isSiteNavBarValid = isSiteNavBarValid;
})(Site || (Site = {}));
/**
* @module Site
*/
var Site;
(function (Site) {
    function PageController($scope, $routeParams, $location, $compile, $http, fileExtensionTypeRegistry) {
        var log = Logger.get("Site");
        var pageId = $routeParams["page"];
        if (!pageId) {
            pageId = "README.md";
            /*
            $location.path("/site/doc/index.md");
            return;
            */
        }

        if (!pageId.startsWith("/") && pageId.indexOf(":/") < 0 && pageId.indexOf("app/site/") < 0) {
            // lets assume the page is relative to app/site/
            pageId = "app/site/" + pageId;
        }
        $scope.pageId = pageId;
        $scope.pageFolder = pageId.substring(0, pageId.lastIndexOf('/') + 1);

        log.info("Loading page '" + $scope.pageId + "'");

        $scope.getContents = function (filename, cb) {
            var fullPath = $scope.pageFolder + filename;
            log.info("Loading the contents of: " + fullPath);
            $http.get(fullPath).success(cb).error(function () {
                return cb(" ");
            });
        };

        $http.get($scope.pageId).success(onResults);

        function onResults(contents, status, headers, config) {
            $scope.contents = contents;
            $scope.html = contents;

            var format = Wiki.fileFormat($scope.pageId, fileExtensionTypeRegistry) || "markdown";
            if ("markdown" === format) {
                // lets convert it to HTML
                $scope.html = contents ? marked(contents) : "";
            } else if (format && format.startsWith("html")) {
                $scope.html = contents;
            } else {
                // TODO?
            }
            $compile($scope.html)($scope);
            Core.$apply($scope);
        }
    }
    Site.PageController = PageController;
})(Site || (Site = {}));
/**
* @module Site
*/
var Site;
(function (Site) {
    function IndexController($scope, $location) {
        $scope.slideInterval = 5000;
    }
    Site.IndexController = IndexController;
})(Site || (Site = {}));
/**
* @module JUnit
* @main JUnit
*/
var JUnit;
(function (JUnit) {
    var pluginName = 'junit';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ngGrid', 'datatable', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/junit/tests', { templateUrl: 'app/junit/html/tests.html', reloadOnSearch: false });
    }).factory('inProgressStatus', function () {
        return {
            jhandle: null,
            data: null,
            result: null,
            alertClass: "success"
        };
    }).run(function ($location, workspace, viewRegistry, layoutFull, helpRegistry) {
        viewRegistry['junit'] = 'app/junit/html/layoutJUnitTree.html';

        helpRegistry.addUserDoc('junit', 'app/junit/doc/help.md', function () {
            return JUnit.isJUnitPluginEnabled(workspace);
        });

        workspace.topLevelTabs.push({
            id: "junit",
            content: "JUnit",
            title: "View and run test cases in this process",
            isValid: function (workspace) {
                return JUnit.isJUnitPluginEnabled(workspace);
            },
            href: function () {
                return "#/junit/tests";
            }
        });
        /*
        workspace.subLevelTabs.push({
        content: '<i class="icon-list-alt"></i> JUnit',
        title: "View the logs in this process",
        isValid: (workspace:Workspace) => workspace.hasDomainAndProperties('org.fusesource.insight', {type: 'JUnitQuery'}),
        href: () => "#/logs"
        });
        */
    });

    hawtioPluginLoader.addModule(pluginName);
})(JUnit || (JUnit = {}));
var JUnit;
(function (JUnit) {
    function TreeController($scope, $location, workspace, jolokia, inProgressStatus) {
        var log = Logger.get("JUnit");

        $scope.inProgressData = null;
        $scope.alertClass = "success";

        $scope.testClasses = [];
        $scope.testClassMap = {};

        $scope.gridOptions = {
            selectedItems: [],
            data: 'selectedTests',
            displayFooter: false,
            showFilter: false,
            filterOptions: {
                filterText: ''
            },
            //selectWithCheckboxOnly: true,
            showSelectionCheckbox: true,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Test Class'
                }
            ]
        };

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        $scope.$on('jmxTreeUpdated', function () {
            reloadTree();
        });

        $scope.runAllTests = function () {
            runTests($scope.testClasses);
        };

        $scope.runTests = function () {
            var tests = ($scope.gridOptions.selectedItems || []).map(function (o) {
                return o.id;
            });
            runTests(tests);
        };

        $scope.runTest = function (className) {
            if (className) {
                runTests([className]);
            }
        };

        $scope.clearResults = function () {
            $scope.testResults = null;
            $scope.alertClass = "success";
            inProgressStatus.data = null;
            inProgressStatus.result = null;
            inProgressStatus.alertClass = "success";
        };

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURL($location, $("#junittree"), true);
        }

        reloadTree();

        function selectionChanged(data) {
            var selectionKey = data ? data.key : null;
            log.debug("Selection is now: " + selectionKey);
            var selectedTests = $scope.testClasses;
            $scope.selectionKey = selectionKey;
            if (selectionKey) {
                selectedTests = $scope.testClassMap[selectionKey] || [selectionKey];
            }
            $scope.selectedTests = selectedTests.map(function (t) {
                return { id: t };
            });
            Core.$apply($scope);
        }

        function reloadTree() {
            var mbean = JUnit.getIntrospectorMBean(workspace);
            var domain = "org.unit";
            var rootFolder = new Folder("Test Cases");
            rootFolder.addClass = "testCases";
            rootFolder.typeName = "testCases";
            rootFolder.domain = domain;
            rootFolder.key = "";
            var children = [rootFolder];

            if (mbean) {
                function render(results) {
                    $scope.testClasses = results;
                    $scope.testClassMap = {};

                    angular.forEach(results, function (className) {
                        var paths = className.split(".");
                        var last = paths.length - 1;
                        var folder = rootFolder;
                        var prefix = "";
                        for (var i = 0; i < last; i++) {
                            var path = paths[i];
                            if (prefix) {
                                prefix += ".";
                            }
                            prefix += path;
                            var list = $scope.testClassMap[prefix];
                            if (!list) {
                                list = [];
                                $scope.testClassMap[prefix] = list;
                            }
                            list.push(className);
                            folder = workspace.folderGetOrElse(folder, path);
                            folder.key = prefix;
                        }
                        var lastPath = paths[last];

                        // lets add the test case...
                        var testClass = new Folder(lastPath);
                        testClass.addClass = "testClass";
                        testClass.domain = domain;
                        testClass.key = className;
                        folder.children.push(testClass);
                    });

                    Core.$apply($scope);

                    var treeElement = $("#junittree");
                    Jmx.enableTree($scope, $location, workspace, treeElement, children, true, function (selectedNode) {
                        var data = selectedNode.data;

                        //$scope.select(data);
                        //workspace.updateSelectionNode(data);
                        selectionChanged(data);
                        Core.$apply($scope);
                    });

                    // lets do this asynchronously to avoid Error: $digest already in progress
                    setTimeout(updateSelectionFromURL, 50);
                }

                jolokia.execute(mbean, "findJUnitTestClassNames", onSuccess(render));
            }
        }

        $scope.runningTests = function () {
            if (inProgressStatus.data !== null && inProgressStatus.data.running) {
                // in case we navigate back, then make sure the scope has the last up to date result to use
                $scope.inProgressData = inProgressStatus.data;
                $scope.alertClass = inProgressStatus.alertClass;
                return true;
            } else {
                return false;
            }
        };

        $scope.hasTestResults = function () {
            if (inProgressStatus.result !== null) {
                // in case we navigate back, then make sure the scope has the last up to date result to use
                $scope.testResults = inProgressStatus.result;
                $scope.alertClass = inProgressStatus.alertClass;
                return true;
            } else {
                return false;
            }
        };

        var renderInProgress = function (response) {
            var result = response.value;
            if (result) {
                log.info("Render inProgress: " + result);

                inProgressStatus.data = result;
                $scope.inProgressData = inProgressStatus.data;

                var alertClass = "success";
                if (result.failureCount > 0) {
                    alertClass = "error";
                }
                inProgressStatus.alertClass = alertClass;
                $scope.alertClass = inProgressStatus.alertClass;

                // if we no longer are running then clear handle
                if (!result.running && inProgressStatus.jhandle !== null) {
                    log.info("Unit test done, unreigster jolokia handle");
                    jolokia.unregister(inProgressStatus.jhandle);
                    inProgressStatus.jhandle = null;
                }

                Core.$apply($scope);
            }
        };

        var renderResults = function (result) {
            if (result) {
                log.info("Render results: " + result);

                inProgressStatus.data = null;
                inProgressStatus.alertClass = null;
                inProgressStatus.result = result;

                var alertClass = "success";
                var notificationClass = "success";
                var message = "JUnit testing succeded with " + result.runCount + " runs.";
                if (result.failureCount > 0) {
                    alertClass = "error";
                    notificationClass = "warning";
                    message = "JUnit testing failed with " + result.failureCount + " failures.";
                }
                $scope.alertClass = alertClass;
                $scope.testResults = inProgressStatus.result;

                // publish notification
                notification(notificationClass, message);

                Core.$apply($scope);
            }
        };

        function runTests(listOfClassNames) {
            // reset before running new set of unit tests
            $scope.clearResults();

            var mbean = JUnit.getJUnitMBean(workspace);
            if (mbean && listOfClassNames && listOfClassNames.length) {
                // register callback for doing live update of testing progress
                if (inProgressStatus.jhandle === null) {
                    log.info("Registering jolokia handle");
                    inProgressStatus.jhandle = jolokia.register(renderInProgress, {
                        type: 'exec', mbean: mbean,
                        operation: 'inProgress()',
                        ignoreErrors: true,
                        arguments: []
                    });

                    // execute the unit tests
                    jolokia.execute(mbean, "runTestClasses", listOfClassNames, onSuccess(renderResults));
                }
            }
        }
    }
    JUnit.TreeController = TreeController;
})(JUnit || (JUnit = {}));
/**
* @module JUnit
*/
var JUnit;
(function (JUnit) {
    JUnit.log = Logger.get("JUnit");

    /**
    * Returns true if the JUnit plugin is enabled (both the hawtio insight and JUnit mbeans are available
    */
    function isJUnitPluginEnabled(workspace) {
        return getIntrospectorMBean(workspace) && getJUnitMBean(workspace);
    }
    JUnit.isJUnitPluginEnabled = isJUnitPluginEnabled;

    function getJUnitMBean(workspace) {
        return Core.getMBeanTypeObjectName(workspace, "hawtio", "JUnitFacade");
    }
    JUnit.getJUnitMBean = getJUnitMBean;

    function getIntrospectorMBean(workspace) {
        return Core.getMBeanTypeObjectName(workspace, "hawtio", "Introspector");
    }
    JUnit.getIntrospectorMBean = getIntrospectorMBean;
})(JUnit || (JUnit = {}));
/**
* @module DataTable
* @main DataTable
*/
var DataTable;
(function (DataTable) {
    var pluginName = 'datatable';

    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/datatable/test', { templateUrl: 'app/datatable/html/test.html' });
    }).directive('hawtioSimpleTable', function ($compile) {
        return new DataTable.SimpleDataTable($compile);
    }).directive('hawtioDatatable', function (workspace, $timeout, $filter, $compile) {
        // return the directive link function. (compile function not needed)
        return function (scope, element, attrs) {
            var gridOptions = null;
            var data = null;
            var widget = null;
            var timeoutId = null;
            var initialised = false;
            var childScopes = [];
            var rowDetailTemplate = null;
            var rowDetailTemplateId = null;
            var selectedItems = null;

            // used to update the UI
            function updateGrid() {
                // console.log("updating the grid!!");
                Core.$applyNowOrLater(scope);
            }

            function convertToDataTableColumn(columnDef) {
                var data = {
                    mData: columnDef.field,
                    sDefaultContent: ""
                };
                var name = columnDef.displayName;
                if (name) {
                    data["sTitle"] = name;
                }
                var width = columnDef.width;
                if (angular.isNumber(width)) {
                    data["sWidth"] = "" + width + "px";
                } else if (angular.isString(width) && !width.startsWith("*")) {
                    data["sWidth"] = width;
                }
                var template = columnDef.cellTemplate;
                if (template) {
                    data["fnCreatedCell"] = function (nTd, sData, oData, iRow, iCol) {
                        var childScope = childScopes[iRow];
                        if (!childScope) {
                            childScope = scope.$new(false);
                            childScopes[iRow] = childScope;
                        }
                        var entity = oData;
                        childScope["row"] = {
                            entity: entity,
                            getProperty: function (name) {
                                return entity[name];
                            }
                        };

                        var elem = $(nTd);
                        elem.html(template);
                        var contents = elem.contents();
                        contents.removeClass("ngCellText");
                        $compile(contents)(childScope);
                    };
                } else {
                    var cellFilter = columnDef.cellFilter;
                    var render = columnDef.render;
                    if (cellFilter && !render) {
                        var filter = $filter(cellFilter);
                        if (filter) {
                            render = function (data, type, full) {
                                return filter(data);
                            };
                        }
                    }
                    if (render) {
                        data["mRender"] = render;
                    }
                }
                return data;
            }

            function destroyChildScopes() {
                angular.forEach(childScopes, function (childScope) {
                    childScope.$destroy();
                });
                childScopes = [];
            }

            function selectHandler(selection) {
                if (selection && selectedItems) {
                    selectedItems.splice(0, selectedItems.length);
                    selectedItems.push(selection);
                    Core.$apply(scope);
                }
            }

            function onTableDataChange(value) {
                gridOptions = value;
                if (gridOptions) {
                    selectedItems = gridOptions.selectedItems;
                    rowDetailTemplate = gridOptions.rowDetailTemplate;
                    rowDetailTemplateId = gridOptions.rowDetailTemplateId;

                    // TODO deal with updating the gridOptions on the fly?
                    if (widget === null) {
                        var widgetOptions = {
                            selectHandler: selectHandler,
                            disableAddColumns: true,
                            rowDetailTemplateId: rowDetailTemplateId,
                            ignoreColumns: gridOptions.ignoreColumns,
                            flattenColumns: gridOptions.flattenColumns
                        };

                        // lets find a child table element
                        // or lets add one if there's not one already
                        var rootElement = $(element);
                        var tableElement = rootElement.children("table");
                        if (!tableElement.length) {
                            $("<table class='table table-bordered table-condensed'></table>").appendTo(rootElement);
                            tableElement = rootElement.children("table");
                        }
                        tableElement.removeClass('table-striped');
                        tableElement.addClass('dataTable');
                        var trElement = Core.getOrCreateElements(tableElement, ["thead", "tr"]);

                        destroyChildScopes();

                        // convert the column configurations
                        var columns = [];
                        var columnCounter = 1;
                        var extraLeftColumn = rowDetailTemplate || rowDetailTemplateId;
                        if (extraLeftColumn) {
                            columns.push({
                                "mDataProp": null,
                                "sClass": "control center",
                                "sWidth": "30px",
                                "sDefaultContent": '<i class="icon-plus"></i>'
                            });

                            var th = trElement.children("th");
                            if (th.length < columnCounter++) {
                                $("<th></th>").appendTo(trElement);
                            }
                        }

                        var columnDefs = gridOptions.columnDefs;
                        if (angular.isString(columnDefs)) {
                            // TODO watch this value?
                            columnDefs = scope[columnDefs];
                        }

                        angular.forEach(columnDefs, function (columnDef) {
                            // if there's not another <tr> then lets add one
                            th = trElement.children("th");
                            if (th.length < columnCounter++) {
                                var name = columnDef.displayName || "";
                                $("<th>" + name + "</th>").appendTo(trElement);
                            }
                            columns.push(convertToDataTableColumn(columnDef));
                        });
                        widget = new TableWidget(scope, workspace, columns, widgetOptions);
                        widget.tableElement = tableElement;

                        var sortInfo = gridOptions.sortInfo;
                        if (sortInfo && columnDefs) {
                            var sortColumns = [];
                            var field = sortInfo.field;
                            if (field) {
                                var idx = columnDefs.findIndex({ field: field });
                                if (idx >= 0) {
                                    if (extraLeftColumn) {
                                        idx += 1;
                                    }
                                    var asc = sortInfo.direction || "asc";
                                    asc = asc.toLowerCase();
                                    sortColumns.push([idx, asc]);
                                }
                            }
                            if (sortColumns.length) {
                                widget.sortColumns = sortColumns;
                            }
                        }

                        // if all the column definitions have an sWidth then lets turn off
                        // the auto-width calculations
                        if (columns.every(function (col) {
                            return col.sWidth;
                        })) {
                            widget.dataTableConfig.bAutoWidth = false;
                        }

                        /*
                        // lets avoid word wrap
                        widget.dataTableConfig["fnCreatedRow"] = function( nRow, aData, iDataIndex ) {
                        var cells = $(nRow).children("td");
                        cells.css("overflow", "hidden");
                        cells.css("white-space", "nowrap");
                        cells.css("text-overflow", "ellipsis");
                        };
                        */
                        var filterText = null;
                        var filterOptions = gridOptions.filterOptions;
                        if (filterOptions) {
                            filterText = filterOptions.filterText;
                        }
                        if (filterText || (angular.isDefined(gridOptions.showFilter) && !gridOptions.showFilter)) {
                            // disable the text filter box
                            widget.dataTableConfig.sDom = 'Rlrtip';
                        }
                        if (filterText) {
                            scope.$watch(filterText, function (value) {
                                var dataTable = widget.dataTable;
                                if (dataTable) {
                                    dataTable.fnFilter(value);
                                }
                            });
                        }
                        if (angular.isDefined(gridOptions.displayFooter) && !gridOptions.displayFooter && widget.dataTableConfig.sDom) {
                            // remove the footer
                            widget.dataTableConfig.sDom = widget.dataTableConfig.sDom.replace('i', '');
                        }
                        // TODO....
                        // lets make sure there is enough th headers for the columns!
                    }
                    if (!data) {
                        // TODO deal with the data name changing one day?
                        data = gridOptions.data;
                        if (data) {
                            var listener = function (value) {
                                if (initialised || (value && (!angular.isArray(value) || value.length))) {
                                    initialised = true;
                                    destroyChildScopes();
                                    widget.populateTable(value);
                                    updateLater();
                                }
                            };
                            scope.$watch(data, listener);

                            // lets add a separate event so we can force updates
                            // if we find cases where the delta logic doesn't work
                            // (such as for nested hawtioinput-input-table)
                            scope.$on("hawtio.datatable." + data, function (args) {
                                var value = Core.pathGet(scope, data);
                                listener(value);
                            });
                        }
                    }
                }
                updateGrid();
            }

            // watch the expression, and update the UI on change.
            scope.$watch(attrs.hawtioDatatable, onTableDataChange);

            // schedule update in one second
            function updateLater() {
                // save the timeoutId for canceling
                timeoutId = $timeout(function () {
                    updateGrid(); // update DOM
                }, 300);
            }

            // listen on DOM destroy (removal) event, and cancel the next UI update
            // to prevent updating ofter the DOM element was removed.
            element.bind('$destroy', function () {
                destroyChildScopes();
                $timeout.cancel(timeoutId);
            });

            updateLater(); // kick off the UI update process.
        };
    }).run(function (helpRegistry) {
        helpRegistry.addDevDoc(pluginName, 'app/datatable/doc/developer.md');
    });

    hawtioPluginLoader.addModule(pluginName);
})(DataTable || (DataTable = {}));
/**
* @module DataTable
* @main DataTable
*/
var DataTable;
(function (DataTable) {
    DataTable.log = Logger.get("DataTable");

    var SimpleDataTable = (function () {
        function SimpleDataTable($compile) {
            var _this = this;
            this.$compile = $compile;
            this.restrict = 'A';
            this.scope = {
                config: '=hawtioSimpleTable',
                target: '@',
                showFiles: '@'
            };
            // necessary to ensure 'this' is this object <sigh>
            this.link = function ($scope, $element, $attrs) {
                return _this.doLink($scope, $element, $attrs);
            };
        }
        SimpleDataTable.prototype.doLink = function ($scope, $element, $attrs) {
            var defaultPrimaryKeyFn = function (entity, idx) {
                // default function to use id/_id/name as primary key, and fallback to use index
                return entity["id"] || entity["_id"] || entity["name"] || idx;
            };

            var config = $scope.config;

            var dataName = config.data || "data";

            // need to remember which rows has been selected as the config.data / config.selectedItems
            // so we can re-select them when data is changed/updated, and entity may be new instances
            // so we need a primary key function to generate a 'primary key' of the entity
            var primaryKeyFn = config.primaryKeyFn || defaultPrimaryKeyFn;
            $scope.rows = [];

            var scope = $scope.$parent || $scope;

            var listener = function (otherValue) {
                var value = Core.pathGet(scope, dataName);
                if (value && !angular.isArray(value)) {
                    value = [value];
                    Core.pathSet(scope, dataName, value);
                }

                if (!('sortInfo' in config)) {
                    config['sortInfo'] = {
                        sortBy: config.columnDefs.first()['field'],
                        ascending: true
                    };
                }

                var sortInfo = $scope.config.sortInfo;

                // enrich the rows with information about their index
                var idx = -1;
                $scope.rows = (value || []).sortBy(sortInfo.sortBy, !sortInfo.ascending).map(function (entity) {
                    idx++;
                    return {
                        entity: entity,
                        index: idx,
                        getProperty: function (name) {
                            return entity[name];
                        }
                    };
                });

                Core.pathSet(scope, ['hawtioSimpleTable', dataName, 'rows'], $scope.rows);

                // okay the data was changed/updated so we need to re-select previously selected items
                // and for that we need to evaluate the primary key function so we can match new data with old data.
                var reSelectedItems = [];
                $scope.rows.forEach(function (row, idx) {
                    var rpk = primaryKeyFn(row.entity, row.index);
                    var selected = config.selectedItems.some(function (s) {
                        var spk = primaryKeyFn(s, s.index);
                        return angular.equals(rpk, spk);
                    });
                    if (selected) {
                        // need to enrich entity with index, as we push row.entity to the re-selected items
                        row.entity.index = row.index;
                        reSelectedItems.push(row.entity);
                        DataTable.log.debug("Data changed so keep selecting row at index " + row.index);
                    }
                });
                config.selectedItems = reSelectedItems;
            };

            scope.$watch(dataName, listener);

            // lets add a separate event so we can force updates
            // if we find cases where the delta logic doesn't work
            // (such as for nested hawtioinput-input-table)
            scope.$on("hawtio.datatable." + dataName, listener);

            function getSelectionArray() {
                var selectionArray = config.selectedItems;
                if (!selectionArray) {
                    selectionArray = [];
                    config.selectedItems = selectionArray;
                }
                if (angular.isString(selectionArray)) {
                    var name = selectionArray;
                    selectionArray = Core.pathGet(scope, name);
                    if (!selectionArray) {
                        selectionArray = [];
                        scope[name] = selectionArray;
                    }
                }
                return selectionArray;
            }

            function isMultiSelect() {
                var multiSelect = $scope.config.multiSelect;
                if (angular.isUndefined(multiSelect)) {
                    multiSelect = true;
                }
                return multiSelect;
            }

            $scope.toggleAllSelections = function () {
                var allRowsSelected = $scope.config.allRowsSelected;
                var newFlag = allRowsSelected;
                var selectionArray = getSelectionArray();
                selectionArray.splice(0, selectionArray.length);
                angular.forEach($scope.rows, function (row) {
                    row.selected = newFlag;
                    if (allRowsSelected) {
                        selectionArray.push(row.entity);
                    }
                });
            };

            $scope.toggleRowSelection = function (row) {
                if (row) {
                    var selectionArray = getSelectionArray();
                    if (!isMultiSelect()) {
                        // lets clear all other selections
                        selectionArray.splice(0, selectionArray.length);
                        angular.forEach($scope.rows, function (r) {
                            if (r !== row) {
                                r.selected = false;
                            }
                        });
                    }
                    var entity = row.entity;
                    if (entity) {
                        var idx = selectionArray.indexOf(entity);
                        if (row.selected) {
                            if (idx < 0) {
                                selectionArray.push(entity);
                            }
                        } else {
                            // clear the all selected checkbox
                            $scope.config.allRowsSelected = false;
                            if (idx >= 0) {
                                selectionArray.splice(idx, 1);
                            }
                        }
                    }
                }
            };

            $scope.sortBy = function (field) {
                if ($scope.config.sortInfo.sortBy === field) {
                    $scope.config.sortInfo.ascending = !$scope.config.sortInfo.ascending;
                } else {
                    $scope.config.sortInfo.sortBy = field;
                    $scope.config.sortInfo.ascending = true;
                }
                $scope.$emit("hawtio.datatable." + dataName);
            };

            $scope.getClass = function (field) {
                if ('sortInfo' in $scope.config) {
                    if ($scope.config.sortInfo.sortBy === field) {
                        if ($scope.config.sortInfo.ascending) {
                            return 'asc';
                        } else {
                            return 'desc';
                        }
                    }
                }

                return '';
            };

            $scope.showRow = function (row) {
                var filter = Core.pathGet($scope, ['config', 'filterOptions', 'filterText']);
                if (Core.isBlank(filter)) {
                    return true;
                }
                var rowJson = angular.toJson(row);
                return rowJson.toLowerCase().has(filter.toLowerCase());
            };

            $scope.isSelected = function (row) {
                return config.selectedItems.some(row.entity);
            };

            $scope.onRowSelected = function (row) {
                var idx = config.selectedItems.indexOf(row.entity);
                if (idx >= 0) {
                    DataTable.log.debug("De-selecting row at index " + row.index);
                    config.selectedItems.splice(idx, 1);
                } else {
                    if (!config.multiSelect) {
                        config.selectedItems = [];
                    }
                    DataTable.log.debug("Selecting row at index " + row.index);

                    // need to enrich entity with index, as we push row.entity to the selected items
                    row.entity.index = row.index;
                    config.selectedItems.push(row.entity);
                }
            };

            // lets add the header and row cells
            var rootElement = $element;
            rootElement.empty();

            var showCheckBox = firstValueDefined(config, ["showSelectionCheckbox", "displaySelectionCheckbox"], true);
            var enableRowClickSelection = firstValueDefined(config, ["enableRowClickSelection"], false);

            var onMouseDown;
            if (enableRowClickSelection) {
                onMouseDown = "ng-mousedown='onRowSelected(row)' ";
            } else {
                onMouseDown = "";
            }
            var headHtml = "<thead><tr>";

            // use a function to check if a row is selected so the UI can be kept up to date asap
            var bodyHtml = "<tbody><tr ng-repeat='row in rows track by $index' ng-show='showRow(row)' " + onMouseDown + "ng-class=\"{'selected': isSelected(row)}\" >";
            var idx = 0;
            if (showCheckBox) {
                var toggleAllHtml = isMultiSelect() ? "<input type='checkbox' ng-show='rows.length' ng-model='config.allRowsSelected' ng-change='toggleAllSelections()'>" : "";

                headHtml += "\n<th>" + toggleAllHtml + "</th>";
                bodyHtml += "\n<td><input type='checkbox' ng-model='row.selected' ng-change='toggleRowSelection(row)'></td>";
            }
            angular.forEach(config.columnDefs, function (colDef) {
                var field = colDef.field;
                var cellTemplate = colDef.cellTemplate || '<div class="ngCellText" title="{{row.entity.' + field + '}}">{{row.entity.' + field + '}}</div>';

                headHtml += "\n<th class='clickable no-fade table-header' ng-click=\"sortBy('" + field + "')\" ng-class=\"getClass('" + field + "')\">{{config.columnDefs[" + idx + "].displayName}}<span class='indicator'></span></th>";

                bodyHtml += "\n<td>" + cellTemplate + "</td>";
                idx += 1;
            });
            var html = headHtml + "\n</tr></thead>\n" + bodyHtml + "\n</tr></tbody>";

            var newContent = this.$compile(html)($scope);
            rootElement.html(newContent);
        };
        return SimpleDataTable;
    })();
    DataTable.SimpleDataTable = SimpleDataTable;

    /**
    * Returns the first property value defined in the given object or the default value if none are defined
    *
    * @param object the object to look for properties
    * @param names the array of property names to look for
    * @param defaultValue the value if no property values are defined
    * @return {*} the first defined property value or the defaultValue if none are defined
    */
    function firstValueDefined(object, names, defaultValue) {
        var answer = defaultValue;
        var found = false;
        angular.forEach(names, function (name) {
            var value = object[name];
            if (!found && angular.isDefined(value)) {
                answer = value;
                found = true;
            }
        });
        return answer;
    }
})(DataTable || (DataTable = {}));
/**
* @module OpenEJB
* @main OpenEJB
*/
var OpenEJB;
(function (OpenEJB) {
    var pluginName = 'openejb';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).config(function ($routeProvider) {
        // TODO custom tomcat views go here...
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['openojb'] = "app/openejb/html/layoutOpenEJBTree.html";
        helpRegistry.addUserDoc('openejb', 'app/openejb/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("openejb");
        });

        workspace.topLevelTabs.push({
            id: "openejb",
            content: "OpenEJB",
            title: "Manage your OpenEJB resources",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("openejb");
            },
            href: function () {
                return "#/jmx/attributes?tab=openejb";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("openejb");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(OpenEJB || (OpenEJB = {}));
/**
* @module OpenEJB
*/
var OpenEJB;
(function (OpenEJB) {
    function TreeController($scope, $location, workspace) {
        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        $scope.$watch('workspace.tree', function () {
            if (workspace.moveIfViewInvalid())
                return;

            var children = [];
            var tree = workspace.tree;
            if (tree) {
                var nodes = tree.children;
                angular.forEach(nodes, function (node) {
                    var nodeChildren = node.children;
                    if (node.title.startsWith("openejb") && nodeChildren) {
                        children = children.concat(nodeChildren);
                    }
                });
            }
            var treeElement = $("#openejbTree");
            Jmx.enableTree($scope, $location, workspace, treeElement, children, true);

            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURL($location, $("#openejbTree"), true);
        }
    }
    OpenEJB.TreeController = TreeController;
})(OpenEJB || (OpenEJB = {}));
var Camin;
(function (Camin) {
    function Controller($scope, jolokia, localStorage, $routeParams) {
        $scope.query = "";
        $scope.result = "";
        $scope.breadcrumbs = [];

        $scope.onQueryChange = function () {
            $scope.result = "Querying exchanges related to " + $scope.query;
            $scope.breadcrumbs = [$scope.query];
            request();
        };

        var request = function () {
            var queryStr = "exchange.id:\"" + $scope.breadcrumbs.join("\" or exchange.id:\"") + "\" or " + "exchange.in.headers.ExtendedBreadcrumb:\"" + $scope.breadcrumbs.join("\" or exchange.in.headers.ExtendedBreadcrumb:\"") + "\" or " + "exchange.out.headers.ExtendedBreadcrumb:\"" + $scope.breadcrumbs.join("\" or exchange.out.headers.ExtendedBreadcrumb:\"") + "\"";
            var query = {
                "query": { "query_string": { "query": queryStr } },
                "fields": ["exchange.id", "exchange.in.headers.ExtendedBreadcrumb", "exchange.out.headers.ExtendedBreadcrumb"],
                "from": 0,
                "size": 1000
            };
            var jreq = {
                type: 'exec',
                mbean: 'org.elasticsearch:service=restjmx',
                operation: 'exec',
                arguments: ['POST', '/_all/camel/_search', angular.toJson(query)] };
            jolokia.request(jreq, {
                method: 'POST',
                error: function (response) {
                    $scope.result = $scope.result + "<br/>" + "Error: " + angular.toJson(response);
                },
                success: function (response) {
                    var data = jQuery.parseJSON(response.value);
                    var oldsize = $scope.breadcrumbs.length;
                    for (var i = 0; i < data['hits']['hits'].length; i++) {
                        var fields = data['hits']['hits'][i].fields;
                        var concat = function (breadcrumbs) {
                            if (breadcrumbs) {
                                if (typeof breadcrumbs === 'string') {
                                    breadcrumbs = [breadcrumbs];
                                }
                                for (var j = 0; j < breadcrumbs.length; j++) {
                                    var id = breadcrumbs[j];
                                    if ($scope.breadcrumbs.indexOf(id) < 0) {
                                        $scope.breadcrumbs.push(id);
                                    }
                                }
                            }
                        };
                        concat(fields["exchange.in.headers.ExtendedBreadcrumb"]);
                        concat(fields["exchange.out.headers.ExtendedBreadcrumb"]);
                    }
                    $scope.result = $scope.result + "<br/>" + "Found " + data.hits.total + " ids";
                    if (oldsize != $scope.breadcrumbs.length) {
                        request();
                    } else {
                        var ids = [];
                        for (var i = 0; i < data['hits']['hits'].length; i++) {
                            var id = data['hits']['hits'][i].fields["exchange.id"];
                            if (ids.indexOf(id) < 0) {
                                ids.push(id);
                            }
                        }
                        var queryStr = "exchange.id:\"" + ids.join("\" or exchange.id:\"") + "\"";
                        $scope.result = $scope.result + "<br/>" + query;
                        var query = {
                            "query": { "query_string": { "query": queryStr } },
                            "from": 0,
                            "size": 1000,
                            "sort": ["timestamp"]
                        };
                        var jreq = {
                            type: 'exec',
                            mbean: 'org.elasticsearch:service=restjmx',
                            operation: 'exec',
                            arguments: ['POST', '/_all/camel/_search', angular.toJson(query)] };
                        jolokia.request(jreq, {
                            method: 'POST',
                            error: function (response) {
                                $scope.result = $scope.result + "<br/>" + "Error: " + angular.toJson(response);
                            },
                            success: function (response) {
                                var data = jQuery.parseJSON(response.value);
                                $scope.result = $scope.result + "<br/>" + "Found " + data['hits']['total'] + " exchanges";
                                var events = [];
                                for (var i = 0; i < data['hits']['hits'].length; i++) {
                                    var e = data['hits']['hits'][i]._source;
                                    events.push(e);
                                }
                                draw(events);
                            }
                        });
                    }
                } });
        };

        var isoDate = function (date) {
            var timestamp, struct, minutesOffset = 0;
            var numericKeys = [1, 4, 5, 6, 7, 10, 11];

            // ES5 §15.9.4.2 states that the string should attempt to be parsed as a Date Time String Format string
            // before falling back to any implementation-specific date parsing, so that’s what we do, even if native
            // implementations could be faster
            //              1 YYYY                2 MM       3 DD           4 HH    5 mm       6 ss        7 msec        8 Z 9 ±    10 tzHH    11 tzmm
            if ((struct = /^(\d{4}|[+\-]\d{6})(?:-(\d{2})(?:-(\d{2}))?)?(?:T(\d{2}):(\d{2})(?::(\d{2})(?:\.(\d{3}))?)?(?:(Z)|([+\-])(\d{2})(?::(\d{2}))?)?)?$/.exec(date))) {
                for (var i = 0, k; (k = numericKeys[i]); ++i) {
                    struct[k] = +struct[k] || 0;
                }

                // allow undefined days and months
                struct[2] = (+struct[2] || 1) - 1;
                struct[3] = +struct[3] || 1;

                if (struct[8] !== 'Z' && struct[9] !== undefined) {
                    minutesOffset = struct[10] * 60 + struct[11];

                    if (struct[9] === '+') {
                        minutesOffset = 0 - minutesOffset;
                    }
                }

                timestamp = Date.UTC(struct[1], struct[2], struct[3], struct[4], struct[5] + minutesOffset, struct[6], struct[7]);
            } else {
                timestamp = Date.parse(date);
            }

            return timestamp;
        };

        var buildSequence = function (events) {
            var sequence = new Camin.Sequence();
            var exchangeToExec = {};

            // Sort events
            events = events.sort(function (a, b) {
                return isoDate(a.timestamp) - isoDate(b.timestamp);
            });

            for (var i = 0; i < events.length; i++) {
                if (events[i].event === 'Created') {
                    var evtCreated = events[i];
                    var evtCompleted = null;
                    for (var j = 0; j < events.length; j++) {
                        if (events[j].event === 'Completed' && evtCreated.exchange.id === events[j].exchange.id) {
                            evtCompleted = events[j];
                            break;
                        }
                    }
                    if (evtCompleted === null) {
                        console.log('Could not find matching Completed exchange for ' + evtCreated.exchange.id);
                        continue;
                    }

                    // We use the completed event here because the created event may miss the routeId information
                    var endpoint = sequence.endpoint(evtCompleted.exchange.fromEndpoint, evtCompleted.exchange.routeId, evtCompleted.exchange.contextId, evtCompleted.host);
                    var exec = sequence.exec(evtCreated.exchange.id, endpoint, isoDate(evtCreated.timestamp), isoDate(evtCompleted.timestamp));
                    exchangeToExec[evtCreated.exchange.id] = exec;
                }
            }

            // Extract calls
            var calls = {};
            for (var i = 0; i < events.length; i++) {
                if (events[i].event === 'Sending' && events[i].exchange.in && events[i].exchange.in.headers) {
                    var callId = events[i].exchange.in.headers.AuditCallId;
                    if (callId && calls[callId] === undefined) {
                        var evtSending = events[i];
                        var evtSent = null;
                        var evtCreated = null;
                        for (var j = 0; j < events.length; j++) {
                            if (events[j].event === 'Sent' && evtSending.exchange.id === events[j].exchange.id && events[j].exchange.in.headers.AuditCallId === callId) {
                                evtSent = events[j];
                                break;
                            }
                        }
                        for (var j = 0; j < events.length; j++) {
                            if (events[j].event === 'Created' && evtSending.exchange.id !== events[j].exchange.id && events[j].exchange.in.headers.AuditCallId === callId) {
                                evtCreated = events[j];
                                break;
                            }
                        }
                        var execA = exchangeToExec[evtSending.exchange.id];
                        var execB = evtCreated ? exchangeToExec[evtCreated.exchange.id] : null;
                        if (evtSent !== null && evtCreated !== null && execA !== null && execB != null) {
                            var call = sequence.call(callId, execA, execB, isoDate(evtSending.timestamp), isoDate(evtSent.timestamp));
                            calls[callId] = call;
                        } else {
                            console.log("Could not find Execution for exchange " + evtSending.exchange.id);
                        }
                    }
                }
            }
            return sequence;
        };

        var buildDiagram = function (sequence) {
            var diagram = new Camin.Diagram();
            var actors = {};
            var signals = [];
            var base = sequence.start();
            for (var i = 0; i < sequence.endpoints.length; i++) {
                var actor = diagram.actor("ep" + i);
                var ep = sequence.endpoints[i];
                var key = ep.url + "|" + ep.routeId + "|" + ep.contextId + "|" + ep.host;
                actors[key] = actor;
            }
            for (var i = 0; i < sequence.calls.length; i++) {
                var call = sequence.calls[i];
                if (call.execB) {
                    var epA = call.execA.endpoint;
                    var keyA = epA.url + "|" + epA.routeId + "|" + epA.contextId + "|" + epA.host;
                    var epB = call.execB.endpoint;
                    var keyB = epB.url + "|" + epB.routeId + "|" + epB.contextId + "|" + epB.host;
                    var actorA = actors[keyA];
                    var actorB = actors[keyB];
                    var start1 = call.start - base;
                    var stop1 = call.execB.start - base;
                    var start2 = call.execB.stop - base;
                    var stop2 = call.stop - base;
                    signals.push({
                        actorA: actorA,
                        actorB: actorB,
                        message: start1 + "ms - " + stop1 + "ms",
                        timestamp: start1 });
                    signals.push({
                        actorA: actorB,
                        actorB: actorA,
                        message: start2 + "ms - " + stop2 + "ms",
                        timestamp: start2 });
                }
            }
            signals = signals.sort(function (a, b) {
                return a.timestamp - b.timestamp;
            });
            for (var i = 0; i < signals.length; i++) {
                diagram.signal(signals[i].actorA, signals[i].actorB, signals[i].message);
            }
            return diagram;
        };

        var buildGantt = function (sequence) {
            var gantt = new Camin.Gantt();
            for (var i = 0; i < sequence.endpoints.length; i++) {
                var endpoint = sequence.endpoints[i];
                var resource = gantt.resource(endpoint);
                for (var j = 0; j < sequence.execs.length; j++) {
                    var exec = sequence.execs[j];
                    if (exec.endpoint === endpoint) {
                        gantt.task(resource, exec.start, exec.stop, exec);
                    }
                }
            }
            for (var i = 0; i < sequence.calls.length; i++) {
                var call = sequence.calls[i];
                if (call.execB) {
                    var taskA = gantt.taskByData(call.execA);
                    var taskB = gantt.taskByData(call.execB);
                    gantt.link(call.start, taskA, call.stop, taskB, call);
                }
            }
            gantt.layout();
            return gantt;
        };

        var eventTypeValue = { "Created": 0, "Sending": 1, "Sent": 2, "Completed": 3 };

        var draw = function (events) {
            $scope.definition = "";

            events = events.sort(function (a, b) {
                return isoDate(a.timestamp) - isoDate(b.timestamp);
            });
            console.log(events);

            var sequence = buildSequence(events);
            console.log(sequence);

            var gantt = buildGantt(sequence);
            console.log(gantt);
            $('#gantt').html('');
            drawGantt('#gantt', gantt);

            var diagram = buildDiagram(sequence);
            console.log(diagram);
            $('#diagram').html('');
            drawDiagram('#diagram', diagram);
        };

        var drawDiagram = function (container, diagram) {
            var arrow_size = 10;
            var margin = 10;
            var actor_width = 100;
            var actor_margin = 30;
            var actor_height = 40;
            var signal_height = 30;
            var actor_font = 20;
            var signal_font = 14;
            var width = diagram.actors.length * (actor_width + actor_margin * 2);
            var height = (diagram.signals.length + 1) * signal_height + actor_height * 2 + margin * 2;

            var svg = d3.select(container).append('svg').attr('width', width + 2 * margin).attr('height', height + 2 * margin);
            var g = svg.append('g').attr('text-anchor', 'middle');
            for (var i = 0; i < diagram.actors.length; i++) {
                var actor = diagram.actors[i];
                var gu = g.append('g').attr('transform', 'translate(' + (i * (actor_width + actor_margin * 2) + actor_margin) + ',' + actor_height + ')');
                gu.append('rect').attr('width', actor_width).attr('height', actor_height).attr('stroke', '#000').attr('stroke-width', '2').attr('fill', '#FFFFFF');
                gu.append('text').attr('x', actor_width / 2).attr('y', actor_height / 2).attr('stroke-width', '0').attr('dominant-baseline', 'middle').attr('font-size', actor_font).text(actor.name);

                g.append('line').attr('x1', i * (actor_width + actor_margin * 2) + actor_width / 2 + actor_margin).attr('y1', actor_height * 2).attr('x2', i * (actor_width + actor_margin * 2) + actor_width / 2 + actor_margin).attr('y2', height - actor_height).attr('stroke', '#000').attr('stroke-width', '2');

                var gu = g.append('g').attr('transform', 'translate(' + (i * (actor_width + actor_margin * 2) + actor_margin) + ',' + (height - actor_height) + ')');
                gu.append('rect').attr('width', actor_width).attr('height', actor_height).attr('stroke', '#000').attr('stroke-width', '2').attr('fill', 'white');
                gu.append('text').attr('x', actor_width / 2).attr('y', actor_height / 2).attr('stroke-width', '0').attr('dominant-baseline', 'middle').attr('font-size', actor_font).text(actor.name);
            }
            for (var i = 0; i < diagram.signals.length; i++) {
                var x;
                var y;
                var length;
                var direction;
                var text;

                x = diagram.signals[i].actorA.index * (actor_width + actor_margin * 2) + actor_width / 2 + actor_margin;
                y = (i + 1) * signal_height + actor_height * 2;
                length = Math.abs(diagram.signals[i].actorA.index - diagram.signals[i].actorB.index) * (actor_width + actor_margin * 2);
                direction = diagram.signals[i].actorB.index > diagram.signals[i].actorA.index ? +1 : -1;
                text = diagram.signals[i].message;

                var gu = g.append('g').attr('transform', 'translate(' + x + ',' + y + ')').attr('stroke-width', '2');
                gu.append('rect').attr('x', Math.min(3, length * direction + 3)).attr('y', '-16').attr('width', Math.abs((length - 6) * direction)).attr('height', '19').attr('stroke', 'white').attr('stroke-width', '0').attr('fill', 'white');
                gu.append('line').attr('x1', 0).attr('y1', 0).attr('x2', length * direction).attr('y2', 0).attr('stroke', '#000').attr('stroke-width', '2');
                gu.append('line').attr('x1', length * direction - arrow_size * direction).attr('y1', -arrow_size).attr('x2', length * direction).attr('y2', 0).attr('stroke', '#000').attr('stroke-width', '2');
                gu.append('line').attr('x1', length * direction).attr('y1', 0).attr('x2', length * direction - arrow_size * direction).attr('y2', arrow_size).attr('stroke', '#000').attr('stroke-width', '2');
                gu.append('text').attr('x', length * direction / 2).attr('y', -8).attr('stroke-width', '0').attr('dominant-baseline', 'middle').attr('font-size', signal_font).text(text);
            }
        };

        var drawGantt = function (container, gantt) {
            var lineHeight = 35;
            var lineMargin = 3;
            var arrowWidth = 4;

            var width = 800;
            var height = lineHeight * gantt.resources.length;
            var margin = {
                top: 20,
                right: 40,
                bottom: 20,
                left: 250
            };

            var begin = gantt.start;
            var end = gantt.stop;

            var x = d3.scale.linear().domain([begin - (end - begin) * 0.1, end + (end - begin) * 0.1]).range([0, width]);
            var yt = function (t) {
                return t.resource.index * lineHeight + lineMargin + t.index * (lineHeight - 2 * lineMargin) / (t.max + 1);
            };
            var ht = function (t) {
                return 2 * (lineHeight - 2 * lineMargin) / (t.max + 1);
            };

            var svg = d3.select(container).append('svg').attr('width', width + margin.left + margin.right).attr('height', height + margin.top + margin.bottom);

            var text = svg.append('g').attr('width', width).attr('height', height).attr('transform', 'translate(0,' + margin.top + ')').selectAll('text').data(gantt.resources).enter();
            text.append('text').attr('x', 0).attr('y', function (r) {
                return r.index * lineHeight + lineHeight / 2;
            }).attr('dy', '-0.2em').attr('text-anchor', 'start').text(function (r) {
                var endpoint = r.data;
                var text = endpoint.url;
                if (text.indexOf("Endpoint[") == 0) {
                    text = text.substring(9, text.length - 1);
                }
                return text;
            });
            text.append('text').attr('x', 0).attr('y', function (r) {
                return r.index * lineHeight + lineHeight / 2;
            }).attr('dy', '0.8em').attr('text-anchor', 'start').text(function (r) {
                var endpoint = r.data;
                return endpoint.host + "/" + endpoint.contextId + "/" + endpoint.routeId;
            });

            var g = svg.append('g').attr('width', width).attr('height', height).attr('transform', 'translate(' + margin.left + ',' + margin.top + ')');

            g.append('g').attr('width', width).attr('height', height).selectAll('rect').data(gantt.tasks).enter().append('rect').attr('rx', lineMargin * 2).attr('ry', lineMargin * 2).attr('x', function (t) {
                return x(t.start);
            }).attr('y', yt).attr('height', ht).attr('width', function (t) {
                return x(t.stop) - x(t.start);
            }).attr('stroke', '#000000').attr('stroke-width', '2').attr('fill', function (t) {
                return d3.hsl(Math.random() * 360, 0.8, 0.8).toString();
            });

            var lines = g.append('g').attr('width', width).attr('height', height).attr('stroke', '#404040').attr('stroke-width', '2').selectAll('line').data(gantt.links).enter();
            lines.append('line').attr('x1', function (l) {
                return x(l.start);
            }).attr('y1', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            }).attr('x2', function (l) {
                return x(l.start);
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.start);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.taskB.start);
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.taskB.start);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.taskB.start) - arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2 - arrowWidth;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.taskB.start);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.taskB.start) - arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2 + arrowWidth;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.taskB.stop);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.stop);
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.stop);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.stop);
            }).attr('y2', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.stop);
            }).attr('y1', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            }).attr('x2', function (l) {
                return x(l.stop) - arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskA) + ht(l.taskA) + arrowWidth;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.stop);
            }).attr('y1', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            }).attr('x2', function (l) {
                return x(l.stop) + arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskA) + ht(l.taskA) + arrowWidth;
            });
        };

        if ($routeParams["exchangeId"]) {
            $scope.query = $routeParams["exchangeId"];
            $scope.onQueryChange();
        }
    }
    Camin.Controller = Controller;
})(Camin || (Camin = {}));
/**
* Camel Insight Plugin
*
* @module Camin
* @main Camin
*/
var Camin;
(function (Camin) {
    var pluginName = 'camin';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ngGrid', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/camin', { templateUrl: 'app/camin/html/camin.html' }).when('/camin/:exchangeId', { templateUrl: 'app/camin/html/camin.html' });
    }).run(function (workspace, viewRegistry, helpRegistry) {
        viewRegistry["camin"] = "app/camin/html/layoutCamin.html";
        helpRegistry.addUserDoc('camin', 'app/camin/doc/help.md', function () {
            return Fabric.hasFabric(workspace);
        });

        workspace.topLevelTabs.push({
            id: "camin",
            content: "Camel",
            title: "Insight into Camel",
            isValid: function (workspace) {
                return Fabric.hasFabric(workspace);
            },
            href: function () {
                return "#/camin";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("camin");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Camin || (Camin = {}));
var Camin;
(function (Camin) {
    var Gantt = (function () {
        function Gantt() {
            this.resources = [];
            this.tasks = [];
            this.links = [];
        }
        Gantt.prototype.resource = function (data) {
            var resource = new Resource(data, this.resources.length);
            this.resources.push(resource);
            return resource;
        };

        Gantt.prototype.task = function (resource, start, stop, data) {
            var task = resource.task(start, stop, data);
            this.tasks.push(task);
            return task;
        };

        Gantt.prototype.link = function (start, taskA, stop, taskB, data) {
            var link = new Link(start, taskA, stop, taskB, data);
            this.links.push(link);
            return link;
        };

        Gantt.prototype.layout = function () {
            for (var i = 0; i < this.resources.length; i++) {
                this.resources[i].layout();
                this.start = this.start ? Math.min(this.start, this.resources[i].start) : this.resources[i].start;
                this.stop = this.stop ? Math.max(this.stop, this.resources[i].stop) : this.resources[i].stop;
            }
            for (var i = 0; i < this.links.length; i++) {
                this.start = this.start ? Math.min(this.start, this.links[i].start) : this.links[i].start;
                this.stop = this.stop ? Math.max(this.stop, this.links[i].stop) : this.links[i].stop;
            }
        };

        Gantt.prototype.taskByData = function (data) {
            for (var i = 0; i < this.tasks.length; i++) {
                if (this.tasks[i].data === data) {
                    return this.tasks[i];
                }
            }
            return undefined;
        };
        return Gantt;
    })();
    Camin.Gantt = Gantt;

    var Resource = (function () {
        function Resource(data, index) {
            this.tasks = [];
            this.index = index;
            this.data = data;
        }
        Resource.prototype.task = function (start, stop, data) {
            var task = new Task(start, stop, data, this);
            this.tasks.push(task);
            return task;
        };

        Resource.prototype.layout = function () {
            this.tasks.sort(function (ta, tb) {
                return ta.start - tb.start;
            });
            var bands = [];
            for (var i = 0; i < this.tasks.length; i++) {
                this.start = this.start ? Math.min(this.start, this.tasks[i].start) : this.tasks[i].start;
                this.stop = this.stop ? Math.max(this.stop, this.tasks[i].stop) : this.tasks[i].stop;
                for (var j = 0; j < bands.length; j++) {
                    if (bands[j] < this.tasks[i].start) {
                        bands[j] = this.tasks[i].stop;
                        this.tasks[i].index = j;
                        break;
                    }
                }
                if (!this.tasks[i].index) {
                    var index = bands.length;
                    this.tasks[i].index = index;
                    bands[index] = this.tasks[i].stop;
                }
            }
            for (var i = 0; i < this.tasks.length; i++) {
                this.tasks[i].max = bands.length;
            }
        };
        return Resource;
    })();
    Camin.Resource = Resource;

    var Task = (function () {
        function Task(start, stop, data, resource) {
            this.start = start;
            this.stop = stop;
            this.data = data;
            this.resource = resource;
        }
        return Task;
    })();
    Camin.Task = Task;

    var Link = (function () {
        function Link(start, taskA, stop, taskB, data) {
            this.start = start;
            this.stop = stop;
            this.taskA = taskA;
            this.taskB = taskB;
            this.data = data;
        }
        return Link;
    })();
    Camin.Link = Link;
})(Camin || (Camin = {}));
var Camin;
(function (Camin) {
    var Diagram = (function () {
        function Diagram() {
            this.actors = [];
            this.signals = [];
        }
        Diagram.prototype.actor = function (name) {
            for (var i = 0; i < this.actors.length; i++) {
                if (this.actors[i].name === name) {
                    return this.actors[i];
                }
            }
            var actor = new Actor(name, this.actors.length);
            this.actors.push(actor);
            return actor;
        };

        Diagram.prototype.signal = function (actorA, actorB, message) {
            var signal = new Signal(actorA, actorB, message);
            this.signals.push(signal);
            return signal;
        };
        return Diagram;
    })();
    Camin.Diagram = Diagram;

    var Actor = (function () {
        function Actor(name, index) {
            this.name = name;
            this.index = index;
        }
        return Actor;
    })();
    Camin.Actor = Actor;

    var Signal = (function () {
        function Signal(actorA, actorB, message) {
            this.actorA = actorA;
            this.actorB = actorB;
            this.message = message;
        }
        Signal.prototype.isSelf = function () {
            return this.actorA.index === this.actorB.index;
        };
        return Signal;
    })();
    Camin.Signal = Signal;
})(Camin || (Camin = {}));
var Camin;
(function (Camin) {
    var Sequence = (function () {
        function Sequence() {
            this.endpoints = [];
            this.execs = [];
            this.calls = [];
        }
        Sequence.prototype.endpoint = function (url, routeId, contextId, host) {
            for (var i = 0; i < this.endpoints.length; i++) {
                if (this.endpoints[i].url === url && this.endpoints[i].routeId === routeId && this.endpoints[i].contextId === contextId && this.endpoints[i].host === host) {
                    return this.endpoints[i];
                }
            }
            var endpoint = new Endpoint(url, routeId, contextId, host);
            this.endpoints.push(endpoint);
            return endpoint;
        };

        Sequence.prototype.exec = function (exchangeId, endpoint, start, stop) {
            var exec = new Execution(exchangeId, endpoint, start, stop);
            this.execs.push(exec);
            return exec;
        };

        Sequence.prototype.call = function (callId, execA, execB, start, stop) {
            var call = new Call(callId, execA, execB, start, stop);
            this.calls.push(call);
            return call;
        };

        Sequence.prototype.start = function () {
            var start;
            for (var i = 0; i < this.execs.length; i++) {
                start = start ? Math.min(start, this.execs[i].start) : this.execs[i].start;
            }
            for (var i = 0; i < this.calls.length; i++) {
                start = start ? Math.min(start, this.calls[i].start) : this.calls[i].start;
            }
            return start;
        };

        Sequence.prototype.stop = function () {
            var stop;
            for (var i = 0; i < this.execs.length; i++) {
                stop = stop ? Math.max(stop, this.execs[i].stop) : this.execs[i].stop;
            }
            for (var i = 0; i < this.calls.length; i++) {
                stop = stop ? Math.max(stop, this.calls[i].stop) : this.calls[i].stop;
            }
            return stop;
        };
        return Sequence;
    })();
    Camin.Sequence = Sequence;

    var Endpoint = (function () {
        function Endpoint(url, routeId, contextId, host) {
            this.url = url;
            this.routeId = routeId;
            this.contextId = contextId;
            this.host = host;
        }
        return Endpoint;
    })();
    Camin.Endpoint = Endpoint;

    var Execution = (function () {
        function Execution(exchangeId, endpoint, start, stop) {
            this.exchangeId = exchangeId;
            this.endpoint = endpoint;
            this.start = start;
            this.stop = stop;
        }
        return Execution;
    })();
    Camin.Execution = Execution;

    var Call = (function () {
        function Call(callId, execA, execB, start, stop) {
            this.callId = callId;
            this.execA = execA;
            this.execB = execB;
            this.start = start;
            this.stop = stop;
        }
        return Call;
    })();
    Camin.Call = Call;
})(Camin || (Camin = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function ViewController($scope, $location, workspace, jolokia) {
        $scope.$watch('workspace.tree', function () {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        });

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            setTimeout(loadData, 50);
        });

        function loadData() {
        }
    }
    Maven.ViewController = ViewController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function ArtifactController($scope, $routeParams, workspace, jolokia) {
        $scope.row = {
            groupId: $routeParams["group"] || "",
            artifactId: $routeParams["artifact"] || "",
            version: $routeParams["version"] || "",
            classifier: $routeParams["classifier"] || "",
            packaging: $routeParams["packaging"] || ""
        };
        var row = $scope.row;

        $scope.id = Maven.getName(row);

        Maven.addMavenFunctions($scope, workspace);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            updateTableContents();
        });

        function updateTableContents() {
            var mbean = Maven.getMavenIndexerMBean(workspace);

            // lets query the name and description of the GAV
            if (mbean) {
                jolokia.execute(mbean, "search", row.groupId, row.artifactId, row.version, row.packaging, row.classifier, "", onSuccess(render));
            } else {
                console.log("No MavenIndexerMBean!");
            }
        }

        function render(response) {
            if (response && response.length) {
                var first = response[0];
                row.name = first.name;
                row.description = first.description;
            }
            Core.$apply($scope);
        }
    }
    Maven.ArtifactController = ArtifactController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function DependenciesController($scope, $routeParams, $location, workspace, jolokia) {
        $scope.artifacts = [];
        $scope.group = $routeParams["group"] || "";
        $scope.artifact = $routeParams["artifact"] || "";
        $scope.version = $routeParams["version"] || "";
        $scope.classifier = $routeParams["classifier"] || "";
        $scope.packaging = $routeParams["packaging"] || "";

        $scope.dependencyTree = null;

        Maven.addMavenFunctions($scope, workspace);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            updateTableContents();
        });

        $scope.onSelectNode = function (node) {
            $scope.selected = node;
        };

        $scope.onRootNode = function (rootNode) {
            // process the rootNode
        };

        $scope.validSelection = function () {
            return $scope.selected && $scope.selected !== $scope.rootDependency;
        };

        $scope.viewDetails = function () {
            var dependency = Core.pathGet($scope.selected, ["dependency"]);
            var link = $scope.detailLink(dependency);
            if (link) {
                var path = Core.trimLeading(link, "#");
                console.log("going to view " + path);
                $location.path(path);
            }
        };

        function updateTableContents() {
            var mbean = Maven.getAetherMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "resolveJson(java.lang.String,java.lang.String,java.lang.String,java.lang.String,java.lang.String)", $scope.group, $scope.artifact, $scope.version, $scope.packaging, $scope.classifier, onSuccess(render));
            } else {
                console.log("No AetherMBean!");
            }
        }

        function render(response) {
            if (response) {
                var json = JSON.parse(response);
                if (json) {
                    //console.log("Found json: " + JSON.stringify(json, null, "  "));
                    $scope.dependencyTree = new Folder("Dependencies");
                    $scope.dependencyActivations = [];
                    addChildren($scope.dependencyTree, json);
                    $scope.dependencyActivations.reverse();
                    $scope.rootDependency = $scope.dependencyTree.children[0];
                }
            }
            Core.$apply($scope);
        }

        function addChildren(folder, dependency) {
            var name = Maven.getName(dependency);
            var node = new Folder(name);
            node.key = name.replace(/\//g, '_');
            node["dependency"] = dependency;
            $scope.dependencyActivations.push(node.key);

            /*
            var imageUrl = Camel.getRouteNodeIcon(value);
            node.icon = imageUrl;
            //node.tooltip = tooltip;
            */
            folder.children.push(node);

            var children = dependency["children"];
            angular.forEach(children, function (child) {
                addChildren(node, child);
            });
        }
    }
    Maven.DependenciesController = DependenciesController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function SearchController($scope, $location, workspace, jolokia) {
        var log = Logger.get("Maven");

        $scope.artifacts = [];
        $scope.selected = [];
        $scope.done = false;
        $scope.inProgress = false;
        $scope.form = {
            searchText: ""
        };
        $scope.search = "";
        $scope.searchForm = 'app/maven/html/searchForm.html';

        Maven.addMavenFunctions($scope, workspace);

        var columnDefs = [
            {
                field: 'groupId',
                displayName: 'Group'
            },
            {
                field: 'artifactId',
                displayName: 'Artifact',
                cellTemplate: '<div class="ngCellText" title="Name: {{row.entity.name}}">{{row.entity.artifactId}}</div>'
            },
            {
                field: 'version',
                displayName: 'Version',
                cellTemplate: '<div class="ngCellText" title="Name: {{row.entity.name}}"><a ng-href="{{detailLink(row.entity)}}">{{row.entity.version}}</a</div>'
            }
        ];

        $scope.gridOptions = {
            data: 'artifacts',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            columnDefs: columnDefs,
            rowDetailTemplateId: "artifactDetailTemplate",
            filterOptions: {
                filterText: 'search'
            }
        };

        $scope.hasAdvancedSearch = function (form) {
            return form.searchGroup || form.searchArtifact || form.searchVersion || form.searchPackaging || form.searchClassifier || form.searchClassName;
        };

        $scope.doSearch = function () {
            $scope.done = false;
            $scope.inProgress = true;
            $scope.artifacts = [];

            // ensure ui is updated with search in progress...
            setTimeout(function () {
                Core.$apply($scope);
            }, 50);

            var mbean = Maven.getMavenIndexerMBean(workspace);
            var form = $scope.form;
            if (mbean) {
                var searchText = form.searchText;
                var kind = form.artifactType;
                if (kind) {
                    if (kind === "className") {
                        log.debug("Search for: " + form.searchText + " className");
                        jolokia.execute(mbean, "searchClasses", searchText, onSuccess(render));
                    } else {
                        var paths = kind.split('/');
                        var packaging = paths[0];
                        var classifier = paths[1];
                        log.debug("Search for: " + form.searchText + " packaging " + packaging + " classifier " + classifier);
                        jolokia.execute(mbean, "searchTextAndPackaging", searchText, packaging, classifier, onSuccess(render));
                    }
                } else if (searchText) {
                    log.debug("Search text is: " + form.searchText);
                    jolokia.execute(mbean, "searchText", form.searchText, onSuccess(render));
                } else if ($scope.hasAdvancedSearch(form)) {
                    log.debug("Searching for " + form.searchGroup + "/" + form.searchArtifact + "/" + form.searchVersion + "/" + form.searchPackaging + "/" + form.searchClassifier + "/" + form.searchClassName);

                    jolokia.execute(mbean, "search", form.searchGroup || "", form.searchArtifact || "", form.searchVersion || "", form.searchPackaging || "", form.searchClassifier || "", form.searchClassName || "", onSuccess(render));
                }
            } else {
                notification("error", "Cannot find the Maven Indexer MBean!");
            }
        };

        var RESPONSE_LIMIT = 50;

        function render(response) {
            log.debug("Search done, preparing result.");
            $scope.done = true;
            $scope.inProgress = false;

            // let's limit the reponse to avoid blowing up
            // the browser until we start using a widget
            // that supports pagination
            if (response.length > RESPONSE_LIMIT) {
                $scope.tooManyResponses = "This search returned " + response.length + " artifacts, showing the first " + RESPONSE_LIMIT + ", please refine your search";
            } else {
                $scope.tooManyResponses = "";
            }
            $scope.artifacts = response.first(RESPONSE_LIMIT);

            Core.$apply($scope);
        }
    }
    Maven.SearchController = SearchController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function VersionsController($scope, $routeParams, workspace, jolokia) {
        $scope.artifacts = [];
        $scope.group = $routeParams["group"] || "";
        $scope.artifact = $routeParams["artifact"] || "";
        $scope.version = "";
        $scope.classifier = $routeParams["classifier"] || "";
        $scope.packaging = $routeParams["packaging"] || "";

        var id = $scope.group + "/" + $scope.artifact;
        if ($scope.classifier) {
            id += "/" + $scope.classifier;
        }
        if ($scope.packaging) {
            id += "/" + $scope.packaging;
        }
        var columnTitle = id + " versions";

        var columnDefs = [
            {
                field: 'versionNumber',
                displayName: columnTitle,
                cellTemplate: '<div class="ngCellText"><a href="#/maven/artifact/{{row.entity.groupId}}/{{row.entity.artifactId}}/{{row.entity.version}}">{{row.entity.version}}</a></div>'
            }
        ];

        $scope.gridOptions = {
            data: 'artifacts',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            columnDefs: columnDefs,
            rowDetailTemplateId: "artifactDetailTemplate",
            sortInfo: { field: 'versionNumber', direction: 'DESC' },
            filterOptions: {
                filterText: 'search'
            }
        };

        Maven.addMavenFunctions($scope, workspace);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            updateTableContents();
        });

        function updateTableContents() {
            var mbean = Maven.getMavenIndexerMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "versionComplete", $scope.group, $scope.artifact, $scope.version, $scope.packaging, $scope.classifier, onSuccess(render));
            } else {
                console.log("No MavenIndexerMBean!");
            }
        }

        function render(response) {
            $scope.artifacts = [];
            angular.forEach(response, function (version) {
                var versionNumberArray = Core.parseVersionNumbers(version);
                var versionNumber = 0;
                for (var i = 0; i <= 4; i++) {
                    var num = (i >= versionNumberArray.length) ? 0 : versionNumberArray[i];
                    versionNumber *= 1000;
                    versionNumber += num;
                }

                $scope.artifacts.push({
                    groupId: $scope.group,
                    artifactId: $scope.artifact,
                    packaging: $scope.packaging,
                    classifier: $scope.classifier,
                    version: version,
                    versionNumber: versionNumber
                });
            });
            Core.$apply($scope);
        }
    }
    Maven.VersionsController = VersionsController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    Maven.log = Logger.get("Maven");

    /**
    * Returns the maven indexer mbean (from the hawtio-maven-indexer library)
    * @method getMavenIndexerMBean
    * @for Maven
    * @param {Core.Workspace} workspace
    * @return {String}
    */
    function getMavenIndexerMBean(workspace) {
        if (workspace) {
            var mavenStuff = workspace.mbeanTypesToDomain["Indexer"] || {};
            var object = mavenStuff["hawtio"] || {};
            return object.objectName;
        } else
            return null;
    }
    Maven.getMavenIndexerMBean = getMavenIndexerMBean;

    function getAetherMBean(workspace) {
        if (workspace) {
            var mavenStuff = workspace.mbeanTypesToDomain["AetherFacade"] || {};
            var object = mavenStuff["hawtio"] || {};
            return object.objectName;
        } else
            return null;
    }
    Maven.getAetherMBean = getAetherMBean;

    function mavenLink(url) {
        var path = null;
        if (url) {
            if (url.startsWith("mvn:")) {
                path = url.substring(4);
            } else {
                var idx = url.indexOf(":mvn:");
                if (idx > 0) {
                    path = url.substring(idx + 5);
                }
            }
        }
        return path ? "#/maven/artifact/" + path : null;
    }
    Maven.mavenLink = mavenLink;

    function getName(row) {
        var id = (row.group || row.groupId) + "/" + (row.artifact || row.artifactId);
        if (row.version) {
            id += "/" + row.version;
        }
        if (row.classifier) {
            id += "/" + row.classifier;
        }
        if (row.packaging) {
            id += "/" + row.packaging;
        }
        return id;
    }
    Maven.getName = getName;

    function completeMavenUri($q, $scope, workspace, jolokia, query) {
        var mbean = getMavenIndexerMBean(workspace);
        if (!angular.isDefined(mbean)) {
            return $q.when([]);
        }

        var parts = query.split('/');
        if (parts.length === 1) {
            // still searching the groupId
            return Maven.completeGroupId(mbean, $q, $scope, workspace, jolokia, query, null, null);
        }
        if (parts.length === 2) {
            // have the groupId, guess we're looking for the artifactId
            return Maven.completeArtifactId(mbean, $q, $scope, workspace, jolokia, parts[0], parts[1], null, null);
        }
        if (parts.length === 3) {
            // guess we're searching for the version
            return Maven.completeVersion(mbean, $q, $scope, workspace, jolokia, parts[0], parts[1], parts[2], null, null);
        }

        return $q.when([]);
    }
    Maven.completeMavenUri = completeMavenUri;

    function completeVersion(mbean, $q, $scope, workspace, jolokia, groupId, artifactId, partial, packaging, classifier) {
        /*
        if (partial.length < 5) {
        return $q.when([]);
        }
        */
        var deferred = $q.defer();

        jolokia.request({
            type: 'exec',
            mbean: mbean,
            operation: 'versionComplete(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)',
            arguments: [groupId, artifactId, partial, packaging, classifier]
        }, {
            method: 'POST',
            success: function (response) {
                $scope.$apply(function () {
                    deferred.resolve(response.value.sortBy().first(15));
                });
            },
            error: function (response) {
                $scope.$apply(function () {
                    console.log("got back an error: ", response);
                    deferred.reject();
                });
            }
        });

        return deferred.promise;
    }
    Maven.completeVersion = completeVersion;

    function completeArtifactId(mbean, $q, $scope, workspace, jolokia, groupId, partial, packaging, classifier) {
        var deferred = $q.defer();

        jolokia.request({
            type: 'exec',
            mbean: mbean,
            operation: 'artifactIdComplete(java.lang.String, java.lang.String, java.lang.String, java.lang.String)',
            arguments: [groupId, partial, packaging, classifier]
        }, {
            method: 'POST',
            success: function (response) {
                $scope.$apply(function () {
                    deferred.resolve(response.value.sortBy().first(15));
                });
            },
            error: function (response) {
                $scope.$apply(function () {
                    console.log("got back an error: ", response);
                    deferred.reject();
                });
            }
        });

        return deferred.promise;
    }
    Maven.completeArtifactId = completeArtifactId;

    function completeGroupId(mbean, $q, $scope, workspace, jolokia, partial, packaging, classifier) {
        // let's go easy on the indexer
        if (partial.length < 5) {
            return $q.when([]);
        }

        var deferred = $q.defer();

        jolokia.request({
            type: 'exec',
            mbean: mbean,
            operation: 'groupIdComplete(java.lang.String, java.lang.String, java.lang.String)',
            arguments: [partial, packaging, classifier]
        }, {
            method: 'POST',
            success: function (response) {
                $scope.$apply(function () {
                    deferred.resolve(response.value.sortBy().first(15));
                });
            },
            error: function (response) {
                console.log("got back an error: ", response);
                $scope.$apply(function () {
                    deferred.reject();
                });
            }
        });

        return deferred.promise;
    }
    Maven.completeGroupId = completeGroupId;

    function addMavenFunctions($scope, workspace) {
        $scope.detailLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var version = row.version || "";
            var classifier = row.classifier || "";
            var packaging = row.packaging || "";
            if (group && artifact) {
                return "#/maven/artifact/" + group + "/" + artifact + "/" + version + "/" + classifier + "/" + packaging;
            }
            return "";
        };

        $scope.javadocLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var version = row.version;
            if (group && artifact && version) {
                return "javadoc/" + group + ":" + artifact + ":" + version + "/";
            }
            return "";
        };

        $scope.versionsLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var classifier = row.classifier || "";
            var packaging = row.packaging || "";
            if (group && artifact) {
                return "#/maven/versions/" + group + "/" + artifact + "/" + classifier + "/" + packaging;
            }
            return "";
        };

        $scope.dependenciesLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var classifier = row.classifier || "";
            var packaging = row.packaging || "";
            var version = row.version;
            if (group && artifact) {
                return "#/maven/dependencies/" + group + "/" + artifact + "/" + version + "/" + classifier + "/" + packaging;
            }
            return "";
        };

        $scope.hasDependencyMBean = function () {
            var mbean = Maven.getAetherMBean(workspace);
            return angular.isDefined(mbean);
        };

        $scope.sourceLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var version = row.version;
            if (group && artifact && version) {
                return "#/source/index/" + group + ":" + artifact + ":" + version + "/";
            }
            return "";
        };
    }
    Maven.addMavenFunctions = addMavenFunctions;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function TestController($scope, workspace, jolokia, $q, $templateCache) {
        $scope.html = "text/html";

        $scope.someUri = '';
        $scope.uriParts = [];
        $scope.mavenCompletion = $templateCache.get("mavenCompletionTemplate");

        $scope.$watch('someUri', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.uriParts = newValue.split("/");
            }
        });

        $scope.$watch('uriParts', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (newValue.length === 1 && newValue.length < oldValue.length) {
                    if (oldValue.last() !== '' && newValue.first().has(oldValue.last())) {
                        var merged = oldValue.first(oldValue.length - 1).include(newValue.first());
                        $scope.someUri = merged.join('/');
                    }
                }
            }
        }, true);

        $scope.doCompletionMaven = function (something) {
            return Maven.completeMavenUri($q, $scope, workspace, jolokia, something);
        };
    }
    Maven.TestController = TestController;
})(Maven || (Maven = {}));
/**
* @module Maven
* @main Maven
*/
var Maven;
(function (Maven) {
    var pluginName = 'maven';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'datatable', 'tree', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/maven/search', { templateUrl: 'app/maven/html/search.html' }).when('/maven/advancedSearch', { templateUrl: 'app/maven/html/advancedSearch.html' }).when('/maven/artifact/:group/:artifact/:version/:classifier/:packaging', { templateUrl: 'app/maven/html/artifact.html' }).when('/maven/artifact/:group/:artifact/:version/:classifier', { templateUrl: 'app/maven/html/artifact.html' }).when('/maven/artifact/:group/:artifact/:version', { templateUrl: 'app/maven/html/artifact.html' }).when('/maven/dependencies/:group/:artifact/:version/:classifier/:packaging', { templateUrl: 'app/maven/html/dependencies.html' }).when('/maven/dependencies/:group/:artifact/:version/:classifier', { templateUrl: 'app/maven/html/dependencies.html' }).when('/maven/dependencies/:group/:artifact/:version', { templateUrl: 'app/maven/html/dependencies.html' }).when('/maven/versions/:group/:artifact/:classifier/:packaging', { templateUrl: 'app/maven/html/versions.html' }).when('/maven/view/:group/:artifact/:version/:classifier/:packaging', { templateUrl: 'app/maven/html/view.html' }).when('/maven/test', { templateUrl: 'app/maven/html/test.html' });
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['maven'] = "app/maven/html/layoutMaven.html";

        workspace.topLevelTabs.push({
            id: "maven",
            content: "Maven",
            title: "Search maven repositories for artifacts",
            isValid: function (workspace) {
                return Maven.getMavenIndexerMBean(workspace);
            },
            href: function () {
                return "#/maven/search";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("/maven");
            }
        });

        helpRegistry.addUserDoc('maven', 'app/maven/doc/help.md', function () {
            return Maven.getMavenIndexerMBean(workspace) !== null;
        });
        helpRegistry.addDevDoc("maven", 'app/maven/doc/developer.md');
    });

    hawtioPluginLoader.addModule(pluginName);
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function PomXmlController($scope) {
        $scope.mavenPomXml = "\n" + "  <dependency>\n" + "    <groupId>" + orBlank($scope.row.groupId) + "</groupId>\n" + "    <artifactId>" + orBlank($scope.row.artifactId) + "</artifactId>\n" + "    <version>" + orBlank($scope.row.version) + "</version>\n" + "  </dependency>\n";

        function orBlank(text) {
            return text || "";
        }
    }
    Maven.PomXmlController = PomXmlController;
})(Maven || (Maven = {}));
/**
* @module API
*/
var API;
(function (API) {
    function WsdlViewController($scope, $location, jolokia) {
        var log = Logger.get("API");

        API.initScope($scope, $location, jolokia);
        var wsdlNamespace = "http://schemas.xmlsoap.org/wsdl/";

        $scope.url = $location.search()["wsdl"];
        API.loadXml($scope.url, onWsdl);

        $scope.$watch("services", enrichApiDocsWithSchema);
        $scope.$watch("jsonSchema", enrichApiDocsWithSchema);

        function enrichApiDocsWithSchema() {
            var services = $scope.services;
            var jsonSchema = $scope.jsonSchema;
            if (services && jsonSchema) {
                log.info("We have services and jsonSchema!");
                enrichServices(jsonSchema, services);
            }
        }

        function enrichServices(jsonSchema, services) {
            angular.forEach(services, function (service) {
                angular.forEach(service.operations, function (method) {
                    angular.forEach(API.concatArrays([method.inputs, method.outputs]), function (object) {
                        enrichRepresentation(jsonSchema, object);
                    });
                });
            });
        }

        function enrichRepresentation(jsonSchema, representation) {
            var defs = jsonSchema ? jsonSchema["definitions"] : null;
            if (defs && representation) {
                var name = representation["name"];
                if (name) {
                    var foundDef = defs[name];
                    if (foundDef) {
                        // unwrap arrays
                        if (angular.isArray(foundDef) && foundDef.length > 0) {
                            foundDef = foundDef[0];
                        }
                        log.info("Found def " + angular.toJson(foundDef) + " for name " + name);
                        representation["schema"] = foundDef;
                    }
                }
            }
        }

        function onWsdl(response) {
            $scope.services = [];
            var root = response.documentElement;
            var targetNamespace = root ? root.getAttribute("targetNamespace") : null;
            var name = root ? root.getAttribute("name") : null;
            var portTypes = response.getElementsByTagNameNS(wsdlNamespace, "portType");
            var services = response.getElementsByTagNameNS(wsdlNamespace, "service");
            var bindings = response.getElementsByTagNameNS(wsdlNamespace, "binding");

            angular.forEach(portTypes, function (portType) {
                var service = {
                    name: name,
                    targetNamespace: targetNamespace,
                    portName: portType.getAttribute("name") || "Unknown",
                    operations: []
                };
                $scope.services.push(service);
                var operations = portType.getElementsByTagNameNS(wsdlNamespace, "operation");
                angular.forEach(operations, function (operation) {
                    var input = operation.getElementsByTagNameNS(wsdlNamespace, "input");
                    var output = operation.getElementsByTagNameNS(wsdlNamespace, "output");

                    function createMessageData(data) {
                        var answer = [];
                        angular.forEach(data, function (item) {
                            var name = item.getAttribute("name");
                            if (name) {
                                answer.push({
                                    name: name
                                });
                            }
                        });
                        return answer;
                    }

                    var opData = {
                        name: operation.getAttribute("name") || "Unknown",
                        inputs: createMessageData(input),
                        outputs: createMessageData(output)
                    };
                    service.operations.push(opData);
                });
            });
            Core.$apply($scope);
        }
    }
    API.WsdlViewController = WsdlViewController;
})(API || (API = {}));
/**
* @module API
*/
var API;
(function (API) {
    function WadlViewController($scope, $location, $http, jolokia) {
        API.initScope($scope, $location, jolokia);

        $scope.url = $location.search()["wadl"];
        API.loadXml($scope.url, onWsdl);

        $scope.$watch("apidocs", enrichApiDocsWithSchema);
        $scope.$watch("jsonSchema", enrichApiDocsWithSchema);

        $scope.tryInvoke = function (resource, method) {
            if (resource) {
                var path = resource.fullPath || resource.path;
                if (path) {
                    // lets substitue the parameters
                    angular.forEach(resource.param, function (param) {
                        var name = param.name;
                        if (name) {
                            var value = param.value;
                            if (angular.isUndefined(value) || value === null) {
                                value = "";
                            }
                            value = value.toString();
                            API.log.debug("replacing " + name + " with '" + value + "'");
                            path = path.replace(new RegExp("{" + name + "}", "g"), value);
                        }
                    });
                    API.log.info("Lets invoke resource: " + path);
                    var url = Core.useProxyIfExternal(path);
                    var methodName = method.name || "GET";
                    method.invoke = {
                        url: url,
                        running: true
                    };
                    var requestData = {
                        method: methodName,
                        url: url,
                        headers: {}
                    };
                    if (methodName === "POST" || methodName === "PUT") {
                        // lets see if we can find a payload
                        angular.forEach(method.request, function (request) {
                            if (!requestData["data"]) {
                                requestData["data"] = request.value;
                            }
                            if (!requestData.headers["Content-Type"]) {
                                requestData.headers["Content-Type"] = request.contentType;
                            }
                        });
                    }
                    API.log.info("About to make request: " + angular.toJson(requestData));
                    $http(requestData).success(function (data, status, headers, config) {
                        API.log.info("Worked!" + data);
                        method.invoke = {
                            url: url,
                            realUrl: path,
                            success: true,
                            data: data,
                            dataMode: textFormat(headers),
                            status: status,
                            headers: headers(),
                            config: config
                        };
                        Core.$apply($scope);
                    }).error(function (data, status, headers, config) {
                        // called asynchronously if an error occurs
                        // or server returns response with an error status.
                        API.log.info("Failed: " + status);
                        method.invoke = {
                            url: url,
                            realUrl: path,
                            data: data,
                            dataMode: textFormat(headers),
                            status: status,
                            headers: headers(),
                            config: config
                        };
                        Core.$apply($scope);
                    });
                }
            }
        };

        function textFormat(headers) {
            return contentTypeTextFormat(headers("content-type"));
        }

        function contentTypeTextFormat(contentType) {
            if (contentType) {
                if (contentType.endsWith("xml")) {
                    return "xml";
                }
                if (contentType.endsWith("html")) {
                    return "html";
                }
                if (contentType.endsWith("json")) {
                    return "json";
                }
            }
            return null;
        }

        function enrichApiDocsWithSchema() {
            var apidocs = $scope.apidocs;
            var jsonSchema = $scope.jsonSchema;
            if (apidocs && jsonSchema) {
                enrichResources(jsonSchema, apidocs.resources);
            }
        }

        function enrichResources(jsonSchema, resources, parentUri) {
            if (typeof parentUri === "undefined") { parentUri = null; }
            angular.forEach(resources, function (resource) {
                var base = resource.base;
                if (base) {
                    if (parentUri) {
                        base = parentUri + base;
                    }
                } else {
                    base = parentUri;
                }
                var path = resource.path;
                if (base && path) {
                    if (!base.endsWith("/") && !path.startsWith("/")) {
                        base += "/";
                    }
                    base += path;
                    resource["fullPath"] = base;
                }
                var childResources = resource.resource;
                if (childResources) {
                    enrichResources(jsonSchema, childResources, base);
                }
                angular.forEach(API.concatArrays([resource.method, resource.operation]), function (method) {
                    // lets remove any empty requests
                    var request = method.request;
                    if (request) {
                        var count = request.count(function (n) {
                            return n["representation"];
                        });
                        if (!count) {
                            delete method.request;
                        }
                    }
                    angular.forEach(API.concatArrays([method.request, method.response]), function (object) {
                        var element = object["element"];
                        var representations = object["representation"];
                        if (representations) {
                            var mediaTypes = representations.map(function (r) {
                                return r["mediaType"];
                            });
                            object["mediaTypes"] = mediaTypes;
                            if (mediaTypes && mediaTypes.length) {
                                object["contentType"] = mediaTypes[0];
                            }
                        }
                        angular.forEach(representations, function (representation) {
                            if (!element) {
                                element = representation["element"];
                            }
                            enrichRepresentation(jsonSchema, representation);
                        });
                        if (element) {
                            object["element"] = element;
                        }
                    });
                });
            });
        }

        function enrichRepresentation(jsonSchema, representation) {
            var defs = jsonSchema ? jsonSchema["definitions"] : null;
            if (defs && representation) {
                var contentType = representation["mediaType"];
                if (contentType) {
                    representation["dataMode"] = contentTypeTextFormat(contentType);
                }

                // TODO find a class name in the representation?
                var element = representation["element"];
                if (element) {
                    var idx = element.indexOf(':');
                    if (idx >= 0) {
                        element = element.substring(idx + 1);
                    }

                    // lets see if we can find a definition which ends with this element
                    var elementPostfix = "." + element;
                    var foundDef = null;
                    angular.forEach(defs, function (value, key) {
                        if (!foundDef && (key === element || key.endsWith(elementPostfix))) {
                            foundDef = value;
                            representation["schema"] = foundDef;
                            representation["typeName"] = element;
                            representation["javaClass"] = key;
                        }
                    });
                }
            }
        }

        function onWsdl(response) {
            $scope.apidocs = API.onWadlXmlLoaded(response);

            //log.info("API docs: " + JSON.stringify($scope.apidocs, null, "  "));
            Core.$apply($scope);
        }
    }
    API.WadlViewController = WadlViewController;
})(API || (API = {}));
/**
* API plugin for browsing WSDL and WADL
*
* @module API
* @main API
*/
var API;
(function (API) {
    var pluginName = 'api';
    angular.module(pluginName, ['bootstrap', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/api/wsdl', { templateUrl: 'app/api/html/wsdl.html' }).when('/api/wadl', { templateUrl: 'app/api/html/wadl.html' });
    }).run(function ($location, workspace, viewRegistry, layoutFull, helpRegistry) {
        viewRegistry['api'] = layoutFull;
        /*
        helpRegistry.addUserDoc('log', 'app/wsdl/doc/help.md', () => {
        return workspace.treeContainsDomainAndProperties('org.fusesource.insight', {type: 'LogQuery'});
        });
        */
    });

    hawtioPluginLoader.addModule(pluginName);
})(API || (API = {}));
/**
* @module API
*/
var API;
(function (API) {
    API.log = Logger.get("API");

    API.wadlNamespace = "http://schemas.xmlsoap.org/wsdl/";

    /**
    * Loads the XML for the given url if its defined or ignore if not valid
    * @method loadXml
    * @for API
    * @static
    * @param {String} url
    * @param {Function} onXml
    *
    */
    function loadXml(url, onXml) {
        if (url) {
            API.log.info("Loading XML: " + url);

            var ajaxParams = {
                type: "GET",
                url: url,
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('Authorization', null);
                },
                dataType: "xml",
                contextType: "text/xml",
                success: onXml,
                error: function (jqXHR, textStatus, errorThrow) {
                    API.log.error("Failed to query XML for: " + url + " status:" + textStatus + " error: " + errorThrow);
                }
            };
            $.ajax(ajaxParams);
        }
    }
    API.loadXml = loadXml;

    var wadlXmlToJavaConfig = {};

    function parseJson(json) {
        var answer = null;
        try  {
            //console.log("got JSON: " + responseJson);
            answer = JSON.parse(json);
        } catch (e) {
            API.log.info("Failed to parse JSON " + e);
            API.log.info("JSON: " + json);
        }
        return answer;
    }
    API.parseJson = parseJson;

    function initScope($scope, $location, jolokia) {
        var search = $location.search();
        $scope.container = search["container"];
        $scope.objectName = search["objectName"];

        $scope.showHide = function (resource) {
            if (resource) {
                resource.hide = resource.hide ? false : true;
            }
        };

        $scope.showOperations = function (resource) {
            showHideOperations(resource, false);
        };

        $scope.expandOperations = function (resource) {
            showHideOperations(resource, true);
        };

        function showHideOperations(resource, flag) {
            if (resource) {
                resource.hide = false;
                angular.forEach(resource.resource, function (childResource) {
                    showHideOperations(childResource, flag);
                });
                angular.forEach(resource.method || resource.operations, function (method) {
                    method.expanded = flag;
                });
            }
        }

        $scope.autoFormat = function (codeMirror) {
            if (!codeMirror) {
                // lets try find the codeMirror in a child scope as a hack :)
                codeMirror = findChildScopeValue($scope, "codeMirror");
            }
            if (codeMirror) {
                setTimeout(function () {
                    CodeEditor.autoFormatEditor(codeMirror);
                }, 50);
            }
        };

        /**
        * Note using this method is usually a dirty hack ;)
        */
        function findChildScopeValue(scope, name) {
            var answer = scope[name];
            var childScope = scope.$$childHead;
            while (childScope && !answer) {
                answer = findChildScopeValue(childScope, name);
                childScope = childScope.$$nextSibling;
            }
            return answer;
        }

        if ($scope.container && $scope.objectName) {
            Fabric.containerJolokia(jolokia, $scope.container, function (remoteJolokia) {
                $scope.remoteJolokia = remoteJolokia;
                if (remoteJolokia) {
                    API.loadJsonSchema(remoteJolokia, $scope.objectName, function (jsonSchema) {
                        //log.info("Got JSON Schema: " + JSON.stringify(jsonSchema, null, "  "));
                        $scope.jsonSchema = jsonSchema;
                        Core.$apply($scope);
                    });
                } else {
                    API.log.info("No Remote Jolokia!");
                }
            });
        } else {
            API.log.info("No container or objectName");
        }
        API.log.info("container: " + $scope.container + " objectName: " + $scope.objectName + " url: " + $scope.url);
    }
    API.initScope = initScope;

    /**
    * Loads the JSON schema from a given CXF endpoint mbean
    * @method loadJsonSchema
    * @for API
    * @static
    * @paran {*} jolokia
    * @param {String} mbean
    * @param {Function} onJsonSchemaFn
    */
    function loadJsonSchema(jolokia, mbean, onJsonSchemaFn) {
        function onResults(response) {
            var schema = {};
            if (response) {
                var json = response;
                if (json) {
                    schema = parseJson(json);
                }
            }
            onJsonSchemaFn(schema);
        }
        if (mbean) {
            return jolokia.execute(mbean, "getJSONSchema", onSuccess(onResults));
        } else {
            var schema = {};
            onJsonSchemaFn(schema);
            return schema;
        }
    }
    API.loadJsonSchema = loadJsonSchema;

    /**
    * When a WADL XML document is loaded, lets convert it to JSON and return it
    * @method onWadlXmlLoaded
    * @for API
    * @static
    * @param {any} response
    * @return {any}
    */
    function onWadlXmlLoaded(response) {
        var root = response.documentElement;
        var output = {};
        return API.convertWadlToJson(root, output);
    }
    API.onWadlXmlLoaded = onWadlXmlLoaded;

    /**
    * Converts the given XML element from WADL to JSON
    * @method convertWadlToJson
    * @for API
    * @static
    * @param {any} element
    * @param {any} obj
    * @return {any}
    */
    function convertWadlToJson(element, obj) {
        if (typeof obj === "undefined") { obj = {}; }
        return API.convertXmlToJson(element, obj, wadlXmlToJavaConfig);
    }
    API.convertWadlToJson = convertWadlToJson;

    function convertWadlJsonToSwagger(object) {
        // now lets convert to swagger style json
        var apis = [];
        var basePath = null;
        var resourcePath = null;

        var resources = Core.pathGet(object, ["resources", 0]);
        if (resources) {
            basePath = resources.base;
            angular.forEach(resources.resource, function (resource) {
                var path = resource.path;
                var operations = [];
                angular.forEach(resource.method, function (method) {
                    var name = method.name;
                    var responseMessages = [];

                    /*
                    {
                    "code": 404,
                    "message": "There are no businesses"
                    }
                    */
                    var parameters = [];

                    /*
                    {
                    "name": "query",
                    "description": "a text query to search across facilities",
                    "required": false,
                    "allowMultiple": false,
                    "dataType": "string",
                    "paramType": "query"
                    }
                    
                    */
                    operations.push({
                        "method": method.name,
                        "summary": method.summary,
                        "notes": method.notes,
                        "nickname": method.nickname,
                        "type": method.type,
                        "parameters": parameters,
                        "produces": [
                            "application/json"
                        ],
                        "responseMessages": responseMessages
                    });
                });

                apis.push({
                    path: path,
                    operations: operations
                });
            });
        }
        return {
            "apiVersion": "1.0",
            "swaggerVersion": "1.2",
            "basePath": basePath,
            "resourcePath": resourcePath,
            "produces": [
                "application/json"
            ],
            apis: apis
        };
    }
    API.convertWadlJsonToSwagger = convertWadlJsonToSwagger;

    function nodeName(owner, node) {
        return node ? node.localName : null;
    }

    /**
    * Converts the given child elements or attributes into properties on the object
    * to convert the XML into JSON using the given config to customise which properties should
    * be considered singular
    * @method convertXmlToJson
    * @for API
    * @static
    * @param {any} element
    * @param {any} obj
    * @param {any} config
    * @return {any}
    */
    function convertXmlToJson(element, obj, config) {
        var elementProperyFn = config.elementToPropertyName || nodeName;
        var attributeProperyFn = config.attributeToPropertyName || nodeName;

        angular.forEach(element.childNodes, function (child) {
            if (child.nodeType === 1) {
                var propertyName = elementProperyFn(element, child);
                if (propertyName) {
                    // TODO should we assume everything is a list and then flatten later?
                    var array = obj[propertyName] || [];
                    if (!angular.isArray(array)) {
                        array = [array];
                    }
                    var value = {};
                    convertXmlToJson(child, value, config);
                    array.push(value);
                    obj[propertyName] = array;
                }
            }
        });
        angular.forEach(element.attributes, function (attr) {
            var propertyName = attributeProperyFn(element, attr);
            if (propertyName) {
                var value = attr.nodeValue;
                obj[propertyName] = value;
            }
        });
        return obj;
    }
    API.convertXmlToJson = convertXmlToJson;

    /**
    * Concatenate all the non-null arrays into a single array
    * @param arrays an array of arrays
    * @return the single flatten arrays with any null/undefined values ignored
    */
    function concatArrays(arrays) {
        var answer = [];
        angular.forEach(arrays, function (array) {
            if (array) {
                if (angular.isArray(array)) {
                    answer = answer.concat(array);
                } else {
                    answer.push(array);
                }
            }
        });
        return answer;
    }
    API.concatArrays = concatArrays;
})(API || (API = {}));
/**
* @module Dozer
* @main Dozer
*/
var Dozer;
(function (Dozer) {
    /**
    * The JMX domain for Dozer
    * @property jmxDomain
    * @for Dozer
    * @type String
    */
    Dozer.jmxDomain = 'net.sourceforge.dozer';

    /**
    * Don't try and load properties for these types
    * @property excludedPackages
    * @for Dozer
    * @type {Array}
    */
    Dozer.excludedPackages = [
        'java.lang',
        'int',
        'double',
        'long'
    ];

    /**
    * Lets map the class names to element names
    * @property elementNameMappings
    * @for Dozer
    * @type {Array}
    */
    Dozer.elementNameMappings = {
        "Mapping": "mapping",
        "MappingClass": "class",
        "Field": "field"
    };

    Dozer.log = Logger.get("Dozer");

    /**
    * Converts the XML string or DOM node to a Dozer model
    * @method loadDozerModel
    * @for Dozer
    * @static
    * @param {Object} xml
    * @param {String} pageId
    * @return {Mappings}
    */
    function loadDozerModel(xml, pageId) {
        var doc = xml;
        if (angular.isString(xml)) {
            doc = $.parseXML(xml);
        }
        console.log("Has Dozer XML document: " + doc);

        var model = new Dozer.Mappings(doc);
        var mappingsElement = doc.documentElement;
        copyAttributes(model, mappingsElement);

        $(mappingsElement).children("mapping").each(function (idx, element) {
            var mapping = createMapping(element);
            model.mappings.push(mapping);
        });

        return model;
    }
    Dozer.loadDozerModel = loadDozerModel;

    function saveToXmlText(model) {
        // lets copy the original doc then replace the mapping elements
        var element = model.doc.documentElement.cloneNode(false);
        appendElement(model.mappings, element, null, 1);
        Dozer.addTextNode(element, "\n");
        var xmlText = Core.xmlNodeToString(element);
        return '<?xml version="1.0" encoding="UTF-8"?>\n' + xmlText;
    }
    Dozer.saveToXmlText = saveToXmlText;

    function findUnmappedFields(workspace, mapping, fn) {
        // lets find the fields which are unmapped
        var className = mapping.class_a.value;
        findProperties(workspace, className, null, function (properties) {
            var answer = [];
            angular.forEach(properties, function (property) {
                console.log("got property " + JSON.stringify(property, null, "  "));
                var name = property.name;
                if (name) {
                    if (mapping.hasFromField(name)) {
                        // ignore this one
                    } else {
                        // TODO auto-detect this property name in the to classes?
                        answer.push(new Dozer.UnmappedField(name, property));
                    }
                }
            });
            fn(answer);
        });
    }
    Dozer.findUnmappedFields = findUnmappedFields;

    /**
    * Finds the properties on the given class and returns them; and either invokes the given function
    * or does a sync request and returns them
    * @method findProperties
    * @for Dozer
    * @static
    * @param {Core.Workspace} workspace
    * @param {String} className
    * @param {String} filter
    * @param {Function} fn
    * @return {any}
    */
    function findProperties(workspace, className, filter, fn) {
        if (typeof filter === "undefined") { filter = null; }
        if (typeof fn === "undefined") { fn = null; }
        var mbean = getIntrospectorMBean(workspace);
        if (mbean) {
            if (filter) {
                return workspace.jolokia.execute(mbean, "findProperties", className, filter, onSuccess(fn));
            } else {
                return workspace.jolokia.execute(mbean, "getProperties", className, onSuccess(fn));
            }
        } else {
            if (fn) {
                return fn([]);
            } else {
                return [];
            }
        }
    }
    Dozer.findProperties = findProperties;

    /**
    * Finds class names matching the given search text and either invokes the function with the results
    * or does a sync request and returns them.
    * @method findClassNames
    * @for Dozer
    * @static
    * @param {Core.Workspace} workspace
    * @param {String} searchText
    * @param {Number} limit @default 20
    * @param {Function} fn
    * @return {any}
    */
    function findClassNames(workspace, searchText, limit, fn) {
        if (typeof limit === "undefined") { limit = 20; }
        if (typeof fn === "undefined") { fn = null; }
        var mbean = getIntrospectorMBean(workspace);
        if (mbean) {
            return workspace.jolokia.execute(mbean, "findClassNames", searchText, limit, onSuccess(fn));
        } else {
            if (fn) {
                return fn([]);
            } else {
                return [];
            }
        }
    }
    Dozer.findClassNames = findClassNames;

    function getIntrospectorMBean(workspace) {
        return Core.getMBeanTypeObjectName(workspace, "hawtio", "Introspector");
    }
    Dozer.getIntrospectorMBean = getIntrospectorMBean;

    function loadModelFromTree(rootTreeNode, oldModel) {
        oldModel.mappings = [];
        angular.forEach(rootTreeNode.childList, function (treeNode) {
            var mapping = Core.pathGet(treeNode, ["data", "entity"]);
            if (mapping) {
                oldModel.mappings.push(mapping);
            }
        });
        return oldModel;
    }
    Dozer.loadModelFromTree = loadModelFromTree;

    function createDozerTree(model) {
        var id = "mappings";
        var folder = new Folder(id);
        folder.addClass = "net-sourceforge-dozer-mappings";
        folder.domain = Dozer.jmxDomain;
        folder.typeName = "mappings";
        folder.entity = model;

        folder.key = Core.toSafeDomID(id);

        angular.forEach(model.mappings, function (mapping) {
            var mappingFolder = createMappingFolder(mapping, folder);
            folder.children.push(mappingFolder);
        });
        return folder;
    }
    Dozer.createDozerTree = createDozerTree;

    function createMappingFolder(mapping, parentFolder) {
        var mappingName = mapping.name();
        var mappingFolder = new Folder(mappingName);
        mappingFolder.addClass = "net-sourceforge-dozer-mapping";
        mappingFolder.typeName = "mapping";
        mappingFolder.domain = Dozer.jmxDomain;
        mappingFolder.key = (parentFolder ? parentFolder.key + "_" : "") + Core.toSafeDomID(mappingName);
        mappingFolder.parent = parentFolder;
        mappingFolder.entity = mapping;
        mappingFolder.icon = url("/app/dozer/img/class.gif");

        /*
        mappingFolder.tooltip = nodeSettings["tooltip"] || nodeSettings["description"] || id;
        */
        angular.forEach(mapping.fields, function (field) {
            addMappingFieldFolder(field, mappingFolder);
        });
        return mappingFolder;
    }
    Dozer.createMappingFolder = createMappingFolder;

    function addMappingFieldFolder(field, mappingFolder) {
        var name = field.name();
        var fieldFolder = new Folder(name);
        fieldFolder.addClass = "net-sourceforge-dozer-field";
        fieldFolder.typeName = "field";
        fieldFolder.domain = Dozer.jmxDomain;
        fieldFolder.key = mappingFolder.key + "_" + Core.toSafeDomID(name);
        fieldFolder.parent = mappingFolder;
        fieldFolder.entity = field;
        fieldFolder.icon = url("/app/dozer/img/attribute.gif");

        /*
        fieldFolder.tooltip = nodeSettings["tooltip"] || nodeSettings["description"] || id;
        */
        mappingFolder.children.push(fieldFolder);
        return fieldFolder;
    }
    Dozer.addMappingFieldFolder = addMappingFieldFolder;

    function createMapping(element) {
        var mapping = new Dozer.Mapping();
        var elementJQ = $(element);
        mapping.class_a = createMappingClass(elementJQ.children("class-a"));
        mapping.class_b = createMappingClass(elementJQ.children("class-b"));
        elementJQ.children("field").each(function (idx, fieldElement) {
            var field = createField(fieldElement);
            mapping.fields.push(field);
        });
        copyAttributes(mapping, element);
        return mapping;
    }

    function createField(element) {
        if (element) {
            var jqe = $(element);
            var a = jqe.children("a").text();
            var b = jqe.children("b").text();
            var field = new Dozer.Field(new Dozer.FieldDefinition(a), new Dozer.FieldDefinition(b));
            copyAttributes(field, element);
            return field;
        }
        return new Dozer.Field(new Dozer.FieldDefinition(""), new Dozer.FieldDefinition(""));
    }

    function createMappingClass(jqElement) {
        if (jqElement && jqElement[0]) {
            var element = jqElement[0];
            var text = element.textContent;
            if (text) {
                var mappingClass = new Dozer.MappingClass(text);
                copyAttributes(mappingClass, element);
                return mappingClass;
            }
        }

        // lets create a default empty mapping
        return new Dozer.MappingClass("");
    }

    function copyAttributes(object, element) {
        var attributeMap = element.attributes;
        for (var i = 0; i < attributeMap.length; i++) {
            // TODO hacky work around for compiler issue ;)
            //var attr = attributeMap.item(i);
            var attMap = attributeMap;
            var attr = attMap.item(i);
            if (attr) {
                var name = attr.localName;
                var value = attr.value;
                if (name && !name.startsWith("xmlns")) {
                    var safeName = Forms.safeIdentifier(name);
                    object[safeName] = value;
                }
            }
        }
    }

    function appendAttributes(object, element, ignorePropertyNames) {
        angular.forEach(object, function (value, key) {
            if (ignorePropertyNames.any(key)) {
                //console.log("Ignored key " + key);
            } else {
                // lets add an attribute value
                if (value) {
                    var text = value.toString();

                    // lets replace any underscores with dashes
                    var name = key.replace(/_/g, '-');
                    element.setAttribute(name, text);
                }
            }
        });
    }
    Dozer.appendAttributes = appendAttributes;

    /**
    * Adds a new child element for this mapping to the given element
    * @method appendElement
    * @for Dozer
    * @static
    * @param {any} object
    * @param {any} element
    * @param {String} elementName
    * @param {Number} indentLevel
    * @return the last child element created
    */
    function appendElement(object, element, elementName, indentLevel) {
        if (typeof elementName === "undefined") { elementName = null; }
        if (typeof indentLevel === "undefined") { indentLevel = 0; }
        var answer = null;
        if (angular.isArray(object)) {
            angular.forEach(object, function (child) {
                answer = appendElement(child, element, elementName, indentLevel);
            });
        } else if (object) {
            if (!elementName) {
                var className = Core.pathGet(object, ["constructor", "name"]);
                if (!className) {
                    console.log("WARNING: no class name for value " + object);
                } else {
                    elementName = Dozer.elementNameMappings[className];
                    if (!elementName) {
                        console.log("WARNING: could not map class name " + className + " to an XML element name");
                    }
                }
            }
            if (elementName) {
                if (indentLevel) {
                    var text = indentText(indentLevel);
                    Dozer.addTextNode(element, text);
                }
                var doc = element.ownerDocument || document;
                var child = doc.createElement(elementName);

                // navigate child properties...
                var fn = object.saveToElement;
                if (fn) {
                    fn.apply(object, [child]);
                } else {
                    angular.forEach(object, function (value, key) {
                        console.log("has key " + key + " value " + value);
                    });
                }

                // if we have any element children then add newline text node
                if ($(child).children().length) {
                    //var text = indentText(indentLevel - 1);
                    var text = indentText(indentLevel);
                    Dozer.addTextNode(child, text);
                }
                element.appendChild(child);
                answer = child;
            }
        }
        return answer;
    }
    Dozer.appendElement = appendElement;

    function nameOf(object) {
        var text = angular.isObject(object) ? object["value"] : null;
        if (!text && angular.isString(object)) {
            text = object;
        }
        return text || "?";
    }
    Dozer.nameOf = nameOf;

    function addTextNode(element, text) {
        if (text) {
            var doc = element.ownerDocument || document;
            var child = doc.createTextNode(text);
            element.appendChild(child);
        }
    }
    Dozer.addTextNode = addTextNode;

    function indentText(indentLevel) {
        var text = "\n";
        for (var i = 0; i < indentLevel; i++) {
            text += "  ";
        }
        return text;
    }
})(Dozer || (Dozer = {}));
/**
* @module Dozer
*/
var Dozer;
(function (Dozer) {
    /**
    * Configures the JSON schemas to improve the UI models
    * @method schemaConfigure
    * @for Dozer
    */
    function schemaConfigure() {
        io_hawt_dozer_schema_Field["tabs"] = {
            'Fields': ['a.value', 'b.value'],
            'From Field': ['a\\..*'],
            'To Field': ['b\\..*'],
            'Field Configuration': ['*']
        };
        io_hawt_dozer_schema_Mapping["tabs"] = {
            'Classes': ['class-a.value', 'class-b.value'],
            'From Class': ['class-a\\..*'],
            'To Class': ['class-b\\..*'],
            'Class Configuration': ['*']
        };

        // hide the fields table from the class configuration tab
        io_hawt_dozer_schema_Mapping.properties.fieldOrFieldExclude.hidden = true;

        Core.pathSet(io_hawt_dozer_schema_Field, ["properties", "a", "properties", "value", "label"], "From Field");
        Core.pathSet(io_hawt_dozer_schema_Field, ["properties", "b", "properties", "value", "label"], "To Field");

        Core.pathSet(io_hawt_dozer_schema_Mapping, ["properties", "class-a", "properties", "value", "label"], "From Class");
        Core.pathSet(io_hawt_dozer_schema_Mapping, ["properties", "class-b", "properties", "value", "label"], "To Class");

        // ignore prefixes in the generated labels
        Core.pathSet(io_hawt_dozer_schema_Field, ["properties", "a", "ignorePrefixInLabel"], true);
        Core.pathSet(io_hawt_dozer_schema_Field, ["properties", "b", "ignorePrefixInLabel"], true);
        Core.pathSet(io_hawt_dozer_schema_Mapping, ["properties", "class-a", "ignorePrefixInLabel"], true);
        Core.pathSet(io_hawt_dozer_schema_Mapping, ["properties", "class-b", "ignorePrefixInLabel"], true);

        // add custom widgets
        Core.pathSet(io_hawt_dozer_schema_Mapping, ["properties", "class-a", "properties", "value", "formTemplate"], classNameWidget("class_a"));
        Core.pathSet(io_hawt_dozer_schema_Mapping, ["properties", "class-b", "properties", "value", "formTemplate"], classNameWidget("class_b"));

        Core.pathSet(io_hawt_dozer_schema_Field, ["properties", "a", "properties", "value", "formTemplate"], '<input type="text" ng-model="dozerEntity.a.value" ' + 'typeahead="title for title in fromFieldNames($viewValue) | filter:$viewValue" ' + 'typeahead-editable="true"  title="The Java class name"/>');
        Core.pathSet(io_hawt_dozer_schema_Field, ["properties", "b", "properties", "value", "formTemplate"], '<input type="text" ng-model="dozerEntity.b.value" ' + 'typeahead="title for title in toFieldNames($viewValue) | filter:$viewValue" ' + 'typeahead-editable="true"  title="The Java class name"/>');

        function classNameWidget(propertyName) {
            return '<input type="text" ng-model="dozerEntity.' + propertyName + '.value" ' + 'typeahead="title for title in classNames($viewValue) | filter:$viewValue" ' + 'typeahead-editable="true"  title="The Java class name"/>';
        }
    }
    Dozer.schemaConfigure = schemaConfigure;
})(Dozer || (Dozer = {}));
/**
* @module Dozer
*/
var Dozer;
(function (Dozer) {
    /**
    * @class Mappings
    */
    var Mappings = (function () {
        function Mappings(doc, mappings) {
            if (typeof mappings === "undefined") { mappings = []; }
            this.doc = doc;
            this.mappings = mappings;
        }
        return Mappings;
    })();
    Dozer.Mappings = Mappings;

    /**
    * @class Mapping
    */
    var Mapping = (function () {
        function Mapping() {
            this.fields = [];
            this.map_id = Core.getUUID();
            this.class_a = new MappingClass('');
            this.class_b = new MappingClass('');
        }
        Mapping.prototype.name = function () {
            return Dozer.nameOf(this.class_a) + " -> " + Dozer.nameOf(this.class_b);
        };

        Mapping.prototype.hasFromField = function (name) {
            return this.fields.find(function (f) {
                return name === f.a.value;
            });
        };

        Mapping.prototype.hasToField = function (name) {
            return this.fields.find(function (f) {
                return name === f.b.value;
            });
        };

        Mapping.prototype.saveToElement = function (element) {
            Dozer.appendElement(this.class_a, element, "class-a", 2);
            Dozer.appendElement(this.class_b, element, "class-b", 2);
            Dozer.appendElement(this.fields, element, "field", 2);
            Dozer.appendAttributes(this, element, ["class_a", "class_b", "fields"]);
        };
        return Mapping;
    })();
    Dozer.Mapping = Mapping;

    /**
    * @class MappingClass
    */
    var MappingClass = (function () {
        function MappingClass(value) {
            this.value = value;
        }
        MappingClass.prototype.saveToElement = function (element) {
            Dozer.addTextNode(element, this.value);
            Dozer.appendAttributes(this, element, ["value", "properties", "error"]);
        };
        return MappingClass;
    })();
    Dozer.MappingClass = MappingClass;

    /**
    * @class Field
    */
    var Field = (function () {
        function Field(a, b) {
            this.a = a;
            this.b = b;
        }
        Field.prototype.name = function () {
            return Dozer.nameOf(this.a) + " -> " + Dozer.nameOf(this.b);
        };

        Field.prototype.saveToElement = function (element) {
            Dozer.appendElement(this.a, element, "a", 3);
            Dozer.appendElement(this.b, element, "b", 3);
            Dozer.appendAttributes(this, element, ["a", "b"]);
        };
        return Field;
    })();
    Dozer.Field = Field;

    /**
    * @class FieldDefinition
    */
    var FieldDefinition = (function () {
        function FieldDefinition(value) {
            this.value = value;
        }
        FieldDefinition.prototype.saveToElement = function (element) {
            Dozer.addTextNode(element, this.value);
            Dozer.appendAttributes(this, element, ["value", "properties", "error"]);
        };
        return FieldDefinition;
    })();
    Dozer.FieldDefinition = FieldDefinition;

    /**
    * @class UnmappedField
    */
    var UnmappedField = (function () {
        function UnmappedField(fromField, property, toField) {
            if (typeof toField === "undefined") { toField = null; }
            this.fromField = fromField;
            this.property = property;
            this.toField = toField;
        }
        return UnmappedField;
    })();
    Dozer.UnmappedField = UnmappedField;
})(Dozer || (Dozer = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    /**
    * The default dashboard definition if no saved dashboards are available
    *
    * @property defaultDashboards
    * @for Dashboard
    * @type {any}
    */
    var defaultDashboards = [
        {
            "title": "Monitor",
            "group": "Personal",
            "widgets": [
                {
                    "id": "w1",
                    "title": "",
                    "row": 1,
                    "col": 1,
                    "size_x": 3,
                    "size_y": 4,
                    "path": "jmx/attributes",
                    "include": "app/jmx/html/attributes.html",
                    "search": {
                        "nid": "root-java.lang-OperatingSystem"
                    },
                    "hash": ""
                },
                {
                    "id": "w3",
                    "title": "Java Heap Memory",
                    "row": 1,
                    "col": 6,
                    "size_x": 2,
                    "size_y": 2,
                    "path": "jmx/widget/donut",
                    "include": "app/jmx/html/donutChart.html",
                    "search": {},
                    "hash": "",
                    "routeParams": "{\"type\":\"donut\",\"title\":\"Java Heap Memory\",\"mbean\":\"java.lang:type=Memory\",\"attribute\":\"HeapMemoryUsage\",\"total\":\"Max\",\"terms\":\"Used\",\"remaining\":\"Free\"}"
                },
                {
                    "id": "w4",
                    "title": "Java Non Heap Memory",
                    "row": 1,
                    "col": 8,
                    "size_x": 2,
                    "size_y": 2,
                    "path": "jmx/widget/donut",
                    "include": "app/jmx/html/donutChart.html",
                    "search": {},
                    "hash": "",
                    "routeParams": "{\"type\":\"donut\",\"title\":\"Java Non Heap Memory\",\"mbean\":\"java.lang:type=Memory\",\"attribute\":\"NonHeapMemoryUsage\",\"total\":\"Max\",\"terms\":\"Used\",\"remaining\":\"Free\"}"
                },
                {
                    "id": "w5",
                    "title": "",
                    "row": 3,
                    "col": 4,
                    "size_x": 6,
                    "size_y": 2,
                    "path": "jmx/charts",
                    "include": "app/jmx/html/charts.html",
                    "search": {
                        "size": "%7B%22size_x%22%3A2%2C%22size_y%22%3A2%7D",
                        "title": "Java%20Non%20Heap%20Memory",
                        "routeParams": "%7B%22type%22%3A%22donut%22%2C%22title%22%3A%22Java%20Non%20Heap%20Memory%22%2C%22mbean%22%3A%22java.lang%3Atype",
                        "nid": "root-java.lang-Threading"
                    },
                    "hash": ""
                },
                {
                    "id": "w6",
                    "title": "System CPU Load",
                    "row": 1,
                    "col": 4,
                    "size_x": 2,
                    "size_y": 2,
                    "path": "jmx/widget/area",
                    "include": "app/jmx/html/areaChart.html",
                    "search": {},
                    "hash": "",
                    "routeParams": "{\"type\":\"area\",\"title\":\"System CPU Load\",\"mbean\":\"java.lang:type=OperatingSystem\",\"attribute\":\"SystemCpuLoad\"}"
                }
            ],
            "id": "4e9d116173ca41767e"
        }
    ];

    

    /**
    * Registry of dashboard repositories that delegates to the current effective
    * dashboard repository
    *
    * @class DefaultDashboardRepository
    * @uses DashboardRepository
    */
    var DefaultDashboardRepository = (function () {
        function DefaultDashboardRepository(workspace, jolokia, localStorage) {
            this.workspace = workspace;
            this.jolokia = jolokia;
            this.localStorage = localStorage;
            this.repository = null;
        }
        DefaultDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            this.getRepository().putDashboards(array, commitMessage, fn);
        };

        DefaultDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            this.getRepository().deleteDashboards(array, fn);
        };

        /**
        * Loads the dashboards then asynchronously calls the function with the data
        * @method getDashboards
        * @param {Function} fn
        */
        DefaultDashboardRepository.prototype.getDashboards = function (fn) {
            this.getRepository().getDashboards(function (values) {
                fn(values);
            });
        };

        /**
        * Loads the given dashboard and invokes the given function with the result
        * @method getDashboard
        * @param {String} id
        * @param {Function} onLoad
        */
        DefaultDashboardRepository.prototype.getDashboard = function (id, onLoad) {
            this.getRepository().getDashboard(id, onLoad);
        };

        DefaultDashboardRepository.prototype.createDashboard = function (options) {
            return this.getRepository().createDashboard(options);
        };

        DefaultDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            return this.getRepository().cloneDashboard(dashboard);
        };

        DefaultDashboardRepository.prototype.getType = function () {
            return this.getRepository().getType();
        };

        DefaultDashboardRepository.prototype.isValid = function () {
            return this.getRepository().isValid();
        };

        /**
        * Looks up the MBean in the JMX tree
        * @method getRepository
        * @return {DashboardRepository}
        */
        DefaultDashboardRepository.prototype.getRepository = function () {
            if (this.repository && this.repository.isValid()) {
                return this.repository;
            }
            if (Fabric.hasFabric(this.workspace)) {
                this.repository = new Dashboard.FabricDashboardRepository(this.workspace, this.jolokia, this.localStorage);
                return this.repository;
            }
            var git = Git.createGitRepository(this.workspace, this.jolokia, this.localStorage);
            if (git) {
                this.repository = new GitDashboardRepository(this.workspace, git);
                return this.repository;
            }
            this.repository = new LocalDashboardRepository(this.workspace);
            return this.repository;
        };
        return DefaultDashboardRepository;
    })();
    Dashboard.DefaultDashboardRepository = DefaultDashboardRepository;

    /**
    * @class LocalDashboardRepository
    * @uses DashboardRepository
    */
    var LocalDashboardRepository = (function () {
        function LocalDashboardRepository(workspace) {
            this.workspace = workspace;
            this.localStorage = null;
            this.localStorage = workspace.localStorage;

            if ('userDashboards' in this.localStorage) {
                // log.info("Found previously saved dashboards");
            } else {
                this.storeDashboards(defaultDashboards);
            }
        }
        LocalDashboardRepository.prototype.loadDashboards = function () {
            var answer = angular.fromJson(localStorage['userDashboards']);
            Dashboard.log.debug("returning dashboards: ", answer);
            return answer;
        };

        LocalDashboardRepository.prototype.storeDashboards = function (dashboards) {
            Dashboard.log.debug("storing dashboards: ", dashboards);
            localStorage['userDashboards'] = angular.toJson(dashboards);
            return this.loadDashboards();
        };

        LocalDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            var dashboards = this.loadDashboards();

            array.forEach(function (dash) {
                var existing = dashboards.findIndex(function (d) {
                    return d.id === dash.id;
                });
                if (existing >= 0) {
                    dashboards[existing] = dash;
                } else {
                    dashboards.push(dash);
                }
            });
            fn(this.storeDashboards(dashboards));
        };

        LocalDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            var dashboards = this.loadDashboards();
            angular.forEach(array, function (item) {
                dashboards.remove(function (i) {
                    return i.id === item.id;
                });
            });
            fn(this.storeDashboards(dashboards));
        };

        LocalDashboardRepository.prototype.getDashboards = function (fn) {
            fn(this.loadDashboards());
        };

        LocalDashboardRepository.prototype.getDashboard = function (id, fn) {
            var dashboards = this.loadDashboards();
            var dashboard = dashboards.find(function (dashboard) {
                return dashboard.id === id;
            });
            fn(dashboard);
        };

        LocalDashboardRepository.prototype.createDashboard = function (options) {
            var answer = {
                title: "New Dashboard",
                group: "Personal",
                widgets: []
            };
            answer = angular.extend(answer, options);
            answer['id'] = Core.getUUID();
            return answer;
        };

        LocalDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            var newDashboard = Object.clone(dashboard);
            newDashboard['id'] = Core.getUUID();
            newDashboard['title'] = "Copy of " + dashboard.title;
            return newDashboard;
        };

        LocalDashboardRepository.prototype.getType = function () {
            return 'container';
        };

        LocalDashboardRepository.prototype.isValid = function () {
            return !Fabric.hasFabric(this.workspace) && !Git.hasGit(this.workspace);
        };
        return LocalDashboardRepository;
    })();
    Dashboard.LocalDashboardRepository = LocalDashboardRepository;

    /**
    * @class GitDashboardRepository
    * @uses DashboardRepository
    */
    var GitDashboardRepository = (function () {
        function GitDashboardRepository(workspace, git) {
            this.workspace = workspace;
            this.git = git;
            this.branch = null;
        }
        GitDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            var _this = this;
            var toPut = array.length;
            var maybeCallback = function () {
                toPut = toPut - 1;
                if (toPut === 0) {
                    _this.getDashboards(fn);
                }
            };

            angular.forEach(array, function (dash) {
                var path = _this.getDashboardPath(dash);
                var contents = JSON.stringify(dash, null, "  ");
                _this.git.write(_this.branch, path, commitMessage, contents, function () {
                    maybeCallback();
                });
            });
        };

        GitDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            var _this = this;
            var toDelete = array.length;
            var maybeCallback = function () {
                toDelete = toDelete - 1;
                if (toDelete === 0) {
                    _this.getDashboards(fn);
                }
            };
            angular.forEach(array, function (dash) {
                var path = _this.getDashboardPath(dash);
                var commitMessage = "Removing dashboard " + path;
                _this.git.remove(_this.branch, path, commitMessage, function () {
                    maybeCallback();
                });
            });
        };

        GitDashboardRepository.prototype.createDashboard = function (options) {
            var answer = {
                title: "New Dashboard",
                group: "Personal",
                widgets: []
            };
            answer = angular.extend(answer, options);
            answer['id'] = Core.getUUID();
            return answer;
        };

        GitDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            var newDashboard = Object.clone(dashboard);
            newDashboard['id'] = Core.getUUID();
            newDashboard['title'] = "Copy of " + dashboard.title;
            return newDashboard;
        };

        GitDashboardRepository.prototype.getType = function () {
            return 'git';
        };

        GitDashboardRepository.prototype.isValid = function () {
            return Git.hasGit(this.workspace);
        };

        GitDashboardRepository.prototype.getDashboardPath = function (dash) {
            // TODO assume a user dashboard for now
            // ideally we'd look up the teams path based on the group
            var id = dash.id || Core.getUUID();
            var path = this.getUserDashboardPath(id);
            return path;
        };

        GitDashboardRepository.prototype.getDashboards = function (fn) {
            var _this = this;
            // TODO lets look in each team directory as well and combine the results...
            var path = this.getUserDashboardDirectory();
            var dashboards = [];
            this.git.read(this.branch, path, function (details) {
                var files = details.children;

                var toRead = files.length;

                var maybeCallback = function () {
                    toRead = toRead - 1;
                    if (toRead === 0) {
                        fn(dashboards);
                    }
                };

                // we now have all the files we need; lets read all their contents
                angular.forEach(files, function (file, idx) {
                    var path = file.path;
                    if (!file.directory && path.endsWith(".json")) {
                        _this.git.read(_this.branch, path, function (details) {
                            // lets parse the contents
                            var content = details.text;
                            if (content) {
                                try  {
                                    var json = JSON.parse(content);
                                    json.uri = path;
                                    dashboards.push(json);
                                } catch (e) {
                                    console.log("Failed to parse: " + content + " due to: " + e);
                                }
                            }
                            Dashboard.log.debug("git - read ", idx, " files, total: ", files.length);
                            maybeCallback();
                        });
                    }
                });
            });
        };

        GitDashboardRepository.prototype.getDashboard = function (id, fn) {
            var path = this.getUserDashboardPath(id);
            this.git.read(this.branch, path, function (details) {
                var dashboard = null;
                var content = details.text;
                if (content) {
                    try  {
                        dashboard = JSON.parse(content);
                    } catch (e) {
                        console.log("Failed to parse: " + content + " due to: " + e);
                    }
                }
                fn(dashboard);
            });
        };

        GitDashboardRepository.prototype.getUserDashboardDirectory = function () {
            // TODO until we fix #96 lets default to a common user name so
            // all the dashboards are shared for all users for now
            //return "/dashboards/user/" + this.git.getUserName();
            return "/dashboards/team/all";
        };

        GitDashboardRepository.prototype.getUserDashboardPath = function (id) {
            return this.getUserDashboardDirectory() + "/" + id + ".json";
        };
        return GitDashboardRepository;
    })();
    Dashboard.GitDashboardRepository = GitDashboardRepository;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function NavBarController($scope, $routeParams, $rootScope, workspace, dashboardRepository) {
        $scope.hash = workspace.hash();
        $scope._dashboards = [];

        $scope.activeDashboard = $routeParams['dashboardId'];

        $rootScope.$on('loadDashboards', loadDashboards);

        $rootScope.$on('dashboardsUpdated', dashboardLoaded);

        $scope.dashboards = function () {
            return $scope._dashboards;
        };

        $scope.isActive = function (dash) {
            return workspace.isLinkActive("#/dashboard/id/" + dash.id);
        };

        $scope.isEditing = function () {
            return workspace.isLinkActive("#/dashboard/edit");
        };

        $scope.onTabRenamed = function (dash) {
            dashboardRepository.putDashboards([dash], "Renamed dashboard", function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        function dashboardLoaded(event, dashboards) {
            Dashboard.log.debug("navbar dashboardLoaded: ", dashboards);
            $scope._dashboards = dashboards;
            if (event === null) {
                $rootScope.$broadcast('dashboardsUpdated', dashboards);
                Core.$apply($scope);
            }
        }

        function loadDashboards(event) {
            dashboardRepository.getDashboards(function (dashboards) {
                // prevent the broadcast from happening...
                dashboardLoaded(event, dashboards);
                Core.$apply($scope);
            });
        }
    }
    Dashboard.NavBarController = NavBarController;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function ShareController($scope, $location, $routeParams, workspace, dashboardRepository) {
        var id = $routeParams["dashboardId"];
        dashboardRepository.getDashboard(id, onDashboardLoad);

        var options = {
            mode: {
                name: "javascript"
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);

        function onDashboardLoad(dashboard) {
            $scope.dashboard = Dashboard.cleanDashboardData(dashboard);

            $scope.json = {
                "description": "hawtio dashboards",
                "public": true,
                "files": {
                    "dashboards.json": {
                        "content": JSON.stringify($scope.dashboard, null, "  ")
                    }
                }
            };

            $scope.source = JSON.stringify($scope.dashboard, null, "  ");
            Core.$applyNowOrLater($scope);
        }
    }
    Dashboard.ShareController = ShareController;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    /**
    * Implements the ng.ILocationService interface and is used by the dashboard to supply
    * controllers with a saved URL location
    *
    * @class RectangleLocation
    */
    var RectangleLocation = (function () {
        function RectangleLocation(delegate, path, search, hash) {
            this.delegate = delegate;
            this._path = path;
            this._search = search;
            this._hash = hash;
        }
        RectangleLocation.prototype.absUrl = function () {
            return this.protocol() + this.host() + ":" + this.port() + this.path() + this.search();
        };

        RectangleLocation.prototype.hash = function (newHash) {
            if (typeof newHash === "undefined") { newHash = null; }
            if (newHash) {
                return this.delegate.hash(newHash).search('tab', null);
                //this._hash = newHash;
            }
            return this._hash;
        };

        RectangleLocation.prototype.host = function () {
            return this.delegate.host();
        };

        RectangleLocation.prototype.path = function (newPath) {
            if (typeof newPath === "undefined") { newPath = null; }
            if (newPath) {
                return this.delegate.path(newPath).search('tab', null);
            }
            return this._path;
        };

        RectangleLocation.prototype.port = function () {
            return this.delegate.port();
        };

        RectangleLocation.prototype.protocol = function () {
            return this.delegate.port();
        };

        RectangleLocation.prototype.replace = function () {
            // TODO
            return this;
        };

        RectangleLocation.prototype.search = function (parametersMap) {
            if (typeof parametersMap === "undefined") { parametersMap = null; }
            if (parametersMap) {
                return this.delegate.search(parametersMap);
            }
            return this._search;
        };

        RectangleLocation.prototype.url = function (newValue) {
            if (typeof newValue === "undefined") { newValue = null; }
            if (newValue) {
                return this.delegate.url(newValue).search('tab', null);
            }
            return this.absUrl();
        };
        return RectangleLocation;
    })();
    Dashboard.RectangleLocation = RectangleLocation;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
* @main Dashboard
*/
var Dashboard;
(function (Dashboard) {
    Dashboard.templatePath = 'app/dashboard/html/';
    Dashboard.pluginName = 'dashboard';

    angular.module(Dashboard.pluginName, ['bootstrap', 'ngResource', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/dashboard/add', { templateUrl: Dashboard.templatePath + 'addToDashboard.html' }).when('/dashboard/edit', { templateUrl: Dashboard.templatePath + 'editDashboards.html' }).when('/dashboard/idx/:dashboardIndex', { templateUrl: Dashboard.templatePath + 'dashboard.html' }).when('/dashboard/id/:dashboardId', { templateUrl: Dashboard.templatePath + 'dashboard.html' }).when('/dashboard/id/:dashboardId/share', { templateUrl: Dashboard.templatePath + 'share.html' }).when('/dashboard/import', { templateUrl: Dashboard.templatePath + 'import.html' });
    }).value('ui.config', {
        // The ui-jq directive namespace
        jq: {
            gridster: {
                widget_margins: [10, 10],
                widget_base_dimensions: [140, 140]
            }
        }
    }).factory('dashboardRepository', function (workspace, jolokia, localStorage) {
        return new Dashboard.DefaultDashboardRepository(workspace, jolokia, localStorage);
    }).directive('hawtioDashboard', function () {
        return new Dashboard.GridsterDirective();
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['dashboard'] = 'app/dashboard/html/layoutDashboard.html';
        helpRegistry.addUserDoc('dashboard', 'app/dashboard/doc/help.md');

        workspace.topLevelTabs.push({
            id: "dashboard",
            content: "Dashboard",
            title: "View and edit your own custom dashboards",
            isValid: function (workspace) {
                return workspace.hasMBeans();
            },
            href: function () {
                return "#/dashboard/idx/0?tab=dashboard";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("dashboard");
            }
        });
    });

    hawtioPluginLoader.addModule(Dashboard.pluginName);
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function EditDashboardsController($scope, $routeParams, $route, $location, $rootScope, dashboardRepository, jolokia, workspace) {
        $scope.hash = workspace.hash();
        $scope.selectedItems = [];
        $scope.repository = dashboardRepository;
        $scope.duplicateDashboards = new Core.Dialog();
        $scope.selectedProfilesDialog = [];
        $scope._dashboards = [];

        $rootScope.$on('dashboardsUpdated', dashboardLoaded);

        $scope.hasUrl = function () {
            return ($scope.url) ? true : false;
        };

        $scope.hasSelection = function () {
            return $scope.selectedItems.length !== 0;
        };

        $scope.gridOptions = {
            selectedItems: $scope.selectedItems,
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                filterText: ''
            },
            data: '_dashboards',
            selectWithCheckboxOnly: true,
            showSelectionCheckbox: true,
            columnDefs: [
                {
                    field: 'title',
                    displayName: 'Dashboard',
                    cellTemplate: '<div class="ngCellText"><a ng-href="#/dashboard/id/{{row.getProperty(' + "'id'" + ')}}{{hash}}"><editable-property class="inline-block" on-save="onDashRenamed(row.entity)" property="title" ng-model="row.entity"></editable-property></a></div>'
                },
                {
                    field: 'group',
                    displayName: 'Group'
                }
            ]
        };

        $scope.onDashRenamed = function (dash) {
            dashboardRepository.putDashboards([dash], "Renamed dashboard", function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        // helpers so we can enable/disable parts of the UI depending on how
        // dashboard data is stored
        $scope.usingGit = function () {
            return dashboardRepository.getType() === 'git';
        };

        $scope.usingFabric = function () {
            return dashboardRepository.getType() === 'fabric';
        };

        $scope.usingLocal = function () {
            return dashboardRepository.getType() === 'container';
        };

        if ($scope.usingFabric()) {
            $scope.container = Fabric.getCurrentContainer(jolokia, ['versionId', 'profileIds']);

            $scope.gridOptions.columnDefs.add([
                {
                    field: 'versionId',
                    displayName: 'Version'
                }, {
                    field: 'profileId',
                    displayName: 'Profile'
                }, {
                    field: 'fileName',
                    displayName: 'File Name'
                }]);
        }

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateData, 100);
        });

        $scope.goBack = function () {
            var href = Core.trimLeading($scope.url, "#");
            if (href) {
                $location.url(href);
            }
        };

        $scope.duplicateToProfiles = function () {
            if ($scope.hasSelection()) {
                $scope.duplicateDashboards.open();
            }
        };

        $scope.doDuplicateToProfiles = function () {
            $scope.duplicateDashboards.close();

            var newDashboards = [];

            $scope.selectedItems.forEach(function (dashboard) {
                $scope.selectedProfilesDialog.forEach(function (profile) {
                    var newDash = dashboardRepository.cloneDashboard(dashboard);
                    newDash['profileId'] = profile.id;
                    newDash['title'] = dashboard.title;
                    newDashboards.push(newDash);
                });
            });

            var commitMessage = "Duplicating " + $scope.selectedItems.length + " dashboards to " + $scope.selectedProfilesDialog.length + " profiles";

            dashboardRepository.putDashboards(newDashboards, commitMessage, function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        $scope.addViewToDashboard = function () {
            var nextHref = null;
            angular.forEach($scope.selectedItems, function (selectedItem) {
                // TODO this could be a helper function
                var text = $scope.url;
                var query = null;
                if (text) {
                    var idx = text.indexOf('?');
                    if (idx && idx > 0) {
                        query = text.substring(idx + 1);
                        text = text.substring(0, idx);
                    }
                    text = Core.trimLeading(text, "#");
                }
                var search = {};
                if (query) {
                    var expressions = query.split("&");
                    angular.forEach(expressions, function (expression) {
                        if (expression) {
                            var names = expression.split("=");
                            var key = names[0];
                            var value = names.length > 1 ? names[1] : null;
                            if (value) {
                                value = encodeURIComponent(value);
                            }
                            var old = search[key];
                            if (old) {
                                if (!angular.isArray(old)) {
                                    old = [old];
                                    search[key] = old;
                                }
                                old.push(value);
                            } else {
                                search[key] = value;
                            }
                        }
                    });
                }

                //console.log("path is: " + text + " the search is " + JSON.stringify(search));
                if ($route && $route.routes) {
                    var value = $route.routes[text];
                    if (value) {
                        var templateUrl = value["templateUrl"];
                        if (templateUrl) {
                            if (!selectedItem.widgets) {
                                selectedItem.widgets = [];
                            }
                            var nextNumber = selectedItem.widgets.length + 1;
                            var widget = {
                                id: "w" + nextNumber, title: "",
                                row: 1,
                                col: 1,
                                size_x: 1,
                                size_y: 1,
                                path: Core.trimLeading(text, "/"),
                                include: templateUrl,
                                search: search,
                                hash: ""
                            };

                            if ($scope.widgetTitle) {
                                widget.title = $scope.widgetTitle;
                            }

                            // figure out the width of the dash
                            var gridWidth = 0;

                            selectedItem.widgets.forEach(function (w) {
                                var rightSide = w.col + w.size_x;
                                if (rightSide > gridWidth) {
                                    gridWidth = rightSide;
                                }
                            });

                            if ($scope.preferredSize) {
                                widget.size_x = parseInt($scope.preferredSize['size_x']);
                                widget.size_y = parseInt($scope.preferredSize['size_y']);
                            }

                            var found = false;

                            var left = function (w) {
                                return w.col;
                            };

                            var right = function (w) {
                                return w.col + w.size_x - 1;
                            };

                            var top = function (w) {
                                return w.row;
                            };

                            var bottom = function (w) {
                                return w.row + w.size_y - 1;
                            };

                            var collision = function (w1, w2) {
                                return !(left(w2) > right(w1) || right(w2) < left(w1) || top(w2) > bottom(w1) || bottom(w2) < top(w1));
                            };

                            if (selectedItem.widgets.isEmpty()) {
                                found = true;
                            }

                            while (!found) {
                                widget.col = 1;
                                for (; (widget.col + widget.size_x) <= gridWidth; widget.col++) {
                                    if (!selectedItem.widgets.any(function (w) {
                                        var c = collision(w, widget);
                                        return c;
                                    })) {
                                        found = true;
                                        break;
                                    }
                                }
                                if (!found) {
                                    widget.row = widget.row + 1;
                                }

                                // just in case, keep the script from running away...
                                if (widget.row > 50) {
                                    found = true;
                                }
                            }

                            if ($scope.routeParams) {
                                widget['routeParams'] = $scope.routeParams;
                            }
                            selectedItem.widgets.push(widget);

                            if (!nextHref && selectedItem.id) {
                                nextHref = "/dashboard/id/" + selectedItem.id;
                            }
                        }
                    } else {
                        // TODO we need to be able to match URI templates...
                    }
                }
            });

            // now lets update the actual dashboard config
            var commitMessage = "Add widget";
            dashboardRepository.putDashboards($scope.selectedItems, commitMessage, function (dashboards) {
                if (nextHref) {
                    // remove any dodgy query
                    delete $location.search()["href"];
                    $location.path(nextHref);
                    Core.$apply($scope);
                }
            });
        };

        $scope.create = function () {
            var counter = dashboards().length + 1;
            var title = "Untitled" + counter;
            var newDash = dashboardRepository.createDashboard({ title: title });

            dashboardRepository.putDashboards([newDash], "Created new dashboard: " + title, function (dashboards) {
                $scope.selectedItems.push(newDash);
                dashboardLoaded(null, dashboards);
            });
        };

        $scope.duplicate = function () {
            var newDashboards = [];
            var commitMessage = "Duplicated dashboard(s) ";
            angular.forEach($scope.selectedItems, function (item, idx) {
                // lets unselect this item
                var commitMessage = "Duplicated dashboard " + item.title;
                var newDash = dashboardRepository.cloneDashboard(item);
                newDashboards.push(newDash);
            });

            // let's just be safe and ensure there's no selections
            $scope.selectedItems = [];

            commitMessage = commitMessage + newDashboards.map(function (d) {
                return d.title;
            }).join(',');
            dashboardRepository.putDashboards(newDashboards, commitMessage, function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        $scope.delete = function () {
            if ($scope.hasSelection()) {
                dashboardRepository.deleteDashboards($scope.selectedItems, function (dashboards) {
                    $scope.selectedItems = [];
                    dashboardLoaded(null, dashboards);
                });
            }
        };

        $scope.gist = function () {
            if ($scope.selectedItems.length > 0) {
                var id = $scope.selectedItems[0].id;
                $location.path("/dashboard/id/" + id + "/share");
            }
        };

        function updateData() {
            var url = $routeParams["href"];
            if (url) {
                $scope.url = decodeURIComponent(url);
            }

            var routeParams = $routeParams["routeParams"];
            if (routeParams) {
                $scope.routeParams = decodeURIComponent(routeParams);
            }
            var size = $routeParams["size"];
            if (size) {
                size = decodeURIComponent(size);
                $scope.preferredSize = angular.fromJson(size);
            }
            var title = $routeParams["title"];
            if (title) {
                title = decodeURIComponent(title);
                $scope.widgetTitle = title;
            }

            dashboardRepository.getDashboards(function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        }

        function dashboardLoaded(event, dashboards) {
            $scope._dashboards = dashboards;
            if (event === null) {
                $scope.$emit('dashboardsUpdated', dashboards);
            }
            Core.$apply($scope);
        }

        function dashboards() {
            return $scope._dashboards;
        }

        updateData();
        /*
        // TODO for case where we navigate to the add view
        // for some reason the route update event isn't enough...
        // and we need to do this async to avoid the size calculation being wrong
        // bit of a hack - would love to remove! :)
        setTimeout(updateData, 100);
        */
    }
    Dashboard.EditDashboardsController = EditDashboardsController;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    var GridsterDirective = (function () {
        function GridsterDirective() {
            this.restrict = 'A';
            this.replace = true;
            this.controller = function ($scope, $element, $attrs, $location, $routeParams, $injector, $route, $templateCache, workspace, dashboardRepository, $compile) {
                $scope.route = $route;
                $scope.injector = $injector;

                var gridSize = 150;
                var gridMargin = 6;
                var gridHeight;

                $scope.gridX = gridSize;
                $scope.gridY = gridSize;

                $scope.widgetMap = {};

                $scope.$on('$destroy', function () {
                    angular.forEach($scope.widgetMap, function (value, key) {
                        if ('scope' in value) {
                            var scope = value['scope'];
                            scope.$destroy();
                        }
                    });
                });

                updateWidgets();

                $scope.removeWidget = function (widget) {
                    var gridster = getGridster();
                    var widgetElem = null;

                    // lets destroy the widgets's scope
                    var widgetData = $scope.widgetMap[widget.id];
                    if (widgetData) {
                        delete $scope.widgetMap[widget.id];
                        var scope = widgetData.scope;
                        widgetElem = widgetData.widget;
                        if (scope) {
                            scope.$destroy();
                        }
                    }
                    if (!widgetElem) {
                        // lets get the li parent element of the template
                        widgetElem = $("div").find("[data-widgetId='" + widget.id + "']").parent();
                    }
                    if (gridster && widgetElem) {
                        gridster.remove_widget(widgetElem);
                    }

                    // no need to remove it...
                    //widgetElem.remove();
                    // lets trash the JSON metadata
                    if ($scope.dashboard) {
                        var widgets = $scope.dashboard.widgets;
                        if (widgets) {
                            widgets.remove(widget);
                        }
                    }

                    updateDashboardRepository("Removed widget " + widget.title);
                };

                function changeWidgetSize(widget, sizefunc, savefunc) {
                    var gridster = getGridster();
                    var entry = $scope.widgetMap[widget.id];
                    var w = entry.widget;
                    var scope = entry.scope;
                    sizefunc(entry);
                    gridster.resize_widget(w, entry.size_x, entry.size_y);
                    gridster.set_dom_grid_height();

                    setTimeout(function () {
                        var template = $templateCache.get("widgetTemplate");
                        var div = $('<div></div>');
                        div.html(template);
                        w.html($compile(div.contents())(scope));

                        makeResizable();
                        Core.$apply($scope);

                        setTimeout(function () {
                            savefunc(widget);
                        }, 50);
                    }, 30);
                }

                $scope.onWidgetRenamed = function (widget) {
                    updateDashboardRepository("Renamed widget to " + widget.title);
                };

                function updateWidgets() {
                    $scope.id = $routeParams["dashboardId"];
                    $scope.idx = $routeParams["dashboardIndex"];
                    if ($scope.id) {
                        $scope.$emit('loadDashboards');
                        dashboardRepository.getDashboard($scope.id, onDashboardLoad);
                    } else {
                        dashboardRepository.getDashboards(function (dashboards) {
                            $scope.$emit('dashboardsUpdated', dashboards);

                            var idx = $scope.idx ? parseInt($scope.idx) : 0;
                            var id = null;
                            if (dashboards.length > 0) {
                                var dashboard = dashboards.length > idx ? dashboards[idx] : dashboard[0];
                                id = dashboard.id;
                            }
                            if (id) {
                                $location.path("/dashboard/id/" + id);
                            } else {
                                $location.path("/dashboard/edit?tab=dashboard");
                            }
                            Core.$apply($scope);
                        });
                    }
                }

                function onDashboardLoad(dashboard) {
                    $scope.dashboard = dashboard;
                    var widgets = ((dashboard) ? dashboard.widgets : null) || [];

                    var minHeight = 10;
                    var minWidth = 6;

                    angular.forEach(widgets, function (widget) {
                        if (angular.isDefined(widget.row) && minHeight < widget.row) {
                            minHeight = widget.row + 1;
                        }
                        if (angular.isDefined(widget.size_x && angular.isDefined(widget.col))) {
                            var rightEdge = widget.col + widget.size_x;
                            if (rightEdge > minWidth) {
                                minWidth = rightEdge + 1;
                            }
                        }
                    });

                    var gridster = $element.gridster({
                        widget_margins: [gridMargin, gridMargin],
                        widget_base_dimensions: [$scope.gridX, $scope.gridY],
                        extra_rows: minHeight,
                        extra_cols: minWidth,
                        max_size_x: minWidth,
                        max_size_y: minHeight,
                        draggable: {
                            stop: function (event, ui) {
                                if (serializeDashboard()) {
                                    updateDashboardRepository("Changing dashboard layout");
                                }
                            }
                        }
                    }).data('gridster');

                    var template = $templateCache.get("widgetTemplate");
                    angular.forEach(widgets, function (widget) {
                        var childScope = $scope.$new(false);
                        childScope.widget = widget;
                        var path = widget.path;
                        var search = null;
                        if (widget.search) {
                            search = Dashboard.decodeURIComponentProperties(widget.search);
                        }
                        var hash = widget.hash;
                        var location = new Dashboard.RectangleLocation($location, path, search, hash);
                        var routeParams = null;
                        if (widget.routeParams) {
                            routeParams = angular.fromJson(widget.routeParams);
                        }

                        var childWorkspace = workspace.createChildWorkspace(location);

                        //var childWorkspace = workspace;
                        childWorkspace.$location = location;

                        // now we need to update the selection from the location search()
                        if (search) {
                            var key = location.search()['nid'];
                            if (key && workspace.tree) {
                                // lets find the node for this key...
                                childWorkspace.selection = workspace.keyToNodeMap[key];
                                if (!childWorkspace.selection) {
                                    var decodedKey = decodeURIComponent(key);
                                    childWorkspace.selection = workspace.keyToNodeMap[decodedKey];
                                }
                            }
                        }

                        var $$scopeInjections = {
                            workspace: childWorkspace,
                            location: location,
                            $location: location,
                            $routeParams: routeParams
                        };
                        childScope.$$scopeInjections = $$scopeInjections;
                        childScope.inDashboard = true;

                        if (!widget.size_x || widget.size_x < 1) {
                            widget.size_x = 1;
                        }
                        if (!widget.size_y || widget.size_y < 1) {
                            widget.size_y = 1;
                        }
                        var div = $('<div></div>');
                        div.html(template);

                        var outerDiv = $('<li class="grid-block" style="display: list-item; position: absolute"></li>');
                        outerDiv.html($compile(div.contents())(childScope));
                        var w = gridster.add_widget(outerDiv, widget.size_x, widget.size_y, widget.col, widget.row);

                        $scope.widgetMap[widget.id] = {
                            widget: w,
                            scope: childScope
                        };
                    });

                    makeResizable();
                    getGridster().enable();

                    Core.$apply($scope);
                }

                function serializeDashboard() {
                    var gridster = getGridster();
                    if (gridster) {
                        var data = gridster.serialize();

                        //console.log("got data: " + JSON.stringify(data));
                        var widgets = $scope.dashboard.widgets || [];

                        // console.log("Widgets: ", widgets);
                        // lets assume the data is in the order of the widgets...
                        angular.forEach(widgets, function (widget, idx) {
                            var value = data[idx];
                            if (value && widget) {
                                // lets copy the values across
                                angular.forEach(value, function (attr, key) {
                                    return widget[key] = attr;
                                });
                            }
                        });
                        return true;
                    }
                    return false;
                }

                function makeResizable() {
                    var blocks = $('.grid-block');
                    blocks.resizable('destroy');

                    blocks.resizable({
                        grid: [gridSize + (gridMargin * 2), gridSize + (gridMargin * 2)],
                        animate: false,
                        minWidth: gridSize,
                        minHeight: gridSize,
                        autoHide: false,
                        start: function (event, ui) {
                            gridHeight = getGridster().$el.height();
                        },
                        resize: function (event, ui) {
                            //set new grid height along the dragging period
                            var g = getGridster();
                            var delta = gridSize + gridMargin * 2;
                            if (event.offsetY > g.$el.height()) {
                                var extra = Math.floor((event.offsetY - gridHeight) / delta + 1);
                                var newHeight = gridHeight + extra * delta;
                                g.$el.css('height', newHeight);
                            }
                        },
                        stop: function (event, ui) {
                            var resized = $(this);
                            setTimeout(function () {
                                resizeBlock(resized);
                            }, 300);
                        }
                    });

                    $('.ui-resizable-handle').hover(function () {
                        getGridster().disable();
                    }, function () {
                        getGridster().enable();
                    });
                }

                function resizeBlock(elmObj) {
                    //var elmObj = $(elmObj);
                    var area = elmObj.find('.widget-area');
                    var w = elmObj.width() - gridSize;
                    var h = elmObj.height() - gridSize;

                    for (var grid_w = 1; w > 0; w -= (gridSize + (gridMargin * 2))) {
                        grid_w++;
                    }

                    for (var grid_h = 1; h > 0; h -= (gridSize + (gridMargin * 2))) {
                        grid_h++;
                    }

                    var widget = {
                        id: area.attr('data-widgetId')
                    };

                    changeWidgetSize(widget, function (widget) {
                        widget.size_x = grid_w;
                        widget.size_y = grid_h;
                    }, function (widget) {
                        if (serializeDashboard()) {
                            updateDashboardRepository("Changed size of widget: " + widget.id);
                        }
                    });
                }

                function updateDashboardRepository(message) {
                    if ($scope.dashboard) {
                        var commitMessage = message;
                        if ($scope.dashboard && $scope.dashboard.title) {
                            commitMessage += " on dashboard " + $scope.dashboard.title;
                        }
                        dashboardRepository.putDashboards([$scope.dashboard], commitMessage, Dashboard.onOperationComplete);
                    }
                }

                function getGridster() {
                    return $element.gridster().data('gridster');
                }
            };
        }
        return GridsterDirective;
    })();
    Dashboard.GridsterDirective = GridsterDirective;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    var FabricDashboardRepository = (function () {
        function FabricDashboardRepository(workspace, jolokia, localStorage) {
            this.workspace = workspace;
            this.jolokia = jolokia;
            this.localStorage = localStorage;
            this.details = this.getBranchAndProfiles();
        }
        FabricDashboardRepository.prototype.getBranchAndProfiles = function () {
            if (Fabric.fabricCreated(this.workspace)) {
                var container = Fabric.getCurrentContainer(this.jolokia, ['id', 'versionId', 'profiles']);
                var profiles = [];
                if (container.profiles) {
                    profiles = container.profiles.unique();
                    profiles = Fabric.filterProfiles(this.jolokia, container.versionId, profiles);
                }

                return {
                    branch: container.versionId,
                    profiles: profiles
                };
            } else {
                return {
                    branch: "1.0",
                    profiles: []
                };
            }
        };

        FabricDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            var _this = this;
            var jolokia = this.jolokia;
            var details = this.details;

            var toPut = array.length;

            var maybeCallback = function () {
                toPut = toPut - 1;
                if (toPut === 0) {
                    _this.getDashboards(fn);
                }
            };

            array.forEach(function (dashboard) {
                // console.log("Saving dash: ", dashboard);
                var data = angular.toJson(dashboard, true);
                var profileId = dashboard.profileId;
                if (!profileId) {
                    // TODO maybe not just pick the first one :-)
                    profileId = details.profiles.first();
                }
                var fileName = dashboard.fileName;
                if (!fileName) {
                    fileName = Core.getUUID() + ".dashboard";
                }
                Fabric.saveConfigFile(jolokia, details.branch, profileId, fileName, data.encodeBase64(), function () {
                    maybeCallback();
                    //notification('success', "Saved dashboard " + dashboard.title);
                }, function (response) {
                    Dashboard.log.error("Failed to store dashboard: ", dashboard.title, " due to: ", response.error, " stack trace: ", response.stacktrace);
                    maybeCallback();
                });
            });
        };

        FabricDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            var _this = this;
            var jolokia = this.jolokia;
            var details = this.details;

            var toDelete = array.length;

            var maybeCallback = function () {
                toDelete = toDelete - 1;
                if (toDelete === 0) {
                    _this.getDashboards(fn);
                }
            };

            array.forEach(function (dashboard) {
                var profileId = dashboard.profileId;
                var fileName = dashboard.fileName;
                if (profileId && fileName) {
                    Fabric.deleteConfigFile(jolokia, details.branch, profileId, fileName, function () {
                        maybeCallback();
                    }, function (response) {
                        Dashboard.log.error("Failed to delete dashboard: ", dashboard.title, " due to: ", response.error, " stack trace: ", response.stacktrace);
                        maybeCallback();
                    });
                }
            });
        };

        FabricDashboardRepository.prototype.createDashboard = function (options) {
            var answer = {
                title: "New Dashboard",
                group: "Fabric",
                versionId: this.details.branch,
                profileId: this.details.profiles.first(),
                widgets: []
            };
            answer = angular.extend(answer, options);
            var uuid = Core.getUUID();
            answer['id'] = uuid;
            answer['fileName'] = uuid + ".dashboard";
            return answer;
        };

        FabricDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            var newDashboard = Object.clone(dashboard);
            var uuid = Core.getUUID();
            newDashboard['id'] = uuid;
            newDashboard['fileName'] = uuid + ".dashboard";
            newDashboard['title'] = "Copy of " + dashboard.title;
            return newDashboard;
        };

        FabricDashboardRepository.prototype.getType = function () {
            return 'fabric';
        };

        FabricDashboardRepository.prototype.isValid = function () {
            return Fabric.hasFabric(this.workspace);
        };

        FabricDashboardRepository.prototype.getDashboards = function (fn) {
            var _this = this;
            var jolokia = this.jolokia;
            var details = this.details;
            var dashboards = [];

            jolokia.request({
                type: 'exec',
                mbean: Fabric.managerMBean,
                operation: 'getConfigurationFiles',
                arguments: [details.branch, details.profiles, ".*dashboard"]
            }, {
                method: 'POST',
                success: function (response) {
                    angular.forEach(response.value, function (value, profile) {
                        angular.forEach(value, function (value, fileName) {
                            var dashboard = angular.fromJson(value.decodeBase64());
                            dashboard['versionId'] = details.branch;
                            dashboard['profileId'] = profile;
                            dashboard['fileName'] = fileName;
                            dashboards.push(dashboard);
                        });
                    });

                    if (dashboards.isEmpty()) {
                        dashboards.push(_this.createDashboard({}));
                    }

                    fn(dashboards);
                },
                error: function (response) {
                    Dashboard.log.error("Failed to load dashboard data: error: ", response.error, " stack trace: ", response.stacktrace);
                    fn([]);
                }
            });
        };

        FabricDashboardRepository.prototype.getDashboard = function (id, fn) {
            this.getDashboards(function (dashboards) {
                var dashboard = dashboards.find(function (dashboard) {
                    return dashboard.id === id;
                });
                fn(dashboard);
            });
        };
        return FabricDashboardRepository;
    })();
    Dashboard.FabricDashboardRepository = FabricDashboardRepository;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function ImportController($scope, $location, $routeParams, workspace, dashboardRepository) {
        $scope.placeholder = "Paste the JSON here for the dashboard configuration to import...";
        $scope.source = $scope.placeholder;

        var options = {
            mode: {
                name: "javascript"
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);

        $scope.isValid = function () {
            return $scope.source && $scope.source !== $scope.placeholder;
        };

        $scope.importJSON = function () {
            var json = [];

            try  {
                json = JSON.parse($scope.source);
            } catch (e) {
                notification("error", "Could not parse the JSON\n" + e);
                json = [];
            }
            var array = [];
            if (angular.isArray(json)) {
                array = json;
            } else if (angular.isObject(json)) {
                array.push(json);
            }

            if (array.length) {
                // lets ensure we have some valid ids and stuff...
                angular.forEach(array, function (dash, index) {
                    angular.copy(dash, dashboardRepository.createDashboard(dash));
                });
                dashboardRepository.putDashboards(array, "Imported dashboard JSON", Dashboard.onOperationComplete);
                $location.path("/dashboard/edit");
            }
        };
    }
    Dashboard.ImportController = ImportController;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    Dashboard.log = Logger.get('Dashboard');

    /**
    * Returns the cleaned up version of the dashboard data without any UI selection state
    * @method cleanDashboardData
    * @static
    * @for Dashboard
    * @param {any} item
    * @return {any}
    */
    function cleanDashboardData(item) {
        var cleanItem = {};
        angular.forEach(item, function (value, key) {
            if (!angular.isString(key) || (!key.startsWith("$") && !key.startsWith("_"))) {
                cleanItem[key] = value;
            }
        });
        return cleanItem;
    }
    Dashboard.cleanDashboardData = cleanDashboardData;

    /**
    * Runs decodeURIComponent() on each value in the object
    * @method decodeURIComponentProperties
    * @static
    * @for Dashboard
    * @param {any} hash
    * @return {any}
    */
    function decodeURIComponentProperties(hash) {
        if (!hash) {
            return hash;
        }
        var decodeHash = {};
        angular.forEach(hash, function (value, key) {
            decodeHash[key] = value ? decodeURIComponent(value) : value;
        });
        return decodeHash;
    }
    Dashboard.decodeURIComponentProperties = decodeURIComponentProperties;

    function onOperationComplete(result) {
        console.log("Completed adding the dashboard with response " + JSON.stringify(result));
    }
    Dashboard.onOperationComplete = onOperationComplete;
})(Dashboard || (Dashboard = {}));
/**
* @module Jvm
*/
var JVM;
(function (JVM) {
    function ConnectController($scope, $location, localStorage, workspace) {
        JVM.configureScope($scope, $location, workspace);

        $scope.forms = {};

        $scope.chromeApp = Core.isChromeApp();
        $scope.useProxy = $scope.chromeApp ? false : true;

        $scope.settings = {
            last: 1,
            lastConnection: ''
        };

        // load settings like current tab, last used connection
        if (JVM.connectControllerKey in localStorage) {
            try  {
                $scope.settings = angular.fromJson(localStorage[JVM.connectControllerKey]);
            } catch (e) {
                // corrupt config
                delete localStorage[JVM.connectControllerKey];
            }
        }

        // load connection settings
        // TODO add known default configurations here...
        $scope.connectionConfigs = {};

        if (JVM.connectionSettingsKey in localStorage) {
            try  {
                $scope.connectionConfigs = angular.fromJson(localStorage[JVM.connectionSettingsKey]);
            } catch (e) {
                // corrupt config
                delete localStorage[JVM.connectionSettingsKey];
            }
        }

        /*
        log.debug("Controller settings: ", $scope.settings);
        log.debug("Current config: ", $scope.currentConfig);
        log.debug("All connection settings: ", $scope.connectionConfigs);
        */
        $scope.formConfig = {
            properties: {
                connectionName: {
                    type: 'java.lang.String',
                    description: 'Name for this connection',
                    'input-attributes': {
                        'placeholder': 'Unnamed...'
                    }
                },
                scheme: {
                    type: 'java.lang.String',
                    description: 'HTTP or HTTPS',
                    required: true
                },
                host: {
                    type: 'java.lang.String',
                    description: 'Target host to connect to',
                    required: true
                },
                port: {
                    type: 'java.lang.Integer',
                    description: 'The HTTP port used to connect to the server',
                    'input-attributes': {
                        'min': '0'
                    },
                    required: true
                },
                path: {
                    type: 'java.lang.String',
                    description: "The URL path used to connect to Jolokia on the remote server"
                },
                userName: {
                    type: 'java.lang.String',
                    description: "The user name to be used when connecting to Jolokia"
                },
                password: {
                    type: 'password',
                    description: 'The password to be used when connecting to Jolokia'
                },
                useProxy: {
                    type: 'java.lang.Boolean',
                    description: 'Whether or not we should use a proxy to connect to the remote Server',
                    'control-attributes': {
                        'ng-hide': 'chromeApp'
                    }
                }
            },
            type: 'void'
        };

        function newConfig() {
            var answer = {
                scheme: 'http',
                host: 'localhost',
                path: 'jolokia',
                port: '8181',
                userName: '',
                password: ''
            };

            if ($scope.chromeApp) {
                answer['useProxy'] = false;
            } else {
                answer['useProxy'] = true;
            }
            return answer;
        }

        $scope.clearSettings = function () {
            delete localStorage[JVM.connectControllerKey];
            delete localStorage[JVM.connectionSettingsKey];
            window.location.reload();
        };

        $scope.newConnection = function () {
            $scope.settings.lastConnection = '';
        };

        $scope.deleteConnection = function () {
            Core.removeRegex($scope.settings.lastConnection);
            delete $scope.connectionConfigs[$scope.settings.lastConnection];
            var tmp = Object.extended($scope.connectionConfigs);
            if (tmp.size() === 0) {
                $scope.settings.lastConnection = '';
            } else {
                $scope.settings.lastConnection = tmp.keys().first();
            }
            localStorage[JVM.connectionSettingsKey] = angular.toJson($scope.connectionConfigs);
        };

        $scope.$watch('settings', function (newValue, oldValue) {
            if (Core.isBlank($scope.settings['lastConnection'])) {
                $scope.currentConfig = newConfig();
            } else {
                $scope.currentConfig = Object.extended($scope.connectionConfigs[$scope.settings['lastConnection']]).clone();
            }

            if (newValue !== oldValue) {
                localStorage[JVM.connectControllerKey] = angular.toJson(newValue);
            }
        }, true);

        $scope.save = function () {
            $scope.gotoServer($scope.currentConfig, null, true);
        };

        $scope.gotoServer = function (json, form, saveOnly) {
            if (json) {
                var jsonCloned = Object.extended(json).clone(true);

                JVM.log.debug("json: ", jsonCloned);

                // new connection created via the form, let's save it
                var connectionName = jsonCloned['connectionName'];
                if (Core.isBlank(connectionName)) {
                    connectionName = "Unnamed" + $scope.settings.last++;
                    jsonCloned['connectionName'] = connectionName;
                }

                var regexs = Core.getRegexs();

                var hasFunc = function (r) {
                    return r['name'] === $scope.settings.lastConnection;
                };

                if ($scope.settings.lastConnection !== connectionName && !Core.isBlank($scope.settings.lastConnection)) {
                    //we're updating an existing connection...
                    delete $scope.connectionConfigs[$scope.settings.lastConnection];

                    // clean up any similarly named regex
                    regexs = regexs.exclude(hasFunc);
                }

                $scope.connectionConfigs[connectionName] = jsonCloned;
                localStorage[JVM.connectionSettingsKey] = angular.toJson($scope.connectionConfigs);
                if (regexs && !regexs.any(hasFunc)) {
                    Core.storeConnectionRegex(regexs, connectionName, jsonCloned);
                }

                // let's default to saved connections now that we've a new connection
                $scope.currentConfig = jsonCloned;
                $scope.settings.lastConnection = connectionName;
            }

            if (saveOnly === true) {
                Core.$apply($scope);
                return;
            }

            var options = new Core.ConnectToServerOptions();
            var host = $scope.currentConfig['host'] || 'localhost';

            // lets trim any http:// prefix or / postfix
            var idx = host.indexOf("://");
            if (idx >= 0) {
                host = host.substring(idx + 3);
            }
            idx = host.indexOf("/");
            if (idx >= 0) {
                host = host.substring(0, idx);
            }

            JVM.log.info("using scheme: " + $scope.currentConfig['scheme'] + " and host name: " + host + " and user: " + $scope.currentConfig['userName'] + " and password: " + ($scope.currentConfig['password'] ? "********" : $scope.currentConfig['password']));
            options.scheme = $scope.currentConfig['scheme'];
            options.host = host;
            options.port = $scope.currentConfig['port'];
            options.path = $scope.currentConfig['path'];
            options.userName = $scope.currentConfig['userName'];
            options.password = $scope.currentConfig['password'];
            options.useProxy = $scope.currentConfig['useProxy'];

            Core.$apply($scope);

            Core.connectToServer(localStorage, options);
        };

        function init() {
            JVM.log.debug("Initializing");
            var schemeEnum = ['http', 'https'];
            Core.pathSet($scope.formConfig, ['properties', 'scheme', 'enum'], schemeEnum);
        }

        init();
    }
    JVM.ConnectController = ConnectController;
})(JVM || (JVM = {}));
/**
* @module Jvm
*/
var JVM;
(function (JVM) {
    function JVMsController($scope, $window, $location, workspace, jolokia, mbeanName) {
        JVM.configureScope($scope, $location, workspace);
        $scope.data = [];
        $scope.deploying = false;
        $scope.status = '';

        $scope.fetch = function () {
            notification('info', 'Discovering local JVM processes, please wait...');
            jolokia.request({
                type: 'exec', mbean: mbeanName,
                operation: 'listLocalJVMs()',
                arguments: []
            }, {
                success: render,
                error: function (response) {
                    $scope.data = [];
                    $scope.status = 'Could not discover local JVM processes: ' + response.error;
                    Core.$apply($scope);
                }
            });
        };

        $scope.stopAgent = function (pid) {
            notification('info', "Attempting to detach agent from PID " + pid);
            jolokia.request({
                type: 'exec', mbean: mbeanName,
                operation: 'stopAgent(java.lang.String)',
                arguments: [pid]
            }, onSuccess(function () {
                notification('success', "Detached agent from PID " + pid);
                $scope.fetch();
            }));
        };

        $scope.startAgent = function (pid) {
            notification('info', "Attempting to attach agent to PID " + pid);
            jolokia.request({
                type: 'exec', mbean: mbeanName,
                operation: 'startAgent(java.lang.String)',
                arguments: [pid]
            }, onSuccess(function () {
                notification('success', "Attached agent to PID " + pid);
                $scope.fetch();
            }));
        };

        $scope.connectTo = function (url) {
            $window.open("?url=" + encodeURIComponent(url));
        };

        function render(response) {
            $scope.data = response.value;
            if ($scope.data.length === 0) {
                $scope.status = 'Could not discover local JVM processes';
            }
            Core.$apply($scope);
        }

        $scope.fetch();
    }
    JVM.JVMsController = JVMsController;
})(JVM || (JVM = {}));
/**
* @module Jvm
* @main Jvm
*/
var Jvm;
(function (Jvm) {
    var pluginName = 'jvm';

    angular.module(pluginName, ['bootstrap', 'ngResource', 'datatable', 'hawtioCore', 'hawtio-forms', 'ui']).config(function ($routeProvider) {
        $routeProvider.when('/jvm/connect', { templateUrl: 'app/jvm/html/connect.html' }).when('/jvm/local', { templateUrl: 'app/jvm/html/local.html' });
    }).constant('mbeanName', 'hawtio:type=JVMList').run(function ($location, workspace, viewRegistry, layoutFull, helpRegistry) {
        viewRegistry[pluginName] = layoutFull;
        helpRegistry.addUserDoc('jvm', 'app/jvm/doc/help.md');

        workspace.topLevelTabs.push({
            id: "connect",
            content: "Connect",
            title: "Connect to other JVMs",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return '#/jvm/connect';
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("jvm");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Jvm || (Jvm = {}));
/**
* @module Jvm
*/
var JVM;
(function (JVM) {
    JVM.log = Logger.get("JVM");

    JVM.connectControllerKey = "jvmConnectSettings";
    JVM.connectionSettingsKey = "jvmConnect";

    /**
    * Adds common properties and functions to the scope
    * @method configureScope
    * @for Jvm
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    */
    function configureScope($scope, $location, workspace) {
        $scope.isActive = function (href) {
            var tidy = Core.trimLeading(href, "#");
            var loc = $location.path();
            return loc === tidy;
        };

        $scope.isValid = function (link) {
            return link && link.isValid(workspace);
        };

        $scope.hasLocalMBean = function () {
            return JVM.hasLocalMBean(workspace);
        };

        $scope.breadcrumbs = [
            {
                content: '<i class=" icon-signin"></i> Remote',
                title: "Connect to a remote JVM running Jolokia",
                isValid: function (workspace) {
                    return true;
                },
                href: "#/jvm/connect"
            },
            {
                content: '<i class="icon-list-ul"></i> Local',
                title: "View a diagram of the route",
                isValid: function (workspace) {
                    return hasLocalMBean(workspace);
                },
                href: "#/jvm/local"
            }
        ];
    }
    JVM.configureScope = configureScope;

    function hasLocalMBean(workspace) {
        return workspace.treeContainsDomainAndProperties('hawtio', { type: 'JVMList' });
    }
    JVM.hasLocalMBean = hasLocalMBean;
})(JVM || (JVM = {}));
/**
* @module JBoss
* @main JBoss
*/
var JBoss;
(function (JBoss) {
    var pluginName = 'jboss';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ui.bootstrap.dialog', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/jboss/server', { templateUrl: 'app/jboss/html/server.html' }).when('/jboss/applications', { templateUrl: 'app/jboss/html/applications.html' }).when('/jboss/dmr', { templateUrl: 'app/jboss/html/dmr.html' }).when('/jboss/connectors', { templateUrl: 'app/jboss/html/connectors.html' });
    }).filter('jbossIconClass', function () {
        return JBoss.iconClass;
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['jboss'] = "app/jboss/html/layoutJBossTabs.html";
        helpRegistry.addUserDoc(pluginName, 'app/' + pluginName + '/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("jboss.as") || workspace.treeContainsDomainAndProperties("jboss.jta") || workspace.treeContainsDomainAndProperties("jboss.modules");
        });

        workspace.topLevelTabs.push({
            id: "jboss",
            content: "JBoss",
            title: "Manage your JBoss container",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("jboss.as") || workspace.treeContainsDomainAndProperties("jboss.jta") || workspace.treeContainsDomainAndProperties("jboss.modules");
            },
            href: function () {
                return "#/jboss/applications";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("jboss");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(JBoss || (JBoss = {}));
var JBoss;
(function (JBoss) {
    function JBossController($scope, $location, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | jbossIconClass}}"></i></div>';

        $scope.uninstallDialog = new Core.Dialog();

        $scope.webapps = [];
        $scope.selected = [];

        var columnDefs = [
            {
                field: 'status',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'name',
                displayName: 'Name',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'contextPath',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'webapps',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            }
        };

        function render(response) {
            $scope.webapps = [];
            $scope.mbeanIndex = {};
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    var mbean = obj.mbean;
                    if (mbean) {
                        obj.name = JBoss.cleanWebAppName(obj.name);
                        obj.contextPath = JBoss.cleanContextPath(obj.name);
                        var idx = $scope.mbeanIndex[mbean];
                        if (angular.isDefined(idx)) {
                            $scope.webapps[mbean] = obj;
                        } else {
                            $scope.mbeanIndex[mbean] = $scope.webapps.length;
                            $scope.webapps.push(obj);
                        }
                        Core.$apply($scope);
                    }
                }
            }

            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: ["name", "status"] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        // function to control the web applications
        $scope.controlWebApps = function (op) {
            // grab id of mbean names to control
            var mbeanNames = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(mbeanNames)) {
                mbeanNames = [mbeanNames];
            }

            // execute operation on each mbean
            var lastIndex = (mbeanNames.length || 1) - 1;
            angular.forEach(mbeanNames, function (mbean, idx) {
                var onResponse = (idx >= lastIndex) ? $scope.onLastResponse : $scope.onResponse;
                jolokia.request({
                    type: 'exec',
                    mbean: mbean,
                    operation: op,
                    arguments: null
                }, onSuccess(onResponse, { error: onResponse }));
            });
        };

        $scope.start = function () {
            $scope.controlWebApps('deploy');
        };

        $scope.stop = function () {
            $scope.controlWebApps('undeploy');
        };

        $scope.reload = function () {
            $scope.controlWebApps('redeploy');
        };

        $scope.uninstall = function () {
            $scope.controlWebApps('remove');
            $scope.uninstallDialog.close();
        };

        // function to trigger reloading page
        $scope.onLastResponse = function (response) {
            $scope.onResponse(response);

            // we only want to force updating the data on the last response
            loadData();
        };

        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading JBoss webapp data...");
            jolokia.search("jboss.as:deployment=*", onSuccess(render));
        }

        // grab server information once
        $scope.jbossServerVersion = "";
        $scope.jbossServerName = "";
        $scope.jbossServerLaunchType = "";

        // lookup jboss 6 and 7
        var servers = jolokia.search("jboss.as:management-root=*");
        if (servers && servers.length === 1) {
            $scope.jbossServerVersion = jolokia.getAttribute(servers[0], "releaseVersion");
            $scope.jbossServerName = jolokia.getAttribute(servers[0], "name");
            $scope.jbossServerLaunchType = jolokia.getAttribute(servers[0], "launchType");
        } else {
            // wildfly is changed
            var wildflyMBean = 'jboss.as:management-root=server';
            var response = jolokia.request({ type: "read", mbean: wildflyMBean, attribute: ["releaseVersion", "name", "launchType"] });
            if (response) {
                var obj = response.value;
                if (obj) {
                    $scope.jbossServerVersion = obj.releaseVersion;
                    $scope.jbossServerName = obj.name;
                    $scope.jbossServerLaunchType = obj.launchType;
                }
            } else {
                console.log("Cannot find JBoss/Wildfly server or there was more than one server");
            }
        }
    }
    JBoss.JBossController = JBossController;
})(JBoss || (JBoss = {}));
var JBoss;
(function (JBoss) {
    function DmrController($scope, $location, workspace) {
        var search = $location.search();
        var connectUrl = url("/proxy/localhost/9990/management");
        var user = search["_user"] || "";
        var pwd = search["_pwd"] || "";
        if (user) {
            connectUrl += "?_user=" + user;
            if (pwd) {
                connectUrl += "&_pwd=" + pwd;
            }
        }

        var isDmr = "dmr" === search["_format"];
        var data = null;
        var format = "application/dmr-encoded";
        if (isDmr) {
            // create an operation
            /*
            var op = new dmr.ModelNode();
            //op.get("operation").set("read-resource");
            op.get("operation").set("read-attribute");
            op.get("address").setEmptyList();
            op.get("name").set("release-version");
            */
            var op = new dmr.ModelNode();
            op.get("operation").set("read-attribute");
            op.get("address").setEmptyList();
            op.get("name").set("release-version");

            data = op.toBase64String();
        } else {
            format = "application/json";
            var request = {
                "operation": "read-resource"
            };
            data = JSON.stringify(request);
        }

        console.log("Using dmr: " + isDmr + " with content type: " + format + " and data " + data);

        $.ajax({
            url: connectUrl,
            data: data,
            processData: false,
            type: "POST",
            dataType: "text",
            contentType: format,
            accepts: format,
            headers: {
                "Content-type": format,
                "Accept": format
            }
        }).done(onData);

        function onData(data) {
            if (data) {
                var json = null;
                if (isDmr) {
                    var response = dmr.ModelNode.fromBase64(data);
                    var jsonText = response.toJSONString();
                    json = JSON.parse(jsonText);
                } else {
                    json = JSON.parse(data);
                    json = json.result;
                }

                $scope.row = json;
                Core.$apply($scope);
                console.log("Response: " + JSON.stringify(json, null, "  "));
            }
        }
    }
    JBoss.DmrController = DmrController;
})(JBoss || (JBoss = {}));
var JBoss;
(function (JBoss) {
    function ConnectorsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | jbossIconClass}}"></i></div>';

        $scope.connectors = [];

        var columnDefs = [
            {
                field: 'bound',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'name',
                displayName: 'Name',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'port',
                displayName: 'Port',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'connectors',
            displayFooter: false,
            displaySelectionCheckbox: false,
            canSelectRows: false,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            }
        };

        function render(response) {
            $scope.connectors = [];

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    if (!obj.port) {
                        obj.port = obj.boundPort;
                    }
                    if (!obj.name) {
                        // special hack for mail-smtp, it only has port
                        obj.name = "mail-smtp";
                    }
                    $scope.connectors.push(obj);
                    Core.$apply($scope);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                if (mbean.toString() !== "jboss.as:socket-binding-group=standard-sockets") {
                    if (mbean.toString().lastIndexOf("management") > 0) {
                        // management mbean do not have port
                        jolokia.request({ type: "read", mbean: mbean, attribute: ["boundPort", "name", "bound"] }, onSuccess(onAttributes));
                    } else if (mbean.toString().lastIndexOf("mail-smtp") > 0) {
                        // special hack for mail-smtp, it only has port
                        jolokia.request({ type: "read", mbean: mbean, attribute: ["port"] }, onSuccess(onAttributes));
                    } else {
                        jolokia.request({ type: "read", mbean: mbean, attribute: ["port", "name", "bound"] }, onSuccess(onAttributes));
                    }
                }
            });
            Core.$apply($scope);
        }
        ;

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading JBoss connector data...");
            jolokia.search("jboss.as:socket-binding-group=standard-sockets,*", onSuccess(render));
        }
    }
    JBoss.ConnectorsController = ConnectorsController;
})(JBoss || (JBoss = {}));
var JBoss;
(function (JBoss) {
    function cleanWebAppName(name) {
        // JBoss may include .war as the application name, so remove that
        if (name && name.lastIndexOf(".war") > -1) {
            return name.replace(".war", "");
        } else {
            return name;
        }
    }
    JBoss.cleanWebAppName = cleanWebAppName;

    function cleanContextPath(contextPath) {
        if (contextPath) {
            return "/" + cleanWebAppName(contextPath);
        } else {
            return "";
        }
    }
    JBoss.cleanContextPath = cleanContextPath;

    function iconClass(state) {
        if (state) {
            switch (state.toString().toLowerCase()) {
                case 'started':
                    return "green icon-play-circle";
                case 'ok':
                    return "green icon-play-circle";
                case 'true':
                    return "green icon-play-circle";
            }
        }
        return "orange icon-off";
    }
    JBoss.iconClass = iconClass;
})(JBoss || (JBoss = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function MBeansController($scope, $location, workspace) {
        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        $scope.select = function (node) {
            $scope.workspace.updateSelectionNode(node);
            Core.$apply($scope);
        };

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURL($location, $("#jmxtree"));
        }

        $scope.populateTree = function () {
            var treeElement = $("#jmxtree");
            $scope.tree = workspace.tree;
            Jmx.enableTree($scope, $location, workspace, treeElement, $scope.tree.children, true);
            setTimeout(updateSelectionFromURL, 50);
        };

        $scope.$on('jmxTreeUpdated', $scope.populateTree);

        $scope.populateTree();
    }
    Jmx.MBeansController = MBeansController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function ChartController($scope, $element, $location, workspace, localStorage, jolokiaUrl, jolokiaParams) {
        $scope.metrics = [];
        $scope.updateRate = 1000; //parseInt(localStorage['updateRate']);

        $scope.context = null;
        $scope.jolokia = null;
        $scope.charts = null;

        $scope.reset = function () {
            if ($scope.context) {
                $scope.context.stop();
                $scope.context = null;
            }
            if ($scope.jolokia) {
                $scope.jolokia.stop();
                $scope.jolokia = null;
            }
            if ($scope.charts) {
                $scope.charts.empty();
                $scope.charts = null;
            }
        };

        $scope.$on('$destroy', function () {
            try  {
                $scope.deregRouteChange();
            } catch (error) {
                // ignore
            }
            try  {
                $scope.dereg();
            } catch (error) {
                // ignore
            }
            $scope.reset();
        });

        $scope.errorMessage = function () {
            if ($scope.updateRate === 0) {
                return "updateRate";
            }

            if ($scope.metrics.length === 0) {
                return "metrics";
            }
        };

        var doRender = Core.throttled(render, 200);

        $scope.deregRouteChange = $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            doRender();
        });
        $scope.dereg = $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;
            doRender();
        });

        doRender();

        function render() {
            var node = workspace.selection;
            if (!angular.isDefined(node) || !angular.isDefined($scope.updateRate) || $scope.updateRate === 0) {
                // Called render too early, let's retry
                setTimeout(doRender, 500);
                Core.$apply($scope);
                return;
            }
            var width = 594;
            var charts = $element.find('#charts');
            if (charts) {
                width = charts.width();
            } else {
                // Called render too early, let's retry
                setTimeout(doRender, 500);
                Core.$apply($scope);
                return;
            }

            // clear out any existing context
            $scope.reset();

            $scope.charts = charts;
            $scope.jolokia = new Jolokia(jolokiaParams);
            $scope.jolokia.start($scope.updateRate);

            var mbean = node.objectName;
            $scope.metrics = [];

            var context = cubism.context().serverDelay($scope.updateRate).clientDelay($scope.updateRate).step($scope.updateRate).size(width);

            $scope.context = context;
            $scope.jolokiaContext = context.jolokia($scope.jolokia);
            var search = $location.search();
            var attributeNames = toSearchArgumentArray(search["att"]);

            if (mbean) {
                // TODO make generic as we can cache them; they rarely ever change
                // lets get the attributes for this mbean
                // we need to escape the mbean path for list
                var listKey = encodeMBeanPath(mbean);

                //console.log("Looking up mbeankey: " + listKey);
                var meta = $scope.jolokia.list(listKey);
                if (meta) {
                    var attributes = meta.attr;
                    if (attributes) {
                        var foundNames = [];
                        for (var key in attributes) {
                            var value = attributes[key];
                            if (value) {
                                var typeName = value['type'];
                                if (isNumberTypeName(typeName)) {
                                    foundNames.push(key);
                                }
                            }
                        }

                        // lets filter the attributes
                        // if we find none then the att search attribute is invalid
                        // so lets discard the filter - as it must be for some other mbean
                        if (attributeNames.length) {
                            var filtered = foundNames.filter(function (key) {
                                return attributeNames.indexOf(key) >= 0;
                            });
                            if (filtered.length) {
                                foundNames = filtered;
                            }
                        }
                        angular.forEach(foundNames, function (key) {
                            var metric = $scope.jolokiaContext.metric({
                                type: 'read',
                                mbean: mbean,
                                attribute: key
                            }, humanizeValue(key));
                            if (metric) {
                                $scope.metrics.push(metric);
                            }
                        });
                    }
                }
            } else {
                // lets try pull out the attributes and elements from the URI and use those to chart
                var elementNames = toSearchArgumentArray(search["el"]);
                if (attributeNames && attributeNames.length && elementNames && elementNames.length) {
                    // first lets map the element names to mbean names to keep the URI small
                    var mbeans = {};
                    elementNames.forEach(function (elementName) {
                        var child = node.get(elementName);
                        if (!child && node.children) {
                            child = node.children.find(function (n) {
                                return elementName === n["title"];
                            });
                        }
                        if (child) {
                            var mbean = child.objectName;
                            if (mbean) {
                                mbeans[elementName] = mbean;
                            }
                        }
                    });

                    // lets create the metrics
                    attributeNames.forEach(function (key) {
                        angular.forEach(mbeans, function (mbean, name) {
                            var attributeTitle = humanizeValue(key);

                            // for now lets always be verbose
                            var title = name + ": " + attributeTitle;

                            var metric = $scope.jolokiaContext.metric({
                                type: 'read',
                                mbean: mbean,
                                attribute: key
                            }, title);
                            if (metric) {
                                $scope.metrics.push(metric);
                            }
                        });
                    });
                }

                // if we've children and none of the query arguments matched any metrics
                // lets redirect back to the edit view
                if (node.children.length && !$scope.metrics.length) {
                    // lets forward to the chart selection UI if we have some children; they may have
                    // chartable attributes
                    $location.path("jmx/chartEdit");
                }
            }

            if ($scope.metrics.length > 0) {
                var d3Selection = d3.select(charts.get(0));
                var axisEl = d3Selection.selectAll(".axis");

                var bail = false;

                axisEl.data(["top", "bottom"]).enter().append("div").attr("class", function (d) {
                    return d + " axis";
                }).each(function (d) {
                    if (bail) {
                        return;
                    }
                    try  {
                        d3.select(this).call(context.axis().ticks(12).orient(d));
                    } catch (error) {
                        // still rendering at not the right time...
                        // log.debug("error: ", error);
                        if (!bail) {
                            bail = true;
                        }
                    }
                });

                if (bail) {
                    $scope.reset();
                    setTimeout(doRender, 500);
                    Core.$apply($scope);
                    return;
                }

                d3Selection.append("div").attr("class", "rule").call(context.rule());

                context.on("focus", function (i) {
                    try  {
                        d3Selection.selectAll(".value").style("right", i === null ? null : context.size() - i + "px");
                    } catch (error) {
                        Jmx.log.info("error: ", error);
                    }
                });

                $scope.metrics.forEach(function (metric) {
                    d3Selection.call(function (div) {
                        div.append("div").data([metric]).attr("class", "horizon").call(context.horizon());
                    });
                });
            } else {
                $scope.reset();
            }

            Core.$apply($scope);
        }
        ;
    }
    Jmx.ChartController = ChartController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    Jmx.propertiesColumnDefs = [
        {
            field: 'name', displayName: 'Property', width: "27%",
            cellTemplate: '<div class="ngCellText" title="{{row.entity.attrDesc}}" ' + 'data-placement="bottom"><div ng-show="!inDashboard" class="inline" compile="getDashboardWidgets(row.entity)"></div>{{row.entity.name}}</div>' },
        {
            field: 'value', displayName: 'Value', width: "70%",
            cellTemplate: '<div class="ngCellText" ng-click="onViewAttribute(row.entity)" title="{{row.entity.tooltip}}" ng-bind-html-unsafe="row.entity.summary"></div>'
        }
    ];

    Jmx.foldersColumnDefs = [
        {
            displayName: 'Name',
            cellTemplate: '<div class="ngCellText"><a href="{{folderHref(row)}}"><i class="{{folderIconClass(row)}}"></i> {{row.getProperty("title")}}</a></div>'
        }
    ];

    function AttributesController($scope, $element, $location, workspace, jolokia, jmxWidgets, jmxWidgetTypes) {
        $scope.searchText = '';
        $scope.columnDefs = [];
        $scope.selectedItems = [];

        $scope.lastKey = null;
        $scope.attributesInfoCache = {};

        $scope.entity = {};
        $scope.attributeSchema = {};

        var attributeSchemaBasic = {
            properties: {
                'key': {
                    description: 'Key',
                    tooltip: 'Attribute key',
                    type: 'string',
                    readOnly: 'true'
                },
                'description': {
                    description: 'Description',
                    tooltip: 'Attribute description',
                    type: 'string',
                    formTemplate: "<textarea class='input-xlarge' rows='2' readonly='true'></textarea>"
                },
                'type': {
                    description: 'Type',
                    tooltip: 'Attribute type',
                    type: 'string',
                    readOnly: 'true'
                }
            }
        };

        $scope.gridOptions = {
            selectedItems: $scope.selectedItems,
            showFilter: false,
            canSelectRows: false,
            enableRowSelection: true,
            keepLastSelected: false,
            multiSelect: true,
            showColumnMenu: true,
            displaySelectionCheckbox: false,
            filterOptions: {
                filterText: ''
            },
            // TODO disabled for now as it causes https://github.com/hawtio/hawtio/issues/262
            //sortInfo: { field: 'name', direction: 'asc'},
            data: 'gridData',
            columnDefs: 'columnDefs'
        };

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid()) {
                Core.unregister(jolokia, $scope);
                return;
            }
            updateTableContents();
        });

        $scope.hasWidget = function (row) {
            console.log("Row: ", row);
            return true;
        };

        $scope.onCancelAttribute = function () {
            // clear entity
            $scope.entity = {};
        };

        $scope.onUpdateAttribute = function () {
            var value = $scope.entity["attrValueEdit"];
            var key = $scope.entity["key"];

            // clear entity
            $scope.entity = {};

            // TODO: check if value changed
            // update the attribute on the mbean
            var mbean = workspace.getSelectedMBeanName();
            if (mbean) {
                jolokia.setAttribute(mbean, key, value, onSuccess(function (response) {
                    notification("success", "Updated attribute " + key);
                }));
            }
        };

        $scope.onViewAttribute = function (row) {
            // create entity and populate it with data from the selected row
            $scope.entity = {};
            $scope.entity["key"] = row.key;
            $scope.entity["description"] = row.attrDesc;
            $scope.entity["type"] = row.type;
            $scope.entity["rw"] = row.rw;
            var type = asJsonSchemaType(row.type, row.key);
            var readOnly = !row.rw;

            // calculate a textare with X number of rows that usually fit the value to display
            var len = row.summary.length;
            var rows = (len / 40) + 1;
            if (rows > 10) {
                // cap at most 10 rows to not make the dialog too large
                rows = 10;
            }

            if (readOnly) {
                // if the value is empty its a &nbsp; as we need this for the table to allow us to click on the empty row
                if (row.summary === '&nbsp;') {
                    $scope.entity["attrValueView"] = '';
                } else {
                    $scope.entity["attrValueView"] = row.summary;
                }

                // clone from the basic schema to the new schema we create on-the-fly
                // this is needed as the dialog have problems if reusing the schema, and changing the schema afterwards
                // so its safer to create a new schema according to our needs
                $scope.attributeSchemaView = {};
                for (var i in attributeSchemaBasic) {
                    $scope.attributeSchemaView[i] = attributeSchemaBasic[i];
                }

                // and add the new attrValue which is dynamic computed
                $scope.attributeSchemaView.properties.attrValueView = {
                    description: 'Value',
                    label: "Value",
                    tooltip: 'Attribute value',
                    type: 'string',
                    formTemplate: "<textarea class='input-xlarge' rows='" + rows + "' readonly='true'></textarea>"
                };

                // just to be safe, then delete not needed part of the scema
                if ($scope.attributeSchemaView) {
                    delete $scope.attributeSchemaView.properties.attrValueEdit;
                }
            } else {
                // if the value is empty its a &nbsp; as we need this for the table to allow us to click on the empty row
                if (row.summary === '&nbsp;') {
                    $scope.entity["attrValueEdit"] = '';
                } else {
                    $scope.entity["attrValueEdit"] = row.summary;
                }

                // clone from the basic schema to the new schema we create on-the-fly
                // this is needed as the dialog have problems if reusing the schema, and changing the schema afterwards
                // so its safer to create a new schema according to our needs
                $scope.attributeSchemaEdit = {};
                for (var i in attributeSchemaBasic) {
                    $scope.attributeSchemaEdit[i] = attributeSchemaBasic[i];
                }

                // and add the new attrValue which is dynamic computed
                $scope.attributeSchemaEdit.properties.attrValueEdit = {
                    description: 'Value',
                    label: "Value",
                    tooltip: 'Attribute value',
                    type: 'string',
                    formTemplate: "<textarea class='input-xlarge' rows='" + rows + "'></textarea>"
                };

                // just to be safe, then delete not needed part of the scema
                if ($scope.attributeSchemaEdit) {
                    delete $scope.attributeSchemaEdit.properties.attrValueView;
                }
            }

            $scope.showAttributeDialog = true;
        };

        $scope.getDashboardWidgets = function (row) {
            var mbean = workspace.getSelectedMBeanName();
            if (!mbean) {
                return '';
            }
            var potentialCandidates = jmxWidgets.filter(function (widget) {
                return mbean === widget.mbean;
            });

            if (potentialCandidates.isEmpty()) {
                return '';
            }

            potentialCandidates = potentialCandidates.filter(function (widget) {
                return widget.attribute === row.key || widget.total === row.key;
            });

            if (potentialCandidates.isEmpty()) {
                return '';
            }

            var rc = [];
            potentialCandidates.forEach(function (widget) {
                var widgetType = Jmx.getWidgetType(widget);
                rc.push("<i class=\"" + widgetType['icon'] + " clickable\" title=\"" + widgetType['title'] + "\" ng-click=\"addChartToDashboard(row.entity, '" + widgetType['type'] + "')\"></i>");
            });
            return rc.join() + "&nbsp;";
        };

        $scope.addChartToDashboard = function (row, widgetType) {
            var mbean = workspace.getSelectedMBeanName();
            var candidates = jmxWidgets.filter(function (widget) {
                return mbean === widget.mbean;
            });

            candidates = candidates.filter(function (widget) {
                return widget.attribute === row.key || widget.total === row.key;
            });

            candidates = candidates.filter(function (widget) {
                return widget.type === widgetType;
            });

            // hmmm, we really should only have one result...
            var widget = candidates.first();
            var type = Jmx.getWidgetType(widget);

            //console.log("widgetType: ", type, " widget: ", widget);
            $location.url(Jmx.createDashboardLink(type, widget));
        };

        /*
        * Returns the toolBar template HTML to use for the current selection
        */
        $scope.toolBarTemplate = function () {
            // lets lookup the list of helpers by domain
            var answer = Jmx.getAttributeToolBar(workspace.selection);

            // TODO - maybe there's a better way to determine when to enable selections
            /*
            if (answer.startsWith("app/camel") && workspace.selection.children.length > 0) {
            $scope.selectToggle.setSelect(true);
            } else {
            $scope.selectToggle.setSelect(false);
            }
            */
            return answer;
        };

        $scope.invokeSelectedMBeans = function (operationName, completeFunction) {
            if (typeof completeFunction === "undefined") { completeFunction = null; }
            var queries = [];
            angular.forEach($scope.selectedItems || [], function (item) {
                var mbean = item["_id"];
                if (mbean) {
                    var opName = operationName;
                    if (angular.isFunction(operationName)) {
                        opName = operationName(item);
                    }

                    //console.log("Invoking operation " + opName + " on " + mbean);
                    queries.push({ type: "exec", operation: opName, mbean: mbean });
                }
            });
            if (queries.length) {
                var callback = function () {
                    if (completeFunction) {
                        completeFunction();
                    } else {
                        operationComplete();
                    }
                };
                jolokia.request(queries, onSuccess(callback, { error: callback }));
            }
        };

        $scope.folderHref = function (row) {
            var key = row.getProperty("key");
            if (key) {
                return Core.createHref($location, "#" + $location.path() + "?nid=" + key, ["nid"]);
            } else {
                return "";
            }
        };

        $scope.folderIconClass = function (row) {
            // TODO lets ignore the classes property for now
            // as we don't have an easy way to know if there is an icon defined for an icon or not
            // and we want to make sure there always is an icon shown
            /*
            var classes = (row.getProperty("addClass") || "").trim();
            if (classes) {
            return classes;
            }
            */
            return row.getProperty("objectName") ? "icon-cog" : "icon-folder-close";
        };

        function operationComplete() {
            updateTableContents();
        }

        function updateTableContents() {
            // lets clear any previous queries just in case!
            Core.unregister(jolokia, $scope);

            $scope.gridData = [];
            $scope.mbeanIndex = null;
            var mbean = workspace.getSelectedMBeanName();
            var request = null;
            var node = workspace.selection;
            if (node === null || angular.isUndefined(node) || node.key !== $scope.lastKey) {
                // cache attributes info, so we know if the attribute is read-only or read-write, and also the attribute description
                $scope.attributesInfoCache = null;
                if (mbean) {
                    var asQuery = function (node) {
                        var path = escapeMBeanPath(node);
                        var query = {
                            type: "LIST",
                            method: "post",
                            path: path,
                            ignoreErrors: true
                        };
                        return query;
                    };
                    var infoQuery = asQuery(mbean);
                    jolokia.request(infoQuery, onSuccess(function (response) {
                        $scope.attributesInfoCache = response.value;
                        Jmx.log.debug("Updated attributes info cache for mbean " + mbean);
                    }));
                }
            }

            if (mbean) {
                request = { type: 'read', mbean: mbean };
                if (node.key !== $scope.lastKey) {
                    $scope.columnDefs = Jmx.propertiesColumnDefs;
                }
            } else if (node) {
                if (node.key !== $scope.lastKey) {
                    $scope.columnDefs = [];
                }

                // lets query each child's details
                var children = node.children;
                if (children) {
                    var childNodes = children.map(function (child) {
                        return child.objectName;
                    });
                    var mbeans = childNodes.filter(function (mbean) {
                        return mbean;
                    });
                    if (mbeans) {
                        var typeNames = Jmx.getUniqueTypeNames(children);
                        if (typeNames.length <= 1) {
                            var query = mbeans.map(function (mbean) {
                                return { type: "READ", mbean: mbean, ignoreErrors: true };
                            });
                            if (query.length > 0) {
                                request = query;

                                // deal with multiple results
                                $scope.mbeanIndex = {};
                                $scope.mbeanRowCounter = 0;
                                $scope.mbeanCount = mbeans.length;
                                //$scope.columnDefs = [];
                            }
                        } else {
                            console.log("Too many type names " + typeNames);
                        }
                    }
                }
            }

            //var callback = onSuccess(render, { error: render });
            var callback = onSuccess(render);
            if (request) {
                $scope.request = request;
                Core.register(jolokia, $scope, request, callback);
            } else if (node) {
                if (node.key !== $scope.lastKey) {
                    $scope.columnDefs = Jmx.foldersColumnDefs;
                }
                $scope.gridData = node.children;
            }
            if (node) {
                $scope.lastKey = node.key;
            }
        }

        function render(response) {
            var data = response.value;
            var mbeanIndex = $scope.mbeanIndex;
            var mbean = response.request['mbean'];
            Jmx.log.debug("mbean: ", mbean);
            if (mbean) {
                // lets store the mbean in the row for later
                data["_id"] = mbean;
            }
            if (mbeanIndex) {
                if (mbean) {
                    var idx = mbeanIndex[mbean];
                    if (!angular.isDefined(idx)) {
                        idx = $scope.mbeanRowCounter;
                        mbeanIndex[mbean] = idx;
                        $scope.mbeanRowCounter += 1;
                    }
                    if (idx === 0) {
                        // this is to force the table to repaint
                        $scope.selectedIndices = $scope.selectedItems.map(function (item) {
                            return $scope.gridData.indexOf(item);
                        });
                        $scope.gridData = [];

                        if (!$scope.columnDefs.length) {
                            // lets update the column definitions based on any configured defaults
                            var key = workspace.selectionConfigKey();
                            var defaultDefs = workspace.attributeColumnDefs[key] || [];
                            var defaultSize = defaultDefs.length;
                            var map = {};
                            angular.forEach(defaultDefs, function (value, key) {
                                var field = value.field;
                                if (field) {
                                    map[field] = value;
                                }
                            });

                            var extraDefs = [];
                            angular.forEach(data, function (value, key) {
                                if (includePropertyValue(key, value)) {
                                    if (!map[key]) {
                                        extraDefs.push({
                                            field: key,
                                            displayName: key === '_id' ? 'Object name' : humanizeValue(key),
                                            visible: defaultSize === 0
                                        });
                                    }
                                }
                            });

                            // the additional columns (which are not pre-configured), should be sorted
                            // so the column menu has a nice sorted list instead of random ordering
                            extraDefs = extraDefs.sort(function (def, def2) {
                                // make sure _id is last
                                if (def.field.startsWith('_')) {
                                    return 1;
                                } else if (def2.field.startsWith('_')) {
                                    return -1;
                                }
                                return def.field.localeCompare(def2.field);
                            });
                            extraDefs.forEach(function (e) {
                                defaultDefs.push(e);
                            });

                            $scope.columnDefs = defaultDefs;
                        }
                    }

                    // assume 1 row of data per mbean
                    $scope.gridData[idx] = data;

                    var count = $scope.mbeanCount;
                    if (!count || idx + 1 >= count) {
                        // only cause a refresh on the last row
                        var newSelections = $scope.selectedIndices.map(function (idx) {
                            return $scope.gridData[idx];
                        }).filter(function (row) {
                            return row;
                        });
                        $scope.selectedItems.splice(0, $scope.selectedItems.length);
                        $scope.selectedItems.push.apply($scope.selectedItems, newSelections);

                        //console.log("Would have selected " + JSON.stringify($scope.selectedItems));
                        Core.$apply($scope);
                    }
                    // if the last row, then fire an event
                } else {
                    console.log("No mbean name in request " + JSON.stringify(response.request));
                }
            } else {
                $scope.columnDefs = Jmx.propertiesColumnDefs;
                var showAllAttributes = true;
                if (angular.isObject(data)) {
                    var properties = [];
                    angular.forEach(data, function (value, key) {
                        if (showAllAttributes || includePropertyValue(key, value)) {
                            // always skip keys which start with _
                            if (!key.startsWith("_")) {
                                // lets format the ObjectName nicely dealing with objects with
                                // nested object names or arrays of object names
                                if (key === "ObjectName") {
                                    value = unwrapObjectName(value);
                                }

                                // lets unwrap any arrays of object names
                                if (angular.isArray(value)) {
                                    value = value.map(function (v) {
                                        return unwrapObjectName(v);
                                    });
                                }
                                var data = { key: key, name: humanizeValue(key), value: safeNull(value) };

                                generateSummaryAndDetail(key, data);
                                properties.push(data);
                            }
                        }
                    });
                    if (!properties.any(function (p) {
                        return p['key'] === 'ObjectName';
                    })) {
                        var objectName = {
                            key: "ObjectName",
                            name: "Object Name",
                            value: mbean
                        };
                        generateSummaryAndDetail(objectName.key, objectName);
                        properties.push(objectName);
                    }
                    properties = properties.sortBy("name");
                    $scope.selectedItems = [data];
                    data = properties;
                }
                $scope.gridData = data;

                // log.debug("gridData: ", $scope.gridData);
                Core.$apply($scope);
            }
        }

        function unwrapObjectName(value) {
            if (!angular.isObject(value)) {
                return value;
            }
            var keys = Object.keys(value);
            if (keys.length === 1 && keys[0] === "objectName") {
                return value["objectName"];
            }
            return value;
        }

        function generateSummaryAndDetail(key, data) {
            var value = data.value;
            if (!angular.isArray(value) && angular.isObject(value)) {
                var detailHtml = "<table class='table table-striped'>";
                var summary = "";
                var object = value;
                var keys = Object.keys(value).sort();
                angular.forEach(keys, function (key) {
                    var value = object[key];
                    detailHtml += "<tr><td>" + humanizeValue(key) + "</td><td>" + value + "</td></tr>";
                    summary += "" + humanizeValue(key) + ": " + value + "  ";
                });
                detailHtml += "</table>";
                data.summary = summary;
                data.detailHtml = detailHtml;
                data.tooltip = summary;
            } else {
                var text = value;

                // if the text is empty then use a no-break-space so the table allows us to click on the row,
                // otherwise if the text is empty, then you cannot click on the row
                if (text === '') {
                    text = '&nbsp;';
                    data.tooltip = "";
                } else {
                    data.tooltip = text;
                }
                data.summary = "" + text + "";
                data.detailHtml = "<pre>" + text + "</pre>";
                if (angular.isArray(value)) {
                    var html = "<ul>";
                    angular.forEach(value, function (item) {
                        html += "<li>" + item + "</li>";
                    });
                    html += "</ul>";
                    data.detailHtml = html;
                }
            }

            // enrich the data with information if the attribute is read-only/read-write, and the JMX attribute description (if any)
            data.rw = false;
            data.attrDesc = data.name;
            data.type = "string";
            if ($scope.attributesInfoCache != null && 'attr' in $scope.attributesInfoCache) {
                var info = $scope.attributesInfoCache.attr[key];
                if (angular.isDefined(info)) {
                    data.rw = info.rw;
                    data.attrDesc = info.desc;
                    data.type = info.type;
                }
            }
        }

        function includePropertyValue(key, value) {
            return !angular.isObject(value);
        }

        function asJsonSchemaType(typeName, id) {
            if (typeName) {
                var lower = typeName.toLowerCase();
                if (lower.startsWith("int") || lower === "long" || lower === "short" || lower === "byte" || lower.endsWith("int")) {
                    return "integer";
                }
                if (lower === "double" || lower === "float" || lower === "bigdecimal") {
                    return "number";
                }
                if (lower === "boolean" || lower === "java.lang.boolean") {
                    return "boolean";
                }
                if (lower === "string" || lower === "java.lang.String") {
                    return "string";
                }
            }

            // fallback as string
            return "string";
        }
    }
    Jmx.AttributesController = AttributesController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
* @main Jmx
*/
var Jmx;
(function (Jmx) {
    var pluginName = 'jmx';

    Jmx.currentProcessId = '';

    angular.module(pluginName, ['bootstrap', 'ui.bootstrap', 'ui.bootstrap.modal', 'ngResource', 'ngGrid', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/jmx/attributes', { templateUrl: 'app/jmx/html/attributes.html' }).when('/jmx/operations', { templateUrl: 'app/jmx/html/operations.html' }).when('/jmx/charts', { templateUrl: 'app/jmx/html/charts.html' }).when('/jmx/chartEdit', { templateUrl: 'app/jmx/html/chartEdit.html' }).when('/jmx/help/:tabName', { templateUrl: 'app/core/html/help.html' }).when('/jmx/widget/donut', { templateUrl: 'app/jmx/html/donutChart.html' }).when('/jmx/widget/area', { templateUrl: 'app/jmx/html/areaChart.html' });
    }).factory('jmxTreeLazyLoadRegistry', function () {
        return Jmx.lazyLoaders;
    }).factory('jmxWidgetTypes', function () {
        return Jmx.jmxWidgetTypes;
    }).factory('jmxWidgets', function () {
        return Jmx.jmxWidgets;
    }).run(function ($location, workspace, viewRegistry, layoutTree, jolokia, pageTitle, helpRegistry) {
        viewRegistry['jmx'] = layoutTree;
        helpRegistry.addUserDoc('jmx', 'app/jmx/doc/help.md');

        pageTitle.addTitleElement(function () {
            if (Jmx.currentProcessId === '') {
                try  {
                    Jmx.currentProcessId = jolokia.getAttribute('java.lang:type=Runtime', 'Name');
                } catch (e) {
                    // ignore
                }
                if (Jmx.currentProcessId && Jmx.currentProcessId.has("@")) {
                    Jmx.currentProcessId = "pid:" + Jmx.currentProcessId.split("@")[0];
                }
            }
            return Jmx.currentProcessId;
        });

        workspace.topLevelTabs.push({
            id: "jmx",
            content: "JMX",
            title: "View the JMX MBeans in this process",
            isValid: function (workspace) {
                return workspace.hasMBeans();
            },
            href: function () {
                return "#/jmx/attributes";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("jmx");
            }
        });

        workspace.subLevelTabs.push({
            content: '<i class="icon-list"></i> Attributes',
            title: "View the attribute values on your selection",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return "#/jmx/attributes";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-leaf"></i> Operations',
            title: "Execute operations on your selection",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return "#/jmx/operations";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-bar-chart"></i> Chart',
            title: "View a chart of the metrics on your selection",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return "#/jmx/charts";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-cog"></i> Edit Chart',
            title: "Edit the chart configuration",
            isValid: function (workspace) {
                return workspace.isLinkActive("jmx/chart");
            },
            href: function () {
                return "#/jmx/chartEdit";
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function ChartEditController($scope, $location, workspace, jolokia) {
        $scope.selectedAttributes = [];
        $scope.selectedMBeans = [];
        $scope.metrics = {};
        $scope.mbeans = {};

        // TODO move this function to $routeScope
        $scope.size = function (value) {
            if (angular.isObject(value)) {
                return Object.size(value);
            } else if (angular.isArray(value)) {
                return value.length;
            } else
                return 1;
        };

        $scope.canViewChart = function () {
            return $scope.selectedAttributes.length && $scope.selectedMBeans.length && $scope.size($scope.mbeans) > 0 && $scope.size($scope.metrics) > 0;
        };

        $scope.showAttributes = function () {
            return $scope.canViewChart() && $scope.size($scope.metrics) > 1;
        };

        $scope.showElements = function () {
            return $scope.canViewChart() && $scope.size($scope.mbeans) > 1;
        };

        $scope.viewChart = function () {
            // lets add the attributes and mbeans into the URL so we can navigate back to the charts view
            var search = $location.search();

            // if we have selected all attributes, then lets just remove the attribute
            if ($scope.selectedAttributes.length === $scope.size($scope.metrics)) {
                delete search["att"];
            } else {
                search["att"] = $scope.selectedAttributes;
            }

            // if we are on an mbean with no children lets discard an unnecessary parameter
            if ($scope.selectedMBeans.length === $scope.size($scope.mbeans) && $scope.size($scope.mbeans) === 1) {
                delete search["el"];
            } else {
                search["el"] = $scope.selectedMBeans;
            }
            $location.search(search);
            $location.path("jmx/charts");
        };

        $scope.$watch('workspace.selection', render);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(render, 50);
        });

        function render() {
            var node = workspace.selection;
            if (!angular.isDefined(node)) {
                return;
            }

            $scope.selectedAttributes = [];
            $scope.selectedMBeans = [];
            $scope.metrics = {};
            $scope.mbeans = {};
            var mbeanCounter = 0;
            var resultCounter = 0;

            // lets iterate through all the children if the current node is not an mbean
            var children = node.children;
            if (!children || !children.length || node.objectName) {
                children = [node];
            }
            if (children) {
                children.forEach(function (mbeanNode) {
                    var mbean = mbeanNode.objectName;
                    var name = mbeanNode.title;
                    if (name && mbean) {
                        mbeanCounter++;
                        $scope.mbeans[name] = name;

                        // we need to escape the mbean path for list
                        var listKey = escapeMBeanPath(mbean);

                        //var listKey = encodeMBeanPath(mbean);
                        jolokia.list(listKey, onSuccess(function (meta) {
                            var attributes = meta.attr;
                            if (attributes) {
                                for (var key in attributes) {
                                    var value = attributes[key];
                                    if (value) {
                                        var typeName = value['type'];
                                        if (isNumberTypeName(typeName)) {
                                            if (!$scope.metrics[key]) {
                                                //console.log("Number attribute " + key + " for " + mbean);
                                                $scope.metrics[key] = key;
                                            }
                                        }
                                    }
                                }
                                if (++resultCounter >= mbeanCounter) {
                                    // TODO do we need to sort just in case?
                                    // lets look in the search URI to default the selections
                                    var search = $location.search();
                                    var attributeNames = toSearchArgumentArray(search["att"]);
                                    var elementNames = toSearchArgumentArray(search["el"]);
                                    if (attributeNames && attributeNames.length) {
                                        attributeNames.forEach(function (name) {
                                            if ($scope.metrics[name]) {
                                                $scope.selectedAttributes.push(name);
                                            }
                                        });
                                    }
                                    if (elementNames && elementNames.length) {
                                        elementNames.forEach(function (name) {
                                            if ($scope.mbeans[name]) {
                                                $scope.selectedMBeans.push(name);
                                            }
                                        });
                                    }

                                    // default selections if there are none
                                    if ($scope.selectedMBeans.length < 1) {
                                        $scope.selectedMBeans = Object.keys($scope.mbeans);
                                    }
                                    if ($scope.selectedAttributes.length < 1) {
                                        var attrKeys = Object.keys($scope.metrics).sort();
                                        if ($scope.selectedMBeans.length > 1) {
                                            $scope.selectedAttributes = [attrKeys.first()];
                                        } else {
                                            $scope.selectedAttributes = attrKeys;
                                        }
                                    }

                                    // lets update the sizes using jquery as it seems AngularJS doesn't support it
                                    $("#attributes").attr("size", Object.size($scope.metrics));
                                    $("#mbeans").attr("size", Object.size($scope.mbeans));
                                    Core.$apply($scope);
                                }
                            }
                        }));
                    }
                });
            }
        }
    }
    Jmx.ChartEditController = ChartEditController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    // IOperationControllerScope
    function OperationController($scope, workspace, jolokia, $document) {
        $scope.title = $scope.item.humanReadable;
        $scope.desc = $scope.item.desc;
        $scope.operationResult = '';
        $scope.executeIcon = "icon-ok";
        $scope.mode = "text";

        var sanitize = function (args) {
            if (args) {
                args.forEach(function (arg) {
                    switch (arg.type) {
                        case "int":
                        case "long":
                            arg.formType = "number";
                            break;
                        default:
                            arg.formType = "text";
                    }
                });
            }

            return args;
        };

        $scope.args = sanitize($scope.item.args);

        $scope.dump = function (data) {
            console.log(data);
        };

        $scope.ok = function () {
            $scope.operationResult = '';
        };

        $scope.reset = function () {
            if ($scope.item.args) {
                $scope.item.args.forEach(function (arg) {
                    arg.value = "";
                });
            }
            $scope.ok();
        };

        $scope.handleResponse = function (response) {
            $scope.executeIcon = "icon-ok";
            $scope.operationStatus = "success";

            if (response === null || 'null' === response) {
                $scope.operationResult = "Operation Succeeded!";
            } else if (typeof response === 'string') {
                $scope.operationResult = response;
            } else {
                $scope.operationResult = angular.toJson(response, true);
            }

            $scope.mode = CodeEditor.detectTextFormat($scope.operationResult);

            Core.$apply($scope);
        };

        $scope.execute = function () {
            var node = workspace.selection;

            if (!node) {
                return;
            }

            var objectName = node.objectName;

            if (!objectName) {
                return;
            }

            var args = [objectName, $scope.item.name];
            if ($scope.item.args) {
                $scope.item.args.forEach(function (arg) {
                    args.push(arg.value);
                });
            }

            args.push(onSuccess($scope.handleResponse, {
                error: function (response) {
                    $scope.executeIcon = "icon-ok";
                    $scope.operationStatus = "error";
                    var error = response.error;
                    $scope.operationResult = error;
                    var stacktrace = response.stacktrace;
                    if (stacktrace) {
                        //console.log(stacktrace);
                        $scope.operationResult = stacktrace;
                    }
                    Core.$apply($scope);
                }
            }));

            $scope.executeIcon = "icon-spinner icon-spin";
            var fn = jolokia.execute;
            fn.apply(jolokia, args);
        };
    }
    Jmx.OperationController = OperationController;

    function OperationsController($scope, $routeParams, workspace, jolokia) {
        $scope.operations = {};
        $scope.methodFilter = '';

        var sanitize = function (value) {
            for (var item in value) {
                item = "" + item;
                value[item].name = item;
                value[item].humanReadable = humanizeValue(item);
            }
            return value;
        };

        var asQuery = function (node) {
            var path = escapeMBeanPath(node);
            var query = {
                type: "LIST",
                method: "post",
                path: path,
                ignoreErrors: true
            };
            return query;
        };

        $scope.isOperationsEmpty = function () {
            return $.isEmptyObject($scope.operations);
        };

        $scope.doFilter = function (item) {
            Jmx.log.debug("item: ", item);
            if (Core.isBlank($scope.methodFilter)) {
                return true;
            }
            if (item.name.toLowerCase().has($scope.methodFilter.toLowerCase()) || item.humanReadable.toLowerCase().has($scope.methodFilter.toLowerCase())) {
                return true;
            }
            return false;
        };

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(render, 50);
        });

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;
            render();
        });

        function render() {
            var node = workspace.selection;
            if (!node) {
                return;
            }

            var objectName = node.objectName;
            if (!objectName) {
                return;
            }

            var query = asQuery(objectName);

            var update_values = function (response) {
                var ops = response.value.op;

                var answer = {};

                var getArgs = function (args) {
                    return "(" + args.map(function (arg) {
                        return arg.type;
                    }).join() + ")";
                };

                angular.forEach(ops, function (value, key) {
                    if (angular.isArray(value)) {
                        angular.forEach(value, function (value, index) {
                            answer[key + getArgs(value.args)] = value;
                        });
                    } else {
                        answer[key + getArgs(value.args)] = value;
                    }
                });
                $scope.operations = sanitize(answer);
                Core.$apply($scope);
            };

            jolokia.request(query, onSuccess(update_values, {
                error: function (response) {
                    notification('error', 'Failed to query available operations: ' + response.error);
                }
            }));
        }

        render();
    }
    Jmx.OperationsController = OperationsController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function AreaChartController($scope, $routeParams, jolokia, $templateCache, localStorage, $element) {
        $scope.mbean = $routeParams['mbean'];
        $scope.attribute = $routeParams['attribute'];
        $scope.duration = localStorage['updateRate'];

        $scope.width = 308;
        $scope.height = 296;

        $scope.template = "";

        $scope.entries = [];

        $scope.data = {
            entries: $scope.entries
        };

        $scope.req = [{ type: 'read', mbean: $scope.mbean, attribute: $scope.attribute }];

        $scope.render = function (response) {
            $scope.entries.push({
                time: response.timestamp,
                count: response.value
            });

            $scope.entries = $scope.entries.last(15);

            if ($scope.template === "") {
                $scope.template = $templateCache.get("areaChart");
            }

            $scope.data = {
                _type: "date_histogram",
                entries: $scope.entries
            };

            Core.$apply($scope);
        };

        Core.register(jolokia, $scope, $scope.req, onSuccess($scope.render));
    }
    Jmx.AreaChartController = AreaChartController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    Jmx.log = Logger.get("JMX");

    var attributesToolBars = {};

    Jmx.lazyLoaders = {};

    function findLazyLoadingFunction(workspace, folder) {
        var factories = workspace.jmxTreeLazyLoadRegistry[folder.domain];
        var lazyFunction = null;
        if (factories && factories.length) {
            angular.forEach(factories, function (customLoader) {
                if (!lazyFunction) {
                    lazyFunction = customLoader(folder);
                }
            });
        }
        return lazyFunction;
    }
    Jmx.findLazyLoadingFunction = findLazyLoadingFunction;

    function registerLazyLoadHandler(domain, lazyLoaderFactory) {
        if (!Jmx.lazyLoaders) {
            Jmx.lazyLoaders = {};
        }
        var array = Jmx.lazyLoaders[domain];
        if (!array) {
            array = [];
            Jmx.lazyLoaders[domain] = array;
        }
        array.push(lazyLoaderFactory);
    }
    Jmx.registerLazyLoadHandler = registerLazyLoadHandler;

    function unregisterLazyLoadHandler(domain, lazyLoaderFactory) {
        if (Jmx.lazyLoaders) {
            var array = Jmx.lazyLoaders[domain];
            if (array) {
                array.remove(lazyLoaderFactory);
            }
        }
    }
    Jmx.unregisterLazyLoadHandler = unregisterLazyLoadHandler;

    /**
    * Registers a toolbar template for the given plugin name, jmxDomain.
    * @method addAttributeToolBar
    * @for Jmx
    * @param {String} pluginName used so that we can later on remove this function when the plugin is removed
    * @param {String} jmxDomain the JMX domain to avoid having to evaluate too many functions on each selection
    * @param {Function} fn the function used to decide which attributes tool bar should be used for the given select
    */
    function addAttributeToolBar(pluginName, jmxDomain, fn) {
        var array = attributesToolBars[jmxDomain];
        if (!array) {
            array = [];
            attributesToolBars[jmxDomain] = array;
        }
        array.push(fn);
    }
    Jmx.addAttributeToolBar = addAttributeToolBar;

    /**
    * Try find a custom toolbar HTML template for the given selection or returns the default value
    * @method getAttributeToolbar
    * @for Jmx
    * @param {Core.NodeSelection} node
    * @param {String} defaultValue
    */
    function getAttributeToolBar(node, defaultValue) {
        if (typeof defaultValue === "undefined") { defaultValue = "app/jmx/html/attributeToolBar.html"; }
        var answer = null;
        var jmxDomain = (node) ? node.domain : null;
        if (jmxDomain) {
            var array = attributesToolBars[jmxDomain];
            if (array) {
                for (var idx in array) {
                    var fn = array[idx];
                    answer = fn(node);
                    if (answer)
                        break;
                }
            }
        }
        return (answer) ? answer : defaultValue;
    }
    Jmx.getAttributeToolBar = getAttributeToolBar;

    function updateTreeSelectionFromURL($location, treeElement, activateIfNoneSelected) {
        if (typeof activateIfNoneSelected === "undefined") { activateIfNoneSelected = false; }
        updateTreeSelectionFromURLAndAutoSelect($location, treeElement, null, activateIfNoneSelected);
    }
    Jmx.updateTreeSelectionFromURL = updateTreeSelectionFromURL;

    function updateTreeSelectionFromURLAndAutoSelect($location, treeElement, autoSelect, activateIfNoneSelected) {
        if (typeof activateIfNoneSelected === "undefined") { activateIfNoneSelected = false; }
        var dtree = treeElement.dynatree("getTree");
        if (dtree) {
            var node = null;
            var key = $location.search()['nid'];
            if (key) {
                try  {
                    node = dtree.activateKey(key);
                } catch (e) {
                    // tree not visible we suspect!
                }
            }
            if (node) {
                node.expand(true);
            } else {
                if (!treeElement.dynatree("getActiveNode")) {
                    // lets expand the first node
                    var root = treeElement.dynatree("getRoot");
                    var children = root ? root.getChildren() : null;
                    if (children && children.length) {
                        var first = children[0];
                        first.expand(true);

                        // invoke any auto select function, and use its result as new first, if any returned
                        if (autoSelect) {
                            var result = autoSelect(first);
                            if (result) {
                                first = result;
                            }
                        }
                        if (activateIfNoneSelected) {
                            first.expand();
                            first.activate();
                        }
                    } else {
                        /*
                        causes NPE :)
                        
                        var first = children[0];
                        first.expand(true);
                        if (activateIfNoneSelected) {
                        first.activate();
                        }
                        */
                    }
                }
            }
        }
    }
    Jmx.updateTreeSelectionFromURLAndAutoSelect = updateTreeSelectionFromURLAndAutoSelect;

    function getUniqueTypeNames(children) {
        var typeNameMap = {};
        angular.forEach(children, function (mbean) {
            var typeName = mbean.typeName;
            if (typeName) {
                typeNameMap[typeName] = mbean;
            }
        });

        // only query if all the typenames are the same
        var typeNames = Object.keys(typeNameMap);
        return typeNames;
    }
    Jmx.getUniqueTypeNames = getUniqueTypeNames;

    function enableTree($scope, $location, workspace, treeElement, children, redraw, onActivateFn) {
        if (typeof redraw === "undefined") { redraw = false; }
        if (typeof onActivateFn === "undefined") { onActivateFn = null; }
        //$scope.workspace = workspace;
        if (treeElement.length) {
            if (!onActivateFn) {
                onActivateFn = function (node) {
                    var data = node.data;

                    //$scope.select(data);
                    workspace.updateSelectionNode(data);
                    Core.$apply($scope);
                };
            }
            workspace.treeElement = treeElement;
            treeElement.dynatree({
                /*
                * The event handler called when a different node in the tree is selected
                */
                onActivate: onActivateFn,
                onLazyRead: function (treeNode) {
                    var folder = treeNode.data;
                    var plugin = null;
                    if (folder) {
                        plugin = Jmx.findLazyLoadingFunction(workspace, folder);
                    }
                    if (plugin) {
                        console.log("Lazy loading folder " + folder.title);
                        var oldChildren = folder.childen;
                        plugin(workspace, folder, function () {
                            treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                            var newChildren = folder.children;
                            if (newChildren !== oldChildren) {
                                treeNode.removeChildren();
                                angular.forEach(newChildren, function (newChild) {
                                    treeNode.addChild(newChild);
                                });
                            }
                        });
                    } else {
                        treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                    }
                },
                onClick: function (node, event) {
                    if (event["metaKey"]) {
                        event.preventDefault();
                        var url = $location.absUrl();
                        if (node && node.data) {
                            var key = node.data["key"];
                            if (key) {
                                var hash = $location.search();
                                hash["nid"] = key;

                                // TODO this could maybe be a generic helper function?
                                // lets trim after the ?
                                var idx = url.indexOf('?');
                                if (idx <= 0) {
                                    url += "?";
                                } else {
                                    url = url.substring(0, idx + 1);
                                }
                                url += $.param(hash);
                            }
                        }
                        window.open(url, '_blank');
                        window.focus();
                        return false;
                    }
                    return true;
                },
                persist: false,
                debugLevel: 0,
                //children: $scope.workspace.tree.children
                children: children
            });

            if (redraw) {
                workspace.redrawTree();
            }
        }
    }
    Jmx.enableTree = enableTree;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function AttributeController($scope, jolokia) {
        $scope.init = function (mbean, attribute) {
            $scope.mbean = mbean;
            $scope.attribute = attribute;

            if (angular.isDefined($scope.mbean) && angular.isDefined($scope.attribute)) {
                Core.register(jolokia, $scope, {
                    type: 'read', mbean: $scope.mbean, attribute: $scope.attribute
                }, onSuccess(render));
            }
        };

        function render(response) {
            if (!Object.equal($scope.data, response.value)) {
                $scope.data = safeNull(response.value);
                Core.$apply($scope);
            }
        }
    }
    Jmx.AttributeController = AttributeController;

    function AttributeChartController($scope, jolokia, $document) {
        $scope.init = function (mbean, attribute) {
            $scope.mbean = mbean;
            $scope.attribute = attribute;

            if (angular.isDefined($scope.mbean) && angular.isDefined($scope.attribute)) {
                Core.register(jolokia, $scope, {
                    type: 'read', mbean: $scope.mbean, attribute: $scope.attribute
                }, onSuccess(render));
            }
        };

        function render(response) {
            if (!angular.isDefined($scope.chart)) {
                $scope.chart = $($document.find("#" + $scope.attribute)[0]);
                if ($scope.chart) {
                    $scope.width = $scope.chart.width();
                }
            }

            if (!angular.isDefined($scope.context)) {
                console.log("Got: ", response);

                $scope.context = cubism.context().serverDelay(0).clientDelay(0).step(1000).size($scope.width);
                $scope.jcontext = $scope.context.jolokia(jolokia);

                $scope.metrics = [];

                Object.extended(response.value).keys(function (key, value) {
                    $scope.metrics.push($scope.jcontext.metric({
                        type: 'read',
                        mbean: $scope.mbean,
                        attribute: $scope.attribute,
                        path: key
                    }, $scope.attribute));
                });

                d3.select("#" + $scope.attribute).call(function (div) {
                    div.append("div").data($scope.metrics).call($scope.context.horizon());
                });

                // let cubism take over at this point...
                Core.unregister(jolokia, $scope);
                Core.$apply($scope);
            }
        }
    }
    Jmx.AttributeChartController = AttributeChartController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function createDashboardLink(widgetType, widget) {
        var href = "#" + widgetType.route;
        var routeParams = angular.toJson(widget);
        var title = widget.title;
        var size = angular.toJson({
            size_x: widgetType.size_x,
            size_y: widgetType.size_y
        });

        return "/dashboard/add?tab=dashboard" + "&href=" + encodeURIComponent(href) + "&size=" + encodeURIComponent(size) + "&title=" + encodeURIComponent(title) + "&routeParams=" + encodeURIComponent(routeParams);
    }
    Jmx.createDashboardLink = createDashboardLink;

    function getWidgetType(widget) {
        return Jmx.jmxWidgetTypes.find(function (type) {
            return type.type === widget.type;
        });
    }
    Jmx.getWidgetType = getWidgetType;

    Jmx.jmxWidgetTypes = [
        {
            type: "donut",
            icon: "icon-smile",
            route: "/jmx/widget/donut",
            size_x: 2,
            size_y: 2,
            title: "Add Donut chart to Dashboard"
        },
        {
            type: "area",
            icon: "icon-bar-chart",
            route: "/jmx/widget/area",
            size_x: 4,
            size_y: 2,
            title: "Add Area chart to Dashboard"
        }
    ];

    Jmx.jmxWidgets = [
        {
            type: "donut",
            title: "Java Heap Memory",
            mbean: "java.lang:type=Memory",
            attribute: "HeapMemoryUsage",
            total: "Max",
            terms: "Used",
            remaining: "Free"
        },
        {
            type: "donut",
            title: "Java Non Heap Memory",
            mbean: "java.lang:type=Memory",
            attribute: "NonHeapMemoryUsage",
            total: "Max",
            terms: "Used",
            remaining: "Free"
        },
        {
            type: "donut",
            title: "File Descriptor Usage",
            mbean: "java.lang:type=OperatingSystem",
            total: "MaxFileDescriptorCount",
            terms: "OpenFileDescriptorCount",
            remaining: "Free"
        },
        {
            type: "donut",
            title: "Loaded Clases",
            mbean: "java.lang:type=ClassLoading",
            total: "TotalLoadedClassCount",
            terms: "LoadedClassCount,UnloadedClassCount",
            remaining: "-"
        },
        {
            type: "donut",
            title: "Swap Size",
            mbean: "java.lang:type=OperatingSystem",
            total: "TotalSwapSpaceSize",
            terms: "FreeSwapSpaceSize",
            remaining: "Used Swap"
        },
        {
            type: "area",
            title: "Process CPU Time",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "ProcessCpuTime"
        },
        {
            type: "area",
            title: "Process CPU Load",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "ProcessCpuLoad"
        },
        {
            type: "area",
            title: "System CPU Load",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "SystemCpuLoad"
        },
        {
            type: "area",
            title: "System CPU Time",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "SystemCpuTime"
        }
    ];
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function DonutChartController($scope, $routeParams, jolokia, $templateCache) {
        /*
        console.log("routeParams: ", $routeParams);
        
        
        // using multiple attributes
        $scope.mbean = "java.lang:type=OperatingSystem";
        $scope.total = "MaxFileDescriptorCount";
        $scope.terms = "OpenFileDescriptorCount";
        */
        // using a single attribute with multiple paths
        /*
        $scope.mbean = "java.lang:type=Memory";
        $scope.total = "Max";
        $scope.attribute = "HeapMemoryUsage";
        $scope.terms = "Used";
        */
        $scope.mbean = $routeParams['mbean'];
        $scope.total = $routeParams['total'];
        $scope.attribute = $routeParams['attribute'];
        $scope.terms = $routeParams['terms'];

        $scope.remainder = $routeParams['remaining'];

        $scope.template = "";
        $scope.termsArray = $scope.terms.split(",");

        $scope.data = {
            total: 0,
            terms: []
        };

        if (!$scope.attribute) {
            $scope.reqs = [{ type: 'read', mbean: $scope.mbean, attribute: $scope.total }];

            $scope.termsArray.forEach(function (term) {
                $scope.reqs.push({ type: 'read', mbean: $scope.mbean, attribute: term });
                $scope.data.terms.push({
                    term: term,
                    count: 0
                });
            });
        } else {
            var terms = $scope.termsArray.include($scope.total);
            $scope.reqs = [{ type: 'read', mbean: $scope.mbean, attribute: $scope.attribute, paths: terms.join(",") }];

            $scope.termsArray.forEach(function (term) {
                $scope.data.terms.push({
                    term: term,
                    count: 0
                });
            });
        }

        if ($scope.remainder && $scope.remainder !== "-") {
            $scope.data.terms.push({
                term: $scope.remainder,
                count: 0
            });
        }

        /*
        $scope.data = {
        total: 100,
        terms: [{
        term: "One",
        count: 25
        }, {
        term: "Two",
        count: 75
        }]
        };
        */
        $scope.render = function (response) {
            //console.log("got: ", response);
            var freeTerm = null;
            if ($scope.remainder && $scope.remainder !== "-") {
                freeTerm = $scope.data.terms.find(function (term) {
                    return term.term === $scope.remainder;
                });
            }

            if (!$scope.attribute) {
                if (response.request.attribute === $scope.total) {
                    $scope.data.total = response.value;
                } else {
                    var term = $scope.data.terms.find(function (term) {
                        return term.term === response.request.attribute;
                    });
                    if (term) {
                        term.count = response.value;
                    }

                    if (freeTerm) {
                        freeTerm.count = $scope.data.total;
                        $scope.data.terms.forEach(function (term) {
                            if (term.term !== $scope.remainder) {
                                freeTerm.count = freeTerm.count - term.count;
                            }
                        });
                    }
                }
            } else {
                if (response.request.attribute === $scope.attribute) {
                    $scope.data.total = response.value[$scope.total.toLowerCase()];

                    $scope.data.terms.forEach(function (term) {
                        if (term.term !== $scope.remainder) {
                            term.count = response.value[term.term.toLowerCase()];
                        }
                    });

                    if (freeTerm) {
                        freeTerm.count = $scope.data.total;
                        $scope.data.terms.forEach(function (term) {
                            if (term.term !== $scope.remainder) {
                                freeTerm.count = freeTerm.count - term.count;
                            }
                        });
                    }
                }
            }
            if ($scope.template === "") {
                $scope.template = $templateCache.get("donut");
            }

            // console.log("Data: ", $scope.data);
            $scope.data = Object.clone($scope.data);
            Core.$apply($scope);
        };

        Core.register(jolokia, $scope, $scope.reqs, onSuccess($scope.render));
    }
    Jmx.DonutChartController = DonutChartController;
})(Jmx || (Jmx = {}));
/**
* @module Source
*/
var Source;
(function (Source) {
    Source.log = Logger.get("Source");

    /**
    * @method getInsightMBean
    * @for Source
    * @param {Core.Workspace} workspace
    * @returns {*}
    */
    function getInsightMBean(workspace) {
        var mavenStuff = workspace.mbeanTypesToDomain["LogQuery"] || {};
        var insight = mavenStuff["org.fusesource.insight"] || {};
        var mbean = insight.objectName;
        return mbean;
    }
    Source.getInsightMBean = getInsightMBean;

    /**
    * @method createBreadcrumbLinks
    * @for Source
    * @param {String} mavenCoords
    * @param {pathName} pathName
    * @returns {Array}
    */
    function createBreadcrumbLinks(mavenCoords, pathName) {
        var linkPrefix = "#/source/index/" + mavenCoords;
        var answer = [{ href: linkPrefix, name: "root" }];
        if (pathName) {
            var pathNames = pathName.split("/");
            var fullPath = "";
            angular.forEach(pathNames, function (path) {
                fullPath += "/" + path;
                var href = linkPrefix + fullPath;
                if (!path.isBlank()) {
                    answer.push({ href: href, name: path || "/", fileName: "/" + fullPath });
                }
            });
        }
        return answer;
    }
    Source.createBreadcrumbLinks = createBreadcrumbLinks;
})(Source || (Source = {}));
/**
* @module Source
*/
var Source;
(function (Source) {
    function JavaDocController($scope, $location, $routeParams, workspace, fileExtensionTypeRegistry, jolokia) {
        $scope.pageId = Wiki.pageId($routeParams, $location);
        var mavenCoords = $routeParams["mavenCoords"];
        var fileName = $scope.pageId;

        $scope.loadingMessage = "Loading javadoc code for file <b>" + fileName + "</b> from artifacts <b>" + mavenCoords + "</b>";

        $scope.breadcrumbs = [];

        // TODO load breadcrumbs
        // $scope.breadcrumbs.push({href: "#" + loc, name: name});
        $scope.$watch('workspace.tree', function () {
            if (!$scope.git && Git.getGitMBean(workspace)) {
                // lets do this asynchronously to avoid Error: $digest already in progress
                //console.log("Reloading the view as we now seem to have a git mbean!");
                setTimeout(updateView, 50);
            }
        });

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateView, 50);
        });

        function viewContents(response) {
            $scope.source = response;
            $scope.loadingMessage = null;
            if (!response) {
                var time = new Date().getTime();
                if (!$scope.lastErrorTime || time - $scope.lastErrorTime > 3000) {
                    $scope.lastErrorTime = time;
                    notification("error", "Could not download the source code for the maven artifacts: " + mavenCoords);
                }
            }
            Core.$apply($scope);
        }

        function updateView() {
            var mbean = Source.getInsightMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "getJavaDoc", mavenCoords, fileName, onSuccess(viewContents));
            }
        }
    }
    Source.JavaDocController = JavaDocController;
})(Source || (Source = {}));
/**
* @module Source
*/
var Source;
(function (Source) {
    function SourceController($scope, $location, $routeParams, workspace, fileExtensionTypeRegistry, jolokia) {
        $scope.pageId = Wiki.pageId($routeParams, $location);
        $scope.format = Wiki.fileFormat($scope.pageId, fileExtensionTypeRegistry);
        var lineNumber = $location.search()["line"] || 1;
        var mavenCoords = $routeParams["mavenCoords"];
        var className = $routeParams["className"] || "";
        var fileName = $scope.pageId || "/";
        var classNamePath = className.replace(/\./g, '/');

        $scope.loadingMessage = "Loading source code for class <b>" + className + "</b> from artifacts <b>" + mavenCoords + "</b>";

        //console.log("Source format is " + $scope.format + " line " + lineNumber + " className " + className + " file " + fileName);
        $scope.breadcrumbs = [];

        var idx = fileName.lastIndexOf('/');
        var path = "/";
        var name = fileName;
        if (idx > 0) {
            path = fileName.substring(0, idx);
            name = fileName.substring(idx + 1);
        } else if (className && className.indexOf('.') > 0) {
            path = classNamePath;
            idx = path.lastIndexOf('/');
            if (idx > 0) {
                name = path.substring(idx + 1);
                path = path.substring(0, idx);
            }
        }
        $scope.breadcrumbs = Source.createBreadcrumbLinks(mavenCoords, path);
        $scope.breadcrumbs.push({ href: $location.url(), name: name, active: true });

        $scope.javaDocLink = function () {
            var path = classNamePath;
            if (!path && fileName && fileName.endsWith(".java")) {
                path = fileName.substring(0, fileName.length - 5);
            }
            if (path) {
                return "javadoc/" + mavenCoords + "/" + path + ".html";
            }
            return null;
        };

        function updateLineSelection() {
            var codeMirror = $scope.codeMirror;
            if (codeMirror && lineNumber) {
                var line = lineNumber - 1;
                var lineText = codeMirror.getLine(line);
                var endChar = (lineText) ? lineText.length : 1000;
                var start = { line: line, ch: 0 };
                var end = { line: line, ch: endChar };
                codeMirror.scrollIntoView(start);
                codeMirror.setCursor(start);
                codeMirror.setSelection(start, end);
                codeMirror.refresh();
                codeMirror.focus();
            }
        }

        $scope.$watch('workspace.tree', function (oldValue, newValue) {
            if (!$scope.git && Git.getGitMBean(workspace)) {
                // lets do this asynchronously to avoid Error: $digest already in progress
                //console.log("Reloading the view as we now seem to have a git mbean!");
                setTimeout(maybeUpdateView, 50);
            }
        });

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(maybeUpdateView, 50);
        });

        function viewContents(response) {
            if (response) {
                Source.log.debug("Downloaded file for the maven artifact: " + mavenCoords);
                $scope.source = response;
                $scope.loadingMessage = null;
            } else {
                // we could not download the source code
                $scope.source = null;
                $scope.loadingMessage = "Cannot download file, please see logging console for details.";
                Source.log.error("Failed to download the source code for the Maven artifact: ", mavenCoords);
            }
            Core.$apply($scope);

            // lets update the line selection asynchronously to check we've properly loaded by now
            setTimeout(updateLineSelection, 100);
        }

        function updateView() {
            var mbean = Source.getInsightMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "getSource", mavenCoords, className, fileName, {
                    success: viewContents,
                    error: function (response) {
                        Source.log.error("Failed to download the source code for the Maven artifact: ", mavenCoords);
                        Source.log.info("Stack trace: ", response.stacktrace);
                        $scope.loadingMessage = "Cannot not download file, please see logging console for details.";
                        Core.$apply($scope);
                    }
                });
            }
        }

        var maybeUpdateView = Core.throttled(updateView, 1000);
        setTimeout(maybeUpdateView, 50);
    }
    Source.SourceController = SourceController;
})(Source || (Source = {}));
/**
* @module Source
*/
var Source;
(function (Source) {
    function IndexController($scope, $location, $routeParams, workspace, jolokia) {
        $scope.pageId = Wiki.pageId($routeParams, $location);
        $scope.mavenCoords = $routeParams["mavenCoords"];
        var fileName = $scope.pageId;
        if (fileName === '/') {
            fileName = undefined;
        }

        $scope.loadingMessage = "Loading source code from artifacts <b>" + $scope.mavenCoords + "</b>";

        createBreadcrumbs();

        $scope.setFileName = function (breadcrumb) {
            fileName = Core.trimLeading(breadcrumb.fileName, "/");
            fileName = Core.trimLeading(fileName, "/");
            console.log("selected fileName '" + fileName + "'");
            createBreadcrumbs();
            filterFileNames();
        };

        $scope.$watch('workspace.tree', function (newValue, oldValue) {
            if (!$scope.git && Git.getGitMBean(workspace)) {
                // lets do this asynchronously to avoid Error: $digest already in progress
                //console.log("Reloading the view as we now seem to have a git mbean!");
                setTimeout(maybeUpdateView, 50);
            }
        });

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(maybeUpdateView, 50);
        });

        function filterFileNames() {
            if (fileName) {
                $scope.sourceFiles = $scope.allSourceFiles.filter(function (n) {
                    return n && n.startsWith(fileName);
                }).map(function (n) {
                    return n.substring(fileName.length + 1);
                }).filter(function (n) {
                    return n;
                });
            } else {
                $scope.sourceFiles = $scope.allSourceFiles;
            }
        }

        $scope.sourceLinks = function (aFile) {
            var name = aFile;
            var paths = null;
            var idx = aFile.lastIndexOf('/');
            if (idx > 0) {
                name = aFile.substring(idx + 1);
                paths = aFile.substring(0, idx);
            }
            var answer = "";
            var fullName = fileName || "";
            if (paths) {
                angular.forEach(paths.split("/"), function (path) {
                    if (fullName) {
                        fullName += "/";
                    }
                    fullName += path;
                    answer += "<a href='#/source/index/" + $scope.mavenCoords + "/" + fullName + "'>" + path + "</a>/";
                });
            }
            answer += "<a href='#/source/view/" + $scope.mavenCoords + "/" + fullName + "/" + name + "'>" + name + "</a>";
            return answer;
        };

        function createBreadcrumbs() {
            $scope.breadcrumbs = Source.createBreadcrumbLinks($scope.mavenCoords, fileName);
            angular.forEach($scope.breadcrumbs, function (breadcrumb) {
                breadcrumb.active = false;
            });
            $scope.breadcrumbs.last().active = true;
        }

        function viewContents(response) {
            if (response) {
                $scope.allSourceFiles = response.split("\n").map(function (n) {
                    return n.trim();
                }).filter(function (n) {
                    return n;
                });
            } else {
                $scope.allSourceFiles = [];
            }
            filterFileNames();
            $scope.loadingMessage = null;
            Core.$apply($scope);
        }

        function updateView() {
            if (!$scope.mavenCoords) {
                return;
            }
            var mbean = Source.getInsightMBean(workspace);
            Source.log.debug("In update view, mbean: ", mbean);
            if (mbean) {
                jolokia.execute(mbean, "getSource", $scope.mavenCoords, null, "/", {
                    success: viewContents,
                    error: function (response) {
                        Source.log.error("Failed to download the source code for the maven artifact: ", $scope.mavenCoords);
                        Source.log.info("Stack trace: ", response.stacktrace);
                        $scope.loadingMessage = "Could not download index, please see console for details";
                        Core.$apply($scope);
                    }
                });
            }
        }

        var maybeUpdateView = Core.throttled(updateView, 1000);
        setTimeout(maybeUpdateView, 50);
    }
    Source.IndexController = IndexController;
})(Source || (Source = {}));
/**
* @module Source
* @main Source
*/
var Source;
(function (Source) {
    var pluginName = 'source';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore', 'wiki']).config(function ($routeProvider) {
        $routeProvider.when('/source/index/:mavenCoords', { templateUrl: 'app/source/html/index.html' }).when('/source/index/:mavenCoords/*page', { templateUrl: 'app/source/html/index.html' }).when('/source/view/:mavenCoords/class/:className/*page', { templateUrl: 'app/source/html/source.html' }).when('/source/view/:mavenCoords/*page', { templateUrl: 'app/source/html/source.html' }).when('/source/javadoc/:mavenCoords/*page', { templateUrl: 'app/source/html/javadoc.html' });
    }).run(function ($location, workspace, viewRegistry, jolokia, localStorage, layoutFull, helpRegistry) {
        viewRegistry['source'] = layoutFull;
        helpRegistry.addUserDoc('source', 'app/source/doc/help.md');
    });

    hawtioPluginLoader.addModule(pluginName);
})(Source || (Source = {}));
/**
* @module Tree
* @main Tree
*/
var Tree;
(function (Tree) {
    var pluginName = 'tree';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).directive('hawtioTree', function (workspace, $timeout, $location, $filter, $compile) {
        // return the directive link function. (compile function not needed)
        return function (scope, element, attrs) {
            var tree = null;
            var data = null;
            var widget = null;
            var timeoutId = null;
            var onSelectFn = lookupFunction("onselect");
            var onDragStartFn = lookupFunction("ondragstart");
            var onDragEnterFn = lookupFunction("ondragenter");
            var onDropFn = lookupFunction("ondrop");

            function lookupFunction(attrName) {
                var answer = null;
                var fnName = attrs[attrName];
                if (fnName) {
                    answer = Core.pathGet(scope, fnName);
                    if (!angular.isFunction(answer)) {
                        answer = null;
                    }
                }
                return answer;
            }

            // watch the expression, and update the UI on change.
            var data = attrs.hawtioTree;
            var queryParam = data;

            scope.$watch(data, onWidgetDataChange);

            // lets add a separate event so we can force updates
            // if we find cases where the delta logic doesn't work
            scope.$on("hawtio.tree." + data, function (args) {
                var value = Core.pathGet(scope, data);
                onWidgetDataChange(value);
            });

            // listen on DOM destroy (removal) event, and cancel the next UI update
            // to prevent updating ofter the DOM element was removed.
            element.bind('$destroy', function () {
                $timeout.cancel(timeoutId);
            });

            updateLater(); // kick off the UI update process.

            // used to update the UI
            function updateWidget() {
                // console.log("updating the grid!!");
                Core.$applyNowOrLater(scope);
            }

            function onWidgetDataChange(value) {
                tree = value;
                if (tree && !widget) {
                    // lets find a child table element
                    // or lets add one if there's not one already
                    var treeElement = $(element);
                    var children = Core.asArray(tree);
                    var hideRoot = attrs["hideroot"];
                    if ("true" === hideRoot) {
                        children = tree['children'];
                    }
                    var config = {
                        clickFolderMode: 3,
                        /*
                        * The event handler called when a different node in the tree is selected
                        */
                        onActivate: function (node) {
                            var data = node.data;
                            if (onSelectFn) {
                                onSelectFn(data, node);
                            } else {
                                workspace.updateSelectionNode(data);
                            }
                            Core.$apply(scope);
                        },
                        /*
                        onLazyRead: function(treeNode) {
                        var folder = treeNode.data;
                        var plugin = null;
                        if (folder) {
                        plugin = Jmx.findLazyLoadingFunction(workspace, folder);
                        }
                        if (plugin) {
                        console.log("Lazy loading folder " + folder.title);
                        var oldChildren = folder.childen;
                        plugin(workspace, folder, () => {
                        treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                        var newChildren = folder.children;
                        if (newChildren !== oldChildren) {
                        treeNode.removeChildren();
                        angular.forEach(newChildren, newChild => {
                        treeNode.addChild(newChild);
                        });
                        }
                        });
                        } else {
                        treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                        }
                        },
                        */
                        onClick: function (node, event) {
                            if (event["metaKey"]) {
                                event.preventDefault();
                                var url = $location.absUrl();
                                if (node && node.data) {
                                    var key = node.data["key"];
                                    if (key) {
                                        var hash = $location.search();
                                        hash[queryParam] = key;

                                        // TODO this could maybe be a generic helper function?
                                        // lets trim after the ?
                                        var idx = url.indexOf('?');
                                        if (idx <= 0) {
                                            url += "?";
                                        } else {
                                            url = url.substring(0, idx + 1);
                                        }
                                        url += $.param(hash);
                                    }
                                }
                                window.open(url, '_blank');
                                window.focus();
                                return false;
                            }
                            return true;
                        },
                        persist: false,
                        debugLevel: 0,
                        children: children,
                        dnd: {
                            onDragStart: onDragStartFn ? onDragStartFn : function (node) {
                                /* This function MUST be defined to enable dragging for the tree.
                                *  Return false to cancel dragging of node.
                                */
                                console.log("onDragStart!");
                                return true;
                            },
                            onDragEnter: onDragEnterFn ? onDragEnterFn : function (node, sourceNode) {
                                console.log("onDragEnter!");
                                return true;
                            },
                            onDrop: onDropFn ? onDropFn : function (node, sourceNode, hitMode) {
                                console.log("onDrop!");

                                /* This function MUST be defined to enable dropping of items on
                                *  the tree.
                                */
                                sourceNode.move(node, hitMode);
                                return true;
                            }
                        }
                    };
                    if (!onDropFn && !onDragEnterFn && !onDragStartFn) {
                        delete config["dnd"];
                    }
                    widget = treeElement.dynatree(config);

                    var activatedNode = false;
                    var activateNodeName = attrs["activatenodes"];
                    if (activateNodeName) {
                        var values = scope[activateNodeName];
                        var tree = treeElement.dynatree("getTree");
                        if (values && tree) {
                            angular.forEach(Core.asArray(values), function (value) {
                                //tree.selectKey(value, true);
                                tree.activateKey(value);
                                activatedNode = true;
                            });
                        }
                    }
                    var root = treeElement.dynatree("getRoot");
                    if (root) {
                        var onRootName = attrs["onroot"];
                        if (onRootName) {
                            var fn = scope[onRootName];
                            if (fn) {
                                fn(root);
                            }
                        }

                        // select and activate first child if we have not activated any others
                        if (!activatedNode) {
                            var children = root['getChildren']();
                            if (children && children.length) {
                                var child = children[0];
                                child.expand(true);
                                child.activate(true);
                            }
                        }
                    }
                }
                updateWidget();
            }

            // schedule update in one second
            function updateLater() {
                // save the timeoutId for canceling
                timeoutId = $timeout(function () {
                    updateWidget(); // update DOM
                }, 300);
            }
        };
    }).run(function (helpRegistry) {
        helpRegistry.addDevDoc(pluginName, 'app/tree/doc/developer.md');
    });

    hawtioPluginLoader.addModule(pluginName);
})(Tree || (Tree = {}));
var Camel;
(function (Camel) {
    function SendMessageController($route, $scope, $element, $timeout, workspace, jolokia, localStorage, $location) {
        var log = Logger.get("Camel");

        log.info("Loaded page!");

        $scope.noCredentials = false;
        $scope.showChoose = false;
        $scope.profileFileNames = [];
        $scope.profileFileNameToProfileId = {};
        $scope.selectedFiles = {};
        $scope.container = {};

        // bind model values to search params...
        Core.bindModelToSearchParam($scope, $location, "tab", "subtab", "compose");
        Core.bindModelToSearchParam($scope, $location, "searchText", "q", "");

        // only reload the page if certain search parameters change
        Core.reloadWhenParametersChange($route, $scope, $location);

        if ($location.path().has('activemq')) {
            if (!localStorage['activemqUserName'] || !localStorage['activemqPassword']) {
                $scope.noCredentials = true;
            }
        }

        var LANGUAGE_FORMAT_PREFERENCE = "defaultLanguageFormat";
        var sourceFormat = workspace.getLocalStorage(LANGUAGE_FORMAT_PREFERENCE) || "javascript";
        $scope.message = "\n\n\n\n";

        // TODO Remove this if possible
        $scope.codeMirror = undefined;
        var options = {
            mode: {
                name: sourceFormat
            },
            // Quick hack to get the codeMirror instance.
            onChange: function (codeMirror) {
                if (!$scope.codeMirror) {
                    $scope.codeMirror = codeMirror;
                }
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);

        $scope.headers = [];

        $scope.addHeader = function () {
            $scope.headers.push({ name: "", value: "" });

            // lets set the focus to the last header
            if ($element) {
                $timeout(function () {
                    var lastHeader = $element.find("input.headerName").last();
                    lastHeader.focus();
                }, 100);
            }
        };

        $scope.removeHeader = function (header) {
            $scope.headers = $scope.headers.remove(header);
        };

        $scope.defaultHeaderNames = function () {
            var answer = [];

            function addHeaderSchema(schema) {
                angular.forEach(schema.definitions.headers.properties, function (value, name) {
                    answer.push(name);
                });
            }

            if (isJmsEndpoint()) {
                addHeaderSchema(Camel.jmsHeaderSchema);
            }
            if (isCamelEndpoint()) {
                addHeaderSchema(Camel.camelHeaderSchema);
            }
            return answer;
        };

        $scope.$watch('workspace.selection', function () {
            // if the current JMX selection does not support sending messages then lets redirect the page
            workspace.moveIfViewInvalid();

            if (Fabric.fabricCreated(workspace)) {
                loadProfileConfigurationFiles();
            }
        });

        /* save the sourceFormat in preferences for later
        * Note, this would be controller specific preferences and not the global, overriding, preferences */
        // TODO Use ng-selected="changeSourceFormat()" - Although it seemed to fire multiple times..
        $scope.$watch('codeMirrorOptions.mode.name', function (newValue, oldValue) {
            workspace.setLocalStorage(LANGUAGE_FORMAT_PREFERENCE, newValue);
        });

        var sendWorked = function () {
            $scope.message = "";
            notification("success", "Message sent!");
        };

        $scope.autoFormat = function () {
            setTimeout(function () {
                CodeEditor.autoFormatEditor($scope.codeMirror);
            }, 50);
        };

        $scope.sendMessage = function () {
            var body = $scope.message;
            doSendMessage(body, sendWorked);
        };

        function doSendMessage(body, onSendCompleteFn) {
            var selection = workspace.selection;
            if (selection) {
                var mbean = selection.objectName;
                if (mbean) {
                    var headers = null;
                    if ($scope.headers.length) {
                        headers = {};
                        angular.forEach($scope.headers, function (object) {
                            var key = object.name;
                            if (key) {
                                headers[key] = object.value;
                            }
                        });
                        log.info("About to send headers: " + JSON.stringify(headers));
                    }

                    var callback = onSuccess(onSendCompleteFn);
                    if (selection.domain === "org.apache.camel") {
                        var target = Camel.getContextAndTargetEndpoint(workspace);
                        var uri = target['uri'];
                        mbean = target['mbean'];
                        if (mbean && uri) {
                            if (headers) {
                                jolokia.execute(mbean, "sendBodyAndHeaders(java.lang.String, java.lang.Object, java.util.Map)", uri, body, headers, callback);
                            } else {
                                jolokia.execute(mbean, "sendStringBody(java.lang.String, java.lang.String)", uri, body, callback);
                            }
                        } else {
                            if (!mbean) {
                                notification("error", "Could not find CamelContext MBean!");
                            } else {
                                notification("error", "Failed to determine endpoint name!");
                            }
                            log.debug("Parsed context and endpoint: ", target);
                        }
                    } else {
                        var user = localStorage["activemqUserName"];
                        var pwd = localStorage["activemqPassword"];
                        if (headers) {
                            jolokia.execute(mbean, "sendTextMessage(java.util.Map, java.lang.String, java.lang.String, java.lang.String)", headers, body, user, pwd, callback);
                        } else {
                            jolokia.execute(mbean, "sendTextMessage(java.lang.String, java.lang.String, java.lang.String)", body, user, pwd, callback);
                        }
                    }
                }
            }
        }

        $scope.fileSelection = function () {
            var answer = [];
            angular.forEach($scope.selectedFiles, function (value, key) {
                if (value) {
                    answer.push(key);
                }
            });
            return answer;
        };

        $scope.sendSelectedFiles = function () {
            var filesToSend = $scope.fileSelection();
            var fileCount = filesToSend.length;
            var version = $scope.container.versionId || "1.0";

            function onSendFileCompleted(response) {
                if (filesToSend.length) {
                    var fileName = filesToSend.pop();
                    if (fileName) {
                        // lets load the file data...
                        var profile = $scope.profileFileNameToProfileId[fileName];
                        if (profile) {
                            var body = Fabric.getConfigFile(jolokia, version, profile, fileName);
                            if (!body) {
                                log.warn("No body for message " + fileName);
                                body = "";
                            }
                            doSendMessage(body, onSendFileCompleted);
                        }
                    }
                } else {
                    var text = Core.maybePlural(fileCount, "Message") + " sent!";
                    notification("success", text);
                }
            }

            // now lets start sending
            onSendFileCompleted(null);
        };

        function isCamelEndpoint() {
            // TODO check for the camel or if its an activemq endpoint
            return true;
        }

        function isJmsEndpoint() {
            // TODO check for the jms/activemq endpoint in camel or if its an activemq endpoint
            return true;
        }

        function loadProfileConfigurationFiles() {
            if (Fabric.fabricCreated(workspace)) {
                $scope.container = Fabric.getCurrentContainer(jolokia, ['versionId', 'profileIds']);
                jolokia.execute(Fabric.managerMBean, "currentContainerConfigurationFiles", onSuccess(onFabricConfigFiles));
            }
        }

        function onFabricConfigFiles(response) {
            $scope.profileFileNameToProfileId = response;
            $scope.profileFileNames = Object.keys(response).sort();
            $scope.showChoose = $scope.profileFileNames.length ? true : false;
            $scope.selectedFiles = {};
            Core.$apply($scope);
        }
    }
    Camel.SendMessageController = SendMessageController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function DebugRouteController($scope, $element, workspace, jolokia) {
        $scope.camelMaximumTraceOrDebugBodyLength = Camel.maximumTraceOrDebugBodyLength(localStorage);

        // ignore the cached stuff in camel.ts as it seems to bork the node ids for some reason...
        $scope.ignoreRouteXmlNode = true;

        $scope.startDebugging = function () {
            setDebugging(true);
        };

        $scope.stopDebugging = function () {
            setDebugging(false);
        };

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(reloadData, 50);
        });

        $scope.$on("camel.diagram.selectedNodeId", function (event, value) {
            $scope.selectedDiagramNodeId = value;
            updateBreakpointFlag();
        });

        $scope.$on("camel.diagram.layoutComplete", function (event, value) {
            updateBreakpointIcons();

            $($element).find("g.node").dblclick(function (n) {
                var id = this.getAttribute("data-cid");
                $scope.toggleBreakpoint(id);
            });
        });

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;
            reloadData();
        });

        $scope.toggleBreakpoint = function (id) {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean && id) {
                var method = isBreakpointSet(id) ? "removeBreakpoint" : "addBreakpoint";
                jolokia.execute(mbean, method, id, onSuccess(breakpointsChanged));
            }
        };

        $scope.addBreakpoint = function () {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean && $scope.selectedDiagramNodeId) {
                jolokia.execute(mbean, "addBreakpoint", $scope.selectedDiagramNodeId, onSuccess(breakpointsChanged));
            }
        };

        $scope.removeBreakpoint = function () {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean && $scope.selectedDiagramNodeId) {
                jolokia.execute(mbean, "removeBreakpoint", $scope.selectedDiagramNodeId, onSuccess(breakpointsChanged));
            }
        };

        $scope.resume = function () {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "resumeAll", onSuccess(clearStoppedAndResume));
            }
        };

        $scope.suspend = function () {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "suspendAll", onSuccess(clearStoppedAndResume));
            }
        };

        $scope.step = function () {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            var stepNode = getStoppedBreakpointId();
            if (mbean && stepNode) {
                jolokia.execute(mbean, "stepBreakpoint(java.lang.String)", stepNode, onSuccess(clearStoppedAndResume));
            }
        };

        // TODO refactor into common code with trace.ts?
        // START
        $scope.messages = [];
        $scope.mode = 'text';

        $scope.messageDialog = new Core.Dialog();

        $scope.gridOptions = Camel.createBrowseGridOptions();
        $scope.gridOptions.selectWithCheckboxOnly = false;
        $scope.gridOptions.showSelectionCheckbox = false;
        $scope.gridOptions.multiSelect = false;
        $scope.gridOptions.afterSelectionChange = onSelectionChanged;
        $scope.gridOptions.columnDefs.push({
            field: 'toNode',
            displayName: 'To Node'
        });

        $scope.openMessageDialog = function (message) {
            var idx = Core.pathGet(message, ["rowIndex"]);
            $scope.selectRowIndex(idx);
            if ($scope.row) {
                var body = $scope.row.body;
                $scope.mode = angular.isString(body) ? CodeEditor.detectTextFormat(body) : "text";
                $scope.messageDialog.open();
            }
        };

        $scope.selectRowIndex = function (idx) {
            $scope.rowIndex = idx;
            var selected = $scope.gridOptions.selectedItems;
            selected.splice(0, selected.length);
            if (idx >= 0 && idx < $scope.messages.length) {
                $scope.row = $scope.messages[idx];
                if ($scope.row) {
                    selected.push($scope.row);
                }
            } else {
                $scope.row = null;
            }
            onSelectionChanged();
        };

        // END
        function onSelectionChanged() {
            var toNode = getStoppedBreakpointId();
            if (toNode) {
                // lets highlight the node in the diagram
                var nodes = getDiagramNodes();
                Camel.highlightSelectedNode(nodes, toNode);
            }
        }

        function reloadData() {
            $scope.debugging = false;
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean) {
                $scope.debugging = jolokia.getAttribute(mbean, "Enabled", onSuccess(null));
                if ($scope.debugging) {
                    jolokia.execute(mbean, "getBreakpoints", onSuccess(onBreakpoints));

                    // get the breakpoints...
                    $scope.graphView = "app/camel/html/routes.html";
                    $scope.tableView = "app/camel/html/browseMessages.html";

                    Core.register(jolokia, $scope, {
                        type: 'exec', mbean: mbean,
                        operation: 'getDebugCounter' }, onSuccess(onBreakpointCounter));
                } else {
                    $scope.graphView = null;
                    $scope.tableView = null;
                }
            }
        }

        function onBreakpointCounter(response) {
            var counter = response.value;
            if (counter && counter !== $scope.breakpointCounter) {
                $scope.breakpointCounter = counter;
                loadCurrentStack();
            }
        }

        /*
        * lets load current 'stack' of which breakpoints are active
        * and what is the current message content
        */
        function loadCurrentStack() {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean) {
                console.log("getting suspended breakpoints!");
                jolokia.execute(mbean, "getSuspendedBreakpointNodeIds", onSuccess(onSuspendedBreakpointNodeIds));
            }
        }

        function onSuspendedBreakpointNodeIds(response) {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            $scope.suspendedBreakpoints = response;
            $scope.stopped = response && response.length;
            var stopNodeId = getStoppedBreakpointId();
            if (mbean && stopNodeId) {
                jolokia.execute(mbean, 'dumpTracedMessagesAsXml', stopNodeId, onSuccess(onMessages));

                // lets update the diagram selection to the newly stopped node
                $scope.selectedDiagramNodeId = stopNodeId;
            }
            updateBreakpointIcons();
            Core.$apply($scope);
        }

        function onMessages(response) {
            console.log("onMessage! ");
            $scope.messages = [];
            if (response) {
                var xml = response;
                if (angular.isString(xml)) {
                    // lets parse the XML DOM here...
                    var doc = $.parseXML(xml);
                    var allMessages = $(doc).find("fabricTracerEventMessage");
                    if (!allMessages || !allMessages.length) {
                        // lets try find another element name
                        allMessages = $(doc).find("backlogTracerEventMessage");
                    }

                    allMessages.each(function (idx, message) {
                        var messageData = Camel.createMessageFromXml(message);
                        var toNode = $(message).find("toNode").text();
                        if (toNode) {
                            messageData["toNode"] = toNode;
                        }
                        $scope.messages.push(messageData);
                    });
                }
            } else {
                console.log("WARNING: dumpTracedMessagesAsXml() returned no results!");
            }

            // lets update the selection and selected row for the message detail view
            updateMessageSelection();
            console.log("has messages " + $scope.messages.length + " selected row " + $scope.row + " index " + $scope.rowIndex);
            Core.$apply($scope);
            updateBreakpointIcons();
        }

        function updateMessageSelection() {
            $scope.selectRowIndex($scope.rowIndex);
            if (!$scope.row && $scope.messageDialog.show) {
                // lets make a dummy empty row
                // so we can keep the detail view while resuming
                $scope.row = {
                    headers: {},
                    body: ""
                };
            }
        }

        function clearStoppedAndResume() {
            $scope.messages = [];
            $scope.suspendedBreakpoints = [];
            $scope.stopped = false;
            updateMessageSelection();
            Core.$apply($scope);
            updateBreakpointIcons();
        }

        /*
        * Return the current node id we are stopped at
        */
        function getStoppedBreakpointId() {
            var stepNode = null;
            var stepNodes = $scope.suspendedBreakpoints;
            if (stepNodes && stepNodes.length) {
                stepNode = stepNodes[0];
                if (stepNodes.length > 1 && isSuspendedAt($scope.selectedDiagramNodeId)) {
                    // TODO should consider we stepping from different nodes based on the call thread or selection?
                    stepNode = $scope.selectedDiagramNodeId;
                }
            }
            return stepNode;
        }

        /*
        * Returns true if the execution is currently suspended at the given node
        */
        function isSuspendedAt(nodeId) {
            return containsNodeId($scope.suspendedBreakpoints, nodeId);
        }

        function onBreakpoints(response) {
            $scope.breakpoints = response;
            updateBreakpointFlag();

            // update the breakpoint icons...
            var nodes = getDiagramNodes();
            if (nodes.length) {
                updateBreakpointIcons(nodes);
            }
            Core.$apply($scope);
        }

        /*
        * Returns true if there is a breakpoint set at the given node id
        */
        function isBreakpointSet(nodeId) {
            return containsNodeId($scope.breakpoints, nodeId);
        }

        function updateBreakpointFlag() {
            $scope.hasBreakpoint = isBreakpointSet($scope.selectedDiagramNodeId);
        }

        function containsNodeId(breakpoints, nodeId) {
            return nodeId && breakpoints && breakpoints.some(nodeId);
        }

        function getDiagramNodes() {
            var svg = d3.select("svg");
            return svg.selectAll("g .node");
        }

        var breakpointImage = url("/app/camel/doc/img/debug/breakpoint.gif");
        var suspendedBreakpointImage = url("/app/camel/doc/img/debug/breakpoint-suspended.gif");

        function updateBreakpointIcons(nodes) {
            if (typeof nodes === "undefined") { nodes = getDiagramNodes(); }
            nodes.each(function (object) {
                // add breakpoint icon
                var nodeId = object.cid;
                var thisNode = d3.select(this);
                var icons = thisNode.selectAll("image.breakpoint");
                var isSuspended = isSuspendedAt(nodeId);
                var isBreakpoint = isBreakpointSet(nodeId);
                if (isBreakpoint || isSuspended) {
                    var imageUrl = isSuspended ? suspendedBreakpointImage : breakpointImage;

                    // lets add an icon image if we don't already have one
                    if (!icons.length || !icons[0].length) {
                        thisNode.append("image").attr("xlink:href", function (d) {
                            return imageUrl;
                        }).attr("class", "breakpoint").attr("x", -12).attr("y", -20).attr("height", 24).attr("width", 24);
                    } else {
                        icons.attr("xlink:href", function (d) {
                            return imageUrl;
                        });
                    }
                } else {
                    icons.remove();
                }
            });
        }

        function breakpointsChanged(response) {
            reloadData();
            Core.$apply($scope);
        }

        function setDebugging(flag) {
            var mbean = Camel.getSelectionCamelDebugMBean(workspace);
            if (mbean) {
                var method = flag ? "enableDebugger" : "disableDebugger";
                var max = $scope.camelMaximumTraceOrDebugBodyLength;
                jolokia.setAttribute(mbean, "BodyMaxChars", max);
                jolokia.execute(mbean, method, onSuccess(breakpointsChanged));
            }
        }
    }
    Camel.DebugRouteController = DebugRouteController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    Camel.jmsHeaderSchema = {
        definitions: {
            headers: {
                properties: {
                    JMSCorrelationID: {
                        type: "java.lang.String"
                    },
                    JMSDeliveryMode: {
                        "type": "string",
                        "enum": [
                            "PERSISTENT",
                            "NON_PERSISTENT"
                        ]
                    },
                    JMSDestination: {
                        type: "javax.jms.Destination"
                    },
                    JMSExpiration: {
                        type: "long"
                    },
                    JMSPriority: {
                        type: "int"
                    },
                    JMSReplyTo: {
                        type: "javax.jms.Destination"
                    },
                    JMSType: {
                        type: "java.lang.String"
                    },
                    JMSXGroupId: {
                        type: "java.lang.String"
                    },
                    AMQ_SCHEDULED_CRON: {
                        type: "java.lang.String"
                    },
                    AMQ_SCHEDULED_DELAY: {
                        type: "java.lang.String"
                    },
                    AMQ_SCHEDULED_PERIOD: {
                        type: "java.lang.String"
                    },
                    AMQ_SCHEDULED_REPEAT: {
                        type: "java.lang.String"
                    }
                }
            },
            "javax.jms.Destination": {
                type: "java.lang.String"
            }
        }
    };
})(Camel || (Camel = {}));
/// <reference path="../../jmx/js/jmxHelpers.ts"/>
/**
* That magic <reference> in the top of this file must be there!!!
*
* @module Camel
* @main Camel
*/
var Camel;
(function (Camel) {
    var jmxModule = Jmx;

    var pluginName = 'camel';

    var routeToolBar = "app/camel/html/attributeToolBarRoutes.html";
    var contextToolBar = "app/camel/html/attributeToolBarContext.html";

    angular.module(pluginName, [
        'bootstrap', 'ui.bootstrap',
        'ui.bootstrap.dialog', 'ui.bootstrap.tabs', 'ui.bootstrap.typeahead', 'ngResource', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/camel/browseEndpoint', { templateUrl: 'app/camel/html/browseEndpoint.html' }).when('/camel/endpoint/browse/:contextId/*endpointPath', { templateUrl: 'app/camel/html/browseEndpoint.html' }).when('/camel/createEndpoint', { templateUrl: 'app/camel/html/createEndpoint.html' }).when('/camel/route/diagram/:contextId/:routeId', { templateUrl: 'app/camel/html/routes.html' }).when('/camel/routes', { templateUrl: 'app/camel/html/routes.html' }).when('/camel/fabricDiagram', { templateUrl: 'app/camel/html/fabricDiagram.html', reloadOnSearch: false }).when('/camel/sendMessage', { templateUrl: 'app/camel/html/sendMessage.html', reloadOnSearch: false }).when('/camel/source', { templateUrl: 'app/camel/html/source.html' }).when('/camel/traceRoute', { templateUrl: 'app/camel/html/traceRoute.html' }).when('/camel/debugRoute', { templateUrl: 'app/camel/html/debug.html' }).when('/camel/profileRoute', { templateUrl: 'app/camel/html/profileRoute.html' }).when('/camel/properties', { templateUrl: 'app/camel/html/properties.html' });
    }).factory('tracerStatus', function () {
        return {
            jhandle: null,
            messages: []
        };
    }).filter('camelIconClass', function () {
        return Camel.iconClass;
    }).run(function (workspace, jolokia, viewRegistry, layoutFull, helpRegistry) {
        viewRegistry['camel/endpoint/'] = layoutFull;
        viewRegistry['camel/route/'] = layoutFull;
        viewRegistry['camel/fabricDiagram'] = layoutFull;
        viewRegistry['camel'] = 'app/camel/html/layoutCamelTree.html';

        helpRegistry.addUserDoc('camel', 'app/camel/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties(Camel.jmxDomain);
        });

        Jmx.addAttributeToolBar(pluginName, Camel.jmxDomain, function (selection) {
            // TODO there should be a nicer way to do this!
            var typeName = selection.typeName;
            if (typeName) {
                if (typeName.startsWith("context"))
                    return contextToolBar;
                if (typeName.startsWith("route"))
                    return routeToolBar;
            }
            var folderNames = selection.folderNames;
            if (folderNames && selection.domain === Camel.jmxDomain) {
                var last = folderNames.last();
                if ("routes" === last)
                    return routeToolBar;
                if ("context" === last)
                    return contextToolBar;
            }
            return null;
        });

        // register default attribute views
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | camelIconClass}}"></i></div>';
        var stateColumn = {
            field: 'State', displayName: 'State',
            cellTemplate: stateTemplate,
            width: 56,
            minWidth: 56,
            maxWidth: 56,
            resizable: false
        };

        var attributes = workspace.attributeColumnDefs;
        attributes[Camel.jmxDomain + "/context/folder"] = [
            stateColumn,
            { field: 'CamelId', displayName: 'Context' },
            { field: 'Uptime', displayName: 'Uptime', visible: false },
            { field: 'CamelVersion', displayName: 'Version', visible: false },
            { field: 'ExchangesCompleted', displayName: 'Completed #' },
            { field: 'ExchangesFailed', displayName: 'Failed #' },
            { field: 'ExchangesTotal', displayName: 'Total #', visible: false },
            { field: 'InflightExchanges', displayName: 'Inflight #' },
            { field: 'MeanProcessingTime', displayName: 'Mean Time' },
            { field: 'MinProcessingTime', displayName: 'Min Time' },
            { field: 'MaxProcessingTime', displayName: 'Max Time' },
            { field: 'TotalProcessingTime', displayName: 'Total Time', visible: false },
            { field: 'DeltaProcessingTime', displayName: 'Delta Time', visible: false },
            { field: 'LastExchangeCompletedTimestamp', displayName: 'Last completed', visible: false },
            { field: 'LastExchangeFailedTimestamp', displayName: 'Last failed', visible: false }
        ];
        attributes[Camel.jmxDomain + "/routes/folder"] = [
            stateColumn,
            { field: 'CamelId', displayName: 'Context' },
            { field: 'RouteId', displayName: 'Route' },
            { field: 'ExchangesCompleted', displayName: 'Completed #' },
            { field: 'ExchangesFailed', displayName: 'Failed #' },
            { field: 'ExchangesTotal', displayName: 'Total #', visible: false },
            { field: 'InflightExchanges', displayName: 'Inflight #' },
            { field: 'MeanProcessingTime', displayName: 'Mean Time' },
            { field: 'MinProcessingTime', displayName: 'Min Time' },
            { field: 'MaxProcessingTime', displayName: 'Max Time' },
            { field: 'TotalProcessingTime', displayName: 'Total Time', visible: false },
            { field: 'DeltaProcessingTime', displayName: 'Delta Time', visible: false },
            { field: 'LastExchangeCompletedTimestamp', displayName: 'Last completed', visible: false },
            { field: 'LastExchangeFailedTimestamp', displayName: 'Last failed', visible: false }
        ];
        attributes[Camel.jmxDomain + "/processors/folder"] = [
            stateColumn,
            { field: 'CamelId', displayName: 'Context' },
            { field: 'RouteId', displayName: 'Route' },
            { field: 'ProcessorId', displayName: 'Processor' },
            { field: 'ExchangesCompleted', displayName: 'Completed #' },
            { field: 'ExchangesFailed', displayName: 'Failed #' },
            { field: 'ExchangesTotal', displayName: 'Total #', visible: false },
            { field: 'InflightExchanges', displayName: 'Inflight #' },
            { field: 'MeanProcessingTime', displayName: 'Mean Time' },
            { field: 'MinProcessingTime', displayName: 'Min Time' },
            { field: 'MaxProcessingTime', displayName: 'Max Time' },
            { field: 'TotalProcessingTime', displayName: 'Total Time', visible: false },
            { field: 'DeltaProcessingTime', displayName: 'Delta Time', visible: false },
            { field: 'LastExchangeCompletedTimestamp', displayName: 'Last completed', visible: false },
            { field: 'LastExchangeFailedTimestamp', displayName: 'Last failed', visible: false }
        ];
        attributes[Camel.jmxDomain + "/components/folder"] = [
            stateColumn,
            { field: 'CamelId', displayName: 'Context' },
            { field: 'ComponentName', displayName: 'Name' }
        ];
        attributes[Camel.jmxDomain + "/consumers/folder"] = [
            stateColumn,
            { field: 'CamelId', displayName: 'Context' },
            { field: 'RouteId', displayName: 'Route' },
            { field: 'EndpointUri', displayName: 'Endpoint URI', width: "**" },
            { field: 'Suspended', displayName: 'Suspended', resizable: false },
            { field: 'InflightExchanges', displayName: 'Inflight #' }
        ];
        attributes[Camel.jmxDomain + "/services/folder"] = [
            stateColumn,
            { field: 'CamelId', displayName: 'Context' },
            { field: 'RouteId', displayName: 'Route' },
            { field: 'Suspended', displayName: 'Suspended', resizable: false },
            { field: 'SupportsSuspended', displayName: 'Can Suspend', resizable: false }
        ];
        attributes[Camel.jmxDomain + "/endpoints/folder"] = [
            stateColumn,
            { field: 'CamelId', displayName: 'Context' },
            { field: 'EndpointUri', displayName: 'Endpoint URI', width: "***" },
            { field: 'Singleton', displayName: 'Singleton', resizable: false }
        ];
        attributes[Camel.jmxDomain + "/threadpools/folder"] = [
            { field: 'Id', displayName: 'Id', width: "**" },
            { field: 'ActiveCount', displayName: 'Active #' },
            { field: 'PoolSize', displayName: 'Pool Size' },
            { field: 'CorePoolSize', displayName: 'Core Pool Size' },
            { field: 'TaskQueueSize', displayName: 'Task Queue Size' },
            { field: 'TaskCount', displayName: 'Task #' },
            { field: 'CompletedTaskCount', displayName: 'Completed Task #' }
        ];
        attributes[Camel.jmxDomain + "/errorhandlers/folder"] = [
            { field: 'CamelId', displayName: 'Context' },
            { field: 'DeadLetterChannel', displayName: 'Dead Letter' },
            { field: 'DeadLetterChannelEndpointUri', displayName: 'Endpoint URI', width: "**", resizable: true },
            { field: 'MaximumRedeliveries', displayName: 'Max Redeliveries' },
            { field: 'RedeliveryDelay', displayName: 'Redelivery Delay' },
            { field: 'MaximumRedeliveryDelay', displayName: 'Max Redeliveries Delay' }
        ];

        workspace.topLevelTabs.push({
            id: "camel",
            content: "Camel",
            title: "Manage your Apache Camel applications",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties(Camel.jmxDomain);
            },
            href: function () {
                return "#/jmx/attributes?tab=camel";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("camel");
            }
        });

        // add sub level tabs
        workspace.subLevelTabs.push({
            content: '<i class="icon-picture"></i> Diagram',
            title: "View a diagram of the Camel routes",
            isValid: function (workspace) {
                return workspace.isRoute();
            },
            href: function () {
                return "#/camel/routes";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-picture"></i> Diagram',
            title: "View the entire JVMs camel flows",
            isValid: function (workspace) {
                return workspace.isTopTabActive("camel") && !workspace.isRoute();
            },
            href: function () {
                return "#/camel/fabricDiagram";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class=" icon-file-alt"></i> Source',
            title: "View the source of the Camel routes",
            isValid: function (workspace) {
                return !workspace.isEndpointsFolder() && (workspace.isRoute() || workspace.isRoutesFolder() || workspace.isCamelContext());
            },
            href: function () {
                return "#/camel/source";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class=" icon-edit"></i> Properties',
            title: "View the pattern properties",
            isValid: function (workspace) {
                return Camel.getSelectedRouteNode(workspace);
            },
            href: function () {
                return "#/camel/properties";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-envelope"></i> Browse',
            title: "Browse the messages on the endpoint",
            isValid: function (workspace) {
                return workspace.isEndpoint();
            },
            href: function () {
                return "#/camel/browseEndpoint";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-stethoscope"></i> Debug',
            title: "Debug the Camel route",
            isValid: function (workspace) {
                return workspace.isRoute() && Camel.getSelectionCamelDebugMBean(workspace);
            },
            href: function () {
                return "#/camel/debugRoute";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-envelope"></i> Trace',
            title: "Trace the messages flowing through the Camel route",
            isValid: function (workspace) {
                return workspace.isRoute() && Camel.getSelectionCamelTraceMBean(workspace);
            },
            href: function () {
                return "#/camel/traceRoute";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-bar-chart"></i> Profile',
            title: "Profile the messages flowing through the Camel route",
            isValid: function (workspace) {
                return workspace.isRoute() && Camel.getSelectionCamelTraceMBean(workspace);
            },
            href: function () {
                return "#/camel/profileRoute";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-pencil"></i> Send',
            title: "Send a message to this endpoint",
            isValid: function (workspace) {
                return workspace.isEndpoint();
            },
            href: function () {
                return "#/camel/sendMessage";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-plus"></i> Endpoint',
            title: "Create a new endpoint",
            isValid: function (workspace) {
                return workspace.isEndpointsFolder();
            },
            href: function () {
                return "#/camel/createEndpoint";
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);

    // register the jmx lazy loader here as it won't have been invoked in the run methot
    hawtioPluginLoader.loadPlugins(function () {
        jmxModule.registerLazyLoadHandler(Camel.jmxDomain, function (folder) {
            if (Camel.jmxDomain === folder.domain && "routes" === folder.typeName) {
                return function (workspace, folder, onComplete) {
                    if ("routes" === folder.typeName) {
                        Camel.processRouteXml(workspace, workspace.jolokia, folder, function (route) {
                            if (route) {
                                Camel.addRouteChildren(folder, route);
                            }
                            onComplete();
                        });
                    } else {
                        onComplete();
                    }
                };
            }
            return null;
        });
    });
})(Camel || (Camel = {}));
/**
* @module Camel
*/
var Camel;
(function (Camel) {
    /**
    * Define the default categories for endpoints and map them to endpoint names
    * @property
    * @for Camel
    * @type {ObjecT}
    */
    Camel.endpointCategories = {
        bigdata: {
            label: "Big Data",
            endpoints: ["hdfs", "hbase", "lucene", "solr"],
            endpointIcon: "/app/camel/img/endpointRepository24.png"
        },
        database: {
            label: "Database",
            endpoints: ["couchdb", "elasticsearch", "hbase", "jdbc", "jpa", "hibernate", "mongodb", "mybatis", "sql"],
            endpointIcon: "/app/camel/img/endpointRepository24.png"
        },
        cloud: {
            label: "Cloud",
            endpoints: [
                "aws-cw", "aws-ddb", "aws-sdb", "aws-ses", "aws-sns", "aws-sqs", "aws-s3",
                "gauth", "ghhtp", "glogin", "gtask",
                "jclouds"]
        },
        core: {
            label: "Core",
            endpoints: ["bean", "direct", "seda"]
        },
        messaging: {
            label: "Messaging",
            endpoints: ["jms", "activemq", "amqp", "cometd", "cometds", "mqtt", "netty", "vertx", "websocket"],
            endpointIcon: "/app/camel/img/endpointQueue24.png"
        },
        mobile: {
            label: "Mobile",
            endpoints: ["apns"]
        },
        sass: {
            label: "SaaS",
            endpoints: ["salesforce", "sap-netweaver"]
        },
        social: {
            label: "Social",
            endpoints: ["atom", "facebook", "irc", "ircs", "rss", "smpp", "twitter", "weather"]
        },
        storage: {
            label: "Storage",
            endpointIcon: "/app/camel/img/endpointFolder24.png",
            endpoints: ["file", "ftp", "sftp", "scp", "jsch"]
        },
        template: {
            label: "Templating",
            endpoints: ["freemarker", "velocity", "xquery", "xslt", "scalate", "string-template"]
        }
    };

    /**
    * Maps endpoint names to a category object
    * @property
    * @for Camel
    * @type {ObjecT}
    */
    Camel.endpointToCategory = {};

    Camel.endpointIcon = "/app/camel/img/endpoint24.png";

    /**
    *  specify custom label & icon properties for endpoint names
    * @property
    * @for Camel
    * @type {ObjecT}
    */
    Camel.endpointConfigurations = {
        drools: {
            icon: "/app/camel/img/endpointQueue24.png"
        },
        quartz: {
            icon: "/app/camel/img/endpointTimer24.png"
        },
        facebook: {
            icon: "/app/camel/img/endpoints/facebook24.jpg"
        },
        salesforce: {
            icon: "/app/camel/img/endpoints/salesForce24.png"
        },
        sap: {
            icon: "/app/camel/img/endpoints/SAPe24.png"
        },
        "sap-netweaver": {
            icon: "/app/camel/img/endpoints/SAPNetweaver24.jpg"
        },
        timer: {
            icon: "/app/camel/img/endpointTimer24.png"
        },
        twitter: {
            icon: "/app/camel/img/endpoints/twitter24.png"
        },
        weather: {
            icon: "/app/camel/img/endpoints/weather24.jpg"
        }
    };

    /**
    * Define the default form configurations
    * @property
    * @for Camel
    * @type {ObjecT}
    */
    Camel.endpointForms = {
        file: {
            tabs: {
                //'Core': ['key', 'value'],
                'Options': ['*']
            }
        },
        activemq: {
            tabs: {
                'Connection': ['clientId', 'transacted', 'transactedInOut', 'transactionName', 'transactionTimeout'],
                'Producer': ['timeToLive', 'priority', 'allowNullBody', 'pubSubNoLocal', 'preserveMessageQos'],
                'Consumer': ['concurrentConsumers', 'acknowledgementModeName', 'selector', 'receiveTimeout'],
                'Reply': ['replyToDestination', 'replyToDeliveryPersistent', 'replyToCacheLevelName', 'replyToDestinationSelectorName'],
                'Options': ['*']
            }
        }
    };

    Camel.endpointForms["jms"] = Camel.endpointForms.activemq;

    angular.forEach(Camel.endpointCategories, function (category, catKey) {
        category.id = catKey;
        angular.forEach(category.endpoints, function (endpoint) {
            Camel.endpointToCategory[endpoint] = category;
        });
    });

    /**
    * Override the EIP pattern tabs...
    * @property
    * @for Camel
    * @type {ObjecT}
    */
    var camelModelTabExtensions = {
        route: {
            'Overview': ['id', 'description'],
            'Advanced': ['*']
        }
    };

    function getEndpointIcon(endpointName) {
        var value = Camel.getEndpointConfig(endpointName, null);
        var answer = Core.pathGet(value, ["icon"]);
        if (!answer) {
            var category = getEndpointCategory(endpointName);
            answer = Core.pathGet(category, ["endpointIcon"]);
        }
        return answer || Camel.endpointIcon;
    }
    Camel.getEndpointIcon = getEndpointIcon;

    function getEndpointConfig(endpointName, category) {
        var answer = Camel.endpointConfigurations[endpointName];
        if (!answer) {
            answer = {};
            Camel.endpointConfigurations[endpointName] = answer;
        }
        if (!answer.label) {
            answer.label = endpointName;
        }
        if (!answer.icon) {
            answer.icon = Core.pathGet(category, ["endpointIcon"]) || Camel.endpointIcon;
        }
        if (!answer.category) {
            answer.category = category;
        }
        return answer;
    }
    Camel.getEndpointConfig = getEndpointConfig;

    function getEndpointCategory(endpointName) {
        return Camel.endpointToCategory[endpointName] || Camel.endpointCategories.core;
    }
    Camel.getEndpointCategory = getEndpointCategory;

    function getConfiguredCamelModel() {
        var schema = _apacheCamelModel;
        var definitions = schema["definitions"];
        if (definitions) {
            angular.forEach(camelModelTabExtensions, function (tabs, name) {
                var model = definitions[name];
                if (model) {
                    if (!model["tabs"]) {
                        model["tabs"] = tabs;
                    }
                }
            });
        }
        return schema;
    }
    Camel.getConfiguredCamelModel = getConfiguredCamelModel;

    function initEndpointChooserScope($scope, $location, localStorage, workspace, jolokia) {
        $scope.selectedComponentName = null;
        $scope.endpointParameters = {};
        $scope.endpointPath = "";

        $scope.schema = {
            definitions: {}
        };

        $scope.jolokia = jolokia;

        // lets see if we need to use a remote jolokia container
        var versionId = $scope.branch;
        var profileId = Fabric.pagePathToProfileId($scope.pageId);
        if (profileId && versionId) {
            Fabric.profileJolokia(jolokia, profileId, versionId, function (profileJolokia) {
                if (!profileJolokia) {
                    // TODO we should expose this to the user somewhere nicely!
                    Camel.log.info("No container is running for profile " + profileId + " and version " + versionId + " so using current container for endpoint completion");
                    profileJolokia = jolokia;
                }
                $scope.jolokia = profileJolokia;

                // force a reload
                $scope.profileWorkspace = null;
                $scope.loadEndpointNames();
            });
        }

        var silentOptions = { silent: true };

        $scope.$watch('workspace.selection', function () {
            $scope.loadEndpointNames();
        });

        $scope.$watch('selectedComponentName', function () {
            if ($scope.selectedComponentName !== $scope.loadedComponentName) {
                $scope.endpointParameters = {};
                $scope.loadEndpointSchema($scope.selectedComponentName);
                $scope.loadedComponentName = $scope.selectedComponentName;
            }
        });

        $scope.endpointCompletions = function (completionText) {
            var answer = null;
            var mbean = findCamelContextMBean();
            var componentName = $scope.selectedComponentName;
            var endpointParameters = {};
            if (mbean && componentName && completionText) {
                answer = $scope.jolokia.execute(mbean, 'completeEndpointPath', componentName, endpointParameters, completionText, onSuccess(null, silentOptions));
            }
            return answer || [];
        };

        $scope.loadEndpointNames = function () {
            $scope.componentNames = null;
            var mbean = findCamelContextMBean();
            if (mbean) {
                //$scope.jolokia.execute(mbean, 'findComponentNames', onSuccess(onComponents, silentOptions));
                $scope.jolokia.execute(mbean, 'findComponentNames', onSuccess(onComponents, { silent: true }));
                /*
                $scope.jolokia.execute(mbean, 'findComponentNames', onSuccess(onComponents, {error: function (response) {
                console.log("FAILED: " + response);
                }}));
                */
            } else {
                console.log("WARNING: No camel context mbean so cannot load component names");
            }
        };

        $scope.loadEndpointSchema = function (componentName) {
            var mbean = findCamelContextMBean();
            if (mbean && componentName && componentName !== $scope.loadedEndpointSchema) {
                $scope.selectedComponentName = componentName;
                $scope.jolokia.execute(mbean, 'componentParameterJsonSchema', componentName, onSuccess(onEndpointSchema, silentOptions));
            }
        };

        function onComponents(response) {
            $scope.componentNames = response;
            Camel.log.info("onComponents: " + response);
            $scope.hasComponentNames = $scope.componentNames ? true : false;
            Core.$apply($scope);
        }

        function onEndpointSchema(response) {
            if (response) {
                try  {
                    //console.log("got JSON: " + response);
                    var json = JSON.parse(response);
                    var endpointName = $scope.selectedComponentName;
                    configureEndpointSchema(endpointName, json);
                    $scope.endpointSchema = json;
                    $scope.schema.definitions[endpointName] = json;
                    $scope.loadedEndpointSchema = endpointName;
                    Core.$apply($scope);
                } catch (e) {
                    console.log("Failed to parse JSON " + e);
                    console.log("JSON: " + response);
                }
            }
        }

        function configureEndpointSchema(endpointName, json) {
            console.log("======== configuring schema for " + endpointName);
            var config = Camel.endpointForms[endpointName];
            if (config && json) {
                if (config.tabs) {
                    json.tabs = config.tabs;
                }
            }
        }

        function findCamelContextMBean() {
            var profileWorkspace = $scope.profileWorkspace;
            if (!profileWorkspace) {
                var removeJolokia = $scope.jolokia;
                if (removeJolokia) {
                    profileWorkspace = Core.createRemoteWorkspace(removeJolokia, $location, localStorage);
                    $scope.profileWorkspace = profileWorkspace;
                }
            }
            if (!profileWorkspace) {
                Camel.log.info("No profileWorkspace found so defaulting it to workspace for now");
                profileWorkspace = workspace;
            }

            // TODO we need to find the MBean for the CamelContext / Route we are editing!
            var componentName = $scope.selectedComponentName;
            var selectedCamelContextId;
            var selectedRouteId;
            if (angular.isDefined($scope.camelSelectionDetails)) {
                selectedCamelContextId = $scope.camelSelectionDetails.selectedCamelContextId;
                selectedRouteId = $scope.camelSelectionDetails.selectedRouteId;
            }

            console.log("==== componentName " + componentName + " selectedCamelContextId: " + selectedCamelContextId + " selectedRouteId: " + selectedRouteId);

            var contextsById = Camel.camelContextMBeansById(profileWorkspace);
            if (selectedCamelContextId) {
                var mbean = Core.pathGet(contextsById, [selectedCamelContextId, "mbean"]);
                if (mbean) {
                    return mbean;
                }
            }
            if (selectedRouteId) {
                var map = Camel.camelContextMBeansByRouteId(profileWorkspace);
                var mbean = Core.pathGet(map, [selectedRouteId, "mbean"]);
                if (mbean) {
                    return mbean;
                }
            }
            if (componentName) {
                var map = Camel.camelContextMBeansByComponentName(profileWorkspace);
                var mbean = Core.pathGet(map, [componentName, "mbean"]);
                if (mbean) {
                    return mbean;
                }
            }

            // NOTE we don't really know which camel context to pick, so lets just find the first one?
            var answer = null;
            angular.forEach(contextsById, function (details, id) {
                var mbean = details.mbean;
                if (!answer && mbean)
                    answer = mbean;
            });
            return answer;
            /*
            // we could be remote to lets query jolokia
            var results = $scope.jolokia.search("org.apache.camel:*,type=context", onSuccess(null));
            //var results = $scope.jolokia.search("org.apache.camel:*", onSuccess(null));
            if (results && results.length) {
            console.log("===== Got results: " + results);
            return results[0];
            }
            
            var mbean = Camel.getSelectionCamelContextMBean(profileWorkspace);
            if (!mbean && $scope.findProfileCamelContext) {
            // TODO as a hack for now lets just find any camel context we can
            var folder = Core.getMBeanTypeFolder(profileWorkspace, Camel.jmxDomain, "context");
            mbean = Core.pathGet(folder, ["objectName"]);
            }
            return mbean;
            */
        }
    }
    Camel.initEndpointChooserScope = initEndpointChooserScope;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function TreeController($scope, $location, $timeout, workspace) {
        $scope.contextFilterText = $location.search()["cq"];
        $scope.fullScreenViewLink = Camel.linkToFullScreenView(workspace);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        var reloadThrottled = Core.throttled(reloadFunction, 500);

        $scope.$watch('workspace.tree', function () {
            reloadThrottled();
        });

        var reloadOnContextFilterThrottled = Core.throttled(function () {
            reloadFunction(function () {
                $("#camelContextIdFilter").focus();
            });
        }, 500);

        $scope.$watch('contextFilterText', function () {
            if ($scope.contextFilterText != $scope.lastContextFilterText) {
                $timeout(reloadOnContextFilterThrottled, 250);
            }
        });

        $scope.$on('jmxTreeUpdated', function () {
            reloadThrottled();
        });

        function reloadFunction(afterSelectionFn) {
            if (typeof afterSelectionFn === "undefined") { afterSelectionFn = null; }
            $scope.fullScreenViewLink = Camel.linkToFullScreenView(workspace);

            var children = [];
            var domainName = Camel.jmxDomain;

            // lets pull out each context
            var tree = workspace.tree;
            if (tree) {
                var rootFolder = new Folder("Camel Contexts");
                rootFolder.addClass = "org-apache-camel-context-folder";
                rootFolder.children = children;
                rootFolder.typeName = "context";
                rootFolder.key = "camelContexts";
                rootFolder.domain = domainName;

                var contextFilterText = $scope.contextFilterText;
                $scope.lastContextFilterText = contextFilterText;
                Camel.log.info("Reloading the tree for filter: " + contextFilterText);
                var folder = tree.get(domainName);
                if (folder) {
                    angular.forEach(folder.children, function (value, key) {
                        var entries = value.map;
                        if (entries) {
                            var contextsFolder = entries["context"];
                            var routesNode = entries["routes"];
                            var endpointsNode = entries["endpoints"];
                            if (contextsFolder) {
                                var contextNode = contextsFolder.children[0];
                                if (contextNode) {
                                    var title = contextNode.title;
                                    if (!contextFilterText || (title && title.indexOf(contextFilterText) >= 0)) {
                                        var folder = new Folder(title);
                                        folder.addClass = "org-apache-camel-context";
                                        folder.domain = domainName;
                                        folder.objectName = contextNode.objectName;
                                        folder.entries = contextNode.entries;
                                        folder.typeName = contextNode.typeName;
                                        folder.key = contextNode.key;
                                        if (routesNode) {
                                            var routesFolder = new Folder("Routes");
                                            routesFolder.addClass = "org-apache-camel-routes-folder";
                                            routesFolder.parent = contextsFolder;
                                            routesFolder.children = routesNode.children;
                                            angular.forEach(routesFolder.children, function (n) {
                                                return n.addClass = "org-apache-camel-routes";
                                            });
                                            folder.children.push(routesFolder);
                                            routesFolder.typeName = "routes";
                                            routesFolder.key = routesNode.key;
                                            routesFolder.domain = routesNode.domain;
                                        }
                                        if (endpointsNode) {
                                            var endpointsFolder = new Folder("Endpoints");
                                            endpointsFolder.addClass = "org-apache-camel-endpoints-folder";
                                            endpointsFolder.parent = contextsFolder;
                                            endpointsFolder.children = endpointsNode.children;
                                            angular.forEach(endpointsFolder.children, function (n) {
                                                n.addClass = "org-apache-camel-endpoints";
                                                if (!Camel.getContextId(n)) {
                                                    n.entries["context"] = contextNode.entries["context"];
                                                }
                                            });
                                            folder.children.push(endpointsFolder);
                                            endpointsFolder.entries = contextNode.entries;
                                            endpointsFolder.typeName = "endpoints";
                                            endpointsFolder.key = endpointsNode.key;
                                            endpointsFolder.domain = endpointsNode.domain;
                                        }
                                        var jmxNode = new Folder("MBeans");

                                        // lets add all the entries which are not one context/routes/endpoints
                                        angular.forEach(entries, function (jmxChild, name) {
                                            if (name !== "context" && name !== "routes" && name !== "endpoints") {
                                                jmxNode.children.push(jmxChild);
                                            }
                                        });

                                        if (jmxNode.children.length > 0) {
                                            jmxNode.sortChildren(false);
                                            folder.children.push(jmxNode);
                                        }
                                        folder.parent = rootFolder;
                                        children.push(folder);
                                    }
                                }
                            }
                        }
                    });
                }

                var treeElement = $("#cameltree");
                Jmx.enableTree($scope, $location, workspace, treeElement, [rootFolder], true);

                /*
                
                // lets select the first node if we have no selection
                var key = $location.search()['nid'];
                var node = children[0];
                if (!key && node) {
                key = node['key'];
                if (key) {
                var q = $location.search();
                q['nid'] = key;
                $location.search(q);
                }
                }
                if (!key) {
                updateSelectionFromURL();
                }
                */
                // lets do this asynchronously to avoid Error: $digest already in progress
                setTimeout(function () {
                    updateSelectionFromURL();
                    if (angular.isFunction(afterSelectionFn)) {
                        afterSelectionFn();
                    }
                }, 50);
            }
        }

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURLAndAutoSelect($location, $("#cameltree"), function (first) {
                // use function to auto select first Camel context routes if there is only one Camel context
                var contexts = first.getChildren();
                if (contexts && contexts.length === 1) {
                    first = contexts[0];
                    first.expand(true);
                    var children = first.getChildren();
                    if (children && children.length) {
                        var routes = children[0];
                        if (routes.data.typeName === 'routes') {
                            first = routes;
                            return first;
                        }
                    }
                }
                return null;
            }, true);
            $scope.fullScreenViewLink = Camel.linkToFullScreenView(workspace);
        }
    }
    Camel.TreeController = TreeController;
})(Camel || (Camel = {}));
/**
* @module Camel
*/
var Camel;
(function (Camel) {
    Camel.log = Logger.get("Camel");

    Camel.jmxDomain = 'org.apache.camel';

    Camel.defaultMaximumLabelWidth = 34;
    Camel.defaultCamelMaximumTraceOrDebugBodyLength = 5000;

    /**
    * Looks up the route XML for the given context and selected route and
    * processes the selected route's XML with the given function
    * @method processRouteXml
    * @param {Workspace} workspace
    * @param {Object} jolokia
    * @param {Folder} folder
    * @param {Function} onRoute
    */
    function processRouteXml(workspace, jolokia, folder, onRoute) {
        var selectedRouteId = getSelectedRouteId(workspace, folder);
        var mbean = getSelectionCamelContextMBean(workspace);

        function onRouteXml(response) {
            var route = null;
            var data = response ? response.value : null;
            if (data) {
                var doc = $.parseXML(data);
                var routes = $(doc).find("route[id='" + selectedRouteId + "']");
                if (routes && routes.length) {
                    route = routes[0];
                }
            }
            onRoute(route);
        }

        if (mbean && selectedRouteId) {
            jolokia.request({ type: 'exec', mbean: mbean, operation: 'dumpRoutesAsXml()' }, onSuccess(onRouteXml, { error: onRouteXml }));
        } else {
            if (!selectedRouteId) {
                console.log("No selectedRouteId when trying to lazy load the route!");
            }
            onRoute(null);
        }
    }
    Camel.processRouteXml = processRouteXml;

    /**
    * Returns the URI string for the given EIP pattern node or null if it is not applicable
    * @method getRouteNodeUri
    * @param {Object} node
    * @return {String}
    */
    function getRouteNodeUri(node) {
        var uri = null;
        if (node) {
            uri = node.getAttribute("uri");
            if (!uri) {
                var ref = node.getAttribute("ref");
                if (ref) {
                    var method = node.getAttribute("method");
                    if (method) {
                        uri = ref + "." + method + "()";
                    } else {
                        uri = "ref:" + ref;
                    }
                }
            }
        }
        return uri;
    }
    Camel.getRouteNodeUri = getRouteNodeUri;

    /**
    * Returns the JSON data for the camel folder; extracting it from the associated
    * routeXmlNode or using the previously extracted and/or edited JSON
    * @method getRouteFolderJSON
    * @param {Folder} folder
    * @param {Object} answer
    * @return {Object}
    */
    function getRouteFolderJSON(folder, answer) {
        if (typeof answer === "undefined") { answer = {}; }
        var nodeData = folder["camelNodeData"];
        if (!nodeData) {
            var routeXmlNode = folder["routeXmlNode"];
            if (routeXmlNode) {
                nodeData = Camel.getRouteNodeJSON(routeXmlNode);
            }
            if (!nodeData) {
                nodeData = answer;
            }
            folder["camelNodeData"] = nodeData;
        }
        return nodeData;
    }
    Camel.getRouteFolderJSON = getRouteFolderJSON;

    function getRouteNodeJSON(routeXmlNode, answer) {
        if (typeof answer === "undefined") { answer = {}; }
        if (routeXmlNode) {
            angular.forEach(routeXmlNode.attributes, function (attr) {
                answer[attr.name] = attr.value;
            });

            // lets not iterate into routes or top level tags
            var localName = routeXmlNode.localName;
            if (localName !== "route" && localName !== "routes" && localName !== "camelContext") {
                // lets look for nested elements and convert those
                // explicitly looking for expressions
                $(routeXmlNode).children("*").each(function (idx, element) {
                    var nodeName = element.localName;
                    var langSettings = Camel.camelLanguageSettings(nodeName);
                    if (langSettings) {
                        // TODO the expression key could be anything really; how should we know?
                        answer["expression"] = {
                            language: nodeName,
                            expression: element.textContent
                        };
                    } else {
                        if (!isCamelPattern(nodeName)) {
                            var nested = getRouteNodeJSON(element);
                            if (nested) {
                                answer[nodeName] = nested;
                            }
                        }
                    }
                });
            }
        }
        return answer;
    }
    Camel.getRouteNodeJSON = getRouteNodeJSON;

    function increaseIndent(currentIndent, indentAmount) {
        if (typeof indentAmount === "undefined") { indentAmount = "  "; }
        return currentIndent + indentAmount;
    }
    Camel.increaseIndent = increaseIndent;

    function setRouteNodeJSON(routeXmlNode, newData, indent) {
        if (routeXmlNode) {
            var childIndent = increaseIndent(indent);

            function doUpdate(value, key, append) {
                if (typeof append === "undefined") { append = false; }
                if (angular.isArray(value)) {
                    // remove previous nodes
                    $(routeXmlNode).children(key).remove();
                    angular.forEach(value, function (item) {
                        doUpdate(item, key, true);
                    });
                } else if (angular.isObject(value)) {
                    // convert languages to the right xml
                    var textContent = null;
                    if (key === "expression") {
                        var languageName = value["language"];
                        if (languageName) {
                            key = languageName;
                            textContent = value["expression"];
                            value = angular.copy(value);
                            delete value["expression"];
                            delete value["language"];
                        }
                    }

                    // TODO deal with nested objects...
                    var nested = $(routeXmlNode).children(key);
                    var element = null;
                    if (append || !nested || !nested.length) {
                        var doc = routeXmlNode.ownerDocument || document;
                        routeXmlNode.appendChild(doc.createTextNode("\n" + childIndent));
                        element = doc.createElement(key);
                        if (textContent) {
                            element.appendChild(doc.createTextNode(textContent));
                        }
                        routeXmlNode.appendChild(element);
                    } else {
                        element = nested[0];
                    }
                    setRouteNodeJSON(element, value, childIndent);
                    if (textContent) {
                        nested.text(textContent);
                    }
                } else {
                    if (value) {
                        if (key.startsWith("_")) {
                            // ignore
                        } else {
                            var text = value.toString();
                            routeXmlNode.setAttribute(key, text);
                        }
                    } else {
                        routeXmlNode.removeAttribute(key);
                    }
                }
            }

            angular.forEach(newData, function (value, key) {
                return doUpdate(value, key, false);
            });
        }
    }
    Camel.setRouteNodeJSON = setRouteNodeJSON;

    function getRouteNodeIcon(nodeSettingsOrXmlNode) {
        var nodeSettings = null;
        if (nodeSettingsOrXmlNode) {
            var nodeName = nodeSettingsOrXmlNode.localName;
            if (nodeName) {
                nodeSettings = getCamelSchema(nodeName);
            } else {
                nodeSettings = nodeSettingsOrXmlNode;
            }
        }
        if (nodeSettings) {
            var imageName = nodeSettings["icon"] || "generic24.png";
            return url("/app/camel/img/" + imageName);
        } else {
            return null;
        }
    }
    Camel.getRouteNodeIcon = getRouteNodeIcon;

    /**
    * Parse out the currently selected endpoint's name to be used when invoking on a
    * context operation that wants an endpoint name
    * @method getSelectedEndpointName
    * @param {Workspace} workspace
    * @return {any} either a string that is the endpoint name or null if it couldn't be parsed
    */
    function getSelectedEndpointName(workspace) {
        var selection = workspace.selection;
        if (selection && selection['objectName'] && selection['typeName'] && selection['typeName'] === 'endpoints') {
            var mbean = Core.parseMBean(selection['objectName']);
            if (!mbean) {
                return null;
            }
            var attributes = mbean['attributes'];
            if (!attributes) {
                return null;
            }

            if (!('name' in attributes)) {
                return null;
            }

            var uri = attributes['name'];
            uri = uri.replace("\\?", "?");
            if (uri.startsWith("\"")) {
                uri = uri.last(uri.length - 1);
            }
            if (uri.endsWith("\"")) {
                uri = uri.first(uri.length - 1);
            }
            return uri;
        } else {
            return null;
        }
    }
    Camel.getSelectedEndpointName = getSelectedEndpointName;

    /**
    * Returns the mbean for the currently selected camel context and the name of the currently
    * selected endpoint for JMX operations on a context that require an endpoint name.
    * @method
    * @param workspace
    * @return {{uri: string, mbean: string}} either value could be null if there's a parse failure
    */
    function getContextAndTargetEndpoint(workspace) {
        return {
            uri: Camel.getSelectedEndpointName(workspace),
            mbean: Camel.getSelectionCamelContextMBean(workspace)
        };
    }
    Camel.getContextAndTargetEndpoint = getContextAndTargetEndpoint;

    /**
    * Returns the cached Camel XML route node stored in the current tree selection Folder
    * @method
    */
    function getSelectedRouteNode(workspace) {
        var selection = workspace.selection;
        return (selection && Camel.jmxDomain === selection.domain) ? selection["routeXmlNode"] : null;
    }
    Camel.getSelectedRouteNode = getSelectedRouteNode;

    /**
    * Flushes the cached Camel XML route node stored in the selected tree Folder
    * @method
    * @param workspace
    */
    function clearSelectedRouteNode(workspace) {
        var selection = workspace.selection;
        if (selection && Camel.jmxDomain === selection.domain) {
            delete selection["routeXmlNode"];
        }
    }
    Camel.clearSelectedRouteNode = clearSelectedRouteNode;

    /**
    * Looks up the given node name in the Camel schema
    * @method
    */
    function getCamelSchema(nodeIdOrDefinition) {
        return (angular.isObject(nodeIdOrDefinition)) ? nodeIdOrDefinition : Forms.lookupDefinition(nodeIdOrDefinition, _apacheCamelModel);
    }
    Camel.getCamelSchema = getCamelSchema;

    /**
    * Returns true if the given nodeId is a route, endpoint or pattern
    * (and not some nested type like a data format)
    * @method
    */
    function isCamelPattern(nodeId) {
        return Forms.isJsonType(nodeId, _apacheCamelModel, "org.apache.camel.model.OptionalIdentifiedDefinition");
    }
    Camel.isCamelPattern = isCamelPattern;

    /**
    * Returns true if the given node type prefers adding the next sibling as a child
    * @method
    */
    function isNextSiblingAddedAsChild(nodeIdOrDefinition) {
        var definition = getCamelSchema(nodeIdOrDefinition);
        if (definition) {
            return definition["nextSiblingAddedAsChild"] || false;
        }
        return null;
    }
    Camel.isNextSiblingAddedAsChild = isNextSiblingAddedAsChild;

    function acceptInput(nodeIdOrDefinition) {
        var definition = getCamelSchema(nodeIdOrDefinition);
        if (definition) {
            return definition["acceptInput"] || false;
        }
        return null;
    }
    Camel.acceptInput = acceptInput;

    function acceptOutput(nodeIdOrDefinition) {
        var definition = getCamelSchema(nodeIdOrDefinition);
        if (definition) {
            return definition["acceptOutput"] || false;
        }
        return null;
    }
    Camel.acceptOutput = acceptOutput;

    /**
    * Looks up the Camel language settings for the given language name
    * @method
    */
    function camelLanguageSettings(nodeName) {
        return _apacheCamelModel.languages[nodeName];
    }
    Camel.camelLanguageSettings = camelLanguageSettings;

    function isCamelLanguage(nodeName) {
        return (camelLanguageSettings(nodeName) || nodeName === "expression") ? true : false;
    }
    Camel.isCamelLanguage = isCamelLanguage;

    /**
    * Converts the XML string or DOM node to a camel tree
    * @method
    */
    function loadCamelTree(xml, key) {
        var doc = xml;
        if (angular.isString(xml)) {
            doc = $.parseXML(xml);
        }

        // TODO get id from camelContext
        var id = "camelContext";
        var folder = new Folder(id);
        folder.addClass = "org-apache-camel-context";
        folder.domain = Camel.jmxDomain;
        folder.typeName = "context";

        folder.key = Core.toSafeDomID(key);

        var context = $(doc).find("camelContext");
        if (!context || !context.length) {
            context = $(doc).find("routes");
        }

        if (context && context.length) {
            folder["xmlDocument"] = doc;
            folder["routeXmlNode"] = context;
            $(context).children("route").each(function (idx, route) {
                var id = route.getAttribute("id");
                if (!id) {
                    id = "route" + idx;
                    route.setAttribute("id", id);
                }
                var routeFolder = new Folder(id);
                routeFolder.addClass = "org-apache-camel-route";
                routeFolder.typeName = "routes";
                routeFolder.domain = Camel.jmxDomain;
                routeFolder.key = folder.key + "_" + Core.toSafeDomID(id);
                routeFolder.parent = folder;
                var nodeSettings = getCamelSchema("route");
                if (nodeSettings) {
                    var imageUrl = getRouteNodeIcon(nodeSettings);
                    routeFolder.tooltip = nodeSettings["tooltip"] || nodeSettings["description"] || id;
                    routeFolder.icon = imageUrl;
                }
                folder.children.push(routeFolder);

                addRouteChildren(routeFolder, route);
            });
        }
        return folder;
    }
    Camel.loadCamelTree = loadCamelTree;

    /**
    * Adds the route children to the given folder for each step in the route
    * @method
    */
    function addRouteChildren(folder, route) {
        folder.children = [];
        folder["routeXmlNode"] = route;
        route.setAttribute("_cid", folder.key);
        $(route).children("*").each(function (idx, n) {
            addRouteChild(folder, n);
        });
    }
    Camel.addRouteChildren = addRouteChildren;

    /**
    * Adds a child to the given folder / route
    * @method
    */
    function addRouteChild(folder, n) {
        var nodeName = n.localName;
        if (nodeName) {
            var nodeSettings = getCamelSchema(nodeName);
            if (nodeSettings) {
                var imageUrl = getRouteNodeIcon(nodeSettings);

                var child = new Folder(nodeName);
                child.domain = Camel.jmxDomain;
                child.typeName = "routeNode";
                updateRouteNodeLabelAndTooltip(child, n, nodeSettings);

                // TODO should maybe auto-generate these?
                child.parent = folder;
                child.folderNames = folder.folderNames;
                var id = n.getAttribute("id") || nodeName;
                var key = folder.key + "_" + Core.toSafeDomID(id);

                // lets find the next key thats unique
                var counter = 1;
                var notFound = true;
                while (notFound) {
                    var tmpKey = key + counter;
                    if (folder.children.some({ key: tmpKey })) {
                        counter += 1;
                    } else {
                        notFound = false;
                        key = tmpKey;
                    }
                }
                child.key = key;
                child.icon = imageUrl;
                child["routeXmlNode"] = n;
                if (!folder.children) {
                    folder.children = [];
                }
                folder.children.push(child);
                addRouteChildren(child, n);
                return child;
            }
        }
        return null;
    }
    Camel.addRouteChild = addRouteChild;

    /**
    * Returns the root JMX Folder of the camel mbeans
    */
    function getRootCamelFolder(workspace) {
        var tree = workspace ? workspace.tree : null;
        if (tree) {
            return tree.get(Camel.jmxDomain);
        }
        return null;
    }
    Camel.getRootCamelFolder = getRootCamelFolder;

    /**
    * Returns the JMX folder for the camel context
    */
    function getCamelContextFolder(workspace, camelContextId) {
        var answer = null;
        var root = getRootCamelFolder(workspace);
        if (root && camelContextId) {
            angular.forEach(root.children, function (contextFolder) {
                if (!answer && camelContextId === contextFolder.title) {
                    answer = contextFolder;
                }
            });
        }
        return answer;
    }
    Camel.getCamelContextFolder = getCamelContextFolder;

    /**
    * Returns the mbean for the given camel context ID or null if it cannot be found
    */
    function getCamelContextMBean(workspace, camelContextId) {
        var contextsFolder = getCamelContextFolder(workspace, camelContextId);
        if (contextsFolder) {
            var contextFolder = contextsFolder.navigate("context");
            if (contextFolder && contextFolder.children && contextFolder.children.length) {
                var contextItem = contextFolder.children[0];
                return contextItem.objectName;
            }
        }
        return null;
    }
    Camel.getCamelContextMBean = getCamelContextMBean;

    /**
    * Given a selection in the workspace try figure out the URL to the
    * full screen view
    */
    function linkToFullScreenView(workspace) {
        var answer = null;
        var selection = workspace.selection;
        if (selection) {
            var entries = selection.entries;
            if (entries) {
                var contextId = entries["context"];
                var name = entries["name"];
                var type = entries["type"];
                if ("endpoints" === type) {
                    return linkToBrowseEndpointFullScreen(contextId, name);
                }
                if ("routes" === type) {
                    return linkToRouteDiagramFullScreen(contextId, name);
                }
                // TODO a default page for a context?
            }
        }
        return answer;
    }
    Camel.linkToFullScreenView = linkToFullScreenView;

    /**
    * Returns the link to browse the endpoint full screen
    */
    function linkToBrowseEndpointFullScreen(contextId, endpointPath) {
        var answer = null;
        if (contextId && endpointPath) {
            answer = "#/camel/endpoint/browse/" + contextId + "/" + endpointPath;
        }
        return answer;
    }
    Camel.linkToBrowseEndpointFullScreen = linkToBrowseEndpointFullScreen;

    /**
    * Returns the link to the route diagram full screen
    */
    function linkToRouteDiagramFullScreen(contextId, routeId) {
        var answer = null;
        if (contextId && routeId) {
            answer = "#/camel/route/diagram/" + contextId + "/" + routeId;
        }
        return answer;
    }
    Camel.linkToRouteDiagramFullScreen = linkToRouteDiagramFullScreen;

    function getFolderCamelNodeId(folder) {
        var answer = Core.pathGet(folder, ["routeXmlNode", "localName"]);
        return ("from" === answer || "to" === answer) ? "endpoint" : answer;
    }
    Camel.getFolderCamelNodeId = getFolderCamelNodeId;

    /**
    * Rebuilds the DOM tree from the tree node and performs all the various hacks
    * to turn the folder / JSON / model into valid camel XML
    * such as renaming language elements from <language expression="foo" language="bar/>
    * to <bar>foo</bar>
    * and changing <endpoint> into either <from> or <to>
    * @method
    * @param treeNode is either the Node from the tree widget (with the real Folder in the data property) or a Folder
    */
    function createFolderXmlTree(treeNode, xmlNode, indent) {
        if (typeof indent === "undefined") { indent = Camel.increaseIndent(""); }
        var folder = treeNode.data || treeNode;
        var count = 0;
        var parentName = getFolderCamelNodeId(folder);
        if (folder) {
            if (!xmlNode) {
                xmlNode = document.createElement(parentName);
                var rootJson = Camel.getRouteFolderJSON(folder);
                if (rootJson) {
                    Camel.setRouteNodeJSON(xmlNode, rootJson, indent);
                }
            }
            var doc = xmlNode.ownerDocument || document;
            var namespaceURI = xmlNode.namespaceURI;

            var from = parentName !== "route";
            var childIndent = Camel.increaseIndent(indent);
            angular.forEach(treeNode.children || treeNode.getChildren(), function (childTreeNode) {
                var childFolder = childTreeNode.data || childTreeNode;
                var name = Camel.getFolderCamelNodeId(childFolder);
                var json = Camel.getRouteFolderJSON(childFolder);
                if (name && json) {
                    var language = false;
                    if (name === "endpoint") {
                        if (from) {
                            name = "to";
                        } else {
                            name = "from";
                            from = true;
                        }
                    }
                    if (name === "expression") {
                        var languageName = json["language"];
                        if (languageName) {
                            name = languageName;
                            language = true;
                        }
                    }

                    // lets create the XML
                    xmlNode.appendChild(doc.createTextNode("\n" + childIndent));
                    var newNode = doc.createElementNS(namespaceURI, name);

                    Camel.setRouteNodeJSON(newNode, json, childIndent);
                    xmlNode.appendChild(newNode);
                    count += 1;
                    createFolderXmlTree(childTreeNode, newNode, childIndent);
                }
            });
            if (count) {
                xmlNode.appendChild(doc.createTextNode("\n" + indent));
            }
        }
        return xmlNode;
    }
    Camel.createFolderXmlTree = createFolderXmlTree;

    function updateRouteNodeLabelAndTooltip(folder, routeXmlNode, nodeSettings) {
        var localName = routeXmlNode.localName;
        var id = routeXmlNode.getAttribute("id");
        var label = nodeSettings["title"] || localName;

        // lets use the ID for routes and other things we give an id
        var tooltip = nodeSettings["tooltip"] || nodeSettings["description"] || label;
        if (id) {
            label = id;
        } else {
            var uri = getRouteNodeUri(routeXmlNode);
            if (uri) {
                // Don't use from/to as it gets odd if you drag/drop and reorder
                // label += " " + uri;
                label = uri;
                var split = uri.split("?");
                if (split && split.length > 1) {
                    label = split[0];
                }
                tooltip += " " + uri;
            } else {
                var children = $(routeXmlNode).children("*");
                if (children && children.length) {
                    var child = children[0];
                    var childName = child.localName;
                    var expression = null;
                    if (Camel.isCamelLanguage(childName)) {
                        expression = child.textContent;
                        if (!expression) {
                            expression = child.getAttribute("expression");
                        }
                    }
                    if (expression) {
                        label += " " + expression;
                        tooltip += " " + childName + " expression";
                    }
                }
            }
        }
        folder.title = label;
        folder.tooltip = tooltip;
        return label;
    }
    Camel.updateRouteNodeLabelAndTooltip = updateRouteNodeLabelAndTooltip;

    /**
    * Returns the selected camel context mbean for the given selection or null if it cannot be found
    * @method
    */
    // TODO should be a service
    function getSelectionCamelContextMBean(workspace) {
        if (workspace) {
            var contextId = getContextId(workspace);
            var selection = workspace.selection;
            var tree = workspace.tree;
            if (tree && selection) {
                var domain = selection.domain;
                if (domain && contextId) {
                    var result = tree.navigate(domain, contextId, "context");
                    if (result && result.children) {
                        var contextBean = result.children.first();
                        if (contextBean.title) {
                            var contextName = contextBean.title;
                            return "" + domain + ":context=" + contextId + ',type=context,name="' + contextName + '"';
                        }
                    }
                }
            }
        }
        return null;
    }
    Camel.getSelectionCamelContextMBean = getSelectionCamelContextMBean;

    function getSelectionCamelContextEndpoints(workspace) {
        if (workspace) {
            var contextId = getContextId(workspace);
            var selection = workspace.selection;
            var tree = workspace.tree;
            if (tree && selection) {
                var domain = selection.domain;
                if (domain && contextId) {
                    return tree.navigate(domain, contextId, "endpoints");
                }
            }
        }
        return null;
    }
    Camel.getSelectionCamelContextEndpoints = getSelectionCamelContextEndpoints;

    /**
    * Returns the selected camel trace mbean for the given selection or null if it cannot be found
    * @method
    */
    // TODO Should be a service
    function getSelectionCamelTraceMBean(workspace) {
        if (workspace) {
            var contextId = getContextId(workspace);
            var selection = workspace.selection;
            var tree = workspace.tree;
            if (tree && selection) {
                var domain = selection.domain;
                if (domain && contextId) {
                    // look for the Camel 2.11 mbean which we prefer
                    var result = tree.navigate(domain, contextId, "tracer");
                    if (result && result.children) {
                        var mbean = result.children.find(function (m) {
                            return m.title.startsWith("BacklogTracer");
                        });
                        if (mbean) {
                            return mbean.objectName;
                        }
                    }

                    // look for the fuse camel fabric mbean
                    var fabricResult = tree.navigate(domain, contextId, "fabric");
                    if (fabricResult && fabricResult.children) {
                        var mbean = fabricResult.children.first();
                        return mbean.objectName;
                    }
                }
            }
        }
        return null;
    }
    Camel.getSelectionCamelTraceMBean = getSelectionCamelTraceMBean;

    function getSelectionCamelDebugMBean(workspace) {
        if (workspace) {
            var contextId = getContextId(workspace);
            var selection = workspace.selection;
            var tree = workspace.tree;
            if (tree && selection) {
                var domain = selection.domain;
                if (domain && contextId) {
                    var result = tree.navigate(domain, contextId, "tracer");
                    if (result && result.children) {
                        var mbean = result.children.find(function (m) {
                            return m.title.startsWith("BacklogDebugger");
                        });
                        if (mbean) {
                            return mbean.objectName;
                        }
                    }
                }
            }
        }
        return null;
    }
    Camel.getSelectionCamelDebugMBean = getSelectionCamelDebugMBean;

    // TODO should be a service
    function getContextId(workspace) {
        var selection = workspace.selection;
        if (selection) {
            var tree = workspace.tree;
            var folderNames = selection.folderNames;
            var entries = selection.entries;
            var contextId;
            if (tree) {
                if (folderNames && folderNames.length > 1) {
                    contextId = folderNames[1];
                } else if (entries) {
                    contextId = entries["context"];
                }
            }
        }
        return contextId;
    }
    Camel.getContextId = getContextId;

    /**
    * Returns true if the state of the item begins with the given state - or one of the given states
    * @method
    * @param item the item which has a State
    * @param state a value or an array of states
    */
    function isState(item, state) {
        var value = (item.State || "").toLowerCase();
        if (angular.isArray(state)) {
            return state.any(function (stateText) {
                return value.startsWith(stateText);
            });
        } else {
            return value.startsWith(state);
        }
    }
    Camel.isState = isState;

    function iconClass(state) {
        if (state) {
            switch (state.toLowerCase()) {
                case 'started':
                    return "green icon-play-circle";
                case 'suspended':
                    return "icon-pause";
            }
        }
        return "orange icon-off";
    }
    Camel.iconClass = iconClass;

    function getSelectedRouteId(workspace, folder) {
        if (typeof folder === "undefined") { folder = null; }
        var selection = folder || workspace.selection;
        var selectedRouteId = null;
        if (selection) {
            if (selection && selection.entries) {
                var typeName = selection.entries["type"];
                var name = selection.entries["name"];
                if ("routes" === typeName && name) {
                    selectedRouteId = trimQuotes(name);
                }
            }
        }
        return selectedRouteId;
    }
    Camel.getSelectedRouteId = getSelectedRouteId;

    /**
    * Returns the selected camel route mbean for the given route id
    * @method
    */
    // TODO Should be a service
    function getSelectionRouteMBean(workspace, routeId) {
        if (workspace) {
            var contextId = getContextId(workspace);
            var selection = workspace.selection;
            var tree = workspace.tree;
            if (tree && selection) {
                var domain = selection.domain;
                if (domain && contextId) {
                    var result = tree.navigate(domain, contextId, "routes");
                    if (result && result.children) {
                        var mbean = result.children.find(function (m) {
                            return m.title === routeId;
                        });
                        if (mbean) {
                            return mbean.objectName;
                        }
                    }
                }
            }
        }
        return null;
    }
    Camel.getSelectionRouteMBean = getSelectionRouteMBean;

    function getCamelVersion(workspace, jolokia) {
        var mbean = getSelectionCamelContextMBean(workspace);
        if (mbean) {
            // must use onSuccess(null) that means sync as we need the version asap
            return jolokia.getAttribute(mbean, "CamelVersion", onSuccess(null));
        } else {
            return null;
        }
    }
    Camel.getCamelVersion = getCamelVersion;

    function createMessageFromXml(exchange) {
        var exchangeElement = $(exchange);
        var uid = exchangeElement.children("uid").text();
        var timestamp = exchangeElement.children("timestamp").text();
        var messageData = {
            headers: {},
            headerTypes: {},
            id: null,
            uid: uid,
            timestamp: timestamp,
            headerHtml: ""
        };
        var message = exchangeElement.children("message")[0];
        if (!message) {
            message = exchange;
        }
        var messageElement = $(message);
        var headers = messageElement.find("header");
        var headerHtml = "";
        headers.each(function (idx, header) {
            var key = header.getAttribute("key");
            var typeName = header.getAttribute("type");
            var value = header.textContent;
            if (key) {
                if (value)
                    messageData.headers[key] = value;
                if (typeName)
                    messageData.headerTypes[key] = typeName;

                headerHtml += "<tr><td class='property-name'>" + key + "</td>" + "<td class='property-value'>" + (value || "") + "</td></tr>";
            }
        });

        messageData.headerHtml = headerHtml;
        var id = messageData.headers["breadcrumbId"];
        if (!id) {
            var postFixes = ["MessageID", "ID", "Path", "Name"];
            angular.forEach(postFixes, function (postfix) {
                if (!id) {
                    angular.forEach(messageData.headers, function (value, key) {
                        if (!id && key.endsWith(postfix)) {
                            id = value;
                        }
                    });
                }
            });

            // lets find the first header with a name or Path in it
            // if still no value, lets use the first :)
            angular.forEach(messageData.headers, function (value, key) {
                if (!id)
                    id = value;
            });
        }
        messageData.id = id;
        var body = messageElement.children("body")[0];
        if (body) {
            var bodyText = body.textContent;
            var bodyType = body.getAttribute("type");
            messageData["body"] = bodyText;
            messageData["bodyType"] = bodyType;
        }
        return messageData;
    }
    Camel.createMessageFromXml = createMessageFromXml;

    function createBrowseGridOptions() {
        return {
            selectedItems: [],
            data: 'messages',
            displayFooter: false,
            showFilter: false,
            showColumnMenu: true,
            enableColumnResize: true,
            enableColumnReordering: true,
            filterOptions: {
                filterText: ''
            },
            selectWithCheckboxOnly: true,
            showSelectionCheckbox: true,
            maintainColumnRatios: false,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'ID',
                    // for ng-grid
                    //width: '50%',
                    // for hawtio-datatable
                    // width: "22em",
                    cellTemplate: '<div class="ngCellText"><a ng-click="openMessageDialog(row)">{{row.entity.id}}</a></div>'
                }
            ]
        };
    }
    Camel.createBrowseGridOptions = createBrowseGridOptions;

    function loadRouteXmlNodes($scope, doc, selectedRouteId, nodes, links, width) {
        var allRoutes = $(doc).find("route");
        var routeDelta = width / allRoutes.length;
        var rowX = 0;
        allRoutes.each(function (idx, route) {
            var routeId = route.getAttribute("id");
            if (!selectedRouteId || !routeId || selectedRouteId === routeId) {
                Camel.addRouteXmlChildren($scope, route, nodes, links, null, rowX, 0);
                rowX += routeDelta;
            }
        });
    }
    Camel.loadRouteXmlNodes = loadRouteXmlNodes;

    function addRouteXmlChildren($scope, parent, nodes, links, parentId, parentX, parentY, parentNode) {
        if (typeof parentNode === "undefined") { parentNode = null; }
        var delta = 150;
        var x = parentX;
        var y = parentY + delta;
        var rid = parent.getAttribute("id");
        var siblingNodes = [];
        var parenNodeName = parent.localName;
        $(parent).children().each(function (idx, route) {
            var id = nodes.length;

            // from acts as a parent even though its a previous sibling :)
            var nodeId = route.localName;
            if (nodeId === "from" && !parentId) {
                parentId = id;
            }
            var nodeSettings = getCamelSchema(nodeId);
            var node = null;
            if (nodeSettings) {
                var label = nodeSettings["title"] || nodeId;
                var uri = getRouteNodeUri(route);
                if (uri) {
                    label += " " + uri.split("?")[0];
                }
                var tooltip = nodeSettings["tooltip"] || nodeSettings["description"] || label;
                if (uri) {
                    tooltip += " " + uri;
                }
                var elementID = route.getAttribute("id");
                var labelSummary = label;
                if (elementID) {
                    var customId = route.getAttribute("customId");
                    if ($scope.camelIgnoreIdForLabel || (!customId || customId === "false")) {
                        labelSummary = "id: " + elementID;
                    } else {
                        label = elementID;
                    }
                }

                // lets check if we need to trim the label
                var labelLimit = $scope.camelMaximumLabelWidth || Camel.defaultMaximumLabelWidth;
                var length = label.length;
                if (length > labelLimit) {
                    labelSummary = label + "\n\n" + labelSummary;
                    label = label.substring(0, labelLimit) + "..";
                }

                var imageUrl = getRouteNodeIcon(nodeSettings);
                if ((nodeId === "from" || nodeId === "to") && uri) {
                    var uriIdx = uri.indexOf(":");
                    if (uriIdx > 0) {
                        var componentScheme = uri.substring(0, uriIdx);

                        //console.log("lets find the endpoint icon for " + componentScheme);
                        if (componentScheme) {
                            var value = Camel.getEndpointIcon(componentScheme);
                            if (value) {
                                imageUrl = url(value);
                            }
                        }
                    }
                }

                //console.log("Image URL is " + imageUrl);
                var cid = route.getAttribute("_cid") || route.getAttribute("id");
                node = {
                    "name": name, "label": label, "labelSummary": labelSummary, "group": 1, "id": id, "elementId": elementID,
                    "x": x, "y:": y, "imageUrl": imageUrl, "cid": cid, "tooltip": tooltip, "type": nodeId };
                if (rid) {
                    node["rid"] = rid;
                    if (!$scope.routeNodes)
                        $scope.routeNodes = {};
                    $scope.routeNodes[rid] = node;
                }
                if (!cid) {
                    cid = nodeId + (nodes.length + 1);
                }
                if (cid) {
                    node["cid"] = cid;
                    if (!$scope.nodes)
                        $scope.nodes = {};
                    $scope.nodes[cid] = node;
                }

                // only use the route id on the first from node
                rid = null;
                nodes.push(node);
                if (parentId !== null && parentId !== id) {
                    if (siblingNodes.length === 0 || parenNodeName === "choice") {
                        links.push({ "source": parentId, "target": id, "value": 1 });
                    } else {
                        siblingNodes.forEach(function (nodeId) {
                            links.push({ "source": nodeId, "target": id, "value": 1 });
                        });
                        siblingNodes.length = 0;
                    }
                }
            } else {
                // ignore non EIP nodes, though we should add expressions...
                var langSettings = Camel.camelLanguageSettings(nodeId);
                if (langSettings && parentNode) {
                    // lets add the language kind
                    var name = langSettings["name"] || nodeId;
                    var text = route.textContent;
                    if (text) {
                        parentNode["tooltip"] = parentNode["label"] + " " + name + " " + text;
                        parentNode["label"] = text;
                    } else {
                        parentNode["label"] = parentNode["label"] + " " + name;
                    }
                }
            }
            var siblings = addRouteXmlChildren($scope, route, nodes, links, id, x, y, node);
            if (parenNodeName === "choice") {
                siblingNodes = siblingNodes.concat(siblings);
                x += delta;
            } else if (nodeId === "choice") {
                siblingNodes = siblings;
                y += delta;
            } else {
                siblingNodes = [nodes.length - 1];
                y += delta;
            }
        });
        return siblingNodes;
    }
    Camel.addRouteXmlChildren = addRouteXmlChildren;

    function getCanvasHeight(canvasDiv) {
        var height = canvasDiv.height();
        if (height < 300) {
            console.log("browse thinks the height is only " + height + " so calculating offset from doc height");
            var offset = canvasDiv.offset();
            height = $(document).height() - 5;
            if (offset) {
                var top = offset['top'];
                if (top) {
                    height -= top;
                }
            }
        }
        return height;
    }
    Camel.getCanvasHeight = getCanvasHeight;

    /**
    * Recursively add all the folders which have a cid value into the given map
    * @method
    */
    function addFoldersToIndex(folder, map) {
        if (typeof map === "undefined") { map = {}; }
        if (folder) {
            var key = folder.key;
            if (key) {
                map[key] = folder;
            }
            angular.forEach(folder.children, function (child) {
                return addFoldersToIndex(child, map);
            });
        }
        return map;
    }
    Camel.addFoldersToIndex = addFoldersToIndex;

    /**
    * Re-generates the XML document using the given Tree widget Node or Folder as the source
    * @method
    */
    function generateXmlFromFolder(treeNode) {
        var folder = (treeNode && treeNode.data) ? treeNode.data : treeNode;
        if (!folder)
            return null;
        var doc = folder["xmlDocument"];
        var context = folder["routeXmlNode"];

        if (context && context.length) {
            var element = context[0];
            var children = element.childNodes;
            var routeIndices = [];
            for (var i = 0; i < children.length; i++) {
                var node = children[i];
                var name = node.localName;
                if ("route" === name && parent) {
                    routeIndices.push(i);
                }
            }

            while (routeIndices.length) {
                var idx = routeIndices.pop();
                var nextIndex = idx + 1;
                while (true) {
                    var node = element.childNodes[nextIndex];
                    if (Core.isTextNode(node)) {
                        element.removeChild(node);
                    } else {
                        break;
                    }
                }
                if (idx < element.childNodes.length) {
                    element.removeChild(element.childNodes[idx]);
                }
                for (var i = idx - 1; i >= 0; i--) {
                    var node = element.childNodes[i];
                    if (Core.isTextNode(node)) {
                        element.removeChild(node);
                    } else {
                        break;
                    }
                }
            }
            Camel.createFolderXmlTree(treeNode, context[0]);
        }
        return doc;
    }
    Camel.generateXmlFromFolder = generateXmlFromFolder;

    /**
    * Returns an object of all the CamelContext MBeans keyed by their id
    * @method
    */
    function camelContextMBeansById(workspace) {
        var answer = {};
        var tree = workspace.tree;
        if (tree) {
            var camelTree = tree.navigate(Camel.jmxDomain);
            if (camelTree) {
                angular.forEach(camelTree.children, function (contextsFolder) {
                    var contextFolder = contextsFolder.navigate("context");
                    if (contextFolder && contextFolder.children && contextFolder.children.length) {
                        var contextItem = contextFolder.children[0];
                        var id = Core.pathGet(contextItem, ["entries", "name"]) || contextItem.key;
                        if (id) {
                            answer[id] = {
                                folder: contextItem,
                                mbean: contextItem.objectName
                            };
                        }
                    }
                });
            }
        }
        return answer;
    }
    Camel.camelContextMBeansById = camelContextMBeansById;

    /**
    * Returns an object of all the CamelContext MBeans keyed by the component name
    * @method
    */
    function camelContextMBeansByComponentName(workspace) {
        return camelContextMBeansByRouteOrComponentId(workspace, "components");
    }
    Camel.camelContextMBeansByComponentName = camelContextMBeansByComponentName;

    /**
    * Returns an object of all the CamelContext MBeans keyed by the route ID
    * @method
    */
    function camelContextMBeansByRouteId(workspace) {
        return camelContextMBeansByRouteOrComponentId(workspace, "routes");
    }
    Camel.camelContextMBeansByRouteId = camelContextMBeansByRouteId;

    function camelContextMBeansByRouteOrComponentId(workspace, componentsOrRoutes) {
        var answer = {};
        var tree = workspace.tree;
        if (tree) {
            var camelTree = tree.navigate(Camel.jmxDomain);
            if (camelTree) {
                angular.forEach(camelTree.children, function (contextsFolder) {
                    var contextFolder = contextsFolder.navigate("context");
                    var componentsFolder = contextsFolder.navigate(componentsOrRoutes);
                    if (contextFolder && componentsFolder && contextFolder.children && contextFolder.children.length) {
                        var contextItem = contextFolder.children[0];
                        var mbean = contextItem.objectName;
                        if (mbean) {
                            var contextValues = {
                                folder: contextItem,
                                mbean: mbean
                            };
                            angular.forEach(componentsFolder.children, function (componentFolder) {
                                var id = componentFolder.title;
                                if (id) {
                                    answer[id] = contextValues;
                                }
                            });
                        }
                    }
                });
            }
        }
        return answer;
    }

    /**
    * Returns true if we should ignore ID values for labels in camel diagrams
    * @method
    */
    function ignoreIdForLabel(localStorage) {
        var value = localStorage["camelIgnoreIdForLabel"];
        return value && (value === "true" || value === true);
    }
    Camel.ignoreIdForLabel = ignoreIdForLabel;

    /**
    * Returns the maximum width of a label before we start to truncate
    * @method
    */
    function maximumLabelWidth(localStorage) {
        var value = localStorage["camelMaximumLabelWidth"];
        if (angular.isString(value)) {
            value = parseInt(value);
        }
        if (!value) {
            value = Camel.defaultMaximumLabelWidth;
        }
        return value;
    }
    Camel.maximumLabelWidth = maximumLabelWidth;

    /**
    * Returns the max body length for tracer and debugger
    * @method
    */
    function maximumTraceOrDebugBodyLength(localStorage) {
        var value = localStorage["camelMaximumTraceOrDebugBodyLength"];
        if (angular.isString(value)) {
            value = parseInt(value);
        }
        if (!value) {
            value = Camel.defaultCamelMaximumTraceOrDebugBodyLength;
        }
        return value;
    }
    Camel.maximumTraceOrDebugBodyLength = maximumTraceOrDebugBodyLength;

    /**
    * Function to highlight the selected toNode in the nodes graph
    *
    * @param nodes the nodes
    * @param toNode the node to highlight
    */
    function highlightSelectedNode(nodes, toNode) {
        // lets clear the selected node first
        nodes.attr("class", "node");

        nodes.filter(function (item) {
            if (item) {
                var cid = item["cid"];
                var rid = item["rid"];
                var type = item["type"];
                var elementId = item["elementId"];

                // if its from then match on rid
                if ("from" === type) {
                    return toNode === rid;
                }

                // okay favor using element id as the cids can become
                // undefined or mangled with mbean object names, causing this to not work
                // where as elementId when present works fine
                if (elementId) {
                    // we should match elementId if defined
                    return toNode === elementId;
                }

                // then fallback to cid
                if (cid) {
                    return toNode === cid;
                } else {
                    // and last rid
                    return toNode === rid;
                }
            }
            return null;
        }).attr("class", "node selected");
    }
    Camel.highlightSelectedNode = highlightSelectedNode;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function BreadcrumbBarController($scope, $routeParams, workspace, jolokia) {
        $scope.workspace = workspace;
        $scope.contextId = $routeParams["contextId"];
        $scope.endpointPath = $routeParams["endpointPath"];
        $scope.endpointName = tidyJmxName($scope.endpointPath);
        $scope.routeId = $routeParams["routeId"];

        $scope.treeViewLink = linkToTreeView();

        var defaultChildEntity = $scope.endpointPath ? "endpoints" : "routes";
        var childEntityToolTips = {
            "endpoints": "Camel Endpoint",
            "routes": "Camel Route"
        };

        /**
        * The array of breadcrumbs so that each item in the list of bookmarks can be switched for fast navigation and
        * we can easily render the navigation path
        */
        $scope.breadcrumbs = [
            {
                name: $scope.contextId,
                items: findContexts(),
                tooltip: "Camel Context"
            },
            {
                name: defaultChildEntity,
                items: findChildEntityTypes($scope.contextId),
                tooltip: "Entity inside a Camel Context"
            },
            {
                name: $scope.endpointName || tidyJmxName($scope.routeId),
                items: findChildEntityLinks($scope.contextId, currentChildEntity()),
                tooltip: childEntityToolTips[defaultChildEntity]
            }
        ];

        // lets find all the camel contexts
        function findContexts() {
            var answer = [];
            var rootFolder = Camel.getRootCamelFolder(workspace);
            if (rootFolder) {
                angular.forEach(rootFolder.children, function (contextFolder) {
                    var id = contextFolder.title;
                    if (id && id !== $scope.contextId) {
                        var name = id;
                        var link = createLinkToFirstChildEntity(id, currentChildEntity());
                        answer.push({
                            name: name,
                            tooltip: "Camel Context",
                            link: link
                        });
                    }
                });
            }
            return answer;
        }

        // lets find all the the child entities of a camel context
        function findChildEntityTypes(contextId) {
            var answer = [];
            angular.forEach(["endpoints", "routes"], function (childEntityName) {
                if (childEntityName && childEntityName !== currentChildEntity()) {
                    var link = createLinkToFirstChildEntity(contextId, childEntityName);
                    answer.push({
                        name: childEntityName,
                        tooltip: "Entity inside a Camel Context",
                        link: link
                    });
                }
            });
            return answer;
        }

        function currentChildEntity() {
            var answer = Core.pathGet($scope, ["breadcrumbs", "childEntity"]);
            return answer || defaultChildEntity;
        }

        /**
        * Based on the current child entity type, find the child links for the given context id and
        * generate a link to the first child; used when changing context or child entity type
        */
        function createLinkToFirstChildEntity(id, childEntityValue) {
            var links = findChildEntityLinks(id, childEntityValue);

            // TODO here we should switch to a default context view if there's no endpoints available...
            var link = links.length > 0 ? links[0].link : Camel.linkToBrowseEndpointFullScreen(id, "noEndpoints");
            return link;
        }

        function findChildEntityLinks(contextId, childEntityValue) {
            if ("endpoints" === childEntityValue) {
                return findEndpoints(contextId);
            } else {
                return findRoutes(contextId);
            }
        }

        // lets find all the endpoints for the given context id
        function findEndpoints(contextId) {
            var answer = [];
            var contextFolder = Camel.getCamelContextFolder(workspace, contextId);
            if (contextFolder) {
                var endpoints = (contextFolder["children"] || []).find(function (n) {
                    return "endpoints" === n.title;
                });
                if (endpoints) {
                    angular.forEach(endpoints.children, function (endpointFolder) {
                        var entries = endpointFolder ? endpointFolder.entries : null;
                        if (entries) {
                            var endpointPath = entries["name"];
                            if (endpointPath) {
                                var name = tidyJmxName(endpointPath);
                                var link = Camel.linkToBrowseEndpointFullScreen(contextId, endpointPath);

                                answer.push({
                                    contextId: contextId,
                                    path: endpointPath,
                                    name: name,
                                    tooltip: "Endpoint",
                                    link: link
                                });
                            }
                        }
                    });
                }
            }
            return answer;
        }

        // lets find all the routes for the given context id
        function findRoutes(contextId) {
            var answer = [];
            var contextFolder = Camel.getCamelContextFolder(workspace, contextId);
            if (contextFolder) {
                var folders = (contextFolder["children"] || []).find(function (n) {
                    return "routes" === n.title;
                });
                if (folders) {
                    angular.forEach(folders.children, function (folder) {
                        var entries = folder ? folder.entries : null;
                        if (entries) {
                            var routeId = entries["name"];
                            if (routeId) {
                                var name = tidyJmxName(routeId);
                                var link = Camel.linkToRouteDiagramFullScreen(contextId, routeId);
                                answer.push({
                                    contextId: contextId,
                                    path: routeId,
                                    name: name,
                                    tooltip: "Camel Route",
                                    link: link
                                });
                            }
                        }
                    });
                }
            }
            return answer;
        }

        /**
        * Creates a link to the tree view version of this view
        */
        function linkToTreeView() {
            var answer = null;
            if ($scope.contextId) {
                var node = null;
                var tab = null;
                if ($scope.endpointPath) {
                    tab = "browseEndpoint";
                    node = workspace.findMBeanWithProperties(Camel.jmxDomain, {
                        context: $scope.contextId,
                        type: "endpoints",
                        name: $scope.endpointPath
                    });
                } else if ($scope.routeId) {
                    tab = "routes";
                    node = workspace.findMBeanWithProperties(Camel.jmxDomain, {
                        context: $scope.contextId,
                        type: "routes",
                        name: $scope.routeId
                    });
                }
                var key = node ? node["key"] : null;
                if (key && tab) {
                    answer = "#/camel/" + tab + "?tab=camel&nid=" + key;
                }
            }
            return answer;
        }

        function tidyJmxName(jmxName) {
            return jmxName ? trimQuotes(jmxName) : jmxName;
        }
    }
    Camel.BreadcrumbBarController = BreadcrumbBarController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    // NOTE this file is code generated by the ide-codegen module in Fuse IDE
    Camel.camelHeaderSchema = {
        definitions: {
            headers: {
                properties: {
                    "CamelAuthentication": {
                        type: "java.lang.String"
                    },
                    "CamelAuthenticationFailurePolicyId": {
                        type: "java.lang.String"
                    },
                    "CamelAcceptContentType": {
                        type: "java.lang.String"
                    },
                    "CamelAggregatedSize": {
                        type: "java.lang.String"
                    },
                    "CamelAggregatedTimeout": {
                        type: "java.lang.String"
                    },
                    "CamelAggregatedCompletedBy": {
                        type: "java.lang.String"
                    },
                    "CamelAggregatedCorrelationKey": {
                        type: "java.lang.String"
                    },
                    "CamelAggregationStrategy": {
                        type: "java.lang.String"
                    },
                    "CamelAggregationCompleteAllGroups": {
                        type: "java.lang.String"
                    },
                    "CamelAggregationCompleteAllGroupsInclusive": {
                        type: "java.lang.String"
                    },
                    "CamelAsyncWait": {
                        type: "java.lang.String"
                    },
                    "CamelBatchIndex": {
                        type: "java.lang.String"
                    },
                    "CamelBatchSize": {
                        type: "java.lang.String"
                    },
                    "CamelBatchComplete": {
                        type: "java.lang.String"
                    },
                    "CamelBeanMethodName": {
                        type: "java.lang.String"
                    },
                    "CamelBeanMultiParameterArray": {
                        type: "java.lang.String"
                    },
                    "CamelBinding": {
                        type: "java.lang.String"
                    },
                    "breadcrumbId": {
                        type: "java.lang.String"
                    },
                    "CamelCharsetName": {
                        type: "java.lang.String"
                    },
                    "CamelCreatedTimestamp": {
                        type: "java.lang.String"
                    },
                    "Content-Encoding": {
                        type: "java.lang.String"
                    },
                    "Content-Length": {
                        type: "java.lang.String"
                    },
                    "Content-Type": {
                        type: "java.lang.String"
                    },
                    "CamelCorrelationId": {
                        type: "java.lang.String"
                    },
                    "CamelDataSetIndex": {
                        type: "java.lang.String"
                    },
                    "org.apache.camel.default.charset": {
                        type: "java.lang.String"
                    },
                    "CamelDestinationOverrideUrl": {
                        type: "java.lang.String"
                    },
                    "CamelDisableHttpStreamCache": {
                        type: "java.lang.String"
                    },
                    "CamelDuplicateMessage": {
                        type: "java.lang.String"
                    },
                    "CamelExceptionCaught": {
                        type: "java.lang.String"
                    },
                    "CamelExceptionHandled": {
                        type: "java.lang.String"
                    },
                    "CamelEvaluateExpressionResult": {
                        type: "java.lang.String"
                    },
                    "CamelErrorHandlerHandled": {
                        type: "java.lang.String"
                    },
                    "CamelExternalRedelivered": {
                        type: "java.lang.String"
                    },
                    "CamelFailureHandled": {
                        type: "java.lang.String"
                    },
                    "CamelFailureEndpoint": {
                        type: "java.lang.String"
                    },
                    "CamelFailureRouteId": {
                        type: "java.lang.String"
                    },
                    "CamelFilterNonXmlChars": {
                        type: "java.lang.String"
                    },
                    "CamelFileLocalWorkPath": {
                        type: "java.lang.String"
                    },
                    "CamelFileName": {
                        type: "java.lang.String"
                    },
                    "CamelFileNameOnly": {
                        type: "java.lang.String"
                    },
                    "CamelFileNameProduced": {
                        type: "java.lang.String"
                    },
                    "CamelFileNameConsumed": {
                        type: "java.lang.String"
                    },
                    "CamelFilePath": {
                        type: "java.lang.String"
                    },
                    "CamelFileParent": {
                        type: "java.lang.String"
                    },
                    "CamelFileLastModified": {
                        type: "java.lang.String"
                    },
                    "CamelFileLength": {
                        type: "java.lang.String"
                    },
                    "CamelFilterMatched": {
                        type: "java.lang.String"
                    },
                    "CamelFileLockFileAcquired": {
                        type: "java.lang.String"
                    },
                    "CamelFileLockFileName": {
                        type: "java.lang.String"
                    },
                    "CamelGroupedExchange": {
                        type: "java.lang.String"
                    },
                    "CamelHttpBaseUri": {
                        type: "java.lang.String"
                    },
                    "CamelHttpCharacterEncoding": {
                        type: "java.lang.String"
                    },
                    "CamelHttpMethod": {
                        type: "java.lang.String"
                    },
                    "CamelHttpPath": {
                        type: "java.lang.String"
                    },
                    "CamelHttpProtocolVersion": {
                        type: "java.lang.String"
                    },
                    "CamelHttpQuery": {
                        type: "java.lang.String"
                    },
                    "CamelHttpResponseCode": {
                        type: "java.lang.String"
                    },
                    "CamelHttpUri": {
                        type: "java.lang.String"
                    },
                    "CamelHttpUrl": {
                        type: "java.lang.String"
                    },
                    "CamelHttpChunked": {
                        type: "java.lang.String"
                    },
                    "CamelHttpServletRequest": {
                        type: "java.lang.String"
                    },
                    "CamelHttpServletResponse": {
                        type: "java.lang.String"
                    },
                    "CamelInterceptedEndpoint": {
                        type: "java.lang.String"
                    },
                    "CamelInterceptSendToEndpointWhenMatched": {
                        type: "java.lang.String"
                    },
                    "CamelLanguageScript": {
                        type: "java.lang.String"
                    },
                    "CamelLogDebugBodyMaxChars": {
                        type: "java.lang.String"
                    },
                    "CamelLogDebugStreams": {
                        type: "java.lang.String"
                    },
                    "CamelLoopIndex": {
                        type: "java.lang.String"
                    },
                    "CamelLoopSize": {
                        type: "java.lang.String"
                    },
                    "CamelMaximumCachePoolSize": {
                        type: "java.lang.String"
                    },
                    "CamelMaximumEndpointCacheSize": {
                        type: "java.lang.String"
                    },
                    "CamelMessageHistory": {
                        type: "java.lang.String"
                    },
                    "CamelMulticastIndex": {
                        type: "java.lang.String"
                    },
                    "CamelMulticastComplete": {
                        type: "java.lang.String"
                    },
                    "CamelNotifyEvent": {
                        type: "java.lang.String"
                    },
                    "CamelOnCompletion": {
                        type: "java.lang.String"
                    },
                    "CamelOverruleFileName": {
                        type: "java.lang.String"
                    },
                    "CamelParentUnitOfWork": {
                        type: "java.lang.String"
                    },
                    "CamelRecipientListEndpoint": {
                        type: "java.lang.String"
                    },
                    "CamelReceivedTimestamp": {
                        type: "java.lang.String"
                    },
                    "CamelRedelivered": {
                        type: "java.lang.String"
                    },
                    "CamelRedeliveryCounter": {
                        type: "java.lang.String"
                    },
                    "CamelRedeliveryMaxCounter": {
                        type: "java.lang.String"
                    },
                    "CamelRedeliveryExhausted": {
                        type: "java.lang.String"
                    },
                    "CamelRedeliveryDelay": {
                        type: "java.lang.String"
                    },
                    "CamelRollbackOnly": {
                        type: "java.lang.String"
                    },
                    "CamelRollbackOnlyLast": {
                        type: "java.lang.String"
                    },
                    "CamelRouteStop": {
                        type: "java.lang.String"
                    },
                    "CamelSoapAction": {
                        type: "java.lang.String"
                    },
                    "CamelSkipGzipEncoding": {
                        type: "java.lang.String"
                    },
                    "CamelSlipEndpoint": {
                        type: "java.lang.String"
                    },
                    "CamelSplitIndex": {
                        type: "java.lang.String"
                    },
                    "CamelSplitComplete": {
                        type: "java.lang.String"
                    },
                    "CamelSplitSize": {
                        type: "java.lang.String"
                    },
                    "CamelTimerCounter": {
                        type: "java.lang.String"
                    },
                    "CamelTimerFiredTime": {
                        type: "java.lang.String"
                    },
                    "CamelTimerName": {
                        type: "java.lang.String"
                    },
                    "CamelTimerPeriod": {
                        type: "java.lang.String"
                    },
                    "CamelTimerTime": {
                        type: "java.lang.String"
                    },
                    "CamelToEndpoint": {
                        type: "java.lang.String"
                    },
                    "CamelTraceEvent": {
                        type: "java.lang.String"
                    },
                    "CamelTraceEventNodeId": {
                        type: "java.lang.String"
                    },
                    "CamelTraceEventTimestamp": {
                        type: "java.lang.String"
                    },
                    "CamelTraceEventExchange": {
                        type: "java.lang.String"
                    },
                    "Transfer-Encoding": {
                        type: "java.lang.String"
                    },
                    "CamelUnitOfWorkExhausted": {
                        type: "java.lang.String"
                    },
                    "CamelUnitOfWorkProcessSync": {
                        type: "java.lang.String"
                    },
                    "CamelXsltFileName": {
                        type: "java.lang.String"
                    }
                }
            }
        }
    };
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function SourceController($scope, workspace) {
        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateRoutes, 50);
        });

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;
            updateRoutes();
        });

        var options = {
            mode: {
                name: 'xml'
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);

        function getSource(routeXmlNode) {
            function removeCrappyHeaders(idx, e) {
                var answer = e.getAttribute("customId");
                if (e.nodeName === 'route') {
                    // always keep id on <route> element
                    answer = "true";
                }
                if (!answer || answer !== "true") {
                    e.removeAttribute("id");
                }

                // just always remove customId, _cid, and group
                e.removeAttribute("customId");
                e.removeAttribute("_cid");
                e.removeAttribute("group");
            }
            var copy = $(routeXmlNode).clone();
            copy.each(removeCrappyHeaders);
            copy.find("*").each(removeCrappyHeaders);
            var newNode = (copy && copy.length) ? copy[0] : routeXmlNode;
            return Core.xmlNodeToString(newNode);
        }

        function updateRoutes() {
            // did we select a single route
            var routeXmlNode = Camel.getSelectedRouteNode(workspace);
            if (routeXmlNode) {
                $scope.source = getSource(routeXmlNode);
                Core.$apply($scope);
            } else {
                // no then try to find the camel context and get all the routes code
                $scope.mbean = Camel.getSelectionCamelContextMBean(workspace);
                if (!$scope.mbean) {
                    // maybe the parent is the camel context folder (when we have selected the routes folder),
                    // then grab the object name from parent
                    var parent = workspace.selection.parent;
                    if (parent && parent.title === "context") {
                        $scope.mbean = parent.children[0].objectName;
                    }
                }
                if ($scope.mbean) {
                    var jolokia = workspace.jolokia;
                    jolokia.request({ type: 'exec', mbean: $scope.mbean, operation: 'dumpRoutesAsXml()' }, onSuccess(populateTable));
                }
            }
        }

        var populateTable = function (response) {
            var data = response.value;
            var selectedRouteId = Camel.getSelectedRouteId(workspace);
            if (data && selectedRouteId) {
                var doc = $.parseXML(data);
                var routes = $(doc).find('route[id="' + selectedRouteId + '"]');
                if (routes && routes.length) {
                    var selectedRoute = routes[0];

                    // TODO turn into XML?
                    var routeXml = getSource(selectedRoute);
                    if (routeXml) {
                        data = routeXml;
                    }
                }
            }
            $scope.source = data;
            Core.$apply($scope);
        };

        var saveWorked = function () {
            notification("success", "Route updated!");

            // lets clear the cached route XML so we reload the new value
            Camel.clearSelectedRouteNode(workspace);
            updateRoutes();
        };

        $scope.saveRouteXml = function () {
            var routeXml = $scope.source;
            if (routeXml) {
                var decoded = decodeURIComponent(routeXml);
                Camel.log.debug("addOrUpdateRoutesFromXml xml decoded: " + decoded);
                var jolokia = workspace.jolokia;
                var mbean = Camel.getSelectionCamelContextMBean(workspace);
                if (mbean) {
                    jolokia.execute(mbean, "addOrUpdateRoutesFromXml(java.lang.String)", decoded, onSuccess(saveWorked));
                } else {
                    notification("error", "Could not find CamelContext MBean!");
                }
            }
        };
    }
    Camel.SourceController = SourceController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function RouteController($scope, $routeParams, $element, $timeout, workspace, jolokia, localStorage) {
        var log = Logger.get("Camel");

        $scope.routes = [];
        $scope.routeNodes = {};

        $scope.contextId = $routeParams["contextId"];
        $scope.routeId = trimQuotes($routeParams["routeId"]);

        $scope.isJmxTab = !$routeParams["contextId"] || !$routeParams["routeId"];

        $scope.camelIgnoreIdForLabel = Camel.ignoreIdForLabel(localStorage);
        $scope.camelMaximumLabelWidth = Camel.maximumLabelWidth(localStorage);

        var updateRoutes = Core.throttled(doUpdateRoutes, 1000);

        // lets delay a little updating the routes to avoid timing issues where we've not yet
        // fully loaded the workspace and/or the XML model
        var delayUpdatingRoutes = 300;

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            $timeout(updateRoutes, delayUpdatingRoutes);
        });

        $scope.$watch('workspace.selection', function () {
            if ($scope.isJmxTab && workspace.moveIfViewInvalid())
                return;
            $timeout(updateRoutes, delayUpdatingRoutes);
        });

        $scope.$on('jmxTreeUpdated', function () {
            $timeout(updateRoutes, delayUpdatingRoutes);
        });

        $scope.$watch('nodeXmlNode', function () {
            if ($scope.isJmxTab && workspace.moveIfViewInvalid())
                return;
            $timeout(updateRoutes, delayUpdatingRoutes);
        });

        function doUpdateRoutes() {
            var routeXmlNode = null;
            if (!$scope.ignoreRouteXmlNode) {
                routeXmlNode = Camel.getSelectedRouteNode(workspace);
                if (!routeXmlNode) {
                    routeXmlNode = $scope.nodeXmlNode;
                }
                if (routeXmlNode && routeXmlNode.localName !== "route") {
                    var wrapper = document.createElement("route");
                    wrapper.appendChild(routeXmlNode.cloneNode(true));
                    routeXmlNode = wrapper;
                }
            }
            $scope.mbean = Camel.getSelectionCamelContextMBean(workspace);
            if (!$scope.mbean && $scope.contextId) {
                $scope.mbean = Camel.getCamelContextMBean(workspace, $scope.contextId);
            }
            if (routeXmlNode) {
                // lets show the remaining parts of the diagram of this route node
                $scope.nodes = {};
                var nodes = [];
                var links = [];
                Camel.addRouteXmlChildren($scope, routeXmlNode, nodes, links, null, 0, 0);
                showGraph(nodes, links);
            } else if ($scope.mbean) {
                jolokia.request({ type: 'exec', mbean: $scope.mbean, operation: 'dumpRoutesAsXml()' }, onSuccess(populateTable));
            } else {
                log.info("No camel context bean! Selection: " + workspace.selection);
            }
        }

        var populateTable = function (response) {
            var data = response.value;

            // routes is the xml data of the routes
            $scope.routes = data;

            // nodes and routeNodes is the GUI nodes for the processors and routes shown in the diagram
            $scope.nodes = {};
            $scope.routeNodes = {};
            var nodes = [];
            var links = [];
            var selectedRouteId = $scope.routeId;
            if (!selectedRouteId) {
                selectedRouteId = Camel.getSelectedRouteId(workspace);
            }
            if (data) {
                var doc = $.parseXML(data);
                Camel.loadRouteXmlNodes($scope, doc, selectedRouteId, nodes, links, getWidth());
                showGraph(nodes, links);
            } else {
                console.log("No data from route XML!");
            }
            Core.$apply($scope);
        };

        var postfix = " selected";

        function isSelected(node) {
            if (node) {
                var className = node.getAttribute("class");
                return className && className.endsWith(postfix);
            }
            return false;
        }

        function setSelected(node, flag) {
            var answer = false;
            if (node) {
                var className = node.getAttribute("class");
                var selected = className && className.endsWith(postfix);
                if (selected) {
                    className = className.substring(0, className.length - postfix.length);
                } else {
                    if (!flag) {
                        // no need to change!
                        return answer;
                    }
                    className = className + postfix;
                    answer = true;
                }
                node.setAttribute("class", className);
            }
            return answer;
        }

        function showGraph(nodes, links) {
            var canvasDiv = $($element);
            var width = getWidth();
            var height = getHeight();
            var svg = canvasDiv.children("svg")[0];
            $scope.graphData = Core.dagreLayoutGraph(nodes, links, width, height, svg);

            var gNodes = canvasDiv.find("g.node");
            gNodes.click(function () {
                var selected = isSelected(this);

                // lets clear all selected flags
                gNodes.each(function (idx, element) {
                    setSelected(element, false);
                });

                var cid = null;
                if (!selected) {
                    cid = this.getAttribute("data-cid");
                    setSelected(this, true);
                }
                $scope.$emit("camel.diagram.selectedNodeId", cid);
                Core.$apply($scope);
            });

            if ($scope.mbean) {
                Core.register(jolokia, $scope, {
                    type: 'exec', mbean: $scope.mbean,
                    operation: 'dumpRoutesStatsAsXml',
                    arguments: [true, true]
                }, onSuccess(statsCallback, { silent: true, error: false }));
            }
            $scope.$emit("camel.diagram.layoutComplete");
            return width;
        }

        function getWidth() {
            var canvasDiv = $($element);
            return canvasDiv.width();
        }

        function getHeight() {
            var canvasDiv = $($element);
            return Camel.getCanvasHeight(canvasDiv);
        }

        function statsCallback(response) {
            var data = response.value;
            if (data) {
                var doc = $.parseXML(data);

                var allStats = $(doc).find("routeStat");
                allStats.each(function (idx, stat) {
                    addTooltipToNode(true, stat);
                });

                var allStats = $(doc).find("processorStat");
                allStats.each(function (idx, stat) {
                    addTooltipToNode(false, stat);
                });

                // now lets try update the graph
                Core.dagreUpdateGraphData($scope.graphData);
            }

            function addTooltipToNode(isRoute, stat) {
                // we could have used a function instead of the boolean isRoute parameter (but sometimes that is easier)
                var id = stat.getAttribute("id");
                var completed = stat.getAttribute("exchangesCompleted");
                var tooltip = "";
                if (id && completed) {
                    var container = isRoute ? $scope.routeNodes : $scope.nodes;
                    var node = container[id];
                    if (!node) {
                        angular.forEach(container, function (value, key) {
                            if (!node && id === value.elementId) {
                                node = value;
                            }
                        });
                    }
                    if (node) {
                        var total = 0 + parseInt(completed);
                        var failed = stat.getAttribute("exchangesFailed");
                        if (failed) {
                            total += parseInt(failed);
                        }
                        var last = stat.getAttribute("lastProcessingTime");
                        var mean = stat.getAttribute("meanProcessingTime");
                        var min = stat.getAttribute("minProcessingTime");
                        var max = stat.getAttribute("maxProcessingTime");
                        tooltip = "last: " + last + " (ms)\nmean: " + mean + " (ms)\nmin: " + min + " (ms)\nmax: " + max + " (ms)";

                        node["counter"] = total;
                        var labelSummary = node["labelSummary"];
                        if (labelSummary) {
                            tooltip = labelSummary + "\n\n" + tooltip;
                        }
                        node["tooltip"] = tooltip;
                    } else {
                        // we are probably not showing the route for these stats
                        /*
                        var keys = Object.keys(container).sort();
                        log.info("Warning, could not find node for " + id + " when keys were: " + keys);
                        */
                    }
                }
            }
        }
    }
    Camel.RouteController = RouteController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function BrowseEndpointController($scope, $routeParams, workspace, jolokia) {
        $scope.workspace = workspace;

        $scope.forwardDialog = new Core.Dialog();

        $scope.showMessageDetails = false;
        $scope.mode = 'text';

        $scope.gridOptions = Camel.createBrowseGridOptions();

        $scope.contextId = $routeParams["contextId"];
        $scope.endpointPath = $routeParams["endpointPath"];

        $scope.isJmxTab = !$routeParams["contextId"] || !$routeParams["endpointPath"];

        $scope.$watch('workspace.selection', function () {
            if ($scope.isJmxTab && workspace.moveIfViewInvalid())
                return;
            loadData();
        });

        // TODO can we share these 2 methods from activemq browse / camel browse / came trace?
        $scope.openMessageDialog = function (message) {
            var idx = Core.pathGet(message, ["rowIndex"]);
            $scope.selectRowIndex(idx);
            if ($scope.row) {
                $scope.mode = CodeEditor.detectTextFormat($scope.row.body);
                $scope.showMessageDetails = true;
            }
        };

        $scope.selectRowIndex = function (idx) {
            $scope.rowIndex = idx;
            var selected = $scope.gridOptions.selectedItems;
            selected.splice(0, selected.length);
            if (idx >= 0 && idx < $scope.messages.length) {
                $scope.row = $scope.messages[idx];
                if ($scope.row) {
                    selected.push($scope.row);
                }
            } else {
                $scope.row = null;
            }
        };

        $scope.forwardMessagesAndCloseForwardDialog = function () {
            var mbean = Camel.getSelectionCamelContextMBean(workspace);
            var selectedItems = $scope.gridOptions.selectedItems;
            var uri = $scope.endpointUri;
            if (mbean && uri && selectedItems && selectedItems.length) {
                //console.log("Creating a new endpoint called: " + uri + " just in case!");
                jolokia.execute(mbean, "createEndpoint(java.lang.String)", uri, onSuccess(intermediateResult));

                $scope.message = "Forwarded " + Core.maybePlural(selectedItems.length, "message" + " to " + uri);
                angular.forEach(selectedItems, function (item, idx) {
                    var callback = (idx + 1 < selectedItems.length) ? intermediateResult : operationSuccess;
                    var body = item.body;
                    var headers = item.headers;

                    //console.log("sending to uri " + uri + " headers: " + JSON.stringify(headers) + " body: " + body);
                    jolokia.execute(mbean, "sendBodyAndHeaders(java.lang.String, java.lang.Object, java.util.Map)", uri, body, headers, onSuccess(callback));
                });
            }
            $scope.forwardDialog.close();
        };

        $scope.endpointUris = function () {
            var endpointFolder = Camel.getSelectionCamelContextEndpoints(workspace);
            return (endpointFolder) ? endpointFolder.children.map(function (n) {
                return n.title;
            }) : [];
        };

        $scope.refresh = loadData;

        function intermediateResult() {
        }

        function operationSuccess() {
            if ($scope.messageDialog) {
                $scope.messageDialog.close();
            }
            $scope.gridOptions.selectedItems.splice(0);
            notification("success", $scope.message);
            setTimeout(loadData, 50);
        }

        function loadData() {
            var mbean = null;
            if ($scope.contextId && $scope.endpointPath) {
                var node = workspace.findMBeanWithProperties(Camel.jmxDomain, {
                    context: $scope.contextId,
                    type: "endpoints",
                    name: $scope.endpointPath
                });
                if (node) {
                    mbean = node.objectName;
                }
            }
            if (!mbean) {
                mbean = workspace.getSelectedMBeanName();
            }
            if (mbean) {
                Camel.log.info("MBean: " + mbean);
                var options = onSuccess(populateTable);
                jolokia.execute(mbean, 'browseAllMessagesAsXml(java.lang.Boolean)', true, options);
            }
        }

        function populateTable(response) {
            var data = [];
            if (angular.isString(response)) {
                // lets parse the XML DOM here...
                var doc = $.parseXML(response);
                var allMessages = $(doc).find("message");

                allMessages.each(function (idx, message) {
                    var messageData = Camel.createMessageFromXml(message);
                    data.push(messageData);
                });
            }
            $scope.messages = data;
            Core.$apply($scope);
        }
    }
    Camel.BrowseEndpointController = BrowseEndpointController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function AttributesToolBarController($scope, workspace, jolokia) {
        $scope.deleteDialog = false;

        $scope.start = function () {
            $scope.invokeSelectedMBeans(function (item) {
                return Camel.isState(item, "suspend") ? "resume()" : "start()";
            });
        };

        $scope.pause = function () {
            $scope.invokeSelectedMBeans("suspend()");
        };

        $scope.stop = function () {
            $scope.invokeSelectedMBeans("stop()", function () {
                // lets navigate to the parent folder!
                // as this will be going way
                workspace.removeAndSelectParentNode();
            });
        };

        /*
        * Only for routes!
        */
        $scope.delete = function () {
            $scope.invokeSelectedMBeans("remove()", function () {
                // force a reload of the tree
                $scope.workspace.operationCounter += 1;
                Core.$apply($scope);
            });
        };

        $scope.anySelectionHasState = function (state) {
            var selected = $scope.selectedItems || [];
            return selected.length && selected.any(function (s) {
                return Camel.isState(s, state);
            });
        };

        $scope.everySelectionHasState = function (state) {
            var selected = $scope.selectedItems || [];
            return selected.length && selected.every(function (s) {
                return Camel.isState(s, state);
            });
        };
    }
    Camel.AttributesToolBarController = AttributesToolBarController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function EndpointController($scope, $location, localStorage, workspace, jolokia) {
        Camel.initEndpointChooserScope($scope, $location, localStorage, workspace, jolokia);

        $scope.workspace = workspace;
        $scope.message = "";

        $scope.createEndpoint = function (name) {
            var jolokia = workspace.jolokia;
            if (jolokia) {
                var mbean = Camel.getSelectionCamelContextMBean(workspace);
                if (mbean) {
                    $scope.message = "Creating endpoint " + name;
                    var operation = "createEndpoint(java.lang.String)";
                    jolokia.execute(mbean, operation, name, onSuccess(operationSuccess));
                } else {
                    notification("error", "Could not find the CamelContext MBean!");
                }
            }
        };

        $scope.createEndpointFromData = function () {
            if ($scope.selectedComponentName && $scope.endpointPath) {
                var name = $scope.selectedComponentName + "://" + $scope.endpointPath;
                console.log("Have endpoint data " + JSON.stringify($scope.endpointParameters));

                var params = "";
                angular.forEach($scope.endpointParameters, function (value, key) {
                    var prefix = params ? "&" : "";
                    params += prefix + key + "=" + value;
                });
                if (params) {
                    name += "?" + params;
                }

                // TODO use form data too for URIs parameters...
                $scope.createEndpoint(name);
            }
        };

        $scope.deleteEndpoint = function () {
            var jolokia = workspace.jolokia;
            var selection = workspace.selection;
            var entries = selection.entries;
            if (selection && jolokia && entries) {
                var domain = selection.domain;
                var brokerName = entries["BrokerName"];
                var name = entries["Destination"];
                var isQueue = "Topic" !== entries["Type"];
                if (domain && brokerName) {
                    var mbean = "" + domain + ":BrokerName=" + brokerName + ",Type=Broker";
                    $scope.message = "Deleting " + (isQueue ? "queue" : "topic") + " " + name;
                    var operation = "removeEndpoint(java.lang.String)";
                    jolokia.execute(mbean, operation, name, onSuccess(deleteSuccess));
                }
            }
        };

        function operationSuccess() {
            $scope.endpointName = "";
            $scope.workspace.operationCounter += 1;
            Core.$apply($scope);
            notification("success", $scope.message);
        }

        function deleteSuccess() {
            // lets set the selection to the parent
            if (workspace.selection) {
                var parent = workspace.selection.parent;
                if (parent) {
                    $scope.workspace.updateSelectionNode(parent);
                }
            }
            $scope.workspace.operationCounter += 1;
            Core.$apply($scope);
            notification("success", $scope.message);
        }
    }
    Camel.EndpointController = EndpointController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function FabricDiagramController($scope, $compile, $location, localStorage, jolokia, workspace) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        var isFmc = Fabric.isFMCContainer(workspace);
        $scope.isFmc = isFmc;

        $scope.selectedNode = null;

        var defaultFlags = {
            panel: true,
            popup: false,
            label: true,
            container: false,
            endpoint: true,
            route: true,
            consumer: true,
            producer: true
        };

        $scope.viewSettings = {};

        $scope.shapeSize = {
            context: 14
        };

        var graphBuilder = new ForceGraph.GraphBuilder();

        Core.bindModelToSearchParam($scope, $location, "searchFilter", "q", "");

        angular.forEach(defaultFlags, function (defaultValue, key) {
            var modelName = "viewSettings." + key;

            // bind model values to search params...
            function currentValue() {
                var answer = $location.search()[paramName] || defaultValue;
                return answer === "false" ? false : answer;
            }

            var paramName = key;
            var value = currentValue();
            Core.pathSet($scope, modelName, value);

            $scope.$watch(modelName, function () {
                var current = Core.pathGet($scope, modelName);
                var old = currentValue();
                if (current !== old) {
                    var defaultValue = defaultFlags[key];
                    if (current !== defaultValue) {
                        if (!current) {
                            current = "false";
                        }
                        $location.search(paramName, current);
                    } else {
                        $location.search(paramName, null);
                    }
                }
                redrawGraph();
            });
        });

        $scope.connectToContext = function () {
            var selectedNode = $scope.selectedNode;
            if (selectedNode) {
                var container = selectedNode["container"] || selectedNode;
                var postfix = null;
                connectToContainer(container, postfix);
            }
        };

        $scope.connectToEndpoint = function () {
            var selectedNode = $scope.selectedNode;
            if (selectedNode) {
                var container = selectedNode["container"] || selectedNode;
                var postfix = null;

                /*
                var brokerName = selectedNode["brokerName"];
                var destinationType = selectedNode["destinationType"];
                var destinationName = selectedNode["destinationName"];
                if (brokerName && destinationType && destinationName) {
                postfix = "nid=root-org.apache.activemq-Broker-" + brokerName + "-" + destinationType + "-" + destinationName;
                }
                */
                connectToContainer(container, postfix);
            }
        };

        function connectToContainer(container, postfix, viewPrefix) {
            if (typeof viewPrefix === "undefined") { viewPrefix = "/jmx/attributes?tab=camel"; }
            var view = viewPrefix;
            if (postfix) {
                view += postfix;
            }

            // TODO if local just link to local view!
            $scope.doConnect(container, view);
        }

        $scope.$on('$destroy', function (event) {
            stopOldJolokia();
        });

        function stopOldJolokia() {
            var oldJolokia = $scope.selectedNodeJolokia;
            if (oldJolokia && oldJolokia !== jolokia) {
                oldJolokia.stop();
            }
        }

        $scope.$watch("selectedNode", function (newValue, oldValue) {
            // lets cancel any previously registered thingy
            if ($scope.unregisterFn) {
                $scope.unregisterFn();
                $scope.unregisterFn = null;
            }
            var node = $scope.selectedNode;
            if (node) {
                var mbean = node.objectName;
                var container = node.container || {};
                var nodeJolokia = node.jolokia || container.jolokia || jolokia;
                if (nodeJolokia !== $scope.selectedNodeJolokia) {
                    stopOldJolokia();
                    $scope.selectedNodeJolokia = nodeJolokia;
                    if (nodeJolokia !== jolokia) {
                        var rate = Core.parseIntValue(localStorage['updateRate'] || "2000", "update rate");
                        if (rate) {
                            nodeJolokia.start(rate);
                        }
                    }
                }
                var dummyResponse = { value: node.panelProperties || {} };
                if (mbean && nodeJolokia) {
                    $scope.unregisterFn = Core.register(nodeJolokia, $scope, {
                        type: 'read', mbean: mbean
                    }, onSuccess(renderNodeAttributes, { error: function (response) {
                            // probably we've got a wrong mbean name?
                            // so lets render at least
                            renderNodeAttributes(dummyResponse);
                            Core.defaultJolokiaErrorHandler(response);
                        } }));
                } else {
                    renderNodeAttributes(dummyResponse);
                }
            }
        });

        var ignoreNodeAttributes = [
            "CamelId", "CamelManagementName"
        ];

        var ignoreNodeAttributesByType = {
            context: ["ApplicationContextClassName", "CamelId", "ClassResolver", "ManagementName", "PackageScanClassResolver", "Properties"],
            endpoint: ["Camel", "Endpoint"],
            route: ["Description"]
        };

        var onlyShowAttributesByType = {
            broker: []
        };

        function renderNodeAttributes(response) {
            var properties = [];
            if (response) {
                var value = response.value || {};
                $scope.selectedNodeAttributes = value;
                var selectedNode = $scope.selectedNode || {};
                var container = selectedNode['container'] || {};
                var nodeType = selectedNode["type"];
                var brokerName = selectedNode["brokerName"];
                var containerId = container["id"];
                var group = selectedNode["group"] || container["group"];
                var jolokiaUrl = selectedNode["jolokiaUrl"] || container["jolokiaUrl"];
                var profile = selectedNode["profile"] || container["profile"];
                var version = selectedNode["version"] || container["version"];

                var isBroker = nodeType && nodeType.startsWith("broker");
                var ignoreKeys = ignoreNodeAttributes.concat(ignoreNodeAttributesByType[nodeType] || []);
                var onlyShowKeys = onlyShowAttributesByType[nodeType];

                angular.forEach(value, function (v, k) {
                    if (onlyShowKeys ? onlyShowKeys.indexOf(k) >= 0 : ignoreKeys.indexOf(k) < 0) {
                        var formattedValue = Core.humanizeValueHtml(v);
                        properties.push({ key: humanizeValue(k), value: formattedValue });
                    }
                });
                properties = properties.sortBy("key");

                /*
                var brokerProperty: any = null;
                if (brokerName) {
                var html = brokerName;
                if (version && profile) {
                var brokerLink = Fabric.brokerConfigLink(workspace, jolokia, localStorage, version, profile, brokerName);
                if (brokerLink) {
                html = $compile('<a target="broker" ng-click="connectToContext()">' +
                '<img title="Apache ActiveMQ" src="app/fabric/img/message_broker.png"> ' + brokerName +
                '</a> <a title="configuration settings" target="brokerConfig" href="' + brokerLink +
                '"><i class="icon-tasks"></i></a>')($scope);
                }
                }
                brokerProperty = {key: "Broker", value: html};
                if (!isBroker) {
                properties.splice(0, 0, brokerProperty);
                }
                }
                
                */
                if (containerId && isFmc) {
                    var containerModel = "selectedNode.container";
                    properties.splice(0, 0, { key: "Container", value: $compile('<div fabric-container-link="' + containerModel + '"></div>')($scope) });
                }

                /*
                var destinationName = value["DestinationName"] || selectedNode["destinationName"];
                if (destinationName && (nodeType !== "queue" && nodeType !== "topic")) {
                var destinationTypeName = getDestinationTypeName(value);
                var html = createDestinationLink(destinationName, destinationTypeName);
                properties.splice(0, 0, {key: destinationTypeName, value: html});
                }
                */
                var typeLabel = selectedNode["typeLabel"];
                var name = selectedNode["name"] || selectedNode["id"] || selectedNode['objectName'];
                if (typeLabel) {
                    var html = name;
                    if (nodeType === "queue" || nodeType === "topic") {
                        html = createDestinationLink(name, nodeType);
                    }
                    var typeProperty = { key: typeLabel, value: html };

                    /*
                    if (isBroker && brokerProperty) {
                    typeProperty = brokerProperty;
                    }
                    */
                    properties.splice(0, 0, typeProperty);
                }
            }
            $scope.selectedNodeProperties = properties;
            Core.$apply($scope);
        }

        /**
        * Generates the HTML for a link to the destination
        */
        function createDestinationLink(destinationName, destinationType) {
            if (typeof destinationType === "undefined") { destinationType = "queue"; }
            return $compile('<a target="destination" title="' + destinationName + '" ng-click="connectToEndpoint()">' + destinationName + '</a>')($scope);
        }

        $scope.$watch("searchFilter", function (newValue, oldValue) {
            redrawGraph();
        });

        if (isFmc) {
            $scope.versionId = Fabric.getDefaultVersionId(jolokia);
            var fields = ["id", "alive", "parentId", "profileIds", "versionId", "provisionResult", "jolokiaUrl", "jmxDomains"];
            Fabric.getContainersFields(jolokia, fields, onFabricContainerData);
        } else {
            // lets just use the current stuff from the workspace
            $scope.$watch('workspace.tree', function () {
                reloadLocalJmxTree();
            });

            $scope.$on('jmxTreeUpdated', function () {
                reloadLocalJmxTree();
            });
        }

        function reloadLocalJmxTree() {
            var localContainer = {
                jolokia: jolokia
            };
            $scope.activeContainers = {
                "local": localContainer
            };
            redrawGraph();
            $scope.containerCount = 1;
        }

        function onFabricContainerData(response) {
            if (response) {
                var responseJson = angular.toJson(response);
                if ($scope.responseJson === responseJson) {
                    return;
                }
                $scope.responseJson = responseJson;

                var containersToDelete = $scope.activeContainers || {};
                $scope.activeContainers = (response || {}).filter(function (c) {
                    return c.jmxDomains.any(Camel.jmxDomain);
                });
                $scope.containerCount = $scope.activeContainers.length;

                // query containers which have camel...
                redrawGraph();
            } else {
                $scope.containerCount = 0;
            }
        }

        function redrawGraph() {
            graphBuilder = new ForceGraph.GraphBuilder();

            // TODO delete any nodes from dead containers in containersToDelete
            angular.forEach($scope.activeContainers, function (container, id) {
                var containerJolokia = container.jolokia;
                if (!containerJolokia) {
                    var jolokiaUrl = container["jolokiaUrl"];
                    if (jolokiaUrl) {
                        var url = Core.useProxyIfExternal(jolokiaUrl);
                        containerJolokia = Fabric.createJolokia(url);
                    }
                }
                if (containerJolokia) {
                    onContainerJolokia(containerJolokia, container);
                } else {
                    Fabric.containerJolokia(jolokia, id, function (containerJolokia) {
                        return onContainerJolokia(containerJolokia, container);
                    });
                }
            });
            $scope.graph = graphBuilder.buildGraph();
            Core.$apply($scope);
        }

        /**
        * Returns true if the given CamelContext ID matches the current search filter
        */
        function matchesContextId(contextId) {
            if (contextId) {
                return !$scope.searchFilter || contextId.indexOf($scope.searchFilter) >= 0;
            }
            return false;
        }

        function onContainerJolokia(containerJolokia, container) {
            if (containerJolokia) {
                container.jolokia = containerJolokia;
                var containerId = container.id || "local";
                var idPrefix = containerId + ":";

                var endpointUriToObject = {};

                function getOrCreateCamelContext(contextId) {
                    var answer = null;
                    if (matchesContextId(contextId)) {
                        // try guess the mbean name
                        var contextMBean = Camel.jmxDomain + ':context=' + contextId + ',type=context,name="' + contextId + '"';
                        var contextAttributes = {
                            contextId: contextId
                        };

                        answer = getOrAddNode("context", idPrefix + contextId, contextAttributes, function () {
                            return {
                                name: contextId,
                                typeLabel: "CamelContext",
                                container: container,
                                objectName: contextMBean,
                                jolokia: containerJolokia,
                                popup: {
                                    title: "CamelContext: " + contextId,
                                    content: ""
                                }
                            };
                        });
                    }
                    return answer;
                }

                // find endpoints
                if ($scope.viewSettings.endpoint) {
                    containerJolokia.search("org.apache.camel:type=endpoints,*", onSuccess(function (response) {
                        angular.forEach(response, function (objectName) {
                            var details = Core.parseMBean(objectName);
                            var attributes = details['attributes'];

                            //log.info("attributes: " + angular.toJson(attributes));
                            var contextId = attributes["context"];
                            var uri = trimQuotes(attributes["name"]);
                            Camel.log.info("context " + contextId + " endpoint " + uri);
                            attributes["uri"] = uri;
                            attributes["mbean"] = objectName;
                            attributes["container"] = container;

                            if (uri && matchesContextId(contextId)) {
                                var endpoint = getOrAddNode("endpoint", idPrefix + uri, attributes, function () {
                                    return {
                                        name: uri,
                                        typeLabel: "Endpoint",
                                        container: container,
                                        objectName: objectName,
                                        jolokia: containerJolokia,
                                        popup: {
                                            title: "Endpoint: " + uri,
                                            content: "<p>context: " + contextId + "</p>"
                                        }
                                    };
                                });
                                if (endpoint) {
                                    endpointUriToObject[uri] = endpoint;
                                }
                                var camelContext = getOrCreateCamelContext(contextId);
                                addLink(camelContext, endpoint, "endpoint");
                            }
                        });
                        graphModelUpdated();
                    }));
                }

                // find routes
                if ($scope.viewSettings.route) {
                    containerJolokia.request({ type: "read", mbean: "org.apache.camel:type=routes,*", attribute: ["EndpointUri"] }, onSuccess(function (response) {
                        angular.forEach(response.value, function (properties, objectName) {
                            var details = Core.parseMBean(objectName);
                            var attributes = details['attributes'];
                            Camel.log.info("route attributes: " + angular.toJson(attributes) + " properties: " + angular.toJson(properties));
                            var contextId = attributes["context"];
                            var routeId = trimQuotes(attributes["name"]);
                            Camel.log.info("context " + contextId + " routeId " + routeId);
                            attributes["routeId"] = routeId;
                            attributes["mbean"] = objectName;
                            attributes["container"] = container;
                            attributes["type"] = "route";

                            if (routeId && matchesContextId(contextId)) {
                                var route = getOrAddNode("route", idPrefix + routeId, attributes, function () {
                                    return {
                                        name: routeId,
                                        typeLabel: "Route",
                                        container: container,
                                        objectName: objectName,
                                        jolokia: containerJolokia,
                                        popup: {
                                            title: "Route: " + routeId,
                                            content: "<p>context: " + contextId + "</p>"
                                        }
                                    };
                                });
                                var uri = properties["EndpointUri"];
                                if (uri && route) {
                                    var endpoint = endpointUriToObject[uri];
                                    Camel.log.info("found route endpoint " + endpoint + " for uri " + uri);
                                    addLink(route, endpoint, "consumer");
                                }
                                var camelContext = getOrCreateCamelContext(contextId);
                                addLink(camelContext, route, "route");
                            }
                        });
                        graphModelUpdated();
                    }));
                }
            }
        }

        function graphModelUpdated() {
            $scope.graph = graphBuilder.buildGraph();
            Core.$apply($scope);
        }

        function getOrAddNode(typeName, id, properties, createFn) {
            var node = null;
            if (id) {
                var nodeId = typeName + ":" + id;
                node = graphBuilder.getNode(nodeId);
                if (!node) {
                    var nodeValues = createFn();
                    node = angular.copy(properties);
                    angular.forEach(nodeValues, function (value, key) {
                        return node[key] = value;
                    });

                    node['id'] = nodeId;
                    if (!node['type']) {
                        node['type'] = typeName;
                    }
                    if (!node['name']) {
                        node['name'] = id;
                    }
                    if (node) {
                        var size = $scope.shapeSize[typeName];
                        if (size && !node['size']) {
                            node['size'] = size;
                        }
                        if (!node['summary']) {
                            node['summary'] = node['popup'] || "";
                        }
                        if (!$scope.viewSettings.popup) {
                            delete node['popup'];
                        }
                        if (!$scope.viewSettings.label) {
                            delete node['name'];
                        }

                        // lets not add nodes which are defined as being disabled
                        var enabled = $scope.viewSettings[typeName];
                        if (enabled || !angular.isDefined(enabled)) {
                            //log.info("Adding node " + nodeId + " of type + " + typeName);
                            graphBuilder.addNode(node);
                        } else {
                            //log.info("Ignoring node " + nodeId + " of type + " + typeName);
                        }
                    }
                }
            }
            return node;
        }

        function addLink(object1, object2, linkType) {
            if (object1 && object2) {
                addLinkIds(object1.id, object2.id, linkType);
            }
        }

        function addLinkIds(id1, id2, linkType) {
            if (id1 && id2) {
                graphBuilder.addLink(id1, id2, linkType);
            }
        }

        /**
        * Avoid the JMX type property clashing with the ForceGraph type property; used for associating css classes with nodes on the graph
        *
        * @param properties
        */
        function renameTypeProperty(properties) {
            properties.mbeanType = properties['type'];
            delete properties['type'];
        }

        function configureEndpointProperties(properties) {
            renameTypeProperty(properties);
            var destinationType = properties.destinationType || "Queue";
            var typeName = destinationType.toLowerCase();
            properties.isQueue = !typeName.startsWith("t");
            properties['destType'] = typeName;
        }
    }
    Camel.FabricDiagramController = FabricDiagramController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function TraceRouteController($scope, workspace, jolokia, localStorage, tracerStatus) {
        var log = Logger.get("CamelTracer");

        $scope.camelMaximumTraceOrDebugBodyLength = Camel.maximumTraceOrDebugBodyLength(localStorage);
        $scope.tracing = false;
        $scope.messages = [];
        $scope.graphView = null;
        $scope.tableView = null;
        $scope.mode = 'text';

        $scope.messageDialog = new Core.Dialog();

        $scope.gridOptions = Camel.createBrowseGridOptions();
        $scope.gridOptions.selectWithCheckboxOnly = false;
        $scope.gridOptions.showSelectionCheckbox = false;
        $scope.gridOptions.multiSelect = false;
        $scope.gridOptions.afterSelectionChange = onSelectionChanged;
        $scope.gridOptions.columnDefs.push({
            field: 'toNode',
            displayName: 'To Node'
        });

        $scope.startTracing = function () {
            log.info("Start tracing");
            setTracing(true);
        };

        $scope.stopTracing = function () {
            log.info("Stop tracing");
            setTracing(false);
        };

        $scope.clear = function () {
            log.debug("Clear messages");
            tracerStatus.messages = [];
            $scope.messages = [];
            Core.$apply($scope);
        };

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid()) {
                return;
            }
            $scope.messages = tracerStatus.messages;
            reloadTracingFlag();
        });

        // TODO can we share these 2 methods from activemq browse / camel browse / came trace?
        $scope.openMessageDialog = function (message) {
            var idx = Core.pathGet(message, ["rowIndex"]);
            $scope.selectRowIndex(idx);
            if ($scope.row) {
                $scope.mode = CodeEditor.detectTextFormat($scope.row.body);
                $scope.messageDialog.open();
            }
        };

        $scope.selectRowIndex = function (idx) {
            $scope.rowIndex = idx;
            var selected = $scope.gridOptions.selectedItems;
            selected.splice(0, selected.length);
            if (idx >= 0 && idx < $scope.messages.length) {
                $scope.row = $scope.messages[idx];
                if ($scope.row) {
                    selected.push($scope.row);
                }
            } else {
                $scope.row = null;
            }
            onSelectionChanged();
        };

        function reloadTracingFlag() {
            $scope.tracing = false;

            // clear any previous polls
            if (tracerStatus.jhandle != null) {
                log.debug("Unregistering jolokia handle");
                jolokia.unregister(tracerStatus.jhandle);
                tracerStatus.jhandle = null;
            }

            var mbean = Camel.getSelectionCamelTraceMBean(workspace);
            if (mbean) {
                $scope.tracing = jolokia.getAttribute(mbean, "Enabled", onSuccess(null));

                if ($scope.tracing) {
                    var traceMBean = mbean;
                    if (traceMBean) {
                        // register callback for doing live update of tracing
                        if (tracerStatus.jhandle === null) {
                            log.debug("Registering jolokia handle");
                            tracerStatus.jhandle = jolokia.register(populateRouteMessages, {
                                type: 'exec', mbean: traceMBean,
                                operation: 'dumpAllTracedMessagesAsXml()',
                                ignoreErrors: true,
                                arguments: []
                            });
                        }
                    }
                    $scope.graphView = "app/camel/html/routes.html";
                    $scope.tableView = "app/camel/html/browseMessages.html";
                } else {
                    tracerStatus.messages = [];
                    $scope.messages = [];
                    $scope.graphView = null;
                    $scope.tableView = null;
                }
            }
        }

        function populateRouteMessages(response) {
            log.debug("Populating response " + response);

            // filter messages due CAMEL-7045 but in camel-core
            // see https://github.com/hawtio/hawtio/issues/292
            var selectedRouteId = Camel.getSelectedRouteId(workspace);

            var xml = response.value;
            if (angular.isString(xml)) {
                // lets parse the XML DOM here...
                var doc = $.parseXML(xml);
                var allMessages = $(doc).find("fabricTracerEventMessage");
                if (!allMessages || !allMessages.length) {
                    // lets try find another element name
                    allMessages = $(doc).find("backlogTracerEventMessage");
                }

                allMessages.each(function (idx, message) {
                    var routeId = $(message).find("routeId").text();
                    if (routeId === selectedRouteId) {
                        var messageData = Camel.createMessageFromXml(message);
                        var toNode = $(message).find("toNode").text();
                        if (toNode) {
                            messageData["toNode"] = toNode;
                        }
                        log.debug("Adding new message to trace table with id " + messageData["id"]);
                        $scope.messages.push(messageData);
                    }
                });

                // keep state of the traced messages on tracerStatus
                tracerStatus.messages = $scope.messages;
                Core.$apply($scope);
            }
        }

        function onSelectionChanged() {
            angular.forEach($scope.gridOptions.selectedItems, function (selected) {
                if (selected) {
                    var toNode = selected["toNode"];
                    if (toNode) {
                        // lets highlight the node in the diagram
                        var nodes = d3.select("svg").selectAll("g .node");
                        Camel.highlightSelectedNode(nodes, toNode);
                    }
                }
            });
        }

        function tracingChanged(response) {
            reloadTracingFlag();
            Core.$apply($scope);
        }

        function setTracing(flag) {
            var mbean = Camel.getSelectionCamelTraceMBean(workspace);
            if (mbean) {
                // set max only supported on BacklogTracer
                // (the old fabric tracer does not support max length)
                if (mbean.toString().endsWith("BacklogTracer")) {
                    var max = $scope.camelMaximumTraceOrDebugBodyLength;
                    jolokia.setAttribute(mbean, "BodyMaxChars", max);
                }
                jolokia.setAttribute(mbean, "Enabled", flag, onSuccess(tracingChanged));
            }
        }

        log.info("Re-activating tracer with " + tracerStatus.messages.length + " existing messages");
        $scope.messages = tracerStatus.messages;
        $scope.tracing = tracerStatus.jhandle != null;
    }
    Camel.TraceRouteController = TraceRouteController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function ProfileRouteController($scope, $location, workspace, jolokia) {
        $scope.data = [];
        $scope.calcManually = true;
        $scope.icons = {};
        $scope.selectedRouteId = "";

        var columnDefs = [
            {
                field: 'id',
                displayName: 'Id',
                cellTemplate: '<div class="ngCellText" ng-bind-html-unsafe="rowIcon(row.entity.id)"></div>',
                cellFilter: null,
                width: "**",
                resizable: true
            },
            {
                field: 'count',
                displayName: 'Count',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'last',
                displayName: 'Last',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'delta',
                displayName: 'Delta',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'mean',
                displayName: 'Mean',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'min',
                displayName: 'Min',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'max',
                displayName: 'Max',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'total',
                displayName: 'Total',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'self',
                displayName: 'Self',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.rowIcon = function (id) {
            var entry = $scope.icons[id];
            if (entry) {
                return entry.img + " " + id;
            } else {
                return id;
            }
        };

        $scope.gridOptions = {
            data: 'data',
            displayFooter: true,
            displaySelectionCheckbox: false,
            canSelectRows: false,
            enableSorting: false,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            }
        };

        var populateProfileMessages = function (response) {
            var updatedData = [];

            // its xml structure so we need to parse it
            var xml = response.value;
            if (angular.isString(xml)) {
                // lets parse the XML DOM here...
                var doc = $.parseXML(xml);

                var routeMessages = $(doc).find("routeStat");

                routeMessages.each(function (idx, message) {
                    var messageData = {
                        id: {},
                        count: {},
                        last: {},
                        delta: {},
                        mean: {},
                        min: {},
                        max: {},
                        total: {},
                        self: {}
                    };

                    // compare counters, as we only update if we have new data
                    messageData.id = message.getAttribute("id");

                    var total = 0;
                    total += +message.getAttribute("exchangesCompleted");
                    total += +message.getAttribute("exchangesFailed");
                    messageData.count = total;
                    messageData.last = message.getAttribute("lastProcessingTime");

                    // delta is only avail from Camel 2.11 onwards
                    var delta = message.getAttribute("deltaProcessingTime");
                    if (delta) {
                        messageData.delta = delta;
                    } else {
                        messageData.delta = 0;
                    }
                    messageData.mean = message.getAttribute("meanProcessingTime");
                    messageData.min = message.getAttribute("minProcessingTime");
                    messageData.max = message.getAttribute("maxProcessingTime");
                    messageData.total = message.getAttribute("totalProcessingTime");

                    // self is pre calculated from Camel 2.11 onwards
                    var self = message.getAttribute("selfProcessingTime");
                    if (self) {
                        messageData.self = self;
                    } else {
                        // we need to calculate this manually
                        $scope.calcManually = true;
                        messageData.self = "0";
                    }

                    updatedData.push(messageData);
                });

                var processorMessages = $(doc).find("processorStat");

                processorMessages.each(function (idx, message) {
                    var messageData = {
                        id: {},
                        count: {},
                        last: {},
                        delta: {},
                        mean: {},
                        min: {},
                        max: {},
                        total: {},
                        self: {}
                    };

                    messageData.id = message.getAttribute("id");
                    var total = 0;
                    total += +message.getAttribute("exchangesCompleted");
                    total += +message.getAttribute("exchangesFailed");
                    messageData.count = total;
                    messageData.last = message.getAttribute("lastProcessingTime");

                    // delta is only avail from Camel 2.11 onwards
                    var delta = message.getAttribute("deltaProcessingTime");
                    if (delta) {
                        messageData.delta = delta;
                    } else {
                        messageData.delta = 0;
                    }
                    messageData.mean = message.getAttribute("meanProcessingTime");
                    messageData.min = message.getAttribute("minProcessingTime");
                    messageData.max = message.getAttribute("maxProcessingTime");

                    // total time for processors is pre calculated as accumulated from Camel 2.11 onwards
                    var apt = message.getAttribute("accumulatedProcessingTime");
                    if (apt) {
                        messageData.total = apt;
                    } else {
                        messageData.total = "0";
                    }

                    // self time for processors is their total time
                    messageData.self = message.getAttribute("totalProcessingTime");

                    updatedData.push(messageData);
                });
            }

            // for Camel 2.10 or older we need to run through the data and calculate the self/total times manually
            if ($scope.calcManually) {
                // sort the data accordingly to order in the icons map
                updatedData.sort(function (e1, e2) {
                    var entry1 = $scope.icons[e1.id];
                    var entry2 = $scope.icons[e2.id];
                    if (entry1 && entry2) {
                        return entry1.index - entry2.index;
                    } else {
                        return 0;
                    }
                });

                var accTotal = 0;
                updatedData.reverse().forEach(function (data, idx) {
                    // update accTotal with self time
                    if (idx < updatedData.length - 1) {
                        // each processor should have the total updated with the accumulated total
                        accTotal += +data.self;
                        data.total = accTotal;
                    } else {
                        // the last row is the route, which should have self calculated as follows
                        data.self = +(data.total - accTotal);

                        // just to be safe we dont want negative values self value for the route
                        if (data.self < 0) {
                            data.self = 0;
                        }
                    }
                });

                // reverse back again
                updatedData.reverse();
            }

            // TODO: need a way to update data without flickering
            // if we do as below with the forEach then the data does not update
            // replace data with updated data
            $scope.data = updatedData;

            Core.$apply($scope);
        };

        // function to trigger reloading page
        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
            loadData();
        };

        $scope.$watch('workspace.tree', function () {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        });

        function initIdToIcon() {
            console.log("initializing id and icons");

            $scope.icons = {};
            var routeXml = Core.pathGet(workspace.selection, ["routeXmlNode"]);
            if (routeXml) {
                // add route id first
                var entry = {
                    img: "",
                    index: 0
                };
                entry.index = -1;
                entry.img = "<img src='app/camel/img/camel_route.png'>";
                $scope.icons[$scope.selectedRouteId] = entry;

                // then each processor id and icons
                $(routeXml).find('*').each(function (idx, element) {
                    var id = element.getAttribute("id");
                    if (id) {
                        var entry = {
                            img: "",
                            index: 0
                        };
                        entry.index = idx;
                        var icon = Camel.getRouteNodeIcon(element);
                        if (icon) {
                            entry.img = "<img src='" + icon + "'>";
                        } else {
                            entry.img = "";
                        }
                        $scope.icons[id] = entry;
                    }
                });
            }
        }

        function loadData() {
            console.log("Loading Camel route profile data...");
            $scope.selectedRouteId = Camel.getSelectedRouteId(workspace);
            var routeMBean = Camel.getSelectionRouteMBean(workspace, $scope.selectedRouteId);
            console.log("Selected route is " + $scope.selectedRouteId);

            var camelVersion = Camel.getCamelVersion(workspace, jolokia);
            if (camelVersion) {
                console.log("Camel version " + camelVersion);
                camelVersion += "camel-";
                var numbers = Core.parseVersionNumbers(camelVersion);
                if (Core.compareVersionNumberArrays(numbers, [2, 11]) >= 0) {
                    // this is Camel 2.11 or better so we dont need to calculate data manually
                    console.log("Camel 2.11 or better detected");
                    $scope.calcManually = false;
                } else {
                    console.log("Camel 2.10 or older detected");
                    $scope.calcManually = true;
                }
            }

            initIdToIcon();
            console.log("Initialized icons, with " + $scope.icons.length + " icons");

            // schedule update the profile data, based on the configured interval
            // TOOD: the icons is not initialized the first time, for some reason, the routeXmlNode is empty/undefined
            // TODO: have cellFilter with bar grey-scale for highlighting the scales between the numbers
            // TODO: have the icons indent, there is some CSS ninja crack to do this
            var query = { type: 'exec', mbean: routeMBean, operation: 'dumpRouteStatsAsXml(boolean,boolean)', arguments: [false, true] };
            scopeStoreJolokiaHandle($scope, jolokia, jolokia.register(populateProfileMessages, query));
        }
    }
    Camel.ProfileRouteController = ProfileRouteController;
})(Camel || (Camel = {}));
var Camel;
(function (Camel) {
    function PropertiesController($scope, workspace) {
        $scope.viewTemplate = null;
        $scope.schema = _apacheCamelModel;

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateData, 50);
        });

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;
            updateData();
        });

        function updateData() {
            var routeXmlNode = Camel.getSelectedRouteNode(workspace);
            $scope.nodeData = Camel.getRouteNodeJSON(routeXmlNode);

            if (routeXmlNode) {
                var nodeName = routeXmlNode.nodeName;
                $scope.model = Camel.getCamelSchema(nodeName);

                if ($scope.model) {
                    console.log("data is: " + JSON.stringify($scope.nodeData, null, "  "));
                    console.log("model schema is: " + JSON.stringify($scope.model, null, "  "));

                    $scope.viewTemplate = "app/camel/html/nodePropertiesView.html";
                }
            }
        }
    }
    Camel.PropertiesController = PropertiesController;
})(Camel || (Camel = {}));
var Core;
(function (Core) {
    var PageTitle = (function () {
        function PageTitle() {
            this.titleElements = [];
        }
        PageTitle.prototype.addTitleElement = function (element) {
            this.titleElements.push(element);
        };

        PageTitle.prototype.getTitle = function () {
            return this.getTitleExcluding([], ' ');
        };

        PageTitle.prototype.getTitleWithSeparator = function (separator) {
            return this.getTitleExcluding([], separator);
        };

        PageTitle.prototype.getTitleExcluding = function (excludes, separator) {
            return this.getTitleArrayExcluding(excludes).join(separator);
        };

        PageTitle.prototype.getTitleArrayExcluding = function (excludes) {
            return this.titleElements.map(function (element) {
                var answer = '';
                if (element) {
                    answer = element();
                    if (answer === null) {
                        return '';
                    }
                }
                return answer;
            }).exclude(excludes).exclude('');
        };
        return PageTitle;
    })();
    Core.PageTitle = PageTitle;
})(Core || (Core = {}));
var Forms;
(function (Forms) {
    var SubmitForm = (function () {
        function SubmitForm() {
            var _this = this;
            this.restrict = 'A';
            this.scope = true;
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        SubmitForm.prototype.doLink = function (scope, element, attrs) {
            var el = $(element);

            var target = 'form[name=' + attrs['hawtioSubmit'] + ']';

            el.click(function () {
                $(target).submit();
                return false;
            });
        };
        return SubmitForm;
    })();
    Forms.SubmitForm = SubmitForm;
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    var ResetForm = (function () {
        function ResetForm() {
            var _this = this;
            this.restrict = 'A';
            this.scope = true;
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        ResetForm.prototype.doLink = function (scope, element, attrs) {
            var el = $(element);

            var target = 'form[name=' + attrs['hawtioReset'] + ']';

            el.click(function () {
                var forms = $(target);
                for (var i = 0; i < forms.length; i++) {
                    forms[i].reset();
                }
                return false;
            });
        };
        return ResetForm;
    })();
    Forms.ResetForm = ResetForm;
})(Forms || (Forms = {}));
/**
* @module Forms
*/
var Forms;
(function (Forms) {
    /**
    * Create a DOM widget tree for the given set of form configuration data.
    *
    * This will include either the standard AngularJS widgets or custom widgets
    * @method createWidget
    * @param {String} propTypeName
    * @param {any} property
    * @param {any} schema
    * @param {any} config
    * @param {String} id
    * @param {Boolean ignorePrefixInLabel
    * @param {String} configScoepName
    * @param {Boolean} wrapInGroup
    */
    function createWidget(propTypeName, property, schema, config, id, ignorePrefixInLabel, configScopeName, wrapInGroup) {
        if (typeof wrapInGroup === "undefined") { wrapInGroup = true; }
        var input = null;
        var group = null;

        function copyElementAttributes(element, propertyName) {
            var propertyAttributes = property[propertyName];
            if (propertyAttributes) {
                angular.forEach(propertyAttributes, function (value, key) {
                    if (angular.isString(value)) {
                        element.attr(key, value);
                    }
                });
            }
        }
        function copyAttributes() {
            copyElementAttributes(input, "input-attributes");
            angular.forEach(property, function (value, key) {
                if (angular.isString(value) && key.indexOf("$") < 0 && key !== "type") {
                    var html = Core.escapeHtml(value);
                    input.attr(key, html);
                }
            });
        }

        // lets try to create standard widget markup by default
        // as they work better than the hawtio wrappers when inside forms...
        var options = {
            valueConverter: null
        };
        var safeId = Forms.safeIdentifier(id);

        var inputMarkup = createStandardWidgetMarkup(propTypeName, property, schema, config, options, safeId);

        // Note if for whatever reason we need to go back to the old way of using hawtio directives for standard
        // angularjs directives, just clear inputMarker to null here ;)
        // inputMarkup = null;
        if (inputMarkup) {
            input = $(inputMarkup);

            copyAttributes();

            id = safeId;

            var modelName = config.model || Core.pathGet(property, ["input-attributes", "ng-model"]);
            if (!modelName) {
                modelName = config.getEntity() + "." + id;
            }
            input.attr("ng-model", modelName);

            input.attr('name', id);

            try  {
                if (config.isReadOnly()) {
                    input.attr('readonly', 'true');
                }
            } catch (e) {
                // ignore missing read only function
            }
            var title = property.tooltip || property.label;
            if (title) {
                input.attr('title', title);
            }

            // allow the prefix to be trimmed from the label if enabled
            var defaultLabel = id;
            if (ignorePrefixInLabel || property.ignorePrefixInLabel) {
                var idx = id.lastIndexOf('.');
                if (idx > 0) {
                    defaultLabel = id.substring(idx + 1);
                }
            }

            // figure out which things to not wrap in a group and label etc...
            if (input.attr("type") !== "hidden" && wrapInGroup) {
                group = this.getControlGroup(config, config, id);
                var labelElement = Forms.getLabel(config, config, property.title || property.label || humanizeValue(defaultLabel));
                if (title) {
                    labelElement.attr('title', title);
                }
                group.append(labelElement);
                copyElementAttributes(labelElement, "label-attributes");

                var controlDiv = Forms.getControlDiv(config);
                controlDiv.append(input);
                controlDiv.append(Forms.getHelpSpan(config, config, id));

                group.append(controlDiv);

                // allow control level directives, such as ng-show / ng-hide
                copyElementAttributes(controlDiv, "control-attributes");
                copyElementAttributes(group, "control-group-attributes");

                var scope = config.scope;
                if (scope && modelName) {
                    var onModelChange = function (newValue) {
                        scope.$emit("hawtio.form.modelChange", modelName, newValue);
                    };
                    var fn = onModelChange;

                    // allow custom converters
                    var converterFn = options.valueConverter;
                    if (converterFn) {
                        fn = function () {
                            converterFn(scope, modelName);
                            var newValue = Core.pathGet(scope, modelName);
                            onModelChange(newValue);
                        };
                    }
                    scope.$watch(modelName, fn);
                }
            }
        } else {
            input = $('<div></div>');
            input.attr(Forms.normalize(propTypeName, property, schema), '');

            copyAttributes();

            input.attr('entity', config.getEntity());
            input.attr('mode', config.getMode());

            var fullSchemaName = config.schemaName;
            if (fullSchemaName) {
                input.attr('schema', fullSchemaName);
            }

            if (configScopeName) {
                input.attr('data', configScopeName);
            }

            if (ignorePrefixInLabel || property.ignorePrefixInLabel) {
                input.attr('ignore-prefix-in-label', true);
            }
            input.attr('name', id);
        }

        var label = property.label;
        if (label) {
            input.attr('title', label);
        }

        // TODO check for id in the schema["required"] array too!
        // as required can be specified either via either of these approaches
        /*
        var schema = {
        required: ["foo", "bar"],
        properties: {
        something: {
        required: true,
        type: "string"
        }
        }
        }
        */
        if (property.required) {
            // don't mark checkboxes as required
            if (input[0].localName === "input" && input.attr("type") === "checkbox") {
                // lets not set required on a checkbox, it doesn't make any sense ;)
            } else {
                input.attr('required', 'true');
            }
        }
        return group ? group : input;
    }
    Forms.createWidget = createWidget;

    /**
    * Lets try create the standard angular JS widgets markup
    * @method createStandardWidgetMarkup
    * @param {String} propTypeName
    * @param {any} property
    * @param {any} schema
    * @param {any} config
    * @param {any} options
    * @param {String} id
    */
    function createStandardWidgetMarkup(propTypeName, property, schema, config, options, id) {
        // lets try use standard widgets first...
        var type = Forms.resolveTypeNameAlias(propTypeName, schema);
        if (!type) {
            return '<input type="text"/>';
        }
        var custom = Core.pathGet(property, ["formTemplate"]);
        if (custom) {
            return null;
        }
        var inputElement = Core.pathGet(property, ["input-element"]);
        if (inputElement) {
            return "<" + inputElement + "></" + inputElement + ">";
        }
        var enumValues = Core.pathGet(property, ["enum"]);
        if (enumValues) {
            var required = true;
            var valuesScopeName = null;
            var attributes = "";
            if (enumValues) {
                // calculate from input attributes...
                var scope = config.scope;
                var data = config.data;
                if (data && scope) {
                    // this is a big ugly - would be nice to expose this a bit easier...
                    // maybe nested objects should expose the model easily...
                    var fullSchema = scope[config.schemaName];
                    var model = angular.isString(data) ? scope[data] : data;

                    // now we need to keep walking the model to find the enum values
                    var paths = id.split(".");
                    var property = null;
                    angular.forEach(paths, function (path) {
                        property = Core.pathGet(model, ["properties", path]);
                        var typeName = Core.pathGet(property, ["type"]);
                        var alias = Forms.lookupDefinition(typeName, fullSchema);
                        if (alias) {
                            model = alias;
                        }
                    });
                    var values = Core.pathGet(property, ["enum"]);
                    valuesScopeName = "$values_" + id.replace(/\./g, "_");
                    scope[valuesScopeName] = values;
                }
            }
            if (valuesScopeName) {
                attributes += ' ng-options="value for value in ' + valuesScopeName + '"';
            }
            var defaultOption = required ? "" : '<option value=""></option>';
            return '<select' + attributes + '>' + defaultOption + '</select>';
        }

        if (angular.isArray(type)) {
            // TODO union of tabbed forms such as Marshal / Unmarshal in camel...
            return null;
        }
        if (!angular.isString(type)) {
            return null;
        }
        var defaultValueConverter = null;
        var defaultValue = property.default;
        if (defaultValue) {
            // lets add a default value
            defaultValueConverter = function (scope, modelName) {
                var value = Core.pathGet(scope, modelName);
                if (!value) {
                    Core.pathSet(scope, modelName, property.default);
                }
            };
            options.valueConverter = defaultValueConverter;
        }

        function getModelValueOrDefault(scope, modelName) {
            var value = Core.pathGet(scope, modelName);
            if (!value) {
                var defaultValue = property.default;
                if (defaultValue) {
                    value = defaultValue;
                    Core.pathSet(scope, modelName, value);
                }
            }
            return value;
        }

        switch (type.toLowerCase()) {
            case "int":
            case "integer":
            case "long":
            case "short":
            case "java.lang.integer":
            case "java.lang.long":
            case "float":
            case "double":
            case "java.lang.float":
            case "java.lang.double":
                // lets add a value conversion watcher...
                options.valueConverter = function (scope, modelName) {
                    var value = getModelValueOrDefault(scope, modelName);
                    if (value && angular.isString(value)) {
                        var numberValue = Number(value);
                        Core.pathSet(scope, modelName, numberValue);
                    }
                };
                return '<input type="number"/>';

            case "array":
            case "java.lang.array":
            case "java.lang.iterable":
            case "java.util.list":
            case "java.util.collection":
            case "java.util.iterator":
            case "java.util.set":
            case "object[]":
                // TODO hack for now - objects should not really use the table, thats only really for arrays...
                /*
                case "object":
                case "java.lang.object":
                */
                //return "hawtio-form-array";
                return null;

            case "boolean":
            case "bool":
            case "java.lang.boolean":
                // lets add a value conversion watcher...
                options.valueConverter = function (scope, modelName) {
                    var value = getModelValueOrDefault(scope, modelName);
                    if (value && "true" === value) {
                        //console.log("coercing String to boolean for " + modelName);
                        Core.pathSet(scope, modelName, true);
                    }
                };
                return '<input type="checkbox"/>';

            case "password":
                return '<input type="password"/>';

            case "hidden":
                return '<input type="hidden"/>';
            default:
                // lets check if this name is an alias to a definition in the schema
                return '<input type="text"/>';
        }
    }
    Forms.createStandardWidgetMarkup = createStandardWidgetMarkup;

    function normalize(type, property, schema) {
        type = Forms.resolveTypeNameAlias(type, schema);
        if (!type) {
            return "hawtio-form-text";
        }
        var custom = Core.pathGet(property, ["formTemplate"]);
        if (custom) {
            return "hawtio-form-custom";
        }
        var enumValues = Core.pathGet(property, ["enum"]);
        if (enumValues) {
            // TODO could use different kinds of radio / combo box
            return "hawtio-form-select";
        }

        if (angular.isArray(type)) {
            // TODO union of tabbed forms such as Marshal / Unmarshal in camel...
            return null;
        }
        if (!angular.isString(type)) {
            try  {
                console.log("Unsupported JSON schema type value " + JSON.stringify(type));
            } catch (e) {
                console.log("Unsupported JSON schema type value " + type);
            }
            return null;
        }
        switch (type.toLowerCase()) {
            case "int":
            case "integer":
            case "long":
            case "short":
            case "java.lang.integer":
            case "java.lang.long":
            case "float":
            case "double":
            case "java.lang.float":
            case "java.lang.double":
                return "hawtio-form-number";

            case "array":
            case "java.lang.array":
            case "java.lang.iterable":
            case "java.util.list":
            case "java.util.collection":
            case "java.util.iterator":
            case "java.util.set":
            case "object[]":
                // TODO hack for now - objects should not really use the table, thats only really for arrays...
                /*
                case "object":
                case "java.lang.object":
                */
                var items = property.items;
                if (items) {
                    var typeName = items.type;
                    if (typeName && typeName === "string") {
                        return "hawtio-form-string-array";
                    }
                }
                return "hawtio-form-array";
            case "boolean":
            case "bool":
            case "java.lang.boolean":
                return "hawtio-form-checkbox";
            case "password":
                return "hawtio-form-password";
            case "hidden":
                return "hawtio-form-hidden";
            default:
                // lets check if this name is an alias to a definition in the schema
                return "hawtio-form-text";
        }
    }
    Forms.normalize = normalize;
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    Forms.pluginName = 'hawtio-forms';
    var log = Logger.get("Forms");

    angular.module(Forms.pluginName, ['bootstrap', 'ngResource', 'hawtioCore', 'datatable', 'ui.bootstrap', 'ui.bootstrap.dialog', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/forms/test', { templateUrl: 'app/forms/html/test.html' }).when('/forms/testTable', { templateUrl: 'app/forms/html/testTable.html' });
    }).directive('simpleForm', function (workspace, $compile) {
        return new Forms.SimpleForm(workspace, $compile);
    }).directive('hawtioInputTable', function (workspace, $compile) {
        return new Forms.InputTable(workspace, $compile);
    }).directive('hawtioFormText', function (workspace, $compile) {
        return new Forms.TextInput(workspace, $compile);
    }).directive('hawtioFormPassword', function (workspace, $compile) {
        return new Forms.PasswordInput(workspace, $compile);
    }).directive('hawtioFormHidden', function (workspace, $compile) {
        return new Forms.HiddenText(workspace, $compile);
    }).directive('hawtioFormNumber', function (workspace, $compile) {
        return new Forms.NumberInput(workspace, $compile);
    }).directive('hawtioFormSelect', function (workspace, $compile) {
        return new Forms.SelectInput(workspace, $compile);
    }).directive('hawtioFormArray', function (workspace, $compile) {
        return new Forms.ArrayInput(workspace, $compile);
    }).directive('hawtioFormStringArray', function (workspace, $compile) {
        return new Forms.StringArrayInput(workspace, $compile);
    }).directive('hawtioFormCheckbox', function (workspace, $compile) {
        return new Forms.BooleanInput(workspace, $compile);
    }).directive('hawtioFormCustom', function (workspace, $compile) {
        return new Forms.CustomInput(workspace, $compile);
    }).directive('hawtioSubmit', function () {
        return new Forms.SubmitForm();
    }).directive('hawtioReset', function () {
        return new Forms.ResetForm();
    }).run(function (helpRegistry) {
        helpRegistry.addDevDoc("forms", 'app/forms/doc/developer.md');
    });

    hawtioPluginLoader.addModule(Forms.pluginName);
})(Forms || (Forms = {}));
/**
* @module Forms
*/
var Forms;
(function (Forms) {
    Forms.log = Logger.get("Forms");

    /**
    * Default any values in the schema on the entity if they are not already present
    * @method defaultValues
    * @param {any} entity
    * @param {any} schema
    */
    function defaultValues(entity, schema) {
        if (entity && schema) {
            angular.forEach(schema.properties, function (property, key) {
                var defaultValue = property.default;
                if (defaultValue && !entity[key]) {
                    console.log("===== defaulting value " + defaultValue + " into entity[" + key + "]");
                    entity[key] = defaultValue;
                }
            });
        }
    }
    Forms.defaultValues = defaultValues;

    /**
    * If the type name refers to an alias in the schemas definitions then perform the lookup and return the real type name
    * @method resolveTypeNAmeAlias
    * @param {String} type
    * @param {any} schema
    *
    */
    function resolveTypeNameAlias(type, schema) {
        if (type && schema) {
            var alias = lookupDefinition(type, schema);
            if (alias) {
                var realType = alias["type"];
                if (realType) {
                    type = realType;
                }
            }
        }
        return type;
    }
    Forms.resolveTypeNameAlias = resolveTypeNameAlias;

    /**
    * Walks the base class hierarchy checking if the given type is an instance of the given type name
    * @method isJsonType
    * @param {String} name
    * @param {any} schema
    * @param {String} typeName
    * @return {Boolean}
    */
    function isJsonType(name, schema, typeName) {
        var definition = lookupDefinition(name, schema);
        while (definition) {
            var extendsTypes = Core.pathGet(definition, ["extends", "type"]);
            if (extendsTypes) {
                if (typeName === extendsTypes) {
                    return true;
                } else {
                    definition = lookupDefinition(extendsTypes, schema);
                }
            } else {
                return false;
            }
        }
        return false;
    }
    Forms.isJsonType = isJsonType;

    /**
    * Removes any dodgy characters for a valid identifier in angularjs such as for '-' characters
    * which are replaced with '_'
    * @method safeIdentifier
    * @param {String} id
    * @return {String}
    */
    function safeIdentifier(id) {
        if (id) {
            return id.replace(/-/g, "_");
        }
        return id;
    }
    Forms.safeIdentifier = safeIdentifier;

    /**
    * Looks up the given type name in the schemas definitions
    * @method lookupDefinition
    * @param {String} name
    * @param {any} schema
    */
    function lookupDefinition(name, schema) {
        if (schema) {
            var defs = schema.definitions;
            if (defs) {
                var answer = defs[name];
                if (answer) {
                    var fullSchema = answer["fullSchema"];
                    if (fullSchema) {
                        return fullSchema;
                    }

                    // we may extend another, if so we need to copy in the base properties
                    var extendsTypes = Core.pathGet(answer, ["extends", "type"]);
                    if (extendsTypes) {
                        fullSchema = angular.copy(answer);
                        fullSchema.properties = fullSchema.properties || {};
                        if (!angular.isArray(extendsTypes)) {
                            extendsTypes = [extendsTypes];
                        }
                        angular.forEach(extendsTypes, function (extendType) {
                            if (angular.isString(extendType)) {
                                var extendDef = lookupDefinition(extendType, schema);
                                var properties = Core.pathGet(extendDef, ["properties"]);
                                if (properties) {
                                    angular.forEach(properties, function (property, key) {
                                        fullSchema.properties[key] = property;
                                    });
                                }
                            }
                        });
                        answer["fullSchema"] = fullSchema;
                        return fullSchema;
                    }
                }
                return answer;
            }
        }
        return null;
    }
    Forms.lookupDefinition = lookupDefinition;

    /**
    * For an array property, find the schema of the items which is either nested inside this property
    * in the 'items' property; or the type name is used to lookup in the schemas definitions
    * @method findArrayItemsSchema
    * @param {String} property
    * @param {any} schema
    */
    function findArrayItemsSchema(property, schema) {
        var items = null;
        if (property && schema) {
            items = property.items;
            if (items) {
                var typeName = items["type"];
                if (typeName) {
                    var definition = lookupDefinition(typeName, schema);
                    if (definition) {
                        return definition;
                    }
                }
            }

            // are we a json schema properties with a link to the schema doc?
            var additionalProperties = property.additionalProperties;
            if (additionalProperties) {
                if (additionalProperties["$ref"] === "#") {
                    return schema;
                }
            }
        }
        return items;
    }
    Forms.findArrayItemsSchema = findArrayItemsSchema;

    /**
    * Returns true if the given schema definition is an object
    * @method isObjectType
    * @param {any} definition
    */
    function isObjectType(definition) {
        var typeName = Core.pathGet(definition, "type");
        return typeName && "object" === typeName;
    }
    Forms.isObjectType = isObjectType;

    /**
    * Returns true if the given property represents a nested object or array of objects
    * @method isArrayOrNestedObject
    * @param {any} property
    * @param {any} schema
    */
    function isArrayOrNestedObject(property, schema) {
        if (property) {
            var propType = resolveTypeNameAlias(property["type"], schema);
            if (propType) {
                if (propType === "object" || propType === "array") {
                    return true;
                }
            }
        }
        return false;
    }
    Forms.isArrayOrNestedObject = isArrayOrNestedObject;

    function configure(config, scopeConfig, attrs) {
        if (angular.isDefined(scopeConfig)) {
            config = angular.extend(config, scopeConfig);
        }
        return angular.extend(config, attrs);
    }
    Forms.configure = configure;

    function getControlGroup(config, arg, id) {
        var rc = $('<div class="' + config.controlgroupclass + '"></div>');
        if (angular.isDefined(arg.description)) {
            rc.attr('title', arg.description);
        }

        Forms.log.debug("getControlGroup, config:", config, " arg: ", arg, " id: ", id);
        if (config['properties'] && config['properties'][id]) {
            var elementConfig = config['properties'][id];
            Forms.log.debug("elementConfig: ", elementConfig);
            if (elementConfig && 'control-attributes' in elementConfig) {
                angular.forEach(elementConfig['control-attributes'], function (value, key) {
                    rc.attr(key, value);
                });
            }
        }

        return rc;
    }
    Forms.getControlGroup = getControlGroup;

    function getLabel(config, arg, label) {
        return $('<label class="' + config.labelclass + '">' + label + ': </label>');
    }
    Forms.getLabel = getLabel;

    function getControlDiv(config) {
        return $('<div class="' + config.controlclass + '"></div>');
    }
    Forms.getControlDiv = getControlDiv;

    function getHelpSpan(config, arg, id) {
        var rc = '';
        if (angular.isDefined(arg.type) && config.showtypes !== 'false') {
            $('<span class="help-block">Type: ' + arg.type + '</span>');
        }
        return rc;
    }
    Forms.getHelpSpan = getHelpSpan;
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    var InputTableConfig = (function () {
        function InputTableConfig() {
            this.name = 'form';
            this.method = 'post';
            // the name of the attribute in the scope which is the data to be editted
            this.entity = 'entity';
            // the name of the attribute in the scope which is the table configuration
            this.tableConfig = 'tableConfig';
            // set to 'view' or 'create' for different modes
            this.mode = 'edit';
            // the definition of the form
            this.data = {};
            this.json = undefined;
            this.properties = [];
            this.action = '';
            this.tableclass = 'table table-striped inputTable';
            this.controlgroupclass = 'control-group';
            this.controlclass = 'controls pull-right';
            this.labelclass = 'control-label';
            this.showtypes = 'true';
            this.removeicon = 'icon-remove';
            this.editicon = 'icon-edit';
            this.addicon = 'icon-plus';
            this.removetext = 'Remove';
            this.edittext = 'Edit';
            this.addtext = 'Add';
            this.onadd = 'onadd';
            this.onedit = 'onedit';
            this.onremove = 'onRemove';
        }
        // TODO - add toggles to turn off add or edit buttons
        InputTableConfig.prototype.getTableConfig = function () {
            return this.tableConfig || "tableConfig";
        };
        return InputTableConfig;
    })();
    Forms.InputTableConfig = InputTableConfig;

    var InputTable = (function () {
        function InputTable(workspace, $compile) {
            var _this = this;
            this.workspace = workspace;
            this.$compile = $compile;
            this.restrict = 'A';
            this.scope = true;
            this.replace = true;
            this.transclude = true;
            this.attributeName = 'hawtioInputTable';
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        InputTable.prototype.doLink = function (scope, element, attrs) {
            var _this = this;
            var config = new InputTableConfig;

            var configName = attrs[this.attributeName];
            var tableConfig = Core.pathGet(scope, configName);
            config = Forms.configure(config, tableConfig, attrs);

            var entityName = attrs["entity"] || config.data || "entity";
            var propertyName = attrs["property"] || "arrayData";
            var entityPath = entityName + "." + propertyName;

            // TODO better name?
            var tableName = config["title"] || entityName;

            if (angular.isDefined(config.json)) {
                config.data = $.parseJSON(config.json);
            } else {
                config.data = scope[config.data];
            }

            scope.selectedItems = [];

            var div = $("<div></div>");

            // TODO lets ensure we have some default columns in the column configuration?
            var tableConfig = Core.pathGet(scope, configName);
            if (!tableConfig) {
                console.log("No table configuration for table " + tableName);
            } else {
                tableConfig["selectedItems"] = scope.selectedItems;
            }

            var table = this.createTable(config, configName);

            var group = this.getControlGroup(config, {}, "");
            var controlDiv = this.getControlDiv(config);
            controlDiv.addClass('btn-group');
            group.append(controlDiv);

            function updateData(action) {
                var data = Core.pathGet(scope, entityPath);

                // lets coerce the data to an array if its empty or an object
                if (!data) {
                    data = [];
                }
                if (!angular.isArray(data) && data) {
                    data = [data];
                }
                data = action(data);
                Core.pathSet(scope, entityPath, data);

                // TODO for some reason this doesn't notify the underlying hawtio-datatable that the table has changed
                // so lets force it with a notify...
                scope.$emit("hawtio.datatable." + entityPath, data);
                Core.$apply(scope);
            }

            function removeSelected(data) {
                angular.forEach(scope.selectedItems, function (selected) {
                    var id = selected["_id"];
                    if (angular.isArray(data)) {
                        data = data.remove(function (value) {
                            return Object.equal(value, selected);
                        });
                        delete selected["_id"];
                        data = data.remove(function (value) {
                            return Object.equal(value, selected);
                        });
                    } else {
                        delete selected["_id"];
                        if (id) {
                            delete data[id];
                        } else {
                            // lets iterate for the value
                            var found = false;
                            angular.forEach(data, function (value, key) {
                                if (!found && (Object.equal(value, selected))) {
                                    console.log("Found row to delete! " + key);
                                    delete data[key];
                                    found = true;
                                }
                            });
                            if (!found) {
                                console.log("Could not find " + JSON.stringify(selected) + " in " + JSON.stringify(data));
                            }
                        }
                    }
                });
                return data;
            }

            var add = null;
            var edit = null;
            var remove = null;
            var addDialog = null;
            var editDialog = null;
            var readOnly = attrs["readonly"];
            if (!readOnly) {
                var property = null;
                var dataName = attrs["data"];
                var dataModel = dataName ? Core.pathGet(scope, dataName) : null;
                var schemaName = attrs["schema"] || dataName;
                var schema = schemaName ? Core.pathGet(scope, schemaName) : null;
                if (propertyName && dataModel) {
                    property = Core.pathGet(dataModel, ["properties", propertyName]);
                }

                add = this.getAddButton(config);

                scope.addDialogOptions = {
                    backdropFade: true,
                    dialogFade: true
                };
                scope.showAddDialog = false;

                scope.openAddDialog = function () {
                    // lets lazily create the add dialog
                    scope.addEntity = {};
                    scope.addFormConfig = Forms.findArrayItemsSchema(property, schema);

                    var childDataModelName = "addFormConfig";
                    if (!addDialog) {
                        var title = "Add " + tableName;
                        addDialog = $('<div modal="showAddDialog" close="closeAddDialog()" options="addDialogOptions">\n' + '<div class="modal-header"><h4>' + title + '</h4></div>\n' + '<div class="modal-body"><div simple-form="addFormConfig" entity="addEntity" data="' + childDataModelName + '" schema="' + schemaName + '"></div></div>\n' + '<div class="modal-footer">' + '<button class="btn btn-primary add" type="button" ng-click="addAndCloseDialog()">Add</button>' + '<button class="btn btn-warning cancel" type="button" ng-click="closeAddDialog()">Cancel</button>' + '</div></div>');
                        div.append(addDialog);
                        _this.$compile(addDialog)(scope);
                    }
                    scope.showAddDialog = true;
                    Core.$apply(scope);
                };

                scope.closeAddDialog = function () {
                    scope.showAddDialog = false;
                    scope.addEntity = {};
                };

                scope.addAndCloseDialog = function () {
                    var newData = scope.addEntity;
                    Forms.log.info("About to add the new entity " + JSON.stringify(newData));
                    if (newData) {
                        updateData(function (data) {
                            // TODO deal with non arrays
                            data.push(newData);
                            return data;
                        });
                    }
                    scope.closeAddDialog();
                };

                edit = this.getEditButton(config);

                scope.editDialogOptions = {
                    backdropFade: true,
                    dialogFade: true
                };
                scope.showEditDialog = false;

                scope.openEditDialog = function () {
                    var selected = scope.selectedItems;

                    // lets make a deep copy for the value being edited
                    var editObject = {};
                    if (selected && selected.length) {
                        angular.copy(selected[0], editObject);
                    }
                    scope.editEntity = editObject;
                    scope.editFormConfig = Forms.findArrayItemsSchema(property, schema);

                    // lets lazily create the edit dialog
                    if (!editDialog) {
                        var title = "Edit " + tableName;
                        editDialog = $('<div modal="showEditDialog" close="closeEditDialog()" options="editDialogOptions">\n' + '<div class="modal-header"><h4>' + title + '</h4></div>\n' + '<div class="modal-body"><div simple-form="editFormConfig" entity="editEntity"></div></div>\n' + '<div class="modal-footer">' + '<button class="btn btn-primary save" type="button" ng-click="editAndCloseDialog()">Save</button>' + '<button class="btn btn-warning cancel" type="button" ng-click="closeEditDialog()">Cancel</button>' + '</div></div>');
                        div.append(editDialog);
                        _this.$compile(editDialog)(scope);
                    }
                    scope.showEditDialog = true;
                    Core.$apply(scope);
                };

                scope.closeEditDialog = function () {
                    scope.showEditDialog = false;
                    scope.editEntity = {};
                };

                scope.editAndCloseDialog = function () {
                    var newData = scope.editEntity;
                    console.log("About to edit the new entity " + JSON.stringify(newData));
                    if (newData) {
                        updateData(function (data) {
                            data = removeSelected(data);

                            // TODO deal with non arrays
                            data.push(newData);
                            return data;
                        });
                    }
                    scope.closeEditDialog();
                };

                remove = this.getRemoveButton(config);
            }

            var findFunction = function (scope, func) {
                if (angular.isDefined(scope[func]) && angular.isFunction(scope[func])) {
                    return scope;
                }
                if (angular.isDefined(scope.$parent) && scope.$parent !== null) {
                    return findFunction(scope.$parent, func);
                } else {
                    return null;
                }
            };

            function maybeGet(scope, func) {
                if (scope !== null) {
                    return scope[func];
                }
                return null;
            }

            var onRemoveFunc = config.onremove.replace('(', '').replace(')', '');
            var onEditFunc = config.onedit.replace('(', '').replace(')', '');
            var onAddFunc = config.onadd.replace('(', '').replace(')', '');

            var onRemove = maybeGet(findFunction(scope, onRemoveFunc), onRemoveFunc);
            var onEdit = maybeGet(findFunction(scope, onEditFunc), onEditFunc);
            var onAdd = maybeGet(findFunction(scope, onAddFunc), onAddFunc);

            if (onRemove === null) {
                onRemove = function () {
                    updateData(function (data) {
                        return removeSelected(data);
                    });
                };
            }
            if (onEdit === null) {
                onEdit = function () {
                    scope.openEditDialog();
                };
            }
            if (onAdd === null) {
                onAdd = function (form) {
                    scope.openAddDialog();
                };
            }
            if (add) {
                add.click(function (event) {
                    onAdd();
                    return false;
                });
                controlDiv.append(add);
            }
            if (edit) {
                edit.click(function (event) {
                    onEdit();
                    return false;
                });
                controlDiv.append(edit);
            }
            if (remove) {
                remove.click(function (event) {
                    onRemove();
                    return false;
                });
                controlDiv.append(remove);
            }

            $(div).append(group);
            $(div).append(table);
            $(element).append(div);

            // compile the template
            this.$compile(div)(scope);
        };

        InputTable.prototype.getAddButton = function (config) {
            return $('<button type="button" class="btn add"><i class="' + config.addicon + '"></i> ' + config.addtext + '</button>');
        };

        InputTable.prototype.getEditButton = function (config) {
            return $('<button type="button" class="btn edit" ng-disabled="!selectedItems.length"><i class="' + config.editicon + '"></i> ' + config.edittext + '</button>');
        };

        InputTable.prototype.getRemoveButton = function (config) {
            return $('<button type="remove" class="btn remove" ng-disabled="!selectedItems.length"><i class="' + config.removeicon + '"></i> ' + config.removetext + '</button>');
        };

        InputTable.prototype.createTable = function (config, tableConfig) {
            //var tableType = "hawtio-datatable";
            var tableType = "hawtio-simple-table";
            var table = $('<div class="' + config.tableclass + '" ' + tableType + '="' + tableConfig + '">');

            //table.find('fieldset').append(this.getLegend(config));
            return table;
        };

        InputTable.prototype.getLegend = function (config) {
            var description = Core.pathGet(config, "data.description");
            if (description) {
                return '<legend>' + config.data.description + '</legend>';
            }
            return '';
        };

        InputTable.prototype.getControlGroup = function (config, arg, id) {
            var rc = $('<div class="' + config.controlgroupclass + '"></div>');
            if (angular.isDefined(arg.description)) {
                rc.attr('title', arg.description);
            }
            return rc;
        };

        InputTable.prototype.getControlDiv = function (config) {
            return $('<div class="' + config.controlclass + '"></div>');
        };

        InputTable.prototype.getHelpSpan = function (config, arg, id) {
            var rc = $('<span class="help-block"></span>');
            if (angular.isDefined(arg.type) && config.showtypes !== 'false') {
                rc.append('Type: ' + arg.type);
            }
            return rc;
        };
        return InputTable;
    })();
    Forms.InputTable = InputTable;
})(Forms || (Forms = {}));
/**
* @module Forms
*/
var Forms;
(function (Forms) {
    /**
    * @class InputBaseConfig
    */
    var InputBaseConfig = (function () {
        function InputBaseConfig() {
            this.name = 'input';
            this.type = '';
            this.description = '';
            this._default = '';
            this.scope = null;
            // Can also be 'view'
            this.mode = 'edit';
            // the name of the full schema
            this.schemaName = "schema";
            this.controlgroupclass = 'control-group';
            this.controlclass = 'controls';
            this.labelclass = 'control-label';
            this.showtypes = 'false';
            /**
            * Custom template for custom form controls
            * @property
            * @type String
            */
            this.formtemplate = null;
            /**
            * the name of the attribute in the scope which is the data to be edited
            * @property
            * @type String
            */
            this.entity = 'entity';
            /**
            * the model expression to bind to. If omitted this defaults to entity + "." + name
            * @property
            * @type String
            */
            this.model = undefined;
        }
        InputBaseConfig.prototype.getEntity = function () {
            return this.entity || "entity";
        };

        InputBaseConfig.prototype.getMode = function () {
            return this.mode || "edit";
        };

        InputBaseConfig.prototype.isReadOnly = function () {
            return this.getMode() === "view";
        };
        return InputBaseConfig;
    })();
    Forms.InputBaseConfig = InputBaseConfig;

    var InputBase = (function () {
        function InputBase(workspace, $compile) {
            var _this = this;
            this.workspace = workspace;
            this.$compile = $compile;
            this.restrict = 'A';
            this.scope = true;
            this.replace = false;
            this.transclude = false;
            this.attributeName = '';
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        InputBase.prototype.doLink = function (scope, element, attrs) {
            var config = new InputBaseConfig;
            config = Forms.configure(config, null, attrs);
            config.scope = scope;
            config.schemaName = attrs["schema"] || "schema";

            var id = Forms.safeIdentifier(config.name);
            var group = this.getControlGroup(config, config, id);

            var modelName = config.model;
            if (!angular.isDefined(modelName)) {
                // TODO always use 2 way binding?
                modelName = config.getEntity() + "." + id;
            }

            // allow the prefix to be trimmed from the label
            var defaultLabel = id;
            if ("true" === attrs["ignorePrefixInLabel"]) {
                var idx = id.lastIndexOf('.');
                if (idx > 0) {
                    defaultLabel = id.substring(idx + 1);
                }
            }
            group.append(Forms.getLabel(config, config, attrs["title"] || humanizeValue(defaultLabel)));
            var controlDiv = Forms.getControlDiv(config);
            controlDiv.append(this.getInput(config, config, id, modelName));
            controlDiv.append(Forms.getHelpSpan(config, config, id));
            group.append(controlDiv);
            $(element).append(this.$compile(group)(scope));

            if (scope && modelName) {
                scope.$watch(modelName, onModelChange);
            }
            function onModelChange(newValue) {
                scope.$emit("hawtio.form.modelChange", modelName, newValue);
            }
        };

        InputBase.prototype.getControlGroup = function (config1, config2, id) {
            return Forms.getControlGroup(config1, config2, id);
        };

        InputBase.prototype.getInput = function (config, arg, id, modelName) {
            var rc = $('<span class="form-data"></span>');
            if (modelName) {
                rc.attr('ng-model', modelName);
                rc.append('{{' + modelName + '}}');
            }
            return rc;
        };
        return InputBase;
    })();
    Forms.InputBase = InputBase;

    var TextInput = (function (_super) {
        __extends(TextInput, _super);
        function TextInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
            this.type = "text";
        }
        /*public getControlGroup(config1, config2, id) {
        return super.getControlGroup(config1, config2, id);
        }*/
        TextInput.prototype.getInput = function (config, arg, id, modelName) {
            if (config.isReadOnly()) {
                return _super.prototype.getInput.call(this, config, arg, id, modelName);
            }
            var rc = $('<input type="' + this.type + '">');
            rc.attr('name', id);
            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }
            var required = config.$attr["required"];
            if (required && required !== "false") {
                rc.attr('required', 'true');
            }
            return rc;
        };
        return TextInput;
    })(InputBase);
    Forms.TextInput = TextInput;

    var HiddenText = (function (_super) {
        __extends(HiddenText, _super);
        function HiddenText(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
            this.type = "hidden";
        }
        HiddenText.prototype.getControlGroup = function (config1, config2, id) {
            var group = _super.prototype.getControlGroup.call(this, config1, config2, id);
            group.css({ 'display': 'none' });
            return group;
        };

        HiddenText.prototype.getInput = function (config, arg, id, modelName) {
            var rc = _super.prototype.getInput.call(this, config, arg, id, modelName);
            rc.attr('readonly', 'true');
            return rc;
        };
        return HiddenText;
    })(TextInput);
    Forms.HiddenText = HiddenText;

    var PasswordInput = (function (_super) {
        __extends(PasswordInput, _super);
        function PasswordInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
            this.type = "password";
        }
        return PasswordInput;
    })(TextInput);
    Forms.PasswordInput = PasswordInput;

    var CustomInput = (function (_super) {
        __extends(CustomInput, _super);
        function CustomInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        CustomInput.prototype.getInput = function (config, arg, id, modelName) {
            var template = arg.formtemplate;
            template = Core.unescapeHtml(template);
            var rc = $(template);
            if (!rc.attr("name")) {
                rc.attr('name', id);
            }
            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }
            return rc;
        };
        return CustomInput;
    })(InputBase);
    Forms.CustomInput = CustomInput;

    var SelectInput = (function (_super) {
        __extends(SelectInput, _super);
        function SelectInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        SelectInput.prototype.getInput = function (config, arg, id, modelName) {
            if (config.isReadOnly()) {
                return _super.prototype.getInput.call(this, config, arg, id, modelName);
            }

            // TODO calculate from input attributes...
            var required = true;

            // TODO we could configure the null option...
            var defaultOption = required ? "" : '<option value=""></option>';
            var rc = $('<select>' + defaultOption + '</select>');
            rc.attr('name', id);

            var scope = config.scope;
            var data = config.data;
            if (data && scope) {
                // this is a big ugly - would be nice to expose this a bit easier...
                // maybe nested objects should expose the model easily...
                var fullSchema = scope[config.schemaName];
                var model = scope[data];

                // now we need to keep walking the model to find the enum values
                var paths = id.split(".");
                var property = null;
                angular.forEach(paths, function (path) {
                    property = Core.pathGet(model, ["properties", path]);
                    var typeName = Core.pathGet(property, ["type"]);
                    var alias = Forms.lookupDefinition(typeName, fullSchema);
                    if (alias) {
                        model = alias;
                    }
                });
                var values = Core.pathGet(property, ["enum"]);
                scope["$selectValues"] = values;
                rc.attr("ng-options", "value for value in $selectValues");
            }
            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }
            return rc;
        };
        return SelectInput;
    })(InputBase);
    Forms.SelectInput = SelectInput;

    var NumberInput = (function (_super) {
        __extends(NumberInput, _super);
        function NumberInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        NumberInput.prototype.getInput = function (config, arg, id, modelName) {
            if (config.isReadOnly()) {
                return _super.prototype.getInput.call(this, config, arg, id, modelName);
            }
            var rc = $('<input type="number">');
            rc.attr('name', id);

            if (angular.isDefined(arg.def)) {
                rc.attr('value', arg.def);
            }

            if (angular.isDefined(arg.minimum)) {
                rc.attr('min', arg.minimum);
            }

            if (angular.isDefined(arg.maximum)) {
                rc.attr('max', arg.maximum);
            }

            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }

            // lets coerce any string values to numbers so that they work properly with the UI
            var scope = config.scope;
            if (scope) {
                function onModelChange() {
                    var value = Core.pathGet(scope, modelName);
                    if (value && angular.isString(value)) {
                        var numberValue = Number(value);
                        Core.pathSet(scope, modelName, numberValue);
                    }
                }
                scope.$watch(modelName, onModelChange);
                onModelChange();
            }
            return rc;
        };
        return NumberInput;
    })(InputBase);
    Forms.NumberInput = NumberInput;

    /**
    * Generates a list of strings which can be added / edited / removed
    * @class StringArrayInput
    */
    var StringArrayInput = (function (_super) {
        __extends(StringArrayInput, _super);
        function StringArrayInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        StringArrayInput.prototype.getInput = function (config, arg, id, modelName) {
            var rowScopeName = "_" + id;
            var ngRepeat = rowScopeName + ' in ' + modelName;

            var readOnlyWidget = '{{' + rowScopeName + '}}';
            if (config.isReadOnly()) {
                return $('<ul><li ng-repeat="' + rowScopeName + ' in ' + modelName + '">' + readOnlyWidget + '</li></ul>');
            } else {
                // TODO there should be an easier way to find the property / schema!
                var scope = config.scope;
                var fallbackSchemaName = (arg.$attr || {}