/* Flot plugin for stacking data sets rather than overlyaing them.

Copyright (c) 2007-2012 IOLA and Ole Laursen.
Licensed under the MIT license.

The plugin assumes the data is sorted on x (or y if stacking horizontally).
For line charts, it is assumed that if a line has an undefined gap (from a
null point), then the line above it should have the same gap - insert zeros
instead of "null" if you want another behaviour. This also holds for the start
and end of the chart. Note that stacking a mix of positive and negative values
in most instances doesn't make sense (so it looks weird).

Two or more series are stacked when their "stack" attribute is set to the same
key (which can be any number or string or just "true"). To specify the default
stack, you can set the stack option like this:

	series: {
		stack: null or true or key (number/string)
	}

You can also specify it for a single series, like this:

	$.plot( $("#placeholder"), [{
		data: [ ... ],
		stack: true
	}])

The stacking order is determined by the order of the data series in the array
(later series end up on top of the previous).

Internally, the plugin modifies the datapoints in each series, adding an
offset to the y value. For line series, extra data points are inserted through
interpolation. If there's a second y value, it's also adjusted (e.g for bar
charts or filled areas).

*/

(function ($) {
    var options = {
        series: { stack: null } // or number/string
    };
    
    function init(plot) {
        function findMatchingSeries(s, allseries) {
            var res = null;
            for (var i = 0; i < allseries.length; ++i) {
                if (s == allseries[i])
                    break;
                
                if (allseries[i].stack == s.stack)
                    res = allseries[i];
            }
            
            return res;
        }
        
		var accumulated = { };

        function stackData(plot, s, datapoints) {
            if (s.stack == null)
                return;

			if (accumulated[s.stack] == null) {
				accumulated[s.stack] = { }
				_.each(plot.getData(), function(sr) {
					if (s.stack == sr.stack) {
						for (var i = 0; i < sr.datapoints.points.length; i += sr.datapoints.pointsize) {
							if (sr.datapoints.points[i] != null) {
								accumulated[sr.stack][sr.datapoints.points[i + (sr.bars.horizontal ? 1 : 0)]] = 0;
							}
						}
					}
				});
			}
			var ps = datapoints.pointsize,
                points = datapoints.points,
                keyOffset = horizontal ? 1 : 0,
                accumulateOffset = horizontal ? 0 : 1,
                horizontal = s.bars.horizontal,
				withlines = s.lines.show,
                withsteps = withlines && s.lines.steps;
			if (true | withlines) {
				var newpoints = [];
				var newacc = { }
				for (var key in accumulated[s.stack]) {
					newacc[key] = accumulated[s.stack][key];
				}
				for (var i = 0; i < points.length; i += ps) {
					if (points[i] != null) {
						newacc[points[i + keyOffset]] += points[i + accumulateOffset];
					}
				}
				for (var key in accumulated[s.stack]) {
					if (withlines || newacc[key] != accumulated[s.stack][key]) {
						var l = newpoints.length;
						for (var m = 0; m < ps; m++) {
							newpoints.push(0);
						}
						newpoints[l + keyOffset] += parseInt(key);
						newpoints[l + accumulateOffset] += newacc[key];
						newpoints[l + 2] += accumulated[s.stack][key];
					} else {
						for (var m = 0; m < ps; m++) {
							newpoints.push(null);
						}
					}
				}
				accumulated[s.stack] = newacc;
	            datapoints.points = newpoints;
			} else {
				for (var i = 0; i < points.length; i += ps) {
					if (points[i] != null) {
						points[i + accumulateOffset] += accumulated[s.stack][points[i + keyOffset]];
						points[i + 2] += accumulated[s.stack][points[i + keyOffset]];
						accumulated[s.stack][points[i + keyOffset]] = points[i + accumulateOffset];
					}
				}
			}
			return;

            var other = findMatchingSeries(s, plot.getData());
            if (!other)
                return;

            var ps = datapoints.pointsize,
                points = datapoints.points,
                otherps = other.datapoints.pointsize,
                otherpoints = other.datapoints.points,
                newpoints = [],
                px, py, intery, qx, qy, bottom,
                withlines = s.lines.show,
                horizontal = s.bars.horizontal,
                withbottom = ps > 2 && (horizontal ? datapoints.format[2].x : datapoints.format[2].y),
                withsteps = withlines && s.lines.steps,
                fromgap = true,
                keyOffset = horizontal ? 1 : 0,
                accumulateOffset = horizontal ? 0 : 1,
                i = 0, j = 0, l, m;

            while (true) {
                if (i >= points.length)
                    break;

                l = newpoints.length;

                if (points[i] == null) {
                    // copy gaps
                    for (m = 0; m < ps; ++m)
                        newpoints.push(points[i + m]);
                    i += ps;
                }
                else if (j >= otherpoints.length) {
                    // for lines, we can't use the rest of the points
                    if (!withlines) {
                        for (m = 0; m < ps; ++m)
                            newpoints.push(points[i + m]);
                    }
                    i += ps;
                }
                else if (otherpoints[j] == null) {
                    // oops, got a gap
                    for (m = 0; m < ps; ++m)
                        newpoints.push(null);
                    fromgap = true;
                    j += otherps;
                }
                else {
                    // cases where we actually got two points
                    px = points[i + keyOffset];
                    py = points[i + accumulateOffset];
                    qx = otherpoints[j + keyOffset];
                    qy = otherpoints[j + accumulateOffset];
                    bottom = 0;

                    if (px == qx) {
                        for (m = 0; m < ps; ++m)
                            newpoints.push(points[i + m]);

                        newpoints[l + accumulateOffset] += qy;
                        bottom = qy;
                        
                        i += ps;
                        j += otherps;
                    }
                    else if (px > qx) {
                        // we got past point below, might need to
                        // insert interpolated extra point
                        if (withlines && i > 0 && points[i - ps] != null) {
                            intery = py + (points[i - ps + accumulateOffset] - py) * (qx - px) / (points[i - ps + keyOffset] - px);
                            newpoints.push(qx);
                            newpoints.push(intery + qy);
                            for (m = 2; m < ps; ++m)
                                newpoints.push(points[i + m]);
                            bottom = qy; 
                        }

                        j += otherps;
                    }
                    else { // px < qx
                        if (fromgap && withlines) {
                            // if we come from a gap, we just skip this point
                            i += ps;
                            continue;
                        }
                            
                        for (m = 0; m < ps; ++m)
                            newpoints.push(points[i + m]);
                        
                        // we might be able to interpolate a point below,
                        // this can give us a better y
                        if (withlines && j > 0 && otherpoints[j - otherps] != null)
                            bottom = qy + (otherpoints[j - otherps + accumulateOffset] - qy) * (px - qx) / (otherpoints[j - otherps + keyOffset] - qx);

                        newpoints[l + accumulateOffset] += bottom;
                        
                        i += ps;
                    }

                    fromgap = false;
                    
                    if (l != newpoints.length && withbottom)
                        newpoints[l + 2] += bottom;
                }

                // maintain the line steps invariant
                if (withsteps && l != newpoints.length && l > 0
                    && newpoints[l] != null
                    && newpoints[l] != newpoints[l - ps]
                    && newpoints[l + 1] != newpoints[l - ps + 1]) {
                    for (m = 0; m < ps; ++m)
                        newpoints[l + ps + m] = newpoints[l + m];
                    newpoints[l + 1] = newpoints[l - ps + 1];
                }
            }

            datapoints.points = newpoints;
        }
        
        plot.hooks.processDatapoints.push(stackData);
    }
    
    $.plot.plugins.push({
        init: init,
        options: options,
        name: 'stack',
        version: '1.2'
    });
})(jQuery);

(function ($) {
    var options = {
        series: {
            stackpercent: null
        } // or number/string
    };

    function init(plot) {

        // will be built up dynamically as a hash from x-value, or y-value if horizontal
        var stackBases = {};
        var processed = false;
        var stackSums = {};

        //set percentage for stacked chart
        function processRawData(plot, series, data, datapoints) {
            if (!processed) {
                processed = true;
                stackSums = getStackSums(plot.getData());
            }
            if (series.stackpercent == true) {
                var num = data.length;
                series.percents = [];
                var key_idx = 0;
                var value_idx = 1;
                if (series.bars && series.bars.horizontal && series.bars.horizontal === true) {
                    key_idx = 1;
                    value_idx = 0;
                }
                for (var j = 0; j < num; j++) {
                    var sum = stackSums[data[j][key_idx] + ""];
                    if (sum > 0) {
                        series.percents.push(data[j][value_idx] * 100 / sum);
                    } else {
                        series.percents.push(0);
                    }
                }
            }
        }

        //calculate summary
        function getStackSums(_data) {
            var data_len = _data.length;
            var sums = {};
            if (data_len > 0) {
                //caculate summary
                for (var i = 0; i < data_len; i++) {
                    if (_data[i].stackpercent) {
                        var key_idx = 0;
                        var value_idx = 1;
                        if (_data[i].bars && _data[i].bars.horizontal && _data[i].bars.horizontal === true) {
                            key_idx = 1;
                            value_idx = 0;
                        }
                        var num = _data[i].data.length;
                        for (var j = 0; j < num; j++) {
                            var value = 0;
                            if (_data[i].data[j][1] != null) {
                                value = _data[i].data[j][value_idx];
                            }
                            if (sums[_data[i].data[j][key_idx] + ""]) {
                                sums[_data[i].data[j][key_idx] + ""] += value;
                            } else {
                                sums[_data[i].data[j][key_idx] + ""] = value;
                            }

                        }
                    }
                }
            }
            return sums;
        }

        function stackData(plot, s, datapoints) {
            if (!s.stackpercent) return;
            if (!processed) {
                stackSums = getStackSums(plot.getData());
            }
            var newPoints = [];
            
            
            var key_idx = 0;
            var value_idx = 1;
            if (s.bars && s.bars.horizontal && s.bars.horizontal === true) {
                key_idx = 1;
                value_idx = 0;
            }

            for (var i = 0; i < datapoints.points.length; i += 3) {
                // note that the values need to be turned into absolute y-values.
                // in other words, if you were to stack (x, y1), (x, y2), and (x, y3),
                // (each from different series, which is where stackBases comes in),
                // you'd want the new points to be (x, y1, 0), (x, y1+y2, y1), (x, y1+y2+y3, y1+y2)
                // generally, (x, thisValue + (base up to this point), + (base up to this point))
                if (!stackBases[datapoints.points[i + key_idx]]) {
                    stackBases[datapoints.points[i + key_idx]] = 0;
                }
                newPoints[i + key_idx] = datapoints.points[i + key_idx];
                newPoints[i + value_idx] = datapoints.points[i + value_idx] + stackBases[datapoints.points[i + key_idx]];
                newPoints[i + 2] = stackBases[datapoints.points[i + key_idx]];
                stackBases[datapoints.points[i + key_idx]] += datapoints.points[i + value_idx];
                // change points to percentage values
                // you may need to set yaxis:{ max = 100 }
                if ( stackSums[newPoints[i+key_idx]+""] > 0 ){
                    newPoints[i + value_idx] = newPoints[i + value_idx] * 100 / stackSums[newPoints[i + key_idx] + ""];
                    newPoints[i + 2] = newPoints[i + 2] * 100 / stackSums[newPoints[i + key_idx] + ""];
                } else {
                    newPoints[i + value_idx] = 0;
                    newPoints[i + 2] = 0;
                }
            }

            datapoints.points = newPoints;
        }

        plot.hooks.processRawData.push(processRawData);
        plot.hooks.processDatapoints.push(stackData);
    }

    $.plot.plugins.push({
        init: init,
        options: options,
        name: 'stackpercent',
        version: '0.1'
    });
})(jQuery);
