/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.store;

import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.SegmentCommitInfo;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.SimpleFSDirectory;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.Version;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.ImmutableCollection;
import org.elasticsearch.common.collect.ImmutableList;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.collect.Iterables;
import org.elasticsearch.common.compress.Compressor;
import org.elasticsearch.common.compress.CompressorFactory;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.lucene.Directories;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.CloseableIndexComponent;
import org.elasticsearch.index.codec.CodecService;
import org.elasticsearch.index.settings.IndexSettings;
import org.elasticsearch.index.shard.AbstractIndexShardComponent;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.DirectoryService;
import org.elasticsearch.index.store.DistributorDirectory;
import org.elasticsearch.index.store.IndexStore;
import org.elasticsearch.index.store.StoreFileMetaData;
import org.elasticsearch.index.store.StoreStats;
import org.elasticsearch.index.store.distributor.Distributor;
import org.elasticsearch.index.store.support.ForceSyncDirectory;

public class Store
extends AbstractIndexShardComponent
implements CloseableIndexComponent,
Closeable {
    private static final String CODEC = "store";
    private static final int VERSION = 0;
    private static final String CORRUPTED = "corrupted_";
    private final AtomicBoolean isClosed = new AtomicBoolean(false);
    private final AtomicInteger refCount = new AtomicInteger(1);
    private final IndexStore indexStore;
    private final CodecService codecService;
    private final DirectoryService directoryService;
    private final StoreDirectory directory;
    private final boolean sync;
    private final DistributorDirectory distributorDirectory;
    private static final String CHECKSUMS_PREFIX = "_checksums-";

    @Inject
    public Store(ShardId shardId, @IndexSettings Settings indexSettings, IndexStore indexStore, CodecService codecService, DirectoryService directoryService, Distributor distributor) throws IOException {
        super(shardId, indexSettings);
        this.indexStore = indexStore;
        this.codecService = codecService;
        this.directoryService = directoryService;
        this.sync = this.componentSettings.getAsBoolean("sync", (Boolean)true);
        this.distributorDirectory = new DistributorDirectory(distributor);
        this.directory = new StoreDirectory(this.distributorDirectory);
    }

    public IndexStore indexStore() {
        this.ensureOpen();
        return this.indexStore;
    }

    public Directory directory() {
        this.ensureOpen();
        return this.directory;
    }

    public SegmentInfos readLastCommittedSegmentsInfo() throws IOException {
        return Store.readLastCommittedSegmentsInfo(this.directory());
    }

    private static SegmentInfos readLastCommittedSegmentsInfo(Directory directory) throws IOException {
        try {
            return Lucene.readSegmentInfos(directory);
        }
        catch (EOFException eof) {
            throw new CorruptIndexException("Read past EOF while reading segment infos", eof);
        }
    }

    private final void ensureOpen() {
        if (this.refCount.get() <= 0) {
            throw new AlreadyClosedException("Store is already closed");
        }
    }

    public MetadataSnapshot getMetadata() throws IOException {
        this.ensureOpen();
        this.failIfCorrupted();
        try {
            return new MetadataSnapshot(this.distributorDirectory, this.logger);
        }
        catch (CorruptIndexException ex) {
            this.markStoreCorrupted(ex);
            throw ex;
        }
    }

    public void deleteContent() throws IOException {
        this.ensureOpen();
        String[] files = this.distributorDirectory.listAll();
        IOException lastException = null;
        for (String file : files) {
            try {
                this.distributorDirectory.deleteFile(file);
            }
            catch (FileNotFoundException | NoSuchFileException e) {
            }
            catch (IOException e) {
                lastException = e;
            }
        }
        if (lastException != null) {
            throw lastException;
        }
    }

    public StoreStats stats() throws IOException {
        this.ensureOpen();
        return new StoreStats(Directories.estimateSize(this.directory), this.directoryService.throttleTimeInNanos());
    }

    public void renameFile(String from, String to) throws IOException {
        this.ensureOpen();
        this.distributorDirectory.renameFile(this.directoryService, from, to);
    }

    public boolean suggestUseCompoundFile() {
        return false;
    }

    public final void incRef() {
        if (!this.tryIncRef()) {
            throw new AlreadyClosedException("Store is already closed can't increment refCount current count [" + this.refCount.get() + "]");
        }
    }

    public final boolean tryIncRef() {
        int i;
        while ((i = this.refCount.get()) > 0) {
            if (!this.refCount.compareAndSet(i, i + 1)) continue;
            return true;
        }
        return false;
    }

    public final void decRef() {
        int i = this.refCount.decrementAndGet();
        assert (i >= 0);
        if (i == 0) {
            this.closeInternal();
        }
    }

    @Override
    public void close() {
        if (this.isClosed.compareAndSet(false, true)) {
            this.decRef();
        }
    }

    private void closeInternal() {
        try {
            this.directory.innerClose();
        }
        catch (IOException e) {
            this.logger.debug("failed to close directory", e, new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static MetadataSnapshot readMetadataSnapshot(File[] indexLocations, ESLogger logger) throws IOException {
        Closeable[] dirs = new Directory[indexLocations.length];
        try {
            for (int i = 0; i < indexLocations.length; ++i) {
                dirs[i] = new SimpleFSDirectory(indexLocations[i]);
            }
            DistributorDirectory dir = new DistributorDirectory((Directory[])dirs);
            Store.failIfCorrupted(dir, new ShardId("", 1));
            MetadataSnapshot metadataSnapshot = new MetadataSnapshot(dir, logger);
            return metadataSnapshot;
        }
        finally {
            IOUtils.close(dirs);
        }
    }

    public IndexOutput createVerifyingOutput(String filename, IOContext context, StoreFileMetaData metadata) throws IOException {
        if (metadata.hasLegacyChecksum() || metadata.checksum() == null) {
            this.logger.debug("create legacy output for {}", filename);
            return this.directory().createOutput(filename, context);
        }
        assert (metadata.writtenBy() != null);
        assert (metadata.writtenBy().onOrAfter(Version.LUCENE_48));
        return new VerifyingIndexOutput(metadata, this.directory().createOutput(filename, context));
    }

    public static void verify(IndexOutput output) throws IOException {
        if (output instanceof VerifyingIndexOutput) {
            ((VerifyingIndexOutput)output).verify();
        }
    }

    public boolean checkIntegrity(StoreFileMetaData md) {
        if (md.writtenBy() != null && md.writtenBy().onOrAfter(Version.LUCENE_48)) {
            try (IndexInput input = this.directory().openInput(md.name(), IOContext.READONCE);){
                CodecUtil.checksumEntireFile(input);
            }
            catch (IOException e) {
                return false;
            }
        }
        return true;
    }

    public boolean isMarkedCorrupted() throws IOException {
        String[] files;
        this.ensureOpen();
        for (String file : files = this.directory().listAll()) {
            if (!file.startsWith(CORRUPTED)) continue;
            return true;
        }
        return false;
    }

    public void failIfCorrupted() throws IOException {
        this.ensureOpen();
        Store.failIfCorrupted(this.directory, this.shardId);
    }

    private static final void failIfCorrupted(Directory directory, ShardId shardId) throws IOException {
        String[] files = directory.listAll();
        ArrayList<CorruptIndexException> ex = new ArrayList<CorruptIndexException>();
        for (String file : files) {
            if (!file.startsWith(CORRUPTED)) continue;
            try (ChecksumIndexInput input = directory.openChecksumInput(file, IOContext.READONCE);){
                CodecUtil.checkHeader(input, CODEC, 0, 0);
                String msg = input.readString();
                StringBuilder builder = new StringBuilder(shardId.toString());
                builder.append(" Corrupted index [");
                builder.append(file).append("] caused by: ");
                builder.append(msg);
                ex.add(new CorruptIndexException(builder.toString()));
                CodecUtil.checkFooter(input);
            }
        }
        if (!ex.isEmpty()) {
            ExceptionsHelper.rethrowAndSuppress(ex);
        }
    }

    public static final boolean isChecksum(String name) {
        return name.startsWith(CHECKSUMS_PREFIX) || name.endsWith(".cks");
    }

    public static String digestToString(long digest) {
        return Long.toString(digest, 36);
    }

    public void deleteQuiet(String ... files) {
        for (String file : files) {
            try {
                this.directory().deleteFile(file);
            }
            catch (Throwable ex) {
                // empty catch block
            }
        }
    }

    public void markStoreCorrupted(CorruptIndexException exception) throws IOException {
        this.ensureOpen();
        if (!this.isMarkedCorrupted()) {
            String uuid = CORRUPTED + Strings.randomBase64UUID();
            try (IndexOutput output = this.directory().createOutput(uuid, IOContext.DEFAULT);){
                CodecUtil.writeHeader(output, CODEC, 0);
                output.writeString(ExceptionsHelper.detailedMessage(exception, true, 0));
                CodecUtil.writeFooter(output);
            }
            catch (IOException ex) {
                this.logger.warn("Can't mark store as corrupted", ex, new Object[0]);
            }
            this.directory().sync(Collections.singleton(uuid));
        }
    }

    static class VerifyingIndexOutput
    extends IndexOutput {
        private final StoreFileMetaData metadata;
        private final IndexOutput output;
        private long writtenBytes;
        private final long checksumPosition;
        private String actualChecksum;

        VerifyingIndexOutput(StoreFileMetaData metadata, IndexOutput actualOutput) {
            this.metadata = metadata;
            this.output = actualOutput;
            this.checksumPosition = metadata.length() - 8L;
        }

        @Override
        public void flush() throws IOException {
            this.output.flush();
        }

        @Override
        public void close() throws IOException {
            this.output.close();
        }

        @Override
        public long getFilePointer() {
            return this.output.getFilePointer();
        }

        @Override
        public long getChecksum() throws IOException {
            return this.output.getChecksum();
        }

        @Override
        public long length() throws IOException {
            return this.output.length();
        }

        public void verify() throws IOException {
            if (this.metadata.checksum().equals(this.actualChecksum) && this.writtenBytes == this.metadata.length()) {
                return;
            }
            throw new CorruptIndexException("verification failed (hardware problem?) : expected=" + this.metadata.checksum() + " actual=" + this.actualChecksum + " writtenLength=" + this.writtenBytes + " expectedLength=" + this.metadata.length() + " (resource=" + this.metadata.toString() + ")");
        }

        @Override
        public void writeByte(byte b) throws IOException {
            if (this.writtenBytes++ == this.checksumPosition) {
                this.readAndCompareChecksum();
            }
            this.output.writeByte(b);
        }

        private void readAndCompareChecksum() throws IOException {
            this.actualChecksum = Store.digestToString(this.getChecksum());
            if (!this.metadata.checksum().equals(this.actualChecksum)) {
                throw new CorruptIndexException("checksum failed (hardware problem?) : expected=" + this.metadata.checksum() + " actual=" + this.actualChecksum + " (resource=" + this.metadata.toString() + ")");
            }
        }

        @Override
        public void writeBytes(byte[] b, int offset, int length) throws IOException {
            if (this.writtenBytes + (long)length > this.checksumPosition && this.actualChecksum == null) {
                assert (this.writtenBytes <= this.checksumPosition);
                int bytesToWrite = (int)(this.checksumPosition - this.writtenBytes);
                this.output.writeBytes(b, offset, bytesToWrite);
                this.readAndCompareChecksum();
                offset += bytesToWrite;
                length -= bytesToWrite;
                this.writtenBytes += (long)bytesToWrite;
            }
            this.output.writeBytes(b, offset, length);
            this.writtenBytes += (long)length;
        }
    }

    public static final class LegacyChecksums {
        private final Map<String, String> legacyChecksums = new HashMap<String, String>();

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void add(StoreFileMetaData metaData) throws IOException {
            if (metaData.hasLegacyChecksum()) {
                LegacyChecksums legacyChecksums = this;
                synchronized (legacyChecksums) {
                    this.legacyChecksums.put(metaData.name(), metaData.checksum());
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void write(Store store) throws IOException {
            DistributorDirectory distributorDirectory = store.distributorDirectory;
            synchronized (distributorDirectory) {
                Map<String, String> stringStringMap = MetadataSnapshot.readLegacyChecksums(store.distributorDirectory);
                stringStringMap.putAll(this.legacyChecksums);
                if (!stringStringMap.isEmpty()) {
                    this.writeChecksums(store.directory, stringStringMap);
                }
            }
        }

        synchronized void writeChecksums(Directory directory, Map<String, String> checksums) throws IOException {
            String checksumName = Store.CHECKSUMS_PREFIX + System.currentTimeMillis();
            while (directory.fileExists(checksumName)) {
                checksumName = Store.CHECKSUMS_PREFIX + System.currentTimeMillis();
            }
            try (IndexOutput output = directory.createOutput(checksumName, IOContext.DEFAULT);){
                output.writeInt(0);
                output.writeStringStringMap(checksums);
            }
            directory.sync(Collections.singleton(checksumName));
        }

        public void clear() {
            this.legacyChecksums.clear();
        }

        public void remove(String name) {
            this.legacyChecksums.remove(name);
        }
    }

    public static final class MetadataSnapshot
    implements Iterable<StoreFileMetaData> {
        private final ImmutableMap<String, StoreFileMetaData> metadata;

        MetadataSnapshot(Directory directory, ESLogger logger) throws IOException {
            this.metadata = this.buildMetadata(directory, logger);
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        ImmutableMap<String, StoreFileMetaData> buildMetadata(Directory directory, ESLogger logger) throws IOException {
            ImmutableMap.Builder<String, StoreFileMetaData> builder = ImmutableMap.builder();
            Map<String, String> checksumMap = MetadataSnapshot.readLegacyChecksums(directory);
            try {
                SegmentInfos segmentCommitInfos;
                try {
                    segmentCommitInfos = Store.readLastCommittedSegmentsInfo(directory);
                }
                catch (FileNotFoundException | NoSuchFileException ex) {
                    logger.trace("Can't read segment infos", ex, new Object[0]);
                    return ImmutableMap.of();
                }
                Version maxVersion = Version.LUCENE_3_0;
                HashSet<String> added = new HashSet<String>();
                for (SegmentCommitInfo info : segmentCommitInfos) {
                    Version version = Lucene.parseVersionLenient(info.info.getVersion(), Version.LUCENE_3_0);
                    if (version.onOrAfter(maxVersion)) {
                        maxVersion = version;
                    }
                    for (String file : Iterables.concat(info.info.files(), info.files())) {
                        if (added.contains(file)) continue;
                        String legacyChecksum = checksumMap.get(file);
                        if (version.onOrAfter(Version.LUCENE_4_8) && legacyChecksum == null) {
                            MetadataSnapshot.checksumFromLuceneFile(directory, file, builder, logger, version);
                        } else {
                            builder.put(file, new StoreFileMetaData(file, directory.fileLength(file), legacyChecksum, null));
                        }
                        added.add(file);
                    }
                }
                for (String file : Arrays.asList(segmentCommitInfos.getSegmentsFileName(), "segments.gen")) {
                    if (added.contains(file)) continue;
                    try {
                        String legacyChecksum = checksumMap.get(file);
                        if (maxVersion.onOrAfter(Version.LUCENE_4_8) && legacyChecksum == null) {
                            MetadataSnapshot.checksumFromLuceneFile(directory, file, builder, logger, maxVersion);
                        } else {
                            builder.put(file, new StoreFileMetaData(file, directory.fileLength(file), legacyChecksum, null));
                        }
                        added.add(file);
                    }
                    catch (FileNotFoundException | NoSuchFileException ex) {
                        if ("segments.gen".equals(file)) continue;
                        throw ex;
                        return builder.build();
                    }
                }
            }
            catch (CorruptIndexException ex) {
                throw ex;
            }
            catch (FileNotFoundException | NoSuchFileException ex) {
                logger.warn("Can't open file to read checksums", ex, new Object[0]);
                return ImmutableMap.of();
            }
            catch (Throwable ex) {
                try {
                    Lucene.checkSegmentInfoIntegrity(directory);
                    throw ex;
                }
                catch (CorruptIndexException cex) {
                    cex.addSuppressed(ex);
                    throw cex;
                }
                catch (Throwable e) {
                    // empty catch block
                }
                throw ex;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        static Map<String, String> readLegacyChecksums(Directory directory) throws IOException {
            Directory directory2 = directory;
            synchronized (directory2) {
                long lastFound = -1L;
                for (String name : directory.listAll()) {
                    long current;
                    if (!Store.isChecksum(name) || (current = Long.parseLong(name.substring(Store.CHECKSUMS_PREFIX.length()))) <= lastFound) continue;
                    lastFound = current;
                }
                if (lastFound <= -1L) {
                    return new HashMap<String, String>();
                }
                try (IndexInput indexInput = directory.openInput(Store.CHECKSUMS_PREFIX + lastFound, IOContext.READONCE);){
                    indexInput.readInt();
                    Map<String, String> map = indexInput.readStringStringMap();
                    return map;
                }
            }
        }

        private static void checksumFromLuceneFile(Directory directory, String file, ImmutableMap.Builder<String, StoreFileMetaData> builder, ESLogger logger, Version version) throws IOException {
            try (IndexInput in = directory.openInput(file, IOContext.READONCE);){
                try {
                    if (in.length() < (long)CodecUtil.footerLength()) {
                        throw new CorruptIndexException("Can't retrieve checksum from file: " + file + " file length must be >= " + CodecUtil.footerLength() + " but was: " + in.length());
                    }
                    String checksum = Store.digestToString(CodecUtil.retrieveChecksum(in));
                    builder.put(file, new StoreFileMetaData(file, directory.fileLength(file), checksum, version));
                }
                catch (Throwable ex) {
                    logger.debug("Can retrieve checksum from file [{}]", ex, file);
                    throw ex;
                }
            }
        }

        @Override
        public Iterator<StoreFileMetaData> iterator() {
            return ((ImmutableCollection)this.metadata.values()).iterator();
        }

        public StoreFileMetaData get(String name) {
            return this.metadata.get(name);
        }

        public Map<String, StoreFileMetaData> asMap() {
            return this.metadata;
        }
    }

    public class StoreDirectory
    extends FilterDirectory
    implements ForceSyncDirectory {
        StoreDirectory(Directory delegateDirectory) throws IOException {
            super(delegateDirectory);
        }

        public ShardId shardId() {
            this.ensureOpen();
            return Store.this.shardId();
        }

        @Nullable
        public CodecService codecService() {
            this.ensureOpen();
            return Store.this.codecService;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        @Override
        public IndexInput openInput(String name, IOContext context) throws IOException {
            IndexInput in = super.openInput(name, context);
            boolean success = false;
            try {
                Compressor compressor;
                if ((name.endsWith(".fdt") || name.endsWith(".tvf")) && (compressor = CompressorFactory.compressor(in)) != null) {
                    return compressor.indexInput(in);
                }
                success = true;
                if (success) return in;
            }
            catch (Throwable throwable) {
                if (success) throw throwable;
                IOUtils.closeWhileHandlingException(in);
                throw throwable;
            }
            IOUtils.closeWhileHandlingException(in);
            return in;
        }

        @Override
        public void close() throws IOException {
            assert (false) : "Nobody should close this directory except of the Store itself";
        }

        @Override
        public void sync(Collection<String> names) throws IOException {
            if (Store.this.sync) {
                super.sync(names);
            }
        }

        private void innerClose() throws IOException {
            super.close();
        }

        @Override
        public void forceSync(String name) throws IOException {
            this.sync(ImmutableList.of(name));
        }

        @Override
        public String toString() {
            return "store(" + this.in.toString() + ")";
        }
    }
}

