/*
 * Copyright 2015 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */
package org.jbosson.plugins.fuse;

import java.util.*;

/**
 * Collection of values for creating Container traits.
 *
 * @author dbokde
 */
class FabricContainerTraits {
    private String containerVersion;
    private final Set<String> profiles = new HashSet<String>();
    private final Map<String, List<String>> parentProfileMap = new HashMap<String, List<String>>();
    private final Set<String> clusters = new HashSet<String>();

    public String getContainerVersion() {
        return containerVersion;
    }

    public void setContainerVersion(String containerVersion) {
        this.containerVersion = containerVersion;
    }

    public void setProfiles(String[] profileNames) {
        for (String profileName : profileNames) {
            // add container version to name, since DynaGroups don't support multiple traits in expression
            profiles.add("[" + profileName + "{"+ containerVersion + "}]");
        }
    }

    public boolean setParentProfiles(String child, String[] parents) {
        boolean newChild = false;
        if (!parentProfileMap.containsKey(child)) {
            parentProfileMap.put(child, Arrays.asList(parents));
            newChild = true;
        }
        return newChild;
    }

    public void addCluster(String clusterName) {
        clusters.add("[" + clusterName + "]");
    }

    public String getProfileTrait() {
        return getValue(profiles);
    }

    public String getParentProfilesTrait() {
        // add container version to name, since DynaGroups don't support multiple traits in expression
        final Set<String> parentValues = new HashSet<String>();
        for (Map.Entry<String, List<String>> entry : parentProfileMap.entrySet()) {
            final StringBuilder value = new StringBuilder();
            value.append("[");
            value.append(entry.getKey());
            value.append("{");
            value.append(containerVersion);
            value.append("}]");
            for (String parent : entry.getValue()) {
                // add parent marker
                value.append(":");
                value.append("[");
                value.append(parent);
                value.append("{");
                value.append(containerVersion);
                value.append("}]");
            }
            parentValues.add(value.toString());
        }

        return getValue(parentValues);
    }

    public String getClustersTrait() {
        return getValue(clusters);
    }

    private String getValue(Set<String> valueSet) {
        final StringBuffer value = new StringBuffer();
        int total = valueSet.size(), i = 0;
        for (String profile : valueSet) {
            value.append(profile);
            if (++i < total) {
                value.append(",");
            }
        }
        return value.toString();
    }

    public boolean hasChildProfile(String child) {
        return parentProfileMap.containsKey(child);
    }

}
