/*
 * Copyright 2013 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */
package org.jbosson.plugins.fuse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mc4j.ems.connection.EmsConnection;
import org.mc4j.ems.connection.bean.EmsBean;
import org.rhq.core.domain.configuration.Configuration;
import org.rhq.core.domain.configuration.PropertySimple;
import org.rhq.core.domain.resource.ResourceType;
import org.rhq.core.pluginapi.inventory.DiscoveredResourceDetails;
import org.rhq.core.pluginapi.inventory.ResourceDiscoveryContext;
import org.rhq.plugins.jmx.JMXComponent;
import org.rhq.plugins.jmx.util.ObjectNameQueryUtility;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Adds support for alternative objectNames in property objectName1
 *
 * @author dbokde
 */
public class ActiveMQDiscoveryComponent<T extends JMXComponent<?>> extends FuseMBeanDiscoveryComponent<T> {

    protected final Log log = LogFactory.getLog(this.getClass());
    private static final String PROPERTY_OBJECT_NAME1 = "objectName1";

    protected ResourceDiscoveryContext<T> discoveryContext;

    @Override
    public Set<DiscoveredResourceDetails> discoverResources(ResourceDiscoveryContext<T> context) {
        this.discoveryContext = context;
        return super.discoverResources(context);
    }

    @Override
    public Set<DiscoveredResourceDetails> performDiscovery(Configuration pluginConfiguration,
                                                           JMXComponent parentResourceComponent,
                                                           ResourceType resourceType, boolean skipUnknownProps) {

        String objectNameQueryTemplateOrig = pluginConfiguration.getSimple(PROPERTY_OBJECT_NAME).getStringValue();
        Set<DiscoveredResourceDetails> detailsSet = performDiscovery(pluginConfiguration, parentResourceComponent,
            resourceType, skipUnknownProps, objectNameQueryTemplateOrig);
        if (detailsSet.isEmpty()) {
            // try property objectName1 only if the default objectName didn't work
            objectNameQueryTemplateOrig = pluginConfiguration.getSimple(PROPERTY_OBJECT_NAME1).getStringValue();
            detailsSet = performDiscovery(pluginConfiguration, parentResourceComponent, resourceType,
                skipUnknownProps, objectNameQueryTemplateOrig);
        }

        return detailsSet;
    }

    /**
     * Shamelessly plagiarized from MBeanResourceDiscoveryComponent.performDiscovery(),
     * since that method has a fixed reference to objectName property
     * <p/>
     * NOTE that this method also overrides the discovered resource's objectName to be the actual bean name,
     * hence the default MBeanResourceComponent will still work
     */
    protected Set<DiscoveredResourceDetails> performDiscovery(Configuration pluginConfiguration,
                                                              JMXComponent parentResourceComponent,
                                                              ResourceType resourceType, boolean skipUnknownProps,
                                                              String objectNameQueryTemplateOrig) {

        log.debug("Discovering MBean resources with object name query template: " + objectNameQueryTemplateOrig);

        EmsConnection connection = parentResourceComponent.getEmsConnection();

        if (connection == null) {
            throw new NullPointerException("The parent resource component [" + parentResourceComponent
                + "] returned a null connection - cannot discover MBeans without a connection");
        }

        final Set<DiscoveredResourceDetails> services = new HashSet<DiscoveredResourceDetails>();
        String templates[] = objectNameQueryTemplateOrig.split("\\|");
        for (String objectNameQueryTemplate : templates) {
            // Get the query template, replacing the parent key variables with the values from the parent configuration
            ObjectNameQueryUtility queryUtility = new ObjectNameQueryUtility(objectNameQueryTemplate,
                (this.discoveryContext != null) ? this.discoveryContext.getParentResourceContext()
                    .getPluginConfiguration() : null);

            List<EmsBean> beans = connection.queryBeans(queryUtility.getTranslatedQuery());
            if (log.isDebugEnabled()) {
                log.debug("Found [" + beans.size() + "] mbeans for query [" + queryUtility.getTranslatedQuery() + "].");
            }
            for (EmsBean bean : beans) {
                if (queryUtility.setMatchedKeyValues(bean.getBeanName().getKeyProperties())) {
                    // Only use beans that have all the properties we've made variables of

                    // Don't match beans that have unexpected properties
                    if (skipUnknownProps
                        && queryUtility.isContainsExtraKeyProperties(bean.getBeanName().getKeyProperties().keySet())) {
                        continue;
                    }

                    String resourceKey = bean.getBeanName().getCanonicalName(); // The detected object name

                    String nameTemplate = (pluginConfiguration.getSimple(PROPERTY_NAME_TEMPLATE) != null) ?
                        pluginConfiguration.getSimple(PROPERTY_NAME_TEMPLATE).getStringValue() : null;

                    String descriptionTemplate = (pluginConfiguration.getSimple(PROPERTY_DESCRIPTION_TEMPLATE) != null)
                        ? pluginConfiguration.getSimple(PROPERTY_DESCRIPTION_TEMPLATE).getStringValue() : null;

                    String name = resourceKey;
                    if (nameTemplate != null) {
                        name = queryUtility.formatMessage(nameTemplate);
                    }

                    String description = null;
                    if (descriptionTemplate != null) {
                        description = queryUtility.formatMessage(descriptionTemplate);
                    }

                    DiscoveredResourceDetails service = new DiscoveredResourceDetails(resourceType, resourceKey, name,
                        "", description, null, null);
                    Configuration config = service.getPluginConfiguration();
                    config.put(new PropertySimple(PROPERTY_OBJECT_NAME, bean.getBeanName().toString()));

                    Map<String, String> mappedVariableValues = queryUtility.getVariableValues();
                    for (String key : mappedVariableValues.keySet()) {
                        config.put(new PropertySimple(key, mappedVariableValues.get(key)));
                    }

                    services.add(service);

                    // Clear out the variables for the next bean detected
                    queryUtility.resetVariables();
                }
            }

            if (log.isDebugEnabled()) {
                log.debug("[" + services.size() + "] services have been added");
            }

        }

        return services;
    }
}
