/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3native;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BufferedFSInputStream;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.s3.S3Exception;
import org.apache.hadoop.fs.s3native.FileMetadata;
import org.apache.hadoop.fs.s3native.Jets3tNativeFileSystemStore;
import org.apache.hadoop.fs.s3native.NativeFileSystemStore;
import org.apache.hadoop.fs.s3native.PartialListing;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.io.retry.RetryProxy;
import org.apache.hadoop.util.Progressable;

public class NativeS3FileSystem
extends FileSystem {
    public static final Log LOG = LogFactory.getLog(NativeS3FileSystem.class);
    private static final String FOLDER_SUFFIX = "_$folder$";
    private static final long MAX_S3_FILE_SIZE = 0x140000000L;
    static final String PATH_DELIMITER = "/";
    private static final int S3_MAX_LISTING_LENGTH = 1000;
    private URI uri;
    private NativeFileSystemStore store;
    private Path workingDir;

    public NativeS3FileSystem() {
    }

    public NativeS3FileSystem(NativeFileSystemStore store2) {
        this.store = store2;
    }

    @Override
    public void initialize(URI uri, Configuration conf) throws IOException {
        super.initialize(uri, conf);
        if (this.store == null) {
            this.store = NativeS3FileSystem.createDefaultStore(conf);
        }
        this.store.initialize(uri, conf);
        this.setConf(conf);
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.workingDir = new Path("/user", System.getProperty("user.name")).makeQualified(this);
    }

    private static NativeFileSystemStore createDefaultStore(Configuration conf) {
        Jets3tNativeFileSystemStore store2 = new Jets3tNativeFileSystemStore();
        RetryPolicy basePolicy = RetryPolicies.retryUpToMaximumCountWithFixedSleep(conf.getInt("fs.s3.maxRetries", 4), conf.getLong("fs.s3.sleepTimeSeconds", 10L), TimeUnit.SECONDS);
        HashMap<Class<? extends Exception>, RetryPolicy> exceptionToPolicyMap = new HashMap<Class<? extends Exception>, RetryPolicy>();
        exceptionToPolicyMap.put(IOException.class, basePolicy);
        exceptionToPolicyMap.put(S3Exception.class, basePolicy);
        RetryPolicy methodPolicy = RetryPolicies.retryByException(RetryPolicies.TRY_ONCE_THEN_FAIL, exceptionToPolicyMap);
        HashMap<String, RetryPolicy> methodNameToPolicyMap = new HashMap<String, RetryPolicy>();
        methodNameToPolicyMap.put("storeFile", methodPolicy);
        return (NativeFileSystemStore)RetryProxy.create(NativeFileSystemStore.class, (Object)store2, methodNameToPolicyMap);
    }

    private static String pathToKey(Path path) {
        if (!path.isAbsolute()) {
            throw new IllegalArgumentException("Path must be absolute: " + path);
        }
        return path.toUri().getPath().substring(1);
    }

    private static Path keyToPath(String key) {
        return new Path(PATH_DELIMITER + key);
    }

    private Path makeAbsolute(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return new Path(this.workingDir, path);
    }

    @Override
    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        throw new IOException("Not supported");
    }

    @Override
    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        if (this.exists(f) && !overwrite) {
            throw new IOException("File already exists:" + f);
        }
        Path absolutePath = this.makeAbsolute(f);
        String key = NativeS3FileSystem.pathToKey(absolutePath);
        return new FSDataOutputStream(new NativeS3FsOutputStream(this.getConf(), this.store, key, progress, bufferSize), this.statistics);
    }

    @Override
    @Deprecated
    public boolean delete(Path path) throws IOException {
        return this.delete(path, true);
    }

    @Override
    public boolean delete(Path f, boolean recursive) throws IOException {
        FileStatus status;
        try {
            status = this.getFileStatus(f);
        }
        catch (FileNotFoundException e) {
            return false;
        }
        Path absolutePath = this.makeAbsolute(f);
        String key = NativeS3FileSystem.pathToKey(absolutePath);
        if (status.isDir()) {
            FileStatus[] contents = this.listStatus(f);
            if (!recursive && contents.length > 0) {
                throw new IOException("Directory " + f.toString() + " is not empty.");
            }
            for (FileStatus p : contents) {
                if (this.delete(p.getPath(), recursive)) continue;
                return false;
            }
            this.store.delete(key + FOLDER_SUFFIX);
        } else {
            this.store.delete(key);
        }
        return true;
    }

    @Override
    public FileStatus getFileStatus(Path f) throws IOException {
        Path absolutePath = this.makeAbsolute(f);
        String key = NativeS3FileSystem.pathToKey(absolutePath);
        if (key.length() == 0) {
            return this.newDirectory(absolutePath);
        }
        FileMetadata meta = this.store.retrieveMetadata(key);
        if (meta != null) {
            return this.newFile(meta, absolutePath);
        }
        if (this.store.retrieveMetadata(key + FOLDER_SUFFIX) != null) {
            return this.newDirectory(absolutePath);
        }
        PartialListing listing = this.store.list(key, 1);
        if (listing.getFiles().length > 0 || listing.getCommonPrefixes().length > 0) {
            return this.newDirectory(absolutePath);
        }
        throw new FileNotFoundException(absolutePath + ": No such file or directory.");
    }

    @Override
    public URI getUri() {
        return this.uri;
    }

    @Override
    public FileStatus[] listStatus(Path f) throws IOException {
        PartialListing listing;
        FileMetadata meta;
        Path absolutePath = this.makeAbsolute(f);
        String key = NativeS3FileSystem.pathToKey(absolutePath);
        if (key.length() > 0 && (meta = this.store.retrieveMetadata(key)) != null) {
            return new FileStatus[]{this.newFile(meta, absolutePath)};
        }
        URI pathUri = absolutePath.toUri();
        TreeSet<FileStatus> status = new TreeSet<FileStatus>();
        String priorLastKey = null;
        do {
            String relativePath;
            Path subpath;
            listing = this.store.list(key, 1000, priorLastKey);
            for (FileMetadata fileMetadata : listing.getFiles()) {
                subpath = NativeS3FileSystem.keyToPath(fileMetadata.getKey());
                relativePath = pathUri.relativize(subpath.toUri()).getPath();
                if (relativePath.endsWith(FOLDER_SUFFIX)) {
                    status.add(this.newDirectory(new Path(absolutePath, relativePath.substring(0, relativePath.indexOf(FOLDER_SUFFIX)))));
                    continue;
                }
                status.add(this.newFile(fileMetadata, subpath));
            }
            for (String commonPrefix : listing.getCommonPrefixes()) {
                subpath = NativeS3FileSystem.keyToPath(commonPrefix);
                relativePath = pathUri.relativize(subpath.toUri()).getPath();
                status.add(this.newDirectory(new Path(absolutePath, relativePath)));
            }
        } while ((priorLastKey = listing.getPriorLastKey()) != null);
        if (status.isEmpty() && this.store.retrieveMetadata(key + FOLDER_SUFFIX) == null) {
            return null;
        }
        return status.toArray(new FileStatus[0]);
    }

    private FileStatus newFile(FileMetadata meta, Path path) {
        return new FileStatus(meta.getLength(), false, 1, 0x140000000L, meta.getLastModified(), path.makeQualified(this));
    }

    private FileStatus newDirectory(Path path) {
        return new FileStatus(0L, true, 1, 0x140000000L, 0L, path.makeQualified(this));
    }

    @Override
    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        Path absolutePath = this.makeAbsolute(f);
        ArrayList<Path> paths = new ArrayList<Path>();
        do {
            paths.add(0, absolutePath);
        } while ((absolutePath = absolutePath.getParent()) != null);
        boolean result = true;
        for (Path path : paths) {
            result &= this.mkdir(path);
        }
        return result;
    }

    private boolean mkdir(Path f) throws IOException {
        try {
            FileStatus fileStatus = this.getFileStatus(f);
            if (!fileStatus.isDir()) {
                throw new IOException(String.format("Can't make directory for path %s since it is a file.", f));
            }
        }
        catch (FileNotFoundException e) {
            String key = NativeS3FileSystem.pathToKey(f) + FOLDER_SUFFIX;
            this.store.storeEmptyFile(key);
        }
        return true;
    }

    @Override
    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        if (!this.exists(f)) {
            throw new FileNotFoundException(f.toString());
        }
        Path absolutePath = this.makeAbsolute(f);
        String key = NativeS3FileSystem.pathToKey(absolutePath);
        return new FSDataInputStream(new BufferedFSInputStream(new NativeS3FsInputStream(this.store.retrieve(key), key), bufferSize));
    }

    private void createParent(Path path) throws IOException {
        String key;
        Path parent = path.getParent();
        if (parent != null && (key = NativeS3FileSystem.pathToKey(this.makeAbsolute(parent))).length() > 0) {
            this.store.storeEmptyFile(key + FOLDER_SUFFIX);
        }
    }

    private boolean existsAndIsFile(Path f) throws IOException {
        Path absolutePath = this.makeAbsolute(f);
        String key = NativeS3FileSystem.pathToKey(absolutePath);
        if (key.length() == 0) {
            return false;
        }
        FileMetadata meta = this.store.retrieveMetadata(key);
        if (meta != null) {
            return true;
        }
        if (this.store.retrieveMetadata(key + FOLDER_SUFFIX) != null) {
            return false;
        }
        PartialListing listing = this.store.list(key, 1, null);
        if (listing.getFiles().length > 0 || listing.getCommonPrefixes().length > 0) {
            return false;
        }
        throw new FileNotFoundException(absolutePath + ": No such file or directory");
    }

    @Override
    public boolean rename(Path src, Path dst) throws IOException {
        String dstKey;
        String srcKey = NativeS3FileSystem.pathToKey(this.makeAbsolute(src));
        if (srcKey.length() == 0) {
            return false;
        }
        try {
            boolean dstIsFile = this.existsAndIsFile(dst);
            if (dstIsFile) {
                return false;
            }
            dstKey = NativeS3FileSystem.pathToKey(this.makeAbsolute(new Path(dst, src.getName())));
        }
        catch (FileNotFoundException e) {
            dstKey = NativeS3FileSystem.pathToKey(this.makeAbsolute(dst));
            try {
                if (!this.getFileStatus(dst.getParent()).isDir()) {
                    return false;
                }
            }
            catch (FileNotFoundException ex) {
                return false;
            }
        }
        try {
            boolean srcIsFile = this.existsAndIsFile(src);
            if (srcIsFile) {
                this.store.rename(srcKey, dstKey);
            } else {
                PartialListing listing;
                this.store.delete(srcKey + FOLDER_SUFFIX);
                this.store.storeEmptyFile(dstKey + FOLDER_SUFFIX);
                String priorLastKey = null;
                do {
                    listing = this.store.listAll(srcKey, 1000, priorLastKey);
                    for (FileMetadata file : listing.getFiles()) {
                        this.store.rename(file.getKey(), dstKey + file.getKey().substring(srcKey.length()));
                    }
                } while ((priorLastKey = listing.getPriorLastKey()) != null);
            }
            this.createParent(src);
            return true;
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    @Override
    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir;
    }

    @Override
    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    private class NativeS3FsOutputStream
    extends OutputStream {
        private Configuration conf;
        private String key;
        private File backupFile;
        private OutputStream backupStream;
        private MessageDigest digest;
        private boolean closed;

        public NativeS3FsOutputStream(Configuration conf, NativeFileSystemStore store2, String key, Progressable progress, int bufferSize) throws IOException {
            this.conf = conf;
            this.key = key;
            this.backupFile = this.newBackupFile();
            try {
                this.digest = MessageDigest.getInstance("MD5");
                this.backupStream = new BufferedOutputStream(new DigestOutputStream(new FileOutputStream(this.backupFile), this.digest));
            }
            catch (NoSuchAlgorithmException e) {
                LOG.warn((Object)"Cannot load MD5 digest algorithm,skipping message integrity check.", (Throwable)e);
                this.backupStream = new BufferedOutputStream(new FileOutputStream(this.backupFile));
            }
        }

        private File newBackupFile() throws IOException {
            File dir = new File(this.conf.get("fs.s3.buffer.dir"));
            if (!dir.mkdirs() && !dir.exists()) {
                throw new IOException("Cannot create S3 buffer directory: " + dir);
            }
            File result = File.createTempFile("output-", ".tmp", dir);
            result.deleteOnExit();
            return result;
        }

        @Override
        public void flush() throws IOException {
            this.backupStream.flush();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public synchronized void close() throws IOException {
            if (this.closed) {
                return;
            }
            this.backupStream.close();
            try {
                byte[] md5Hash = this.digest == null ? null : this.digest.digest();
                NativeS3FileSystem.this.store.storeFile(this.key, this.backupFile, md5Hash);
            }
            finally {
                if (!this.backupFile.delete()) {
                    LOG.warn((Object)("Could not delete temporary s3n file: " + this.backupFile));
                }
                super.close();
                this.closed = true;
            }
        }

        @Override
        public void write(int b) throws IOException {
            this.backupStream.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.backupStream.write(b, off, len);
        }
    }

    private class NativeS3FsInputStream
    extends FSInputStream {
        private InputStream in;
        private final String key;
        private long pos = 0L;

        public NativeS3FsInputStream(InputStream in, String key) {
            this.in = in;
            this.key = key;
        }

        @Override
        public synchronized int read() throws IOException {
            int result = this.in.read();
            if (result != -1) {
                ++this.pos;
            }
            return result;
        }

        @Override
        public synchronized int read(byte[] b, int off, int len) throws IOException {
            int result = this.in.read(b, off, len);
            if (result > 0) {
                this.pos += (long)result;
            }
            return result;
        }

        @Override
        public void close() throws IOException {
            this.in.close();
        }

        @Override
        public synchronized void seek(long pos) throws IOException {
            this.in.close();
            this.in = NativeS3FileSystem.this.store.retrieve(this.key, pos);
            this.pos = pos;
        }

        @Override
        public synchronized long getPos() throws IOException {
            return this.pos;
        }

        @Override
        public boolean seekToNewSource(long targetPos) throws IOException {
            return false;
        }
    }
}

