/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.impl;

import org.apache.camel.ContextTestSupport;
import org.apache.camel.util.URISupport;

/**
 * @version
 */
public class DefaultEndpointTest extends ContextTestSupport {

    public void testSanitizeUri() {
        assertNull(URISupport.sanitizeUri(null));
        assertEquals("", URISupport.sanitizeUri(""));
        assertSanitizedUriUnchanged("http://camel.apache.org");
        assertSanitizedUriUnchanged("irc://irc.codehaus.org/camel");
        assertSanitizedUriUnchanged("direct:foo?bar=123&cheese=yes");
        assertSanitizedUriUnchanged("https://issues.apache.org/activemq/secure/AddComment!default.jspa?id=33239");
        assertEquals("ftp://host.mysite.com/records?passiveMode=true&user=someuser&password=******",
            URISupport.sanitizeUri("ftp://host.mysite.com/records?passiveMode=true&user=someuser&password=superSecret"));
        assertEquals("sftp://host.mysite.com/records?user=someuser&privateKeyFile=key.file&privateKeyFilePassphrase=******&knownHostsFile=hosts.list",
            URISupport.sanitizeUri("sftp://host.mysite.com/records?user=someuser&privateKeyFile=key.file&privateKeyFilePassphrase=superSecret&knownHostsFile=hosts.list"));
        assertEquals("aws-sqs://MyQueue?accessKey=1672t4rflhnhli3&secretKey=******",
            URISupport.sanitizeUri("aws-sqs://MyQueue?accessKey=1672t4rflhnhli3&secretKey=qi472qfberu33dqjncq"));
    }

    /**
     * Ensures that the Uri was not changed because no password was found.
     *
     * @param uri The uri to test.
     */
    private void assertSanitizedUriUnchanged(String uri) {
        assertEquals(uri, URISupport.sanitizeUri(uri));
    }

}
