/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.dataformat.csv;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.camel.Exchange;
import org.apache.camel.spi.DataFormat;
import org.apache.camel.util.ExchangeHelper;
import org.apache.camel.util.ObjectHelper;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVStrategy;
import org.apache.commons.csv.writer.CSVConfig;
import org.apache.commons.csv.writer.CSVField;
import org.apache.commons.csv.writer.CSVWriter;

/**
 * CSV Data format.
 * <p/>
 * By default, columns are autogenerated in the resulting CSV. Subsequent
 * messages use the previously created columns with new fields being added at
 * the end of the line. Thus, field order is the same from message to message.
 * Autogeneration can be disabled. In this case, only the fields defined in
 * csvConfig are written on the output.
 *
 * @version $Revision: 3239 $
 */
public class CsvDataFormat implements DataFormat {
    private CSVStrategy strategy = CSVStrategy.DEFAULT_STRATEGY;
    private CSVConfig config = new CSVConfig();
    private boolean autogenColumns = true;

    public void marshal(Exchange exchange, Object object, OutputStream outputStream) throws Exception {
        ObjectHelper.notNull(config, "config");

        Map map = ExchangeHelper.convertToMandatoryType(exchange, Map.class, object);
        OutputStreamWriter out = new OutputStreamWriter(outputStream);
        try {
            if (autogenColumns) {
                // no specific config has been set so lets add fields
                Set set = map.keySet();
                updateFieldsInConfig(set, exchange);
            }
            CSVWriter writer = new CSVWriter(config);
            writer.setWriter(out);
            writer.writeRecord(map);
        } finally {
            out.close();
        }
    }

    public Object unmarshal(Exchange exchange, InputStream inputStream) throws Exception {
        InputStreamReader in = new InputStreamReader(inputStream);
        try {
            CSVParser parser = new CSVParser(in, getStrategy());
            List<List<String>> list = new ArrayList<List<String>>();
            while (true) {
                String[] strings = parser.getLine();
                if (strings == null) {
                    break;
                }
                List<String> line = Arrays.asList(strings);
                list.add(line);
            }
            if (list.size() == 1) {
                return list.get(0);
            } else {
                return list;
            }
        } finally {
            in.close();
        }
    }

    public void setConfig(CSVConfig config) {
        this.config = config;
    }

    public CSVStrategy getStrategy() {
        return strategy;
    }

    public void setStrategy(CSVStrategy strategy) {
        this.strategy = strategy;
    }

    public boolean isAutogenColumns() {
        return autogenColumns;
    }

    /**
     * Auto generate columns.
     *
     * @param autogenColumns set to false to disallow column autogeneration (default true)
     */
    public void setAutogenColumns(boolean autogenColumns) {
        this.autogenColumns = autogenColumns;
    }

    protected CSVConfig createConfig() {
        return new CSVConfig();
    }

    private synchronized void updateFieldsInConfig(Set set, Exchange exchange) {
        for (Object value : set) {
            if (value != null) {
                String text = exchange.getContext().getTypeConverter().convertTo(String.class, value);
                // do not add field twice
                if (config.getField(text) == null) {
                    CSVField field = new CSVField(text);
                    config.addField(field);
                }
            }
        }
    }

}
