/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.cxf.transport.servlet;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.http.HttpServletResponse;

import org.apache.cxf.Bus;
import org.apache.cxf.message.Message;
import org.apache.cxf.message.MessageImpl;
import org.apache.cxf.service.model.EndpointInfo;
import org.apache.cxf.transport.ConduitInitiator;
import org.apache.cxf.transport.MessageObserver;
import org.apache.cxf.transport.http.AbstractHTTPDestination;


public class ServletDestination extends AbstractHTTPDestination {
    
    static final Logger LOG = Logger.getLogger(ServletDestination.class.getName());
        
    private static final long serialVersionUID = 1L;        
    
    
    /**
     * Constructor, allowing subsititution of configuration.
     * 
     * @param b the associated Bus
     * @param ci the associated conduit initiator
     * @param ei the endpoint info of the destination 
     * @param cfg the configuration
     * @throws IOException
     */    
    public ServletDestination(Bus b,
                              ConduitInitiator ci,
                              EndpointInfo ei)
        throws IOException {
        // would add the default port to the address
        super(b, ci, ei, false);
    }
    
    
    protected Logger getLogger() {
        return LOG;
    }

    /**
     * Copy the response headers into the response.
     * 
     * @param message the current message
     * @param headers the current set of headers
     */
    protected void copyResponseHeaders(Message message, HttpServletResponse response) {
        String ct = (String) message.get(Message.CONTENT_TYPE);
        String enc = (String) message.get(Message.ENCODING);
        
        if (null != ct) {
            if (enc != null && ct.indexOf("charset=") == -1) {
                ct = ct + "; charset=" + enc;
            }
            response.setContentType(ct);
        } else if (enc != null) {
            response.setContentType("text/xml; charset=" + enc);
        }
        
    }
    
    protected void doMessage(MessageImpl inMessage) throws IOException {
        try {
            
            setHeaders(inMessage);
            
            inMessage.setDestination(this);
            
            incomingObserver.onMessage(inMessage);
            
        } finally {
            if (LOG.isLoggable(Level.INFO)) {
                LOG.info("Finished servicing http request on thread: " + Thread.currentThread());
            }
        }        
    }
    
    public MessageObserver getMessageObserver() {
        return this.incomingObserver;
    }

    public EndpointInfo getEndpointInfo() {
        return endpointInfo;
    }
}
