/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.cxf.jaxrs.utils;


import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.PathSegment;
import javax.ws.rs.core.Request;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.core.UriInfo;
import javax.ws.rs.ext.ContextResolver;
import javax.ws.rs.ext.MessageBodyWorkers;
import javax.xml.bind.JAXBContext;

import org.apache.cxf.jaxrs.Customer;
import org.apache.cxf.jaxrs.JAXBContextProvider;
import org.apache.cxf.jaxrs.JAXRSServiceFactoryBean;
import org.apache.cxf.jaxrs.JAXRSServiceImpl;
import org.apache.cxf.jaxrs.SimpleFactory;
import org.apache.cxf.jaxrs.impl.HttpHeadersImpl;
import org.apache.cxf.jaxrs.impl.MetadataMap;
import org.apache.cxf.jaxrs.impl.PathSegmentImpl;
import org.apache.cxf.jaxrs.impl.ProvidersImpl;
import org.apache.cxf.jaxrs.impl.RequestImpl;
import org.apache.cxf.jaxrs.impl.SecurityContextImpl;
import org.apache.cxf.jaxrs.impl.UriInfoImpl;
import org.apache.cxf.jaxrs.impl.tl.ThreadLocalProxy;
import org.apache.cxf.jaxrs.impl.tl.ThreadLocalUriInfo;
import org.apache.cxf.jaxrs.lifecycle.PerRequestResourceProvider;
import org.apache.cxf.jaxrs.lifecycle.SingletonResourceProvider;
import org.apache.cxf.jaxrs.model.ClassResourceInfo;
import org.apache.cxf.jaxrs.model.MethodDispatcher;
import org.apache.cxf.jaxrs.model.OperationResourceInfo;
import org.apache.cxf.jaxrs.model.URITemplate;
import org.apache.cxf.jaxrs.provider.ProviderFactory;
import org.apache.cxf.message.Message;
import org.apache.cxf.message.MessageImpl;
import org.apache.cxf.transport.http.AbstractHTTPDestination;
import org.easymock.EasyMock;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class JAXRSUtilsTest extends Assert {
    
    @Before
    public void setUp() {
    }

    @Test
    public void testSelectBetweenMultipleResourceClasses() throws Exception {
        JAXRSServiceFactoryBean sf = new JAXRSServiceFactoryBean();
        sf.setResourceClasses(org.apache.cxf.jaxrs.resources.BookStoreNoSubResource.class,
                              org.apache.cxf.jaxrs.resources.BookStore.class);
        sf.create();        
        List<ClassResourceInfo> resources = ((JAXRSServiceImpl)sf.getService()).getClassResourceInfos();
        MultivaluedMap<String, String> map = new MetadataMap<String, String>();
        ClassResourceInfo bStore = JAXRSUtils.selectResourceClass(resources, "/bookstore", map);
        assertEquals(bStore.getResourceClass(), org.apache.cxf.jaxrs.resources.BookStore.class);
        
        bStore = JAXRSUtils.selectResourceClass(resources, "/bookstore/", map);
        assertEquals(bStore.getResourceClass(), 
                     org.apache.cxf.jaxrs.resources.BookStore.class);
        
        bStore = JAXRSUtils.selectResourceClass(resources, "/bookstore/bar", map);
        assertEquals(bStore.getResourceClass(), 
                     org.apache.cxf.jaxrs.resources.BookStoreNoSubResource.class);
    }
    
    @Test
    public void testSelectBetweenMultipleResourceClasses2() throws Exception {
        JAXRSServiceFactoryBean sf = new JAXRSServiceFactoryBean();
        sf.setResourceClasses(org.apache.cxf.jaxrs.resources.TestResourceTemplate1.class,
                              org.apache.cxf.jaxrs.resources.TestResourceTemplate2.class);
        sf.create();        
        List<ClassResourceInfo> resources = ((JAXRSServiceImpl)sf.getService()).getClassResourceInfos();
        MultivaluedMap<String, String> map = new MetadataMap<String, String>();
        ClassResourceInfo bStore = JAXRSUtils.selectResourceClass(resources, "/1", map);
        assertEquals(bStore.getResourceClass(), org.apache.cxf.jaxrs.resources.TestResourceTemplate1.class);
        
        bStore = JAXRSUtils.selectResourceClass(resources, "/1/", map);
        assertEquals(bStore.getResourceClass(), 
                     org.apache.cxf.jaxrs.resources.TestResourceTemplate1.class);
        
        bStore = JAXRSUtils.selectResourceClass(resources, "/1/foo", map);
        assertEquals(bStore.getResourceClass(), 
                     org.apache.cxf.jaxrs.resources.TestResourceTemplate2.class);
        
        bStore = JAXRSUtils.selectResourceClass(resources, "/1/foo/bar", map);
        assertEquals(bStore.getResourceClass(), 
                     org.apache.cxf.jaxrs.resources.TestResourceTemplate2.class);
    }
    
    @Test
    public void testSelectBetweenMultipleResourceClasses3() throws Exception {
        JAXRSServiceFactoryBean sf = new JAXRSServiceFactoryBean();
        sf.setResourceClasses(org.apache.cxf.jaxrs.resources.TestResourceTemplate4.class,
                              org.apache.cxf.jaxrs.resources.TestResourceTemplate3.class);
        sf.create();        
        List<ClassResourceInfo> resources = ((JAXRSServiceImpl)sf.getService()).getClassResourceInfos();
        MultivaluedMap<String, String> map = new MetadataMap<String, String>();
        ClassResourceInfo bStore = JAXRSUtils.selectResourceClass(resources, "/", map);
        assertEquals(bStore.getResourceClass(), org.apache.cxf.jaxrs.resources.TestResourceTemplate3.class);
        
        bStore = JAXRSUtils.selectResourceClass(resources, "/test", map);
        assertEquals(bStore.getResourceClass(), 
                     org.apache.cxf.jaxrs.resources.TestResourceTemplate4.class);
        
    }
    
    @Test
    public void testFindTargetResourceClass() throws Exception {
        JAXRSServiceFactoryBean sf = new JAXRSServiceFactoryBean();
        sf.setResourceClasses(org.apache.cxf.jaxrs.resources.BookStoreNoSubResource.class);
        sf.create();        
        List<ClassResourceInfo> resources = ((JAXRSServiceImpl)sf.getService()).getClassResourceInfos();

        String contentTypes = "*/*";
        
        //If acceptContentTypes does not specify a specific Mime type, the  
        //method is declared with a most specific ProduceMime type is selected.
        OperationResourceInfo ori = findTargetResourceClass(resources, null, 
             "/bookstore/1/books/123/", "GET", new MetadataMap<String, String>(), contentTypes, 
             getTypes("*/*"));       
        assertNotNull(ori);
        assertEquals("getBookJSON", ori.getMethodToInvoke().getName());
        
        //test
        ori = findTargetResourceClass(resources, null, "/bookstore/1/books/123",
             "GET", new MetadataMap<String, String>(), contentTypes, getTypes("application/json"));        
        assertNotNull(ori);
        assertEquals("getBookJSON", ori.getMethodToInvoke().getName());
        
        //test 
        ori = findTargetResourceClass(resources, null, "/bookstore/1/books/123",
              "GET", new MetadataMap<String, String>(), contentTypes, getTypes("application/xml"));        
        assertNotNull(ori);
        assertEquals("getBook", ori.getMethodToInvoke().getName());
        
        //test 
        ori = findTargetResourceClass(resources, null, "/bookstore/1/books",
                      "GET", new MetadataMap<String, String>(), contentTypes, 
                      getTypes("application/xml"));        
        assertNotNull(ori);
        assertEquals("getBooks", ori.getMethodToInvoke().getName());
        
        //test find POST
        ori = findTargetResourceClass(resources, null, "/bookstore/1/books",
                 "POST", new MetadataMap<String, String>(), contentTypes, getTypes("application/xml"));       
        assertNotNull(ori);
        assertEquals("addBook", ori.getMethodToInvoke().getName());
        
        //test find PUT
        ori = findTargetResourceClass(resources, null, "/bookstore/1/books",
            "PUT", new MetadataMap<String, String>(), contentTypes, getTypes("application/xml"));  
        assertEquals("updateBook", ori.getMethodToInvoke().getName());
        
        //test find DELETE
        ori = findTargetResourceClass(resources, null, "/bookstore/1/books/123",
             "DELETE", new MetadataMap<String, String>(), contentTypes, getTypes("application/xml"));        
        assertNotNull(ori);
        assertEquals("deleteBook", ori.getMethodToInvoke().getName());     
        
    }
    
    private List<MediaType> getTypes(String types) {
        return JAXRSUtils.parseMediaTypes(types);
    }
    
    @Test
    public void testFindTargetResourceClassWithTemplates() throws Exception {
        JAXRSServiceFactoryBean sf = new JAXRSServiceFactoryBean();
        sf.setResourceClasses(org.apache.cxf.jaxrs.resources.BookStoreTemplates.class);
        sf.create();        
        List<ClassResourceInfo> resources = ((JAXRSServiceImpl)sf.getService()).getClassResourceInfos();

        String contentTypes = "*/*";
        
        //If acceptContentTypes does not specify a specific Mime type, the  
        //method is declared with a most specific ProduceMime type is selected.
        MetadataMap<String, String> values = new MetadataMap<String, String>();
        OperationResourceInfo ori = findTargetResourceClass(resources, null, "/1/2/",
             "GET", values, contentTypes, getTypes("*/*"));       
        assertNotNull(ori);
        assertEquals("getBooks", ori.getMethodToInvoke().getName());
        assertEquals("Only id and final match groups should be there", 2, values.size());
        assertEquals("2 {id} values should've been picked up", 2, values.get("id").size());
        assertEquals("FINAL_MATCH_GROUP should've been picked up", 1, 
                     values.get(URITemplate.FINAL_MATCH_GROUP).size());
        assertEquals("First {id} is 1", "1", values.getFirst("id"));
        assertEquals("Second id is 2", "2", values.get("id").get(1));
        
        values = new MetadataMap<String, String>();
        ori = findTargetResourceClass(resources, null, "/2",
             "POST", values, contentTypes, getTypes("*/*"));       
        assertNotNull(ori);
        assertEquals("updateBookStoreInfo", ori.getMethodToInvoke().getName());
        assertEquals("Only id and final match groups should be there", 2, values.size());
        assertEquals("Only single {id} should've been picked up", 1, values.get("id").size());
        assertEquals("FINAL_MATCH_GROUP should've been picked up", 1, 
                     values.get(URITemplate.FINAL_MATCH_GROUP).size());
        assertEquals("Only the first {id} should've been picked up", "2", values.getFirst("id"));
        
        values = new MetadataMap<String, String>();
        ori = findTargetResourceClass(resources, null, "/3/4",
             "PUT", values, contentTypes, getTypes("*/*"));       
        assertNotNull(ori);
        assertEquals("updateBook", ori.getMethodToInvoke().getName());
        assertEquals("Only the first {id} should've been picked up", 3, values.size());
        assertEquals("Only the first {id} should've been picked up", 1, values.get("id").size());
        assertEquals("Only the first {id} should've been picked up", 1, values.get("bookId").size());
        assertEquals("Only the first {id} should've been picked up", 1, 
                     values.get(URITemplate.FINAL_MATCH_GROUP).size());
        assertEquals("Only the first {id} should've been picked up", "3", values.getFirst("id"));
        assertEquals("Only the first {id} should've been picked up", "4", values.getFirst("bookId"));
    }
    
    @Test
    public void testFindTargetResourceClassWithSubResource() throws Exception {
        JAXRSServiceFactoryBean sf = new JAXRSServiceFactoryBean();
        sf.setResourceClasses(org.apache.cxf.jaxrs.resources.BookStore.class);
        sf.create();        
        List<ClassResourceInfo> resources = ((JAXRSServiceImpl)sf.getService()).getClassResourceInfos();

        String contentTypes = "*/*";
        
        OperationResourceInfo ori = findTargetResourceClass(resources,
               null, "/bookstore/books/123", "GET", new MetadataMap<String, String>(), contentTypes,
               getTypes("*/*"));       
        assertNotNull(ori);
        assertEquals("getBook", ori.getMethodToInvoke().getName());
        
        ori = findTargetResourceClass(resources, null, 
            "/bookstore/books/123/true/chapter/1", "GET", new MetadataMap<String, String>(), contentTypes,
            getTypes("*/*"));       
        assertNotNull(ori);
        assertEquals("getNewBook", ori.getMethodToInvoke().getName());
        
        ori = findTargetResourceClass(resources, null, "/bookstore/books",
            "POST", new MetadataMap<String, String>(), contentTypes, getTypes("*/*"));      
        assertNotNull(ori);
        assertEquals("addBook", ori.getMethodToInvoke().getName());
        
        ori = findTargetResourceClass(resources, null, "/bookstore/books",
             "PUT", new MetadataMap<String, String>(), contentTypes, getTypes("*/*"));        
        assertNotNull(ori);
        assertEquals("updateBook", ori.getMethodToInvoke().getName());
        
        ori = findTargetResourceClass(resources, null, "/bookstore/books/123",
            "DELETE", new MetadataMap<String, String>(), contentTypes, getTypes("*/*"));        
        assertNotNull(ori);
        assertEquals("deleteBook", ori.getMethodToInvoke().getName());
    }

    @Test
    public void testIntersectMimeTypes() throws Exception {
        //test basic
        List<MediaType> methodMimeTypes = new ArrayList<MediaType>(
             JAXRSUtils.parseMediaTypes("application/mytype,application/xml,application/json"));
        
        MediaType acceptContentType = MediaType.valueOf("application/json");
        List <MediaType> candidateList = JAXRSUtils.intersectMimeTypes(methodMimeTypes, 
                                                 MediaType.valueOf("application/json"));  

        assertEquals(1, candidateList.size());
        assertTrue(candidateList.get(0).toString().equals("application/json"));
        
        //test basic       
        methodMimeTypes = JAXRSUtils.parseMediaTypes(
            "application/mytype, application/json, application/xml");
        candidateList = JAXRSUtils.intersectMimeTypes(methodMimeTypes, 
                                                      MediaType.valueOf("application/json"));  

        assertEquals(1, candidateList.size());
        assertTrue(candidateList.get(0).toString().equals("application/json"));
        
        //test accept wild card */*       
        candidateList = JAXRSUtils.intersectMimeTypes(
            "application/mytype,application/json,application/xml", "*/*");  

        assertEquals(3, candidateList.size());
        
        //test accept wild card application/*       
        methodMimeTypes = JAXRSUtils.parseMediaTypes("text/html,text/xml,application/xml");
        acceptContentType = MediaType.valueOf("text/*");
        candidateList = JAXRSUtils.intersectMimeTypes(methodMimeTypes, acceptContentType);  

        assertEquals(2, candidateList.size());
        for (MediaType type : candidateList) {
            assertTrue("text/html".equals(type.toString()) 
                       || "text/xml".equals(type.toString()));            
        }
        
        //test produce wild card */*
        candidateList = JAXRSUtils.intersectMimeTypes("*/*", "application/json");

        assertEquals(1, candidateList.size());
        assertTrue("application/json".equals(candidateList.get(0).toString()));
        
        //test produce wild card application/*
        candidateList = JAXRSUtils.intersectMimeTypes("application/*", "application/json");  

        assertEquals(1, candidateList.size());
        assertTrue("application/json".equals(candidateList.get(0).toString()));        
        
        //test produce wild card */*, accept wild card */*
        candidateList = JAXRSUtils.intersectMimeTypes("*/*", "*/*");  

        assertEquals(1, candidateList.size());
        assertTrue("*/*".equals(candidateList.get(0).toString()));
    }
    
    @Test
    public void testIntersectMimeTypesTwoArray() throws Exception {
        //test basic
        List <MediaType> acceptedMimeTypes = 
            JAXRSUtils.parseMediaTypes("application/mytype, application/xml, application/json");
        
        List <MediaType> candidateList = 
            JAXRSUtils.intersectMimeTypes(acceptedMimeTypes, JAXRSUtils.ALL_TYPES);

        assertEquals(3, candidateList.size());
        for (MediaType type : candidateList) {
            assertTrue("application/mytype".equals(type.toString()) 
                       || "application/xml".equals(type.toString())
                       || "application/json".equals(type.toString()));
        }
        
        //test basic
        acceptedMimeTypes = Collections.singletonList(JAXRSUtils.ALL_TYPES);
        List<MediaType> providerMimeTypes = 
            JAXRSUtils.parseMediaTypes("application/mytype, application/xml, application/json");

        candidateList = JAXRSUtils.intersectMimeTypes(acceptedMimeTypes, providerMimeTypes);

        assertEquals(3, candidateList.size());
        for (MediaType type : candidateList) {
            assertTrue("application/mytype".equals(type.toString()) 
                       || "application/xml".equals(type.toString())
                       || "application/json".equals(type.toString()));
        }
        
        //test empty
        acceptedMimeTypes = JAXRSUtils.parseMediaTypes("application/mytype,application/xml");
        
        candidateList = JAXRSUtils.intersectMimeTypes(acceptedMimeTypes, 
                                                      MediaType.valueOf("application/json"));

        assertEquals(0, candidateList.size());
    }
    
    @Test
    public void testParseMediaTypes() throws Exception {
        List<MediaType> types = JAXRSUtils.parseMediaTypes("*");
        assertTrue(types.size() == 1 
                   && types.get(0).equals(JAXRSUtils.ALL_TYPES));
        types = JAXRSUtils.parseMediaTypes("text/*");
        assertTrue(types.size() == 1 && types.get(0).equals(new MediaType("text", "*")));
        types = JAXRSUtils.parseMediaTypes("text/*,text/plain;q=.2,text/xml,TEXT/BAR");
        assertTrue(types.size() == 4
                   && "text/*".equals(types.get(0).toString())
                   && "text/plain;q=.2".equals(types.get(1).toString())
                   && "text/xml".equals(types.get(2).toString())
                   && "text/bar".equals(types.get(3).toString()));
        
    }
    
    @Test
    public void testSortMediaTypes() throws Exception {
        List<MediaType> types = 
            JAXRSUtils.sortMediaTypes("text/*,text/plain;q=.2,text/xml,TEXT/BAR");
        assertTrue(types.size() == 4
                   && "text/bar".equals(types.get(0).toString())
                   && "text/plain;q=.2".equals(types.get(1).toString())
                   && "text/xml".equals(types.get(2).toString())
                   && "text/*".equals(types.get(3).toString()));
    }
    
    @Test
    public void testCompareMediaTypes() throws Exception {
        MediaType m1 = MediaType.valueOf("text/xml");
        MediaType m2 = MediaType.valueOf("text/*");
        assertTrue("text/xml is more specific than text/*", 
                   JAXRSUtils.compareMediaTypes(m1, m2) < 0);
        assertTrue("text/* is less specific than text/*", 
                   JAXRSUtils.compareMediaTypes(m2, m1) > 0);
        assertTrue("text/xml should be equal to itself", 
                   JAXRSUtils.compareMediaTypes(m1, new MediaType("text", "xml")) == 0);
        assertTrue("text/* should be equal to itself", 
                   JAXRSUtils.compareMediaTypes(m2, new MediaType("text", "*")) == 0);
        
        assertTrue("text/plain is alphabetically earlier than text/xml", 
                   JAXRSUtils.compareMediaTypes(MediaType.valueOf("text/plain"), m1) < 0);
        assertTrue("text/xml is alphabetically later than text/plain", 
                   JAXRSUtils.compareMediaTypes(m1, MediaType.valueOf("text/plain")) > 0);
        assertTrue("*/* is less specific than text/xml", 
                   JAXRSUtils.compareMediaTypes(JAXRSUtils.ALL_TYPES, m1) > 0);
        assertTrue("*/* is less specific than text/xml", 
                   JAXRSUtils.compareMediaTypes(m1, JAXRSUtils.ALL_TYPES) < 0);
        assertTrue("*/* is less specific than text/*", 
                   JAXRSUtils.compareMediaTypes(JAXRSUtils.ALL_TYPES, m2) > 0);
        assertTrue("*/* is less specific than text/*", 
                   JAXRSUtils.compareMediaTypes(m2, JAXRSUtils.ALL_TYPES) < 0);
        
        MediaType m3 = MediaType.valueOf("text/xml;q=0.2");
        assertTrue("text/xml should be more preferred than than text/xml;q=0.2", 
                   JAXRSUtils.compareMediaTypes(m1, m3) < 0);
        MediaType m4 = MediaType.valueOf("text/xml;q=.3");
        assertTrue("text/xml;q=.3 should be more preferred than than text/xml;q=0.2", 
                   JAXRSUtils.compareMediaTypes(m4, m3) < 0);
    }
    
    @Test
    public void testCompareSortedMediaTypes() throws Exception {
        MediaType m1 = MediaType.valueOf("text/xml");
        MediaType m2 = MediaType.valueOf("text/*");
        assertTrue("text/xml is more specific than text/*", 
                   JAXRSUtils.compareSortedMediaTypes(Collections.singletonList(m1), 
                                                      Collections.singletonList(m2)) < 0);
        assertTrue("text/* is less specific than text/xml", 
                   JAXRSUtils.compareSortedMediaTypes(Collections.singletonList(m2), 
                                                      Collections.singletonList(m1)) > 0);
        
        assertTrue("text/xml is the same as text/xml", 
                   JAXRSUtils.compareSortedMediaTypes(Collections.singletonList(m1), 
                                                      Collections.singletonList(m1)) == 0);
        
        List<MediaType> sortedList1 = new ArrayList<MediaType>();
        sortedList1.add(m1);
        sortedList1.add(m2);
                
        List<MediaType> sortedList2 = new ArrayList<MediaType>();
        sortedList2.add(m1);
        sortedList2.add(m2);
        
        assertTrue("lists should be equal", 
                   JAXRSUtils.compareSortedMediaTypes(sortedList1, sortedList2) == 0);
        
        sortedList1.add(MediaType.valueOf("*/*"));
        assertTrue("first list should be less specific", 
                   JAXRSUtils.compareSortedMediaTypes(sortedList1, sortedList2) > 0);
        sortedList1.add(MediaType.valueOf("*/*"));
        assertTrue("second list should be more specific", 
                   JAXRSUtils.compareSortedMediaTypes(sortedList2, sortedList1) < 0);
    }
    
    @Test
    public void testAcceptTypesMatch() throws Exception {
        
        Method m = Customer.class.getMethod("test", new Class[]{});
        ClassResourceInfo cr = new ClassResourceInfo(Customer.class);
        
        assertTrue("text/xml can not be matched",
                   JAXRSUtils.matchMimeTypes(JAXRSUtils.ALL_TYPES, 
                                             new MediaType("text", "xml"), 
                                             new OperationResourceInfo(m, cr)));
        assertTrue("text/xml can not be matched",
                   JAXRSUtils.matchMimeTypes(JAXRSUtils.ALL_TYPES, 
                                             new MediaType("text", "*"), 
                                             new OperationResourceInfo(m, cr)));
        assertTrue("text/xml can not be matched",
                   JAXRSUtils.matchMimeTypes(JAXRSUtils.ALL_TYPES, 
                                             new MediaType("*", "*"), 
                                             new OperationResourceInfo(m, cr)));
        assertFalse("text/plain was matched",
                   JAXRSUtils.matchMimeTypes(JAXRSUtils.ALL_TYPES, 
                                             new MediaType("text", "plain"), 
                                             new OperationResourceInfo(m, cr)));
    }
 
    
    @Test
    public void testQueryParameters() throws Exception {
        Class[] argType = {String.class, Integer.TYPE};
        Method m = Customer.class.getMethod("testQuery", argType);
        MessageImpl messageImpl = new MessageImpl();
        
        messageImpl.put(Message.QUERY_STRING, "query=24");
        List<Object> params = JAXRSUtils.processParameters(new OperationResourceInfo(m, null),
                                                           null, 
                                                           messageImpl);
        assertEquals("Query Parameter was not matched correctly", "24", params.get(0));
        assertEquals("Primitive Query Parameter was not matched correctly", 24, params.get(1));
        
        
    }
    
    @Test
    public void testQueryParametersBean() throws Exception {
        Class[] argType = {Customer.CustomerBean.class};
        Method m = Customer.class.getMethod("testQueryBean", argType);
        MessageImpl messageImpl = new MessageImpl();
        
        messageImpl.put(Message.QUERY_STRING, "a=aValue&b=123");
        List<Object> params = JAXRSUtils.processParameters(new OperationResourceInfo(m, null),
                                                           null, 
                                                           messageImpl);
        assertEquals("Bean should be created", 1, params.size());
        Customer.CustomerBean cb = (Customer.CustomerBean)params.get(0);
        assertNotNull(cb);
        
        assertEquals("aValue", cb.getA());
        assertEquals(new Long(123), cb.getB());
    }
    
    @Test
    public void testPathParametersBean() throws Exception {
        Class[] argType = {Customer.CustomerBean.class};
        Method m = Customer.class.getMethod("testPathBean", argType);
        MessageImpl messageImpl = new MessageImpl();
        
        MultivaluedMap<String, String> pathTamplates = new MetadataMap<String, String>();
        pathTamplates.add("a", "aValue");
        pathTamplates.add("b", "123");
        List<Object> params = JAXRSUtils.processParameters(new OperationResourceInfo(m, null),
                                                           pathTamplates, 
                                                           messageImpl);
        assertEquals("Bean should be created", 1, params.size());
        Customer.CustomerBean cb = (Customer.CustomerBean)params.get(0);
        assertNotNull(cb);
        
        assertEquals("aValue", cb.getA());
        assertEquals(new Long(123), cb.getB());
    }
    
    @Test
    public void testMatrixParametersBean() throws Exception {
        Class[] argType = {Customer.CustomerBean.class};
        Method m = Customer.class.getMethod("testMatrixBean", argType);
        MessageImpl messageImpl = new MessageImpl();
        messageImpl.put(Message.REQUEST_URI, "/bar;a=aValue/baz;b=123");
        List<Object> params = JAXRSUtils.processParameters(new OperationResourceInfo(m, null),
                                                           new MetadataMap<String, String>(), 
                                                           messageImpl);
        assertEquals("Bean should be created", 1, params.size());
        Customer.CustomerBean cb = (Customer.CustomerBean)params.get(0);
        assertNotNull(cb);
        
        assertEquals("aValue", cb.getA());
        assertEquals(new Long(123), cb.getB());
    }
    
    
    @Test
    public void testMultipleQueryParameters() throws Exception {
        Class[] argType = {String.class, String.class, Long.class, 
                           Boolean.TYPE, String.class};
        Method m = Customer.class.getMethod("testMultipleQuery", argType);
        MessageImpl messageImpl = new MessageImpl();
        
        messageImpl.put(Message.QUERY_STRING, 
                        "query=first&query2=second&query3=3&query4=true&query5");
        List<Object> params = JAXRSUtils.processParameters(new OperationResourceInfo(m, null), 
                                                           null, messageImpl);
        assertEquals("First Query Parameter of multiple was not matched correctly", "first", 
                     params.get(0));
        assertEquals("Second Query Parameter of multiple was not matched correctly", 
                     "second", params.get(1));
        assertEquals("Third Query Parameter of multiple was not matched correctly", 
                     new Long(3), params.get(2));
        assertEquals("Fourth Query Parameter of multiple was not matched correctly", 
                     Boolean.TRUE, params.get(3));
        assertEquals("Fourth Query Parameter of multiple was not matched correctly", 
                     "", params.get(4));
    }
    
    @SuppressWarnings("unchecked")
    @Test
    public void testMatrixParameters() throws Exception {
        Class[] argType = {String.class, String.class, String.class, String.class, List.class};
        Method m = Customer.class.getMethod("testMatrixParam", argType);
        MessageImpl messageImpl = new MessageImpl();
        
        messageImpl.put(Message.REQUEST_URI, "/foo;p4=0;p3=3/bar;p1=1;p2/baz;p4=4;p4=5");
        List<Object> params = JAXRSUtils.processParameters(new OperationResourceInfo(m, null), 
                                                           null, messageImpl);
        assertEquals("5 Matrix params should've been identified", 5, params.size());
        
        assertEquals("First Matrix Parameter not matched correctly", 
                     "1", params.get(0));
        assertEquals("Second Matrix Parameter was not matched correctly", 
                     "", params.get(1));
        assertEquals("Third Matrix Parameter was not matched correctly", 
                     "3", params.get(2));
        assertEquals("Fourth Matrix Parameter was not matched correctly", 
                     "0", params.get(3));
        List<String> list = (List<String>)params.get(4);
        assertEquals(3, list.size());
        assertEquals("0", list.get(0));
        assertEquals("4", list.get(1));
        assertEquals("5", list.get(2));
    }
    
    @SuppressWarnings("unchecked")
    @Test
    public void testMatrixAndPathSegmentParameters() throws Exception {
        Class[] argType = {PathSegment.class, String.class};
        Method m = Customer.class.getMethod("testPathSegment", argType);
        MessageImpl messageImpl = new MessageImpl();
        messageImpl.put(Message.REQUEST_URI, "/bar%20foo;p4=0%201");
        MultivaluedMap<String, String> values = new MetadataMap<String, String>();
        values.add("ps", "bar%20foo;p4=0%201");
        List<Object> params = JAXRSUtils.processParameters(new OperationResourceInfo(m, null), 
                                                           values, 
                                                           messageImpl);
        assertEquals("2 params should've been identified", 2, params.size());
        
        PathSegment ps = (PathSegment)params.get(0);
        assertEquals("bar foo", ps.getPath());
        assertEquals(1, ps.getMatrixParameters().size());
        assertEquals("0 1", ps.getMatrixParameters().getFirst("p4"));
        assertEquals("bar foo", params.get(1));
    }
    
    
    @Test
    public void testSelectResourceMethod() throws Exception {
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class);
        OperationResourceInfo ori1 = new OperationResourceInfo(
                                         Customer.class.getMethod("getItAsXML", new Class[]{}), 
                                         cri);
        ori1.setHttpMethod("GET");
        ori1.setURITemplate(new URITemplate("/"));
        OperationResourceInfo ori2 = new OperationResourceInfo(
                                         Customer.class.getMethod("getItPlain", new Class[]{}), 
                                         cri);
        ori2.setHttpMethod("GET");
        ori2.setURITemplate(new URITemplate("/"));
        MethodDispatcher md = new MethodDispatcher(); 
        md.bind(ori1, Customer.class.getMethod("getItAsXML", new Class[]{}));
        md.bind(ori2, Customer.class.getMethod("getItPlain", new Class[]{}));
        cri.setMethodDispatcher(md);
        
        OperationResourceInfo ori = JAXRSUtils.findTargetMethod(cri, "/", "GET", 
              new MetadataMap<String, String>(), "*/*", getTypes("text/plain"));
        
        assertSame(ori, ori2);
        
        ori = JAXRSUtils.findTargetMethod(cri, "/", "GET", new MetadataMap<String, String>(), 
                                              "*/*", getTypes("text/xml"));
                         
        assertSame(ori, ori1);
        
        ori = JAXRSUtils.findTargetMethod(cri, "/", "GET", new MetadataMap<String, String>(), 
                                          "*/*", getTypes("*,text/plain,text/xml"));
                     
        assertSame(ori, ori2);
        ori = JAXRSUtils.findTargetMethod(cri, "/", "GET", new MetadataMap<String, String>(), 
                                          "*/*", getTypes("*,x/y,text/xml,text/plain"));
                     
        assertSame(ori, ori2);
    }
    
    @SuppressWarnings("unchecked")
    @Test
    public void testHttpContextParameters() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        OperationResourceInfo ori = 
            new OperationResourceInfo(
                Customer.class.getMethod("testParams", 
                                         new Class[]{UriInfo.class, 
                                                     HttpHeaders.class, 
                                                     Request.class,
                                                     SecurityContext.class,
                                                     MessageBodyWorkers.class,
                                                     String.class,
                                                     List.class}), 
                cri);
        ori.setHttpMethod("GET");
        MultivaluedMap<String, String> headers = new MetadataMap<String, String>();
        headers.add("Foo", "bar, baz");
        
        Message m = new MessageImpl();
        m.put(Message.PROTOCOL_HEADERS, headers);
        
        List<Object> params = 
            JAXRSUtils.processParameters(ori, new MetadataMap<String, String>(), m);
        assertEquals("7 parameters expected", 7, params.size());
        assertSame(UriInfoImpl.class, params.get(0).getClass());
        assertSame(HttpHeadersImpl.class, params.get(1).getClass());
        assertSame(RequestImpl.class, params.get(2).getClass());
        assertSame(SecurityContextImpl.class, params.get(3).getClass());
        assertSame(ProvidersImpl.class, params.get(4).getClass());
        assertSame(String.class, params.get(5).getClass());
        assertEquals("Wrong header param", "bar", params.get(5));
        List<String> values = (List<String>)params.get(6);
        assertEquals("Wrong headers size", 2, values.size());
        assertEquals("Wrong 1st header param", "bar", values.get(0));
        assertEquals("Wrong 2nd header param", "baz", values.get(1));
    }
    
    @Test
    public void testHttpContextParametersFromInterface() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        Method methodToInvoke = 
            Customer.class.getMethod("setUriInfoContext", 
                                     new Class[]{UriInfo.class});
        OperationResourceInfo ori = 
            new OperationResourceInfo(methodToInvoke, cri);
        ori.setHttpMethod("GET");
        ori.setAnnotatedMethod(AnnotationUtils.getAnnotatedMethod(methodToInvoke));
        
        
        Message m = new MessageImpl();
        
        List<Object> params = 
            JAXRSUtils.processParameters(ori, new MetadataMap<String, String>(), m);
        assertEquals("1 parameters expected", 1, params.size());
        assertSame(UriInfoImpl.class, params.get(0).getClass());
    }
    
    @Test
    public void testServletContextParameters() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        OperationResourceInfo ori = 
            new OperationResourceInfo(
                Customer.class.getMethod("testServletParams", 
                                         new Class[]{HttpServletRequest.class, 
                                                     HttpServletResponse.class, 
                                                     ServletContext.class,
                                                     ServletConfig.class}), 
                cri);
        ori.setHttpMethod("GET");
        HttpServletRequest request = EasyMock.createMock(HttpServletRequest.class);
        HttpServletResponse response = EasyMock.createMock(HttpServletResponse.class);
        ServletContext context = EasyMock.createMock(ServletContext.class);
        ServletConfig config = EasyMock.createMock(ServletConfig.class);        
        
        EasyMock.replay(request);
        EasyMock.replay(response);
        EasyMock.replay(context);
        EasyMock.replay(config);
        
        Message m = new MessageImpl();
        m.put(AbstractHTTPDestination.HTTP_REQUEST, request);
        m.put(AbstractHTTPDestination.HTTP_RESPONSE, response);
        m.put(AbstractHTTPDestination.HTTP_CONTEXT, context);
        m.put(AbstractHTTPDestination.HTTP_CONFIG, config);
        
        List<Object> params = 
            JAXRSUtils.processParameters(ori, new MetadataMap<String, String>(), m);
        assertEquals("4 parameters expected", 4, params.size());
        assertSame(request.getClass(), params.get(0).getClass());
        assertSame(response.getClass(), params.get(1).getClass());
        assertSame(context.getClass(), params.get(2).getClass());
        assertSame(config.getClass(), params.get(3).getClass());
        
    }
    
    @Test
    public void testPerRequestHttpContextFields() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        cri.setResourceProvider(new PerRequestResourceProvider(Customer.class));
        OperationResourceInfo ori = new OperationResourceInfo(null, cri);
        
        Customer c = new Customer();
        
        Message m = new MessageImpl();
        m.put(Message.PROTOCOL_HEADERS, new HashMap<String, List<String>>());
        
        InjectionUtils.injectContextFields(c, ori.getClassResourceInfo(), m);
        assertSame(UriInfoImpl.class, c.getUriInfo2().getClass());
        assertSame(HttpHeadersImpl.class, c.getHeaders().getClass());
        assertSame(RequestImpl.class, c.getRequest().getClass());
        assertSame(SecurityContextImpl.class, c.getSecurityContext().getClass());
        assertSame(ProvidersImpl.class, c.getBodyWorkers().getClass());
        
    }
    
    @Test
    public void testSingletonHttpContextFields() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        Customer c = new Customer();
        cri.setResourceProvider(new SingletonResourceProvider(c));
        
        Message m = new MessageImpl();
        m.put(Message.PROTOCOL_HEADERS, new HashMap<String, List<String>>());
        ServletContext servletContextMock = EasyMock.createNiceMock(ServletContext.class);
        m.put(AbstractHTTPDestination.HTTP_CONTEXT, servletContextMock);
        HttpServletRequest httpRequest = EasyMock.createNiceMock(HttpServletRequest.class);
        m.put(AbstractHTTPDestination.HTTP_REQUEST, httpRequest);
        HttpServletResponse httpResponse = EasyMock.createNiceMock(HttpServletResponse.class);
        m.put(AbstractHTTPDestination.HTTP_RESPONSE, httpResponse);
        
        InjectionUtils.injectContextProxies(cri, cri.getResourceProvider().getInstance());
        InjectionUtils.injectContextFields(c, cri, m);
        InjectionUtils.injectContextMethods(c, cri, m);
        assertSame(ThreadLocalUriInfo.class, c.getUriInfo2().getClass());
        assertSame(UriInfoImpl.class, 
                   ((ThreadLocalProxy)c.getUriInfo2()).get().getClass());
        assertSame(HttpHeadersImpl.class, 
                   ((ThreadLocalProxy)c.getHeaders()).get().getClass());
        assertSame(RequestImpl.class, 
                   ((ThreadLocalProxy)c.getRequest()).get().getClass());
        assertSame(SecurityContextImpl.class, 
                   ((ThreadLocalProxy)c.getSecurityContext()).get().getClass());
        assertSame(ProvidersImpl.class, 
                   ((ThreadLocalProxy)c.getBodyWorkers()).get().getClass());

        assertSame(servletContextMock, 
                   ((ThreadLocalProxy)c.getThreadLocalServletContext()).get());

        assertSame(servletContextMock, 
                   ((ThreadLocalProxy)c.getServletContext()).get());
        
        assertSame(httpRequest, 
                   ((ThreadLocalProxy)c.getServletRequest()).get());
        assertSame(httpResponse, 
                   ((ThreadLocalProxy)c.getServletResponse()).get());
    }
    
    @Test
    public void testSingletonHttpResourceFields() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        Customer c = new Customer();
        cri.setResourceProvider(new SingletonResourceProvider(c));
                
        Message m = new MessageImpl();
        ServletContext servletContextMock = EasyMock.createNiceMock(ServletContext.class);
        m.put(AbstractHTTPDestination.HTTP_CONTEXT, servletContextMock);
        HttpServletRequest httpRequest = EasyMock.createNiceMock(HttpServletRequest.class);
        m.put(AbstractHTTPDestination.HTTP_REQUEST, httpRequest);
        HttpServletResponse httpResponse = EasyMock.createNiceMock(HttpServletResponse.class);
        m.put(AbstractHTTPDestination.HTTP_RESPONSE, httpResponse);
        InjectionUtils.injectContextProxies(cri, cri.getResourceProvider().getInstance());
        InjectionUtils.injectResourceFields(c, cri, m);
        assertSame(servletContextMock, 
                   ((ThreadLocalProxy)c.getServletContextResource()).get());
        assertSame(httpRequest, 
                   ((ThreadLocalProxy)c.getServletRequestResource()).get());
        assertSame(httpResponse, 
                   ((ThreadLocalProxy)c.getServletResponseResource()).get());
    }
    
    @Test
    public void testContextAnnotationOnMethod() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        Customer c = new Customer();
        cri.setResourceProvider(new SingletonResourceProvider(c));
        InjectionUtils.injectContextProxies(cri, cri.getResourceProvider().getInstance());
        
        OperationResourceInfo ori = new OperationResourceInfo(Customer.class.getMethods()[0],
                                                              cri); 
        
        
        JAXRSUtils.handleSetters(ori, c, new MessageImpl());
        assertNotNull(c.getUriInfo());
        assertSame(ThreadLocalUriInfo.class, c.getUriInfo().getClass());
        assertSame(UriInfoImpl.class, 
                   ((ThreadLocalProxy)c.getUriInfo()).get().getClass());
    }
    
    @Test
    public void testParamAnnotationOnMethod() throws Exception {

        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        Customer c = new Customer();
        OperationResourceInfo ori = new OperationResourceInfo(Customer.class.getMethods()[0],
                                                              cri);
        Message m = new MessageImpl();
        m.put(Message.QUERY_STRING, "a=aValue&query2=b");
        JAXRSUtils.handleSetters(ori, c, m);
        assertEquals("aValue", c.getQueryParam());
        
    }
    
    @Test
    public void testParamAnnotationOnField() throws Exception {

        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        Customer c = new Customer();
        OperationResourceInfo ori = new OperationResourceInfo(Customer.class.getMethods()[0],
                                                              cri);
        Message m = new MessageImpl();
        m.put(Message.QUERY_STRING, "b=bValue");
        JAXRSUtils.handleSetters(ori, c, m);
        assertEquals("bValue", c.getB());
        
    }

    @Test
    public void testContextResolverParam() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        OperationResourceInfo ori = 
            new OperationResourceInfo(
                Customer.class.getMethod("testContextResolvers", 
                                         new Class[]{ContextResolver.class}), 
                                         cri);
        ori.setHttpMethod("GET");
        
        ContextResolver<JAXBContext> cr = new JAXBContextProvider();
        ProviderFactory.getInstance().registerUserProvider(cr);
        
        Message m = new MessageImpl();
        m.put(Message.BASE_PATH, "/");    
        List<Object> params = 
            JAXRSUtils.processParameters(ori, new MetadataMap<String, String>(), m);
        assertEquals("1 parameters expected", 1, params.size());
        assertSame(cr.getClass(), params.get(0).getClass());
    }
    
    @Test
    public void testContextResolverFields() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        cri.setResourceProvider(new PerRequestResourceProvider(Customer.class));
        OperationResourceInfo ori = new OperationResourceInfo(null, cri);
        
        Customer c = new Customer();
        ContextResolver<JAXBContext> cr = new JAXBContextProvider();
        ProviderFactory.getInstance().registerUserProvider(cr);
        
        Message m = new MessageImpl();
        m.put(Message.BASE_PATH, "/");    
        InjectionUtils.injectContextFields(c, ori.getClassResourceInfo(), m);
        assertSame(cr.getClass(), c.getContextResolver().getClass());
    }
    
    @Test
    public void testServletResourceFields() throws Exception {
        
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        cri.setResourceProvider(new PerRequestResourceProvider(Customer.class));
        OperationResourceInfo ori = new OperationResourceInfo(null, cri);
        
        Customer c = new Customer();
        
        // Creating mocks for the servlet request, response and context
        HttpServletRequest request = EasyMock.createMock(HttpServletRequest.class);
        HttpServletResponse response = EasyMock.createMock(HttpServletResponse.class);
        ServletContext context = EasyMock.createMock(ServletContext.class);
        EasyMock.replay(request);
        EasyMock.replay(response);
        EasyMock.replay(context);
        
        Message m = new MessageImpl();
        m.put(AbstractHTTPDestination.HTTP_REQUEST, request);
        m.put(AbstractHTTPDestination.HTTP_RESPONSE, response);
        m.put(AbstractHTTPDestination.HTTP_CONTEXT, context);
        
        InjectionUtils.injectResourceFields(c, ori.getClassResourceInfo(), m);
        assertSame(request.getClass(), c.getServletRequestResource().getClass());
        assertSame(response.getClass(), c.getServletResponseResource().getClass());
        assertSame(context.getClass(), c.getServletContextResource().getClass());
        assertNull(c.getServletRequest());
        assertNull(c.getServletResponse());
        assertNull(c.getServletContext());
        
        c = new Customer();
        InjectionUtils.injectContextFields(c, ori.getClassResourceInfo(), m);
        assertNull(c.getServletRequestResource());
        assertNull(c.getServletResponseResource());
        assertNull(c.getServletContextResource());
        assertSame(request.getClass(), c.getServletRequest().getClass());
        assertSame(response.getClass(), c.getServletResponse().getClass());
        assertSame(context.getClass(), c.getServletContext().getClass());
    }
    
    @Test
    public void testConversion() throws Exception {
        ClassResourceInfo cri = new ClassResourceInfo(Customer.class, true);
        OperationResourceInfo ori = 
            new OperationResourceInfo(
                Customer.class.getMethod("testConversion", 
                                         new Class[]{PathSegmentImpl.class, 
                                                     SimpleFactory.class}), 
                cri);
        ori.setHttpMethod("GET");
        ori.setURITemplate(new URITemplate("{id1}/{id2}"));
        MultivaluedMap<String, String> values = new MetadataMap<String, String>();
        values.putSingle("id1", "1");
        values.putSingle("id2", "2");
        
        Message m = new MessageImpl();
        
        
        List<Object> params = 
            JAXRSUtils.processParameters(ori, values, m);
        PathSegment ps = (PathSegment)params.get(0);
        assertEquals("1", ps.getPath());
        
        SimpleFactory sf = (SimpleFactory)params.get(1);
        assertEquals(2, sf.getId());
    }
    
    private static OperationResourceInfo findTargetResourceClass(List<ClassResourceInfo> resources,
                                                                Message message,
                                                                String path, 
                                                                String httpMethod,
                                                                MultivaluedMap<String, String> values,
                                                                String requestContentType, 
                                                                List<MediaType> acceptContentTypes) {
        
        ClassResourceInfo resource = JAXRSUtils.selectResourceClass(resources, path, values);
        
        if (resource != null) {
            String subResourcePath = values.getFirst(URITemplate.FINAL_MATCH_GROUP);
            OperationResourceInfo ori = JAXRSUtils.findTargetMethod(resource, subResourcePath, httpMethod, 
                                                   values, requestContentType, acceptContentTypes);
            if (ori != null) {
                return ori;
            }
        }
        
        return null;
    }
}
