/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.cxf.bus.extension;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.cxf.Bus;
import org.apache.cxf.common.util.StringUtils;

public class Extension {
   
    protected String className;
    protected ClassLoader classloader;
    protected Class<?> clazz;
    protected String interfaceName;
    protected boolean deferred;
    protected Collection<String> namespaces = new ArrayList<String>();
    protected Object obj;
    
    public Extension() {
    }
    public Extension(Extension ext) {
        className = ext.className;
        interfaceName = ext.interfaceName;
        deferred = ext.deferred;
        namespaces = ext.namespaces;
        obj = ext.obj;
        clazz = ext.clazz;
        classloader = ext.classloader;
    }
    
    public String getName() {
        return StringUtils.isEmpty(interfaceName) ? className : interfaceName;
    }
    public Object getLoadedObject() {
        return obj;
    }
    
    public Extension cloneNoObject() {
        Extension ext = new Extension(this);
        ext.obj = null;
        ext.clazz = null;
        return ext;
    }
    
    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append("class: ");
        buf.append(className);
        buf.append(", interface: ");
        buf.append(interfaceName);
        buf.append(", deferred: ");
        buf.append(deferred ? "true" : "false");
        buf.append(", namespaces: (");
        int n = 0;
        for (String ns : namespaces) {
            if (n > 0) {
                buf.append(", ");
            }
            buf.append(ns);
            n++;
        }
        buf.append(")");
        return buf.toString();        
    }
    
    public String getClassname() {
        return className;
    }
    
    public void setClassname(String i) {
        clazz = null;
        className = i;
    }
       
    public String getInterfaceName() {
        return interfaceName;
    }

    public void setInterfaceName(String i) {
        interfaceName = i;
    }

    public boolean isDeferred() {
        return deferred;
    }
    
    public void setDeferred(boolean d) {
        deferred = d;
    }
    
    public Collection<String> getNamespaces() {
        return namespaces;
    }
    
    
    
    
    public Class<?> getClassObject(ClassLoader cl) {
        if (clazz == null) {
            if (classloader != null) {
                try {
                    clazz = classloader.loadClass(className);
                    return clazz;
                } catch (ClassNotFoundException nex) {
                    //ignore, fall into the stuff below
                }
            }                
            try {
                clazz = cl.loadClass(className);
            } catch (ClassNotFoundException ex) {
                try {
                    // using the extension classloader as a fallback
                    clazz = this.getClass().getClassLoader().loadClass(className);
                } catch (ClassNotFoundException nex) {
                    throw new ExtensionException(nex);
                }
            }
        }
        return clazz;
    }
    public Object load(ClassLoader cl, Bus b) {
        try {
            Class<?> cls = getClassObject(cl);
            try {
                //if there is a Bus constructor, use it.
                if (b != null) {
                    Constructor con = cls.getConstructor(Bus.class);
                    obj = con.newInstance(b);
                    return obj;
                }
            } catch (Exception ex) {
                //ignore
            }
            obj = cls.newInstance();
        } catch (IllegalAccessException ex) {
            throw new ExtensionException(ex);
        } catch (InstantiationException ex) {
            throw new ExtensionException(ex);
        }
        
        return obj;
    }
    
    public Class loadInterface(ClassLoader cl) {
        Class<?> cls = null;
        if (classloader != null) {
            try {
                return classloader.loadClass(interfaceName);
            } catch (ClassNotFoundException nex) {
                //ignore, fall into the stuff below
            }
        }                

        try {
            cls = cl.loadClass(interfaceName);
        } catch (ClassNotFoundException ex) {
            try {
                // using the extension classloader as a fallback
                cls = this.getClass().getClassLoader().loadClass(interfaceName);
            } catch (ClassNotFoundException nex) {
                throw new ExtensionException(nex);
            }
        }
        return cls;
    }
    
    
}
